/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a cob of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include <unistd.h>
#include <QDir>
#include <QStringList>

#include "Settings.h"
#include "Util.h"
	

//======================================================================
void Util::setSetting(const QString &key, const QVariant &value)
{
	Settings::setUserSetting(key, value);
}
//---------------------------------------------------------------------
QVariant Util::getSetting(const QString &key, const QVariant &defaultValue)
{
	return Settings::getUserSetting(key, defaultValue);
}

//======================================================================
bool Util::isDirWritable (const QDir &dir)
{
	if (! dir.exists())
		return false;
		
	// try yo write a file
	FILE *fd;
	QString tmpfname = dir.absolutePath() + "/jgj13642hygg54hjgiouhg43.tmp";
	fd = fopen( qPrintable(tmpfname), "w");
	if (fd != NULL
			&& fwrite(&tmpfname,1,1,fd)==1 )
	{
		fclose(fd);
		unlink( qPrintable(tmpfname) );
		return true;
	}
	else
		return false;
}
    
//======================================================================
QString Util::formatTemperature(float tempKelvin)
{
    QString tunit = Util::getSetting("unitsTemp", "").toString();
    QString unit = (tunit=="") ? QObject::tr("°C") : tunit;
    QString r;
    if (unit == QObject::tr("°C")) {
        r.sprintf("%.1f ", tempKelvin-273.15);
    }
    else if (unit == QObject::tr("°F")) {
        r.sprintf("%.1f ", 1.8*(tempKelvin-273.15)+32.0);
    }
    else  {   // if (unit == QObject::tr("°K"))
        unit = QObject::tr("°K");
        r.sprintf("%.1f ", tempKelvin);
    }
    return r+unit;
}
//-------------------------------------------------------
QString Util::formatTemperature_short(float tempKelvin)
{
    QString tunit = Util::getSetting("unitsTemp", "").toString();
    QString unit = (tunit=="") ? QObject::tr("°C") : tunit;
    QString r;
    if (unit == QObject::tr("°C")) {
        r.sprintf("%d", qRound(tempKelvin-273.15) );
    }
    else if (unit == QObject::tr("°F")) {
        r.sprintf("%d", qRound(1.8*(tempKelvin-273.15)+32.0) );
    }
    else  {   // if (unit == QObject::tr("°K"))
        unit = QObject::tr("°K");
        r.sprintf("%d", qRound(tempKelvin) );
    }
    return r; //+unit;
}
//----------------------------------------------------------------
QString Util::formatSpeed(float meterspersecond)
{
    QString tunit = Util::getSetting("unitsWindSpeed", "").toString();
    QString unit = (tunit=="") ? QObject::tr("km/h") : tunit;
    QString r;
    if (unit == "m/s") {
        r.sprintf("%.1f ", meterspersecond);
        r += QObject::tr("m/s");
    }
    else if (unit == "km/h") {
        r.sprintf("%.1f ", meterspersecond*3.6);
        r += QObject::tr("km/h");
    }
    else  {   // if (unit == QObject::tr("nœuds"))
        r.sprintf("%.1f ", meterspersecond*3.6/1.852);
        r += QObject::tr("nœuds");
    }
    return r;
}
//----------------------------------------------------------------
QString Util::formatDistance(float mille)
{
    QString tunit = Util::getSetting("unitsDistance", "").toString();
    QString unit = (tunit=="") ? QObject::tr("km") : tunit;
    QString r, unite;
    float d;
    if (unit == QObject::tr("km")) {
        unite = "km";
        d= mille*1.852;
    }
    else  {
        unite = "NM";
        d = mille;
    }
    if (d<10)
        r.sprintf("%5.2f %s", d, qPrintable(unite));
    else if (d<100)
        r.sprintf("%5.1f %s", d, qPrintable(unite));
    else
        r.sprintf("%5.0f %s", d, qPrintable(unite));
    return r;
}
//----------------------------------------------------------------
QString Util::formatAltitude(float meter)
{
    QString tunit = Util::getSetting("unitsAltitude", "").toString();
    QString unit = (tunit=="") ? QObject::tr("m") : tunit;
    QString r, unite;
    float d;
    if (unit == QObject::tr("m")) {
        unite = "m";
        d= meter;
    }
    else {
        unite = "ft";
        d= 10*((int) (meter/0.3048/10.0));
    }
	r.sprintf("%.0f %s", d, qPrintable(unite));
    return r;
}
//----------------------------------------------------------------
QString Util::formatSnowDepth(float meter)
{
    QString tunit = Util::getSetting("unitsAltitude", "").toString();
    QString unit = (tunit=="") ? QObject::tr("m") : tunit;
    QString r, unite;
    float d;
    if (unit == QObject::tr("m")) {
        unite = "cm";
        d= meter*100.0;
        if (d < 10)
			r.sprintf("%.1f %s", d, qPrintable(unite));
		else
			r.sprintf("%.0f %s", d, qPrintable(unite));
    }
    else {
        unite = "ft";
        d= meter/0.3048;
		r.sprintf("%.2f %s", d, qPrintable(unite));
    }
    return r;
}
//----------------------------------------------------------------
QString Util::formatDegres(float x, bool inf100)     // 123.4 -> 123°24.00'
{
	const char *cdeg = "°";
    QString tunit = Util::getSetting("unitsPosition", "").toString();
    QString unit = (tunit=="") ? QObject::tr("dd°mm'ss\"") : tunit;
    
    QString r;
    if (unit == QObject::tr("dd°mm,mm'"))
    {
        int deg = (int) fabs(x);
        float min = (fabs(x) - deg)*60.0;
        char sign = (x<0) ? '-' : ' ';
        if (inf100)
        	r.sprintf("%c%02d%s%05.2f'", sign,deg,cdeg, min);
		else
        	r.sprintf("%c%03d%s%05.2f'", sign,deg,cdeg, min);
    }
    else if (unit == QObject::tr("dd°mm'ss\""))
    {
        int sec = (int) fabs(x*3600.0);  // total en secondes
        int min = sec / 60;              // nombre entier de minutes
        int deg = min / 60;              // nombre entier de degrés
        min = min % 60;                  // reste en minutes
        sec = sec % 60;                  // reste en secondes
        char sign = (x<0) ? '-' : ' ';
        if (inf100)
	        r.sprintf("%c%02d%s%02d'%02d\"", sign,deg,cdeg, min,sec);
		else
	        r.sprintf("%c%03d%s%02d'%02d\"", sign,deg,cdeg, min,sec);
    }
    else // if (unit == QObject::tr("dd,dd°"))
    {
        if (inf100)
    	    r.sprintf("%05.2f%s",x,cdeg);
		else
    	    r.sprintf("%06.2f%s",x,cdeg);
    }
    return r;
}
//---------------------------------------------------------------------
QString Util::formatPosition(float x, float y)  // 123°24.00'W 45°67.89'N
{
    if ( Util::getSetting("orderLatitudeLongitude", true).toBool() )
		return formatLatitude(y)+" "+formatLongitude(x);
	else
		return formatLongitude(x)+" "+formatLatitude(y);
}
//---------------------------------------------------------------------
QString Util::formatLongitude(float x)
{
    QString dir = Util::getSetting("longitudeDirection", "").toString();
	while (x > 360)
		x -= 360;
	while (x < -360)
		x += 360;
    if (dir == "West+") {
    	if (x <= 0)
	    	return formatDegres(-x, false)+"W";
		else
	    	return formatDegres(360-x, false)+"W";
	}
	else if (dir == "East+") {
    	if (x >= 0)
	    	return formatDegres(x, false)+"E";
		else
	    	return formatDegres(360+x, false)+"E";
	}
	else {
		// Mode automatique
		if (x > 0) {
			if (x <= 180)
    			return formatDegres(x, false)+"E";
			else
    			return formatDegres(360-x, false)+"W";
		}
		else {
			if (x >= -180)
	    		return formatDegres(-x, false)+"W";
			else
    			return formatDegres(x+360, false)+"E";
		}
	}
}
//---------------------------------------------------------------------
QString Util::formatLatitude(float y)
{
    QString dir = Util::getSetting("latitudeDirection", "").toString();
    if (dir == "South+")
	    return formatDegres(-y, true)+"S";
	else if (dir == "North+")
	    return formatDegres(y, true)+"N";
	else {
		// Mode automatique
		if (y > 0)
		    return formatDegres(y, true)+"N";
		else	
	    	return formatDegres(-y, true)+"S";
	}
}
//---------------------------------------------------------------------
QString Util::formatPercentValue(float v)
{
    QString r;
    if (v<0)
        v=0;
    else if (v>100)
        v=100;    
    r.sprintf("%d %%", (int)floor(v+0.5));
    return r;
}

//======================================================================
QString Util::formatDayName(QDate dt)
{
	QString  sday;
	if (getSetting("appLanguage", "").toString() == "fr")
	{
		switch (dt.dayOfWeek()) {	// car la locale french ajoute un pt. inutile
			case 1: sday="Lun"; break;
			case 2: sday="Mar"; break;
			case 3: sday="Mer"; break;
			case 4: sday="Jeu"; break;
			case 5: sday="Ven"; break;
			case 6: sday="Sam"; break;
			case 7: sday="Dim"; break;
			default: sday=""; break;
		}
	}
    else {
		QLocale loc;
		sday = loc.toString(dt, "ddd");
	}
	return sday;
}
//------------------------------------------------
QString Util::formatDateLong(time_t t)
{
    QDateTime dt = applyTimeZone(t);
	return formatDayName(dt.date())+" "+formatDateShort(t);
}
//---------------------------------------------------------------------
QString Util::formatDateShort(time_t t)
{
    QDateTime dt = applyTimeZone(t);
	if (getSetting("appLanguage", "").toString() == "fr")
	    return dt.toString("dd-MM-yyyy");
	else
	    return dt.toString("yyyy-MM-dd");
}
//---------------------------------------------------------------------
QString Util::formatDateTimeLong(time_t t)
{
	return formatDateLong(t)+" "+formatTime(t);
}
//---------------------------------------------------------------------
QString Util::formatDateTimeShort(time_t t)
{
	return formatDateShort(t)+" "+formatTime(t);
}
//---------------------------------------------------------------------
QString Util::formatTime(time_t t)
{
	QString suffix;
    QDateTime dt = applyTimeZone(t, &suffix);
    return dt.toString("hh:mm ")+suffix;
}
//---------------------------------------------------------------------
QDateTime Util::applyTimeZone(time_t t, QString *suffix)
{
    QDateTime dt;
    dt.setTime_t(t);
    //dt.setTimeSpec(Qt::UTC);
	dt = dt.toUTC();
	
	QString tmzone =  Util::getSetting("timeZone", "UTC").toString();
	if (tmzone == "LOC") {
		dt = dt.toLocalTime();
		if (suffix != NULL)
			*suffix = "LOC";
	}
	else if (tmzone.left(4)=="UTC+" || tmzone.left(4)=="UTC-")
	{    // UTC-12 UTC-11 ... UTC+1 UTC+2 UTC+3 ... UTC+14
		int dec = tmzone.mid(3,-1).toInt();
		if (dec==0 || dec<-12 || dec>14) {
			if (suffix != NULL)
				*suffix = "UTC";
		}
		else {
			dt = dt.addSecs(dec*3600);
			if (suffix != NULL)
				*suffix = tmzone;
		}
	}
	else
	{	// default timezone : UTC
		if (suffix != NULL)
			*suffix = "UTC";
	}
    
    return dt;
}

//============================================================================
double Util::distancePointSegment (double a,double b,     // point
								  double x0, double y0,  // segment start
								  double x1, double y1   // segment end
							)
{
	double dx, dy, t, dist;
	double hx, hy;

	dx = x1 - x0;
	dy = y1 - y0;

	if (dx == 0 && dy == 0) {	//  the segment is a single point
		dx = a - x0;
		dy = b - y0;
		dist = sqrt(dx*dx + dy*dy);
	}
	else
	{
		// Projection on segment line
		t = ((a-x0)*dx + (b-y0)*dy) / (dx*dx + dy*dy);
		if (t < 0) {
			dx = a-x0;
			dy = b-y0;
			hx = x0;
			hy = y0;
		}
		else if (t > 1) {
			dx = a-x1;
			dy = b-y1;
			hx = x1;
			hy = y1;
		}
		else {
			hx = x0 + t*dx;
			hy = y0 + t*dy;
			dx = a-hx;
			dy = b-hy;
		}
		dist = sqrt(dx*dx + dy*dy);
	}
	return dist;
}








