/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include <QMessageBox>
#include <cmath>
#include <cassert>

#include "MeteoTable.h"
#include "GribPlot.h"
#include "Util.h"
#include "Settings.h"


//-------------------------------------------------------------------------------
MeteoTableWidget::MeteoTableWidget(GribPlot *gribplot, double lon, double lat, QWidget *parent)
	: QWidget(parent)
{
	this->gribplot = gribplot;
	this->lon = lon;
	this->lat = lat;

	layout = new QGridLayout(this);
	layout->setContentsMargins(0,0,0,0);
	layout->setSpacing(0);
	
	createTable();
}

//-------------------------------------------------------------------------------
MeteoTableWidget::~MeteoTableWidget()
{
	Util::cleanListPointers(lspinfos);
	qDeleteAll(listVisibleData);
}

//-------------------------------------------------------------------------------
void MeteoTableWidget::createTable()
{
	GribReader *gribread = gribplot->getGribReader();
	std::set<time_t> sdates = gribread->getListDates();
	std::set<time_t>::iterator iter;
	std::set<time_t>::iterator iter2;
	int lig, col, colspan;
	QString dstr;
	//-----------------------------------------------
	// Titre 1 : une colonne par jour, regroupant plusieurs horaires
	//-----------------------------------------------
	time_t now = time(NULL);
	struct tm *pstm;
	pstm = gmtime(&now);
	now = mktime(pstm);
//printf("ref=%d\n", (int) gribread->getRefDate());

	time_t dateproche = 0;
	uint difftime, difftimemin=0xFFFFFFFF;
	
	col = 0;
	lig = 0;
	addCell_title("", true, layout, lig,col, 2,1);
	col ++;
	QString actuel = "";
	for (iter=sdates.begin(); iter!=sdates.end(); iter++)
	{
		time_t daterecord = *iter;
		// repère le record le plus proche de now (si dans la même journée)
		difftime = (daterecord>now) ? daterecord-now : now-daterecord;
		if (difftimemin>difftime && (difftime/3600)<=24) {
			difftimemin = difftime;
			dateproche = daterecord;
		}
		// note la date du jour
		dstr = Util::formatDateLong(daterecord);
		if (dstr != actuel)
		{
			colspan = 0;
			actuel = dstr;
			iter2 = iter;
			do
			{
				colspan ++;
				iter2 ++;
				dstr = Util::formatDateLong(*iter2);
			} while (actuel==dstr);
			
			addCell_title(actuel, true, layout, lig,col, 1,colspan);
			col += colspan;
		}
	}
	//-----------------------------------------------
	// Titre 2 : une colonne par date+horaires
	//-----------------------------------------------
	GribPointInfo *pinfo;
	col = 1;
	lig = 1;
	for (iter=sdates.begin(); iter!=sdates.end(); iter++)
	{
		time_t date = *iter;
		addCell_title(Util::formatTime(date), false, layout, lig,col, 1,1,
						dateproche==date);
		col ++;

		// Grib data for this point and this date
		pinfo = new GribPointInfo(gribread, lon,lat, date);
		lspinfos.push_back(pinfo);
	}
	//-----------------------------------------------
	// Contenus
	//-----------------------------------------------
	lig ++;
	createListVisibleGribData();
	
	QList <MTGribData *>::iterator it;
	for (it=listVisibleData.begin(); it!=listVisibleData.end(); it++) {
		MTGribData *gr = *it;
		uchar dataType = GribCode::getDataType (gr->gribCode);
		uchar levelType = GribCode::getLevelType (gr->gribCode);
		uint  levelValue = GribCode::getLevelValue (gr->gribCode);
		
		if (dataType==GRB_WIND_XY2D && levelType==LV_ABOV_GND && levelValue==10){
			addLine_Wind (lig++);
		}
		else if (dataType==GRB_PRESSURE && levelType==LV_MSL && levelValue==0){
			addLine_Pressure (lig++);
		}
		else if (dataType==GRB_CLOUD_TOT && levelType==LV_ATMOS_ALL && levelValue==0){
			addLine_CloudCover (lig++);
		}
		else if (dataType==GRB_PRECIP_TOT && levelType==LV_GND_SURF && levelValue==0){
			addLine_Rain (lig++);
		}
		else if (dataType==GRB_TEMP && levelType==LV_ABOV_GND && levelValue==2){
			addLine_Temperature (GRB_TEMP, lig++);
		}
		else if (dataType==GRB_TMIN && levelType==LV_ABOV_GND && levelValue==2){
			addLine_Temperature (GRB_TMIN, lig++);
		}
		else if (dataType==GRB_TMAX && levelType==LV_ABOV_GND && levelValue==2){
			addLine_Temperature (GRB_TMAX, lig++);
		}
		else if (dataType==GRB_TPOT && levelType==LV_SIGMA && levelValue==9950){
			addLine_Temperature (GRB_TPOT, lig++);
		}
		else if (dataType==GRB_GEOPOT_HGT  && levelType==LV_ISOTHERM0 && levelValue==0){
			addLine_Isotherm0Height (lig++);
		}
		else if (dataType==GRB_DEWPOINT && levelType==LV_ABOV_GND && levelValue==2){
			addLine_DewPoint (lig++);
		}
		else if (dataType==GRB_DIFF_TEMPDEW && levelType==LV_ABOV_GND && levelValue==2){
			addLine_DeltaTemperature (GRB_DIFF_TEMPDEW, lig++);
		}
		else if (dataType==GRB_HUMID_REL && levelType==LV_ABOV_GND && levelValue==2){
			addLine_HumidRel (lig++);
		}
		else if (dataType==GRB_SNOW_CATEG && levelType==LV_GND_SURF && levelValue==0){
			addLine_Categorical (GRB_SNOW_CATEG, lig++);
		}
		else if (dataType==GRB_FRZRAIN_CATEG && levelType==LV_GND_SURF && levelValue==0){
			addLine_Categorical (GRB_FRZRAIN_CATEG, lig++);
		}
		else if (dataType==GRB_SNOW_DEPTH && levelType==LV_GND_SURF && levelValue==0){
			addLine_SnowDepth (lig++);
		}
		else if (dataType==GRB_CAPE && levelType==LV_GND_SURF && levelValue==0){
			addLine_CAPEsfc (lig++);
		}
	}
}

//-----------------------------------------------------------------
bool lessThanMTGribData(const MTGribData *a, const MTGribData *b)
{
	return (a->pos < b->pos);
}

//-----------------------------------------------------------------
void MeteoTableWidget::createListVisibleGribData ()
{
	qDeleteAll(listVisibleData);
	
	QStringList listKeys = Settings::getAllKeys();
	QStringList::const_iterator cstit;
	bool foundMTableData = false;
	for (cstit = listKeys.constBegin(); cstit != listKeys.constEnd(); cstit++) {
		QString key = *cstit;
		if (key.startsWith("MTableData_vis_"))
		{
			foundMTableData = true;
			if (Util::getSetting(key, false).toBool()) {
				// key like MTableData_vis_0002690b
				QString gribCodeStr = key;
				gribCodeStr = gribCodeStr.replace("MTableData_vis_","");	// extracts gribcode

				QString keyPos = QString("MTableData_pos_")+gribCodeStr;
				
				int pos = Util::getSetting(keyPos, 1000).toInt();

				MTGribData *group = new MTGribData(gribCodeStr, pos);
				listVisibleData.append(group);
			}
		}
	}
	if (!foundMTableData) {
		// No MTableData_... in settings : initialize first list (arbitrary)
		int pos = 0;
    	listVisibleData.append( new MTGribData (
    				GribCode::makeCode(GRB_WIND_XY2D,LV_ABOV_GND,10), pos++) );
    	listVisibleData.append( new MTGribData (
    				GribCode::makeCode(GRB_CLOUD_TOT,LV_ATMOS_ALL,0), pos++) );
    	listVisibleData.append( new MTGribData (
    				GribCode::makeCode(GRB_PRECIP_TOT,LV_GND_SURF,0), pos++) );
    	listVisibleData.append( new MTGribData (
    				GribCode::makeCode(GRB_TEMP,LV_ABOV_GND,2), pos++) );
    	listVisibleData.append( new MTGribData (
    				GribCode::makeCode(GRB_DIFF_TEMPDEW,LV_ABOV_GND,2), pos++) );
    	listVisibleData.append( new MTGribData (
    				GribCode::makeCode(GRB_GEOPOT_HGT,LV_ISOTHERM0,0), pos++) );
    	listVisibleData.append( new MTGribData (
					GribCode::makeCode(GRB_PRESSURE,LV_MSL,0), pos++) );
	}
	// Sort visible data by position
	qSort(listVisibleData.begin(), listVisibleData.end(), lessThanMTGribData);
}

//-----------------------------------------------------------------
void MeteoTableWidget::addLine_Isotherm0Height(int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	addCell_title(tr("Isotherme 0°"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		txt = "";
		if (pinfo->hasIsotherm0HGT()) {
			double v = pinfo->isotherm0HGT;
			txt.sprintf("%.0f ", v);
			txt += tr("m");
			bgColor = QColor(gribplot->getAltitudeColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}

//-----------------------------------------------------------------
void MeteoTableWidget::addLine_Pressure(int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	addCell_title(tr("Pression"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		txt = "";
		if (pinfo->hasPressure()) {
			double v = pinfo->pressure;
			v = v/100.0;
			txt.sprintf("%.1f ", v);
			txt += tr("hPa");
			bgColor = QColor(gribplot->getPressureColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_Temperature(uchar type, int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	double v;
	switch (type) {
		case GRB_TEMP:
			addCell_title(tr("Température"), true, layout, lig,col);
			break;
		case GRB_TMIN:
			addCell_title(tr("Temp. min"), true, layout, lig,col);
			break;
		case GRB_TMAX:
			addCell_title(tr("Temp. max"), true, layout, lig,col);
			break;
		case GRB_TPOT:
			addCell_title(tr("Temp. pot"), true, layout, lig,col);
			break;
	}
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		switch (type) {
			case GRB_TMIN:
				v = pinfo->tempMin;
				break;
			case GRB_TMAX:
				v = pinfo->tempMax;
				break;
			case GRB_TPOT:
				v = pinfo->tempPot;
				break;
			case GRB_TEMP:
			default:
				v = pinfo->temp;
				break;
		}
		txt = "";
		if (v != GRIB_NOTDEF) {
			txt = Util::formatTemperature(v);
			bgColor = QColor(gribplot->getTemperatureColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_DeltaTemperature(uchar type, int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	double v;
	switch (type) {
		case GRB_DIFF_TEMPDEW:
		default:
			addCell_title(tr("Ecart temp-rosée"), true, layout, lig,col);
			break;
	}
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		switch (type) {
			case GRB_DIFF_TEMPDEW:
			default:
				v = fabs(pinfo->temp - pinfo->dewPoint);
				break;
		}
		txt = "";
		if (pinfo->temp != GRIB_NOTDEF && pinfo->dewPoint != GRIB_NOTDEF) {
			txt = Util::formatTemperature(v + 273.15);
			bgColor = QColor(gribplot->getDeltaTemperaturesColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_DewPoint(int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	addCell_title(tr("Point de rosée"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		txt = "";
		if (pinfo->hasDewPoint()) {
			double v = pinfo->dewPoint;
			txt = Util::formatTemperature(v);
			bgColor = QColor(gribplot->getTemperatureColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_CAPEsfc(int lig)
{
	std::list<GribPointInfo *>::iterator iter;
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	addCell_title(tr("CAPE (surface)"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		txt = "";
		if (pinfo->hasCAPEsfc()) {
			double v = pinfo->CAPEsfc;
			txt.sprintf("%d ", (int) v);
			txt += tr("J/kg");
			bgColor = QColor(gribplot->getCAPEColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_Rain(int lig)
{
	std::list<GribPointInfo *>::iterator iter;
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	addCell_title(tr("Précipitations"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		txt = "";
		if (pinfo->hasRain()) {
			double v = pinfo->rain;
			txt.sprintf("%.2f ", v);
			txt += tr("mm/h");
			bgColor = QColor(gribplot->getRainColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_CloudCover (int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	addCell_title(tr("Nébulosité"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		txt = "";
		double v = 0;
		if (pinfo->hasCloud()) {
			v = pinfo->cloud;
			txt = Util::formatPercentValue(v);
			bgColor = QColor(gribplot->getCloudColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor, MTABLE_CLOUD_CELL, v);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_Wind(int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	addCell_title(tr("Vent"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		double vx = GRIB_NOTDEF;
		double vy = GRIB_NOTDEF;
		GribPointInfo * pinfo = *iter;
		txt = "";
		if (pinfo->hasWind())
		{
			vx = pinfo->vx;
			vy = pinfo->vy;
			double v = sqrt(vx*vx+vy*vy);
			double dir = -atan2(-vx, vy) *180.0/M_PI + 180;
			if (dir < 0)
				dir += 360.0;
			if (dir >= 360)
				dir -= 360.0;

			QString tmp;
			txt = "";
			tmp.sprintf("%.0f", dir);
			txt += tmp + tr(" °") + "\n";
			txt += Util::formatSpeed(v) + "\n";
			tmp.sprintf("%2d", Util::kmhToBeaufort(v*3.6));
			txt += tmp + tr(" Bf");

			bgColor = QColor(gribplot->getWindColor(v*3.6, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor, MTABLE_WIND_CELL,vx,vy);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_HumidRel (int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	addCell_title(tr("Humidité relative"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		txt = "";
		double v = 0;
		if (pinfo->hasHumid()) {
			v = pinfo->humid;
			txt = Util::formatPercentValue(v);
			bgColor = QColor(gribplot->getHumidColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_Categorical(uchar type, int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	double v = -1000;
	switch (type) {
		case GRB_FRZRAIN_CATEG:
			addCell_title(tr("Verglas (risque)"), true, layout, lig,col);
			break;
		case GRB_SNOW_CATEG:
			addCell_title(tr("Neige (risque)"), true, layout, lig,col);
			break;
	}
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		switch (type) {
			case GRB_FRZRAIN_CATEG:
				v = pinfo->frzRainCateg;
				break;
			case GRB_SNOW_CATEG:
				v = pinfo->snowCateg;
				break;
		}
		txt = "";
		if (v >= 0) {
			txt.sprintf("%.1f", v);
			bgColor = QColor(gribplot->getSnowDepthColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}
//-----------------------------------------------------------------
void MeteoTableWidget::addLine_SnowDepth(int lig)
{
	std::list<GribPointInfo *>::iterator iter; 
	QColor    bgColor = Qt::white;
	QString   txt;
	int col = 0;
	double v = -1000;
	addCell_title(tr("Neige"), true, layout, lig,col);
	col ++;
	for (iter=lspinfos.begin(); iter!=lspinfos.end(); iter++, col++)
	{
		GribPointInfo * pinfo = *iter;
		v = pinfo->snowDepth;
		txt = "";
		if (v >= 0) {
			txt.sprintf("%.1f", v);
			bgColor = QColor(gribplot->getSnowDepthColor(v, true));
		}
		addCell_content(txt, layout,lig,col, 1,1, bgColor);
	}
}


//===================================================================
void MeteoTableWidget::addCell_content (
				QString txt,
				QGridLayout *layout,int lig,int col,
				int    rowspan,
				int    colspan,
				QColor bgcolor,
				int    cellType,
				double  vx,
				double  vy
				)
{
	TableCell *cell;
 	if (cellType == MTABLE_WIND_CELL) {
	 	cell = new TableCell_Wind(vx, vy, (lat<0), gribplot,
	 				this, txt, false, bgcolor);
 	}
 	else if (cellType == MTABLE_CLOUD_CELL) {
	 	cell = new TableCell_Clouds ( vx, gribplot,
	 				this, txt, false, bgcolor);
 	}
 	else {
	 	cell = new TableCell(this, txt, false, bgcolor);
	}
	if (lig==0 && col==0)
		cell->setBorders(TableCell::all);
	else if (lig==0)
		cell->setBorders(TableCell::south+TableCell::east+TableCell::north);
	else if (col==0)
		cell->setBorders(TableCell::west+TableCell::south+TableCell::east);
	else
		cell->setBorders(TableCell::south+TableCell::east);
	layout->addWidget(cell, lig,col, rowspan,colspan );
}

//----------------------------------------------------------------
void MeteoTableWidget::addCell_title( QString txt, bool bold,
				QGridLayout *layout,int lig,int col, int rowspan,int colspan,
				bool isNowDate)
{
	QColor bgcolor(200,200,255);
	if (isNowDate) {
		bgcolor = QColor(250,250,100);
	}
	
	TableCell *cell = new TableCell(this, txt, bold, bgcolor);	
	if (lig==0 && col==0)
		cell->setBorders(TableCell::all);
	else if (lig==0)
		cell->setBorders(TableCell::south+TableCell::east+TableCell::north);
	else if (col==0)
		cell->setBorders(TableCell::west+TableCell::south+TableCell::east);
	else
		cell->setBorders(TableCell::south+TableCell::east);
	layout->addWidget(cell, lig,col, rowspan,colspan );
}

//===================================================================
// TableCell : case seule
//===================================================================
TableCell::TableCell(QWidget *parent, QString txt, bool bold,
						QColor bgcolor)
	: QWidget(parent)
{
	this->bgcolor = bgcolor;
	this->bordercolor = QColor(100,100,100);
	this->borders = TableCell::none;
	
	QGridLayout *layout = new QGridLayout();
	
	label = new QLabel(txt, this);
	label->setAlignment(Qt::AlignHCenter);
	
	if (bold) {
		QFont font;
		font.setBold(true);
		label->setFont(font);
	}
	setContrastedTextColor(this->bgcolor);
	
	layout->addWidget(label, 0,0, Qt::AlignHCenter|Qt::AlignBottom);
	this->setLayout(layout);
}
//----------------------------------------------------------
void    TableCell::setContrastedTextColor(QColor bgcolor)
{
	// Ecriture contrastée
	double gris = 0.30*bgcolor.redF() + 0.59*bgcolor.greenF() + 0.11*bgcolor.blueF();
	QColor fgcolor;
	if (gris < 0.45)
		fgcolor = QColor(230,230,230);
	else
		fgcolor = Qt::black;
	QPalette p;
	p.setBrush(QPalette::Active, QPalette::WindowText, fgcolor);
	p.setBrush(QPalette::Inactive, QPalette::WindowText, fgcolor);
	label->setPalette(p);
}
//---------------------------------------------------------
void TableCell::paintEvent(QPaintEvent * /*event*/)
{
    QPainter pnt(this);
	
	pnt.fillRect(0,0,width(),height(), QBrush(bgcolor));
	
	QPen pen(bordercolor);
	pen.setWidth(1);
	pnt.setPen(pen);
	
	if (borders & TableCell::north)
		pnt.drawLine(0,0, width()-1,0);
	if (borders & TableCell::south)
		pnt.drawLine(0,height()-1, width()-1,height()-1);
	
	if (borders & TableCell::west)
		pnt.drawLine(0,0, 0,height()-1);
	if (borders & TableCell::east)
		pnt.drawLine(width()-1,0, width()-1,height()-1);
	
}


//===================================================================
// TableCell_Wind : case seule spécialisée pour le vent (flêche+barbules)
//===================================================================
TableCell_Wind::TableCell_Wind (double vx, double vy, bool south,
        			GribPlot *gribplot,
        			QWidget *parent, QString txt, bool bold,
        			QColor bgcolor )
	: TableCell(parent, txt, bold, bgcolor)
{
	this->vx = vx;
	this->vy = vy;
	this->south = south;
	this->gribplot = gribplot;

	windArrowsColor = QColor(40,40,40);
	showWindArrows = Util::getSetting("MTABLE_showWindArrows", true).toBool();

	if (showWindArrows)
		setMinimumHeight(label->minimumSizeHint().height()+50);
}	
//---------------------------------------------------------
void TableCell_Wind::paintEvent(QPaintEvent * e)
{
	TableCell::paintEvent(e);
    QPainter pnt(this);
	pnt.setRenderHint(QPainter::Antialiasing, true);

	if (showWindArrows && vx != GRIB_NOTDEF && vx != GRIB_NOTDEF)
	{
    	gribplot->drawWindArrowWithBarbs(
    			pnt, width()/2, 25, vx, vy, south, windArrowsColor);
	}
}



//===================================================================
// TableCell_Clouds : case seule spécialisée pour la nébulosité
//===================================================================
TableCell_Clouds::TableCell_Clouds (
					double   val,
        			GribPlot *gribplot,
        			QWidget  *parent, QString txt, bool bold,
        			QColor bgcolor )
	: TableCell(parent, txt, bold, bgcolor)
{
	this->gribplot = gribplot;
	this->val = val;

	// Color = seaColor + bgColor
	QColor seaColor (50,50,200, 255);
	QImage img (1,1, QImage::Format_ARGB32_Premultiplied);
	QPainter pnt (&img);
	
	// cloudColorMode = 1 : 100 % -> white
	// cloudColorMode = 2 : 100 % -> dark
	int cloudColorMode;
	if (Util::getSetting("MTABLE_cloudsColorMode", "black").toString() == "white")
		cloudColorMode = 1;
	else
		cloudColorMode = 2;
	
	QColor cloudColor = QColor::fromRgba(gribplot->getCloudColor(val, true, cloudColorMode));
		
	pnt.fillRect (0,0, 1,1, seaColor );
	pnt.fillRect (0,0, 1,1, cloudColor);
	this->bgcolor = QColor(img.pixel(0,0));
	
	setContrastedTextColor(this->bgcolor);
}	
//---------------------------------------------------------
void TableCell_Clouds::paintEvent(QPaintEvent * e)
{
	TableCell::paintEvent(e);
    QPainter pnt(this);
	pnt.setRenderHint(QPainter::Antialiasing, true);

}


//===================================================================
// MeteoTable : dialog + MeteoTableWidget
//===================================================================
MeteoTableDialog::MeteoTableDialog(GribPlot *gribplot, double lon, double lat, QString posName)
	: QDialog()
{
	this->gribplot = gribplot;
	this->lon = lon;
	this->lat = lat;
    
    if (!gribplot->isGribReaderOk()) {
        QMessageBox::critical (this,
			tr("Erreur"),tr("Création du Météotable impossible:\n\nPas de fichier GRIB ouvert."));
		delete this;
		return;
    }
    GribReader *gribread = gribplot->getGribReader();
    GribRecord *grbrec;
	if ((grbrec=gribread->getFirstGribRecord()) == NULL) {
        QMessageBox::critical (this,
            	tr("Erreur"),tr("Création du Météotable impossible:\n\nZone GRIB indéterminée."));
		delete this;
		return;
    }
	if (!grbrec->isPointInMap(lon, lat)) {
		QMessageBox::critical (this,
				tr("Erreur"),tr("Création du Météotable impossible :\n\nPoint en dehors de la zone couverte par le fichier GRIB."));
		delete this;
		return;
	}

	//----------------------------------------------
	setModal(false);
	
	meteoTableWidget = new MeteoTableWidget(gribplot, lon,lat, this);
	assert(meteoTableWidget);
	optionsDialog = new DialogMeteotableOptions();
	assert(optionsDialog);
	
	scrollArea = new QScrollArea();
	assert(scrollArea);
	scrollArea->setWidget(meteoTableWidget);
	
	QVBoxLayout *layout = new QVBoxLayout(this);

	QString position = Util::formatPosition(lon, lat);
	QLabel *lbpos;
	if (posName == "") {
		setWindowTitle(position);
		lbpos = new QLabel(tr("Position : ") + position);
	}
	else {
		setWindowTitle(posName);
		lbpos = new QLabel(tr("Position : <b>") +posName + "</b> : " + position);
	}
	QLabel *lbdate = new QLabel(tr("Date de référence : ")
							+ Util::formatDateTimeLong(gribread->getRefDate()));
	//-------------------------------
	btClose = new QPushButton(tr("Fermer"));
	btOptions = new QPushButton(tr("Options..."));
    assert(btClose);
    assert(btOptions);
    QFrame *framebts = new QFrame(this);
    assert(framebts);
    QHBoxLayout *hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(btClose);
        hlay->addWidget(btOptions);
        framebts->setLayout(hlay);
        
	connect(btClose, SIGNAL(clicked()), this, SLOT(reject()));
	connect(btOptions, SIGNAL(clicked()), this, SLOT(slotBtOptions()));
	connect(optionsDialog, SIGNAL(accepted()), this, SLOT(slotOptionsChanged()));
							
	//-------------------------------
	layout->addWidget( lbpos);
	layout->addWidget( lbdate);
	layout->addWidget(scrollArea);	
	layout->addWidget(framebts, 0, Qt::AlignHCenter);
	
	// taille par défaut pour la 1ère ouverture
	adjustSize();
	int w = 800;
	int h = this->height()+80;
    resize( Util::getSetting("meteoTableDialogSize", QSize(w,h)).toSize() );
	show();
}
//-----------------------------------------
MeteoTableDialog::~MeteoTableDialog()
{
	Util::setSetting("meteoTableDialogSize", size());
	delete optionsDialog;
}

//-----------------------------------------
void MeteoTableDialog::slotBtOptions()
{
	optionsDialog->show();
}

//-----------------------------------------
void MeteoTableDialog::slotOptionsChanged()
{
	delete meteoTableWidget;
	meteoTableWidget = new MeteoTableWidget(gribplot, lon,lat, this);
	assert(meteoTableWidget);
	scrollArea->setWidget(meteoTableWidget);
}






    

















