<?php declare(strict_types = 0);
/*
** Copyright (C) 2001-2024 Zabbix SIA
**
** This program is free software: you can redistribute it and/or modify it under the terms of
** the GNU Affero General Public License as published by the Free Software Foundation, version 3.
**
** This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
** without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU Affero General Public License for more details.
**
** You should have received a copy of the GNU Affero General Public License along with this program.
** If not, see <https://www.gnu.org/licenses/>.
**/


namespace Widgets\ItemNavigator\Includes;

use Zabbix\Widgets\CWidgetField;

class CWidgetFieldItemGrouping extends CWidgetField {

	public const DEFAULT_VIEW = CWidgetFieldItemGroupingView::class;
	public const DEFAULT_VALUE = [];

	public const GROUP_BY_HOST_GROUP = 0;
	public const GROUP_BY_HOST_NAME = 1;
	public const GROUP_BY_HOST_TAG = 2;
	public const GROUP_BY_ITEM_TAG = 3;

	public const MAX_ROWS = 10;

	public function __construct(string $name, string $label = null) {
		parent::__construct($name, $label);

		$this
			->setDefault(self::DEFAULT_VALUE)
			->setValidationRules(['type' => API_OBJECTS, 'length' => self::MAX_ROWS, 'fields' => [
				'attribute'	=> ['type' => API_INT32, 'flags' => API_REQUIRED, 'in' => implode(',', [self::GROUP_BY_HOST_GROUP, self::GROUP_BY_HOST_NAME, self::GROUP_BY_HOST_TAG, self::GROUP_BY_ITEM_TAG])],
				'tag_name'	=> ['type' => API_STRING_UTF8, 'length' => $this->getMaxLength()]
			]]);
	}

	public function validate(bool $strict = false): array {
		$errors = parent::validate($strict);

		if ($errors) {
			return $errors;
		}

		foreach ($this->getValue() as $value) {
			if ($value['attribute'] == self::GROUP_BY_HOST_GROUP || $value['attribute'] == self::GROUP_BY_HOST_NAME) {
				continue;
			}

			if ($value['tag_name'] === '') {
				$errors[] = _s('Invalid parameter "%1$s": %2$s.', _('Group by'), _('tag cannot be empty'));

				break;
			}
		}

		$unique_groupings = [];

		foreach ($this->getValue() as $value) {
			$attribute = $value['attribute'];
			$tag_name = $attribute == self::GROUP_BY_HOST_TAG || $attribute == self::GROUP_BY_ITEM_TAG
				? $value['tag_name']
				: '';

			if (array_key_exists($attribute, $unique_groupings) && $unique_groupings[$attribute] === $tag_name) {
				$errors[] = _s('Invalid parameter "%1$s": %2$s.', _('Group by'), _('rows must be unique'));

				break;
			}
			else {
				$unique_groupings[$attribute] = $tag_name;
			}
		}

		return $errors;
	}

	public function toApi(array &$widget_fields = []): void {
		foreach ($this->getValue() as $index => $value) {
			$widget_fields[] = [
				'type' => ZBX_WIDGET_FIELD_TYPE_INT32,
				'name' => $this->name.'.'.$index.'.'.'attribute',
				'value' => $value['attribute']
			];

			if ($value['attribute'] == self::GROUP_BY_HOST_TAG || $value['attribute'] == self::GROUP_BY_ITEM_TAG) {
				$widget_fields[] = [
					'type' => $this->save_type,
					'name' => $this->name.'.'.$index.'.'.'tag_name',
					'value' => $value['tag_name']
				];
			}
		}
	}
}
