/*******************************************************************
 * File: timer.c
 *
 * Implement a one second timer that will call functions in a linked list.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: timer.c,v 1.10 2006/02/23 22:26:53 chessing Exp $
 * $Date: 2006/02/23 22:26:53 $
 * $Log: timer.c,v $
 * Revision 1.10  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.9  2005/10/13 18:57:28  chessing
 * Removed some leftover debug strings.  Cleaned up some compiler warnings. Fixed an association bug in the madwifi driver.
 *
 * Revision 1.8  2005/10/13 18:46:47  chessing
 * Fixed to the Madwifi driver to allow it to do dynamic WEP again.  Fixed up the wext driver to behave correctly again. ;)
 *
 * Revision 1.7  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.6  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <inttypes.h>
#include <unistd.h>

#include "profile.h"
#include "xsup_debug.h"
#include "timer.h"

struct timer_data
{
  uint16_t seconds_left;     // How many seconds are left before this timer
                             // expires.
  uint16_t timer_id;         // A numeric identifier that indicates what
                             // type of timer this is.  (See timer.h)
  int8_t (*timer_tick)(struct interface_data *);
  int8_t (*timer_expired)(struct interface_data *);

  struct timer_data *next;
};

struct timer_data *timer_head;

struct timer_ids_struct {
  uint16_t timer_id;
  char *timer_id_name;
};

struct timer_ids_struct timer_ids[] = {
  {0, "invalid timer"},
  {COUNTERMEASURE_TIMER, "countermeasure timer"},
  {REKEY_PROB_TIMER, "rekey problem timer"},
  {ASSOCIATION_TIMER, "association timer"},
  {STALE_KEY_WARN_TIMER, "stale key warning timer"},
  {PASSIVE_SCAN_TIMER, "passive scan timer"},
};

/*****************************
 * 
 * Set up the linked list of timers that we will be using, and make sure
 * it is empty.
 *
 *****************************/
void timer_init()
{
  timer_head = NULL;
}

/*****************************
 *
 * Locate the timer structure by id number.
 *
 *****************************/
struct timer_data *timer_get_by_id(uint16_t id)
{
  struct timer_data *cur;

  cur = timer_head;

  while ((cur != NULL) && (cur->timer_id != id))
    {
      cur = cur->next;
    }

  return cur;
}

/*****************************
 *
 * Reset the "seconds_left" value of a timer.
 *
 *****************************/
uint8_t timer_reset_timer_count(uint16_t timer_id, uint16_t seconds_left)
{
  struct timer_data *timer;
  char *timername;

  timer = timer_get_by_id(timer_id);

  if (timer == NULL)
    {
      timername = timer_get_name_from_id(timer_id);
      if (timername == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't reset timeout for unknown "
		       "timer with id %d!\n", timer_id);
	  return FALSE;
	} else {
	  debug_printf(DEBUG_NORMAL, "Couldn't reset timeout for timer '%s'!"
		       "\n", timername);
	  return FALSE;
	}
    }

  timer->seconds_left = seconds_left;
  return TRUE;
}

/*****************************
 *
 * Given a timer id #, look up the string name of the ID.
 *
 *****************************/
char *timer_get_name_from_id(uint16_t id)
{
  uint16_t i = 0;

  while ((i<=NUM_TIMER_IDS) && (!(timer_ids[i].timer_id == id)))
    {
      i++;
    }

  if (timer_ids[i].timer_id == id)
    {
      return timer_ids[i].timer_id_name;
    }

  return NULL;
}

/*****************************
 *
 * Check to see if the timer already exists in the list.  Return FALSE if
 * the timer is not currently in the list, and TRUE if it is.
 *
 *****************************/
uint8_t timer_check_existing(uint16_t timertype)
{
  struct timer_data *cur;

  cur = timer_head;

  if (cur == NULL) 
    {
      debug_printf(DEBUG_EVERYTHING, "No existing timers in the list!\n");
      return FALSE;
    }

  while (cur != NULL)
    {
      if (cur->timer_id == timertype) 
	{
	  debug_printf(DEBUG_EVERYTHING, "Matched existing timer! (Timer "
		       "was : '%s')\n", timer_get_name_from_id(timertype));
	  return TRUE;
	}

      cur = cur->next;
    }

  return FALSE;
}

/*****************************
 *
 * Add a node to our timer list.
 *
 *****************************/
void timer_add_timer(uint16_t timertype, uint16_t timeout, void *timertick, 
		     void *timerexpired)
{
  struct timer_data *cur = NULL;

  if (timer_head == NULL)
    {
      timer_head = (struct timer_data *)malloc(sizeof(struct timer_data));
      if (timer_head == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory to add timer "
		       "to our list!  We may have strange results!\n");
	  return;
	}

      cur = timer_head;
    } else {
      if (timer_check_existing(timertype) == TRUE)
	{
	  debug_printf(DEBUG_NORMAL, "Attempt to add a timer that already "
		       "exists!  Ignoring!\n");
	  return;
	}

      cur = timer_head;

      while (cur->next != NULL)
	{
	  cur = cur->next;
	}

      cur->next = (struct timer_data *)malloc(sizeof(struct timer_data));
      if (cur->next == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for timer "
		       "node!  We may have strange results!\n");
	  return;
	}

      cur = cur->next;
    }

  // At this point, cur should point to a node that was just created.
  cur->next = NULL;
  cur->timer_id = timertype;
  cur->seconds_left = timeout;
  cur->timer_tick = timertick;
  cur->timer_expired = timerexpired;
}

/*****************************
 *
 * Called once a second, this function should execute the "on_tick" function
 * in the timer.
 *
 *****************************/
void timer_tick(struct interface_data *intdata)
{
  struct timer_data *cur;

  intdata->tick = FALSE;

  cur = timer_head;

  if (cur == NULL)
    {
      // We don't have any timers to tick.
      return;
    }

  // Otherwise, tick all of our timers.
  while (cur != NULL)
    {
      cur->seconds_left--;
      
      if (cur->timer_tick != NULL)
	{
	  cur->timer_tick(intdata);
	}

      if (cur->seconds_left <= 0)
	{
	  if (cur->timer_expired != NULL)
	    {
	      cur->timer_expired(intdata);
	    }
	}

      cur = cur->next;
    }
}


/*****************************
 *
 * Cancel an existing timer by removing it from the linked list.
 *
 *****************************/
void timer_cancel(uint16_t timertype)
{
  struct timer_data *cur, *prev;
  char *timername;

  // Be sure we have something to work with.
  if (timer_head == NULL)
    {
      timername = timer_get_name_from_id(timertype);
      if (timername == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "No timers to cancel for unknown timer "
		       "type %d!\n", timertype);
	  return;
	} else {
	  debug_printf(DEBUG_NORMAL, "No timers to cancel for '%s'!\n",
		       timername);
	  timername = NULL;
	  return;
	}
    }

  // See if the first node has what we are looking for.
  if (timer_head->timer_id == timertype)
    {
      // We found the timer to cancel.
      cur = timer_head;
      timer_head = timer_head->next;
      
      free(cur);

      timername = timer_get_name_from_id(timertype);
      if (timername == NULL)
	{
	  debug_printf(DEBUG_EVERYTHING, "Canceled timer for unknown timer id %d!\n", timertype);
	} else {
	  debug_printf(DEBUG_EVERYTHING, "Canceled timer for '%s'!\n",
		       timername);
	}
      return;
    }

  // Now, check the rest.
  cur = timer_head->next;
  prev = timer_head;

  while ((cur != NULL) && (cur->timer_id != timertype))
    {
      prev = cur;
      cur = cur->next;
    }

  if (cur == NULL)
    {
      debug_printf(DEBUG_STATE, "Attempted to cancel timer that doesn't "
		   "exist!  (Timer was : '%s')\n", 
		   timer_get_name_from_id(timertype));
      return;
    }

  if (cur->timer_id == timertype)
    {
      prev->next = cur->next;

      free(cur);

      timername = timer_get_name_from_id(timertype);
      if (timername == NULL)
	{
	  debug_printf(DEBUG_EVERYTHING, "Canceled timer for unknown timer id"
		       " %d!\n", timertype);
	} else {
	  debug_printf(DEBUG_EVERYTHING, "Canceled timer for '%s'!\n",
		       timername);
	}
    } else {
      timername = timer_get_name_from_id(timertype);
      if (timername == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Request to cancel timer for unknown "
		       "timer id %d, failed because the timer is not in the "
		       "list!\n", timertype);
	} else {
	  debug_printf(DEBUG_NORMAL, "Request to cancel timer '%s' failed "
		       "because the timer is not in the list!\n", timername);
	}
    }
  return;
}

/*****************************
 *
 * Free up the memory being used in the linked list.
 *
 *****************************/
void timer_cleanup()
{
  struct timer_data *cur, *next;

  cur = timer_head;

  while (cur != NULL)
    {
      next = cur->next;

      free(cur);
      cur = next;
    }
}

/*******************************************************************
 *
 * Replacement sleep function that will actually work with signals going off.
 * NOTE: This sleep function will not be accurate to the second!!!  If you 
 * need something that is accurate like that, look elsewhere. ;)
 *
 *******************************************************************/
void timer_sleep(uint16_t secs)
{
  uint16_t i;

  // The normal sleep() call will terminate when an signal is triggered.  So,
  // rather than sleep for a specific amount of time, we should only sleep
  // for a second, since our SIGALRM should only trigger once a second.  The
  // result of this is that we will only sleep for approximately the amount
  // of time that is defined by "secs".
  for (i=0;i<secs;i++)
    {
      sleep(1);
    }
}
