/*******************************************************************
 * EAP layer implementation.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: eap.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eap.c,v 1.125 2006/03/08 00:16:04 chessing Exp $
 * $Date: 2006/03/08 00:16:04 $
 * $Log: eap.c,v $
 * Revision 1.125  2006/03/08 00:16:04  chessing
 * Fixed EAP hints code to work correctly when the request ID packet is padded out with null bytes.  (Observed in Aruba APs.)  Some changes/fixes for the EAP-AKA module.
 *
 * Revision 1.124  2006/03/02 04:52:41  chessing
 * Added the ability to load a config based on the hints that are included in an EAP request identity message.  Should consider having a way to override listening to these messages.
 *
 * Revision 1.123  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.122  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.121  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.120  2005/09/26 02:04:07  shaftoe
 * Yet another round of log output cleanup.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.119  2005/09/17 19:08:27  chessing
 * Updates to the MADwifi driver so that it works correctly with dynamic WEP again.  Because of a requirement in the driver, the normal wext calls will authenticate, but only broadcast traffic will be able to pass.  So, all MAdwifi users should use -D madwifi to make everything work correctly.
 *
 * Revision 1.118  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.117  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.116  2005/08/09 01:39:13  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <netinet/in.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>

#include "snmp.h"
#include "frame_structs.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "xsupconfig.h"
#include "profile.h"
#include "eap.h"
#include "eapol.h"
#include "xsup_ipc.h"

// Header files for auth types we know about.
#include "eap_types/md5/eapmd5.h"
#include "eap_types/tls/eaptls.h"
#include "eap_types/ttls/eapttls.h"
#include "eap_types/mschapv2/eapmschapv2.h"
#include "eap_types/peap/eappeap.h"
#include "eap_types/leap/eapleap.h"
#include "eap_types/otp/eapotp.h"

#ifdef EAP_SIM_ENABLE
#include "winscard.h"
#include "eap_types/sim/eapsim.h"
#include "eap_types/aka/eapaka.h"
#endif

struct eap_type_handler {
  int eap_auth_type;
  char *eapname;
  int (*eap_auth_setup)(struct generic_eap_data *);
  int (*eap_auth_handlers)(struct generic_eap_data *, u_char *, int, u_char *,
			   int*);
  int (*eap_auth_get_keys)(struct interface_data *);
  int (*eap_auth_failed)(struct generic_eap_data *);
  int (*eap_auth_cleanup)(struct generic_eap_data *);
};

struct eap_type_handler eaphandlers[] = {
  {EAP_TYPE_MD5, "EAP_MD5", eapmd5_setup, eapmd5_process, eapmd5_get_keys,
   eapmd5_failed, eapmd5_cleanup},
  {EAP_TYPE_TLS, "EAP_TLS", eaptls_setup, eaptls_process, eaptls_get_keys,
   eaptls_failed, eaptls_cleanup},
  {EAP_TYPE_TTLS, "EAP_TTLS", eapttls_setup, eapttls_process, eapttls_get_keys,
   eapttls_failed, eapttls_cleanup},
  {EAP_TYPE_MSCHAPV2, "EAP_MSCHAPV2", eapmschapv2_setup, eapmschapv2_process, 
   eapmschapv2_get_keys, eapmschapv2_failed, eapmschapv2_cleanup},
  {EAP_TYPE_PEAP, "EAP_PEAP", eappeap_setup, eappeap_process, eappeap_get_keys,
   eappeap_failed, eappeap_cleanup},
  {EAP_TYPE_LEAP, "EAP_LEAP", eapleap_setup, eapleap_process, eapleap_get_keys,
   eapleap_failed, eapleap_cleanup},
#ifdef EAP_SIM_ENABLE
  {EAP_TYPE_SIM, "EAP_SIM", eapsim_setup, eapsim_process, eapsim_get_keys, 
   eapsim_failed, eapsim_cleanup},
  {EAP_TYPE_AKA, "EAP_AKA", eapaka_setup, eapaka_process, eapaka_get_keys,
   eapaka_failed, eapaka_cleanup},
#endif
  {EAP_TYPE_OTP, "EAP_OTP", eapotp_setup, eapotp_process, eapotp_get_keys,
   NULL, eapotp_cleanup},
  {EAP_TYPE_GTC, "EAP_GTC", eapotp_setup, eapotp_process, eapotp_get_keys,
   NULL, eapotp_cleanup},
  {NO_EAP_AUTH, NULL, NULL, NULL, NULL, NULL, NULL}
};

/***************************************************
 *
 * Initalize anything needed for EAP.
 *
 ***************************************************/
void eap_init(struct interface_data *thisint)
{

}

/***************************************************
 *
 * This is called when the low level state machine resets.
 *
 ***************************************************/
void eap_reset(struct interface_data *thisint)
{
  // We have let the active EAP method know that we reset.
  thisint->statemachine->eapRestart = FALSE;
}

/***************************************************
 *
 * Look through the EAP array, and see if we can find the index to the
 * EAP method in question.
 *
 ***************************************************/
int eap_find_type(int eap_type)
{
  int eapmethod;

  // Determine which authenticator in our array is the right one.
  eapmethod = 0;

  while ((eaphandlers[eapmethod].eap_auth_type != NO_EAP_AUTH) &&
	 (eaphandlers[eapmethod].eap_auth_type != eap_type))
    {
      eapmethod++;
    }
     
  if ((eaphandlers[eapmethod].eap_auth_type == NO_EAP_AUTH) && (eap_type > 3))
    {
      debug_printf(DEBUG_NORMAL, 
		   "No EAP Type Handler found for EAP Type %d!\n",
		   eap_type);
    }
  
  return eapmethod;
}

/***************************************************
 *
 * Return the name of the EAP type given the number that was in the packet.
 * When using this function, be sure not to try to free() the returned
 * pointer!  It will make a mess!
 *
 ***************************************************/
char *eap_get_name_from_type(char eap_type)
{
  int eapidx;

  eapidx = eap_find_type(eap_type);
  
  return eaphandlers[eapidx].eapname;
}

/***************************************************
 *
 * Respond to a request Auth.
 *
 ***************************************************/
char eap_response_auth(struct interface_data *thisint)
{
  int eapsize, eapolver;
  struct config_network *network_data;

  debug_printf(DEBUG_STATE, "Building EAPOL-Response-Authentication\n");
  
  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to %s:%d!\n",
		   __FUNCTION__, __LINE__);
      return XEGENERROR;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid configuration available for this "
		   "network!\n");
      return XEBADCONFIG;
    }

  if (eap_create_active_method(&network_data->activemethod,
			       network_data->identity,
			       thisint->tempPassword,
			       thisint->intName) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't build active method! Authentication"
		   " will not happen!\n");
      return XEGENERROR;
    }

  switch (eap_request_auth(network_data->activemethod,
			   network_data->methods, 
			   (char *) &thisint->recvframe[OFFSET_TO_EAP],
			   thisint->recv_size, 
			   (char *) &thisint->sendframe[OFFSET_TO_EAP], &eapsize))
    {
    case XINNERSUCCESS:
      // We got a LEAP success, so we need to tell the state machine to
      // create the keying material.
      debug_printf(DEBUG_AUTHTYPES, "LEAP successful!\n");
      thisint->statemachine->eapSuccess = TRUE;
      thisint->statemachine->eapolEap = FALSE;
      eapleap_get_keys(thisint);
      break;
      
    case XEINVALIDEAP:
      debug_printf(DEBUG_AUTHTYPES, "Invalid EAP type, or couldn't initialize "
		   "EAP!\n");
      
      // We failed for some reason, so clear the received buffer so we don't
      // continue to try to process things.
      thisint->recv_size = 0;
      bzero(thisint->recvframe, 1520);
      break;
      
    case XEGENERROR:
      thisint->recv_size = 0;
      bzero(thisint->recvframe, 1520);
      break;
    }
  
  if (eapsize != 0)
    {
      eapolver = eapol_get_eapol_ver();

      eapol_build_header(EAP_PACKET, eapsize, eapolver, (char *) thisint->sendframe);
      thisint->send_size = eapsize+OFFSET_TO_EAP;
      return XENONE;
    }

  debug_printf(DEBUG_INT, "No response frame was built!\n");
  return XENOFRAMES;
}

/***************************************************
 *
 * Respond to a request ID.
 *
 ***************************************************/
char eap_response_id(struct interface_data *thisint)
{
  int eapsize, eapolver;
  struct eap_header *eapdata;
  struct config_network *network_data;
  
  debug_printf(DEBUG_STATE, "Building EAP-Response-ID!\n");
  
  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid value passed to %s:%d!\n",
		   __FUNCTION__, __LINE__);
      return XEGENERROR;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid network config for this network!"
		   "\n");
      return XEBADCONFIG;
    }

  // See if we need to prepopulate the ID
  eap_prepopulate_id(thisint);

  eapdata = (struct eap_header *)&thisint->recvframe[OFFSET_TO_EAP];

  if (network_data->identity == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No identity!  Make sure you have an "
		   "identity value in all of your network profiles!\n");
      return XEGENERROR;
    }

  debug_printf(DEBUG_CONFIG, "Identity : %s\n", network_data->identity);

  eap_request_id(network_data->identity,
		 eapdata->eap_identifier,
		 (char *) &thisint->sendframe[OFFSET_TO_EAP], &eapsize);

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAP_PACKET, eapsize, eapolver, (char *) thisint->sendframe);
  thisint->send_size = eapsize+OFFSET_TO_EAP;

  return XENONE;
}

/***************************************************
 *
 * Cleanup the active EAP type, and anything else that we set up for using
 * EAP.
 *
 ***************************************************/
void eap_cleanup(struct generic_eap_data **clearmethod)
{
  int searchval;
  struct generic_eap_data *activemethod;

  if ((!clearmethod) || (!(*clearmethod))) 
    {
      debug_printf(DEBUG_AUTHTYPES, "There was no active method in eap_cleanup()!\n");
      return;
    }

  activemethod = *clearmethod;

  debug_printf(DEBUG_EVERYTHING, "Calling EAP-Cleanup!\n");
  searchval = 0;

  if (activemethod->eapNum != 0)
    {
      searchval = eap_find_type(activemethod->eapNum);

      if (eaphandlers[searchval].eap_auth_type != NO_EAP_AUTH)
	{
	  (*eaphandlers[searchval].eap_auth_cleanup)(activemethod);
	  activemethod->eapNum = 0;
	} else {
	  debug_printf(DEBUG_NORMAL, "Couldn't clean up after active EAP type! (Type : %d)\n",activemethod->eapNum);
	  debug_printf(DEBUG_NORMAL, "This shouldn't be possible!  Please report it to the XSupplicant list!\n");
	}

      if (activemethod->identity != NULL)
	{
	  free(activemethod->identity);
	  activemethod->identity = NULL;
	}

      if (*clearmethod != NULL)
	{
	  free(*clearmethod);
	  *clearmethod = NULL;
	}
    }
}


/*******************************************
 *
 * We got an EAP-Notify message.  Parse, and display it for now.
 *
 *******************************************/
void eap_do_notify(struct interface_data *thisint)
{
  struct eap_header *myeap;
  char myval[255];
  u_char *inframe;
  int insize;

  if (!thisint)
    {
      debug_printf(DEBUG_EVERYTHING, "Got a Notify, but interface context "
		   "is invalid!\n");
      return;
    }

  inframe = thisint->recvframe;
  insize = thisint->recv_size;

  myeap = (struct eap_header *)&inframe[OFFSET_TO_EAP];

  bzero(&myval[0], 255);
  
  // We need to determine how long the string that we were returned is.
  // So, take the EAP length value, and subtract 5 to account for the EAP
  // header.
  strncpy(&myval[0], (char *) &inframe[OFFSET_TO_DATA], (ntohs(myeap->eap_length)-5));

  debug_printf(DEBUG_NORMAL, "EAP Notification : %s\n", &myval[0]);
  bzero(thisint->recvframe, thisint->recv_size);
  thisint->recv_size = 0;             // So we don't process this again.
  xsup_ipc_send_eap_notify(thisint, myval);
}

/*******************************************
 *
 * If the length of the EAP request ID packet is longer than the space needed
 * to store the basic EAP request, then we may have some network hints.  These
 * hints may include a network name that we can use to determine which network
 * profile to read from the configuration.
 *
 *******************************************/
void eap_process_hints(struct interface_data *ctx)
{
  struct eap_header     *myeap;
  char                  *inframe;
  char                  *hints;
  char                  *networkid, *nidonly;
  struct config_globals *globals;

  inframe = ctx->recvframe;

  myeap = (struct eap_header *)&inframe[OFFSET_TO_EAP];

  if (ctx->recv_size > (OFFSET_TO_EAP + sizeof(struct eap_header) + 1))
    {
      // We may have some data.  The character that trails the end of the
      // EAP packet should be 0x00.  So, skip it and show the results.
      hints = (char *)&inframe[OFFSET_TO_EAP + sizeof(struct eap_header)+1];

      globals = config_get_globals();
      if (!globals)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't get global configuration "
		       "options.  Assuming we should use EAP hints.\n");
	}

      if ((!globals) || 
	  (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_EAP_HINTS)))
	{

	  if (hints[0] != 0x00)
	    {
	      networkid = strstr(hints, "networkid=");
	      
	      if (networkid != NULL)
		{
		  nidonly = strtok(networkid, ",");
		  
		  nidonly += strlen("networkid=");
		  
		  if (nidonly[0] != 0x00)
		    {
		      debug_printf(DEBUG_NORMAL, "Network ID from EAP hint : %s\n", nidonly);
		      debug_printf(DEBUG_NORMAL, "If this is a wired network, the above ID "
				   "can be used in the configuration file to identify this "
				   "network.\n");
		      
		      if (config_build(ctx, nidonly) != TRUE)
			{
			  debug_printf(DEBUG_NORMAL, "Couldn't build config for network %s!\n",
				       nidonly);
			}
		    }
		}
	    } 
	}
    }
}

/*******************************************
 *
 * Process the EAP piece of the packet, determine what type of EAP packet it is
 * and set state machine variables accordingly.  The variables set will
 * cause the state machine to know what to do next.
 *
 *******************************************/
int eap_process_header(struct interface_data *thisint)
{
  struct eap_header *myeap;
  u_char *inframe;
  struct config_network *network_data;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context in %s!\n",
		   __FUNCTION__);
      return XEGENERROR;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No available network configuration "
		   "information!\n");
      return XEBADCONFIG;
    }

  if (!thisint->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Statemachine not initialized in %s()!\n",
		   __FUNCTION__);
      return XEGENERROR;
    }

  inframe = thisint->recvframe;

  myeap = (struct eap_header *)&inframe[OFFSET_TO_EAP];

  switch (myeap->eap_code)
    {
    case EAP_REQUEST:
      if (eap_get_name_from_type(myeap->eap_type) != NULL)
	{
	  debug_printf(DEBUG_EVERYTHING, "Got EAP-Request for type %d (%s).\n",
		       myeap->eap_type,
		       eap_get_name_from_type(myeap->eap_type));
	}

      // We have a data packet of some sort, so set eapolEap.
      thisint->statemachine->eapolEap = TRUE;
      thisint->statemachine->eapSuccess = FALSE;
      thisint->statemachine->eapFail = FALSE;
      thisint->statemachine->lastEapType = myeap->eap_type;

      switch (myeap->eap_type)
	{
	case EAP_TYPE_IDENTITY:
	  debug_printf(DEBUG_EVERYTHING, "Got EAP-Request-Identification.\n");
	  eap_process_hints(thisint);
	  return EAP_REQUEST_ID;
	  break;
	  
	case EAP_TYPE_NOTIFY:
	  debug_printf(DEBUG_EVERYTHING, "Got an EAP-Notify.\n");
	  return EAP_REQUEST_NOTIFY;
	  break;
	  
	default:
	  debug_printf(DEBUG_EVERYTHING, "Got EAP-Request-Authentication.\n");
	  if (ntohs(myeap->eap_length) <= 4)
	    {
	      debug_printf(DEBUG_NORMAL, "Got invalid EAP packet, ignoring!\n");
	      snmp_dot1xSuppEapLengthErrorFramesRx();
	    } else {
	    return EAP_REQUEST_AUTH;
	    }
	  break;
	}
      break;

    case EAP_RESPONSE:
      if (myeap->eap_type == EAP_TYPE_LEAP) {
	debug_printf(DEBUG_EVERYTHING, "Got LEAP Response Packet.  Ready for AP verification!\n");
	thisint->statemachine->eapolEap = TRUE;
	thisint->statemachine->eapSuccess = FALSE;
	thisint->statemachine->eapFail = FALSE;

	return EAP_REQUEST_AUTH;
      } else {
	debug_printf(DEBUG_EVERYTHING, "Got EAP-Response, ignoring!\n");
      }
      break;

    case EAP_SUCCESS:
      debug_printf(DEBUG_EVERYTHING, "Got EAP-Success!\n");
      thisint->statemachine->eapSuccess = TRUE;

      if (network_data->activemethod == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid activemethod after success packet recv!\n");
	  break;
	}

      snmp_eapol_success_rx();
      if (network_data->activemethod->eapNum == EAP_TYPE_LEAP) {
	myeap->eap_type = EAP_TYPE_LEAP;

	// We aren't quite done.. (This will keep us from being stuck in
	// success state.)
	thisint->statemachine->eapSuccess = FALSE;
	thisint->statemachine->eapolEap = TRUE;
	return EAP_REQUEST_AUTH;
      } else {
	debug_printf(DEBUG_NORMAL, "Successfully authenticated %s\n", 
			thisint->intName);
	thisint->statemachine->eapSuccess = TRUE;
      }

      // And get our keying material
      eap_get_keying_material(thisint);

      // Finally, dump the SNMP scoreboard if we need to.
      snmp_dump_stats(thisint->intName);
      break;

    case EAP_FAILURE:
      debug_printf(DEBUG_EVERYTHING, "Got EAP-Failure!\n");
      debug_printf(DEBUG_NORMAL, "Failed to authenticate %s\n", 
		      thisint->intName);
      snmp_eapol_fail_rx();
      thisint->statemachine->eapFail = TRUE;
      eap_do_fail(network_data->activemethod);
      
      // Dump the SNMP scoreboard if we need to.
      snmp_dump_stats(thisint->intName);
      break;
    }
  return XENONE;
}

/**********************************************************************
 *
 * For certain EAP methods, such as SIM and AKA, we can populate the ID
 * based on something other than the config file.  (Such as a smartcard.)
 * 
 **********************************************************************/
void eap_prepopulate_id(struct interface_data *thisint)
{
#ifdef EAP_SIM_ENABLE
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Interface Data struct is invalid in "
		   "eap_prepopulate_id()!\n");
      return;
    }

  if (!network_data)
    {
      debug_printf(DEBUG_NORMAL, "No valid network config is set! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  // If we have SIM enabled, there is no username, and the primary EAP method
  // is SIM, then ask the SIM card for it's IMSI to use as the username.
  if ((network_data->identity == NULL) && 
      (network_data->methods->method_num == EAP_TYPE_SIM))
    {
      network_data->identity = (char *)malloc(50);
      if (network_data->identity == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for identity!\n");
	  return;
	}
      eapsim_get_username(thisint);
    }

  // Same is true for AKA.
  if ((network_data->identity == NULL) &&
      (network_data->methods->method_num == EAP_TYPE_AKA))
    {
      network_data->identity = (char *)malloc(50);
      if (network_data->identity == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for identity!\n");
	  return;
	}
      eapaka_get_username(thisint);
    }
#endif
}

/************************************************
 *
 * Process an EAP Request ID, and respond with the username information that
 * we have configured.  (If nothing is configured, we should ignore the
 * packet, and just return.  Returning an outsize of 0 means that we are
 * ignoring things.)
 *
 ************************************************/
void eap_request_id(char *identity, int eapid, char *outframe, 
		    int *eapsize)
{
  struct eap_header *myeap;
  char *username_ofs;

  if (!identity)
    {
      debug_printf(DEBUG_NORMAL, "NULL identity passed to %s!\n", __FUNCTION__);
      return;
    }

  if (!outframe)
    {
      debug_printf(DEBUG_NORMAL, "Invalid return frame buffer in %s!\n",
		   __FUNCTION__);
      return;
    }

  if (!eapsize)
    {
      debug_printf(DEBUG_NORMAL, "Invalid EAP size return buffer in %s!\n",
		   __FUNCTION__);
      return;
    }

  myeap = (struct eap_header *)outframe;

  myeap->eap_code = EAP_RESPONSE;
  myeap->eap_identifier = eapid; 

  if (identity == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No identity has been specified!  Authentication will not happen!\n");
      return;
    }

  *eapsize = (strlen(identity)+sizeof(struct eap_header));
  myeap->eap_length = htons(*eapsize);
  myeap->eap_type = EAP_TYPE_IDENTITY;

  username_ofs = (char *)&outframe[sizeof(struct eap_header)];
  strncpy(username_ofs, identity, strlen(identity));
}

/************************************************
 *
 * Create/update the active method struct.
 *
 ************************************************/
int eap_create_active_method(struct generic_eap_data **activemethod,
			     char *identity, char *tempPwd, char *intname)
{
  struct generic_eap_data *mymethod;

  if (activemethod == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid pointer passed to %s!\n",
		   __FUNCTION__);
      return XEMALLOC;
    }

  mymethod = *activemethod;

  if (mymethod == NULL)
    {
      *activemethod = (struct generic_eap_data *)malloc(sizeof(struct generic_eap_data));
      mymethod = *activemethod;

      if (mymethod == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory in %s:%d!\n",
		       __FUNCTION__, __LINE__);
	  return XEMALLOC;
	}

      memset(mymethod, 0, sizeof(struct generic_eap_data));

      mymethod->eap_conf_data = NULL;
      mymethod->eap_data = NULL;
   
      mymethod->identity = (char *)malloc(strlen(identity)+1);
      
      if (mymethod->identity == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory to copy identity!\n");
	} else {
	  strcpy(mymethod->identity, identity);
	}
    }

  mymethod->tempPwd = tempPwd;
  mymethod->intName = intname;

  return XENONE;
}

/************************************************
 *
 * Process an authentication request.  Based on the information in the packet,
 * we call the correct EAP type.  We return an error if it is an EAP type
 * that we don't know.
 *
 ************************************************/
int eap_request_auth(struct generic_eap_data *activemethod,
		     struct config_eap_method *eapConfig,
		     char *inframe, int insize, char *outframe, int *eapsize)
{
  struct eap_header *myouteap, *myineap;
  int eapmethod, done, valideap, working_eap_type, eapinsize = 0;
  struct config_eap_method *start=NULL, *cur=NULL, *newmethod = NULL;
  char *tosendframe;
  int retVal = XENONE, pwd_needed;

  if (!outframe)
    {
      debug_printf(DEBUG_NORMAL, "Invalid value for 'outframe' in %s:%d!\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (!eapsize)
    {
      debug_printf(DEBUG_NORMAL, "Invalid value for 'eapsize' in %s:%d!\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  eapmethod = 0;
  *eapsize = 0;
  done = FALSE;
  valideap = FALSE;

  myineap = (struct eap_header *)inframe;
  tosendframe = (char *)&inframe[sizeof(struct eap_header)];
  working_eap_type = myineap->eap_type;
  
  myouteap = (struct eap_header *)outframe;

  // Make sure we have valid method data.
  if (eapConfig == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No EAP methods available in "
		   "eap_request_auth()!\n");
      return XEMALLOC;
    }

  // Check to make sure that the type requested is in our list of valid
  // types.
  
  start = eapConfig;
  cur = start;

  while ((cur != NULL) && (cur->method_num != working_eap_type))
    {
      cur = cur->next;
    }

  // If we have a type that matches, then go ahead...
  if (cur != NULL)
    {
      valideap = TRUE;
      newmethod = cur;
      activemethod->eap_conf_data = newmethod->method_data;
    } else {
      valideap = FALSE;
      activemethod->eap_conf_data = NULL;
    }

  // If the requested EAP type isn't valid, then send a NAK.
  if ((valideap == FALSE) && (ntohs(myineap->eap_length)>4))
    {
      debug_printf(DEBUG_STATE, "Unsupported EAP type requested. (%d)  Sending NAK!\n",myineap->eap_type);
      myouteap->eap_code = EAP_RESPONSE;
      myouteap->eap_identifier = myineap->eap_identifier;
      myouteap->eap_length = htons(6);
      myouteap->eap_type = EAP_TYPE_NAK;

      if (eapConfig == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "There are no authentication methods defined for this interface!  Make sure you have at least one valid EAP type defined in your configuration.\n");
	  return XEBADCONFIG;
	}

      outframe[sizeof(struct eap_header)] = eapConfig->method_num;
      *eapsize = 6;
      return *eapsize;
    }

  // Now, determine which authenticator in our array is the right one.
  eapmethod = eap_find_type(working_eap_type);

  if (eaphandlers[eapmethod].eap_auth_type == NO_EAP_AUTH)
    {
      debug_printf(DEBUG_NORMAL, "No valid EAP type could be found in %s:%d!\n",
		   __FUNCTION__, __LINE__);
      // We got an error.
      return XEINVALIDEAP;
    }

  // If we had an EAP type before, and we have changed this time through,
  // make sure we call the cleanup methods.
  if ((activemethod->eapNum > 0) && 
      (activemethod->eapNum != eaphandlers[eapmethod].eap_auth_type))
    {
      debug_printf(DEBUG_AUTHTYPES, "EAP Type Changed!  Cleaning up old type!\n");
      eap_clear_active_method(activemethod);
    }

  // If this is a new EAP type, call the setup method.
  if (activemethod->eapNum == 0)
    {
      if (((*eaphandlers[eapmethod].eap_auth_setup)(activemethod)) != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "EAP method failed to set up properly! "
		       "Calling cleanup routine.\n");
	  eap_cleanup(&activemethod);
	  
	  return XEINVALIDEAP;
	}
	  
      activemethod->eapNum = eaphandlers[eapmethod].eap_auth_type;

      if (activemethod->eap_data == NULL)
	{
	  debug_printf(DEBUG_AUTHTYPES, "This EAP type didn't set up any state information!?\n");
	}
    } 

  activemethod->eapid = myineap->eap_identifier;
  eapinsize = ntohs(myineap->eap_length)-5;

  pwd_needed = activemethod->need_password;

  if ((pwd_needed != 1) || (activemethod->tempPwd != NULL))
    {
      switch ((*eaphandlers[eapmethod].eap_auth_handlers)(activemethod, 
						      (u_char *) tosendframe, eapinsize, 
						  (u_char *) &outframe[sizeof(struct eap_header)], 
						  eapsize))
	{
	case XINNERSUCCESS:
	  retVal = XINNERSUCCESS;
	  break;

	case XTLSNEEDDATA:
	  break;

	case XENONE:
	  break;

	default:
	  debug_printf(DEBUG_NORMAL, "XEGENERROR\n");
	  return XEGENERROR;
	}
    }

  // See if an EAP type requested a password.
  if ((activemethod->need_password == 1) && (pwd_needed == 0))
    {
      debug_printf(DEBUG_AUTHTYPES, "Requesting password from GUI!\n");

      xsup_ipc_gui_prompt(activemethod->intName, activemethod->tempPwd, 
			  activemethod->eaptype, activemethod->eapchallenge);

      *eapsize = 0;
      free(activemethod->eaptype);
      activemethod->eaptype = NULL;
      free(activemethod->eapchallenge);
      activemethod->eapchallenge = NULL;

      return XENONE;
    };

  if ((activemethod->need_password == 1) && (pwd_needed == 1) &&
      (!activemethod->tempPwd)) *eapsize = 0;

  // If we are using LEAP, we need to make some extra calls here.
  if (eaphandlers[eapmethod].eap_auth_type == EAP_TYPE_LEAP)
    {
      if (eapleap_done(activemethod) == 1)
	{
	  retVal = XINNERSUCCESS;
	}
    }

  if (*eapsize > 0)
    {
      *eapsize = *eapsize + (sizeof(struct eap_header));
      myouteap->eap_length = htons(*eapsize);
      if (eaphandlers[eapmethod].eap_auth_type == EAP_TYPE_LEAP &&
	  myineap->eap_code == EAP_SUCCESS) {
	myouteap->eap_code = EAP_REQUEST;
	myouteap->eap_identifier = myineap->eap_identifier;
      } else {
	myouteap->eap_code = EAP_RESPONSE;
	if (myineap == NULL)
	  {
	    debug_printf(DEBUG_NORMAL, "Invalid packet! (%s:%d)\n",
			 __FUNCTION__, __LINE__);
	    return XENOFRAMES;
	  }
	myouteap->eap_identifier = myineap->eap_identifier;
      }
      myouteap->eap_type = activemethod->eapNum;
    } 

  return retVal;
}


/************************************************************************
 *
 * Clear the active EAP type.  This will be called when the EAP type we are
 * using has changed, or we have encountered another event (such as an
 * essid change) that should require a completely new authentication!
 *
 ************************************************************************/
int eap_clear_active_method(struct generic_eap_data *activemethod)
{
  int eapmethod = 0;

  // First, make sure we have something to clean up.
  if (activemethod == NULL)
    {
      debug_printf(DEBUG_AUTHTYPES, "There was nothing to clean up in eap_clear_active_method!\n");
      return XENONE;
    }

  if (activemethod->eapNum == 0)
    {
      debug_printf(DEBUG_NORMAL, "Invalid EAP type %d!\n", activemethod->eapNum);
      return -1;
    }

  eapmethod = eap_find_type(activemethod->eapNum);

  (*eaphandlers[eapmethod].eap_auth_cleanup)(activemethod);
  activemethod->eapNum = 0;

  return XENONE;
}

/*************************************************
 *
 * Ask the EAP method to give us keying material.
 *
 *************************************************/
int eap_get_keying_material(struct interface_data *thisint)
{
  int eapmethod = 0;
  struct config_network *network_data;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure in "
		   "eap_get_keying_material()!\n");
      return XEMALLOC;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration information! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (network_data->activemethod == NULL)
    {
      debug_printf(DEBUG_AUTHTYPES, "The EAP type doesn't seem to exist anymore!\n");
      return XENONE;
    }

  eapmethod = eap_find_type(network_data->activemethod->eapNum);

  if (eaphandlers[eapmethod].eap_auth_type == NO_EAP_AUTH)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't find an EAP type handler to clean up from!\n");
      debug_printf(DEBUG_NORMAL, "We will probably leak memory!\n");
      return XENONE;
    }

  (*eaphandlers[eapmethod].eap_auth_get_keys)(thisint);

  thisint->statemachine->PMK = (char *) thisint->keyingMaterial;
  debug_printf(DEBUG_INT, "PMK Keys (%d) :\n", thisint->keyingLength);
  debug_hex_dump(DEBUG_INT, (u_char *) thisint->statemachine->PMK, thisint->keyingLength);

  debug_printf(DEBUG_INT, "WEP Keys (%d) :\n", thisint->keyingLength);
  debug_hex_dump(DEBUG_INT, thisint->keyingMaterial, thisint->keyingLength);
  return XENONE;
}

/************************************************************************
 *
 * Notify the eap method that the attempt has failed.  This should be used
 * for things such as destroying a password that has failed, so it will be
 * requested again.  Or, for resetting a context in the case of TLS based
 * authentication methods.
 *
 ************************************************************************/
int eap_do_fail(struct generic_eap_data *activemethod)
{
  int eapmethod = 0;

  // Make sure that we have something to clear.
  if (activemethod == NULL)
    {
      debug_printf(DEBUG_AUTHTYPES, "There was no method defined for executing a failure!\n");
      return XENONE;
    }

  eapmethod = eap_find_type(activemethod->eapNum);

  if (eaphandlers[eapmethod].eap_auth_type == NO_EAP_AUTH)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't find an EAP type handler to notify of the failure!\n");
      return XENONE;
    }

  if (eaphandlers[eapmethod].eap_auth_failed == NULL) 
    {
      debug_printf(DEBUG_AUTHTYPES, "EAP handler didn't have a failure method!\n");
      return XENONE;
    }

  (*eaphandlers[eapmethod].eap_auth_failed)(activemethod);

  return XENONE;
}
