/*******************************************************************
 *
 * File: xsup_ipc.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu, Terry.Simons@utah.edu
 *
 * $Id: xsup_ipc.c,v 1.34 2005/08/20 19:26:42 chessing Exp $
 * $Date: 2005/08/20 19:26:42 $
 * $Log: xsup_ipc.c,v $
 * Revision 1.34  2005/08/20 19:26:42  chessing
 * Added option -s that will force removal of old unix domain control socket.  This option should only be used on startup, since it could cause some strange problems if it is used when an existing Xsupplicant instance is in operation.
 *
 * Revision 1.33  2005/08/18 19:12:48  chessing
 * Patch from Carsten Grohmann to fix xsup_ipc to use the correct socket name.
 *
 * Revision 1.32  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <net/if.h>

#include "profile.h"
#include "config.h"
#include "xsup_ipc.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "ipc_callout.h"

#define XSUP_SOCKET "/tmp/xsupplicant.sock."

static int ipc_sock;

// mset is the master set of sockets in use.
static fd_set mset;
static int max_sock = 0;
char socknamestr[256];

/******************************************************************
 *
 * Initalize the socket that we will use to communicate with a client/clients.
 * Also, set up any structures that may be needed.
 *
 ******************************************************************/
int xsup_ipc_init(struct interface_data *intdata)
{
  int sockErr = 0, ipc_gid = 0;
  char *error = NULL;
  struct sockaddr_un sa;

  sa.sun_family = AF_LOCAL;
  bzero(socknamestr, 256);
  strncpy(socknamestr, XSUP_SOCKET, 256-(strlen(intdata->intName)+1));
  strcat (socknamestr, intdata->intName);
  strncpy(sa.sun_path, socknamestr, sizeof(sa.sun_path));

  if (TEST_FLAG(intdata->flags, CLEAR_IPC))
    {
      // We need to clear the socket file if it exists.

      debug_printf(DEBUG_INT, "Clearing control socket %s.\n", socknamestr);
      remove(socknamestr);
    }

  // Socket we will be using to communicate.
  ipc_sock = socket(PF_UNIX, SOCK_STREAM, 0);

  if (ipc_sock == -1) {
    debug_printf(DEBUG_NORMAL, "Couldn't establish handler to daemon socket!\n");
    return XENOSOCK;
  } 

  max_sock = ipc_sock;
  FD_SET(ipc_sock, &mset);

  debug_printf(DEBUG_INT, "Opened socket descriptor #%d\n", ipc_sock);

  sockErr = bind(ipc_sock, (struct sockaddr *)&sa, sizeof(sa));
  if (sockErr == -1) 
    {
      error = strerror(errno);
      debug_printf(DEBUG_NORMAL, "An error occured binding to socket.  (Error : %s)\n", error);
      close(ipc_sock);
      return XENOSOCK;
    }

  sockErr = listen(ipc_sock, 10);
  if (sockErr < 0)
    {
      error = strerror(errno);
      debug_printf(DEBUG_NORMAL, "An error occured listening on the socket! "
		   "(Error : %s)\n", error);
      close(ipc_sock);
      return XENOSOCK;
    }

  // Set the rights on the file.
  if (chmod(socknamestr, S_IREAD | S_IWRITE | S_IEXEC | S_IRGRP | S_IWGRP |
	     S_IXGRP | S_IROTH | S_IXOTH) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Can't set rights on socket file %s! (Error"
		   " : %s)\n", socknamestr, strerror(errno));
    }

  // Set the correct group ownership. 
  ipc_gid = config_get_ipc_gid();

  if (ipc_gid >= 0)
    {
      if (chown(socknamestr, -1, ipc_gid) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "Can't set group ownership on socket "
                       "file %s! (Error : %s)\n", socknamestr, strerror(errno));
	}
    }

  return XENONE;
}

/**************************************************************
 *
 * Send a message to a client.
 *
 **************************************************************/
void xsup_ipc_send_message(int skfd, char *tosend, int tolen)
{
  int err;

  if ((!tosend) || (tolen <= 0))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to xsup_ipc_send_message()!\n");
      return;
    }

  debug_printf(DEBUG_INT, "(IPC) Sending : \n");
  debug_hex_dump(DEBUG_INT, tosend, tolen);

  err = send(skfd, tosend, tolen, 0);
  if (err < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error writing to client! (Error : %s)\n",
		   strerror(errno));
      return;
    }

  if (err < tolen)
    {
      debug_printf(DEBUG_NORMAL, "Runt message sent to client!\n");
    }
}

/**************************************************************
 *
 * Get a message from a client.  Validate it, and return the payload.
 * outsize should be passed in a valid that is the maximum size of
 * outbuf.  outsize will then be changed to the size of the result
 * buffer.
 *
 **************************************************************/
void xsup_ipc_get_message(int sockdesc, char *outbuf, int *outsize)
{
  int readStat = -1;

  if ((!outbuf) || (!outsize))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to xsup_ipc_get_message()!\n");
      return;
    }

  readStat = recv(sockdesc, outbuf, *outsize, 0);

  if (readStat <= 0)
    {
      if (readStat == 0)
	{
	  debug_printf(DEBUG_NORMAL, "Disconnecting socket!\n");
	  FD_CLR(sockdesc, &mset);
	} else {
	  debug_printf(DEBUG_NORMAL, "Couldn't read message from a client! (Error %d -- %s)\n", errno,
		       strerror(errno));
	}

      *outsize = 0;
    } else {
      debug_printf(DEBUG_INT, "(IPC) Got : \n");
      debug_hex_dump(DEBUG_INT, outbuf, readStat);
      // Verify that the packet we got is valid.
      
      *outsize = readStat;
    }
}

/**********************************************************     
 *
 * Send a message to all registered clients.
 *
 ***********************************************************/
int xsup_ipc_send_all(char *message, int msglen)
{
  int i;

  for (i=0;i<max_sock+1;i++)
    {
      if(msglen <= 0)
	{
	  printf("Invalid message length: %d\n", msglen);
	  
	  if(message != NULL)
	    {
	      printf("\tfor message: %s\n", message);
	    }
	}
      else if ((FD_ISSET(i, &mset)) && (i != ipc_sock))
	{
	  xsup_ipc_send_message(i, message, msglen);
	}
    }
  
  return XENONE;
}

/***********************************************************
 *
 * Push the current state to all clients.
 *
 ***********************************************************/
void xsup_ipc_send_auth_state(struct interface_data *workint)
{
  char buffer[50];
  int bufptr;
  struct ipc_cmd *cmd;

  bufptr = 0;
  cmd = (struct ipc_cmd *)buffer;
  cmd->version = IPC_VERSION_NUM;
  cmd->int_idx = workint->intIndex;
  cmd->attribute = AUTH_STATE;
  cmd->getset = IPC_SET;          // This is an unrequested push.
  cmd->len = 1;
  bufptr += sizeof(struct ipc_cmd);
  buffer[bufptr] = workint->statemachine->curState;
  bufptr += 1;

  xsup_ipc_send_all(buffer, bufptr);
}

/***********************************************************
 *
 * Push an EAP notification to any connected clients.
 *
 ***********************************************************/
void xsup_ipc_send_eap_notify(struct interface_data *workint, char *notify)
{
  char buffer[50];
  int bufptr;
  struct ipc_cmd *cmd;

  bufptr = 0;
  cmd = (struct ipc_cmd *)buffer;
  cmd->version = IPC_VERSION_NUM;
  cmd->int_idx = workint->intIndex;
  cmd->attribute = NOTIFY;
  cmd->getset = IPC_SET;          // This is an unrequested push.
  cmd->len = strlen(notify);
  bufptr += sizeof(struct ipc_cmd);
  strncpy((char *)&buffer[bufptr], notify, strlen(notify));
  bufptr += strlen(notify);

  xsup_ipc_send_all(buffer, bufptr);
}

/***********************************************************
 *
 * Handle the requests the client sent us.
 *
 ***********************************************************/
void xsup_ipc_process_message(struct interface_data *workint, int skfd,
			      char *buffer, int bufsize)
{
  char sendbuf[1520];
  int sendsize, bufptr;
  struct ipc_cmd *cmd;

  sendsize = 0;

  bufptr = 0;
  
  while (bufptr < bufsize)
    {
      cmd = (struct ipc_cmd *)&buffer[bufptr];

      if (workint != NULL)
	{
	  switch (cmd->attribute)
	    {
	    case AUTH_STATE:
	      debug_printf(DEBUG_NORMAL, "Checking auth state!\n");
	      ipc_callout_auth_state(workint, &bufptr, buffer, 
				     bufsize, (char *)&sendbuf, &sendsize);
	      break;
	      
	    case CONFIG:
	      debug_printf(DEBUG_EVERYTHING, "Processing config!\n");
	      ipc_callout_process_conf(workint, &bufptr, buffer, 
				       bufsize, (char *)&sendbuf, &sendsize);
	      break;
	      
	    case PROFILE:
	      // Get or set a profile.
	      debug_printf(DEBUG_EVERYTHING, "Loading new profile...\n");
	      ipc_callout_getset_profile(workint, &bufptr, buffer,
					 bufsize, (char *)&sendbuf, &sendsize);
	      break;
	      
	    case TEMPPASSWORD:
	      // Set a temporary password.
	      debug_printf(DEBUG_EVERYTHING, "Setting temporary password.\n");
	      ipc_callout_set_password(workint, &bufptr, buffer,
				       bufsize, (char *)&sendbuf, &sendsize);
	      break;
	      
	    case ERROR_MSG:
	      debug_printf(DEBUG_NORMAL, "Got an error message from the client.  "
			   "Your client is probably broken!\n");
	      break;
	      
	    default:
	      debug_printf(DEBUG_NORMAL, "Unknown command %02X!\n",
			   buffer[bufptr]);
	      cmd = (struct ipc_cmd *)&buffer[bufptr];
	      
	      if ((bufptr+(cmd->len+sizeof(struct ipc_cmd))) > bufsize)
		{
		  debug_printf(DEBUG_NORMAL, "Size included in command is too "
			       "high! Ignoring remaining part of packet!\n");
		  bufptr = bufsize;
		} else {
		  bufptr+= (cmd->len+sizeof(struct ipc_cmd));
		}
	    }
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "Error getting working interface "
		       "dumping current packet: \n");

	  debug_hex_dump(DEBUG_NORMAL, buffer, bufsize);

	  debug_printf(DEBUG_NORMAL, "Ignoring remaining parts of packet!\n");
	  bufptr = bufsize+1;
	}
    }
  
  // Make sure we actually have something to return.
  if (sendsize >= (sizeof(struct ipc_cmd)))
    {
      debug_printf(DEBUG_EVERYTHING, "Sending IPC response!\n");
      xsup_ipc_send_message(skfd, sendbuf, sendsize);
    } else {
      debug_printf(DEBUG_EVERYTHING, "sendsize = %d  -- Nothing to send!\n", 
		   sendsize);
    }
}

/***********************************************************
 *
 * Process any IPC messages, and respond accordingly.
 *
 ***********************************************************/
void xsup_ipc_process(struct interface_data *thisint)
{
  char buffer[1520];   // We shouldn't have a message larger than this.
  char result_buf[1520];
  int bufsize, resbufptr, numcmds, selres, i, newsock, len;
  fd_set tset;
  struct timeval tv;
  struct sockaddr sa;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct in xsup_ipc_process()!\n");
      return;
    }

  bufsize = 1520;
  resbufptr = 0;
  numcmds = 0;

  bzero(&result_buf[0], 1520);

  tset = mset;
  tv.tv_sec = 0;
  tv.tv_usec = 0;

  selres = select(max_sock+1, &tset, NULL, NULL, &tv);
  if (selres <= 0)
    {
      // Nothing to do.
      return;
    }

  // Otherwise, loop through available sockets, and check for events.
  for (i=0;i<max_sock+1;i++)
    {
      if (FD_ISSET(i, &tset))
	{
	  if (i == ipc_sock)
	    {
	      // We got a request for a new IPC client connection.
	      debug_printf(DEBUG_INT, "Got a request to connect a new client.\n");
	      newsock = accept(ipc_sock, (struct sockaddr *)&sa, &len);
	      if (newsock <= 0)
		{
		  debug_printf(DEBUG_NORMAL, "Got a request for a new IPC client "
			       "connection.  But, accept() returned an error!\n");
		} else {
		  FD_SET(newsock, &mset);
		  if (newsock > max_sock)
		    {
		      max_sock = newsock;
		    }
		}
	    } else {
	      bufsize = 1520;
	      xsup_ipc_get_message(i, (char *)&buffer, &bufsize);

	      if(DEBUG_EVERYTHING)
		{
		  xsup_ipc_debug((char *)buffer, bufsize);
		}
	      
	      xsup_ipc_process_message(thisint, i, (char *)&buffer, bufsize);
	    }
	}
    }
}

/***********************************************************
 *
 * Clean up any structures used, and close out the communication socket.
 *
 ***********************************************************/
void xsup_ipc_cleanup()
{
  char *error;

  debug_printf(DEBUG_EVERYTHING, "Shutting down IPC socket!\n");
  debug_printf(DEBUG_INT, "Closing socket descriptor #%d\n", ipc_sock);
  if (close(ipc_sock) < 0)
    {
      error = strerror(errno);
      debug_printf(DEBUG_NORMAL, "Error closing socket!  (Error : %s)\n",
		   error);
    }

  unlink(socknamestr);
}

/***********************************************************
 *
 * Dump an IPC control message.  Useful for debugging clients.
 *
 ***********************************************************/
void xsup_ipc_debug(char *message, int length)
{
  printf("xsup_ipc_debug stub\n");
}

/*******************************************************************
 * 
 * Send a message to any GUI that is listening to let it know that we need
 * a password.
 *
 *******************************************************************/
int xsup_ipc_gui_prompt(char *interface, char *tempPwd, char *eapType, 
			char *challenge)
{
  char packet[512];
  int bufptr = 0, intidx;

  if (tempPwd == NULL)
    {
      // Ask the GUI to prompt for a password.
      debug_printf(DEBUG_AUTHTYPES, "Asking the GUI for a password.\n");
      debug_printf(DEBUG_AUTHTYPES, "EAP type is %s, challenge is %s\n",
		   eapType, challenge);

      bzero((char *)&packet[0], 512);

      if(interface)
	{
	  intidx = if_nametoindex(interface);
	  debug_printf(DEBUG_AUTHTYPES, "Interface : %s  Index : %d\n",
		       interface, intidx);
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "Invalid interface data in %s Line: %d\n",
		       __FUNCTION__, __LINE__);
	  return XEBADCONFIG;
	}

      ipc_callout_request_password(intidx, &bufptr, (char *)&packet[0], 512, 
				   eapType, challenge);
      debug_printf(DEBUG_AUTHTYPES, "Sending : \n");
      debug_hex_dump(DEBUG_AUTHTYPES, packet, bufptr);

      xsup_ipc_send_all((char *)&packet, bufptr);
      
      // Let the caller know we are asking for a password.
      return XPROMPT;
    }

  return XENONE;
}

