/*******************************************************************
 * Handle keying for WPA keys.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: eapol_key_type254.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapol_key_type254.c,v 1.41 2005/08/09 01:39:14 chessing Exp $
 * $Date: 2005/08/09 01:39:14 $
 * $Log: eapol_key_type254.c,v $
 * Revision 1.41  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 ******************************************************************/

#include <string.h>
#include <stdlib.h>

#include "xsup_debug.h"
#include "xsup_err.h"
#include "profile.h"
#include "config.h"
#include "key_statemachine.h"
#include "eapol_key_type254.h"
#include "eapol.h"
#include "frame_structs.h"
#include "wpa.h"
#include "wpa_common.h"
#include "psk.h"
#include "mic.h"
#include "snmp.h"
#include "cardif/cardif.h"
#include "eap_types/mschapv2/mschapv2.h"

u_char group_key_ver = 0, pairwise_key_ver = 0;

/**************************************************************
 *
 * Given a frame, parse all of the data that is contained in it, and
 * provide a human readable output that is useful for debugging.
 *
 **************************************************************/
void eapol_key_type254_dump(char *framedata)
{
  uint16_t value16=0;
  int need_comma = 0;
  struct wpa_key_packet *keydata;

  keydata = (struct wpa_key_packet *)&framedata[OFFSET_TO_EAPOL+4];

  debug_printf(DEBUG_INT, "Key Descriptor      = %d\n", keydata->key_descriptor);
  memcpy(&value16, keydata->key_information, 2);
  debug_printf(DEBUG_INT, "Key Information     = %04X  (Flags : ", ntohs(value16));
  if (ntohs(value16) & WPA_PAIRWISE_KEY)
    {
      debug_printf_nl(DEBUG_INT, "Pairwise Key");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_INSTALL_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Install Key");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_KEY_ACK_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Key Ack");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_KEY_MIC_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "MIC");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_SECURE_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Secure");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_ERROR_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Error");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_REQUEST_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Request");
      need_comma = 1;
    }

  debug_printf_nl(DEBUG_INT, ")\n");

  switch (ntohs(value16) & WPA_KEYTYPE_MASK)
    {
    case 1:
      debug_printf(DEBUG_INT, "Key Descriptor Version : HMAC-MD5 for MIC and RC4 for encryption.\n");
      break;

    case 2:
      debug_printf(DEBUG_INT, "Key Descriptor Version : HMAC-SHA1-128 for MIC and AES for encryption.\n");
      break;
    }

  debug_printf(DEBUG_INT, "Key Length          = %d\n", 
	       ntohs(keydata->key_length));

  debug_printf(DEBUG_INT, "Key Replay Counter  = ");  
  debug_hex_printf(DEBUG_INT, keydata->key_replay_counter, 8);

  debug_printf(DEBUG_INT, "Key NONCE           = ");
  debug_hex_printf(DEBUG_INT, keydata->key_nonce, 32);

  debug_printf(DEBUG_INT, "Key IV              = ");
  debug_hex_printf(DEBUG_INT, keydata->key_iv, 16);
  
  debug_printf(DEBUG_INT, "Key RSC             = ");
  debug_hex_printf(DEBUG_INT, keydata->key_rsc, 8);
  
  debug_printf(DEBUG_INT, "Key ID              = ");
  debug_hex_printf(DEBUG_INT, keydata->key_id, 8);

  debug_printf(DEBUG_INT, "Key MIC             = ");
  debug_hex_printf(DEBUG_INT, keydata->key_mic, 16);

  value16 = ntohs(keydata->key_material_len);
  debug_printf(DEBUG_INT, "Key Material Length = %d\n", value16);

  if (value16 > 0)
    {
      debug_printf(DEBUG_INT, "Key Data : (%d)\n", value16);
      debug_hex_dump(DEBUG_INT, keydata->keydata, value16);
    }
}

/*******************************************************
 *
 * Generate the pre-Temporal key. (PTK) Using the authenticator, and 
 * supplicant nonces.  (Anonce, and Snonce.)  The PTK is used for keying
 * when we are ready.
 *
 *******************************************************/
char *eapol_key_type254_gen_ptk(struct interface_data *intdata, char *Anonce)
{
  char prfdata[76];  // 6*2 (MAC addrs) + 32*2 (nonces)
  char *retval;

  bzero((char *)&prfdata, 76);

  if (memcmp((char *)&intdata->source_mac, (char *)&intdata->dest_mac, 6) < 0)
    {
      memcpy((char *)&prfdata[0], (char *)&intdata->source_mac, 6);
      memcpy((char *)&prfdata[6], (char *)&intdata->dest_mac, 6);
    } else if (memcmp((char *)&intdata->source_mac, (char *)&intdata->dest_mac,
		      6) > 0)
      {
	memcpy((char *)&prfdata[0], (char *)&intdata->dest_mac, 6);
	memcpy((char *)&prfdata[6], (char *)&intdata->source_mac, 6);
      } else {
	debug_printf(DEBUG_NORMAL, "Source and Destination MAC addresses "
		     "match!  The PTK won't be valid!\n");
	return NULL;
      }

  if (memcmp(intdata->statemachine->SNonce, Anonce, 32) < 0)
    {
      memcpy((char *)&prfdata[12], intdata->statemachine->SNonce, 32);
      memcpy((char *)&prfdata[44], Anonce, 32);
    } else if (memcmp(intdata->statemachine->SNonce, Anonce, 32) > 0)
      {
	memcpy((char *)&prfdata[12], Anonce, 32);
	memcpy((char *)&prfdata[44], intdata->statemachine->SNonce, 32);
      } else {
	debug_printf(DEBUG_NORMAL, "ANonce and SNonce match!  The PTK won't"
		     " be valid!\n");
	return NULL;
      }
  
  retval = (char *)malloc(80);
  if (retval == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for retval in %s "
		   "at %d!\n", __FUNCTION__, __LINE__);
      return NULL;
    }

  debug_printf(DEBUG_INT, "PMK : ");
  debug_hex_printf(DEBUG_INT, intdata->statemachine->PMK, 32);
  wpa_PRF(intdata->statemachine->PMK, 32, "Pairwise key expansion", 22,
	  (char *)&prfdata, 76, retval, 64);

  debug_printf(DEBUG_INT, "PTK : ");
  debug_hex_printf(DEBUG_INT, retval, 64);

  return retval;
}

/*****************************************************************
 *
 * When a MIC failure occures, we need to send the AP a request for
 * a new key. (Reference 802.11i-D3.0.pdf page 43, line 8)
 *
 *****************************************************************/
void eapol_key_type254_request_new_key(struct interface_data *intdata, 
				       char unicast)
{
  struct wpa_key_packet *outkeydata;
  uint16_t value16, keyindex, len;
  int eapolver;
  char key[16];
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error getting network configuration data! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  outkeydata = (struct wpa_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa_key_packet));

  outkeydata->key_descriptor = WPA_KEY_TYPE;

  value16 = (WPA_REQUEST_FLAG | WPA_ERROR_FLAG);

  if (unicast == 1)
    {
      // Set the key flags to indicate this is a pairwise key, with an
      // index of 0.
      keyindex = (WPA_PAIRWISE_KEY | pairwise_key_ver);
    } else {
      // Set the key flags to indicate this is a group key. We don't know the
      // index, so for now we will use 1.
      keyindex = ((1 << 4) | group_key_ver);
    }

  value16 = (value16 | keyindex);

  memcpy(outkeydata->key_information, &value16, 2);

  // Build the response.
  len = sizeof(struct wpa_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  eapolver = network_data->force_eapol_ver;

  // If we have not hard set the eapol version, then use the last
  // version that the authenticator gave us.  This isn't in line with
  // the 802.1X-REV-2004-d11 standard, but many authenticators are checking
  // version #s, and breaking when it isn't 1. :-/
  if ((eapolver < 1) || (eapolver > MAX_EAPOL_VER))
    eapolver = snmp_get_dot1xSuppLastEapolFrameVersion();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, intdata->sendframe); 
  
  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate(intdata->sendframe, intdata->send_size+4, key, 16);

  cardif_sendframe(intdata);
  intdata->statemachine->eapolEap = FALSE;
}

/****************************************************************
 *
 * When we have completed the PTK piece, and the pairwise key has been
 * applied to the interface, we need to get the group key.  The authenticator
 * will send us a group key that is encrypted.  We should decrypt it, apply
 * it to our interface, and send the authenticator a message to let it know
 * that we have a group key.
 *
 ****************************************************************/
void eapol_key_type254_do_gtk(struct interface_data *intdata)
{
  struct wpa_key_packet *inkeydata, *outkeydata;
  uint16_t value16, keyflags, version, keyindex, len;
  int eapolver;
  unsigned char *keydata;
  char key[16], rc4_ek[32];
  u_char allfs[6] = {0xff,0xff,0xff,0xff,0xff,0xff};
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network data struct! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  inkeydata = (struct wpa_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa_key_packet));

  outkeydata->key_descriptor = WPA_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);

  keyflags = 0;
  keyflags = (value16 & WPA_KEYTYPE_MASK);
  version = keyflags;
  group_key_ver = version;
  keyindex = ((value16 & WPA_KEY_INDEX) >> 4);

  // Verify that our flags are correct.  (We don't check for the install flag,
  // for now.  Since the WPA spec doesn't explicitly require that the install
  // flag be set.)
  if (!((value16 & WPA_KEY_ACK_FLAG) &&
	(value16 & WPA_KEY_MIC_FLAG) && (value16 & WPA_SECURE_FLAG)))
    {
      debug_printf(DEBUG_NORMAL, "Invalid flags in GTK message 1!\n");
      return;
    }

  value16 = ntohs(inkeydata->key_material_len);
  
  keydata = (unsigned char *)malloc(value16);
  if (keydata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for key data!\n");
      return;
    }

  memcpy(keydata, 
	 &intdata->recvframe[OFFSET_TO_EAPOL+4+sizeof(struct wpa_key_packet)],
	 value16);

  debug_printf(DEBUG_INT, "Setting GTK! (Version : %d  Length : %d)\n", 
	       version, value16);

  switch (version)
    {
    case 1:
      // Decrypt the GTK.
      bzero(rc4_ek, 32);
      memcpy(rc4_ek, inkeydata->key_iv, 16);
      memcpy(&rc4_ek[16], &intdata->statemachine->PTK[16], 16);
      rc4_skip(rc4_ek, 32, 256, keydata, value16);
      
      wpa_common_swap_rx_tx_mic(keydata);
      cardif_set_tkip_key(intdata, allfs, keyindex, FALSE, inkeydata->key_rsc,
			  6, keydata, value16);
      break;

    case 2:
      // First, decrypt the GTK
      bzero(key, 16);
      aes_unwrap(&intdata->statemachine->PTK[16], (value16-8)/8, keydata,
		 key);

      wpa_common_swap_rx_tx_mic(keydata);
      cardif_set_ccmp_key(intdata, NULL, keyindex, FALSE, inkeydata->key_rsc,
     			  6, key, (value16-8));
      break;
    }

  // Build the response.
  len = sizeof(struct wpa_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  value16 = ((version | (keyindex << 4)) | WPA_KEY_MIC_FLAG | WPA_SECURE_FLAG);
  value16 = htons(value16);
  memcpy(&outkeydata->key_information, &value16, 2);

  outkeydata->key_length = inkeydata->key_length;
  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);

  eapolver = network_data->force_eapol_ver;

  // If we have not hard set the eapol version, then use the last
  // version that the authenticator gave us.  This isn't in line with
  // the 802.1X-REV-2004-d11 standard, but many authenticators are checking
  // version #s, and breaking when it isn't 1. :-/
  if ((eapolver < 1) || (eapolver > MAX_EAPOL_VER))
    eapolver = snmp_get_dot1xSuppLastEapolFrameVersion();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, intdata->sendframe); 
  
  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate(intdata->sendframe, intdata->send_size+4, key, 16);

  // Dump what we built.
  eapol_key_type254_dump(intdata->sendframe);

  if (network_data->methods->method_num == WPA_PSK)
    {
      // If we are using PSK, and we made it here, then we are in 
      // AUTHENTICATED state.
      intdata->statemachine->eapSuccess = TRUE;
    }

  // Drop unencrypted frames.
  cardif_drop_unencrypted(intdata, 1);
}

/***************************************************************
 *
 * Handle the first packet in the four-way handshake.
 *
 ***************************************************************/
void eapol_key_type254_do_type1(struct interface_data *intdata)
{
  struct wpa_key_packet *inkeydata, *outkeydata;
  uint16_t keyflags, len, value16;
  int i, version, eapolver, ielen;
  char key[16];
  char wpa_ie[26];
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration struct! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  inkeydata = (struct wpa_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa_key_packet));

  // XXX Need to do this better.  Tie it in with Nonce code from SIM/AKA.
  for (i=0;i<32;i++)
    {
      outkeydata->key_nonce[i] = rand();
    }
  intdata->statemachine->SNonce = (char *)malloc(32);
  if (intdata->statemachine->SNonce == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for SNonce in "
		   "%s at %d!\n", __FUNCTION__, __LINE__);
      return;
    }
  memcpy(intdata->statemachine->SNonce, (char *)&outkeydata->key_nonce[0],
	 32);

  // Calculate the PTK.
  if (intdata->statemachine->PTK != NULL)
    {
      free(intdata->statemachine->PTK);
      intdata->statemachine->PTK = NULL;
    }
  intdata->statemachine->PTK = eapol_key_type254_gen_ptk(intdata,
							 (char *)&inkeydata->key_nonce);

  outkeydata->key_descriptor = WPA_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);

  keyflags = 0;
  keyflags = (value16 & WPA_KEYTYPE_MASK);
  version = keyflags;
  pairwise_key_ver = version;
  keyflags |= (WPA_PAIRWISE_KEY | WPA_KEY_MIC_FLAG);
  keyflags = htons(keyflags);

  memcpy(&outkeydata->key_information, &keyflags, 2);
  
  len = sizeof(struct wpa_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  outkeydata->key_length = inkeydata->key_length;

  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);

  cardif_get_wpa_ie(intdata, wpa_ie, &ielen);

  memcpy(&intdata->sendframe[OFFSET_TO_EAPOL+4+sizeof(struct wpa_key_packet)], 
	 &wpa_ie, ielen);
  value16 = ielen;
  value16 = htons(value16);
  intdata->send_size += ielen;

  outkeydata->key_material_len = value16;

  eapolver = network_data->force_eapol_ver;

  // If we have not hard set the eapol version, then use the last
  // version that the authenticator gave us.  This isn't in line with
  // the 802.1X-REV-2004-d11 standard, but many authenticators are checking
  // version #s, and breaking when it isn't 1. :-/
  if ((eapolver < 1) || (eapolver > MAX_EAPOL_VER))
    eapolver = snmp_get_dot1xSuppLastEapolFrameVersion();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, intdata->sendframe);

  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate(intdata->sendframe, intdata->send_size+4, key, 16);

  // Dump what we built.
  eapol_key_type254_dump(intdata->sendframe);
}

/********************************************************
 *
 * Handle the third packet in the 4 way handshake.  We should be able to
 * generate the pairwise key at this point.
 *
 ********************************************************/
void eapol_key_type254_do_type3(struct interface_data *intdata)
{
  struct wpa_key_packet *inkeydata, *outkeydata;
  uint16_t keyflags, len, value16, keyindex;
  int eapolver, version;
  char key[32];
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid configuration data! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  bzero(key, 32);

  inkeydata = (struct wpa_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL], 
	sizeof(struct wpa_key_packet)+28);

  outkeydata->key_descriptor = WPA_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);
  
  keyflags = 0;
  keyflags = (value16 & WPA_KEYTYPE_MASK);
  version = keyflags;  

  keyflags = 0;
  keyflags = (value16 & WPA_KEYTYPE_MASK);
  keyflags |= (WPA_PAIRWISE_KEY | WPA_KEY_MIC_FLAG);
  keyindex = ((value16 & WPA_KEY_INDEX) >> 4);

  // If the authenticator sets the secure flag, we need to do the same.
  if (value16 & WPA_SECURE_FLAG) keyflags |= WPA_SECURE_FLAG;
  keyflags = htons(keyflags);

  memcpy(&outkeydata->key_information, &keyflags, 2);
  
  len = sizeof(struct wpa_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  outkeydata->key_length = inkeydata->key_length;

  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);
  memcpy(&outkeydata->key_nonce, intdata->statemachine->SNonce, 32);

  memcpy(outkeydata->key_nonce, intdata->statemachine->SNonce,
	 32);

  eapolver = network_data->force_eapol_ver;
  
  // If we have not hard set the eapol version, then use the last
  // version that the authenticator gave us.  This isn't in line with
  // the 802.1X-REV-2004-d11 standard, but many authenticators are checking
  // version #s, and breaking when it isn't 1. :-/
  if ((eapolver < 1) || (eapolver > MAX_EAPOL_VER))
    eapolver = snmp_get_dot1xSuppLastEapolFrameVersion();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, intdata->sendframe); 

  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate(intdata->sendframe, intdata->send_size+4, key, 16);  

  // Dump what we built.
  eapol_key_type254_dump(intdata->sendframe);

  // Get TK1
  value16 = ntohs(inkeydata->key_length);
  memcpy(key, (char *)&intdata->statemachine->PTK[32], value16);
  
  debug_printf(DEBUG_INT, "TK1 : ");
  debug_hex_printf(DEBUG_INT, key, value16);
  
  cardif_sendframe(intdata);
  intdata->statemachine->eapolEap = FALSE;
  intdata->send_size = 0;

  debug_printf(DEBUG_INT, "Setting PTK1!\n");

  switch (version) 
    {
    case 1:
      // We need to swap the TX/RX MIC values since we are the supplicant.  
      // (Thanks to Jouni Malinen for pointing this out!)
      wpa_common_swap_rx_tx_mic(key);
      cardif_set_tkip_key(intdata, intdata->dest_mac, keyindex, TRUE, 
			  inkeydata->key_rsc, 6, key, value16);
      break;

    case 2:
      cardif_set_ccmp_key(intdata, intdata->dest_mac, keyindex, TRUE, 
			  inkeydata->key_rsc, 6, key, value16);
      break;
    }
}

void eapol_key_type254_determine_key(struct interface_data *intdata)
{
  struct wpa_key_packet *keydata;
  int keyflags, eapolver;
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid configuration data! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  keydata = (struct wpa_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];  
  memcpy(&keyflags, keydata->key_information, 2);

  keyflags = ntohs(keyflags);

  if (keyflags & WPA_KEY_MIC_FLAG)
    {
      if (mic_wpa_validate(intdata->recvframe, intdata->recv_size, 
			   intdata->statemachine->PTK, 16) == FALSE)
	{
	  intdata->statemachine->MICVerified = TRUE;
	  intdata->statemachine->IntegrityFailed = TRUE;
	  debug_printf(DEBUG_INT, "MIC failure!\n");
       	  return;   // Silently discard.
	} else {
	  intdata->statemachine->IntegrityFailed = FALSE;
	  intdata->statemachine->MICVerified = TRUE;
	}
    }
  if ((keyflags & WPA_PAIRWISE_KEY) && (keyflags & WPA_KEY_ACK_FLAG) &&
      (keyflags & WPA_KEY_MIC_FLAG) && (keyflags & WPA_INSTALL_FLAG))
    {
      debug_printf(DEBUG_INT, "Key Packet #3 (response) :\n");
      eapol_key_type254_do_type3(intdata);
    } else if ((keyflags & WPA_PAIRWISE_KEY) && (keyflags & WPA_KEY_ACK_FLAG))
    {
      debug_printf(DEBUG_INT, "Key Packet #1 (response) :\n");
      eapol_key_type254_do_type1(intdata);
    } else if ((keyflags & WPA_KEY_MIC_FLAG) && (keyflags & WPA_PAIRWISE_KEY))
    {
      debug_printf(DEBUG_NORMAL, "Got Key Packet #2 or #4!  (This shouldn't happen!)\n");
    } else if (!(keyflags & WPA_PAIRWISE_KEY))
      {
	// We have a group key packet.
	eapol_key_type254_do_gtk(intdata);
      }

  if (intdata->recv_size > 0)
    {
      eapolver = network_data->force_eapol_ver;

      // If we have not hard set the eapol version, then use the last
      // version that the authenticator gave us.  This isn't in line with
      // the 802.1X-REV-2004-d11 standard, but many authenticators are checking
      // version #s, and breaking when it isn't 1. :-/
      if ((eapolver < 1) || (eapolver > MAX_EAPOL_VER))
	eapolver = snmp_get_dot1xSuppLastEapolFrameVersion();

      eapol_build_header(EAPOL_KEY, (intdata->recv_size-OFFSET_TO_EAPOL-4), 
			 eapolver, intdata->recvframe);
      cardif_sendframe(intdata);
      intdata->statemachine->eapolEap = FALSE;
    }      
}

/*************************************************************
 *
 * Process a WPA frame that we get from the authenticator.
 *
 *************************************************************/
void eapol_key_type254_process(struct interface_data *intdata)
{
  struct config_wpa_psk *psk;
  u_char *inframe;
  int insize;
  char tpmk[32];
  struct config_network *network_data;

  if (!intdata)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data passed to %s:%d!\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network data structure! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  inframe = intdata->recvframe;
  insize = intdata->recv_size;

  debug_printf(DEBUG_INT, "Processing WPA key!\n");

  if (intdata->statemachine->PMK == NULL)
    {
      if (network_data->methods->method_num == WPA_PSK)
	{
	  psk = (struct config_wpa_psk *)network_data->methods->method_data;
	  if (psk->key)
	    {
	      debug_printf(DEBUG_NORMAL, "Key : %s   Cur ESSID : %s\n", 
			   psk->key, intdata->cur_essid);

	      if (intdata->cur_essid == NULL)
		{
		  debug_printf(DEBUG_NORMAL, "Unknown SSID, checking!\n");
		  intdata->cur_essid = (char *)malloc(99);
		  if (intdata->cur_essid == NULL)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory! "
				   "(%s:%d)\n", __FUNCTION__, __LINE__);
		      return;
		    }
		  if (cardif_GetSSID(intdata, intdata->cur_essid) != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't get ESSID!\n");
		      free(intdata->cur_essid);
		      return;
		    }
		}

	      // We have an ASCII password, so calculate it.
	      if (psk_wpa_pbkdf2(psk->key, intdata->cur_essid, 
				 strlen(intdata->cur_essid), (char *)&tpmk) 
		  == TRUE)
		{
		  intdata->statemachine->PMK = (char *)malloc(32);
		  if (intdata->statemachine->PMK == NULL)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for"
				   " intdata->statemachine->PMK in %s:%d!\n",
				   __FUNCTION__, __LINE__);
		      return;
		    }
		  
		  memcpy(intdata->statemachine->PMK, (char *)&tpmk, 32);
		}
	    } else {
	      // We have a hex key, we need to convert it from ASCII to real
	      // hex.
	      if (strlen(psk->hex_key) != 64)
		{
		  debug_printf(DEBUG_NORMAL, "Invalid HEX key defined for "
			       "WPA-PSK!\n");
		  return;
		}
	      process_hex(psk->hex_key, strlen(psk->hex_key), (char *)&tpmk);
	      intdata->statemachine->PMK = (char *)malloc(32);
	      if (intdata->statemachine->PMK == NULL)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for "
			       "intdata->statemachine->PMK in %s:%d!\n",
			       __FUNCTION__, __LINE__);
		  return;
		}
	    }
	} else {
	  debug_printf(DEBUG_NORMAL, "There is no PMK available!  WPA cannot"
		       " continue!\n");
	  return;
	}
    } 

  eapol_key_type254_dump(inframe);

  eapol_key_type254_determine_key(intdata);

  if (intdata->send_size > 0)
    {
      cardif_sendframe(intdata);
      intdata->statemachine->eapolEap = FALSE;
    }
}
