/*******************************************************************
 * TLS En/Decrypt Function implementations
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: tls_crypt.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: tls_crypt.c,v 1.14 2005/08/09 01:39:18 chessing Exp $
 * $Date: 2005/08/09 01:39:18 $
 * $Log: tls_crypt.c,v $
 * Revision 1.14  2005/08/09 01:39:18  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <string.h>
#include <strings.h>
#include <openssl/ssl.h>
#include <stdint.h>
#include <netinet/in.h>

#include "config.h"
#include "profile.h"
#include "eap.h"
#include "eaptls.h"
#include "tls_funcs.h"
#include "../../xsup_debug.h"
#include "../../xsup_err.h"

u_char *tls_crypt_gen_keyblock(struct generic_eap_data *thisint, char *sesskey,
			      int sesskeylen)
{
  u_char seed[SSL3_RANDOM_SIZE*2];
  u_char *p = seed;
  struct tls_vars *mytls_vars;
  u_char *retblock;

  debug_printf(DEBUG_EVERYTHING, "Generating key block!\n");

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to tls_crypt_gen_keyblock()!\n");
      return NULL;
    }

  if (!sesskey)
    {
      debug_printf(DEBUG_NORMAL, "Invalid session constant!\n");
      return NULL;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;

  if (!mytls_vars->ssl)
    {
      debug_printf(DEBUG_NORMAL, "No valid SSL context found!\n");
      return NULL;
    }

  debug_printf(DEBUG_EVERYTHING, "Using session key const of : %s\n",
	       sesskey);

  retblock = (u_char *)malloc(TLS_SESSION_KEY_SIZE);
  if (!retblock)
    return NULL;

  memcpy(p, mytls_vars->ssl->s3->client_random, SSL3_RANDOM_SIZE);
  p+= SSL3_RANDOM_SIZE;
  memcpy(p, mytls_vars->ssl->s3->server_random, SSL3_RANDOM_SIZE);
  tls_funcs_PRF(SSL_get_session(mytls_vars->ssl)->master_key, 
		SSL_get_session(mytls_vars->ssl)->master_key_length,
		sesskey, sesskeylen, seed, 
		SSL3_RANDOM_SIZE * 2, retblock, 
		TLS_SESSION_KEY_SIZE);

  return retblock;
}

// This function written by Danielle Brevi
int tls_crypt_decrypt(struct generic_eap_data *thisint, u_char *in_data, int in_size, u_char *out_data, int *out_size)
{
  struct tls_vars *mytls_vars;
  int rc=0;
  u_char p[1000];

  if ((!thisint) || (!thisint->eap_data) || (!in_data) || (!out_data) ||
      (!out_size))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to tls_crypt_decrypt()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  bzero(p,1000);

  if (BIO_reset(mytls_vars->ssl_in) <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt.c, BIO_reset(mytls_vars->ssl_in) failed.\n");
      tls_funcs_process_error();
      return XETLSCRYPTFAIL;
    }

  rc=BIO_write(mytls_vars->ssl_in, in_data, in_size);

  if (BIO_reset(mytls_vars->ssl_out) <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt.c, BIO_reset(mytls_vars->ssl_out) failed.\n");
      tls_funcs_process_error();
      return XETLSCRYPTFAIL;
    }

  rc=SSL_read(mytls_vars->ssl, out_data, 1000);
  if (rc <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt.c, SSL_read(mytls_vars->ssl, out_data, 1000) failed.\n");
      tls_funcs_process_error();
      return XETLSCRYPTFAIL;
    }

  *out_size = rc;

  return XENONE;
}


int tls_crypt_encrypt(struct generic_eap_data *thisint, u_char *in_data, int in_size, u_char *out_data, int *out_size)
{
  struct tls_vars *mytls_vars;
  int rc=0;
  u_char *p;
  int to_send_size = 0;
  uint64_t length;

  if ((!thisint) || (!thisint->eap_data) || (!in_data) || (!out_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to tls_crypt_encrypt()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;

  // We need to modify this, to read more when there is more to be returned.
  p = (u_char *)malloc(1000);
  if (p == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error with malloc of \"p\" in tls_crypt_encrypt().\n");
      return -1;
    }

  bzero(p,1000);
  
  if (BIO_reset(mytls_vars->ssl_in) <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt.c, BIO_reset(mytls_vars->ssl_in) failed.\n");
      tls_funcs_process_error();
      return -1;
    }

  if (BIO_reset(mytls_vars->ssl_out) <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt.c, BIO_reset(mytls_vars->ssl_out) failed.\n");
      tls_funcs_process_error();
      return -1;
    }

  rc=SSL_write(mytls_vars->ssl, in_data, in_size);
  if (rc <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt.c, SSL_write in encrypt failed!\n");
      tls_funcs_process_error();
      return -1;
    }

  rc = BIO_read(mytls_vars->ssl_out, p, 1000);   // Allow largest possible read.
  if (rc <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt.c, BIO_read in encrypt failed!\n");
      tls_funcs_process_error();
      return -1;
    }

  to_send_size = rc;

  out_data[0] = EAPTLS_LENGTH_INCL;  // No more to send.
  length = ntohl(to_send_size+5);
  memcpy(&out_data[1], &length, 4);
  memcpy(&out_data[5], p, to_send_size);

  *out_size = to_send_size+5;
  if(p)
    {
      free(p);
      p = NULL;
    }
  return XENONE;
}

int tls_crypt_encrypt_nolen(struct generic_eap_data *thisint, u_char *in_data, int in_size, u_char *out_data, int *out_size)
{
  struct tls_vars *mytls_vars;
  int rc=0;
  u_char *p;
  int to_send_size = 0;

  if ((!thisint) || (!thisint->eap_data) || (!in_data) || (!out_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to tls_crypt_encrypt()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;

  // We need to modify this, to read more when there is more to be returned.
  p = (u_char *)malloc(1000);
  if (p == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error with malloc of \"p\" in tls_crypt_encrypt().\n");
      return -1;
    }

  bzero(p,1000);
  
  if (BIO_reset(mytls_vars->ssl_in) <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt (nolen), BIO_reset failed in encrypt!\n");
      tls_funcs_process_error();
      return -1;
    }

  if (BIO_reset(mytls_vars->ssl_out) <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt (nolen), BIO_reset (2) failed in encrypt!\n");
      tls_funcs_process_error();
      return -1;
    }

  rc=SSL_write(mytls_vars->ssl, in_data, in_size);
  if (rc <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt (nolen), SSL_write failed in encrypt!\n");
      tls_funcs_process_error();
    }

  rc = BIO_read(mytls_vars->ssl_out, p, 1000);   // Allow largest possible read.
  if (rc <= 0)
    {
      debug_printf(DEBUG_NORMAL, "In tls_crypt (nolen), BIO_read failed in encrypt!\n");
      tls_funcs_process_error();
      return -1;
    }

  to_send_size = rc;

  out_data[0] = 0x00;  // No more to send.
  memcpy(&out_data[1], p, to_send_size);

  *out_size = to_send_size+1;
  if(p)
    {
      free(p);
      p = NULL;
    }
  return XENONE;
}
