/*******************************************************************
 * Implementation for handling SSID related config pieces.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: config_ssid.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: config_ssid.c,v 1.11 2005/08/09 01:39:13 chessing Exp $
 * $Date: 2005/08/09 01:39:13 $
 * $Log: config_ssid.c,v $
 * Revision 1.11  2005/08/09 01:39:13  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 * Revision 1.10  2005/05/01 22:09:07  chessing
 * Small updates that are needed prior to the 1.2-pre1 release.
 *
 *******************************************************************/

#include <string.h>
#include <stdlib.h>

#include "xsup_debug.h"
#include "config.h"
#include "config_ssid.h"

struct found_ssids *ssids=NULL;

/**************************************
 *
 *  Dump all of the information about the SSID we are looking at.
 *
 **************************************/
void config_ssid_dump()
{
  if (ssids == NULL)
    {
      debug_printf(DEBUG_EVERYTHING, "NO VALID SSID INFORMATION AVAILABLE TO "
		   "DUMP!!\n");
      return;
    }

  if (ssids->ssid_name != NULL)
    {
      debug_printf(DEBUG_EVERYTHING, "ESSID : %s\n", ssids->ssid_name);
      debug_printf(DEBUG_EVERYTHING, "Abilities : %02X\n", ssids->abilities);
      
      if ((ssids->wpa_ie != NULL) && ((ssids->wpa_ie_len > 0) && (ssids->wpa_ie_len < 255)))
	{
	  debug_printf(DEBUG_EVERYTHING, "WPA IE (%d) : ", ssids->wpa_ie_len);
	  debug_hex_printf(DEBUG_EVERYTHING, ssids->wpa_ie, ssids->wpa_ie_len);
	  debug_printf(DEBUG_EVERYTHING, "\n");
	}

      if ((ssids->rsn_ie != NULL) && ((ssids->rsn_ie_len > 0) && (ssids->rsn_ie_len < 255)))
	{
	  debug_printf(DEBUG_EVERYTHING, "RSN IE (%d) : ", ssids->rsn_ie_len);
	  debug_hex_printf(DEBUG_EVERYTHING, ssids->rsn_ie, ssids->rsn_ie_len);
	  debug_printf(DEBUG_EVERYTHING, "\n");
	}
    }
}

/**************************************
 *
 * Clear out the list of SSIDs that we know about.  This will usually be 
 * called at the time that a scan is started.  To allow for new data to
 * be compiled.
 *
 **************************************/
void config_ssid_clear()
{
  struct found_ssids *cur, *next;

  cur = ssids;

  while (cur != NULL)
    {
      if (cur->ssid_name != NULL)
	{
	  free(cur->ssid_name);
	  cur->ssid_name = NULL;
	}

      next = cur->next;
      free(cur);
      cur = next;
    }

  ssids = NULL;
}

/******************************************
 *
 * Add an SSID, and it's abilities to a list of SSIDs we know about in the
 * area.  
 *
 ******************************************/
void config_ssids_add(char *ssid_name, u_char ability, u_char *wpa_ie, 
		      u_char wpa_ie_len, u_char *rsn_ie, u_char rsn_ie_len,
		      unsigned int freq, u_char *mac)
{
  struct found_ssids *cur;

  if (strlen(ssid_name) == 0)
    {
      // The SSID name we got isn't long enough.  Ignore it..
      debug_printf(DEBUG_NORMAL, "Invalid ESSID!\n");
      return;
    }

  cur = ssids;

  if (cur == NULL)
    {
      ssids = (struct found_ssids *)malloc(sizeof(struct found_ssids));
      if (ssids == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory! (%s:%d)\n",
		       __FUNCTION__, __LINE__);
	  return;
	}
      cur = ssids;

    } else {
      while (cur->next != NULL) cur = cur->next;

      cur->next = (struct found_ssids *)malloc(sizeof(struct found_ssids));
      if (cur == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory! (%s:%d)\n",
		       __FUNCTION__, __LINE__);
	  return;
	}

      cur = cur->next;
    }

  cur->ssid_name = (char *)malloc(strlen(ssid_name)+1);
  if (cur->ssid_name != NULL)
    {
      bzero(cur->ssid_name, strlen(ssid_name)+1);
      strcpy(cur->ssid_name, ssid_name);
    }

  cur->abilities = ability;
  
  cur->freq = freq;
  memcpy(cur->mac, mac, 6);

  if (wpa_ie != NULL)
    {
      cur->wpa_ie = (char *)malloc(wpa_ie_len);
      if (cur->wpa_ie == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for wpa_ie! "
		       "(%s:%d)\n", __FUNCTION__, __LINE__);
	  return;
	}
      bzero(cur->wpa_ie, wpa_ie_len);
      memcpy(cur->wpa_ie, wpa_ie, wpa_ie_len);
      cur->wpa_ie_len = wpa_ie_len;
    } else {
      wpa_ie = NULL;
      wpa_ie_len = 0;
    }

  if (rsn_ie != NULL)
    {
      cur->rsn_ie = (char *)malloc(rsn_ie_len);
      if (cur->rsn_ie == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for rsn_ie! "
		       "(%s:%d)\n", __FUNCTION__, __LINE__);
	  return;
	}
      bzero(cur->rsn_ie, wpa_ie_len);
      memcpy(cur->rsn_ie, rsn_ie, rsn_ie_len);
      cur->rsn_ie_len = rsn_ie_len;
    } else {
      rsn_ie = NULL;
      rsn_ie_len = 0;
    }

  cur->next = NULL;
}

/******************************************
 *
 * Return the SSID name for the SSID we want to connect to.
 *
 ******************************************/
char *config_ssid_get_desired_ssid()
{
  if (ssids == NULL) return NULL;

  return ssids->ssid_name;
}

/******************************************
 *
 * Return an unsigned char with bitflags that indicate what capabilities this
 * SSID has. (Supporting of WEP, WPA, 802.11I, etc.)
 *
 ******************************************/
u_char config_ssid_get_ssid_abilities()
{
  if (ssids == NULL) return 0x00;

  return ssids->abilities;
}

/******************************************
 *
 * Return the wpa_ie and the wpa_ie_len for the selected SSID.
 *
 ******************************************/
void config_ssid_get_wpa_ie(u_char *wpa_ie, u_char *wpa_ie_len)
{
  if (ssids == NULL) 
    {
      wpa_ie = NULL;
      wpa_ie_len = 0;
      return;
    }

  wpa_ie = ssids->wpa_ie;
  *wpa_ie_len = ssids->wpa_ie_len;
}

/*******************************************************
 *
 * Return the frequency that this AP is on.
 *
 *******************************************************/
unsigned int config_ssid_get_freq()
{
  return ssids->freq;
}

/*******************************************************
 *
 * Return the MAC address of the AP for this SSID.
 *
 *******************************************************/
u_char *config_ssid_get_mac()
{
  if (ssids == NULL)
    {
      // This may not actually be a problem.  Rather, it could be that the
      // user wanted to use an SSID that we didn't find in a scan.
      debug_printf(DEBUG_INT, "Invalid SSID struct!  (%s:%d)\n", 
		   __FUNCTION__, __LINE__);
      return NULL;
    }

  return (u_char *)&ssids->mac;
}
