/*-
# MOTIF/X-BASED PANEX(tm)
#
#  xpanex.c
#
###
#
#  Copyright (c) 1996 - 2007	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

#include "file.h"
#ifdef WINVER
#include "PanexP.h"
#define TITLE "wpanex"

static PanexRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->panex.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#ifdef MOUSEBITMAPS
#include "pixmaps/mouse-l.xbm"
#include "pixmaps/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Panex.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "pixmaps/panex.t.xpm"
#include "pixmaps/panex.p.xpm"
#include "pixmaps/panex.s.xpm"
#include "pixmaps/panex.m.xpm"
#include "pixmaps/panex.l.xpm"
#include "pixmaps/panex.xpm"
#define RESIZE_XPM(s) ((char **) ((s)<=32)?\
(((s)<=22)?(((s)<=16)?panex_t_xpm:panex_p_xpm):\
(((s)<=24)?panex_s_xpm:panex_m_xpm)):\
(((s)<=48)?panex_l_xpm:panex_xpm))
#endif
#include "pixmaps/panex.xbm"
#define DEFINE_XBM (char *) panex_bits, panex_width, panex_height
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef WINVER
static const char aboutHelp[] = {
"Panex Version 7.3.2\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html\n"
"I got some help from Rene Jansen <rene.j.jansen@bigfoot.com> "
"and Nick Baxter <nickb@baxterweb.com>.\n"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-tile {color}] [-pyramid{0|1} {color}]\n"
"[-{border|bd} {color}] [-delay msecs] [-[no]sound]\n"
"[-moveSound {filename}] [-{font|fn} {fontname}]\n"
"[-tiles {int}] [-{mode {int}|hanoi|panex}]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]\n"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"The original Panex puzzle is from the Japanese Magic Company "
"estimated from the 1980's.  Mathematicians at Bell\n"
"Laboratories estimated the number of moves to swap 2 columns "
"of order 10 to be  27,564 <= N <= 31,537.  It came\n"
"in two varieties: one with a magenta and a yellow pyramid "
"of order 10 on silver tiles; in the gold version, the\n"
"gold version pieces of each color look alike (i.e. no "
"pyramid is drawn on them), this is a little harder.\n"
"The original Tower of Hanoi puzzle is the invention of "
"Edouard Lucas and temple of Benares of the god Brahma is\n"
"also his invention.\n"
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a tile in the top tile "
"of a column.  Release \"mouse-left\" button on\n"
"another column to move the tile to that column.  It will "
"not move if blocked.\n"
"\n"
"Click \"mouse-right\" button, or press \"C\" or \"c\" keys, "
"to clear the puzzle.\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"S\" or \"s\" keys to auto-solve.  Unfortunately, its "
"only implemented from the starting position.\n"
"\n"
"Press \"M\" or \"m\" keys to toggle between hanoi "
"(one pyramid column) and panex (two pyramid columns) modes\n"
"(they each have different rules).\n"
"In Hanoi, one can not place  larger trapezoid on a smaller "
"trapezoid.  Here the goal is to move the pile from\n"
"the left peg to the rightmost peg.\n"
"In Panex, a tile can not go lower that its initial starting "
"point.  Here the goal is to exchange the 2 piles.\n"
"\n"
"Press \"I\" or \"i\" keys to increase the number of tiles.\n"
"\n"
"Press \"D\" or \"d\" keys to decrease the number of tiles.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Unlike other puzzles in the collection there is no way to "
"move pieces without drag and drop.\n"
};

static const char referencesHelp[] = {
"Mark Manasse & Danny Sleator of AT&T Bell Laboratories  "
"and Victor K. Wei of Bell Communications Research,\n"
"Some Results on the Panex Puzzle, Murray Hill, NJ, 1985 "
"20 pp. (unpublished).\n"
"Vladimir Dubrovsky,  Nesting Puzzles  Part 1: Moving oriental "
" Quantum  January/February  1996  pp 55-57,\n"
"50-51.\n"
"L.  E.  Horden,  Sliding  Piece  Puzzles  (Recreations  in  "
"Mathematics  Series),  Oxford University Press  1986,\n"
"pp 144, 145.\n"
"Jerry Slocum & Jack Botermans,  Puzzles Old & New (How to Make "
"and  Solve Them),  University of Washington\n"
"Press, Seattle 1987, p 135.\n"
};
#endif

#ifndef SCOREFILE
#define SCOREFILE "panex.scores"
#endif

#define MAXTILES 10
#define NEVER (-1)
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[MAXMODES][MAXTILES - MINTILES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
static Widget moves, record, message, modes[MAXMODES], tile, speed;
static char buff[21];
static const char *modeString[] =
{
	"Hanoi", "Panex"
};
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget clearDialog;
static Arg arg[3];
#else
static char titleDsp[TITLELEN] = "";
#endif
static char scoreFileName[FILENAMELEN] = SCOREFILE;
static char fileName[FILENAMELEN];
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char userNameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap pixmap = None;
static Widget topLevel, puzzle;
static char *progDsp;
static char userNameDsp[USERNAMELEN] = "";

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
PrintRecords(void)
{
	int i, mode;

	(void) printf("   HANOI/PANEX  HALL OF FAME\n\n");
	(void) printf("PANEX TILES USER            MOVES\n");
	for (mode = 0; mode < MAXMODES; mode++)
		for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
			if (puzzleRecord[mode][i].score > 0)
				(void) printf("%5d%6d %-12s%9d\n",
					mode, i + 1,
					puzzleRecord[mode][i].name,
					puzzleRecord[mode][i].score);
		}
}
#endif

static void
InitRecords(void)
{
	int i, mode;

	for (mode = 0; mode < MAXMODES; mode++)
		for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
			puzzleRecord[mode][i].score = NEVER;
			(void) strncpy(puzzleRecord[mode][i].name, NOACCESS,
				USERNAMELEN);
		}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, mode, n;
	char userName[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USERNAMELEN);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USERNAMELEN);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (mode = 0; mode < MAXMODES; mode++)
		for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
			(void) fscanf(fp, "%d %s\n", &n, userName);
			if (n <= puzzleRecord[mode][i].score ||
					puzzleRecord[mode][i].score <= NEVER) {
				puzzleRecord[mode][i].score = n;
				(void) strncpy(puzzleRecord[mode][i].name,
					userName, USERNAMELEN);
			}
		}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, mode;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILENAMELEN];

		(void) strncpy(lockFile, fileName, FILENAMELEN - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (mode = 0; mode < MAXMODES; mode++) {
			for (i = 0; i < MAXTILES - MINTILES + 1; i++)
				(void) fprintf(fp, "%d %s\n",
					puzzleRecord[mode][i].score,
					puzzleRecord[mode][i].name);
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
PrintRecord(int tiles, int mode)
{
	int i = tiles - MINTILES, j = mode - HANOI;

	if (tiles > MAXTILES)
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	else if (puzzleRecord[j][i].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[j][i].score, puzzleRecord[j][i].name);

	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static void
PrintStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int tiles
#endif
		)
{
#ifdef HAVE_MOTIF
	PrintState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	PrintState(moves, buff);
#else
	(void) sprintf(titleDsp, "%s: %d@ (%d/%s) - %s",
		progDsp, tiles, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(puzzle), XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
HandleSolved(int counter, int tiles, int mode)
{
	int i = tiles - MINTILES, j = mode - HANOI;

	if (tiles <= MAXTILES && (counter < puzzleRecord[j][i].score ||
			puzzleRecord[j][i].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		puzzleRecord[j][i].score = counter;
		(void) strncpy(puzzleRecord[j][i].name, userNameDsp,
			USERNAMELEN);
		WriteRecords();
		PrintRecord(tiles, mode);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
PanexWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int tiles, mode;
	char *userName, *scoreFile;

#ifdef WINVER
	InitializePuzzle(w, brush);

	tiles = w->panex.tiles;
	mode = w->panex.mode;
	userName = w->panex.userName;
	scoreFile = w->panex.scoreFile;
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNtiles, &tiles,
		XtNmode, &mode,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf(aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
	if (scoreOnly) {
		InitRecords();
		ReadRecords();
		PrintRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
	{
		int delay;

		if (tiles > MAXTILES)
			XtVaSetValues(tile, XmNmaximum, tiles, NULL);
		XmScaleSetValue(tile, tiles);
		XmToggleButtonSetState(modes[mode - HANOI], True, False);
		XtVaGetValues(puzzle, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
	/* SET_STARTED(w, False); */
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USERNAMELEN);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USERNAMELEN);
#else
		char *login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	PrintRecord(tiles, mode);
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, tiles
#endif
		);
}

#ifdef WINVER
void
setPuzzle(PanexWidget w, int reason)
#else
static void
CallbackPuzzle(Widget w, caddr_t clientData, panexCallbackStruct *callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int tiles, mode;
	Boolean cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	tiles = w->panex.tiles;
	mode = w->panex.mode;
	cheat = w->panex.cheat;
#else
	XtVaGetValues(w,
		XtNtiles, &tiles,
		XtNmode, &mode,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
		break;
#ifndef WINVER
	case ACTION_CLEAR_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(clearDialog);
#else
		XtVaSetValues(puzzle, XtNmenu, ACTION_CLEAR, NULL);
#endif
		break;
#endif
	case ACTION_RESTORE:
	case ACTION_RESET:
		movesDsp = 0;
		break;
	case ACTION_ILLEGAL:
		(void) strncpy(messageDsp, "Illegal move", MESSAGELEN);
		break;
	case ACTION_BLOCKED:
		(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
		break;
	case ACTION_SPACE:
#if 0
		/* Too annoying */
		(void) strncpy(messageDsp, "Spaces can not slide",
			MESSAGELEN);
#endif
		break;
	case ACTION_IGNORE:
		(void) strncpy(messageDsp, "Clear to start",
			MESSAGELEN);
		break;
	case ACTION_MOVED:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	case ACTION_SOLVED:
		if (cheat)
			(void) sprintf(messageDsp,
				"No cheating %s!!", userNameDsp);
		else if (HandleSolved(movesDsp, tiles, mode))
			(void) sprintf(messageDsp,
				"Congratulations %s!!", userNameDsp);
		else
			(void) strncpy(messageDsp, "Solved!",
				MESSAGELEN);
		SET_STARTED(w, False);
		break;
	case ACTION_COMPUTED:
		SET_STARTED(w, False);
		break;
	case ACTION_INCREMENT:
		movesDsp = 0;
		tiles++;
		PrintRecord(tiles, mode);
#ifdef WINVER
		w->panex.tiles = tiles;
#else
		XtVaSetValues(w, XtNtiles, tiles, NULL);
#ifdef HAVE_MOTIF
		if (tiles > MAXTILES)
			XtVaSetValues(tile, XmNmaximum, tiles, NULL);
		XmScaleSetValue(tile, tiles);
#endif
#endif
		break;
	case ACTION_DECREMENT:
		movesDsp = 0;
		tiles--;
		PrintRecord(tiles, mode);
#ifdef WINVER
		w->panex.tiles = tiles;
#else
		XtVaSetValues(w, XtNtiles, tiles, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(tile, tiles);
		if (tiles >= MAXTILES)
			XtVaSetValues(tile, XmNmaximum, tiles, NULL);
#endif
#endif
		break;
	case ACTION_MODE:
		movesDsp = 0;
		mode = (mode == PANEX) ? HANOI : PANEX;
		PrintRecord(tiles, mode);
#ifdef WINVER
		w->panex.mode = mode;
#else
		XtVaSetValues(w, XtNmode, mode, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(modes[mode - HANOI], True,
			True);
#endif
#endif
		break;
	case ACTION_UNDO:
		movesDsp--;
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		SET_STARTED(w, True);
		break;
#ifdef HAVE_MOTIF
	case ACTION_SPEED:
		{
			int oldDelay, delay;

			XtVaGetValues(puzzle, XtNdelay, &delay, NULL);
			oldDelay = delay;
			if (delay > MAXSPEED - MINSPEED)
				delay = MAXSPEED - MINSPEED;
			if (delay < 0)
				delay = 0;
			if (delay != oldDelay) {
				XtVaSetValues(w,
					XtNdelay, delay, NULL);
			}
			XmScaleSetValue(speed,
				MAXSPEED + MINSPEED - delay - 1);
		}
		return;
#endif
	}
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, tiles
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
	case WM_COMMAND:
		if (LOWORD(wParam) == IDOK) {
			(void) EndDialog(hDlg, TRUE);
			return TRUE;
		}
		break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	static Boolean mousePress = False;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.panex.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			EnterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.panex.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			LeavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		Initialize(&widget, brush);
		break;
	case WM_DESTROY:
		DestroyPuzzle(brush);
		break;
	case WM_SIZE:
		ResizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		ExposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		ClearPuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		mousePress = True;
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		SelectPuzzle(&widget, LOWORD(lParam));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_MOUSEMOVE:
		if (!mousePress)
			break;
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		MotionPuzzle(&widget, LOWORD(lParam));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		mousePress = False;
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		ReleasePuzzle(&widget, LOWORD(lParam));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			GetPuzzle(&widget);
			ResizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			WritePuzzle(&widget);
			break;
		case ACTION_EXIT:
			DestroyPuzzle(brush);
			break;
		case ACTION_HIDE:
			HidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			UndoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			RedoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ClearPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			SolvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_MODE:
			(void) ModePuzzle(&widget);
			SizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_INCREMENT:
		case ACTION_DECREMENT:
			((LOWORD(wParam) == ACTION_DECREMENT) ?
				DecrementPuzzle(&widget) :
				IncrementPuzzle(&widget));
			SizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SPEED:
			SpeedPuzzle(&widget);
			break;
		case ACTION_SLOW:
			SlowPuzzle(&widget);
			break;
		case ACTION_SOUND:
			SoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) About);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char *programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*panex.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*panex.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*panex.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*panex.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*panex.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*panex.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*panex.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*panex.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*panex.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*panex.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*panex.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-pyramid0", (char *) "*panex.pyramidColor0", XrmoptionSepArg, NULL},
	{(char *) "-pyramid1", (char *) "*panex.pyramidColor1", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*panex.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*panex.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*panex.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-moveSound", (char *) "*panex.moveSound", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*panex.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*panex.font", XrmoptionSepArg, NULL},
	{(char *) "-tiles", (char *) "*panex.tiles", XrmoptionSepArg, NULL},
	{(char *) "-mode", (char *) "*panex.mode", XrmoptionSepArg, NULL},
	{(char *) "-hanoi", (char *) "*panex.mode", XrmoptionNoArg, (char *) "0"},
	{(char *) "-panex", (char *) "*panex.mode", XrmoptionNoArg, (char *) "1"},
	{(char *) "-userName", (char *) "*panex.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*panex.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*panex.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*panex.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
CallbackPuzzleClear(Widget w, XtPointer clientData, XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(puzzle, XtNmenu, ACTION_CLEAR, NULL);
	}
}

static void
TileSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int tiles = cbs->value, mode, old;

	XtVaGetValues(puzzle,
		XtNtiles, &old,
		XtNmode, &mode, NULL);
	if (old == tiles)
		return;
	XtVaSetValues(puzzle,
		XtNtiles, tiles, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, mode);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
ModeToggle(Widget w, int mode, XmToggleButtonCallbackStruct *cbs)
{
	int tiles, old;

	XtVaGetValues(puzzle,
		XtNtiles, &tiles,
		XtNmode, &old, NULL);
	if (old == mode)
		return;
	XtVaSetValues(puzzle,
		XtNmode, mode + HANOI, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, mode + HANOI);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(puzzle,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(puzzle, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_GET;

	if (val == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle, XtNmenu, val, NULL);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_UNDO;

	XtVaSetValues(puzzle, XtNmenu, val, NULL);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}

static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, radioRowCol, tilesRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, solveString;
	XmString mode, incrementString, decrementString;
	XmString speedString, slowString, soundString;
	unsigned int i;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Panex",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu",
		0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	solveString = XmStringCreateSimple((char *) "Auto-Solve");
	mode = XmStringCreateSimple((char *) "Mode (Hanoi/Panex)");
	incrementString = XmStringCreateSimple((char *) "Increment");
	decrementString = XmStringCreateSimple((char *) "Decrement");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu",
		1, playCB,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, mode, 'M', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(solveString);
	XmStringFree(mode);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	clearDialog = createQuery(topLevel,
		(char *) "Are you sure you want to start over?",
		(char *) "Clear Query",
		(XtCallbackProc) CallbackPuzzleClear);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Clear", 6, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	radioRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN,
		XmNradioBehavior, True, NULL);
	for (i = 0; i < XtNumber(modeString); i++) {
		modes[i] = XtVaCreateManagedWidget(modeString[i],
			xmToggleButtonGadgetClass, radioRowCol,
			XmNradioBehavior, True, NULL);
		XtAddCallback(modes[i], XmNvalueChangedCallback,
			(XtCallbackProc) ModeToggle, (XtPointer) i);
	}
	tilesRowCol = XtVaCreateManagedWidget("tilesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tile = XtVaCreateManagedWidget("tile",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size", 5,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tile, XmNvalueChangedCallback,
		(XtCallbackProc) TileSlider, (XtPointer) NULL);
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Panex!",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle = XtCreateManagedWidget("panex",
		panexWidgetClass, mainPanel, NULL, 0);
#else
	puzzle = XtCreateManagedWidget("panex",
		panexWidgetClass, topLevel, NULL, 0);
#endif
	XtVaGetValues(puzzle,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
	XtAddCallback(puzzle, XtNselectCallback,
		(XtCallbackProc) CallbackPuzzle, (XtPointer) NULL);
	Initialize(puzzle);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(puzzle), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle),
		XCreateFontCursor(XtDisplay(puzzle), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
