/*
* Copyright (C) 2012 Canonical Ltd
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License version 3 as
* published by the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*/

#include <vector>
#include <queue>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/bind.hpp>

#include "xpathselect.h"
#include "utils.h"

namespace xpathselect
{
    // anonymous namespace for internal-only utility class:
    namespace
    {
        // Stores a part of an XPath query.
        class XPathQueryPart
        {
        public:
            XPathQueryPart(std::string const& query_part)
            {
                std::vector<std::string> part_pieces;
                boost::algorithm::split(part_pieces,
                    query_part,
                    boost::algorithm::is_any_of("[]="),
                    boost::algorithm::token_compress_on);

                // Boost's split() implementation does not match it's documentation! According to the
                // docs, it's not supposed to add empty strings, but it does, which is a PITA. This
                // next line removes them:
                part_pieces.erase( std::remove_if( part_pieces.begin(),
                    part_pieces.end(),
                    boost::bind( &std::string::empty, _1 ) ),
                  part_pieces.end());

                if (part_pieces.size() == 1)
                {
                    node_name_ = part_pieces.at(0);
                }
                else if (part_pieces.size() == 3)
                {
                    node_name_ = part_pieces.at(0);
                    param_name_ = part_pieces.at(1);
                    param_value_ = part_pieces.at(2);
                }
                else
                {
                    // assume it's just a node name:
                    node_name_ = query_part;
                }
            }

            XPathQueryPart(XPathQueryPart const& other)
            : node_name_(other.node_name_)
            , param_name_(other.param_name_)
            , param_value_(other.param_value_)
            {}

            bool Matches(Node::Ptr const& node) const
            {
                bool matches = (node_name_ == "*" || node->GetName() == node_name_);
                if (!param_name_.empty())
                {
                    matches &= node->MatchProperty(param_name_, param_value_);
                }

                return matches;
            }
        private:
            std::string node_name_;
            std::string param_name_;
            std::string param_value_;
        };

        typedef std::vector<XPathQueryPart> QueryList;

        QueryList GetQueryPartsFromQuery(std::string const& query)
        {
            QueryList query_parts;

            // split query into parts
            std::list<std::string> query_strings;
            boost::algorithm::split(query_strings,
                query,
                boost::algorithm::is_any_of("/"),
                boost::algorithm::token_compress_on);

            // Boost's split() implementation does not match it's documentation! According to the
            // docs, it's not supposed to add empty strings, but it does, which is a PITA. This
            // next line removes them:
            query_strings.erase( std::remove_if( query_strings.begin(),
                query_strings.end(),
                boost::bind( &std::string::empty, _1 ) ),
              query_strings.end());

            for(std::string part : query_strings)
            {
                query_parts.push_back(XPathQueryPart(part));
            }

            return query_parts;
        }

        NodeList MatchAbsoluteQuery(Node::Ptr const& root, QueryList const& query_parts)
        {
            NodeList matches;

            if (query_parts.front().Matches(root))
            {
                matches.push_back(root);
            }
            return matches;
        }

        NodeList MatchRelativeQuery(Node::Ptr const& root, QueryList const& query_parts)
        {
            NodeList matches;
            // non-recursive BFS traversal to find starting points:
            std::queue<Node::Ptr> queue;
            queue.push(root);
            while (!queue.empty())
            {
                Node::Ptr node = queue.front();
                queue.pop();
                if (query_parts.front().Matches(node))
                {
                    // found one. We keep going deeper, as there may be another node beneath this one
                    // with the same node name.
                    matches.push_back(node);
                }
                // Add all children of current node to queue.
                for(Node::Ptr child : node->Children())
                {
                    queue.push(child);
                }
            }
            return matches;
        }

        NodeList ProcessStartPoints(NodeList start_points, QueryList query_parts)
        {
            query_parts.erase(query_parts.begin());
            typedef std::pair<Node::Ptr, QueryList::iterator> node_match_pair;

            std::queue<node_match_pair> traverse_queue;
            for(Node::Ptr node : start_points)
            {
                traverse_queue.push(node_match_pair(node, query_parts.begin()));
            }
            start_points.clear();

            while (!traverse_queue.empty())
            {
                node_match_pair p = traverse_queue.front();
                traverse_queue.pop();

                Node::Ptr node = p.first;
                auto query_it = p.second;

                if (query_it == query_parts.end())
                {
                    // found a match:
                    start_points.push_back(node);
                }
                else
                {
                    // push all children of current node to start of queue, advance search iterator, and loop again.
                    for (Node::Ptr child : node->Children())
                    {
                        if (query_it->Matches(child))
                        {
                            auto it_copy(query_it);
                            ++it_copy;
                            traverse_queue.push(node_match_pair(child, it_copy));
                        }
                    }
                }
            }
            return start_points;
        }
    }

    NodeList SelectNodes(Node::Ptr const& root, std::string query)
    {
        // allow users to be lazy when specifying tree root:
        if (query == "" || query == "/" || query == "//")
        {
            query = "/" + root->GetName();
        }

        bool is_absolute = utils::IsQueryAbsolute(query);
        QueryList query_parts = GetQueryPartsFromQuery(query);

        NodeList start_nodes;
        if (is_absolute)
        {
            start_nodes = MatchAbsoluteQuery(root, query_parts);
        }
        else
        {
            start_nodes = MatchRelativeQuery(root, query_parts);
        }

        return ProcessStartPoints(start_nodes, query_parts);
    }
}
