/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>

#include <xmms/configfile.h>

#include "singit_debug.h"
#include "singit/macros.h"

#include "singit/config_xmms.h"
#include "singit_main_config.h"

#include "singit_main.h"

#include "singit_tools.h"

#include "editor_status.h"

/*
#define def_html_header g_strconcat("<HTML>\\n<HEAD>\\n  <TITLE>", _("SingIt generated Lyrics"), \
		"</TITLE>\\n</HEAD>\\n", "<BODY>\\n<TABLE BORDER=\"0\" CELLSPACINNG=\"2\">\\n", NULL)
#define def_html_tail "</TABLE>\\n</BODY>\\n</HTML>\\n"
*/
#define def_lyric_line "<TR><TD>%t</TD><TD>%l</TD></TR>"

#define def_html_header g_strconcat("<HTML>\\n<HEAD>\\n  <TITLE>", _("SingIt generated Lyrics"), \
	"</TITLE>\\n<STYLE TYPE=\"text/ccs\">\\n<!--\\nfont.lyrictag {\\n  color:#CC0000;\\n}\\n//-->\\n</STYLE>\\n", \
	"</HEAD>\\n<BODY>\\n", NULL)
#define def_html_tail "\\n</BODY>\\n</HTML>\\n"

enum {
	SC_SAVE_EDITOR,
	SC_SAVE_POSITIONS,
	SC_SAVE_PLUGINS,
};

static gpointer singit_config_init_event(SingitConfigGen *scg, gpointer user_data)
{
	SingitConfigData *singit_config_data = g_malloc(sizeof(SingitConfigData));

	singit_config_data->debugEnable = FALSE;
	singit_config_data->debugLevelExcl = FALSE;
	singit_config_data->debugLevel = 9;

	singit_config_data->basePath = NULL;
	singit_config_data->constructFilename = NULL;
	singit_config_data->lyricExtension = NULL;
	singit_config_data->lyricFontName = NULL;

	singit_config_data->backgroundColor = NULL;
	singit_config_data->normalTextColor = NULL;
	singit_config_data->normalBarColor = NULL;
	singit_config_data->progressTextColor = NULL;
	singit_config_data->progressBarColor = NULL;

	singit_config_data->htmlHeader = NULL;
	singit_config_data->htmlTail = NULL;
	singit_config_data->lyricLine = NULL;

	singit_config_data->enabled_dplugins = NULL;

	singit_config_data->submittingAuthor = NULL;

	singit_config_data->framerateLimitEnabled = TRUE;
	singit_config_data->framerateLimit = 32;

	singit_config_data->httpProxyUsage = FALSE;
	singit_config_data->httpProxyServer = NULL;
	singit_config_data->httpProxyPort = 80;
	singit_config_data->httpProxyUser = NULL;
	singit_config_data->httpProxyPasswd = NULL;
	singit_config_data->httpProxySavePasswd = FALSE;

	return singit_config_data;
}

#define FREE(var) \
	if (var != NULL) { g_free(var); }

#define FREE_N(var) \
	if (var != NULL) { g_free(var); var = NULL; }

static void singit_config_free_event(SingitConfigGen *scg, SingitConfigData *cfg_data, gpointer user_data)
{
	FREE(cfg_data->basePath);
	FREE(cfg_data->constructFilename);
	FREE(cfg_data->lyricExtension);
	FREE(cfg_data->lyricFontName);

	FREE(cfg_data->backgroundColor);
	FREE(cfg_data->normalTextColor);
	FREE(cfg_data->normalBarColor);
	FREE(cfg_data->progressTextColor);
	FREE(cfg_data->progressBarColor);

	FREE(cfg_data->htmlHeader);
	FREE(cfg_data->htmlTail);
	FREE(cfg_data->lyricLine);
	FREE(cfg_data->enabled_dplugins);

	FREE(cfg_data->submittingAuthor);

	FREE(cfg_data->httpProxyServer);
	FREE(cfg_data->httpProxyUser);
	FREE(cfg_data->httpProxyPasswd);

	g_free(cfg_data);
	cfg_data = NULL;
}

#define CFG_DATA	cfg_data
#define CFG_FILE	cfg_file
#define SECTION		"SingIt"

static void singit_config_load_event(SingitConfigGen *scg, ConfigFile* cfg_file, SingitConfigData *cfg_data, gpointer user_data)
{
	gchar *tmp;

	READ_CFG_DEF(boolean, hideIfNotFound, FALSE);
	READ_CFG_DEF(boolean, toggleYZ, FALSE);
	READ_CFG_DEF(boolean, showEmpty, FALSE);
	READ_CFG_DEF(boolean, autoResize, TRUE);
	READ_CFG_DEF(boolean, guessSyncLyrics, TRUE);
	READ_CFG_DEF(boolean, showJumpingBall, TRUE);
	READ_CFG_DEF(boolean, findOptimalFont, FALSE);
	READ_CFG_DEF(boolean, useXMMSuSleep, FALSE);
	READ_CFG_DEF(boolean, showTimeIndicator, TRUE);

#ifdef CODEDEBUG
	READ_CFG_DEF(boolean, debugEnable, FALSE);
	READ_CFG_DEF(boolean, debugLevelExcl, FALSE);
	READ_CFG_DEF(int, debugLevel, 8);
#endif

	READ_CFG_DEF(string, basePath, g_strdup("~/lyrics/,~~/,~~/lyrics/"));
	READ_CFG_DEF(string, constructFilename, g_strdup("%1"));
	READ_CFG_DEF(string, lyricExtension, g_strdup("txt,tag"));
	READ_CFG_DEF(string, lyricFontName, g_strdup("-adobe-helvetica-bold-r-*-*-14-*"));
	READ_CFG_DEF(string, enabled_dplugins, g_strdup("libdisplayer_karaoke_window"));

	READ_CFG_DEF(color, backgroundColor, g_strdup("white"));
	READ_CFG_DEF(color, normalTextColor, g_strdup("black"));
	READ_CFG_DEF(color, normalBarColor, g_strdup("white"));
	READ_CFG_DEF(color, progressTextColor, g_strdup("white"));
	READ_CFG_DEF(color, progressBarColor, g_strdup("lightblue"));

	READ_CFG_DEF(int, configWinPosX, -1);
	READ_CFG_DEF(int, configWinPosY, -1);

	READ_CFG_DEF(boolean, framerateLimitEnabled, TRUE);
	READ_CFG_DEF(int, framerateLimit, 32);

	// * From lyrics exporter *
	READ_CFG_DEF(string, htmlHeader, g_strdup(def_html_header));
	tmp = tools_replace_string(cfg_data->htmlHeader, "\\n", "\n");
	g_free(cfg_data->htmlHeader);
	cfg_data->htmlHeader = tmp;

	READ_CFG_DEF(string, htmlTail, g_strdup(def_html_tail));
	tmp = tools_replace_string(cfg_data->htmlTail, "\\n", "\n");
	g_free(cfg_data->htmlTail);
	cfg_data->htmlTail = tmp;

	READ_CFG_DEF(string, lyricLine, g_strdup(def_lyric_line));
	READ_CFG_DEF(int, exportMode, 0);

	// * From lyrics editor *
	READ_CFG_DEF(int, reactionTime, 500);
	READ_CFG_DEF(string, submittingAuthor, g_strdup(""));

	READ_CFG_DEF(boolean, httpProxyUsage, FALSE);
	READ_CFG_DEF(string, httpProxyServer, g_strdup(""));
	READ_CFG_DEF(int, httpProxyPort, 80);
	READ_CFG_DEF(string, httpProxyUser, g_strdup(""));
	READ_CFG_DEF(string, httpProxyPasswd, g_strdup(""));
	READ_CFG_DEF(boolean, httpProxySavePasswd, FALSE);
}

static void singit_config_save_event(SingitConfigGen *scg,
	ConfigFile* cfg_file, SingitConfigData *cfg_data, gpointer user_data)
{
//	g_print("Save\n");

	//  * boolean values *
	WRITE_CFG(boolean, toggleYZ);
	WRITE_CFG(boolean, showEmpty);
	WRITE_CFG(boolean, autoResize);
	WRITE_CFG(boolean, hideIfNotFound);
	WRITE_CFG(boolean, showJumpingBall);
	WRITE_CFG(boolean, guessSyncLyrics);
#ifdef CODEDEBUG
	WRITE_CFG(boolean, debugEnable);
	WRITE_CFG(boolean, debugLevelExcl);
	WRITE_CFG(int, debugLevel);
#endif
	WRITE_CFG(boolean, findOptimalFont);
	WRITE_CFG(boolean, useXMMSuSleep);
	WRITE_CFG(boolean, showTimeIndicator);

	// * string values *
	WRITE_CFG(del_string, basePath);
	WRITE_CFG(del_string, constructFilename);
	WRITE_CFG(del_string, lyricExtension);
	WRITE_CFG(del_string, lyricFontName);

	WRITE_CFG(del_string, backgroundColor);
	WRITE_CFG(del_string, normalTextColor);
	WRITE_CFG(del_string, normalBarColor);
	WRITE_CFG(del_string, progressTextColor);
	WRITE_CFG(del_string, progressBarColor);

	WRITE_CFG(del_string, enabled_dplugins);

	//* integer values *
	WRITE_CFG(int, framerateLimit);
	WRITE_CFG(int, configWinPosX);
	WRITE_CFG(int, configWinPosY);

	// * Editor config *
	WRITE_CFG(int, reactionTime);
	WRITE_CFG(del_string, submittingAuthor);

	WRITE_CFG(boolean, httpProxyUsage);
	WRITE_CFG(string, httpProxyServer);
	WRITE_CFG(int, httpProxyPort);
	WRITE_CFG(string, httpProxyUser);

	if (cfg_data->httpProxySavePasswd == TRUE)
		WRITE_CFG(del_string, httpProxyPasswd);
	else
		singit_config_xmms_remove_key
			(CFG_FILE, SECTION, "httpProxyPasswd");

	WRITE_CFG(boolean, httpProxySavePasswd);
}

static void singit_config_update_event(SingitConfigGen *scg, SingitConfigData *cfg_data, gpointer user_data)
{
	if (EDT_STATUS != NULL)
		singit_config_gen_emit_update(EDT_STATUS->config);

	STATUS->config_update = TRUE;
}

static void singit_config_save_part_event(SingitConfigGen *scg, ConfigFile* cfg_file, SingitConfigData *cfg_data, gint part, gpointer user_data)
{
	switch (part) {
	case SC_SAVE_POSITIONS:
		WRITE_CFG(int, configWinPosX);
		WRITE_CFG(int, configWinPosY);
		break;
	case SC_SAVE_PLUGINS:
		WRITE_CFG(del_string, enabled_dplugins);
		break;
	default:
		break;
	}
}

#undef SECTION
#undef CFG_FILE
#undef CFG_DATA

SingitConfigGen* singit_config_new(void)
{
//	g_return_val_if_fail((STATUS == NULL) && (STATUS->config == NULL),
//		SINGIT_CONFIG_GEN(STATUS->config));

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_singit_config.c [singit_config_new]\n"));
#endif

	STATUS->config = singit_config_xmms_new();
	singit_config_gen_clear_initial(&STATUS->config);
	singit_config_gen_set_filename
		(STATUS->config, CONFIG_FILENAME);

	gtk_signal_connect(GTK_OBJECT(STATUS->config), "init_data",
		GTK_SIGNAL_FUNC(singit_config_init_event), NULL);
	gtk_signal_connect(GTK_OBJECT(STATUS->config), "free_data",
		GTK_SIGNAL_FUNC(singit_config_free_event), NULL);
	gtk_signal_connect(GTK_OBJECT(STATUS->config), "load",
		GTK_SIGNAL_FUNC(singit_config_load_event), NULL);
	gtk_signal_connect(GTK_OBJECT(STATUS->config), "save",
		GTK_SIGNAL_FUNC(singit_config_save_event), NULL);
	gtk_signal_connect(GTK_OBJECT(STATUS->config), "save_part",
		GTK_SIGNAL_FUNC(singit_config_save_part_event), NULL);
	gtk_signal_connect(GTK_OBJECT(STATUS->config), "update",
		GTK_SIGNAL_FUNC(singit_config_update_event), NULL);

	return STATUS->config;
}

ConfigFile* singit_config_open(void)
{
	ConfigFile *cfg;

	if (!singit_config_gen_attach(STATUS->config)) {
#ifdef CODEDEBUG
		DEBUG(DLV_ALL, ("Failed\n"));
#endif
		return FALSE;
	}

	cfg = xmms_cfg_open_file((gchar*) CONFIG_FILENAME);
	if (!cfg)
		cfg = xmms_cfg_new();

	return cfg;
}

gboolean singit_config_close(ConfigFile *cfg, gboolean save)
{
	SingitConfigGen *scg_o;

	if (!cfg) {
#ifdef CODEDEBUG
		DEBUG(DLV_ALL, ("Failed\n"));
#endif
		return FALSE;
	}

	if (save)
		xmms_cfg_write_file(cfg, (gchar*) CONFIG_FILENAME);
	xmms_cfg_free(cfg);

	scg_o = STATUS->config;
	singit_config_gen_detach(&scg_o);

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("Ok\n"));
#endif

	return TRUE;
}

gboolean singit_config_save_editor(void)
{
	if (STATUS->config != NULL) {
#ifdef CODEDEBUG
		DEBUG(DLV_ALL, ("dlg_singit_config.c [singit_config_save_editor]\n"));
#endif
		singit_config_gen_save_part
			(STATUS->config, SC_SAVE_EDITOR);
		return (singit_config_gen_get_error_code
			(STATUS->config) == SCGS_OK);
	}
	return FALSE;
}

gboolean singit_config_save_positions(void)
{
	SCGStatus error;

	if (STATUS->config != NULL) {
#ifdef CODEDEBUG
		DEBUG(DLV_ALL, ("dlg_singit_config.c [singit_config_save_positions] : "));
#endif
		singit_config_gen_save_part
			(SINGIT_CONFIG_GEN(STATUS->config), SC_SAVE_POSITIONS);

		error = singit_config_gen_get_error_code
			(SINGIT_CONFIG_GEN(STATUS->config));
		singit_config_gen_clear_error_code
			(SINGIT_CONFIG_GEN(STATUS->config));
		return (error == SCGS_OK);
	}
	return FALSE;
}

gboolean singit_config_save_plugins(void)
{
	SCGStatus error;

	if (STATUS->config != NULL) {
#ifdef CODEDEBUG
		DEBUG(DLV_ALL, ("dlg_singit_config.c [singit_config_save_plugins] : "));
#endif
		singit_config_gen_save_part
			(SINGIT_CONFIG_GEN(STATUS->config), SC_SAVE_PLUGINS);

		error = singit_config_gen_get_error_code
			(SINGIT_CONFIG_GEN(STATUS->config));
		singit_config_gen_clear_error_code
			(SINGIT_CONFIG_GEN(STATUS->config));
		return (error == SCGS_OK);
	}
	return FALSE;
}

gboolean singit_config_load_plugin_position(const gchar* plugin, gint *posX, gint *posY, gint *sizeX, gint *sizeY)
{
	ConfigFile *cfg;
	gboolean result = TRUE;
	gchar *load_str;

	cfg = singit_config_open();
	if (!(cfg && plugin))
		{ return FALSE; }

	if (posX && result) {
		load_str = g_strconcat("dp_", plugin, "_posX", NULL);
		if (!(singit_config_xmms_read(int, cfg, "SingIt", load_str, posX)))
			{ result = FALSE; }
		g_free(load_str);
	}

	if (posY && result) {
		load_str = g_strconcat("dp_", plugin, "_posY", NULL);
		if (!(singit_config_xmms_read(int, cfg, "SingIt", load_str, posY)))
			{ result = FALSE; }
		g_free(load_str);
	}

	if (sizeX && result) {
		load_str = g_strconcat("dp_", plugin, "_sizeX", NULL);
		if (!(singit_config_xmms_read(int, cfg, "SingIt", load_str, sizeX)))
			{ result = FALSE; }
		g_free(load_str);
	}

	if (sizeY && result) {
		load_str = g_strconcat("dp_", plugin, "_sizeY", NULL);
		if (!(singit_config_xmms_read(int, cfg, "SingIt", load_str, sizeY)))
			{ result = FALSE; }
		g_free(load_str);
	}

	singit_config_close(cfg, FALSE);
	return result;
}

gboolean singit_config_save_plugin_position(const gchar* plugin, gint posX, gint posY, gint sizeX, gint sizeY)
{
	ConfigFile *cfg;
	gchar *save_str;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_singit_config.c [singit_config_save_plugin_position] :\n"));
#endif

	if (!plugin) { return FALSE; }

	cfg = singit_config_open();
	if (!cfg) { return FALSE; }

	if (posX >= 0) {
		save_str = g_strconcat("dp_", plugin, "_posX", NULL);
		singit_config_xmms_write(int, cfg, "SingIt", save_str, posX);
		g_free(save_str);
	}

	if (posY >= 0) {
		save_str = g_strconcat("dp_", plugin, "_posY", NULL);
		singit_config_xmms_write(int, cfg, "SingIt", save_str, posY);
		g_free(save_str);
	}

	if (sizeX >= 0) {
		save_str = g_strconcat("dp_", plugin, "_sizeX", NULL);
		singit_config_xmms_write(int, cfg, "SingIt", save_str, sizeX);
		g_free(save_str);
	}

	if (sizeY >= 0) {
		save_str = g_strconcat("dp_", plugin, "_sizeY", NULL);
		singit_config_xmms_write(int, cfg, "SingIt", save_str, sizeY);
		g_free(save_str);
	}

	return singit_config_close(cfg, TRUE);
}
