/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <math.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include "singit_debug.h"

#include "editor_singit_main.h"
#include "editor_synced_lyrics.h"
#include "editor_config.h"
#include "editor_status.h"

enum {
	ARG_0,
	ARG_CURSOR_TEXT_OFFSET
};

static SingitEditorViewClass *parent_class = NULL;

extern VisPlugin singit_vp;

void editor_synced_lyrics_text_changed_event (GtkEditable *editable, gpointer user_data);
void editor_synced_lyrics_text_value_changed_event(GtkAdjustment *adj, gpointer user_data);
gboolean before_text_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
gboolean after_text_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);

static void editor_synced_lyrics_destroy (GtkObject *object);
static gchar *editor_synced_lyrics_get_text(SingitEditorView *sev);
static void editor_synced_lyrics_set_text(SingitEditorView *sev, const gchar *text);

static void editor_synced_lyrics_class_init (EditorSyncedLyricsClass *klass)
{
	GtkObjectClass *object_klass;
	SingitEditorViewClass *sev_klass;

	object_klass = (GtkObjectClass*) klass;
	sev_klass = (SingitEditorViewClass*) klass;

	parent_class = gtk_type_class(TYPE_SINGIT_EDITOR_VIEW);

	object_klass->destroy = editor_synced_lyrics_destroy;
	sev_klass->get_text = editor_synced_lyrics_get_text;
	sev_klass->set_text = editor_synced_lyrics_set_text;
}

static void editor_synced_lyrics_init (EditorSyncedLyrics *esl)
{
	GtkWidget *editor_synced_lyrics_vbox, *editor_before_text_hbox,
		*editor_before_text_vscrollbar, *editor_after_text_hbox,
		*editor_after_text_vscrollbar;

	editor_synced_lyrics_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(esl), editor_synced_lyrics_vbox);
	gtk_widget_show(editor_synced_lyrics_vbox);

	editor_before_text_hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(editor_synced_lyrics_vbox), editor_before_text_hbox);
	gtk_widget_show (editor_before_text_hbox);

	esl->before_text = gtk_text_new (NULL, NULL);
	gtk_box_pack_start(GTK_BOX(editor_before_text_hbox), esl->before_text, TRUE, TRUE, 0);
	gtk_text_set_editable(GTK_TEXT(esl->before_text), FALSE);
	gtk_text_set_line_wrap(GTK_TEXT(esl->before_text), FALSE);
	gtk_signal_connect(GTK_OBJECT(esl->before_text), "button_press_event",
		GTK_SIGNAL_FUNC(before_text_button_press_event), esl);
	gtk_widget_set_sensitive(esl->before_text, FALSE);
	gtk_widget_show (esl->before_text);

	editor_before_text_vscrollbar = gtk_vscrollbar_new (GTK_TEXT(esl->before_text)->vadj);
	gtk_box_pack_start(GTK_BOX(editor_before_text_hbox), editor_before_text_vscrollbar, FALSE, FALSE, 0);
	gtk_widget_show(editor_before_text_vscrollbar);

	esl->percent_text_adj = gtk_adjustment_new (0, 0, 101, 1, 1, 1);
	gtk_signal_connect (GTK_OBJECT (esl->percent_text_adj), "value_changed",
		GTK_SIGNAL_FUNC (editor_synced_lyrics_text_value_changed_event), esl);
	esl->percent_text_scale = gtk_hscale_new( GTK_ADJUSTMENT(esl->percent_text_adj) );
	gtk_box_pack_start(GTK_BOX(editor_synced_lyrics_vbox),
		esl->percent_text_scale, FALSE, FALSE, 0);
	gtk_widget_show(esl->percent_text_scale);

	editor_after_text_hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(editor_synced_lyrics_vbox), editor_after_text_hbox);
	gtk_widget_show (editor_after_text_hbox);

	esl->after_text = gtk_text_new (NULL, NULL);
	gtk_box_pack_start(GTK_BOX(editor_after_text_hbox), esl->after_text, TRUE, TRUE, 0);
	gtk_text_set_editable(GTK_TEXT(esl->after_text), FALSE);
	gtk_text_set_line_wrap(GTK_TEXT(esl->after_text), FALSE);
	gtk_signal_connect_after(GTK_OBJECT(esl->after_text), "button_press_event",
		GTK_SIGNAL_FUNC(after_text_button_press_event), esl);
	gtk_widget_show (esl->after_text);

	editor_after_text_vscrollbar = gtk_vscrollbar_new (GTK_TEXT(esl->after_text)->vadj);
	gtk_box_pack_start(GTK_BOX(editor_after_text_hbox), editor_after_text_vscrollbar, FALSE, FALSE, 0);
	gtk_widget_show(editor_after_text_vscrollbar);
}

GtkType editor_synced_lyrics_get_type (void)
{
	static guint ect_type = 0;

	if (!ect_type) {
		GtkTypeInfo ect_info = {
			(gchar*) "EditorSyncedLyrics",
			sizeof (EditorSyncedLyrics),
			sizeof (EditorSyncedLyricsClass),
			(GtkClassInitFunc) editor_synced_lyrics_class_init,
			(GtkObjectInitFunc) editor_synced_lyrics_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		ect_type = gtk_type_unique (TYPE_SINGIT_EDITOR_VIEW, &ect_info);
	}

	return ect_type;
}

static void editor_synced_lyrics_destroy (GtkObject *object)
{
	EditorSyncedLyrics *esl;

	g_return_if_fail (esl = EDITOR_SYNCED_LYRICS (object));

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

gboolean before_text_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	if (event->type != GDK_BUTTON_PRESS) // skip double and triple click events
		{ return TRUE; }

	switch(event->button) {
	case 0:

		break;
	case 1:
		break;
	}

	return TRUE;
}

static gchar* find_next_break_start(gchar *text)
{
	const gchar *breaker = "\n";
	gchar *temp;

	if (!text) { return NULL; }
	if (strlen(breaker) == 1) { return (text + 1); }
	else {
		temp = text;
		if (temp[0] == breaker[0]) { temp++; }
		while (temp[0] && (strchr(breaker, temp[0]) != NULL) && (temp[0] != breaker[strlen(breaker) - 1]))
			{ temp++; }
		return temp;
	}
}

static gint find_end_pos(gchar *text, gboolean *is_breaker)
{
	gchar *pos_copy;
	const gchar *breaker = "\n";

	if (!text) { return -1; }

	pos_copy = text;

	if (pos_copy[0] == ' ') {
		while (pos_copy[0] && (pos_copy[0] == ' ')) { pos_copy++; }
		if (is_breaker) {
			if (strchr(breaker, pos_copy[0]) != NULL) { *is_breaker = TRUE; }
			else { *is_breaker = FALSE; }
		}
		return (pos_copy - text);
	}
	else if (strchr(breaker, pos_copy[0]) != NULL) {
		pos_copy = find_next_break_start(pos_copy);
		if (is_breaker) { *is_breaker = TRUE; }
		return (pos_copy - text);
	}
	else {
		while (pos_copy[0] && (pos_copy[0] != breaker[0]) && (pos_copy[0] != ' ')) { pos_copy++; }
		if (pos_copy[0] == ' ') {
			while (pos_copy[0] && (pos_copy[0] == ' ')) { pos_copy++; }
		}
		if (is_breaker) {
			if (strchr(breaker, pos_copy[0]) != NULL) { *is_breaker = TRUE; }
			else { *is_breaker = FALSE; }
		}
		return (pos_copy - text);
	}
	return -1;
}

gboolean after_text_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	EditorSyncedLyrics *esl;
	gchar *after_string;
	gint time;
	gchar time_stamp[12];
	gint pos_end, breaker_end;
	gint after_text_length, before_text_length;
	gboolean is_breaker;

	g_return_val_if_fail(event != NULL, FALSE);
	g_return_val_if_fail(IS_EDITOR_SYNCED_LYRICS(user_data), FALSE);

	// skip double and triple click events
	if (event->type != GDK_BUTTON_PRESS)
		{ return TRUE; }

	esl = EDITOR_SYNCED_LYRICS(user_data);

	// Do not leak this
	after_string = gtk_editable_get_chars(GTK_EDITABLE(esl->after_text), 0, -1);

	time = xmms_remote_get_output_time(singit_vp.xmms_session);

	pos_end = find_end_pos(after_string, &is_breaker);

	switch (event->button) {
	case 1: // Left mouse button
		if (time > GET_SCD->reactionTime) { time -= GET_SCD->reactionTime; }
		if (GET_ECD->extendedLyrics) {
			time_stamp[11] = '\0';
			sprintf(time_stamp, "[%.2i:%.2i:%.3i]", time / 60000, time / 1000 % 60, time % 1000);
		}
		else {
			time_stamp[7] = '\0';
			sprintf(time_stamp, "[%.2i:%.2i]", time / 60000, time / 1000 % 60);
		}
		gtk_text_set_point(GTK_TEXT(esl->before_text),
				gtk_text_get_length(GTK_TEXT(esl->before_text)));
		gtk_text_insert(GTK_TEXT(esl->before_text), NULL,
			&esl->after_text->style->black, NULL, &time_stamp[0], strlen(time_stamp));
		singit_editor_view_modify(SINGIT_EDITOR_VIEW(esl), TRUE);
		if ((after_string == NULL) || (strlen(after_string) == 0))
			{ break; }

	case 3: // Right mouse button
		if (pos_end > 0) {
			// If CTRL jump to the end of the row / start of the next row
			if (event->state & GDK_CONTROL_MASK) {
				breaker_end = pos_end;
				while ((pos_end > 0) && !is_breaker) {
					pos_end = find_end_pos(after_string + breaker_end, &is_breaker);
					breaker_end += pos_end;
				}
				pos_end = breaker_end;
			}
			gtk_text_set_point(GTK_TEXT(esl->before_text),
				gtk_text_get_length(GTK_TEXT(esl->before_text)));
			if ((after_string != NULL) && (strlen(after_string) != 0)) {
				gtk_text_insert(GTK_TEXT(esl->before_text), NULL,
					&esl->after_text->style->black, NULL, after_string, pos_end);
			}
			else {
				gtk_text_insert(GTK_TEXT(esl->before_text), NULL,
					&esl->after_text->style->black, NULL, "\n", pos_end);
			}
		}
		break;
	}

	if (pos_end > 0) {
		gtk_editable_delete_text(GTK_EDITABLE(esl->after_text), 0, pos_end);
	}

	after_text_length = gtk_text_get_length(GTK_TEXT(esl->after_text));
	before_text_length = gtk_text_get_length(GTK_TEXT(esl->before_text));

	if (after_text_length + before_text_length == 0) {
		GTK_ADJUSTMENT(esl->percent_text_adj)->value = 0;
	}
	else {
		GTK_ADJUSTMENT(esl->percent_text_adj)->value =
			before_text_length / (double) (after_text_length + before_text_length) * 100;
	}
	gtk_adjustment_changed(GTK_ADJUSTMENT(esl->percent_text_adj));

	g_free(after_string);

	return TRUE;
}

void editor_synced_lyrics_text_value_changed_event(GtkAdjustment *adj, gpointer user_data)
{
	const gchar *breaker = "\n";
	EditorSyncedLyrics *esl = EDITOR_SYNCED_LYRICS(user_data);
	gchar *before_string = gtk_editable_get_chars(GTK_EDITABLE(esl->before_text), 0, -1);
	gchar *after_string = gtk_editable_get_chars(GTK_EDITABLE(esl->after_text), 0, -1);
	gchar *pos_insert;

	gint before_length = strlen(before_string);
	gint after_length = strlen(after_string);

	gint new_length = abs((before_length + after_length) * adj->value / 100.0);

	if (new_length < before_length) {
		pos_insert = &before_string[new_length];
		while (((pos_insert - before_string) > 0) && strchr(breaker, pos_insert[0]) != NULL) { pos_insert--; }
		while (((pos_insert - before_string) > 0) && (pos_insert[0] != breaker[strlen(breaker) - 1]) &&
			(pos_insert[0] != ' ')) { pos_insert--; }
		new_length = pos_insert - before_string;
		if (pos_insert[0] == ' ' ||
			((pos_insert[0] == breaker[0]) && strchr(breaker, pos_insert[0]) == NULL)) { new_length++; }

		gtk_editable_delete_text(GTK_EDITABLE(esl->before_text), new_length, -1);
		gtk_text_set_point(GTK_TEXT(esl->after_text), 0);
		gtk_text_insert(GTK_TEXT(esl->after_text), NULL,
				&esl->after_text->style->black,
				NULL, &before_string[new_length], before_length - new_length);
		gtk_text_set_point(GTK_TEXT(esl->after_text), 0);
		gtk_widget_queue_draw(esl->after_text);
	}
	else if (new_length > before_length) {
		pos_insert = &after_string[new_length - before_length];
		while (pos_insert[0] && ((pos_insert - after_string) > 0) && (pos_insert[0] != breaker[strlen(breaker) - 1]) &&
			(pos_insert[0] != ' ')) { pos_insert--; }
		new_length = pos_insert - after_string + before_length;
		if (pos_insert[0] == ' ' || (pos_insert[0] == breaker[0])) { new_length++; }

		gtk_editable_delete_text(GTK_EDITABLE(esl->after_text), 0, new_length - before_length);
		gtk_text_set_point(GTK_TEXT(esl->before_text),
			gtk_text_get_length(GTK_TEXT(esl->before_text)));
		gtk_text_insert(GTK_TEXT(esl->before_text), NULL,
				&esl->before_text->style->black,
				NULL, after_string, new_length - before_length);
	}

//	gtk_editable_set_position(GTK_EDITABLE(esl->before_text), -1);
//	gtk_editable_set_position(GTK_EDITABLE(esl->after_text), 0);
	g_free(after_string);
	g_free(before_string);
}

static gchar *editor_synced_lyrics_get_text(SingitEditorView *sev)
{
	gchar *result = NULL, *concat_str;
	EditorSyncedLyrics *esl;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_synced_lyrics.c [editor_synced_lyrics_get_text]\n"));
#endif

	g_return_val_if_fail(IS_SINGIT_EDITOR_VIEW(sev), NULL);

	esl = EDITOR_SYNCED_LYRICS(sev);

	concat_str = gtk_editable_get_chars(GTK_EDITABLE(esl->before_text), 0, -1);
	result = g_strconcat(concat_str, gtk_editable_get_chars
		(GTK_EDITABLE(esl->after_text), 0, -1), NULL);

	sev->cursor_text_offset = strlen(concat_str);

	g_free(concat_str);
	return result;
}

static void editor_synced_lyrics_set_text(SingitEditorView *sev, const gchar *text)
{
	gint before_text_length, after_text_length, char_offset;
	const gchar *breaker = "\n";
	EditorSyncedLyrics *esl;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_synced_lyrics.c [editor_synced_lyrics_set_text]\n"));
#endif

	g_return_if_fail(IS_SINGIT_EDITOR_VIEW(sev));

	esl = EDITOR_SYNCED_LYRICS(sev);

	gtk_text_freeze(GTK_TEXT(esl->before_text));
	gtk_text_freeze(GTK_TEXT(esl->after_text));
	if (gtk_text_get_length(GTK_TEXT(esl->before_text)) > 0) {
		gtk_text_set_point(GTK_TEXT(esl->before_text), 0);
		gtk_text_forward_delete(GTK_TEXT(esl->before_text),
			gtk_text_get_length(GTK_TEXT(esl->before_text)));
	}
	if (gtk_text_get_length(GTK_TEXT(esl->after_text)) > 0) {
		gtk_text_set_point(GTK_TEXT(esl->after_text), 0);
		gtk_text_forward_delete(GTK_TEXT(esl->after_text),
			gtk_text_get_length(GTK_TEXT(esl->after_text)));
	}

	char_offset = sev->cursor_text_offset;

	if (text) {
		if (char_offset > 0) {
			while ((text[char_offset] != breaker[strlen(breaker) - 1]) &&
				(text[char_offset] != ' '))
			{
				char_offset--;
				if (char_offset == -1)
					{ break; }
			}
			char_offset++;
		}
		if (char_offset > 0) {
			before_text_length = MIN((gint) strlen(text), char_offset);
			gtk_text_insert(GTK_TEXT(esl->before_text), NULL,
				&esl->before_text->style->black,
				NULL, text, before_text_length);
			gtk_text_set_point(GTK_TEXT(esl->before_text), before_text_length);
			after_text_length = strlen(text) - char_offset;
		}
		else {
			after_text_length = -1;
			before_text_length = 0;
		}

		if (after_text_length != 0) {
			gtk_text_insert(GTK_TEXT(esl->after_text), NULL,
				&esl->after_text->style->black,
				NULL, (gchar*) (text + before_text_length),
				after_text_length);
			gtk_text_set_point(GTK_TEXT(esl->after_text), 0);
		}
		if (after_text_length + before_text_length == 0) { GTK_ADJUSTMENT(esl->percent_text_adj)->value = 0; }
		else {
			GTK_ADJUSTMENT(esl->percent_text_adj)->value =
				before_text_length / (double) (after_text_length + before_text_length) * 100;
		}
//		gtk_editable_set_position(GTK_EDITABLE(esl->before_text), -1);
//		gtk_editable_set_position(GTK_EDITABLE(esl->after_text), 0);
		gtk_adjustment_changed(GTK_ADJUSTMENT(esl->percent_text_adj));
	}
	gtk_text_thaw(GTK_TEXT(esl->after_text));
	gtk_text_thaw(GTK_TEXT(esl->before_text));
}

GtkWidget* editor_synced_lyrics_new(void)
{
	GtkWidget *editor;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_synced_lyrics.c [editor_synced_lyrics_new]\n"));
#endif

	editor = gtk_type_new(TYPE_EDITOR_SYNCED_LYRICS);
	gtk_widget_set_usize(editor, 300, 160);

	return editor;
}
