/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Based on
 * mfl.c - General parts of Minimal Font Library
 *
 * Copyright (C) 2000 Boris Gjenero
 *
 * You may contact Boris Gjenero at bgjenero@sympatico.ca
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "libmfl.h"

// * Include the default font data *
#include "mfl_deffont.inc"

struct mfl_font_s {

	int height;
	unsigned char *data;
	int is_default;
};

struct mfl_context_s {

	void *buf;

	unsigned int bpp;
	unsigned int width;
	unsigned int height;
	unsigned int bpl;

	unsigned long color;
	mfl_font font;
	int opmode;
};

mfl_font mfl_LoadRawFont(const char *fname) {

	mfl_font f = NULL;
	unsigned long int l;
	FILE *ff;

	if (fname == NULL) {
		l = mfl_deffont_size;
		goto allocate_font;
	}

	// * Open font file *
	ff = fopen(fname, "rb");
	if (ff == NULL)
		goto lrf_open_fault;

	// * Get length of font file *
	if (fseek(ff, 0, SEEK_END) != 0) goto lrf_fault;
	l = ftell(ff);

	// * Seek to start *
	if (fseek(ff, 0, SEEK_SET) != 0) goto lrf_fault;

	// * Determine font height *
	if (l & 0xff) goto lrf_fault;	// * Unknown length *

allocate_font:
	// * Allocate data, fill out structure *
	f = malloc(sizeof(struct mfl_font_s));
	f->height = l >> 8;
	f->is_default = (fname == NULL);

	// * Set font data *
	if (!f->is_default) {
		f->data = malloc(l);
		if (fread(f->data, 1, l, ff) != l) {
			free(f->data);
			free(f);
			f = NULL;
		}
	}
	else {
		f->data = (unsigned char *) mfl_deffont;
		goto lrf_open_fault;
	}

 lrf_fault:
	// * Close input file *
	fclose(ff);

 lrf_open_fault:
	return f;
}

void mfl_DestroyFont(mfl_font f) {

	if (!f->is_default)
		free(f->data);
	free(f);
}

void mfl_UpdateContext(mfl_context cx, void *buf, unsigned int bpp,
	unsigned int bpl, unsigned int width, unsigned int height) {

	cx->buf = buf;
	cx->bpp = bpp;
	cx->bpl = bpl;
	cx->width = width;
	cx->height = height;
}

mfl_context mfl_CreateContext(void *buf, unsigned int bpp,
	unsigned int bpl, unsigned int width, unsigned int height) {

	mfl_context cx = malloc(sizeof(struct mfl_context_s));

	mfl_UpdateContext(cx, buf, bpp, bpl, width, height);

	cx->font = NULL;
	cx->opmode = MFL_NORMAL;

	return cx;
}

void mfl_DestroyContext(mfl_context cx) {

	free(cx);
}

void mfl_SetTextColor(mfl_context cx, unsigned long c) {

	cx->color = c;
}

void mfl_SetFont(mfl_context cx, mfl_font f) {

	cx->font = f;
}

void mfl_SetDrawMode(mfl_context cx, int mode) {

	cx->opmode = mode;
}

static inline unsigned int mfl_GetTextWidthInline
	(const mfl_context cx, const char *s, int l) {

	return l * 8 * (cx->bpp / 8);
}

unsigned int mfl_GetTextWidthL
	(const mfl_context cx, const char *s, int l) {

	return mfl_GetTextWidthInline(cx, s, l);
}

unsigned int mfl_GetTextWidth
	(const mfl_context cx, const char *s) {

	return mfl_GetTextWidthInline(cx, s, strlen(s));
}

#undef P_SZ
#define P_SZ	1
#include "mfl_draw_inc.c"
#undef P_SZ

#define P_SZ	2
#include "mfl_draw_inc.c"
#undef P_SZ

#define P_SZ	4
#include "mfl_draw_inc.c"
#undef P_SZ

void mfl_OutChar(const mfl_context cx, long x, long y, char c) {

	switch (cx->bpp) {
	case 8: mfl_OutChar8(cx, x, y, c); break;
	case 16: mfl_OutChar16(cx, x, y, c); break;
	case 32: mfl_OutChar32(cx, x, y, c); break;
	};
}


void mfl_OutTextL(const mfl_context cx, long x, long y, const char *s, int l) {

	switch (cx->bpp) {
	case 8: mfl_OutText8L(cx, x, y, s, l); break;
	case 16: mfl_OutText16L(cx, x, y, s, l); break;
	case 32: mfl_OutText32L(cx, x, y, s, l); break;
	};
}

void mfl_OutText(const mfl_context cx, long x, long y, const char *s) {

	switch (cx->bpp) {
	case 8: mfl_OutText8L(cx, x, y, s, strlen(s)); break;
	case 16: mfl_OutText16L(cx, x, y, s, strlen(s)); break;
	case 32: mfl_OutText32L(cx, x, y, s, strlen(s)); break;
	};
}
