
#include "Hashtable.h"

#include "Eg Common.h"

#include <stdlib.h>



long Hashtable::sTableSizes[ NUM_SIZES ] = { 23, 97, 397, 797, 3203, 12853, 51437, 205759, 1646237, 3292489, 6584983, 13169977, 52679969 };

	

Hashtable::Hashtable( int inLoadFactor ) {
	mTableSize		= 0;
	mTable			= nil;
	mNumEntries		= 0;
	mLoadFactor		= inLoadFactor * 64 / 100;
	mHashParam		= -1;
	
	// Don't let the client kill himself with a bad load factor
	// The load factor is the max ratio of filled slots to total slots
	if ( mLoadFactor > 63 )
		mLoadFactor = 63;
	else if ( mLoadFactor < 7 )
		mLoadFactor = 7;
		
	Rehash();
}


Hashtable::~Hashtable() {
	RemoveAll();
	
	if ( mTable )
		delete []mTable;
}

	
void Hashtable::Rehash() {
	long		i, index, oldSize = mTableSize;
	KEntry*		temp, *entry;
	KEntry**	oldTable = mTable;
	
	// Find the next bigger table size
	for ( i = 0; mTableSize <= oldSize; i++ )
		mTableSize = sTableSizes[ i ];

	
	// Alloc the new table and make it empty	
	mTable = new KEntry*[ mTableSize ];
	for ( i = 0; i < mTableSize; i++ )
		mTable[ i ] = nil;
	
	// Rehash all the old values into the new table
	for ( i = 0; i < oldSize; i++ ) {
		for ( entry = oldTable[ i ]; entry; ) {
			index = entry -> mKey % mTableSize;
			temp = entry -> mNext;
			entry -> mNext = mTable[ index ];
			mTable[ index ]	= entry;
			entry = temp;
		}	
	}
	
	// We don't need the old table anymore
	if ( oldTable )
		delete []oldTable;
}





void* Hashtable::put( long inKey, const Hashable* inHKey, void* inValue, long inFlags ) {
	long			index;
	KEntry*			entry;
	void*			oldVal;

	// See if we need to make the hash table bigger
	if ( mNumEntries >= ( ( mTableSize * ( (long) mLoadFactor ) ) >> 6 ) )
		Rehash();
	
	// If we already have the key, replace the value and pass the old one back
	entry = fetchEntry( inKey, inHKey );
	if ( entry ) {
		oldVal			= entry -> mValue;
		
		// If this value is marked to be owned by this Hashtable, delete it.
		if ( entry -> mFlags & HT_VALUE_OWNED ) {
			delete ((DictValue*) oldVal);
			oldVal = nil;
		}
				
		// If this key is to be owned by this Hashtable, delete it.
		if ( ( inFlags & HT_KEY_OWNED ) && inHKey ) {
			
			// If the current key isn't owned and the incoming key is, avoid having to delete by using the owned key
			if ( ( entry -> mFlags & HT_KEY_OWNED ) == 0 )
				entry -> mHashable = inHKey; 
			else
				delete inHKey;  
		}
		
		// Carry over the HT_KEY_OWNED bit b/c it remain on if the current key is owned, regardless of incoming key ownership
		inFlags |= entry -> mFlags & HT_KEY_OWNED;

		}
	else { 
		oldVal				= nil;
		entry				= new KEntry;
		index				= ((unsigned long) inKey) % mTableSize;
		entry -> mKey		= ((unsigned long) inKey);
		entry -> mHashable	= inHKey;
		entry -> mNext		= mTable[ index ];
		mTable[ index ]		= entry;
		mNumEntries++;
	}

	entry -> mFlags		= inFlags;
	entry -> mValue		= inValue;
	
	return oldVal;
}






long Hashtable::get( long inKey, const Hashable* inHashable, void** outValue ) const {
	KEntry*	entry = fetchEntry( inKey, inHashable );
	long retFlags = 0;

	if ( entry ) {
	
		retFlags = entry -> mFlags | HT_KEY_FOUND;
		
		if ( outValue ) 
			*outValue = entry -> mValue;
	}
	
	
	return retFlags;
}



bool Hashtable::setFlags( long inKey, const Hashable* inHashable, long inFlags ) {
	KEntry*	entry = fetchEntry( inKey, inHashable );

	if ( entry ) {
	
		// Clear the user flags then OR that with the new user flags
		entry -> mFlags = ( entry -> mFlags & ( ~ HT_USER_FLAGS ) ) | ( inFlags & HT_USER_FLAGS );
	}
	
	return ( entry != nil );
}







void Hashtable::GetValues( XPtrList& outValues ) {
	KEntry** entryP = mTable, *entry;
	int i;
	
	outValues.Dim( NumEntries() );
	
	for ( i = 0; i < mTableSize; i++ ) {
		entry = *entryP;
		while ( entry ) {
			outValues.Add( entry -> mValue );
			entry = entry -> mNext;
		}

		entryP++;
	}
}


void Hashtable::GetKeys( XPtrList& outKeys ) {
	KEntry** entryP = mTable, *entry;
	int i;
	
	outKeys.Dim( NumEntries() );
	
	for ( i = 0; i < mTableSize; i++ ) {
		entry = *entryP;
		while ( entry ) {
			outKeys.Add( ( entry -> mHashable ) ? entry -> mHashable : ( (const void*) entry -> mKey ) );
			entry = entry -> mNext;
		}

		entryP++;
	}
}


const Hashable* Hashtable::GetKey( const Hashable* inKey ) {
	KEntry* entry = fetchEntry( inKey -> Hash( mHashParam ), inKey );
	
	return entry ? entry -> mHashable : nil;
}




void* Hashtable::GetValue( long inN ) {
	KEntry** entryP = mTable, *entry;
	long i, n = 0;
	
	for ( i = 0; i < mTableSize; i++ ) {
		entry = *entryP;
		while ( entry ) {
			if ( n == inN ) 
				return entry -> mValue;
			n++;
		}

		entryP++;
	}
	
	return nil;
}


KEntry*	Hashtable::fetchEntry( long inKey, const Hashable* inHKey ) const {
	long		index = ( (unsigned long) inKey ) % mTableSize;
	KEntry*		entry = mTable[ index ];

	// Look thru the chain for the key and the entry holding that key-value pair
	while ( entry ) {
		if ( entry -> mKey == inKey ) {
		
			// Instant match if there's no hashables to check or if the hashables are a trivial match (ie, why call Equals() if they're the same ptr?)
			if ( ! entry -> mHashable || ! inHKey || entry -> mHashable == inHKey )
				return entry;
			else if ( inHKey -> Equals( entry -> mHashable, mHashParam ) )
				return entry;
		}
		entry = entry -> mNext;
	}
	
	return nil;
}



void Hashtable::Iterate( IteratorFuncT inFunc, void* inArg ) {
	long i;
	KEntry*	entry;
	
	// Step theu the dict table and delete all the KEntries
	for ( i = 0; i < mTableSize; i++ ) {
		for ( entry = mTable[ i ]; entry; ) {
		
			//  Perform the callback
			inFunc( entry, inArg );
			
			entry = entry -> mNext;
		}
	}
}






void Hashtable::RemoveAll() {
	long i;
	KEntry*	entry, *temp;
	
	// Step thru the dict table and delete each KEntries (and data attached to it)
	for ( i = 0; i < mTableSize; i++ ) {
		entry = mTable[ i ];
		if ( entry ) {
			while ( entry ) {
				temp = entry -> mNext;
				deleteEntry( entry );
				entry = temp;
			}
			mTable[ i ] = nil;
		}
	}
	mNumEntries = 0;
}



void* Hashtable::remove( long inKey, const Hashable* inHKey ) {
	long		index = ( (unsigned long) inKey ) % mTableSize;
	KEntry*		entry = mTable[ index ], *prev = nil;
	bool		isEqual;
	void*		retVal;

	// Look thru the chain for a matching key and delete it
	while ( entry ) {
		if ( entry -> mKey == inKey ) {
			if ( inHKey && entry -> mHashable )
				isEqual = inHKey -> Equals( entry -> mHashable, mHashParam );
			else
				isEqual = true;
				
			if ( isEqual ) {
		
				if ( prev == nil ) 
					mTable[ index ] = nil;
				else 
					prev -> mNext = entry -> mNext;

				retVal = deleteEntry( entry );

				mNumEntries--;
				return retVal;
			}
		}
		prev = entry;
		entry = entry -> mNext;
	}
	
	return nil;
}



void* Hashtable::deleteEntry( KEntry* inEntry ) {
	void* value = inEntry -> mValue;
	
	// If the key or value is owned (or both), delete them///
	if ( ( inEntry -> mFlags & HT_KEY_OWNED ) && inEntry -> mHashable )
		delete inEntry -> mHashable;
		
	if ( inEntry -> mFlags & HT_VALUE_OWNED ) {
		delete ( (DictValue*) value );
		value = nil;
	}

	delete inEntry;
	
	return value;
}



long& Hashtable::operator[] ( const long inKey ) {
	KEntry*	entry = fetchEntry( inKey, nil );
	
	if ( ! entry ) {
		Put( inKey, HT_NO_FLAGS );
		entry = fetchEntry( inKey, nil );
	}
	
	return (long&) entry -> mValue;
}



void*& Hashtable::operator[] ( const void* inKey ) {
	KEntry*	entry = fetchEntry( (long) inKey, nil );
	
	if ( ! entry ) {
		Put( (long) inKey, HT_NO_FLAGS );
		entry = fetchEntry( (long) inKey, nil );
	}
	
	return entry -> mValue;
}






void Hashtable::Rank( XPtrList& outKeys, ANSI_QSCompFcnT inCompFcn, long inNumToRank ) {
	long i, n = NumEntries();
	KEntry** entryP = mTable, *entry;
	const void **p, **temp = new const void*[ 2 * n ];
	
	if ( inNumToRank < 0 )
		inNumToRank = n;
	inNumToRank = _MIN( inNumToRank, n );

	// To rank, we must sort by value, with a tag on each element of its key
	p = temp;
	for ( i = 0; i < mTableSize; i++ ) {
		entry = *entryP;
		while ( entry ) {
			*p = entry -> mValue;  
			p++; 
			*p = ( entry -> mHashable ) ? entry -> mHashable : (const void*) entry -> mKey;
			p++;
			entry = entry -> mNext;
		}

		entryP++;
	}

	// Default to long-long ranking
	if ( ! inCompFcn )
		inCompFcn = sLongComparitor;
	
	// Sort the floats
	qsort( temp, n, 8, inCompFcn );
	
	// Put the sorted results in the destination
	outKeys.RemoveAll();
	p = temp + 1;
	for ( i = 0; i < n; i++ ) {
		outKeys.Add( *p );
		p += 2;
	}
	
	// Cleanup
	delete []temp;
}





int Hashtable::sLongComparitor( const void* inA, const void* inB ) {
	return ((long) inB - (long) inA);
}
