/* MSA spectrum analyzer
 * Copyright (C) 2000 Michal Kunikowski <kunik@poczta.onet.pl>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bmp.h"
#include "msa.h"
#include "msa_skin.h"

#include "titlebar.xpm"
#include "logo.xpm"

#include "debug.h"

static GdkPixmap *titlebar_pixmap = NULL;
static GdkPixmap *logo_pixmap = NULL;
static GdkPixmap *ledsoff_pixmap = NULL,	*ledson_pixmap = NULL,
		 *ledsoffrev_pixmap = NULL,	*ledsonrev_pixmap = NULL,
                 *ledsoffmir_pixmap = NULL,	*ledsonmir_pixmap = NULL,
		 *ledsoffrevmir_pixmap = NULL,	*ledsonrevmir_pixmap = NULL;

// loads image first tries to load
// adding an extension .gif, .png, .bmp and .xpm
GdkPixmap *loadimage(GdkWindow *window, gchar *filename)
{
	GdkPixmap *pixmap = NULL;
	gchar *fn;
	fn = (gchar*)g_malloc((strlen(filename)+5)*sizeof(gchar));

	DPRINT(1, "loadimage(%p, \"%s\")\n",window,filename);

	// try to load file with .bmp extension
	strcpy(fn, filename);
	strcat(fn, ".bmp");
	pixmap = msa_read_bmp(window, fn);
	if (pixmap != NULL) goto end;

	// try to load file with .xpm extension
	strcpy(fn, filename);
	strcat(fn, ".xpm");
	pixmap = gdk_pixmap_create_from_xpm(window, NULL, NULL, fn);
	if (pixmap != NULL) goto end;
	
	DPRINT(1, "loadimage -> failed to load %s\n", filename);

end:	g_free(fn);
	return pixmap;
}

void set_logo_pixmap(GdkGC *gc, GdkPixmap *pixmap)
{
	DPRINT(1, "set_logo_pixmap(%p,%p)\n",gc,pixmap);

	gdk_draw_pixmap(pixmap, gc, logo_pixmap,
		0, 0, 0, 0,
		msaskincfg.window_width, msaskincfg.window_height);
}

void add_suitable_titlebar_to_pixmap(GdkGC *gc, GdkPixmap *pixmap)
{
	int i=0;
	
	DPRINT(1, "add_suitable_titlebar_to_pixmap(%p,%p)\n",gc,pixmap);
	
	if (msacfg.isplaying && msacfg.hasfocus) i=0;
	else if (msacfg.isplaying && !msacfg.hasfocus) i=msaskincfg.titlebar_height*1;
	else if (!msacfg.isplaying && msacfg.hasfocus) i=msaskincfg.titlebar_height*2;
	else if (!msacfg.isplaying && !msacfg.hasfocus) i=msaskincfg.titlebar_height*3;

	gdk_draw_pixmap(pixmap, gc, titlebar_pixmap,
		0, i,
		msaskincfg.titlebar_x, msaskincfg.titlebar_y,
		msaskincfg.titlebar_width, msaskincfg.titlebar_height);
}

void set_suitable_analyzer_pixmaps(GdkPixmap **leds_off_pixmap, GdkPixmap **leds_on_pixmap)
{
	DPRINT(1, "set_suitable_analyzer_pixmaps(%p,%p)\n",leds_off_pixmap,leds_on_pixmap);
	
	if (msacfg.reverse_mode && msacfg.mirror_mode) {
		DPRINT(1, "set_suitable_analyzer_pixmaps: reversed & mirrored pixmaps\n");
		*leds_on_pixmap  = ledsonrevmir_pixmap;
		*leds_off_pixmap = ledsoffrevmir_pixmap;
	}
	else if (msacfg.reverse_mode && !msacfg.mirror_mode) {
		DPRINT(1, "set_suitable_analyzer_pixmaps: reversed pixmaps\n");
		*leds_on_pixmap  = ledsonrev_pixmap;
		*leds_off_pixmap = ledsoffrev_pixmap;
	}
	else if (!msacfg.reverse_mode && msacfg.mirror_mode) {
		DPRINT(1, "set_suitable_analyzer_pixmaps: mirrored pixmaps\n");
		*leds_on_pixmap  = ledsonmir_pixmap;
		*leds_off_pixmap = ledsoffmir_pixmap;
	}
	else if (!msacfg.reverse_mode && !msacfg.mirror_mode) {
		DPRINT(1, "set_suitable_analyzer_pixmaps: normal pixmaps\n");
		*leds_on_pixmap  = ledson_pixmap;
		*leds_off_pixmap = ledsoff_pixmap;
	}
	else {
		DPRINT(0, "set_suitable_analyzer_pixmaps: !!! IMPOSSIBLE !!!\n");
	}
}


// *** SKIN CONFIG ***

void msaskin_setdefaultcfg(msaskinconfig *skincfg)
{
	DPRINT(1, "msaskin_setdefaultcfg(%p)\n",skincfg);
	
	if (skincfg==NULL) return;

	skincfg->ver=0;

	skincfg->num_bands=32;

	skincfg->peak_height=2;

	skincfg->logo_no=0;
	skincfg->bands_off_no=1;
	skincfg->bands_on_no=2;
	skincfg->bands_off_rev_no=1;
	skincfg->bands_on_rev_no=2;

	skincfg->window_width=275;
	skincfg->window_height=116;

	skincfg->titlebar_width=275;
	skincfg->titlebar_height=13;
	skincfg->titlebar_x=0;
	skincfg->titlebar_y=0;
	
	skincfg->exitbutton_width=9;
	skincfg->exitbutton_height=9;
	skincfg->exitbutton_x=264;
	skincfg->exitbutton_y=3;
	
	skincfg->menubutton_width=9;
	skincfg->menubutton_height=9;
	skincfg->menubutton_x=6;
	skincfg->menubutton_y=3;

	skincfg->analyzer_width=256;
	skincfg->analyzer_height=92;
	skincfg->analyzer_x=10;
	skincfg->analyzer_y=15;
};

void splitstr(char *dst1, char *dst2, char *src, char delim)
{
	int i=0, j=0, srclen=0;
	srclen=strlen(src);
	for (i=0; i<srclen && src[i]!=delim; i++) dst1[i]=src[i];
	dst1[i++]='\0';
        for (j=0; i<srclen; i++,j++) dst2[j]=src[i];
	dst2[j++]='\0';
}

gboolean msaskin_loadskinconfig(msaskinconfig *skincfg, char *skindir)
{
	FILE *f = NULL;
	char file[MSA_MAX_PATH_SIZE], line[MSA_MAX_PATH_SIZE],
	     idstr[MSA_MAX_PATH_SIZE], valstr[MSA_MAX_PATH_SIZE];
	int v;
	
	strncpy(file, skindir, MSA_MAX_PATH_SIZE);
	strncat(file, "/config", MSA_MAX_PATH_SIZE);

	DPRINT(1, "xmms plugin: loading config from file %s\n", file);

	msaskin_setdefaultcfg(skincfg);
	
	if ( (f=fopen(file, "r")) == NULL ) {
		DPRINT(1, "xmms plugin: error loading config from file %s\n", file);
		return FALSE;
	}

	while (fgets(line, MSA_MAX_PATH_SIZE, f)) {
		g_strstrip(line);
		if (line[0] != '#') {
			splitstr(idstr, valstr, line, '=');
			g_strstrip(idstr);
			g_strup(idstr);
			g_strstrip(valstr);
			v=atoi(valstr);

			if      (strcmp(idstr, "VERSION")==0)           skincfg->ver              =v;
			else if (strcmp(idstr, "NUM_BANDS")==0)         skincfg->num_bands        =v;
			else if (strcmp(idstr, "PEAK_HEIGHT")==0)       skincfg->peak_height      =v;			
			else if (strcmp(idstr, "LOGO_NO")==0)           skincfg->logo_no          =v;
			else if (strcmp(idstr, "BANDS_OFF_NO")==0)      skincfg->bands_off_no     =v;
			else if (strcmp(idstr, "BANDS_ON_NO")==0)       skincfg->bands_on_no      =v;
			else if (strcmp(idstr, "REV_BANDS_OFF_NO")==0)  skincfg->bands_off_rev_no =v;
			else if (strcmp(idstr, "REV_BANDS_ON_NO")==0)   skincfg->bands_on_rev_no  =v;
			else if (strcmp(idstr, "WINDOW_WIDTH")==0)      skincfg->window_width     =v;
			else if (strcmp(idstr, "WINDOW_HEIGHT")==0)     skincfg->window_height    =v;
//			else if (strcmp(idstr, "TITLEBAR_WIDTH")==0)    skincfg->titlebar_width   =v;
			else if (strcmp(idstr, "TITLEBAR_HEIGHT")==0)   skincfg->titlebar_height  =v;
//			else if (strcmp(idstr, "TITLEBAR_X")==0)        skincfg->titlebar_x       =v;
//			else if (strcmp(idstr, "TITLEBAR_Y")==0)        skincfg->titlebar_y       =v;
			else if (strcmp(idstr, "ANALYZER_WIDTH")==0)    skincfg->analyzer_width   =v;
			else if (strcmp(idstr, "ANALYZER_HEIGHT")==0)   skincfg->analyzer_height  =v;
			else if (strcmp(idstr, "ANALYZER_X")==0)        skincfg->analyzer_x       =v;
			else if (strcmp(idstr, "ANALYZER_Y")==0)        skincfg->analyzer_y       =v;
			else if (strcmp(idstr, "EXITBUTTON_WIDTH")==0)  skincfg->exitbutton_width =v;
			else if (strcmp(idstr, "EXITBUTTON_HEIGHT")==0) skincfg->exitbutton_height=v;
			else if (strcmp(idstr, "EXITBUTTON_X")==0)      skincfg->exitbutton_x     =v;
			else if (strcmp(idstr, "EXITBUTTON_Y")==0)      skincfg->exitbutton_y     =v;
			else if (strcmp(idstr, "MENUBUTTON_WIDTH")==0)  skincfg->menubutton_width =v;
			else if (strcmp(idstr, "MENUBUTTON_HEIGHT")==0) skincfg->menubutton_height=v;
			else if (strcmp(idstr, "MENUBUTTON_X")==0)      skincfg->menubutton_x     =v;
			else if (strcmp(idstr, "MENUBUTTON_Y")==0)      skincfg->menubutton_y     =v;
		}
	}
	return TRUE;
}

// *** DEFAULT SKIN PIXMAPS ***

void msaskin_unref_skin_pixmaps()
{
	DPRINT(1, "msaskin_unref_skin_pixmaps()\n");

	unrefpixmap(&titlebar_pixmap);
	unrefpixmap(&logo_pixmap);
	unrefpixmap(&ledsoff_pixmap);
	unrefpixmap(&ledson_pixmap);
	unrefpixmap(&ledsoffrev_pixmap);
	unrefpixmap(&ledsonrev_pixmap);
	unrefpixmap(&ledsoffmir_pixmap);
	unrefpixmap(&ledsonmir_pixmap);
	unrefpixmap(&ledsoffrevmir_pixmap);
	unrefpixmap(&ledsonrevmir_pixmap);
}

static gint msaskin_hexdigit2decdigit(gchar c)
{
	if (c>='0'&&c<='9') return (gint)(c-'0');
	else if (c>='a'&&c<='f') return (gint)(c-'a'+10);
	else if (c>='A'&&c<='F') return (gint)(c-'A'+10);
	return 0;
}

static void msaskin_color_parse(const gchar *spec, GdkColor *color)
{
	color->red   = (msaskin_hexdigit2decdigit(spec[0])*0x10+msaskin_hexdigit2decdigit(spec[1])) * 0x100;
	color->green = (msaskin_hexdigit2decdigit(spec[2])*0x10+msaskin_hexdigit2decdigit(spec[3])) * 0x100;
	color->blue  = (msaskin_hexdigit2decdigit(spec[4])*0x10+msaskin_hexdigit2decdigit(spec[5])) * 0x100;
}

static void msaskin_analyzer_colors_parse(const gchar *spec, msa_analyzer_colors *colors)
{
	msaskin_color_parse(spec, &(colors->bg));
	msaskin_color_parse(&spec[7], &(colors->low));
	msaskin_color_parse(&spec[14], &(colors->med));
	msaskin_color_parse(&spec[21], &(colors->high));
}

static void set_default_skin_pixmaps(GdkWindow *window, GdkGC *gc, gchar *colorspec)
{
	msa_analyzer_colors color;
	GdkPixmap *tmp_pixmap = NULL;
	int i,j;

	DPRINT(1, "set_default_skin_pixmaps(%p,%p,%s)\n",window,gc,colorspec);
	
	msaskin_analyzer_colors_parse(colorspec, &color);

	msaskin_unref_skin_pixmaps();

	logo_pixmap          = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoff_pixmap       = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledson_pixmap        = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoffrev_pixmap    = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsonrev_pixmap     = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoffmir_pixmap    = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsonmir_pixmap     = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoffrevmir_pixmap = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsonrevmir_pixmap  = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());

	titlebar_pixmap = gdk_pixmap_create_from_xpm_d(window, NULL, NULL, msa_default_titlebar_xpm);

	msaskin_set_default_border_pixmap(gc, logo_pixmap);
	gdk_draw_pixmap(ledsoff_pixmap,       gc, logo_pixmap, 0, 0, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledson_pixmap,        gc, logo_pixmap, 0, 0, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsoffrev_pixmap,    gc, logo_pixmap, 0, 0, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsonrev_pixmap,     gc, logo_pixmap, 0, 0, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsoffmir_pixmap,    gc, logo_pixmap, 0, 0, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsonmir_pixmap,     gc, logo_pixmap, 0, 0, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsoffrevmir_pixmap, gc, logo_pixmap, 0, 0, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsonrevmir_pixmap,  gc, logo_pixmap, 0, 0, 0, 0, msaskincfg.window_width, msaskincfg.window_height);

	tmp_pixmap = gdk_pixmap_create_from_xpm_d(window, NULL, NULL, msa_default_logo_xpm);

	// how to get pixmap width and heigth out of pixmap ???
	i=102;
	j=70;
	gdk_draw_pixmap(logo_pixmap, gc, tmp_pixmap, 0, 0,
		(msaskincfg.window_width-i)/2, msaskincfg.titlebar_height/2 + msaskincfg.window_height/2-j/2,
		i, j);
	unrefpixmap(&tmp_pixmap);

	msaskin_set_std_analyzer_pixmap(gc,          &color, ledsoff_pixmap, ledson_pixmap);
	msaskin_set_std_rev_analyzer_pixmap(gc,      &color, ledsoffrev_pixmap, ledsonrev_pixmap);
	msaskin_set_mirrored_analyzer_pixmap(gc,     &color, ledsoffmir_pixmap, ledsonmir_pixmap);
	msaskin_set_mirrored_rev_analyzer_pixmap(gc, &color, ledsoffrevmir_pixmap, ledsonrevmir_pixmap);
}

void set_skin_from_file_v1(GdkWindow *window, GdkGC *gc, gchar *skindir)
{
	GdkPixmap *analyzer_pixmap = NULL;
	
	char file_analyzer[MSA_MAX_PATH_SIZE], file_titlebar[MSA_MAX_PATH_SIZE];
	gboolean stat = TRUE;
	
	DPRINT(1, "set_skin_from_file_v1(%p,%p,%s)\n",window,gc,skindir);

	msaskin_unref_skin_pixmaps();

	strncpy(file_analyzer, msacfg.skindir, MSA_MAX_PATH_SIZE);
	strncat(file_analyzer, "/analyzer", MSA_MAX_PATH_SIZE);
	strncpy(file_titlebar, msacfg.skindir, MSA_MAX_PATH_SIZE);
	strncat(file_titlebar, "/titlebar", MSA_MAX_PATH_SIZE);

	stat =	(titlebar_pixmap = loadimage(window, file_titlebar))!=NULL &&
		(analyzer_pixmap = loadimage(window, file_analyzer))!=NULL;

	if (!stat) {
		unrefpixmap(&titlebar_pixmap);
		unrefpixmap(&analyzer_pixmap);
		msaskin_setdefaultcfg(&msaskincfg);
		set_default_skin_pixmaps(window, gc, DEFAULT_SKIN_COLOR);
		return;
	}

	logo_pixmap       = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoff_pixmap    = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledson_pixmap     = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoffrev_pixmap = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsonrev_pixmap  = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());

	gdk_draw_pixmap(logo_pixmap,       gc, analyzer_pixmap, 0, msaskincfg.window_height*msaskincfg.logo_no,          0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsoff_pixmap,    gc, analyzer_pixmap, 0, msaskincfg.window_height*msaskincfg.bands_off_no,     0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledson_pixmap,     gc, analyzer_pixmap, 0, msaskincfg.window_height*msaskincfg.bands_on_no,      0, 0, msaskincfg.window_width, msaskincfg.window_height);	
	gdk_draw_pixmap(ledsoffrev_pixmap, gc, analyzer_pixmap, 0, msaskincfg.window_height*msaskincfg.bands_off_rev_no, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsonrev_pixmap,  gc, analyzer_pixmap, 0, msaskincfg.window_height*msaskincfg.bands_on_rev_no,  0, 0, msaskincfg.window_width, msaskincfg.window_height);
	ledsoffmir_pixmap = ledsoff_pixmap;
	ledsonmir_pixmap  = ledson_pixmap;
	ledsoffrevmir_pixmap = ledsoffrev_pixmap;
	ledsonrevmir_pixmap  = ledsonrev_pixmap;

	unrefpixmap(&analyzer_pixmap);
}

void set_skin_from_file_v2(GdkWindow *window, GdkGC *gc, gchar *skindir)
{
	GdkPixmap *analyzer_pixmap = NULL;
	
	char file_analyzer[MSA_MAX_PATH_SIZE], file_titlebar[MSA_MAX_PATH_SIZE];
	gboolean stat = TRUE;

	DPRINT(1, "set_skin_from_file_v2(%p,%p,%s)\n",window,gc,skindir);

	msaskin_unref_skin_pixmaps();

	strncpy(file_analyzer, msacfg.skindir, MSA_MAX_PATH_SIZE);
	strncat(file_analyzer, "/analyzer", MSA_MAX_PATH_SIZE);
	strncpy(file_titlebar, msacfg.skindir, MSA_MAX_PATH_SIZE);
	strncat(file_titlebar, "/titlebar", MSA_MAX_PATH_SIZE);

	stat =	(titlebar_pixmap = loadimage(window, file_titlebar))!=NULL &&
		(analyzer_pixmap = loadimage(window, file_analyzer))!=NULL;

	if (!stat) {
		unrefpixmap(&titlebar_pixmap);
		unrefpixmap(&analyzer_pixmap);
		msaskin_setdefaultcfg(&msaskincfg);
		set_default_skin_pixmaps(window, gc, DEFAULT_SKIN_COLOR);
		return;
	}

	logo_pixmap       = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoff_pixmap    = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledson_pixmap     = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoffrev_pixmap = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsonrev_pixmap  = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());

	ledsoffmir_pixmap    = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsonmir_pixmap     = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsoffrevmir_pixmap = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());
	ledsonrevmir_pixmap  = gdk_pixmap_new(window, msaskincfg.window_width, msaskincfg.window_height, gdk_visual_get_best_depth());

	gdk_draw_pixmap(logo_pixmap,       gc, analyzer_pixmap, 0, msaskincfg.window_height*msaskincfg.logo_no,          0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsoff_pixmap,    gc, analyzer_pixmap,
		msaskincfg.window_width*0+0, msaskincfg.window_height*1+1,     0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledson_pixmap,     gc, analyzer_pixmap,
		msaskincfg.window_width*1+1, msaskincfg.window_height*1+1,      0, 0, msaskincfg.window_width, msaskincfg.window_height);	
	gdk_draw_pixmap(ledsoffrev_pixmap, gc, analyzer_pixmap,
		msaskincfg.window_width*0+0, msaskincfg.window_height*2+2, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsonrev_pixmap,  gc, analyzer_pixmap,
		msaskincfg.window_width*1+1, msaskincfg.window_height*2+2,  0, 0, msaskincfg.window_width, msaskincfg.window_height);

	gdk_draw_pixmap(ledsoffmir_pixmap,    gc, analyzer_pixmap,
		msaskincfg.window_width*0+0, msaskincfg.window_height*3+3,     0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsonmir_pixmap,     gc, analyzer_pixmap,
		msaskincfg.window_width*1+1, msaskincfg.window_height*3+3,      0, 0, msaskincfg.window_width, msaskincfg.window_height);	
	gdk_draw_pixmap(ledsoffrevmir_pixmap, gc, analyzer_pixmap,
		msaskincfg.window_width*0+0, msaskincfg.window_height*4+4, 0, 0, msaskincfg.window_width, msaskincfg.window_height);
	gdk_draw_pixmap(ledsonrevmir_pixmap,  gc, analyzer_pixmap,
		msaskincfg.window_width*1+1, msaskincfg.window_height*4+4,  0, 0, msaskincfg.window_width, msaskincfg.window_height);

	unrefpixmap(&analyzer_pixmap);
}

void set_skin_from_file(GdkWindow *window, GdkGC *gc, gchar *skindir)
{
	DPRINT(1, "set_skin_from_file(%p,%p,%s)\n",window,gc,skindir);

	if (msaskin_loadskinconfig(&msaskincfg, skindir)==FALSE) {
		set_default_skin_pixmaps(window, gc, DEFAULT_SKIN_COLOR);
		return;
	}
	
	DPRINT(1, "msa [set_skin_from_file]: ver = %d\n",msaskincfg.ver);
	
	if (msaskincfg.ver==2) set_skin_from_file_v2(window, gc, skindir);
	else set_skin_from_file_v1(window, gc, skindir);
}

void set_default_skin(GdkWindow *window, GdkGC *gc)
{
	gchar *colorspec;
	int len;

	DPRINT(1, "set_default_skin(%p,%p)\n",window,gc);
	
	len=strlen(msacfg.skindir);
	if (len==0) {
		colorspec = g_strdup_printf("%02x%02x%02x-%02x%02x%02x-%02x%02x%02x-%02x%02x%02x",
			msacfg.color[0][0], msacfg.color[0][1], msacfg.color[0][2],
			msacfg.color[1][0], msacfg.color[1][1], msacfg.color[1][2],
			msacfg.color[2][0], msacfg.color[2][1], msacfg.color[2][2],
			msacfg.color[3][0], msacfg.color[3][1], msacfg.color[3][2]);
		msaskin_setdefaultcfg(&msaskincfg);
		set_default_skin_pixmaps(window, gc, colorspec);
		g_free(colorspec);
	}
	else if (len>=29 && msacfg.skindir[len-29]=='[' && msacfg.skindir[len-1]==']') {
		colorspec=&(msacfg.skindir[len-28]);
		msaskin_setdefaultcfg(&msaskincfg);
		set_default_skin_pixmaps(window, gc, colorspec);
	}
	else {
		DPRINT(1, "msa [set_default_skin]: ver = %d\n", msaskincfg.ver);

		set_skin_from_file(window, gc, msacfg.skindir);
	}
}

void msaskin_set_default_border_pixmap(GdkGC *gc, GdkPixmap *pixmap)
{
	int ww=msaskincfg.window_width;
	int wh=msaskincfg.window_height;
	GdkColor color;

	DPRINT(1, "msaskin_set_default_border_pixmap(%p,%p)\n",gc,pixmap);
	
	if (gc==NULL || pixmap==NULL) return;

	msaskin_color_parse("000000", &color);
	gdk_color_alloc(gdk_colormap_get_system(), &color);
	gdk_gc_set_foreground(gc,&color);
	gdk_draw_rectangle(pixmap, gc, TRUE, 0, 0, ww, wh);

	msaskin_color_parse("313439", &color);
	gdk_color_alloc(gdk_colormap_get_system(), &color);
	gdk_gc_set_foreground(gc,&color);
	gdk_draw_line(pixmap, gc, 3,    3,    ww-2, 3);
	gdk_draw_line(pixmap, gc, 3,    3,    3,    wh-2);
	gdk_draw_line(pixmap, gc, 4,    wh-2, ww-2, wh-2);
	gdk_draw_line(pixmap, gc, ww-2, 4,    ww-2, wh-2);

	msaskin_color_parse("4a5052", &color);
	gdk_color_alloc(gdk_colormap_get_system(), &color);
	gdk_gc_set_foreground(gc,&color);
	gdk_draw_line(pixmap, gc, 2,    2,    ww-2, 2);
	gdk_draw_line(pixmap, gc, 2,    2,    2,    wh-2);
	gdk_draw_line(pixmap, gc, 4,    wh-3, ww-3, wh-3);
	gdk_draw_line(pixmap, gc, ww-3, 4,    ww-3, wh-3);

	msaskin_color_parse("8b919c", &color);
	gdk_color_alloc(gdk_colormap_get_system(), &color);
	gdk_gc_set_foreground(gc,&color);
	gdk_draw_line(pixmap, gc, 1,    1,    ww-2, 1);
	gdk_draw_line(pixmap, gc, 1,    1,    1,    wh-2);
	gdk_draw_line(pixmap, gc, 4,    wh-4, ww-4, wh-4);
	gdk_draw_line(pixmap, gc, ww-4, 4,    ww-4, wh-4);
}

void msaskin_set_std_analyzer_pixmap(GdkGC *gc, msa_analyzer_colors *colors, GdkPixmap *off_pixmap, GdkPixmap *on_pixmap)
{
	int aw=msaskincfg.analyzer_width;
	int ah=msaskincfg.analyzer_height;
	int ax=msaskincfg.analyzer_x;
	int ay=msaskincfg.analyzer_y;
	int bw=msaskincfg.analyzer_width/msaskincfg.num_bands;
	int i, j;

	DPRINT(1, "msaskin_set_std_analyzer_pixmap(%p,%p,%p,%p)\n",gc,colors,off_pixmap,on_pixmap);
	
	gdk_color_alloc(gdk_colormap_get_system(), &colors->bg);
	gdk_gc_set_foreground(gc, &colors->bg);
	gdk_draw_rectangle(off_pixmap, gc, TRUE, ax, ay, aw, ah);
	gdk_draw_rectangle(on_pixmap,  gc, TRUE, ax, ay, aw, ah);

	// draw scale
	gdk_color_alloc(gdk_colormap_get_system(), &colors->med);
	gdk_gc_set_foreground(gc, &colors->med);
        for(i = 0; i <= aw; i+=4*bw) {
		gdk_draw_point(on_pixmap,  gc, ax+i-1, ay+ah+1);
		gdk_draw_point(off_pixmap, gc, ax+i-1, ay+ah+1);
	}
	for(i = 0; i <= aw; i+=4) {
		gdk_draw_point(on_pixmap,  gc, ax+i-1, ay+ah+2);
		gdk_draw_point(off_pixmap, gc, ax+i-1, ay+ah+2);
	}
        for(i = 0; i <= ah; i+=2) {
		gdk_draw_point(on_pixmap,  gc, ax-4,    ay+i-1);
		gdk_draw_point(on_pixmap,  gc, ax+aw+2, ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax-4,    ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax+aw+2, ay+i-1);
	}
        for(i = 0, j = 4; i <= ah; i+=j, j+=2) {
		if (i+j>ah) i=ah;
		gdk_draw_point(on_pixmap,  gc, ax-3,    ay+i-1);
		gdk_draw_point(on_pixmap,  gc, ax+aw+1, ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax-3,    ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax+aw+1, ay+i-1);
	}

	// leds off
	gdk_color_alloc(gdk_colormap_get_system(), &colors->low);
	gdk_gc_set_foreground(gc, &colors->low);
        for(i=0; i<ah; i+=2) gdk_draw_line(off_pixmap, gc, ax, ay+i, ax+aw-1, ay+i);

	// ledson - high
	gdk_color_alloc(gdk_colormap_get_system(), &colors->high);
	gdk_gc_set_foreground(gc, &colors->high);
        for(i=0; i<ah; i+=2) gdk_draw_line(on_pixmap, gc, ax, ay+i, ax+aw-1, ay+i);

	gdk_color_alloc(gdk_colormap_get_system(), &colors->bg);
	gdk_gc_set_foreground(gc, &colors->bg);
	for(i=bw-1; i<aw; i+=bw) {
		gdk_draw_line(on_pixmap,  gc, ax+i, ay, ax+i, ay+ah);
		gdk_draw_line(off_pixmap, gc, ax+i, ay, ax+i, ay+ah);
	}
}

void msaskin_set_std_rev_analyzer_pixmap(GdkGC *gc, msa_analyzer_colors *colors, GdkPixmap *off_pixmap, GdkPixmap *on_pixmap)
{
	int aw=msaskincfg.analyzer_width;
	int ah=msaskincfg.analyzer_height;
	int ax=msaskincfg.analyzer_x;
	int ay=msaskincfg.analyzer_y;
	int bw=msaskincfg.analyzer_width/msaskincfg.num_bands;
	int i, j;

	DPRINT(1, "msaskin_set_std_rev_analyzer_pixmap(%p,%p,%p,%p)\n",gc,colors,off_pixmap,on_pixmap);
	
	gdk_color_alloc(gdk_colormap_get_system(), &colors->bg);
	gdk_gc_set_foreground(gc, &colors->bg);
	gdk_draw_rectangle(off_pixmap, gc, TRUE, ax, ay, aw, ah);
	gdk_draw_rectangle(on_pixmap,  gc, TRUE, ax, ay, aw, ah);

	// draw scale
	gdk_color_alloc(gdk_colormap_get_system(), &colors->med);
	gdk_gc_set_foreground(gc, &colors->med);
        for(i = 0; i <= aw; i+=4*bw) {
		gdk_draw_point(on_pixmap,  gc, ax+i-1, ay+ah+1);
		gdk_draw_point(off_pixmap, gc, ax+i-1, ay+ah+1);
	}
	for(i = 0; i <= aw; i+=4) {
		gdk_draw_point(on_pixmap,  gc, ax+i-1, ay+ah+2);
		gdk_draw_point(off_pixmap, gc, ax+i-1, ay+ah+2);
	}
        for(i = 0; i <= ah; i+=2) {
		gdk_draw_point(on_pixmap,  gc, ax-4,    ay+i-1);
		gdk_draw_point(on_pixmap,  gc, ax+aw+2, ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax-4,    ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax+aw+2, ay+i-1);
	}
        for(i = ah, j = 4; i >= 0; i-=j, j+=2) {
		if (i-j<0) i=0;
		gdk_draw_point(on_pixmap,  gc, ax-3,    ay+i-1);
		gdk_draw_point(on_pixmap,  gc, ax+aw+1, ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax-3,    ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax+aw+1, ay+i-1);
	}

	// leds off
	gdk_color_alloc(gdk_colormap_get_system(), &colors->low);
	gdk_gc_set_foreground(gc, &colors->low);
        for(i=0; i<ah; i+=2) gdk_draw_line(off_pixmap, gc, ax, ay+i, ax+aw-1, ay+i);

	// ledson - high
	gdk_color_alloc(gdk_colormap_get_system(), &colors->high);
	gdk_gc_set_foreground(gc, &colors->high);
        for(i=0; i<ah; i+=2) gdk_draw_line(on_pixmap, gc, ax, ay+i, ax+aw-1, ay+i);

	gdk_color_alloc(gdk_colormap_get_system(), &colors->bg);
	gdk_gc_set_foreground(gc, &colors->bg);
	for(i=bw-1; i<aw; i+=bw) {
		gdk_draw_line(on_pixmap,  gc, ax+i, ay, ax+i, ay+ah);
		gdk_draw_line(off_pixmap, gc, ax+i, ay, ax+i, ay+ah);
	}
}

void msaskin_set_mirrored_analyzer_pixmap(GdkGC *gc, msa_analyzer_colors *colors, GdkPixmap *off_pixmap, GdkPixmap *on_pixmap)
{
	int aw=msaskincfg.analyzer_width;
	int ah=msaskincfg.analyzer_height;
	int ah23=2*msaskincfg.analyzer_height/3;
	int ax=msaskincfg.analyzer_x;
	int ay=msaskincfg.analyzer_y;
	int bw=msaskincfg.analyzer_width/msaskincfg.num_bands;
	int i, j;

	DPRINT(1, "msaskin_set_mirrored_analyzer_pixmap(%p,%p,%p,%p)\n",gc,colors,off_pixmap,on_pixmap);
	
	gdk_color_alloc(gdk_colormap_get_system(), &colors->bg);
	gdk_gc_set_foreground(gc, &colors->bg);
	gdk_draw_rectangle(off_pixmap, gc, TRUE, ax, ay, aw, ah);
	gdk_draw_rectangle(on_pixmap,  gc, TRUE, ax, ay, aw, ah);

	// draw scale
	gdk_color_alloc(gdk_colormap_get_system(), &colors->med);
	gdk_gc_set_foreground(gc, &colors->med);
	// horizontal
        for(i = 0; i <= aw; i+=4*bw) {
		gdk_draw_point(on_pixmap,  gc, ax+i-1, ay+ah+1);
		gdk_draw_point(off_pixmap, gc, ax+i-1, ay+ah+1);
	}
	for(i = 0; i <= aw; i+=4) {
		gdk_draw_point(on_pixmap,  gc, ax+i-1, ay+ah+2);
		gdk_draw_point(off_pixmap, gc, ax+i-1, ay+ah+2);
	}
	// vertical
        for(i = 0; i <= ah; i+=2) {
		gdk_draw_point(on_pixmap,  gc, ax-4,    ay+i-1);
		gdk_draw_point(on_pixmap,  gc, ax+aw+2, ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax-4,    ay+i-1);
		gdk_draw_point(off_pixmap, gc, ax+aw+2, ay+i-1);
	}
        for(i = ah23, j = 4; i > 0; i-=j, j+=2) {
		if (i-j<=0) i=-1;
		gdk_draw_point(on_pixmap,  gc, ax-3,    ay+i);
		gdk_draw_point(on_pixmap,  gc, ax+aw+1, ay+i);
		gdk_draw_point(off_pixmap, gc, ax-3,    ay+i);
		gdk_draw_point(off_pixmap, gc, ax+aw+1, ay+i);
	}
        for(i = ah23, j = 4; i < ah; i+=j, j+=2) {
		if (i+j>=ah) i=ah-1;
		gdk_draw_point(on_pixmap,  gc, ax-3,    ay+i);
		gdk_draw_point(on_pixmap,  gc, ax+aw+1, ay+i);
		gdk_draw_point(off_pixmap, gc, ax-3,    ay+i);
		gdk_draw_point(off_pixmap, gc, ax+aw+1, ay+i);
	}

	// leds off
	gdk_color_alloc(gdk_colormap_get_system(), &colors->low);
	gdk_gc_set_foreground(gc, &colors->low);
        for(i=0; i<ah; i+=2) gdk_draw_line(off_pixmap, gc, ax, ay+i, ax+aw-1, ay+i);

	// ledson - high
	gdk_color_alloc(gdk_colormap_get_system(), &colors->high);
	gdk_gc_set_foreground(gc, &colors->high);
        for(i=0; i<ah23; i+=2) gdk_draw_line(on_pixmap, gc, ax, ay+i, ax+aw-1, ay+i);

	// ledson - mirror
	gdk_color_alloc(gdk_colormap_get_system(), &colors->med);
	gdk_gc_set_foreground(gc, &colors->med);
	for (; i<ah; i+=2) gdk_draw_line(on_pixmap, gc, ax, ay+i, ax+aw-1, ay+i);

	gdk_color_alloc(gdk_colormap_get_system(), &colors->bg);
	gdk_gc_set_foreground(gc, &colors->bg);
	for(i=bw-1; i<aw; i+=bw) {
		gdk_draw_line(on_pixmap,  gc, ax+i, ay, ax+i, ay+ah);
		gdk_draw_line(off_pixmap, gc, ax+i, ay, ax+i, ay+ah);
	}
}

void msaskin_set_mirrored_rev_analyzer_pixmap(GdkGC *gc, msa_analyzer_colors *colors, GdkPixmap *off_pixmap, GdkPixmap *on_pixmap)
{
	DPRINT(1, "msaskin_set_mirrored_rev_analyzer_pixmap(%p,%p,%p,%p)\n",gc,colors,off_pixmap,on_pixmap);

	msaskin_set_mirrored_analyzer_pixmap(gc, colors, off_pixmap, on_pixmap);
}
