/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  Bump Scope - Visualization Plugin for XMMS
 *  Copyright (C) 1999 Zinx Verituse
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <pthread.h>
#include <stdlib.h>
#include "math.h"
#include <xmms/plugin.h>
#include <xmms/util.h>
#include <xmms/configfile.h>
#include <xmms/fullscreen.h>
#include <xmms/xmmsctrl.h>
#include "xmms_logo.h"
#include "bump_scope.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#define VERSION ""
#endif

pthread_mutex_t bumpscope_res_lock;
gboolean bumpscope_have_mutex = FALSE;

GtkWidget *bumpscope_window = NULL;
static GtkWidget *area = NULL;
static GtkItemFactory *bumpscope_menu = NULL;
static gint bumpscope_need_draw = 0;

static void bumpscope_init(void);
static void bumpscope_cleanup(void);
static void bumpscope_playback_start(void);
static void bumpscope_playback_stop(void);
static void bumpscope_render_pcm(gint16 data[2][512]);
static void bumpscope_about(void);
static void bumpscope_render_light(gint lx, gint ly);

BumpScopeConfig bumpscope_cfg;

VisPlugin bumpscope_vp;

VisPlugin *get_vplugin_info(void) {
	memset(&bumpscope_vp, 0, sizeof(bumpscope_vp));
	bumpscope_vp.description =		"Bump Scope "VERSION;
	bumpscope_vp.num_pcm_chs_wanted =	1;
	bumpscope_vp.num_freq_chs_wanted =	0;
	
	bumpscope_vp.init =			bumpscope_init;
	bumpscope_vp.cleanup =			bumpscope_cleanup;
	bumpscope_vp.configure =		bumpscope_configure;
	bumpscope_vp.playback_start =		bumpscope_playback_start;
	bumpscope_vp.playback_stop =		bumpscope_playback_stop;
	bumpscope_vp.render_pcm =		bumpscope_render_pcm;
	return &bumpscope_vp;
}

#define BUMPSCOPE_MENU_COLORCYC 1
#define BUMPSCOPE_MENU_LIGHTMOV 2
#define BUMPSCOPE_MENU_DIAMONDL 3
#define BUMPSCOPE_MENU_AUTOFULL 4
#define BUMPSCOPE_MENU_FULLSCRN 5
#define BUMPSCOPE_MENU_CONFIGUR 6
#define BUMPSCOPE_MENU_ABOUT    7

void bumpscope_menu_cb(gpointer cb_data, guint action, GtkWidget *w);

#define BUMPSCOPE_MENU_ENTRIES 9
GtkItemFactoryEntry bumpscope_menu_entries[] = {
	{ "/Color cycling",	NULL, bumpscope_menu_cb, BUMPSCOPE_MENU_COLORCYC,	"<ToggleItem>" },
	{ "/Moving light",	NULL, bumpscope_menu_cb, BUMPSCOPE_MENU_LIGHTMOV,	"<ToggleItem>" },
	{ "/Diamond light",	NULL, bumpscope_menu_cb, BUMPSCOPE_MENU_DIAMONDL,	"<ToggleItem>" },
	{ "/Auto Fullscreen",	NULL, bumpscope_menu_cb, BUMPSCOPE_MENU_AUTOFULL,	"<ToggleItem>" },
	{ "/Fullscreen",	NULL, bumpscope_menu_cb, BUMPSCOPE_MENU_FULLSCRN,	"<ToggleItem>" },
	{ "/-",			NULL, NULL,		  0,				"<Separator>" },
	{ "/Config dialog...",	NULL, bumpscope_menu_cb, BUMPSCOPE_MENU_CONFIGUR,	"<Item>" },
	{ "/-",			NULL, NULL,		  0,				"<Separator>" },
	{ "/About BumpScope",	NULL, bumpscope_menu_cb, BUMPSCOPE_MENU_ABOUT,		"<Item>" },
};

#define min(x,y) ((x)<(y)?(x):(y))
#define BPL	((MAX_WIDTH + 2))

static guchar phongdat[MAX_PHONGRES][MAX_PHONGRES];
static guchar rgb_buf[(MAX_WIDTH + 2) * (MAX_HEIGHT + 2)];
static guchar rgb_buf2[(MAX_WIDTH + 2) * (MAX_HEIGHT + 2)];
static gdouble intense1[256], intense2[256];
static GdkRgbCmap *cmap = NULL;
gint bumpscope_win_w, bumpscope_win_h;

#ifndef X86_ASM_OPT
void bumpscope_blur_8(guchar *ptr,gint w, gint h, gint bpl)
{
	register guint i,sum;
	register guchar *iptr;

	iptr = ptr + bpl + 1;
	i = bpl * h;
	while(i--)
	{
		sum = (iptr[-bpl] + iptr[-1] + iptr[1] + iptr[bpl]) >> 2;
		if(sum > 2)
			sum -= 2;
		*(iptr++) = sum;
	}


}
#else
extern void bumpscope_blur_8(guchar *ptr,gint w, gint h, gint bpl);
#endif

static void bumpscope_generate_intense() {
	guint32 i;
	gdouble in;

	for(i = 255; i > 0; i--)
	{
		intense1[i] = cos(((gdouble)(255-i)*M_PI)/512.0);
		intense2[i] = pow(intense1[i], 250)*150;
	}
	intense1[0] = intense1[1];
	intense2[0] = intense2[1];
}

void bumpscope_generate_cmap(guint32 color)
{
	guint32 colors[256],i,red,blue,green,r,g,b;
	if (bumpscope_window) {
		red = (guint32)(color / 0x10000);
		green = (guint32)((color % 0x10000)/0x100);
		blue = (guint32)(color % 0x100);

		for(i = 255; i > 0; i--)
		{
			r = ((gdouble)(100*red/255)*intense1[i]+intense2[i]);
			if (r > 255) r = 255;
			g = ((gdouble)(100*green/255)*intense1[i]+intense2[i]);
			if (g > 255) g = 255;
			b = ((gdouble)(100*blue/255)*intense1[i]+intense2[i]);
			if (b > 255) b = 255;

			colors[i] = (((guint32)(r) << 16) | ((guint32)(g) << 8) | ((guint32)(b)));
		}
		colors[0] = colors[1];

		/* This function is always called inside a BUMP_LOCK() */
		if (cmap) gdk_rgb_cmap_free(cmap);
		cmap = gdk_rgb_cmap_new(colors, 256);

		bumpscope_need_draw = 1;
	}
}

void bumpscope_generate_phongdat() {
	gint y, x;
	gdouble i, i2;

	for (y = 0; y < (PHONGRAD); y++) {
		for (x = 0; x < (PHONGRAD); x++) {
			i = (gdouble)x/((gdouble)PHONGRAD)-1;
			i2 = (gdouble)y/((gdouble)PHONGRAD)-1;
			if (bumpscope_cfg.diamond)
				i = 1 - pow(i*i2,.75) - i*i - i2*i2;
			else
				i = 1 - i*i - i2*i2;

			if (i >= 0) {
				if (bumpscope_cfg.diamond)
					i = i*i*i * 255.0;
				else
					i = i*i*i * 255.0;

				if (i > 255) i = 255;
				phongdat[y][x] = i;
				phongdat[(PHONGRES-1)-y][x] = i;
				phongdat[y][(PHONGRES-1)-x] = i;
				phongdat[(PHONGRES-1)-y][(PHONGRES-1)-x] = i;
			} else {
				phongdat[y][x] = 0;
				phongdat[(PHONGRES-1)-y][x] = 0;
				phongdat[y][(PHONGRES-1)-x] = 0;
				phongdat[(PHONGRES-1)-y][(PHONGRES-1)-x] = 0;
			}
		}
	}

	bumpscope_need_draw = 1;
}

/* this is _very_ unoptimized.. bumpscope_xmms_logo should be
   in xmms_logo.h as a GIMP C-Source style image */
void bumpscope_draw_xmms_logo() {
	gint x, y, xd, yd;

	memset(rgb_buf, 0, (MAX_WIDTH+2)*(MAX_HEIGHT+2));
	for (y = 1, yd = -(((gint)(HEIGHT+2)-(gint)bumpscope_xmms_logo.height)/(gint)2);
		y < (HEIGHT+1); y++, yd++) {

		for (x = 1, xd = -(((gint)(WIDTH+2)-(gint)bumpscope_xmms_logo.width)/(gint)2);
			x < (WIDTH+1); x++, xd++) {

			if (xd>=0 && xd<bumpscope_xmms_logo.width &&
			    yd>=0 && yd<bumpscope_xmms_logo.height) {
			    	rgb_buf[y*BPL+x] =
bumpscope_xmms_logo.pixel_data[(yd*bumpscope_xmms_logo.width+xd)*bumpscope_xmms_logo.bytes_per_pixel];
			} else {
				rgb_buf[y*BPL+x] = 0;
			}
		}
	}

	bumpscope_need_draw = 1;
}

static pthread_t main_thread;

static void bumpscope_translate(gint x, gint y, gint *xo, gint *yo, gint *xd, gint *yd, gint *angle) {
	/* try setting y to both maxes */
	*yo = HEIGHT/2;
	*angle = asin((float)(y-(HEIGHT/2))/(float)*yo)/(M_PI/180.0);
	*xo = (x-(WIDTH/2))/cos(*angle*(M_PI/180.0));

	if (*xo>=-(WIDTH/2) && *xo<=(WIDTH/2)) {
		*xd = (*xo>0)?-1:1;
		*yd = 0;
		return;
	}

	*yo = -*yo;
	*angle = asin((float)(y-(HEIGHT/2))/(float)*yo)/(M_PI/180.0);
	*xo = (x-(WIDTH/2))/cos(*angle*(M_PI/180.0));

	if (*xo>=-(WIDTH/2) && *xo<=(WIDTH/2)) {
		*xd = (*xo>0)?-1:1;
		*yd = 0;
		return;
	}

	/* try setting x to both maxes */
	*xo = WIDTH/2;
	*angle = acos((float)(x-(WIDTH/2))/(float)*xo)/(M_PI/180.0);
	*yo = (y-(HEIGHT/2))/sin(*angle*(M_PI/180.0));

	if (*yo>=-(HEIGHT/2) && *yo<=(HEIGHT/2)) {
		*yd = (*yo>0)?-1:1;
		*xd = 0;
		return;
	}

	*xo = -*xo;
	*angle = acos((float)(x-(WIDTH/2))/(float)*xo)/(M_PI/180.0);
	*yo = (y-(HEIGHT/2))/sin(*angle*(M_PI/180.0));

	/* if this isn't right, it's out of our range and we don't care */
	*yd = (*yo>0)?-1:1;
	*xd = 0;
}

/* rgb <-> hsv conversion from gtkcolorsel.c */
static void bumpscope_rgb_to_hsv (guint32 color,
	    gdouble *h, gdouble *s, gdouble *v)
{
  gdouble max, min, delta, r, g, b;

  r = (gdouble)(color>>16) / 255.0;
  g = (gdouble)((color>>8)&0xff) / 255.0;
  b = (gdouble)(color&0xff) / 255.0;

  max = r;
  if (g > max) max = g;
  if (b > max) max = b;

  min = r;
  if (g < min) min = g;
  if (b < min) min = b;

  *v = max;

  if (max != 0.0) *s = (max - min) / max;
  else *s = 0.0;

  if (*s == 0.0) *h = 0.0;
  else
    {
      delta = max - min;

      if (r == max) *h = (g - b) / delta;
      else if (g == max) *h = 2.0 + (b - r) / delta;
      else if (b == max) *h = 4.0 + (r - g) / delta;

      *h = *h * 60.0;

      if (*h < 0.0) *h = *h + 360;
    }
}

static void bumpscope_hsv_to_rgb (gdouble  h, gdouble  s, gdouble  v,
                                guint32 *color)
{
  gint i;
  gdouble f, w, q, t, r, g, b;

  if (s == 0.0)
    s = 0.000001;

  if (h == -1.0)
    {
      r = v; g = v; b = v;
    }
  else
    {
      if (h == 360.0) h = 0.0;
      h = h / 60.0;
      i = (gint) h;
      f = h - i;
      w = v * (1.0 - s);
      q = v * (1.0 - (s * f));
      t = v * (1.0 - (s * (1.0 - f)));

      switch (i)
        {
        case 0: r = v; g = t; b = w; break;
        case 1: r = q; g = v; b = w; break;
        case 2: r = w; g = v; b = t; break;
        case 3: r = w; g = q; b = v; break;
        case 4: r = t; g = w; b = v; break;
        /*case 5: use default to keep gcc from complaining */
	default: r = v; g = w; b = q; break;
        }
    }

  *color = ((guint32)((gdouble)r*255)<<16) | ((guint32)((gdouble)g*255)<<8) | ((guint32)((gdouble)b*255));
}

static void bumpscope_destroy_cb(GtkWidget *w,gpointer data) {
	bumpscope_vp.disable_plugin(&bumpscope_vp);
}

static int mouse_move = 0;
static int am_fullscreen = 0;

void bumpscope_menu_cb(gpointer cb_data, guint action, GtkWidget *w) {
	switch (action) {
		case BUMPSCOPE_MENU_ABOUT:
			bumpscope_about(); break;
		case BUMPSCOPE_MENU_CONFIGUR:
			bumpscope_configure();
			break;
		case BUMPSCOPE_MENU_COLORCYC:
			bumpscope_cfg.color_cycle = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Color cycling"))->active;
			if (!bumpscope_cfg.color_cycle) {
				BUMP_LOCK();
				bumpscope_generate_cmap(bumpscope_cfg.color);
				BUMP_UNLOCK();
			}
			bumpscope_write_cfg();
			break;
		case BUMPSCOPE_MENU_LIGHTMOV:
			bumpscope_cfg.moving_light = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Moving light"))->active;
			bumpscope_write_cfg();
			break;
		case BUMPSCOPE_MENU_DIAMONDL:
			bumpscope_cfg.diamond = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Diamond light"))->active;
			bumpscope_generate_phongdat();
			bumpscope_write_cfg();
			break;
		case BUMPSCOPE_MENU_AUTOFULL:
			bumpscope_cfg.auto_fullscreen = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Auto Fullscreen"))->active;
			bumpscope_write_cfg();
			break;
		case BUMPSCOPE_MENU_FULLSCRN:
			am_fullscreen = !am_fullscreen;
			bumpscope_win_w = WIDTH; bumpscope_win_h = HEIGHT;
			if (am_fullscreen) {
				am_fullscreen = xmms_fullscreen_enter(bumpscope_window, &bumpscope_win_w, &bumpscope_win_h);
			} else {
				xmms_fullscreen_leave(bumpscope_window);
			}
			GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Fullscreen"))->active = am_fullscreen;
			break;
		default:
			/* ugh */
	}
}


static void bumpscope_keypress_cb(GtkWidget *w, GdkEventKey *event, gpointer data) {
	switch (event->keyval) {
		case GDK_Z:
		case GDK_z:
			xmms_remote_playlist_prev(bumpscope_vp.xmms_session);
			break;
		case GDK_X:
		case GDK_x:
			xmms_remote_play(bumpscope_vp.xmms_session);
			break;
		case GDK_C:
		case GDK_c:
			xmms_remote_pause(bumpscope_vp.xmms_session);
			break;
		case GDK_V:
		case GDK_v:
			xmms_remote_stop(bumpscope_vp.xmms_session);
			break;
		case GDK_B:
		case GDK_b:
			xmms_remote_playlist_next(bumpscope_vp.xmms_session);
			break;
	}
}

static void bumpscope_press_cb(GtkWidget *w, GdkEventButton *event, gpointer data) {
	if (event->button == 1) {
		mouse_move = 1;

		bumpscope_cfg.x = event->x-(bumpscope_win_w-WIDTH)/2;
		bumpscope_cfg.y = event->y-(bumpscope_win_h-HEIGHT)/2;
		if (!bumpscope_cfg.moving_light) bumpscope_need_draw = 1;
	} else if (event->button == 3) {
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Color cycling"))->active = bumpscope_cfg.color_cycle;
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Moving light"))->active = bumpscope_cfg.moving_light;
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Diamond light"))->active = bumpscope_cfg.diamond;
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Auto Fullscreen"))->active = bumpscope_cfg.auto_fullscreen;
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(bumpscope_menu, "/Fullscreen"))->active = am_fullscreen;

		gtk_widget_set_sensitive(GTK_WIDGET(gtk_item_factory_get_widget(bumpscope_menu, "/About BumpScope")), !am_fullscreen);
		gtk_widget_set_sensitive(GTK_WIDGET(gtk_item_factory_get_widget(bumpscope_menu, "/Config dialog...")), !am_fullscreen);
		util_item_factory_popup(bumpscope_menu, (guint)event->x_root, (guint)event->y_root + 2, 3, GDK_CURRENT_TIME);
	}
}

static void bumpscope_release_cb(GtkWidget *w, GdkEventButton *event, gpointer data) {
	if (event->button == 2) {
		bumpscope_win_w = WIDTH; bumpscope_win_h = HEIGHT;
		if (!am_fullscreen) {
			am_fullscreen = xmms_fullscreen_enter(w, &bumpscope_win_w, &bumpscope_win_h);
		} else {
			xmms_fullscreen_leave(w);
			am_fullscreen = 0;
		}
		return;
	}

	if (event->button != 1) return;
	mouse_move = 0;
	bumpscope_write_cfg();
}

static void bumpscope_motion_cb(GtkWidget *w, GdkEventMotion *event, gpointer data) {
	if (!mouse_move) return;

	bumpscope_cfg.x = event->x-(bumpscope_win_w-WIDTH)/2;
	bumpscope_cfg.y = event->y-(bumpscope_win_h-HEIGHT)/2;
	if (!bumpscope_cfg.moving_light) bumpscope_need_draw = 1;
}

static gboolean bumpscope_expose_cb(GtkWidget *w, GdkEventExpose *event, gpointer data) {
	bumpscope_need_draw = 1;
	return TRUE;
}

static void do_gtk_init() {
	GdkColor black = {0, 0, 0, 0};

	bumpscope_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(bumpscope_window),"Bump Scope");
	gtk_window_set_policy(GTK_WINDOW(bumpscope_window), FALSE, FALSE, FALSE);
	gtk_widget_set_events(bumpscope_window,GDK_BUTTON_MOTION_MASK|GDK_BUTTON_PRESS_MASK|GDK_BUTTON_RELEASE_MASK|GDK_KEY_PRESS_MASK);
	gtk_widget_set_usize(bumpscope_window, WIDTH, HEIGHT);
	gtk_widget_realize(bumpscope_window);

	gtk_signal_connect(GTK_OBJECT(bumpscope_window),"destroy",GTK_SIGNAL_FUNC(bumpscope_destroy_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(bumpscope_window),"key-press-event",GTK_SIGNAL_FUNC(bumpscope_keypress_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(bumpscope_window),"button-press-event",GTK_SIGNAL_FUNC(bumpscope_press_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(bumpscope_window),"button-release-event",GTK_SIGNAL_FUNC(bumpscope_release_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(bumpscope_window),"motion-notify-event",GTK_SIGNAL_FUNC(bumpscope_motion_cb),NULL);

	area = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(bumpscope_window), area);
	gtk_widget_realize(area);

	gtk_signal_connect(GTK_OBJECT(area),"expose-event",GTK_SIGNAL_FUNC(bumpscope_expose_cb),NULL);

	gdk_window_set_background(area->window, &black);

	bumpscope_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<BumpScope>", NULL);
	gtk_item_factory_create_items(bumpscope_menu, BUMPSCOPE_MENU_ENTRIES, bumpscope_menu_entries, NULL);

	bumpscope_win_w = WIDTH; bumpscope_win_h = HEIGHT;

	gtk_widget_show(area);
	gtk_widget_show(bumpscope_window);

	gdk_window_clear(area->window);
}

static int bumpscope_quit = 0;

static void *bumpscope_main_thread(void *arg) {
	gint angle, xo, yo, xd, yd, lx, ly;
	gint was_moving = 0, was_color = 0;
	gdouble h, s, v, sd = 0;
	gint hd = 0;
	guint32 color;

	/*
	 * Hack to keep bump scope from crashing xmms while
	 * xmms creates its windows
	 */
	while (!gtk_main_level())
		xmms_usleep(10000);

	for (;!bumpscope_quit;) {
		lx = bumpscope_cfg.x;
		ly = bumpscope_cfg.y;
		if (bumpscope_cfg.moving_light) {
			if (!was_moving) {
				bumpscope_translate(lx, ly, &xo, &yo, &xd, &yd, &angle);
				was_moving = 1;
			}

			lx = WIDTH/2+cos(angle*(M_PI/180.0))*xo;
			ly = HEIGHT/2+sin(angle*(M_PI/180.0))*yo;

			bumpscope_need_draw = 1;

			angle += 2; if (angle >= 360) angle = 0;

			xo += xd;
			if ((gint)xo > ((gint)WIDTH/(gint)2) || (gint)xo < -((gint)WIDTH/(gint)2)) {
				xo = (xo>0)?(WIDTH/2):-(WIDTH/2);
				if (random()&1) {
					xd = (xd>0)?-1:1;
					yd = 0;
				} else {
					yd = (yd>0)?-1:1;
					xd = 0;
				}
			}

			yo += yd;
			if ((gint)yo > ((gint)HEIGHT/(gint)2) || (gint)yo < -((gint)HEIGHT/(gint)2)) {
				yo = (yo>0)?(HEIGHT/2):-(HEIGHT/2);
				if (random()&1) {
					xd = (xd>0)?-1:1;
					yd = 0;
				} else {
					yd = (yd>0)?-1:1;
					xd = 0;
				}
			}
		} else {
			if (was_moving) {
				bumpscope_need_draw = 1;
				was_moving = 0;
			}
		}

		if (bumpscope_cfg.color_cycle) {
			if (!was_color) {
				bumpscope_rgb_to_hsv(bumpscope_cfg.color, &h, &s, &v);
				was_color = 1;

				if (random()&1) {
					hd = (random()&1)*2-1;
					sd = 0;
				} else {
					sd = 0.01 * ((random()&1)*2-1);
					hd = 0;
				}
			}

			bumpscope_hsv_to_rgb(h, s, v, &color);

			GDK_THREADS_ENTER();
			BUMP_LOCK();
			bumpscope_generate_cmap(color);
			BUMP_UNLOCK();
			GDK_THREADS_LEAVE();

			if (hd) {
				h += hd;
				if (h >= 360) h = 0;
				if (h < 0) h = 359;
				if ((random()%150) == 0) {
					if (random()&1) {
						hd = (random()&1)*2-1;
						sd = 0;
					} else {
						sd = 0.01 * ((random()&1)*2-1);
						hd = 0;
					}
				}
			} else {
				s += sd;

				if (s <= 0 || s >= 0.5) {
					if (s < 0) s = 0;

					if (s > 0.52) {
						sd = -0.01;
					} else if (s == 0) {
						h = random()%360;
						sd = 0.01;
					} else {
						if (random()&1) {
							hd = (random()&1)*2-1;
							sd = 0;
						} else {
							sd = 0.01 * ((random()&1)*2-1);
							hd = 0;
						}
					}
				}
			}
		} else if (was_color) {
			GDK_THREADS_ENTER();
			BUMP_LOCK();
			bumpscope_generate_cmap(bumpscope_cfg.color);
			BUMP_UNLOCK();
			GDK_THREADS_LEAVE();
			was_color = 0;
		}

		if (bumpscope_need_draw) {
			bumpscope_need_draw = 0;
			bumpscope_render_light(lx, ly);
		}

		/* This should make it run at a bit below 50fps
			(1000000/20000 == 50) */
		xmms_usleep(20000);
	}

	pthread_exit(NULL);
}

static void bumpscope_init(void)
{
	if (bumpscope_window) return;

	if (!bumpscope_have_mutex) {
		pthread_mutex_init(&bumpscope_res_lock, NULL);
		bumpscope_have_mutex = TRUE;
	}

	bumpscope_read_config();
	bumpscope_quit = 0;

	BUMP_LOCK();
	do_gtk_init();
	bumpscope_generate_phongdat();
	bumpscope_generate_intense();
	bumpscope_generate_cmap(bumpscope_cfg.color);
	BUMP_UNLOCK();

	bumpscope_draw_xmms_logo();

	if (!xmms_fullscreen_init(bumpscope_window)) {
		gtk_widget_set_sensitive(GTK_WIDGET(gtk_item_factory_get_widget(bumpscope_menu, "/Auto Fullscreen")), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(gtk_item_factory_get_widget(bumpscope_menu, "/Fullscreen")), FALSE);
	}

	pthread_create(&main_thread, NULL, bumpscope_main_thread, NULL);
}

static void bumpscope_cleanup(void)
{
	if (bumpscope_quit) return;

	GDK_THREADS_LEAVE();
	bumpscope_quit = 1;
	pthread_join(main_thread, NULL);
	GDK_THREADS_ENTER();

	bumpscope_write_cfg();

	if (bumpscope_window)
		xmms_fullscreen_leave(bumpscope_window);

	xmms_fullscreen_cleanup(bumpscope_window);

	BUMP_LOCK();
	if (bumpscope_window) {
		gtk_widget_destroy(bumpscope_window);
		gtk_object_destroy(GTK_OBJECT(bumpscope_menu));
		bumpscope_window = NULL;
	}

	if (cmap) {
		gdk_rgb_cmap_free(cmap);
		cmap = NULL;
	}
	BUMP_UNLOCK();
}

static void bumpscope_playback_start(void) {
	BUMP_LOCK();
	if (GTK_WIDGET_REALIZED(bumpscope_window) && bumpscope_cfg.auto_fullscreen) {
		bumpscope_win_w = WIDTH; bumpscope_win_h = HEIGHT;
		am_fullscreen = xmms_fullscreen_enter(bumpscope_window, &bumpscope_win_w, &bumpscope_win_h);
	}
	BUMP_UNLOCK();
}

static void bumpscope_playback_stop(void) {
	BUMP_LOCK();
	if (GTK_WIDGET_REALIZED(bumpscope_window) && bumpscope_cfg.auto_fullscreen) {
		bumpscope_win_w = WIDTH; bumpscope_win_h = HEIGHT;
		xmms_fullscreen_leave(bumpscope_window);
		am_fullscreen = 0;
	}
	BUMP_UNLOCK();
	bumpscope_draw_xmms_logo();
}

static inline void draw_vert_line(guchar *buffer, gint x, gint y1, gint y2)
{
	int y;
	guchar *p;
	if(y1 < y2)
	{
		p = buffer+((y1+1)*BPL)+x+1;
		for(y = y1; y <= y2; y++) {
			*p = 0xff;
			p += BPL;
		}
	}
	else if(y2 < y1)
	{
		p = buffer+((y2+1)*BPL)+x+1;
		for(y = y2; y <= y1; y++) {
			*p = 0xff;
			p += BPL;
		}
	}
	else {
		buffer[((y1+1)*BPL)+x+1] = 0xff;
	}
}

static void bumpscope_render_light(gint lx, gint ly) {
	gint i, j, prev_y, dy, dx, xq, yq;

	prev_y = BPL + 1;
	for (dy = (-ly)+(PHONGRES/2), j = 0; j < HEIGHT; j++, dy++, prev_y+=BPL-WIDTH) {
		for (dx = (-lx)+(PHONGRES/2), i = 0; i < WIDTH; i++, dx++, prev_y++) {
			xq = (rgb_buf[prev_y-1]-rgb_buf[prev_y+1])+dx;
			yq = (rgb_buf[prev_y-BPL]-rgb_buf[prev_y+BPL])+dy;
			if (yq<0 || yq>=PHONGRES ||
			    xq<0 || xq>=PHONGRES) {
				rgb_buf2[prev_y] = 0;
			    	continue;
			}
			rgb_buf2[prev_y] = phongdat[yq][xq];
		}
	}

	/* GDK LOCK MUST COME FIRST. (unless you do some nasty hacks) */
	GDK_THREADS_ENTER();
	BUMP_LOCK();
	if (GTK_WIDGET_REALIZED(bumpscope_window)) {
		gdk_draw_indexed_image(area->window,area->style->white_gc,(bumpscope_win_w-WIDTH)/2,(bumpscope_win_h-HEIGHT)/2,WIDTH,HEIGHT,GDK_RGB_DITHER_NONE,rgb_buf2 + BPL + 1, BPL, cmap);
	}
	BUMP_UNLOCK();
	GDK_THREADS_LEAVE();
}

static void bumpscope_render_pcm(gint16 data[2][512])
{
	gint i, y, prev_y;

	prev_y = (gint)HEIGHT/(gint)2 + ((gint)data[0][0]*(gint)HEIGHT)/(gint)0x10000;

	if (prev_y < 0) prev_y = 0;
	if (prev_y >= HEIGHT) prev_y = HEIGHT-1;

	for(i = 0; i < WIDTH; i++)
	{
		y = (i*511)/(WIDTH-1);

		y = (gint)HEIGHT/(gint)2 + ((gint)data[0][y]*(gint)HEIGHT)/(gint)0x10000;

		/* This should _never_ happen (and doesn't), but I test anyway. */
		if (y < 0) y = 0;
		if(y >= HEIGHT) y = HEIGHT - 1;

		draw_vert_line(rgb_buf,i,prev_y,y);
		prev_y = y;
	}
	bumpscope_blur_8(rgb_buf, WIDTH, HEIGHT, BPL);

	bumpscope_need_draw = 1;
	return;
}

static void bumpscope_about(void) {
	GtkWidget *dialog, *button, *label;

	dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog), "About Bump Scope "VERSION);
	gtk_container_border_width(GTK_CONTAINER(dialog), 5);
	label = gtk_label_new(
"Bump Scope - Visualization Plugin for XMMS\n\
by Zinx Verituse <zinx@xmms.org>\n\
Copyright (C) 1999-2001 Zinx Verituse\n\
\n\
This program is free software; you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation; either version 2 of the License, or\n\
(at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with this program; if not, write to the Free Software\n\
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307  USA"
);

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	button = gtk_button_new_with_label(" Close ");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	gtk_widget_show(dialog);
	gtk_widget_grab_focus(button);
}
