#!/bin/sh
#
# @(#)install.sh	6.125 00/01/14
#
# Script to install the software binary and support files on
# the target system.
#
# Command line options:
#
# -n	Do not clean up distribution files after installing
# -b	Run in non-interactive mode, use default answers for all prompts.
#	The following environment variables can be used to override the
#	default answers:
#	BATCH_BINDIR		Binary executable installation directory
#	BATCH_LIBDIR		X11 library directory
#	BATCH_XMCDLIB		Xmcd support files top level directory
#	BATCH_CDDBDIR		Local CD database top level directory
#	BATCH_DISCOGDIR		Local discographies top level directory
#	BATCH_MANDIR		Manual page top level directory
#	BATCH_MANSUFFIX		Manual page file suffix
#
#    xmcd  - Motif(tm) CD Audio Player
#    cda   - Command-line CD Audio Player
#
#    Copyright (C) 1993-2000  Ti Kan
#    E-mail: ti@amb.org
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

OPATH="$PATH"
PATH=/bin:/usr/bin:/sbin:/usr/sbin:/etc:/usr/local/bin
export PATH

XMCD_VER=2.6
DEMODB=4102560a
XMCD_URL=http://metalab.unc.edu/tkan/xmcd/
DIRPERM=755
SCRPERM=755
FILEPERM=444
BINPERM=711
XBINPERM=4711
OWNER=bin
GROUP=bin
BINOWNER=bin
BINGROUP=bin
XBINOWNER=root
XBINGROUP=bin
ERRFILE=/tmp/xmcd.err
TMPFILE=/tmp/xmcdinst.$$
STARTUP_SCRIPT=.xmcd_start

#
# Utility functions
#

do_exit()
{
	if [ $1 -eq 0 ]
	then
	    $ECHO "\nInstallation of xmcd is now complete.  Please read"
	    $ECHO "$XMCDLIB/docs/README for further information."
	    if [ -n "$BATCH" ]
	    then
		$ECHO "\nBefore using xmcd/cda for the first time, you must"
		$ECHO "set up the software by running the following program:"
		$ECHO "\n\t${XMCDLIB}/config/config.sh\n"
		$ECHO "Xmcd/cda will not work until that is done."
	    fi
	    $ECHO "\nYou may also want to set up cron(1M) to run the"
	    $ECHO "$XMCDLIB/scripts/genidx script periodically (e.g., nightly)"
	    $ECHO "to re-generate the Local Discography category index pages."
	else
	    $ECHO "\nErrors have occurred in the installation."
	    if [ $ERRFILE != /dev/null ]
	    then
		$ECHO "See $ERRFILE for an error log."
	    fi
	fi
	exit $1
}

log_err()
{
	if [ "$1" = "-p" ]
	then
		$ECHO "Error: $2"
	fi
	$ECHO "$2" >>$ERRFILE
	ERROR=1
}

get_str()
{
	$ECHO "$* \c"
	read ANS
	if [ -n "$ANS" ]
	then
		return 0
	else
		return 1
	fi
}

get_yn()
{
	if [ -z "$YNDEF" ]
	then
		YNDEF=y
	fi

	while :
	do
		$ECHO "$*? [${YNDEF}] \c"
		read ANS
		if [ -n "$ANS" ]
		then
			case $ANS in
			[yY])
				RET=0
				break
				;;
			[nN])
				RET=1
				break
				;;
			*)
				$ECHO "Please answer y or n"
				;;
			esac
		else
			if [ $YNDEF = y ]
			then
				RET=0
			else
				RET=1
			fi
			break
		fi
	done

	YNDEF=
	return $RET
}

do_link()
{
	source=$1
	target=$2

	# Try symlink first
	ln -s $source $target 2>/dev/null
	if [ $? != 0 ]
	then
		# Use hard link
		ln $source $target 2>/dev/null
	fi
	RETSTAT=$?
	if [ $RETSTAT != 0 ]
	then
		log_err -p "Cannot link $source -> $target"
	fi
	return $RETSTAT
}

do_chown()
{
	if [ $1 != _default_ ]
	then
		chown $1 $2 2>/dev/null
	fi
}

do_chgrp()
{
	if [ $1 != _default_ ]
	then
		chgrp $1 $2 2>/dev/null
	fi
}

do_chmod()
{
	if [ $1 != _default_ ]
	then
		chmod $1 $2 2>/dev/null
	fi
}

make_dir()
{
	dir=$1
	perm=$2
	owner=$3
	group=$4

	$ECHO "\t$dir"
	if [ ! -d $dir ]
	then
		mkdir -p $dir
	fi
	do_chown $owner $dir
	do_chgrp $group $dir
	do_chmod $perm $dir
	return 0
}

inst_file()
{
	source=$1
	target=$2
	perm=$3
	owner=$4
	group=$5
	tdir=`dirname $target`

	if [ -n "$tdir" -a -d "$tdir" -a -w "$tdir" ]
	then
		if [ ! -f $source ]
		then
			$ECHO "\t$target NOT installed"
			log_err -n "Cannot install ${target}: file missing."
			return 1
		fi

		$ECHO "\t$target"
		if [ -f $target ]
		then
			rm -f $target
		fi

		cp $source $target
		if [ $? != 0 ]
		then
			log_err -n "Cannot install $target: file copy error."
			return 1
		fi

		if [ -f $target ]
		then
			do_chown $owner $target 2>/dev/null
			do_chgrp $group $target 2>/dev/null
			do_chmod $perm $target 2>/dev/null
		fi
	else
		$ECHO "\t$target NOT installed"
		log_err -n \
		    "Cannot install $target: target directory not writable."
		return 1
	fi
	return 0
}

link_prompt()
{
	$ECHO "\nFor security reasons, setuid programs (such as xmcd) search"
	$ECHO "only /usr/lib and/or /usr/ccs/lib for dynamic libraries."
	$ECHO "Some of the dynamic libraries that xmcd needs may not be in"
	$ECHO "the standard locations, thus xmcd may not be able to find"
	$ECHO "them."

	$ECHO "\nSymbolic links can be created now to correct this problem."

	$ECHO "\nDo you want this installation procedure to search your"
	$ECHO "system for needed dynamic libraries, and create symbolic links"
	$ECHO "of these libraries to /usr/lib\c"
	YNDEF=y
	get_yn " "
	if [ $? != 0 ]
	then
		$ECHO "\nNo links will be created.\n"
		$ECHO "If you encounter difficulty starting xmcd, see the FAQ"
		$ECHO "file in the xmcd distribution for further information."
		return 1
	fi
	$ECHO ""
	return 0
}

link_dynlibs()
{
	LINKFOUND=False

	#
	# Set LD_LIBRARY_PATH to point to all conceivable places where
	# dynamic libraries can hide
	#
	LD_LIBRARY_PATH=$LIBDIR:/usr/X/lib:/usr/X11/lib:/usr/X386/lib:/usr/X11R5/lib:/usr/X11R6/lib:/usr/openwin/lib:/usr/X/desktop:/usr/Motif/lib:/usr/Motif1.1/lib:/usr/Motif1.2/lib:/usr/Motif2.0/lib:/usr/dt/lib:/usr/lib/X11:/usr/ccs/lib:/usr/lib:/lib
	export LD_LIBRARY_PATH

	# Find the ldd program
	for i in /bin /usr/bin /usr/ccs/bin
	do
		if [ -x $i/ldd ]
		then
			LDD=$i/ldd
		fi
	done

	if [ -z "$LDD" ]
	then
		# Can't locate ldd
		return
	fi

	if [ ! -r xmcd_d/xmcd ]
	then
		# Can't read xmcd binary
		return
	fi

	# Run ldd to determine its dynamic library configuration
	$LDD xmcd_d/xmcd >$TMPFILE 2>/dev/null

	if fgrep '=>' $TMPFILE >/dev/null 2>&1
	then
		# BSD/SunOS/SVR5 style ldd output
		DYNLIBS="`fgrep -v 'not found' $TMPFILE | \
			fgrep -v 'xmcd needs:' | \
			sed -e 's/^.*=> *//' -e 's/(.*)//' | tr '\015' ' '`"
		ERRLIBS="`fgrep 'not found' $TMPFILE | $AWK '{ print $1 }' | \
			tr '\015' ' '`"
	else
		# SVR4 style ldd output
		DYNLIBS="`fgrep 'loaded:' $TMPFILE | sed 's/^.*: //' | \
			tr '\015' ' '`"
		ERRLIBS="`fgrep 'error opening' $TMPFILE | \
			sed 's/^.*opening //' | tr '\015' ' '`"
	fi

	# Remove temp files
	rm -f $TMPFILE

	for i in $ERRLIBS _xoxo_
	do
		if [ "$i" = _xoxo_ ]
		then
			break
		fi

		# A needed library is not found in LD_LIBRARY_PATH
		log_err -p "\nNeeded library $i not found!  See the xmcd FAQ."
	done

	for i in $DYNLIBS _xoxo_
	do
		if [ "$i" = _xoxo_ ]
		then
			# Done processing
			break
		fi

		LIBNAME=`basename $i`
		DIRNAME=`dirname $i`

		if [ "$DIRNAME" = /usr/lib -o "$DIRNAME" = /usr/ccs/lib ]
		then
			# This is the standard library location
			continue
		fi

		if [ -f /usr/lib/$LIBNAME -o -f /usr/ccs/lib/$LIBNAME ]
		then
			# Link already there
			continue
		fi

		if [ $LINKFOUND = False ]
		then
			LINKFOUND=True

			if [ -z "$BATCH" ]
			then
				link_prompt
				if [ $? != 0 ]
				then
					return
				fi
			fi
		fi

		YNDEF=y
		if [ -n "$BATCH" ] || \
			get_yn "Link $DIRNAME/$LIBNAME to /usr/lib/$LIBNAME"
		then
			do_link $DIRNAME/$LIBNAME /usr/lib/$LIBNAME
		else
			$ECHO "$DIRNAME/$LIBNAME not linked."
		fi

	done
}


#
# Main execution starts here
#

# Catch some signals
trap "rm -f $TMPFILE; exit 1" 1 2 3 5 15

#
# Get platform information
#
OS_SYS=`(uname -s) 2>/dev/null`
OS_REL=`(uname -r) 2>/dev/null`
OS_MACH=`(uname -m) 2>/dev/null`

if [ -z "$OS_SYS" ]
then
	OS_SYS=unknown
fi
if [ -z "$OS_REL" ]
then
	OS_REL=unknown
fi
if [ -z "$OS_MACH" ]
then
	OS_MACH=unknown
fi

OS_SYS_T=`echo "$OS_SYS" | sed -e 's/\//_/g' -e 's/-/_/g' -e 's/[ 	]/_/g'`
OS_MACH_T=`echo "$OS_MACH" | sed -e 's/\//_/g' -e 's/-/_/g' -e 's/[ 	]/_/g'`
OS_REL_T=`echo "$OS_REL" | sed -e 's/\//_/g' -e 's/-/_/g' -e 's/[ 	]/_/g'`

# Use Sysv echo if possible
if [ -x /usr/5bin/echo ]
then
	ECHO=/usr/5bin/echo				# SunOS SysV echo
elif [ -z "`(echo -e a) 2>/dev/null | fgrep e`" ]
then
	ECHO="echo -e"					# GNU bash, etc.
else
	ECHO=echo					# generic SysV
fi
if [ "$OS_SYS" = QNX ]
then
	ECHO=echo
fi

# If awk doesn't work well on your system, try changing the
# following to nawk or gawk.
AWK=awk

# Remove old error log file
ERROR=0
rm -f $ERRFILE
if [ -f $ERRFILE ]
then
	$ECHO "Cannot remove old $ERRFILE: error logging not enabled."
	ERRFILE=/dev/null
fi

# Parse command line args
NOREMOVE=0
BATCH=
if [ $# -gt 1 ]
then
	for i in $*
	do
		case "$i" in
		-n)
			# Whether to remove distribution files after
			# installation
			NOREMOVE=1
			;;
		-b)
			# Specify batch mode
			BATCH=$i
			;;
		*)
			log_err -p "Unknown option: $i"
			do_exit 1
			;;
		esac
	done
elif [ -f docs_d/INSTALL -a -f docs_d/PORTING ]
then
	# Do not remove if running from source code directory
	NOREMOVE=1
fi


# Implement platform-specific features and deal with OS quirks
LINKLIBS=False
SHELL=/bin/sh
SCO=False

if [ "$OS_SYS" = A/UX ]
then
	# Apple A/UX
	SHELL=/bin/ksh
elif [ "$OS_SYS" = FreeBSD ]
then
	# FreeBSD
	OS_REL_T=`echo "$OS_REL_T" | sed 's/\..*$//'`
	OS_SYS_T=${OS_SYS_T}_${OS_REL_T}
elif [ "$OS_SYS" = HP-UX ]
then
	# HP-UX
	SHELL=/bin/ksh
elif [ "$OS_SYS" = SunOS -o "$OS_SYS" = Solaris ]
then
	case "$OS_REL" in
	4.*)
		# SunOS 4.x
		LINKLIBS=True
		;;
	5.*)
		# SunOS 5.x
		LINKLIBS=True
		OS_SYS_T=Solaris
		;;
	*)
		;;
	esac
elif [ "$OS_SYS" = ULTRIX ]
then
	# Digital Ultrix
	SHELL=/bin/sh5
elif [ -x /bin/ftx ] && /bin/ftx
then
	case "$OS_REL" in
	4.*)
		# Stratus FTX SVR4
		LINKLIBS=True
		;;
	*)
		;;
	esac
elif [ -x /bin/i386 -o -x /sbin/i386 ] && i386
then
	case "$OS_REL" in
	3.*)
		if (uname -X | fgrep "Release = 3.2") >/dev/null 2>&1
		then
			# SCO UNIX/ODT
			SCO=True
			OS_SYS_T=SCO_SV
		fi
		;;
	4.*)
		# UNIX SVR4.x
		LINKLIBS=True
		OS_SYS_T=${OS_SYS_T}_${OS_REL_T}
		;;
	4*MP)
		# UNIX SVR4 MP
		LINKLIBS=True
		OS_SYS_T=${OS_SYS_T}_${OS_REL_T}
		;;
	5)
		# UNIX SVR5
		LINKLIBS=True
		OS_SYS_T=${OS_SYS_T}_${OS_REL_T}
		;;
	5.*)
		# UNIX SVR5.x
		LINKLIBS=True
		OS_SYS_T=${OS_SYS_T}_${OS_REL_T}
		;;
	*)
		;;
	esac
elif [ -x /bin/m88k ] && m88k
then
	case "$OS_REL" in
	4.*)
		# Motorola SVR4/m88k
		LINKLIBS=True
		;;
	*)
		;;
	esac
fi


$ECHO "\nInstalling \"xmcd\" Motif CD Player version $XMCD_VER by Ti Kan"
$ECHO "-------------------------------------------------------"
$ECHO "\nThis is free software and comes with no warranty."
$ECHO "See the GNU General Public License in the COPYING file"
$ECHO "for details."
$ECHO "\nPlease visit the xmcd web site:"
$ECHO "\n\t$XMCD_URL"

# Check privilege
(id | fgrep 'uid=0(root)') >/dev/null 2>&1
if [ $? != 0 ]
then
	$ECHO "\n\nYou should be the super user to install xmcd."

	YNDEF=n
	if [ -z "$BATCH" ] && get_yn "\n  Proceed with installation anyway"
	then
		$ECHO "\nWARNING: Without super-user privilege, some files may"
		$ECHO "not be properly installed, or they may be installed"
		$ECHO "with incorrect permissions."

		XBINPERM=711
		XBINOWNER=_default_
		BINOWNER=_default_
		OWNER=_default_
		GROUP=_default_
	else
		log_err -p "Not super user: installation aborted by user."
		do_exit 1
	fi
fi


# Check existence of binaries

MISSING=
for i in \
	xmcd_d/xmcd \
	cda_d/cda \
	util_d/cddbcmd \
	util_d/dp2xmcd.sh \
	util_d/wm2xmcd
do
	if [ ! -f $i ]
	then
		MISSING="$MISSING $i"
	fi
done

if [ -n "$MISSING" ]
then
	$ECHO "\n\nThe following executable binaries are missing:\n"
	for i in $MISSING
	do
		$ECHO "\t$i"
	done
	$ECHO "\nIf you have the xmcd source code distribution, make sure"
	$ECHO "you compile the source code to generate the binaries first."
	$ECHO "See the INSTALL file for details."
	$ECHO "\nIf you have the xmcd binary distribution, it is probably"
	$ECHO "corrupt."

	YNDEF=n
	if [ -z "$BATCH" ] && get_yn "\n  Proceed with installation anyway"
	then
		$ECHO "\nThe missing files will not be installed."
	else
		log_err -p "Missing binaries: installation aborted by user."
		do_exit 1
	fi
fi


# Determine BINDIR

BINTRYDIRS="\
	/usr/bin/X11 \
	/usr/X/bin \
	/usr/X11/bin \
	/usr/X11R6/bin \
	/usr/X11R5/bin \
	/usr/X386/bin \
	/usr/openwin/bin \
	/usr/local/bin/X11 \
	/usr/local/bin \
	/usr/lbin \
"

if [ -z "$BINDIR" ]
then
	for i in $BINTRYDIRS
	do
		if [ -d $i ]
		then
			BINDIR=$i
			break
		fi
	done

	if [ -z "$BINDIR" ]
	then
		BINDIR=/usr/bin/X11
	fi
else
	BINDIR=`echo $BINDIR | sed 's/\/\//\//g'`
fi


$ECHO "\n\nThe X binary directory is where the executable binary files"
$ECHO "will be installed."

if [ -n "$BATCH" ]
then
	if [ -n "$BATCH_BINDIR" ]
	then
		BINDIR=$BATCH_BINDIR
	fi
	$ECHO "\nUsing $BINDIR"
else
	while :
	do
		if get_str "\n  Enter X binary directory\n  [${BINDIR}]:"
		then
			if [ -d "$ANS" ]
			then
				BINDIR=$ANS
				break
			else
				$ECHO "  Error: $ANS does not exist."
			fi
		else
			break
		fi
	done
fi


# Determine LIBDIR

LIBTRYDIRS="\
	/usr/lib/X11 \
	/usr/X/lib \
	/usr/X11/lib \
	/usr/X11R6/lib/X11 \
	/usr/X11R5/lib/X11 \
	/usr/X386/lib \
	/usr/openwin/lib \
	/usr/local/lib/X11 \
"

if [ -z "$LIBDIR" ]
then
	for i in $LIBTRYDIRS
	do
		if [ -d $i ]
		then
			LIBDIR=$i
			break
		fi
	done

	if [ -z "$LIBDIR" ]
	then
		LIBDIR=/usr/lib/X11
	fi
else
	LIBDIR=`echo $LIBDIR | sed 's/\/\//\//g'`
fi

$ECHO "\n\nThe X library directory is where the X window system"
$ECHO "configuration files are located."

if [ -n "$BATCH" ]
then
	if [ -n "$BATCH_LIBDIR" ]
	then
		LIBDIR=$BATCH_LIBDIR
	fi
	$ECHO "\nUsing $LIBDIR"
else
	while :
	do
		if get_str "\n  Enter X library directory\n  [${LIBDIR}]:"
		then
			if [ -d "$ANS" ]
			then
				LIBDIR=$ANS
				break
			else
				$ECHO "  Error: $ANS does not exist."
			fi
		else
			break
		fi
	done
fi


# Determine xmcd libdir

$ECHO "\n\nThe xmcd library directory is where xmcd/cda support files"
$ECHO "will be installed."

XMCDLIB=$LIBDIR/xmcd

if [ -n "$BATCH" ]
then
	if [ -n "$BATCH_XMCDLIB" ]
	then
		XMCDLIB=$BATCH_XMCDLIB
	fi
	$ECHO "\nUsing $XMCDLIB"
else
	while :
	do
		if get_str "\n  Enter xmcd library directory\n  [${XMCDLIB}]:"
		then
			UDIR="$ANS"
			if [ -d "$UDIR" ]
			then
				XMCDLIB=$UDIR
				break
			else
				if get_yn \
				"  Directory $UDIR does not exist.  Create it"
				then
					XMCDLIB="$UDIR"
					break
				else
					$ECHO \
					"  ERROR: Cannot install in ${UDIR}."
				fi
			fi
		else
			break
		fi
	done
fi


# Determine CDDBDIR

CDDBDIR=$XMCDLIB/cddb

if [ ! -d $CDDBDIR ]
then
	$ECHO "\n\nThe CD database directory is where the local CD information"
	$ECHO "cache will be maintained."

	if [ -n "$BATCH" ]
	then
		if [ -n "$BATCH_CDDBDIR" ]
		then
			CDDBDIR=$BATCH_CDDBDIR
		fi
		$ECHO "\nUsing $CDDBDIR"
	else
		if get_str "\n  Enter CD database directory\n  [${CDDBDIR}]:"
		then
			CDDBDIR=$ANS
		fi
	fi
fi


# Determine DISCOGDIR

DISCOGDIR=$XMCDLIB/discog

if [ ! -d $DISCOGDIR ]
then
	$ECHO "\n\nThe Local Discographies directory is where documents,"
	$ECHO "images, sound clips and other files related to each CD may"
	$ECHO "be stored.  These can then be viewed ot played using the"
	$ECHO "wwwWarp feature of xmcd."

	if [ -n "$BATCH" ]
	then
		if [ -n "$BATCH_DISCOGDIR" ]
		then
			DISCOGDIR=$BATCH_DISCOGDIR
		fi
		$ECHO "\nUsing $DISCOGDIR"
	else
		if get_str \
		"\n  Enter Local Discographies directory\n  [${DISCOGDIR}]:"
		then
			DISCOGDIR=$ANS
		fi
	fi
fi


# Determine MANDIR

if [ -z "$MANDIR" ]
then
	for i in	/usr/man/man.LOCAL \
			/usr/share/man/man1 \
			/usr/X11/man/man1 \
			/usr/X/man/man1 \
			/usr/X11R6/man/man1 \
			/usr/X11R5/man/man1 \
			/usr/X386/man/man1 \
			/usr/local/man/man1
	do
		if [ -d $i ]
		then
			MANDIR=$i
			break
		fi
	done

	if [ -z "$MANDIR" ]
	then
		MANDIR=/usr/man/man1
	fi

else
	MANDIR=`echo $MANDIR | sed 's/\/\//\//g'`
fi

$ECHO "\n\nThe on-line manual directory is where the man pages in"
$ECHO "in the xmcd package will be installed."

if [ -n "$BATCH" ]
then
	if [ -n "$BATCH_MANDIR" ]
	then
		MANDIR=$BATCH_MANDIR
	fi
	$ECHO "\nUsing $MANDIR"
else
	if get_str "\n  Enter on-line manual directory\n  [${MANDIR}]:"
	then
		MANDIR=$ANS
	fi
fi

if [ ! -d $MANDIR -a -z "$BATCH" ]
then
	YNDEF=y
	get_yn "  Directory $MANDIR does not exist.  Create it"
	if [ $? -ne 0 ]
	then
		$ECHO "  The xmcd on-line manual will not be installed."
		MANDIR=
	fi
fi

# Determine MANSUFFIX

if [ -n "$MANDIR" ]
then
	if [ -z "$MANSUFFIX" ]
	then
		MANSUFFIX=1
	fi

	if [ -n "$BATCH" ]
	then
		if [ -n "$BATCH_MANSUFFIX" ]
		then
			MANSUFFIX=$BATCH_MANSUFFIX
		fi
		$ECHO "Using ManSuffix $MANSUFFIX"
	else
		if get_str \
		"\n  Enter on-line manual file name suffix\n  [${MANSUFFIX}]:"
		then
			MANSUFFIX=$ANS
		fi
	fi
fi


# Remove old xmcd components

$ECHO "\n\nChecking for old xmcd components..."

# Old binaries
dirs=`echo "$OPATH" | $AWK -F: '{ for (i = 1; i <= NF; i++) print $i }'`
for i in $BINTRYDIRS
do
	dirs=`$ECHO "$dirs\n$i"`
done
dirs=`($ECHO "$dirs" | \
       sed -e 's/^[ 	]*//' -e '/^$/d' | \
       sort | uniq) 2>/dev/null`

if [ -n "$dirs" ]
then
	for i in $dirs
	do
	    if [ "$i" = "$BINDIR" -o "$i" = "." ]
	    then
		    continue
	    fi

	    for j in xmcd cda cddbcmd wm2xmcd dp2xmcd $STARTUP_SCRIPT
	    do
		tryfile=${i}/${j}
		if [ -f $tryfile -a -x $tryfile ]
		then
		    if [ -z "$BATCH" ]
		    then
			YNDEF=y
			if get_yn "Remove old executable $tryfile"
			then
			    rm -f $tryfile
			    if [ $? -ne 0 ]
			    then
				$ECHO "Cannot remove $tryfile."
			    fi
			fi
		    else
			rm -f $tryfile
		    fi
		fi
	    done
	done
fi

# Old xmcd app-defaults files
dirs=`for i in $LIBTRYDIRS $LIBDIR; do echo "$i"; done | sort | uniq`
for i in $dirs
do
	tryfile=${i}/app-defaults/XMcd
	if [ -f "$tryfile" ]
	then
		if [ -z "$BATCH" ]
		then
		    YNDEF=y
		    if get_yn "Remove old xmcd resource file $tryfile"
		    then
			rm -f $tryfile
			if [ $? -ne 0 ]
			then
			    $ECHO "Cannot remove ${tryfile}."
			fi
		    fi
		else
		    rm -f $tryfile
		fi
	fi
done


# Set architecture-specific binary directory
ARCHBIN="${XMCDLIB}/bin-${OS_SYS_T}-${OS_MACH_T}"


# Make all necessary directories
$ECHO "\n\nMaking directories..."

make_dir $XMCDLIB $DIRPERM $OWNER $GROUP
make_dir $XMCDLIB/app-defaults $DIRPERM $OWNER $GROUP
make_dir $XMCDLIB/config $DIRPERM $OWNER $GROUP
make_dir $XMCDLIB/config/.tbl $DIRPERM $OWNER $GROUP
make_dir $XMCDLIB/docs $DIRPERM $OWNER $GROUP
make_dir $XMCDLIB/help $DIRPERM $OWNER $GROUP
make_dir $XMCDLIB/pixmaps $DIRPERM $OWNER $GROUP
make_dir $XMCDLIB/scripts $DIRPERM $OWNER $GROUP
make_dir $ARCHBIN $DIRPERM $OWNER $GROUP
make_dir $CDDBDIR $DIRPERM $OWNER $GROUP
make_dir $DISCOGDIR $DIRPERM $OWNER $GROUP

if [ "$CDDBDIR" != "$XMCDLIB/cddb" ]
then
	ln -s "$CDDBDIR" "$XMCDLIB/cddb"
	$ECHO "\t$XMCDLIB/cddb"
fi
if [ "$DISCOGDIR" != "$XMCDLIB/discog" ]
then
	ln -s "$DISCOGDIR" "$XMCDLIB/discog"
	$ECHO "\t$XMCDLIB/discog"
fi

if [ -n "$MANDIR" ]
then
	make_dir $MANDIR $DIRPERM $OWNER $GROUP
fi


# Install files

SHELL_S=`echo $SHELL | sed 's/\//\\\\\//g'`
BINDIR_S=`echo $BINDIR | sed 's/\//\\\\\//g'`
LIBDIR_S=`echo $LIBDIR | sed 's/\//\\\\\//g'`
XMCDLIB_S=`echo $XMCDLIB | sed 's/\//\\\\\//g'`
CDDBDIR_S=`echo $CDDBDIR | sed 's/\//\\\\\//g'`
DISCOGDIR_S=`echo $DISCOGDIR | sed 's/\//\\\\\//g'`

$ECHO "\nInstalling xmcd files..."

# Private binaries
inst_file xmcd_d/xmcd $ARCHBIN/xmcd $XBINPERM $XBINOWNER $GROUP
inst_file cda_d/cda $ARCHBIN/cda $XBINPERM $XBINOWNER $GROUP
inst_file util_d/cddbcmd $ARCHBIN/cddbcmd $BINPERM $BINOWNER $GROUP
inst_file util_d/dp2xmcd.sh $ARCHBIN/dp2xmcd $SCRPERM $OWNER $GROUP
inst_file util_d/wm2xmcd $ARCHBIN/wm2xmcd $BINPERM $BINOWNER $GROUP

# X resource defaults file
inst_file xmcd_d/XMcd.ad $XMCDLIB/app-defaults/XMcd $FILEPERM $OWNER $GROUP

# XKeysymDB file
inst_file xmcd_d/XKeysymDB $XMCDLIB/app-defaults/XKeysymDB $FILEPERM $OWNER $GROUP

# Documentation
rm -f $XMCDLIB/docs/*
(cd docs_d ;\
for i in * ;\
do \
	case $i in
	SCCS)
		;;
	RCS)
		;;
	CVS)
		;;
	*)
		inst_file $i $XMCDLIB/docs/$i $FILEPERM $OWNER $GROUP ;\
		;;
	esac
done)

# Help files
rm -f $XMCDLIB/help/*
for i in xmcd_d/hlpfiles/*.btn xmcd_d/hlpfiles/*.lbl xmcd_d/hlpfiles/*.txw \
	 xmcd_d/hlpfiles/*.scl xmcd_d/hlpfiles/*.lsw xmcd_d/hlpfiles/*.rbx \
	 xmcd_d/hlpfiles/*.cbx xmcd_d/hlpfiles/*.opt
do
	j=`echo $i | sed 's/xmcd\_d\/hlpfiles\///'`
	inst_file $i $XMCDLIB/help/$j $FILEPERM $OWNER $GROUP
done

# Icon/pixmap files
for i in xmcd.icon xmcd_a.px xmcd_b.px xmcd.xpm
do
	inst_file misc_d/$i $XMCDLIB/pixmaps/$i $FILEPERM $OWNER $GROUP
done

# Configuration files
inst_file libdi_d/common.cfg $XMCDLIB/config/common.cfg \
	$FILEPERM $OWNER $GROUP
inst_file libdi_d/device.cfg $XMCDLIB/config/device.cfg \
	$FILEPERM $OWNER $GROUP
inst_file misc_d/sites $XMCDLIB/config/sites \
	$FILEPERM $OWNER $GROUP

rm -f $XMCDLIB/config/.tbl/*
ENTRIES=`(cd libdi_d/cfgtbl; echo * | \
	sed -e 's/Imakefile//' -e 's/Makefile//' -e 's/SCCS//' -e 's/RCS//')`
for i in $ENTRIES
do
	if (fgrep "tblver=" libdi_d/cfgtbl/$i) >/dev/null 2>&1
	then
		inst_file libdi_d/cfgtbl/$i $XMCDLIB/config/.tbl/$i \
			$FILEPERM $OWNER $GROUP
	fi
done

# Configuration script
sed -e "s/^#!\/bin\/sh.*/#!$SHELL_S/" \
    -e "s/^BINDIR=.*/BINDIR=$BINDIR_S/" \
    -e "s/^LIBDIR=.*/LIBDIR=$LIBDIR_S/" \
    -e "s/^XMCDLIB=.*/XMCDLIB=$XMCDLIB_S/" \
    -e "s/^CDDBDIR=.*/CDDBDIR=$CDDBDIR_S/" \
    -e "s/^DISCOGDIR=.*/DISCOGDIR=$DISCOGDIR_S/" \
    -e "s/^OWNER=.*/OWNER=$OWNER/" \
    -e "s/^GROUP=.*/GROUP=$GROUP/" \
    <libdi_d/config.sh >/tmp/xmcdcfg.$$

rm -f $XMCDLIB/config/configure.sh
inst_file /tmp/xmcdcfg.$$ $XMCDLIB/config/config.sh $SCRPERM $OWNER $GROUP
rm -f /tmp/xmcdcfg.$$

# Convenience link to config.sh
if [ "$SCO" = True ]
then
	if [ -w /usr/lib/mkdev ]
	then
		$ECHO "\t/usr/lib/mkdev/xmcd"
		rm -f /usr/lib/mkdev/xmcd
		do_link $XMCDLIB/config/config.sh /usr/lib/mkdev/xmcd
	fi
fi

# Shell scripts
inst_file misc_d/ncsarmt.sh $XMCDLIB/scripts/ncsarmt $SCRPERM $OWNER $GROUP
inst_file misc_d/ncsawrap.sh $XMCDLIB/scripts/ncsawrap $SCRPERM $OWNER $GROUP
inst_file misc_d/nswrap.sh $XMCDLIB/scripts/nswrap $SCRPERM $OWNER $GROUP
sed -e "s/^#!\/bin\/sh.*/#!$SHELL_S/" \
    -e "s/^BINDIR=.*/BINDIR=$BINDIR_S/" \
    -e "s/^LIBDIR=.*/LIBDIR=$LIBDIR_S/" \
    -e "s/^XMCDLIB=.*/XMCDLIB=$XMCDLIB_S/" \
    -e "s/^CDDBDIR=.*/CDDBDIR=$CDDBDIR_S/" \
    -e "s/^DISCOGDIR=.*/DISCOGDIR=$DISCOGDIR_S/" \
    -e "s/^OWNER=.*/OWNER=$OWNER/" \
    -e "s/^GROUP=.*/GROUP=$GROUP/" \
    <misc_d/genidx.sh >$XMCDLIB/scripts/genidx
do_chown $OWNER $XMCDLIB/scripts/genidx
do_chgrp $GROUP $XMCDLIB/scripts/genidx
do_chmod $SCRPERM $XMCDLIB/scripts/genidx
$ECHO "\t$XMCDLIB/scripts/genidx"

# Local discographies
inst_file misc_d/bkgnd.gif $DISCOGDIR/bkgnd.gif $FILEPERM $OWNER $GROUP
inst_file misc_d/xmcdlogo.gif $DISCOGDIR/xmcdlogo.gif $FILEPERM $OWNER $GROUP
rm -f $DISCOGDIR/discog.html
sed -e "s/\$SHELL/$SHELL_S/g" \
    -e "s/\$BINDIR/$BINDIR_S/g" \
    -e "s/\$LIBDIR/$LIBDIR_S/g" \
    -e "s/\$XMCDLIB/$XMCDLIB_S/g" \
    -e "s/\$CDDBDIR/$CDDBDIR_S/g" \
    -e "s/file:\/\/localhost\/\$DISCOGDIR/file:\/\/localhost$DISCOGDIR_S/g" \
    -e "s/\$DISCOGDIR/$DISCOGDIR_S/g" \
    <misc_d/discog.htm >$DISCOGDIR/discog.html
do_chown $OWNER $DISCOGDIR/discog.html
do_chgrp $GROUP $DISCOGDIR/discog.html
do_chmod $FILEPERM $DISCOGDIR/discog.html
$ECHO "\t$DISCOGDIR/discog.html"

# Startup wrapper script
rm -f $BINDIR/$STARTUP_SCRIPT
sed -e "s/XMCD_LIBDIR=.*/XMCD_LIBDIR=$XMCDLIB_S/" \
	misc_d/start.sh >$BINDIR/$STARTUP_SCRIPT
do_chown $BINOWNER $BINDIR/$STARTUP_SCRIPT
do_chgrp $BINGROUP $BINDIR/$STARTUP_SCRIPT
do_chmod $SCRPERM $BINDIR/$STARTUP_SCRIPT
$ECHO "\t$BINDIR/$STARTUP_SCRIPT"

# Use startup wrapper script to invoke all supported programs
for i in xmcd cda cddbcmd dp2xmcd wm2xmcd
do
	rm -f $BINDIR/$i
	do_link $BINDIR/$STARTUP_SCRIPT $BINDIR/$i
	$ECHO "\t$BINDIR/$i"
done

# Manual page files
if [ -n "$MANDIR" -a -n "$MANSUFFIX" ]
then
	for i in xmcd cda
	do
		inst_file ${i}_d/${i}.man $MANDIR/${i}.$MANSUFFIX \
			$FILEPERM $OWNER $GROUP
	done
	for i in cddbcmd dp2xmcd wm2xmcd
	do
		inst_file util_d/${i}.man $MANDIR/${i}.$MANSUFFIX \
			$FILEPERM $OWNER $GROUP
	done
fi

if [ -z "$BATCH" ]
then
	# Run device-dependent config script
	if [ -r $XMCDLIB/config/config.sh ]
	then
		$SHELL $XMCDLIB/config/config.sh
		if [ $? != 0 ]
		then
			log_err -n "$XMCDLIB/config/config.sh failed."
		fi
	else
		log_err -p "Cannot execute $XMCDLIB/config/config.sh"
	fi
fi

# Demo cddb file
if [ -d $CDDBDIR/misc ]
then
	$ECHO "\nInstalling demo mode CDDB file..."
	inst_file misc_d/demo.db $CDDBDIR/misc/$DEMODB \
		$FILEPERM $OWNER $GROUP
fi

# Set up dynamic library links
if [ $LINKLIBS = True ]
then
	link_dynlibs
fi

# Clean up
if [ $NOREMOVE = 0 ]
then
	rm -rf common_d libdi_d misc_d xmcd_d cda_d util_d docs_d install.sh \
		xmcd.tar
fi

do_exit $ERROR

