/*

   xlog - GTK+ logging program for amateur radio operators
   Copyright (C) 2001 - 2007 Joop Stakenborg <pg4i@amsat.org>

   This file is part of xlog.

   Xlog is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   Xlog is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with xlog.  If not, see <http://www.gnu.org/licenses/>.

*/

/* gui_importlogdialog.c - creation and destruction of the importlog dialog 
 *
 * This dialog shows 2 entries where log names should be filled in, clicking
 * on OK will open them in one page. After that, the user can decide to sort
 * the log or not.
 */

#include <gtk/gtk.h>
#include <string.h>

#include "gui_mergedialog.h"
#include "support.h"
#include "preferences.h"
#include "log.h"
#include "utils.h"
#include "gui_utils.h"
#include "main.h"
#include "dxcc.h"

extern GtkWidget *mainwindow, *scorewindow;
extern GtkWidget *mainnotebook;
extern preferencestype preferences;
extern programstatetype programstate;
extern GList *logwindowlist;

GtkWidget *importlogdialog;
static GtkWidget *previewscrolledwindow;

/* 
 * in case we open a non-xlog file, we have to know the xlog filename
 */
static gchar *find_logname (const gchar *selected)
{
	gchar *basen, *dirn, **split, *l;

	basen = g_path_get_basename (selected);
	dirn = g_path_get_dirname (selected);

	if (g_strrstr(basen, "."))
	{
		split = g_strsplit (basen, ".", 0);
		l = g_strconcat (dirn, G_DIR_SEPARATOR_S, split[0], ".xlog", NULL);
		g_strfreev (split);
	}
	else
	{
		l = g_strconcat (dirn, G_DIR_SEPARATOR_S, basen, ".xlog", NULL);
	}
	g_free (basen);
	g_free (dirn);
	return l;
}

/* callback for the preview widget, read the first 10 lines */
static void
preview_update (GtkFileChooser *chooser, gpointer data)
{
	GtkWidget *preview;
	gchar *filename, buf[80];
	gboolean have_preview = FALSE;
	gint i;
	FILE *in;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;

	preview = GTK_WIDGET (data);
	filename = gtk_file_chooser_get_preview_filename (chooser);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(preview));
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	gtk_text_buffer_delete (buffer, &start, &end);

	in = fopen (filename, "r");
	if (in)
	{
		have_preview = TRUE;
		for (i = 0; i < 10; i++)
		{
			if (fgets (buf, 80, in) == NULL) 
				break;
			else if (g_utf8_validate (buf, -1, NULL))
				gtk_text_buffer_insert_with_tags_by_name 
					(buffer, &start, buf, -1, "tiny", NULL);
		}
	}
	gtk_widget_set_size_request (previewscrolledwindow, 200, -1);
	gtk_file_chooser_set_preview_widget_active (chooser, have_preview);
}

static void
on_importlogbutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *fileselection, *logentry, *preview;
	GtkTextBuffer *buffer;
	gchar *sellog;
	gint response;

	fileselection = gtk_file_chooser_dialog_new (_("xlog - select a log for import"),
    	GTK_WINDOW(mainwindow), GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL, 
		GTK_RESPONSE_CANCEL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);
	gtk_file_chooser_set_current_folder
		(GTK_FILE_CHOOSER (fileselection), preferences.savedir);

	/* preview widget */
	previewscrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (previewscrolledwindow),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER);
	preview = gtk_text_view_new ();
	gtk_text_view_set_editable (GTK_TEXT_VIEW(preview), FALSE);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(preview), FALSE);
	gtk_container_add (GTK_CONTAINER (previewscrolledwindow), preview);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(preview));
	gtk_text_buffer_create_tag (buffer, "tiny", "size", 8 * PANGO_SCALE, NULL);
	gtk_file_chooser_set_preview_widget (GTK_FILE_CHOOSER (fileselection),
		previewscrolledwindow);
	gtk_widget_set_size_request (previewscrolledwindow, 0, -1);
	gtk_widget_show_all (previewscrolledwindow);
	g_signal_connect
		(fileselection, "update-preview", G_CALLBACK (preview_update), preview);

	gtk_widget_show_all (fileselection);
	response = gtk_dialog_run (GTK_DIALOG(fileselection));
	if (response == GTK_RESPONSE_ACCEPT)
	{
		sellog = gtk_file_chooser_get_filename 
			(GTK_FILE_CHOOSER (fileselection));
		logentry = lookup_widget (importlogdialog, "logentry");
		gtk_entry_set_text (GTK_ENTRY(logentry), sellog);
		g_free (sellog);
	}
	gtk_widget_destroy (fileselection);
}

void
on_menu_import_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *vbox, *label1, *logentry, *loghbox, *logbutton, *combobox,
		*sep, *label2, *rementry;
	gint i, response, filetype;
	gchar *log, *logn, *temp, *xlogfile;
	LOGDB *lp;
	logtype *logw;
	gboolean logexists = FALSE;
	GtkTreePath *path;

	importlogdialog = gtk_dialog_new_with_buttons (_("xlog - import"),
		GTK_WINDOW(mainwindow), GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		_("_Import"), GTK_RESPONSE_OK,
		NULL);
	gtk_window_set_default_size (GTK_WINDOW (importlogdialog), 400, 120);
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add
		(GTK_CONTAINER (GTK_DIALOG (importlogdialog)->vbox), vbox);
	label1 = gtk_label_new (_("Select log and logtype for import"));
	gtk_box_pack_start (GTK_BOX (vbox), label1, FALSE, FALSE, 0);
	loghbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), loghbox, FALSE, FALSE, 0);
	logentry = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (loghbox), logentry, TRUE, TRUE, 0);
	gtk_entry_set_activates_default (GTK_ENTRY (logentry), TRUE);

	logbutton = gtk_button_new_from_stock (GTK_STOCK_OPEN);
	gtk_box_pack_start (GTK_BOX (loghbox), logbutton, FALSE, FALSE, 0);

	combobox = gtk_combo_box_new_text ();
	gtk_box_pack_start (GTK_BOX (loghbox), combobox, FALSE, FALSE, 0);

	/* Keep entries in same order as enum log_file_type in logfile/logfile.h */
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "twlog");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "adif");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "cabrillo");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "editest");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "trlog");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "EDI");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "oh1aa");
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox), 1);

	sep = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (vbox), sep, FALSE, FALSE, 5);
	label2 = gtk_label_new (_("When importing from cabrillo add the following remark"));
	gtk_box_pack_start (GTK_BOX (vbox), label2, TRUE, TRUE, 0);
	rementry = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (vbox), rementry, TRUE, TRUE, 0);
	if (strlen(programstate.importremark) > 0)
		gtk_entry_set_text (GTK_ENTRY (rementry), programstate.importremark);
	gtk_entry_set_activates_default (GTK_ENTRY (rementry), TRUE);

	g_signal_connect ((gpointer) logbutton, "clicked",
		G_CALLBACK (on_importlogbutton_clicked), NULL);
	GLADE_HOOKUP_OBJECT (importlogdialog, logentry, "logentry");

	gtk_dialog_set_default_response (GTK_DIALOG (importlogdialog),
		GTK_RESPONSE_OK);
	gtk_widget_show_all (importlogdialog);
	response = gtk_dialog_run (GTK_DIALOG (importlogdialog));
	if (response == GTK_RESPONSE_OK)
	{
		log = gtk_editable_get_chars (GTK_EDITABLE (logentry), 0, -1);
		filetype = gtk_combo_box_get_active (GTK_COMBO_BOX (combobox));
		programstate.importremark =
			gtk_editable_get_chars (GTK_EDITABLE (rementry), 0, -1);
		gtk_widget_destroy (importlogdialog);
		/* check by filename and if log exist jump to the page in the notebook */
		xlogfile = find_logname (log);
		for (i = 0; i < g_list_length (logwindowlist); i++)
		{
			logw = g_list_nth_data (logwindowlist, i);
			if (g_ascii_strcasecmp (xlogfile, logw->filename) == 0)
			{
				gtk_notebook_set_current_page (GTK_NOTEBOOK(mainnotebook), i);
				set_qsoframe (logw);
				logexists = TRUE;
				update_statusbar
					(_("A log with the same name is already open"));
				break;
			}
		}
		if (!logexists)
		{
			programstate.qsos = 0;
			lp = log_file_open (log, filetype + 1);
			if (lp)
			{
				logn = logname (g_path_get_basename (log));
				logw =	openlogwindow (lp, logn, programstate.logwindows++);
				gtk_notebook_set_current_page 
					(GTK_NOTEBOOK(mainnotebook), programstate.logwindows - 1);
				log_file_qso_foreach (lp, fillin_list, logw);
				log_file_close (lp);
				g_free (logn);
				logw->filename = g_strconcat
					(preferences.savedir, G_DIR_SEPARATOR_S, logw->logname, ".xlog", NULL);
				lp->type = filetype + 1;
				logwindowlist =	g_list_append (logwindowlist, logw);
				gtk_notebook_set_current_page 
					(GTK_NOTEBOOK(mainnotebook), programstate.logwindows - 1);
				set_qsoframe (logw);
				set_tabs_menu ();

				if (GTK_WIDGET_VISIBLE(scorewindow))
				{
					fill_scoring_arrays ();
					update_dxccscoring ();
					if (preferences.awardswac == 1)
						update_wacscoring ();
				}

				path = gtk_tree_path_new_from_string ("0");
					gtk_tree_view_scroll_to_cell
			(GTK_TREE_VIEW (logw->treeview), path, NULL, TRUE, 0.5, 0.0);
				gtk_tree_path_free (path);

				set_font (preferences.logfont);
				temp = g_strdup_printf (_("%d QSO's imported"), programstate.qsos);
				update_statusbar (temp);
				g_free (temp);
			}
		}
		g_free (xlogfile);
		g_free (log);
	}
	else
		gtk_widget_destroy (importlogdialog);
}
