// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_single_editor.cc
// *
// * Purpose: A single list of phonebook or SMS entries that can be edited
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 8.3.2000
// *************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xgsm_single_editor.h"
#include "xgsm_multi_editor.h"
#include "xgsm_util.h"
#include "xgsm_dialogs.h"
#include "xgsm_pref.h"

#include <gtk--/base.h>
#include <gtk--/frame.h>
#include <gtk--/button.h>
#include <gtk--/main.h>
#include <gnome--/dialog.h>
#include <typeinfo>
#include <algorithm>
#include <gsmlib/gsm_util.h>
#include <limits.h>
#include <iostream>

extern "C" {
#include "interface.h"
#include "support.h"
}

using namespace std;
using namespace Xgsm;
using namespace SigC;
using namespace gsmlib;

// SingleEditor members

SingleEditor::SingleEditor(MultiEditor *multiEditor, 
                           Gtk::Frame *frame,
                           string prefix, string messageText, string extension,
                           SortOrder sortOrder) :
  DeviceHelper(messageText),
  _multiEditor(multiEditor), _frame(frame),
  _sourceMenuSetting(0), _storeNameActivateEvent(false),
  _sortOrder(sortOrder),
  _prefix(prefix), _extension(extension),
  _rowEdited(-1), _deviceIdle(false)
{
  // lookup widgets
  LOOKUP_WIDGET(_clist, _prefix + "_clist", GtkCList, GTK_IS_CLIST, _frame);
  LOOKUP_WIDGET(_openFileMI, _prefix + "_open_file", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_openDeviceMI, _prefix + "_open_device", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_closeMI, _prefix + "_close", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_undoMI, _prefix + "_undo", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_redoMI, _prefix + "_redo", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_addMI, _prefix + "_add_mi", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_cutMI, _prefix + "_cut", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_copyMI, _prefix + "_copy", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_pasteMI, _prefix + "_paste", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_selectAllMI, _prefix + "_select_all", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_unselectAllMI, _prefix + "_unselect_all", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  LOOKUP_WIDGET(_addButton, _prefix + "_add_button", GtkButton,
                GTK_IS_BUTTON, _frame);
  LOOKUP_WIDGET(_cutButton, _prefix + "_cut_button", GtkButton,
                GTK_IS_BUTTON, _frame);
  LOOKUP_WIDGET(_resetButton, _prefix + "_reset_button", GtkButton,
                GTK_IS_BUTTON, _frame);
  LOOKUP_WIDGET(_openButton, _prefix + "_open_button", GtkButton,
                GTK_IS_BUTTON, _frame);
  LOOKUP_WIDGET(_storeNameEntry, _prefix + "_store_name", GnomeEntry,
                GNOME_IS_ENTRY, _frame);
  LOOKUP_WIDGET(_sourceMenu, _prefix + "_source", GtkOptionMenu,
                GTK_IS_OPTION_MENU, _frame);
  
  // change clist selection style to extended if necessary
  if (! config.getMultipleListSelectionStyle())
    _clist->set_selection_mode(GTK_SELECTION_EXTENDED);

  // connect handlers
  _selectAllMI->activate.connect(_clist->select_all.slot());
  _selectAllMI->activate.connect(slot(this, &SingleEditor::onCompleteEdit));
  _unselectAllMI->activate.connect(_clist->unselect_all.slot());
  _unselectAllMI->activate.connect(slot(this, &SingleEditor::onCompleteEdit));
  _closeMI->activate.connect(slot(this, &SingleEditor::onClose));
  _openFileMI->activate.connect(slot(this, &SingleEditor::onOpenFile));
  _openDeviceMI->activate.connect(slot(this, &SingleEditor::onOpenDevice));
  _openButton->clicked.connect(slot(this, &SingleEditor::onOpen));
  ((Gtk::Entry*)_storeNameEntry->gtk_entry())->
    activate.connect(slot(this, &SingleEditor::onStoreNameActivate));
  _resetButton->clicked.connect(_clist->unselect_all.slot());
  _frame->destroy.connect(slot(this, &SingleEditor::onDestroy));
  _sourceMenu->get_menu()->
    selection_done.connect(slot(this, &SingleEditor::onSourceMenu));
  
  // make columns auto-resize
  for (int i = 0; i < 3; ++i) _clist->set_column_auto_resize(i, true);

  // attention: this only calls SingleEditor::sensitive()
  // (not XXSingleEditor::sensitive(),
  // but the store-specific widgets are insensitive by default)
  sensitive(false);
}

void SingleEditor::startEdit(int row)
{
  _rowEdited = row;
  _addButton->set_sensitive(false);
  _addMI->set_sensitive(false);
}

void SingleEditor::stopEdit()
{
  _rowEdited = -1;
  _addButton->set_sensitive(_deviceIdle);
  _addMI->set_sensitive(_deviceIdle);
}

void SingleEditor::sensitive(bool deviceIdle)
{
  DEBUG_START(1);
  cout << _messageText << " sensitive(" << deviceIdle << ") called" << endl;
  DEBUG_END;
  int rs = rowsSelected();
  _deviceIdle = deviceIdle;

  // gui objects to open devices
  bool canOpen = (deviceIdle || _dev.isnull()) && ! _waitingForOpen;
  _openFileMI->set_sensitive(canOpen);
  _openDeviceMI->set_sensitive(canOpen);
  _openButton->set_sensitive(canOpen);
  _storeNameEntry->set_sensitive(canOpen);
  _sourceMenu->set_sensitive(canOpen);

  // gui objects that need an existing, idle device
  _clist->set_sensitive(deviceIdle && ! _dev.isnull());
  _closeMI->set_sensitive(deviceIdle && ! _dev.isnull());
  _undoMI->set_sensitive(deviceIdle && ! _dev.isnull());
  _redoMI->set_sensitive(deviceIdle && ! _dev.isnull());
  _addMI->set_sensitive(! editInProgress() && deviceIdle && ! _dev.isnull());
  _cutMI->set_sensitive(rs > 0 && deviceIdle && ! _dev.isnull());
  _copyMI->set_sensitive(rs > 0 && deviceIdle && ! _dev.isnull());
  _pasteMI->set_sensitive(deviceIdle && ! _dev.isnull());
  _selectAllMI->set_sensitive(deviceIdle && ! _dev.isnull());
  _unselectAllMI->set_sensitive(deviceIdle && ! _dev.isnull());
  _cutButton->set_sensitive(rs > 0 && deviceIdle && ! _dev.isnull());
  _resetButton->set_sensitive(deviceIdle && ! _dev.isnull());
  _addButton->set_sensitive(! editInProgress() && deviceIdle &&
                            ! _dev.isnull());
  _multiEditor->sensitive(this, deviceIdle && ! _dev.isnull(),
                          (deviceIdle && ! _dev.isnull()) || _waitingForOpen);
}

void SingleEditor::onAddDone(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    Gnome::Dialogs::error
      ((string)_("Error adding entry\n(") +
       ((ErrorResponse&)response()).exception().what() + ")")->run_and_close();
    close();
  }
  else
  {
    assert(typeid(response()) == typeid(AddPbEntryResponse) ||
           typeid(response()) == typeid(AddSmsEntryResponse));
    if (_destroyed) return;
    refresh();
  }
}

void SingleEditor::onCutDone(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    Gnome::Dialogs::error
      ((string)_("Error deleting entries\n(") +
       ((ErrorResponse&)response()).exception().what() + ")")->run_and_close();
    close();
  }
  else
  {
    assert(typeid(response()) == typeid(DeletePbEntriesResponse) ||
           typeid(response()) == typeid(DeleteSmsEntriesResponse));
    if (_destroyed) return;
    vector<int> saveSelection;
    for (Gtk::CList_Helpers::SelectionIterator i =
           _clist->selection().begin();
         i != _clist->selection().end(); ++i)
      saveSelection.push_back(i->get_row_num());
    sort(saveSelection.begin(), saveSelection.end());
    _clist->freeze();
    for (vector<int>::reverse_iterator i = saveSelection.rbegin();
         i != saveSelection.rend(); ++i)
    {
      _clist->remove_row(*i);
      _entries->erase(_entries->begin() + *i);
    }
    _clist->thaw();
  }
}

void SingleEditor::onOpenDone(ResponseRef response)
{
  DeviceHelper::onOpenDone(response);
  if (typeid(response()) == typeid(OpenResponse))
  {
    if (_destroyed) return;

    // init() called or user entered empty store name
    if (currentStoreName() == "")
      close();

    // update history: make sure that available stores are in the history
    // and therefore selectable by the user
    vector<string> storeNames = getStoreNames((OpenResponse&)response());
    for (vector<string>::iterator i = storeNames.begin();
         i != storeNames.end(); ++i)
      if (_storeNameHistory.find(*i) == _storeNameHistory.end())
      {
        _storeNameHistory.insert(*i);
        _storeNameEntry->append_history(FALSE, *i);
      }
    dumpEntries();
    _storeNameActivateEvent = false;

    ((Gtk::Entry*)_storeNameEntry->gtk_entry())->set_text(currentStoreName());
    refresh();
  }
}

void SingleEditor::onEditDone(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    Gnome::Dialogs::error
      (stringPrintf(_("Error updating %s \n("), _messageText.c_str()) +
       ((ErrorResponse&)response()).exception().what() + ")")->run_and_close();
    close();
  }
  else
  {
    assert(typeid(response()) == typeid(UpdatePbEntryResponse));
    if (_destroyed) return;
    stopEdit();
  }
}

void SingleEditor::onDestroy()
{
  _destroyed = true;
}

void SingleEditor::onClose()
{
  DeviceHelper::onClose();
  if (! _destroyed) refresh();
}

void SingleEditor::onSourceMenu()
{
  _sourceMenuSetting = 0;
  for (Gtk::Menu_Helpers::MenuList::iterator
         i = _sourceMenu->get_menu()->items().begin();
       i != _sourceMenu->get_menu()->items().end(); ++i, ++_sourceMenuSetting)
    if (_sourceMenu->get_menu()->get_active() == *i)
      break;
}

void SingleEditor::onOpen()
{
  string newSource = ((Gtk::Entry*)_storeNameEntry->gtk_entry())->get_text();

  // update history
  if (newSource != "" &&
      _storeNameHistory.find(newSource) == _storeNameHistory.end())
  {
    _storeNameHistory.insert(newSource);
    if (! _storeNameActivateEvent)
      _storeNameEntry->prepend_history(TRUE, newSource);
  }
  dumpEntries();

  // open new phonebook
  if (_sourceMenuSetting == 0)  // open file
    open(newSource);
  else                          // open device
    open(_deviceName, config.getBaudRate(), config.getInitString(),
         config.getSoftwareHandshake(), newSource);
}

void SingleEditor::onStoreNameActivate()
{
  _storeNameActivateEvent = true;
  onOpen();
}

void SingleEditor::onOpenFile()
{
  string newSource = runFileSelectionDialog(
    stringPrintf(_("Open %s file"), _messageText.c_str()));
  if (newSource == "") return;

  // update history
  if (_storeNameHistory.find(newSource) == _storeNameHistory.end())
  {
    _storeNameHistory.insert(newSource);
    _storeNameEntry->prepend_history(TRUE, newSource);
  }
  dumpEntries();

  open(newSource);
}

void SingleEditor::onOpenDevice()
{
  string baudRate, initString;
  bool swHandshake;
  string newSource = runDeviceSelectionDialog(
    stringPrintf(_("Open %s device"), _messageText.c_str()), baudRate,
    initString, swHandshake);
  if (newSource == "") return;

  open(newSource, baudRate, initString, swHandshake, "");
}

int SingleEditor::rowsSelected()
{
  int result = 0;
  for (Gtk::CList_Helpers::SelectionIterator i = _clist->selection().begin();
       i != _clist->selection().end(); ++i)
    ++result;
  return result;
}

void SingleEditor::setPanedPosition(int position)
{
  // do nothing
}

SingleEditor::~SingleEditor()
{
  onDestroy();
  close();
}
