// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_main.cc
// *
// * Purpose: Main windows GUI class
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 8.3.2000
// *************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xgsm_main.h"
#include "xgsm_util.h"
#include "xgsm_pref.h"
#include "xgsm_multi_editor.h"
#include "xgsm_send_sms.h"
#include "xgsm_status_dialog.h"

#include <gnome.h>
#include <gnome--/client.h>
#include <gnome--/main.h>
#include <gnome--/about.h>
#include <gnome--/dialog.h>
#include <gtk--/menuitem.h>
#include <gtk--/tearoffmenuitem.h>
#include <gtk--/base.h>
#include <gtk--/label.h>
#include <gtk--/style.h>
#include <gtk--/scrollbar.h>
#include <gdk--/color.h>
#include <gsmlib/gsm_error.h>
#include <gsmlib/gsm_util.h>
#include <typeinfo>
#include <iostream>

extern "C" {
#include "interface.h"
#include "support.h"
}

using namespace std;
using namespace Xgsm;
using namespace gsmlib;

// Main members

void Main::onPhonebook()
{
  MultiEditor *m = new PbMultiEditor();
  m->leftEditor()->init();
  m->rightEditor()->init();
}

void Main::onDefaultPhonebook()
{
  openDefaultPhonebook();
}

void Main::onSMSStore()
{
  MultiEditor *m = new SmsMultiEditor();
  m->leftEditor()->init();
  m->rightEditor()->init();
}

void Main::onDefaultSMSStore()
{
  openDefaultSMSStore();
}

void Main::onSendNewSMS()
{
  new SendSMS;
}

void Main::onCallForwarding()
{
  // FIXME
}

void Main::onStatus()
{
  new StatusDialog();
}

void Main::onPreferences()
{
  new Preferences();
}

void Main::onAbout()
{
  Gtk::wrap((GnomeAbout*)create_about_dialog())->show_all();
}

void Main::onDestroy()
{
  DEBUG_START(1);
  cout << "onDestroy called" << endl;
  DEBUG_END;

  ToplevelHelper::closeAllDialogs();

  // create abort dialog, connect Abort button and set
  Gtk::Widget *closeDialog = Gtk::wrap(create_close_dialog());

  CONNECT_SIGNAL("close_dialog_abort", GtkButton, GTK_IS_BUTTON, clicked,
                 closeDialog, Main::onCloseDialogAbort);
  
  LOOKUP_WIDGET(_closeDialogProgressBar, "close_dialog_progress_bar",
                GtkProgressBar, GTK_IS_PROGRESS_BAR, closeDialog);
  
  closeDialog->show_all();
  _closeDialogProgressIncrement = 0.05;

  _waitingForClose = true;
}

void Main::onStop()
{
  string deviceName =
    ((Gtk::Label*)_stopMenu->get_active()->get_child())->get_text();
  DeviceRef dev = Device::getDevice(deviceName, false);

  message(gsmlib::stringPrintf(_("Trying to stop device %s..."),
                               deviceName.c_str()));
  if (! dev.isnull())
    dev->interrupt();
}

void Main::onCloseDialogAbort()
{
  Gnome::Main::quit();
}

gint Main::onConfigureEvent(GdkEventConfigure* event)
{
  config.setMainSizes(event->width, event->height);
  return TRUE;
}

gint Main::idle()
{
  if (_waitingForClose)
  {
    // animate close dialog
    if (_closeDialogProgressBar != NULL)
    {
      gfloat percentage = _closeDialogProgressBar->get_current_percentage();
      percentage += _closeDialogProgressIncrement;
      if (percentage > 1.0)
        percentage = 1.0;
      else if (percentage < 0.0)
        percentage = 0.0;

      if (percentage > 0.99)
      {
        _closeDialogProgressIncrement = -_closeDialogProgressIncrement;
        
        if (_closeDialogProgressBar->get_orientation() ==
            GTK_PROGRESS_LEFT_TO_RIGHT)
          _closeDialogProgressBar->set_orientation(GTK_PROGRESS_RIGHT_TO_LEFT);
        else
          _closeDialogProgressBar->set_orientation(GTK_PROGRESS_LEFT_TO_RIGHT);
        percentage = 1.0;
      }
      else if (percentage < 0.01)
      {
        _closeDialogProgressIncrement = -_closeDialogProgressIncrement;
        
        percentage = 0.0;
      }
      _closeDialogProgressBar->set_percentage(percentage);
    }

    // quit if there is no device activity
    if (Device::noBusyDevices())
    {
      cerr << endl;
      Gnome::Main::quit();
    }
  }
  else
  {
    // process the status bar
    if (pthread_mutex_trylock(&_appbarMtx) == 0)
    {
      string newStatusMessage;
      gfloat newProgress;
      if (_stateQueue.size() == 0)
      {
        newStatusMessage = "";
        newProgress = 0.0;
      }
      else
      {
        newStatusMessage = _stateQueue.front()._statusMessage;
        newProgress = _stateQueue.front()._progress;
      }
      if (newStatusMessage != _lastIdleStatusMessage)
      {
        _appbar->set_status(newStatusMessage);
        _lastIdleStatusMessage = newStatusMessage;
      }
      if (newProgress != _lastIdleProgress)
      {
        _appbar->get_progress()->set_percentage(newProgress);
        _lastIdleProgress = newProgress;
      }
      pthread_mutex_unlock(&_appbarMtx);
    }

    // process the message window
    if (pthread_mutex_trylock(&_messagesMtx) == 0)
    {
      if (_messages.size() > 0)
      {
        _messageWindow->freeze();
        for (vector<Message>::iterator i = _messages.begin();
             i != _messages.end(); ++i)
        {
          string s = i->_message + "\n";
          if (i->_warning)
          {
            Gtk::Text_Helpers::Context gc;
            gc.set_foreground(Gdk_Color(config.getWarningColor()));
            _messageWindow->insert(gc, s);
          }
          else
            _messageWindow->insert(s);
        }
        _messageWindow->set_point(_messageWindow->get_length());
        _messageWindow->thaw();
        _messages.clear();

        // scroll down the message window
        Gtk::ScrolledWindow *w;
        LOOKUP_WIDGET(w, "main_messages_scrolledwindow", GtkScrolledWindow,
                      GTK_IS_SCROLLED_WINDOW, this);
        w->get_vscrollbar()->default_vmotion(0, 100);
      }
      pthread_mutex_unlock(&_messagesMtx);
    }
  }
  return TRUE;
}

void Main::sessionDie()
{
  Gnome::Main::quit();
}

gint Main::saveSession(gint phase, 
                       GnomeSaveStyle saveStyle,
                       gint isShutdown, GnomeInteractStyle interactStyle,
                       gint isFast)
{
  // FIXME
  return true;
}

Main::Main() :
  Gtk::Widget(create_main_window()),
  _client(Gnome::Client::master_client()),
  _closeDialogProgressBar(NULL),
  _waitingForClose(false), _nextProgressId(0), _lastIdleProgress(-1)
{
  // connect handlers
  _client->save_yourself.connect(slot(this, &Main::saveSession));
  _client->die.connect(slot(this, &Main::sessionDie));
  destroy.connect(slot(this, &Main::onDestroy));
  
  CONNECT_SIGNAL("main_open_phonebook", GtkMenuItem,
                 GTK_IS_MENU_ITEM, activate, this, Main::onPhonebook);
  CONNECT_SIGNAL("main_open_default_phonebook",
                 GtkMenuItem, GTK_IS_MENU_ITEM, activate,
                 this, Main::onDefaultPhonebook);
  CONNECT_SIGNAL("main_open_sms_store", GtkMenuItem,
                 GTK_IS_MENU_ITEM, activate, this, Main::onSMSStore);
  CONNECT_SIGNAL("main_open_default_sms_store",
                 GtkMenuItem, GTK_IS_MENU_ITEM,
                 activate, this, Main::onDefaultSMSStore);
  CONNECT_SIGNAL("main_send_new_sms", GtkMenuItem, GTK_IS_MENU_ITEM,
                 activate, this, Main::onSendNewSMS);
  //  CONNECT_SIGNAL("main_call_forwarding", GtkMenuItem, GTK_IS_MENU_ITEM,
  //                 activate, this, Main::onCallForwarding);
  CONNECT_SIGNAL("main_quit", GtkMenuItem, GTK_IS_MENU_ITEM,
                 activate, this, Gtk::Object::destroy_);
  CONNECT_SIGNAL("main_status", GtkMenuItem, GTK_IS_MENU_ITEM,
                 activate, this, Main::onStatus);
  CONNECT_SIGNAL("main_preferences", GtkMenuItem, GTK_IS_MENU_ITEM,
                 activate, this, Main::onPreferences);
  CONNECT_SIGNAL("main_about", GtkMenuItem, GTK_IS_MENU_ITEM,
                 activate, this, Main::onAbout);
  configure_event.connect(slot(this, &Main::onConfigureEvent));

  // initialize stop menu
  Gtk::MenuItem *mi;
  LOOKUP_WIDGET(mi, "main_stop", GtkMenuItem, GTK_IS_MENU_ITEM, this);
  _stopMenu = new Gtk::Menu();
  mi->set_submenu(*_stopMenu);
  Gtk::TearoffMenuItem *tmi = new Gtk::TearoffMenuItem();
  _stopMenu->insert(*tmi, 0);
  tmi->show_now();

  // initialize appbar interface
  LOOKUP_WIDGET(_appbar, "main_appbar", GnomeAppBar, GNOME_IS_APPBAR, this);
  pthread_mutex_init(&_appbarMtx, NULL);

  // initialize messages
  pthread_mutex_init(&_messagesMtx, NULL);
  LOOKUP_WIDGET(_messageWindow, "main_messages", GtkText, GTK_IS_TEXT, this);
}

void Main::showAll(bool configError)
{
  // set size
  int width, height;
  if (config.getMainSizes(width, height))
    ((Gtk::Window*)this)->set_default_size(width, height);

  // start idle functions
  Gtk::Main::timeout.connect(slot(this, &Main::idle), 100);

  // show the window
  show_all();

  // startup preferences window if there were errors in the config file
  if (configError) 
    new Preferences(true);
}

int Main::initProgress(string statusMessage)
{
  pthread_mutex_lock(&_appbarMtx);
  int result = _nextProgressId++;
  AppbarState as;
  as._statusMessage = statusMessage;
  as._progress = 0;
  as._progressId = result;
  _stateQueue.push_front(as);
  pthread_mutex_unlock(&_appbarMtx);
  return result;
}
    
void Main::updateProgress(int progressId, gfloat percentage)
{
  assert(progressId >= 0);
  pthread_mutex_lock(&_appbarMtx);
  for (deque<AppbarState>::iterator i = _stateQueue.begin();
       i != _stateQueue.end(); ++i)
    if (i->_progressId == progressId)
    {
      i->_progress = percentage;
      break;
    }
  pthread_mutex_unlock(&_appbarMtx);
}
    
void Main::finishProgress(int progressId)
{
  assert(progressId >= 0);
  pthread_mutex_lock(&_appbarMtx);
  for (deque<AppbarState>::iterator i = _stateQueue.begin();
       i != _stateQueue.end(); ++i)
    if (i->_progressId == progressId)
    {
      _stateQueue.erase(i);
      break;
    }
  pthread_mutex_unlock(&_appbarMtx);
}

void Main::onEditDone(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
    // handle error response
    Gnome::Dialogs::error
      (stringPrintf(_("Error updating file or device %s \n("),
                    ((ErrorResponse&)response()).sourceName().c_str()) +
       ((ErrorResponse&)response()).exception().what() + ")")->run_and_close();
  else
  {
    assert(typeid(response()) == typeid(UpdatePbEntryResponse));
  }
}

void Main::registerBusyDevice(string deviceName)
{
  if (! _waitingForClose && deviceName != "")
  {
    Gtk::MenuItem *m = new Gtk::MenuItem(deviceName);
    _stopMenu->insert(*m, 1);
    m->show_now();
    m->activate.connect(slot(this, &Main::onStop));
  }
}

void Main::unregisterBusyDevice(string deviceName)
{
  if (! _waitingForClose && deviceName != "")
  {
    // this roundabout way of deleting is necessary because iterators are
    // invalidated by the remove
    bool deleted;
    do
    {
      deleted = false;
      for (Gtk::Menu_Helpers::MenuList::iterator i =
             _stopMenu->items().begin();
           i != _stopMenu->items().end(); ++i)
        if (*i != NULL && (*i)->get_child() != NULL &&
            Gtk::Label::isA((*i)->get_child()) &&
            ((Gtk::Label*)(*i)->get_child())->get() == deviceName)
        {
          _stopMenu->remove(**i);
          deleted = true;
          break;
        }
    }
    while (deleted);
  }
}

void Main::message(string message, bool isWarning)
{
  // FIXME isWarning
  pthread_mutex_lock(&_messagesMtx);
  Message m;
  m._warning = isWarning;
  m._message = message;
  _messages.push_back(m);
  pthread_mutex_unlock(&_messagesMtx);
}

MultiEditor *Main::openDefaultPhonebook()
{
  MultiEditor *m = new PbMultiEditor();

  // open left phonebook pane with defaults
  if (config.getPhonebookDataSource(Left) == FromFile)
  {
    if (config.getPhonebookStoreName(Left) != "")
      m->leftEditor()->open(config.getPhonebookStoreName(Left));
  }
  else
  {
    if (config.getPhoneDevice() != "" &&
        config.getPhonebookStoreName(Left) != "")
      m->leftEditor()->open(config.getPhoneDevice(),
                            config.getBaudRate(),
                            config.getInitString(),
                            config.getSoftwareHandshake(),
                            config.getPhonebookStoreName(Left));
  }

  // open right phonebook pane with defaults
  if (config.getPhonebookDataSource(Right) == FromFile)
  {
    if (config.getPhonebookStoreName(Right) != "")
      m->rightEditor()->open(config.getPhonebookStoreName(Right));
  }
  else
  {
    if (config.getPhoneDevice() != "" &&
        config.getPhonebookStoreName(Right) != "")
      m->rightEditor()->open(config.getPhoneDevice(),
                             config.getBaudRate(),
                             config.getInitString(),
                             config.getSoftwareHandshake(),
                             config.getPhonebookStoreName(Right));
  }
  return m;
}

MultiEditor *Main::openDefaultSMSStore()
{
  MultiEditor *m = new SmsMultiEditor();

  // open left SMS pane with defaults
  if (config.getSMSDataSource(Left) == FromFile)
  {
    if (config.getSMSStoreName(Left) != "")
      m->leftEditor()->open(config.getSMSStoreName(Left));
  }
  else
  {
    if (config.getPhoneDevice() != "" &&
        config.getSMSStoreName(Left) != "")
      m->leftEditor()->open(config.getPhoneDevice(),
                            config.getBaudRate(),
                            config.getInitString(),
                            config.getSoftwareHandshake(),
                            config.getSMSStoreName(Left));
  }

  // open right SMS pane with defaults
  if (config.getSMSDataSource(Right) == FromFile)
  {
    if (config.getSMSStoreName(Right) != "")
      m->rightEditor()->open(config.getSMSStoreName(Right));
  }
  else
  {
    if (config.getPhoneDevice() != "" &&
        config.getSMSStoreName(Right) != "")
      m->rightEditor()->open(config.getPhoneDevice(),
                             config.getBaudRate(),
                             config.getInitString(),
                             config.getSoftwareHandshake(),
                             config.getSMSStoreName(Right));
  }
  return m;
}

#ifndef NDEBUG
int Xgsm::xgsmDebugLevel = 0;
#endif

// main program

Main *Xgsm::mainWindow;

int main (int argc, char *argv[])
{
  // create the Gnome application object
  string gnomeApp = XGSM_CONFIG_PATH;
  string gnomeAppVersion = "0.1";
  Gnome::Main gnomeMain(gnomeApp, gnomeAppVersion, argc, argv, NULL, 0, 0);
  
#ifndef NDEBUG
  char *s;
  if ((s = getenv("XGSM_DEBUG")) != NULL)
  {
    try
    {
      xgsmDebugLevel = checkNumber(s);
    }
    catch (GsmException &e)
    {
      Gnome::Dialogs::error(_("XGSM_DEBUG must be a number"));
    }
  }
#endif

  // create main window
  mainWindow = new Main();
  
  // load config file
  bool configError = false;
  try
  {
    config.load();
  }
  catch (GsmException &ge)
  {
    Gnome::Dialogs::error(ge.what())->run_and_close();
    configError = true;
  }

  mainWindow->showAll(configError);

  gnomeMain.run();

  DEBUG_START(1);
  cout << "before main() quit" << endl;
  DEBUG_END;

  exit(0);                      // FIXME

  return 0;
}
