
/*
 *  icontheme -   Gtkdoc comments in .h file.
 *
 *  Copyright (c) 2004 Edscott Wilson Garcia <edscott@imp.mx>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <sys/types.h>
#include <sys/wait.h>

#include <dbh.h>


static pid_t cache_manager=0;
static GList *base_dirs=NULL;
static GList *theme_list=NULL;
static DBHashTable *cache = NULL;
static int cache_size=0;
static const gchar *cache_file=NULL;
static gchar *requested_theme=NULL;

#ifndef HAVE_LIBXFCEGUI4
static
GdkPixbuf *
xfce_pixbuf_new_from_file_at_size(const gchar *filename, gint width, gint height,
		GError **error)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gdk_pixbuf_new_from_file_at_size(filename, width, height, error);
#else
	GdkPixbuf *pix, *tmp;
	gint w, h;
	
	pix = gdk_pixbuf_new_from_file(filename, error);
	if(pix) {
		w = gdk_pixbuf_get_width(pix);
		h = gdk_pixbuf_get_height(pix);
		if(w != width || h != height) {
			tmp = gdk_pixbuf_scale_simple(pix, width, height,
					GDK_INTERP_BILINEAR);
			g_object_unref(G_OBJECT(pix));
			pix = tmp;
		}
	}
	
	return pix;
#endif
}
#endif

static
const gchar *
get_supported_regex(void){
    GSList *pix_formats, *l;
    gchar **pix_extensions,**p; 
    static gchar *reg=NULL,*r=NULL;
    /* check SVG format support */
    if ((pix_formats = gdk_pixbuf_get_formats()) != NULL) {
	for(l = pix_formats; l; l = l->next) {
	    GdkPixbufFormat *fmt = l->data;
	    pix_extensions = gdk_pixbuf_format_get_extensions(fmt);
	    for(p=pix_extensions; *p; p++) {
		TRACE("supported=%s\n",*p);
		if (reg) {
		    g_free(r);
		    r=reg;
		    reg=g_strconcat(r,"|",*p,NULL);
		} else reg=g_strdup(*p);
	    }
	    g_strfreev(pix_extensions);
	}
	g_slist_free(pix_formats);
    }
    if (!reg) return "\\.(png|xpm)$)";
    g_free(r);
    r=g_strconcat("\\.(",reg,")$",NULL);
    g_free(reg); reg=NULL;

    TRACE("regex=%s\n",r);
    return (const gchar *) r;
}
    

static
gint  
compare_theme_info(gconstpointer a,gconstpointer b){
    themehash_info_t *theme_info_a = (themehash_info_t *)a;
    themehash_info_t *theme_info_b = (themehash_info_t *)b;
    return strcmp(theme_info_a->id, theme_info_b->id);
}


static 
GList * 
free_string_list (GList *list){	
    if (list){
	GList *tmp;
	for (tmp=list; tmp; tmp = tmp->next) g_free(tmp->data);
	g_list_free(list);
    }
    return NULL;
}



static 
void 
clear_bighash(gpointer key, gpointer value, gpointer user_data)
{
    icon_info_t *p,*icon_info_p = (icon_info_t *)value;
    g_free(key);
    if (!value) return;
    do {
	g_free(icon_info_p->path);
	g_free(icon_info_p->type);
	p=icon_info_p;
	icon_info_p = icon_info_p->next;
	g_free(p);
    } while (icon_info_p);
    return;
}

    
/*frees Glist of themes */
static 
GList * 
free_theme_list (GList *list){
    GList *tmp=list, *c_list;
    if (!list) return NULL;
    for (tmp=list; tmp; tmp = tmp->next) {
	    themehash_info_t *themehash_info_p;
	    themehash_info_p= (themehash_info_t *)tmp->data;
	    /* smallhashes has no allocated memory in hash table. All pointers
	     * belong to the bighash. */
	    for (c_list=themehash_info_p->smallhashes; c_list; c_list=c_list->next){
		smallhash_t *smallhash = (smallhash_t *)c_list->data;
		g_free(smallhash->context);
		/* pointers within small hashes are duplicated in bighash,
		 * so that they need not be freed at this step */
		g_hash_table_destroy(smallhash->hash);
		g_free(smallhash);
	    }
	    /* now we free memory */
	    g_list_free(themehash_info_p->smallhashes);
	    g_hash_table_foreach(themehash_info_p->bighash, clear_bighash, NULL);
            g_hash_table_destroy(themehash_info_p->bighash);	    
	    g_free(themehash_info_p->id);
	    g_free(themehash_info_p);
    }
    return NULL;
}

static
gchar *
dump_if_duplicate(gchar *X,gchar *Y){
    if (Y) {
	if (strcmp(X,Y)==0) {
	    g_free(Y);
	    return NULL;
	}
    }
    return Y;
}

#ifdef DEBUG
static
void 
showme_themeinfo(GList *list, gchar *tag){
    {
	GList *tmp=list;
	DBG("%s:\n",tag);
	for (; tmp; tmp=tmp->next){
	    themehash_info_t *theme_info = (themehash_info_t *)tmp->data;
	    if (!theme_info) continue;
	    DBG("\t%s\n",(gchar *)theme_info->id);
	}
    }
}
static
void 
showme(GList *list, gchar *tag){
    {
	GList *tmp=list;
	DBG("%s:\n",tag);
	for (; tmp; tmp=tmp->next)
	    DBG("\t%s\n",(gchar *)tmp->data);
    }
}

#endif

static 
void 
chop (gchar *g){
    if (!g || strlen(g)<2) return;
    if (*(g+(strlen(g)-1)) == G_DIR_SEPARATOR) *(g+(strlen(g)-1)) = 0;
}

static
GList *
get_base_dirs(GList *base_dirs){
    gchar *kdedir=NULL, *datadir=NULL, *datadir2=NULL, *homedir=NULL;
    gchar **p, **paths;
    
    base_dirs = free_string_list (base_dirs);
    
    if (VALID_KDEDIR) kdedir=g_build_filename(KDEDIR,NULL);
    if (PACKAGE_DATA_DIR) datadir=g_build_filename(PACKAGE_DATA_DIR, "icons", NULL);
    if (PACKAGE_DATA_DIR) datadir2=g_build_filename(PACKAGE_DATA_DIR, "pixmaps", NULL);
    homedir=g_build_filename(HOMEDIR, NULL);
    
    /* XFCE_RESOURCE_DATA, "icons/ */
    paths = xfce_resource_lookup_all(XFCE_RESOURCE_DATA, "icons/");
    if (!paths) TRACE("TRACE: paths is NULL\n");
    for (p=paths; *p; p++) {
	chop(*p);
	TRACE("TRACE: path:%s \n",*p);
	if (g_file_test(*p,G_FILE_TEST_IS_DIR))
	    base_dirs=g_list_append(base_dirs,(gpointer)g_strdup(*p));
	else TRACE("!G_FILE_TEST_EXISTS");
	datadir=dump_if_duplicate(*p,datadir);
	datadir2=dump_if_duplicate(*p,datadir2);
	kdedir=dump_if_duplicate(*p,kdedir);
	homedir=dump_if_duplicate(*p,homedir);
    }
    g_strfreev(paths); 
    /* KDE path and datadir */
    if (kdedir) base_dirs=g_list_append(base_dirs,(gpointer)kdedir);
    if (datadir) base_dirs=g_list_append(base_dirs,(gpointer)datadir);
    if (datadir2) base_dirs=g_list_append(base_dirs,(gpointer)datadir2);
    if (homedir) base_dirs=g_list_prepend(base_dirs,(gpointer)homedir);
#ifdef DEBUG
    showme(base_dirs,"base_dirs");
#endif
    return base_dirs;
}

static
gchar ** 
get_rc_info(gchar *index,gchar *rc_info){
    XfceRc *themerc;
    gchar **info=NULL;
    if((themerc = xfce_rc_simple_open(index, TRUE)) == NULL) return NULL;
    if(xfce_rc_has_group(themerc, "Icon Theme")) {
        xfce_rc_set_group(themerc, "Icon Theme");
        info = xfce_rc_read_list_entry(themerc, rc_info, ",");
    }
    xfce_rc_close(themerc);
    return info;
}

static
GList *
add_theme_to_list(GList *list, const gchar *id){
    themehash_info_t *themehash_info_p=(themehash_info_t *)malloc(sizeof(themehash_info_t));
    themehash_info_p->id = g_strdup(id);
    themehash_info_p->smallhashes = NULL;
    themehash_info_p->bighash = g_hash_table_new(g_str_hash, g_str_equal);
    list = g_list_append(list,(gpointer)themehash_info_p);
    return list;
}

static 
gchar *
theme_index(gchar *base_dir, gchar *theme){
    gchar *index;
    chop(base_dir);
    index=g_build_filename(base_dir, theme, NULL);
    TRACE("doing theme_index for %s",index);
    if (!g_file_test(index,G_FILE_TEST_IS_DIR)){
        g_free(index);
        return NULL;
    }
    g_free(index);
    index=g_build_filename(base_dir, theme, "index.theme", NULL);
    TRACE("looking for theme_index for %s",index);
    if (!g_file_test(index,G_FILE_TEST_EXISTS)){
        g_warning("%s does not exist",index);
        g_free(index);
        return NULL;
    }
    return index;
}

static
GList *
add_theme_name(GList *list, gchar *base_dir, gchar *theme){
    gchar *index=theme_index(base_dir, theme);
    if (!index) return list;
    else {
	themehash_info_t theme_info;
	theme_info.id=theme;
	if (!g_list_find_custom (theme_list,&theme_info, compare_theme_info)){
	    list = add_theme_to_list(list,theme);
	}
	TRACE("using theme defined with %s",index);
	g_free(index);
	return list;
    }
}

static
gboolean 
read_icon_directory(gchar *directory, themehash_info_t *themehash_info_p, 
		int size, const gchar *type, const gchar *context){
    GDir *dir;
    icon_info_t *icon_info_p;
    static regex_t supported;
    static gboolean regex_compiled=FALSE;
    const gchar *name;
    gchar *key;
    
    if (!regex_compiled) {
	const gchar *regex=get_supported_regex();
	if (regcomp(&supported, regex, REG_EXTENDED | REG_ICASE | REG_NOSUB)==0) regex_compiled=TRUE;
	else regex_compiled=FALSE;
    }
    dir = g_dir_open(directory, 0, NULL);
    TRACE("reading %s...(%s)\n",directory,((dir)?"yes":"no"));
    if (!dir) return FALSE;
     
    while((name = g_dir_read_name(dir))) {
        /*check for supported types*/
        if (regex_compiled && regexec(&supported, name, 0, NULL, 0)) {
            DBG("%s not supported",name);
            continue;
        }
        key = g_strdup(name);
        if (strchr(key,'.')) *strrchr(key,'.')=0;

        /* put in bighash */
        icon_info_p = g_hash_table_lookup(themehash_info_p->bighash,key);
        if (!icon_info_p){
	        icon_info_p = (icon_info_t *)malloc(sizeof(icon_info_t));
        } else {
		while (icon_info_p->next)icon_info_p= icon_info_p->next;
	        icon_info_p->next = (icon_info_t *)malloc(sizeof(icon_info_t));
	        icon_info_p = icon_info_p->next;
        }
        icon_info_p->next=NULL;
        icon_info_p->size=size;
        icon_info_p->type=g_strdup(type);
	icon_info_p->path=g_build_filename(directory,name,NULL);
        if (!g_hash_table_lookup(themehash_info_p->bighash,key)){
		TRACE("TRACE: putting into bighash: %s, size=%d\n",key,size);
            g_hash_table_insert(themehash_info_p->bighash, key, (gpointer) icon_info_p);		
        } else {
            /* key is already in the bighash */
		TRACE("key is already in the bighash: %s, size=%d\n",key,size);
            g_free(key);
            continue;
        }

	    /* put in smallhash */
        {
            GList *list;
            smallhash_t *smallhash_p; 
            for (list=themehash_info_p->smallhashes; list; list=list->next){
        	smallhash_p=(smallhash_t *) list->data; 
	    	if (strcmp(smallhash_p->context,context)==0) break;
	    }
	    if (!list){ /* find small hash associated to context */
		smallhash_p = (smallhash_t *)malloc(sizeof(smallhash_t));
		smallhash_p->context = g_strdup(context);
		smallhash_p->hash = g_hash_table_new(g_str_hash, g_str_equal);
		themehash_info_p->smallhashes = g_list_append(themehash_info_p->smallhashes,smallhash_p);
	    }
	    /* put entry key into smallhash too */
	    if (!g_hash_table_lookup(smallhash_p->hash,key)){
		g_hash_table_insert(smallhash_p->hash, key, (gpointer) icon_info_p);		
	    }
	}
    } /* end while read dir */
    g_dir_close(dir);
    return TRUE;
}

static
gboolean 
add_fallback (gchar *base_dir, const gchar *fallback){
    gchar **p,*directories[]={
	"48x48/apps",
	"48x48/stock/generic",
	"scalable/apps",
	NULL
    };
    themehash_info_t theme_info, *themehash_info_p=NULL;
    GList *list;

    if (!g_file_test(base_dir,G_FILE_TEST_IS_DIR)) return FALSE; 
    {
	gchar *g=g_build_filename(base_dir,fallback,NULL);
	if (!g_file_test(g,G_FILE_TEST_IS_DIR)){
	   g_free(g);
	   return FALSE; 
	}
	g_free(g);
    }
    
    theme_info.id=(gchar *)fallback;
    list=g_list_find_custom (theme_list,&theme_info, compare_theme_info);
	
    if (!list){
	list = theme_list= add_theme_to_list(theme_list,fallback);
    }
    if (list) themehash_info_p = (themehash_info_t *)list->data;
    
    if (themehash_info_p) for (p=directories; *p; p++){
	gchar *directory;
	directory=g_build_filename(base_dir,fallback,*p,NULL);
	read_icon_directory(directory,themehash_info_p,48,"Threshold","Misc");
	g_free(directory);
    } /* foreach directory */

    
    return TRUE;
}


static
gboolean 
add_theme_directories (gchar *base_dir, gchar *theme){
    gchar *index=theme_index(base_dir, theme);
    gchar **p,**directories;
    themehash_info_t theme_info, *themehash_info_p=NULL;
    GList *list;
    XfceRc *themerc;

    gchar *homedir;
    
    theme_info.id=theme;
    list=g_list_find_custom (theme_list,&theme_info, compare_theme_info);
	
    if (!list){
	return FALSE;
    }
    themehash_info_p = (themehash_info_t *)list->data;
   
    homedir=g_build_filename(HOMEDIR,NULL);
    if (strcmp(base_dir, homedir)==0){
	read_icon_directory(homedir,themehash_info_p,48,"Threshold","Misc");
	g_free(homedir);
	return TRUE;
    }
    g_free(homedir);
 
    TRACE("TRACE: add_theme_directories %s %s\n",base_dir, theme);
    
    if (!index) return FALSE;
	
    if ((directories=get_rc_info(index, "Directories"))==NULL){
	g_free(index);
	return FALSE;
    }
    TRACE("TRACE adding directories for %s/%s\n",base_dir,theme);

    
 
    themerc = xfce_rc_simple_open(index, TRUE);
    if(!themerc) g_assert_not_reached();
    for (p=directories; *p; p++){
	int size;
	const gchar *context, *type;
	gchar *directory;
	
	if(!xfce_rc_has_group(themerc, *p)) continue;
	xfce_rc_set_group(themerc, *p);
	
	/* All Rodent and gnome directories are of type scalable...?*/
	type = xfce_rc_read_entry(themerc, "Type", "Threshold");
	context = xfce_rc_read_entry(themerc, "Context", "Misc");
	size = atoi(xfce_rc_read_entry(themerc, "Size", "0"));

	directory=g_build_filename(base_dir,theme,*p,NULL);
	read_icon_directory(directory,themehash_info_p,size,type,context);
	g_free(directory);
    } /* foreach directory */
    xfce_rc_close(themerc);    
    g_free(index);
    return TRUE;
}


static
void
add_theme_inherits(gchar *base_dir, gchar *theme){
    gchar *index=theme_index(base_dir, theme);
    themehash_info_t theme_info;
    if (index) {
	gchar **inherits=NULL;
    
	/* get the inherit themes and add 'em in 
	 * we do not follow inherits from inherits to 
	 * avoid loop conditions from buggy icon.theme files.*/
	if ((inherits = get_rc_info(index,"Inherits"))!= NULL){
	    gchar **p;
	    for (p=inherits; *p; p++) {
		TRACE("inherits=%s\n",*p);
		theme_info.id = *p;
		if (!g_list_find_custom (theme_list, &theme_info, compare_theme_info)){
		    theme_list = add_theme_name(theme_list, base_dir, *p);
		    add_theme_directories(base_dir, *p);
		}
		else TRACE("inherits already listed=%s\n",*p);
	    }
	    g_strfreev(inherits);
	}	
	g_free(index);
    }
#ifdef HAVE_RODENT_ICONS     
    /* if Rodent is around, it will be the first fallback... */
    theme_info.id = "Rodent";
    if (!g_list_find_custom (theme_list, &theme_info, compare_theme_info)){
	theme_list = add_theme_name(theme_list, base_dir, theme_info.id);
	add_theme_directories(base_dir, theme_info.id);
    }
#else
#ifdef HAVE_GNOME_ICONS
    /* no Rodent, must be gnome lying around... */
    theme_info.id = "gnome";
    if (!g_list_find_custom (theme_list, &theme_info, compare_theme_info)){
	theme_list = add_theme_name(theme_list, base_dir, theme_info.id);
	add_theme_directories(base_dir, theme_info.id);
    }
#endif
#endif    
    /* hicolor is fallback for all themes, like it or not... */
    theme_info.id = "hicolor";
    if (!g_list_find_custom (theme_list, &theme_info, compare_theme_info)){
	theme_list = add_theme_name(theme_list, base_dir, theme_info.id);
	add_theme_directories(base_dir, theme_info.id);
    }
}

static
gboolean 
add_theme(GList *list,gchar *theme){
    GList *tmp;
    
    if (!theme_list) theme_list = add_theme_to_list(theme_list,theme);
    
    for (tmp=list; tmp; tmp=tmp->next){
	gchar *base_dir=(gchar *)tmp->data;
	theme_list = add_theme_name(theme_list, base_dir, theme);
	add_theme_directories(base_dir, theme);
    }
    
    /* inherit themes should go after all instances of requested theme,
     * that's why two loops...*/
    for (tmp=list; tmp; tmp=tmp->next){
	gchar *base_dir=(gchar *)tmp->data;
	add_theme_inherits(base_dir, theme);	
	add_fallback(base_dir,FALLBACK);
    }
    /* ultimate fallback, old style icon directories (pixmaps, icons) */
    {
	GList *tmp;
	themehash_info_t theme_info, *themehash_info_p=NULL;
	theme_info.id=(gchar *)FALLBACK;
	tmp=g_list_find_custom (theme_list,&theme_info, compare_theme_info);
	if (tmp) themehash_info_p = (themehash_info_t *)tmp->data;
	for (tmp=base_dirs; tmp && themehash_info_p; tmp=tmp->next){
	    TRACE("ultimate read: \t%s\n",(gchar *)tmp->data);
	    read_icon_directory((gchar *)tmp->data,themehash_info_p,48,"Threshold","Misc");
	}
    }
    
    
    return TRUE;
}
    
static
icon_info_t *
find_bighash(const gchar *key){
    icon_info_t *icon_info_p=NULL;
    GList *tmp;
    /* first try theme, then inherits */
    for (tmp=theme_list; tmp; tmp=tmp->next){
	themehash_info_t *themehash_info_p=(themehash_info_t *)tmp->data;
	TRACE ("TRACE:theme=%s\n",themehash_info_p->id);
	TRACE("hashsize=%d\n", g_hash_table_size   (themehash_info_p->bighash));
	icon_info_p = g_hash_table_lookup(themehash_info_p->bighash,key);
	if (icon_info_p) break;
    }
    return icon_info_p;
}

static
icon_info_t *
find_smallhash(const gchar *context,const gchar *key){
    icon_info_t *icon_info_p=NULL;
    GList *tmp,*l;
    /* first try theme, then inherits */
    for (tmp=theme_list; tmp; tmp=tmp->next){
	themehash_info_t *themehash_info_p=(themehash_info_t *)tmp->data;
	for (l=themehash_info_p->smallhashes; l; l = l->next) {
	    smallhash_t *smallhash_p=(smallhash_t *)l->data;
	    if (strcmp(context,smallhash_p->context)==0){
		icon_info_p = g_hash_table_lookup(smallhash_p->hash,key);
		break;
	    }
	}
	TRACE("theme=%s\n",themehash_info_p->id);
	if (icon_info_p) break;
    }
    return icon_info_p;
}

static 
const gchar *
select_best_match(icon_info_t *icon_info_pp, int size){
    int best_norm=999999;
    const gchar *best_match=NULL;
    icon_info_t *icon_info_p=icon_info_pp;
    for (;icon_info_p; icon_info_p = icon_info_p->next){
	int pseudosize;
	int norm;
	/* break on first exact size match, whatever the mimetype,
	 * (svg, png, bmp, xpm, etc) */
	if (size==icon_info_p->size) {
		TRACE("%s is exact match (%d)",icon_info_p->path,icon_info_p->size);
	    return (const gchar *)icon_info_p->path;
	}
#if 0
	/* disregard non-scalables (from index.theme definitions)*/
	/* this is currently disabled since some index.theme files
	 * (like gnome) may not use the Threshold type for pngs...  */
	if (strcmp("Scalable",icon_info_p->type)!=0) pseudosize=size;
	  else pseudosize=icon_info_p->size; 
#else
	/* prefer svg icons before any other type (this replaces the
	 * disabled code above)*/
	if (g_str_has_suffix(icon_info_p->path,".svg")) pseudosize=size;
	else pseudosize=icon_info_p->size;
#endif
	/* get closest match */
	norm = abs(size - pseudosize);
	if (!best_match || norm < best_norm) {
	    best_match = icon_info_p->path;
	    best_norm = norm;
	    if (strstr(icon_info_p->path,"gnome-fs-directory")) 
		TRACE("%s looks like best match (%d) %d < %d",
			icon_info_p->path,pseudosize,norm, best_norm);
	}
    }
    return best_match;
}

static
const gchar *
find_icon_path_priv(const gchar *key,int size,const gchar *context, gboolean pirate){
    icon_info_t *icon_info_p=NULL;
    const gchar *best_match=NULL;
    
    /* shortcircuit with cache  */
    if (cache_file != NULL){
	GString *gs;
	static gchar *cache_hit=NULL;
	/* obtain file lock for cache */
	
	if((cache = DBH_open((char *)cache_file)) == NULL) {
	    /* if cache file is corrupted by third party, dump it
	     * and fall back to bighash behaviour for this session.*/
	    DBG("removing corrupted cache file...");
	    unlink(cache_file);
	    cache_file=NULL;
	    base_dirs = get_base_dirs(base_dirs);
	    open_theme(requested_theme, 0);
	} else {
	    /* obtain lock */
	    TRACE("obtain lock for cache file...");
	    lockf(fileno(cache->database),F_LOCK, 0);
	    TRACE("lock obtained for cache file");
	    
	    gs = g_string_new(key);
	    sprintf((char *)DBH_KEY(cache), "%10u", g_string_hash(gs));
	    g_string_free(gs, TRUE);
	    
	    if (!DBH_load(cache)){
		DBH_close(cache);
		if (!pirate) {
		    DBG("icon not found: %s", key);
		    return NULL;
		}
		return find_icon_path_priv("pirate",size,context,FALSE);
	    }
	    g_free(cache_hit);
	    cache_hit = g_strdup((gchar *)cache->data);
	    DBH_close(cache);
	    return (const gchar *) cache_hit;
	}
    }
    
    if (!theme_list){
	TRACE("Cannot find icon for %s (theme is not open)",key);
	return NULL;
    }
    if (!key) return NULL;
    if (context) icon_info_p = find_smallhash(context,key);
    if (!icon_info_p) icon_info_p=find_bighash(key);
    if (!icon_info_p) {
	TRACE("Cannot find icon for %s",key);
	return NULL;
    }
    /* now select the best match... */
    best_match=select_best_match(icon_info_p,size);
    if (!best_match) {
	g_warning("no icon match for %s",key);
    }
    return best_match;
}


/***********************************************************************************/
/*cache stuff*/
static 
int 
check_dir(char *path)
{
    struct stat st;
    if(stat(path, &st) < 0)
    {
	if(mkdir(path, 0770) < 0)
	    return FALSE;
	return TRUE;
    }
    if(S_ISDIR(st.st_mode))
    {
	if(access(path, W_OK) < 0)
	    return FALSE;
	return TRUE;
    }
    return FALSE;
} 

static
const gchar *
get_cache_path (const gchar *theme, int size){
    static gchar *cache_path=NULL;
    gchar *cache_dir;
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);

    cache_dir = g_build_filename(xdg_dir,CACHE_DIR,NULL);    
    g_free(xdg_dir);
    if(!check_dir(cache_dir)) {
	    g_free(cache_dir);
	    return NULL;
    }
    if (cache_path)  g_free(cache_path);
    
    cache_path = g_strdup_printf("%s%c%s.%d.cache.dbh",cache_dir,G_DIR_SEPARATOR,theme,size);
    g_free(cache_dir);
    TRACE("using cache: %s\n",cache_path);
    return (const gchar *)cache_path;
}


static 
void 
add2cache(gpointer key, gpointer value, gpointer user_data)
{
    icon_info_t *icon_info_p = (icon_info_t *)value;
    GString *gs;
    if (!value || !cache) return;
    gs = g_string_new(key);
    sprintf((char *)DBH_KEY(cache), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    if (!DBH_load(cache)){
	const gchar *best_match=select_best_match(icon_info_p, cache_size);
	if (best_match){
		TRACE("size=%d, adding %s (%s) to cache: %s\n",cache_size,(char *)key,cache->key,best_match);
	    strcpy((gchar *)cache->data,best_match);
	    DBH_set_recordsize(cache, strlen(best_match)+1);
	    DBH_update(cache);
	}      
    }
    else TRACE("TRACE: %s already in cache\n",(char *)key);
    return;
}


static long long recurse_basedir_sum(gchar *dir){
    long long sum=0;
    const char *file;
    GDir *gdir;
    if ((gdir = g_dir_open(dir, 0, NULL)) != NULL) {
      while((file = g_dir_read_name(gdir))) {
	gchar *path = g_build_filename(dir, file, NULL);
	if (g_file_test(path,G_FILE_TEST_IS_SYMLINK)){
	    g_free(path);
	    continue;
	}
	if (g_file_test(path, G_FILE_TEST_IS_DIR))
	{
	    struct stat st;
	    /* recurse */
	    sum += recurse_basedir_sum(path);
	    if (stat(path,&st)==0) {
		sum += st.st_mtime;
		sum += st.st_dev;
	    }
	}
	g_free(path);
      }
      g_dir_close(gdir);
    }
    return sum;
}

/* must get info for all directories within */

static long long get_basedir_sum(void){
    long long basedir_sum=0;
    GList *list;
    for (list=base_dirs;list;list=list->next){
	struct stat st;

	if (stat((gchar *)list->data,&st)==0) {
	    TRACE("stat ok");
	    basedir_sum += st.st_mtime;
	    basedir_sum += st.st_dev;
	}
	basedir_sum += recurse_basedir_sum((gchar *)list->data);
	DBG("basedir summing %s (%ld %ld)",(gchar *)list->data,(long)st.st_mtime,(long)st.st_dev);
    }
   basedir_sum +=SERIAL;
   return basedir_sum;
}

static 
void
save_cache_info(gchar *theme, int size){
    FILE *info;
    gchar *cache_info_path=g_strconcat(cache_file,".info",NULL);
    cache_info_t cache_info;
    if (!cache_file) return;
    cache_info_path=g_strconcat(cache_file,".info",NULL);
    TRACE("saving %s",cache_info_path);
    
    cache_info.basedir_sum=get_basedir_sum();
    strncpy(cache_info.supported_regex,get_supported_regex(),255);
    cache_info.supported_regex[255]=0;
    info=fopen(cache_info_path,"wb");
    if (info) {
	if (fwrite(&cache_info,sizeof(cache_info_t),1,info)<1){
	    g_warning("cannot write to %s",cache_info_path);
	}
	fclose(info);
    } else g_warning("cannot write to %s",cache_info_path);
    TRACE("wrote basedir sum is %lld",cache_info.basedir_sum);
    g_free(cache_info_path);
}

/* criteria for cache regeneration:
 * 1- does not exist
 * 2- one of the base_dirs has been modified
 * 3- supported regexp has changed.
 *
 * This info will be kept in file "theme.size.cache.info"
 * */
static 
gboolean
compare_cache_info(gchar *theme, int size){
    FILE *info;
    gchar *cache_info_path;
    cache_info_t cache_info,disk_cache_info;
	
    DBG("comparing cache info for %s (%d)",theme, size);

    if (!g_file_test(cache_file,G_FILE_TEST_EXISTS)) {
	DBG("%s not exists",cache_file);
	return FALSE;
    }
    
    cache_info_path=g_strconcat(cache_file,".info",NULL);
    if ((info=fopen(cache_info_path,"rb"))==NULL){
	DBG("%s not exists",cache_info_path);
	g_free(cache_info_path);
	return FALSE;
    }
    g_free(cache_info_path);
    
    fread(&disk_cache_info,sizeof(cache_info_t),1,info);
    fclose(info);
    DBG("read basedir cache sum is %lld",disk_cache_info.basedir_sum);
    
    cache_info.basedir_sum=get_basedir_sum();
    DBG("cache sum is %lld",cache_info.basedir_sum);
    if (cache_info.basedir_sum != disk_cache_info.basedir_sum) {
	DBG("cache_info.basedir_sum(%lld) != disk_cache_info.basedir_sum(%lld)",cache_info.basedir_sum, disk_cache_info.basedir_sum);
	return FALSE;
    }
    
    strncpy(cache_info.supported_regex,get_supported_regex(),255);
    cache_info.supported_regex[255]=0;
    if (strcmp(cache_info.supported_regex,disk_cache_info.supported_regex)) {
	DBG("cache_info.supported_regex != disk_cache_info.supported_regex");
	return FALSE;
    }
    return TRUE;
}


static 
gboolean
generate_cache (gchar *theme, int size){
    GList *list;
    if (!cache_file || !base_dirs) return FALSE;
    cache_size = size;
    /*if((cache = DBH_open(cache_path)) == NULL)*/
    {
	if((cache = DBH_create((char *)cache_file, 11)) == NULL) return FALSE;
    }
    for (list=theme_list; list; list=list->next){
	themehash_info_t *themehash_info_p = (themehash_info_t *)list->data;
	/*if (strcmp(theme,themehash_info_p->id)) continue;*/
	g_hash_table_foreach(themehash_info_p->bighash, add2cache, NULL);	
	TRACE("%s, hashsize=%d\n", themehash_info_p->id, g_hash_table_size   (themehash_info_p->bighash));
    }
    DBH_close(cache);
    save_cache_info(theme,size);
    return TRUE;
}


#if 0

/* for testing... 
 * uncomment this stuff and create a makefile:
 * the result is a program that queries you for the
 * icon name and returns the best match, trying first
 * the Mimetypes context, then all contexts. Then it 
 * gives you the path for best matches on sizes 36, 48, 58.
 * Size 48 should generally give an exact match, other
 * sizes an svg icon, if available.
 *
 * Regular expression for supported icon types that is
 * generated on my box (FYI):
 *
 * \.(wbmp|jpeg|jpe|jpg|ani|bmp|gif|ico|cur|pcx|png|pnm|pbm|pgm|ppm|ras|tga|targa|xbm|xpm|svg|svgz)$
 * 
 * */

int main(int argc, char ** argv){
    gchar *theme;
    if (argc < 2) {
	g_warning("Theme not specified, using Rodent");
	theme="Rodent";
    } 
    else theme=argv[1];
    if (!open_theme(theme,48)){
	g_error("cannot open theme %s",theme);
    }
    TRACE("Load ok\n"); 
	
#if DEBUG
    while (1) {
	icon_info_t *icon_info_p;
	gchar buf[256];
	TRACE("key:"); fflush(NULL);
	scanf("%s",buf);
	if (strchr(buf,'\n')) *strchr(buf,'\n')=0;
#ifdef DEBUG
	icon_info_p=find_smallhash("MimeTypes",buf);
	if (!icon_info_p) icon_info_p=find_bighash(buf);
	if (icon_info_p) {
	    icon_info_t *tmp;
	    for (tmp=icon_info_p; tmp; tmp=tmp->next){
		TRACE("size=%d,path=%s\n",tmp->size,tmp->path);
	    }
	} else TRACE("not found\n");
#endif

	TRACE("Best match for size 36 is = %s\n",find_icon_path(buf,36,NULL));
	TRACE("Best match for size 48 is = %s\n",find_icon_path(buf,48,NULL));
	TRACE("Best match for size 58 is = %s\n",find_icon_path(buf,58,NULL));
    }
#endif
}

#endif
