/*
 * Copyright (C) 2002-5 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define XFFM_PRIMARY_C

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <regex.h>
#include <dirent.h>
#include <X11/Xlib.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>


#include "constants.h"
#include "types.h"

#include "primary.h"
#include "primary.i"

G_MODULE_EXPORT
gchar *workdir = NULL;

static gchar *text_editors[]={
    "gvim",
    "emacs",
    "xedit",
    NULL
};
static gchar *text_types[]={
    "text/",
    "application/x-csh",
    "application/x-bsh",
    "application/x-ksh",
    "application/x-python",
    "application/x-shellscript",
    "application/x-tsh",
    "application/x-zsh",
    "application/xhtml+xml",
    "application/rtf",
    "application/x-cgi",
    "application/x-perl",
    "application/x-php",
    "application/x-theme",
    "application/x-trash",
    NULL
};

G_MODULE_EXPORT
gchar **text_type_OK (const gchar *path)
{ 
  const gchar *t=MIME_get_type((const gchar *)path, TRUE);
  if (t) {
      int i;
      TRACE("text_type_OK: mimetype for %s=%s",path,t);
      for (i=0;text_types[i];i++) if (strncmp(text_types[i],t,strlen(text_types[i]))==0){
	  TRACE(" %s==%s",text_types[i],t);	  
	  return text_editors;
      }
  }
  return NULL;
}



/**
 * primary_entry.c:
 *
 * shared entry structure functions.
 * @entry_compare
 * @mk_entry
 * @mk_entry_path
 * @stat_entry
 * @mk_net_entry
 * @destroy_entry
 *
 * **/

/**
 * entry_compare:
 *
 * Compare function for quick sort of array of entries
 *
 * @caso:  sort column id. GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID for default.
 * 
 * **/


static int ascending=0;
G_MODULE_EXPORT
void set_ascending(int asc){
    ascending=asc;
}
    
static 
gint
pathv_compare	(const gchar *a, const gchar *b){
    gboolean asc;
    /*int ascending = GET_ASCENDING;*/
    switch (ascending %3){
	case 0:
	case 1:	  asc=TRUE; break; 
	default:  asc=FALSE;
    }
    if (!a && !b) return 0;
    if (!a ) return (asc)?-1:1;
    if (!b ) return (asc)?1:-1;
    if (strchr(a,G_DIR_SEPARATOR)) a=strrchr(a,G_DIR_SEPARATOR)+1;
    if (strchr(b,G_DIR_SEPARATOR)) b=strrchr(b,G_DIR_SEPARATOR)+1;
    return strcmp(a,b);
}


G_MODULE_EXPORT
gint entry_compare (	int caso,
			record_entry_t *en_a,
			record_entry_t *en_b)
{
    gboolean d1d, d2d; /* d1, d2 is directory flag */
    gboolean asc;
    /*int ascending = GET_ASCENDING;*/
    switch (ascending %3){
	case 0:
	case 1:	  asc=TRUE; break; 
	default:  asc=FALSE;
    }
   	if (!en_a && !en_b) return 0;
	if (!en_a ) return (asc)?-1:1;
	if (!en_b ) return (asc)?1:-1;
	if (IS_DUMMY_TYPE(en_a->type)) return (asc)?-1:1;
	if (IS_DUMMY_TYPE(en_b->type)) return (asc)?1:-1;

	if (!en_a->path && !en_b->path) return 0;
	if (!en_a->path ) return (asc)?-1:1;
	if (!en_b->path ) return (asc)?1:-1;
	
	d1d = IS_DIR(en_a->type) | IS_NETDIR(en_a->subtype);
	d2d = IS_DIR(en_b->type) | IS_NETDIR(en_b->subtype);

	if (d1d && !d2d)  return (asc)?-1:1;
	if (!d1d && d2d)  return (asc)?1:-1;
	
	
	/* subsorting... */
	if (caso ==GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID  && IS_FILE(en_a->type) && IS_FILE(en_b->type)){
	    char *a,*b;
	    a=strrchr(en_a->path,'.');
	    b=strrchr(en_b->path,'.');
	    if (a || b) {
		if (!a) return -1;
		if (!b) return 1;
		if (strcmp(a,b)) return strcmp(a,b); 
	    }
	}
		
	

	switch (caso){   
	    case GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID:
	    case NAME_COLUMN: 
	    {
		gchar *a=en_a->path,*b=en_b->path;
		if (strchr(a,G_DIR_SEPARATOR)) a=strrchr(a,G_DIR_SEPARATOR)+1;
		if (strchr(b,G_DIR_SEPARATOR)) b=strrchr(b,G_DIR_SEPARATOR)+1;
		return strcmp(a,b);
	    }
	    default:
	  	if (!en_a->st && !en_b->st) return 0;
		if (!en_a->st ) return (asc)?-1:1;
		if (!en_b->st ) return (asc)?1:-1;
		switch (caso) {
		    case SIZE_COLUMN: return en_a->st->st_size - en_b->st->st_size;
		    case DATE_COLUMN: return en_a->st->st_mtime - en_b->st->st_mtime;
		    case OWNER_COLUMN: return en_a->st->st_uid - en_b->st->st_uid;
		    case GROUP_COLUMN: return en_a->st->st_gid - en_b->st->st_gid;
		    case MODE_COLUMN: return en_a->st->st_mode - en_b->st->st_mode;
		}
	}
	return 0;   
}

static int relative_sort_column=GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID;
G_MODULE_EXPORT
void set_sort_column(int column)
{	
    relative_sort_column = column;
}

G_MODULE_EXPORT
int
xfdir_compare(		const void *a, 
			const void *b)
{
    int result;
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    if (!d1 && !d2) return 0;
    if (!d1) return -1;
    if (!d2) return 1;

    if (ascending%3==2){
        /* equivalent to no sort (treeview uses 3 states:
         * no sort is ascending by name and
         * subsorted by filetype).*/
        d2 = (const dir_t *)a;
        d1 = (const dir_t *)b;
    }
    /* we sort here by displayed name, not by the path's basename,
     * which is what entry compare will do... */
    
    if (relative_sort_column == NAME_COLUMN) {
	if (d1->en && d2->en) { /* directories on top */
	    gboolean d1d = IS_DIR(d1->en->type) | IS_NETDIR(d1->en->subtype);
	    gboolean d2d = IS_DIR(d2->en->type) | IS_NETDIR(d2->en->subtype);
	    if (d1d && !d2d)  return -1;
	    if (!d1d && d2d)  return 1;
	}

	result = pathv_compare(d1->pathv, d2->pathv);
	if (result) return result;
	/* break name ties with date: */
	return (entry_compare(DATE_COLUMN,d1->en,d2->en));
    }
    result = entry_compare(relative_sort_column,d1->en,d2->en);
    if (result) return result;
    /* break other ties with name */
    return pathv_compare(d1->pathv, d2->pathv);
    
    return (entry_compare(relative_sort_column,d1->en,d2->en));
}


G_MODULE_EXPORT
record_entry_t *
copy_entry(record_entry_t *src_en)
{
    record_entry_t *en;
    if (!src_en) return NULL;
    en = (record_entry_t *) malloc(sizeof(record_entry_t));
    if(!en) assert_not_reached();
    en->type = src_en->type;
    en->subtype = src_en->subtype;
    en->count = src_en->count;
    
    if (src_en->st) {
	en->st = (struct stat *)malloc(sizeof(struct stat));
	memcpy(en->st,src_en->st,sizeof(struct stat));
    } else en->st = NULL;
    
    en->tag = (src_en->tag)?g_strdup(src_en->tag):NULL;
    en->path = (src_en->path)?g_strdup(src_en->path):NULL;
    en->filter = (src_en->filter)?g_strdup(src_en->filter):NULL;
    en->module = (src_en->module)?g_strdup(src_en->module):NULL;
    en->load_time = src_en->load_time;
    en->checksum = src_en->checksum;
    en->icon = (src_en->icon)?g_strdup(src_en->icon):NULL;
    en->exec = (src_en->exec)?g_strdup(src_en->exec):NULL;
    
    return en;
}

G_MODULE_EXPORT
record_entry_t *
mk_entry (		int type)
{
    record_entry_t *en;
    en = (record_entry_t *) malloc(sizeof(record_entry_t));
    if(!en)
	assert_not_reached();
    en->type = 0;
    en->subtype = 0;
    en->filter = g_strdup("*");
    en->count = -1;
    en->load_time = 0;
    en->module = NULL;
    en->path = NULL;
    en->tag = NULL;
    en->st = NULL;
    en->icon = NULL;
    en->exec = NULL;
    INHERIT_TYPE(en->type, type);	    
    if (xffm_details->preferences & IMAGE_PREVIEW) SET_SHOWS_IMAGES(en->type);
    if (xffm_details->preferences & SHOW_DOT) SET_SHOWS_HIDDEN(en->type);

    return en;
}

G_MODULE_EXPORT
record_entry_t *
mk_entry_path (		const gchar *path, 
			int type)
{
    record_entry_t *en;
    en = mk_entry(type);
    en->path = g_strdup(path);
    if(access(path, W_OK) < 0)
    {
	SET_NOWRITE(en->type);
    }
    if(access(path, R_OK) < 0)
    {
	SET_NOACCESS(en->type);
    }
    if(access(path, X_OK) >= 0)
    {
	/* this fails for priviledged users so retest */
	struct stat s;
	if (stat (path, &s) != -1 && 
	    ((s.st_mode & S_IXUSR) || 
	     (s.st_mode & S_IXGRP) || 
	     (s.st_mode & S_IXOTH)) ){
		SET_EXE(en->type);
	}
    }
    return en;
}

G_MODULE_EXPORT
record_entry_t *
stat_entry (		const gchar *path, 
			int type)
{
    record_entry_t *en;
    struct stat *st, s;
    st = (struct stat *)malloc(sizeof(struct stat));

    en = mk_entry_path(path, type);
    /*SET_LOCAL_TYPE(en->type);  default */

    if(!st)
	assert_not_reached();
    else if(stat(path, st) < 0)
    {
	if(lstat(path, st) < 0)
	{
	    destroy_entry(en);	    
	    g_free(st);
	    st=NULL;
	    return NULL;
	}
	SET_BROKEN_LNK(en->type);
	en->st = st;
	return en;
    }
    en->st = st;
    if (lstat(path, &s) < 0){
	    destroy_entry(en);	    
	    g_free(st);
	    st=NULL;
	    return NULL;
    }
    
    if(S_ISLNK(s.st_mode))
    {
	SET_XF_LNK(en->type);
    }
    else if(S_ISDIR(s.st_mode))
    {
	SET_XF_DIR(en->type);
    }
    else if(S_ISSOCK(s.st_mode))
    {
	SET_XF_SOCK(en->type);
    }
    else if(S_ISBLK(s.st_mode))
    {
	SET_XF_BLK(en->type);
    }
    else if(S_ISCHR(s.st_mode))
    {
	SET_XF_CHR(en->type);
    }
    else if(S_ISFIFO(s.st_mode))
    {
	SET_XF_FIFO(en->type);
    }
    else
    {
	SET_XF_REG(en->type);
    }

    if(S_ISDIR(st->st_mode))
    {
	SET_DIR(en->type);
	en->count = count_files(en->path);
    }

    return en;
}

G_MODULE_EXPORT
record_entry_t *
mk_net_entry (		const gchar *path,
			unsigned type)
{
   record_entry_t *en;
    char *p,*t;
   
    /* posible stuff (printers not yet defined as netstuff):
	     * XF_NETWS    smb://XXX@YYY:
	     * XF_NETSHARE smb://XXX@YYY:ZZZ
	     * NETFILE     smb://XXX@YYY:ZZZ/UUU
	     * NETDIR      smb://XXX@YYY:ZZZ/VVV/
     * */

   p=g_strdup(path+strlen("smb://"));

   en=mk_entry(type);
   if (p[strlen(p)-1]==':') SET_XF_NETWS(en->subtype);
   else if (p[strlen(p)-1]=='/') SET_NETDIR(en->subtype);
   else if (strchr(p,'/')) SET_NETFILE(en->subtype);
   else SET_XF_NETSHARE(en->subtype);
			  
	  	  
   en->st=(struct stat *)malloc(sizeof(struct stat));
   en->st->st_size=0;
   en->st->st_mtime=0;
   en->st->st_ctime=0;
   en->st->st_gid=-1;
   en->st->st_uid=-1;
   en->st->st_mode=0;
   t=strtok(p,"@");
   if (!t) assert_not_reached();
   en->tag=g_strdup(t);

   if (IS_XF_NETWS(en->subtype)){
       t=strtok(NULL,":"); 
   } 
   else {
        t=t+strlen(t)+1;
        *(strchr(t,':'))='/';
   }
   en->path=(char *)malloc(strlen(t)+3);
   sprintf(en->path,"//%s",t);
   g_free(p);
   p=NULL;
   return en;
}


G_MODULE_EXPORT
void 
destroy_entry (		record_entry_t * en)
{
    if(!en) return;
    g_free(en->filter);
    g_free(en->st);
    g_free(en->path);
    g_free(en->tag);
    g_free(en->exec);
    g_free(en->icon);
    g_free(en);
    en=NULL;
}

G_MODULE_EXPORT
void set_entry_tag (record_entry_t * en, off_t tama) 
{
    int hcount;

    hcount = count_hidden_files(en->path);
    if(hcount)
    {
	SET_HAS_HIDDEN(en->type);
    }
    if (en->tag) g_free(en->tag);
    if(SHOWS_HIDDEN(en->type))
	en->tag = g_strdup_printf("%s (%s %s)", my_utf_string(FILENAME(en)), sizetag(tama, en->count), _("Showing hidden."));
    else
    {
	if(hcount)
	    en->tag = g_strdup_printf("%s (%s %d %s)",  my_utf_string(FILENAME(en)), sizetag(tama, en->count), hcount, _("hidden."));

	else
	    en->tag = g_strdup_printf("%s (%s %s)",  my_utf_string(FILENAME(en)), sizetag(tama, en->count), _("No hidden."));
    }
}

/* gui */
static gboolean pending_busy=FALSE;
G_MODULE_EXPORT
gboolean processing_pending (void){
	return pending_busy;
}

G_MODULE_EXPORT
void set_processing_pending (void){
	pending_busy=TRUE;
}
G_MODULE_EXPORT
void unset_processing_pending (void){
	pending_busy=FALSE;
}

G_MODULE_EXPORT
void process_pending_gtk (){
  int i;
  static gboolean recursive=FALSE;
  /* breakout if recursive call */
  if (recursive) return;
  if (processing_pending()) return;
  recursive=TRUE;
  for (i=0;i<256;i++){
	  if (!gtk_events_pending()) break;
	  else gtk_main_iteration();
  }
  recursive=FALSE;  
}
G_MODULE_EXPORT
void 
cursor_wait(	GtkWidget *w)
{
    static GdkCursor *cursor = NULL;
    if(!w) return;
    if(!cursor)	cursor = gdk_cursor_new(GDK_WATCH);
    gdk_window_set_cursor(w->window, cursor);
    gdk_flush();
}


G_MODULE_EXPORT
void 
cursor_reset(	GtkWidget *w)
{
    if(!w) return;
    gdk_window_set_cursor(w->window, NULL);

}


/* local files */
G_MODULE_EXPORT
gint 
count_files (gchar *path)
{
    DIR *directory;
    struct dirent *dir;
    gint count = 0;

    directory = opendir(path);
    if(!directory)
	return -1;
    while((dir = readdir(directory)) != NULL)
    {
	if(strcmp(dir->d_name, ".") && strcmp(dir->d_name, ".."))		
		count++;
    }
    closedir(directory);
    return (count);
}


G_MODULE_EXPORT
gboolean 
get_xfdir_local(	xfdir_t *xfdir_p,
			record_entry_t *en,
			const gchar *filter)
{
    const regex_t *preg=NULL;
    gint j;
    off_t tama = 0;
     

    xfdir_p->gl=NULL;
    
    if(!en){
	g_warning("en != NULL");
	return FALSE;
    }
    
    if (filter && strcmp(filter,"*") != 0) {
	g_free(en->filter);
	en->filter = g_strdup(filter);
	preg = compile_regex_filter(en->filter,SHOWS_HIDDEN(en->type));
    }
    
    /* cache short circuit (only if not using filter) */
    if (!preg && read_xfdir_cache(en,xfdir_p)){
	/* calculate checksums for the directory (for later use by monitor) 
	 * */
	TRACE("---------cache read");
	if (!xfdir_p->pathc) return TRUE;
	for(j = 0; j < xfdir_p->pathc; j++) if (xfdir_p->gl[j].en && xfdir_p->gl[j].en->st)
	{
	    tama += xfdir_p->gl[j].en->st->st_size;	
	    en->checksum += (xfdir_p->gl[j].en->st->st_mtime + xfdir_p->gl[j].en->st->st_ctime);
	}
	set_entry_tag(en, tama);
	return TRUE;
    }
    
    /* read part */
    /* count step */
    xfdir_p->pathc = read_files_local(en->path, en->type, preg, NULL);
    if(xfdir_p->pathc < 0) {
	return FALSE;
    }
    
    if(xfdir_p->pathc)
    {
	xfdir_p->gl = (dir_t *) malloc(xfdir_p->pathc * sizeof(dir_t));
	if(!xfdir_p->gl)
	{
	    return FALSE;
	} else {
	    memset(xfdir_p->gl,0,xfdir_p->pathc * sizeof(dir_t));
	}

	if(read_files_local(en->path, en->type, preg, xfdir_p) < 0)
	{
	    return FALSE;
	}

	/* calculate checksums for the directory (for later use by monitor) 
	 * */
	for(j = 0; j < xfdir_p->pathc; j++) 
	    if (xfdir_p->gl[j].en && xfdir_p->gl[j].en->st)
	{
	    tama += xfdir_p->gl[j].en->st->st_size;	
	    en->checksum += (xfdir_p->gl[j].en->st->st_mtime + xfdir_p->gl[j].en->st->st_ctime);
	}
	set_entry_tag(en, tama);
	if (xfdir_p->pathc > 52) save_xfdir_cache(en,xfdir_p);
	
    }
    return TRUE;
}

/* 
 * field xfdir->gl[i].en not freed since it is transferred to the 
 * treemodel row or girdview population.
 * */
G_MODULE_EXPORT
void 
xfdirfree (	xfdir_t * xfdir)
{
    if (!xfdir) return;
    if(xfdir->gl != NULL)
    {
	int i;
	for(i = 0; i < xfdir->pathc; i++) g_free(xfdir->gl[i].pathv);
	g_free(xfdir->gl);
	xfdir->gl=NULL;
    }
}

G_MODULE_EXPORT
void rm_cache_local(	const gchar *directory){
    if (!directory) return;
    unlink((char *)get_local_cache_path((char *)directory));
}

G_MODULE_EXPORT
const regex_t *
compile_regex_filter(		const gchar *filter,
				gboolean shows_hidden)
{
#ifndef USE_FILTER_BAR
    return NULL;
#else
    static regex_t preg;
    const char *regex;
    static gboolean compiled=FALSE;

        
    if (!filter || !strlen(filter) || strcmp(filter,"*")==0) return NULL;
    regex = filter2regex(filter, shows_hidden);
    

    /*printf("compile_regex_filter(%s)\n",filter);*/
    if (compiled) regfree(&preg);
    if (regcomp(&preg, regex, REG_EXTENDED | REG_ICASE | REG_NOSUB)==0)
	compiled=TRUE;
    else 
	compiled=FALSE;
    return &preg;
#endif
}

G_MODULE_EXPORT    
gint 
all_files_count(void)
{
    return all_files;
}

G_MODULE_EXPORT
void init_xffm_env(void){
	int i;
	for (i=0;env_vars[i];i++)
		env_string[i]=NULL;
	return;
}

G_MODULE_EXPORT
void xffm_setenv(const char *name,
		char *value,
		gboolean verbose){
  int which;
  gboolean go_ahead=FALSE;
  for (which=0;env_vars[which];which++)
	    if (strcmp(name,env_vars[which]) == 0) break;
  if (!env_vars[which]) return;
  if (!value || !strlen(value)) {
	  /*printf("TRACE: erasing %s\n",name);*/
        g_free(env_string[which]);
	env_string[which]=NULL;
/* environment handling is a blast:
 * freebsd only clears environment with unsetenv, not putenv
 * */
	
#ifdef HAVE_SETENV
	unsetenv(name);
        env_string[which]=NULL;
#else
        env_string[which]= g_strconcat(name,"=",NULL);
        putenv(env_string[which]);	
#endif
	
           if (verbose){	
	     if (strcmp(name, "SMB_USER") == 0) {
		    gchar *m=g_strdup_printf("Mcs manager changed xffm environment: %s",name);  
		    /*XXX keep this g_message */
		    g_message(m);
		    g_free(m);
	     }
	     else {
		    /*XXX keep this g_message */
		    g_message("Mcs manager changed xffm environment: %s=%s",name, ((value)?value:" "));
	     }
	   }
	return;
  }
  if (strcmp(name, "XFFM_MAX_PREVIEW_SIZE")==0)
  {
        if (is_number(value)) go_ahead=TRUE;
  } else if (strcmp(name, "TERMCMD")==0) {
	  /*if (getenv(name) && access(getenv(name),X_OK)==0)*/
	  /*printf("TRACE:TERMCMD=%s!\n",value);*/

	  if (value && strlen(value)){
		gchar *c,*t=g_strdup(value);
    		t=g_strstrip(t);
		if (strchr(t,' '))t=strtok(t," ");
		c=g_find_program_in_path(t);
		if (c && access(c,X_OK)==0) {
		  go_ahead=TRUE;
      		}
  		g_free(c);
		c=NULL;
		g_free(t);
		t=NULL;
	  }
  }
  else go_ahead=TRUE;
  if (go_ahead){
     	  g_free(env_string[which]);
	  env_string[which]=NULL;
	  if (strcmp(name,"SMB_USER") == 0 && !strchr(value,'%')) {
             env_string[which]= g_strconcat(name,"=",value,"%",NULL);		
	  } else {
             env_string[which]= g_strconcat(name,"=",value,NULL);
	  }
          putenv(env_string[which]);
	  /*printf("TRACE: mapping %s=%s\n",name,value);*/
		   if (verbose){		   
		     if (strcmp(name, "SMB_USER") == 0) {
			gchar *m=g_strdup_printf("Mcs manager changed xffm environment: %s",name);  
			/*XXX keep this g_message */
		        g_message(m);
		        g_free(m);
		     }
		     else {
			/*XXX keep this g_message */
			g_message("Mcs manager changed xffm environment: %s=%s",name, ((value)?value:" "));
		     }
		   }
  } else { /* not go_ahead */
	g_warning("Mcs manager failed to change xffm environment: %s",name);
  } 
  return;	  
}

G_MODULE_EXPORT
const gchar *
get_local_cache_path(gchar *directory){
    gchar *xdg_dir;
    gchar *cache_dir;
    GString *gs;
    gchar key[11];
    static gchar *cache_path=NULL;
    
    xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    cache_dir = g_build_filename(xdg_dir,"xffm","cache",NULL);  
    g_free(xdg_dir);
    mkdir(cache_dir, 0770);
    
    if (!g_file_test(cache_dir,G_FILE_TEST_IS_DIR)) {
	g_free(cache_dir);
	return NULL;
    }

    gs = g_string_new(directory);
    sprintf(key, "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    g_free(cache_path);
    cache_path=g_build_filename(cache_dir,key,NULL);
    g_free(cache_dir);
    return (const gchar *)cache_path;
}


#ifdef DEBUG
#include <sys/times.h>
static struct tms *last_clock=NULL;

const gchar *
profile(void){
    struct tms this_clock;
    static gchar *p=NULL;
    g_free(p);
    if (!last_clock) {
	last_clock = (struct tms *)malloc(sizeof(struct tms));
	times(last_clock);
    }
    times(&this_clock);
    p=g_strdup_printf("\n\tPROFILE*** user=%ld, system=%ld",
	(long)(this_clock.tms_utime - last_clock->tms_utime),
	(long)(this_clock.tms_stime - last_clock->tms_stime));
    memcpy(last_clock,&this_clock,sizeof(struct tms));
	
    return (const gchar *)p;   
}
#endif
