/*
 * Copyright (C) 2004 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define SPLIT_VIEW
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#define __TREEVIEW_C__

#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

#include <errno.h>
#include <grp.h>
#include <pwd.h>
#include <dirent.h>


#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>


#include "constants.h"
#include "types.h"

#include "primary.h"
#include "treeview.h"
#include "actions_lib.h"

#include "tubo.h"

G_MODULE_EXPORT
gboolean 
easy_chgrp(GtkCellRendererText *cell,
	     const gchar         *path_string,
	     const gchar         *new_text,
	     gpointer             data)
{             
    GtkTreeView *treeview=(GtkTreeView *)data;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
    record_entry_t *en;
    GtkTreeIter iter;
    gchar *old_text;

    if (!path_string || !path || !gtk_tree_model_get_iter (treemodel,&iter,path)) {
	/*printf("TRACE: !path || !gtk_tree_model_get_iter (treemodel,&iter,path)\n");*/	
	goto end_easy;
    }
    gtk_tree_model_get (treemodel, &iter, GROUP_COLUMN, &old_text, -1);
    if (strcmp(old_text,new_text)==0) {
	/*printf("TRACE: text does not differ\n");*/
	goto end_easy;
    }
    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);

    if (!en || !en->path) {
	/*printf("TRACE: !en || !en->path\n");	*/
	goto end_easy;
    }
	
    if (IS_PATH(en->type)){
	struct group *grbuf;
	uid_t pw=-1;
	gid_t gr=-1;
	grbuf = getgrnam(new_text);
	if (grbuf) gr=grbuf->gr_gid;
	else {
	    gchar *p;
	    gboolean OK=TRUE;
	    for (p=(gchar *)new_text;*p;p++){
	        if (!isdigit((int)*p)){
		   OK=FALSE;
		   break;
		}
	    }
	    if (OK) {
		gr=atoi(new_text);
		grbuf = getgrgid(gr);
	    } else {
		print_diagnostics(&(xffm_details->arbol->widgets),"xfce/error",strerror(EINVAL),
		    ":\n","chgrp ",new_text," ",en->path,"\n",NULL);
		goto end_easy;
	    }
	    
	}
	
	if (chown(en->path, pw, gr) == -1){
	    char group_s[64]; /* needs to be fork safe variable */
	    sprintf(group_s,"%d",gr);
	    xffm_try_sudo(&(xffm_details->arbol->widgets),"chgrp",group_s,en->path);
	}
	else {
	    update_text_cell_for_row(GROUP_COLUMN,treemodel, &iter,new_text);
	    /*local_monitor(TRUE);*/ /* to be sure */
	}	    
    }
end_easy:
    if (path) gtk_tree_path_free(path);
    return FALSE;
}

G_MODULE_EXPORT
gboolean 
easy_chown(GtkCellRendererText *cell,
	     const gchar         *path_string,
	     const gchar         *new_text,
	     gpointer             data)
{             
    GtkTreeView *treeview=(GtkTreeView *)data;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
    record_entry_t *en;
    GtkTreeIter iter;
    gchar *old_text;

    if (!path_string || !path || !gtk_tree_model_get_iter (treemodel,&iter,path)) {
	/*printf("TRACE: !path || !gtk_tree_model_get_iter (treemodel,&iter,path)\n");*/	
	goto end_easy;
    }
    
    gtk_tree_model_get (treemodel, &iter, OWNER_COLUMN, &old_text, -1);
    if (strcmp(old_text,new_text)==0) {
	/*printf("TRACE: text does not differ\n");*/
	goto end_easy;
    }
    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);

    if (!en || !en->path) {
	/*printf("TRACE: !en || !en->path\n");	*/
	goto end_easy;
    }
	
    if (IS_PATH(en->type)){
	struct passwd *pwbuf;
	uid_t pw=-1;
	gid_t gr=-1;
	pwbuf = getpwnam(new_text);
	if (pwbuf) pw=pwbuf->pw_uid;	
	else {
	    gchar *p;
	    gboolean OK=TRUE;
	    for (p=(gchar *)new_text;*p;p++){
	        if (!isdigit((int)*p)){
		   OK=FALSE;
		   break;
		}
	    }
	    if (OK) {
		pw=atoi(new_text);
		pwbuf = getpwuid(pw);
	    } else {
		print_diagnostics(&(xffm_details->arbol->widgets),"xfce/error",strerror(EINVAL),
		    ":\n","chown ",new_text," ",en->path,"\n",NULL);
		goto end_easy;
	    }
	}

	if (chown(en->path, pw, gr) == -1){
	    char pw_s[64]; /* needs to be fork safe variable */
	    sprintf(pw_s,"%d",pw);
	    xffm_try_sudo(&(xffm_details->arbol->widgets),"chown",pw_s,en->path);
	}
	else {
	    update_text_cell_for_row(OWNER_COLUMN,treemodel,&iter,new_text);
	    /*local_monitor(TRUE);*//* to be sure */
	}	    
    }
end_easy:
    if (path) gtk_tree_path_free(path);
    return FALSE;
}

static mode_t 
parse_mode_string(const gchar *mode_string){
    mode_t mode=0;
    int i,section=0;
    
    for (i=0;i<strlen(mode_string) && i<strlen("drwxrwxrwx") && section<=8;i++){
	if (mode_string[i]!='-' && mode_string[i]!='r' &&
	    mode_string[i]!='w' && mode_string[i]!='x' &&
	    mode_string[i]!='s' && mode_string[i]!='S' &&	
	    mode_string[i]!='t' && mode_string[i]!='T') continue;
	if (mode_string[i]=='-'){
	    if (i) section++;
	    continue;
	}
	if (section%3 == 0 && mode_string[i]!='r') section++;
	if (section%3 == 1 && mode_string[i]!='w') section++;
	if (section%3 == 2 && mode_string[i]!='x' && mode_string[i]!='s' && mode_string[i]!='S' && mode_string[i]!='t' && mode_string[i]!='T') section++;    

	switch (section) {
	    /* user */
	    case 0: if (mode_string[i]=='r') mode |= S_IRUSR;
		    break;
	    case 1: if (mode_string[i]=='w') mode |= S_IWUSR;
		    break;
	    case 2: if (mode_string[i]=='x') mode |= S_IXUSR;
		    if (mode_string[i]=='s') mode |= (S_IXUSR|S_ISUID);
		    if (mode_string[i]=='S') mode |= S_ISUID;
		    break;
	    /* group */
	    case 3: if (mode_string[i]=='r') mode |= S_IRGRP;
		    break;
	    case 4: if (mode_string[i]=='w') mode |= S_IWGRP;
		    break;
	    case 5: if (mode_string[i]=='x') mode |= S_IXGRP;
		    if (mode_string[i]=='s') mode |= (S_IXGRP|S_ISGID);
		    if (mode_string[i]=='S') mode |= S_ISGID;
		    break;
	    /* others */
	    case 6: if (mode_string[i]=='r') mode |= S_IROTH;
		    break;
	    case 7: if (mode_string[i]=='w') mode |= S_IWOTH;
		    break;
	    case 8: if (mode_string[i]=='x') mode |= S_IXOTH;
		    if (mode_string[i]=='t') mode |= (S_IXOTH|S_ISVTX);
		    if (mode_string[i]=='T') mode |= S_ISVTX;
		    break;
	    default: break;
	}
	section++;
    }

    return mode;
}

static gboolean 
is_valid_mode_string(const gchar *mode_string){
    int i;
    for (i=1;i<strlen(mode_string);i++){
	switch (mode_string[i]){
	    case 'd': /* ignored */
	    case 'c': /* ignored */
	    case 'b': /* ignored */
	    case ' ': /* ignored */
	    case '-':
	    case 'r':
	    case 'w':
	    case 'x':
	    case 's':
	    case 'S':
	    case 't':
	    case 'T':
		break;
	    default:
		goto invalid_mode;
	}
    }
    return TRUE;
invalid_mode:
    print_diagnostics(&(xffm_details->arbol->widgets),"xfce/error",strerror(EINVAL),
	    ":\n",mode_string,"\n",NULL);
    return FALSE;
}

G_MODULE_EXPORT
gboolean 
easy_remode(GtkCellRendererText *cell,
	     const gchar         *path_string,
	     const gchar         *new_text,
	     gpointer             data)
{             
    GtkTreeView *treeview=(GtkTreeView *)data;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
    record_entry_t *en;
    GtkTreeIter iter;
    gchar *old_text;

    if (!path_string || !path || !gtk_tree_model_get_iter (treemodel,&iter,path)) goto end_easy;
    gtk_tree_model_get (treemodel, &iter, MODE_COLUMN, &old_text, -1);
    if (strcmp(old_text,new_text)==0) goto end_easy;
    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);

    if (!en || !en->path) goto end_easy;
	
    if (IS_PATH(en->type)){
	if (!is_valid_mode_string(new_text)) goto end_easy;
	if (chmod(en->path, parse_mode_string(new_text)) == -1){
	    char mode_s[64];
	    sprintf(mode_s,"0%o",parse_mode_string(new_text));
	    xffm_try_sudo(&(xffm_details->arbol->widgets),"chmod",mode_s,en->path);
	}
	else {
	    update_text_cell_for_row(MODE_COLUMN,treemodel,&iter,new_text);
	    local_monitor(TRUE); /* to be sure */
	}	    
    }
end_easy:
    if (path) gtk_tree_path_free(path);
    return FALSE;
}

G_MODULE_EXPORT
gboolean 
easy_rename(GtkCellRendererText *cell,
	     const gchar         *path_string,
	     const gchar         *new_text,
	     gpointer             data)
{             
    GtkTreeView *treeview=(GtkTreeView *)data;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
    record_entry_t *en;
    GtkTreeIter iter;
    gchar *old_text;

    if (!path_string || !path || !gtk_tree_model_get_iter (treemodel,&iter,path)) {
	TRACE("!path_string || !path || !gtk_tree_model_get_iter()");
	goto end_easy;
    }
    gtk_tree_model_get (treemodel, &iter, NAME_COLUMN, &old_text, -1);
    if (strcmp(old_text,new_text)==0) {
	TRACE("strcmp(old_text,new_text)==0");
	goto end_easy;
    }

    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);

    if (!en || !en->path) {
	goto end_easy;
    }
    if (IS_PATH(en->type)){
	char *p,*d,*n;
	n=g_path_get_basename(en->path);
	if (strcmp(n,"..Wastebasket")==0){
	    g_free(n);
	    TRACE("strcmp(n,\"..Wastebasket\")==0");
	    goto end_easy;
	}
	d=g_path_get_dirname(en->path);
	p=g_strconcat(d,G_DIR_SEPARATOR_S,my_utf2local_string(new_text),NULL);
	if (xffm_rename(&(xffm_details->arbol->widgets), p, en->path)) {
	    TRACE("rename ok");
	    g_free(en->path);
	    en->path = p;
	    update_text_cell_for_row (	NAME_COLUMN,treemodel, &iter,new_text);
	} else {
	    TRACE("rename not ok");
	    g_free(p);
	}
	g_free(n);
	g_free(d);
    } else {
	TRACE("!IS_PATH(en->type)");
    }
end_easy:
    if (path) gtk_tree_path_free(path);
    return FALSE;
}


