/* Copyright 2005 Edscott Wilson Garca. 
 * Distributed with GPL licence.*/

static void graphics_fix_tip_bug(icon_view_t *icon_view_p)
{
	int x,y;
	int p_x, p_y;
	gint root_w,root_h;

	if (!icon_view_p || !icon_view_p->tips ||! icon_view_p->tips->tip_window||! icon_view_p->tips->tip_window->window) return;
	
	gdk_window_get_geometry  (gdk_get_default_root_window(),
				    NULL,NULL,&root_w,&root_h,NULL);        
	gtk_window_get_position (GTK_WINDOW(icon_view_p->widgets.window),&p_x,&p_y);
	gtk_window_get_position (GTK_WINDOW(icon_view_p->tips->tip_window),&x,&y);
	TRACE("tip window= 0x%x; parent= 0x%x", 
		(unsigned) icon_view_p->tips->tip_window,
		(unsigned) icon_view_p->widgets.window);
	TRACE("tip window position=%d,%d; parent= %d,%d", x,y,p_x,p_y);
	gdk_window_get_position (icon_view_p->widgets.window->window,&p_x,&p_y);
	if (p_y - 75 > 0) p_y -= 75;
	else if (p_y  + icon_view_p->widgets.window->allocation.height < root_h + 20){
	    p_y  += icon_view_p->widgets.window->allocation.height;
	}
        gdk_window_move (icon_view_p->tips->tip_window->window,x,p_y); 
	gdk_window_show(icon_view_p->tips->tip_window->window);
}

static void
graphics_set_default_tooltip(icon_view_t *icon_view_p, record_entry_t *en)
{
    const gchar *prg=xffm_details->argv[0];

    {
	gchar *p=NULL;
	p=g_strdup_printf(_("Welcome to %s"),prg);
	gtk_tooltips_set_tip (GTK_TOOLTIPS (icon_view_p->tips), 
		icon_view_p->paper,p,"");
	g_free(p);
    }
}

typedef struct tip_timer_t{
    icon_view_t *icon_view_p;
    int tip_timer;
}tip_timer_t;

static 
int
graphics_set_entry_tip(gpointer data)
{
    const gchar *tip=NULL;
    tip_timer_t *tip_timer_p = data;
    icon_view_t *icon_view_p;
    record_entry_t *en;
    TRACE("gtk_tooltips_enable");
    if (!data) return FALSE;
    icon_view_p = tip_timer_p->icon_view_p;
    if (tip_timer_p->tip_timer != icon_view_p->tip_timer) {
	/* wrong timeout condition */
	g_free(data);
	return FALSE;
    }
    if (!icon_view_p || !icon_view_p->saturated_p || !icon_view_p->tip_timer) {
	/* no timer should be active condition */
	g_free(data);
	return FALSE;
    }
    en = icon_view_p->saturated_p->en;
    if (icon_view_p->tips && icon_view_p->tips->tips_data_list)
	    icon_view_p->tips->active_tips_data = icon_view_p->tips->tips_data_list->data;

    gtk_tooltips_enable(icon_view_p->tips);      
    gtk_tooltips_force_window (icon_view_p->tips);
    if (icon_view_p->module_name) {
	tip=function_natural("plugins",icon_view_p->module_name,en,"entry_tip");
    }
    else if (en && en->module) {
	tip=function_natural("plugins",en->module,en,"plugin_info");
    }
    if (!tip && en && en->path && strlen(en->path)) {
	tip=path_info(en);
    }
    if (!tip || !strlen(tip)) {
	/*graphics_set_default_tooltip(icon_view_p, NULL);*/
	/*graphics_set_default_tooltip(icon_view_p, en);*/
	goto over;
    }
    gtk_tooltips_set_tip (GTK_TOOLTIPS (icon_view_p->tips), 
		icon_view_p->paper,tip,"");
    
    /* now, work around the tooltip placement bug: */
    graphics_fix_tip_bug(icon_view_p);
over:
    icon_view_p->tip_timer=0;
    g_free(data);
    return FALSE;
}


static void
graphics_unset_entry_tip(icon_view_t *icon_view_p)
{

    /*graphics_set_default_tooltip(icon_view_p,NULL);*/
    /*graphics_fix_tip_bug(icon_view_p);*/
    if (icon_view_p && icon_view_p->tips && icon_view_p->tips->tip_window && icon_view_p->tips->tip_window->window) {
	gdk_window_hide(icon_view_p->tips->tip_window->window);
    }
}



/* define this to make windows appear at saved location.
 * must create a dummy window for pixmaps so that this can
 * be done without flashing (annoying).
 *
 * This is buggy still...*/
/*#define RESTORE_POSITION*/

#ifdef RESTORE_POSITION
static gboolean iconview_is_mapped(icon_view_t *icon_view_pp){
    GList *tmp;
    
    if (!icon_view_pp) {
	TRACE(" mapped: !icon_view_pp");
	return FALSE;
    }
	
    for (tmp=iconview_list; tmp; tmp=tmp->next){
	icon_view_t *icon_view_p=(icon_view_t *)tmp->data;
	if (icon_view_pp==tmp->data) continue;
        if (!icon_view_pp->en){
	    if (!tmp->data) {
		TRACE(" mapped: !tmp->data");
		return TRUE;
	    }
	    return FALSE;
	}
        if (!icon_view_pp->en->path){
	    if (icon_view_p->en && !icon_view_p->en->path) {
		TRACE(" mapped: !icon_view_p->en->path");
		return TRUE;
	    }
	    return FALSE;
	}

	if (icon_view_p->en && icon_view_p->en->path && strcmp(icon_view_pp->en->path,icon_view_p->en->path)==0) {
	    TRACE(" mapped: %s",icon_view_p->en->path);
	    return TRUE;
	}
    }
    return FALSE;
}
#endif

static
iconview_preferences_t *
get_iconview_preferences(const gchar *path){
    DBHashTable *preferences;
    const gchar *key;
    GString *gs;
    static iconview_preferences_t iconview_preferences;
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *f=g_build_filename(xdg_dir,GRIDVIEW_PREFERENCES_DBH_FILE,NULL);

    if (path) key=path;
    else key="ROOT_ICONVIEW";
    TRACE("looking for preferences with key=%s",key);
    
    preferences = DBH_open(f);
    g_free(xdg_dir);
    g_free(f);
    
    if(!preferences){
	TRACE("no preferences file");
	return NULL;
    }
        
    gs = g_string_new(key);	
    sprintf((char *)DBH_KEY(preferences), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    
    
    if(!DBH_load(preferences)) {
	TRACE("no preferences record for %s (%s)", key, (char *)DBH_KEY(preferences));
	DBH_close(preferences);
	return NULL;
    }
    memcpy(&iconview_preferences, DBH_DATA(preferences), sizeof(iconview_preferences_t));
    TRACE("got preferences with key=%s  (preferences=0x%x, sort_column%d)\n",
	    key,iconview_preferences.preferences,
	    iconview_preferences.sortcolumn);
    DBH_close(preferences);
    return &iconview_preferences;
}

G_MODULE_EXPORT
void 
save_iconview_preferences(icon_view_t *icon_view_p){
    iconview_preferences_t iconview_preferences;
    DBHashTable *preferences;
    GString *gs;
    gchar *xdg_dir;
    gchar *f;
    const gchar *key;
    if (!icon_view_p) return;

    if (!icon_view_p->en || !icon_view_p->en->path) key="ROOT_ICONVIEW";
    else key=icon_view_p->en->path;
    
    xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    f=g_build_filename(xdg_dir,GRIDVIEW_PREFERENCES_DBH_FILE,NULL);
    
    preferences = DBH_open(f);
    if (!preferences) preferences = DBH_create(f, 11);

    g_free(xdg_dir);
    g_free(f);
    if (!preferences) return;
    
    gs = g_string_new(key);	
    sprintf((char *)DBH_KEY(preferences), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);

    iconview_preferences.preferences=icon_view_p->preferences;
    iconview_preferences.sortcolumn=icon_view_p->sortcolumn;
	    
    memcpy(DBH_DATA(preferences),&iconview_preferences,  sizeof(iconview_preferences_t));
    DBH_set_recordsize(preferences, sizeof(iconview_preferences_t));
    
    if (!DBH_update(preferences))	{
	g_warning("!DBH_update(preferences)");
    }
    DBH_close(preferences);
    TRACE("saved preferences with key=%s\n",key);
    return;
}

static
iconview_geometry_t *
get_iconview_geometry_p(icon_view_t *icon_view_p){
    DBHashTable *geometry;
    const gchar *key;
    GString *gs;
    static iconview_geometry_t iconview_geometry;
    gchar *xdg_dir;
    gchar *f;
    gboolean no_record=FALSE;

    if (!icon_view_p) return NULL;
    if (icon_view_p->en && icon_view_p->en->path && strlen(icon_view_p->en->path)) key=icon_view_p->en->path;
    else key="ROOT_ICONVIEW";

again:
    xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    f=g_build_filename(xdg_dir,GRIDVIEW_GEOMETRY_DBH_FILE,NULL);
    
    TRACE("looking for geometry with key=%s",key);
    
    geometry = DBH_open(f);
    g_free(xdg_dir);
    g_free(f);
    
    if(!geometry){
	TRACE("no geometry file");
	return NULL;
    }
        
    gs = g_string_new(key);	
    sprintf((char *)DBH_KEY(geometry), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    
    
    if(!DBH_load(geometry)) {
	TRACE("no geometry record for %s (%s)", key, (char *)DBH_KEY(geometry));
	DBH_close(geometry);
	
	if (strcmp(key,"ROOT_ICONVIEW")==0) return NULL;
	else {
	    key="ROOT_ICONVIEW";
	    no_record=TRUE;
	    goto again;
	}
    }
    memcpy(&iconview_geometry, DBH_DATA(geometry), sizeof(iconview_geometry_t));
    /* no placement for non records...*/
    if (no_record){
	iconview_geometry.x = iconview_geometry.y = -1;
    }
    TRACE("got geometry with key=%s  (%lf,position=%d,%d, size=%d,%d)\n",
	    key,iconview_geometry.scrollX,
	    iconview_geometry.x,iconview_geometry.y,
	    iconview_geometry.w,iconview_geometry.h);
    DBH_close(geometry);
    return &iconview_geometry;
}

static
void 
save_iconview_geometry_p(icon_view_t *icon_view_p){
    iconview_geometry_t iconview_geometry;
    DBHashTable *geometry;
    GString *gs;
    gchar *xdg_dir;
    gchar *f;
    const gchar *key;
    if (!icon_view_p) return;

    if (!icon_view_p->en || !icon_view_p->en->path) key="ROOT_ICONVIEW";
    else key=icon_view_p->en->path;
    
    xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    f=g_build_filename(xdg_dir,GRIDVIEW_GEOMETRY_DBH_FILE,NULL);
    
    geometry = DBH_open(f);
    if (!geometry) geometry = DBH_create(f, 11);

    g_free(xdg_dir);
    g_free(f);
    if (!geometry) {
	g_warning("cannot create geometry file");
	return;
    }
    
    gs = g_string_new(key);	
    sprintf((char *)DBH_KEY(geometry), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);

    iconview_geometry.w=icon_view_p->widgets.window->allocation.width;
    iconview_geometry.h=icon_view_p->widgets.window->allocation.height;
    iconview_geometry.scrollX=gtk_adjustment_get_value (gtk_scrolled_window_get_vadjustment(icon_view_p->scrolled_window));
    TRACE("saving ajustment at %lf (%s)",iconview_geometry.scrollX, key);
    gtk_window_get_position ((GtkWindow *)icon_view_p->widgets.window,&(iconview_geometry.x),&(iconview_geometry.y));
    /*iconview_geometry.preferences=icon_view_p->preferences;*/
	    
    TRACE("saving windowposition %d,%d size %d,%d",
	    iconview_geometry.x,iconview_geometry.y,
	    iconview_geometry.w,iconview_geometry.h);
 


    memcpy(DBH_DATA(geometry),&iconview_geometry,  sizeof(iconview_geometry_t));
    DBH_set_recordsize(geometry, sizeof(iconview_geometry_t));
    
    if (!DBH_update(geometry))	{
	g_warning("!DBH_update(geometry)");
    }
    DBH_close(geometry);
    TRACE("saved geometry with key=%s (%lf,%d,%d)\n",
	    key,
	    iconview_geometry.scrollX,
	    iconview_geometry.w,
	    iconview_geometry.h);
    return;
}

static 
void
restore_iconview_size (icon_view_t *icon_view_p){
    double a=0.0;
    /*record_entry_t *en=icon_view_p->en;*/
    iconview_geometry_t *iconview_geometry_p;
    iconview_geometry_p=get_iconview_geometry_p(icon_view_p);
    TRACE("redlight2=%d",icon_view_p->redlight2);
    /* figure out the x,y landing position before the resize to
     * avoid race conditions and the inability of gtk to sync
     * easily with X server */
    if (iconview_geometry_p) {
	
	gint x,y;
	gint root_w,root_h;
	if (iconview_geometry_p->w && iconview_geometry_p->h) {
	    GdkWindow *win=gtk_widget_get_parent_window(icon_view_p->widgets.status);


	    gtk_window_get_position((GtkWindow *)icon_view_p->widgets.window,&x,&y);
	    gdk_window_get_geometry  (gdk_get_default_root_window(),
				    NULL,NULL,&root_w,&root_h,NULL);
	    TRACE("x=%d y=%d, w+x=%d (root_w=%d), w+y=%d (root_h=%d)",x,y,iconview_geometry_p->w+x , root_w ,iconview_geometry_p->h+y , root_h); 
#ifdef RESTORE_POSITION
	    if (!iconview_is_mapped(icon_view_p)) {
	        x=iconview_geometry_p->x,y=iconview_geometry_p->y;
	        gdk_window_move_resize (win, x, y, iconview_geometry_p->w,iconview_geometry_p->h);
	    }
#else
	    if (iconview_geometry_p->w+x > root_w || iconview_geometry_p->h+y > root_h){
	        x=iconview_geometry_p->x,y=iconview_geometry_p->y;
	    }
	    gdk_window_move_resize (win, x, y, iconview_geometry_p->w,iconview_geometry_p->h);
#endif
	    while (gtk_events_pending()) gtk_main_iteration();
	    gdk_flush();
	    TRACE("resize to %d,%d",iconview_geometry_p->w,iconview_geometry_p->h);
	    if (iconview_geometry_p->scrollX < icon_view_p->paper->allocation.height){
		a=iconview_geometry_p->scrollX;
	    }
	    gtk_adjustment_set_value (gtk_scrolled_window_get_vadjustment(icon_view_p->scrolled_window),a);
	    gtk_adjustment_changed (gtk_scrolled_window_get_vadjustment(icon_view_p->scrolled_window));
	    TRACE("set adjustment to %lf",a); 
	}
    }
    TRACE("redlight2=%d",icon_view_p->redlight2);
}

static 
int
select_population_rectangle(icon_view_t *icon_view_p,
			    GdkRectangle *rect)
{
    int result=0;
    int row,column;
    int bottom_row=rect->y/CELLSIZE;
    int bottom_column=rect->x/CELLSIZE;
    int top_row=(rect->y+rect->height)/CELLSIZE;
    int top_column=(rect->x+rect->width)/CELLSIZE;

    int low_X=rect->x, low_Y=rect->y; 
    int	high_X=rect->x+rect->width, high_Y=rect->y+rect->height;

    if (!rect->width || !rect->height) return result;

    /* first find the affected grid elements. we
     * later can figure out if the icon is actually
     * selected. */
    if (!icon_view_p->population_pp) return result;
    
    for (row=bottom_row; row <= top_row; row++) {
	for (column=bottom_column; column <= top_column; column++){
	    population_t *population_p;
	    int element=row*icon_view_p->grid_columns+column;
	    if (element > icon_view_p->max_elements) continue;
	    if (column >= icon_view_p->grid_columns)  continue;
	    population_p = *(icon_view_p->population_pp+element);
	    if (!population_p) continue;
	    TRACE("row, column ok (%d,%d)",row,column);
	    /* update row, column to consider posible window resizes 
	     * that may have occured. */
	    population_p->row=row;
	    population_p->column=column;
	    /* figure out if the icon is actually in the selection box */
	    {
		int low_x = column*CELLSIZE + CELLSPACING;
		int high_x = column*CELLSIZE + CELLSPACING 
		    + population_p->pixbufW;
		int low_y = row*CELLSIZE;
		int high_y = row*CELLSIZE + population_p->pixbufH;
		if (low_X - high_x < -1 && low_x - high_X < -1 &&
		    low_Y - high_y < -1 && low_y - high_Y < -1){
		    TRACE("found in selection box: %s",
			    population_p->en->path);
		    select_pixbuf(icon_view_p,population_p);

		    result++;
		}
			
	    }
	    
	}
    }
    return result;
}

static
void
draw_window(icon_view_t *icon_view_p, int x, int y, gboolean draw){
    static GdkGC *toggleGC=NULL;
    if (icon_view_p->down_X == -1 && icon_view_p->down_Y == -1) return;

    if (!toggleGC){
	GdkColor pen_color={7,65535,30000,65535}; 
	toggleGC = gdk_gc_new (icon_view_p->paper->window);
	gdk_gc_set_colormap (toggleGC,icon_view_p->cmap);
	gdk_colormap_alloc_color(icon_view_p->cmap,&pen_color,TRUE,TRUE);
	gdk_gc_set_foreground (toggleGC, &pen_color);	
	gdk_gc_set_function (toggleGC, GDK_XOR);
    }
    if (icon_view_p->old_X == -1 && icon_view_p->old_Y== -1) {
	    icon_view_p->old_X=icon_view_p->down_X;
	    icon_view_p->old_Y=icon_view_p->down_Y;
    } else {
	    int lowX,highX,lowY,highY;
	    lowX=(icon_view_p->old_X>icon_view_p->down_X)?
		icon_view_p->down_X:icon_view_p->old_X;
	    lowY=(icon_view_p->old_Y>icon_view_p->down_Y)?
		icon_view_p->down_Y:icon_view_p->old_Y;
	    highX=(icon_view_p->old_X<icon_view_p->down_X)?
		icon_view_p->down_X:icon_view_p->old_X;
	    highY=(icon_view_p->old_Y<icon_view_p->down_Y)?
		icon_view_p->down_Y:icon_view_p->old_Y;
	    /* clean old rectangle */
	    /* draw new rectangle */
	    gdk_draw_rectangle (icon_view_p->paper->window, 
			toggleGC, FALSE, 
			lowX,lowY,
			highX-lowX,highY-lowY);
	    if (!draw) return;
	    /* draw new rectangle */
	    icon_view_p->old_X=x;
	    icon_view_p->old_Y=y;
	    lowX=(icon_view_p->old_X>icon_view_p->down_X)?
		icon_view_p->down_X:icon_view_p->old_X;
	    lowY=(icon_view_p->old_Y>icon_view_p->down_Y)?
		icon_view_p->down_Y:icon_view_p->old_Y;
	    highX=(icon_view_p->old_X<icon_view_p->down_X)?
		icon_view_p->down_X:icon_view_p->old_X;
	    highY=(icon_view_p->old_Y<icon_view_p->down_Y)?
		icon_view_p->down_Y:icon_view_p->old_Y;
	    gdk_draw_rectangle (icon_view_p->paper->window, 
			toggleGC, FALSE, 
			lowX,lowY,
			highX-lowX,highY-lowY);
	    
    }

}

static 
int
select_population_area(	icon_view_t *icon_view_p)
{
    /* FIXME next 
     * just select all items found in range, unselect everything
     * that is not in range (if selected previously). 
     * Use icon_view_p->down_X, icon_view_p->down_Y, icon_view_p->old_X, icon_view_p->old_Y for area */
    GdkRectangle rect;
    int lowX=(icon_view_p->old_X>icon_view_p->down_X)?
	icon_view_p->down_X:icon_view_p->old_X;
    int lowY=(icon_view_p->old_Y>icon_view_p->down_Y)?
	icon_view_p->down_Y:icon_view_p->old_Y;
    int highX=(icon_view_p->old_X<icon_view_p->down_X)?
	icon_view_p->down_X:icon_view_p->old_X;
    int highY=(icon_view_p->old_Y<icon_view_p->down_Y)?
	icon_view_p->down_Y:icon_view_p->old_Y;
    rect.x = lowX;
    rect.y = lowY;
    rect.width = highX - lowX;
    rect.height = highY - lowY;
    select_population_rectangle(icon_view_p,&rect);
    return 1;
}

static 
void
set_iconview_restart(icon_view_t *icon_view_p)
{
	GList *tmp;
	int i=0;
	int argc=1+2*g_list_length(iconview_list);
	gchar **argv;
	argv = (gchar **)malloc((argc+1)*sizeof(gchar *));
	argv[i++]=xffm_details->argv[0];
	argv[argc]=NULL;
	
	for (tmp=iconview_list; tmp; tmp=tmp->next,i+=2){
	    icon_view_t *icon_view_p=(icon_view_t *)tmp->data;
	    if (!icon_view_p->module_name){
		argv[i+1]="null";
	    } else {
		argv[i+1]=(gchar *)icon_view_p->module_name;
	    }  
	    if (!icon_view_p->en){
		argv[i]="null";
	    } else {
		if (!icon_view_p->en->path) argv[i]="null";
		else  argv[i]=icon_view_p->en->path;
	    }
	}
	set_restart_command(icon_view_p->paper,argc,argv);
	g_free(argv);
}


static
void
set_iconview_title(	icon_view_t *icon_view_p)
{
    record_entry_t *en=icon_view_p->en;
    const gchar *path;

    if (en && en->path && strlen(en->path)){
	path=en->path;
    } else {
	path=OUR_HOST_NAME(&(icon_view_p->widgets));
    }
    
    if (icon_view_p->module_name && function_void("plugins",icon_view_p->module_name,"exec_name")) path=function_void("plugins",icon_view_p->module_name,"exec_name");
    set_icon_name(icon_view_p->widgets.window,path);
    set_application_icon(&(icon_view_p->widgets),en);
    
    /* session management */
    set_iconview_restart(icon_view_p);
    
    
    return;
}

static 
void
graphics_expose_item(	icon_view_t *icon_view_p,
			int row,
			int column){
	    int element=row*icon_view_p->grid_columns+column;
	    population_t *population_p;
	    if (icon_view_p->population_pp)
		population_p=icon_view_p->population_pp[element];
	    else
		population_p=NULL;
	    TRACE("exposing element %d,%d",row,column);
	    select_pen(icon_view_p,BACKGROUND_COLOR);
	    gdk_draw_rectangle (icon_view_p->paper->window, 
			icon_view_p->penGC, TRUE, 
			column*CELLSIZE,row*CELLSIZE,
			CELLSIZE,CELLSIZE);
	    /* if element is out of bounds, skip it */
	    if (element >= icon_view_p->max_elements) return;
	    /* if element is repeated as first column of next row, skip it */
	    if (element == (row+1)*icon_view_p->grid_columns) return;
	        
	    if ( population_p ){
		int text_color=TEXT_COLOR;
		/* contents */

		if (population_p->en && population_p->en->path && !g_utf8_validate (population_p->en->path,-1,NULL)) 
		    text_color=INVALID_UTF8_TEXT_COLOR;
		TRACE("drawing element %d,%d (0x%x)",row,column,
			(unsigned)population_p);
		
		if (population_p==icon_view_p->label_p)text_color=SATURATE_LABEL_COLOR;
	        if (population_p->pixbuf) {
		    gdk_draw_pixbuf (icon_view_p->paper->window,
		    icon_view_p->penGC,
		    population_p->pixbuf,
		    0,0,
		    column*CELLSIZE+CELLSPACING,
		    row*CELLSIZE,
		    population_p->pixbufW,
		    population_p->pixbufH,
                    GDK_RGB_DITHER_NONE,0,0);
		}
		
		if (population_p->selected) text_color=SELECT_TEXT_COLOR;
		else if (population_p==icon_view_p->saturated_p)
		    text_color=SATURATE_TEXT_COLOR;
		select_pen(icon_view_p,text_color);
		if (population_p->layout) 
		    gdk_draw_layout(icon_view_p->paper->window,
		    icon_view_p->penGC, 
		    column*CELLSIZE + (CELLSIZE-population_p->logical_rect.width)/2,
		    row*CELLSIZE+TEXTSPACING+population_p->pixbufH,
		    population_p->layout);
		if (population_p->layout2) 
		    gdk_draw_layout(icon_view_p->paper->window,
		    icon_view_p->penGC, 
		    column*CELLSIZE + (CELLSIZE-population_p->logical_rect2.width)/2,
		    row*CELLSIZE+TEXTSPACING+population_p->pixbufH+population_p->logical_rect2.height,
		    population_p->layout2);

	    }
}

/**
 * graphics_expose:
 *
 *    redraws a range of grid elements.
 *    
 *    To redraw by x,y coordinates, use
 *    gtk_widget_draw_queue_area(). This will in turn come
 *    back to this function with the correct grid range parameters.
 *    
 *    **/

static 
void
graphics_expose(	icon_view_t *icon_view_p,
			int bottom_row,
			int bottom_column,
			int top_row,
			int top_column)
{
    int row,column;
    TRACE("expose for grid elements %d,%d -> %d,%d",
	    bottom_row,bottom_column,top_row,top_column);
    if (!icon_view_p) return;
    for (column=bottom_column; column<=top_column; column++){
	for (row=bottom_row; row<=top_row; row++){
	    graphics_expose_item(icon_view_p,row,column);
	}
    }
}

/*static 
void
graphics_expose_all(	icon_view_t *icon_view_p){
    TRACE("exposeall for grid");
    graphics_expose(icon_view_p,0,0,icon_view_p->grid_rows,icon_view_p->grid_columns);
}*/

/**
 * graphics_on_size:
 *
 * figures out the number of columns and rows in the paper
 * from resize event data *
 * **/

static
gboolean    
graphics_on_size (	icon_view_t *icon_view_p, 
			int width, 
			int height)
{
    int w,h;
    TRACE("area ...%d x %d = %d\n",width,height,width*height);
    if (!icon_view_p || !icon_view_p->paper) return TRUE;
    
    w=icon_view_p->paper->allocation.width;
    h=icon_view_p->paper->allocation.height;  
    TRACE("paper area...%d\n",w*h);
    
    icon_view_p->grid_columns = w / CELLSIZE;
    icon_view_p->grid_rows = icon_view_p->grid_area / icon_view_p->grid_columns;
    if (icon_view_p->grid_rows * icon_view_p->grid_columns 
	    < 
	icon_view_p->grid_area)
    {
	icon_view_p->grid_rows++;
    }
    
    TRACE("xxx rows and columns: %d,%d",icon_view_p->grid_rows,icon_view_p->grid_columns);
    {
	GtkAdjustment *adjustment=gtk_scrolled_window_get_vadjustment(icon_view_p->scrolled_window);
	TRACE("adjustment upper=%lf, adjustment pagesize=%lf",
		adjustment->upper,adjustment->page_size);
	TRACE("rows=%d, columns=%d, rows*CELLSIZE=%d",icon_view_p->grid_rows,icon_view_p->grid_columns,icon_view_p->grid_rows*CELLSIZE);
	adjustment->upper=icon_view_p->grid_rows*CELLSIZE;
	
    }
	
    return TRUE;
}

/**
 * select_pixbuf:
 *
 * Selects the icon which belongs to the parameter @population_p
 * in the iconview window @icon_view_p
 *
 * **/

static 
void 
select_pixbuf(	icon_view_t *icon_view_p,
			population_t *population_p)
{
    if (!population_p) {
	TRACE("!population_p");
	return;
    }
    if (!population_p->en) return;
    
    if (icon_view_p->module_name) {
	/* ask the module whether the element is selectable 
	 * (by default they will not be)*/
	if (!function_natural("plugins",icon_view_p->module_name,
		    population_p->en,"is_selectable")) 
	{
	    return;
	}
    }
    if (IS_DUMMY_TYPE(population_p->en->type)) return;
    /* this might not be convenient: */
    if (IS_ROOT_TYPE(population_p->en->type))
    {
	if (!population_p->en->module) return;
	if (!function_natural("plugins",population_p->en->module,
		    population_p->en,"get_dnd_path")) return;
    }
    /* this might not be convenient: */
    /*if (!icon_view_p->en) return;*/ /* root elements */
    
    
    if (!population_p->selected_pixbuf && population_p->normal_pixbuf){
	population_p->selected_pixbuf = gdk_pixbuf_copy (population_p->normal_pixbuf);
	gdk_pixbuf_saturate_and_pixelate(population_p->selected_pixbuf,population_p->selected_pixbuf,-8.0,TRUE); 
    }
    population_p->pixbuf = population_p->selected_pixbuf;
    gtk_widget_queue_draw_area (icon_view_p->paper, 
		population_p->column*CELLSIZE, population_p->row*CELLSIZE,
		CELLSIZE,CELLSIZE);	
    if (!population_p->selected) icon_view_p->selection_count++;
    population_p->selected=TRUE;
    if (!g_list_find(icon_view_p->selection_list,population_p->en))
	icon_view_p->selection_list = g_list_append(icon_view_p->selection_list,population_p->en);
    icon_view_p->selected_p=population_p;
    
}

static 
void 
unselect_pixbuf(	icon_view_t *icon_view_p,
			population_t *population_p)
{
    if (!population_p) {
	g_warning("!population_p");
	return;
    }   
    population_p->pixbuf = population_p->normal_pixbuf;
    
    gtk_widget_queue_draw_area (icon_view_p->paper, 
		population_p->column*CELLSIZE, 
		population_p->row*CELLSIZE,
		CELLSIZE,CELLSIZE);	
    if (population_p->selected) icon_view_p->selection_count--;
    population_p->selected=FALSE;
    if (icon_view_p->selection_list){
	icon_view_p->selection_list = g_list_remove(icon_view_p->selection_list,population_p->en);
	if (!g_list_length(icon_view_p->selection_list)){
	    g_list_free(icon_view_p->selection_list);
	    icon_view_p->selection_list=NULL;
	}
    }
}

static 
void 
unselect_all_pixbuf(	icon_view_t *icon_view_p)
{
    GList *tmp;
    for (tmp=icon_view_p->population_list; tmp; tmp=tmp->next){
	population_t *population_p = (population_t *)tmp->data;
	if (!population_p) continue;
	if (population_p == icon_view_p->doing_drag_p) continue;
	if (population_p->selected){
	    unselect_pixbuf(icon_view_p,population_p);
	}
    }
    if (icon_view_p->selection_list) g_list_free(icon_view_p->selection_list);
    icon_view_p->selection_list=NULL;
}
   

static 
void 
saturate_pixbuf(	icon_view_t *icon_view_p,
			population_t *population_p)
{
    if (!population_p) {
	g_warning("!population_p");
	return;
    }
    if (population_p->selected) return;
    if (!population_p->saturated_pixbuf && population_p->normal_pixbuf){
	population_p->saturated_pixbuf = gdk_pixbuf_copy (population_p->normal_pixbuf);
	gdk_pixbuf_saturate_and_pixelate(population_p->saturated_pixbuf,population_p->saturated_pixbuf,38.0,TRUE);
    }
    population_p->pixbuf = population_p->saturated_pixbuf;
    icon_view_p->saturated_p=population_p;
    TRACE("saturate row, column=(%d,%d)",population_p->row,population_p->column);
    gtk_widget_queue_draw_area (icon_view_p->paper, 
		population_p->column*CELLSIZE, population_p->row*CELLSIZE,
		CELLSIZE,CELLSIZE);
    if (getenv("XFFM_DISABLE_TIPS") && strlen(getenv("XFFM_DISABLE_TIPS"))) {
	/* tips disabled condition */
	TRACE("tips disabled by environment variable");
    }
    else {
	tip_timer_t *tip_timer_p = (tip_timer_t *)malloc(sizeof(tip_timer_t));
	memset(tip_timer_p, 0, sizeof(tip_timer_t));
	tip_timer_p->icon_view_p = icon_view_p;
	tip_timer_p->tip_timer = icon_view_p->tip_timer =
	    g_timeout_add (500,graphics_set_entry_tip, tip_timer_p);
    }	
}

static 
void 
unsaturate_pixbuf(	icon_view_t *icon_view_p)
{
    if (icon_view_p->tip_timer) {
	
    }
    if (!icon_view_p->saturated_p) {
	TRACE("!icon_view_p->saturated_p");
	return;
    }  
    if (icon_view_p->saturated_p->en && g_list_find(icon_view_p->selection_list,icon_view_p->saturated_p->en)){
	icon_view_p->saturated_p=NULL;
	TRACE("population is selected...");
	return;
    }
    icon_view_p->saturated_p->pixbuf = icon_view_p->saturated_p->normal_pixbuf;
    gtk_widget_queue_draw_area (icon_view_p->paper, 
		icon_view_p->saturated_p->column*CELLSIZE, 
		icon_view_p->saturated_p->row*CELLSIZE,
		CELLSIZE,CELLSIZE);	
    icon_view_p->saturated_p=NULL;
    graphics_unset_entry_tip(icon_view_p);
}


static 
void 
graphics_saturation_event(	icon_view_t *icon_view_p,
				population_t *population_p,
				double x,
				double y)
{
    if (!icon_view_p) return;
    if (!population_p && icon_view_p->saturated_p){
	TRACE("unsaturating");
	unsaturate_pixbuf(icon_view_p);
	return;
    }
   
    if (icon_view_p->doing_drag_p && x>0 && y>0){
	gdouble h=(x - icon_view_p->mouseX)*(x - icon_view_p->mouseX)+(y - icon_view_p->mouseY)*(y - icon_view_p->mouseY);
	TRACE("h=%f",h);
	/* reverse 1:2 to change behaviour from control to no-control */
	if (h >= 4) enter_drag_state(icon_view_p);
#if 0
	/* if not valid drop target, return */
	if (icon_view_p->module_name){
	    if (!function_natural("plugins",icon_view_p->module_name,population_p->en,"valid_drop_site")) return;
	} else { /* local */
	    if (!IS_DIR(population_p->en->type)) return;
	}
#endif
    }
    if (!population_p || population_p->selected) {
	graphics_unset_entry_tip(icon_view_p);
	return;
    }
    if (population_p && !icon_view_p->saturated_p){
	TRACE("saturating");
	saturate_pixbuf(icon_view_p,population_p);
    }
}
static 
void 
graphics_label_event(	icon_view_t *icon_view_p,
				population_t *population_p)
{
    if (!icon_view_p) return;
    if (!population_p && icon_view_p->label_p){
	TRACE("unsaturating label");
	gtk_widget_queue_draw_area (icon_view_p->paper, 
		icon_view_p->label_p->column*CELLSIZE, 
		icon_view_p->label_p->row*CELLSIZE,
		CELLSIZE,CELLSIZE);	
	icon_view_p->label_p=NULL;
	return;
    }
    if (!population_p || population_p->selected) return;
   
    if (population_p && !icon_view_p->label_p && population_p->en && !IS_ROOT_TYPE(population_p->en->type) && !IS_DUMMY_TYPE(population_p->en->type)){
	if (icon_view_p->module_name){
	    TRACE("FIXME: ask module");
	} else if (IS_PATH(population_p->en->type)) {
	    TRACE("saturating label");
	    icon_view_p->label_p=population_p;
	    gtk_widget_queue_draw_area (icon_view_p->paper, 
		icon_view_p->label_p->column*CELLSIZE, 
		icon_view_p->label_p->row*CELLSIZE,
		CELLSIZE,CELLSIZE);	
	}
    }
}


static
void 
select_pen(		icon_view_t *icon_view_p,
			int p)
{
  static GdkColor pen_color[]={
      {0,65535,65535,65535}, 	/*white*/
      {1,0,0,0}, 		/*black*/
      {2,45000,0,0}, 		/*red*/
      {3,0,30000,0}, 		/*green*/
      {4,0,0,65535}, 		/*blue*/
      {5,28000,28000,0}, 	/*brown*/
      {6,65535,0,65535}, 	/*magenta*/
  };
  if (p < 0 ||  p > 6) {
      g_warning("pen %d out of range.",p);
      return;
  }
	    
  if (!icon_view_p->cmap) {
    int i;
      
    icon_view_p->cmap = gdk_drawable_get_colormap (icon_view_p->paper->window);
    for (i=0; i<7; i++){
	if (!gdk_colormap_alloc_color(icon_view_p->cmap,pen_color+i,TRUE,TRUE)) 
	    g_error ("couldn't allocate color");
    }

    icon_view_p->penGC = gdk_gc_new (icon_view_p->paper->window);
    gdk_gc_set_colormap (icon_view_p->penGC,icon_view_p->cmap);
  }
  gdk_gc_set_foreground (icon_view_p->penGC, pen_color+p);	
}

static
void
graphics_mk_layouts(icon_view_t *icon_view_p,gchar *tag,PangoLayout **layout,PangoLayout **layout2 ){
    PangoRectangle logical_rect;
    gchar *tag1=NULL,*the_tag=NULL;
    
    
    if (!tag || !strlen(tag)) {
	*layout=*layout2=NULL;
	return;
    }
    the_tag=g_strdup(my_utf_string(tag));
    
    tag1=g_strdup(the_tag);

    
#if 10
    /* this has a memory leak... */
    
    if (layout) {
      *layout = gtk_widget_create_pango_layout(icon_view_p->paper, tag1);
      pango_layout_get_pixel_extents(*layout, NULL, &(logical_rect));
      
      while (strlen(tag1)>1 && logical_rect.width+1 > CELLSIZE - 2*TEXTSPACING) {
	gchar *rear;
	int length=strlen(tag1);
	gchar *p=tag1+length-1;
	rear=g_utf8_find_prev_char(tag1,p);	
	length=strlen (rear);
	memset(rear,0,length);
	g_object_unref(*layout);
	*layout = gtk_widget_create_pango_layout(icon_view_p->paper, tag1);
	pango_layout_get_pixel_extents(*layout, NULL, &(logical_rect));	
      }
    }
   
    if (layout2) {
      gchar *p,*tag2=NULL;
      if (strlen(tag1) && strlen(the_tag) > strlen(tag1)){
	p=the_tag+(strlen(tag1));
	
	*layout2 = gtk_widget_create_pango_layout(icon_view_p->paper, p);
	pango_layout_get_pixel_extents(*layout2, NULL, &(logical_rect));
	while (strlen(p)>1 && logical_rect.width+1 > CELLSIZE - 2*TEXTSPACING)
	{
	    if (g_utf8_find_next_char (p,NULL) && p != g_utf8_find_next_char (p,NULL)){
		p=g_utf8_find_next_char (p,NULL);
		
	    }
	    else { p++;}
	    tag2=g_strconcat(my_utf_string("~"),p,NULL);
	    g_object_unref(*layout2);
	    *layout2 = gtk_widget_create_pango_layout(icon_view_p->paper,tag2);
	    pango_layout_get_pixel_extents(*layout2, NULL, &(logical_rect));	
	    g_free(tag2);
	}	
	
      } else 
	  *layout2=NULL;
    }
    g_free(the_tag);
    g_free(tag1);
#else
    /* this is no good for japanese (but does not leak)...*/
    {
	gchar *p,*tag2=NULL;
    if (layout) {
      *layout = gtk_widget_create_pango_layout(icon_view_p->paper, tag1);
      pango_layout_get_pixel_extents(*layout, NULL, &(logical_rect));
      if (logical_rect.width+1 > CELLSIZE - 2*TEXTSPACING) {
	gchar *c,*c_set=". _-";
	for (c=c_set+1; *c; c++){
	    if (strrchr(tag1,*c)){
		g_object_unref(*layout);
		tag2=g_strdup(strrchr(tag1,*c));	
		*strrchr(tag1,*c)=0;
		*layout = gtk_widget_create_pango_layout(icon_view_p->paper, tag1);
		pango_layout_get_pixel_extents(*layout, NULL, &(logical_rect));	
		break;
	    }  
	}
      }
      while (strlen(tag1)>1 && logical_rect.width+1 > CELLSIZE - 2*TEXTSPACING)
      {
	g_free(tag2); tag2=NULL;
	g_object_unref(*layout);
	tag1[strlen(tag1)-2]='-';
	tag1[strlen(tag1)-1]=0;
	*layout = gtk_widget_create_pango_layout(icon_view_p->paper, tag1);
	pango_layout_get_pixel_extents(*layout, NULL, &(logical_rect));	
      }
    }
    
    if (layout2) {
      if (strlen(tag1) && strlen(tag) > strlen(tag1)){
	if (tag2) p=tag2;
	else p=tag+(strlen(tag1)-1);
	*layout2 = gtk_widget_create_pango_layout(icon_view_p->paper, p);
	pango_layout_get_pixel_extents(*layout2, NULL, &(logical_rect));
	while (strlen(p)>1 && logical_rect.width+1 > CELLSIZE - 2*TEXTSPACING)
	{
	    g_object_unref(*layout2);
	    p++;
	    *p='~';
	    *layout2 = gtk_widget_create_pango_layout(icon_view_p->paper, p);
	    pango_layout_get_pixel_extents(*layout2, NULL, &(logical_rect));	
	}	
      } else *layout2=NULL;
    }
    g_free(tag1);
    g_free(tag2);
    }


#endif

     return;
}

static
void
graphics_mk_logical_rect(PangoLayout *layout, PangoRectangle *logical_rect)
{
    if (layout){
	pango_layout_get_pixel_extents(layout, NULL, logical_rect);
	/* ++ because zero is actually unity, not nonexistance */
	logical_rect->width++;
	logical_rect->height++;
    }
}

void 
static
graphics_layout(	icon_view_t *icon_view_p,
			population_t *population_p,
			gchar *actual_tag)
{
    if (!icon_view_p || !population_p){
	g_warning("!icon_view_p || !population_p");
	return;
    }
    graphics_mk_layouts(icon_view_p,actual_tag,
	    &(population_p->layout),&(population_p->layout2));
    graphics_mk_logical_rect(population_p->layout,
	    &(population_p->logical_rect));
    graphics_mk_logical_rect(population_p->layout2,
	    &(population_p->logical_rect2));
    return;
}

static
population_t **
graphics_mk_grid_elements(int grid_elements){
    population_t **population_pp;
    TRACE("grid_elements=%d (elements)",grid_elements);
    population_pp=
	(population_t **)malloc(grid_elements*sizeof(population_t *));
    memset((void *)(population_pp),0,grid_elements*sizeof(population_t *));
    return population_pp;
}

static
gboolean 
graphics_destroy_event	(icon_view_t *icon_view_p)
{
    GList *tmp;
    for (tmp=icon_view_p->go_list; tmp; tmp=tmp->next) {
	destroy_entry((record_entry_t *)(tmp->data));
    }
    g_list_free(icon_view_p->go_list);
    destroy_population(icon_view_p);
    /*if (icon_view_p->pixmap) g_object_unref(icon_view_p->pixmap);*/
    destroy_entry(icon_view_p->en);
    g_free(icon_view_p->population_pp);
    icon_view_p->population_pp=NULL;
    iconview_list=g_list_remove (iconview_list,icon_view_p);
    gtk_widget_destroy(icon_view_p->widgets.window);
    g_free(icon_view_p);
    
    if (!iconview_list) {
	TRACE("No more iconviews on screen.");
	gtk_main_quit();
    }
    else {
	xffm_details->icon_view_p = (icon_view_t *)iconview_list->data;
	set_iconview_restart(xffm_details->icon_view_p);
    }
    return TRUE;
}

