#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>

#include <fox-1.4/fx.h>
#include <fox-1.4/fxkeys.h>
#include <fox-1.4/FXPNGIcon.h>

#include "xfedefs.h"
#include "icons.h"
#include "DirPanel.h"
#include "DirList.h"
#include "Properties.h"
#include "HistInputDialog.h"
#include "File.h"
#include "XFileExplorer.h"
#include "MessageBox.h"


// Global variables
extern FXbool confirm_del;
extern FXString clipboard;
extern FXuint clptype;
extern FXMainWindow *mainWindow;
#if defined(linux)
extern FXStringDict* fsdevices;
extern FXStringDict* mtdevices;
#endif

// Map
FXDEFMAP(DirPanel) DirPanelMap[]={
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_EXPANDTREE,DirPanel::onExpandTree),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_COLLAPSETREE,DirPanel::onCollapseTree),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_PROPERTIES,DirPanel::onCmdProperties),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_ARCHIVE,DirPanel::onCmdAddToArch),
                                   FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,DirPanel::ID_DIR_PANEL,DirPanel::onPopupMenu),
                               	   FXMAPFUNC(SEL_CLICKED,DirPanel::ID_DIR_PANEL,DirPanel::onCmdDirectory),
                               	   FXMAPFUNC(SEL_EXPANDED,DirPanel::ID_DIR_PANEL,DirPanel::onExpand),
                               	   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_COPY,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_COPYTO,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_MOVE,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_RENAME,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_SYMLINK,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_DELETE,DirPanel::onCmdDirDel),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_NEW_DIR,DirPanel::onCmdNewDir),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_XTERM,DirPanel::onCmdXTerm),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_COPY_CLP,DirPanel::onCmdClp),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_CUT_CLP,DirPanel::onCmdClp),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_PASTE_CLP,DirPanel::onCmdPaste),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_SORT_REVERSE,DirPanel::onCmdSortReverse),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_TOGGLE_HIDDEN,DirPanel::onCmdToggleHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_TOGGLE_TREE,DirPanel::onCmdToggleTree),
#if defined(linux)
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_MOUNT,DirPanel::onCmdMount),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_UMOUNT,DirPanel::onCmdMount),
                               	   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_MOUNT,DirPanel::onUpdMount),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_UMOUNT,DirPanel::onUpdUnmount),
#endif								   
                               	   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_PASTE_CLP,DirPanel::onUpdPaste),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_TOGGLE_HIDDEN,DirPanel::onUpdToggleHidden),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_TOGGLE_TREE,DirPanel::onUpdToggleTree),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_CUT_CLP,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_ARCHIVE,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_DIR_MOVE,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_DIR_RENAME,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_DIR_DELETE,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_NEW_DIR,DirPanel::onUpdMenu),
                                    FXMAPFUNC(SEL_UPDATE,DirPanel::ID_STATUS,DirPanel::onUpdStatus),
                          		};

// Object implementation 
FXIMPLEMENT(DirPanel,FXVerticalFrame,DirPanelMap,ARRAYNUMBER(DirPanelMap))

// Construct Directory Panel
DirPanel::DirPanel(FXComposite *p,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXVerticalFrame(p,opts,x,y,w,h,0,0,0,0)
{
	// Create some icons
	attricon=new FXPNGIcon(getApp(),attrib);
    archicon=new FXPNGIcon(getApp(),arch);
    sorticon=new FXPNGIcon(getApp(),sort);
    newfoldericon=new FXPNGIcon(getApp(),newfolder);
	delicon=new FXPNGIcon(getApp(),filedelete);	
	copyicon=new FXPNGIcon(getApp(),copy_clp);	
	cuticon=new FXPNGIcon(getApp(),cut_clp);	
	pasteicon=new FXPNGIcon(getApp(),paste_clp);	
	mapicon=new FXPNGIcon(getApp(),maphost);	
	unmapicon=new FXPNGIcon(getApp(),unmaphost);	
	xtermicon=new FXPNGIcon(getApp(),shell);
	symlinkicon=new FXPNGIcon(getApp(),minilink);
	renameicon=new FXPNGIcon(getApp(),renameit);
	moveicon=new FXPNGIcon(getApp(),moveit);
	exptreeicon=new FXPNGIcon(getApp(),exptree);
	expfoldericon=new FXPNGIcon(getApp(),expfolder);
	colltreeicon=new FXPNGIcon(getApp(),colltree);
	collfoldericon=new FXPNGIcon(getApp(),collfolder);

	FXIcon *showhiddenicon=new FXPNGIcon(getApp(),showhidden);
	FXIcon *hidehiddenicon=new FXPNGIcon(getApp(),hidehidden);

	// Construct directory panel
    FXVerticalFrame* cont=new FXVerticalFrame(this,LAYOUT_FILL_Y|LAYOUT_FILL_X|FRAME_SUNKEN,0,0,0,0, 0,0,0,0, 1,1);
	FXPacker* packer=new FXHorizontalFrame(cont,JUSTIFY_LEFT|LAYOUT_FILL_X|FRAME_RAISED,0,0,0,0, 0,0,0,0);
	new FXLabel(packer,_("Folders"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X);

    list=new DirList(cont,this,ID_DIR_PANEL,LAYOUT_FILL_X|LAYOUT_FILL_Y|TREELIST_BROWSESELECT|TREELIST_SHOWS_LINES|TREELIST_SHOWS_BOXES|ICONLIST_BIG_ICONS);

	statusbar=new FXHorizontalFrame(cont,JUSTIFY_LEFT|LAYOUT_FILL_X,0,0,0,0, 3,3,3,3);
    statusbar->setTarget(this);
    statusbar->setSelector(FXSEL(SEL_UPDATE,DirPanel::ID_STATUS));
    new FXToggleButton(statusbar,_("\tShow hidden folders (Ctrl-F5)"),_("\tHide hidden folders (Ctrl-F5)"),showhiddenicon,hidehiddenicon,this,
	                   ID_TOGGLE_HIDDEN,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);
    status=new FXLabel(statusbar,_("Status"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_SUNKEN);
}


// Destructor
DirPanel::~DirPanel()
{
	delete attricon;
	delete archicon;
	delete sorticon;
	delete newfoldericon;
	delete delicon;
	delete copyicon;
	delete cuticon;
	delete pasteicon;
	delete mapicon;
	delete unmapicon;
	delete xtermicon;
	delete symlinkicon;
	delete renameicon;
	delete moveicon;
	delete exptreeicon;
	delete expfoldericon;
	delete colltreeicon;
	delete collfoldericon;
	delete list;
	delete statusbar;
	delete status;
}

// Create X window
void DirPanel::create()
{
	FXVerticalFrame::create();
}

// To pass the expand message to DirList
long DirPanel::onExpand(FXObject*,FXSelector,void* ptr)
{
	list->handle(this,FXSEL(SEL_EXPANDED,0),(void*)ptr);
	return 1;
}


// Change the directory when clicking on the tree list
long DirPanel::onCmdDirectory(FXObject*,FXSelector,void* ptr)
{
    TreeItem *item=(TreeItem*)ptr;
    if(item)
    {
        FXString directory=list->getItemPathname(item);
        if(!::isReadExecutable(directory))
        {
            MessageBox::error(this,MBOX_OK,_("Error"),_(" Permission to: %s denied."), directory.text());
            return 0;
        }
		FilePanel* currentpanel=((XFileExplorer*) mainWindow)->lpanel->current;
		FXComboBox* address=((XFileExplorer*) mainWindow)->address;
        currentpanel->list->setDirectory(directory);
        
		// Remember latest directory in the location address
       	FXString item;
		FXint i=0;
		FXint count=address->getNumItems();
       	FXString p=currentpanel->list->getDirectory();

		if(!count)
		{
			count++;
			address->insertItem(0,address->getText());
		}
       	while(i < count)
		{
       		item=address->getItem(i++);
       		if(::streq((const char*)&p[0],(const char*)&item[0]))
			{
				i--;
				break;
			}
       	}
       	if(i==count)
			address->insertItem(0,currentpanel->list->getDirectory());
    }
    return 1;
}

// Toggle hidden files
long DirPanel::onCmdToggleHidden(FXObject*,FXSelector,void*)
{
    list->showHiddenFiles(!list->showHiddenFiles());
    return 1;
}


// Update toggle hidden files widget
long DirPanel::onUpdToggleHidden(FXObject* sender,FXSelector,void*)
{
    if(list->showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


long DirPanel::onCmdToggleTree(FXObject* sender,FXSelector sel,void* ptr)
{
	return this->handle(sender,FXSEL(SEL_COMMAND,FXWindow::ID_TOGGLESHOWN),ptr);
}

long DirPanel::onUpdToggleTree(FXObject* sender,FXSelector sel,void* ptr)
{
    if(this->shown())
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_CHECK),ptr);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_UNCHECK),ptr);
    return 1;
}


// Directory list context menu
long DirPanel::onPopupMenu(FXObject* o,FXSelector s,void* p)
{
	// Current item becomes item under cursor
	FXint x, y, xitem, yitem;
    FXuint state;
	getRoot()->getCursorPosition(x,y,state);
	list->getCursorPosition(xitem,yitem,state);   	    
	DirItem* item=(DirItem*)list->getItemAt(xitem,yitem);

	// If no item under cursor, then return
	if (item==NULL) return 0;
	
	// If item, then set it current and set directory in DirList and FileList
	list->setCurrentItem(item,TRUE);
	FXString dir=list->getItemPathname((TreeItem*)item);
	list->setDirectory(dir,TRUE);
	((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(dir);

    // Popup menu pane
    FXMenuPane menu(this);
    
	// Menu
#if defined(linux)
	if (::isLink(dir))
		dir=readlink(dir);
    if(fsdevices->find(dir.text()))
	{
		new FXMenuCommand(&menu,_("&Mount"),mapicon,this,ID_MOUNT);
		new FXMenuCommand(&menu,_("Unmount"),unmapicon,this,ID_UMOUNT);
    	new FXMenuSeparator(&menu);
	}
#endif
    new FXMenuCommand(&menu,_("&Terminal"),xtermicon,this,DirPanel::ID_XTERM);
	new FXMenuCommand(&menu,_("New f&older..."),newfoldericon,this,DirPanel::ID_NEW_DIR);
	new FXMenuCommand(&menu,_("&Add to archive..."),archicon,this,DirPanel::ID_ARCHIVE);
    new FXMenuSeparator(&menu);
	new FXMenuCommand(&menu,_("&Copy"),copyicon,this,DirPanel::ID_COPY_CLP);
	new FXMenuCommand(&menu,_("C&ut"),cuticon,this,DirPanel::ID_CUT_CLP);
	new FXMenuCommand(&menu,_("&Paste"),pasteicon,this,DirPanel::ID_PASTE_CLP);
    new FXMenuSeparator(&menu);
	new FXMenuCommand(&menu,_("Re&name..."),renameicon,this,DirPanel::ID_DIR_RENAME);
	new FXMenuCommand(&menu,_("Copy &to..."),copyicon,this,DirPanel::ID_DIR_COPYTO);
	new FXMenuCommand(&menu,_("&Move..."),moveicon,this,DirPanel::ID_DIR_MOVE);
	new FXMenuCommand(&menu,_("Symlin&k..."),symlinkicon,this,DirPanel::ID_DIR_SYMLINK);
	new FXMenuCommand(&menu,_("&Delete"),delicon,this,DirPanel::ID_DIR_DELETE);
    new FXMenuSeparator(&menu);
	new FXMenuCheck(&menu,_("&Hidden folders"),this,DirPanel::ID_TOGGLE_HIDDEN);
	new FXMenuCommand(&menu,_("&Reverse order"),sorticon,this,DirPanel::ID_SORT_REVERSE);
	new FXMenuCommand(&menu,_("E&xpand tree"),exptreeicon,this,DirPanel::ID_EXPANDTREE);
	new FXMenuCommand(&menu,_("Collap&se tree"),colltreeicon,this,DirPanel::ID_COLLAPSETREE);
    new FXMenuSeparator(&menu);
	new FXMenuCommand(&menu,_("Proper&ties"),attricon,this,DirPanel::ID_PROPERTIES);
	
    menu.create();
    menu.popup(NULL,x,y);
	getApp()->runModalWhileShown(&menu);
    return 1;
}




// Helper function used to explore the directory tree and expand or collapse it
long DirPanel::exploreUp(DirItem *item, const DirItem *rootitem, const FXint task)
{
	DirItem* parentitem=item;
	
	if (task==ID_EXPANDTREE)
		list->expandTree((TreeItem*)item,TRUE);
	else
		list->collapseTree((TreeItem*)item,TRUE);
	
	item=(DirItem*)item->getFirst();
	if (!item)
		exploreDown(parentitem,rootitem,task);
	else
		exploreUp(item,rootitem,task);
	return 1;	
}


// Helper function used to explore the directory tree and expand or collapse it
long DirPanel::exploreDown(DirItem *item, const DirItem *rootitem, const FXint task)
{
	if (item==rootitem)
		return 1;
	
	DirItem* parentitem=(DirItem*)item->getParent();
	
	if (task==ID_EXPANDTREE)
		list->expandTree((TreeItem*)item,TRUE);
	else
		list->collapseTree((TreeItem*)item,TRUE);
	item=(DirItem*)item->getNext();
	
	if (!item)
		exploreDown(parentitem,rootitem,task);
	else
	{
		if (task==ID_EXPANDTREE)
			list->expandTree((TreeItem*)item,TRUE);
		else
			list->collapseTree((TreeItem*)item,TRUE);
		if (!list->isItemLeaf(item))
			exploreUp(item,rootitem,task);
		else
			exploreDown(item,rootitem,task);
	}
	return 1;		
}
				

// Expand the directory tree under cursor
long DirPanel::onExpandTree(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem* rootitem=(DirItem*)list->getCurrentItem();
	getApp()->beginWaitCursor();
	exploreUp(rootitem,rootitem,ID_EXPANDTREE);
	getApp()->endWaitCursor();
	
	return 1;
}


// Collapse the directory tree under cursor
long DirPanel::onCollapseTree(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem* rootitem=(DirItem*)list->getCurrentItem();
	getApp()->beginWaitCursor();
	exploreUp(rootitem,rootitem,ID_COLLAPSETREE);
	getApp()->endWaitCursor();
	
	return 1;
}


// Directory properties
long DirPanel::onCmdProperties(FXObject* sender,FXSelector,void* ptr)
{
    // Current item
	DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);
	
	PropertiesBox* attrdlg=new PropertiesBox(this,FXFile::name(pathname),FXFile::directory(pathname));	
   	if(attrdlg->execute(PLACEMENT_SCREEN))
        list->setDirectory(pathname,TRUE);
    delete attrdlg;
   	return 1;
}


// Add files or directory to an archive
long DirPanel::onCmdAddToArch(FXObject* o,FXSelector sel,void* ptr)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString name=list->getItemText(item);
	FXString pathname=list->getItemPathname((TreeItem*)item);	
	FXString parentdir=FXFile::directory(pathname);
	FXString archive="", cmd;
	
    static char history[ADD_TO_ARCH_HIST_SIZE][100];
    static int n=0;
    int i;
	File *f;	
    HistInputDialog* dialog;
    
	dialog=new HistInputDialog(this,archive,
	       _("Create new archive: (e.g. archive.tar.gz)\n=>Supported formats : tar, tgz, tbz2, taz, zip, rar, lzh"),
	       _("Add To Archive"));
    for(int i=0;i<n;i++)
        dialog->appendItem(history[i]);
    dialog->CursorEnd();
    if(dialog->execute())
    {
        archive=parentdir+PATHSEPSTRING+dialog->getText();

        // Handle zip and bz2 archives
        FXString ext=archive.rafter('.',1);
		ext.lower();
		if (ext=="zip")
            cmd="zip -r ";
        else if (ext=="bz2" || ext=="tbz2")
            cmd="tar -jcvf ";
		else if (ext=="rar")
			cmd="rar a ";
		else if (ext=="lzh")
			cmd="lha a ";
		else if (ext=="z")
			cmd="tar -Zcvf ";

		// Archive command name
		cmd+=archive+" "+"\""+name+"\"";
					
		// File object
		f=new File(this,_("Create archive"),ARCHIVE);
		f->create();
				   
		// Create archive
		chdir(parentdir.text());
		f->archive(archive,cmd);
 
		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,MBOX_OK,_("Error"),_("Archive creation cancelled!"));
		}					
		delete f; 

        // Manage dialog history size
		n=dialog->getHistorySize();
        n++;
        cmd=dialog->getText();
        if(n>ADD_TO_ARCH_HIST_SIZE)
            n=ADD_TO_ARCH_HIST_SIZE;
        for(i=0;i<n-1;i++)
            if(::streq(history[i],cmd.text()))
                break;
        if(i==n-1)
        {
            strlcpy(history[0],cmd.text(),sizeof(history[0]));
            for(i=1;i<n;i++)
                strlcpy(history[i],dialog->getHistoryItem(i-1).text(),dialog->getHistoryItem(i-1).length()+1);
        }
        else
            n--;

		// Display parent directory in DirList and FileList
		list->setDirectory(parentdir,TRUE);
		((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(parentdir);
    }
    delete dialog;
    return 1;
}


// Copy/Move/Rename/Symlink directory
long DirPanel::onCmdDirMan(FXObject* sender,FXSelector sel,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	
	// Parameters passed to XFileExplorer::onCmdFileMan
	// -1 is used to tell that we are calling from DirPanel
	FXString source=list->getItemPathname((TreeItem*)item);	
    FXString target=FXFile::directory(source);
	FXString param=target+"\n"+"-1"+"\n"+source;

    switch(FXSELID(sel))
    {
    case ID_DIR_COPY:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_COPY);
        break;
    case ID_DIR_COPYTO:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_COPYTO);
        break;
    case ID_DIR_MOVE:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_MOVE);
        break;
    case ID_DIR_RENAME:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_RENAME);
        break;
    case ID_DIR_SYMLINK:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_SYMLINK);
        break;
    }
    mainWindow->handle(this,sel,strdup(param.text()));

	// Force dirpanel refresh
    list->onRefresh(0,0,0);
    return 1;
}

// Reverse the sort order
long DirPanel::onCmdSortReverse(FXObject* sender,FXSelector sel,void*)
{
        list->handle(this,FXSEL(SEL_COMMAND,DirList::ID_SORT_REVERSE),NULL);
		return 1;
}


// Delete directory
long DirPanel::onCmdDirDel(FXObject*,FXSelector,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);	
	FXString parentdir=FXFile::directory(pathname);	

    // If we don't have permission to write to the parent directory
    if(!::isWritable(parentdir))
    {
        MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s: Permission denied"), parentdir.text());
        return 0;
    }

    if(confirm_del)
    {
        FXPNGIcon icon(getApp(),delete_big);
		FXString message=_("Delete folder ") + pathname + " ?";
		MessageBox box(this,_("Confirm Delete"),message,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
        if(box.execute(PLACEMENT_CURSOR) != MBOX_CLICKED_OK)
            return 0;
    }

	// File object
   	File* f=new File(this,_("File Delete"),DELETE);
   	f->create();

    // If we don't have permission to write to the directory
 	if (!::isLWritable(pathname))
    {
		// Get process uid and directory uid
		FXuint uid=getuid();
		struct stat buf;
		lstatout(pathname.text(),&buf);
		FXuint diruid=buf.st_uid;

		// If we are owner of the directory
		if (diruid==uid)
		{
			// Dialog to confirm directory deletion
			f->allowPdialog=FALSE;
			f->hide();
			FXPNGIcon icon(getApp(),errorIcon);
			FXString str=_("Folder ") + pathname + _(" is write-protected, delete it anyway?");
        	MessageBox box(this,_("Confirm Delete"),str,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);        	
			FXuint answer=box.execute(PLACEMENT_SCREEN);
			if(answer == MBOX_CLICKED_OK)
			{
				f->remove(pathname); 	

				// If action is cancelled in progress dialog
				if (f->isCancelled)
					MessageBox::error(this,MBOX_OK,_("Error"),_("Delete folder operation cancelled!"));
				
				// Return to parent directory in DirList and FileList
				list->setDirectory(parentdir,TRUE);
				((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(parentdir);
			}
		}
					
		// If we are not owner of the directory
		else
		{
			f->allowPdialog=FALSE;
			f->hide();
        	MessageBox::error(this,MBOX_OK,_("Error"),_("Can't delete folder %s: Permission denied"), pathname.text());
    	}
	}

	// If we have permission to write
	else
	{
		f->remove(pathname); 	

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,MBOX_OK,_("Error"),_("Delete file operation cancelled!"));
		}
		// Return to parent directory in DirList and FileList
		list->setDirectory(parentdir,TRUE);
		((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(parentdir);
	}

	delete f;

	// Force DirPanel and FilePanel refresh
 	//mainWindow->handle(this,FXSEL(SEL_COMMAND,XFileExplorer::ID_REFRESH),NULL);
	list->onRefresh(0,0,0);
    return 1;
}


// Create new directory
long DirPanel::onCmdNewDir(FXObject*,FXSelector,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();

    FXString dirname="";	
	FXString dirpath=list->getItemPathname((TreeItem*)item);
    if(dirpath!=ROOTDIR)
        dirpath+=PATHSEPSTRING;
    
    InputDialog* dialog=new InputDialog(this,dirname,_("Create new folder..."),_("New Folder"));
    dialog->CursorEnd();
    if(dialog->execute(PLACEMENT_CURSOR))
    {
		dirname=dirpath+dialog->getText();		
        if(dirname!=dirpath)
        {
			// Create the new dir
			FXint ret=mkdir(dirname.text(),0755);
			if (ret==-1)
			{
         		MessageBox::error(this,MBOX_OK,_("Error"),"%s",strerror(errno));
    			return 0;
			}
		}							
    }
    delete dialog;
	
	// Force dirpanel refresh
	list->onRefresh(0,0,0);
    return 1;
}


// Run Terminal in the selected directory
long  DirPanel::onCmdXTerm(FXObject*,FXSelector,void*)
{
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString buf=list->getItemPathname((TreeItem*)item);
	chdir(buf.text());
    FXString cmd=getApp()->reg().readStringEntry("PROGS","xterm","xterm -sb");
    cmd += " &";
    system(cmd.text());
    return 1;
}

// Paste file(s)
long DirPanel::onCmdPaste(FXObject*,FXSelector sel,void*)
{
    if(clipboard.empty())
        return 1;

    FXString param;
    FXString targetdir;

	targetdir=((XFileExplorer*) mainWindow)->lpanel->current->list->getDirectory();
    param=targetdir+"\n"+clipboard;

    switch(clptype)
    {
    case COPY_CLP:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_COPY);
        break;
    case CUT_CLP:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_CUT);
        break;
    }
    mainWindow->handle(this,sel,strdup(param.text()));

	// Force dirpanel refresh
	list->onRefresh(0,0,0);

    return 1;
}


// Clipboard
long DirPanel::onCmdClp(FXObject*,FXSelector sel,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);

	// Empty clipboard
	clipboard="";

	// Clipboard type
	if (FXSELID(sel)==ID_CUT_CLP)
		clptype=CUT_CLP;
	else
		clptype=COPY_CLP;

	clipboard+=pathname+"\n";
    clipboard=FXStringVal(1)+"\n"+clipboard;

    return 1;
}


#if defined(linux)
// Mount/Unmount directory
long DirPanel::onCmdMount(FXObject*,FXSelector sel,void*)
{
    FXString cmd, msg, text;
	unsigned int op;
	File *f;
	
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString dir=list->getItemPathname((TreeItem*)item);

	// If symbolic link
	if (::isLink(dir))
		dir=FXFile::symlink(dir);

    // Select the command and set the appropriate message
    if(FXSELID(sel)==ID_MOUNT)
    {
        op=MOUNT;
		msg=_("Mount");
        cmd="mount ";
    }
    else
    {
        op=UNMOUNT;
		msg=_("Unmount");
        cmd="umount ";
    }
    cmd+=dir;
    cmd+=" 2>&1";
    chdir(ROOTDIR);

	// Wait cursor
	getApp()->beginWaitCursor();

	// File object
	text=msg + _(" file system...");
	f=new File(this,text.text(),op);
	f->create();
				   
	// Mount/unmount file system
	text=msg + _(" the folder :");
	f->mount(dir,text,cmd,op);
 
	// If action is cancelled in progress dialog
	if (f->isCancelled)
	{
		f->hide();
		text=msg + _(" operation cancelled!");
		MessageBox::error(this,MBOX_OK,_("Error"),text.text());
		delete f;
		return 0;
	}					
	
	getApp()->endWaitCursor();
	delete f; 

	// Force dirpanel refresh
    list->onRefresh(0,0,0);

    return 1;
}

// Update the Mount menu item
long DirPanel::onUpdMount(FXObject* o,FXSelector sel,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString dir=list->getItemPathname((TreeItem*)item);

    if(fsdevices->find(dir.text()) && !mtdevices->find(dir.text()))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Update the Unmount menu item
long DirPanel::onUpdUnmount(FXObject* o,FXSelector sel,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString dir=list->getItemPathname((TreeItem*)item);

    if(fsdevices->find(dir.text()) && mtdevices->find(dir.text()))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}
#endif


// Update the paste button and menus
long DirPanel::onUpdPaste(FXObject* o,FXSelector,void*)
{
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
		return 0;
	
	// Path name of the selected item
	FXString dir=list->getItemPathname(item);	

    return 1;
}


// Update menu items and toolbar buttons that are related to file operations
long DirPanel::onUpdMenu(FXObject* o,FXSelector,void*)
{
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
		return 0;
	
	// Path name of the selected item
	FXString dir=list->getItemPathname(item);	
    return 1;
}


// Update the status bar and the directory path name
long DirPanel::onUpdStatus(FXObject* sender,FXSelector,void*)
{
	unsigned long long dsize;
	char buf[MAXPATHLEN+1], size[64];
	FXString hsize;
	
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
	{
		status->setText("--");
		return 0;
	}
	
	// Path name of the selected item
	FXString path=list->getItemPathname(item);	

	// Size of the files present in the directory
	path=quote(path,FALSE);
	strlcpy(buf,path.text(),path.length()+1);
	dsize=dirsize(buf);

	// Size in human readable form
	snprintf(size,sizeof(size)-1,"%llu",dsize);
	hsize=::hSize(size);

	// Update the status label
	snprintf(buf,sizeof(buf)-1,_("%s"),hsize.text());
	status->setText(buf);

	// Update the path in the window title
	if(getuid()==0)
		mainWindow->setTitle("Xfe (root) - " + path);
	else
		mainWindow->setTitle("Xfe - " + path);
    return 1;
}
