/*
 *  xfce4-notifyd
 *
 *  Copyright (c) 2008 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <gio/gio.h>
#include <gio/gdesktopappinfo.h>
#include <gtk/gtk.h>
#include <gtk/gtkx.h>

#include <xfconf/xfconf.h>
#include <libxfce4ui/libxfce4ui.h>
#include <libnotify/notify.h>

#include <common/xfce-notify-common.h>
#include <common/xfce-notify-log.h>

#include "xfce4-notifyd-config.ui.h"
#include "xfce4-notifyd-config-known-app.ui.h"

#define LOG_DISPLAY_LIMIT             100

typedef struct
{
    GtkWidget  *log_level;
    GtkWidget  *log_level_apps;
    GtkWidget  *log_level_apps_label;
    GtkWidget  *infobar_label;
    GtkWidget  *log_listbox;
    GtkToolbar *log_toolbar;
} NotificationLogWidgets;

typedef struct
{
    GtkWidget  *do_slideout;
    GtkWidget  *do_slideout_label;
} NotificationSlideoutWidgets;

typedef struct
{
    const gchar *app_id;
    gchar *app_name;
    gint count;
} LogAppCount;

typedef struct
{
    XfconfChannel *channel;
    gchar *known_application;
    gchar *known_application_name;

    GtkWidget *app_label;
    GtkWidget *mute_switch;
    GtkWidget *allow_criticals;
    GtkWidget *include_in_log;
} KnownApplicationSignalData;

static void handle_app_switch_property_changed(XfconfChannel *channel,
                                               const gchar *property_name,
                                               const GValue *value,
                                               KnownApplicationSignalData *signal_data);

static void
xfce_notifyd_config_show_notification_callback(NotifyNotification *notification,
                                               const char         *action,
                                               gpointer            unused)
{
  /* Don't do anything, we just have a button to show its style */
}

static void
xfce_notifyd_config_show_notification_preview(GtkWindow *parent_window)
{
    NotifyNotification *notification;
    GError             *error = NULL;

    notification =
        notify_notification_new(_("Notification Preview"),
                                _("This is what notifications will look like"),
                                "org.xfce.notification");

    notify_notification_add_action(notification,
                                   "button",
                                   _("Button"),
                                   (NotifyActionCallback) xfce_notifyd_config_show_notification_callback,
                                   NULL,
                                   NULL);

    if (!notify_notification_show(notification, &error)) {
        xfce_dialog_show_error(parent_window, error,
                               _("Notification preview failed"));

        g_error_free(error);
    }

    g_object_unref(notification);
}

static gchar *
xfce4_notifyd_slider_format_value(GtkScale *slider,
                                  gdouble value,
                                  gpointer user_data)
{
    return g_strdup_printf("%d%%", (gint)(value * 100));
}

static void
xfce4_notifyd_config_theme_combo_changed(GtkComboBox *theme_combo,
                                         gpointer user_data)
{
    XfconfChannel *channel = user_data;
    GtkTreeModel *model;
    GtkTreeIter iter;
    gchar *theme = NULL;

    if(!gtk_combo_box_get_active_iter(theme_combo, &iter))
        return;

    model = gtk_combo_box_get_model (theme_combo);

    gtk_tree_model_get(model, &iter, 0, &theme, -1);
    xfconf_channel_set_string(channel, "/theme", theme);
    g_free(theme);
}

static void
xfce4_notifyd_config_theme_changed(XfconfChannel *channel,
                                   const gchar *property,
                                   const GValue *value,
                                   gpointer user_data)
{
    GtkWidget *theme_combo = user_data;
    GtkListStore *ls;
    GtkTreeIter iter;
    gchar *theme;
    const gchar *new_theme;

    new_theme = G_VALUE_TYPE(value) ? g_value_get_string(value) : "Default";

    ls = GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(theme_combo)));

    for(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(ls), &iter);
        gtk_list_store_iter_is_valid(ls, &iter);
        gtk_tree_model_iter_next(GTK_TREE_MODEL(ls), &iter))
    {
        gtk_tree_model_get(GTK_TREE_MODEL(ls), &iter, 0, &theme, -1);
        if(!strcmp(theme, new_theme)) {

            gtk_combo_box_set_active_iter(GTK_COMBO_BOX(theme_combo),
                                          &iter);
            g_free(theme);

            xfce_notifyd_config_show_notification_preview(GTK_WINDOW(gtk_widget_get_toplevel(theme_combo)));

            return;
        }
        g_free(theme);
    }

    gtk_list_store_append(ls, &iter);
    gtk_list_store_set(ls, &iter, 0, new_theme, -1);
    gtk_combo_box_set_active_iter(GTK_COMBO_BOX(theme_combo), &iter);
}

static void
list_store_add_themes_in_dir(GtkListStore *ls,
                             const gchar *path,
                             const gchar *current_theme,
                             GHashTable *themes_seen,
                             GtkTreeIter *current_theme_iter)
{
    GDir *dir;
    const gchar *file;
    gchar *old_filename = NULL;

    dir = g_dir_open(path, 0, NULL);
    if(!dir)
        return;

    while((file = g_dir_read_name(dir))) {
        gchar *filename;

        if(g_hash_table_lookup(themes_seen, file))
            continue;

        filename =
            g_build_filename(path, file, "xfce-notify-4.0", "gtk.css", NULL);

        if (g_file_test(filename, G_FILE_TEST_IS_REGULAR)) {
            GtkTreeIter iter;
            if (old_filename != NULL) {
                if (g_ascii_strcasecmp (old_filename, filename) < 0)
                    gtk_list_store_append(ls, &iter);
                else
                    gtk_list_store_prepend(ls, &iter);
                g_free (old_filename);
                old_filename = NULL;
            }
            else
                gtk_list_store_append(ls, &iter);

            gtk_list_store_set(ls, &iter, 0, file, -1);
            g_hash_table_insert(themes_seen, g_strdup(file),
                                GUINT_TO_POINTER(1));

            if(!strcmp(file, current_theme))
                memcpy(current_theme_iter, &iter, sizeof(iter));
        }

        g_free (old_filename);
        old_filename = filename;
    }
    g_free (old_filename);

    g_dir_close(dir);
}

static void
xfce4_notifyd_config_setup_theme_combo(GtkWidget *theme_combo,
                                       const gchar *current_theme)
{
    GtkListStore *ls;
    gchar *dirname, **dirnames;
    GHashTable *themes_seen;
    gint i;
    GtkTreeIter current_theme_iter;

    ls = gtk_list_store_new(1, G_TYPE_STRING);
    themes_seen = g_hash_table_new_full(g_str_hash, g_str_equal,
                                        (GDestroyNotify)g_free, NULL);

    dirname = g_build_filename(xfce_get_homedir(), ".themes", NULL);
    list_store_add_themes_in_dir(ls, dirname, current_theme,
                                 themes_seen, &current_theme_iter);
    g_free(dirname);

    dirnames = xfce_resource_lookup_all(XFCE_RESOURCE_DATA, "themes/");
    for(i = 0; dirnames && dirnames[i]; ++i)
        list_store_add_themes_in_dir(ls, dirnames[i], current_theme,
                                     themes_seen, &current_theme_iter);
    g_strfreev(dirnames);

    g_hash_table_destroy(themes_seen);

    gtk_combo_box_set_model(GTK_COMBO_BOX(theme_combo), GTK_TREE_MODEL(ls));

    if(gtk_list_store_iter_is_valid(ls, &current_theme_iter))
        gtk_combo_box_set_active_iter(GTK_COMBO_BOX(theme_combo),
                                      &current_theme_iter);

    g_object_unref(G_OBJECT(ls));
}

static void
xfce_notifyd_config_dialog_response(GtkWidget *dialog, gint response, gpointer unused)
{
  if (response == 0)
      g_signal_stop_emission_by_name (dialog, "response");
}

static void
xfce_notifyd_config_preview_clicked(GtkButton *button)
{
  GtkWidget *window = gtk_widget_get_toplevel (GTK_WIDGET (button));

  xfce_notifyd_config_show_notification_preview (GTK_WINDOW (window));
}

/* Shows a separator before each row. */
static void
display_header_func (GtkListBoxRow *row,
                     GtkListBoxRow *before,
                     gpointer       user_data)
{
  if (before != NULL)
    {
      GtkWidget *header = gtk_separator_new (GTK_ORIENTATION_HORIZONTAL);
      gtk_widget_show (header);

      gtk_list_box_row_set_header (row, header);
    }
}

static gboolean
xfce_g_strv_remove(gchar **strv,
                   const gchar *to_remove)
{
    gint i;

    for (i = 0; strv[i] != NULL; ++i) {
        if (strcmp(strv[i], to_remove) == 0) {
            break;
        }
    }

    if (G_LIKELY(strv[i] != NULL)) {
        gint remaining = g_strv_length(&strv[i+1]);

        g_free(strv[i]);

        if (remaining > 0) {
            memmove(&strv[i], &strv[i+1], remaining * sizeof(gchar *));
        }
        strv[i+remaining] = NULL;

        return TRUE;
    } else {
        return FALSE;
    }
}

static void
xfce_add_to_string_list_property(XfconfChannel *channel,
                                 const gchar *property_name,
                                 gchar **strv,
                                 const gchar *to_add)
{
    if (strv == NULL || !g_strv_contains((const gchar *const *)strv, to_add)) {
        gint len = strv == NULL ? 0 : g_strv_length(strv);
        gchar **new_strv = g_new(gchar *, len + 2);
        if (len > 0) {
            memcpy(new_strv, strv, len * sizeof(gchar *));
        }
        new_strv[len] = (gchar *)to_add;
        new_strv[len + 1] = NULL;
        xfconf_channel_set_string_list(channel, property_name, (const gchar *const *)new_strv);
        g_free(new_strv);
    }
}

static void
xfce_remove_from_string_list_property(XfconfChannel *channel,
                                      const gchar *property_name,
                                      gchar **strv,
                                      const gchar *to_remove)
{
    if (strv != NULL) {
        if (xfce_g_strv_remove(strv, to_remove)) {
            if (strv[0] == NULL) {
                xfconf_channel_reset_property(channel, property_name, FALSE);
            } else {
                xfconf_channel_set_string_list(channel, property_name, (const gchar *const *)strv);
            }
        }
    }
}

static void
handle_app_switch_widget_toggled(KnownApplicationSignalData *signal_data,
                                 const gchar *property_name,
                                 gboolean add_to_list)
{
    gchar **strs = xfconf_channel_get_string_list(signal_data->channel, property_name);

    g_signal_handlers_block_by_func(signal_data->channel,
                                    handle_app_switch_property_changed,
                                    signal_data);
    if (add_to_list) {
        xfce_add_to_string_list_property(signal_data->channel, property_name, strs, signal_data->known_application);
    } else {
        xfce_remove_from_string_list_property(signal_data->channel, property_name, strs, signal_data->known_application);
    }
    g_signal_handlers_unblock_by_func(signal_data->channel,
                                      handle_app_switch_property_changed,
                                      signal_data);


    g_strfreev(strs);
}

static void
xfce4_notifyd_mute_switch_activated(GtkSwitch *sw,
                                    gboolean state,
                                    KnownApplicationSignalData *signal_data)
{
    handle_app_switch_widget_toggled(signal_data, MUTED_APPLICATIONS_PROP, state);
    gtk_widget_set_sensitive(signal_data->app_label, !state);
}

static void
xfce4_notifyd_allow_criticals_switch_activated(GtkSwitch *sw,
                                               gboolean state,
                                               KnownApplicationSignalData *signal_data)
{
    handle_app_switch_widget_toggled(signal_data, DENIED_CRITICAL_NOTIFICATIONS_PROP, !state);
}

static void
xfce4_notifyd_include_in_log_switch_activated(GtkSwitch *sw,
                                              gboolean state,
                                              KnownApplicationSignalData *signal_data)
{
    handle_app_switch_widget_toggled(signal_data, EXCLUDED_FROM_LOG_APPLICATIONS_PROP, !state);
}

static void
handle_app_switch_property_changed(XfconfChannel *channel,
                                   const gchar *property_name,
                                   const GValue *value,
                                   KnownApplicationSignalData *signal_data)
{
    gboolean found = FALSE;
    GtkWidget *switch_to_set;
    GtkWidget *label_to_sensitize = NULL;
    gpointer callback_to_block;
    gboolean reverse_sense;

    if (g_strcmp0(property_name, MUTED_APPLICATIONS_PROP) == 0) {
        switch_to_set = signal_data->mute_switch;
        label_to_sensitize = signal_data->app_label;
        callback_to_block = xfce4_notifyd_mute_switch_activated;
        reverse_sense = FALSE;
    } else if (g_strcmp0(property_name, DENIED_CRITICAL_NOTIFICATIONS_PROP) == 0) {
        switch_to_set = signal_data->allow_criticals;
        callback_to_block = xfce4_notifyd_allow_criticals_switch_activated;
        reverse_sense = TRUE;
    } else if (g_strcmp0(property_name, EXCLUDED_FROM_LOG_APPLICATIONS_PROP) == 0) {
        switch_to_set = signal_data->include_in_log;
        callback_to_block = xfce4_notifyd_include_in_log_switch_activated;
        reverse_sense = TRUE;
    } else {
        return;
    }

    if (G_VALUE_HOLDS(value, G_TYPE_PTR_ARRAY)) {
        GPtrArray *strs = g_value_get_boxed(value);
        for (guint i = 0; i < strs->len; ++i) {
            const GValue *str = g_ptr_array_index(strs, i);
            if (G_VALUE_HOLDS_STRING(str)) {
                if (g_strcmp0(g_value_get_string(str), signal_data->known_application) == 0) {
                    found = TRUE;
                    break;
                }
            }
        }
    }

    g_signal_handlers_block_by_func(signal_data->mute_switch,
                                    callback_to_block,
                                    signal_data);
    gtk_switch_set_active(GTK_SWITCH(switch_to_set), reverse_sense ? !found : found);
    g_signal_handlers_unblock_by_func(signal_data->mute_switch,
                                      callback_to_block,
                                      signal_data);
    if (label_to_sensitize != NULL) {
        gtk_widget_set_sensitive(label_to_sensitize, reverse_sense ? found : !found);
    }
}

static void
listbox_remove_all (GtkWidget *widget, gpointer user_data)
{
    GtkWidget *container = user_data;
    gtk_container_remove (GTK_CONTAINER (container), widget);
}

static gint
xfce_notify_sort_apps_in_log (gconstpointer a, gconstpointer b)
{
    const LogAppCount *entry1 = *((LogAppCount **) a);
    const LogAppCount *entry2 = *((LogAppCount **) b);

    gint count_diff = entry2->count - entry1->count;
    if (count_diff != 0) {
        return count_diff;
    } else {
        return g_utf8_collate(entry1->app_name, entry2->app_name);
    }
}

static GPtrArray *
xfce_notify_count_apps_in_log (GKeyFile *notify_log,
                               GPtrArray *known_applications)
{
    GPtrArray *log_stats = g_ptr_array_new();

    if (known_applications != NULL) {
        GHashTable *counts = g_hash_table_new(g_str_hash, g_str_equal);
        GList *keys;

        for (guint i = 0; i < known_applications->len; ++i) {
            GValue *known_application = g_ptr_array_index(known_applications, i);
            if (G_VALUE_HOLDS_STRING(known_application)) {
                g_hash_table_insert(counts, (gchar *)g_value_get_string(known_application), GUINT_TO_POINTER(0));
            }
        }

        if (notify_log != NULL) {
            gsize num_groups = 0;
            gchar **groups = g_key_file_get_groups(notify_log, &num_groups);

            for (gsize i = 0; i < num_groups; ++i) {
                gchar *app_name = g_key_file_get_string(notify_log, groups[i], "app_name", NULL);

                if (g_hash_table_contains(counts, app_name)) {
                    g_hash_table_replace(counts, app_name, GUINT_TO_POINTER(GPOINTER_TO_UINT(g_hash_table_lookup(counts, app_name)) + 1));
                }
            }
        }

        keys = g_hash_table_get_keys(counts);
        for (GList *l = keys; l != NULL; l = l->next) {
            LogAppCount *entry = g_new0(LogAppCount, 1);
            const gchar *app_id = l->data;
            gchar *app_name;

            app_name = notify_get_from_desktop_file(app_id, G_KEY_FILE_DESKTOP_KEY_NAME);
            if (app_name == NULL) {
                /* Fallback: Just set the name provided by the application */
                app_name = g_strdup(app_id);
            }

            entry->app_id = app_id;
            entry->app_name = app_name;
            entry->count = GPOINTER_TO_UINT(g_hash_table_lookup(counts, app_id));
            g_ptr_array_add(log_stats, entry);
        }
        g_list_free(keys);

        g_hash_table_destroy(counts);
        g_ptr_array_sort(log_stats, xfce_notify_sort_apps_in_log);
    }

    return log_stats;
}

static void
known_application_signal_data_free(KnownApplicationSignalData *signal_data)
{
    g_signal_handlers_disconnect_by_func(signal_data->channel,
                                         handle_app_switch_property_changed,
                                         signal_data);

    g_object_unref(signal_data->channel);
    g_free(signal_data->known_application);
    g_free(signal_data->known_application_name);
    g_slice_free(KnownApplicationSignalData, signal_data);
}

static void
known_application_delete_clicked(GtkWidget *button,
                                 KnownApplicationSignalData *signal_data)
{
    gchar *confirm_text = g_strdup_printf( _("Are you sure you want to delete notification settings for application \"%s\"?"),
                                          signal_data->known_application_name);
    if (xfce_dialog_confirm(GTK_WINDOW(gtk_widget_get_toplevel(button)),
                            "edit-delete",
                            _("Delete"),
                            confirm_text,
                            _("Forget Application")))
    {
        gchar **known_applications = xfconf_channel_get_string_list(signal_data->channel, KNOWN_APPLICATIONS_PROP);

        if (G_LIKELY(known_applications != NULL)) {
            gchar **muted_applications = xfconf_channel_get_string_list(signal_data->channel, MUTED_APPLICATIONS_PROP);
            gchar **denied_critical_notifications = xfconf_channel_get_string_list(signal_data->channel, DENIED_CRITICAL_NOTIFICATIONS_PROP);

            xfce_remove_from_string_list_property(signal_data->channel, KNOWN_APPLICATIONS_PROP, known_applications, signal_data->known_application);
            xfce_remove_from_string_list_property(signal_data->channel, MUTED_APPLICATIONS_PROP, muted_applications, signal_data->known_application);
            xfce_remove_from_string_list_property(signal_data->channel, DENIED_CRITICAL_NOTIFICATIONS_PROP, denied_critical_notifications, signal_data->known_application);

            g_strfreev(known_applications);
            g_strfreev(muted_applications);
            g_strfreev(denied_critical_notifications);
        }
    }

    g_free(confirm_text);
}

static gboolean
test_gicon_exists(GIcon *gicon,
                  gint size,
                  gint scale)
{
    GtkIconInfo *icon_info = gtk_icon_theme_lookup_by_gicon_for_scale(gtk_icon_theme_get_default(),
                                                                      gicon,
                                                                      size,
                                                                      scale,
                                                                      GTK_ICON_LOOKUP_USE_BUILTIN);
    gboolean exists = icon_info != NULL;

    if (icon_info != NULL) {
        g_object_unref(icon_info);
    }

    return exists;
}

static void
xfce4_notifyd_known_application_insert_row (XfconfChannel *channel,
                                            GtkWidget *known_applications_listbox,
                                            GPtrArray *muted_applications,
                                            const gchar *const *denied_critical_applications,
                                            const gchar *const *excluded_from_log_applications,
                                            const gchar *known_application,
                                            const gchar *app_name,
                                            gint count)
{
    GtkBuilder *builder;
    GtkWidget *row;
    GtkWidget *expander;
    GtkWidget *app_label;
    GtkWidget *log_count_label;
    GtkWidget *icon;
    GtkWidget *mute_switch;
    GtkWidget *allow_criticals;
    GtkWidget *include_in_log;
    GtkWidget *delete_button;
    guint i;
    gint icon_width, icon_height, icon_size;
    gint scale_factor = gtk_widget_get_scale_factor(known_applications_listbox);
    gchar *desktop_icon_name = NULL;

    gtk_icon_size_lookup(GTK_ICON_SIZE_LARGE_TOOLBAR, &icon_width, &icon_height);
    icon_size = MIN(icon_width, icon_height);

    builder = gtk_builder_new_from_string(xfce4_notifyd_config_known_app_ui, xfce4_notifyd_config_known_app_ui_length);
    if (G_UNLIKELY(builder == NULL)) {
        g_critical("Unable to load known app UI description");
        return;
    }

    expander = GTK_WIDGET(gtk_builder_get_object(builder, "app_item"));
    icon = GTK_WIDGET(gtk_builder_get_object(builder, "app_icon"));
    app_label = GTK_WIDGET(gtk_builder_get_object(builder, "app_name"));
    log_count_label = GTK_WIDGET(gtk_builder_get_object(builder, "app_log_count"));
    mute_switch = GTK_WIDGET(gtk_builder_get_object(builder, "app_mute"));
    allow_criticals = GTK_WIDGET(gtk_builder_get_object(builder, "allow_critical_notifications"));
    include_in_log = GTK_WIDGET(gtk_builder_get_object(builder, "include_in_log"));
    delete_button = GTK_WIDGET(gtk_builder_get_object(builder, "delete_app"));

    row = gtk_list_box_row_new();
    gtk_list_box_insert(GTK_LIST_BOX(known_applications_listbox), row, -1);
    gtk_container_add(GTK_CONTAINER(row), expander);

    /* Number of notifications in the log (if enabled) */
    if (count > 0) {
        gtk_label_set_text (GTK_LABEL(log_count_label), g_strdup_printf("%d", count));
    }

    /* All applications that don't supply their name at all */
    if (xfce_str_is_empty (known_application)) {
        const char *format = "<span style=\"italic\">\%s</span>";
        char *markup;

        known_application = g_strdup(_("Unspecified applications"));
        markup = g_markup_printf_escaped (format, known_application);
        gtk_label_set_markup (GTK_LABEL (app_label), markup);
        g_free (markup);

        gtk_widget_set_sensitive(mute_switch, FALSE);
        gtk_widget_set_sensitive(allow_criticals, FALSE);
        gtk_widget_set_sensitive(include_in_log, FALSE);
        gtk_widget_set_sensitive(delete_button, FALSE);
    } else {
        KnownApplicationSignalData *signal_data = g_slice_new0(KnownApplicationSignalData);
        GIcon *gicon = NULL;

        signal_data->channel = g_object_ref(channel);
        signal_data->known_application = g_strdup(known_application);
        signal_data->app_label = app_label;
        signal_data->mute_switch = mute_switch;
        signal_data->allow_criticals = allow_criticals;
        signal_data->include_in_log = include_in_log;

        /* Try to find the correct icon based on the desktop file */
        desktop_icon_name = notify_get_from_desktop_file (known_application, G_KEY_FILE_DESKTOP_KEY_ICON);
        if (desktop_icon_name != NULL) {
            if (g_path_is_absolute(desktop_icon_name)
                && g_file_test(desktop_icon_name, G_FILE_TEST_EXISTS)
                && !g_file_test(desktop_icon_name, G_FILE_TEST_IS_DIR))
            {
                GFile *file = g_file_new_for_path(desktop_icon_name);
                gicon = g_file_icon_new(file);
                g_object_unref(file);
            } else {
                gicon = g_themed_icon_new_with_default_fallbacks(desktop_icon_name);
            }

            if (!test_gicon_exists(gicon, icon_size, scale_factor)) {
                g_clear_object(&gicon);
            }
        }

        /* Fallback: Try to naively load icon names */
        if (gicon == NULL) {
            /* Find icons in the right priority:
               1. normal icon name with fallback
               2. lowercase icon name with fallback */

            gicon = g_themed_icon_new_with_default_fallbacks(known_application);
            if (!test_gicon_exists(gicon, icon_size, scale_factor)) {
                g_clear_object(&gicon);
            }

            if (gicon == NULL) {
                gchar *icon_name_dashed = g_strdelimit(g_strdup(known_application), " .", '-');

                gicon = g_themed_icon_new_with_default_fallbacks(icon_name_dashed);
                if (!test_gicon_exists(gicon, icon_size, scale_factor)) {
                    gchar *icon_name_dashed_lower = g_ascii_strdown(icon_name_dashed, -1);

                    g_clear_object(&gicon);
                    gicon = g_themed_icon_new_with_default_fallbacks(icon_name_dashed_lower);
                    if (!test_gicon_exists(gicon, icon_size, scale_factor)) {
                        g_clear_object(&gicon);
                    }

                    g_free(icon_name_dashed_lower);
                }

                g_free(icon_name_dashed);
            }
        }

        if (G_LIKELY(gicon != NULL)) {
            gtk_image_set_pixel_size(GTK_IMAGE(icon), icon_size);
            gtk_image_set_from_gicon(GTK_IMAGE(icon), gicon, GTK_ICON_SIZE_LARGE_TOOLBAR);
            g_object_unref(gicon);
        }

        signal_data->known_application_name = g_strdup(app_name);
        gtk_label_set_text(GTK_LABEL(app_label), app_name);

        /* Set correct initial value as to whether an application is muted */
        if (muted_applications != NULL) {
            for (i = 0; i < muted_applications->len; i++) {
                GValue *muted_application;
                muted_application = g_ptr_array_index (muted_applications, i);
                if (g_str_match_string (g_value_get_string (muted_application), known_application, FALSE) == TRUE) {
                    gtk_widget_set_sensitive(app_label, FALSE);
                    gtk_switch_set_active (GTK_SWITCH (mute_switch), TRUE);
                    break;
                }
            }
        }
        g_signal_connect(mute_switch, "state-set",
                         G_CALLBACK(xfce4_notifyd_mute_switch_activated), signal_data);
        g_signal_connect(channel, "property-changed::" MUTED_APPLICATIONS_PROP,
                         G_CALLBACK(handle_app_switch_property_changed), signal_data);

        gtk_switch_set_active(GTK_SWITCH(allow_criticals),
                              denied_critical_applications == NULL
                              || !g_strv_contains(denied_critical_applications, known_application));
        g_signal_connect(allow_criticals, "state-set",
                         G_CALLBACK(xfce4_notifyd_allow_criticals_switch_activated), signal_data);
        g_signal_connect(channel, "property-changed::" DENIED_CRITICAL_NOTIFICATIONS_PROP,
                         G_CALLBACK(handle_app_switch_property_changed), signal_data);

        gtk_switch_set_active(GTK_SWITCH(include_in_log),
                              excluded_from_log_applications == NULL
                              || !g_strv_contains(excluded_from_log_applications, known_application));
        g_signal_connect(include_in_log, "state-set",
                         G_CALLBACK(xfce4_notifyd_include_in_log_switch_activated), signal_data);
        g_signal_connect(channel, "property-changed::" EXCLUDED_FROM_LOG_APPLICATIONS_PROP,
                         G_CALLBACK(handle_app_switch_property_changed), signal_data);

        g_signal_connect(delete_button, "clicked",
                         G_CALLBACK(known_application_delete_clicked), signal_data);

        g_object_set_data_full(G_OBJECT(row),
                               "xfce4-notifyd-known-application-signal-data", signal_data,
                               (GDestroyNotify)known_application_signal_data_free);
    }

    g_free(desktop_icon_name);
}

static void
xfce4_notifyd_known_applications_changed (XfconfChannel *channel,
                               const gchar *property,
                               const GValue *value,
                               gpointer user_data)
{
    GtkWidget *known_applications_listbox = user_data;
    GtkCallback func = listbox_remove_all;
    GPtrArray *known_applications;
    GPtrArray *known_applications_sorted;
    GPtrArray *muted_applications;
    gchar **denied_critical_applications;
    gchar **excluded_from_log_applications;
    GKeyFile *notify_log;
    guint i;

    known_applications = xfconf_channel_get_arrayv (channel, KNOWN_APPLICATIONS_PROP);
    muted_applications = xfconf_channel_get_arrayv (channel, MUTED_APPLICATIONS_PROP);
    denied_critical_applications = xfconf_channel_get_string_list(channel, DENIED_CRITICAL_NOTIFICATIONS_PROP);
    excluded_from_log_applications = xfconf_channel_get_string_list(channel, EXCLUDED_FROM_LOG_APPLICATIONS_PROP);

    /* TODO: Check the old list versus the new one and only add/remove rows
             as needed instead instead of cleaning up the whole widget */
    /* Clean up the list and re-fill it */
    gtk_container_foreach (GTK_CONTAINER (known_applications_listbox), func, known_applications_listbox);

    if (known_applications != NULL) {
        /* Sort the apps based on their appearance in the log */
        notify_log = xfce_notify_log_get();
        known_applications_sorted = xfce_notify_count_apps_in_log(notify_log, known_applications);
        g_key_file_unref (notify_log);

        for (i = 0; i < known_applications_sorted->len; i++) {
            LogAppCount *application = g_ptr_array_index(known_applications_sorted, i);
            xfce4_notifyd_known_application_insert_row(channel,
                                                       known_applications_listbox,
                                                       muted_applications,
                                                       (const gchar *const *)denied_critical_applications,
                                                       (const gchar *const *)excluded_from_log_applications,
                                                       application->app_id,
                                                       application->app_name,
                                                       application->count);
            g_free(application->app_name);
        }
        g_ptr_array_free(known_applications_sorted, TRUE);
    }

    xfconf_array_free (known_applications);
    xfconf_array_free (muted_applications);
    g_strfreev(denied_critical_applications);
    g_strfreev(excluded_from_log_applications);
    gtk_widget_show_all (known_applications_listbox);
}

static void
xfce4_notifyd_do_not_disturb_activated (GtkSwitch *do_not_disturb_switch,
                                        gboolean state,
                                        gpointer user_data)
{
    GtkWidget *do_not_disturb_info = user_data;

    gtk_revealer_set_reveal_child (GTK_REVEALER (do_not_disturb_info),
                                    gtk_switch_get_active (GTK_SWITCH (do_not_disturb_switch)));
    gtk_switch_set_active (GTK_SWITCH (do_not_disturb_switch), state);
}

static void
xfce4_notifyd_do_fadeout_activated (GtkSwitch *do_fadeout,
                                    gboolean state,
                                    gpointer user_data)
{
    NotificationSlideoutWidgets *slideout_widgets = user_data;

    /* The sliding out animation is only available along with fade-out */
    if (state == FALSE)
        gtk_switch_set_active (GTK_SWITCH (slideout_widgets->do_slideout), FALSE);
    gtk_widget_set_sensitive (slideout_widgets->do_slideout, state);
    gtk_widget_set_sensitive (slideout_widgets->do_slideout_label, state);
    gtk_switch_set_active (GTK_SWITCH (do_fadeout), state);
}

static void
xfce4_notifyd_log_activated (GtkSwitch *log_switch,
                             gboolean state,
                             gpointer user_data)
{
    NotificationLogWidgets *log_widgets = user_data;
    const char *format = _("<b>Currently only urgent notifications are shown.</b>\nNotification logging is \%s.");
    char *markup;

    gtk_switch_set_state (GTK_SWITCH (log_switch), state);
    gtk_widget_set_sensitive (GTK_WIDGET (log_widgets->log_level), state);
    gtk_widget_set_sensitive (GTK_WIDGET (log_widgets->log_level_apps), state);
    gtk_widget_set_sensitive (GTK_WIDGET (log_widgets->log_level_apps_label), state);
    markup = g_markup_printf_escaped (format, state ? _("enabled") : _("disabled"));
    gtk_label_set_markup (GTK_LABEL (log_widgets->infobar_label), markup);
    g_free (markup);
}

static void
xfce4_notifyd_log_open (GtkButton *button, gpointer user_data) {
    gchar *notify_log_path;

    notify_log_path = xfce_resource_lookup (XFCE_RESOURCE_CACHE, XFCE_NOTIFY_LOG_FILE);
    if (notify_log_path) {
        gchar *uri = g_strdup_printf ("file://%s", notify_log_path);
        if (!g_app_info_launch_default_for_uri (uri, NULL, NULL))
            g_warning ("Could not open the log file: %s", notify_log_path);
        g_free (uri);
        g_free (notify_log_path);
    }
}

static void
xfce4_notifyd_log_populate (NotificationLogWidgets *log_widgets)
{
    GtkWidget *const log_listbox = log_widgets->log_listbox;
    GKeyFile *notify_log;
    GtkCallback func = listbox_remove_all;
    gint i;
    GDateTime *today;
    gchar *timestamp;
    GtkWidget *limit_button;
    gsize num_groups = 0;
    GdkPixbuf *pixbuf = NULL;
    gchar *notify_log_icon_folder;
    gchar *notify_log_icon_path;
    gint scale_factor = gtk_widget_get_scale_factor (log_listbox);

    today = g_date_time_new_now_local ();
    timestamp = g_date_time_format (today, "%F");

    gtk_container_foreach (GTK_CONTAINER (log_listbox), func, log_listbox);
    notify_log = xfce_notify_log_get();
    notify_log_icon_folder = xfce_resource_save_location (XFCE_RESOURCE_CACHE,
                                                          XFCE_NOTIFY_ICON_PATH, TRUE);

    if (notify_log) {
        gchar **groups;
        gboolean yesterday = FALSE;
        int log_length;

        groups = g_key_file_get_groups (notify_log, &num_groups);
        log_length = GPOINTER_TO_UINT(num_groups) - LOG_DISPLAY_LIMIT;
        if (log_length < 0)
            log_length = 0;

        /* Notifications are only shown until LOG_DISPLAY_LIMIT is hit */
        for (i = log_length; groups && groups[i]; i += 1) {
            GtkWidget *grid;
            GtkWidget *summary, *body, *app_icon;
            const gchar *group = groups[i];
            const char *format = "<b>\%s</b>";
            const char *tooltip_format_simple = "<b>\%s</b> - \%s";
            char *markup;
            gchar *app_name;
            gchar *tooltip_timestamp = NULL;
            gchar *tmp;
            GDateTime *log_timestamp;
            GDateTime *local_timestamp = NULL;

            log_timestamp = g_date_time_new_from_iso8601 (group, NULL);
            if (log_timestamp != NULL)
            {
                local_timestamp = g_date_time_to_local (log_timestamp);
                g_date_time_unref (log_timestamp);
            }

            if (local_timestamp != NULL) {
                tooltip_timestamp = g_date_time_format (local_timestamp, "%c");
                g_date_time_unref (local_timestamp);
            }

            if (g_ascii_strncasecmp (timestamp, group, 10) == 0 && yesterday == FALSE) {
                GtkWidget *header;
                header = gtk_label_new (_("Yesterday and before"));
                gtk_widget_set_sensitive (header, FALSE);
                gtk_widget_set_margin_top (header, 3);
                gtk_widget_set_margin_bottom (header, 3);
                gtk_list_box_insert (GTK_LIST_BOX (log_listbox), header, 0);
                yesterday = TRUE;
            }

            app_name = g_key_file_get_string (notify_log, group, "app_name", NULL);
            tmp = g_key_file_get_string (notify_log, group, "summary", NULL);
            markup = g_markup_printf_escaped (format, tmp);
            g_free (tmp);
            summary = gtk_label_new (NULL);
            if (xfce_notify_is_markup_valid(markup)) {
                gtk_label_set_markup (GTK_LABEL (summary), markup);
            } else {
                gtk_label_set_text(GTK_LABEL(summary), markup);
            }
            gtk_label_set_xalign (GTK_LABEL (summary), 0);
            g_free (markup);
            tmp = g_key_file_get_string (notify_log, group, "body", NULL);
            body = gtk_label_new (NULL);
            g_object_ref (body);
            if (xfce_notify_is_markup_valid(tmp)) {
                gtk_label_set_markup (GTK_LABEL (body), tmp);
            } else {
                gtk_label_set_text(GTK_LABEL(body), tmp);
            }
            g_free (tmp);
            gtk_label_set_xalign (GTK_LABEL (body), 0);
            gtk_label_set_ellipsize (GTK_LABEL (body), PANGO_ELLIPSIZE_END);
            tmp = g_key_file_get_string (notify_log, group, "app_icon", NULL);
            notify_log_icon_path = g_strconcat (notify_log_icon_folder , tmp, ".png", NULL);
            if (g_file_test (notify_log_icon_path, G_FILE_TEST_EXISTS)) {
                pixbuf = gdk_pixbuf_new_from_file_at_scale (notify_log_icon_path,
                                                            24 * scale_factor, 24 * scale_factor,
                                                            TRUE, NULL);
                if (G_LIKELY (pixbuf != NULL)) {
                    cairo_surface_t *surface = gdk_cairo_surface_create_from_pixbuf (pixbuf, scale_factor, NULL);
                    app_icon = gtk_image_new_from_surface (surface);
                    g_object_unref (pixbuf);
                    cairo_surface_destroy (surface);
                } else {
                    app_icon = gtk_image_new ();
                }
            } else {
                app_icon = gtk_image_new_from_icon_name (tmp, GTK_ICON_SIZE_LARGE_TOOLBAR);
                gtk_image_set_pixel_size (GTK_IMAGE (app_icon), 24);
            }
            g_free (notify_log_icon_path);
            g_free (tmp);
            gtk_widget_set_margin_start (app_icon, 3);
            // TODO: actions and timeout are missing (timeout is only interesting for urgent messages) - do we need that?
            grid = gtk_grid_new ();
            gtk_grid_set_column_spacing (GTK_GRID (grid), 6);
            gtk_grid_attach (GTK_GRID (grid), GTK_WIDGET (app_icon), 0, 0 , 1, 2);

            /* Handle icon-only notifications */
            tmp = g_key_file_get_string (notify_log, group, "body", NULL);
            if (g_strcmp0 (tmp, "") == 0) {
                gtk_grid_attach (GTK_GRID (grid), GTK_WIDGET (summary), 1, 0, 1, 2);
                if (tooltip_timestamp != NULL) {
                    markup = g_strdup_printf (tooltip_format_simple, app_name, tooltip_timestamp);
                }
                else {
                    markup = g_strdup_printf (format, app_name);
                }
            }
            else {
                gtk_grid_attach (GTK_GRID (grid), GTK_WIDGET (summary), 1, 0, 1, 1);
                gtk_grid_attach (GTK_GRID (grid), GTK_WIDGET (body), 1, 1, 1, 1);
                if (tooltip_timestamp != NULL) {
                    markup = g_strdup_printf (tooltip_format_simple, app_name, tooltip_timestamp);
                }
                else {
                    markup = g_strdup_printf (format, app_name);
                }
            }
            g_free (tmp);
            g_free (app_name);
            g_free (tooltip_timestamp);
            g_object_unref (body);
            body = NULL;

            gtk_widget_set_tooltip_markup (grid, markup);
            g_free (markup);

            gtk_list_box_insert (GTK_LIST_BOX (log_listbox), grid, 0);
        }
        if (log_length > 0) {
            gchar *limit_label;
            limit_label = g_strdup_printf("Showing %d of %d notifications. Click to open full log.",
                                          LOG_DISPLAY_LIMIT, GPOINTER_TO_UINT(num_groups));
            limit_button = gtk_button_new_with_label (limit_label);
            g_free (limit_label);
            limit_label = NULL;
            gtk_widget_set_margin_bottom (limit_button, 3);
            gtk_widget_set_margin_top (limit_button, 3);
            gtk_button_set_relief (GTK_BUTTON (limit_button), GTK_RELIEF_NONE);
            g_signal_connect (G_OBJECT (limit_button), "clicked",
                              G_CALLBACK (xfce4_notifyd_log_open), log_listbox);
            gtk_list_box_insert (GTK_LIST_BOX (log_listbox), limit_button, LOG_DISPLAY_LIMIT + 1);
        }
        g_strfreev (groups);
        g_key_file_free (notify_log);
    }

    g_free (notify_log_icon_folder);
    g_date_time_unref (today);
    g_free (timestamp);
    if (notify_log)
        g_key_file_unref (notify_log);

    gtk_widget_show_all (log_listbox);

    /* Update "open file" and "clear log" buttons sensitivity */
    gtk_widget_set_sensitive (GTK_WIDGET (gtk_toolbar_get_nth_item (log_widgets->log_toolbar, 1)), notify_log != NULL);
    gtk_widget_set_sensitive (GTK_WIDGET (gtk_toolbar_get_nth_item (log_widgets->log_toolbar, 2)), num_groups > 0);
}

static void
xfce4_notifyd_log_refresh (GtkButton *button, gpointer user_data) {
    xfce4_notifyd_log_populate (user_data);
}

static void xfce_notify_log_clear_button_clicked (GtkButton *button, gpointer user_data) {
    GtkWidget *log_listbox = ((NotificationLogWidgets *) user_data)->log_listbox;
    GtkCallback func = listbox_remove_all;
    GtkWidget *dialog;
    gint result;

    dialog = xfce_notify_clear_log_dialog ();
    result = gtk_dialog_run (GTK_DIALOG (dialog));
    /* Clear the listbox widget too in case the log is cleared */
    if (result == GTK_RESPONSE_OK)
        gtk_container_foreach (GTK_CONTAINER (log_listbox), func, log_listbox);
    gtk_widget_destroy (dialog);
}

static void xfce4_notifyd_show_help(GtkButton *button,
                                    GtkWidget *dialog)
{
    xfce_dialog_show_help_with_version(GTK_WINDOW(dialog), "notifyd", "start", NULL, NULL);
}

static void xfce_notify_bus_name_appeared_cb (GDBusConnection *connection,
                                              const gchar *name,
                                              const gchar *name_owner,
                                              gpointer user_data)
{
    GtkWidget  *notifyd_running = user_data;

    gtk_revealer_set_reveal_child (GTK_REVEALER (notifyd_running), FALSE);
}

static void xfce_notify_bus_name_vanished_cb (GDBusConnection *connection,
                                              const gchar *name,
                                              gpointer user_data)
{
    GtkWidget *notifyd_running = user_data;

    gtk_revealer_set_reveal_child (GTK_REVEALER (notifyd_running), TRUE);
}

static GtkWidget *
placeholder_label_new (gchar *place_holder_text)
{
    GtkWidget *label;
    label = gtk_label_new ("");
    gtk_label_set_markup (GTK_LABEL (label), place_holder_text);
    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);
    gtk_widget_set_sensitive (label, FALSE);
    gtk_widget_set_margin_start (label, 24);
    gtk_widget_set_margin_end (label, 24);
    gtk_widget_set_margin_top (label, 24);
    gtk_widget_set_margin_bottom (label, 24);
    return label;
}

static GtkWidget *
xfce4_notifyd_config_setup_dialog(GtkBuilder *builder)
{
    XfconfChannel *channel;
    GtkWidget *dlg;
    GtkWidget *btn;
    GtkWidget *sbtn;
    GtkWidget *slider;
    GtkWidget *theme_combo;
    GtkWidget *position_combo;
    GtkWidget *help_button;
    GtkWidget *known_applications_scrolled_window;
    GtkWidget *known_applications_listbox;
    GtkWidget *placeholder_label;
    GtkWidget *do_not_disturb_switch;
    GtkWidget *do_not_disturb_info;
    GtkWidget *log_switch;
    GtkWidget *log_scrolled_window;
    GtkToolItem *log_clear_button;
    GtkToolItem *log_refresh_button;
    GtkToolItem *log_open_button;
    GtkWidget *icon;
    GtkWidget *primary_monitor;
    GtkWidget *mute_sounds;
    GtkWidget *do_fadeout;
    GtkWidget *show_text_with_gauge;
    GtkAdjustment *adj;
    GError *error = NULL;
    gchar *current_theme;
    NotificationLogWidgets log_widgets;
    NotificationSlideoutWidgets slideout_widgets;

    gtk_builder_connect_signals(builder, NULL);

    dlg = GTK_WIDGET(gtk_builder_get_object(builder, "notifyd_settings_dlg"));
    g_signal_connect(G_OBJECT(dlg), "response",
                     G_CALLBACK(xfce_notifyd_config_dialog_response), NULL);

    btn = GTK_WIDGET(gtk_builder_get_object(builder, "close_btn"));
    g_signal_connect_swapped(G_OBJECT(btn), "clicked",
                             G_CALLBACK(gtk_dialog_response), dlg);

    help_button = GTK_WIDGET(gtk_builder_get_object(builder, "help_btn"));
    g_signal_connect(G_OBJECT(help_button), "clicked",
                    G_CALLBACK(xfce4_notifyd_show_help), dlg);

    if(!xfconf_init(&error)) {
        xfce_message_dialog(NULL, _("Xfce Notify Daemon"),
                            "dialog-error",
                            _("Settings daemon is unavailable"),
                            error->message,
                            "application-exit", GTK_RESPONSE_ACCEPT,
                            NULL);
        exit(EXIT_FAILURE);
    }

    channel = xfconf_channel_new("xfce4-notifyd");

    /**************
        GENERAL   *
     **************/
    // Behavior
    do_not_disturb_switch = GTK_WIDGET (gtk_builder_get_object (builder, "do_not_disturb"));
    xfconf_g_property_bind (channel, "/do-not-disturb", G_TYPE_BOOLEAN,
                            G_OBJECT (do_not_disturb_switch), "active");
    /* Manually control the revealer for the infobar because of https://bugzilla.gnome.org/show_bug.cgi?id=710888 */
    do_not_disturb_info = GTK_WIDGET (gtk_builder_get_object (builder, "do_not_disturb_info"));
    gtk_revealer_set_reveal_child (GTK_REVEALER (do_not_disturb_info),
                                   gtk_switch_get_active (GTK_SWITCH (do_not_disturb_switch)));
    g_signal_connect (G_OBJECT (do_not_disturb_switch), "state-set",
                      G_CALLBACK (xfce4_notifyd_do_not_disturb_activated), do_not_disturb_info);

    primary_monitor = GTK_WIDGET(gtk_builder_get_object(builder, "primary_monitor"));
    xfconf_g_property_bind(channel, "/primary-monitor", G_TYPE_UINT,
                           G_OBJECT(primary_monitor), "active");
    if(gtk_combo_box_get_active(GTK_COMBO_BOX(primary_monitor)) == -1)
        gtk_combo_box_set_active(GTK_COMBO_BOX(primary_monitor), 0);

    mute_sounds = GTK_WIDGET(gtk_builder_get_object(builder, "mute_sounds"));
#ifdef ENABLE_SOUND
    xfconf_g_property_bind(channel, MUTE_SOUNDS_PROP, G_TYPE_BOOLEAN,
                           G_OBJECT(mute_sounds), "active");
#else
    gtk_widget_hide(GTK_WIDGET(gtk_builder_get_object(builder, "mute_sounds_label")));
    gtk_widget_hide(mute_sounds);
#endif

    // Appearance
    theme_combo = GTK_WIDGET(gtk_builder_get_object(builder, "theme_combo"));
    current_theme = xfconf_channel_get_string(channel, "/theme", "Default");
    xfce4_notifyd_config_setup_theme_combo(theme_combo, current_theme);
    g_free(current_theme);
    g_signal_connect(G_OBJECT(theme_combo), "changed",
                     G_CALLBACK(xfce4_notifyd_config_theme_combo_changed),
                     channel);
    g_signal_connect(G_OBJECT(channel), "property-changed::/theme",
                     G_CALLBACK(xfce4_notifyd_config_theme_changed),
                     theme_combo);

    position_combo = GTK_WIDGET(gtk_builder_get_object(builder, "position_combo"));
    xfconf_g_property_bind(channel, "/notify-location", G_TYPE_UINT,
                           G_OBJECT(position_combo), "active");
    if(gtk_combo_box_get_active(GTK_COMBO_BOX(position_combo)) == -1)
        gtk_combo_box_set_active(GTK_COMBO_BOX(position_combo), GTK_CORNER_TOP_RIGHT);

    slider = GTK_WIDGET(gtk_builder_get_object(builder, "opacity_slider"));
    g_signal_connect(G_OBJECT(slider), "format-value",
                     G_CALLBACK(xfce4_notifyd_slider_format_value), NULL);
    adj = gtk_range_get_adjustment(GTK_RANGE(slider));
    xfconf_g_property_bind(channel, "/initial-opacity", G_TYPE_DOUBLE,
                           G_OBJECT(adj), "value");

    sbtn = GTK_WIDGET(gtk_builder_get_object(builder, "expire_timeout_sbtn"));
    xfconf_g_property_bind(channel, "/expire-timeout", G_TYPE_INT,
                           G_OBJECT(sbtn), "value");

    do_fadeout = GTK_WIDGET(gtk_builder_get_object(builder, "do_fadeout"));
    gtk_switch_set_active (GTK_SWITCH (do_fadeout), TRUE);
    xfconf_g_property_bind(channel, "/do-fadeout", G_TYPE_BOOLEAN,
                           G_OBJECT(do_fadeout), "active");

    slideout_widgets.do_slideout_label = GTK_WIDGET(gtk_builder_get_object(builder, "do_slideout_label"));
    slideout_widgets.do_slideout = GTK_WIDGET(gtk_builder_get_object(builder, "do_slideout"));
    xfconf_g_property_bind(channel, "/do-slideout", G_TYPE_BOOLEAN,
                           G_OBJECT(slideout_widgets.do_slideout), "active");
    g_signal_connect (G_OBJECT (do_fadeout), "state-set",
                      G_CALLBACK (xfce4_notifyd_do_fadeout_activated), &slideout_widgets);
    if (gtk_switch_get_active (GTK_SWITCH (do_fadeout)) == FALSE) {
        gtk_widget_set_sensitive (slideout_widgets.do_slideout_label, FALSE);
        gtk_widget_set_sensitive (slideout_widgets.do_slideout, FALSE);
    }

    show_text_with_gauge = GTK_WIDGET(gtk_builder_get_object(builder, "show_text_with_gauge"));
    xfconf_g_property_bind(channel, "/show-text-with-gauge", G_TYPE_BOOLEAN,
                           G_OBJECT(show_text_with_gauge), "active");

    btn = GTK_WIDGET(gtk_builder_get_object(builder, "preview_button"));
    g_signal_connect(G_OBJECT(btn), "clicked",
                     G_CALLBACK(xfce_notifyd_config_preview_clicked), NULL);

    /*******************
        APPLICATIONS   *
     *******************/
    known_applications_scrolled_window = GTK_WIDGET (gtk_builder_get_object (builder, "known_applications_scrolled_window"));
    known_applications_listbox = gtk_list_box_new ();
    gtk_list_box_set_selection_mode(GTK_LIST_BOX(known_applications_listbox), GTK_SELECTION_NONE);
    gtk_container_add (GTK_CONTAINER (known_applications_scrolled_window), known_applications_listbox);
    gtk_list_box_set_header_func (GTK_LIST_BOX (known_applications_listbox), display_header_func, NULL, NULL);

    placeholder_label = placeholder_label_new (_("<big><b>Currently there are no known applications.</b></big>"
                                                 "\nAs soon as an application sends a notification"
                                                 "\nit will appear in this list."));
    /* Initialize the list of known applications */
    xfce4_notifyd_known_applications_changed (channel, KNOWN_APPLICATIONS_PROP, NULL, known_applications_listbox);
    gtk_list_box_set_placeholder (GTK_LIST_BOX (known_applications_listbox), placeholder_label);
    gtk_widget_show_all (placeholder_label);
    g_signal_connect (G_OBJECT (channel),
                      "property-changed::" KNOWN_APPLICATIONS_PROP,
                      G_CALLBACK (xfce4_notifyd_known_applications_changed), known_applications_listbox);

    /**********
        LOG   *
     **********/
    log_widgets.log_level = GTK_WIDGET (gtk_builder_get_object (builder, "log_level"));
    log_widgets.log_level_apps = GTK_WIDGET (gtk_builder_get_object (builder, "log_level_apps"));
    log_widgets.log_level_apps_label = GTK_WIDGET (gtk_builder_get_object (builder, "log_level_apps_label"));
    log_widgets.infobar_label = GTK_WIDGET (gtk_builder_get_object (builder, "infobar_label"));
    log_switch = GTK_WIDGET (gtk_builder_get_object (builder, "log_switch"));
    xfconf_g_property_bind (channel, "/notification-log", G_TYPE_BOOLEAN,
                            G_OBJECT (log_switch), "active");
    g_signal_connect (G_OBJECT (log_switch), "state-set",
                      G_CALLBACK (xfce4_notifyd_log_activated), &log_widgets);
    xfconf_g_property_bind(channel, "/log-level", G_TYPE_UINT,
                           G_OBJECT(log_widgets.log_level), "active");
    xfconf_g_property_bind(channel, "/log-level-apps", G_TYPE_UINT,
                          G_OBJECT(log_widgets.log_level_apps), "active");
    sbtn = GTK_WIDGET (gtk_builder_get_object (builder, "log_max_size"));
    xfconf_g_property_bind(channel, "/log-max-size", G_TYPE_UINT,
                           G_OBJECT(sbtn), "value");

    /* Initialize the settings' states correctly */
    if(gtk_combo_box_get_active(GTK_COMBO_BOX(log_widgets.log_level)) == -1)
        gtk_combo_box_set_active(GTK_COMBO_BOX(log_widgets.log_level), 0);
    if(gtk_combo_box_get_active(GTK_COMBO_BOX(log_widgets.log_level_apps)) == -1)
        gtk_combo_box_set_active(GTK_COMBO_BOX(log_widgets.log_level_apps), 0);
    xfce4_notifyd_log_activated (GTK_SWITCH (log_switch), gtk_switch_get_active (GTK_SWITCH(log_switch)), &log_widgets);

    log_scrolled_window = GTK_WIDGET (gtk_builder_get_object (builder, "log_scrolled_window"));
    log_widgets.log_listbox = gtk_list_box_new ();
    gtk_container_add (GTK_CONTAINER (log_scrolled_window), log_widgets.log_listbox);
    gtk_list_box_set_header_func (GTK_LIST_BOX (log_widgets.log_listbox), display_header_func, NULL, NULL);
    placeholder_label = placeholder_label_new (_("<big><b>Empty log</b></big>"
                                                 "\nNo notifications have been logged yet."));
    gtk_list_box_set_placeholder (GTK_LIST_BOX (log_widgets.log_listbox), placeholder_label);
    gtk_widget_show_all (placeholder_label);

    log_widgets.log_toolbar = GTK_TOOLBAR (gtk_builder_get_object (builder, "log_toolbar"));
    icon = gtk_image_new_from_icon_name ("view-refresh-symbolic", GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_image_set_pixel_size (GTK_IMAGE (icon), 16);
    log_refresh_button = gtk_tool_button_new (icon, _("Refresh"));
    gtk_widget_set_tooltip_text (GTK_WIDGET (log_refresh_button), _("Refresh the notification log"));
    gtk_toolbar_insert(log_widgets.log_toolbar, GTK_TOOL_ITEM(log_refresh_button), 0);
    g_signal_connect (G_OBJECT (log_refresh_button), "clicked",
                      G_CALLBACK (xfce4_notifyd_log_refresh), &log_widgets);
    icon = gtk_image_new_from_icon_name ("document-open-symbolic", GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_image_set_pixel_size (GTK_IMAGE (icon), 16);
    log_open_button = gtk_tool_button_new (icon, _("Open"));
    gtk_widget_set_tooltip_text (GTK_WIDGET (log_open_button), _("Open the notification log in an external editor"));
    gtk_toolbar_insert(log_widgets.log_toolbar, GTK_TOOL_ITEM(log_open_button), 1);
    g_signal_connect (G_OBJECT (log_open_button), "clicked",
                      G_CALLBACK (xfce4_notifyd_log_open), log_widgets.log_listbox);
    icon = gtk_image_new_from_icon_name ("edit-clear-symbolic", GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_image_set_pixel_size (GTK_IMAGE (icon), 16);
    log_clear_button = gtk_tool_button_new (icon, _("Clear"));
    gtk_widget_set_tooltip_text (GTK_WIDGET (log_clear_button), _("Clear the notification log"));
    gtk_toolbar_insert(log_widgets.log_toolbar, GTK_TOOL_ITEM(log_clear_button), 2);
    g_signal_connect (G_OBJECT (log_clear_button), "clicked",
                      G_CALLBACK (xfce_notify_log_clear_button_clicked), &log_widgets);
    gtk_widget_show_all (GTK_WIDGET(log_widgets.log_toolbar));

    xfce4_notifyd_log_populate (&log_widgets);

    return dlg;
}

int
main(int argc,
     char **argv)
{
    GtkWidget *settings_dialog = NULL;
    GtkWidget *notifyd_running;
    GtkBuilder *builder;
    gboolean opt_version = FALSE;
    gint32 opt_socket_id = 0;
    guint watch_handle_id;
    GOptionEntry option_entries[] = {
        { "version", 'V', G_OPTION_FLAG_IN_MAIN, G_OPTION_ARG_NONE, &opt_version, N_("Display version information"), NULL },
        { "socket-id", 's', G_OPTION_FLAG_IN_MAIN, G_OPTION_ARG_INT, &opt_socket_id, N_("Settings manager socket"), N_("SOCKET_ID") },
        { NULL, },
    };
    GError *error = NULL;

    xfce_textdomain(GETTEXT_PACKAGE, LOCALEDIR, "UTF-8");

    if(!gtk_init_with_args(&argc, &argv, NULL, option_entries, PACKAGE, &error)) {
        if(G_LIKELY(error)) {
            g_printerr("%s: %s.\n", G_LOG_DOMAIN, error->message);
            g_printerr(_("Type '%s --help' for usage."), G_LOG_DOMAIN);
            g_printerr("\n");
            g_error_free(error);
        } else
            g_error("Unable to open display.");

        return EXIT_FAILURE;
    }

    if(G_UNLIKELY(opt_version)) {
        g_print("%s %s\n", G_LOG_DOMAIN, VERSION);
        g_print("Copyright (c) 2010 Brian Tarricone <bjt23@cornell.edu>\n");
        g_print("Copyright (c) 2010 Jérôme Guelfucci <jeromeg@xfce.org>\n");
        g_print("Copyright (c) 2016 Ali Abdallah <ali@xfce.org>\n");
        g_print("Copyright (c) 2016 Simon Steinbeiß <simon@xfce.org>\n");
        g_print(_("Released under the terms of the GNU General Public License, version 2\n"));
        g_print(_("Please report bugs to %s.\n"), PACKAGE_BUGREPORT);

        return EXIT_SUCCESS;
    }

    if (!notify_init ("Xfce4-notifyd settings")) {
      g_error ("Failed to initialize libnotify.");
      return EXIT_FAILURE;
    }

    builder = gtk_builder_new();
    gtk_builder_add_from_string(builder, xfce4_notifyd_config_ui, xfce4_notifyd_config_ui_length, NULL);
    if(G_UNLIKELY(!builder)) {
        g_error("Unable to read embedded UI definition file");
        return EXIT_FAILURE;
    }

    settings_dialog = xfce4_notifyd_config_setup_dialog(builder);

    notifyd_running = GTK_WIDGET (gtk_builder_get_object (builder, "notifyd_running"));
    gtk_revealer_set_reveal_child (GTK_REVEALER (notifyd_running), FALSE);
    watch_handle_id = g_bus_watch_name (G_BUS_TYPE_SESSION,
                                        "org.freedesktop.Notifications",
                                        G_BUS_NAME_WATCHER_FLAGS_NONE,
                                        xfce_notify_bus_name_appeared_cb,
                                        xfce_notify_bus_name_vanished_cb,
                                        notifyd_running,
                                        NULL);

    if(opt_socket_id) {
        GtkWidget *plug, *plug_child;

        plug = gtk_plug_new(opt_socket_id);
        gtk_widget_show(plug);
        g_signal_connect(G_OBJECT(plug), "delete-event",
                         G_CALLBACK(gtk_main_quit), NULL);

        plug_child = GTK_WIDGET(gtk_builder_get_object(builder, "plug-child"));

        /* In the glade file, plug_child has parent, so remove it first */
        gtk_container_remove (GTK_CONTAINER(gtk_widget_get_parent(plug_child)), plug_child);
        gtk_container_add (GTK_CONTAINER(plug), plug_child);

        gdk_notify_startup_complete();
        g_object_unref(G_OBJECT(builder));
        gtk_widget_destroy(settings_dialog);

        gtk_main();
    } else {
        g_object_unref(G_OBJECT(builder));

        gtk_dialog_run(GTK_DIALOG(settings_dialog));
    }

    g_bus_unwatch_name (watch_handle_id);
    notify_uninit();
    xfconf_shutdown();

    return EXIT_SUCCESS;
}
