#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <inttypes.h>
#include <sys/time.h>

#include "xenner.h"

/* -------------------------------------------------------------- */

#define MILLISECS(_ms)  ((uint64_t)((_ms) * 1000000ULL))

static inline uint32_t div_frac(uint32_t dividend, uint32_t divisor)
{
    uint32_t quotient, remainder;

    __asm__ ( 
        "divl %4"
        : "=a" (quotient), "=d" (remainder)
        : "0" (0), "1" (dividend), "r" (divisor) );

#if 0
    fprintf(stderr, "%s: %u<<32 / %u -> %u +%u\n", __FUNCTION__,
	    dividend, divisor, quotient, remainder);
#endif
    return quotient;
}   

static void set_time_scale(uint64_t ticks_per_sec, int *shift,
			   uint32_t *mul_frac)
{
    uint64_t tps64 = ticks_per_sec;
    uint32_t tps32;

    *shift = 0;

    while ( tps64 > (MILLISECS(1000)*2) )
    {
        tps64 >>= 1;
        (*shift)--;
    }

    tps32 = (uint32_t)tps64;
    while ( tps32 <= (uint32_t)MILLISECS(1000) )
    {
        tps32 <<= 1;
        (*shift)++;
    }

    *mul_frac = div_frac(MILLISECS(1000), tps32);
}

/* -------------------------------------------------------------- */

#define TSC_USECS 100000  /* 0.1 sec */

#define rdtscll(val) do {  \
     unsigned int __a,__d; \
     asm volatile("rdtsc" : "=a" (__a), "=d" (__d)); \
     (val) = ((uint64_t)__a) | (((uint64_t)__d)<<32); \
} while(0)

void tsc_calibrate(struct xenvm *xen, int *shift, uint32_t *mul_frac)
{
    uint64_t tsc_start, tsc_end, tsc_tics;
    struct timeval tv_start, tv_end;
    int64_t usecs;

    logprintf(xen, "calibrate tsc ... ");
    gettimeofday(&tv_start, NULL);
    rdtscll(tsc_start);

    do {
	gettimeofday(&tv_end, NULL);
	rdtscll(tsc_end);
	usecs  = tv_end.tv_usec - tv_start.tv_usec;
	usecs += (tv_end.tv_sec - tv_start.tv_sec) * 1000000;
    } while (usecs < TSC_USECS);

    tsc_tics = (tsc_end - tsc_start) * 1000000 / usecs;
    set_time_scale(tsc_tics, shift, mul_frac);
    logprintf(xen, "done, %d MHz, shift: %d, mul_frac: %u\n",
	      (int)(tsc_tics / 1000000), *shift, *mul_frac);
}

#ifdef STANDALONE
int main(int argc, char *argv[])
{
    uint32_t mul_frac;
    int shift;
    
    tsc_calibrate(&shift, &mul_frac);
    return 0;
}
#endif
