#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <dlfcn.h>
#include <stdarg.h>
#include <assert.h>
#include <dirent.h>
#include <inttypes.h>
#include <fcntl.h>

#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/types.h>

#include <xenctrl.h>

#include "xennerctrl.h"
#include "shared.h"

/* --------------------------------------------------------------------- */

struct xennermap {
    struct xennerdom        *dom;
    struct grant_entry      *grant;
    void                    *addr;
    void                    *munmap;

    struct list_head        next;
};
struct gntpriv {
    int                     handle;
    struct list_head        maps;
    int                     mapcount;
    pthread_mutex_t         maplock;

    struct list_head        list;
};
static LIST_HEAD(privs);
static pthread_mutex_t privlock;

/* --------------------------------------------------------------------- */

static void *map_ref_dom(struct gntpriv *p,
			 int domid, void *addr, uint32_t ref, int prot)
{
    struct xennerdom *dom;
    struct xennermap *map;
    struct grant_entry *e;
    uint64_t offset;

    dom = xenner_get_dom(domid);
    
    if (!dom->parsed)
	return NULL;
    if (ref >= dom->grant_entries)
	return NULL;

    e = dom->grant_table + ref;
    if ((e->flags & GTF_type_mask) != GTF_permit_access)
	goto err;
    offset = VMCORE_HEADER + e->frame * PAGE_SIZE;
    if (offset >= dom->vmcore_size)
	goto err;

    map = malloc(sizeof(*map));
    if (NULL == map)
	goto err;
    memset(map,0,sizeof(*map));
    map->dom = dom;
    map->grant = e;
    if (addr) {
	map->addr = mmap(addr, PAGE_SIZE, prot,
			 MAP_SHARED | MAP_FIXED,
			 dom->fd, offset);
	if ((void*)-1 == map->addr)
	    goto err2;
	map->munmap = map->addr;
    } else {
	map->addr = dom->vmcore + offset;
    }
    list_add_tail(&map->next, &p->maps);
    p->mapcount++;

    if (libxc_trace)
	fprintf(stderr, "%s: flags 0x%x, domid %d, frame 0x%x -> addr %p %s%s [cnt %d]\n",
		__FUNCTION__,
		(int)dom->grant_table[ref].flags,
		(int)dom->grant_table[ref].domid,
		(int)dom->grant_table[ref].frame,
		map->addr,
		(prot & PROT_READ)  ? "r" : "-",
		(prot & PROT_WRITE) ? "w" : "-",
		p->mapcount);
    return map->addr;

err2:
    free(map);
err:
    fprintf(stderr, "%s: flags 0x%x, domid %d, frame 0x%x -> FAILED\n", __FUNCTION__,
	    (int)dom->grant_table[ref].flags,
	    (int)dom->grant_table[ref].domid,
	    (int)dom->grant_table[ref].frame);
    return NULL;
}

static void unmap_ref(struct gntpriv *p, void *addr)
{
    struct xennermap *map;
    struct list_head *item;

    list_for_each(item, &p->maps) {
	map = list_entry(item, struct xennermap, next);
	if (map->addr != addr)
	    continue;
	if (map->munmap)
	    munmap(map->munmap, PAGE_SIZE);
	xenner_put_dom(map->dom);
	list_del(&map->next);
	p->mapcount--;
	free(map);

	if (libxc_trace)
	    fprintf(stderr, "%s: %p [cnt %d]\n", __FUNCTION__, addr, p->mapcount);
	return;
    }
    fprintf(stderr, "%s: not found: %p\n", __FUNCTION__, addr);
}

/* --------------------------------------------------------------------- */

static struct gntpriv *getpriv(int handle)
{
    struct list_head *item;
    struct gntpriv *p;

    pthread_mutex_lock(&privlock);
    list_for_each(item, &privs) {
	p = list_entry(item, struct gntpriv, list);
	if (p->handle == handle) {
	    pthread_mutex_unlock(&privlock);
	    return p;
	}
    }
    pthread_mutex_unlock(&privlock);
    return NULL;
}

int xc_gnttab_open(void)
{
    static int next_handle = 105;
    struct gntpriv *p;

    p = malloc(sizeof(*p));
    if (NULL == p)
	goto err;
    memset(p,0,sizeof(*p));
    p->handle = next_handle++;
    INIT_LIST_HEAD(&p->maps);

    pthread_mutex_lock(&privlock);
    list_add_tail(&p->list, &privs);
    pthread_mutex_unlock(&privlock);
    return p->handle;

err:
    return -1;
}

int xc_gnttab_close(int xcg_handle)
{
    struct gntpriv *p;

    p = getpriv(xcg_handle);
    pthread_mutex_lock(&privlock);
    list_del(&p->list);
    pthread_mutex_unlock(&privlock);
    if (!list_empty(&p->maps)) {
	fprintf(stderr, "%s: WARNING: %d leftover mappings\n",
		__FUNCTION__, p->mapcount);
    }
    free(p);
    return 0;
}

void *xc_gnttab_map_grant_ref(int xcg_handle, uint32_t domid,
                              uint32_t ref, int prot)
{
    struct gntpriv *p = getpriv(xcg_handle);;
    void *ptr;
    
    pthread_mutex_lock(&p->maplock);
    p = getpriv(xcg_handle);
    ptr = map_ref_dom(p, domid, NULL, ref, prot);
    pthread_mutex_unlock(&p->maplock);
    return ptr;
}

void *xc_gnttab_map_grant_refs(int xcg_handle, uint32_t count,
                               uint32_t *domids, uint32_t *refs, int prot)
{
    struct gntpriv *p = getpriv(xcg_handle);;
    void *ptr;
    int i;

    /* get address space by creating a anonymous mapping ... */
    ptr = mmap(NULL, count * PAGE_SIZE, prot, MAP_ANONYMOUS | MAP_SHARED, -1, 0);
    if ((void*)-1 == ptr)
	return NULL;

    /* ... then replace it page by page with the content we want */
    pthread_mutex_lock(&p->maplock);
    p = getpriv(xcg_handle);
    for (i = 0; i < count; i++) {
	if (NULL == map_ref_dom(p, domids[i], ptr + i * PAGE_SIZE, refs[i], prot))
	    goto err;
    }
    pthread_mutex_unlock(&p->maplock);
    return ptr;

err:
    while (i > 0) {
	i--;
	unmap_ref(p, ptr + i * PAGE_SIZE);
    }
    pthread_mutex_unlock(&p->maplock);
    munmap(ptr, count * PAGE_SIZE);
    return NULL;
}

int xc_gnttab_munmap(int xcg_handle, void *start_address, uint32_t count)
{
    struct gntpriv *p = getpriv(xcg_handle);;
    void *addr = start_address;
    int i = count;

    pthread_mutex_lock(&p->maplock);
    p = getpriv(xcg_handle);
    while (i) {
	unmap_ref(p, addr);
	addr += PAGE_SIZE;
	i--;
    }
    pthread_mutex_unlock(&p->maplock);
    return 0;
}
