 /*
 *  file    : poche_gen.c
 *  project   : xcfa
 *  with      : Gtk-2
 *
 *  copyright : (C) 2003 - 2011 by Claude Bulin
 *
 *  xcfa - GTK+ implementation of the GNU shell command
 *  GNU General Public License
 *
 *  This file is part of XCFA.
 * 
 *  XCFA is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  at your option) any later version.
 * 
 *  XCFA is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with XCFA.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#ifdef ENABLE_NLS
	#include <libintl.h>
	#define _(String) gettext (String)
#endif

#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <glib/gstdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/types.h>

#ifdef ENABLE_STATIC_LIBRARY
	#include "../lib/lib.h"
#endif

#include "global.h"
#include "configuser.h"
#include "win_info.h"
#include "prg_init.h"
#include "conv.h"
#include "poche.h"


/*
*---------------------------------------------------------------------------
* EXTERN
*---------------------------------------------------------------------------
*/

extern int kill (pid_t pid, int sig);




/*
*---------------------------------------------------------------------------
* VARIABLES
*---------------------------------------------------------------------------
*/

typedef struct {				/* LES CHAMPS DE LA POCHETTE	*/
	gchar *author;				/* Auteur			*/
	gchar *title;				/* Titre de la chanson		*/
	gchar *time;				/* Duree			*/
} FIELDS_PS;

typedef struct {
	gchar    *str_package;			/* PARAM_PACKAGE 		*/
	gchar    *str_version;			/* PARAM_VERSION		*/
	gchar    *str_title;			/* PARAM_TITLE			*/
	gboolean  bool_title_cover;		/* PARAM_BOOL_TITLE_COVER	*/
	gboolean  bool_title_back;		/* PARAM_BOOL_TITLE_BACK	*/
	gchar    *str_foottext;			/* PARAM_FOOTTEXT		*/
	gboolean  bool_foottext_cover;		/* PARAM_BOOL_FOOTTEXT_COVER	*/
	gboolean  bool_foottext_back;		/* PARAM_BOOL_FOOTTEXT_BACK	*/
	gboolean  bool_textdata_cover;		/* PARAM_BOOL_TEXTDATA_COVER	*/
	gboolean  bool_textdata_back;		/* PARAM_BOOL_TEXTDATA_BACK	*/
	gboolean  bool_line_cover;		/* PARAM_BOOL_LINE_COVER	*/
	gboolean  bool_line_back;		/* PARAM_BOOL_LINE_BACK		*/
	GList    *glist_fields;			/* PARAM_TEXTDATA		*/
	gint      num_choicefieldscover;	/* PARAM_CHOICEFIELDSCOVER	*/
	gint      num_choicefieldsback;		/* PARAM_CHOICEFIELDSBACK	*/
} PS;


// 
// 
PS *pochegen_remove_struct_ps (PS *p_ps)
{
	GList      *List = NULL;
	FIELDS_PS  *fps = NULL;

	// PRINT_FUNC_LF();

	if (NULL != p_ps) {
		g_free (p_ps->str_package);
		p_ps->str_package = NULL;		/* PARAM_PACKAGE 		*/
		g_free (p_ps->str_version);
		p_ps->str_version = NULL;		/* PARAM_VERSION		*/
		g_free (p_ps->str_title);
		p_ps->str_title = NULL;			/* PARAM_TITLE			*/
		g_free (p_ps->str_foottext);
		p_ps->str_foottext = NULL;		/* PARAM_FOOTTEXT		*/

		List = g_list_first (p_ps->glist_fields);
		while (List) {
			if (NULL != (fps = (FIELDS_PS *)List->data)) {
				g_free (fps->author);	fps->author = NULL;
				g_free (fps->title);	fps->title = NULL;
				g_free (fps->time);	fps->time = NULL;
				g_free (fps);
				fps = List->data = NULL;
			}
			List = g_list_next (List);
		}
		g_list_free (p_ps->glist_fields);
		p_ps->glist_fields = NULL;		/* PARAM_TEXTDATA		*/

		g_free (p_ps);
		p_ps = NULL;
	}
	return (NULL);
}
// 
// 
PS *pochegen_allocate_struct_ps (void)
{
	PS *new = (PS *)g_malloc0 (sizeof (PS));

	// PRINT_FUNC_LF();

	if (new) {
		new->str_package = NULL;		/* PARAM_PACKAGE 		*/
		new->str_version = NULL;		/* PARAM_VERSION		*/
		new->str_title = NULL;			/* PARAM_TITLE			*/
		new->bool_title_cover = FALSE;		/* PARAM_BOOL_TITLE_COVER	*/
		new->bool_title_back = FALSE;		/* PARAM_BOOL_TITLE_BACK	*/
		new->str_foottext = NULL;		/* PARAM_FOOTTEXT		*/
		new->bool_foottext_cover = FALSE;	/* PARAM_BOOL_FOOTTEXT_COVER	*/
		new->bool_foottext_back = FALSE;	/* PARAM_BOOL_FOOTTEXT_BACK	*/
		new->bool_textdata_cover = FALSE;	/* PARAM_BOOL_TEXTDATA_COVER	*/
		new->bool_textdata_back = FALSE;	/* PARAM_BOOL_TEXTDATA_BACK	*/
		new->bool_line_cover = FALSE;		/* PARAM_BOOL_LINE_COVER	*/
		new->bool_line_back = FALSE;		/* PARAM_BOOL_LINE_BACK		*/
		new->glist_fields = NULL;		/* PARAM_TEXTDATA		*/
		new->num_choicefieldscover = 0;		/* PARAM_CHOICEFIELDSCOVER	*/
		new->num_choicefieldsback = 0;		/* PARAM_CHOICEFIELDSBACK	*/
	}
	return ((PS *)new);
}
// 
// Lecture du buffer textview ligne par ligne pour le stocker dans des
// structures 'TEXTDATA' qui seront chainees dans un GList
// 
GList *pochegen_get_glist_from_textview (void)
{
	GList         *List = NULL;
	GList         *NList = NULL;
	GtkTextBuffer *Buf = NULL;
	GtkTextIter    Start;
	GtkTextIter    End;
	gint           Total_Line = 0;
	gint           Cpt;
	gchar         *Ptr_buf = NULL;
	gchar         *Ptr = NULL;
	gchar        **Larrbuf = NULL;
	gint           i;
	gint           separator = 0;
	FIELDS_PS     *ps = NULL;
	gchar         *PtrDummy = NULL;
	

	// PRINT_FUNC_LF();

	/* Suppression des lignes vides en fin de texte */
	for (Cpt = Total_Line; Cpt > 1; Cpt --) {
		Buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (poche.Adr_textview));
		gtk_text_buffer_get_iter_at_line_offset (Buf, &Start, Cpt, 0);
		gtk_text_buffer_get_end_iter (Buf, &End);
		
		PtrDummy = gtk_text_buffer_get_text (Buf, &Start, &End, FALSE);
		// Ptr_buf = g_locale_from_utf8 (PtrDummy, -1, NULL, NULL, NULL);
		Ptr_buf = g_strdup(PtrDummy);
		g_free (PtrDummy);
		PtrDummy = NULL;
		// Ptr_buf = g_locale_from_utf8 (gtk_text_buffer_get_text (Buf, &Start, &End, FALSE), -1, NULL, NULL, NULL);

		if (*Ptr_buf == '\n' || *Ptr_buf == '\0')
			gtk_text_buffer_delete (Buf, &Start, &End);
		else	break;
	}
	g_free (Ptr_buf);
	Ptr_buf = NULL;
	/* Get start-end buffer text with coord. in utf8 */
	Buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (poche.Adr_textview));
	gtk_text_buffer_get_bounds (Buf, &Start, &End);
	
	PtrDummy = gtk_text_buffer_get_text (Buf, &Start, &End, FALSE);
	// Ptr_buf = g_locale_from_utf8 (PtrDummy, -1, NULL, NULL, NULL);
	Ptr_buf = g_strdup(PtrDummy);
	g_free (PtrDummy);
	PtrDummy = NULL;
	// Ptr_buf = g_locale_from_utf8 (gtk_text_buffer_get_text (Buf, &Start, &End, FALSE), -1, NULL, NULL, NULL);
	/* Decompose le buffer en lignes de texte */
	Larrbuf = g_strsplit(Ptr_buf, "\n", 0);
	g_free (Ptr_buf);
	Ptr_buf = NULL;

	for (i=0; Larrbuf[i]; i++) {
		/* delette end cars line */
		if ((Ptr = strrchr (Larrbuf[i], 13))) *Ptr = '\0';
		if ((Ptr = strrchr (Larrbuf[i], 10))) *Ptr = '\0';
		/* total car '-' in Larrbuf[i] pour 'separator' */
		for (separator = 0, Ptr = Larrbuf[i]; ; ) {
			/* Ptr = strchr (Ptr, '-'); */
			Ptr = strstr (Ptr, " - ");
			if (NULL == Ptr) break;
			Ptr++;
			Ptr++;
			Ptr++;
			separator ++;
		}

		ps = (FIELDS_PS *)g_malloc0 (sizeof (FIELDS_PS));

		/* DEUX CHAMPS */
		if (separator == 1) {
			/* champs UN pour: ps->author */
			ps->author = g_strdup (Larrbuf[i]);
			/* Ptr = strchr (ps->author, '-'); */
			Ptr = strstr (ps->author, " - ");
			Ptr--;
			while (Ptr > ps->author && (*Ptr == ' ' || *Ptr == '\t')) Ptr--;
			Ptr++;
			*Ptr = '\0';
			/* champs DEUX pour: ps->title */
			/* Ptr = strchr (Larrbuf[i], '-'); */
			Ptr = strstr (Larrbuf[i], " - ");
			Ptr ++;
			Ptr ++;
			Ptr ++;
			while (*Ptr && (*Ptr == ' ' || *Ptr == '\t')) Ptr++;
			ps->title = g_strdup (Ptr);
			/* champs TROIS FICTIF pour: ps->time */
			ps->time = g_strdup ("00:00");
		}
		/* TROIS CHAMPS */
		else if (separator == 2) {
			/* champs UN pour: td->author */
			ps->author = g_strdup (Larrbuf[i]);
			/* Ptr = strchr (ps->author, '-'); */
			Ptr = strstr (ps->author, " - ");
			Ptr--;
			while (Ptr > ps->author && (*Ptr == ' ' || *Ptr == '\t')) Ptr--;
			Ptr++;
			*Ptr = '\0';
			/* champs DEUX pour: ps->title */
			/* Ptr = strchr (Larrbuf[i], '-'); */
			Ptr = strstr (Larrbuf[i], " - ");
			Ptr ++;
			Ptr ++;
			Ptr ++;
			while (*Ptr && (*Ptr == ' ' || *Ptr == '\t')) Ptr++;
			ps->title = g_strdup (Ptr);
			/* Ptr = strchr (ps->title, '-'); */
			Ptr = strstr (ps->title, " - ");
			Ptr--;
			while (Ptr > ps->title && (*Ptr == ' ' || *Ptr == '\t')) Ptr--;
			Ptr++;
			*Ptr = '\0';
			/* champs TROIS pour: ps->time */
			Ptr = strrchr (Larrbuf[i], '-');
			if (Ptr > Larrbuf[i] && *(Ptr -1) == ' ' && *(Ptr -1) == ' ')
			Ptr ++;
			while (*Ptr && (*Ptr == ' ' || *Ptr == '\t')) Ptr++;
			ps->time = g_strdup (Ptr);
		}
		/* ON NE CONSIDERE QU'UN SEUL CHAMPS */
		else {
			/* champs UN pour: ps->author */
			ps->author = g_strdup (Larrbuf[i]);
			ps->title = g_strdup ("");
			ps->time = g_strdup ("00:00");
		}

		List = g_list_append (List, ps);
	}

	/* Suppression des structures vides en fin de liste */
	NList = g_list_last (List);
	while (NList) {
		ps = (FIELDS_PS *)NList->data;
		if (NULL != ps && *ps->author == '\0' && *ps->title == '\0') {
			if (ps->author) { g_free (ps->author); ps->author = NULL; }
			if (ps->title)  { g_free (ps->title);  ps->title = NULL;  }
			if (ps->time)   { g_free (ps->time);   ps->time = NULL;   }
			g_free (ps);
			ps = NList->data = NULL;
		}
		else {
			break;
		}
		NList = g_list_previous (NList);
	}
	g_strfreev(Larrbuf);

	return (List);
}
// 
// 
PS *pochegen_get_struct_ps (void)
{
	PS   *new = NULL;
	gint  num;

	if ((new = pochegen_allocate_struct_ps ()) == NULL) {
		g_print ("ERREUR D'ALLOCATION MEMOIRE\n");
		return (NULL);
	}

	// PRINT_FUNC_LF();

	new->str_package = g_strdup (PACKAGE);
	new->str_version = g_strdup (VERSION);
	new->str_title = g_strdup (pocheevnt_get_entry_titre_ps ());
	new->str_foottext = g_strdup (pocheevnt_get_entry_foottext_ps ());
	/*
	if (GTK_WIDGET_IS_SENSITIVE(GTK_COMBO_BOX (poche.Adr_combobox_text_recto)) ||
	    GTK_WIDGET_IS_SENSITIVE(GTK_COMBO_BOX (poche.Adr_combobox_text_verso))) {
		new->glist_fields = pochegen_get_glist_from_textview ();
	}
	*/
	new->glist_fields = pochegen_get_glist_from_textview ();
	
	/* Titre sur recto / verso */
	switch (gtk_combo_box_get_active (GTK_COMBO_BOX (poche.Adr_combobox_entry_title))) {
	case 0 :
		new->bool_title_cover = TRUE;
		new->bool_title_back = FALSE;
		break;
	case 1 :
		new->bool_title_cover = FALSE;
		new->bool_title_back = TRUE;
		break;
	case 2 :
		new->bool_title_cover = TRUE;
		new->bool_title_back = TRUE;
		break;
	case 3 :
		new->bool_title_cover = FALSE;
		new->bool_title_back = FALSE;
		break;
	}
	/*
	if (GTK_WIDGET_IS_SENSITIVE(GTK_COMBO_BOX (poche.Adr_combobox_entry_title)) == FALSE) {
		new->bool_title_cover = FALSE;
		new->bool_title_back = FALSE;
	}
	*/
	/* Pied de page sur recto / verso */
	switch (gtk_combo_box_get_active (GTK_COMBO_BOX (poche.Adr_combobox_entry_foottext))) {
	case 0 :
		new->bool_foottext_cover = TRUE;
		new->bool_foottext_back = FALSE;
		break;
	case 1 :
		new->bool_foottext_cover = FALSE;
		new->bool_foottext_back = TRUE;
		break;
	case 2 :
		new->bool_foottext_cover = TRUE;
		new->bool_foottext_back = TRUE;
		break;
	case 3 :
		new->bool_foottext_cover = FALSE;
		new->bool_foottext_back = FALSE;
		break;
	}
	/*
	if (GTK_WIDGET_IS_SENSITIVE(GTK_COMBO_BOX (poche.Adr_combobox_entry_foottext)) == FALSE) {
		new->bool_foottext_cover = FALSE;
		new->bool_foottext_back = FALSE;
	}
	*/

	/* liste au recto */
	num = gtk_combo_box_get_active (GTK_COMBO_BOX (poche.Adr_combobox_text_recto));
	switch (num) {
	case 0 :
	case 1 :
	case 2 :
	case 3 :
	case 4 :
	case 5 :
		new->num_choicefieldscover = num;
		new->bool_textdata_cover = TRUE;
		break;
	default :
		new->num_choicefieldscover = 0;
		new->bool_textdata_cover = FALSE;
	}
	/*
	if (GTK_WIDGET_IS_SENSITIVE(GTK_COMBO_BOX (poche.Adr_combobox_text_recto)) == FALSE) {
		new->num_choicefieldscover = 0;
		new->bool_textdata_cover = FALSE;
	}
	*/

	/* liste au verso */
	num = gtk_combo_box_get_active (GTK_COMBO_BOX (poche.Adr_combobox_text_verso));
	switch (num) {
	case 0 :
	case 1 :
	case 2 :
	case 3 :
	case 4 :
	case 5 :
		new->num_choicefieldsback = num;
		new->bool_textdata_back = TRUE;
		break;
	default :
		new->num_choicefieldsback = 0;
		new->bool_textdata_back = FALSE;
	}
	/*
	if (GTK_WIDGET_IS_SENSITIVE(GTK_COMBO_BOX (poche.Adr_combobox_text_verso)) == FALSE) {
		new->num_choicefieldsback = 0;
		new->bool_textdata_back = FALSE;
	}
	*/

	/* Relier les champs */
	switch (gtk_combo_box_get_active (GTK_COMBO_BOX (poche.Adr_combobox_link_fields))) {
	case 0 :
		new->bool_line_cover = TRUE;
		new->bool_line_back = FALSE;
		break;
	case 1 :
		new->bool_line_cover = FALSE;
		new->bool_line_back = TRUE;
		break;
	case 2 :
		new->bool_line_cover = TRUE;
		new->bool_line_back = TRUE;
		break;
	case 3 :
		new->bool_line_cover = FALSE;
		new->bool_line_back = FALSE;
		break;
	}
	/*
	if (GTK_WIDGET_IS_SENSITIVE(GTK_COMBO_BOX (poche.Adr_combobox_link_fields)) == FALSE) {
		new->bool_line_cover = FALSE;
		new->bool_line_back = FALSE;
	}
	*/

	return ((PS *)new);
}
// 
// GEN CODE IMAGEMAGICK
// 
void pochegen_code_convert_ps (void)
{
	FILE  *fp_preview = NULL;
	guint  line = 0;

	// PRINT_FUNC_LF();
	
	fp_preview = fopen (poche.file_preview_ps, "a");
	while (*code_convert_ps [ line ]) {
		fprintf (fp_preview, "%s\n", code_convert_ps [ line ]);
		line ++;
	}
	fclose (fp_preview);
}


/*
http://groups.google.com/group/fr.comp.lang.postscript/browse_thread/thread/02c45a9323555bce/6ea3bd76a5363612
In article
<20b53f10-2209-494e-96ef-2b934820a...@26g2000hsk.googlegroups.com>,
 Ahmed Ayadi <ayadi.ahmed.e...@gmail.com> wrote:

> Je suis nouveau avec le language Postscript. J'aime bien imprimé des
> documents postscript avec des lettres avec accent comme 'é', 'à' et
> 'ç'.

Bonsoir.
En supposant l'usage de caractères sur 8 bits, il faut procéder au
recodage de la fonte.

Une fonte (Type 1, Type 3, TrueType, OpenType etc...) contient une
association entre un code numérique (code de caractère) et un dessin
vectoriel (glyphe) qui représente en général (au moins dans les langues
alphabétiques) une lettre, un chiffre, un signe de ponctuation etc...
Cette association est l'encodage. (encoding)
Cette association n'est pas fixée et peut être modifié via une
programmation Postscript adéquate. Cette programmation se nomme
'recodage' (re-encoding).
En Postscript les fontes ne sont pas tenues d'avoir un encodage
particulier,  et donc pour avoir des résultats reproductible avec
n'importe quelle fonte, le recodage est indispensable.

Pour une fonte T1 (ou Type 42, expression Postscript d'une TrueType),
l'encodage est donnée par le vecteur dont la clef est /Encoding dans le
dictionnaire de la fonte. Il faut modifier ce vecteur pour recoder la
fonte. Ce vecteur comporte 256 valeurs (indicées par le code de
caractère sur 8 bits, de 0 à 255). Chaque valeur de ce vecteur est un
nom de glyphe, identifiant le dessin qui sera effectué.

Donc pour recoder une fonte, on fait une copie du dictinnaire de fonte
où l'on assigne des valeurs différente aux éléments du vecteur
/Encoding. Les interpréteurs PS prédéfinissent des vecteurs usuels, ce
qui évite de devoir réinventer la roue. Un de ces vecteurs est
/ISOLatin1Encoding (voir Red Book annexe E)

Muni de ces explications, l'exemple du Red Book section 5.9.1 se
comprend mieux (sachant que pour des raisons d'unicité, il n'est pas
permis de cloner l'identificateur de fonte contenu sous clef /FID du
dictionnaire de fonte):

/Helvetica findfont
dup length dict begin
{ 1 index /FID ne
  {def}
  {pop pop}
  ifelse
} forall

/Encoding ISOLatin1Encoding def
currentdict
end
/Helvetica-ISOLatin1 exch definefont pop

/Helvetica-ISOLatin1 findfont 30 scalefont setfont
20 120 moveto (c cedille ç \347 ) show
20 90 moveto  (e grave è \350 ) show
20 60 moveto  (a grave à \340 ) show
showpage

François Robert
*/


/*
http://www.utf8-chartable.de/
	Table de caracteres
*/

/*
page 615 depuis:
	Manuel de reference du langage Postscript Edition: Addison-Wesley
*/
typedef struct {
	gint	car1, car2;
	gint	CarPostscript;
} TABLE_PS;

#define TOTAL_TABLE_PS 95

TABLE_PS TablePS [ TOTAL_TABLE_PS ] = {

 { 0xc2, 0xa1, 241 },	// 	¡	 INVERTED EXCLAMATION MARK
 { 0xc2, 0xa2, 242 },	// 	¢	 CENT SIGN
 { 0xc2, 0xa3, 243 },	// 	£	 POUND SIGN
 { 0xc2, 0xa4, 244 },	// 	¤	 CURRENCY SIGN
 { 0xc2, 0xa5, 245 },	// 	¥	 YEN SIGN
 { 0xc2, 0xa6, 246 },	// 	¦	 BROKEN BAR
 { 0xc2, 0xa7, 247 },	// 	§	 SECTION SIGN
 { 0xc2, 0xa8, 250 },	//	¨	 DIAERESIS
 { 0xc2, 0xa9, 251 },	//	©	 COPYRIGHT SIGN
 { 0xc2, 0xaa, 252 },	//	ª	 FEMININE ORDINAL INDICATOR
 { 0xc2, 0xab, 253 },	//	«	 LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
 { 0xc2, 0xac, 254 },	//	¬	 NOT SIGN
 { 0xc2, 0xad, 255 },	//		 SOFT HYPHEN
 { 0xc2, 0xae, 256 },	//	®	 REGISTERED SIGN
 { 0xc2, 0xaf, 257 },	//	¯	 MACRON
 { 0xc2, 0xb0, 260 },	//	°	 DEGREE SIGN
 { 0xc2, 0xb1, 261 },	//	±	 PLUS-MINUS SIGN
 { 0xc2, 0xb2, 262 },	//	²	 SUPERSCRIPT TWO
 { 0xc2, 0xb3, 263 },	//	³	 SUPERSCRIPT THREE
 { 0xc2, 0xb4, 264 },	//	´	 ACUTE ACCENT
 { 0xc2, 0xb5, 265 },	//	µ	 MICRO SIGN
 { 0xc2, 0xb6, 266 },	//	¶	 PILCROW SIGN
 { 0xc2, 0xb7, 267 },	//	·	 MIDDLE DOT
 { 0xc2, 0xb8, 270 },	//	¸	 CEDILLA
 { 0xc2, 0xb9, 271 },	//	¹	 SUPERSCRIPT ONE
 { 0xc2, 0xba, 272 },	//	º	 MASCULINE ORDINAL INDICATOR
 { 0xc2, 0xbb, 273 },	//	»	 RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
 { 0xc2, 0xbc, 274 },	//	¼	 VULGAR FRACTION ONE QUARTER
 { 0xc2, 0xbd, 275 },	//	½	 VULGAR FRACTION ONE HALF
 { 0xc2, 0xbe, 276 },	//	¾	 VULGAR FRACTION THREE QUARTERS
 { 0xc2, 0xbf, 277 },	//	¿	 INVERTED QUESTION MARK
 { 0xc3, 0x80, 300 },	//	À	 LATIN CAPITAL LETTER A WITH GRAVE
 { 0xc3, 0x81, 301 },	//	Á	 LATIN CAPITAL LETTER A WITH ACUTE
 { 0xc3, 0x82, 302 },	//	Â	 LATIN CAPITAL LETTER A WITH CIRCUMFLEX
 { 0xc3, 0x83, 303 },	//	Ã	 LATIN CAPITAL LETTER A WITH TILDE
 { 0xc3, 0x84, 304 },	//	Ä	 LATIN CAPITAL LETTER A WITH DIAERESIS
 { 0xc3, 0x85, 305 },	//	Å	 LATIN CAPITAL LETTER A WITH RING ABOVE
 { 0xc3, 0x86, 306 },	//	Æ	 LATIN CAPITAL LETTER AE
 { 0xc3, 0x87, 307 },	//	Ç	 LATIN CAPITAL LETTER C WITH CEDILLA
 { 0xc3, 0x88, 310 },	//	È	 LATIN CAPITAL LETTER E WITH GRAVE
 { 0xc3, 0x89, 311 },	//	É	 LATIN CAPITAL LETTER E WITH ACUTE
 { 0xc3, 0x8a, 312 },	//	Ê	 LATIN CAPITAL LETTER E WITH CIRCUMFLEX
 { 0xc3, 0x8b, 313 },	//	Ë	 LATIN CAPITAL LETTER E WITH DIAERESIS
 { 0xc3, 0x8c, 314 },	//	Ì	 LATIN CAPITAL LETTER I WITH GRAVE
 { 0xc3, 0x8d, 315 },	//	Í	 LATIN CAPITAL LETTER I WITH ACUTE
 { 0xc3, 0x8e, 316 },	//	Î	 LATIN CAPITAL LETTER I WITH CIRCUMFLEX
 { 0xc3, 0x8f, 317 },	//	Ï	 LATIN CAPITAL LETTER I WITH DIAERESIS
 { 0xc3, 0x90, 320 },	//	Ð	 LATIN CAPITAL LETTER ETH
 { 0xc3, 0x91, 321 },	//	Ñ	 LATIN CAPITAL LETTER N WITH TILDE
 { 0xc3, 0x92, 322 },	//	Ò	 LATIN CAPITAL LETTER O WITH GRAVE
 { 0xc3, 0x93, 323 },	//	Ó	 LATIN CAPITAL LETTER O WITH ACUTE
 { 0xc3, 0x94, 324 },	//	Ô	 LATIN CAPITAL LETTER O WITH CIRCUMFLEX
 { 0xc3, 0x95, 325 },	//	Õ	 LATIN CAPITAL LETTER O WITH TILDE
 { 0xc3, 0x96, 326 },	//	Ö	 LATIN CAPITAL LETTER O WITH DIAERESIS
 { 0xc3, 0x97, 327 },	// 	×	 MULTIPLICATION SIGN
 { 0xc3, 0x98, 330 },	//	Ø	 LATIN CAPITAL LETTER O WITH STROKE
 { 0xc3, 0x99, 331 },	//	Ù	 LATIN CAPITAL LETTER U WITH GRAVE
 { 0xc3, 0x9a, 332 },	//	Ú	 LATIN CAPITAL LETTER U WITH ACUTE
 { 0xc3, 0x9b, 333 },	//	Û	 LATIN CAPITAL LETTER U WITH CIRCUMFLEX
 { 0xc3, 0x9c, 334 },	//	Ü	 LATIN CAPITAL LETTER U WITH DIAERESIS
 { 0xc3, 0x9d, 335 },	//	Ý	 LATIN CAPITAL LETTER Y WITH ACUTE
 { 0xc3, 0x9e, 336 },	//	Þ	 LATIN CAPITAL LETTER THORN
 { 0xc3, 0x9f, 337 },	//	ß	 LATIN SMALL LETTER SHARP S
 { 0xc3, 0xa0, 340 },	//	à	 LATIN SMALL LETTER A WITH GRAVE
 { 0xc3, 0xa1, 341 },	//	á	 LATIN SMALL LETTER A WITH ACUTE
 { 0xc3, 0xa2, 342 },	//	â	 LATIN SMALL LETTER A WITH CIRCUMFLEX
 { 0xc3, 0xa3, 343 },	//	ã	 LATIN SMALL LETTER A WITH TILDE
 { 0xc3, 0xa4, 344 },	//	ä	 LATIN SMALL LETTER A WITH DIAERESIS
 { 0xc3, 0xa5, 345 },	//	å	 LATIN SMALL LETTER A WITH RING ABOVE
 { 0xc3, 0xa6, 346 },	//	æ	 LATIN SMALL LETTER AE
 { 0xc3, 0xa7, 347 },	//	ç	 LATIN SMALL LETTER C WITH CEDILLA
 { 0xc3, 0xa8, 350 },	//	è	 LATIN SMALL LETTER E WITH GRAVE
 { 0xc3, 0xa9, 351 },	//	é	 LATIN SMALL LETTER E WITH ACUTE
 { 0xc3, 0xaa, 352 },	//	ê	 LATIN SMALL LETTER E WITH CIRCUMFLEX
 { 0xc3, 0xab, 353 },	//	ë	 LATIN SMALL LETTER E WITH DIAERESIS
 { 0xc3, 0xac, 354 },	//	ì	 LATIN SMALL LETTER I WITH GRAVE
 { 0xc3, 0xad, 355 },	//	í	 LATIN SMALL LETTER I WITH ACUTE
 { 0xc3, 0xae, 356 },	//	î	 LATIN SMALL LETTER I WITH CIRCUMFLEX
 { 0xc3, 0xaf, 357 },	//	ï	 LATIN SMALL LETTER I WITH DIAERESIS
 { 0xc3, 0xb0, 360 },	//	ð	 LATIN SMALL LETTER ETH
 { 0xc3, 0xb1, 361 },	//	ñ	 LATIN SMALL LETTER N WITH TILDE
 { 0xc3, 0xb2, 362 },	//	ò	 LATIN SMALL LETTER O WITH GRAVE
 { 0xc3, 0xb3, 363 },	//	ó	 LATIN SMALL LETTER O WITH ACUTE
 { 0xc3, 0xb4, 364 },	//	ô	 LATIN SMALL LETTER O WITH CIRCUMFLEX
 { 0xc3, 0xb5, 365 },	//	õ	 LATIN SMALL LETTER O WITH TILDE
 { 0xc3, 0xb6, 366 },	//	ö	 LATIN SMALL LETTER O WITH DIAERESIS
 { 0xc3, 0xb7, 367 },	//	÷	 DIVISION SIGN
 { 0xc3, 0xb8, 370 },	//	ø	 LATIN SMALL LETTER O WITH STROKE
 { 0xc3, 0xb9, 371 },	//	ù	 LATIN SMALL LETTER U WITH GRAVE
 { 0xc3, 0xba, 372 },	//	ú	 LATIN SMALL LETTER U WITH ACUTE
 { 0xc3, 0xbb, 373 },	//	û	 LATIN SMALL LETTER U WITH CIRCUMFLEX
 { 0xc3, 0xbc, 374 },	//	ü	 LATIN SMALL LETTER U WITH DIAERESIS
 { 0xc3, 0xbd, 375 },	//	ý	 LATIN SMALL LETTER Y WITH ACUTE
 { 0xc3, 0xbe, 376 },	//	þ	 LATIN SMALL LETTER THORN
 { 0xc3, 0xbf, 377 }	//	ÿ	 LATIN SMALL LETTER Y WITH DIAERESIS

};

// 
// 
gint pochegen_get_car_table_ps (gint car1, gint car2)
{
	gint	cpt;
	
	for (cpt = 0; cpt < TOTAL_TABLE_PS; cpt ++) {
		if (car1 == TablePS [ cpt ] . car1 && car2 == TablePS [ cpt ] . car2) break;
	}
	if (cpt >= TOTAL_TABLE_PS) {
		return (40);	// ESPACE
	}
	
	return ( TablePS [ cpt ] . CarPostscript );
}
// 
// 
void pochegen_fprintf (FILE *fp, gchar *buf)
{
	gchar	*Ptr = buf;
	gint	car1, car2;
	
	fprintf (fp, "(");
	while (*Ptr) {
		if (*Ptr < 0) {
			car1 = *Ptr;
			car1 &= 0xff;
			car2 = *(Ptr +1);
			car2 &= 0xff;
			fprintf (fp, "\\%03d", pochegen_get_car_table_ps (car1, car2));
			Ptr ++;
		}
		else {
			fprintf (fp, "%c", *Ptr);
		}
		Ptr ++;
	}
	fprintf (fp, ")");
}
/*
*---------------------------------------------------------------------------
* GEN CODE XCFA
*---------------------------------------------------------------------------
*/

// 
// Generation du fichier PostScript
// 
void pochegen_code_xcfa_ps (void)
{
	FILE		*fp_preview = NULL;
	gint		line = 0;
	PS		*new = NULL;
	GList		*List = NULL;
	FIELDS_PS	*ps = NULL;

	// PRINT_FUNC_LF();

	if ((new = pochegen_get_struct_ps ()) == NULL) {
		g_print ("ERREUR D'ALLOCATION MEMOIRE\n");
		return;
	}

	/* Creation fichier vierge */
	fp_preview = fopen (poche.file_preview_ps, "a");
	line = 0;

	while (*code_xcfa_ps [ line ]) {
		/* PARAM_PACKAGE	*/
		if (strstr (code_xcfa_ps [ line ], "PARAM_PACKAGE")) {
			fprintf (fp_preview, "/PACKAGE (%s) def\n", new->str_package);
		}
		/* PARAM_VERSION	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_VERSION")) {
			fprintf (fp_preview, "/VERSION (%s) def\n", new->str_version);
		}
		/* PARAM_TITLE	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_TITLE")) {
			
			// fprintf (fp_preview, "/title (%s) def\n", new->str_title);
			
			fprintf (fp_preview, "/title ");
			pochegen_fprintf (fp_preview, new->str_title);
			fprintf (fp_preview, " def\n");
		}
		/* PARAM_BOOL_TITLE_COVER	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_BOOL_TITLE_COVER")) {
			fprintf (fp_preview, "/bool_title_cover %s def\n", new->bool_title_cover ? "true" : "false");
		}
		/* PARAM_BOOL_TITLE_BACK	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_BOOL_TITLE_BACK")) {
			fprintf (fp_preview, "/bool_title_back %s def\n", new->bool_title_back ? "true" : "false");
		}
		/* PARAM_FOOTTEXT	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_FOOTTEXT")) {
		
			// fprintf (fp_preview, "/foottext (%s) def\n", new->str_foottext);
			
			fprintf (fp_preview, "/foottext ");
			pochegen_fprintf (fp_preview, new->str_foottext);
			fprintf (fp_preview, " def\n");
		}
		/* PARAM_BOOL_FOOTTEXT_COVER	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_BOOL_FOOTTEXT_COVER")) {
			fprintf (fp_preview, "/bool_foottext_cover %s def\n", new->bool_foottext_cover ? "true" : "false");
		}
		/* PARAM_BOOL_FOOTTEXT_BACK	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_BOOL_FOOTTEXT_BACK")) {
			fprintf (fp_preview, "/bool_foottext_back %s def\n", new->bool_foottext_back ? "true" : "false");
		}
		/* PARAM_BOOL_TEXTDATA_COVER	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_BOOL_TEXTDATA_COVER")) {
			fprintf (fp_preview, "/bool_textdata_cover %s def\n", new->bool_textdata_cover ? "true" : "false");
		}
		/* PARAM_BOOL_TEXTDATA_BACK	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_BOOL_TEXTDATA_BACK")) {
			fprintf (fp_preview, "/bool_textdata_back %s def\n", new->bool_textdata_back ? "true" : "false");
		}
		/* PARAM_CHOICEFIELDSCOVER	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_CHOICEFIELDSCOVER")) {
			fprintf (fp_preview, "/choicefieldscover %d def\n", new->num_choicefieldscover);
		}
		/* PARAM_CHOICEFIELDSBACK	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_CHOICEFIELDSBACK")) {
			fprintf (fp_preview, "/choicefieldsback %d def\n", new->num_choicefieldsback);
		}
		/* PARAM_BOOL_LINE_COVER	*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_BOOL_LINE_COVER")) {
			fprintf (fp_preview, "/bool_line_cover %s def\n", new->bool_line_cover ? "true" : "false");
		}
		/* PARAM_BOOL_LINE_BACK		*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_BOOL_LINE_BACK")) {
			fprintf (fp_preview, "/bool_line_back %s def\n", new->bool_line_back ? "true" : "false");
		}
		/* PARAM_TEXTDATA		*/
		else if (strstr (code_xcfa_ps [ line ], "PARAM_TEXTDATA")) {
			/*
			new->glist_fields = NULL;
			fprintf (fp_preview, "/bool_line_back %s def\n", new->bool_line_back ? "true" : "false");
			*/
			List = g_list_first (new->glist_fields);
			if (List) {
			
				fprintf (fp_preview, "/textdata [\n");
				while (List) {
					if ((ps = (FIELDS_PS *)List->data)) {
						
						// fprintf (fp_preview, "(%s)",  ps->author);
						pochegen_fprintf (fp_preview, ps->author);
												
						// fprintf (fp_preview, " (%s)", ps->title);
						pochegen_fprintf (fp_preview, ps->title);
						
						if (*ps->author || *ps->title)
							fprintf (fp_preview, " (%s)", ps->time);
						else	fprintf (fp_preview, " ()");
						
						fprintf (fp_preview, "\n");
					}
					List = g_list_next (List);
				}
				fprintf (fp_preview, "] def\n");
			}
			else {
				fprintf (fp_preview, "/textdata [\n");
				fprintf (fp_preview, "() () ()\n");
				fprintf (fp_preview, "] def\n");
			}
		}
		else {
			fprintf (fp_preview, "%s\n", code_xcfa_ps [ line ]);
		}
		line ++;
	}

	fclose (fp_preview);

	new = pochegen_remove_struct_ps (new);
}
// 
// GENERATION DU CODE POSTSCRIPT RECTO / VERSO
// 
GLIST_POCHETTE *pochegen_get_elem_actif (gboolean bool_recto)
{
	GLIST_POCHETTE *ptr = NULL;

	if (bool_recto == TRUE)
		ptr = (GLIST_POCHETTE *)poche.ptr_recto;
	else	ptr = (GLIST_POCHETTE *)poche.ptr_verso;

	return ((GLIST_POCHETTE *)ptr);
}
// 
// 
gboolean pochegen_creation_postscript_recto (void)
{
	GLIST_POCHETTE *gl = NULL;
	gchar          *file_recto_tmp = NULL;
	FILE           *fp_recto = NULL;
	FILE           *fp_recto_tmp = NULL;
#define MAX_CARS 255
	gchar  buf   [ MAX_CARS +1 ];
	gchar		*Dummy;

	// PRINT_FUNC_LF();

	if ((gl = pochegen_get_elem_actif (TRUE))) {

		if (libutils_test_file_exist (poche.file_recto_ps) == TRUE) {
			/*g_print ("PAS DE GENERATION RECTO\n");*/
			return (TRUE);
		}
		/*g_print ("GENERATION RECTO: %s\n", poche.file_recto_ps);*/

		file_recto_tmp = g_strdup_printf ("%s/recto_tmp.ps", poche.Path_pochette_ps);
		pocheexec_convert_to_postscript (gl->name_img, file_recto_tmp);
	}
	else {
		g_unlink (poche.file_recto_ps);
		return (FALSE);
	}

	fp_recto = fopen (poche.file_recto_ps, "w");
	fp_recto_tmp = fopen (file_recto_tmp, "r");

	while (fgets (buf, MAX_CARS, fp_recto_tmp) != NULL) {
		if (buf[0]  == 'D' &&
		    buf[1]  == 'i' &&
		    buf[2]  == 's' &&
		    buf[3]  == 'p' &&
		    buf[4]  == 'l' &&
		    buf[5]  == 'a' &&
		    buf[6]  == 'y' &&
		    buf[7]  == 'I' &&
		    buf[8]  == 'm' &&
		    buf[9]  == 'a' &&
		    buf[10] == 'g' &&
		    buf[11] == 'e') {
			/*g_print ("\n%s\n", buf);*/
			break;
		}
	}
	Dummy = fgets (buf, MAX_CARS, fp_recto_tmp);
	Dummy = fgets (buf, MAX_CARS, fp_recto_tmp);
	while (fgets (buf, MAX_CARS, fp_recto_tmp) != NULL) {
		if (buf [ 0 ] == '%' && buf [ 1 ] == '%') break;
		fprintf (fp_recto, "%s", buf);
	}
	fprintf (fp_recto, "\n");

	fclose (fp_recto);
	fclose (fp_recto_tmp);

	g_unlink (file_recto_tmp);

	g_free (file_recto_tmp);
	file_recto_tmp = NULL;

	return (TRUE);
}
// 
// 
gboolean pochegen_creation_postscript_verso (void)
{
	GLIST_POCHETTE *gl = NULL;
	gchar          *file_verso_tmp = NULL;
	FILE           *fp_verso = NULL;
	FILE           *fp_verso_tmp = NULL;
#define MAX_CARS 255
	gchar  buf   [ MAX_CARS +1 ];
	gchar		*Dummy;

	// PRINT_FUNC_LF();

	if ((gl = pochegen_get_elem_actif (FALSE))) {

		if (libutils_test_file_exist (poche.file_verso_ps) == TRUE) {
			/*g_print ("PAS DE GENERATION VERSO\n");*/
			return (TRUE);
		}
		/*g_print ("GENERATION VERSO: %s\n", poche.file_verso_ps);*/

		file_verso_tmp = g_strdup_printf ("%s/verso_tmp.ps", poche.Path_pochette_ps);
		pocheexec_convert_to_postscript (gl->name_img, file_verso_tmp);
	}
	else {
		g_unlink (poche.file_verso_ps);
		return (FALSE);
	}

	fp_verso = fopen (poche.file_verso_ps, "w");
	fp_verso_tmp = fopen (file_verso_tmp, "r");

	while (fgets (buf, MAX_CARS, fp_verso_tmp) != NULL) {
		if (buf[0]  == 'D' &&
		    buf[1]  == 'i' &&
		    buf[2]  == 's' &&
		    buf[3]  == 'p' &&
		    buf[4]  == 'l' &&
		    buf[5]  == 'a' &&
		    buf[6]  == 'y' &&
		    buf[7]  == 'I' &&
		    buf[8]  == 'm' &&
		    buf[9]  == 'a' &&
		    buf[10] == 'g' &&
		    buf[11] == 'e') {
			/*g_print ("\n%s\n", buf);*/
			break;
		}
	}
	Dummy = fgets (buf, MAX_CARS, fp_verso_tmp);
	Dummy = fgets (buf, MAX_CARS, fp_verso_tmp);
	while (fgets (buf, MAX_CARS, fp_verso_tmp) != NULL) {
		if (buf [ 0 ] == '%' && buf [ 1 ] == '%') break;
		fprintf (fp_verso, "%s", buf);
	}
	fprintf (fp_verso, "\n");

	fclose (fp_verso);
	fclose (fp_verso_tmp);

	g_unlink (file_verso_tmp);

	g_free (file_verso_tmp);
	file_verso_tmp = NULL;

	return (TRUE);
}
// 
// 
gchar *pochegen_get_coordonees_recto (void)
{
	gchar *str = NULL;

	str = g_strdup_printf ("%d %d\n%d %d\n",
			poche.tab_recto [ 0 ],
			poche.tab_recto [ 1 ],
			poche.tab_recto [ 2 ],
			poche.tab_recto [ 3 ]);
	strcpy (poche.coord_recto, str);
	g_free (str);
	str = NULL;
	return (&poche.coord_recto[0]);
}
// 
// 
gchar *pochegen_get_coordonees_verso (gint choice)
{
	gchar *str = NULL;

	str = g_strdup_printf ("%d %d\n%d %d\n",
			poche.tab_verso [ 0 ],
			poche.tab_verso [ 1 ],
			poche.tab_verso [ 2 ],
			poche.tab_verso [ 3 ]);
	strcpy (poche.coord_verso, str);
	g_free (str);
	str = NULL;
	return (&poche.coord_verso[0]);
}
// 
// 
void pochegen_code_entete_ps (void)
{
	FILE  *fp_preview = NULL;
	guint  line = 0;

	// PRINT_FUNC_LF();
	
	fp_preview = fopen (poche.file_preview_ps, "w");
	while (*code_entete_ps [ line ]) {
		fprintf (fp_preview, "%s\n", code_entete_ps [ line ]);
		line ++;
	}
	fclose (fp_preview);
}
// 
// GENERATION DU CODE POSTSCRIPT
// 
void pochegen_gen_code_postscript (void)
{
	FILE     *fp_recto = NULL;
	FILE     *fp_verso = NULL;
	FILE     *fp_preview = NULL;
#define MAX_CARS 255
	gchar     buf   [ MAX_CARS +1 ];
	gboolean  bool_file_recto = FALSE;
	gboolean  bool_file_verso = FALSE;
	static gboolean enter_pochegen_gen_code_postscript = FALSE;
	
	if (TRUE == enter_pochegen_gen_code_postscript) return;
	enter_pochegen_gen_code_postscript = TRUE;

	// PRINT_FUNC_LF();

	pochegen_code_entete_ps ();
	bool_file_recto = pochegen_creation_postscript_recto ();
	bool_file_verso = pochegen_creation_postscript_verso ();

	if (bool_file_recto || bool_file_verso) {
		pochegen_code_convert_ps ();
	}

	if (bool_file_verso) {
		fp_verso = fopen (poche.file_verso_ps, "r");
		fp_preview = fopen (poche.file_preview_ps, "a");
		fprintf (fp_preview, "DisplayImage\n");
		fprintf (fp_preview, "%s", pochegen_get_coordonees_verso (1));
		while (fgets (buf, MAX_CARS, fp_verso) != NULL) {
			fprintf (fp_preview, "%s", buf);
		}
		fprintf (fp_preview, "\n");
		fclose (fp_verso);
		fclose (fp_preview);
	}
	
	if (bool_file_recto) {
		fp_recto = fopen (poche.file_recto_ps, "r");
		fp_preview = fopen (poche.file_preview_ps, "a");
		fprintf (fp_preview, "DisplayImage\n");
		fprintf (fp_preview, "%s", pochegen_get_coordonees_recto ());
		while (fgets (buf, MAX_CARS, fp_recto) != NULL) {
			fprintf (fp_preview, "%s", buf);
		}
		fprintf (fp_preview, "\n");
		fclose (fp_recto);
		fclose (fp_preview);
	}

	pochegen_code_xcfa_ps ();
	
	enter_pochegen_gen_code_postscript = FALSE;
}

/*
*---------------------------------------------------------------------------
* AFFICHAGE DE L'IMAGE DU PREVIEW
*---------------------------------------------------------------------------
*/

// 
// Dimensionne l'image et affichage dans le preview
// 
void pochegen_affiche_image (void)
{
	gdouble  percent = pocheevnt_get_percent_zoom ();
	gint     hauteur;
	gint     largeur;
	gchar   *dimension = NULL;
	gchar   *LineCommand = NULL;
	/*GError  *error = NULL;*/
	
	// PRINT_FUNC_LF();

	/* Calculs dimension */
	largeur = (percent * 612.0) / 100.0;
	hauteur = (percent * 792.0) / 100.0;
	dimension = g_strdup_printf ("%dx%d", largeur, hauteur);

	/* Generation fichier preview.png */
	pocheexec_convert_to_png (poche.file_preview_ps, poche.file_preview_png, dimension);
	g_free (dimension);
	dimension = NULL;

	gdk_window_clear (poche.Adr_image_preview->window);

	/* Afficher le fichier preview.png */

	LineCommand = g_strdup (poche.file_preview_png);
	gtk_image_set_from_file (GTK_IMAGE (poche.Adr_image_preview), LineCommand);
	/*gtk_widget_show(poche.Adr_image_preview);*/
	g_free (LineCommand);
	LineCommand = NULL;
	
	poche.bool_with_get_grille_img = TRUE;
}
// 
// 
void pochegen_file_postscript (void)
{
	gchar *newname = NULL;
	gchar *PtrName = NULL;
	
	// PRINT_FUNC_LF();
	
	PtrName = strrchr (poche.file_preview_ps, '/');
	PtrName ++;
	newname = g_strdup_printf ("%s/%s", Config.PathDestFilePostScript, PtrName);
	conv_copy_src_to_dest (poche.file_preview_ps, newname);
	g_free (newname);
	newname = NULL;
}
// 
// 
void pochegen_file_pdf (void)
{
	// PRINT_FUNC_LF();
	
	/* Test si ps2pdf est present */
	if (PrgInit.bool_ps2pdf == FALSE) {
		wininfo_create (
			_("ps2pdf est absent"),
			_("Production du fichier PDF impossible car Le programme"),
			  "\n",
			_("'ps2pdf' est absent de votre configuration."),
			  "\n\n",
		 	_("SOLUTION:"),
		 	  "\n",
			_("Installez 'ps2pdf'."),
			  "");
		return;
	}
	/* Fabrication du PDF */
	pocheexec_make_pdf_file (poche.file_preview_ps, Config.PathDestFilePostScript);
}
// 
// 
void pochegen_file_postscript_pdf (void)
{
	// PRINT_FUNC_LF();
	
	pochegen_file_postscript ();
	pochegen_file_pdf ();
}
















