 /*
 * file      : get_info.c
 * project   : xcfa
 * with      : Gtk-2
 *
 * copyright : (C) 2003 - 2010 by Claude Bulin
 *
 * xcfa - GTK+ implementation of the GNU shell command
 * GNU General Public License
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * OLD ADRESS:
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * NEW ADRESS:
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 */



#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#ifdef ENABLE_NLS
	#include <libintl.h>
	#define _(String) gettext (String)
#endif

#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <pthread.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/utsname.h>

#ifdef ENABLE_STATIC_LIBRARY
	#include "../lib/lib.h"
#endif

#include "global.h"
#include "prg_init.h"
#include "file.h"
#include "configuser.h"
#include "get_info.h"





#define GET_MAX_CARS 1024


typedef struct {

	pid_t		code_fork;		/* Num Code Fork	*/
	int		signal_numchildren;	/* the signal handler	*/
	int		tube [ 2 ];		/* for pipe		*/
	gboolean	bool_percent;

} VAR_GET;

VAR_GET VarGet;


/*
*---------------------------------------------------------------------------
* SIGNAL
*---------------------------------------------------------------------------
*/
void GetInfo_sigchld (gint signum)
{
        gint status;

        wait(&status);

	/* PRINT_FUNC(); */
        /* if there are still children waiting
        *  re-install the signal handler
	*/
	VarGet.signal_numchildren --;
        if (VarGet.signal_numchildren > 0)
        {
                /* re-install the signal handler */
                signal (SIGCHLD, GetInfo_sigchld);
        }
}
// 
// 
gint GetInfo_exec_with_output (gchar **args, pid_t *p, gint Streams)
{
	VarGet.signal_numchildren = 0;

	if (pipe (VarGet.tube) != 0)
	{
		fprintf (stderr, "error: pipe\n");
		exit (1);
	}
	if ((*p = fork()) == 0)
	{
		dup2 (VarGet.tube [ 1 ], Streams); /* STDOUT_FILENO STDERR_FILENO */
		close (VarGet.tube [ 1 ]);
		execvp (args[0], args);
		fprintf (stderr, "error: exec");
		exit (2);
	}
	VarGet.signal_numchildren ++;
	signal (SIGCHLD, GetInfo_sigchld);
	close (VarGet.tube [ 1 ]);
	return (VarGet.tube [ 0 ]);
}


/*
*---------------------------------------------------------------------------
* FUNCTION
*---------------------------------------------------------------------------
*/
extern float roundf(float x);

// RETOURNE, EN KiloOctet LA PLACE DISPONIBLE DU DOSSIER TEMORAIRE
// 
glong GetInfo_level_df (void)
{
	gint   pos = 0;
	gchar *args [ 20 ];
	gint   fd;
	gint   size = -1;
	gchar  buf [ 1024 + 10 ];
	gchar *Path = g_strdup_printf ("%s/", Config.PathnameTMP);
	glong  Size = 0;
	gchar *Ptr = NULL;
	
	args[pos++] = "df";
	args[pos++] = "--block-size=K";
	args[pos++] = Path;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!  NORMALISE -n %s\n", args[ 0 ]);
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/
	
	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);

	do {
		pos = -1;
		do {
			pos++;
			if (pos >= 1024) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= 1024\n", pos);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		if (strstr (buf, "/dev/")) {
			/* 
			buf=/dev/sda5            19686772K  4080996K 14605736K  22% /
			*/
			/* g_print("buf=%s\n", buf); */
			Ptr = strchr (buf, 'K');
			Ptr ++;
			Ptr = strchr (Ptr, 'K');
			Ptr ++;
			while (*Ptr == ' ') Ptr ++;
			/* g_print("Ptr  = '%s'\n", Ptr); */
			Size = (glong)atol (Ptr);
			/* g_print("Size = %lu\n", Size); */
		}

	} while (size != 0);
	
	g_free (Path);
	Path = NULL;

	close(fd);
	VarGet.code_fork = -1;
	return (Size);
}
// 
// 
gboolean GetInfo_level_bool_percent (void)
{
	return (VarGet.bool_percent);
}
// 
// 
gint GetInfo_level_get_from (TYPE_FILE_IS TypeFileIs, gchar *file)
{
	gint   pos = 0;
	gchar *args [ 20 ];
	gint   fd;
	gint   size = -1;
	gchar  buf [ 1024 + 10 ];
	gint   arrondit = -1;
	
	if (TypeFileIs == FILE_IS_WAV || TypeFileIs == FILE_IS_MP3) {
		/*
		normalize-audio -n ./kill_bill.wav
		*/
		args[pos++] = prginit_get_name (NMR_normalize);
		args[pos++] = "-n";
		args[pos++] = file;
		args[pos++] = NULL;
	}
	else if (TypeFileIs == FILE_IS_OGG) {
		/*
		normalize-ogg -n ./kill_bill.ogg
		*/
		args[pos++] = "normalize-ogg";
		args[pos++] = "-n";
		args[pos++] = file;
		args[pos++] = NULL;
	}

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!  NORMALISE -n %s\n", args[ 0 ]);
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/
	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);

	do {
		pos = -1;
		do {
			pos++;
			if (pos >= 1024) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= 1024\n", pos);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		VarGet.bool_percent = TRUE;
		
		if (strstr (buf, "dBFS")) {
			/* g_print ("dBFS\n"); */
			/* g_print ("buf = %s\n", buf); */
			/* g_print ("%f\n", atof(buf)); */
			arrondit = (gint)roundf(atof(buf));
			/* g_print ("%d\n", arrondit); */
		}
		
	} while (size != 0);

	close(fd);
	VarGet.code_fork = -1;
	return (arrondit);
}
// 
// 
GString *GetInfo_file (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);

	args[pos++] = "file";
	args[pos++] = "-b";
	args[pos++] = "-z";
	args[pos++] = "-n";
	args[pos++] = file;
	args[pos++] = NULL;

/*
	g_print ("!-----------------------------------!\n");
	g_print ("!             F I L E               !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);

	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		/*
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
		*/
		buf[pos ++ ] = '\n';
		buf[pos] = '\0';
		g_string_append_printf (gstr, "%s", buf);

	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
gchar *GetInfo_cpu (void)
{	
	gchar	 buf [ GET_MAX_CARS + 10 ];
	FILE	*fp;
	gchar	*Ptr = NULL;
	
	fp = fopen ("/proc/cpuinfo", "r");
	while (fgets (buf,  GET_MAX_CARS, fp) != NULL) {
		// g_print("BUF = %s", buf);
		Ptr = buf;
		if (*(Ptr + 0) == 'm' &&
		    *(Ptr + 1) == 'o' &&
		    *(Ptr + 2) == 'd' &&
		    *(Ptr + 3) == 'e' &&
		    *(Ptr + 4) == 'l' &&
		    *(Ptr + 5) == ' ' &&
		    *(Ptr + 6) == 'n' &&
		    *(Ptr + 7) == 'a' &&
		    *(Ptr + 8) == 'm' &&
		    *(Ptr + 9) == 'e'
		    
		) {
			// g_print("---------BUF = %s", buf);
			
			Ptr = buf;
			while (*Ptr ++)
				if (*Ptr == '\n') *Ptr = '\0';
			for ( Ptr = buf; *Ptr != ':'; Ptr ++ );
			Ptr ++;
			while (*Ptr == ' ') Ptr ++;
			// g_print("---------BUF = %s", Ptr);
			
			fclose (fp);
			return (g_strdup (Ptr));
		}
	}
	fclose (fp);
	return (NULL);
}
// 
// 
gchar *GetInfo_cpu_str (void)
{	
	struct	utsname utsname;
	gchar	*ptruname = NULL;
	gchar	*ModelName = NULL;

	ModelName = GetInfo_cpu ();

	HostConf.NbCpu   = sysconf(_SC_NPROCESSORS_ONLN);
	HostConf.TypeCpu = 8 * sizeof(void*);
	if (uname (&utsname) == 0) {
		
		strcpy (HostConf.Machine, utsname.machine);
		if (strcmp (HostConf.Machine, "x86_64") == 0)
			HostConf.BoolCpuIs64Bits = TRUE;
		else	HostConf.BoolCpuIs64Bits = FALSE;
		ptruname = g_strdup_printf (
				"<span foreground=\"#0000FF\"><b>Informations machine:</b></span>\n\n"
				"sysname\n\t<b> %s</b>\n"
				"nodename\n\t<b> %s</b>\n"
				"release\n\t<b> %s</b>\n"
				"version\n\t<b> %s</b>\n"
				"machine\n\t<b> %s</b>\n"
				"\t<b> %d CPU (%d bits)</b>\n"
				"\t<b> %s</b>\n",
			utsname.sysname, utsname.nodename, utsname.release, utsname.version, utsname.machine,
			HostConf.NbCpu, HostConf.TypeCpu,
			ModelName);
	}
	g_free (ModelName);
	ModelName = NULL;
	return (ptruname);
}
// 
// 
void GetInfo_cpu_print (void)
{	
	struct	utsname utsname;
	gchar	*ptruname = NULL;
	gchar	*ModelName = NULL;

	ModelName = GetInfo_cpu ();

	HostConf.NbCpu   = sysconf(_SC_NPROCESSORS_ONLN);
	HostConf.TypeCpu = 8 * sizeof(void*);
	if (uname (&utsname) == 0) {
		
		strcpy (HostConf.Machine, utsname.machine);
		if (strcmp (HostConf.Machine, "x86_64") == 0)
			HostConf.BoolCpuIs64Bits = TRUE;
		else	HostConf.BoolCpuIs64Bits = FALSE;
		
		ptruname = g_strdup_printf (
				"\nINFORMATIONS MACHINE:\n\n"
				"sysname\t\t= %s\n"
				"nodename\t= %s\n"
				"release\t\t= %s\n"
				"version\t\t= %s\n"
				"machine\t\t= %s\n"
				"       \t\t= %d CPU (%d bits)\n"
				"       \t\t= %s\n",
			utsname.sysname, utsname.nodename, utsname.release, utsname.version, utsname.machine,
			HostConf.NbCpu, HostConf.TypeCpu,
			ModelName);
		g_print("%s",ptruname);
	}
	g_free (ModelName);
	ModelName = NULL;
	g_free (ptruname);
	ptruname = NULL;
}
// 
// 
GString *GetInfo_faad (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);

	args[pos++] = "faad";
	args[pos++] = "-i";
	args[pos++] = file;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!             F A A D               !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDERR_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		/*
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
		*/
		buf[pos ++ ] = '\n';
		buf[pos] = '\0';
		g_string_append_printf (gstr, "%s", buf);

	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
GString *GetInfo_checkmp3 (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = NULL;

	gstr = g_string_new (NULL);

	// args[pos++] = PrgInit.name_checkmp3;
	args[pos++] = prginit_get_name (NMR_checkmp3);
	args[pos++] = "-a";
	args[pos++] = "-v";
	args[pos++] = "-i";
	args[pos++] = file;
	args[pos++] = NULL;
	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!           C H E C K M P 3         !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/
	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
GString *GetInfo_ogginfo (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);

	args[pos++] = "ogginfo";
	args[pos++] = file;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!           O G G I N F O           !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
GString *GetInfo_mediainfo (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);

	args[pos++] = "dvd+rw-mediainfo";
	args[pos++] = file;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!        DVD+RW-MEDIAINFO           !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
GString *GetInfo_cdplay (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);

	args[pos++] = "cdplay";
	args[pos++] = "-d";
	args[pos++] = file;
	args[pos++] = "info";
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!           C D P L A Y             !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
GString *GetInfo_which (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);

	args[pos++] = "which";
	args[pos++] = file;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!             W H I C H             !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
void GetInfo_eject (gchar *StrDevice)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;

	args[pos++] = "eject";
	args[pos++] = "--cdrom";
	args[pos++] = StrDevice;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!              E J E C T            !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDERR_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
	} while (size != 0);

	close(fd);
}
// 
// 
GString *GetInfo_cdparanoia (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);

	args[pos++] = "cdparanoia";
	args[pos++] = "-Q";
	args[pos++] = "-d";
	args[pos++] = file;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!       C D P A R A N O I A         !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDERR_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
GString *GetInfo_metaflac (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);

	args[pos++] = "metaflac";
	args[pos++] = "--show-total-samples";
	args[pos++] = "--show-sample-rate";
	args[pos++] = file;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!         M E T A F L A C           !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
	} while (size != 0);

	close(fd);
	return (gstr);
}
// 
// 
gboolean GetInfo_wget (void)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	gboolean bool_access_ok = TRUE;

	args[pos++] = "wget";
	args[pos++] = "--user-agent=\"Mozilla 22.0\"";
	args[pos++] = "--directory-prefix=/tmp/";
	args[pos++] = "http://www.google.fr/index.html";
	args[pos++] = NULL;
	args[pos++] = NULL;

	/*g_print ("!-----------------------------------!\n");
	g_print ("!             W G E T               !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDERR_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		
		if (NULL != strstr (buf, "Not Found")) {
			bool_access_ok = FALSE;
		}
		
	} while (size != 0);

	close(fd);
	g_unlink ("/tmp/index.html");
	
	return (bool_access_ok);
}
// 
// 
SHNTAG *GetInfo_free_shntool (SHNTAG *ShnTag)
{
	if (NULL != ShnTag) {
		if (NULL != ShnTag->time)	{ g_free (ShnTag->time);	ShnTag->time = NULL; }
		if (NULL != ShnTag->size)	{ g_free (ShnTag->size);	ShnTag->size = NULL; }

		g_free (ShnTag);
		ShnTag = NULL;
	}

	return ((SHNTAG *)NULL);

}
SHNTAG *GetInfo_shntool (gchar *file)
{
	gint     fd;
	gint     pos = 0;
	gchar   *args [ 10 ];
	gchar    buf [ GET_MAX_CARS + 10 ];
	gint     size = 0;
	GString *gstr = g_string_new (NULL);
	SHNTAG	*ShnTag = NULL;
	gchar	*Ptr = NULL;

	guint	 Min;
	guint	 Sec;
	guint	 Hundr;

	args[pos++] = "shntool";
	args[pos++] = "info";
	args[pos++] = file;
	args[pos++] = NULL;

	/*
	g_print ("!-----------------------------------!\n");
	g_print ("!          S H N T O O L            !\n");
	g_print ("!-----------------------------------!\n");
	for (pos = 0; args[ pos ] != NULL; pos++)
		g_print ("args [ %02d ] = '%s'\n", pos, args [ pos ]);
	g_print ("\n");
	*/

	fd = GetInfo_exec_with_output (args, &VarGet.code_fork, STDOUT_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= GET_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= GET_MAX_CARS(%d)\n", pos, GET_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);

		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		if (pos > 0 ) {
			buf[pos ++ ] = '\n';
			buf[pos] = '\0';
			g_string_append_printf (gstr, "%s", buf);
		}
	} while (size != 0);

	close(fd);

	ShnTag = g_malloc0 (sizeof (SHNTAG));

	if ((Ptr = strstr (gstr->str, "Length:")) != NULL) {
		Ptr = strchr (Ptr, ':');
		Ptr ++;
		while (*Ptr == ' ') Ptr ++;

		Min = atoi (Ptr);

		Ptr = strchr (Ptr, ':');
		Ptr ++;
		Sec = atoi (Ptr);

		Ptr = strchr (Ptr, '.');
		Ptr ++;
		Hundr = atoi (Ptr);

		ShnTag->SecTime = (gdouble)(Min * 60) + (gdouble)(Sec) + (gdouble)(Hundr / 10.0);

		ShnTag->time = g_strdup_printf ("%02d:%02d", Min, Sec);

		if ((Ptr = strchr (ShnTag->time, '.')) != NULL) {
			*Ptr = '\0';
		}
	}

	ShnTag->size = g_strdup_printf ("%d Ko", (gint)libutils_get_size_file (file) / 1024);

	g_string_free (gstr, TRUE);

	return ((SHNTAG *)ShnTag);
}
// 
// 
TYPE_FILE_IS GetInfo_file_is (gchar *PathName)
{
	GString		*gstr = NULL;
	TYPE_FILE_IS	TypeFileRet = FILE_IS_NONE;
	
	gstr = GetInfo_file (PathName);
	if (*gstr->str) {
		if (FileIs_aiff (PathName)) {
			TypeFileRet = FILE_IS_AIFF;
		}
		else if (FileIs_shn (PathName)) {
			TypeFileRet = FILE_IS_SHN;
		}
		else if (FileIs_mpc (PathName)) {
			TypeFileRet = FILE_IS_MPC;
		}
		else if (strstr (gstr->str, "AAC")) {
			TypeFileRet = FILE_IS_AAC;
		}
		else if (FileIs_ac3 (PathName)) {
			TypeFileRet = FILE_IS_AC3;
		}
		else if (FileIs_ape (PathName)) {
			TypeFileRet = FILE_IS_APE;
		}
		else if (FileIs_dts (PathName)) {
			TypeFileRet = FILE_IS_DTS;
		}
		else if (strstr (gstr->str, "WAVE audio")) {
			TypeFileRet = FILE_IS_WAV;
		}
	
		/*	XWD X Window Dump image data
		 *	
		 *	http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=511764
		 */
		else if (strstr (gstr->str, "MP3 file") ||
			 strstr (gstr->str, "MP3 encoding") ||
			 strstr (gstr->str, "MPEG ADTS, layer III") ||
			 strstr (gstr->str, "XWD X Window Dump image data")) {
			 TypeFileRet = FILE_IS_MP3;
			 PRINT("IS MP3");
		}
		else if (strstr (gstr->str, "Ogg data") && strstr (gstr->str, "Vorbis audio")) {
			TypeFileRet = FILE_IS_OGG;
		}
		else if (strstr (gstr->str, "ISO Media, MPEG v4 system")) {
			TypeFileRet = FILE_IS_M4A;
		}
		// else if (strstr (gstr->str, "MPEG ADTS, AAC, v4 LC")) {
		// 	TypeFileRet = FILE_IS_AAC;
		// }
		else if (strstr (gstr->str, "Microsoft ASF")) {
			TypeFileRet = FILE_IS_WMA;
		}
		else if (FileIs_wavpack (PathName)) {
			TypeFileRet =  FILE_IS_WAVPACK;
		}
		else if (strstr (gstr->str, "RealMedia file")) {
			TypeFileRet = FILE_IS_RM;
		}
		else if (strstr (gstr->str, "FLAC audio bitstream data")) {
			TypeFileRet = FILE_IS_FLAC;
		}
		// else if (strstr (gstr->str, "data")) {
		// 	TypeFileRet = FILE_IS_DTS;
		// }
		else if (strstr (gstr->str, "image data") || strstr (gstr->str, "PNG image")) {
			TypeFileRet = FILE_IS_IMG;
		}
		else if (strstr (gstr->str, "HTML document text")) {
			TypeFileRet = FILE_IS_HTML;
		}
		else if (strstr (gstr->str, "ASCII") && strstr (gstr->str, "text")) {
			TypeFileRet = FILE_IS_TXT;
		}
		// else if (strstr (gstr->str, "ASCII") && strstr (gstr->str, "text") && infosong_test_xcfa_gen_conf (PathName)) {
		// 	TypeFileRet = FILE_IS_XCFA_GEN_CONF;
		// }
		else {
			TypeFileRet = FILE_IS_NONE;
		}
	}
	g_string_free (gstr, TRUE);
	gstr = NULL;
	
	if (FILE_IS_NONE == TypeFileRet) {
		if (FileIs_ac3 (PathName)) {
			TypeFileRet = FILE_IS_AC3;
		}
		else if (FileIs_ape (PathName)) {
			TypeFileRet = FILE_IS_APE;
		}
		else if (FileIs_mp3 (PathName)) {
			TypeFileRet = FILE_IS_MP3;
		}
		else if (FileIs_aiff (PathName)) {
			TypeFileRet = FILE_IS_AIFF;
		}
		else if (FileIs_flac (PathName)) {
			TypeFileRet = FILE_IS_FLAC;
		}
		else if (FileIs_ape (PathName)) {
			TypeFileRet = FILE_IS_APE;
		}
		else if (FileIs_mpc (PathName)) {
			TypeFileRet = FILE_IS_MPC;
		}
		else if (FileIs_wav (PathName)) {
			TypeFileRet = FILE_IS_WAV;
		}
		else if (FileIs_flac (PathName)) {
			TypeFileRet = FILE_IS_FLAC;
		}
		else if (FileIs_shn (PathName)) {
			TypeFileRet = FILE_IS_SHN;
		}
	}
	
	return (TypeFileRet);
}





