/*
 * Program XBLAST V2.5.11 or higher
 * (C) by Oliver Vogel (e-mail: vogel@ikp.uni-koeln.de)
 * November 17th, 1997
 * started August 1993
 *
 * File: util.c
 * header file for util.c
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public Licences as by published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Publis License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: util.c,v 1.2 1999/07/28 19:28:51 xblast Exp $
 * $Log: util.c,v $
 * Revision 1.2  1999/07/28 19:28:51  xblast
 * environment check only on first call
 *
 * Revision 1.1  1998/01/03 14:08:10  xblast
 * Initial revision
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#define _UTIL_C
#include "util.h"

/* 
 * a simple 32Bit random generator 
 * it will also be used to sync client and server
 */
#define RANDOM_MAX 4294967296.0
static unsigned rnum;

/*
 * public function seed_random
 *   seeds the random number generator
 */
#ifdef __STDC__
void
seed_random (unsigned seed)
#else
void
seed_random (seed)
  unsigned seed;
#endif
{
  rnum = seed;
}

/*
 * public function get_random_seed
 *   get the seed of the random number generator
 */
#ifdef __STDC__
unsigned
get_random_seed (void)
#else
unsigned
get_random_seed ()
#endif
{
  return rnum;
}

/* 
 * local function random
 *   creates a 32 Bit pseudo random number
 */
#ifdef __STDC__
static unsigned 
my_random (void)
#else
static unsigned 
my_random ()
#endif
{
  rnum = 1664525L * rnum + 1013904223L;
  return rnum;
}

/*
 * public function random_number
 *  creates an integer random number between 0 and max_val-1;
 */
#ifdef __STDC__
int 
random_number (int max_val)
#else
int 
random_number (max_val)
  int max_val;
#endif
{
  return (int)(my_random()/RANDOM_MAX * max_val);
}


/*
 * public function: open_file
 *   find and open an xblast data file.
 */
#ifdef __STDC__
XBFile *
open_file (char *path,
	   char *name,
	   char *ext)
#else
XBFile *
open_file (path, name, ext)
  char *path, *name, *ext;
#endif
{
  XBFile *ptr;
  int i;
  struct stat stat_buf;
  static char fname[1024];
  static char *path_list[3] = {
    NULL,
    XBLASTDIR,
    ".",
  };

  /* alloc XBFile data structure */
  if (NULL == (ptr = malloc(sizeof(XBFile) ) ) ) {
    fprintf(stderr,"failed to alloc file data structure\n");
    return NULL;
  }

  /* check environment for xblast search path */
  if (NULL == path_list[0]) {
    path_list[0] = getenv("XBLASTDIR");
  }

  /* 
   * try to open file, first look in $XBLASTDIR, then in the X11-xblast dir
   * and last in . 
   */
  for (i=0; i<3; i++) {
    if (NULL != path_list[i]) {
      /* try to open file directly */
      sprintf(fname, "%s/%s/%s.%s", path_list[i], path, name, ext);
      if (NULL != (ptr->fp = fopen(fname,"r") ) ) {
	/* file opened succesfully, use fclose to close it */
	ptr->close = fclose;
	return ptr;
      } 
      /* try to open gnu-zipped file */
      sprintf(fname, "%s/%s/%s.%s.gz", path_list[i], path, name, ext); 
      if (-1 != stat(fname, &stat_buf) ) {
	sprintf(fname, "gzip -d -c %s/%s/%s.%s.gz 2>/dev/null", 
		path_list[i], path, name, ext);
	if (NULL != (ptr->fp = popen(fname,"r") ) ) {
	  /* pipe opened successfully , use pclose to close it */
	  ptr->close = pclose;
	  return ptr;
	}
      }
    }
  }
  
  free(ptr);
  return NULL;
}



/*
 * local function pbm_swap_bits
 *   swaps bit order of pbm data to xbm format
 */
#ifdef __STDC__
static void
pbm_swap_bits (unsigned char *pbm,
	       int width, int height)
#else
static void
pbm_swap_bits (pbm, width, height)
  unsigned char *pbm;
  int width, height;
#endif
{
  static unsigned *conv_table = NULL;
  int nbytes;

  if (conv_table == NULL) {
    unsigned i, j;
    unsigned *ptr;
    if (NULL == (conv_table = malloc (256*sizeof(unsigned) ) ) ) {
      fprintf (stderr, "%s: alloc failed\n", "pbm_swap_bits");
      exit(2);
    }

    for (i=0, ptr=conv_table; i<256; i++, ptr++) {
      *ptr = 0;
      for (j=0; j<8; j++) {
	if (i & (1<<j)) {
	  *ptr |= (128>>j);
	}
      }
    }
  }

  nbytes = (width*height+7)/8;

  for (;nbytes > 0; nbytes--, pbm++) {
    *pbm = conv_table[*pbm];
  }
}



/*
 * public function read pbm file
 * load portable bitmap into memory
 */
#ifdef __STDC__
unsigned char *
read_pbm_file (char *path, 
	       char *filename,
	       int *width, int *height)
#else
unsigned char *
read_pbm_file (path, filename, width, height)
  char *path, *filename;
  int *width, *height;
#endif
{
  XBFile *fp = NULL;
  unsigned char *pbm = NULL;
  static char zeile [1024];
  int pbm_size;

  if (NULL == (fp = open_file(path, filename, "pbm") ) ) {
    fprintf(stderr, "failed to open file \"%s/%s.%s\".\n",
	    path, filename, "pbm");
    return NULL;
  }

  /* read header */
  if (3 != fscanf(fp->fp, "%s%d%d%*d%*c", zeile, width, height) ) {
    fprintf(stderr, "Failed to read ppm header\n");
    goto Error;
  }

  /* set size */
  pbm_size = ( (*width) * (*height) + 7) / 8;

  /* alloc data */
  if (NULL == (pbm = malloc( pbm_size * sizeof(char) ) ) )  {
    goto Error;
  }
    
  /* read data */
  if (pbm_size != fread (pbm, sizeof(char), pbm_size, fp->fp) ) {
    goto Error;
  }

  /* swap bit order */
  pbm_swap_bits (pbm, *width, *height);
  
  /* close file */
  close_file(fp);

  return pbm;

  /* error handling */
Error:
  if (fp != NULL) {
    close_file(fp);
  }
  if (pbm != NULL){
    free(pbm);
  }
  return NULL;
}



/*
 * public function read pbm file
 * load portable bitmap into memory
 */
#ifdef __STDC__
unsigned char *
read_ppm_file (char *path, 
	       char *filename,
	       int *width, int *height)
#else
unsigned char *
read_ppm_file (path, filename, width, height)
  char *path, *filename;
  int *width, *height;
#endif
{
  XBFile *fp = NULL;
  unsigned char *ppm = NULL;
  static char zeile [1024];
  int n_pixel;

  if (NULL == (fp = open_file(path, filename, "ppm") ) ) {
    fprintf(stderr, "failed to open file \"%s/%s.%s\".\n",
	    path, filename, "ppm");
    goto Error;
  }

  /* read header */
  if (3 != fscanf(fp->fp, "%s%d%d%*d%*c", zeile, width, height)) {
    fprintf(stderr, "Failed to read ppm header\n");
    goto Error;
  }
  n_pixel = 3*(*width)*(*height);

  /* alloc data */
  if (NULL == (ppm = malloc( n_pixel * sizeof(char) ) ) )  {
    goto Error;
  }
    
  /* read data */
  if (n_pixel != fread (ppm, sizeof(char), n_pixel, fp->fp) ) {
    goto Error;
  }
  
  /* close file */
  close_file(fp);

  return ppm;

  /* error handling */
Error:
  if (fp != NULL) {
    close_file(fp);
  }
  if (ppm != NULL){
    free(ppm);
  }
  return NULL;
}



/*
 * public function read_epm_file
 * load  Extended Pixmap (.epm) into memory
 */
#ifdef __STDC__
unsigned char *
read_epm_file (char *path, 
	       char *filename,
	       int *width, int *height, int *depth)
#else
unsigned char *
read_epm_file (path, filename, width, height, depth)
  char *path, *filename;
  int *width, *height, *depth;
#endif
{
  XBFile *fp = NULL;
  unsigned char *epm = NULL;
  unsigned char *buf = NULL;
  static char zeile [1024];
  int n_pixel;

  if (NULL == (fp = open_file(path, filename, "epm") ) ) {
    fprintf(stderr, "failed to open file \"%s/%s.%s\".\n", path,filename, "epm");
    return NULL;
  }

  /* read header */
  if (4 != fscanf(fp->fp, "%s%d%d%*d%d%*c", zeile, width, height, depth) ) {
    fprintf(stderr, "Failed to read epm header\n");
    goto Error;
  }
  /* calc number of pixels */
  n_pixel = (*depth)*(*width)*(*height);
  /* alloc data */
  if (NULL == (epm = malloc( n_pixel * sizeof(char) ) ) ) {
    goto Error;
  }
  /* check magic */  
  if (0 == strcmp (zeile, "PX") ) {
    /* uncompressed data */
    /* read data */
    if (n_pixel != fread (epm, sizeof(char), n_pixel, fp->fp) ) {
      goto Error;
    }
  } else if (0 == strcmp(zeile, "PZ") ) {
    /* compressed data */
    int i, j, n_bytes, zero_count;
    /* alloc input buffer */
    if (NULL == (buf = malloc(2 * n_pixel * sizeof(char) ) ) ) {
      goto Error;
    }
    if (0 == (n_bytes = fread (buf, sizeof(char), 2*n_pixel, fp->fp) ) ) {
      goto Error;
    }
    for (i=0, j=0; i<n_bytes; i++) {
      if (buf[i]) {
	epm[j] = buf[i];
	j++;
      } else {
	zero_count = 0;
	do {
	  i++;
	  zero_count += buf[i];
	} while (buf[i] == 255);
	memset (epm+j, 0, zero_count);
	j += zero_count;
      }
    }
  } else {
    /* wrong magic word */
    fprintf(stderr, "Wrong magic word \"%s\".\n", zeile);
    goto Error;
  }

  
  /* close file */
  close_file(fp);

  return epm;
  
  /* Error handling */
Error:
  if (fp != NULL) {
    close_file(fp);
  }
  if (epm != NULL) {
    free(epm);
  }
if (buf != NULL) {
  free(buf);
} 
  return NULL;
}



/*
 * end of file util.h
 */
