/*  Wordtrans, front-end for several dictionaries, for the Qt toolkit.
    Copyright (C) 2000-2004 Ricardo Villalba <rvm@escomposlinux.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <qglobal.h>
#include "QDiccionarios.h"
#include "QTextDiccionario.h"
#include "QBabyDiccionario.h"
#include "QDictDiccionario.h"
#include "qfunciones.h"
#include "intern.h"
#include <qdir.h>
#include <stdio.h>

#ifndef NO_GRAPHICS_AT_ALL
#include <qpopupmenu.h>
#include "pixmaps/tipo_texto_mini.xpm"
#include "pixmaps/tipo_babylon_mini.xpm"
#include "pixmaps/tipo_dict_mini.xpm"

#if QT_VERSION >= 300
#include "pixmaps/selected.xpm"
#endif

#endif

QDiccionarios::QDiccionarios(const QString & defaultDir) {
#ifndef NO_GRAPHICS_AT_ALL
	icono_texto=NULL;
	icono_babylon=NULL;
	icono_dict=NULL;
#endif

	if (defaultDir!=QString::null) {
		allow_save_configs=false;
		directorio=defaultDir;
		debug( "Using %s as the configuration directory", directorio.latin1());
	} else {
		allow_save_configs=true;
		directorio=qNomPersonal(".wordtrans");
		QDir dir( directorio );

		if (!dir.exists()) {
			debug( "Directory %s doesn't exist. I will create it", directorio.latin1() );
			if (!dir.mkdir( directorio ) ) {
				warning( "Directory %s cannot be created!", directorio.latin1() );
			}
		}

		copiar();
	}
}

QDiccionarios::~QDiccionarios() {
}

void QDiccionarios::copiar(const QString& filtro) {
	QDir d("/etc/wordtrans", filtro ); //"*.conf");

	if (!d.exists() ) {
		debug( "Directory /etc/wordtrans doesn't exist. Exiting from copiar()");
		return;
	}

	d.setFilter( QDir::Files ); //| QDir::Hidden | QDir::NoSymLinks );
	//d.setSorting( QDir::Size | QDir::Reversed );
	const QFileInfoList *list = d.entryInfoList();
	QFileInfoListIterator it( *list );      // create list iterator
	QFileInfo *fi;                          // pointer for traversing

	QString nombre;
	QString destino;
	QFile dest;

	while ( (fi=it.current()) ) {           // for each file...
		//debug("QDiccionarios::copiar: %s", fi->absFilePath().data() );

		nombre=fi->fileName();
		destino=directorio + "/" + nombre;

		dest.setName( destino);
	
		//debug("Comprobando si existe %s...", destino.data() );

		if (!dest.exists()) {
			QCString comando;
			comando= "cp ";
			comando += fi->absFilePath().latin1();
			comando += " ";
			comando += destino.latin1();
			//debug("Comando: %s", comando.data() );
			debug( "Installing %s", destino.latin1() );
			ejecuta( comando.data() );
		}

		++it;                               // goto next list element
	}

}

void QDiccionarios::copiar() {
	copiar("*.conf");
	copiar("*.bconf");
	copiar("*.dconf");
}

void QDiccionarios::load(const QString& filtro, bool baby, bool dictd) {
	QDir d(directorio, filtro ); //"*.conf");
	d.setFilter( QDir::Files ); //| QDir::Hidden | QDir::NoSymLinks );
	//d.setSorting( QDir::Size | QDir::Reversed );
	const QFileInfoList *list = d.entryInfoList();
	QFileInfoListIterator it( *list );      // create list iterator
	QFileInfo *fi;                          // pointer for traversing

	while ( (fi=it.current()) ) {           // for each file...
		//debug("Fichero: %s", fi->absFilePath().data() );

		DiccionarioBase *dict;
		if (dictd)
			dict = new QDictDiccionario();
		else
		if (baby)
			dict = new QBabyDiccionario();
		else
			dict = new QTextDiccionario();

		dict->load( fi->absFilePath() );

		if (dict->getDeleted()) {
			printf( "Dictionary %s is marked as deleted. Not loaded!\n",
                    dict->getAlias().latin1() );
			delete dict;
		} else {
			lista_dict.append( dict );
		}

		if ( (baby) && (filtro=="*.baby") ) {
			printf( _("Warning: the extension '.baby' is obsolete. "
                      "Please, rename it to '.bconf'\n").latin1() );
		}

		++it;                               // goto next list element
	}

}

void QDiccionarios::load() {
	load("*.conf",false,false);
	load("*.bconf",true,false);
	load("*.dconf",false,true);

	load("*.baby",true,false);
}

void QDiccionarios::show_dicts() {
	puts( _("Available dictionaries:").latin1() );
	puts( _("ALIAS      w  c  i  g        FILE").latin1() );
//	puts( _("---------- -  -  -  -----------------"));

	DiccionarioBase *dic;
	QString full_words;
	QString case_sensitive;
	QString invertir;
	QString acentos;
	for ( dic=lista_dict.first(); dic != 0; dic=lista_dict.next() ) {
		if (dic->getPalabrasCompletas()) full_words="+"; else full_words="-";
		if (dic->getCaseSensitive()) case_sensitive="+"; else case_sensitive="-";
		if (dic->getInvertirResultados()) invertir="+"; else invertir="-";
		if (dic->getIgnorarAcentos()) acentos="+"; else acentos="-";

		switch ( dic->getTipo() ) {
			case DiccionarioBase::Dict_dict:
					{
					QDictDiccionario *d= (QDictDiccionario *)dic;
					printf("%-10s =  =  =  =  %s (dict)\n", 
							d->getAlias().latin1(), d->getDatabase().latin1() );
					}
					break;

			case DiccionarioBase::Babylon_dict:
					printf("%-10s =  =  =  =  %s (babylon)\n", 
							dic->getAlias().latin1(), dic->getDiccionario().latin1() );
					break;

			default : printf("%-10s %s  %s  %s  %s  %s\n", dic->getAlias().latin1(), 
						full_words.latin1(), case_sensitive.latin1(), invertir.latin1(),
						acentos.latin1(), dic->getDiccionario().latin1() );
		}
	}
}


void QDiccionarios::add(DiccionarioBase *d) {
	lista_dict.append(d);
}

DiccionarioBase * QDiccionarios::first() {
	return lista_dict.first();
}

DiccionarioBase * QDiccionarios::next() {
	return lista_dict.next();
}

DiccionarioBase * QDiccionarios::last() {
	return lista_dict.last();
}

DiccionarioBase * QDiccionarios::prev() {
	return lista_dict.prev();
}

DiccionarioBase * QDiccionarios::at( unsigned int index) {
	return lista_dict.at(index);
}

#ifndef NO_GRAPHICS_AT_ALL
QPixmap * QDiccionarios::getIconoTipo(DiccionarioBase *dic) {
	if (icono_texto==NULL) icono_texto=new QPixmap( tipo_texto_mini_xpm );
	if (icono_babylon==NULL) icono_babylon=new QPixmap( tipo_babylon_mini_xpm );
	if (icono_dict==NULL) icono_dict=new QPixmap( tipo_dict_mini_xpm );

	QPixmap *icono;
	switch ( dic->getTipo() ) {
		case DiccionarioBase::Babylon_dict: icono=icono_babylon; break;
		case DiccionarioBase::Dict_dict: icono=icono_dict; break;
		default: icono=icono_texto;
	}
	return icono;
}

QPopupMenu * QDiccionarios::creaMenu() {
	QPopupMenu *menu=new QPopupMenu();
	menu->setCheckable(true);

	addMenuItems(menu);

	return menu;
}

void QDiccionarios::addMenuItems(QPopupMenu *menu) {
	DiccionarioBase *dic;
	QString texto;
	for ( dic=first(); dic != 0; dic=next() ) {
		texto=dic->getFullName();
		/*
		texto = dic->getIdioma1();
		texto += " <-> ";
		texto += dic->getIdioma2();
		*/
		QPixmap *icono= getIconoTipo(dic);
#if QT_VERSION < 300
		menu->insertItem( *icono, texto );
#else
		QIconSet iconos( *icono );
		iconos.setPixmap( QPixmap( selected_xpm ), QIconSet::Small,
                          QIconSet::Normal, QIconSet::On );
		menu->insertItem( iconos, texto );
#endif
	}
}

void QDiccionarios::updateMenuItems(QPopupMenu *menu, int index) {
	DiccionarioBase *dic;
	int cuenta=0;
	for ( dic=first(); dic != 0; dic=next() ) {
		QString texto=dic->getFullName(); //dic->getIdioma1() + " <-> " + dic->getIdioma2();
		QPixmap *icono= getIconoTipo(dic);
#if QT_VERSION < 300
#define ICONO *icono
#else
		QIconSet iconos( *icono );
		iconos.setPixmap( QPixmap( selected_xpm ), QIconSet::Small,
                          QIconSet::Normal, QIconSet::On );
#define ICONO iconos
#endif
		int id=menu->idAt( cuenta+index );
		if (id==-1) {
			menu->insertItem( ICONO, texto );
		} else {
			menu->changeItem( id, ICONO, texto );
		}
		cuenta++;
	}
}

/*
void QDiccionarios::updateMenuItem( QPopupMenu *menu, int index ) {
	DiccionarioBase *dic = at( index );
	QString texto=dic->getFullName();
	QPixmap *icono= getIconoTipo(dic);
	int id=menu->idAt( index );
	if (id==-1) { 
		debug("QDiccionarios::updateMenuItem: item %d no existe!", index);
		return;
	}
	menu->changeItem( id, *icono, texto);
}
*/
#endif

void QDiccionarios::save_all_dicts() {
	if (allow_save_configs) {
		debug( "Saving dictionaries" ); //"Guardando estado diccionarios");
		DiccionarioBase *dic;
		for ( dic=first(); dic != 0; dic=next() ) {
			dic->save();
		}
	} else {
		debug( "Not saving any dictionary" ); 
	}
}

DiccionarioBase * QDiccionarios::find( const QString& alias, int &index ) {
	DiccionarioBase *dic;
	for ( dic=first(); dic != 0; dic=next() ) {
		//if (strcmp( dic->getAlias().latin1(), alias.latin1() )==0) {
		if ( dic->getAlias() == alias ) {
			index=lista_dict.at();
			return dic;
		}
	}
	return NULL;
}
