%{

// Note about implementation:
//
//  * We number the colums in the following way:
//      The colum numbers [0..clientcolc[ are the client colums
//      The colum numbers [clientcolc..clientcolc+servercolc[ are 
//      the server colums.
//      

// Define this to print some debug info to stderr      
//#define PRINT_DEBUG_INFO

#include <stdio.h>
#include <cmath>
#include <stdlib.h> // Might be used by yacc

#include "common.h"
#include "wiplcexpr.h"

int yyparse();
void yy_scan_string(const char*);
void yyerror(char*);
int yylex();          // for Debian slink.

//-----------------------------------------------------------------------------
// Generel-purpose classes and definitions.

int max(int a,int b) { return a>b ? a : b; }

// This represents an array with a fixed size and with elments of type T.
// The array supports a "shift-all-elements-to-the-right" operation.
template<class T> class CycArray {
  T* array;
  int len;
  int startpos;

  public:
  // Creates the array with the given size:
  CycArray(int len) {
    array=new T[this->len=len];
    startpos=0;
  }
  
  // All elements are moved one position in the direction of the end of the 
  // array. The last element becomes the first element:
  void shift() {
    startpos--;
    if(startpos<0) startpos=len-1;
  }
  
  int size() {
    return len;
  }
  
  T& operator [] (int idx) {
    return array[(idx+startpos)%len];
  }
};

//-----------------------------------------------------------------------------
// The Dependency class
//
// This class represents what dependencies a given expression has on time 
// values and old colums:
class Dependency {   
  // All these are indexed by a colum number:
  
  int* Keep_col; // Keep_col[n] is the highest number of iterations back
                 // that this expression is depend on the values in colum n.
		 // If this expression is not at alle depend on colum n
		 // then Keep_col[n]==-1. If it is depend on the value
		 // from the same iteration it is 0, if depend on the value
		 // from the previous iteration it is 1 an so on.
  int* ColDep;   // 0 for colums we are depend on in the same iteration
                 // and -1 else. Is used to determine in which order the
		 // different expressions should be evaluated.
  int* Start_col;// Start_col[n] is 0 if this expression is depend on the
                 // the values of colum n as they were when the client
		 // started execurting. I.e.: As they were on the first
		 // iteration. If Start_col[n]!=0 it is -1
  int totalcolc; // Number of client and server colums and number of
                 // elements in the arrays above.
  int Keep_t;    // On how many iterations back we are depend on
                 // time values. Can be -1, 0 or >0.		 
  
  public:
  Dependency();
  ~Dependency() { delete[] Keep_col; delete[] ColDep; delete[] Start_col; }
  
  // Extend what we are depend on to what the argument is depend on:
  void addDependencies(Dependency&);
  
  // Add dependenci to colum x n iterations ago:
  void add_dep_col(int x,int n) { Keep_col[x]=max(Keep_col[x],n); 
                                  if(n==0) ColDep[x]=0; 
		   	        }

  // Add dependcy to the time values n iteration ago:				
  void add_dep_t(int n)         { Keep_t=max(Keep_t,n); }
  
  // Add dependency to the initial value of colum x:
  void add_dep_b(int x)         { Start_col[x]=0; }
  
  // Give information on the current dependecies. See description of
  // the returned variabels above:
  int* get_col_dep() { return ColDep; }
  int* keep_col()    { return Keep_col; }
  int* start_col()   { return Start_col; }
  int  keep_t()      { return Keep_t; }
  
  // Print debug information:
  #ifdef PRINT_DEBUG_INFO
    void print();
  #endif  
};

#ifdef PRINT_DEBUG_INFO
void Dependency::print() {
  fprintf(stderr,"    Time depened          : %i\n",Keep_t);

  fprintf(stderr,"    Colums same iteration :");
  for(int i=0; i<totalcolc; i++) if(ColDep[i]==0) fprintf(stderr," %i",i);
  fprintf(stderr,"\n");

  fprintf(stderr,"    Colums first iteration :");
  for(int i=0; i<totalcolc; i++) if(Start_col[i]==0) fprintf(stderr," %i",i);
  fprintf(stderr,"\n");

  fprintf(stderr,"    Colums other iteration:");
  for(int i=0; i<totalcolc; i++) {
    if(Keep_col[i]!=-1) fprintf(stderr," %i(back:%i)",i,Keep_col[i]);
  }
  fprintf(stderr,"\n");
}
#endif

Dependency::Dependency() {
  totalcolc=WiplcExpr::instance->totalcolc;
  Keep_col=new int[totalcolc];
  ColDep=new int[totalcolc];
  Start_col=new int[totalcolc];
  for(int i=0; i<totalcolc; i++) { 
    Keep_col[i]=-1;
    ColDep[i]=-1;
    Start_col[i]=-1;
  }
  Keep_t=-1;
}

void Dependency::addDependencies(Dependency& d) {
  int i,j;
  
  // Merge col information:
  for(j=0; j<totalcolc; j++) Keep_col[j]=max(Keep_col[j],d.Keep_col[j]);

  // Merge colDep information:
  for(j=0; j<totalcolc; j++) ColDep[j]=max(ColDep[j],d.ColDep[j]);

  // Merge Start_col information:
  for(j=0; j<totalcolc; j++) Start_col[j]=max(Start_col[j],d.Start_col[j]);
  
  // Merge t information:
  Keep_t=max(Keep_t,d.Keep_t);
}

//---------------------------------------------------------------------------
// The ValueServer class
//
// This class remembers all information about old colums etc.

typedef CycArray<double*> ColumArray; // A colum with values from several
                                      // iterations.
typedef CycArray<Time>    TimeArray;  // Times as they were on different
                                      // iterations.

class ValueServer {
  // The following are used to save information for alle the old and
  // new colums etc. We only save information which is needed.

  ColumArray** saved_cols;    // The colums we have. Is 0 for colums
                              // we don't save information about (i.e.
			      // unused server-colums).  
  TimeArray* saved_times;     // The times we have saved. 0 if we don't
                              // save time values.
  double**   saved_first;     // The contents of the conters on the first
                              // iteration. 0 for colums for which we have
			      // not saved this value.
  
  // This contain the adresses coresponding to the colums saved in saved_cols
  // and saved_first:
  mac_addr* macaddr;
  ip_addr*  ipaddr; 
  int       cardc; // And this the number of elements in all these tabels
  
  // Are we in the first iteration?
  int first_iteration;
  
  // Static information:
  card_id_type cit;
  int          clientcolc,servercolc,totalcolc;
  
  // These mainpulates saved_cols and saved_first:
  
  // Copy row i to row j
  void copy_row(int i,int j);
  
  // This is called when a new network card is discoverd. It initiliazes
  // the card to be on row i. If server_values is != 0 saved_first is
  // initialized with that values. Else saved_first is initialized to 0.
  //
  // (server_values should only be != 0 in the first iteration since cards not
  //  in the table should be implicitly assumed to be there but have
  //  all counters set to 0).
  void new_card(int i, const int64* server_values);
  
  // Update the given row with the given information from the server:
  void set_latest_row(int i, const int64* server_values, ip_addr, mac_addr);  
  
  public:
  // These are used when by expressions when they are asked to evaluate
  // themself. They correspond to user functions:  
  double* col(int x,int n) { return (*saved_cols[x])[n]; }
  Time    t(int n)         { return (*saved_times)[n];   }
  double* b(int x)         { return saved_first[x];      }
  
  // Initialize the server.
  //
  // The dependency information isused  to find out which colums etc this
  // class must save:
  ValueServer(Dependency&, int maxcards, int clientcolc, int servercolc, 
              card_id_type cit);
  
  // Every time we start a new period(=iteration) this should be called. It 
  // will update all the values represented by this class:
  void newPeriod(const ServerMem*, Time t);
  
  // This will return the position of the latest version of the colum given
  // as parameter. The user is supposed to evalute the colum expressions and
  // put them directly into this array:
  double* getColum(int i) {
    return col(i,0);
  }
  
  // Return infromation about mac/ip adresses:
  mac_addr* getMacColum() {
    return macaddr;
  }
  ip_addr* getIpColum() {
    return ipaddr;
  }
};

ValueServer::ValueServer(Dependency& d, int maxcardc, int clientcolc, int servercolc, card_id_type cit) {
  // Basic initialization:
  this->cit=cit;
  this->clientcolc=clientcolc;
  this->servercolc=servercolc;
  this->totalcolc=clientcolc+servercolc;
  cardc=0;
  ipaddr=new ip_addr[maxcardc];   
  macaddr=new mac_addr[maxcardc]; 
  first_iteration=1;

  // Allocate all colums and initialize all values in them to 0.
  saved_cols=new ColumArray*[totalcolc];  
  int* cols=d.keep_col();
  for(int i=0; i<totalcolc; i++) {
    if(cols[i]!=-1) { // This colum is used by someone
      saved_cols[i]=new ColumArray(cols[i]+1); 
      for(int j=0; j<=cols[i]; j++) {
        double* d=new double[maxcardc];	
        for(int k=0; k<maxcardc; k++) d[k]=0.0;
        (*saved_cols[i])[j]=d;
      }
    } else {
      saved_cols[i]=0;
    }
  }
  
  // Allocate room for the first colums:
  saved_first=new double*[totalcolc];
  for(int k=0; k<totalcolc; k++) 
    if(d.start_col()[k]!=-1)
      saved_first[k]=new double[maxcardc];
    else
      saved_first[k]=0;
  
  // And allocate the timearray and set default values:
  if(d.keep_t()!=-1) {
    saved_times=new TimeArray(d.keep_t()+1); // constructor sets default time to 0
  } else
    saved_times=0;
}

inline void ValueServer::copy_row(int i,int j) {
  for(int a=0; a<totalcolc; a++) {
    ColumArray* c=saved_cols[a];
    if(!c) continue;
    int d=c->size();
    for(int b=0; b<d; b++) {
      double* e=(*c)[b];      
      e[j]=e[i];
    }
    if(saved_first[a]) {
      saved_first[a][j]=saved_first[a][i];
    }
  }
  ipaddr[j]=ipaddr[i];
  macaddr[j]=macaddr[i];
}

inline void ValueServer::new_card(int i, const int64* counters) {
  for(int a=0; a<totalcolc; a++) {
    ColumArray* c=saved_cols[a];
    if(c) {
      int d=c->size();
      for(int b=0; b<d; b++) {
        double* e=(*c)[b];      
        e[i]=0.0;
      }
    }
    if(saved_first[a]) {
      if(a>=clientcolc && counters!=0)
        saved_first[a][i]=counters[a-clientcolc];
      else
        saved_first[a][i]=0;
    }
  }
}

inline void ValueServer::set_latest_row(int r, const int64* rows, 
                                        ip_addr ip, mac_addr mac) {
  for(int i=clientcolc; i<totalcolc; i++) {
     ColumArray* c=saved_cols[i];
     if(c) {
       (*c)[0][r]=rows[i-clientcolc];
     }
  }
  ipaddr[r]=ip;
  macaddr[r]=mac;
}

void ValueServer::newPeriod(const ServerMem* sm, Time time) {
  // Move everything one step:
  if(saved_times) {
    saved_times->shift();
    (*saved_times)[0]=time; // And update time
  }
  for(int i=0; i<totalcolc; i++) {
    if(saved_cols[i]) saved_cols[i]->shift();
  }
    
  // And copy in the new values. This is a bit complicated since cards might
  // have been inserted or deleted in the server. And in this situation we 
  // must do the same insertion / deltion in all tabels.

  // First we copy the new value of the counters and delete the cards that
  // are no longer in the server:
  int t=0;    // Index in our table we copy from
  int tnew=0; // Index in our table we copy to
  int d=0;    // Index in the daemon area.
  
  while(t<cardc || d<sm->curcardc) {
    // Set dif to the same sign as:
    //   <card id for table> minus <card id for daemon area>
    int dif;
    if(t==cardc)
      dif=1;
    else if(d==sm->curcardc) 
      dif=-1;
    else {
      if(cit==idIP) {
        dif=ip_addr::compare(ipaddr[t],sm->cards[d].ipaddr);
      } else {
        dif=mac_addr::compare(macaddr[t],sm->cards[d].macaddr);
      }
    }
    
    if(dif==0) {
      if(tnew!=t) {
        // Copy card to new location:
        copy_row(t,tnew);      
      }
      
      // We are at the same card - copy information to the row:
      set_latest_row(tnew,sm->cards[d].Counters,sm->cards[d].ipaddr,
                                                sm->cards[d].macaddr);
      
      tnew++; 
      d++; 
      t++;
    } else if(dif<0) {
      // A card in the table should be deleted
      t++;       
    } else if(dif>0) {
      // A card in the daemon should be inserted
      d++;
    }
  }
  
  cardc=tnew;
  
  // Now insert the new cards if any. Note that at this point all cards in 
  // the client area are also in the server area.

  t=cardc-1;        // Copy from this place in client
  d=sm->curcardc-1; // Copy to this place in client / 
                    //   from this place in server
    
  for(;(t!=d);d--) { // Walk through the server table        
    assert(d>t);
    
    // Set dif as above:
    int dif;
    if(t<0) {
      dif=-1;
    } else if(cit==idIP) {
      dif=ip_addr::compare(ipaddr[t],sm->cards[d].ipaddr);
    } else {
      dif=mac_addr::compare(macaddr[t],sm->cards[d].macaddr);
    }
    
    if(dif==0) {
      // We are at the samme card - so just move if neccesarry
      copy_row(t,d);	
      t--;
    } else {
      // We need to insert a card from the server in the client
      assert(dif<0);
      new_card(d,first_iteration ? sm->cards[d].Counters : 0);
      set_latest_row(d,sm->cards[d].Counters,sm->cards[d].ipaddr,
                                             sm->cards[d].macaddr);
    }
  }
  
  cardc=sm->curcardc;
  first_iteration=0;
}

//---------------------------------------------------------------------------
// Expressions.
//
// Note: 
//   Everything calculated in the program is calculated in the same
//   way in every colum. For efficiency reasons we calculate values for 
//   a whole colum at a time (and hope the cache-miss penalty is not
//   too big...).

// This represents an expression on an _array_ of floating point values
// (i.e. for a whole colum):
class FloatExpr {  
  public:  
  // Every expression should set this in it's construtor. In this
  // way dependency information for expressions can be calculated
  // button-up.
  Dependency dep;
  
  // This should evaluate the expression for the first numcards elements
  // of the array:
  virtual double* eval(int numcards)=0;
};  

// An expression with a constant element on all fields:
class ConstExpr:public FloatExpr {
  double* val;

  // This will evaluate the expression for the given number of elements
  // in the array:
  virtual double* eval(int numcards) {
    return val;
  }
  
  public:
  ConstExpr(double value) {
    int i=WiplcExpr::instance->maxcardc;
    val=new double[i];
    for(int a=0; a<i; a++) val[a]=value;
  }
};

// A generic binary operator with two floating point operands
class BinopExpr:public FloatExpr {
  protected:
  double* result;
  FloatExpr &left,&right;

  public:
   
  BinopExpr(FloatExpr& Left, FloatExpr& Right) : left(Left), right(Right) {
    dep.addDependencies(Left.dep);
    dep.addDependencies(Right.dep);
    result=new double[WiplcExpr::instance->maxcardc];
  }
};

class AddExpr:public BinopExpr {
  virtual double* eval(int numcards) {
    double* Left=left.eval(numcards);
    double* Right=right.eval(numcards);
    for(int i=0; i<numcards; i++) {
      result[i]=Left[i]+Right[i];
    }  
    return result;
  }
  public:
  AddExpr(FloatExpr& Left,FloatExpr& Right) : BinopExpr(Left,Right) {}
};

class SubExpr:public BinopExpr {
  virtual double* eval(int numcards) {
    double* Left=left.eval(numcards);
    double* Right=right.eval(numcards);
    for(int i=0; i<numcards; i++) {
      result[i]=Left[i]-Right[i];
    }  
    return result;
  }
  public:
  SubExpr(FloatExpr& Left,FloatExpr& Right) : BinopExpr(Left,Right) {}
};
  
class MulExpr:public BinopExpr {
  virtual double* eval(int numcards) {
    double* Left=left.eval(numcards);
    double* Right=right.eval(numcards);
    for(int i=0; i<numcards; i++) {
      result[i]=Left[i]*Right[i];
    }  
    return result;
  }
  public:
  MulExpr(FloatExpr& Left,FloatExpr& Right) : BinopExpr(Left,Right) {}
};

class DivExpr:public BinopExpr {
  virtual double* eval(int numcards) {
    double* Left=left.eval(numcards);
    double* Right=right.eval(numcards);
    for(int i=0; i<numcards; i++) {      
      result[i]=Left[i]/Right[i];
    }  
    return result;
  }
  public:
  DivExpr(FloatExpr& Left,FloatExpr& Right) : BinopExpr(Left,Right) {}
};

// A generic binary operator with a floating point and an integer oprand.
class ShExpr:public FloatExpr {
  protected:
  double* result;
  FloatExpr &left;
  int right;

  public:
   
  ShExpr(FloatExpr& Left, int Right) : left(Left), right(Right) {
    dep.addDependencies(Left.dep);
    result=new double[WiplcExpr::instance->maxcardc];
  }
};

class ShlExpr:public ShExpr {
  virtual double* eval(int numcards) {
    double* Left=left.eval(numcards);
    for(int i=0; i<numcards; i++) {
      result[i]=ldexp(Left[i],-right);
    }  
    return result;
  }
  public:
  ShlExpr(FloatExpr& Left,int Right) : ShExpr(Left,Right) {}
};

class ShrExpr:public ShExpr {
  virtual double* eval(int numcards) {
    double* Left=left.eval(numcards);
    for(int i=0; i<numcards; i++) {
      result[i]=ldexp(Left[i],right);
    }  
    return result;
  }
  public:
  ShrExpr(FloatExpr& Left,int Right) : ShExpr(Left,Right) {}
};

// Expressions representing the different functions the user can use:
class tExpr:public FloatExpr {
  int n;
  double* val;

  virtual double* eval(int numcards) {
    double v=WiplcExpr::instance->vs->t(n).asDouble();
    for(int a=0; a<numcards; a++) val[a]=v;
    return val;
  }
  
  public:
    
  tExpr(int N) {
    n=N;
    dep.add_dep_t(n);
    val=new double[WiplcExpr::instance->maxcardc];
  }  
};

class colExpr:public FloatExpr {
  int x,n;

  virtual double* eval(int numcards) {
    return WiplcExpr::instance->vs->col(x,n);
  }
  
  public:
    
  colExpr(int X,int N) {
    x=X; n=N;
    dep.add_dep_col(x,n);
  }  
};

class bExpr:public FloatExpr {
  int x;

  virtual double* eval(int numcards) {
    return WiplcExpr::instance->vs->b(x);
  }
  
  public:
    
  bExpr(int X) {
    x=X;
    dep.add_dep_b(x);
  }  
};

//---------------------------------------------------------------------------
// Grammar and semantic actions.

// Result of parsing is put here:
FloatExpr* result;

%}  

%union {
  FloatExpr* nfexpr;
  unsigned   nuintconst;
  float      nfloatconst;
}

%type <nfexpr>     fexpr
%type <nfexpr>     start
%type <nuintconst> uintconst

%token <nfexpr>      NUMFEXPR
%token <nuintconst>  UINTCONST
%token <nfloatconst> FLOATCONST

%token SHL SHR
%token ERROR

%left SHL SHR 
%left '+' '-' 
%left '*' '/' 

%%

start    : fexpr                               { result=$$;               }
fexpr    : '(' fexpr ')'                       { $$=$2;                   }
         | fexpr '+' fexpr                     { $$=new AddExpr(*$1,*$3); }
	 | fexpr '-' fexpr                     { $$=new SubExpr(*$1,*$3); }
	 | fexpr '*' fexpr                     { $$=new MulExpr(*$1,*$3); }
	 | fexpr '/' fexpr                     { $$=new DivExpr(*$1,*$3); }
	 | fexpr SHL uintconst                 { $$=new ShrExpr(*$1,$3);  }
	 | fexpr SHR uintconst                 { $$=new ShlExpr(*$1,$3);  }
	 | uintconst                           { $$=new ConstExpr($1);    }

	 | 'c' '(' uintconst ',' uintconst ')' { 
	         if($3>=WiplcExpr::instance->clientcolc) {
	           yyerror("Invalid client colum number"); YYERROR;
		 }	   
	         $$=new colExpr($3,$5);   }
	 | 'd' '(' uintconst ',' uintconst ')' { 
	         if($3>=WiplcExpr::instance->servercolc) {
	           yyerror("Invalid server counter number"); YYERROR;
		 }	   
	         $$=new colExpr($3+WiplcExpr::instance->clientcolc,$5); }

         | 'b' '(' uintconst ')' {
	         if($3>=WiplcExpr::instance->servercolc) {
	           yyerror("Invalid server counter number"); YYERROR;
		 }	   
	         $$=new bExpr($3+WiplcExpr::instance->clientcolc); }
		 	 		 
	 | 't' '(' uintconst ')' { $$=new tExpr($3);       }
	 | FLOATCONST            { $$=new ConstExpr($1);   }

uintconst: UINTCONST

%%

//-----------------------------------------------------------------------------
// The WiplcExpr class:

#include <assert.h>

// Compiletime error handling:
int anyerror;
int col;
void yyerror(char *s) {
  if(!anyerror) {
    snprintf(errmsg,sizeof(errmsg),"In expression for colum %i: %s",col,s);
    anyerror=1;
  }
}

WiplcExpr* WiplcExpr::instance=0;

WiplcExpr::WiplcExpr(int clientcolc, int servercolc, 
                     int maxcardc, card_id_type cit, int timebackdep) {
  assert(instance==0); // Only one instance is allowed because of global
                       // variabels.
  instance=this;		      
  this->clientcolc=clientcolc;
  this->servercolc=servercolc;
  this->totalcolc=clientcolc+servercolc;
  this->maxcardc=maxcardc;
  this->cit=cit;
  this->timebackdep=timebackdep;
  vs=0;
  colums=new FloatExpr*[clientcolc];
  colnums=new int[clientcolc];
  for(int i=0; i<clientcolc; i++) colums[i]=0;
}

WiplcExpr::~WiplcExpr() {
  instance=0;
}

char* WiplcExpr::setExpression(int colum, char* program) {
  col=colum;
  anyerror=0;
  
  // Parse program
  yy_scan_string(program);
  if(yyparse()) {
    if(!anyerror) yyerror("Parse error"); // make a default error message
    return errmsg;
  }
  
  // And set it:
  colums[colum]=result;
  
  return 0;
}

char* WiplcExpr::noMoreExpressions() {
  // Find common dependencies for all colums:
  Dependency dep;
  
  for(int i=0; i<clientcolc; i++) {
    if(colums[i]!=0) {
      // What this colum is depend on:
      dep.addDependencies(colums[i]->dep);
     
      // We are outself depend on the latest version of this colum because
      // we must be able to give it to the user of this class via the
      // getColum method:
      dep.add_dep_col(i,0);
    }
  }
  
  // Add dependency for the getTimeBack method:
  dep.add_dep_t(timebackdep);
  
  // Create the valueserver:
  vs=new ValueServer(dep,maxcardc,clientcolc,servercolc,cit);

  // Now reorder the colums variable to have the colums in an order
  // in which they can be evaluated. I.e.: Make a topological sort of the
  // dependenci graf:
  
  FloatExpr** result=new FloatExpr*[clientcolc+1];
  int resultc=0;
  
  for(;;) {
    // Try to find a colum we can evaluate:
    int i,j;
    for(i=0;i<clientcolc; i++) {
      // Check if we can evaluate colum i
      if(!colums[i]) continue;
      int* dep=colums[i]->dep.get_col_dep();
      for(j=0; j<clientcolc; j++) {
        if(!colums[j]) continue;
        if(dep[j]==0 && colums[j]!=0) 
          // Colum i is depend on coulm j which is not evalutated yet
  	  break;
      }
      if(j==clientcolc) break; // break if we can evaluate colum i
    }
    
    if(i==clientcolc) {
      // We could not find any colum to evaluate. See if there
      // is any colums left:
      for(int a=0; a<clientcolc; a++)
        if(colums[a]) { // There is      
          delete[] colums;
          return "Cyclic dependencies in colum expressions";
	}
	
      // All colums could be evaluated - save new value of colums:
      result[resultc++]=0;
      delete[] colums;
      colums=result;
      
      #ifdef PRINT_DEBUG_INFO      
        fprintf(stderr,"Colum order: ");
	for(int i=0; result[i]; i++) fprintf(stderr,"%i ",colnums[i]);
	fprintf(stderr,"\n");
	
	fprintf(stderr,"Colum dependencies:\n");
	for(int i=0; i<clientcolc; i++) {
	  if(colums[i]) {
    	    Dependency& dep=colums[i]->dep;
            fprintf(stderr,"  Colum %i:\n",colnums[i]);
	    dep.print();
	  }
	}
	fprintf(stderr,"  Total:\n");
	dep.print();
     #endif
      
      return 0;
    }
    
    // We can evalute colum i:
    colnums[resultc]=i;
    result[resultc++]=colums[i];
    
    // Remove it from list of unevaluated colums:
    colums[i]=0;
  }
}

void WiplcExpr::nextIteration(const ServerMem* sm, Time t) {
  // Shift old colums and copy in new values of the server:
  vs->newPeriod(sm,t);
  cardcntc=sm->curcardc;
};

void WiplcExpr::updateValues() {  
  // Calculate all expressions and give them to the value server:
  for(int i=0; colums[i]; i++) {
    memcpy(vs->getColum(colnums[i]),
           colums[i]->eval(cardcntc),
	   cardcntc*sizeof(double));
  }
}

const double* WiplcExpr::getColum(int n) {
  return vs->getColum(n);
}    

const mac_addr* WiplcExpr::getMacColum() {
  return vs->getMacColum();
}

const ip_addr* WiplcExpr::getIpColum() {
  return vs->getIpColum();
}

Time WiplcExpr::getTimeBack(int n) {
  return vs->t(n);
}
