/*
 * Copyright 2008 Henri Verbeet for CodeWeavers
 * Copyright 2015 Józef Kucia for CodeWeavers
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <assert.h>
#include <float.h>
#include <limits.h>
#include <math.h>
#include <stdlib.h>
#define COBJMACROS
#include "initguid.h"
#include "d3d11_4.h"
#include "wine/heap.h"
#include "wine/test.h"

#define BITS_NNAN 0xffc00000
#define BITS_NAN  0x7fc00000
#define BITS_NINF 0xff800000
#define BITS_INF  0x7f800000
#define BITS_N1_0 0xbf800000
#define BITS_1_0  0x3f800000

#define SWAPCHAIN_FLAG_SHADER_INPUT             0x1

static unsigned int use_adapter_idx;
static BOOL enable_debug_layer;
static BOOL use_warp_adapter;
static BOOL use_mt = TRUE;

static struct test_entry
{
    union
    {
        void (*test)(void);
        void (*test_fl)(D3D_FEATURE_LEVEL fl);
    } u;
    D3D_FEATURE_LEVEL fl;
} *mt_tests;
size_t mt_tests_size, mt_test_count;

struct format_support
{
    DXGI_FORMAT format;
    D3D_FEATURE_LEVEL fl_required;
    D3D_FEATURE_LEVEL fl_optional;
};

static const struct format_support display_format_support[] =
{
    {DXGI_FORMAT_R8G8B8A8_UNORM,             D3D_FEATURE_LEVEL_9_1},
    {DXGI_FORMAT_R8G8B8A8_UNORM_SRGB,        D3D_FEATURE_LEVEL_9_1},
    {DXGI_FORMAT_B8G8R8A8_UNORM,             D3D_FEATURE_LEVEL_9_1},
    {DXGI_FORMAT_B8G8R8A8_UNORM_SRGB,        D3D_FEATURE_LEVEL_9_1},
    {DXGI_FORMAT_R16G16B16A16_FLOAT,         D3D_FEATURE_LEVEL_10_0},
    {DXGI_FORMAT_R10G10B10A2_UNORM,          D3D_FEATURE_LEVEL_10_0},
    {DXGI_FORMAT_R10G10B10_XR_BIAS_A2_UNORM, D3D_FEATURE_LEVEL_11_0, D3D_FEATURE_LEVEL_10_0},
};

struct vec2
{
    float x, y;
};

struct vec3
{
    float x, y, z;
};

struct vec4
{
    float x, y, z, w;
};

struct ivec4
{
    int x, y, z, w;
};

struct uvec4
{
    unsigned int x, y, z, w;
};

struct device_desc
{
    const D3D_FEATURE_LEVEL *feature_level;
    UINT flags;
};

struct resource_desc
{
    D3D11_RESOURCE_DIMENSION dimension;
    unsigned int width;
    unsigned int height;
    unsigned int depth_or_array_size;
    unsigned int level_count;
    DXGI_FORMAT format;
    DXGI_SAMPLE_DESC sample_desc;
    D3D11_USAGE usage;
    unsigned int bind_flags;
    unsigned int cpu_access_flags;
    unsigned int misc_flags;
    unsigned int structure_byte_stride;
};

struct swapchain_desc
{
    BOOL windowed;
    unsigned buffer_count;
    unsigned int width, height;
    DXGI_SWAP_EFFECT swap_effect;
    DWORD flags;
};

static void queue_test_entry(const struct test_entry *t)
{
    if (mt_test_count >= mt_tests_size)
    {
        mt_tests_size = max(16, mt_tests_size * 2);
        mt_tests = heap_realloc(mt_tests, mt_tests_size * sizeof(*t));
    }
    mt_tests[mt_test_count++] = *t;
}

static void queue_test_fl(void (*test)(const D3D_FEATURE_LEVEL fl), D3D_FEATURE_LEVEL fl)
{
    struct test_entry t;

    t.u.test_fl = test;
    t.fl = fl;
    queue_test_entry(&t);
}

static void queue_test(void (*test)(void))
{
    struct test_entry t;

    t.u.test = test;
    t.fl = 0;
    queue_test_entry(&t);
}

static void run_mt_test(const struct test_entry *t)
{
    if (t->fl)
    {
        winetest_push_context("Feature level %#x", t->fl);
        t->u.test_fl(t->fl);
        winetest_pop_context();
    }
    else
    {
        t->u.test();
    }
}

static DWORD WINAPI thread_func(void *ctx)
{
    LONG *i = ctx, j;

    while (*i < mt_test_count)
    {
        j = *i;
        if (InterlockedCompareExchange(i, j + 1, j) == j)
            run_mt_test(&mt_tests[j]);
    }

    return 0;
}

static void run_queued_tests(void)
{
    unsigned int thread_count, i;
    HANDLE *threads;
    SYSTEM_INFO si;
    LONG test_idx;

    if (!use_mt)
    {
        for (i = 0; i < mt_test_count; ++i)
        {
            run_mt_test(&mt_tests[i]);
        }

        return;
    }

    GetSystemInfo(&si);
    thread_count = si.dwNumberOfProcessors;
    threads = heap_calloc(thread_count, sizeof(*threads));
    for (i = 0, test_idx = 0; i < thread_count; ++i)
    {
        threads[i] = CreateThread(NULL, 0, thread_func, &test_idx, 0, NULL);
        ok(!!threads[i], "Failed to create thread %u.\n", i);
    }
    WaitForMultipleObjects(thread_count, threads, TRUE, INFINITE);
    for (i = 0; i < thread_count; ++i)
    {
        CloseHandle(threads[i]);
    }
    heap_free(threads);
}

static void set_box(D3D11_BOX *box, UINT left, UINT top, UINT front, UINT right, UINT bottom, UINT back)
{
    box->left = left;
    box->top = top;
    box->front = front;
    box->right = right;
    box->bottom = bottom;
    box->back = back;
}

static ULONG get_refcount(void *iface)
{
    IUnknown *unknown = iface;
    IUnknown_AddRef(unknown);
    return IUnknown_Release(unknown);
}

#define check_interface(a, b, c, d) check_interface_(__LINE__, a, b, c, d)
static HRESULT check_interface_(unsigned int line, void *iface, REFIID riid, BOOL supported, BOOL is_broken)
{
    HRESULT hr, expected_hr, broken_hr;
    IUnknown *unknown = iface, *out;

    if (supported)
    {
        expected_hr = S_OK;
        broken_hr = E_NOINTERFACE;
    }
    else
    {
        expected_hr = E_NOINTERFACE;
        broken_hr = S_OK;
    }

    hr = IUnknown_QueryInterface(unknown, riid, (void **)&out);
    ok_(__FILE__, line)(hr == expected_hr || broken(is_broken && hr == broken_hr),
            "Got hr %#lx, expected %#lx.\n", hr, expected_hr);
    if (SUCCEEDED(hr))
        IUnknown_Release(out);
    return hr;
}

static BOOL compare_float(float f, float g, unsigned int ulps)
{
    int x = *(int *)&f;
    int y = *(int *)&g;

    if (x < 0)
        x = INT_MIN - x;
    if (y < 0)
        y = INT_MIN - y;

    if (abs(x - y) > ulps)
        return FALSE;

    return TRUE;
}

static BOOL compare_vec4(const struct vec4 *v1, const struct vec4 *v2, unsigned int ulps)
{
    return compare_float(v1->x, v2->x, ulps)
            && compare_float(v1->y, v2->y, ulps)
            && compare_float(v1->z, v2->z, ulps)
            && compare_float(v1->w, v2->w, ulps);
}

static BOOL compare_uint(unsigned int x, unsigned int y, unsigned int max_diff)
{
    unsigned int diff = x > y ? x - y : y - x;

    return diff <= max_diff;
}

static BOOL compare_uvec4(const struct uvec4* v1, const struct uvec4 *v2)
{
    return v1->x == v2->x && v1->y == v2->y && v1->z == v2->z && v1->w == v2->w;
}

static BOOL compare_color(DWORD c1, DWORD c2, BYTE max_diff)
{
    return compare_uint(c1 & 0xff, c2 & 0xff, max_diff)
            && compare_uint((c1 >> 8) & 0xff, (c2 >> 8) & 0xff, max_diff)
            && compare_uint((c1 >> 16) & 0xff, (c2 >> 16) & 0xff, max_diff)
            && compare_uint((c1 >> 24) & 0xff, (c2 >> 24) & 0xff, max_diff);
}

static char const *debugstr_viewport(D3D11_VIEWPORT *vp)
{
    if (!vp) return "(null)";
    return wine_dbg_sprintf("{%.8e, %.8e, %.8e, %.8e, %.8e, %.8e}",
            vp->TopLeftX, vp->TopLeftY, vp->Width, vp->Height, vp->MinDepth, vp->MaxDepth);
}

struct srv_desc
{
    DXGI_FORMAT format;
    D3D11_SRV_DIMENSION dimension;
    unsigned int miplevel_idx;
    unsigned int miplevel_count;
    unsigned int layer_idx;
    unsigned int layer_count;
};

static void get_srv_desc(D3D11_SHADER_RESOURCE_VIEW_DESC *d3d11_desc, const struct srv_desc *desc)
{
    d3d11_desc->Format = desc->format;
    d3d11_desc->ViewDimension = desc->dimension;
    if (desc->dimension == D3D11_SRV_DIMENSION_TEXTURE1D)
    {
        U(*d3d11_desc).Texture1D.MostDetailedMip = desc->miplevel_idx;
        U(*d3d11_desc).Texture1D.MipLevels = desc->miplevel_count;
    }
    else if (desc->dimension == D3D11_SRV_DIMENSION_TEXTURE1DARRAY)
    {
        U(*d3d11_desc).Texture1DArray.MostDetailedMip = desc->miplevel_idx;
        U(*d3d11_desc).Texture1DArray.MipLevels = desc->miplevel_count;
        U(*d3d11_desc).Texture1DArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture1DArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_SRV_DIMENSION_TEXTURE2D)
    {
        U(*d3d11_desc).Texture2D.MostDetailedMip = desc->miplevel_idx;
        U(*d3d11_desc).Texture2D.MipLevels = desc->miplevel_count;
    }
    else if (desc->dimension == D3D11_SRV_DIMENSION_TEXTURE2DARRAY)
    {
        U(*d3d11_desc).Texture2DArray.MostDetailedMip = desc->miplevel_idx;
        U(*d3d11_desc).Texture2DArray.MipLevels = desc->miplevel_count;
        U(*d3d11_desc).Texture2DArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture2DArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_SRV_DIMENSION_TEXTURE2DMSARRAY)
    {
        U(*d3d11_desc).Texture2DMSArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture2DMSArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_SRV_DIMENSION_TEXTURE3D)
    {
        U(*d3d11_desc).Texture3D.MostDetailedMip = desc->miplevel_idx;
        U(*d3d11_desc).Texture3D.MipLevels = desc->miplevel_count;
    }
    else if (desc->dimension == D3D11_SRV_DIMENSION_TEXTURECUBE)
    {
        U(*d3d11_desc).TextureCube.MostDetailedMip = desc->miplevel_idx;
        U(*d3d11_desc).TextureCube.MipLevels = desc->miplevel_count;
    }
    else if (desc->dimension == D3D11_SRV_DIMENSION_TEXTURECUBEARRAY)
    {
        U(*d3d11_desc).TextureCubeArray.MostDetailedMip = desc->miplevel_idx;
        U(*d3d11_desc).TextureCubeArray.MipLevels = desc->miplevel_count;
        U(*d3d11_desc).TextureCubeArray.First2DArrayFace = desc->layer_idx;
        U(*d3d11_desc).TextureCubeArray.NumCubes = desc->layer_count;
    }
    else if (desc->dimension != D3D11_SRV_DIMENSION_UNKNOWN
            && desc->dimension != D3D11_SRV_DIMENSION_TEXTURE2DMS)
    {
        trace("Unhandled view dimension %#x.\n", desc->dimension);
    }
}

#define check_srv_desc(a, b) check_srv_desc_(__LINE__, a, b)
static void check_srv_desc_(unsigned int line, const D3D11_SHADER_RESOURCE_VIEW_DESC *desc,
        const struct srv_desc *expected_desc)
{
    ok_(__FILE__, line)(desc->Format == expected_desc->format,
            "Got format %#x, expected %#x.\n", desc->Format, expected_desc->format);
    ok_(__FILE__, line)(desc->ViewDimension == expected_desc->dimension,
            "Got view dimension %#x, expected %#x.\n", desc->ViewDimension, expected_desc->dimension);

    if (desc->ViewDimension != expected_desc->dimension)
        return;

    if (desc->ViewDimension == D3D11_SRV_DIMENSION_TEXTURE2D)
    {
        ok_(__FILE__, line)(U(*desc).Texture2D.MostDetailedMip == expected_desc->miplevel_idx,
                "Got MostDetailedMip %u, expected %u.\n",
                U(*desc).Texture2D.MostDetailedMip, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).Texture2D.MipLevels == expected_desc->miplevel_count,
                "Got MipLevels %u, expected %u.\n",
                U(*desc).Texture2D.MipLevels, expected_desc->miplevel_count);
    }
    else if (desc->ViewDimension == D3D11_SRV_DIMENSION_TEXTURE2DARRAY)
    {
        ok_(__FILE__, line)(U(*desc).Texture2DArray.MostDetailedMip == expected_desc->miplevel_idx,
                "Got MostDetailedMip %u, expected %u.\n",
                U(*desc).Texture2DArray.MostDetailedMip, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.MipLevels == expected_desc->miplevel_count,
                "Got MipLevels %u, expected %u.\n",
                U(*desc).Texture2DArray.MipLevels, expected_desc->miplevel_count);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.FirstArraySlice == expected_desc->layer_idx,
                "Got FirstArraySlice %u, expected %u.\n",
                U(*desc).Texture2DArray.FirstArraySlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.ArraySize == expected_desc->layer_count,
                "Got ArraySize %u, expected %u.\n",
                U(*desc).Texture2DArray.ArraySize, expected_desc->layer_count);
    }
    else if (desc->ViewDimension == D3D11_SRV_DIMENSION_TEXTURE2DMSARRAY)
    {
        ok_(__FILE__, line)(U(*desc).Texture2DMSArray.FirstArraySlice == expected_desc->layer_idx,
                "Got FirstArraySlice %u, expected %u.\n",
                U(*desc).Texture2DMSArray.FirstArraySlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DMSArray.ArraySize == expected_desc->layer_count,
                "Got ArraySize %u, expected %u.\n",
                U(*desc).Texture2DMSArray.ArraySize, expected_desc->layer_count);
    }
    else if (desc->ViewDimension == D3D11_SRV_DIMENSION_TEXTURE3D)
    {
        ok_(__FILE__, line)(U(*desc).Texture3D.MostDetailedMip == expected_desc->miplevel_idx,
                "Got MostDetailedMip %u, expected %u.\n",
                U(*desc).Texture3D.MostDetailedMip, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).Texture3D.MipLevels == expected_desc->miplevel_count,
                "Got MipLevels %u, expected %u.\n",
                U(*desc).Texture3D.MipLevels, expected_desc->miplevel_count);
    }
    else if (desc->ViewDimension == D3D11_SRV_DIMENSION_TEXTURECUBE)
    {
        ok_(__FILE__, line)(U(*desc).TextureCube.MostDetailedMip == expected_desc->miplevel_idx,
                "Got MostDetailedMip %u, expected %u.\n",
                U(*desc).TextureCube.MostDetailedMip, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).TextureCube.MipLevels == expected_desc->miplevel_count,
                "Got MipLevels %u, expected %u.\n",
                U(*desc).TextureCube.MipLevels, expected_desc->miplevel_count);
    }
    else if (desc->ViewDimension == D3D11_SRV_DIMENSION_TEXTURECUBEARRAY)
    {
        ok_(__FILE__, line)(U(*desc).TextureCubeArray.MostDetailedMip == expected_desc->miplevel_idx,
                "Got MostDetailedMip %u, expected %u.\n",
                U(*desc).TextureCubeArray.MostDetailedMip, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).TextureCubeArray.MipLevels == expected_desc->miplevel_count,
                "Got MipLevels %u, expected %u.\n",
                U(*desc).TextureCubeArray.MipLevels, expected_desc->miplevel_count);
        ok_(__FILE__, line)(U(*desc).TextureCubeArray.First2DArrayFace == expected_desc->layer_idx,
                "Got First2DArrayFace %u, expected %u.\n",
                U(*desc).TextureCubeArray.First2DArrayFace, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).TextureCubeArray.NumCubes == expected_desc->layer_count,
                "Got NumCubes %u, expected %u.\n",
                U(*desc).TextureCubeArray.NumCubes, expected_desc->layer_count);
    }
    else if (desc->ViewDimension != D3D11_SRV_DIMENSION_TEXTURE2DMS)
    {
        trace("Unhandled view dimension %#x.\n", desc->ViewDimension);
    }
}

struct rtv_desc
{
    DXGI_FORMAT format;
    D3D11_RTV_DIMENSION dimension;
    unsigned int miplevel_idx;
    unsigned int layer_idx;
    unsigned int layer_count;
};

static void get_rtv_desc(D3D11_RENDER_TARGET_VIEW_DESC *d3d11_desc, const struct rtv_desc *desc)
{
    d3d11_desc->Format = desc->format;
    d3d11_desc->ViewDimension = desc->dimension;
    if (desc->dimension == D3D11_RTV_DIMENSION_TEXTURE1D)
    {
        U(*d3d11_desc).Texture1D.MipSlice = desc->miplevel_idx;
    }
    else if (desc->dimension == D3D11_RTV_DIMENSION_TEXTURE1DARRAY)
    {
        U(*d3d11_desc).Texture1DArray.MipSlice = desc->miplevel_idx;
        U(*d3d11_desc).Texture1DArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture1DArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_RTV_DIMENSION_TEXTURE2D)
    {
        U(*d3d11_desc).Texture2D.MipSlice = desc->miplevel_idx;
    }
    else if (desc->dimension == D3D11_RTV_DIMENSION_TEXTURE2DARRAY)
    {
        U(*d3d11_desc).Texture2DArray.MipSlice = desc->miplevel_idx;
        U(*d3d11_desc).Texture2DArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture2DArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_RTV_DIMENSION_TEXTURE2DMSARRAY)
    {
        U(*d3d11_desc).Texture2DMSArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture2DMSArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_RTV_DIMENSION_TEXTURE3D)
    {
        U(*d3d11_desc).Texture3D.MipSlice = desc->miplevel_idx;
        U(*d3d11_desc).Texture3D.FirstWSlice = desc->layer_idx;
        U(*d3d11_desc).Texture3D.WSize = desc->layer_count;
    }
    else if (desc->dimension != D3D11_RTV_DIMENSION_UNKNOWN
            && desc->dimension != D3D11_RTV_DIMENSION_TEXTURE2DMS)
    {
        trace("Unhandled view dimension %#x.\n", desc->dimension);
    }
}

#define check_rtv_desc(a, b) check_rtv_desc_(__LINE__, a, b)
static void check_rtv_desc_(unsigned int line, const D3D11_RENDER_TARGET_VIEW_DESC *desc,
        const struct rtv_desc *expected_desc)
{
    ok_(__FILE__, line)(desc->Format == expected_desc->format,
            "Got format %#x, expected %#x.\n", desc->Format, expected_desc->format);
    ok_(__FILE__, line)(desc->ViewDimension == expected_desc->dimension,
            "Got view dimension %#x, expected %#x.\n", desc->ViewDimension, expected_desc->dimension);

    if (desc->ViewDimension != expected_desc->dimension)
        return;

    if (desc->ViewDimension == D3D11_RTV_DIMENSION_TEXTURE2D)
    {
        ok_(__FILE__, line)(U(*desc).Texture2D.MipSlice == expected_desc->miplevel_idx,
                "Got MipSlice %u, expected %u.\n",
                U(*desc).Texture2D.MipSlice, expected_desc->miplevel_idx);
    }
    else if (desc->ViewDimension == D3D11_RTV_DIMENSION_TEXTURE2DARRAY)
    {
        ok_(__FILE__, line)(U(*desc).Texture2DArray.MipSlice == expected_desc->miplevel_idx,
                "Got MipSlice %u, expected %u.\n",
                U(*desc).Texture2DArray.MipSlice, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.FirstArraySlice == expected_desc->layer_idx,
                "Got FirstArraySlice %u, expected %u.\n",
                U(*desc).Texture2DArray.FirstArraySlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.ArraySize == expected_desc->layer_count,
                "Got ArraySize %u, expected %u.\n",
                U(*desc).Texture2DArray.ArraySize, expected_desc->layer_count);
    }
    else if (desc->ViewDimension == D3D11_RTV_DIMENSION_TEXTURE2DMSARRAY)
    {
        ok_(__FILE__, line)(U(*desc).Texture2DMSArray.FirstArraySlice == expected_desc->layer_idx,
                "Got FirstArraySlice %u, expected %u.\n",
                U(*desc).Texture2DMSArray.FirstArraySlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DMSArray.ArraySize == expected_desc->layer_count,
                "Got ArraySize %u, expected %u.\n",
                U(*desc).Texture2DMSArray.ArraySize, expected_desc->layer_count);
    }
    else if (desc->ViewDimension == D3D11_RTV_DIMENSION_TEXTURE3D)
    {
        ok_(__FILE__, line)(U(*desc).Texture3D.MipSlice == expected_desc->miplevel_idx,
                "Got MipSlice %u, expected %u.\n",
                U(*desc).Texture3D.MipSlice, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).Texture3D.FirstWSlice == expected_desc->layer_idx,
                "Got FirstWSlice %u, expected %u.\n",
                U(*desc).Texture3D.FirstWSlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture3D.WSize == expected_desc->layer_count,
                "Got WSize %u, expected %u.\n",
                U(*desc).Texture3D.WSize, expected_desc->layer_count);
    }
    else if (desc->ViewDimension != D3D11_RTV_DIMENSION_TEXTURE2DMS)
    {
        trace("Unhandled view dimension %#x.\n", desc->ViewDimension);
    }
}

struct dsv_desc
{
    DXGI_FORMAT format;
    D3D11_DSV_DIMENSION dimension;
    unsigned int miplevel_idx;
    unsigned int layer_idx;
    unsigned int layer_count;
};

static void get_dsv_desc(D3D11_DEPTH_STENCIL_VIEW_DESC *d3d11_desc, const struct dsv_desc *desc)
{
    d3d11_desc->Format = desc->format;
    d3d11_desc->ViewDimension = desc->dimension;
    d3d11_desc->Flags = 0;
    if (desc->dimension == D3D11_DSV_DIMENSION_TEXTURE1D)
    {
        U(*d3d11_desc).Texture1D.MipSlice = desc->miplevel_idx;
    }
    else if (desc->dimension == D3D11_DSV_DIMENSION_TEXTURE1DARRAY)
    {
        U(*d3d11_desc).Texture1DArray.MipSlice = desc->miplevel_idx;
        U(*d3d11_desc).Texture1DArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture1DArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_DSV_DIMENSION_TEXTURE2D)
    {
        U(*d3d11_desc).Texture2D.MipSlice = desc->miplevel_idx;
    }
    else if (desc->dimension == D3D11_DSV_DIMENSION_TEXTURE2DARRAY)
    {
        U(*d3d11_desc).Texture2DArray.MipSlice = desc->miplevel_idx;
        U(*d3d11_desc).Texture2DArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture2DArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_DSV_DIMENSION_TEXTURE2DMSARRAY)
    {
        U(*d3d11_desc).Texture2DMSArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture2DMSArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension != D3D11_DSV_DIMENSION_UNKNOWN
            && desc->dimension != D3D11_DSV_DIMENSION_TEXTURE2DMS)
    {
        trace("Unhandled view dimension %#x.\n", desc->dimension);
    }
}

#define check_dsv_desc(a, b) check_dsv_desc_(__LINE__, a, b)
static void check_dsv_desc_(unsigned int line, const D3D11_DEPTH_STENCIL_VIEW_DESC *desc,
        const struct dsv_desc *expected_desc)
{
    ok_(__FILE__, line)(desc->Format == expected_desc->format,
            "Got format %#x, expected %#x.\n", desc->Format, expected_desc->format);
    ok_(__FILE__, line)(desc->ViewDimension == expected_desc->dimension,
            "Got view dimension %#x, expected %#x.\n", desc->ViewDimension, expected_desc->dimension);

    if (desc->ViewDimension != expected_desc->dimension)
        return;

    if (desc->ViewDimension == D3D11_DSV_DIMENSION_TEXTURE2D)
    {
        ok_(__FILE__, line)(U(*desc).Texture2D.MipSlice == expected_desc->miplevel_idx,
                "Got MipSlice %u, expected %u.\n",
                U(*desc).Texture2D.MipSlice, expected_desc->miplevel_idx);
    }
    else if (desc->ViewDimension == D3D11_DSV_DIMENSION_TEXTURE2DARRAY)
    {
        ok_(__FILE__, line)(U(*desc).Texture2DArray.MipSlice == expected_desc->miplevel_idx,
                "Got MipSlice %u, expected %u.\n",
                U(*desc).Texture2DArray.MipSlice, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.FirstArraySlice == expected_desc->layer_idx,
                "Got FirstArraySlice %u, expected %u.\n",
                U(*desc).Texture2DArray.FirstArraySlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.ArraySize == expected_desc->layer_count,
                "Got ArraySize %u, expected %u.\n",
                U(*desc).Texture2DArray.ArraySize, expected_desc->layer_count);
    }
    else if (desc->ViewDimension == D3D11_DSV_DIMENSION_TEXTURE2DMSARRAY)
    {
        ok_(__FILE__, line)(U(*desc).Texture2DMSArray.FirstArraySlice == expected_desc->layer_idx,
                "Got FirstArraySlice %u, expected %u.\n",
                U(*desc).Texture2DMSArray.FirstArraySlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DMSArray.ArraySize == expected_desc->layer_count,
                "Got ArraySize %u, expected %u.\n",
                U(*desc).Texture2DMSArray.ArraySize, expected_desc->layer_count);
    }
    else if (desc->ViewDimension != D3D11_DSV_DIMENSION_TEXTURE2DMS)
    {
        trace("Unhandled view dimension %#x.\n", desc->ViewDimension);
    }
}

struct uav_desc
{
    DXGI_FORMAT format;
    D3D11_UAV_DIMENSION dimension;
    unsigned int miplevel_idx;
    unsigned int layer_idx;
    unsigned int layer_count;
};

static void get_uav_desc(D3D11_UNORDERED_ACCESS_VIEW_DESC *d3d11_desc, const struct uav_desc *desc)
{
    d3d11_desc->Format = desc->format;
    d3d11_desc->ViewDimension = desc->dimension;
    if (desc->dimension == D3D11_UAV_DIMENSION_TEXTURE1D)
    {
        U(*d3d11_desc).Texture1D.MipSlice = desc->miplevel_idx;
    }
    else if (desc->dimension == D3D11_UAV_DIMENSION_TEXTURE1DARRAY)
    {
        U(*d3d11_desc).Texture1DArray.MipSlice = desc->miplevel_idx;
        U(*d3d11_desc).Texture1DArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture1DArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_UAV_DIMENSION_TEXTURE2D)
    {
        U(*d3d11_desc).Texture2D.MipSlice = desc->miplevel_idx;
    }
    else if (desc->dimension == D3D11_UAV_DIMENSION_TEXTURE2DARRAY)
    {
        U(*d3d11_desc).Texture2DArray.MipSlice = desc->miplevel_idx;
        U(*d3d11_desc).Texture2DArray.FirstArraySlice = desc->layer_idx;
        U(*d3d11_desc).Texture2DArray.ArraySize = desc->layer_count;
    }
    else if (desc->dimension == D3D11_UAV_DIMENSION_TEXTURE3D)
    {
        U(*d3d11_desc).Texture3D.MipSlice = desc->miplevel_idx;
        U(*d3d11_desc).Texture3D.FirstWSlice = desc->layer_idx;
        U(*d3d11_desc).Texture3D.WSize = desc->layer_count;
    }
    else if (desc->dimension != D3D11_UAV_DIMENSION_UNKNOWN)
    {
        trace("Unhandled view dimension %#x.\n", desc->dimension);
    }
}

#define check_uav_desc(a, b) check_uav_desc_(__LINE__, a, b)
static void check_uav_desc_(unsigned int line, const D3D11_UNORDERED_ACCESS_VIEW_DESC *desc,
        const struct uav_desc *expected_desc)
{
    ok_(__FILE__, line)(desc->Format == expected_desc->format,
            "Got format %#x, expected %#x.\n", desc->Format, expected_desc->format);
    ok_(__FILE__, line)(desc->ViewDimension == expected_desc->dimension,
            "Got view dimension %#x, expected %#x.\n", desc->ViewDimension, expected_desc->dimension);

    if (desc->ViewDimension != expected_desc->dimension)
        return;

    if (desc->ViewDimension == D3D11_UAV_DIMENSION_TEXTURE2D)
    {
        ok_(__FILE__, line)(U(*desc).Texture2D.MipSlice == expected_desc->miplevel_idx,
                "Got MipSlice %u, expected %u.\n",
                U(*desc).Texture2D.MipSlice, expected_desc->miplevel_idx);
    }
    else if (desc->ViewDimension == D3D11_UAV_DIMENSION_TEXTURE2DARRAY)
    {
        ok_(__FILE__, line)(U(*desc).Texture2DArray.MipSlice == expected_desc->miplevel_idx,
                "Got MipSlice %u, expected %u.\n",
                U(*desc).Texture2DArray.MipSlice, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.FirstArraySlice == expected_desc->layer_idx,
                "Got FirstArraySlice %u, expected %u.\n",
                U(*desc).Texture2DArray.FirstArraySlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture2DArray.ArraySize == expected_desc->layer_count,
                "Got ArraySize %u, expected %u.\n",
                U(*desc).Texture2DArray.ArraySize, expected_desc->layer_count);
    }
    else if (desc->ViewDimension == D3D11_UAV_DIMENSION_TEXTURE3D)
    {
        ok_(__FILE__, line)(U(*desc).Texture3D.MipSlice == expected_desc->miplevel_idx,
                "Got MipSlice %u, expected %u.\n",
                U(*desc).Texture3D.MipSlice, expected_desc->miplevel_idx);
        ok_(__FILE__, line)(U(*desc).Texture3D.FirstWSlice == expected_desc->layer_idx,
                "Got FirstWSlice %u, expected %u.\n",
                U(*desc).Texture3D.FirstWSlice, expected_desc->layer_idx);
        ok_(__FILE__, line)(U(*desc).Texture3D.WSize == expected_desc->layer_count,
                "Got WSize %u, expected %u.\n",
                U(*desc).Texture3D.WSize, expected_desc->layer_count);
    }
    else
    {
        trace("Unhandled view dimension %#x.\n", desc->ViewDimension);
    }
}

static void set_viewport(ID3D11DeviceContext *context, float x, float y,
        float width, float height, float min_depth, float max_depth)
{
    D3D11_VIEWPORT vp;

    vp.TopLeftX = x;
    vp.TopLeftY = y;
    vp.Width = width;
    vp.Height = height;
    vp.MinDepth = min_depth;
    vp.MaxDepth = max_depth;

    ID3D11DeviceContext_RSSetViewports(context, 1, &vp);
}

#define create_buffer(a, b, c, d) create_buffer_(__LINE__, a, b, 0, c, d)
#define create_buffer_misc(a, b, c, d, e) create_buffer_(__LINE__, a, b, c, d, e)
static ID3D11Buffer *create_buffer_(unsigned int line, ID3D11Device *device,
        unsigned int bind_flags, unsigned int misc_flags, unsigned int size, const void *data)
{
    D3D11_SUBRESOURCE_DATA resource_data;
    D3D11_BUFFER_DESC buffer_desc;
    ID3D11Buffer *buffer;
    HRESULT hr;

    buffer_desc.ByteWidth = size;
    buffer_desc.Usage = D3D11_USAGE_DEFAULT;
    buffer_desc.BindFlags = bind_flags;
    buffer_desc.CPUAccessFlags = 0;
    buffer_desc.MiscFlags = misc_flags;
    buffer_desc.StructureByteStride = 0;

    resource_data.pSysMem = data;
    resource_data.SysMemPitch = 0;
    resource_data.SysMemSlicePitch = 0;

    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, data ? &resource_data : NULL, &buffer);
    ok_(__FILE__, line)(hr == S_OK, "Failed to create buffer, hr %#lx.\n", hr);
    return buffer;
}

static HRESULT create_resource(ID3D11Device *device, const struct resource_desc *desc,
        const D3D11_SUBRESOURCE_DATA *data, ID3D11Resource **resource)
{
    D3D11_TEXTURE1D_DESC texture1d_desc;
    D3D11_TEXTURE2D_DESC texture2d_desc;
    D3D11_TEXTURE3D_DESC texture3d_desc;
    D3D11_BUFFER_DESC buffer_desc;

    switch (desc->dimension)
    {
        case D3D11_RESOURCE_DIMENSION_BUFFER:
            buffer_desc.ByteWidth = desc->width;
            buffer_desc.Usage = desc->usage;
            buffer_desc.BindFlags = desc->bind_flags;
            buffer_desc.CPUAccessFlags = desc->cpu_access_flags;
            buffer_desc.MiscFlags = desc->misc_flags;
            buffer_desc.StructureByteStride = desc->structure_byte_stride;
            return ID3D11Device_CreateBuffer(device, &buffer_desc, data, (ID3D11Buffer **)resource);

        case D3D11_RESOURCE_DIMENSION_TEXTURE1D:
            texture1d_desc.Width = desc->width;
            texture1d_desc.MipLevels = desc->level_count;
            texture1d_desc.ArraySize = desc->depth_or_array_size;
            texture1d_desc.Format = desc->format;
            texture1d_desc.Usage = desc->usage;
            texture1d_desc.BindFlags = desc->bind_flags;
            texture1d_desc.CPUAccessFlags = desc->cpu_access_flags;
            texture1d_desc.MiscFlags = desc->misc_flags;
            return ID3D11Device_CreateTexture1D(device, &texture1d_desc, data, (ID3D11Texture1D **)resource);

        case D3D11_RESOURCE_DIMENSION_TEXTURE2D:
            texture2d_desc.Width = desc->width;
            texture2d_desc.Height = desc->height;
            texture2d_desc.MipLevels = desc->level_count;
            texture2d_desc.ArraySize = desc->depth_or_array_size;
            texture2d_desc.Format = desc->format;
            texture2d_desc.SampleDesc = desc->sample_desc;
            texture2d_desc.Usage = desc->usage;
            texture2d_desc.BindFlags = desc->bind_flags;
            texture2d_desc.CPUAccessFlags = desc->cpu_access_flags;
            texture2d_desc.MiscFlags = desc->misc_flags;
            return ID3D11Device_CreateTexture2D(device, &texture2d_desc, data, (ID3D11Texture2D **)resource);

        case D3D11_RESOURCE_DIMENSION_TEXTURE3D:
            texture3d_desc.Width = desc->width;
            texture3d_desc.Height = desc->height;
            texture3d_desc.Depth = desc->depth_or_array_size;
            texture3d_desc.MipLevels = desc->level_count;
            texture3d_desc.Format = desc->format;
            texture3d_desc.Usage = desc->usage;
            texture3d_desc.BindFlags = desc->bind_flags;
            texture3d_desc.CPUAccessFlags = desc->cpu_access_flags;
            texture3d_desc.MiscFlags = desc->misc_flags;
            return ID3D11Device_CreateTexture3D(device, &texture3d_desc, data, (ID3D11Texture3D **)resource);

        default:
            return E_INVALIDARG;
    }
}

struct resource_readback
{
    ID3D11Resource *resource;
    D3D11_MAPPED_SUBRESOURCE map_desc;
    ID3D11DeviceContext *immediate_context;
    unsigned int width, height, depth, sub_resource_idx;
};

static void init_resource_readback(ID3D11Resource *resource, ID3D11Resource *readback_resource,
        unsigned int width, unsigned int height, unsigned int depth, unsigned int sub_resource_idx,
        ID3D11Device *device, struct resource_readback *rb)
{
    HRESULT hr;

    rb->resource = readback_resource;
    rb->width = width;
    rb->height = height;
    rb->depth = depth;
    rb->sub_resource_idx = sub_resource_idx;

    ID3D11Device_GetImmediateContext(device, &rb->immediate_context);

    ID3D11DeviceContext_CopyResource(rb->immediate_context, rb->resource, resource);
    if (FAILED(hr = ID3D11DeviceContext_Map(rb->immediate_context,
            rb->resource, sub_resource_idx, D3D11_MAP_READ, 0, &rb->map_desc)))
    {
        trace("Failed to map resource, hr %#lx.\n", hr);
        ID3D11Resource_Release(rb->resource);
        rb->resource = NULL;
        ID3D11DeviceContext_Release(rb->immediate_context);
        rb->immediate_context = NULL;
    }
}

static void get_buffer_readback(ID3D11Buffer *buffer, struct resource_readback *rb)
{
    D3D11_BUFFER_DESC buffer_desc;
    ID3D11Resource *rb_buffer;
    ID3D11Device *device;
    HRESULT hr;

    memset(rb, 0, sizeof(*rb));

    ID3D11Buffer_GetDevice(buffer, &device);

    ID3D11Buffer_GetDesc(buffer, &buffer_desc);
    buffer_desc.Usage = D3D11_USAGE_STAGING;
    buffer_desc.BindFlags = 0;
    buffer_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
    buffer_desc.MiscFlags = 0;
    buffer_desc.StructureByteStride = 0;
    if (FAILED(hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, (ID3D11Buffer **)&rb_buffer)))
    {
        trace("Failed to create staging buffer, hr %#lx.\n", hr);
        ID3D11Device_Release(device);
        return;
    }

    init_resource_readback((ID3D11Resource *)buffer, rb_buffer,
            buffer_desc.ByteWidth, 1, 1, 0, device, rb);

    ID3D11Device_Release(device);
}

static void get_texture1d_readback(ID3D11Texture1D *texture, unsigned int sub_resource_idx,
        struct resource_readback *rb)
{
    D3D11_TEXTURE1D_DESC texture_desc;
    ID3D11Resource *rb_texture;
    unsigned int miplevel;
    ID3D11Device *device;
    HRESULT hr;

    memset(rb, 0, sizeof(*rb));

    ID3D11Texture1D_GetDevice(texture, &device);

    ID3D11Texture1D_GetDesc(texture, &texture_desc);
    texture_desc.Usage = D3D11_USAGE_STAGING;
    texture_desc.BindFlags = 0;
    texture_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
    texture_desc.MiscFlags = 0;
    if (FAILED(hr = ID3D11Device_CreateTexture1D(device, &texture_desc, NULL, (ID3D11Texture1D **)&rb_texture)))
    {
        trace("Failed to create texture, hr %#lx.\n", hr);
        ID3D11Device_Release(device);
        return;
    }

    miplevel = sub_resource_idx % texture_desc.MipLevels;
    init_resource_readback((ID3D11Resource *)texture, rb_texture,
            max(1, texture_desc.Width >> miplevel), 1, 1, sub_resource_idx, device, rb);

    ID3D11Device_Release(device);
}

static void get_texture_readback(ID3D11Texture2D *texture, unsigned int sub_resource_idx,
        struct resource_readback *rb)
{
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11Resource *rb_texture;
    unsigned int miplevel;
    ID3D11Device *device;
    HRESULT hr;

    memset(rb, 0, sizeof(*rb));

    ID3D11Texture2D_GetDevice(texture, &device);

    ID3D11Texture2D_GetDesc(texture, &texture_desc);
    texture_desc.Usage = D3D11_USAGE_STAGING;
    texture_desc.BindFlags = 0;
    texture_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
    texture_desc.MiscFlags = 0;
    if (FAILED(hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, (ID3D11Texture2D **)&rb_texture)))
    {
        trace("Failed to create texture, hr %#lx.\n", hr);
        ID3D11Device_Release(device);
        return;
    }

    miplevel = sub_resource_idx % texture_desc.MipLevels;
    init_resource_readback((ID3D11Resource *)texture, rb_texture,
            max(1, texture_desc.Width >> miplevel),
            max(1, texture_desc.Height >> miplevel),
            1, sub_resource_idx, device, rb);

    ID3D11Device_Release(device);
}

static void get_texture3d_readback(ID3D11Texture3D *texture, unsigned int sub_resource_idx,
        struct resource_readback *rb)
{
    D3D11_TEXTURE3D_DESC texture_desc;
    ID3D11Resource *rb_texture;
    unsigned int miplevel;
    ID3D11Device *device;
    HRESULT hr;

    memset(rb, 0, sizeof(*rb));

    ID3D11Texture3D_GetDevice(texture, &device);

    ID3D11Texture3D_GetDesc(texture, &texture_desc);
    texture_desc.Usage = D3D11_USAGE_STAGING;
    texture_desc.BindFlags = 0;
    texture_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
    texture_desc.MiscFlags = 0;
    if (FAILED(hr = ID3D11Device_CreateTexture3D(device, &texture_desc, NULL, (ID3D11Texture3D **)&rb_texture)))
    {
        trace("Failed to create texture, hr %#lx.\n", hr);
        ID3D11Device_Release(device);
        return;
    }

    miplevel = sub_resource_idx % texture_desc.MipLevels;
    init_resource_readback((ID3D11Resource *)texture, rb_texture,
            max(1, texture_desc.Width >> miplevel),
            max(1, texture_desc.Height >> miplevel),
            max(1, texture_desc.Depth >> miplevel),
            sub_resource_idx, device, rb);

    ID3D11Device_Release(device);
}

static void get_resource_readback(ID3D11Resource *resource,
        unsigned int sub_resource_idx, struct resource_readback *rb)
{
    D3D11_RESOURCE_DIMENSION d;

    ID3D11Resource_GetType(resource, &d);
    switch (d)
    {
        case D3D11_RESOURCE_DIMENSION_BUFFER:
            get_buffer_readback((ID3D11Buffer *)resource, rb);
            return;

        case D3D11_RESOURCE_DIMENSION_TEXTURE1D:
            get_texture1d_readback((ID3D11Texture1D *)resource, sub_resource_idx, rb);
            return;

        case D3D11_RESOURCE_DIMENSION_TEXTURE2D:
            get_texture_readback((ID3D11Texture2D *)resource, sub_resource_idx, rb);
            return;

        case D3D11_RESOURCE_DIMENSION_TEXTURE3D:
            get_texture3d_readback((ID3D11Texture3D *)resource, sub_resource_idx, rb);
            return;

        default:
            memset(rb, 0, sizeof(*rb));
            return;
    }
}

static void *get_readback_data(struct resource_readback *rb,
        unsigned int x, unsigned int y, unsigned int z, unsigned byte_width)
{
    return (BYTE *)rb->map_desc.pData + z * rb->map_desc.DepthPitch + y * rb->map_desc.RowPitch + x * byte_width;
}

static BYTE get_readback_u8(struct resource_readback *rb, unsigned int x, unsigned int y, unsigned int z)
{
    return *(BYTE *)get_readback_data(rb, x, y, z, sizeof(BYTE));
}

static WORD get_readback_u16(struct resource_readback *rb, unsigned int x, unsigned int y, unsigned int z)
{
    return *(WORD *)get_readback_data(rb, x, y, z, sizeof(WORD));
}

static DWORD get_readback_u32(struct resource_readback *rb, unsigned int x, unsigned int y, unsigned int z)
{
    return *(DWORD *)get_readback_data(rb, x, y, z, sizeof(DWORD));
}

static DWORD get_readback_color(struct resource_readback *rb, unsigned int x, unsigned int y, unsigned int z)
{
    return get_readback_u32(rb, x, y, z);
}

static float get_readback_float(struct resource_readback *rb, unsigned int x, unsigned int y)
{
    return *(float *)get_readback_data(rb, x, y, 0, sizeof(float));
}

static const struct vec4 *get_readback_vec4(struct resource_readback *rb, unsigned int x, unsigned int y)
{
    return get_readback_data(rb, x, y, 0, sizeof(struct vec4));
}

static const struct uvec4 *get_readback_uvec4(struct resource_readback *rb, unsigned int x, unsigned int y)
{
    return get_readback_data(rb, x, y, 0, sizeof(struct uvec4));
}

static void release_resource_readback(struct resource_readback *rb)
{
    ID3D11DeviceContext_Unmap(rb->immediate_context, rb->resource, rb->sub_resource_idx);
    ID3D11Resource_Release(rb->resource);
    ID3D11DeviceContext_Release(rb->immediate_context);
}

static DWORD get_texture_color(ID3D11Texture2D *texture, unsigned int x, unsigned int y)
{
    struct resource_readback rb;
    DWORD color;

    get_texture_readback(texture, 0, &rb);
    color = get_readback_color(&rb, x, y, 0);
    release_resource_readback(&rb);

    return color;
}

#define check_readback_data_u8(a, b, c, d) check_readback_data_u8_(__LINE__, a, b, c, d)
static void check_readback_data_u8_(unsigned int line, struct resource_readback *rb,
        const RECT *rect, BYTE expected_value, BYTE max_diff)
{
    unsigned int x = 0, y = 0, z = 0;
    BOOL all_match = FALSE;
    RECT default_rect;
    BYTE value = 0;

    if (!rect)
    {
        SetRect(&default_rect, 0, 0, rb->width, rb->height);
        rect = &default_rect;
    }

    for (z = 0; z < rb->depth; ++z)
    {
        for (y = rect->top; y < rect->bottom; ++y)
        {
            for (x = rect->left; x < rect->right; ++x)
            {
                value = get_readback_u8(rb, x, y, z);
                if (!compare_uint(value, expected_value, max_diff))
                    goto done;
            }
        }
    }
    all_match = TRUE;

done:
    ok_(__FILE__, line)(all_match,
            "Got 0x%02x, expected 0x%02x at (%u, %u, %u), sub-resource %u.\n",
            value, expected_value, x, y, z, rb->sub_resource_idx);
}

#define check_readback_data_u16(a, b, c, d) check_readback_data_u16_(__LINE__, a, b, c, d)
static void check_readback_data_u16_(unsigned int line, struct resource_readback *rb,
        const RECT *rect, WORD expected_value, BYTE max_diff)
{
    unsigned int x = 0, y = 0, z = 0;
    BOOL all_match = FALSE;
    RECT default_rect;
    WORD value = 0;

    if (!rect)
    {
        SetRect(&default_rect, 0, 0, rb->width, rb->height);
        rect = &default_rect;
    }

    for (z = 0; z < rb->depth; ++z)
    {
        for (y = rect->top; y < rect->bottom; ++y)
        {
            for (x = rect->left; x < rect->right; ++x)
            {
                value = get_readback_u16(rb, x, y, z);
                if (!compare_uint(value, expected_value, max_diff))
                    goto done;
            }
        }
    }
    all_match = TRUE;

done:
    ok_(__FILE__, line)(all_match,
            "Got 0x%04x, expected 0x%04x at (%u, %u, %u), sub-resource %u.\n",
            value, expected_value, x, y, z, rb->sub_resource_idx);
}

#define check_readback_data_u24(a, b, c, d, e) check_readback_data_u24_(__LINE__, a, b, c, d, e)
static void check_readback_data_u24_(unsigned int line, struct resource_readback *rb,
        const RECT *rect, unsigned int shift, DWORD expected_value, BYTE max_diff)
{
    unsigned int x = 0, y = 0, z = 0;
    BOOL all_match = FALSE;
    RECT default_rect;
    DWORD value = 0;

    if (!rect)
    {
        SetRect(&default_rect, 0, 0, rb->width, rb->height);
        rect = &default_rect;
    }

    for (z = 0; z < rb->depth; ++z)
    {
        for (y = rect->top; y < rect->bottom; ++y)
        {
            for (x = rect->left; x < rect->right; ++x)
            {
                value = get_readback_u32(rb, x, y, z) >> shift;
                if (!compare_uint(value, expected_value, max_diff))
                    goto done;
            }
        }
    }
    all_match = TRUE;

done:
    ok_(__FILE__, line)(all_match,
            "Got 0x%06lx, expected 0x%06lx at (%u, %u, %u), sub-resource %u.\n",
            value, expected_value, x, y, z, rb->sub_resource_idx);
}

#define check_readback_data_color(a, b, c, d) check_readback_data_color_(__LINE__, a, b, c, d)
static void check_readback_data_color_(unsigned int line, struct resource_readback *rb,
        const RECT *rect, DWORD expected_color, BYTE max_diff)
{
    unsigned int x = 0, y = 0, z = 0;
    BOOL all_match = FALSE;
    RECT default_rect;
    DWORD color = 0;

    if (!rect)
    {
        SetRect(&default_rect, 0, 0, rb->width, rb->height);
        rect = &default_rect;
    }

    for (z = 0; z < rb->depth; ++z)
    {
        for (y = rect->top; y < rect->bottom; ++y)
        {
            for (x = rect->left; x < rect->right; ++x)
            {
                color = get_readback_color(rb, x, y, z);
                if (!compare_color(color, expected_color, max_diff))
                    goto done;
            }
        }
    }
    all_match = TRUE;

done:
    ok_(__FILE__, line)(all_match,
            "Got 0x%08lx, expected 0x%08lx at (%u, %u, %u), sub-resource %u.\n",
            color, expected_color, x, y, z, rb->sub_resource_idx);
}

#define check_texture_sub_resource_color(a, b, c, d, e) check_texture_sub_resource_color_(__LINE__, a, b, c, d, e)
static void check_texture_sub_resource_color_(unsigned int line, ID3D11Texture2D *texture,
        unsigned int sub_resource_idx, const RECT *rect, DWORD expected_color, BYTE max_diff)
{
    struct resource_readback rb;

    get_texture_readback(texture, sub_resource_idx, &rb);
    check_readback_data_color_(line, &rb, rect, expected_color, max_diff);
    release_resource_readback(&rb);
}

#define check_texture_color(t, c, d) check_texture_color_(__LINE__, t, c, d)
static void check_texture_color_(unsigned int line, ID3D11Texture2D *texture,
        DWORD expected_color, BYTE max_diff)
{
    unsigned int sub_resource_idx, sub_resource_count;
    D3D11_TEXTURE2D_DESC texture_desc;

    ID3D11Texture2D_GetDesc(texture, &texture_desc);
    sub_resource_count = texture_desc.ArraySize * texture_desc.MipLevels;
    for (sub_resource_idx = 0; sub_resource_idx < sub_resource_count; ++sub_resource_idx)
        check_texture_sub_resource_color_(line, texture, sub_resource_idx, NULL, expected_color, max_diff);
}

#define check_texture1d_sub_resource_color(a, b, c, d, e) check_texture1d_sub_resource_color_(__LINE__, a, b, c, d, e)
static void check_texture1d_sub_resource_color_(unsigned int line, ID3D11Texture1D *texture,
        unsigned int sub_resource_idx, const RECT *rect, DWORD expected_color, BYTE max_diff)
{
    struct resource_readback rb;

    get_texture1d_readback(texture, sub_resource_idx, &rb);
    check_readback_data_color_(line, &rb, rect, expected_color, max_diff);
    release_resource_readback(&rb);
}

#define check_texture1d_color(t, c, d) check_texture1d_color_(__LINE__, t, c, d)
static void check_texture1d_color_(unsigned int line, ID3D11Texture1D *texture,
        DWORD expected_color, BYTE max_diff)
{
    unsigned int sub_resource_idx, sub_resource_count;
    D3D11_TEXTURE1D_DESC texture_desc;

    ID3D11Texture1D_GetDesc(texture, &texture_desc);
    sub_resource_count = texture_desc.ArraySize * texture_desc.MipLevels;
    for (sub_resource_idx = 0; sub_resource_idx < sub_resource_count; ++sub_resource_idx)
        check_texture1d_sub_resource_color_(line, texture, sub_resource_idx, NULL, expected_color, max_diff);
}

#define check_texture3d_sub_resource_color(a, b, c, d, e) check_texture3d_sub_resource_color_(__LINE__, a, b, c, d, e)
static void check_texture3d_sub_resource_color_(unsigned int line, ID3D11Texture3D *texture,
        unsigned int sub_resource_idx, const RECT *rect, DWORD expected_color, BYTE max_diff)
{
    struct resource_readback rb;

    get_texture3d_readback(texture, sub_resource_idx, &rb);
    check_readback_data_color_(line, &rb, rect, expected_color, max_diff);
    release_resource_readback(&rb);
}

#define check_texture3d_color(t, c, d) check_texture3d_color_(__LINE__, t, c, d)
static void check_texture3d_color_(unsigned int line, ID3D11Texture3D *texture,
        DWORD expected_color, BYTE max_diff)
{
    unsigned int sub_resource_idx, sub_resource_count;
    D3D11_TEXTURE3D_DESC texture_desc;

    ID3D11Texture3D_GetDesc(texture, &texture_desc);
    sub_resource_count = texture_desc.MipLevels;
    for (sub_resource_idx = 0; sub_resource_idx < sub_resource_count; ++sub_resource_idx)
        check_texture3d_sub_resource_color_(line, texture, sub_resource_idx, NULL, expected_color, max_diff);
}

#define check_texture_sub_resource_float(a, b, c, d, e) check_texture_sub_resource_float_(__LINE__, a, b, c, d, e)
static void check_texture_sub_resource_float_(unsigned int line, ID3D11Texture2D *texture,
        unsigned int sub_resource_idx, const RECT *rect, float expected_value, BYTE max_diff)
{
    struct resource_readback rb;
    unsigned int x = 0, y = 0;
    BOOL all_match = TRUE;
    float value = 0.0f;
    RECT default_rect;

    get_texture_readback(texture, sub_resource_idx, &rb);
    if (!rect)
    {
        SetRect(&default_rect, 0, 0, rb.width, rb.height);
        rect = &default_rect;
    }
    for (y = rect->top; y < rect->bottom; ++y)
    {
        for (x = rect->left; x < rect->right; ++x)
        {
            value = get_readback_float(&rb, x, y);
            if (!compare_float(value, expected_value, max_diff))
            {
                all_match = FALSE;
                break;
            }
        }
        if (!all_match)
            break;
    }
    release_resource_readback(&rb);
    ok_(__FILE__, line)(all_match,
            "Got %.8e, expected %.8e at (%u, %u), sub-resource %u.\n",
            value, expected_value, x, y, sub_resource_idx);
}

#define check_texture_float(r, f, d) check_texture_float_(__LINE__, r, f, d)
static void check_texture_float_(unsigned int line, ID3D11Texture2D *texture,
        float expected_value, BYTE max_diff)
{
    unsigned int sub_resource_idx, sub_resource_count;
    D3D11_TEXTURE2D_DESC texture_desc;

    ID3D11Texture2D_GetDesc(texture, &texture_desc);
    sub_resource_count = texture_desc.ArraySize * texture_desc.MipLevels;
    for (sub_resource_idx = 0; sub_resource_idx < sub_resource_count; ++sub_resource_idx)
        check_texture_sub_resource_float_(line, texture, sub_resource_idx, NULL, expected_value, max_diff);
}

#define check_texture_sub_resource_vec4(a, b, c, d, e) check_texture_sub_resource_vec4_(__LINE__, a, b, c, d, e)
static void check_texture_sub_resource_vec4_(unsigned int line, ID3D11Texture2D *texture,
        unsigned int sub_resource_idx, const RECT *rect, const struct vec4 *expected_value, BYTE max_diff)
{
    struct resource_readback rb;
    unsigned int x = 0, y = 0;
    struct vec4 value = {0};
    BOOL all_match = TRUE;
    RECT default_rect;

    get_texture_readback(texture, sub_resource_idx, &rb);
    if (!rect)
    {
        SetRect(&default_rect, 0, 0, rb.width, rb.height);
        rect = &default_rect;
    }
    for (y = rect->top; y < rect->bottom; ++y)
    {
        for (x = rect->left; x < rect->right; ++x)
        {
            value = *get_readback_vec4(&rb, x, y);
            if (!compare_vec4(&value, expected_value, max_diff))
            {
                all_match = FALSE;
                break;
            }
        }
        if (!all_match)
            break;
    }
    release_resource_readback(&rb);
    ok_(__FILE__, line)(all_match,
            "Got {%.8e, %.8e, %.8e, %.8e}, expected {%.8e, %.8e, %.8e, %.8e} at (%u, %u), sub-resource %u.\n",
            value.x, value.y, value.z, value.w,
            expected_value->x, expected_value->y, expected_value->z, expected_value->w,
            x, y, sub_resource_idx);
}

#define check_texture_vec4(a, b, c) check_texture_vec4_(__LINE__, a, b, c)
static void check_texture_vec4_(unsigned int line, ID3D11Texture2D *texture,
        const struct vec4 *expected_value, BYTE max_diff)
{
    unsigned int sub_resource_idx, sub_resource_count;
    D3D11_TEXTURE2D_DESC texture_desc;

    ID3D11Texture2D_GetDesc(texture, &texture_desc);
    sub_resource_count = texture_desc.ArraySize * texture_desc.MipLevels;
    for (sub_resource_idx = 0; sub_resource_idx < sub_resource_count; ++sub_resource_idx)
        check_texture_sub_resource_vec4_(line, texture, sub_resource_idx, NULL, expected_value, max_diff);
}

#define check_texture_sub_resource_uvec4(a, b, c, d) check_texture_sub_resource_uvec4_(__LINE__, a, b, c, d)
static void check_texture_sub_resource_uvec4_(unsigned int line, ID3D11Texture2D *texture,
        unsigned int sub_resource_idx, const RECT *rect, const struct uvec4 *expected_value)
{
    struct resource_readback rb;
    unsigned int x = 0, y = 0;
    struct uvec4 value = {0};
    BOOL all_match = TRUE;
    RECT default_rect;

    get_texture_readback(texture, sub_resource_idx, &rb);
    if (!rect)
    {
        SetRect(&default_rect, 0, 0, rb.width, rb.height);
        rect = &default_rect;
    }
    for (y = rect->top; y < rect->bottom; ++y)
    {
        for (x = rect->left; x < rect->right; ++x)
        {
            value = *get_readback_uvec4(&rb, x, y);
            if (!compare_uvec4(&value, expected_value))
            {
                all_match = FALSE;
                break;
            }
        }
        if (!all_match)
            break;
    }
    release_resource_readback(&rb);
    ok_(__FILE__, line)(all_match,
            "Got {0x%08x, 0x%08x, 0x%08x, 0x%08x}, expected {0x%08x, 0x%08x, 0x%08x, 0x%08x} "
            "at (%u, %u), sub-resource %u.\n",
            value.x, value.y, value.z, value.w,
            expected_value->x, expected_value->y, expected_value->z, expected_value->w,
            x, y, sub_resource_idx);
}

#define check_texture_uvec4(a, b) check_texture_uvec4_(__LINE__, a, b)
static void check_texture_uvec4_(unsigned int line, ID3D11Texture2D *texture,
        const struct uvec4 *expected_value)
{
    unsigned int sub_resource_idx, sub_resource_count;
    D3D11_TEXTURE2D_DESC texture_desc;

    ID3D11Texture2D_GetDesc(texture, &texture_desc);
    sub_resource_count = texture_desc.ArraySize * texture_desc.MipLevels;
    for (sub_resource_idx = 0; sub_resource_idx < sub_resource_count; ++sub_resource_idx)
        check_texture_sub_resource_uvec4_(line, texture, sub_resource_idx, NULL, expected_value);
}

static IDXGIAdapter *create_adapter(void)
{
    IDXGIFactory4 *factory4;
    IDXGIFactory *factory;
    IDXGIAdapter *adapter;
    HRESULT hr;

    if (!use_warp_adapter && !use_adapter_idx)
        return NULL;

    if (FAILED(hr = CreateDXGIFactory1(&IID_IDXGIFactory, (void **)&factory)))
    {
        trace("Failed to create IDXGIFactory, hr %#lx.\n", hr);
        return NULL;
    }

    adapter = NULL;
    if (use_warp_adapter)
    {
        if (SUCCEEDED(hr = IDXGIFactory_QueryInterface(factory, &IID_IDXGIFactory4, (void **)&factory4)))
        {
            hr = IDXGIFactory4_EnumWarpAdapter(factory4, &IID_IDXGIAdapter, (void **)&adapter);
            IDXGIFactory4_Release(factory4);
        }
        else
        {
            trace("Failed to get IDXGIFactory4, hr %#lx.\n", hr);
        }
    }
    else
    {
        hr = IDXGIFactory_EnumAdapters(factory, use_adapter_idx, &adapter);
    }
    IDXGIFactory_Release(factory);
    if (FAILED(hr))
        trace("Failed to get adapter, hr %#lx.\n", hr);
    return adapter;
}

static ID3D11Device *create_device(const struct device_desc *desc)
{
    static const D3D_FEATURE_LEVEL default_feature_level[] =
    {
        D3D_FEATURE_LEVEL_11_0,
        D3D_FEATURE_LEVEL_10_1,
        D3D_FEATURE_LEVEL_10_0,
    };
    const D3D_FEATURE_LEVEL *feature_level;
    UINT flags = desc ? desc->flags : 0;
    unsigned int feature_level_count;
    IDXGIAdapter *adapter;
    ID3D11Device *device;
    HRESULT hr;

    if (desc && desc->feature_level)
    {
        feature_level = desc->feature_level;
        feature_level_count = 1;
    }
    else
    {
        feature_level = default_feature_level;
        feature_level_count = ARRAY_SIZE(default_feature_level);
    }

    if (enable_debug_layer)
        flags |= D3D11_CREATE_DEVICE_DEBUG;

    if ((adapter = create_adapter()))
    {
        hr = D3D11CreateDevice(adapter, D3D_DRIVER_TYPE_UNKNOWN, NULL, flags,
                feature_level, feature_level_count, D3D11_SDK_VERSION, &device, NULL, NULL);
        IDXGIAdapter_Release(adapter);
        return SUCCEEDED(hr) ? device : NULL;
    }

    if (SUCCEEDED(D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, flags,
            feature_level, feature_level_count, D3D11_SDK_VERSION, &device, NULL, NULL)))
        return device;
    if (SUCCEEDED(D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_WARP, NULL, flags,
            feature_level, feature_level_count, D3D11_SDK_VERSION, &device, NULL, NULL)))
        return device;
    if (SUCCEEDED(D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_REFERENCE, NULL, flags,
            feature_level, feature_level_count, D3D11_SDK_VERSION, &device, NULL, NULL)))
        return device;

    return NULL;
}

static void get_device_adapter_desc(ID3D11Device *device, DXGI_ADAPTER_DESC *adapter_desc)
{
    IDXGIDevice *dxgi_device;
    IDXGIAdapter *adapter;
    HRESULT hr;

    hr = ID3D11Device_QueryInterface(device, &IID_IDXGIDevice, (void **)&dxgi_device);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = IDXGIDevice_GetAdapter(dxgi_device, &adapter);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    IDXGIDevice_Release(dxgi_device);
    hr = IDXGIAdapter_GetDesc(adapter, adapter_desc);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    IDXGIAdapter_Release(adapter);
}

static void print_adapter_info(void)
{
    DXGI_ADAPTER_DESC adapter_desc;
    ID3D11Device *device;

    if (!(device = create_device(NULL)))
        return;

    get_device_adapter_desc(device, &adapter_desc);
    trace("Adapter: %s, %04x:%04x.\n", wine_dbgstr_w(adapter_desc.Description),
            adapter_desc.VendorId, adapter_desc.DeviceId);
    ID3D11Device_Release(device);
}

static BOOL is_warp_device(ID3D11Device *device)
{
    DXGI_ADAPTER_DESC adapter_desc;
    get_device_adapter_desc(device, &adapter_desc);
    return !adapter_desc.SubSysId && !adapter_desc.Revision
            && ((!adapter_desc.VendorId && !adapter_desc.DeviceId)
            || (adapter_desc.VendorId == 0x1414 && adapter_desc.DeviceId == 0x008c));
}

static BOOL is_vendor_device(ID3D11Device *device, unsigned int vendor_id)
{
    DXGI_ADAPTER_DESC adapter_desc;

    if (!strcmp(winetest_platform, "wine"))
        return FALSE;

    get_device_adapter_desc(device, &adapter_desc);
    return adapter_desc.VendorId == vendor_id;
}

static BOOL is_amd_device(ID3D11Device *device)
{
    return is_vendor_device(device, 0x1002);
}

static BOOL is_intel_device(ID3D11Device *device)
{
    return is_vendor_device(device, 0x8086);
}

static BOOL is_nvidia_device(ID3D11Device *device)
{
    return is_vendor_device(device, 0x10de);
}

static BOOL is_d3d11_2_runtime(ID3D11Device *device)
{
    ID3D11Device2 *device2;
    HRESULT hr;

    hr = ID3D11Device_QueryInterface(device, &IID_ID3D11Device2, (void **)&device2);
    if (SUCCEEDED(hr))
        ID3D11Device2_Release(device2);
    return hr == S_OK;
}

static BOOL check_compute_shaders_via_sm4_support(ID3D11Device *device)
{
    D3D11_FEATURE_DATA_D3D10_X_HARDWARE_OPTIONS options;

    if (FAILED(ID3D11Device_CheckFeatureSupport(device,
            D3D11_FEATURE_D3D10_X_HARDWARE_OPTIONS, &options, sizeof(options))))
        return FALSE;
    return options.ComputeShaders_Plus_RawAndStructuredBuffers_Via_Shader_4_x;
}

static BOOL check_viewport_array_index_from_any_shader_support(ID3D11Device *device)
{
    D3D11_FEATURE_DATA_D3D11_OPTIONS3 options;

    if (FAILED(ID3D11Device_CheckFeatureSupport(device,
            D3D11_FEATURE_D3D11_OPTIONS3, &options, sizeof(options))))
        return FALSE;
    return options.VPAndRTArrayIndexFromAnyShaderFeedingRasterizer;
}

static BOOL is_buffer(ID3D11Resource *resource)
{
    D3D11_RESOURCE_DIMENSION dimension;
    ID3D11Resource_GetType(resource, &dimension);
    return dimension == D3D11_RESOURCE_DIMENSION_BUFFER;
}

static IDXGISwapChain *create_swapchain(ID3D11Device *device, HWND window,
        const struct swapchain_desc *swapchain_desc)
{
    DXGI_SWAP_CHAIN_DESC dxgi_desc;
    IDXGISwapChain *swapchain;
    IDXGIDevice *dxgi_device;
    IDXGIAdapter *adapter;
    IDXGIFactory *factory;
    HRESULT hr;

    hr = ID3D11Device_QueryInterface(device, &IID_IDXGIDevice, (void **)&dxgi_device);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = IDXGIDevice_GetAdapter(dxgi_device, &adapter);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    IDXGIDevice_Release(dxgi_device);
    hr = IDXGIAdapter_GetParent(adapter, &IID_IDXGIFactory, (void **)&factory);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    IDXGIAdapter_Release(adapter);

    dxgi_desc.BufferDesc.Width = 640;
    dxgi_desc.BufferDesc.Height = 480;
    dxgi_desc.BufferDesc.RefreshRate.Numerator = 60;
    dxgi_desc.BufferDesc.RefreshRate.Denominator = 1;
    dxgi_desc.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    dxgi_desc.BufferDesc.ScanlineOrdering = DXGI_MODE_SCANLINE_ORDER_UNSPECIFIED;
    dxgi_desc.BufferDesc.Scaling = DXGI_MODE_SCALING_UNSPECIFIED;
    dxgi_desc.SampleDesc.Count = 1;
    dxgi_desc.SampleDesc.Quality = 0;
    dxgi_desc.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
    dxgi_desc.BufferCount = 1;
    dxgi_desc.OutputWindow = window;
    dxgi_desc.Windowed = TRUE;
    dxgi_desc.SwapEffect = DXGI_SWAP_EFFECT_DISCARD;
    dxgi_desc.Flags = 0;

    if (swapchain_desc)
    {
        dxgi_desc.Windowed = swapchain_desc->windowed;
        dxgi_desc.SwapEffect = swapchain_desc->swap_effect;
        dxgi_desc.BufferCount = swapchain_desc->buffer_count;
        if (swapchain_desc->width)
            dxgi_desc.BufferDesc.Width = swapchain_desc->width;
        if (swapchain_desc->height)
            dxgi_desc.BufferDesc.Height = swapchain_desc->height;

        if (swapchain_desc->flags & SWAPCHAIN_FLAG_SHADER_INPUT)
            dxgi_desc.BufferUsage |= DXGI_USAGE_SHADER_INPUT;
    }

    hr = IDXGIFactory_CreateSwapChain(factory, (IUnknown *)device, &dxgi_desc, &swapchain);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    IDXGIFactory_Release(factory);

    return swapchain;
}

struct d3d11_test_context
{
    ID3D11Device *device;
    HWND window;
    IDXGISwapChain *swapchain;
    ID3D11Texture2D *backbuffer;
    ID3D11RenderTargetView *backbuffer_rtv;
    ID3D11DeviceContext *immediate_context;

    ID3D11InputLayout *input_layout;
    ID3D11VertexShader *vs;
    const DWORD *vs_code;
    ID3D11Buffer *vs_cb;
    ID3D11Buffer *vb;

    ID3D11PixelShader *ps;
    ID3D11Buffer *ps_cb;
};

#define init_test_context(a, b) init_test_context_(__LINE__, a, b, NULL)
#define init_test_context_ext(a, b, c) init_test_context_(__LINE__, a, b, c)
static BOOL init_test_context_(unsigned int line, struct d3d11_test_context *context,
        const D3D_FEATURE_LEVEL *feature_level, const struct swapchain_desc *swapchain_desc)
{
    unsigned int rt_width, rt_height;
    struct device_desc device_desc;
    HRESULT hr;
    RECT rect;

    memset(context, 0, sizeof(*context));

    device_desc.feature_level = feature_level;
    device_desc.flags = 0;
    if (!(context->device = create_device(&device_desc)))
    {
        skip_(__FILE__, line)("Failed to create device.\n");
        return FALSE;
    }

    rt_width = swapchain_desc && swapchain_desc->width ? swapchain_desc->width : 640;
    rt_height = swapchain_desc && swapchain_desc->height ? swapchain_desc->height : 480;
    SetRect(&rect, 0, 0, rt_width, rt_height);
    AdjustWindowRect(&rect, WS_OVERLAPPEDWINDOW | WS_VISIBLE, FALSE);
    context->window = CreateWindowA("static", "d3d11_test", WS_OVERLAPPEDWINDOW | WS_VISIBLE,
            0, 0, rect.right - rect.left, rect.bottom - rect.top, NULL, NULL, NULL, NULL);
    context->swapchain = create_swapchain(context->device, context->window, swapchain_desc);
    hr = IDXGISwapChain_GetBuffer(context->swapchain, 0, &IID_ID3D11Texture2D, (void **)&context->backbuffer);
    ok_(__FILE__, line)(hr == S_OK, "Failed to get backbuffer, hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(context->device, (ID3D11Resource *)context->backbuffer,
            NULL, &context->backbuffer_rtv);
    ok_(__FILE__, line)(hr == S_OK, "Failed to create rendertarget view, hr %#lx.\n", hr);

    ID3D11Device_GetImmediateContext(context->device, &context->immediate_context);

    ID3D11DeviceContext_OMSetRenderTargets(context->immediate_context, 1, &context->backbuffer_rtv, NULL);

    set_viewport(context->immediate_context, 0.0f, 0.0f, rt_width, rt_height, 0.0f, 1.0f);

    return TRUE;
}

#define release_test_context(context) release_test_context_(__LINE__, context)
static void release_test_context_(unsigned int line, struct d3d11_test_context *context)
{
    ULONG ref;

    if (context->input_layout)
        ID3D11InputLayout_Release(context->input_layout);
    if (context->vs)
        ID3D11VertexShader_Release(context->vs);
    if (context->vs_cb)
        ID3D11Buffer_Release(context->vs_cb);
    if (context->vb)
        ID3D11Buffer_Release(context->vb);
    if (context->ps)
        ID3D11PixelShader_Release(context->ps);
    if (context->ps_cb)
        ID3D11Buffer_Release(context->ps_cb);

    ID3D11DeviceContext_Release(context->immediate_context);
    ID3D11RenderTargetView_Release(context->backbuffer_rtv);
    ID3D11Texture2D_Release(context->backbuffer);
    IDXGISwapChain_Release(context->swapchain);
    DestroyWindow(context->window);

    ref = ID3D11Device_Release(context->device);
    ok_(__FILE__, line)(!ref, "Device has %lu references left.\n", ref);
}

#define draw_quad(context) draw_quad_vs_(__LINE__, context, NULL, 0)
#define draw_quad_vs(a, b, c) draw_quad_vs_(__LINE__, a, b, c)
static void draw_quad_vs_(unsigned int line, struct d3d11_test_context *context,
        const DWORD *vs_code, size_t vs_code_size)
{
    static const D3D11_INPUT_ELEMENT_DESC default_layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    static const DWORD default_vs_code[] =
    {
#if 0
        float4 main(float4 position : POSITION) : SV_POSITION
        {
            return position;
        }
#endif
        0x43425844, 0x4fb19b86, 0x955fa240, 0x1a630688, 0x24eb9db4, 0x00000001, 0x000001e0, 0x00000006,
        0x00000038, 0x00000084, 0x000000d0, 0x00000134, 0x00000178, 0x000001ac, 0x53414e58, 0x00000044,
        0x00000044, 0xfffe0200, 0x00000020, 0x00000024, 0x00240000, 0x00240000, 0x00240000, 0x00240000,
        0x00240000, 0xfffe0200, 0x0200001f, 0x80000005, 0x900f0000, 0x02000001, 0xc00f0000, 0x80e40000,
        0x0000ffff, 0x50414e58, 0x00000044, 0x00000044, 0xfffe0200, 0x00000020, 0x00000024, 0x00240000,
        0x00240000, 0x00240000, 0x00240000, 0x00240000, 0xfffe0200, 0x0200001f, 0x80000005, 0x900f0000,
        0x02000001, 0xc00f0000, 0x80e40000, 0x0000ffff, 0x396e6f41, 0x0000005c, 0x0000005c, 0xfffe0200,
        0x00000034, 0x00000028, 0x00240000, 0x00240000, 0x00240000, 0x00240000, 0x00240001, 0x00000000,
        0xfffe0200, 0x0200001f, 0x80000005, 0x900f0000, 0x04000004, 0xc0030000, 0x90ff0000, 0xa0e40000,
        0x90e40000, 0x02000001, 0xc00c0000, 0x90e40000, 0x0000ffff, 0x52444853, 0x0000003c, 0x00010040,
        0x0000000f, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e, 0x4e475349, 0x0000002c,
        0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f,
        0x49534f50, 0x4e4f4954, 0xababab00, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49,
    };
    static const struct vec3 quad[] =
    {
        {-1.0f, -1.0f, 0.0f},
        {-1.0f,  1.0f, 0.0f},
        { 1.0f, -1.0f, 0.0f},
        { 1.0f,  1.0f, 0.0f},
    };

    ID3D11Device *device = context->device;
    unsigned int stride, offset;
    HRESULT hr;

    if (!vs_code)
    {
        vs_code = default_vs_code;
        vs_code_size = sizeof(default_vs_code);
    }

    if (!context->input_layout)
    {
        hr = ID3D11Device_CreateInputLayout(device, default_layout_desc, ARRAY_SIZE(default_layout_desc),
                vs_code, vs_code_size, &context->input_layout);
        ok_(__FILE__, line)(hr == S_OK, "Failed to create input layout, hr %#lx.\n", hr);
    }

    if (context->vs_code != vs_code)
    {
        if (context->vs)
            ID3D11VertexShader_Release(context->vs);

        hr = ID3D11Device_CreateVertexShader(device, vs_code, vs_code_size, NULL, &context->vs);
        ok_(__FILE__, line)(hr == S_OK, "Failed to create vertex shader, hr %#lx.\n", hr);

        context->vs_code = vs_code;
    }

    if (!context->vb)
        context->vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(quad), quad);

    ID3D11DeviceContext_IASetInputLayout(context->immediate_context, context->input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context->immediate_context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    stride = sizeof(*quad);
    offset = 0;
    ID3D11DeviceContext_IASetVertexBuffers(context->immediate_context, 0, 1, &context->vb, &stride, &offset);
    ID3D11DeviceContext_VSSetShader(context->immediate_context, context->vs, NULL, 0);

    ID3D11DeviceContext_Draw(context->immediate_context, 4, 0);
}

#define draw_quad_z(context, z) draw_quad_z_(__LINE__, context, z)
static void draw_quad_z_(unsigned int line, struct d3d11_test_context *context, float z)
{
    static const DWORD vs_code[] =
    {
#if 0
        float depth;

        void main(float4 in_position : POSITION, out float4 out_position : SV_Position)
        {
            out_position = in_position;
            out_position.z = depth;
        }
#endif
        0x43425844, 0x22d7ff76, 0xd53b167c, 0x1b49ccf1, 0xbebfec39, 0x00000001, 0x00000100, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000b0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x52444853, 0x00000064, 0x00010040,
        0x00000019, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005f, 0x001010b2, 0x00000000,
        0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x05000036, 0x001020b2, 0x00000000, 0x00101c46,
        0x00000000, 0x06000036, 0x00102042, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x0100003e,
    };

    struct vec4 data = {z};

    if (!context->vs_cb)
        context->vs_cb = create_buffer(context->device, D3D11_BIND_CONSTANT_BUFFER, sizeof(data), NULL);

    ID3D11DeviceContext_UpdateSubresource(context->immediate_context,
            (ID3D11Resource *)context->vs_cb, 0, NULL, &data, 0, 0);

    ID3D11DeviceContext_VSSetConstantBuffers(context->immediate_context, 0, 1, &context->vs_cb);
    draw_quad_vs_(__LINE__, context, vs_code, sizeof(vs_code));
}

static void set_quad_color(struct d3d11_test_context *context, const struct vec4 *color)
{
    ID3D11DeviceContext_UpdateSubresource(context->immediate_context,
            (ID3D11Resource *)context->ps_cb, 0, NULL, color, 0, 0);
}

#define draw_color_quad(a, b) draw_color_quad_(__LINE__, a, b, NULL, 0)
#define draw_color_quad_vs(a, b, c, d) draw_color_quad_(__LINE__, a, b, c, d)
static void draw_color_quad_(unsigned int line, struct d3d11_test_context *context,
        const struct vec4 *color, const DWORD *vs_code, size_t vs_code_size)
{
    static const DWORD ps_color_code[] =
    {
#if 0
        float4 color;

        float4 main() : SV_TARGET
        {
            return color;
        }
#endif
        0x43425844, 0xe7ffb369, 0x72bb84ee, 0x6f684dcd, 0xd367d788, 0x00000001, 0x00000158, 0x00000005,
        0x00000034, 0x00000080, 0x000000cc, 0x00000114, 0x00000124, 0x53414e58, 0x00000044, 0x00000044,
        0xffff0200, 0x00000014, 0x00000030, 0x00240001, 0x00300000, 0x00300000, 0x00240000, 0x00300000,
        0x00000000, 0x00000001, 0x00000000, 0xffff0200, 0x02000001, 0x800f0800, 0xa0e40000, 0x0000ffff,
        0x396e6f41, 0x00000044, 0x00000044, 0xffff0200, 0x00000014, 0x00000030, 0x00240001, 0x00300000,
        0x00300000, 0x00240000, 0x00300000, 0x00000000, 0x00000001, 0x00000000, 0xffff0200, 0x02000001,
        0x800f0800, 0xa0e40000, 0x0000ffff, 0x52444853, 0x00000040, 0x00000040, 0x00000010, 0x04000059,
        0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x06000036, 0x001020f2,
        0x00000000, 0x00208e46, 0x00000000, 0x00000000, 0x0100003e, 0x4e475349, 0x00000008, 0x00000000,
        0x00000008, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054,
    };

    ID3D11Device *device = context->device;
    HRESULT hr;

    if (!context->ps)
    {
        hr = ID3D11Device_CreatePixelShader(device, ps_color_code, sizeof(ps_color_code), NULL, &context->ps);
        ok_(__FILE__, line)(hr == S_OK, "Failed to create pixel shader, hr %#lx.\n", hr);
    }

    if (!context->ps_cb)
        context->ps_cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(*color), NULL);

    ID3D11DeviceContext_PSSetShader(context->immediate_context, context->ps, NULL, 0);
    ID3D11DeviceContext_PSSetConstantBuffers(context->immediate_context, 0, 1, &context->ps_cb);

    set_quad_color(context, color);

    draw_quad_vs_(line, context, vs_code, vs_code_size);
}

static void test_create_device(void)
{
    static const D3D_FEATURE_LEVEL default_feature_levels[] =
    {
        D3D_FEATURE_LEVEL_11_0,
        D3D_FEATURE_LEVEL_10_1,
        D3D_FEATURE_LEVEL_10_0,
        D3D_FEATURE_LEVEL_9_3,
        D3D_FEATURE_LEVEL_9_2,
        D3D_FEATURE_LEVEL_9_1,
    };
    D3D_FEATURE_LEVEL feature_level, supported_feature_level;
    DXGI_SWAP_CHAIN_DESC swapchain_desc, obtained_desc;
    ID3D11DeviceContext *immediate_context;
    IDXGISwapChain *swapchain;
    ID3D11Device *device;
    ULONG refcount;
    HWND window;
    HRESULT hr;

    if (FAILED(hr = D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &device, NULL, NULL)))
    {
        skip("Failed to create HAL device.\n");
        if ((device = create_device(NULL)))
        {
            trace("Feature level %#x.\n", ID3D11Device_GetFeatureLevel(device));
            ID3D11Device_Release(device);
        }
        return;
    }

    supported_feature_level = ID3D11Device_GetFeatureLevel(device);
    trace("Feature level %#x.\n", supported_feature_level);
    ID3D11Device_Release(device);

    hr = D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION, NULL, NULL, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);

    hr = D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION, NULL,
            &feature_level, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);
    ok(feature_level == supported_feature_level, "Got feature level %#x, expected %#x.\n",
            feature_level, supported_feature_level);

    hr = D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, default_feature_levels,
            ARRAY_SIZE(default_feature_levels), D3D11_SDK_VERSION, NULL, &feature_level, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);
    ok(feature_level == supported_feature_level, "Got feature level %#x, expected %#x.\n",
            feature_level, supported_feature_level);

    hr = D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION, NULL, NULL,
            &immediate_context);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ok(!!immediate_context, "Expected immediate device context pointer, got NULL.\n");
    refcount = get_refcount(immediate_context);
    ok(refcount == 1, "Got refcount %lu, expected 1.\n", refcount);

    ID3D11DeviceContext_GetDevice(immediate_context, &device);
    refcount = ID3D11Device_Release(device);
    ok(refcount == 1, "Got refcount %lu, expected 1.\n", refcount);

    refcount = ID3D11DeviceContext_Release(immediate_context);
    ok(!refcount, "ID3D11DeviceContext has %lu references left.\n", refcount);

    device = (ID3D11Device *)0xdeadbeef;
    feature_level = 0xdeadbeef;
    immediate_context = (ID3D11DeviceContext *)0xdeadbeef;
    hr = D3D11CreateDevice(NULL, D3D_DRIVER_TYPE_UNKNOWN, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &device, &feature_level, &immediate_context);
    todo_wine ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!device, "Got unexpected device pointer %p.\n", device);
    ok(!feature_level, "Got unexpected feature level %#x.\n", feature_level);
    ok(!immediate_context, "Got unexpected immediate context pointer %p.\n", immediate_context);

    window = CreateWindowA("static", "d3d11_test", 0, 0, 0, 0, 0, 0, 0, 0, 0);

    swapchain_desc.BufferDesc.Width = 800;
    swapchain_desc.BufferDesc.Height = 600;
    swapchain_desc.BufferDesc.RefreshRate.Numerator = 60;
    swapchain_desc.BufferDesc.RefreshRate.Denominator = 60;
    swapchain_desc.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    swapchain_desc.BufferDesc.ScanlineOrdering = DXGI_MODE_SCANLINE_ORDER_UNSPECIFIED;
    swapchain_desc.BufferDesc.Scaling = DXGI_MODE_SCALING_UNSPECIFIED;
    swapchain_desc.SampleDesc.Count = 1;
    swapchain_desc.SampleDesc.Quality = 0;
    swapchain_desc.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
    swapchain_desc.BufferCount = 1;
    swapchain_desc.OutputWindow = window;
    swapchain_desc.Windowed = TRUE;
    swapchain_desc.SwapEffect = DXGI_SWAP_EFFECT_DISCARD;
    swapchain_desc.Flags = 0;

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, NULL, NULL, NULL, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, NULL, NULL, &feature_level, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);
    ok(feature_level == supported_feature_level, "Got feature level %#x, expected %#x.\n",
            feature_level, supported_feature_level);

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, &swapchain, &device, NULL, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    check_interface(swapchain, &IID_IDXGISwapChain1, TRUE, FALSE);

    memset(&obtained_desc, 0, sizeof(obtained_desc));
    hr = IDXGISwapChain_GetDesc(swapchain, &obtained_desc);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(obtained_desc.BufferDesc.Width == swapchain_desc.BufferDesc.Width,
            "Got unexpected BufferDesc.Width %u.\n", obtained_desc.BufferDesc.Width);
    ok(obtained_desc.BufferDesc.Height == swapchain_desc.BufferDesc.Height,
            "Got unexpected BufferDesc.Height %u.\n", obtained_desc.BufferDesc.Height);
    todo_wine ok(obtained_desc.BufferDesc.RefreshRate.Numerator == swapchain_desc.BufferDesc.RefreshRate.Numerator,
            "Got unexpected BufferDesc.RefreshRate.Numerator %u.\n",
            obtained_desc.BufferDesc.RefreshRate.Numerator);
    todo_wine ok(obtained_desc.BufferDesc.RefreshRate.Denominator == swapchain_desc.BufferDesc.RefreshRate.Denominator,
            "Got unexpected BufferDesc.RefreshRate.Denominator %u.\n",
            obtained_desc.BufferDesc.RefreshRate.Denominator);
    ok(obtained_desc.BufferDesc.Format == swapchain_desc.BufferDesc.Format,
            "Got unexpected BufferDesc.Format %#x.\n", obtained_desc.BufferDesc.Format);
    ok(obtained_desc.BufferDesc.ScanlineOrdering == swapchain_desc.BufferDesc.ScanlineOrdering,
            "Got unexpected BufferDesc.ScanlineOrdering %#x.\n", obtained_desc.BufferDesc.ScanlineOrdering);
    ok(obtained_desc.BufferDesc.Scaling == swapchain_desc.BufferDesc.Scaling,
            "Got unexpected BufferDesc.Scaling %#x.\n", obtained_desc.BufferDesc.Scaling);
    ok(obtained_desc.SampleDesc.Count == swapchain_desc.SampleDesc.Count,
            "Got unexpected SampleDesc.Count %u.\n", obtained_desc.SampleDesc.Count);
    ok(obtained_desc.SampleDesc.Quality == swapchain_desc.SampleDesc.Quality,
            "Got unexpected SampleDesc.Quality %u.\n", obtained_desc.SampleDesc.Quality);
    ok(obtained_desc.BufferUsage == swapchain_desc.BufferUsage,
            "Got unexpected BufferUsage %#x.\n", obtained_desc.BufferUsage);
    ok(obtained_desc.BufferCount == swapchain_desc.BufferCount,
            "Got unexpected BufferCount %u.\n", obtained_desc.BufferCount);
    ok(obtained_desc.OutputWindow == swapchain_desc.OutputWindow,
            "Got unexpected OutputWindow %p.\n", obtained_desc.OutputWindow);
    ok(obtained_desc.Windowed == swapchain_desc.Windowed,
            "Got unexpected Windowed %#x.\n", obtained_desc.Windowed);
    ok(obtained_desc.SwapEffect == swapchain_desc.SwapEffect,
            "Got unexpected SwapEffect %#x.\n", obtained_desc.SwapEffect);
    ok(obtained_desc.Flags == swapchain_desc.Flags,
            "Got unexpected Flags %#x.\n", obtained_desc.Flags);

    refcount = IDXGISwapChain_Release(swapchain);
    ok(!refcount, "Swapchain has %lu references left.\n", refcount);

    feature_level = ID3D11Device_GetFeatureLevel(device);
    ok(feature_level == supported_feature_level, "Got feature level %#x, expected %#x.\n",
            feature_level, supported_feature_level);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            NULL, NULL, &device, NULL, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Device_Release(device);

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            NULL, NULL, NULL, NULL, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            NULL, NULL, NULL, &feature_level, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);
    ok(feature_level == supported_feature_level, "Got feature level %#x, expected %#x.\n",
            feature_level, supported_feature_level);

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            NULL, NULL, NULL, NULL, &immediate_context);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_Release(immediate_context);

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, NULL, NULL, NULL, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);

    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, &swapchain, NULL, NULL, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    IDXGISwapChain_Release(swapchain);

    swapchain_desc.OutputWindow = NULL;
    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, NULL, NULL, NULL, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);
    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, NULL, &device, NULL, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Device_Release(device);

    swapchain = (IDXGISwapChain *)0xdeadbeef;
    device = (ID3D11Device *)0xdeadbeef;
    feature_level = 0xdeadbeef;
    immediate_context = (ID3D11DeviceContext *)0xdeadbeef;
    swapchain_desc.OutputWindow = NULL;
    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, &swapchain, &device, &feature_level, &immediate_context);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    ok(!swapchain, "Got unexpected swapchain pointer %p.\n", swapchain);
    ok(!device, "Got unexpected device pointer %p.\n", device);
    ok(!feature_level, "Got unexpected feature level %#x.\n", feature_level);
    ok(!immediate_context, "Got unexpected immediate context pointer %p.\n", immediate_context);

    swapchain = (IDXGISwapChain *)0xdeadbeef;
    device = (ID3D11Device *)0xdeadbeef;
    feature_level = 0xdeadbeef;
    immediate_context = (ID3D11DeviceContext *)0xdeadbeef;
    swapchain_desc.OutputWindow = window;
    swapchain_desc.BufferDesc.Format = DXGI_FORMAT_BC5_UNORM;
    hr = D3D11CreateDeviceAndSwapChain(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, 0, NULL, 0, D3D11_SDK_VERSION,
            &swapchain_desc, &swapchain, &device, &feature_level, &immediate_context);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!swapchain, "Got unexpected swapchain pointer %p.\n", swapchain);
    ok(!device, "Got unexpected device pointer %p.\n", device);
    ok(!feature_level, "Got unexpected feature level %#x.\n", feature_level);
    ok(!immediate_context, "Got unexpected immediate context pointer %p.\n", immediate_context);

    DestroyWindow(window);
}

static void test_device_interfaces(const D3D_FEATURE_LEVEL feature_level)
{
    struct device_desc device_desc;
    IDXGIAdapter *dxgi_adapter;
    IDXGIDevice *dxgi_device;
    ID3D11Device *device;
    IUnknown *iface;
    ULONG refcount;
    HRESULT hr;

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }

    check_interface(device, &IID_IUnknown, TRUE, FALSE);
    check_interface(device, &IID_IDXGIObject, TRUE, FALSE);
    check_interface(device, &IID_IDXGIDevice, TRUE, FALSE);
    check_interface(device, &IID_IDXGIDevice1, TRUE, FALSE);
    check_interface(device, &IID_ID3D10Multithread, TRUE, TRUE); /* Not available on all Windows versions. */
    check_interface(device, &IID_ID3D10Device, FALSE, FALSE);
    check_interface(device, &IID_ID3D10Device1, FALSE, FALSE);
    check_interface(device, &IID_ID3D11InfoQueue, enable_debug_layer, FALSE);

    hr = ID3D11Device_QueryInterface(device, &IID_IDXGIDevice, (void **)&dxgi_device);
    ok(SUCCEEDED(hr), "Device should implement IDXGIDevice.\n");
    hr = IDXGIDevice_GetParent(dxgi_device, &IID_IDXGIAdapter, (void **)&dxgi_adapter);
    ok(SUCCEEDED(hr), "Device parent should implement IDXGIAdapter.\n");
    hr = IDXGIAdapter_GetParent(dxgi_adapter, &IID_IDXGIFactory, (void **)&iface);
    ok(SUCCEEDED(hr), "Adapter parent should implement IDXGIFactory.\n");
    IUnknown_Release(iface);
    IDXGIAdapter_Release(dxgi_adapter);
    hr = IDXGIDevice_GetParent(dxgi_device, &IID_IDXGIAdapter1, (void **)&dxgi_adapter);
    ok(SUCCEEDED(hr), "Device parent should implement IDXGIAdapter1.\n");
    hr = IDXGIAdapter_GetParent(dxgi_adapter, &IID_IDXGIFactory1, (void **)&iface);
    ok(SUCCEEDED(hr), "Adapter parent should implement IDXGIFactory1.\n");
    IUnknown_Release(iface);
    IDXGIAdapter_Release(dxgi_adapter);
    IDXGIDevice_Release(dxgi_device);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);

    device_desc.feature_level = &feature_level;
    device_desc.flags = D3D11_CREATE_DEVICE_DEBUG;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create debug device.\n");
        return;
    }

    todo_wine check_interface(device, &IID_ID3D11InfoQueue, TRUE, FALSE);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_immediate_context(void)
{
    ID3D11DeviceContext *immediate_context, *previous_immediate_context;
    ULONG expected_refcount, refcount;
    ID3D11CommandList *command_list;
    ID3D11Multithread *multithread;
    ID3D11Buffer *buffer[2];
    ID3D11Device *device;
    unsigned int flags;
    BOOL enabled;
    HRESULT hr;

    static const unsigned int buffer_contents[] =
    {
        0x11111111, 0x22222222, 0x33333333, 0x44444444,
        0x55555555, 0x66666666, 0x77777777, 0x88888888,
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    expected_refcount = get_refcount(device) + 1;
    ID3D11Device_GetImmediateContext(device, &immediate_context);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu.\n", refcount);
    previous_immediate_context = immediate_context;

    ID3D11Device_GetImmediateContext(device, &immediate_context);
    ok(immediate_context == previous_immediate_context, "Got different immediate device context objects.\n");
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu.\n", refcount);

    refcount = ID3D11DeviceContext_Release(previous_immediate_context);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3D11DeviceContext_Release(immediate_context);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);

    ID3D11Device_GetImmediateContext(device, &immediate_context);
    ok(immediate_context == previous_immediate_context, "Got different immediate device context objects.\n");
    refcount = ID3D11DeviceContext_Release(immediate_context);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);

    ID3D11Device_GetImmediateContext(device, &immediate_context);
    expected_refcount = get_refcount(immediate_context) + 1;
    hr = ID3D11DeviceContext_QueryInterface(immediate_context, &IID_ID3D11Multithread, (void **)&multithread);
    if (hr == E_NOINTERFACE)
    {
        win_skip("ID3D11Multithread is not supported.\n");
        goto done;
    }
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    refcount = get_refcount(immediate_context);
    ok(refcount == expected_refcount, "Got refcount %lu, expected %lu.\n", refcount, expected_refcount);

    expected_refcount = refcount;
    refcount = get_refcount(multithread);
    ok(refcount == expected_refcount, "Got refcount %lu, expected %lu.\n", refcount, expected_refcount);

    enabled = ID3D11Multithread_GetMultithreadProtected(multithread);
    todo_wine ok(!enabled, "Multithread protection is %#x.\n", enabled);

    ID3D11Multithread_Release(multithread);

    ID3D11Device_GetImmediateContext(device, &immediate_context);

    flags = ID3D11DeviceContext_GetContextFlags(immediate_context);
    ok(!flags, "Got unexpected flags %#x.\n", flags);

    hr = ID3D11DeviceContext_FinishCommandList(immediate_context, FALSE, &command_list);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);

    buffer[0] = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, 16, &buffer_contents[0]);
    buffer[1] = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, 16, &buffer_contents[4]);

    ID3D11DeviceContext_CopyResource(immediate_context, (ID3D11Resource *)buffer[1], (ID3D11Resource *)buffer[0]);

    hr = ID3D11DeviceContext_FinishCommandList(immediate_context, FALSE, &command_list);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);

    ID3D11Buffer_Release(buffer[1]);
    ID3D11Buffer_Release(buffer[0]);
    ID3D11DeviceContext_Release(immediate_context);

done:
    refcount = ID3D11DeviceContext_Release(immediate_context);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_deferred_context(void)
{
    ULONG refcount, expected_refcount;
    struct device_desc device_desc;
    ID3D11DeviceContext *context;
    ID3D11Device *device;
    HRESULT hr;

    device_desc.feature_level = NULL;
    device_desc.flags = D3D11_CREATE_DEVICE_SINGLETHREADED;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create single-threaded device.\n");
        return;
    }

    hr = ID3D11Device_CreateDeferredContext(device, 0, &context);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    if (hr == S_OK)
        ID3D11DeviceContext_Release(context);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateDeferredContext(device, 0, &context);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got refcount %lu, expected %lu.\n", refcount, expected_refcount);
    refcount = get_refcount(context);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    check_interface(context, &IID_IUnknown, TRUE, FALSE);
    check_interface(context, &IID_ID3D11DeviceChild, TRUE, FALSE);
    check_interface(context, &IID_ID3D11DeviceContext, TRUE, FALSE);
    check_interface(context, &IID_ID3D11Multithread, FALSE, FALSE);

    refcount = ID3D11DeviceContext_Release(context);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_texture1d(void)
{
    ULONG refcount, expected_refcount;
    D3D11_SUBRESOURCE_DATA data = {0};
    ID3D11Device *device, *tmp;
    D3D11_TEXTURE1D_DESC desc;
    ID3D11Texture1D *texture;
    unsigned int i;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    desc.Width = 512;
    desc.MipLevels = 1;
    desc.ArraySize = 1;
    desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    desc.Usage = D3D11_USAGE_DEFAULT;
    desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    desc.CPUAccessFlags = 0;
    desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture1D(device, &desc, &data, &texture);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateTexture1D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Texture1D_GetDevice(texture, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    check_interface(texture, &IID_IDXGISurface, TRUE, FALSE);
    ID3D11Texture1D_Release(texture);

    desc.MipLevels = 0;
    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateTexture1D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Texture1D_GetDevice(texture, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    ID3D11Texture1D_GetDesc(texture, &desc);
    ok(desc.Width == 512, "Got unexpected Width %u.\n", desc.Width);
    ok(desc.MipLevels == 10, "Got unexpected MipLevels %u.\n", desc.MipLevels);
    ok(desc.ArraySize == 1, "Got unexpected ArraySize %u.\n", desc.ArraySize);
    ok(desc.Format == DXGI_FORMAT_R8G8B8A8_UNORM, "Got unexpected Format %#x.\n", desc.Format);
    ok(desc.Usage == D3D11_USAGE_DEFAULT, "Got unexpected Usage %u.\n", desc.Usage);
    ok(desc.BindFlags == D3D11_BIND_SHADER_RESOURCE, "Got unexpected BindFlags %#x.\n", desc.BindFlags);
    ok(desc.CPUAccessFlags == 0, "Got unexpected CPUAccessFlags %#x.\n", desc.CPUAccessFlags);
    ok(desc.MiscFlags == 0, "Got unexpected MiscFlags %#x.\n", desc.MiscFlags);

    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    ID3D11Texture1D_Release(texture);

    desc.MipLevels = 1;
    desc.ArraySize = 2;
    hr = ID3D11Device_CreateTexture1D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    ID3D11Texture1D_Release(texture);

    for (i = 0; i < 4; ++i)
    {
        desc.ArraySize = i;
        desc.Format = DXGI_FORMAT_R32G32B32A32_TYPELESS;
        desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
        desc.MiscFlags = 0;
        hr = ID3D11Device_CreateTexture1D(device, &desc, NULL, &texture);
        ok(hr == (i ? S_OK : E_INVALIDARG), "Test %u: Got unexpected hr %#lx.\n", i, hr);
        if (SUCCEEDED(hr))
            ID3D11Texture1D_Release(texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

#define test_dxgi_resource(a) test_dxgi_resource_(__LINE__, a)
static void test_dxgi_resource_(unsigned int line, void *iface)
{
    IDXGIResource *resource, *resource2;
    IDXGISurface *surface, *surface2;
    IUnknown *object = iface, *unk;
    DWORD data;
    HRESULT hr;
    UINT size;

    hr = IUnknown_QueryInterface(object, &IID_IDXGIResource, (void **)&resource);
    ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    if (FAILED(hr)) return;

    if (SUCCEEDED(IUnknown_QueryInterface(object, &IID_IDXGISurface, (void **)&surface)))
    {
        hr = IDXGISurface_QueryInterface(surface, &IID_IDXGIDeviceSubObject, (void **)&unk);
        ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        ok_(__FILE__, line)((IUnknown *)resource == unk, "Unexpected interface pointer.\n");
        IUnknown_Release(unk);

        hr = IDXGISurface_QueryInterface(surface, &IID_IDXGIObject, (void **)&unk);
        ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        ok_(__FILE__, line)((IUnknown *)resource == unk, "Unexpected interface pointer.\n");
        IUnknown_Release(unk);

        hr = IDXGISurface_QueryInterface(surface, &IID_IDXGIResource, (void **)&resource2);
        ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        ok_(__FILE__, line)(resource2 == resource, "Unexpected resource pointer.\n");
        IDXGIResource_Release(resource2);

        hr = IDXGIResource_QueryInterface(resource, &IID_IDXGISurface, (void **)&surface2);
        ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        ok_(__FILE__, line)(surface2 == surface, "Unexpected surface pointer.\n");
        IDXGISurface_Release(surface2);

        hr = IDXGISurface_GetParent(surface, &IID_IDXGIResource, (void **)&resource2);
        ok_(__FILE__, line)(hr == E_NOINTERFACE, "Got unexpected hr %#lx.\n", hr);
        hr = IDXGIResource_GetParent(resource, &IID_IDXGISurface, (void **)&surface2);
        ok_(__FILE__, line)(hr == E_NOINTERFACE, "Got unexpected hr %#lx.\n", hr);

        data = 123;
        hr = IDXGIResource_SetPrivateData(resource, &IID_IUnknown, sizeof(data), &data);
        ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

        size = sizeof(data);
        data = 0;
        hr = IDXGISurface_GetPrivateData(surface, &IID_IUnknown, &size, &data);
        ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        ok_(__FILE__, line)(data == 123, "Unexpected data %#lx.\n", data);

        IDXGISurface_Release(surface);
    }

    IDXGIResource_Release(resource);
}

static void test_texture1d_interfaces(void)
{
    ID3D10Texture1D *d3d10_texture;
    D3D11_TEXTURE1D_DESC desc;
    ID3D11Texture1D *texture;
    ID3D11Device *device;
    unsigned int i;
    ULONG refcount;
    HRESULT hr;

    static const struct test
    {
        BOOL implements_d3d10_interfaces;
        UINT bind_flags;
        UINT misc_flags;
        UINT expected_bind_flags;
        UINT expected_misc_flags;
    }
    desc_conversion_tests[] =
    {
        {
            TRUE,
            D3D11_BIND_SHADER_RESOURCE, 0,
            D3D10_BIND_SHADER_RESOURCE, 0
        },
        {
            TRUE,
            D3D11_BIND_UNORDERED_ACCESS, 0,
            D3D11_BIND_UNORDERED_ACCESS, 0
        },
        {
            FALSE,
            0, D3D11_RESOURCE_MISC_RESOURCE_CLAMP,
            0, 0
        },
        {
            TRUE,
            0, D3D11_RESOURCE_MISC_SHARED_KEYEDMUTEX,
            0, D3D10_RESOURCE_MISC_SHARED_KEYEDMUTEX
        },
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create ID3D11Device, skipping tests.\n");
        return;
    }

    desc.Width = 512;
    desc.MipLevels = 1;
    desc.ArraySize = 1;
    desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    desc.Usage = D3D11_USAGE_DEFAULT;
    desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    desc.CPUAccessFlags = 0;
    desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture1D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    test_dxgi_resource(texture);
    ID3D11Texture1D_Release(texture);

    desc.MipLevels = 0;
    hr = ID3D11Device_CreateTexture1D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    test_dxgi_resource(texture);
    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    check_interface(texture, &IID_IDXGIResource, TRUE, FALSE);
    hr = check_interface(texture, &IID_ID3D10Texture1D, TRUE, TRUE); /* Not available on all Windows versions. */
    ID3D11Texture1D_Release(texture);
    if (FAILED(hr))
    {
        win_skip("1D textures do not implement ID3D10Texture1D, skipping tests.\n");
        ID3D11Device_Release(device);
        return;
    }

    for (i = 0; i < ARRAY_SIZE(desc_conversion_tests); ++i)
    {
        const struct test *current = &desc_conversion_tests[i];
        D3D10_TEXTURE1D_DESC d3d10_desc;
        ID3D10Device *d3d10_device;

        desc.Width = 512;
        desc.MipLevels = 1;
        desc.ArraySize = 1;
        desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
        desc.Usage = D3D11_USAGE_DEFAULT;
        desc.BindFlags = current->bind_flags;
        desc.CPUAccessFlags = 0;
        desc.MiscFlags = current->misc_flags;

        hr = ID3D11Device_CreateTexture1D(device, &desc, NULL, &texture);
        /* Shared resources are not supported by REF and WARP devices. */
        ok(SUCCEEDED(hr) || broken(hr == E_OUTOFMEMORY),
                "Test %u: Failed to create a 1d texture, hr %#lx.\n", i, hr);
        if (FAILED(hr))
        {
            win_skip("Failed to create ID3D11Texture1D, skipping test %u.\n", i);
            continue;
        }

        check_interface(texture, &IID_IDXGISurface, TRUE, FALSE);

        hr = ID3D11Texture1D_QueryInterface(texture, &IID_ID3D10Texture1D, (void **)&d3d10_texture);
        ID3D11Texture1D_Release(texture);

        if (current->implements_d3d10_interfaces)
        {
            ok(SUCCEEDED(hr), "Test %u: Texture should implement ID3D10Texture1D.\n", i);
        }
        else
        {
            todo_wine ok(hr == E_NOINTERFACE, "Test %u: Texture should not implement ID3D10Texture1D.\n", i);
            if (SUCCEEDED(hr)) ID3D10Texture1D_Release(d3d10_texture);
            continue;
        }

        ID3D10Texture1D_GetDesc(d3d10_texture, &d3d10_desc);

        ok(d3d10_desc.Width == desc.Width,
                "Test %u: Got unexpected Width %u.\n", i, d3d10_desc.Width);
        ok(d3d10_desc.MipLevels == desc.MipLevels,
                "Test %u: Got unexpected MipLevels %u.\n", i, d3d10_desc.MipLevels);
        ok(d3d10_desc.ArraySize == desc.ArraySize,
                "Test %u: Got unexpected ArraySize %u.\n", i, d3d10_desc.ArraySize);
        ok(d3d10_desc.Format == desc.Format,
                "Test %u: Got unexpected Format %u.\n", i, d3d10_desc.Format);
        ok(d3d10_desc.BindFlags == current->expected_bind_flags,
                "Test %u: Got unexpected BindFlags %#x.\n", i, d3d10_desc.BindFlags);
        ok(d3d10_desc.CPUAccessFlags == desc.CPUAccessFlags,
                "Test %u: Got unexpected CPUAccessFlags %#x.\n", i, d3d10_desc.CPUAccessFlags);
        ok(d3d10_desc.MiscFlags == current->expected_misc_flags,
                "Test %u: Got unexpected MiscFlags %#x.\n", i, d3d10_desc.MiscFlags);

        d3d10_device = (ID3D10Device *)0xdeadbeef;
        ID3D10Texture1D_GetDevice(d3d10_texture, &d3d10_device);
        ok(!d3d10_device, "Test %u: Got unexpected device pointer %p, expected NULL.\n", i, d3d10_device);
        if (d3d10_device) ID3D10Device_Release(d3d10_device);

        ID3D10Texture1D_Release(d3d10_texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_texture2d(void)
{
    ULONG refcount, expected_refcount;
    D3D11_SUBRESOURCE_DATA data = {0};
    D3D_FEATURE_LEVEL feature_level;
    ID3D11Device *device, *tmp;
    D3D11_TEXTURE2D_DESC desc;
    ID3D11Texture2D *texture;
    UINT quality_level_count;
    unsigned int i;
    HRESULT hr;

    static const struct
    {
        DXGI_FORMAT format;
        UINT array_size;
        D3D11_BIND_FLAG bind_flags;
        UINT misc_flags;
        BOOL succeeds;
        BOOL todo;
    }
    tests[] =
    {
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   1, D3D11_BIND_VERTEX_BUFFER,    0, FALSE, TRUE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   1, D3D11_BIND_INDEX_BUFFER,     0, FALSE, TRUE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   1, D3D11_BIND_CONSTANT_BUFFER,  0, FALSE, TRUE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   0, D3D11_BIND_SHADER_RESOURCE,  0, FALSE, FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   2, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   3, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   3, D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_TEXTURECUBE,
                FALSE, FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   1, D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_TEXTURECUBE,
                FALSE, FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   5, D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_TEXTURECUBE,
                FALSE, FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   6, D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_TEXTURECUBE,
                TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   7, D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_TEXTURECUBE,
                TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,  10, D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_TEXTURECUBE,
                TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,  12, D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_TEXTURECUBE,
                TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   0, D3D11_BIND_RENDER_TARGET,    0, FALSE, FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   2, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   9, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS,   1, D3D11_BIND_DEPTH_STENCIL,    0, FALSE, FALSE},
        {DXGI_FORMAT_R32G32B32A32_UINT,       1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32A32_SINT,       1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32B32_TYPELESS,      1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R16G16B16A16_TYPELESS,   1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R16G16B16A16_TYPELESS,   1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G32_TYPELESS,         1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G8X24_TYPELESS,       1, D3D11_BIND_DEPTH_STENCIL,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32G8X24_TYPELESS,       1, D3D11_BIND_UNORDERED_ACCESS, 0, FALSE, TRUE},
        {DXGI_FORMAT_X32_TYPELESS_G8X24_UINT, 1, D3D11_BIND_UNORDERED_ACCESS, 0, FALSE, TRUE},
        {DXGI_FORMAT_R10G10B10A2_TYPELESS,    1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R10G10B10A2_TYPELESS,    1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R16G16_TYPELESS,         1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R16G16_UNORM,            1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R16G16_SNORM,            1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32_TYPELESS,            0, D3D11_BIND_SHADER_RESOURCE,  0, FALSE, FALSE},
        {DXGI_FORMAT_R32_TYPELESS,            1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R32_TYPELESS,            9, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R32_TYPELESS,            9, D3D11_BIND_SHADER_RESOURCE | D3D11_BIND_DEPTH_STENCIL, 0,
                TRUE,  FALSE},
        {DXGI_FORMAT_R32_TYPELESS,            9, D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_TEXTURECUBE,
                TRUE,  FALSE},
        {DXGI_FORMAT_R32_TYPELESS,            1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32_TYPELESS,            1, D3D11_BIND_RENDER_TARGET | D3D11_BIND_DEPTH_STENCIL, 0,
                FALSE, TRUE},
        {DXGI_FORMAT_R32_TYPELESS,            1, D3D11_BIND_DEPTH_STENCIL,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R32_TYPELESS,            1, D3D11_BIND_DEPTH_STENCIL | D3D11_BIND_UNORDERED_ACCESS, 0,
                FALSE, TRUE},
        {DXGI_FORMAT_R24G8_TYPELESS,          1, D3D11_BIND_VERTEX_BUFFER,    0, FALSE, TRUE},
        {DXGI_FORMAT_R24G8_TYPELESS,          1, D3D11_BIND_INDEX_BUFFER,     0, FALSE, TRUE},
        {DXGI_FORMAT_R24G8_TYPELESS,          1, D3D11_BIND_CONSTANT_BUFFER,  0, FALSE, TRUE},
        {DXGI_FORMAT_R24G8_TYPELESS,          1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R24G8_TYPELESS,          1, D3D11_BIND_DEPTH_STENCIL,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R24G8_TYPELESS,          1, D3D11_BIND_UNORDERED_ACCESS, 0, FALSE, TRUE},
        {DXGI_FORMAT_X24_TYPELESS_G8_UINT,    1, D3D11_BIND_UNORDERED_ACCESS, 0, FALSE, TRUE},
        {DXGI_FORMAT_R8G8_TYPELESS,           1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8_TYPELESS,           1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8_UNORM,              1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8_SNORM,              1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R16_TYPELESS,            1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R16_TYPELESS,            1, D3D11_BIND_DEPTH_STENCIL,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R16_TYPELESS,            1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R16_UINT,                1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R16_SINT,                1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R8_TYPELESS,             1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8B8A8_UNORM,          1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8B8A8_UNORM,          1, D3D11_BIND_DEPTH_STENCIL,    0, FALSE, FALSE},
        {DXGI_FORMAT_R8G8B8A8_UINT,           1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8B8A8_SNORM,          1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8B8A8_SINT,           1, D3D11_BIND_RENDER_TARGET,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8B8A8_UNORM,          1, D3D11_BIND_RENDER_TARGET,    D3D11_RESOURCE_MISC_SHARED | D3D11_RESOURCE_MISC_SHARED_KEYEDMUTEX,
                FALSE, TRUE},
        {DXGI_FORMAT_D24_UNORM_S8_UINT,       1, D3D11_BIND_SHADER_RESOURCE,  0, FALSE, TRUE},
        {DXGI_FORMAT_D24_UNORM_S8_UINT,       1, D3D11_BIND_RENDER_TARGET,    0, FALSE, FALSE},
        {DXGI_FORMAT_D32_FLOAT,               1, D3D11_BIND_SHADER_RESOURCE,  0, FALSE, TRUE},
        {DXGI_FORMAT_D32_FLOAT,               1, D3D11_BIND_SHADER_RESOURCE | D3D11_BIND_DEPTH_STENCIL, 0,
                FALSE, TRUE},
        {DXGI_FORMAT_D32_FLOAT,               1, D3D11_BIND_RENDER_TARGET,    0, FALSE, FALSE},
        {DXGI_FORMAT_D32_FLOAT,               1, D3D11_BIND_DEPTH_STENCIL,    0, TRUE,  FALSE},
        {DXGI_FORMAT_R9G9B9E5_SHAREDEXP,      1, D3D11_BIND_SHADER_RESOURCE,  0, TRUE,  FALSE},
        {DXGI_FORMAT_R9G9B9E5_SHAREDEXP,      1, D3D11_BIND_RENDER_TARGET,    0, FALSE, FALSE},
        {DXGI_FORMAT_R9G9B9E5_SHAREDEXP,      1, D3D11_BIND_DEPTH_STENCIL,    0, FALSE, FALSE},
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    feature_level = ID3D11Device_GetFeatureLevel(device);

    desc.Width = 512;
    desc.Height = 512;
    desc.MipLevels = 1;
    desc.ArraySize = 1;
    desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    desc.SampleDesc.Count = 1;
    desc.SampleDesc.Quality = 0;
    desc.Usage = D3D11_USAGE_DEFAULT;
    desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    desc.CPUAccessFlags = 0;
    desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &desc, &data, &texture);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Texture2D_GetDevice(texture, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    check_interface(texture, &IID_IDXGISurface, TRUE, FALSE);
    ID3D11Texture2D_Release(texture);

    desc.MipLevels = 0;
    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Texture2D_GetDevice(texture, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    ID3D11Texture2D_GetDesc(texture, &desc);
    ok(desc.Width == 512, "Got unexpected Width %u.\n", desc.Width);
    ok(desc.Height == 512, "Got unexpected Height %u.\n", desc.Height);
    ok(desc.MipLevels == 10, "Got unexpected MipLevels %u.\n", desc.MipLevels);
    ok(desc.ArraySize == 1, "Got unexpected ArraySize %u.\n", desc.ArraySize);
    ok(desc.Format == DXGI_FORMAT_R8G8B8A8_UNORM, "Got unexpected Format %#x.\n", desc.Format);
    ok(desc.SampleDesc.Count == 1, "Got unexpected SampleDesc.Count %u.\n", desc.SampleDesc.Count);
    ok(desc.SampleDesc.Quality == 0, "Got unexpected SampleDesc.Quality %u.\n", desc.SampleDesc.Quality);
    ok(desc.Usage == D3D11_USAGE_DEFAULT, "Got unexpected Usage %u.\n", desc.Usage);
    ok(desc.BindFlags == D3D11_BIND_RENDER_TARGET, "Got unexpected BindFlags %#x.\n", desc.BindFlags);
    ok(desc.CPUAccessFlags == 0, "Got unexpected CPUAccessFlags %#x.\n", desc.CPUAccessFlags);
    ok(desc.MiscFlags == 0, "Got unexpected MiscFlags %#x.\n", desc.MiscFlags);

    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    ID3D11Texture2D_Release(texture);

    desc.MipLevels = 1;
    desc.ArraySize = 2;
    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    ID3D11Texture2D_Release(texture);

    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 2, &quality_level_count);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    desc.ArraySize = 1;
    desc.SampleDesc.Count = 2;
    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    if (quality_level_count)
    {
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        ID3D11Texture2D_Release(texture);
        desc.SampleDesc.Quality = quality_level_count;
        hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    }
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    /* We assume 15 samples multisampling is never supported in practice. */
    desc.SampleDesc.Count = 15;
    desc.SampleDesc.Quality = 0;
    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    desc.SampleDesc.Count = 1;
    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        HRESULT expected_hr = tests[i].succeeds ? S_OK : E_INVALIDARG;
        BOOL todo = tests[i].todo;

        if (feature_level < D3D_FEATURE_LEVEL_10_1
                && (tests[i].misc_flags & D3D11_RESOURCE_MISC_TEXTURECUBE)
                && tests[i].array_size > 6)
        {
            expected_hr = E_INVALIDARG;
            todo = TRUE;
        }

        desc.ArraySize = tests[i].array_size;
        desc.Format = tests[i].format;
        desc.BindFlags = tests[i].bind_flags;
        desc.MiscFlags = tests[i].misc_flags;
        hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);

        todo_wine_if(todo)
        ok(hr == expected_hr, "Test %u: Got unexpected hr %#lx (format %#x).\n",
                i, hr, desc.Format);

        if (SUCCEEDED(hr))
            ID3D11Texture2D_Release(texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_texture2d_interfaces(void)
{
    ID3D10Texture2D *d3d10_texture;
    D3D11_TEXTURE2D_DESC desc;
    ID3D11Texture2D *texture;
    ID3D11Device *device;
    unsigned int i;
    ULONG refcount;
    HRESULT hr;

    static const struct test
    {
        BOOL implements_d3d10_interfaces;
        UINT bind_flags;
        UINT misc_flags;
        UINT expected_bind_flags;
        UINT expected_misc_flags;
    }
    desc_conversion_tests[] =
    {
        {
            TRUE,
            D3D11_BIND_SHADER_RESOURCE, 0,
            D3D10_BIND_SHADER_RESOURCE, 0
        },
        {
            TRUE,
            D3D11_BIND_UNORDERED_ACCESS, 0,
            D3D11_BIND_UNORDERED_ACCESS, 0
        },
        {
            FALSE,
            0, D3D11_RESOURCE_MISC_RESOURCE_CLAMP,
            0, 0
        },
        {
            TRUE,
            0, D3D11_RESOURCE_MISC_SHARED_KEYEDMUTEX,
            0, D3D10_RESOURCE_MISC_SHARED_KEYEDMUTEX
        },
        {
            TRUE,
            0, D3D11_RESOURCE_MISC_SHARED_KEYEDMUTEX | D3D11_RESOURCE_MISC_SHARED_NTHANDLE,
            0, D3D10_RESOURCE_MISC_SHARED_KEYEDMUTEX
        },
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create ID3D11Device, skipping tests.\n");
        return;
    }

    desc.Width = 512;
    desc.Height = 512;
    desc.MipLevels = 1;
    desc.ArraySize = 1;
    desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    desc.SampleDesc.Count = 1;
    desc.SampleDesc.Quality = 0;
    desc.Usage = D3D11_USAGE_DEFAULT;
    desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    desc.CPUAccessFlags = 0;
    desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    check_interface(texture, &IID_IDXGISurface, TRUE, FALSE);
    check_interface(texture, &IID_IDXGIResource, TRUE, FALSE);
    test_dxgi_resource(texture);
    ID3D11Texture2D_Release(texture);

    desc.MipLevels = 0;
    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    check_interface(texture, &IID_IDXGIResource, TRUE, FALSE);
    test_dxgi_resource(texture);
    hr = check_interface(texture, &IID_ID3D10Texture2D, TRUE, TRUE); /* Not available on all Windows versions. */
    ID3D11Texture2D_Release(texture);
    if (FAILED(hr))
    {
        win_skip("2D textures do not implement ID3D10Texture2D, skipping tests.\n");
        ID3D11Device_Release(device);
        return;
    }

    for (i = 0; i < ARRAY_SIZE(desc_conversion_tests); ++i)
    {
        const struct test *current = &desc_conversion_tests[i];
        D3D10_TEXTURE2D_DESC d3d10_desc;
        ID3D10Device *d3d10_device;

        desc.Width = 512;
        desc.Height = 512;
        desc.MipLevels = 1;
        desc.ArraySize = 1;
        desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
        desc.SampleDesc.Count = 1;
        desc.SampleDesc.Quality = 0;
        desc.Usage = D3D11_USAGE_DEFAULT;
        desc.BindFlags = current->bind_flags;
        desc.CPUAccessFlags = 0;
        desc.MiscFlags = current->misc_flags;

        hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
        /* Shared resources are not supported by REF and WARP devices. */
        ok(hr == S_OK || broken(hr == E_OUTOFMEMORY), "Test %u: Got unexpected hr %#lx.\n", i, hr);
        if (FAILED(hr))
        {
            win_skip("Failed to create ID3D11Texture2D, skipping test %u.\n", i);
            continue;
        }

        check_interface(texture, &IID_IDXGISurface, TRUE, FALSE);

        hr = ID3D11Texture2D_QueryInterface(texture, &IID_ID3D10Texture2D, (void **)&d3d10_texture);
        ID3D11Texture2D_Release(texture);

        if (current->implements_d3d10_interfaces)
        {
            ok(SUCCEEDED(hr), "Test %u: Texture should implement ID3D10Texture2D.\n", i);
        }
        else
        {
            todo_wine ok(hr == E_NOINTERFACE, "Test %u: Texture should not implement ID3D10Texture2D.\n", i);
            if (SUCCEEDED(hr)) ID3D10Texture2D_Release(d3d10_texture);
            continue;
        }

        ID3D10Texture2D_GetDesc(d3d10_texture, &d3d10_desc);

        ok(d3d10_desc.Width == desc.Width,
                "Test %u: Got unexpected Width %u.\n", i, d3d10_desc.Width);
        ok(d3d10_desc.Height == desc.Height,
                "Test %u: Got unexpected Height %u.\n", i, d3d10_desc.Height);
        ok(d3d10_desc.MipLevels == desc.MipLevels,
                "Test %u: Got unexpected MipLevels %u.\n", i, d3d10_desc.MipLevels);
        ok(d3d10_desc.ArraySize == desc.ArraySize,
                "Test %u: Got unexpected ArraySize %u.\n", i, d3d10_desc.ArraySize);
        ok(d3d10_desc.Format == desc.Format,
                "Test %u: Got unexpected Format %u.\n", i, d3d10_desc.Format);
        ok(d3d10_desc.SampleDesc.Count == desc.SampleDesc.Count,
                "Test %u: Got unexpected SampleDesc.Count %u.\n", i, d3d10_desc.SampleDesc.Count);
        ok(d3d10_desc.SampleDesc.Quality == desc.SampleDesc.Quality,
                "Test %u: Got unexpected SampleDesc.Quality %u.\n", i, d3d10_desc.SampleDesc.Quality);
        ok(d3d10_desc.Usage == (D3D10_USAGE)desc.Usage,
                "Test %u: Got unexpected Usage %u.\n", i, d3d10_desc.Usage);
        ok(d3d10_desc.BindFlags == current->expected_bind_flags,
                "Test %u: Got unexpected BindFlags %#x.\n", i, d3d10_desc.BindFlags);
        ok(d3d10_desc.CPUAccessFlags == desc.CPUAccessFlags,
                "Test %u: Got unexpected CPUAccessFlags %#x.\n", i, d3d10_desc.CPUAccessFlags);
        ok(d3d10_desc.MiscFlags == current->expected_misc_flags,
                "Test %u: Got unexpected MiscFlags %#x.\n", i, d3d10_desc.MiscFlags);

        d3d10_device = (ID3D10Device *)0xdeadbeef;
        ID3D10Texture2D_GetDevice(d3d10_texture, &d3d10_device);
        ok(!d3d10_device, "Test %u: Got unexpected device pointer %p, expected NULL.\n", i, d3d10_device);
        if (d3d10_device) ID3D10Device_Release(d3d10_device);

        ID3D10Texture2D_Release(d3d10_texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_texture3d(void)
{
    ULONG refcount, expected_refcount;
    D3D11_SUBRESOURCE_DATA data = {0};
    ID3D11Device *device, *tmp;
    D3D11_TEXTURE3D_DESC desc;
    ID3D11Texture3D *texture;
    unsigned int i;
    HRESULT hr;

    static const struct
    {
        DXGI_FORMAT format;
        D3D11_BIND_FLAG bind_flags;
        BOOL succeeds;
        BOOL todo;
    }
    tests[] =
    {
        {DXGI_FORMAT_R32G32B32A32_TYPELESS, D3D11_BIND_VERTEX_BUFFER,   FALSE, TRUE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS, D3D11_BIND_INDEX_BUFFER,    FALSE, TRUE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS, D3D11_BIND_CONSTANT_BUFFER, FALSE, TRUE},
        {DXGI_FORMAT_R32G32B32A32_TYPELESS, D3D11_BIND_SHADER_RESOURCE, TRUE,  FALSE},
        {DXGI_FORMAT_R16G16B16A16_TYPELESS, D3D11_BIND_SHADER_RESOURCE, TRUE,  FALSE},
        {DXGI_FORMAT_R10G10B10A2_TYPELESS,  D3D11_BIND_SHADER_RESOURCE, TRUE,  FALSE},
        {DXGI_FORMAT_R8G8B8A8_UNORM,        D3D11_BIND_DEPTH_STENCIL,   FALSE, FALSE},
        {DXGI_FORMAT_D24_UNORM_S8_UINT,     D3D11_BIND_RENDER_TARGET,   FALSE, FALSE},
        {DXGI_FORMAT_D32_FLOAT,             D3D11_BIND_RENDER_TARGET,   FALSE, FALSE},
        {DXGI_FORMAT_R9G9B9E5_SHAREDEXP,    D3D11_BIND_SHADER_RESOURCE, TRUE,  FALSE},
        {DXGI_FORMAT_R9G9B9E5_SHAREDEXP,    D3D11_BIND_RENDER_TARGET,   FALSE, FALSE},
        {DXGI_FORMAT_R9G9B9E5_SHAREDEXP,    D3D11_BIND_DEPTH_STENCIL,   FALSE, FALSE},
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create ID3D11Device, skipping tests.\n");
        return;
    }

    desc.Width = 64;
    desc.Height = 64;
    desc.Depth = 64;
    desc.MipLevels = 1;
    desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    desc.Usage = D3D11_USAGE_DEFAULT;
    desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    desc.CPUAccessFlags = 0;
    desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture3D(device, &desc, &data, &texture);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateTexture3D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Texture3D_GetDevice(texture, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    ID3D11Texture3D_Release(texture);

    desc.MipLevels = 0;
    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateTexture3D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Texture3D_GetDevice(texture, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    ID3D11Texture3D_GetDesc(texture, &desc);
    ok(desc.Width == 64, "Got unexpected Width %u.\n", desc.Width);
    ok(desc.Height == 64, "Got unexpected Height %u.\n", desc.Height);
    ok(desc.Depth == 64, "Got unexpected Depth %u.\n", desc.Depth);
    ok(desc.MipLevels == 7, "Got unexpected MipLevels %u.\n", desc.MipLevels);
    ok(desc.Format == DXGI_FORMAT_R8G8B8A8_UNORM, "Got unexpected Format %#x.\n", desc.Format);
    ok(desc.Usage == D3D11_USAGE_DEFAULT, "Got unexpected Usage %u.\n", desc.Usage);
    ok(desc.BindFlags == D3D11_BIND_RENDER_TARGET, "Got unexpected BindFlags %u.\n", desc.BindFlags);
    ok(desc.CPUAccessFlags == 0, "Got unexpected CPUAccessFlags %u.\n", desc.CPUAccessFlags);
    ok(desc.MiscFlags == 0, "Got unexpected MiscFlags %u.\n", desc.MiscFlags);

    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    ID3D11Texture3D_Release(texture);

    desc.MipLevels = 1;
    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        desc.Format = tests[i].format;
        desc.BindFlags = tests[i].bind_flags;
        hr = ID3D11Device_CreateTexture3D(device, &desc, NULL, &texture);

        todo_wine_if(tests[i].todo)
        ok(hr == (tests[i].succeeds ? S_OK : E_INVALIDARG), "Test %u: Got unexpected hr %#lx.\n", i, hr);

        if (SUCCEEDED(hr))
            ID3D11Texture3D_Release(texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_texture3d_interfaces(void)
{
    ID3D10Texture3D *d3d10_texture;
    D3D11_TEXTURE3D_DESC desc;
    ID3D11Texture3D *texture;
    ID3D11Device *device;
    unsigned int i;
    ULONG refcount;
    HRESULT hr;

    static const struct test
    {
        BOOL implements_d3d10_interfaces;
        UINT bind_flags;
        UINT misc_flags;
        UINT expected_bind_flags;
        UINT expected_misc_flags;
    }
    desc_conversion_tests[] =
    {
        {
            TRUE,
            D3D11_BIND_SHADER_RESOURCE, 0,
            D3D10_BIND_SHADER_RESOURCE, 0
        },
        {
            TRUE,
            D3D11_BIND_UNORDERED_ACCESS, 0,
            D3D11_BIND_UNORDERED_ACCESS, 0
        },
        {
            FALSE,
            0, D3D11_RESOURCE_MISC_RESOURCE_CLAMP,
            0, 0
        },
        {
            TRUE,
            0, D3D11_RESOURCE_MISC_SHARED_KEYEDMUTEX,
            0, D3D10_RESOURCE_MISC_SHARED_KEYEDMUTEX
        },
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create ID3D11Device.\n");
        return;
    }

    desc.Width = 64;
    desc.Height = 64;
    desc.Depth = 64;
    desc.MipLevels = 0;
    desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    desc.Usage = D3D11_USAGE_DEFAULT;
    desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    desc.CPUAccessFlags = 0;
    desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture3D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);
    check_interface(texture, &IID_IDXGIResource, TRUE, FALSE);
    test_dxgi_resource(texture);
    hr = check_interface(texture, &IID_ID3D10Texture3D, TRUE, TRUE); /* Not available on all Windows versions. */
    ID3D11Texture3D_Release(texture);
    if (FAILED(hr))
    {
        win_skip("3D textures do not implement ID3D10Texture3D.\n");
        ID3D11Device_Release(device);
        return;
    }

    for (i = 0; i < ARRAY_SIZE(desc_conversion_tests); ++i)
    {
        const struct test *current = &desc_conversion_tests[i];
        D3D10_TEXTURE3D_DESC d3d10_desc;
        ID3D10Device *d3d10_device;

        desc.Width = 64;
        desc.Height = 64;
        desc.Depth = 64;
        desc.MipLevels = 1;
        desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
        desc.Usage = D3D11_USAGE_DEFAULT;
        desc.BindFlags = current->bind_flags;
        desc.CPUAccessFlags = 0;
        desc.MiscFlags = current->misc_flags;

        hr = ID3D11Device_CreateTexture3D(device, &desc, NULL, &texture);
        /* Shared resources are not supported by REF and WARP devices. */
        ok(hr == S_OK || broken(hr == E_OUTOFMEMORY), "Test %u: Got unexpected hr %#lx.\n", i, hr);
        if (FAILED(hr))
        {
            win_skip("Failed to create ID3D11Texture3D, skipping test %u.\n", i);
            continue;
        }

        check_interface(texture, &IID_IDXGISurface, FALSE, FALSE);

        hr = ID3D11Texture3D_QueryInterface(texture, &IID_ID3D10Texture3D, (void **)&d3d10_texture);
        ID3D11Texture3D_Release(texture);

        if (current->implements_d3d10_interfaces)
        {
            ok(SUCCEEDED(hr), "Test %u: Texture should implement ID3D10Texture3D.\n", i);
        }
        else
        {
            todo_wine ok(hr == E_NOINTERFACE, "Test %u: Texture should not implement ID3D10Texture3D.\n", i);
            if (SUCCEEDED(hr)) ID3D10Texture3D_Release(d3d10_texture);
            continue;
        }

        ID3D10Texture3D_GetDesc(d3d10_texture, &d3d10_desc);

        ok(d3d10_desc.Width == desc.Width,
                "Test %u: Got unexpected Width %u.\n", i, d3d10_desc.Width);
        ok(d3d10_desc.Height == desc.Height,
                "Test %u: Got unexpected Height %u.\n", i, d3d10_desc.Height);
        ok(d3d10_desc.Depth == desc.Depth,
                "Test %u: Got unexpected Depth %u.\n", i, d3d10_desc.Depth);
        ok(d3d10_desc.MipLevels == desc.MipLevels,
                "Test %u: Got unexpected MipLevels %u.\n", i, d3d10_desc.MipLevels);
        ok(d3d10_desc.Format == desc.Format,
                "Test %u: Got unexpected Format %u.\n", i, d3d10_desc.Format);
        ok(d3d10_desc.Usage == (D3D10_USAGE)desc.Usage,
                "Test %u: Got unexpected Usage %u.\n", i, d3d10_desc.Usage);
        ok(d3d10_desc.BindFlags == current->expected_bind_flags,
                "Test %u: Got unexpected BindFlags %#x.\n", i, d3d10_desc.BindFlags);
        ok(d3d10_desc.CPUAccessFlags == desc.CPUAccessFlags,
                "Test %u: Got unexpected CPUAccessFlags %#x.\n", i, d3d10_desc.CPUAccessFlags);
        ok(d3d10_desc.MiscFlags == current->expected_misc_flags,
                "Test %u: Got unexpected MiscFlags %#x.\n", i, d3d10_desc.MiscFlags);

        d3d10_device = (ID3D10Device *)0xdeadbeef;
        ID3D10Texture3D_GetDevice(d3d10_texture, &d3d10_device);
        ok(!d3d10_device, "Test %u: Got unexpected device pointer %p, expected NULL.\n", i, d3d10_device);
        if (d3d10_device) ID3D10Device_Release(d3d10_device);

        ID3D10Texture3D_Release(d3d10_texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_buffer(void)
{
    ID3D10Buffer *d3d10_buffer;
    HRESULT expected_hr, hr;
    D3D11_BUFFER_DESC desc;
    ID3D11Buffer *buffer;
    ID3D11Device *device;
    unsigned int i;
    ULONG refcount;

    static const struct test
    {
        BOOL succeeds;
        BOOL implements_d3d10_interfaces;
        UINT bind_flags;
        UINT misc_flags;
        UINT structure_stride;
        UINT expected_bind_flags;
        UINT expected_misc_flags;
    }
    tests[] =
    {
        {
            TRUE, TRUE,
            D3D11_BIND_VERTEX_BUFFER, 0, 0,
            D3D10_BIND_VERTEX_BUFFER, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_INDEX_BUFFER, 0, 0,
            D3D10_BIND_INDEX_BUFFER, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_CONSTANT_BUFFER, 0, 0,
            D3D10_BIND_CONSTANT_BUFFER, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_SHADER_RESOURCE, 0, 0,
            D3D10_BIND_SHADER_RESOURCE, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_STREAM_OUTPUT, 0, 0,
            D3D10_BIND_STREAM_OUTPUT, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_RENDER_TARGET, 0, 0,
            D3D10_BIND_RENDER_TARGET, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_UNORDERED_ACCESS, 0, 0,
            D3D11_BIND_UNORDERED_ACCESS, 0
        },
        {
            TRUE, TRUE,
            0, D3D11_RESOURCE_MISC_SHARED, 0,
            0, D3D10_RESOURCE_MISC_SHARED
        },
        {
            TRUE, TRUE,
            0, D3D11_RESOURCE_MISC_DRAWINDIRECT_ARGS, 0,
            0, 0
        },
        {
            FALSE, FALSE,
            D3D11_BIND_VERTEX_BUFFER, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS, 0,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_INDEX_BUFFER, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS, 0,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_CONSTANT_BUFFER, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS, 0,
        },
        {
            TRUE, TRUE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS, 0,
            D3D10_BIND_SHADER_RESOURCE, 0
        },
        {
            FALSE, FALSE,
            D3D11_BIND_STREAM_OUTPUT, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS, 0,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_RENDER_TARGET, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS, 0,
        },
        {
            TRUE, TRUE,
            D3D11_BIND_UNORDERED_ACCESS, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS, 0,
            D3D11_BIND_UNORDERED_ACCESS, 0
        },
        {
            FALSE, FALSE,
            0, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS, 0,
        },
        /* Structured buffers do not implement ID3D10Buffer. */
        {
            TRUE, FALSE,
            0, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 16,
        },
        {
            TRUE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 16,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, ~0u,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 0,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 1,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 2,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 3,
        },
        {
            TRUE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 4,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 5,
        },
        {
            TRUE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 8,
        },
        {
            TRUE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 512,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 513,
        },
        {
            TRUE, FALSE,
            D3D11_BIND_SHADER_RESOURCE, D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 1024,
        },
        {
            TRUE, TRUE,
            0, 0, 513,
            0, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_CONSTANT_BUFFER, 0, 513,
            D3D10_BIND_CONSTANT_BUFFER, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_SHADER_RESOURCE, 0, 513,
            D3D10_BIND_SHADER_RESOURCE, 0
        },
        {
            TRUE, TRUE,
            D3D11_BIND_UNORDERED_ACCESS, 0, 513,
            D3D11_BIND_UNORDERED_ACCESS, 0
        },
        {
            FALSE, FALSE,
            0, D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS | D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 16,
        },
        {
            FALSE, FALSE,
            D3D11_BIND_SHADER_RESOURCE,
            D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS | D3D11_RESOURCE_MISC_BUFFER_STRUCTURED, 16,
        },
        {
            TRUE, TRUE,
            0, D3D11_RESOURCE_MISC_SHARED_KEYEDMUTEX, 0,
            0, D3D10_RESOURCE_MISC_SHARED_KEYEDMUTEX
        },
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create ID3D11Device.\n");
        return;
    }

    buffer = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, 1024, NULL);
    hr = check_interface(buffer, &IID_ID3D10Buffer, TRUE, TRUE); /* Not available on all Windows versions. */
    ID3D11Buffer_Release(buffer);
    if (FAILED(hr))
    {
        win_skip("Buffers do not implement ID3D10Buffer.\n");
        ID3D11Device_Release(device);
        return;
    }

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        const struct test *current = &tests[i];
        D3D11_BUFFER_DESC obtained_desc;
        D3D10_BUFFER_DESC d3d10_desc;
        ID3D10Device *d3d10_device;

        desc.ByteWidth = 1024;
        desc.Usage = D3D11_USAGE_DEFAULT;
        desc.BindFlags = current->bind_flags;
        desc.CPUAccessFlags = 0;
        desc.MiscFlags = current->misc_flags;
        desc.StructureByteStride = current->structure_stride;

        hr = ID3D11Device_CreateBuffer(device, &desc, NULL, &buffer);
        expected_hr = current->succeeds ? S_OK : E_INVALIDARG;
        /* Shared resources are not supported by REF and WARP devices. */
        ok(hr == expected_hr || broken(hr == E_OUTOFMEMORY), "Test %u: Got hr %#lx, expected %#lx.\n",
                i, hr, expected_hr);
        if (FAILED(hr))
        {
            if (hr == E_OUTOFMEMORY)
                win_skip("Failed to create a buffer, skipping test %u.\n", i);
            continue;
        }

        check_interface(buffer, &IID_IDXGIResource, TRUE, FALSE);

        if (!(desc.MiscFlags & D3D11_RESOURCE_MISC_BUFFER_STRUCTURED))
            desc.StructureByteStride = 0;

        ID3D11Buffer_GetDesc(buffer, &obtained_desc);

        ok(obtained_desc.ByteWidth == desc.ByteWidth,
                "Test %u: Got unexpected ByteWidth %u.\n", i, obtained_desc.ByteWidth);
        ok(obtained_desc.Usage == desc.Usage,
                "Test %u: Got unexpected Usage %u.\n", i, obtained_desc.Usage);
        ok(obtained_desc.BindFlags == desc.BindFlags,
                "Test %u: Got unexpected BindFlags %#x.\n", i, obtained_desc.BindFlags);
        ok(obtained_desc.CPUAccessFlags == desc.CPUAccessFlags,
                "Test %u: Got unexpected CPUAccessFlags %#x.\n", i, obtained_desc.CPUAccessFlags);
        ok(obtained_desc.MiscFlags == desc.MiscFlags,
                "Test %u: Got unexpected MiscFlags %#x.\n", i, obtained_desc.MiscFlags);
        ok(obtained_desc.StructureByteStride == desc.StructureByteStride,
                "Test %u: Got unexpected StructureByteStride %u.\n", i, obtained_desc.StructureByteStride);

        hr = ID3D11Buffer_QueryInterface(buffer, &IID_ID3D10Buffer, (void **)&d3d10_buffer);
        ID3D11Buffer_Release(buffer);

        if (current->implements_d3d10_interfaces)
        {
            ok(SUCCEEDED(hr), "Test %u: Buffer should implement ID3D10Buffer.\n", i);
        }
        else
        {
            todo_wine ok(hr == E_NOINTERFACE, "Test %u: Buffer should not implement ID3D10Buffer.\n", i);
            if (SUCCEEDED(hr)) ID3D10Buffer_Release(d3d10_buffer);
            continue;
        }

        ID3D10Buffer_GetDesc(d3d10_buffer, &d3d10_desc);

        ok(d3d10_desc.ByteWidth == desc.ByteWidth,
                "Test %u: Got unexpected ByteWidth %u.\n", i, d3d10_desc.ByteWidth);
        ok(d3d10_desc.Usage == (D3D10_USAGE)desc.Usage,
                "Test %u: Got unexpected Usage %u.\n", i, d3d10_desc.Usage);
        ok(d3d10_desc.BindFlags == current->expected_bind_flags,
                "Test %u: Got unexpected BindFlags %#x.\n", i, d3d10_desc.BindFlags);
        ok(d3d10_desc.CPUAccessFlags == desc.CPUAccessFlags,
                "Test %u: Got unexpected CPUAccessFlags %#x.\n", i, d3d10_desc.CPUAccessFlags);
        ok(d3d10_desc.MiscFlags == current->expected_misc_flags,
                "Test %u: Got unexpected MiscFlags %#x.\n", i, d3d10_desc.MiscFlags);

        d3d10_device = (ID3D10Device *)0xdeadbeef;
        ID3D10Buffer_GetDevice(d3d10_buffer, &d3d10_device);
        ok(!d3d10_device, "Test %u: Got unexpected device pointer %p, expected NULL.\n", i, d3d10_device);
        if (d3d10_device) ID3D10Device_Release(d3d10_device);

        ID3D10Buffer_Release(d3d10_buffer);
    }

    memset(&desc, 0, sizeof(desc));
    desc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
    for (i = 0; i <= 32; ++i)
    {
        desc.ByteWidth = i;
        expected_hr = !i || i % 16 ? E_INVALIDARG : S_OK;
        hr = ID3D11Device_CreateBuffer(device, &desc, NULL, &buffer);
        ok(hr == expected_hr, "Got unexpected hr %#lx for constant buffer size %u.\n", hr, i);
        if (SUCCEEDED(hr))
            ID3D11Buffer_Release(buffer);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_depthstencil_view(void)
{
    D3D11_DEPTH_STENCIL_VIEW_DESC dsv_desc;
    D3D11_TEXTURE2D_DESC texture_desc;
    ULONG refcount, expected_refcount;
    ID3D11DepthStencilView *dsview;
    ID3D11Device *device, *tmp;
    ID3D11Texture2D *texture;
    unsigned int i;
    HRESULT hr;

#define FMT_UNKNOWN  DXGI_FORMAT_UNKNOWN
#define D24S8        DXGI_FORMAT_D24_UNORM_S8_UINT
#define R24G8_TL     DXGI_FORMAT_R24G8_TYPELESS
#define DIM_UNKNOWN  D3D11_DSV_DIMENSION_UNKNOWN
#define TEX_1D       D3D11_DSV_DIMENSION_TEXTURE1D
#define TEX_1D_ARRAY D3D11_DSV_DIMENSION_TEXTURE1DARRAY
#define TEX_2D       D3D11_DSV_DIMENSION_TEXTURE2D
#define TEX_2D_ARRAY D3D11_DSV_DIMENSION_TEXTURE2DARRAY
#define TEX_2DMS     D3D11_DSV_DIMENSION_TEXTURE2DMS
#define TEX_2DMS_ARR D3D11_DSV_DIMENSION_TEXTURE2DMSARRAY
    static const struct
    {
        struct
        {
            unsigned int miplevel_count;
            unsigned int array_size;
            DXGI_FORMAT format;
        } texture;
        struct dsv_desc dsv_desc;
        struct dsv_desc expected_dsv_desc;
    }
    tests[] =
    {
        {{ 1, 1, D24S8},    {0},                                    {D24S8, TEX_2D,       0}},
        {{10, 1, D24S8},    {0},                                    {D24S8, TEX_2D,       0}},
        {{10, 1, D24S8},    {FMT_UNKNOWN, TEX_2D,       0},         {D24S8, TEX_2D,       0}},
        {{10, 1, D24S8},    {FMT_UNKNOWN, TEX_2D,       1},         {D24S8, TEX_2D,       1}},
        {{10, 1, D24S8},    {FMT_UNKNOWN, TEX_2D,       9},         {D24S8, TEX_2D,       9}},
        {{ 1, 1, R24G8_TL}, {D24S8,       TEX_2D,       0},         {D24S8, TEX_2D,       0}},
        {{10, 1, R24G8_TL}, {D24S8,       TEX_2D,       0},         {D24S8, TEX_2D,       0}},
        {{ 1, 4, D24S8},    {0},                                    {D24S8, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, D24S8},    {0},                                    {D24S8, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 0, ~0u}, {D24S8, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2D_ARRAY, 1, 0, ~0u}, {D24S8, TEX_2D_ARRAY, 1, 0, 4}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2D_ARRAY, 3, 0, ~0u}, {D24S8, TEX_2D_ARRAY, 3, 0, 4}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2D_ARRAY, 5, 0, ~0u}, {D24S8, TEX_2D_ARRAY, 5, 0, 4}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2D_ARRAY, 9, 0, ~0u}, {D24S8, TEX_2D_ARRAY, 9, 0, 4}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 1, ~0u}, {D24S8, TEX_2D_ARRAY, 0, 1, 3}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 2, ~0u}, {D24S8, TEX_2D_ARRAY, 0, 2, 2}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 3, ~0u}, {D24S8, TEX_2D_ARRAY, 0, 3, 1}},
        {{ 1, 1, D24S8},    {FMT_UNKNOWN, TEX_2DMS},                {D24S8, TEX_2DMS}},
        {{ 1, 4, D24S8},    {FMT_UNKNOWN, TEX_2DMS},                {D24S8, TEX_2DMS}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2DMS},                {D24S8, TEX_2DMS}},
        {{ 1, 1, D24S8},    {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0,  1},  {D24S8, TEX_2DMS_ARR, 0, 0, 1}},
        {{ 1, 1, D24S8},    {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0, ~0u}, {D24S8, TEX_2DMS_ARR, 0, 0, 1}},
        {{10, 1, D24S8},    {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0,  1},  {D24S8, TEX_2DMS_ARR, 0, 0, 1}},
        {{10, 1, D24S8},    {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0, ~0u}, {D24S8, TEX_2DMS_ARR, 0, 0, 1}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0,  1},  {D24S8, TEX_2DMS_ARR, 0, 0, 1}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0,  4},  {D24S8, TEX_2DMS_ARR, 0, 0, 4}},
        {{10, 4, D24S8},    {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0, ~0u}, {D24S8, TEX_2DMS_ARR, 0, 0, 4}},
    };
    static const struct
    {
        struct
        {
            unsigned int miplevel_count;
            unsigned int array_size;
            DXGI_FORMAT format;
        } texture;
        struct dsv_desc dsv_desc;
    }
    invalid_desc_tests[] =
    {
        {{1, 1, D24S8},    {D24S8,       DIM_UNKNOWN}},
        {{6, 4, D24S8},    {D24S8,       DIM_UNKNOWN}},
        {{1, 1, D24S8},    {D24S8,       TEX_1D,       0}},
        {{1, 1, D24S8},    {D24S8,       TEX_1D_ARRAY, 0, 0, 1}},
        {{1, 1, D24S8},    {R24G8_TL,    TEX_2D,       0}},
        {{1, 1, R24G8_TL}, {FMT_UNKNOWN, TEX_2D,       0}},
        {{1, 1, D24S8},    {D24S8,       TEX_2D,       1}},
        {{1, 1, D24S8},    {D24S8,       TEX_2D_ARRAY, 0, 0, 0}},
        {{1, 1, D24S8},    {D24S8,       TEX_2D_ARRAY, 1, 0, 1}},
        {{1, 1, D24S8},    {D24S8,       TEX_2D_ARRAY, 0, 0, 2}},
        {{1, 1, D24S8},    {D24S8,       TEX_2D_ARRAY, 0, 1, 1}},
        {{1, 1, D24S8},    {D24S8,       TEX_2DMS_ARR, 0, 0, 2}},
        {{1, 1, D24S8},    {D24S8,       TEX_2DMS_ARR, 0, 1, 1}},
    };
#undef FMT_UNKNOWN
#undef D24S8
#undef R24G8_TL
#undef DIM_UNKNOWN
#undef TEX_1D
#undef TEX_1D_ARRAY
#undef TEX_2D
#undef TEX_2D_ARRAY
#undef TEX_2DMS
#undef TEX_2DMS_ARR

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    texture_desc.Width = 512;
    texture_desc.Height = 512;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_D24_UNORM_S8_UINT;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_DEPTH_STENCIL;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, NULL, &dsview);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11DepthStencilView_GetDevice(dsview, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    memset(&dsv_desc, 0, sizeof(dsv_desc));
    ID3D11DepthStencilView_GetDesc(dsview, &dsv_desc);
    ok(dsv_desc.Format == texture_desc.Format, "Got unexpected format %#x.\n", dsv_desc.Format);
    ok(dsv_desc.ViewDimension == D3D11_DSV_DIMENSION_TEXTURE2D,
            "Got unexpected view dimension %#x.\n", dsv_desc.ViewDimension);
    ok(!dsv_desc.Flags, "Got unexpected flags %#x.\n", dsv_desc.Flags);
    ok(!U(dsv_desc).Texture2D.MipSlice, "Got unexpected mip slice %u.\n", U(dsv_desc).Texture2D.MipSlice);

    ID3D11DepthStencilView_Release(dsview);
    ID3D11Texture2D_Release(texture);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        D3D11_DEPTH_STENCIL_VIEW_DESC *current_desc;

        texture_desc.MipLevels = tests[i].texture.miplevel_count;
        texture_desc.ArraySize = tests[i].texture.array_size;
        texture_desc.Format = tests[i].texture.format;

        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        if (tests[i].dsv_desc.dimension == D3D11_DSV_DIMENSION_UNKNOWN)
        {
            current_desc = NULL;
        }
        else
        {
            current_desc = &dsv_desc;
            get_dsv_desc(current_desc, &tests[i].dsv_desc);
        }

        expected_refcount = get_refcount(texture);
        hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, current_desc, &dsview);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        refcount = get_refcount(texture);
        ok(refcount == expected_refcount, "Got refcount %lu, expected %lu.\n", refcount, expected_refcount);

        /* Not available on all Windows versions. */
        check_interface(dsview, &IID_ID3D10DepthStencilView, TRUE, TRUE);

        memset(&dsv_desc, 0, sizeof(dsv_desc));
        ID3D11DepthStencilView_GetDesc(dsview, &dsv_desc);
        check_dsv_desc(&dsv_desc, &tests[i].expected_dsv_desc);

        ID3D11DepthStencilView_Release(dsview);
        ID3D11Texture2D_Release(texture);
    }

    for (i = 0; i < ARRAY_SIZE(invalid_desc_tests); ++i)
    {
        texture_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
        texture_desc.ArraySize = invalid_desc_tests[i].texture.array_size;
        texture_desc.Format = invalid_desc_tests[i].texture.format;

        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        dsview = (void *)0xdeadbeef;
        get_dsv_desc(&dsv_desc, &invalid_desc_tests[i].dsv_desc);
        hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, &dsv_desc, &dsview);
        ok(hr == E_INVALIDARG, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        ok(!dsview, "Unexpected pointer %p.\n", dsview);

        ID3D11Texture2D_Release(texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_depthstencil_view_interfaces(void)
{
    D3D10_DEPTH_STENCIL_VIEW_DESC d3d10_dsv_desc;
    D3D11_DEPTH_STENCIL_VIEW_DESC dsv_desc;
    ID3D10DepthStencilView *d3d10_dsview;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DepthStencilView *dsview;
    ID3D11Texture2D *texture;
    ID3D11Device *device;
    ULONG refcount;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    texture_desc.Width = 512;
    texture_desc.Height = 512;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_D24_UNORM_S8_UINT;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_DEPTH_STENCIL;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    dsv_desc.Format = texture_desc.Format;
    dsv_desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2D;
    dsv_desc.Flags = 0;
    U(dsv_desc).Texture2D.MipSlice = 0;

    hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, &dsv_desc, &dsview);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11DepthStencilView_QueryInterface(dsview, &IID_ID3D10DepthStencilView, (void **)&d3d10_dsview);
    ID3D11DepthStencilView_Release(dsview);
    ok(hr == S_OK || broken(hr == E_NOINTERFACE) /* Not available on all Windows versions. */,
            "Got unexpected hr %#lx.\n", hr);

    if (FAILED(hr))
    {
        win_skip("Depth stencil view does not implement ID3D10DepthStencilView.\n");
        goto done;
    }

    ID3D10DepthStencilView_GetDesc(d3d10_dsview, &d3d10_dsv_desc);
    ok(d3d10_dsv_desc.Format == dsv_desc.Format, "Got unexpected format %#x.\n", d3d10_dsv_desc.Format);
    ok(d3d10_dsv_desc.ViewDimension == (D3D10_DSV_DIMENSION)dsv_desc.ViewDimension,
            "Got unexpected view dimension %u.\n", d3d10_dsv_desc.ViewDimension);
    ok(U(d3d10_dsv_desc).Texture2D.MipSlice == U(dsv_desc).Texture2D.MipSlice,
            "Got unexpected mip slice %u.\n", U(d3d10_dsv_desc).Texture2D.MipSlice);

    ID3D10DepthStencilView_Release(d3d10_dsview);

done:
    ID3D11Texture2D_Release(texture);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_rendertarget_view(void)
{
    D3D11_RENDER_TARGET_VIEW_DESC rtv_desc;
    D3D11_TEXTURE3D_DESC texture3d_desc;
    D3D11_TEXTURE2D_DESC texture2d_desc;
    D3D11_SUBRESOURCE_DATA data = {0};
    ULONG refcount, expected_refcount;
    D3D11_BUFFER_DESC buffer_desc;
    ID3D11RenderTargetView *rtview;
    ID3D11Device *device, *tmp;
    ID3D11Texture3D *texture3d;
    ID3D11Texture2D *texture2d;
    ID3D11Resource *texture;
    ID3D11Buffer *buffer;
    unsigned int i;
    HRESULT hr;

#define FMT_UNKNOWN  DXGI_FORMAT_UNKNOWN
#define RGBA8_UNORM  DXGI_FORMAT_R8G8B8A8_UNORM
#define RGBA8_TL     DXGI_FORMAT_R8G8B8A8_TYPELESS
#define DIM_UNKNOWN  D3D11_RTV_DIMENSION_UNKNOWN
#define TEX_1D       D3D11_RTV_DIMENSION_TEXTURE1D
#define TEX_1D_ARRAY D3D11_RTV_DIMENSION_TEXTURE1DARRAY
#define TEX_2D       D3D11_RTV_DIMENSION_TEXTURE2D
#define TEX_2D_ARRAY D3D11_RTV_DIMENSION_TEXTURE2DARRAY
#define TEX_2DMS     D3D11_RTV_DIMENSION_TEXTURE2DMS
#define TEX_2DMS_ARR D3D11_RTV_DIMENSION_TEXTURE2DMSARRAY
#define TEX_3D       D3D11_RTV_DIMENSION_TEXTURE3D
    static const struct
    {
        struct
        {
            unsigned int miplevel_count;
            unsigned int depth_or_array_size;
            DXGI_FORMAT format;
        } texture;
        struct rtv_desc rtv_desc;
        struct rtv_desc expected_rtv_desc;
    }
    tests[] =
    {
        {{ 1, 1, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_2D,       0}},
        {{10, 1, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_2D,       0}},
        {{10, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D,       0},         {RGBA8_UNORM, TEX_2D,       0}},
        {{10, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D,       1},         {RGBA8_UNORM, TEX_2D,       1}},
        {{10, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D,       9},         {RGBA8_UNORM, TEX_2D,       9}},
        {{ 1, 1, RGBA8_TL},    {RGBA8_UNORM, TEX_2D,       0},         {RGBA8_UNORM, TEX_2D,       0}},
        {{10, 1, RGBA8_TL},    {RGBA8_UNORM, TEX_2D,       0},         {RGBA8_UNORM, TEX_2D,       0}},
        {{ 1, 4, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 1, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 1, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 3, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 3, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 5, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 5, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 9, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 9, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 1, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 1, 3}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 2, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 2, 2}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 3, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 3, 1}},
        {{ 1, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS},                {RGBA8_UNORM, TEX_2DMS}},
        {{ 1, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS},                {RGBA8_UNORM, TEX_2DMS}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS},                {RGBA8_UNORM, TEX_2DMS}},
        {{ 1, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0,  1},  {RGBA8_UNORM, TEX_2DMS_ARR, 0, 0, 1}},
        {{ 1, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0, ~0u}, {RGBA8_UNORM, TEX_2DMS_ARR, 0, 0, 1}},
        {{10, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0,  1},  {RGBA8_UNORM, TEX_2DMS_ARR, 0, 0, 1}},
        {{10, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0, ~0u}, {RGBA8_UNORM, TEX_2DMS_ARR, 0, 0, 1}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0,  1},  {RGBA8_UNORM, TEX_2DMS_ARR, 0, 0, 1}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0,  4},  {RGBA8_UNORM, TEX_2DMS_ARR, 0, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0, 0, ~0u}, {RGBA8_UNORM, TEX_2DMS_ARR, 0, 0, 4}},
        {{ 1, 6, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_3D,       0, 0, 6}},
        {{ 2, 6, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_3D,       0, 0, 6}},
        {{ 2, 6, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 0, 6}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       1, 0, 2}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       1, 0, 2}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 1, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 1, 3}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 2, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 2, 2}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 3, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 3, 1}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 1,  1},  {RGBA8_UNORM, TEX_3D,       0, 1, 1}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 1,  1},  {RGBA8_UNORM, TEX_3D,       1, 1, 1}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 1, ~0u}, {RGBA8_UNORM, TEX_3D,       1, 1, 1}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 0, 8}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       1, 0, 4}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       2, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       2, 0, 2}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       3, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       3, 0, 1}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       4, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       4, 0, 1}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       5, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       5, 0, 1}},
    };
    static const struct
    {
        struct
        {
            D3D11_RTV_DIMENSION dimension;
            unsigned int miplevel_count;
            unsigned int depth_or_array_size;
            DXGI_FORMAT format;
        } texture;
        struct rtv_desc rtv_desc;
    }
    invalid_desc_tests[] =
    {
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, DIM_UNKNOWN}},
        {{TEX_2D, 6, 4, RGBA8_UNORM}, {RGBA8_UNORM, DIM_UNKNOWN}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0, ~0u}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_2D,        0}},
        {{TEX_2D, 1, 1, RGBA8_TL},    {FMT_UNKNOWN, TEX_2D,        0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  1, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  2}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 1,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2DMS_ARR,  0, 0,  2}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2DMS_ARR,  0, 1,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_3D,        0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_TL,    TEX_3D,        0, 0,  1}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  9}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        3, 0,  2}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        2, 0,  4}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 0,  8}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 8, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 4, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        2, 2, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        3, 1, ~0u}},
    };
#undef FMT_UNKNOWN
#undef RGBA8_UNORM
#undef RGBA8_TL
#undef DIM_UNKNOWN
#undef TEX_1D
#undef TEX_1D_ARRAY
#undef TEX_2D
#undef TEX_2D_ARRAY
#undef TEX_2DMS
#undef TEX_2DMS_ARR
#undef TEX_3D

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    buffer_desc.ByteWidth = 1024;
    buffer_desc.Usage = D3D11_USAGE_DEFAULT;
    buffer_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    buffer_desc.CPUAccessFlags = 0;
    buffer_desc.MiscFlags = 0;
    buffer_desc.StructureByteStride = 0;

    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, &data, &buffer);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, &buffer);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Buffer_GetDevice(buffer, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    rtv_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_BUFFER;
    U1(U(rtv_desc).Buffer).ElementOffset = 0;
    U2(U(rtv_desc).Buffer).ElementWidth = 64;

    if (!enable_debug_layer)
    {
        rtview = (void *)0xdeadbeef;
        hr = ID3D11Device_CreateRenderTargetView(device, NULL, &rtv_desc, &rtview);
        ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
        ok(!rtview, "Unexpected pointer %p.\n", rtview);
    }

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)buffer, &rtv_desc, &rtview);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11RenderTargetView_GetDevice(rtview, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    /* Not available on all Windows versions. */
    check_interface(rtview, &IID_ID3D10RenderTargetView, TRUE, TRUE);

    ID3D11RenderTargetView_Release(rtview);
    ID3D11Buffer_Release(buffer);

    texture2d_desc.Width = 512;
    texture2d_desc.Height = 512;
    texture2d_desc.SampleDesc.Count = 1;
    texture2d_desc.SampleDesc.Quality = 0;
    texture2d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture2d_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture2d_desc.CPUAccessFlags = 0;
    texture2d_desc.MiscFlags = 0;

    texture3d_desc.Width = 64;
    texture3d_desc.Height = 64;
    texture3d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture3d_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture3d_desc.CPUAccessFlags = 0;
    texture3d_desc.MiscFlags = 0;

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        D3D11_RENDER_TARGET_VIEW_DESC *current_desc;

        if (tests[i].expected_rtv_desc.dimension != D3D11_RTV_DIMENSION_TEXTURE3D)
        {
            texture2d_desc.MipLevels = tests[i].texture.miplevel_count;
            texture2d_desc.ArraySize = tests[i].texture.depth_or_array_size;
            texture2d_desc.Format = tests[i].texture.format;

            hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, NULL, &texture2d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture2d;
        }
        else
        {
            texture3d_desc.MipLevels = tests[i].texture.miplevel_count;
            texture3d_desc.Depth = tests[i].texture.depth_or_array_size;
            texture3d_desc.Format = tests[i].texture.format;

            hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &texture3d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture3d;
        }

        if (tests[i].rtv_desc.dimension == D3D11_RTV_DIMENSION_UNKNOWN)
        {
            current_desc = NULL;
        }
        else
        {
            current_desc = &rtv_desc;
            get_rtv_desc(current_desc, &tests[i].rtv_desc);
        }

        expected_refcount = get_refcount(texture);
        hr = ID3D11Device_CreateRenderTargetView(device, texture, current_desc, &rtview);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        refcount = get_refcount(texture);
        ok(refcount == expected_refcount, "Got refcount %lu, expected %lu.\n", refcount, expected_refcount);

        /* Not available on all Windows versions. */
        check_interface(rtview, &IID_ID3D10RenderTargetView, TRUE, TRUE);

        memset(&rtv_desc, 0, sizeof(rtv_desc));
        ID3D11RenderTargetView_GetDesc(rtview, &rtv_desc);
        check_rtv_desc(&rtv_desc, &tests[i].expected_rtv_desc);

        ID3D11RenderTargetView_Release(rtview);
        ID3D11Resource_Release(texture);
    }

    for (i = 0; i < ARRAY_SIZE(invalid_desc_tests); ++i)
    {
        assert(invalid_desc_tests[i].texture.dimension == D3D11_RTV_DIMENSION_TEXTURE2D
                || invalid_desc_tests[i].texture.dimension == D3D11_RTV_DIMENSION_TEXTURE3D);

        if (invalid_desc_tests[i].texture.dimension != D3D11_RTV_DIMENSION_TEXTURE3D)
        {
            texture2d_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
            texture2d_desc.ArraySize = invalid_desc_tests[i].texture.depth_or_array_size;
            texture2d_desc.Format = invalid_desc_tests[i].texture.format;

            hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, NULL, &texture2d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture2d;
        }
        else
        {
            texture3d_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
            texture3d_desc.Depth = invalid_desc_tests[i].texture.depth_or_array_size;
            texture3d_desc.Format = invalid_desc_tests[i].texture.format;

            hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &texture3d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture3d;
        }

        get_rtv_desc(&rtv_desc, &invalid_desc_tests[i].rtv_desc);
        rtview = (void *)0xdeadbeef;
        hr = ID3D11Device_CreateRenderTargetView(device, texture, &rtv_desc, &rtview);
        ok(hr == E_INVALIDARG, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        ok(!rtview, "Unexpected pointer %p.\n", rtview);

        ID3D11Resource_Release(texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_shader_resource_view(void)
{
    D3D11_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D11_TEXTURE3D_DESC texture3d_desc;
    D3D11_TEXTURE2D_DESC texture2d_desc;
    ULONG refcount, expected_refcount;
    ID3D11ShaderResourceView *srview;
    D3D_FEATURE_LEVEL feature_level;
    D3D11_BUFFER_DESC buffer_desc;
    ID3D11Device *device, *tmp;
    ID3D11Texture3D *texture3d;
    ID3D11Texture2D *texture2d;
    ID3D11Resource *texture;
    ID3D11Buffer *buffer;
    unsigned int i;
    HRESULT hr;

#define FMT_UNKNOWN  DXGI_FORMAT_UNKNOWN
#define RGBA8_UNORM  DXGI_FORMAT_R8G8B8A8_UNORM
#define RGBA8_SRGB   DXGI_FORMAT_R8G8B8A8_UNORM_SRGB
#define RGBA8_UINT   DXGI_FORMAT_R8G8B8A8_UINT
#define RGBA8_TL     DXGI_FORMAT_R8G8B8A8_TYPELESS
#define DIM_UNKNOWN  D3D11_SRV_DIMENSION_UNKNOWN
#define TEX_1D       D3D11_SRV_DIMENSION_TEXTURE1D
#define TEX_1D_ARRAY D3D11_SRV_DIMENSION_TEXTURE1DARRAY
#define TEX_2D       D3D11_SRV_DIMENSION_TEXTURE2D
#define TEX_2D_ARRAY D3D11_SRV_DIMENSION_TEXTURE2DARRAY
#define TEX_2DMS     D3D11_SRV_DIMENSION_TEXTURE2DMS
#define TEX_2DMS_ARR D3D11_SRV_DIMENSION_TEXTURE2DMSARRAY
#define TEX_3D       D3D11_SRV_DIMENSION_TEXTURE3D
#define TEX_CUBE     D3D11_SRV_DIMENSION_TEXTURECUBE
#define CUBE_ARRAY   D3D11_SRV_DIMENSION_TEXTURECUBEARRAY
    static const struct
    {
        struct
        {
            unsigned int miplevel_count;
            unsigned int depth_or_array_size;
            DXGI_FORMAT format;
        } texture;
        struct srv_desc srv_desc;
        struct srv_desc expected_srv_desc;
    }
    tests[] =
    {
        {{10,  1, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, TEX_2D,       0, 10}},
        {{10,  1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D,       0, ~0u},         {RGBA8_UNORM, TEX_2D,       0, 10}},
        {{10,  1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,       0, ~0u},         {RGBA8_UNORM, TEX_2D,       0, 10}},
        {{10,  1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D,       0, 10},          {RGBA8_UNORM, TEX_2D,       0, 10}},
        {{ 1,  1, RGBA8_TL},    {RGBA8_UNORM, TEX_2D,       0, ~0u},         {RGBA8_UNORM, TEX_2D,       0,  1}},
        {{10,  1, RGBA8_TL},    {RGBA8_UNORM, TEX_2D,       0, ~0u},         {RGBA8_UNORM, TEX_2D,       0, 10}},
        {{10,  4, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, TEX_2D_ARRAY, 0, 10, 0, 4}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, ~0u, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 10, 0, 4}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 1, ~0u, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 1,  9, 0, 4}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 3, ~0u, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 3,  7, 0, 4}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 5, ~0u, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 5,  5, 0, 4}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 9, ~0u, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 9,  1, 0, 4}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, ~0u, 1, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 10, 1, 3}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, ~0u, 2, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 10, 2, 2}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, ~0u, 3, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 10, 3, 1}},
        {{ 1,  1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS},                     {RGBA8_UNORM, TEX_2DMS}},
        {{ 1,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS},                     {RGBA8_UNORM, TEX_2DMS}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS},                     {RGBA8_UNORM, TEX_2DMS}},
        {{ 1,  1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0,  1,  0,  1},  {RGBA8_UNORM, TEX_2DMS_ARR, 0,  1, 0, 1}},
        {{ 1,  1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0,  1,  0, ~0u}, {RGBA8_UNORM, TEX_2DMS_ARR, 0,  1, 0, 1}},
        {{10,  1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0,  1,  0,  1},  {RGBA8_UNORM, TEX_2DMS_ARR, 0,  1, 0, 1}},
        {{10,  1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0,  1,  0, ~0u}, {RGBA8_UNORM, TEX_2DMS_ARR, 0,  1, 0, 1}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0,  1,  0,  1},  {RGBA8_UNORM, TEX_2DMS_ARR, 0,  1, 0, 1}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0,  1,  0,  4},  {RGBA8_UNORM, TEX_2DMS_ARR, 0,  1, 0, 4}},
        {{10,  4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2DMS_ARR, 0,  1,  0, ~0u}, {RGBA8_UNORM, TEX_2DMS_ARR, 0,  1, 0, 4}},
        {{ 1, 12, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, TEX_3D,       0,  1}},
        {{ 1, 12, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0,  1},          {RGBA8_UNORM, TEX_3D,       0,  1}},
        {{ 1, 12, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, ~0u},         {RGBA8_UNORM, TEX_3D,       0,  1}},
        {{ 4, 12, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, ~0u},         {RGBA8_UNORM, TEX_3D,       0,  4}},
        {{ 1,  6, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, TEX_CUBE,     0,  1}},
        {{ 2,  6, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, TEX_CUBE,     0,  2}},
        {{ 2,  9, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, TEX_CUBE,     0,  2}},
        {{ 2, 11, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, TEX_CUBE,     0,  2}},
        {{ 2,  6, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_CUBE,     0, ~0u},         {RGBA8_UNORM, TEX_CUBE,     0,  2}},
        {{ 2,  6, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_CUBE,     0,  1},          {RGBA8_UNORM, TEX_CUBE ,    0,  1}},
        {{ 2,  6, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_CUBE,     1,  1},          {RGBA8_UNORM, TEX_CUBE ,    1,  1}},
        {{ 2,  6, RGBA8_UNORM}, {FMT_UNKNOWN, CUBE_ARRAY,   0,  1,  0,  1},  {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 1}},
        {{ 2,  6, RGBA8_UNORM}, {FMT_UNKNOWN, CUBE_ARRAY,   0, ~0u, 0, ~0u}, {RGBA8_UNORM, CUBE_ARRAY,   0,  2, 0, 1}},
        {{ 1,  8, RGBA8_UNORM}, {FMT_UNKNOWN, CUBE_ARRAY,   0, ~0u, 0, ~0u}, {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 1}},
        {{ 1, 12, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 2}},
        {{ 1, 12, RGBA8_UNORM}, {FMT_UNKNOWN, CUBE_ARRAY,   0, ~0u, 0, ~0u}, {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 2}},
        {{ 1, 12, RGBA8_UNORM}, {FMT_UNKNOWN, CUBE_ARRAY,   0, ~0u, 0,  1},  {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 1}},
        {{ 1, 12, RGBA8_UNORM}, {FMT_UNKNOWN, CUBE_ARRAY,   0, ~0u, 0,  2},  {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 2}},
        {{ 1, 13, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 2}},
        {{ 1, 14, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 2}},
        {{ 1, 18, RGBA8_UNORM}, {0},                                         {RGBA8_UNORM, CUBE_ARRAY,   0,  1, 0, 3}},
        {{ 1,  1, RGBA8_UINT},  {0},                                         {RGBA8_UINT,  TEX_2D,       0,  1, 0, 1}},
        {{ 1,  1, RGBA8_TL},    {RGBA8_UINT,  TEX_2D,       0, ~0u},         {RGBA8_UINT,  TEX_2D,       0,  1, 0, 1}},
    };
    static const struct
    {
        struct
        {
            D3D11_SRV_DIMENSION dimension;
            unsigned int miplevel_count;
            unsigned int depth_or_array_size;
            DXGI_FORMAT format;
        } texture;
        struct srv_desc srv_desc;
    }
    invalid_desc_tests[] =
    {
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, DIM_UNKNOWN}},
        {{TEX_2D, 6, 4, RGBA8_UNORM}, {RGBA8_UNORM, DIM_UNKNOWN}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0,  1, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_2D,        0, ~0u}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_2D,        0,  1}},
        {{TEX_2D, 1, 1, RGBA8_TL},    {FMT_UNKNOWN, TEX_2D,        0, ~0u}},
        {{TEX_2D, 1, 1, RGBA8_TL},    {FMT_UNKNOWN, TEX_2D,        0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0,  0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0,  2}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        1,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0,  0, 0,  0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0,  0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0,  1, 0,  0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0,  2, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  1,  1, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0,  1, 0,  2}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0,  1, 1,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2DMS_ARR,  0,  1, 0,  2}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2DMS_ARR,  0,  1, 1,  1}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, TEX_CUBE,      0,  0}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, TEX_CUBE,      0,  2}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, TEX_CUBE,      1,  1}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  0, 0,  0}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  0, 0,  1}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  1, 0,  0}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  1, 0,  0}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  2, 0,  1}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    1,  1, 0,  1}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  1, 1,  1}},
        {{TEX_2D, 1, 6, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  1, 1, ~0u}},
        {{TEX_2D, 1, 7, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  1, 2,  1}},
        {{TEX_2D, 1, 7, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  1, 2, ~0u}},
        {{TEX_2D, 1, 7, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  1, 0,  2}},
        {{TEX_2D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, CUBE_ARRAY,    0,  1, 0,  2}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UINT,  TEX_2D,        0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UINT},  {RGBA8_UNORM, TEX_2D,        0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_SRGB,  TEX_2D,        0,  1}},
        {{TEX_2D, 1, 1, RGBA8_SRGB},  {RGBA8_UNORM, TEX_2D,        0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0,  1, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_CUBE,      0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0,  1, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0,  1, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_CUBE,      0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0,  1, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0,  0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_3D,        0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,        0,  2}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,        1,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0,  2}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1,  1}},
    };
#undef FMT_UNKNOWN
#undef RGBA8_UNORM
#undef RGBA8_SRGB
#undef RGBA8_UINT
#undef RGBA8_TL
#undef DIM_UNKNOWN
#undef TEX_1D
#undef TEX_1D_ARRAY
#undef TEX_2D
#undef TEX_2D_ARRAY
#undef TEX_2DMS
#undef TEX_2DMS_ARR
#undef TEX_3D
#undef TEX_CUBE
#undef CUBE_ARRAY

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }
    feature_level = ID3D11Device_GetFeatureLevel(device);

    buffer = create_buffer(device, D3D11_BIND_SHADER_RESOURCE, 1024, NULL);

    srview = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)buffer, NULL, &srview);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!srview, "Unexpected pointer %p.\n", srview);

    srv_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    srv_desc.ViewDimension = D3D11_SRV_DIMENSION_BUFFER;
    U1(U(srv_desc).Buffer).ElementOffset = 0;
    U2(U(srv_desc).Buffer).ElementWidth = 64;

    srview = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateShaderResourceView(device, NULL, &srv_desc, &srview);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!srview, "Unexpected pointer %p.\n", srview);

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)buffer, &srv_desc, &srview);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11ShaderResourceView_GetDevice(srview, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    /* Not available on all Windows versions. */
    check_interface(srview, &IID_ID3D10ShaderResourceView, TRUE, TRUE);
    check_interface(srview, &IID_ID3D10ShaderResourceView1, TRUE, TRUE);

    ID3D11ShaderResourceView_Release(srview);
    ID3D11Buffer_Release(buffer);

    /* Without D3D11_BIND_SHADER_RESOURCE. */
    buffer = create_buffer(device, 0, 1024, NULL);

    srview = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)buffer, &srv_desc, &srview);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!srview, "Unexpected pointer %p.\n", srview);

    ID3D11Buffer_Release(buffer);

    if (feature_level >= D3D_FEATURE_LEVEL_11_0)
    {
        buffer_desc.ByteWidth = 1024;
        buffer_desc.Usage = D3D11_USAGE_DEFAULT;
        buffer_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
        buffer_desc.CPUAccessFlags = 0;
        buffer_desc.MiscFlags = D3D11_RESOURCE_MISC_BUFFER_STRUCTURED;
        buffer_desc.StructureByteStride = 4;

        hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, &buffer);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

        hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)buffer, NULL, &srview);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

        memset(&srv_desc, 0, sizeof(srv_desc));
        ID3D11ShaderResourceView_GetDesc(srview, &srv_desc);

        ok(srv_desc.Format == DXGI_FORMAT_UNKNOWN, "Got unexpected format %#x.\n", srv_desc.Format);
        ok(srv_desc.ViewDimension == D3D11_SRV_DIMENSION_BUFFER, "Got unexpected view dimension %#x.\n",
                srv_desc.ViewDimension);
        ok(!U1(U(srv_desc).Buffer).FirstElement, "Got unexpected first element %u.\n",
                U1(U(srv_desc).Buffer).FirstElement);
        ok(U2(U(srv_desc).Buffer).NumElements == 256, "Got unexpected num elements %u.\n",
                U2(U(srv_desc).Buffer).NumElements);

        ID3D11ShaderResourceView_Release(srview);
        ID3D11Buffer_Release(buffer);
    }
    else
    {
        skip("Structured buffers require feature level 11_0.\n");
    }

    texture2d_desc.Width = 512;
    texture2d_desc.Height = 512;
    texture2d_desc.SampleDesc.Count = 1;
    texture2d_desc.SampleDesc.Quality = 0;
    texture2d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture2d_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture2d_desc.CPUAccessFlags = 0;

    texture3d_desc.Width = 64;
    texture3d_desc.Height = 64;
    texture3d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture3d_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture3d_desc.CPUAccessFlags = 0;
    texture3d_desc.MiscFlags = 0;

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        D3D11_SHADER_RESOURCE_VIEW_DESC *current_desc;

        if (tests[i].expected_srv_desc.dimension != D3D11_SRV_DIMENSION_TEXTURE3D)
        {
            texture2d_desc.MipLevels = tests[i].texture.miplevel_count;
            texture2d_desc.ArraySize = tests[i].texture.depth_or_array_size;
            texture2d_desc.Format = tests[i].texture.format;
            texture2d_desc.MiscFlags = 0;

            if (tests[i].expected_srv_desc.dimension == D3D11_SRV_DIMENSION_TEXTURECUBE
                    || tests[i].expected_srv_desc.dimension == D3D11_SRV_DIMENSION_TEXTURECUBEARRAY)
                texture2d_desc.MiscFlags |= D3D11_RESOURCE_MISC_TEXTURECUBE;

            if (texture2d_desc.MiscFlags & D3D11_RESOURCE_MISC_TEXTURECUBE
                    && (texture2d_desc.ArraySize != 6
                    || tests[i].expected_srv_desc.dimension == D3D11_SRV_DIMENSION_TEXTURECUBEARRAY)
                    && feature_level < D3D_FEATURE_LEVEL_10_1)
            {
                skip("Test %u: Cube map array textures require feature level 10_1.\n", i);
                continue;
            }

            hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, NULL, &texture2d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture2d;
        }
        else
        {
            texture3d_desc.MipLevels = tests[i].texture.miplevel_count;
            texture3d_desc.Depth = tests[i].texture.depth_or_array_size;
            texture3d_desc.Format = tests[i].texture.format;

            hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &texture3d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture3d;
        }

        if (tests[i].srv_desc.dimension == D3D11_SRV_DIMENSION_UNKNOWN)
        {
            current_desc = NULL;
        }
        else
        {
            current_desc = &srv_desc;
            get_srv_desc(current_desc, &tests[i].srv_desc);
        }

        expected_refcount = get_refcount(texture);
        hr = ID3D11Device_CreateShaderResourceView(device, texture, current_desc, &srview);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        refcount = get_refcount(texture);
        ok(refcount == expected_refcount, "Got refcount %lu, expected %lu.\n", refcount, expected_refcount);

        /* Not available on all Windows versions. */
        check_interface(srview, &IID_ID3D10ShaderResourceView, TRUE, TRUE);
        check_interface(srview, &IID_ID3D10ShaderResourceView1, TRUE, TRUE);

        memset(&srv_desc, 0, sizeof(srv_desc));
        ID3D11ShaderResourceView_GetDesc(srview, &srv_desc);
        check_srv_desc(&srv_desc, &tests[i].expected_srv_desc);

        ID3D11ShaderResourceView_Release(srview);
        ID3D11Resource_Release(texture);
    }

    for (i = 0; i < ARRAY_SIZE(invalid_desc_tests); ++i)
    {
        assert(invalid_desc_tests[i].texture.dimension == D3D11_SRV_DIMENSION_TEXTURE2D
                || invalid_desc_tests[i].texture.dimension == D3D11_SRV_DIMENSION_TEXTURE3D);

        if (invalid_desc_tests[i].texture.dimension == D3D11_SRV_DIMENSION_TEXTURE2D)
        {
            texture2d_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
            texture2d_desc.ArraySize = invalid_desc_tests[i].texture.depth_or_array_size;
            texture2d_desc.Format = invalid_desc_tests[i].texture.format;
            texture2d_desc.MiscFlags = 0;

            if (invalid_desc_tests[i].srv_desc.dimension == D3D11_SRV_DIMENSION_TEXTURECUBE
                    || invalid_desc_tests[i].srv_desc.dimension == D3D11_SRV_DIMENSION_TEXTURECUBEARRAY)
                texture2d_desc.MiscFlags |= D3D11_RESOURCE_MISC_TEXTURECUBE;

            if (invalid_desc_tests[i].srv_desc.dimension == D3D11_SRV_DIMENSION_TEXTURECUBEARRAY
                    && feature_level < D3D_FEATURE_LEVEL_10_1)
            {
                skip("Test %u: Cube map array textures require feature level 10_1.\n", i);
                continue;
            }

            hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, NULL, &texture2d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture2d;
        }
        else
        {
            texture3d_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
            texture3d_desc.Depth = invalid_desc_tests[i].texture.depth_or_array_size;
            texture3d_desc.Format = invalid_desc_tests[i].texture.format;

            hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &texture3d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture3d;
        }

        srview = (void *)0xdeadbeef;
        get_srv_desc(&srv_desc, &invalid_desc_tests[i].srv_desc);
        hr = ID3D11Device_CreateShaderResourceView(device, texture, &srv_desc, &srview);
        ok(hr == E_INVALIDARG, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        ok(!srview, "Unexpected pointer %p.\n", srview);

        ID3D11Resource_Release(texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_shader(const D3D_FEATURE_LEVEL feature_level)
{
#if 0
    float4 light;
    float4x4 mat;

    struct input
    {
        float4 position : POSITION;
        float3 normal : NORMAL;
    };

    struct output
    {
        float4 position : POSITION;
        float4 diffuse : COLOR;
    };

    output main(const input v)
    {
        output o;

        o.position = mul(v.position, mat);
        o.diffuse = dot((float3)light, v.normal);

        return o;
    }
#endif
    static const DWORD vs_4_1[] =
    {
        0x43425844, 0xfce5b27c, 0x965db93d, 0x8c3d0459, 0x9890ebac, 0x00000001, 0x000001c4, 0x00000003,
        0x0000002c, 0x0000007c, 0x000000cc, 0x4e475349, 0x00000048, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000041, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000707, 0x49534f50, 0x4e4f4954, 0x524f4e00, 0x004c414d, 0x4e47534f,
        0x00000048, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x00000041, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f, 0x49534f50,
        0x4e4f4954, 0x4c4f4300, 0xab00524f, 0x52444853, 0x000000f0, 0x00010041, 0x0000003c, 0x0100086a,
        0x04000059, 0x00208e46, 0x00000000, 0x00000005, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f,
        0x00101072, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000001,
        0x08000011, 0x00102012, 0x00000000, 0x00101e46, 0x00000000, 0x00208e46, 0x00000000, 0x00000001,
        0x08000011, 0x00102022, 0x00000000, 0x00101e46, 0x00000000, 0x00208e46, 0x00000000, 0x00000002,
        0x08000011, 0x00102042, 0x00000000, 0x00101e46, 0x00000000, 0x00208e46, 0x00000000, 0x00000003,
        0x08000011, 0x00102082, 0x00000000, 0x00101e46, 0x00000000, 0x00208e46, 0x00000000, 0x00000004,
        0x08000010, 0x001020f2, 0x00000001, 0x00208246, 0x00000000, 0x00000000, 0x00101246, 0x00000001,
        0x0100003e,
    };
    static const DWORD vs_4_0[] =
    {
        0x43425844, 0x3ae813ca, 0x0f034b91, 0x790f3226, 0x6b4a718a, 0x00000001, 0x000001c0,
        0x00000003, 0x0000002c, 0x0000007c, 0x000000cc, 0x4e475349, 0x00000048, 0x00000002,
        0x00000008, 0x00000038, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f,
        0x00000041, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x00000707, 0x49534f50,
        0x4e4f4954, 0x524f4e00, 0x004c414d, 0x4e47534f, 0x00000048, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000041,
        0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f, 0x49534f50, 0x4e4f4954,
        0x4c4f4300, 0xab00524f, 0x52444853, 0x000000ec, 0x00010040, 0x0000003b, 0x04000059,
        0x00208e46, 0x00000000, 0x00000005, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f,
        0x00101072, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x001020f2,
        0x00000001, 0x08000011, 0x00102012, 0x00000000, 0x00101e46, 0x00000000, 0x00208e46,
        0x00000000, 0x00000001, 0x08000011, 0x00102022, 0x00000000, 0x00101e46, 0x00000000,
        0x00208e46, 0x00000000, 0x00000002, 0x08000011, 0x00102042, 0x00000000, 0x00101e46,
        0x00000000, 0x00208e46, 0x00000000, 0x00000003, 0x08000011, 0x00102082, 0x00000000,
        0x00101e46, 0x00000000, 0x00208e46, 0x00000000, 0x00000004, 0x08000010, 0x001020f2,
        0x00000001, 0x00208246, 0x00000000, 0x00000000, 0x00101246, 0x00000001, 0x0100003e,
    };
    static const DWORD vs_3_0[] =
    {
        0xfffe0300, 0x002bfffe, 0x42415443, 0x0000001c, 0x00000077, 0xfffe0300, 0x00000002,
        0x0000001c, 0x00000100, 0x00000070, 0x00000044, 0x00040002, 0x00000001, 0x0000004c,
        0x00000000, 0x0000005c, 0x00000002, 0x00000004, 0x00000060, 0x00000000, 0x6867696c,
        0xabab0074, 0x00030001, 0x00040001, 0x00000001, 0x00000000, 0x0074616d, 0x00030003,
        0x00040004, 0x00000001, 0x00000000, 0x335f7376, 0x4d00305f, 0x6f726369, 0x74666f73,
        0x29522820, 0x534c4820, 0x6853204c, 0x72656461, 0x6d6f4320, 0x656c6970, 0x2e392072,
        0x392e3932, 0x332e3235, 0x00313131, 0x0200001f, 0x80000000, 0x900f0000, 0x0200001f,
        0x80000003, 0x900f0001, 0x0200001f, 0x80000000, 0xe00f0000, 0x0200001f, 0x8000000a,
        0xe00f0001, 0x03000009, 0xe0010000, 0x90e40000, 0xa0e40000, 0x03000009, 0xe0020000,
        0x90e40000, 0xa0e40001, 0x03000009, 0xe0040000, 0x90e40000, 0xa0e40002, 0x03000009,
        0xe0080000, 0x90e40000, 0xa0e40003, 0x03000008, 0xe00f0001, 0xa0e40004, 0x90e40001,
        0x0000ffff,
    };
    static const DWORD vs_2_0[] =
    {
        0xfffe0200, 0x002bfffe, 0x42415443, 0x0000001c, 0x00000077, 0xfffe0200, 0x00000002,
        0x0000001c, 0x00000100, 0x00000070, 0x00000044, 0x00040002, 0x00000001, 0x0000004c,
        0x00000000, 0x0000005c, 0x00000002, 0x00000004, 0x00000060, 0x00000000, 0x6867696c,
        0xabab0074, 0x00030001, 0x00040001, 0x00000001, 0x00000000, 0x0074616d, 0x00030003,
        0x00040004, 0x00000001, 0x00000000, 0x325f7376, 0x4d00305f, 0x6f726369, 0x74666f73,
        0x29522820, 0x534c4820, 0x6853204c, 0x72656461, 0x6d6f4320, 0x656c6970, 0x2e392072,
        0x392e3932, 0x332e3235, 0x00313131, 0x0200001f, 0x80000000, 0x900f0000, 0x0200001f,
        0x80000003, 0x900f0001, 0x03000009, 0xc0010000, 0x90e40000, 0xa0e40000, 0x03000009,
        0xc0020000, 0x90e40000, 0xa0e40001, 0x03000009, 0xc0040000, 0x90e40000, 0xa0e40002,
        0x03000009, 0xc0080000, 0x90e40000, 0xa0e40003, 0x03000008, 0xd00f0000, 0xa0e40004,
        0x90e40001, 0x0000ffff,
    };

#if 0
    float4 main(const float4 color : COLOR) : SV_TARGET
    {
        float4 o;

        o = color;

        return o;
    }
#endif
    static const DWORD ps_4_1[] =
    {
        0x43425844, 0xa1a44423, 0xa4cfcec2, 0x64610832, 0xb7a852bd, 0x00000001, 0x000000d4, 0x00000003,
        0x0000002c, 0x0000005c, 0x00000090, 0x4e475349, 0x00000028, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x4f4c4f43, 0xabab0052, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x0000003c, 0x00000041, 0x0000000f,
        0x0100086a, 0x03001062, 0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_4_0[] =
    {
        0x43425844, 0x08c2b568, 0x17d33120, 0xb7d82948, 0x13a570fb, 0x00000001, 0x000000d0, 0x00000003,
        0x0000002c, 0x0000005c, 0x00000090, 0x4e475349, 0x00000028, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x4f4c4f43, 0xabab0052, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000038, 0x00000040, 0x0000000e,
        0x03001062, 0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_4_0_level_9_0[] =
    {
        0x43425844, 0xbc6626e7, 0x7778dc9d, 0xc8a43be2, 0xe4b53f7a, 0x00000001, 0x00000170,
        0x00000005, 0x00000034, 0x00000080, 0x000000cc, 0x0000010c, 0x0000013c, 0x53414e58,
        0x00000044, 0x00000044, 0xffff0200, 0x00000020, 0x00000024, 0x00240000, 0x00240000,
        0x00240000, 0x00240000, 0x00240000, 0xffff0200, 0x0200001f, 0x80000000, 0xb00f0000,
        0x02000001, 0x800f0800, 0x80e40000, 0x0000ffff, 0x396e6f41, 0x00000044, 0x00000044,
        0xffff0200, 0x00000020, 0x00000024, 0x00240000, 0x00240000, 0x00240000, 0x00240000,
        0x00240000, 0xffff0200, 0x0200001f, 0x80000000, 0xb00f0000, 0x02000001, 0x800f0800,
        0xb0e40000, 0x0000ffff, 0x52444853, 0x00000038, 0x00000040, 0x0000000e, 0x03001062,
        0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x0100003e, 0x4e475349, 0x00000028, 0x00000001,
        0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f,
        0x4f4c4f43, 0xabab0052, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x545f5653, 0x45475241,
        0xabab0054,
    };
    static const DWORD ps_4_0_level_9_1[] =
    {
        0x43425844, 0x275ecf38, 0x4349ff01, 0xa6b0e324, 0x6e54a4fc, 0x00000001, 0x00000120,
        0x00000004, 0x00000030, 0x0000007c, 0x000000bc, 0x000000ec, 0x396e6f41, 0x00000044,
        0x00000044, 0xffff0200, 0x00000020, 0x00000024, 0x00240000, 0x00240000, 0x00240000,
        0x00240000, 0x00240000, 0xffff0200, 0x0200001f, 0x80000000, 0xb00f0000, 0x02000001,
        0x800f0800, 0xb0e40000, 0x0000ffff, 0x52444853, 0x00000038, 0x00000040, 0x0000000e,
        0x03001062, 0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e, 0x4e475349, 0x00000028,
        0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x00000f0f, 0x4f4c4f43, 0xabab0052, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x545f5653,
        0x45475241, 0xabab0054,
    };
    static const DWORD ps_4_0_level_9_3[] =
    {
        0x43425844, 0xc7d541c4, 0x961d4e0e, 0x9ce7ec57, 0x70f47dcb, 0x00000001, 0x00000120,
        0x00000004, 0x00000030, 0x0000007c, 0x000000bc, 0x000000ec, 0x396e6f41, 0x00000044,
        0x00000044, 0xffff0200, 0x00000020, 0x00000024, 0x00240000, 0x00240000, 0x00240000,
        0x00240000, 0x00240000, 0xffff0201, 0x0200001f, 0x80000000, 0xb00f0000, 0x02000001,
        0x800f0800, 0xb0e40000, 0x0000ffff, 0x52444853, 0x00000038, 0x00000040, 0x0000000e,
        0x03001062, 0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e, 0x4e475349, 0x00000028,
        0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x00000f0f, 0x4f4c4f43, 0xabab0052, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x545f5653,
        0x45475241, 0xabab0054,
    };

#if 0
    struct gs_out
    {
        float4 pos : SV_POSITION;
    };

    [maxvertexcount(4)]
    void main(point float4 vin[1] : POSITION, inout TriangleStream<gs_out> vout)
    {
        float offset = 0.1 * vin[0].w;
        gs_out v;

        v.pos = float4(vin[0].x - offset, vin[0].y - offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x - offset, vin[0].y + offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x + offset, vin[0].y - offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x + offset, vin[0].y + offset, vin[0].z, vin[0].w);
        vout.Append(v);
    }
#endif
    static const DWORD gs_4_1[] =
    {
        0x43425844, 0x779daaf5, 0x7e154197, 0xcf5e99da, 0xb502b4d2, 0x00000001, 0x00000240, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x000001a4, 0x00020041,
        0x00000069, 0x0100086a, 0x0400005f, 0x002010f2, 0x00000001, 0x00000000, 0x02000068, 0x00000001,
        0x0100085d, 0x0100285c, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x0200005e, 0x00000004,
        0x0f000032, 0x00100032, 0x00000000, 0x80201ff6, 0x00000041, 0x00000000, 0x00000000, 0x00004002,
        0x3dcccccd, 0x3dcccccd, 0x00000000, 0x00000000, 0x00201046, 0x00000000, 0x00000000, 0x05000036,
        0x00102032, 0x00000000, 0x00100046, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6,
        0x00000000, 0x00000000, 0x01000013, 0x05000036, 0x00102012, 0x00000000, 0x0010000a, 0x00000000,
        0x0e000032, 0x00100052, 0x00000000, 0x00201ff6, 0x00000000, 0x00000000, 0x00004002, 0x3dcccccd,
        0x00000000, 0x3dcccccd, 0x00000000, 0x00201106, 0x00000000, 0x00000000, 0x05000036, 0x00102022,
        0x00000000, 0x0010002a, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000,
        0x00000000, 0x01000013, 0x05000036, 0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x05000036,
        0x00102022, 0x00000000, 0x0010001a, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6,
        0x00000000, 0x00000000, 0x01000013, 0x05000036, 0x00102032, 0x00000000, 0x00100086, 0x00000000,
        0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000, 0x00000000, 0x01000013, 0x0100003e,
    };
    static const DWORD gs_4_0[] =
    {
        0x43425844, 0x000ee786, 0xc624c269, 0x885a5cbe, 0x444b3b1f, 0x00000001, 0x0000023c, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x000001a0, 0x00020040,
        0x00000068, 0x0400005f, 0x002010f2, 0x00000001, 0x00000000, 0x02000068, 0x00000001, 0x0100085d,
        0x0100285c, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x0200005e, 0x00000004, 0x0f000032,
        0x00100032, 0x00000000, 0x80201ff6, 0x00000041, 0x00000000, 0x00000000, 0x00004002, 0x3dcccccd,
        0x3dcccccd, 0x00000000, 0x00000000, 0x00201046, 0x00000000, 0x00000000, 0x05000036, 0x00102032,
        0x00000000, 0x00100046, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000,
        0x00000000, 0x01000013, 0x05000036, 0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x0e000032,
        0x00100052, 0x00000000, 0x00201ff6, 0x00000000, 0x00000000, 0x00004002, 0x3dcccccd, 0x00000000,
        0x3dcccccd, 0x00000000, 0x00201106, 0x00000000, 0x00000000, 0x05000036, 0x00102022, 0x00000000,
        0x0010002a, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000, 0x00000000,
        0x01000013, 0x05000036, 0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x05000036, 0x00102022,
        0x00000000, 0x0010001a, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000,
        0x00000000, 0x01000013, 0x05000036, 0x00102032, 0x00000000, 0x00100086, 0x00000000, 0x06000036,
        0x001020c2, 0x00000000, 0x00201ea6, 0x00000000, 0x00000000, 0x01000013, 0x0100003e,
    };

    ULONG refcount, expected_refcount;
    struct device_desc device_desc;
    ID3D11Device *device, *tmp;
    ID3D11GeometryShader *gs;
    ID3D11ComputeShader *cs;
    ID3D11VertexShader *vs;
    ID3D11DomainShader *ds;
    ID3D11PixelShader *ps;
    ID3D11HullShader *hs;
    HRESULT hr;

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }

    /* level_9 shaders */
    hr = ID3D11Device_CreatePixelShader(device, ps_4_0_level_9_0, sizeof(ps_4_0_level_9_0), NULL, &ps);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);
    ID3D11PixelShader_Release(ps);

    hr = ID3D11Device_CreatePixelShader(device, ps_4_0_level_9_1, sizeof(ps_4_0_level_9_1), NULL, &ps);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);
    ID3D11PixelShader_Release(ps);

    hr = ID3D11Device_CreatePixelShader(device, ps_4_0_level_9_3, sizeof(ps_4_0_level_9_3), NULL, &ps);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);
    ID3D11PixelShader_Release(ps);

    ps = (void *)0xdeadbeef;
    hr = ID3D11Device_CreatePixelShader(device, vs_2_0, sizeof(vs_2_0), NULL, &ps);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!ps, "Unexpected pointer %p.\n", ps);

    /* vertex shader */
    vs = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateVertexShader(device, vs_2_0, sizeof(vs_2_0), NULL, &vs);
    ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
    ok(!vs, "Unexpected pointer %p.\n", vs);

    vs = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateVertexShader(device, vs_3_0, sizeof(vs_3_0), NULL, &vs);
    ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
    ok(!vs, "Unexpected pointer %p.\n", vs);

    vs = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateVertexShader(device, ps_4_0, sizeof(ps_4_0), NULL, &vs);
    ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
    ok(!vs, "Unexpected pointer %p.\n", vs);

    expected_refcount = get_refcount(device) + (feature_level >= D3D_FEATURE_LEVEL_10_0);
    hr = ID3D11Device_CreateVertexShader(device, vs_4_0, sizeof(vs_4_0), NULL, &vs);
    if (feature_level >= D3D_FEATURE_LEVEL_10_0)
        ok(hr == S_OK, "Got hr %#lx.\n", hr);
    else
        ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);

    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n",
            refcount, expected_refcount);
    if (feature_level >= D3D_FEATURE_LEVEL_10_0)
    {
        tmp = NULL;
        expected_refcount = refcount + 1;
        ID3D11VertexShader_GetDevice(vs, &tmp);
        ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
        refcount = get_refcount(device);
        ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n",
                refcount, expected_refcount);
        ID3D11Device_Release(tmp);

        /* Not available on all Windows versions. */
        check_interface(vs, &IID_ID3D10VertexShader, TRUE, TRUE);

        refcount = ID3D11VertexShader_Release(vs);
        ok(!refcount, "Vertex shader has %lu references left.\n", refcount);
    }

    hr = ID3D11Device_CreateVertexShader(device, vs_4_1, sizeof(vs_4_1), NULL, &vs);
    if (feature_level >= D3D_FEATURE_LEVEL_10_1)
    {
        ok(hr == S_OK, "Got hr %#lx.\n", hr);
        refcount = ID3D11VertexShader_Release(vs);
        ok(!refcount, "Vertex shader has %lu references left.\n", refcount);
    }
    else
    {
        todo_wine_if(feature_level >= D3D_FEATURE_LEVEL_10_0)
            ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
        if (SUCCEEDED(hr))
            ID3D11VertexShader_Release(vs);
    }

    /* pixel shader */
    expected_refcount = get_refcount(device) + (feature_level >= D3D_FEATURE_LEVEL_10_0);
    hr = ID3D11Device_CreatePixelShader(device, ps_4_0, sizeof(ps_4_0), NULL, &ps);
    if (feature_level >= D3D_FEATURE_LEVEL_10_0)
        ok(hr == S_OK, "Got hr %#lx.\n", hr);
    else
        ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);

    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n",
            refcount, expected_refcount);
    if (feature_level >= D3D_FEATURE_LEVEL_10_0)
    {
        tmp = NULL;
        expected_refcount = refcount + 1;
        ID3D11PixelShader_GetDevice(ps, &tmp);
        ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
        refcount = get_refcount(device);
        ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n",
                refcount, expected_refcount);
        ID3D11Device_Release(tmp);

        /* Not available on all Windows versions. */
        check_interface(ps, &IID_ID3D10PixelShader, TRUE, TRUE);

        refcount = ID3D11PixelShader_Release(ps);
        ok(!refcount, "Pixel shader has %lu references left.\n", refcount);
    }

    hr = ID3D11Device_CreatePixelShader(device, ps_4_1, sizeof(ps_4_1), NULL, &ps);
    if (feature_level >= D3D_FEATURE_LEVEL_10_1)
    {
        ok(hr == S_OK, "Got hr %#lx.\n", hr);
        refcount = ID3D11PixelShader_Release(ps);
        ok(!refcount, "Pixel shader has %lu references left.\n", refcount);
    }
    else
    {
        todo_wine_if(feature_level >= D3D_FEATURE_LEVEL_10_0)
            ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
        if (SUCCEEDED(hr))
            ID3D11PixelShader_Release(ps);
    }

    /* geometry shader */
    expected_refcount = get_refcount(device) + (feature_level >= D3D_FEATURE_LEVEL_10_0);
    hr = ID3D11Device_CreateGeometryShader(device, gs_4_0, sizeof(gs_4_0), NULL, &gs);
    if (feature_level >= D3D_FEATURE_LEVEL_10_0)
        ok(hr == S_OK, "Got hr %#lx.\n", hr);
    else
        ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);

    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n",
            refcount, expected_refcount);
    if (feature_level >= D3D_FEATURE_LEVEL_10_0)
    {
        tmp = NULL;
        expected_refcount = refcount + 1;
        ID3D11GeometryShader_GetDevice(gs, &tmp);
        ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
        refcount = get_refcount(device);
        ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n",
                refcount, expected_refcount);
        ID3D11Device_Release(tmp);

        /* Not available on all Windows versions. */
        check_interface(gs, &IID_ID3D10GeometryShader, TRUE, TRUE);

        refcount = ID3D11GeometryShader_Release(gs);
        ok(!refcount, "Geometry shader has %lu references left.\n", refcount);
    }

    hr = ID3D11Device_CreateGeometryShader(device, gs_4_1, sizeof(gs_4_1), NULL, &gs);
    if (feature_level >= D3D_FEATURE_LEVEL_10_1)
    {
        ok(hr == S_OK, "Got hr %#lx.\n", hr);
        refcount = ID3D11GeometryShader_Release(gs);
        ok(!refcount, "Geometry shader has %lu references left.\n", refcount);
    }
    else
    {
        todo_wine_if(feature_level >= D3D_FEATURE_LEVEL_10_0)
            ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
        if (SUCCEEDED(hr))
            ID3D11GeometryShader_Release(gs);
    }

    /* Hull shader */
    hs = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateHullShader(device, gs_4_0, sizeof(gs_4_0), NULL, &hs);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!hs, "Unexpected pointer %p.\n", hs);

    /* Domain shader */
    ds = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateDomainShader(device, gs_4_0, sizeof(gs_4_0), NULL, &ds);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!ds, "Unexpected pointer %p.\n", ds);

    /* Compute shader */
    ds = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateComputeShader(device, gs_4_0, sizeof(gs_4_0), NULL, &cs);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!cs, "Unexpected pointer %p.\n", cs);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_sampler_state(void)
{
    static const struct test
    {
        D3D11_FILTER filter;
        D3D10_FILTER expected_filter;
    }
    desc_conversion_tests[] =
    {
        {D3D11_FILTER_MIN_MAG_MIP_POINT, D3D10_FILTER_MIN_MAG_MIP_POINT},
        {D3D11_FILTER_MIN_MAG_POINT_MIP_LINEAR, D3D10_FILTER_MIN_MAG_POINT_MIP_LINEAR},
        {D3D11_FILTER_MIN_POINT_MAG_LINEAR_MIP_POINT, D3D10_FILTER_MIN_POINT_MAG_LINEAR_MIP_POINT},
        {D3D11_FILTER_MIN_POINT_MAG_MIP_LINEAR, D3D10_FILTER_MIN_POINT_MAG_MIP_LINEAR},
        {D3D11_FILTER_MIN_LINEAR_MAG_MIP_POINT, D3D10_FILTER_MIN_LINEAR_MAG_MIP_POINT},
        {D3D11_FILTER_MIN_LINEAR_MAG_POINT_MIP_LINEAR, D3D10_FILTER_MIN_LINEAR_MAG_POINT_MIP_LINEAR},
        {D3D11_FILTER_MIN_MAG_LINEAR_MIP_POINT, D3D10_FILTER_MIN_MAG_LINEAR_MIP_POINT},
        {D3D11_FILTER_MIN_MAG_MIP_LINEAR, D3D10_FILTER_MIN_MAG_MIP_LINEAR},
        {D3D11_FILTER_ANISOTROPIC, D3D10_FILTER_ANISOTROPIC},
        {D3D11_FILTER_COMPARISON_MIN_MAG_MIP_POINT, D3D10_FILTER_COMPARISON_MIN_MAG_MIP_POINT},
        {D3D11_FILTER_COMPARISON_MIN_MAG_POINT_MIP_LINEAR, D3D10_FILTER_COMPARISON_MIN_MAG_POINT_MIP_LINEAR},
        {
            D3D11_FILTER_COMPARISON_MIN_POINT_MAG_LINEAR_MIP_POINT,
            D3D10_FILTER_COMPARISON_MIN_POINT_MAG_LINEAR_MIP_POINT
        },
        {D3D11_FILTER_COMPARISON_MIN_POINT_MAG_MIP_LINEAR, D3D10_FILTER_COMPARISON_MIN_POINT_MAG_MIP_LINEAR},
        {D3D11_FILTER_COMPARISON_MIN_LINEAR_MAG_MIP_POINT, D3D10_FILTER_COMPARISON_MIN_LINEAR_MAG_MIP_POINT},
        {
            D3D11_FILTER_COMPARISON_MIN_LINEAR_MAG_POINT_MIP_LINEAR,
            D3D10_FILTER_COMPARISON_MIN_LINEAR_MAG_POINT_MIP_LINEAR
        },
        {D3D11_FILTER_COMPARISON_MIN_MAG_LINEAR_MIP_POINT, D3D10_FILTER_COMPARISON_MIN_MAG_LINEAR_MIP_POINT},
        {D3D11_FILTER_COMPARISON_MIN_MAG_MIP_LINEAR, D3D10_FILTER_COMPARISON_MIN_MAG_MIP_LINEAR},
        {D3D11_FILTER_COMPARISON_ANISOTROPIC, D3D10_FILTER_COMPARISON_ANISOTROPIC},
    };

    ID3D11SamplerState *sampler_state1, *sampler_state2;
    ID3D10SamplerState *d3d10_sampler_state;
    ULONG refcount, expected_refcount;
    ID3D11Device *device, *tmp;
    D3D11_SAMPLER_DESC desc;
    unsigned int i;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_CreateSamplerState(device, NULL, &sampler_state1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    desc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
    desc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
    desc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
    desc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
    desc.MipLODBias = 0.0f;
    desc.MaxAnisotropy = 16;
    desc.ComparisonFunc = D3D11_COMPARISON_ALWAYS;
    desc.BorderColor[0] = 0.0f;
    desc.BorderColor[1] = 1.0f;
    desc.BorderColor[2] = 0.0f;
    desc.BorderColor[3] = 1.0f;
    desc.MinLOD = 0.0f;
    desc.MaxLOD = 16.0f;

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateSamplerState(device, &desc, &sampler_state1);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateSamplerState(device, &desc, &sampler_state2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(sampler_state1 == sampler_state2, "Got different sampler state objects.\n");
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11SamplerState_GetDevice(sampler_state1, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    ID3D11SamplerState_GetDesc(sampler_state1, &desc);
    ok(desc.Filter == D3D11_FILTER_MIN_MAG_MIP_LINEAR, "Got unexpected filter %#x.\n", desc.Filter);
    ok(desc.AddressU == D3D11_TEXTURE_ADDRESS_WRAP, "Got unexpected address u %u.\n", desc.AddressU);
    ok(desc.AddressV == D3D11_TEXTURE_ADDRESS_WRAP, "Got unexpected address v %u.\n", desc.AddressV);
    ok(desc.AddressW == D3D11_TEXTURE_ADDRESS_WRAP, "Got unexpected address w %u.\n", desc.AddressW);
    ok(!desc.MipLODBias, "Got unexpected mip LOD bias %f.\n", desc.MipLODBias);
    ok(!desc.MaxAnisotropy, "Got unexpected max anisotropy %u.\n", desc.MaxAnisotropy);
    ok(desc.ComparisonFunc == D3D11_COMPARISON_NEVER, "Got unexpected comparison func %u.\n", desc.ComparisonFunc);
    ok(!desc.BorderColor[0] && !desc.BorderColor[1] && !desc.BorderColor[2] && !desc.BorderColor[3],
            "Got unexpected border color {%.8e, %.8e, %.8e, %.8e}.\n",
            desc.BorderColor[0], desc.BorderColor[1], desc.BorderColor[2], desc.BorderColor[3]);
    ok(!desc.MinLOD, "Got unexpected min LOD %f.\n", desc.MinLOD);
    ok(desc.MaxLOD == 16.0f, "Got unexpected max LOD %f.\n", desc.MaxLOD);

    refcount = ID3D11SamplerState_Release(sampler_state2);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3D11SamplerState_Release(sampler_state1);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);

    for (i = 0; i < ARRAY_SIZE(desc_conversion_tests); ++i)
    {
        const struct test *current = &desc_conversion_tests[i];
        D3D10_SAMPLER_DESC d3d10_desc, expected_desc;

        desc.Filter = current->filter;
        desc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
        desc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
        desc.AddressW = D3D11_TEXTURE_ADDRESS_BORDER;
        desc.MipLODBias = 0.0f;
        desc.MaxAnisotropy = 16;
        desc.ComparisonFunc = D3D11_COMPARISON_ALWAYS;
        desc.BorderColor[0] = 0.0f;
        desc.BorderColor[1] = 1.0f;
        desc.BorderColor[2] = 0.0f;
        desc.BorderColor[3] = 1.0f;
        desc.MinLOD = 0.0f;
        desc.MaxLOD = 16.0f;

        hr = ID3D11Device_CreateSamplerState(device, &desc, &sampler_state1);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        hr = ID3D11SamplerState_QueryInterface(sampler_state1, &IID_ID3D10SamplerState,
                (void **)&d3d10_sampler_state);
        ok(hr == S_OK || broken(hr == E_NOINTERFACE) /* Not available on all Windows versions. */,
                "Test %u: Got unexpected hr %#lx.\n", i, hr);
        if (FAILED(hr))
        {
            win_skip("Sampler state does not implement ID3D10SamplerState.\n");
            ID3D11SamplerState_Release(sampler_state1);
            break;
        }

        memcpy(&expected_desc, &desc, sizeof(expected_desc));
        expected_desc.Filter = current->expected_filter;
        if (!D3D11_DECODE_IS_ANISOTROPIC_FILTER(current->filter))
            expected_desc.MaxAnisotropy = 0;
        if (!D3D11_DECODE_IS_COMPARISON_FILTER(current->filter))
            expected_desc.ComparisonFunc = D3D10_COMPARISON_NEVER;

        ID3D10SamplerState_GetDesc(d3d10_sampler_state, &d3d10_desc);
        ok(d3d10_desc.Filter == expected_desc.Filter,
                "Test %u: Got unexpected filter %#x.\n", i, d3d10_desc.Filter);
        ok(d3d10_desc.AddressU == expected_desc.AddressU,
                "Test %u: Got unexpected address u %u.\n", i, d3d10_desc.AddressU);
        ok(d3d10_desc.AddressV == expected_desc.AddressV,
                "Test %u: Got unexpected address v %u.\n", i, d3d10_desc.AddressV);
        ok(d3d10_desc.AddressW == expected_desc.AddressW,
                "Test %u: Got unexpected address w %u.\n", i, d3d10_desc.AddressW);
        ok(d3d10_desc.MipLODBias == expected_desc.MipLODBias,
                "Test %u: Got unexpected mip LOD bias %f.\n", i, d3d10_desc.MipLODBias);
        ok(d3d10_desc.MaxAnisotropy == expected_desc.MaxAnisotropy,
                "Test %u: Got unexpected max anisotropy %u.\n", i, d3d10_desc.MaxAnisotropy);
        ok(d3d10_desc.ComparisonFunc == expected_desc.ComparisonFunc,
                "Test %u: Got unexpected comparison func %u.\n", i, d3d10_desc.ComparisonFunc);
        ok(d3d10_desc.BorderColor[0] == expected_desc.BorderColor[0]
                && d3d10_desc.BorderColor[1] == expected_desc.BorderColor[1]
                && d3d10_desc.BorderColor[2] == expected_desc.BorderColor[2]
                && d3d10_desc.BorderColor[3] == expected_desc.BorderColor[3],
                "Test %u: Got unexpected border color {%.8e, %.8e, %.8e, %.8e}.\n", i,
                d3d10_desc.BorderColor[0], d3d10_desc.BorderColor[1],
                d3d10_desc.BorderColor[2], d3d10_desc.BorderColor[3]);
        ok(d3d10_desc.MinLOD == expected_desc.MinLOD,
                "Test %u: Got unexpected min LOD %f.\n", i, d3d10_desc.MinLOD);
        ok(d3d10_desc.MaxLOD == expected_desc.MaxLOD,
                "Test %u: Got unexpected max LOD %f.\n", i, d3d10_desc.MaxLOD);

        refcount = ID3D10SamplerState_Release(d3d10_sampler_state);
        ok(refcount == 1, "Test %u: Got unexpected refcount %lu.\n", i, refcount);
        refcount = ID3D11SamplerState_Release(sampler_state1);
        ok(!refcount, "Test %u: Got unexpected refcount %lu.\n", i, refcount);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_blend_state(void)
{
    static const D3D11_BLEND_DESC desc_conversion_tests[] =
    {
        {
            FALSE, FALSE,
            {
                {
                    FALSE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD
                },
            },
        },
        {
            FALSE, TRUE,
            {
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    FALSE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_RED
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    FALSE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_GREEN
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
            },
        },
        {
            FALSE, TRUE,
            {
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_SUBTRACT,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    TRUE, D3D11_BLEND_ZERO, D3D11_BLEND_ONE, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ZERO, D3D11_BLEND_ONE, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ONE, D3D11_BLEND_OP_MAX,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    TRUE, D3D11_BLEND_ONE, D3D11_BLEND_ONE, D3D11_BLEND_OP_MIN,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    FALSE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
                {
                    FALSE, D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD,
                    D3D11_BLEND_ONE, D3D11_BLEND_ZERO, D3D11_BLEND_OP_ADD, D3D11_COLOR_WRITE_ENABLE_ALL
                },
            },
        },
    };

    ID3D11BlendState *blend_state1, *blend_state2;
    D3D11_BLEND_DESC desc, obtained_desc;
    ID3D10BlendState *d3d10_blend_state;
    D3D10_BLEND_DESC d3d10_blend_desc;
    ULONG refcount, expected_refcount;
    ID3D11Device *device, *tmp;
    unsigned int i, j;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_CreateBlendState(device, NULL, &blend_state1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    memset(&desc, 0, sizeof(desc));
    desc.AlphaToCoverageEnable = FALSE;
    desc.IndependentBlendEnable = FALSE;
    desc.RenderTarget[0].BlendEnable = FALSE;
    desc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateBlendState(device, &desc, &blend_state1);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateBlendState(device, &desc, &blend_state2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(blend_state1 == blend_state2, "Got different blend state objects.\n");
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11BlendState_GetDevice(blend_state1, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    ID3D11BlendState_GetDesc(blend_state1, &obtained_desc);
    ok(obtained_desc.AlphaToCoverageEnable == FALSE, "Got unexpected alpha to coverage enable %#x.\n",
            obtained_desc.AlphaToCoverageEnable);
    ok(obtained_desc.IndependentBlendEnable == FALSE, "Got unexpected independent blend enable %#x.\n",
            obtained_desc.IndependentBlendEnable);
    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT; ++i)
    {
        ok(obtained_desc.RenderTarget[i].BlendEnable == FALSE,
                "Got unexpected blend enable %#x for render target %u.\n",
                obtained_desc.RenderTarget[i].BlendEnable, i);
        ok(obtained_desc.RenderTarget[i].SrcBlend == D3D11_BLEND_ONE,
                "Got unexpected src blend %u for render target %u.\n",
                obtained_desc.RenderTarget[i].SrcBlend, i);
        ok(obtained_desc.RenderTarget[i].DestBlend == D3D11_BLEND_ZERO,
                "Got unexpected dest blend %u for render target %u.\n",
                obtained_desc.RenderTarget[i].DestBlend, i);
        ok(obtained_desc.RenderTarget[i].BlendOp == D3D11_BLEND_OP_ADD,
                "Got unexpected blend op %u for render target %u.\n",
                obtained_desc.RenderTarget[i].BlendOp, i);
        ok(obtained_desc.RenderTarget[i].SrcBlendAlpha == D3D11_BLEND_ONE,
                "Got unexpected src blend alpha %u for render target %u.\n",
                obtained_desc.RenderTarget[i].SrcBlendAlpha, i);
        ok(obtained_desc.RenderTarget[i].DestBlendAlpha == D3D11_BLEND_ZERO,
                "Got unexpected dest blend alpha %u for render target %u.\n",
                obtained_desc.RenderTarget[i].DestBlendAlpha, i);
        ok(obtained_desc.RenderTarget[i].BlendOpAlpha == D3D11_BLEND_OP_ADD,
                "Got unexpected blend op alpha %u for render target %u.\n",
                obtained_desc.RenderTarget[i].BlendOpAlpha, i);
        ok(obtained_desc.RenderTarget[i].RenderTargetWriteMask == D3D11_COLOR_WRITE_ENABLE_ALL,
                "Got unexpected render target write mask %#x for render target %u.\n",
                obtained_desc.RenderTarget[0].RenderTargetWriteMask, i);
    }

    /* Not available on all Windows versions. */
    check_interface(blend_state1, &IID_ID3D10BlendState, TRUE, TRUE);
    check_interface(blend_state1, &IID_ID3D10BlendState1, TRUE, TRUE);

    refcount = ID3D11BlendState_Release(blend_state1);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3D11BlendState_Release(blend_state2);
    ok(!refcount, "Blend state has %lu references left.\n", refcount);

    for (i = 0; i < ARRAY_SIZE(desc_conversion_tests); ++i)
    {
        const D3D11_BLEND_DESC *current_desc = &desc_conversion_tests[i];

        hr = ID3D11Device_CreateBlendState(device, current_desc, &blend_state1);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        hr = ID3D11BlendState_QueryInterface(blend_state1, &IID_ID3D10BlendState, (void **)&d3d10_blend_state);
        ok(hr == S_OK || broken(hr == E_NOINTERFACE) /* Not available on all Windows versions. */,
                "Test %u: Got unexpected hr %#lx.\n", i, hr);
        if (FAILED(hr))
        {
            win_skip("Blend state does not implement ID3D10BlendState.\n");
            ID3D11BlendState_Release(blend_state1);
            break;
        }

        ID3D10BlendState_GetDesc(d3d10_blend_state, &d3d10_blend_desc);
        ok(d3d10_blend_desc.AlphaToCoverageEnable == current_desc->AlphaToCoverageEnable,
                "Got unexpected alpha to coverage enable %#x for test %u.\n",
                d3d10_blend_desc.AlphaToCoverageEnable, i);
        ok(d3d10_blend_desc.SrcBlend == (D3D10_BLEND)current_desc->RenderTarget[0].SrcBlend,
                "Got unexpected src blend %u for test %u.\n", d3d10_blend_desc.SrcBlend, i);
        ok(d3d10_blend_desc.DestBlend == (D3D10_BLEND)current_desc->RenderTarget[0].DestBlend,
                "Got unexpected dest blend %u for test %u.\n", d3d10_blend_desc.DestBlend, i);
        ok(d3d10_blend_desc.BlendOp == (D3D10_BLEND_OP)current_desc->RenderTarget[0].BlendOp,
                "Got unexpected blend op %u for test %u.\n", d3d10_blend_desc.BlendOp, i);
        ok(d3d10_blend_desc.SrcBlendAlpha == (D3D10_BLEND)current_desc->RenderTarget[0].SrcBlendAlpha,
                "Got unexpected src blend alpha %u for test %u.\n", d3d10_blend_desc.SrcBlendAlpha, i);
        ok(d3d10_blend_desc.DestBlendAlpha == (D3D10_BLEND)current_desc->RenderTarget[0].DestBlendAlpha,
                "Got unexpected dest blend alpha %u for test %u.\n", d3d10_blend_desc.DestBlendAlpha, i);
        ok(d3d10_blend_desc.BlendOpAlpha == (D3D10_BLEND_OP)current_desc->RenderTarget[0].BlendOpAlpha,
                "Got unexpected blend op alpha %u for test %u.\n", d3d10_blend_desc.BlendOpAlpha, i);
        for (j = 0; j < D3D10_SIMULTANEOUS_RENDER_TARGET_COUNT; j++)
        {
            unsigned int k = current_desc->IndependentBlendEnable ? j : 0;
            ok(d3d10_blend_desc.BlendEnable[j] == current_desc->RenderTarget[k].BlendEnable,
                    "Got unexpected blend enable %#x for test %u, render target %u.\n",
                    d3d10_blend_desc.BlendEnable[j], i, j);
            ok(d3d10_blend_desc.RenderTargetWriteMask[j] == current_desc->RenderTarget[k].RenderTargetWriteMask,
                    "Got unexpected render target write mask %#x for test %u, render target %u.\n",
                    d3d10_blend_desc.RenderTargetWriteMask[j], i, j);
        }

        ID3D10BlendState_Release(d3d10_blend_state);

        refcount = ID3D11BlendState_Release(blend_state1);
        ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_depthstencil_state(void)
{
    ID3D11DepthStencilState *ds_state1, *ds_state2;
    ULONG refcount, expected_refcount;
    D3D11_DEPTH_STENCIL_DESC ds_desc;
    ID3D11Device *device, *tmp;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_CreateDepthStencilState(device, NULL, &ds_state1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    ds_desc.DepthEnable = TRUE;
    ds_desc.DepthWriteMask = D3D11_DEPTH_WRITE_MASK_ALL;
    ds_desc.DepthFunc = D3D11_COMPARISON_LESS;
    ds_desc.StencilEnable = FALSE;
    ds_desc.StencilReadMask = D3D11_DEFAULT_STENCIL_READ_MASK;
    ds_desc.StencilWriteMask = D3D11_DEFAULT_STENCIL_WRITE_MASK;
    ds_desc.FrontFace.StencilFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilDepthFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilPassOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilFunc = D3D11_COMPARISON_ALWAYS;
    ds_desc.BackFace.StencilFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilDepthFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilPassOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilFunc = D3D11_COMPARISON_ALWAYS;

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateDepthStencilState(device, &ds_desc, &ds_state1);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateDepthStencilState(device, &ds_desc, &ds_state2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(ds_state1 == ds_state2, "Got different depthstencil state objects.\n");
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11DepthStencilState_GetDevice(ds_state1, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    /* Not available on all Windows versions. */
    check_interface(ds_state1, &IID_ID3D10DepthStencilState, TRUE, TRUE);

    refcount = ID3D11DepthStencilState_Release(ds_state2);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3D11DepthStencilState_Release(ds_state1);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);

    ds_desc.DepthEnable = FALSE;
    ds_desc.DepthWriteMask = D3D11_DEPTH_WRITE_MASK_ZERO;
    ds_desc.DepthFunc = D3D11_COMPARISON_NEVER;
    ds_desc.StencilEnable = FALSE;
    ds_desc.StencilReadMask = 0;
    ds_desc.StencilWriteMask = 0;
    ds_desc.FrontFace.StencilFailOp = D3D11_STENCIL_OP_ZERO;
    ds_desc.FrontFace.StencilDepthFailOp = D3D11_STENCIL_OP_ZERO;
    ds_desc.FrontFace.StencilPassOp = D3D11_STENCIL_OP_ZERO;
    ds_desc.FrontFace.StencilFunc = D3D11_COMPARISON_NEVER;
    ds_desc.BackFace = ds_desc.FrontFace;

    hr = ID3D11Device_CreateDepthStencilState(device, &ds_desc, &ds_state1);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    memset(&ds_desc, 0, sizeof(ds_desc));
    ID3D11DepthStencilState_GetDesc(ds_state1, &ds_desc);
    ok(!ds_desc.DepthEnable, "Got unexpected depth enable %#x.\n", ds_desc.DepthEnable);
    ok(ds_desc.DepthWriteMask == D3D11_DEPTH_WRITE_MASK_ALL,
            "Got unexpected depth write mask %#x.\n", ds_desc.DepthWriteMask);
    ok(ds_desc.DepthFunc == D3D11_COMPARISON_LESS, "Got unexpected depth func %#x.\n", ds_desc.DepthFunc);
    ok(!ds_desc.StencilEnable, "Got unexpected stencil enable %#x.\n", ds_desc.StencilEnable);
    ok(ds_desc.StencilReadMask == D3D11_DEFAULT_STENCIL_READ_MASK,
            "Got unexpected stencil read mask %#x.\n", ds_desc.StencilReadMask);
    ok(ds_desc.StencilWriteMask == D3D11_DEFAULT_STENCIL_WRITE_MASK,
            "Got unexpected stencil write mask %#x.\n", ds_desc.StencilWriteMask);
    ok(ds_desc.FrontFace.StencilDepthFailOp == D3D11_STENCIL_OP_KEEP,
            "Got unexpected front face stencil depth fail op %#x.\n", ds_desc.FrontFace.StencilDepthFailOp);
    ok(ds_desc.FrontFace.StencilPassOp == D3D11_STENCIL_OP_KEEP,
            "Got unexpected front face stencil pass op %#x.\n", ds_desc.FrontFace.StencilPassOp);
    ok(ds_desc.FrontFace.StencilFailOp == D3D11_STENCIL_OP_KEEP,
            "Got unexpected front face stencil fail op %#x.\n", ds_desc.FrontFace.StencilFailOp);
    ok(ds_desc.FrontFace.StencilFunc == D3D11_COMPARISON_ALWAYS,
            "Got unexpected front face stencil func %#x.\n", ds_desc.FrontFace.StencilFunc);
    ok(ds_desc.BackFace.StencilDepthFailOp == D3D11_STENCIL_OP_KEEP,
            "Got unexpected back face stencil depth fail op %#x.\n", ds_desc.BackFace.StencilDepthFailOp);
    ok(ds_desc.BackFace.StencilPassOp == D3D11_STENCIL_OP_KEEP,
            "Got unexpected back face stencil pass op %#x.\n", ds_desc.BackFace.StencilPassOp);
    ok(ds_desc.BackFace.StencilFailOp == D3D11_STENCIL_OP_KEEP,
            "Got unexpected back face stencil fail op %#x.\n", ds_desc.BackFace.StencilFailOp);
    ok(ds_desc.BackFace.StencilFunc == D3D11_COMPARISON_ALWAYS,
            "Got unexpected back face stencil func %#x.\n", ds_desc.BackFace.StencilFunc);

    ID3D11DepthStencilState_Release(ds_state1);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_rasterizer_state(void)
{
    ID3D11RasterizerState *rast_state1, *rast_state2;
    ID3D10RasterizerState *d3d10_rast_state;
    ULONG refcount, expected_refcount;
    D3D10_RASTERIZER_DESC d3d10_desc;
    D3D11_RASTERIZER_DESC desc;
    ID3D11Device *device, *tmp;
    ID3D11Device1 *device1;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_CreateRasterizerState(device, NULL, &rast_state1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    desc.FillMode = D3D11_FILL_SOLID;
    desc.CullMode = D3D11_CULL_BACK;
    desc.FrontCounterClockwise = FALSE;
    desc.DepthBias = 0;
    desc.DepthBiasClamp = 0.0f;
    desc.SlopeScaledDepthBias = 0.0f;
    desc.DepthClipEnable = TRUE;
    desc.ScissorEnable = FALSE;
    desc.MultisampleEnable = FALSE;
    desc.AntialiasedLineEnable = FALSE;

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateRasterizerState(device, &desc, &rast_state1);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRasterizerState(device, &desc, &rast_state2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(rast_state1 == rast_state2, "Got different rasterizer state objects.\n");
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11RasterizerState_GetDevice(rast_state1, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    hr = ID3D11RasterizerState_QueryInterface(rast_state1, &IID_ID3D10RasterizerState, (void **)&d3d10_rast_state);
    ok(SUCCEEDED(hr) || broken(hr == E_NOINTERFACE) /* Not available on all Windows versions. */,
            "Rasterizer state should implement ID3D10RasterizerState.\n");
    if (SUCCEEDED(hr))
    {
        ID3D10RasterizerState_GetDesc(d3d10_rast_state, &d3d10_desc);
        ok(d3d10_desc.FillMode == D3D10_FILL_SOLID, "Got unexpected fill mode %u.\n", d3d10_desc.FillMode);
        ok(d3d10_desc.CullMode == D3D10_CULL_BACK, "Got unexpected cull mode %u.\n", d3d10_desc.CullMode);
        ok(!d3d10_desc.FrontCounterClockwise, "Got unexpected front counter clockwise %#x.\n",
                d3d10_desc.FrontCounterClockwise);
        ok(!d3d10_desc.DepthBias, "Got unexpected depth bias %d.\n", d3d10_desc.DepthBias);
        ok(!d3d10_desc.DepthBiasClamp, "Got unexpected depth bias clamp %f.\n", d3d10_desc.DepthBiasClamp);
        ok(!d3d10_desc.SlopeScaledDepthBias, "Got unexpected slope scaled depth bias %f.\n",
                d3d10_desc.SlopeScaledDepthBias);
        ok(!!d3d10_desc.DepthClipEnable, "Got unexpected depth clip enable %#x.\n", d3d10_desc.DepthClipEnable);
        ok(!d3d10_desc.ScissorEnable, "Got unexpected scissor enable %#x.\n", d3d10_desc.ScissorEnable);
        ok(!d3d10_desc.MultisampleEnable, "Got unexpected multisample enable %#x.\n",
                d3d10_desc.MultisampleEnable);
        ok(!d3d10_desc.AntialiasedLineEnable, "Got unexpected antialiased line enable %#x.\n",
                d3d10_desc.AntialiasedLineEnable);

        refcount = ID3D10RasterizerState_Release(d3d10_rast_state);
        ok(refcount == 2, "Got unexpected refcount %lu.\n", refcount);
    }

    if (ID3D11Device_QueryInterface(device, &IID_ID3D11Device1, (void **)&device1) == S_OK)
    {
        ID3D11RasterizerState1 *state_ex1;
        D3D11_RASTERIZER_DESC1 desc1;

        hr = ID3D11RasterizerState_QueryInterface(rast_state1, &IID_ID3D11RasterizerState1, (void **)&state_ex1);
        ok(hr == S_OK, "Got hr %#lx.\n", hr);

        memset(&desc1, 0xcc, sizeof(desc1));
        ID3D11RasterizerState1_GetDesc1(state_ex1, &desc1);
        ok(!memcmp(&desc1, &desc, sizeof(desc)), "D3D11 desc didn't match.\n");
        ok(!desc1.ForcedSampleCount, "Got forced sample count %u.\n", desc1.ForcedSampleCount);

        ID3D11RasterizerState1_Release(state_ex1);

        memcpy(&desc1, &desc, sizeof(desc));
        desc1.ForcedSampleCount = 0;
        hr = ID3D11Device1_CreateRasterizerState1(device1, &desc1, &state_ex1);
        ok(hr == S_OK, "Got hr %#lx.\n", hr);

        ID3D11RasterizerState1_Release(state_ex1);

        ID3D11Device1_Release(device1);
    }

    refcount = ID3D11RasterizerState_Release(rast_state2);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3D11RasterizerState_Release(rast_state1);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_query(void)
{
    static const struct
    {
        D3D11_QUERY query;
        D3D_FEATURE_LEVEL required_feature_level;
        BOOL is_predicate;
        BOOL can_use_create_predicate;
        BOOL todo;
    }
    tests[] =
    {
        {D3D11_QUERY_EVENT,                         D3D_FEATURE_LEVEL_10_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_OCCLUSION,                     D3D_FEATURE_LEVEL_10_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_TIMESTAMP,                     D3D_FEATURE_LEVEL_10_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_TIMESTAMP_DISJOINT,            D3D_FEATURE_LEVEL_10_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_PIPELINE_STATISTICS,           D3D_FEATURE_LEVEL_10_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_OCCLUSION_PREDICATE,           D3D_FEATURE_LEVEL_10_0, TRUE,  TRUE,  FALSE},
        {D3D11_QUERY_SO_STATISTICS,                 D3D_FEATURE_LEVEL_10_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_SO_OVERFLOW_PREDICATE,         D3D_FEATURE_LEVEL_10_0, TRUE,  TRUE,  TRUE},
        {D3D11_QUERY_SO_STATISTICS_STREAM0,         D3D_FEATURE_LEVEL_11_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_SO_OVERFLOW_PREDICATE_STREAM0, D3D_FEATURE_LEVEL_11_0, TRUE,  FALSE, TRUE},
        {D3D11_QUERY_SO_STATISTICS_STREAM1,         D3D_FEATURE_LEVEL_11_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_SO_OVERFLOW_PREDICATE_STREAM1, D3D_FEATURE_LEVEL_11_0, TRUE,  FALSE, TRUE},
        {D3D11_QUERY_SO_STATISTICS_STREAM2,         D3D_FEATURE_LEVEL_11_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_SO_OVERFLOW_PREDICATE_STREAM2, D3D_FEATURE_LEVEL_11_0, TRUE,  FALSE, TRUE},
        {D3D11_QUERY_SO_STATISTICS_STREAM3,         D3D_FEATURE_LEVEL_11_0, FALSE, FALSE, FALSE},
        {D3D11_QUERY_SO_OVERFLOW_PREDICATE_STREAM3, D3D_FEATURE_LEVEL_11_0, TRUE,  FALSE, TRUE},
    };

    ULONG refcount, expected_refcount;
    D3D_FEATURE_LEVEL feature_level;
    D3D11_QUERY_DESC query_desc;
    ID3D11Predicate *predicate;
    ID3D11Device *device, *tmp;
    HRESULT hr, expected_hr;
    ID3D11Query *query;
    unsigned int i;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }
    feature_level = ID3D11Device_GetFeatureLevel(device);

    hr = ID3D11Device_CreateQuery(device, NULL, &query);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePredicate(device, NULL, &predicate);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        if (tests[i].required_feature_level > feature_level)
        {
            skip("Query type %u requires feature level %#x.\n", tests[i].query, tests[i].required_feature_level);
            continue;
        }

        query_desc.Query = tests[i].query;
        query_desc.MiscFlags = 0;

        hr = ID3D11Device_CreateQuery(device, &query_desc, NULL);
        todo_wine_if(tests[i].todo)
        ok(hr == S_FALSE, "Got unexpected hr %#lx for query type %u.\n", hr, query_desc.Query);

        query_desc.Query = tests[i].query;
        hr = ID3D11Device_CreateQuery(device, &query_desc, &query);
        todo_wine_if(tests[i].todo)
        ok(hr == S_OK, "Got unexpected hr %#lx for query type %u.\n", hr, query_desc.Query);
        if (FAILED(hr))
            continue;

        check_interface(query, &IID_ID3D11Predicate, tests[i].is_predicate, FALSE);
        ID3D11Query_Release(query);

        expected_hr = tests[i].can_use_create_predicate ? S_FALSE : E_INVALIDARG;
        hr = ID3D11Device_CreatePredicate(device, &query_desc, NULL);
        ok(hr == expected_hr, "Got unexpected hr %#lx for query type %u.\n", hr, query_desc.Query);

        expected_hr = tests[i].can_use_create_predicate ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreatePredicate(device, &query_desc, &predicate);
        ok(hr == expected_hr, "Got unexpected hr %#lx for query type %u.\n", hr, query_desc.Query);
        if (SUCCEEDED(hr))
            ID3D11Predicate_Release(predicate);
    }

    query_desc.Query = D3D11_QUERY_OCCLUSION_PREDICATE;
    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreatePredicate(device, &query_desc, &predicate);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Predicate_GetDevice(predicate, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);
    /* Not available on all Windows versions. */
    check_interface(predicate, &IID_ID3D10Predicate, TRUE, TRUE);
    ID3D11Predicate_Release(predicate);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

#define get_query_data(a, b, c, d) get_query_data_(__LINE__, a, b, c, d)
static void get_query_data_(unsigned int line, ID3D11DeviceContext *context,
        ID3D11Asynchronous *query, void *data, unsigned int data_size)
{
    unsigned int i;
    HRESULT hr;

    for (i = 0; i < 500; ++i)
    {
        if ((hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0)) != S_FALSE)
            break;
        Sleep(10);
    }
    ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    memset(data, 0xff, data_size);
    hr = ID3D11DeviceContext_GetData(context, query, data, data_size, 0);
    ok_(__FILE__, line)(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
}

static void test_occlusion_query(void)
{
    static const struct vec4 red = {1.0f, 0.0f, 0.0f, 1.0f};
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};

    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    D3D11_QUERY_DESC query_desc;
    ID3D11Asynchronous *query;
    unsigned int data_size, i;
    ID3D11Texture2D *texture;
    ID3D11Device *device;
    union
    {
        UINT64 uint;
        DWORD dword[2];
    } data;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);

    query_desc.Query = D3D11_QUERY_OCCLUSION;
    query_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&query);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    data_size = ID3D11Asynchronous_GetDataSize(query);
    ok(data_size == sizeof(data), "Got unexpected data size %u.\n", data_size);

    memset(&data, 0xff, sizeof(data));
    hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(data), 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    ok(data.dword[0] == 0xffffffff && data.dword[1] == 0xffffffff,
            "Data was modified 0x%08lx%08lx.\n", data.dword[1], data.dword[0]);

    ID3D11DeviceContext_End(context, query);
    ID3D11DeviceContext_Begin(context, query);
    ID3D11DeviceContext_Begin(context, query);

    memset(&data, 0xff, sizeof(data));
    hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(data), 0);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    ok(data.dword[0] == 0xffffffff && data.dword[1] == 0xffffffff,
            "Data was modified 0x%08lx%08lx.\n", data.dword[1], data.dword[0]);

    draw_color_quad(&test_context, &red);

    ID3D11DeviceContext_End(context, query);
    get_query_data(context, query, &data, sizeof(data));
    ok(data.uint == 640 * 480, "Got unexpected query result 0x%08lx%08lx.\n", data.dword[1], data.dword[0]);

    memset(&data, 0xff, sizeof(data));
    hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(DWORD), 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(WORD), 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(data) - 1, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(data) + 1, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(data.dword[0] == 0xffffffff && data.dword[1] == 0xffffffff,
            "Data was modified 0x%08lx%08lx.\n", data.dword[1], data.dword[0]);

    memset(&data, 0xff, sizeof(data));
    hr = ID3D11DeviceContext_GetData(context, query, &data, 0, 0);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(data.dword[0] == 0xffffffff && data.dword[1] == 0xffffffff,
            "Data was modified 0x%08lx%08lx.\n", data.dword[1], data.dword[0]);

    hr = ID3D11DeviceContext_GetData(context, query, NULL, sizeof(DWORD), 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, query, NULL, sizeof(data), 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_Begin(context, query);
    ID3D11DeviceContext_End(context, query);
    ID3D11DeviceContext_End(context, query);

    get_query_data(context, query, &data, sizeof(data));
    ok(!data.uint, "Got unexpected query result 0x%08lx%08lx.\n", data.dword[1], data.dword[0]);
    hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture_desc.Width = D3D10_REQ_TEXTURE2D_U_OR_V_DIMENSION;
    texture_desc.Height = D3D10_REQ_TEXTURE2D_U_OR_V_DIMENSION;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
    set_viewport(context, 0.0f, 0.0f, texture_desc.Width, texture_desc.Height, 0.0f, 1.0f);

    ID3D11DeviceContext_Begin(context, query);
    for (i = 0; i < 100; i++)
        draw_color_quad(&test_context, &red);
    ID3D11DeviceContext_End(context, query);

    get_query_data(context, query, &data, sizeof(data));
    ok((data.dword[0] == 0x90000000 && data.dword[1] == 0x1)
            || (data.dword[0] == 0xffffffff && !data.dword[1])
            || broken(!data.uint),
            "Got unexpected query result 0x%08lx%08lx.\n", data.dword[1], data.dword[0]);
    hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11Asynchronous_Release(query);

    /* The following test exercises a code path in wined3d. A wined3d context
     * associated with the query is destroyed when the swapchain is released. */
    hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&query);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    set_viewport(context, 0.0f, 0.0f, 64.0f, 64.0f, 0.0f, 1.0f);
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);
    ID3D11DeviceContext_Begin(context, query);
    draw_color_quad(&test_context, &red);
    ID3D11DeviceContext_End(context, query);

    ID3D11RenderTargetView_Release(test_context.backbuffer_rtv);
    ID3D11Texture2D_Release(test_context.backbuffer);
    IDXGISwapChain_Release(test_context.swapchain);
    test_context.swapchain = create_swapchain(device, test_context.window, NULL);
    hr = IDXGISwapChain_GetBuffer(test_context.swapchain, 0, &IID_ID3D11Texture2D,
            (void **)&test_context.backbuffer);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)test_context.backbuffer,
            NULL, &test_context.backbuffer_rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &test_context.backbuffer_rtv, NULL);
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);

    get_query_data(context, query, &data, sizeof(data));
    /* This test occasionally succeeds with CSMT enabled because of a race condition. */
    if (0)
    {
        todo_wine ok(data.dword[0] == 0x1000 && !data.dword[1],
                "Got unexpected query result 0x%08lx%08lx.\n", data.dword[1], data.dword[0]);
    }

    ID3D11Asynchronous_Release(query);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11Texture2D_Release(texture);
    release_test_context(&test_context);
}

static void test_pipeline_statistics_query(void)
{
    static const D3D11_QUERY_DATA_PIPELINE_STATISTICS zero_data = {0};
    static const struct vec4 red = {1.0f, 0.0f, 0.0f, 1.0f};
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};

    D3D11_QUERY_DATA_PIPELINE_STATISTICS data;
    struct d3d11_test_context test_context;
    ID3D11DeviceContext *context;
    D3D11_QUERY_DESC query_desc;
    ID3D11Asynchronous *query;
    unsigned int data_size;
    ID3D11Device *device;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);

    query_desc.Query = D3D11_QUERY_PIPELINE_STATISTICS;
    query_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&query);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    data_size = ID3D11Asynchronous_GetDataSize(query);
    ok(data_size == sizeof(data), "Got unexpected data size %u.\n", data_size);

    hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(data), 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_End(context, query);
    ID3D11DeviceContext_Begin(context, query);
    ID3D11DeviceContext_Begin(context, query);

    memset(&data, 0xff, sizeof(data));
    hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(data), 0);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    ok(data.IAVertices == ~(UINT64)0, "Data was modified.\n");

    draw_quad(&test_context);

    ID3D11DeviceContext_End(context, query);
    get_query_data(context, query, &data, sizeof(data));

    /* WARP devices randomly return all-zeroed structures as if the draw did not happen. Flushing and
     * sleeping a second before ending the query reduces the likelihood of hitting the bug a lot, but
     * does not eliminate it entirely. To make things work reliably ignore such broken results. */
    if (is_warp_device(device) && !memcmp(&data, &zero_data, sizeof(data)))
    {
        win_skip("WARP device randomly returns zeroed query results.\n");
    }
    else
    {
        ok(data.IAVertices == 4, "Got unexpected IAVertices count: %u.\n", (unsigned int)data.IAVertices);
        ok(data.IAPrimitives == 2, "Got unexpected IAPrimitives count: %u.\n", (unsigned int)data.IAPrimitives);
        ok(data.VSInvocations == 4, "Got unexpected VSInvocations count: %u.\n", (unsigned int)data.VSInvocations);
        ok(!data.GSInvocations, "Got unexpected GSInvocations count: %u.\n", (unsigned int)data.GSInvocations);
        ok(!data.GSPrimitives, "Got unexpected GSPrimitives count: %u.\n", (unsigned int)data.GSPrimitives);
        ok(data.CInvocations == 2, "Got unexpected CInvocations count: %u.\n", (unsigned int)data.CInvocations);
        ok(data.CPrimitives == 2, "Got unexpected CPrimitives count: %u.\n", (unsigned int)data.CPrimitives);
        todo_wine
        ok(!data.PSInvocations, "Got unexpected PSInvocations count: %u.\n", (unsigned int)data.PSInvocations);
        ok(!data.HSInvocations, "Got unexpected HSInvocations count: %u.\n", (unsigned int)data.HSInvocations);
        ok(!data.DSInvocations, "Got unexpected DSInvocations count: %u.\n", (unsigned int)data.DSInvocations);
        ok(!data.CSInvocations, "Got unexpected CSInvocations count: %u.\n", (unsigned int)data.CSInvocations);
    }

    ID3D11DeviceContext_Begin(context, query);
    draw_color_quad(&test_context, &red);
    ID3D11DeviceContext_End(context, query);
    get_query_data(context, query, &data, sizeof(data));
    ok(data.IAVertices == 4, "Got unexpected IAVertices count: %u.\n", (unsigned int)data.IAVertices);
    ok(data.IAPrimitives == 2, "Got unexpected IAPrimitives count: %u.\n", (unsigned int)data.IAPrimitives);
    ok(data.VSInvocations == 4, "Got unexpected VSInvocations count: %u.\n", (unsigned int)data.VSInvocations);
    ok(!data.GSInvocations, "Got unexpected GSInvocations count: %u.\n", (unsigned int)data.GSInvocations);
    ok(!data.GSPrimitives, "Got unexpected GSPrimitives count: %u.\n", (unsigned int)data.GSPrimitives);
    ok(data.CInvocations == 2, "Got unexpected CInvocations count: %u.\n", (unsigned int)data.CInvocations);
    ok(data.CPrimitives == 2, "Got unexpected CPrimitives count: %u.\n", (unsigned int)data.CPrimitives);
    ok(data.PSInvocations >= 640 * 480, "Got unexpected PSInvocations count: %u.\n", (unsigned int)data.PSInvocations);
    ok(!data.HSInvocations, "Got unexpected HSInvocations count: %u.\n", (unsigned int)data.HSInvocations);
    ok(!data.DSInvocations, "Got unexpected DSInvocations count: %u.\n", (unsigned int)data.DSInvocations);
    ok(!data.CSInvocations, "Got unexpected CSInvocations count: %u.\n", (unsigned int)data.CSInvocations);

    ID3D11Asynchronous_Release(query);
    release_test_context(&test_context);
}

static void test_timestamp_query(void)
{
    static const struct vec4 red = {1.0f, 0.0f, 0.0f, 1.0f};

    ID3D11Asynchronous *timestamp_query, *timestamp_disjoint_query;
    D3D11_QUERY_DATA_TIMESTAMP_DISJOINT disjoint, prev_disjoint;
    struct d3d11_test_context test_context;
    ID3D11DeviceContext *context;
    D3D11_QUERY_DESC query_desc;
    unsigned int data_size;
    ID3D11Device *device;
    UINT64 timestamp;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    query_desc.Query = D3D11_QUERY_TIMESTAMP;
    query_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&timestamp_query);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    data_size = ID3D11Asynchronous_GetDataSize(timestamp_query);
    ok(data_size == sizeof(UINT64), "Got unexpected data size %u.\n", data_size);

    query_desc.Query = D3D11_QUERY_TIMESTAMP_DISJOINT;
    query_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&timestamp_disjoint_query);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    data_size = ID3D11Asynchronous_GetDataSize(timestamp_disjoint_query);
    ok(data_size == sizeof(disjoint), "Got unexpected data size %u.\n", data_size);

    disjoint.Frequency = 0xdeadbeef;
    disjoint.Disjoint = 0xdeadbeef;
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, NULL, 0, 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint), 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    ok(disjoint.Frequency == 0xdeadbeef, "Frequency data was modified.\n");
    ok(disjoint.Disjoint == 0xdeadbeef, "Disjoint data was modified.\n");

    /* Test a TIMESTAMP_DISJOINT query. */
    ID3D11DeviceContext_Begin(context, timestamp_disjoint_query);

    disjoint.Frequency = 0xdeadbeef;
    disjoint.Disjoint = 0xdeadbeef;
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, NULL, 0, 0);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint), 0);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    ok(disjoint.Frequency == 0xdeadbeef, "Frequency data was modified.\n");
    ok(disjoint.Disjoint == 0xdeadbeef, "Disjoint data was modified.\n");

    ID3D11DeviceContext_End(context, timestamp_disjoint_query);
    get_query_data(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint));
    ok(disjoint.Frequency != ~(UINT64)0, "Frequency data was not modified.\n");
    ok(disjoint.Disjoint == TRUE || disjoint.Disjoint == FALSE, "Got unexpected disjoint %#x.\n", disjoint.Disjoint);

    prev_disjoint = disjoint;

    disjoint.Frequency = 0xdeadbeef;
    disjoint.Disjoint = 0xff;
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint) - 1, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint) + 1, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint) / 2, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint) * 2, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(disjoint.Frequency == 0xdeadbeef, "Frequency data was modified.\n");
    ok(disjoint.Disjoint == 0xff, "Disjoint data was modified.\n");

    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, NULL, 0, 0);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint),
            D3D11_ASYNC_GETDATA_DONOTFLUSH);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(disjoint.Frequency == prev_disjoint.Frequency, "Frequency data mismatch.\n");
    ok(disjoint.Disjoint == prev_disjoint.Disjoint, "Disjoint data mismatch.\n");

    memset(&timestamp, 0xff, sizeof(timestamp));
    hr = ID3D11DeviceContext_GetData(context, timestamp_query, NULL, 0, 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_query, &timestamp, sizeof(timestamp), 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    ok(timestamp == ~(UINT64)0, "Timestamp data was modified.\n");

    /* Test a TIMESTAMP query inside a TIMESTAMP_DISJOINT query. */
    ID3D11DeviceContext_Begin(context, timestamp_disjoint_query);

    memset(&timestamp, 0xff, sizeof(timestamp));
    hr = ID3D11DeviceContext_GetData(context, timestamp_query, &timestamp, sizeof(timestamp), 0);
    ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
    ok(timestamp == ~(UINT64)0, "Timestamp data was modified.\n");

    draw_color_quad(&test_context, &red);

    ID3D11DeviceContext_End(context, timestamp_query);
    get_query_data(context, timestamp_query, &timestamp, sizeof(timestamp));

    timestamp = 0xdeadbeef;
    hr = ID3D11DeviceContext_GetData(context, timestamp_query, &timestamp, sizeof(timestamp) / 2, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(timestamp == 0xdeadbeef, "Timestamp was modified.\n");

    hr = ID3D11DeviceContext_GetData(context, timestamp_query, &timestamp, sizeof(timestamp), 0);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(timestamp != 0xdeadbeef, "Timestamp was not modified.\n");

    timestamp = 0xdeadbeef;
    hr = ID3D11DeviceContext_GetData(context, timestamp_query, &timestamp, sizeof(timestamp) - 1, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_query, &timestamp, sizeof(timestamp) + 1, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_query, &timestamp, sizeof(timestamp) / 2, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11DeviceContext_GetData(context, timestamp_query, &timestamp, sizeof(timestamp) * 2, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(timestamp == 0xdeadbeef, "Timestamp was modified.\n");

    ID3D11DeviceContext_End(context, timestamp_disjoint_query);
    get_query_data(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint));
    disjoint.Frequency = 0xdeadbeef;
    disjoint.Disjoint = 0xff;
    hr = ID3D11DeviceContext_GetData(context, timestamp_disjoint_query, &disjoint, sizeof(disjoint), 0);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(disjoint.Frequency != 0xdeadbeef, "Frequency data was not modified.\n");
    ok(disjoint.Disjoint == TRUE || disjoint.Disjoint == FALSE, "Got unexpected disjoint %#x.\n", disjoint.Disjoint);

    /* It's not strictly necessary for the TIMESTAMP query to be inside a TIMESTAMP_DISJOINT query. */
    ID3D11Asynchronous_Release(timestamp_query);
    query_desc.Query = D3D11_QUERY_TIMESTAMP;
    query_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&timestamp_query);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    draw_color_quad(&test_context, &red);

    ID3D11DeviceContext_End(context, timestamp_query);
    get_query_data(context, timestamp_query, &timestamp, sizeof(timestamp));

    ID3D11Asynchronous_Release(timestamp_query);
    ID3D11Asynchronous_Release(timestamp_disjoint_query);
    release_test_context(&test_context);
}

static void test_so_statistics_query(void)
{
    struct d3d11_test_context test_context;
    D3D11_QUERY_DATA_SO_STATISTICS data;
    ID3D11DeviceContext *context;
    unsigned int vertex_count[4];
    D3D11_QUERY_DESC query_desc;
    ID3D11Buffer *so_buffer[4];
    ID3D11Asynchronous *query;
    ID3D11GeometryShader *gs;
    ID3D11VertexShader *vs;
    unsigned int data_size;
    ID3D11Device *device;
    ID3D11Buffer *cb;
    unsigned int i;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        float4 main(uint id : SV_VertexID) : custom
        {
            return (float4)id;
        }
#endif
        0x43425844, 0x8b0e47b9, 0x6efc9512, 0xd55ca6ff, 0x487c5ef2, 0x00000001, 0x000000d4, 0x00000003,
        0x0000002c, 0x00000060, 0x00000090, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000006, 0x00000001, 0x00000000, 0x00000101, 0x565f5653, 0x65747265, 0x00444978,
        0x4e47534f, 0x00000028, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x74737563, 0xab006d6f, 0x52444853, 0x0000003c, 0x00010040, 0x0000000f,
        0x04000060, 0x00101012, 0x00000000, 0x00000006, 0x03000065, 0x001020f2, 0x00000000, 0x05000056,
        0x001020f2, 0x00000000, 0x00101006, 0x00000000, 0x0100003e,
    };
    static const DWORD gs_code[] =
    {
#if 0
        struct vertex
        {
            float4 data : custom;
        };

        uint4 vertex_count;

        [maxvertexcount(32)]
        void main(point vertex input[1], uint id : SV_PrimitiveID,
                inout PointStream<vertex> output0,
                inout PointStream<vertex> output1,
                inout PointStream<vertex> output2,
                inout PointStream<vertex> output3)
        {
            if (id < vertex_count.x)
                output0.Append(input[0]);
            if (id < vertex_count.y)
                output1.Append(input[0]);
            if (id < vertex_count.z)
                output2.Append(input[0]);
            if (id < vertex_count.w)
                output3.Append(input[0]);
        }
#endif
        0x43425844, 0xd616829d, 0x4355ce2a, 0xd71909e5, 0xdc916d4c, 0x00000001, 0x000002bc, 0x00000003,
        0x0000002c, 0x00000084, 0x0000010c, 0x4e475349, 0x00000050, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x0000003f, 0x00000000, 0x00000007,
        0x00000001, 0xffffffff, 0x00000101, 0x74737563, 0x53006d6f, 0x72505f56, 0x74696d69, 0x49657669,
        0xabab0044, 0x3547534f, 0x00000080, 0x00000004, 0x00000008, 0x00000000, 0x00000078, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000001, 0x00000078, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x00000002, 0x00000078, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x00000003, 0x00000078, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x74737563, 0xab006d6f, 0x58454853, 0x000001a8, 0x00020050, 0x0000006a, 0x0100086a,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0400005f, 0x002010f2, 0x00000001, 0x00000000,
        0x0200005f, 0x0000b000, 0x02000068, 0x00000001, 0x0100085d, 0x0300008f, 0x00110000, 0x00000000,
        0x0100085c, 0x03000065, 0x001020f2, 0x00000000, 0x0300008f, 0x00110000, 0x00000001, 0x0100085c,
        0x03000065, 0x001020f2, 0x00000000, 0x0300008f, 0x00110000, 0x00000002, 0x0100085c, 0x03000065,
        0x001020f2, 0x00000000, 0x0300008f, 0x00110000, 0x00000003, 0x0100085c, 0x03000065, 0x001020f2,
        0x00000000, 0x0200005e, 0x00000020, 0x0700004f, 0x001000f2, 0x00000000, 0x0000b001, 0x00208e46,
        0x00000000, 0x00000000, 0x0304001f, 0x0010000a, 0x00000000, 0x06000036, 0x001020f2, 0x00000000,
        0x00201e46, 0x00000000, 0x00000000, 0x03000075, 0x00110000, 0x00000000, 0x01000015, 0x0304001f,
        0x0010001a, 0x00000000, 0x06000036, 0x001020f2, 0x00000000, 0x00201e46, 0x00000000, 0x00000000,
        0x03000075, 0x00110000, 0x00000001, 0x01000015, 0x0304001f, 0x0010002a, 0x00000000, 0x06000036,
        0x001020f2, 0x00000000, 0x00201e46, 0x00000000, 0x00000000, 0x03000075, 0x00110000, 0x00000002,
        0x01000015, 0x0304001f, 0x0010003a, 0x00000000, 0x06000036, 0x001020f2, 0x00000000, 0x00201e46,
        0x00000000, 0x00000000, 0x03000075, 0x00110000, 0x00000003, 0x01000015, 0x0100003e,
    };
    static const D3D11_SO_DECLARATION_ENTRY so_declaration[] =
    {
        {0, "custom", 0, 0, 4, 0},
        {1, "custom", 0, 0, 4, 1},
        {2, "custom", 0, 0, 4, 2},
        {3, "custom", 0, 0, 4, 3},
    };
    static const unsigned int offset[4] = {0};

    static const struct
    {
        D3D11_QUERY query;
        D3D_FEATURE_LEVEL feature_level;
    }
    tests[] =
    {
        {D3D11_QUERY_SO_STATISTICS,         D3D_FEATURE_LEVEL_10_0},
        {D3D11_QUERY_SO_STATISTICS_STREAM0, D3D_FEATURE_LEVEL_11_0},
        {D3D11_QUERY_SO_STATISTICS_STREAM1, D3D_FEATURE_LEVEL_11_0},
        {D3D11_QUERY_SO_STATISTICS_STREAM2, D3D_FEATURE_LEVEL_11_0},
        {D3D11_QUERY_SO_STATISTICS_STREAM3, D3D_FEATURE_LEVEL_11_0},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        if (ID3D11Device_GetFeatureLevel(device) < tests[i].feature_level)
        {
            skip("Feature level %#x is required.\n", tests[i].feature_level);
            continue;
        }

        query_desc.Query = tests[i].query;
        query_desc.MiscFlags = 0;
        hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&query);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        data_size = ID3D11Asynchronous_GetDataSize(query);
        ok(data_size == sizeof(data), "Got unexpected data size %u.\n", data_size);

        hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0);
        ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
        hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(data), 0);
        ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);

        ID3D11DeviceContext_End(context, query);
        ID3D11DeviceContext_Begin(context, query);
        ID3D11DeviceContext_Begin(context, query);

        memset(&data, 0xff, sizeof(data));
        hr = ID3D11DeviceContext_GetData(context, query, NULL, 0, 0);
        todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
        hr = ID3D11DeviceContext_GetData(context, query, &data, sizeof(data), 0);
        todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);
        ok(data.NumPrimitivesWritten == ~(UINT64)0, "Data was modified.\n");
        ok(data.PrimitivesStorageNeeded == ~(UINT64)0, "Data was modified.\n");

        draw_quad(&test_context);

        ID3D11DeviceContext_End(context, query);
        get_query_data(context, query, &data, sizeof(data));
        ok(!data.NumPrimitivesWritten, "Got unexpected NumPrimitivesWritten: %u.\n",
                (unsigned int)data.NumPrimitivesWritten);
        todo_wine
        ok(!data.PrimitivesStorageNeeded, "Got unexpected PrimitivesStorageNeeded: %u.\n",
                (unsigned int)data.PrimitivesStorageNeeded);

        ID3D11DeviceContext_Begin(context, query);
        draw_quad(&test_context);
        ID3D11DeviceContext_End(context, query);
        get_query_data(context, query, &data, sizeof(data));
        ok(!data.NumPrimitivesWritten, "Got unexpected NumPrimitivesWritten: %u.\n",
                (unsigned int)data.NumPrimitivesWritten);
        todo_wine
        ok(!data.PrimitivesStorageNeeded, "Got unexpected PrimitivesStorageNeeded: %u.\n",
                (unsigned int)data.PrimitivesStorageNeeded);

        ID3D11Asynchronous_Release(query);
    }

    if (ID3D11Device_GetFeatureLevel(device) < D3D_FEATURE_LEVEL_11_0)
    {
        skip("Vertex streams are not supported.\n");
        goto done;
    }

    /* multiple vertex streams */
    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);

    hr = ID3D11Device_CreateGeometryShaderWithStreamOutput(device, gs_code, sizeof(gs_code),
            so_declaration, ARRAY_SIZE(so_declaration),
            NULL, 0, D3D11_SO_NO_RASTERIZED_STREAM, NULL, &gs);
    todo_wine
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    if (FAILED(hr))
    {
        ID3D11VertexShader_Release(vs);
        goto done;
    }
    ID3D11DeviceContext_GSSetShader(context, gs, NULL, 0);

    for (i = 0; i < ARRAY_SIZE(vertex_count); ++i)
        vertex_count[i] = 5;
    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(vertex_count), vertex_count);
    ID3D11DeviceContext_GSSetConstantBuffers(context, 0, 1, &cb);

    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_POINTLIST);

    query_desc.Query = D3D11_QUERY_SO_STATISTICS;
    query_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&query);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_Begin(context, query);
    ID3D11DeviceContext_Draw(context, 5, 0);
    ID3D11DeviceContext_End(context, query);

    memset(&data, 0xff, sizeof(data));
    get_query_data(context, query, &data, sizeof(data));
    ok(!data.NumPrimitivesWritten, "Got unexpected primitives written %u.\n",
            (unsigned int)data.NumPrimitivesWritten);
    ok(data.PrimitivesStorageNeeded == 5, "Got unexpected primitives storage needed %u.\n",
            (unsigned int)data.PrimitivesStorageNeeded);

    for (i = 0; i < ARRAY_SIZE(so_buffer); ++i)
        so_buffer[i] = create_buffer(device, D3D11_BIND_STREAM_OUTPUT, sizeof(struct vec4) * 10, NULL);

    ID3D11DeviceContext_SOSetTargets(context, D3D11_SO_BUFFER_SLOT_COUNT, so_buffer, offset);
    ID3D11DeviceContext_Begin(context, query);
    ID3D11DeviceContext_Draw(context, 16, 0);
    ID3D11DeviceContext_End(context, query);
    memset(&data, 0xff, sizeof(data));
    get_query_data(context, query, &data, sizeof(data));
    ok(data.NumPrimitivesWritten == 5, "Got unexpected primitives written %u.\n",
            (unsigned int)data.NumPrimitivesWritten);
    ok(data.PrimitivesStorageNeeded == 5, "Got unexpected primitives storage needed %u.\n",
            (unsigned int)data.PrimitivesStorageNeeded);

    vertex_count[0] = 3;
    vertex_count[1] = 6;
    vertex_count[2] = 4;
    vertex_count[3] = 12;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, vertex_count, 0, 0);

    ID3D11DeviceContext_SOSetTargets(context, D3D11_SO_BUFFER_SLOT_COUNT, so_buffer, offset);
    ID3D11DeviceContext_Begin(context, query);
    ID3D11DeviceContext_Draw(context, 32, 0);
    ID3D11DeviceContext_End(context, query);
    memset(&data, 0xff, sizeof(data));
    get_query_data(context, query, &data, sizeof(data));
    ok(data.NumPrimitivesWritten == 3, "Got unexpected primitives written %u.\n",
            (unsigned int)data.NumPrimitivesWritten);
    ok(data.PrimitivesStorageNeeded == 3, "Got unexpected primitives storage needed %u.\n",
            (unsigned int)data.PrimitivesStorageNeeded);

    vertex_count[0] = 16;
    vertex_count[1] = 6;
    vertex_count[2] = 4;
    vertex_count[3] = 12;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, vertex_count, 0, 0);

    ID3D11DeviceContext_SOSetTargets(context, D3D11_SO_BUFFER_SLOT_COUNT, so_buffer, offset);
    ID3D11DeviceContext_Begin(context, query);
    ID3D11DeviceContext_Draw(context, 32, 0);
    ID3D11DeviceContext_End(context, query);
    memset(&data, 0xff, sizeof(data));
    get_query_data(context, query, &data, sizeof(data));
    ok(data.NumPrimitivesWritten == 10, "Got unexpected primitives written %u.\n",
            (unsigned int)data.NumPrimitivesWritten);
    ok(data.PrimitivesStorageNeeded == 16, "Got unexpected primitives storage needed %u.\n",
            (unsigned int)data.PrimitivesStorageNeeded);

    ID3D11Asynchronous_Release(query);

    for (i = 0; i < ARRAY_SIZE(so_buffer); ++i)
        ID3D11Buffer_Release(so_buffer[i]);
    ID3D11Buffer_Release(cb);
    ID3D11GeometryShader_Release(gs);
    ID3D11VertexShader_Release(vs);

done:
    release_test_context(&test_context);
}

static void test_device_removed_reason(void)
{
    ID3D11Device *device;
    ULONG refcount;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_GetDeviceRemovedReason(device);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_GetDeviceRemovedReason(device);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_private_data(void)
{
    ULONG refcount, expected_refcount;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D10Texture2D *d3d10_texture;
    ID3D11Device *test_object;
    ID3D11Texture2D *texture;
    IDXGIDevice *dxgi_device;
    IDXGISurface *surface;
    ID3D11Device *device;
    IUnknown *ptr;
    HRESULT hr;
    UINT size;

    static const GUID test_guid =
            {0xfdb37466, 0x428f, 0x4edf, {0xa3, 0x7f, 0x9b, 0x1d, 0xf4, 0x88, 0xc5, 0xfc}};
    static const GUID test_guid2 =
            {0x2e5afac2, 0x87b5, 0x4c10, {0x9b, 0x4b, 0x89, 0xd7, 0xd1, 0x12, 0xe7, 0x2b}};
    static const DWORD data[] = {1, 2, 3, 4};

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    test_object = create_device(NULL);

    texture_desc.Width = 512;
    texture_desc.Height = 512;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Texture2D_QueryInterface(texture, &IID_IDXGISurface, (void **)&surface);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_SetPrivateData(device, &test_guid, 0, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_SetPrivateDataInterface(device, &test_guid, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_SetPrivateData(device, &test_guid, ~0u, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_SetPrivateData(device, &test_guid, ~0u, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_SetPrivateDataInterface(device, &test_guid, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    size = sizeof(ptr) * 2;
    ptr = (IUnknown *)0xdeadbeef;
    hr = ID3D11Device_GetPrivateData(device, &test_guid, &size, &ptr);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(!ptr, "Got unexpected pointer %p.\n", ptr);
    ok(size == sizeof(IUnknown *), "Got unexpected size %u.\n", size);

    hr = ID3D11Device_QueryInterface(device, &IID_IDXGIDevice, (void **)&dxgi_device);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    size = sizeof(ptr) * 2;
    ptr = (IUnknown *)0xdeadbeef;
    hr = IDXGIDevice_GetPrivateData(dxgi_device, &test_guid, &size, &ptr);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(!ptr, "Got unexpected pointer %p.\n", ptr);
    ok(size == sizeof(IUnknown *), "Got unexpected size %u.\n", size);
    IDXGIDevice_Release(dxgi_device);

    refcount = get_refcount(test_object);
    hr = ID3D11Device_SetPrivateDataInterface(device, &test_guid, (IUnknown *)test_object);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    expected_refcount = refcount + 1;
    refcount = get_refcount(test_object);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    hr = ID3D11Device_SetPrivateDataInterface(device, &test_guid, (IUnknown *)test_object);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(test_object);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);

    hr = ID3D11Device_SetPrivateDataInterface(device, &test_guid, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    --expected_refcount;
    refcount = get_refcount(test_object);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);

    hr = ID3D11Device_SetPrivateDataInterface(device, &test_guid, (IUnknown *)test_object);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    size = sizeof(data);
    hr = ID3D11Device_SetPrivateData(device, &test_guid, size, data);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(test_object);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    hr = ID3D11Device_SetPrivateData(device, &test_guid, 42, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_SetPrivateData(device, &test_guid, 42, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_SetPrivateDataInterface(device, &test_guid, (IUnknown *)test_object);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ++expected_refcount;
    size = 2 * sizeof(ptr);
    ptr = NULL;
    hr = ID3D11Device_GetPrivateData(device, &test_guid, &size, &ptr);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(size == sizeof(test_object), "Got unexpected size %u.\n", size);
    ++expected_refcount;
    refcount = get_refcount(test_object);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    IUnknown_Release(ptr);
    --expected_refcount;

    ptr = (IUnknown *)0xdeadbeef;
    size = 1;
    hr = ID3D11Device_GetPrivateData(device, &test_guid, &size, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(size == sizeof(device), "Got unexpected size %u.\n", size);
    size = 2 * sizeof(ptr);
    hr = ID3D11Device_GetPrivateData(device, &test_guid, &size, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(size == sizeof(device), "Got unexpected size %u.\n", size);
    refcount = get_refcount(test_object);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);

    size = 1;
    hr = ID3D11Device_GetPrivateData(device, &test_guid, &size, &ptr);
    ok(hr == DXGI_ERROR_MORE_DATA, "Got unexpected hr %#lx.\n", hr);
    ok(size == sizeof(device), "Got unexpected size %u.\n", size);
    ok(ptr == (IUnknown *)0xdeadbeef, "Got unexpected pointer %p.\n", ptr);
    if (!enable_debug_layer)
    {
        hr = ID3D11Device_GetPrivateData(device, &test_guid2, NULL, NULL);
        ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
        size = 0xdeadbabe;
        hr = ID3D11Device_GetPrivateData(device, &test_guid2, &size, &ptr);
        ok(hr == DXGI_ERROR_NOT_FOUND, "Got unexpected hr %#lx.\n", hr);
        ok(size == 0, "Got unexpected size %u.\n", size);
        ok(ptr == (IUnknown *)0xdeadbeef, "Got unexpected pointer %p.\n", ptr);
        hr = ID3D11Device_GetPrivateData(device, &test_guid, NULL, &ptr);
        ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
        ok(ptr == (IUnknown *)0xdeadbeef, "Got unexpected pointer %p.\n", ptr);
    }

    hr = ID3D11Texture2D_SetPrivateDataInterface(texture, &test_guid, (IUnknown *)test_object);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ptr = NULL;
    size = sizeof(ptr);
    hr = IDXGISurface_GetPrivateData(surface, &test_guid, &size, &ptr);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(ptr == (IUnknown *)test_object, "Got unexpected ptr %p, expected %p.\n", ptr, test_object);
    IUnknown_Release(ptr);

    hr = ID3D11Texture2D_QueryInterface(texture, &IID_ID3D10Texture2D, (void **)&d3d10_texture);
    ok(SUCCEEDED(hr) || broken(hr == E_NOINTERFACE) /* Not available on all Windows versions. */,
            "Texture should implement ID3D10Texture2D.\n");
    if (SUCCEEDED(hr))
    {
        ptr = NULL;
        size = sizeof(ptr);
        hr = ID3D10Texture2D_GetPrivateData(d3d10_texture, &test_guid, &size, &ptr);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        ok(ptr == (IUnknown *)test_object, "Got unexpected ptr %p, expected %p.\n", ptr, test_object);
        IUnknown_Release(ptr);
        ID3D10Texture2D_Release(d3d10_texture);
    }

    IDXGISurface_Release(surface);
    ID3D11Texture2D_Release(texture);
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
    refcount = ID3D11Device_Release(test_object);
    ok(!refcount, "Test object has %lu references left.\n", refcount);
}

static void test_state_refcounting(const D3D_FEATURE_LEVEL feature_level)
{
    ID3D11RasterizerState *rasterizer_state, *tmp_rasterizer_state;
    ID3D11Predicate *predicate, *tmp_predicate;
    ID3D11SamplerState *sampler, *tmp_sampler;
    ID3D11ShaderResourceView *srv, *tmp_srv;
    ID3D11RenderTargetView *rtv, *tmp_rtv;
    D3D11_RASTERIZER_DESC rasterizer_desc;
    D3D11_TEXTURE2D_DESC texture_desc;
    D3D11_QUERY_DESC predicate_desc;
    D3D11_SAMPLER_DESC sampler_desc;
    struct device_desc device_desc;
    ID3D11DeviceContext *context;
    ID3D11Texture2D *texture;
    ID3D11Device *device;
    ULONG refcount;
    HRESULT hr;

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }

    ID3D11Device_GetImmediateContext(device, &context);

    /* ID3D11SamplerState */
    memset(&sampler_desc, 0, sizeof(sampler_desc));
    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
    sampler_desc.MaxLOD = FLT_MAX;
    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &tmp_sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);

    tmp_sampler = sampler;
    refcount = get_refcount(sampler);
    ok(refcount == 1, "Got refcount %lu, expected 1.\n", refcount);
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);
    refcount = ID3D11SamplerState_Release(sampler);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);
    sampler = NULL;
    ID3D11DeviceContext_PSGetSamplers(context, 0, 1, &sampler);
    ok(sampler == tmp_sampler, "Got sampler %p, expected %p.\n", sampler, tmp_sampler);
    refcount = ID3D11SamplerState_Release(sampler);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);

    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &tmp_sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    refcount = ID3D11SamplerState_Release(tmp_sampler);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);

    /* ID3D11RasterizerState */
    memset(&rasterizer_desc, 0, sizeof(rasterizer_desc));
    rasterizer_desc.FillMode = D3D11_FILL_SOLID;
    rasterizer_desc.CullMode = D3D11_CULL_BACK;
    rasterizer_desc.DepthClipEnable = TRUE;
    hr = ID3D11Device_CreateRasterizerState(device, &rasterizer_desc, &rasterizer_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_RSSetState(context, rasterizer_state);
    refcount = ID3D11RasterizerState_Release(rasterizer_state);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);
    ID3D11DeviceContext_RSGetState(context, &tmp_rasterizer_state);
    ok(tmp_rasterizer_state == rasterizer_state, "Got rasterizer state %p, expected %p.\n",
            tmp_rasterizer_state, rasterizer_state);
    refcount = ID3D11RasterizerState_Release(tmp_rasterizer_state);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);

    /* ID3D11ShaderResourceView */
    memset(&texture_desc, 0, sizeof(texture_desc));
    texture_desc.Width = 32;
    texture_desc.Height = 32;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, NULL, &srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Texture2D_Release(texture);

    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);
    refcount = ID3D11ShaderResourceView_Release(srv);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);
    ID3D11DeviceContext_PSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got SRV %p, expected %p.\n", tmp_srv, srv);
    refcount = ID3D11ShaderResourceView_Release(tmp_srv);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);

    /* ID3D11RenderTargetView */
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Texture2D_Release(texture);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
    refcount = ID3D11RenderTargetView_Release(rtv);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);
    ID3D11DeviceContext_OMGetRenderTargets(context, 1, &tmp_rtv, NULL);
    ok(tmp_rtv == rtv, "Got RTV %p, expected %p.\n", tmp_rtv, rtv);
    refcount = ID3D11RenderTargetView_Release(tmp_rtv);
    ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);

    /* ID3D11Predicate */
    if (feature_level >= D3D_FEATURE_LEVEL_10_0)
    {
        predicate_desc.Query = D3D11_QUERY_OCCLUSION_PREDICATE;
        predicate_desc.MiscFlags = 0;
        hr = ID3D11Device_CreatePredicate(device, &predicate_desc, &predicate);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

        ID3D11DeviceContext_SetPredication(context, predicate, TRUE);
        refcount = ID3D11Predicate_Release(predicate);
        ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);
        ID3D11DeviceContext_GetPredication(context, &tmp_predicate, NULL);
        ok(tmp_predicate == predicate, "Got predicate %p, expected %p.\n", tmp_predicate, predicate);
        refcount = ID3D11Predicate_Release(tmp_predicate);
        ok(!refcount, "Got refcount %lu, expected 0.\n", refcount);
    }

    ID3D11DeviceContext_Release(context);
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_device_context_state(void)
{
    static const GUID test_guid =
            {0xfdb37466, 0x428f, 0x4edf, {0xa3, 0x7f, 0x9b, 0x1d, 0xf4, 0x88, 0xc5, 0xfc}};
    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    static const float custom_blend_factor[] = {0.1f, 0.2f, 0.3f, 0.4f};
    static const float default_blend_factor[] = {1.0f, 1.0f, 1.0f, 1.0f};
#if 0
    float4 main(float4 pos : POSITION) : POSITION
    {
        return pos;
    }
#endif
    static const DWORD simple_vs[] =
    {
        0x43425844, 0x66689e7c, 0x643f0971, 0xb7f67ff4, 0xabc48688, 0x00000001, 0x000000d4, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x49534f50, 0x4e4f4954, 0xababab00, 0x52444853, 0x00000038, 0x00010040,
        0x0000000e, 0x0300005f, 0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
#if 0
    struct data
    {
        float4 position : SV_Position;
    };

    struct patch_constant_data
    {
        float edges[3] : SV_TessFactor;
        float inside : SV_InsideTessFactor;
    };

    void patch_constant(InputPatch<data, 3> input, out patch_constant_data output)
    {
        output.edges[0] = output.edges[1] = output.edges[2] = 1.0f;
        output.inside = 1.0f;
    }

    [domain("tri")]
    [outputcontrolpoints(3)]
    [partitioning("integer")]
    [outputtopology("triangle_ccw")]
    [patchconstantfunc("patch_constant")]
    data hs_main(InputPatch<data, 3> input, uint i : SV_OutputControlPointID)
    {
        return input[i];
    }

    [domain("tri")]
    void ds_main(patch_constant_data input,
            float3 tess_coord : SV_DomainLocation,
            const OutputPatch<data, 3> patch,
            out data output)
    {
        output.position = tess_coord.x * patch[0].position
                + tess_coord.y * patch[1].position
                + tess_coord.z * patch[2].position;
    }
#endif
    static const DWORD simple_hs[] =
    {
        0x43425844, 0x42b5df25, 0xfd8aa2b1, 0xbe5490cb, 0xb595f8b1, 0x00000001, 0x0000020c, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x47534350, 0x0000008c,
        0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003, 0x00000000, 0x00000e01,
        0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01, 0x00000068, 0x00000002,
        0x0000000d, 0x00000003, 0x00000002, 0x00000e01, 0x00000076, 0x00000000, 0x0000000e, 0x00000003,
        0x00000003, 0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469,
        0x46737365, 0x6f746361, 0xabab0072, 0x58454853, 0x000000d8, 0x00030050, 0x00000036, 0x01000071,
        0x01001893, 0x01001894, 0x01001095, 0x01000896, 0x01002097, 0x0100086a, 0x01000073, 0x02000099,
        0x00000003, 0x0200005f, 0x00017000, 0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x04000067,
        0x00102012, 0x00000001, 0x00000012, 0x04000067, 0x00102012, 0x00000002, 0x00000013, 0x02000068,
        0x00000001, 0x0400005b, 0x00102012, 0x00000000, 0x00000003, 0x04000036, 0x00100012, 0x00000000,
        0x0001700a, 0x06000036, 0x00902012, 0x0010000a, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e,
        0x01000073, 0x04000067, 0x00102012, 0x00000003, 0x00000014, 0x05000036, 0x00102012, 0x00000003,
        0x00004001, 0x3f800000, 0x0100003e,
    };
    static const DWORD simple_ds[] =
    {
        0x43425844, 0xb7e35b82, 0x1b930ff2, 0x48d3a0f2, 0x375219ed, 0x00000001, 0x000001e0, 0x00000004,
        0x00000030, 0x00000064, 0x000000f8, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d,
        0x00000003, 0x00000000, 0x00000001, 0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001,
        0x00000001, 0x00000068, 0x00000002, 0x0000000d, 0x00000003, 0x00000002, 0x00000001, 0x00000076,
        0x00000000, 0x0000000e, 0x00000003, 0x00000003, 0x00000001, 0x545f5653, 0x46737365, 0x6f746361,
        0x56530072, 0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x4e47534f, 0x0000002c,
        0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x58454853, 0x000000ac, 0x00040050, 0x0000002b, 0x01001893,
        0x01001095, 0x0100086a, 0x0200005f, 0x0001c072, 0x0400005f, 0x002190f2, 0x00000003, 0x00000000,
        0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x02000068, 0x00000001, 0x07000038, 0x001000f2,
        0x00000000, 0x0001c556, 0x00219e46, 0x00000001, 0x00000000, 0x09000032, 0x001000f2, 0x00000000,
        0x0001c006, 0x00219e46, 0x00000000, 0x00000000, 0x00100e46, 0x00000000, 0x09000032, 0x001020f2,
        0x00000000, 0x0001caa6, 0x00219e46, 0x00000002, 0x00000000, 0x00100e46, 0x00000000, 0x0100003e,
    };
#if 0
    struct gs_out
    {
        float4 pos : SV_POSITION;
    };

    [maxvertexcount(4)]
    void main(point float4 vin[1] : POSITION, inout TriangleStream<gs_out> vout)
    {
        float offset = 0.1 * vin[0].w;
        gs_out v;

        v.pos = float4(vin[0].x - offset, vin[0].y - offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x - offset, vin[0].y + offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x + offset, vin[0].y - offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x + offset, vin[0].y + offset, vin[0].z, vin[0].w);
        vout.Append(v);
    }
#endif
    static const DWORD simple_gs[] =
    {
        0x43425844, 0x000ee786, 0xc624c269, 0x885a5cbe, 0x444b3b1f, 0x00000001, 0x0000023c, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x000001a0, 0x00020040,
        0x00000068, 0x0400005f, 0x002010f2, 0x00000001, 0x00000000, 0x02000068, 0x00000001, 0x0100085d,
        0x0100285c, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x0200005e, 0x00000004, 0x0f000032,
        0x00100032, 0x00000000, 0x80201ff6, 0x00000041, 0x00000000, 0x00000000, 0x00004002, 0x3dcccccd,
        0x3dcccccd, 0x00000000, 0x00000000, 0x00201046, 0x00000000, 0x00000000, 0x05000036, 0x00102032,
        0x00000000, 0x00100046, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000,
        0x00000000, 0x01000013, 0x05000036, 0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x0e000032,
        0x00100052, 0x00000000, 0x00201ff6, 0x00000000, 0x00000000, 0x00004002, 0x3dcccccd, 0x00000000,
        0x3dcccccd, 0x00000000, 0x00201106, 0x00000000, 0x00000000, 0x05000036, 0x00102022, 0x00000000,
        0x0010002a, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000, 0x00000000,
        0x01000013, 0x05000036, 0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x05000036, 0x00102022,
        0x00000000, 0x0010001a, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000,
        0x00000000, 0x01000013, 0x05000036, 0x00102032, 0x00000000, 0x00100086, 0x00000000, 0x06000036,
        0x001020c2, 0x00000000, 0x00201ea6, 0x00000000, 0x00000000, 0x01000013, 0x0100003e,
    };
#if 0
    float4 main(float4 color : COLOR) : SV_TARGET
    {
        return color;
    }
#endif
    static const DWORD simple_ps[] =
    {
        0x43425844, 0x08c2b568, 0x17d33120, 0xb7d82948, 0x13a570fb, 0x00000001, 0x000000d0, 0x00000003,
        0x0000002c, 0x0000005c, 0x00000090, 0x4e475349, 0x00000028, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x4f4c4f43, 0xabab0052, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000038, 0x00000040, 0x0000000e,
        0x03001062, 0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
#if 0
    [numthreads(1, 1, 1)]
    void main() { }
#endif
    static const DWORD simple_cs[] =
    {
        0x43425844, 0x1acc3ad0, 0x71c7b057, 0xc72c4306, 0xf432cb57, 0x00000001, 0x00000074, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000020, 0x00050050, 0x00000008, 0x0100086a,
        0x0400009b, 0x00000001, 0x00000001, 0x00000001, 0x0100003e,
    };
    static const struct vec4 constant = {1.257f, 1.885f, 2.513f, 3.770f};

    ID3DDeviceContextState *context_state, *previous_context_state, *tmp_context_state, *context_state2;
    UINT ib_offset, vb_offset, vb_stride, so_offset, offset, stride, sample_mask, stencil_ref, count;
    ID3D11Buffer *cb, *srvb, *uavb, *ib, *vb, *sob, *tmp_cb, *tmp_ib, *tmp_vb, *tmp_sob;
    D3D_FEATURE_LEVEL feature_level, selected_feature_level;
    ID3D11UnorderedAccessView *tmp_uav, *uav, *ps_uav;
    ID3D11Device *d3d11_device, *d3d11_device2;
    ID3D11SamplerState *sampler, *tmp_sampler;
    D3D11_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D11_SHADER_RESOURCE_VIEW_DESC srv_desc;
    ID3D11DeviceContext1 *context, *context2;
    ID3D11ShaderResourceView *tmp_srv, *srv;
    D3D11_DEVICE_CONTEXT_TYPE context_type;
    ID3D11DepthStencilState *tmp_dss, *dss;
    ID3D11RenderTargetView *tmp_rtv, *rtv;
    ID3D11DepthStencilView *tmp_dsv, *dsv;
    ID3D11VertexShader *tmp_vs, *vs, *vs2;
    ID3D11RasterizerState *tmp_rs, *rs;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11GeometryShader *tmp_gs, *gs;
    enum D3D_PRIMITIVE_TOPOLOGY topo;
    ID3D11ComputeShader *tmp_cs, *cs;
    D3D11_DEPTH_STENCIL_DESC ds_desc;
    ID3D11Predicate *tmp_pred, *pred;
    ID3D11DomainShader *tmp_ds, *ds;
    D3D11_SAMPLER_DESC sampler_desc;
    D3D11_QUERY_DESC predicate_desc;
    ID3D11Device1 *device, *device2;
    ID3D11InputLayout *il, *tmp_il;
    ID3D11PixelShader *tmp_ps, *ps;
    D3D11_RASTERIZER_DESC rs_desc;
    ID3D11BlendState *tmp_bs, *bs;
    ID3D11HullShader *tmp_hs, *hs;
    D3D11_VIEWPORT tmp_vp[2], vp;
    D3D11_RECT tmp_rect[2], rect;
    D3D11_BLEND_DESC blend_desc;
    ID3D11Texture2D *texture;
    enum DXGI_FORMAT format;
    float blend_factor[4];
    BOOL pred_value;
    ULONG refcount;
    UINT data_size;
    char data[64];
    HRESULT hr;

    if (!(d3d11_device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_QueryInterface(d3d11_device, &IID_ID3D11Device1, (void **)&device);
    ID3D11Device_Release(d3d11_device);
    if (FAILED(hr))
    {
        skip("ID3D11Device1 is not available.\n");
        return;
    }

    check_interface(device, &IID_ID3D10Device, FALSE, FALSE);
    check_interface(device, &IID_ID3D10Device1, FALSE, FALSE);

    feature_level = ID3D11Device1_GetFeatureLevel(device);
    context = NULL;
    ID3D11Device1_GetImmediateContext1(device, &context);
    ok(!!context, "Failed to get immediate context.\n");

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_ALWAYS;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 1.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 1.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = 16.0f;
    hr = ID3D11Device1_CreateSamplerState(device, &sampler_desc, &sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    feature_level = min(feature_level, D3D_FEATURE_LEVEL_11_1);

    hr = ID3D11Device1_CreateDeviceContextState(device, 0, &feature_level,
            1, D3D11_SDK_VERSION, &IID_ID3D11Device1, NULL, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);

    selected_feature_level = 0xc0de0000;
    hr = ID3D11Device1_CreateDeviceContextState(device, 0, &feature_level, 1,
            D3D11_SDK_VERSION, &IID_ID3D11Device1, &selected_feature_level, NULL);
    ok(hr == S_FALSE, "Got unexpected hr %#lx.\n", hr);
    ok(selected_feature_level == feature_level, "Got unexpected feature level %#x, expected %#x.\n",
            selected_feature_level, feature_level);

    selected_feature_level = 0xc0de0000;
    context_state = (void *)0xc0de0001;
    hr = ID3D11Device1_CreateDeviceContextState(device, 0, &feature_level, 0,
            D3D11_SDK_VERSION, &IID_ID3D11Device1, &selected_feature_level, &context_state);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!selected_feature_level, "Got unexpected feature level %#x.\n", selected_feature_level);
    ok(!context_state, "Got unexpected context state %p.\n", context_state);

    hr = ID3D11Device1_CreateDeviceContextState(device, 0, &feature_level,
            0, D3D11_SDK_VERSION, &IID_ID3D11Device1, NULL, NULL);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device1_CreateDeviceContextState(device, 0, NULL,
            0, D3D11_SDK_VERSION, &IID_ID3D11Device1, NULL, NULL);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device1_CreateDeviceContextState(device, 0, &feature_level,
            1, D3D11_SDK_VERSION, &IID_ID3D11Device1, NULL, &context_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(context_state);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    context_type = ID3D11DeviceContext1_GetType(context);
    ok(context_type == D3D11_DEVICE_CONTEXT_IMMEDIATE, "Unexpected context type %u.\n", context_type);

    check_interface(device, &IID_ID3D10Device, TRUE, FALSE);
    check_interface(device, &IID_ID3D10Device1, TRUE, FALSE);
    check_interface(device, &IID_ID3D11Device, TRUE, FALSE);
    check_interface(device, &IID_ID3D11Device1, TRUE, FALSE);

    cb = create_buffer((ID3D11Device *)device, D3D11_BIND_CONSTANT_BUFFER, sizeof(constant), NULL);
    srvb = create_buffer((ID3D11Device *)device, D3D11_BIND_SHADER_RESOURCE, 1024, NULL);
    uavb = create_buffer((ID3D11Device *)device, D3D11_BIND_UNORDERED_ACCESS, 1024, NULL);
    ib = create_buffer((ID3D11Device *)device, D3D11_BIND_INDEX_BUFFER, 1024, NULL);
    vb = create_buffer((ID3D11Device *)device, D3D11_BIND_VERTEX_BUFFER, 1024, NULL);
    sob = create_buffer((ID3D11Device *)device, D3D11_BIND_STREAM_OUTPUT, 1024, NULL);

    hr = ID3D11Device1_CreateVertexShader(device, simple_vs, sizeof(simple_vs), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device1_CreateGeometryShader(device, simple_gs, sizeof(simple_gs), NULL, &gs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device1_CreatePixelShader(device, simple_ps, sizeof(simple_ps), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    if (feature_level < D3D_FEATURE_LEVEL_11_0) hs = NULL;
    else
    {
        hr = ID3D11Device1_CreateHullShader(device, simple_hs, sizeof(simple_hs), NULL, &hs);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    }

    if (feature_level < D3D_FEATURE_LEVEL_11_0) ds = NULL;
    else
    {
        hr = ID3D11Device1_CreateDomainShader(device, simple_ds, sizeof(simple_ds), NULL, &ds);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    }

    if (feature_level < D3D_FEATURE_LEVEL_11_0) cs = NULL;
    else
    {
        hr = ID3D11Device1_CreateComputeShader(device, simple_cs, sizeof(simple_cs), NULL, &cs);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    }

    srv_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    srv_desc.ViewDimension = D3D11_SRV_DIMENSION_BUFFER;
    U(srv_desc).Buffer.ElementOffset = 0;
    U(srv_desc).Buffer.ElementWidth = 64;
    hr = ID3D11Device1_CreateShaderResourceView(device, (ID3D11Resource *)srvb, &srv_desc, &srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Buffer_Release(srvb);

    uav_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 4;
    U(uav_desc).Buffer.Flags = 0;
    hr = ID3D11Device1_CreateUnorderedAccessView(device, (ID3D11Resource *)uavb, &uav_desc, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Buffer_Release(uavb);

    uavb = create_buffer((ID3D11Device *)device, D3D11_BIND_UNORDERED_ACCESS, 1024, NULL);
    hr = ID3D11Device1_CreateUnorderedAccessView(device, (ID3D11Resource *)uavb, &uav_desc, &ps_uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Buffer_Release(uavb);

    hr = ID3D11Device1_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            simple_vs, sizeof(simple_vs), &il);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ib_offset = 16;
    vb_offset = 16;
    vb_stride = 16;
    so_offset = 16;

    texture_desc.Width = 512;
    texture_desc.Height = 512;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device1_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device1_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Texture2D_Release(texture);

    texture_desc.Format = DXGI_FORMAT_D24_UNORM_S8_UINT;
    texture_desc.BindFlags = D3D11_BIND_DEPTH_STENCIL;
    hr = ID3D11Device1_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device1_CreateDepthStencilView(device, (ID3D11Resource *)texture, NULL, &dsv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11Texture2D_Release(texture);

    memset(&blend_desc, 0, sizeof(blend_desc));
    blend_desc.RenderTarget[0].BlendEnable = TRUE;
    blend_desc.RenderTarget[0].SrcBlend = D3D11_BLEND_SRC_ALPHA;
    blend_desc.RenderTarget[0].DestBlend = D3D11_BLEND_INV_SRC_ALPHA;
    blend_desc.RenderTarget[0].BlendOp = D3D11_BLEND_OP_ADD;
    blend_desc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_SRC_ALPHA;
    blend_desc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_INV_SRC_ALPHA;
    blend_desc.RenderTarget[0].BlendOpAlpha = D3D11_BLEND_OP_ADD;
    blend_desc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;
    hr = ID3D11Device1_CreateBlendState(device, &blend_desc, &bs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ds_desc.DepthEnable = TRUE;
    ds_desc.DepthWriteMask = D3D11_DEPTH_WRITE_MASK_ALL;
    ds_desc.DepthFunc = D3D11_COMPARISON_LESS;
    ds_desc.StencilEnable = FALSE;
    ds_desc.StencilReadMask = D3D11_DEFAULT_STENCIL_READ_MASK;
    ds_desc.StencilWriteMask = D3D11_DEFAULT_STENCIL_WRITE_MASK;
    ds_desc.FrontFace.StencilFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilDepthFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilPassOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilFunc = D3D11_COMPARISON_ALWAYS;
    ds_desc.BackFace.StencilFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilDepthFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilPassOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilFunc = D3D11_COMPARISON_ALWAYS;
    hr = ID3D11Device1_CreateDepthStencilState(device, &ds_desc, &dss);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    rs_desc.FillMode = D3D11_FILL_SOLID;
    rs_desc.CullMode = D3D11_CULL_BACK;
    rs_desc.FrontCounterClockwise = FALSE;
    rs_desc.DepthBias = 0;
    rs_desc.DepthBiasClamp = 0.0f;
    rs_desc.SlopeScaledDepthBias = 0.0f;
    rs_desc.DepthClipEnable = TRUE;
    rs_desc.ScissorEnable = TRUE;
    rs_desc.MultisampleEnable = FALSE;
    rs_desc.AntialiasedLineEnable = FALSE;
    hr = ID3D11Device1_CreateRasterizerState(device, &rs_desc, &rs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    SetRect(&rect, 0, 0, 1, 2);
    vp.TopLeftX = 0;
    vp.TopLeftY = 0;
    vp.Width = 3;
    vp.Height = 4;
    vp.MinDepth = 0.f;
    vp.MaxDepth = 0.01f;

    predicate_desc.Query = D3D11_QUERY_OCCLUSION_PREDICATE;
    predicate_desc.MiscFlags = 0;
    ID3D11Device1_CreatePredicate(device, &predicate_desc, &pred);

    ID3D11DeviceContext1_VSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_VSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext1_VSSetShaderResources(context, 0, 1, &srv);
    refcount = get_refcount(vs);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    ID3D11DeviceContext1_GSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_GSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_GSSetShader(context, gs, NULL, 0);
    ID3D11DeviceContext1_GSSetShaderResources(context, 0, 1, &srv);

    ID3D11DeviceContext1_PSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_PSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext1_PSSetShaderResources(context, 0, 1, &srv);

    ID3D11DeviceContext1_HSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_HSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_HSSetShader(context, hs, NULL, 0);
    ID3D11DeviceContext1_HSSetShaderResources(context, 0, 1, &srv);

    ID3D11DeviceContext1_DSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_DSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_DSSetShader(context, ds, NULL, 0);
    ID3D11DeviceContext1_DSSetShaderResources(context, 0, 1, &srv);

    ID3D11DeviceContext1_CSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_CSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_CSSetShader(context, cs, NULL, 0);
    ID3D11DeviceContext1_CSSetShaderResources(context, 0, 1, &srv);
    ID3D11DeviceContext1_CSSetUnorderedAccessViews(context, 0, 1, &uav, NULL);

    ID3D11DeviceContext1_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    ID3D11DeviceContext1_IASetInputLayout(context, il);
    ID3D11DeviceContext1_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, ib_offset);
    ID3D11DeviceContext1_IASetVertexBuffers(context, 0, 1, &vb, &vb_stride, &vb_offset);

    ID3D11DeviceContext1_OMSetBlendState(context, bs, custom_blend_factor, 0xff00ff00);
    ID3D11DeviceContext1_OMSetDepthStencilState(context, dss, 3);
    ID3D11DeviceContext1_OMSetRenderTargetsAndUnorderedAccessViews(context, 1, &rtv, dsv, 1, 1, &ps_uav, NULL);

    ID3D11DeviceContext1_RSSetScissorRects(context, 1, &rect);
    ID3D11DeviceContext1_RSSetViewports(context, 1, &vp);
    ID3D11DeviceContext1_RSSetState(context, rs);

    ID3D11DeviceContext1_SOSetTargets(context, 1, &sob, &so_offset);
    ID3D11DeviceContext1_SetPredication(context, pred, TRUE);

    previous_context_state = (ID3DDeviceContextState *)0xdeadbeef;
    ID3D11DeviceContext1_SwapDeviceContextState(context, NULL, &previous_context_state);
    ok(previous_context_state == NULL, "Got unexpected state pointer.\n");
    previous_context_state = NULL;
    ID3D11DeviceContext1_SwapDeviceContextState(context, context_state, &previous_context_state);
    ok(previous_context_state != NULL, "Failed to get previous context state\n");
    refcount = get_refcount(vs);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    hr = ID3DDeviceContextState_SetPrivateData(context_state, &test_guid, sizeof(constant), &constant);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = ID3DDeviceContextState_Release(context_state);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
    ID3D11DeviceContext1_SwapDeviceContextState(context, previous_context_state, &context_state);
    data_size = sizeof(data);
    memset(data, 0xa5, sizeof(data));
    hr = ID3DDeviceContextState_GetPrivateData(context_state, &test_guid, &data_size, data);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(data_size == sizeof(constant), "Got private data size %#x, expected %#Ix.\n", data_size, sizeof(constant));
    ok(!memcmp(data, &constant, sizeof(constant)), "Got unexpected private data.\n");
    ID3D11DeviceContext1_SwapDeviceContextState(context, context_state, NULL);

    context_type = ID3D11DeviceContext1_GetType(context);
    ok(context_type == D3D11_DEVICE_CONTEXT_IMMEDIATE, "Unexpected context type %u.\n", context_type);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShader(context, &tmp_vs, NULL, NULL);
    ok(!tmp_vs, "Got unexpected shader %p.\n", tmp_vs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    tmp_gs = (ID3D11GeometryShader *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetShader(context, &tmp_gs, NULL, NULL);
    ok(!tmp_gs, "Got unexpected shader %p.\n", tmp_gs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    tmp_ps = (ID3D11PixelShader *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetShader(context, &tmp_ps, NULL, NULL);
    ok(!tmp_ps, "Got unexpected shader %p.\n", tmp_ps);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    tmp_hs = (ID3D11HullShader *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetShader(context, &tmp_hs, NULL, NULL);
    if (hs) ok(!tmp_hs, "Got unexpected shader %p.\n", tmp_hs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    tmp_ds = (ID3D11DomainShader *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetShader(context, &tmp_ds, NULL, NULL);
    if (ds) ok(!tmp_ds, "Got unexpected shader %p.\n", tmp_ds);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    tmp_cs = (ID3D11ComputeShader *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetShader(context, &tmp_cs, NULL, NULL);
    if (cs) ok(!tmp_cs, "Got unexpected shader %p.\n", tmp_cs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);
    tmp_uav = (ID3D11UnorderedAccessView *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetUnorderedAccessViews(context, 0, 1, &tmp_uav);
    ok(!tmp_uav, "Got unexpected uav %p.\n", tmp_uav);

    topo = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetPrimitiveTopology(context, &topo);
    ok(topo == D3D11_PRIMITIVE_TOPOLOGY_UNDEFINED, "Got unexpected topology %#x.\n", topo);
    tmp_il = (ID3D11InputLayout *)0xdeadbeef;
    ID3D11DeviceContext1_IAGetInputLayout(context, &tmp_il);
    ok(!tmp_il, "Got unexpected input layout %p.\n", tmp_il);
    tmp_ib = (ID3D11Buffer *)0xdeadbeef;
    format = 0xdeadbeef;
    offset = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetIndexBuffer(context, &tmp_ib, &format, &offset);
    ok(!tmp_ib, "Got unexpected input buffer %p.\n", tmp_ib);
    ok(format == DXGI_FORMAT_UNKNOWN, "Got unexpected input buffer format %#x.\n", format);
    ok(offset == 0, "Got unexpected input buffer offset %#x.\n", offset);
    tmp_vb = (ID3D11Buffer *)0xdeadbeef;
    stride = 0xdeadbeef;
    offset = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetVertexBuffers(context, 0, 1, &tmp_vb, &stride, &offset);
    ok(!tmp_vb, "Got unexpected vertex buffer %p.\n", tmp_vb);
    ok(stride == 0, "Got unexpected vertex buffer stride %#x.\n", stride);
    ok(offset == 0, "Got unexpected vertex buffer offset %#x.\n", offset);

    tmp_rtv = (ID3D11RenderTargetView *)0xdeadbeef;
    tmp_dsv = (ID3D11DepthStencilView *)0xdeadbeef;
    tmp_uav = (ID3D11UnorderedAccessView *)0xdeadbeef;
    ID3D11DeviceContext1_OMGetRenderTargetsAndUnorderedAccessViews(context, 1, &tmp_rtv, &tmp_dsv, 1, 1, &tmp_uav);
    ok(!tmp_rtv, "Got unexpected rendertarget view %p.\n", tmp_rtv);
    ok(!tmp_dsv, "Got unexpected depth/stencil view %p.\n", tmp_dsv);
    ok(!tmp_uav, "Got unexpected unordered access view %p.\n", tmp_uav);
    tmp_bs = (ID3D11BlendState *)0xdeadbeef;
    memset(blend_factor, 0xcd, sizeof(blend_factor));
    sample_mask = 0xdeadbeef;
    ID3D11DeviceContext1_OMGetBlendState(context, &tmp_bs, blend_factor, &sample_mask);
    ok(!tmp_bs, "Got unexpected blend state %p.\n", tmp_bs);
    ok(!memcmp(blend_factor, default_blend_factor, sizeof(blend_factor)),
            "Got unexpected blend factor %f,%f,%f,%f.\n",
            blend_factor[0], blend_factor[1], blend_factor[2], blend_factor[3]);
    ok(sample_mask == ~0, "Got unexpected sample mask %#x.\n", sample_mask);
    tmp_dss = (ID3D11DepthStencilState *)0xdeadbeef;
    stencil_ref = 0xdeadbeef;
    ID3D11DeviceContext1_OMGetDepthStencilState(context, &tmp_dss, &stencil_ref);
    ok(!tmp_dss, "Got unexpected depth/stencil state %p.\n", tmp_dss);
    ok(stencil_ref == 0, "Got unexpected stencil ref %#x.\n", stencil_ref);

    tmp_rs = (ID3D11RasterizerState *)0xdeadbeef;
    ID3D11DeviceContext1_RSGetState(context, &tmp_rs);
    ok(!tmp_rs, "Got unexpected rasterizer state %p.\n", tmp_rs);
    memset(tmp_vp, 0xa5, sizeof(tmp_vp));
    count = 2;
    ID3D11DeviceContext1_RSGetViewports(context, &count, tmp_vp);
    ok(count == 0, "Got unexpected viewport count %u.\n", count);
    memset(tmp_rect, 0xa5, sizeof(tmp_rect));
    count = 2;
    ID3D11DeviceContext1_RSGetScissorRects(context, &count, tmp_rect);
    ok(count == 0, "Got unexpected scissor rect count %u.\n", count);

    tmp_sob = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_SOGetTargets(context, 1, &tmp_sob);
    ok(!tmp_sob, "Got unexpected stream output buffer %p.\n", tmp_sob);

    tmp_pred = (ID3D11Predicate *)0xdeadbeef;
    pred_value = 0xdeadbeef;
    ID3D11DeviceContext1_GetPredication(context, &tmp_pred, &pred_value);
    ok(!tmp_pred, "Got unexpected predicate %p.\n", tmp_pred);
    ok(!pred_value, "Got unexpected predicate value %d.\n", pred_value);

    /* updating the device context should also update the device context state */
    hr = ID3D11Device1_CreateVertexShader(device, simple_vs, sizeof(simple_vs), NULL, &vs2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext1_VSSetShader(context, vs2, NULL, 0);
    ID3D11DeviceContext1_SwapDeviceContextState(context, context_state, &tmp_context_state);
    refcount = ID3DDeviceContextState_Release(tmp_context_state);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
    ok(tmp_context_state == context_state, "Got unexpected state pointer.\n");
    tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShader(context, &tmp_vs, NULL, NULL);
    ok(tmp_vs == vs2, "Got shader %p, expected %p.\n", tmp_vs, vs2);
    refcount = ID3D11VertexShader_Release(tmp_vs);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    /* context states may be used with other devices instances too
     *
     * Or at least, no error is returned right away. Using objects like shaders
     * from other device instances causes a segfault later on Nvidia Windows
     * drivers. This suggests the feature tested below is more a bug than a
     * feature.
     *
     * The tests below suggest that a ContextState object stores its own state
     * for every device it is used with. This isn't entirely true, e.g. the
     * primitive topology can be transferred between devices, but will cause
     * odd refcounting behavior afterwards (IAGetPrimitiveTopology will leak 54
     * references on the context's device for example). */
    if (0)
    {
        d3d11_device2 = create_device(NULL);
        ok(!!d3d11_device2, "Failed to create device.\n");
        hr = ID3D11Device_QueryInterface(d3d11_device2, &IID_ID3D11Device1, (void **)&device2);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
        ID3D11Device_Release(d3d11_device2);
        ID3D11Device1_GetImmediateContext1(device2, &context2);
        ok(!!context2, "Failed to get immediate context.\n");

        /* but they track a distinct state on each context */
        ID3D11DeviceContext1_SwapDeviceContextState(context2, context_state, &tmp_context_state);
        ok(!!tmp_context_state, "Failed to get context state.\n");
        tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
        ID3D11DeviceContext1_VSGetShader(context2, &tmp_vs, NULL, NULL);
        ok(!tmp_vs, "Got unexpected shader %p.\n", tmp_vs);

        /* updating context2 vertex shader doesn't update other contexts using the same state */
        ID3D11DeviceContext1_VSSetShader(context2, vs, NULL, 0);
        ID3D11DeviceContext1_VSGetShader(context, &tmp_vs, NULL, NULL);
        ok(tmp_vs == vs2, "Got shader %p, expected %p.\n", tmp_vs, vs2);
        refcount = ID3D11VertexShader_Release(tmp_vs);
        ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

        ID3D11DeviceContext1_SwapDeviceContextState(context2, tmp_context_state, &context_state2);
        refcount = ID3DDeviceContextState_Release(tmp_context_state);
        ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
        refcount = ID3DDeviceContextState_Release(context_state2);
        ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
        ok(context_state2 == context_state, "Got unexpected state pointer.\n");

        /* swapping the default state on context2 effectively clears the vertex shader */
        tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
        ID3D11DeviceContext1_VSGetShader(context2, &tmp_vs, NULL, NULL);
        ok(!tmp_vs, "Got unexpected shader %p.\n", tmp_vs);

        ID3D11DeviceContext1_SwapDeviceContextState(context2, context_state, &tmp_context_state);
        ok(!!tmp_context_state, "Failed to get context state.\n");
        refcount = ID3DDeviceContextState_Release(tmp_context_state);
        ok(!refcount, "Got unexpected refcount %lu.\n", refcount);

        /* clearing the vertex shader on context doesn't have side effect on context2 */
        ID3D11DeviceContext1_VSSetShader(context, NULL, NULL, 0);
        refcount = get_refcount(vs2);
        ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
        tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
        ID3D11DeviceContext1_VSGetShader(context2, &tmp_vs, NULL, NULL);
        ok(tmp_vs == vs, "Got shader %p, expected %p.\n", tmp_vs, vs);
        refcount = ID3D11VertexShader_Release(tmp_vs);
        ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

        /* even after swapping it again */
        ID3D11DeviceContext1_SwapDeviceContextState(context2, context_state, NULL);
        tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
        ID3D11DeviceContext1_VSGetShader(context2, &tmp_vs, NULL, NULL);
        ok(tmp_vs == vs, "Got shader %p, expected %p.\n", tmp_vs, vs);
        refcount = ID3D11VertexShader_Release(tmp_vs);
        ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

        /* swapping the initial state on context2 doesn't have side effect on context either */
        ID3D11DeviceContext1_SwapDeviceContextState(context2, previous_context_state, NULL);
        tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
        ID3D11DeviceContext1_VSGetShader(context, &tmp_vs, NULL, NULL);
        ok(!tmp_vs, "Got unexpected shader %p.\n", tmp_vs);

        refcount = ID3D11DeviceContext1_Release(context2);
        ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
        refcount = ID3D11Device1_Release(device2);
        ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
    }

    ID3D11DeviceContext1_SwapDeviceContextState(context, previous_context_state, &tmp_context_state);
    refcount = ID3DDeviceContextState_Release(previous_context_state);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3DDeviceContextState_Release(tmp_context_state);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3DDeviceContextState_Release(context_state);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
    ok(tmp_context_state == context_state, "Got unexpected state pointer.\n");
    refcount = get_refcount(vs);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    /* ID3DDeviceContextState retains the previous state. */

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_ps = (ID3D11PixelShader *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetShader(context, &tmp_ps, NULL, NULL);
    ok(tmp_ps == ps, "Got shader %p, expected %p.\n", tmp_ps, ps);
    ID3D11PixelShader_Release(tmp_ps);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_cs = (ID3D11ComputeShader *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetShader(context, &tmp_cs, NULL, NULL);
    ok(tmp_cs == cs, "Got shader %p, expected %p.\n", tmp_cs, cs);
    if (cs) ID3D11ComputeShader_Release(tmp_cs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);
    tmp_uav = (ID3D11UnorderedAccessView *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetUnorderedAccessViews(context, 0, 1, &tmp_uav);
    ok(tmp_uav == uav, "Got uav %p, expected %p.\n", tmp_uav, uav);
    ID3D11UnorderedAccessView_Release(tmp_uav);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_ds = (ID3D11DomainShader *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetShader(context, &tmp_ds, NULL, NULL);
    ok(tmp_ds == ds, "Got shader %p, expected %p.\n", tmp_ds, ds);
    if (ds) ID3D11DomainShader_Release(tmp_ds);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_gs = (ID3D11GeometryShader *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetShader(context, &tmp_gs, NULL, NULL);
    ok(tmp_gs == gs, "Got shader %p, expected %p.\n", tmp_gs, gs);
    ID3D11GeometryShader_Release(tmp_gs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_hs = (ID3D11HullShader *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetShader(context, &tmp_hs, NULL, NULL);
    ok(tmp_hs == hs, "Got shader %p, expected %p.\n", tmp_hs, hs);
    if (hs) ID3D11HullShader_Release(tmp_hs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShader(context, &tmp_vs, NULL, NULL);
    ok(tmp_vs == vs, "Got shader %p, expected %p.\n", tmp_vs, vs);
    ID3D11VertexShader_Release(tmp_vs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    topo = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetPrimitiveTopology(context, &topo);
    ok(topo == D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP, "Got topology %#x, expected %#x.\n", topo, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    tmp_il = (ID3D11InputLayout *)0xdeadbeef;
    ID3D11DeviceContext1_IAGetInputLayout(context, &tmp_il);
    ok(tmp_il == il, "Got input layout %p, expected %p.\n", tmp_il, il);
    ID3D11InputLayout_Release(tmp_il);
    tmp_ib = (ID3D11Buffer *)0xdeadbeef;
    format = 0xdeadbeef;
    offset = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetIndexBuffer(context, &tmp_ib, &format, &offset);
    ok(tmp_ib == ib, "Got input buffer %p, expected %p.\n", tmp_ib, ib);
    ID3D11Buffer_Release(tmp_ib);
    ok(format == DXGI_FORMAT_R32_UINT, "Got input buffer format %#x, expected %#x.\n", format, DXGI_FORMAT_R32_UINT);
    ok(offset == 16, "Got input buffer offset %#x, expected 16.\n", offset);
    tmp_vb = (ID3D11Buffer *)0xdeadbeef;
    stride = 0xdeadbeef;
    offset = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetVertexBuffers(context, 0, 1, &tmp_vb, &stride, &offset);
    ok(tmp_vb == vb, "Got vertex buffer %p, expected %p.\n", tmp_vb, vb);
    ID3D11Buffer_Release(tmp_vb);
    ok(stride == 16, "Got vertex buffer stride %#x, expected 16.\n", stride);
    ok(offset == 16, "Got vertex buffer offset %#x, expected 16.\n", offset);

    tmp_rtv = (ID3D11RenderTargetView *)0xdeadbeef;
    tmp_dsv = (ID3D11DepthStencilView *)0xdeadbeef;
    tmp_uav = (ID3D11UnorderedAccessView *)0xdeadbeef;
    ID3D11DeviceContext1_OMGetRenderTargetsAndUnorderedAccessViews(context, 1, &tmp_rtv, &tmp_dsv, 1, 1, &tmp_uav);
    ok(tmp_rtv == rtv, "Got rendertarget view %p, expected %p.\n", tmp_rtv, rtv);
    ID3D11RenderTargetView_Release(tmp_rtv);
    ok(tmp_dsv == dsv, "Got depth/stencil view %p, expected %p.\n", tmp_dsv, dsv);
    ID3D11DepthStencilView_Release(tmp_dsv);
    ok(tmp_uav == ps_uav, "Got unordered access view %p, expected %p.\n", tmp_uav, ps_uav);
    ID3D11UnorderedAccessView_Release(tmp_uav);
    tmp_bs = (ID3D11BlendState *)0xdeadbeef;
    memset(blend_factor, 0xcd, sizeof(blend_factor));
    sample_mask = 0xdeadbeef;
    ID3D11DeviceContext1_OMGetBlendState(context, &tmp_bs, blend_factor, &sample_mask);
    ok(tmp_bs == bs, "Got blend state %p, expected %p.\n", tmp_bs, bs);
    ID3D11BlendState_Release(tmp_bs);
    ok(!memcmp(blend_factor, custom_blend_factor, sizeof(blend_factor)),
            "Got blend factor %f,%f,%f,%f, expected %f,%f,%f,%f.\n",
            blend_factor[0], blend_factor[1], blend_factor[2], blend_factor[3],
            custom_blend_factor[0], custom_blend_factor[1], custom_blend_factor[2], custom_blend_factor[3]);
    ok(sample_mask == 0xff00ff00, "Got sample mask %#x, expected %#x.\n", sample_mask, 0xff00ff00);
    tmp_dss = (ID3D11DepthStencilState *)0xdeadbeef;
    stencil_ref = 0xdeadbeef;
    ID3D11DeviceContext1_OMGetDepthStencilState(context, &tmp_dss, &stencil_ref);
    ok(tmp_dss == dss, "Got depth/stencil state %p, expected %p.\n", tmp_dss, dss);
    ID3D11DepthStencilState_Release(tmp_dss);
    ok(stencil_ref == 3, "Got stencil ref %#x, expected 3.\n", stencil_ref);

    tmp_rs = (ID3D11RasterizerState *)0xdeadbeef;
    ID3D11DeviceContext1_RSGetState(context, &tmp_rs);
    ok(tmp_rs == rs, "Got unexpected rasterizer state %p.\n", tmp_rs);
    ID3D11RasterizerState_Release(tmp_rs);
    memset(tmp_vp, 0xa5, sizeof(tmp_vp));
    count = 2;
    ID3D11DeviceContext1_RSGetViewports(context, &count, tmp_vp);
    ok(count == 1, "Got viewport count %u, expected 1.\n", count);
    ok(!memcmp(tmp_vp, &vp, sizeof(vp)), "Got viewport %s, expected %s.\n",
            debugstr_viewport(tmp_vp), debugstr_viewport(&vp));
    memset(tmp_rect, 0xa5, sizeof(tmp_rect));
    count = 2;
    ID3D11DeviceContext1_RSGetScissorRects(context, &count, tmp_rect);
    ok(count == 1, "Got scissor rect count %u, expected 1.\n", count);
    ok(!memcmp(tmp_rect, &rect, sizeof(rect)), "Got scissor rect %s, expected %s.\n",
            wine_dbgstr_rect(tmp_rect), wine_dbgstr_rect(&rect));

    tmp_sob = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_SOGetTargets(context, 1, &tmp_sob);
    ok(tmp_sob == sob, "Got stream output buffer %p, expected %p.\n", tmp_sob, sob);
    ID3D11Buffer_Release(tmp_sob);

    tmp_pred = (ID3D11Predicate *)0xdeadbeef;
    pred_value = 0xdeadbeef;
    ID3D11DeviceContext1_GetPredication(context, &tmp_pred, &pred_value);
    ok(tmp_pred == pred, "Got predicate %p, expected %p.\n", tmp_pred, pred);
    ID3D11Predicate_Release(tmp_pred);
    ok(pred_value == TRUE, "Got predicate value %#x, expected TRUE.\n", pred_value);

    feature_level = min(feature_level, D3D_FEATURE_LEVEL_10_1);
    hr = ID3D11Device1_CreateDeviceContextState(device, 0, &feature_level, 1, D3D11_SDK_VERSION,
            &IID_ID3D10Device, NULL, &context_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(context_state);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    context_type = ID3D11DeviceContext1_GetType(context);
    ok(context_type == D3D11_DEVICE_CONTEXT_IMMEDIATE, "Unexpected context type %u.\n", context_type);

    /* Enable ID3D10Device behavior. */
    previous_context_state = NULL;
    ID3D11DeviceContext1_SwapDeviceContextState(context, context_state, &previous_context_state);
    refcount = ID3DDeviceContextState_Release(context_state);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
    ok(previous_context_state != NULL, "Failed to get previous context state\n");

    context_type = ID3D11DeviceContext1_GetType(context);
    ok(context_type == D3D11_DEVICE_CONTEXT_IMMEDIATE, "Unexpected context type %u.\n", context_type);

    ID3D11DeviceContext1_VSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_VSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext1_VSSetShaderResources(context, 0, 1, &srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetConstantBuffers(context, 0, 1, &tmp_cb);
    todo_wine ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    if (tmp_cb && tmp_cb != (ID3D11Buffer *)0xdeadbeef) ID3D11Buffer_Release(tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetSamplers(context, 0, 1, &tmp_sampler);
    todo_wine ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    if (tmp_sampler && tmp_sampler != (ID3D11SamplerState *)0xdeadbeef) ID3D11SamplerState_Release(tmp_sampler);
    tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShader(context, &tmp_vs, NULL, NULL);
    todo_wine ok(!tmp_vs, "Got unexpected shader %p.\n", tmp_vs);
    if (tmp_vs && tmp_vs != (ID3D11VertexShader *)0xdeadbeef) ID3D11VertexShader_Release(tmp_vs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShaderResources(context, 0, 1, &tmp_srv);
    todo_wine ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);
    if (tmp_srv && tmp_srv != (ID3D11ShaderResourceView *)0xdeadbeef) ID3D11ShaderResourceView_Release(tmp_srv);

    ID3D11DeviceContext1_GSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_GSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_GSSetShader(context, gs, NULL, 0);
    ID3D11DeviceContext1_GSSetShaderResources(context, 0, 1, &srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetConstantBuffers(context, 0, 1, &tmp_cb);
    todo_wine ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    if (tmp_cb && tmp_cb != (ID3D11Buffer *)0xdeadbeef) ID3D11Buffer_Release(tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetSamplers(context, 0, 1, &tmp_sampler);
    todo_wine ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    if (tmp_sampler && tmp_sampler != (ID3D11SamplerState *)0xdeadbeef) ID3D11SamplerState_Release(tmp_sampler);
    tmp_gs = (ID3D11GeometryShader *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetShader(context, &tmp_gs, NULL, NULL);
    todo_wine ok(!tmp_gs, "Got unexpected shader %p.\n", tmp_gs);
    if (tmp_gs && tmp_gs != (ID3D11GeometryShader *)0xdeadbeef) ID3D11GeometryShader_Release(tmp_gs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetShaderResources(context, 0, 1, &tmp_srv);
    todo_wine ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);
    if (tmp_srv && tmp_srv != (ID3D11ShaderResourceView *)0xdeadbeef) ID3D11ShaderResourceView_Release(tmp_srv);

    ID3D11DeviceContext1_PSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_PSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext1_PSSetShaderResources(context, 0, 1, &srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetConstantBuffers(context, 0, 1, &tmp_cb);
    todo_wine ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    if (tmp_cb && tmp_cb != (ID3D11Buffer *)0xdeadbeef) ID3D11Buffer_Release(tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetSamplers(context, 0, 1, &tmp_sampler);
    todo_wine ok(tmp_sampler == (ID3D11SamplerState *)0xdeadbeef, "Got unexpected sampler %p.\n", tmp_sampler);
    if (tmp_sampler && tmp_sampler != (ID3D11SamplerState *)0xdeadbeef) ID3D11SamplerState_Release(tmp_sampler);
    tmp_ps = (ID3D11PixelShader *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetShader(context, &tmp_ps, NULL, NULL);
    todo_wine ok(!tmp_ps, "Got unexpected shader %p.\n", tmp_ps);
    if (tmp_ps && tmp_ps != (ID3D11PixelShader *)0xdeadbeef) ID3D11PixelShader_Release(tmp_ps);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetShaderResources(context, 0, 1, &tmp_srv);
    todo_wine ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);
    if (tmp_srv && tmp_srv != (ID3D11ShaderResourceView *)0xdeadbeef) ID3D11ShaderResourceView_Release(tmp_srv);

    ID3D11DeviceContext1_HSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_HSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_HSSetShader(context, hs, NULL, 0);
    ID3D11DeviceContext1_HSSetShaderResources(context, 0, 1, &srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetConstantBuffers(context, 0, 1, &tmp_cb);
    todo_wine ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    if (tmp_cb && tmp_cb != (ID3D11Buffer *)0xdeadbeef) ID3D11Buffer_Release(tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetSamplers(context, 0, 1, &tmp_sampler);
    todo_wine ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    if (tmp_sampler && tmp_sampler != (ID3D11SamplerState *)0xdeadbeef) ID3D11SamplerState_Release(tmp_sampler);
    tmp_hs = (ID3D11HullShader *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetShader(context, &tmp_hs, NULL, NULL);
    if (hs) todo_wine ok(!tmp_hs, "Got unexpected shader %p.\n", tmp_hs);
    if (tmp_hs && tmp_hs != (ID3D11HullShader *)0xdeadbeef) ID3D11HullShader_Release(tmp_hs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetShaderResources(context, 0, 1, &tmp_srv);
    todo_wine ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);
    if (tmp_srv && tmp_srv != (ID3D11ShaderResourceView *)0xdeadbeef) ID3D11ShaderResourceView_Release(tmp_srv);

    ID3D11DeviceContext1_DSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_DSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_DSSetShader(context, ds, NULL, 0);
    ID3D11DeviceContext1_DSSetShaderResources(context, 0, 1, &srv);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetConstantBuffers(context, 0, 1, &tmp_cb);
    todo_wine ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    if (tmp_cb && tmp_cb != (ID3D11Buffer *)0xdeadbeef) ID3D11Buffer_Release(tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetSamplers(context, 0, 1, &tmp_sampler);
    todo_wine ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    if (tmp_sampler && tmp_sampler != (ID3D11SamplerState *)0xdeadbeef) ID3D11SamplerState_Release(tmp_sampler);
    tmp_ds = (ID3D11DomainShader *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetShader(context, &tmp_ds, NULL, NULL);
    if (ds) todo_wine ok(!tmp_ds, "Got unexpected shader %p.\n", tmp_ds);
    if (tmp_ds && tmp_ds != (ID3D11DomainShader *)0xdeadbeef) ID3D11DomainShader_Release(tmp_ds);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetShaderResources(context, 0, 1, &tmp_srv);
    todo_wine ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);
    if (tmp_srv && tmp_srv != (ID3D11ShaderResourceView *)0xdeadbeef) ID3D11ShaderResourceView_Release(tmp_srv);

    ID3D11DeviceContext1_CSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext1_CSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext1_CSSetShader(context, cs, NULL, 0);
    ID3D11DeviceContext1_CSSetShaderResources(context, 0, 1, &srv);
    ID3D11DeviceContext1_CSSetUnorderedAccessViews(context, 0, 1, &uav, NULL);

    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetConstantBuffers(context, 0, 1, &tmp_cb);
    todo_wine ok(!tmp_cb, "Got unexpected buffer %p.\n", tmp_cb);
    if (tmp_cb && tmp_cb != (ID3D11Buffer *)0xdeadbeef) ID3D11Buffer_Release(tmp_cb);
    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetSamplers(context, 0, 1, &tmp_sampler);
    todo_wine ok(!tmp_sampler, "Got unexpected sampler %p.\n", tmp_sampler);
    if (tmp_sampler && tmp_sampler != (ID3D11SamplerState *)0xdeadbeef) ID3D11SamplerState_Release(tmp_sampler);
    tmp_cs = (ID3D11ComputeShader *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetShader(context, &tmp_cs, NULL, NULL);
    if (cs) todo_wine ok(!tmp_cs, "Got unexpected shader %p.\n", tmp_cs);
    if (tmp_cs && tmp_cs != (ID3D11ComputeShader *)0xdeadbeef) ID3D11ComputeShader_Release(tmp_cs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetShaderResources(context, 0, 1, &tmp_srv);
    todo_wine ok(!tmp_srv, "Got unexpected srv %p.\n", tmp_srv);
    if (tmp_srv && tmp_srv != (ID3D11ShaderResourceView *)0xdeadbeef) ID3D11ShaderResourceView_Release(tmp_srv);
    tmp_uav = (ID3D11UnorderedAccessView *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetUnorderedAccessViews(context, 0, 1, &tmp_uav);
    todo_wine ok(!tmp_uav, "Got unexpected uav %p.\n", tmp_uav);
    if (tmp_uav && tmp_uav != (ID3D11UnorderedAccessView *)0xdeadbeef) ID3D11UnorderedAccessView_Release(tmp_uav);

    ID3D11DeviceContext1_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    ID3D11DeviceContext1_IASetInputLayout(context, il);
    ID3D11DeviceContext1_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, ib_offset);
    ID3D11DeviceContext1_IASetVertexBuffers(context, 0, 1, &vb, &vb_stride, &vb_offset);

    topo = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetPrimitiveTopology(context, &topo);
    todo_wine ok(topo == D3D11_PRIMITIVE_TOPOLOGY_UNDEFINED, "Got unexpected topology %#x.\n", topo);
    tmp_il = (ID3D11InputLayout *)0xdeadbeef;
    ID3D11DeviceContext1_IAGetInputLayout(context, &tmp_il);
    todo_wine ok(!tmp_il, "Got unexpected input layout %p.\n", tmp_il);
    if (tmp_il) ID3D11InputLayout_Release(tmp_il);
    tmp_ib = (ID3D11Buffer *)0xdeadbeef;
    format = 0xdeadbeef;
    offset = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetIndexBuffer(context, &tmp_ib, &format, &offset);
    todo_wine ok(!tmp_ib, "Got unexpected input buffer %p.\n", tmp_ib);
    if (tmp_ib) ID3D11Buffer_Release(tmp_ib);
    todo_wine ok(format == DXGI_FORMAT_UNKNOWN, "Got unexpected input buffer format %#x.\n", format);
    todo_wine ok(offset == 0, "Got unexpected input buffer offset %#x.\n", offset);
    tmp_vb = (ID3D11Buffer *)0xdeadbeef;
    stride = 0xdeadbeef;
    offset = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetVertexBuffers(context, 0, 1, &tmp_vb, &stride, &offset);
    todo_wine ok(!tmp_vb, "Got unexpected vertex buffer %p.\n", tmp_vb);
    if (tmp_vb) ID3D11Buffer_Release(tmp_vb);
    todo_wine ok(stride == 0, "Got unexpected vertex buffer stride %#x.\n", stride);
    todo_wine ok(offset == 0, "Got unexpected vertex buffer offset %#x.\n", offset);

    ID3D11DeviceContext1_OMSetBlendState(context, bs, custom_blend_factor, 0xff00ff00);
    ID3D11DeviceContext1_OMSetDepthStencilState(context, dss, 3);
    ID3D11DeviceContext1_OMSetRenderTargetsAndUnorderedAccessViews(context, 1, &rtv, dsv, 1, 1, &ps_uav, NULL);

    tmp_rtv = (ID3D11RenderTargetView *)0xdeadbeef;
    tmp_dsv = (ID3D11DepthStencilView *)0xdeadbeef;
    tmp_uav = (ID3D11UnorderedAccessView *)0xdeadbeef;
    ID3D11DeviceContext1_OMGetRenderTargetsAndUnorderedAccessViews(context, 1, &tmp_rtv, &tmp_dsv, 1, 1, &tmp_uav);
    todo_wine ok(!tmp_rtv, "Got unexpected rendertarget view %p.\n", tmp_rtv);
    if (tmp_rtv) ID3D11RenderTargetView_Release(tmp_rtv);
    todo_wine ok(!tmp_dsv, "Got unexpected depth/stencil view %p.\n", tmp_dsv);
    if (tmp_dsv) ID3D11DepthStencilView_Release(tmp_dsv);
    todo_wine ok(!tmp_uav, "Got unexpected unordered access view %p.\n", tmp_uav);
    if (tmp_uav) ID3D11UnorderedAccessView_Release(tmp_uav);
    tmp_bs = (ID3D11BlendState *)0xdeadbeef;
    memset(blend_factor, 0xcd, sizeof(blend_factor));
    sample_mask = 0xdeadbeef;
    ID3D11DeviceContext1_OMGetBlendState(context, &tmp_bs, blend_factor, &sample_mask);
    todo_wine ok(!tmp_bs, "Got unexpected blend state %p.\n", tmp_bs);
    if (tmp_bs) ID3D11BlendState_Release(tmp_bs);
    todo_wine ok(!memcmp(blend_factor, default_blend_factor, sizeof(blend_factor)),
            "Got unexpected blend factor %f,%f,%f,%f.\n",
            blend_factor[0], blend_factor[1], blend_factor[2], blend_factor[3]);
    todo_wine ok(sample_mask == ~0, "Got unexpected sample mask %#x.\n", sample_mask);
    tmp_dss = (ID3D11DepthStencilState *)0xdeadbeef;
    stencil_ref = 0xdeadbeef;
    ID3D11DeviceContext1_OMGetDepthStencilState(context, &tmp_dss, &stencil_ref);
    todo_wine ok(!tmp_dss, "Got unexpected depth/stencil state %p.\n", tmp_dss);
    if (tmp_dss) ID3D11DepthStencilState_Release(tmp_dss);
    todo_wine ok(stencil_ref == 0, "Got unexpected stencil ref %#x.\n", stencil_ref);

    ID3D11DeviceContext1_RSSetScissorRects(context, 1, &rect);
    ID3D11DeviceContext1_RSSetViewports(context, 1, &vp);
    ID3D11DeviceContext1_RSSetState(context, rs);

    ID3D11DeviceContext1_SOSetTargets(context, 1, &sob, &so_offset);
    ID3D11DeviceContext1_SetPredication(context, pred, TRUE);

    tmp_rs = (ID3D11RasterizerState *)0xdeadbeef;
    ID3D11DeviceContext1_RSGetState(context, &tmp_rs);
    todo_wine ok(!tmp_rs, "Got unexpected rasterizer state %p.\n", tmp_rs);
    if (tmp_rs) ID3D11RasterizerState_Release(tmp_rs);
    memset(tmp_vp, 0xa5, sizeof(tmp_vp));
    count = 2;
    ID3D11DeviceContext1_RSGetViewports(context, &count, tmp_vp);
    todo_wine ok(count == 0, "Got unexpected viewport count %u.\n", count);
    memset(tmp_rect, 0xa5, sizeof(tmp_rect));
    count = 2;
    ID3D11DeviceContext1_RSGetScissorRects(context, &count, tmp_rect);
    todo_wine ok(count == 0, "Got unexpected scissor rect count %u.\n", count);

    tmp_sob = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_SOGetTargets(context, 1, &tmp_sob);
    todo_wine ok(!tmp_sob, "Got unexpected stream output buffer %p.\n", tmp_sob);
    if (tmp_sob) ID3D11Buffer_Release(tmp_sob);

    tmp_pred = (ID3D11Predicate *)0xdeadbeef;
    pred_value = 0xdeadbeef;
    ID3D11DeviceContext1_GetPredication(context, &tmp_pred, &pred_value);
    todo_wine ok(!tmp_pred, "Got unexpected predicate %p.\n", tmp_pred);
    if (tmp_pred) ID3D11Predicate_Release(tmp_pred);
    todo_wine ok(!pred_value, "Got unexpected predicate value %d.\n", pred_value);

    check_interface(device, &IID_ID3D10Device, TRUE, FALSE);
    check_interface(device, &IID_ID3D10Device1, TRUE, FALSE);

    context_state = NULL;
    ID3D11DeviceContext1_SwapDeviceContextState(context, previous_context_state, &context_state);
    refcount = ID3DDeviceContextState_Release(context_state);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);
    refcount = ID3DDeviceContextState_Release(previous_context_state);
    ok(!refcount, "Got unexpected refcount %lu.\n", refcount);

    /* ID3DDeviceContextState retains the previous state. */

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_vs = (ID3D11VertexShader *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShader(context, &tmp_vs, NULL, NULL);
    ok(tmp_vs == vs, "Got shader %p, expected %p.\n", tmp_vs, vs);
    ID3D11VertexShader_Release(tmp_vs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_VSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_gs = (ID3D11GeometryShader *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetShader(context, &tmp_gs, NULL, NULL);
    ok(tmp_gs == gs, "Got shader %p, expected %p.\n", tmp_gs, gs);
    ID3D11GeometryShader_Release(tmp_gs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_GSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_ps = (ID3D11PixelShader *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetShader(context, &tmp_ps, NULL, NULL);
    ok(tmp_ps == ps, "Got shader %p, expected %p.\n", tmp_ps, ps);
    ID3D11PixelShader_Release(tmp_ps);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_PSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_hs = (ID3D11HullShader *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetShader(context, &tmp_hs, NULL, NULL);
    ok(tmp_hs == hs, "Got shader %p, expected %p.\n", tmp_hs, hs);
    if (hs) ID3D11HullShader_Release(tmp_hs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_HSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_ds = (ID3D11DomainShader *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetShader(context, &tmp_ds, NULL, NULL);
    ok(tmp_ds == ds, "Got shader %p, expected %p.\n", tmp_ds, ds);
    if (ds) ID3D11DomainShader_Release(tmp_ds);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_DSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);

    tmp_sampler = (ID3D11SamplerState *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetSamplers(context, 0, 1, &tmp_sampler);
    ok(tmp_sampler == sampler, "Got sampler %p, expected %p.\n", tmp_sampler, sampler);
    ID3D11SamplerState_Release(tmp_sampler);
    tmp_cb = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetConstantBuffers(context, 0, 1, &tmp_cb);
    ok(tmp_cb == cb, "Got buffer %p, expected %p.\n", tmp_cb, cb);
    ID3D11Buffer_Release(tmp_cb);
    tmp_cs = (ID3D11ComputeShader *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetShader(context, &tmp_cs, NULL, NULL);
    ok(tmp_cs == cs, "Got shader %p, expected %p.\n", tmp_cs, cs);
    if (cs) ID3D11ComputeShader_Release(tmp_cs);
    tmp_srv = (ID3D11ShaderResourceView *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetShaderResources(context, 0, 1, &tmp_srv);
    ok(tmp_srv == srv, "Got srv %p, expected %p.\n", tmp_srv, srv);
    ID3D11ShaderResourceView_Release(tmp_srv);
    tmp_uav = (ID3D11UnorderedAccessView *)0xdeadbeef;
    ID3D11DeviceContext1_CSGetUnorderedAccessViews(context, 0, 1, &tmp_uav);
    ok(tmp_uav == uav, "Got uav %p, expected %p.\n", tmp_uav, uav);
    ID3D11UnorderedAccessView_Release(tmp_uav);

    topo = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetPrimitiveTopology(context, &topo);
    ok(topo == D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP, "Got topology %#x, expected %#x.\n", topo, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    tmp_il = (ID3D11InputLayout *)0xdeadbeef;
    ID3D11DeviceContext1_IAGetInputLayout(context, &tmp_il);
    ok(tmp_il == il, "Got input layout %p, expected %p.\n", tmp_il, il);
    ID3D11InputLayout_Release(tmp_il);
    tmp_ib = (ID3D11Buffer *)0xdeadbeef;
    format = 0xdeadbeef;
    offset = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetIndexBuffer(context, &tmp_ib, &format, &offset);
    ok(tmp_ib == ib, "Got input buffer %p, expected %p.\n", tmp_ib, ib);
    ID3D11Buffer_Release(tmp_ib);
    ok(format == DXGI_FORMAT_R32_UINT, "Got input buffer format %#x, expected %#x.\n", format, DXGI_FORMAT_R32_UINT);
    ok(offset == 16, "Got input buffer offset %#x, expected 16.\n", offset);
    tmp_vb = (ID3D11Buffer *)0xdeadbeef;
    stride = 0xdeadbeef;
    offset = 0xdeadbeef;
    ID3D11DeviceContext1_IAGetVertexBuffers(context, 0, 1, &tmp_vb, &stride, &offset);
    ok(tmp_vb == vb, "Got vertex buffer %p, expected %p.\n", tmp_vb, vb);
    ID3D11Buffer_Release(tmp_vb);
    ok(stride == 16, "Got vertex buffer stride %#x, expected 16.\n", stride);
    ok(offset == 16, "Got vertex buffer offset %#x, expected 16.\n", offset);

    tmp_rtv = (ID3D11RenderTargetView *)0xdeadbeef;
    tmp_dsv = (ID3D11DepthStencilView *)0xdeadbeef;
    tmp_uav = (ID3D11UnorderedAccessView *)0xdeadbeef;
    ID3D11DeviceContext1_OMGetRenderTargetsAndUnorderedAccessViews(context, 1, &tmp_rtv, &tmp_dsv, 1, 1, &tmp_uav);
    ok(tmp_rtv == rtv, "Got rendertarget view %p, expected %p.\n", tmp_rtv, rtv);
    ID3D11RenderTargetView_Release(tmp_rtv);
    ok(tmp_dsv == dsv, "Got depth/stencil view %p, expected %p.\n", tmp_dsv, dsv);
    ID3D11DepthStencilView_Release(tmp_dsv);
    ok(tmp_uav == ps_uav, "Got unordered access view %p, expected %p.\n", tmp_uav, ps_uav);
    ID3D11UnorderedAccessView_Release(tmp_uav);
    tmp_bs = (ID3D11BlendState *)0xdeadbeef;
    memset(blend_factor, 0xcd, sizeof(blend_factor));
    sample_mask = 0xdeadbeef;
    ID3D11DeviceContext1_OMGetBlendState(context, &tmp_bs, blend_factor, &sample_mask);
    ok(tmp_bs == bs, "Got blend state %p, expected %p.\n", tmp_bs, bs);
    ID3D11BlendState_Release(tmp_bs);
    ok(!memcmp(blend_factor, custom_blend_factor, sizeof(blend_factor)),
            "Got blend factor %f,%f,%f,%f, expected %f,%f,%f,%f.\n",
            blend_factor[0], blend_factor[1], blend_factor[2], blend_factor[3],
            custom_blend_factor[0], custom_blend_factor[1], custom_blend_factor[2], custom_blend_factor[3]);
    ok(sample_mask == 0xff00ff00, "Got sample mask %#x, expected %#x.\n", sample_mask, 0xff00ff00);
    tmp_dss = (ID3D11DepthStencilState *)0xdeadbeef;
    stencil_ref = 0xdeadbeef;
    ID3D11DeviceContext1_OMGetDepthStencilState(context, &tmp_dss, &stencil_ref);
    ok(tmp_dss == dss, "Got depth/stencil state %p, expected %p.\n", tmp_dss, dss);
    ID3D11DepthStencilState_Release(tmp_dss);
    ok(stencil_ref == 3, "Got stencil ref %#x, expected 3.\n", stencil_ref);

    tmp_rs = (ID3D11RasterizerState *)0xdeadbeef;
    ID3D11DeviceContext1_RSGetState(context, &tmp_rs);
    ok(tmp_rs == rs, "Got unexpected rasterizer state %p.\n", tmp_rs);
    ID3D11RasterizerState_Release(tmp_rs);
    memset(tmp_vp, 0xa5, sizeof(tmp_vp));
    count = 2;
    ID3D11DeviceContext1_RSGetViewports(context, &count, tmp_vp);
    ok(count == 1, "Got viewport count %u, expected 1.\n", count);
    ok(!memcmp(tmp_vp, &vp, sizeof(vp)), "Got viewport %s, expected %s.\n",
            debugstr_viewport(tmp_vp), debugstr_viewport(&vp));
    memset(tmp_rect, 0xa5, sizeof(tmp_rect));
    count = 2;
    ID3D11DeviceContext1_RSGetScissorRects(context, &count, tmp_rect);
    ok(count == 1, "Got scissor rect count %u, expected 1.\n", count);
    ok(!memcmp(tmp_rect, &rect, sizeof(rect)), "Got scissor rect %s, expected %s.\n",
            wine_dbgstr_rect(tmp_rect), wine_dbgstr_rect(&rect));

    tmp_sob = (ID3D11Buffer *)0xdeadbeef;
    ID3D11DeviceContext1_SOGetTargets(context, 1, &tmp_sob);
    ok(tmp_sob == sob, "Got stream output buffer %p, expected %p.\n", tmp_sob, sob);
    ID3D11Buffer_Release(tmp_sob);

    tmp_pred = (ID3D11Predicate *)0xdeadbeef;
    pred_value = 0xdeadbeef;
    ID3D11DeviceContext1_GetPredication(context, &tmp_pred, &pred_value);
    ok(tmp_pred == pred, "Got predicate %p, expected %p.\n", tmp_pred, pred);
    ID3D11Predicate_Release(tmp_pred);
    ok(pred_value == TRUE, "Got predicate value %#x, expected TRUE.\n", pred_value);

    check_interface(device, &IID_ID3D10Device, TRUE, FALSE);
    check_interface(device, &IID_ID3D10Device1, TRUE, FALSE);

    ID3D11Predicate_Release(pred);
    ID3D11Buffer_Release(sob);
    ID3D11RasterizerState_Release(rs);
    ID3D11BlendState_Release(bs);
    ID3D11DepthStencilState_Release(dss);
    ID3D11DepthStencilView_Release(dsv);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11UnorderedAccessView_Release(ps_uav);
    ID3D11InputLayout_Release(il);
    ID3D11Buffer_Release(ib);
    ID3D11Buffer_Release(vb);
    if (cs) ID3D11ComputeShader_Release(cs);
    if (ds) ID3D11DomainShader_Release(ds);
    if (hs) ID3D11HullShader_Release(hs);
    ID3D11PixelShader_Release(ps);
    ID3D11GeometryShader_Release(gs);
    ID3D11VertexShader_Release(vs2);
    ID3D11VertexShader_Release(vs);
    ID3D11Buffer_Release(cb);
    ID3D11ShaderResourceView_Release(srv);
    ID3D11UnorderedAccessView_Release(uav);
    ID3D11SamplerState_Release(sampler);
    ID3D11DeviceContext1_Release(context);
    refcount = ID3D11Device1_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_blend(void)
{
    ID3D11BlendState *src_blend, *dst_blend, *dst_blend_factor;
    struct d3d11_test_context test_context;
    ID3D11RenderTargetView *offscreen_rtv;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11InputLayout *input_layout;
    ID3D11DeviceContext *context;
    D3D11_BLEND_DESC blend_desc;
    unsigned int stride, offset;
    ID3D11Texture2D *offscreen;
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *vb;
    DWORD color;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        struct vs_out
        {
            float4 position : SV_POSITION;
            float4 color : COLOR;
        };

        struct vs_out main(float4 position : POSITION, float4 color : COLOR)
        {
            struct vs_out o;

            o.position = position;
            o.color = color;

            return o;
        }
#endif
        0x43425844, 0x5c73b061, 0x5c71125f, 0x3f8b345f, 0xce04b9ab, 0x00000001, 0x00000140, 0x00000003,
        0x0000002c, 0x0000007c, 0x000000d0, 0x4e475349, 0x00000048, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000041, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0x4c4f4300, 0xab00524f, 0x4e47534f,
        0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000,
        0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f, 0x505f5653,
        0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052, 0x52444853, 0x00000068, 0x00010040, 0x0000001a,
        0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x001010f2, 0x00000001, 0x04000067, 0x001020f2,
        0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x05000036, 0x001020f2, 0x00000000,
        0x00101e46, 0x00000000, 0x05000036, 0x001020f2, 0x00000001, 0x00101e46, 0x00000001, 0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        struct vs_out
        {
            float4 position : SV_POSITION;
            float4 color : COLOR;
        };

        float4 main(struct vs_out i) : SV_TARGET
        {
            return i.color;
        }
#endif
        0x43425844, 0xe2087fa6, 0xa35fbd95, 0x8e585b3f, 0x67890f54, 0x00000001, 0x000000f4, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000038, 0x00000040,
        0x0000000e, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000001, 0x0100003e,
    };
    static const struct
    {
        struct vec3 position;
        DWORD diffuse;
    }
    quads[] =
    {
        /* quad1 */
        {{-1.0f, -1.0f, 0.1f}, 0x4000ff00},
        {{-1.0f,  0.0f, 0.1f}, 0x4000ff00},
        {{ 1.0f, -1.0f, 0.1f}, 0x4000ff00},
        {{ 1.0f,  0.0f, 0.1f}, 0x4000ff00},
        /* quad2 */
        {{-1.0f,  0.0f, 0.1f}, 0xc0ff0000},
        {{-1.0f,  1.0f, 0.1f}, 0xc0ff0000},
        {{ 1.0f,  0.0f, 0.1f}, 0xc0ff0000},
        {{ 1.0f,  1.0f, 0.1f}, 0xc0ff0000},
    };
    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"COLOR",    0, DXGI_FORMAT_R8G8B8A8_UNORM,  0, 12, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    static const float blend_factor[] = {0.3f, 0.4f, 0.8f, 0.9f};
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(quads), quads);

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    memset(&blend_desc, 0, sizeof(blend_desc));
    blend_desc.RenderTarget[0].BlendEnable = TRUE;
    blend_desc.RenderTarget[0].SrcBlend = D3D11_BLEND_SRC_ALPHA;
    blend_desc.RenderTarget[0].DestBlend = D3D11_BLEND_INV_SRC_ALPHA;
    blend_desc.RenderTarget[0].BlendOp = D3D11_BLEND_OP_ADD;
    blend_desc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_SRC_ALPHA;
    blend_desc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_INV_SRC_ALPHA;
    blend_desc.RenderTarget[0].BlendOpAlpha = D3D11_BLEND_OP_ADD;
    blend_desc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;

    hr = ID3D11Device_CreateBlendState(device, &blend_desc, &src_blend);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    blend_desc.RenderTarget[0].SrcBlend = D3D11_BLEND_DEST_ALPHA;
    blend_desc.RenderTarget[0].DestBlend = D3D11_BLEND_INV_DEST_ALPHA;
    blend_desc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_DEST_ALPHA;
    blend_desc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_INV_DEST_ALPHA;

    hr = ID3D11Device_CreateBlendState(device, &blend_desc, &dst_blend);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    blend_desc.RenderTarget[0].SrcBlend = D3D11_BLEND_BLEND_FACTOR;
    blend_desc.RenderTarget[0].DestBlend = D3D11_BLEND_INV_BLEND_FACTOR;
    blend_desc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_DEST_ALPHA;
    blend_desc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_INV_DEST_ALPHA;

    hr = ID3D11Device_CreateBlendState(device, &blend_desc, &dst_blend_factor);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    stride = sizeof(*quads);
    offset = 0;
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb, &stride, &offset);
    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

    ID3D11DeviceContext_OMSetBlendState(context, src_blend, NULL, D3D11_DEFAULT_SAMPLE_MASK);
    ID3D11DeviceContext_Draw(context, 4, 0);
    ID3D11DeviceContext_OMSetBlendState(context, dst_blend, NULL, D3D11_DEFAULT_SAMPLE_MASK);
    ID3D11DeviceContext_Draw(context, 4, 4);

    color = get_texture_color(test_context.backbuffer, 320, 360);
    ok(compare_color(color, 0x700040bf, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 120);
    ok(compare_color(color, 0xa080007f, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

    ID3D11DeviceContext_OMSetBlendState(context, dst_blend_factor, blend_factor, D3D11_DEFAULT_SAMPLE_MASK);
    ID3D11DeviceContext_Draw(context, 4, 0);
    ID3D11DeviceContext_Draw(context, 4, 4);

    color = get_texture_color(test_context.backbuffer, 320, 360);
    ok(compare_color(color, 0x600066b3, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 120);
    ok(compare_color(color, 0xa0cc00b3, 1), "Got unexpected colour 0x%08lx.\n", color);

    texture_desc.Width = 128;
    texture_desc.Height = 128;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_B8G8R8X8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE | D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    /* DXGI_FORMAT_B8G8R8X8_UNORM is not supported on all implementations. */
    if (FAILED(ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &offscreen)))
    {
        skip("DXGI_FORMAT_B8G8R8X8_UNORM not supported.\n");
        goto done;
    }

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)offscreen, NULL, &offscreen_rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &offscreen_rtv, NULL);

    set_viewport(context, 0.0f, 0.0f, 128.0f, 128.0f, 0.0f, 1.0f);

    ID3D11DeviceContext_ClearRenderTargetView(context, offscreen_rtv, red);

    ID3D11DeviceContext_OMSetBlendState(context, src_blend, NULL, D3D11_DEFAULT_SAMPLE_MASK);
    ID3D11DeviceContext_Draw(context, 4, 0);
    ID3D11DeviceContext_OMSetBlendState(context, dst_blend, NULL, D3D11_DEFAULT_SAMPLE_MASK);
    ID3D11DeviceContext_Draw(context, 4, 4);

    color = get_texture_color(offscreen, 64, 96) & 0x00ffffff;
    ok(compare_color(color, 0x00bf4000, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(offscreen, 64, 32) & 0x00ffffff;
    ok(compare_color(color, 0x000000ff, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11RenderTargetView_Release(offscreen_rtv);
    ID3D11Texture2D_Release(offscreen);
done:
    ID3D11BlendState_Release(dst_blend_factor);
    ID3D11BlendState_Release(dst_blend);
    ID3D11BlendState_Release(src_blend);
    ID3D11PixelShader_Release(ps);
    ID3D11VertexShader_Release(vs);
    ID3D11Buffer_Release(vb);
    ID3D11InputLayout_Release(input_layout);
    release_test_context(&test_context);
}

static void test_texture1d(void)
{
    struct shader
    {
        const DWORD *code;
        size_t size;
    };
    struct texture
    {
        UINT width;
        UINT miplevel_count;
        UINT array_size;
        DXGI_FORMAT format;
        D3D11_SUBRESOURCE_DATA data[3];
    };

    D3D11_SHADER_RESOURCE_VIEW_DESC srv_desc;
    struct d3d11_test_context test_context;
    const struct texture *current_texture;
    D3D11_TEXTURE1D_DESC texture_desc;
    D3D11_SAMPLER_DESC sampler_desc;
    const struct shader *current_ps;
    D3D_FEATURE_LEVEL feature_level;
    ID3D11ShaderResourceView *srv;
    ID3D11DeviceContext *context;
    ID3D11SamplerState *sampler;
    struct resource_readback rb;
    ID3D11Texture1D *texture;
    struct vec4 ps_constant;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    unsigned int i, x;
    ID3D11Buffer *cb;
    DWORD color;
    HRESULT hr;

    static const DWORD ps_ld_code[] =
    {
#if 0
        Texture1D t;

        float miplevel;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float2 p;
            t.GetDimensions(miplevel, p.x, p.y);
            p.y = miplevel;
            p *= float2(position.x / 640.0f, 1.0f);
            return t.Load(int2(p));
        }
#endif
        0x43425844, 0x7b0c6359, 0x598178f6, 0xef2ddbdb, 0x88fc794c, 0x00000001, 0x000001ac, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000010f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000110, 0x00000040,
        0x00000044, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x04001058, 0x00107000, 0x00000000,
        0x00005555, 0x04002064, 0x00101012, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x0600001c, 0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000,
        0x0700003d, 0x001000f2, 0x00000000, 0x0010000a, 0x00000000, 0x00107e46, 0x00000000, 0x07000038,
        0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x0010100a, 0x00000000, 0x06000036, 0x001000e2,
        0x00000000, 0x00208006, 0x00000000, 0x00000000, 0x0a000038, 0x001000f2, 0x00000000, 0x00100e46,
        0x00000000, 0x00004002, 0x3acccccd, 0x3f800000, 0x3f800000, 0x3f800000, 0x0500001b, 0x001000f2,
        0x00000000, 0x00100e46, 0x00000000, 0x0700002d, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000,
        0x00107e46, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_ld = {ps_ld_code, sizeof(ps_ld_code)};
    static const DWORD ps_ld_sint8_code[] =
    {
#if 0
        Texture1D<int4> t;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float2 p, s;
            int4 c;

            p = float2(position.x / 640.0f, 0.0f);
            t.GetDimensions(0, s.x, s.y);
            p *= s;

            c = t.Load(int2(p));
            return (max(c / (float4)127, (float4)-1) + (float4)1) / 2.0f;
        }
#endif
        0x43425844, 0x65a13d1e, 0x8a0bfc92, 0xa2f2708a, 0x0bafafb6, 0x00000001, 0x00000234, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000010f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000198, 0x00000040,
        0x00000066, 0x04001058, 0x00107000, 0x00000000, 0x00003333, 0x04002064, 0x00101012, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000002, 0x0700003d, 0x001000f2,
        0x00000000, 0x00004001, 0x00000000, 0x00107e46, 0x00000000, 0x07000038, 0x00100012, 0x00000001,
        0x0010100a, 0x00000000, 0x00004001, 0x3acccccd, 0x08000036, 0x001000e2, 0x00000001, 0x00004002,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x07000038, 0x001000f2, 0x00000000, 0x00100fc6,
        0x00000000, 0x00100e46, 0x00000001, 0x0500001b, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000,
        0x0700002d, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x00107e46, 0x00000000, 0x0500002b,
        0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x0a000038, 0x001000f2, 0x00000000, 0x00100e46,
        0x00000000, 0x00004002, 0x3c010204, 0x3c010204, 0x3c010204, 0x3c010204, 0x0a000034, 0x001000f2,
        0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0xbf800000, 0xbf800000, 0xbf800000, 0xbf800000,
        0x0a000000, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0x3f800000, 0x3f800000,
        0x3f800000, 0x3f800000, 0x0a000038, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x00004002,
        0x3f000000, 0x3f000000, 0x3f000000, 0x3f000000, 0x0100003e,
    };
    static const struct shader ps_ld_sint8 = {ps_ld_sint8_code, sizeof(ps_ld_sint8_code)};
    static const DWORD ps_ld_uint8_code[] =
    {
#if 0
        Texture1D<uint4> t;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float2 p, s;

            p = float2(position.x / 640.0f, 0.0f);
            t.GetDimensions(0, s.x, s.y);
            p *= s;

            return t.Load(int2(p)) / (float4)255;
        }
#endif
        0x43425844, 0x35186c1f, 0x55bad4fd, 0xb7c97a57, 0x99c060e7, 0x00000001, 0x000001bc, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000010f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000120, 0x00000040,
        0x00000048, 0x04001058, 0x00107000, 0x00000000, 0x00004444, 0x04002064, 0x00101012, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000002, 0x0700003d, 0x001000f2,
        0x00000000, 0x00004001, 0x00000000, 0x00107e46, 0x00000000, 0x07000038, 0x00100012, 0x00000001,
        0x0010100a, 0x00000000, 0x00004001, 0x3acccccd, 0x08000036, 0x001000e2, 0x00000001, 0x00004002,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x07000038, 0x001000f2, 0x00000000, 0x00100fc6,
        0x00000000, 0x00100e46, 0x00000001, 0x0500001b, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000,
        0x0700002d, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x00107e46, 0x00000000, 0x05000056,
        0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x0a000038, 0x001020f2, 0x00000000, 0x00100e46,
        0x00000000, 0x00004002, 0x3b808081, 0x3b808081, 0x3b808081, 0x3b808081, 0x0100003e,
    };
    static const struct shader ps_ld_uint8 = {ps_ld_uint8_code, sizeof(ps_ld_uint8_code)};
    static DWORD ps_ld_array_code[] =
    {
#if 0
        Texture1DArray t;

        float miplevel;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float3 p;
            t.GetDimensions(miplevel, p.x, p.y, p.z);
            p.y = 1;
            p.z = miplevel;
            p *= float3(position.x / 640.0f, 1.0f, 1.0f);
            return t.Load(int3(p));
        }
#endif
        0x43425844, 0xbfccadc4, 0xc00ff13d, 0x2ba75365, 0xf747cbee, 0x00000001, 0x000001c0, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000010f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000124, 0x00000040,
        0x00000049, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x04003858, 0x00107000, 0x00000000,
        0x00005555, 0x04002064, 0x00101012, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x0600001c, 0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000,
        0x0700003d, 0x001000f2, 0x00000000, 0x0010000a, 0x00000000, 0x00107e46, 0x00000000, 0x07000038,
        0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x0010100a, 0x00000000, 0x06000036, 0x001000c2,
        0x00000000, 0x00208006, 0x00000000, 0x00000000, 0x0a000038, 0x00100072, 0x00000000, 0x00100386,
        0x00000000, 0x00004002, 0x3acccccd, 0x3f800000, 0x3f800000, 0x00000000, 0x0500001b, 0x001000d2,
        0x00000000, 0x00100906, 0x00000000, 0x05000036, 0x00100022, 0x00000000, 0x00004001, 0x00000001,
        0x0700002d, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x00107e46, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_ld_array = {ps_ld_array_code, sizeof(ps_ld_array_code)};

    static const DWORD rgba_level_0[] =
    {
        0xff0000ff, 0xff00ffff, 0xff00ff00, 0xffffff00,
    };
    static const DWORD rgba_level_1[] =
    {
        0xffffffff, 0xff0000ff,
    };
    static const DWORD rgba_level_2[] =
    {
        0xffff0000,
    };
    static const DWORD srgb_data[] =
    {
        0x00000000, 0xffffffff, 0xff000000, 0x7f7f7f7f,
    };
    static const DWORD r32_uint[] =
    {
          0,   1,   2,   3,
    };
    static const DWORD r9g9b9e5_data[] =
    {
        0x80000100, 0x80020000, 0x84000000, 0x84000100,
    };
    static const DWORD array_data0[] =
    {
        0xff0000ff, 0xff00ffff, 0xff00ff00, 0xffffff00,
    };
    static const DWORD array_data1[] =
    {
        0x00ffff00, 0xff000000, 0x00ff0000, 0x000000ff,
    };
    static const DWORD array_data2[] =
    {
        0x000000ff, 0xffff00ff, 0x0000ff00, 0xff000000,
    };
    static const struct texture rgba_texture =
    {
        4, 3, 1, DXGI_FORMAT_R8G8B8A8_UNORM,
        {
            {rgba_level_0, 4 * sizeof(*rgba_level_0), 0},
            {rgba_level_1, 2 * sizeof(*rgba_level_1), 0},
            {rgba_level_2,     sizeof(*rgba_level_2), 0},
        }
    };
    static const struct texture srgb_texture = {4, 1, 1, DXGI_FORMAT_R8G8B8A8_UNORM_SRGB,
            {{srgb_data, 4 * sizeof(*srgb_data)}}};
    static const struct texture sint8_texture = {4, 1, 1, DXGI_FORMAT_R8G8B8A8_SINT,
            {{rgba_level_0, 4 * sizeof(*rgba_level_0)}}};
    static const struct texture uint8_texture = {4, 1, 1, DXGI_FORMAT_R8G8B8A8_UINT,
            {{rgba_level_0, 4 * sizeof(*rgba_level_0)}}};
    static const struct texture r32u_typeless = {4, 1, 1, DXGI_FORMAT_R32_TYPELESS,
        {{r32_uint, 4 * sizeof(*r32_uint)}}};
    static const struct texture r9g9b9e5_texture = {4, 1, 1, DXGI_FORMAT_R9G9B9E5_SHAREDEXP,
        {{r9g9b9e5_data, 4 * sizeof(*r9g9b9e5_data)}}};
    static const struct texture array_texture = {4, 1, 3, DXGI_FORMAT_R8G8B8A8_UNORM_SRGB,
        {
            {array_data0, 4 * sizeof(*array_data0)},
            {array_data1, 4 * sizeof(*array_data1)},
            {array_data2, 4 * sizeof(*array_data2)},
        }
    };

    static const DWORD level_1_colors[] =
    {
        0xffffffff, 0xffffffff, 0xff0000ff, 0xff0000ff,
    };
    static const DWORD level_2_colors[] =
    {
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
    };
    static const DWORD srgb_colors[] =
    {
        0x00000001, 0xffffffff, 0xff000000, 0x7f363636,
    };
    static const DWORD sint8_colors[] =
    {
        0x7e80807e, 0x7e807e7e, 0x7e807e80, 0x7e7e7e80,
    };
    static const DWORD r32u_colors[4] =
    {
        0x01000000, 0x01000001, 0x01000002, 0x01000003,
    };
    static const DWORD r9g9b9e5_colors[4] =
    {
        0xff0000ff, 0xff00ff00, 0xffff0000, 0xffff00ff,
    };
    static const DWORD zero_colors[4] = {0};
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};
    static const struct texture_test
    {
        const struct shader *ps;
        const struct texture *texture;
        D3D11_FILTER filter;
        float lod_bias;
        float min_lod;
        float max_lod;
        float ps_constant;
        const DWORD *expected_colors;
    }
    texture_tests[] =
    {
#define POINT        D3D11_FILTER_MIN_MAG_MIP_POINT
#define POINT_LINEAR D3D11_FILTER_MIN_MAG_POINT_MIP_LINEAR
#define MIP_MAX      D3D11_FLOAT32_MAX
        {&ps_ld,              &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  0.0f, rgba_level_0},
        {&ps_ld,              &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  1.0f, level_1_colors},
        {&ps_ld,              &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  2.0f, level_2_colors},
        {&ps_ld,              &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  3.0f, zero_colors},
        {&ps_ld,              &srgb_texture,     POINT,        0.0f, 0.0f,    0.0f,  0.0f, srgb_colors},
        {&ps_ld,              &r9g9b9e5_texture, POINT,        0.0f, 0.0f,    0.0f,  0.0f, r9g9b9e5_colors},
        {&ps_ld,              NULL,              POINT,        0.0f, 0.0f,    0.0f,  0.0f, zero_colors},
        {&ps_ld,              NULL,              POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, zero_colors},
        {&ps_ld_sint8,        &sint8_texture,    POINT,        0.0f, 0.0f,    0.0f,  0.0f, sint8_colors},
        {&ps_ld_uint8,        &uint8_texture,    POINT,        0.0f, 0.0f,    0.0f,  0.0f, rgba_level_0},
        {&ps_ld_array,        &array_texture,    POINT,        0.0f, 0.0f,    0.0f,  0.0f, array_data1},
    };
#undef POINT
#undef POINT_LINEAR
#undef MIP_MAX
    static const struct srv_test
    {
        const struct shader *ps;
        const struct texture *texture;
        struct srv_desc srv_desc;
        float ps_constant;
        const DWORD *expected_colors;
    }
    srv_tests[] =
    {
#define TEX_1D              D3D11_SRV_DIMENSION_TEXTURE1D
#define R32_UINT            DXGI_FORMAT_R32_UINT
        {&ps_ld_uint8,        &r32u_typeless,    {R32_UINT,            TEX_1D,       0, 1},       0.0f, r32u_colors},
#undef TEX_1D
#undef R32_UINT
#undef FMT_UNKNOWN
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;
    feature_level = ID3D11Device_GetFeatureLevel(device);

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(ps_constant), NULL);

    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = D3D11_FLOAT32_MAX;

    ps = NULL;
    srv = NULL;
    sampler = NULL;
    texture = NULL;
    current_ps = NULL;
    current_texture = NULL;
    for (i = 0; i < ARRAY_SIZE(texture_tests); ++i)
    {
        const struct texture_test *test = &texture_tests[i];

        if (current_ps != test->ps)
        {
            if (ps)
                ID3D11PixelShader_Release(ps);

            current_ps = test->ps;

            hr = ID3D11Device_CreatePixelShader(device, current_ps->code, current_ps->size, NULL, &ps);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

            ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
        }

        if (current_texture != test->texture)
        {
            if (texture)
                ID3D11Texture1D_Release(texture);
            if (srv)
                ID3D11ShaderResourceView_Release(srv);

            current_texture = test->texture;

            if (current_texture)
            {
                texture_desc.Width = current_texture->width;
                texture_desc.MipLevels = current_texture->miplevel_count;
                texture_desc.ArraySize = current_texture->array_size;
                texture_desc.Format = current_texture->format;

                hr = ID3D11Device_CreateTexture1D(device, &texture_desc, current_texture->data, &texture);
                ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

                hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, NULL, &srv);
                ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            }
            else
            {
                texture = NULL;
                srv = NULL;
            }

            ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);
        }

        if (!sampler || (sampler_desc.Filter != test->filter
                || sampler_desc.MipLODBias != test->lod_bias
                || sampler_desc.MinLOD != test->min_lod
                || sampler_desc.MaxLOD != test->max_lod))
        {
            if (sampler)
                ID3D11SamplerState_Release(sampler);

            sampler_desc.Filter = test->filter;
            sampler_desc.MipLODBias = test->lod_bias;
            sampler_desc.MinLOD = test->min_lod;
            sampler_desc.MaxLOD = test->max_lod;

            hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

            ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);
        }

        ps_constant.x = test->ps_constant;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &ps_constant, 0, 0);

        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

        draw_quad(&test_context);

        get_texture_readback(test_context.backbuffer, 0, &rb);
        for (x = 0; x < 4; ++x)
        {
            color = get_readback_color(&rb, 80 + x * 160, 0, 0);
            ok(compare_color(color, test->expected_colors[x], 2),
                    "Test %u: Got unexpected colour 0x%08lx at (%u).\n", i, color, x);
        }
        release_resource_readback(&rb);
    }
    if (srv)
        ID3D11ShaderResourceView_Release(srv);
    ID3D11SamplerState_Release(sampler);
    if (texture)
        ID3D11Texture1D_Release(texture);
    ID3D11PixelShader_Release(ps);

    if (is_warp_device(device) && feature_level < D3D_FEATURE_LEVEL_11_0)
    {
        win_skip("SRV tests are broken on WARP.\n");
        ID3D11Buffer_Release(cb);
        release_test_context(&test_context);
        return;
    }

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = D3D11_FLOAT32_MAX;

    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);

    ps = NULL;
    srv = NULL;
    texture = NULL;
    current_ps = NULL;
    current_texture = NULL;
    for (i = 0; i < ARRAY_SIZE(srv_tests); ++i)
    {
        const struct srv_test *test = &srv_tests[i];

        if (current_ps != test->ps)
        {
            if (ps)
                ID3D11PixelShader_Release(ps);

            current_ps = test->ps;

            hr = ID3D11Device_CreatePixelShader(device, current_ps->code, current_ps->size, NULL, &ps);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

            ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
        }

        if (current_texture != test->texture)
        {
            if (texture)
                ID3D11Texture1D_Release(texture);

            current_texture = test->texture;

            texture_desc.Width = current_texture->width;
            texture_desc.MipLevels = current_texture->miplevel_count;
            texture_desc.ArraySize = current_texture->array_size;
            texture_desc.Format = current_texture->format;

            hr = ID3D11Device_CreateTexture1D(device, &texture_desc, current_texture->data, &texture);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        }

        if (srv)
            ID3D11ShaderResourceView_Release(srv);

        get_srv_desc(&srv_desc, &test->srv_desc);
        hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, &srv_desc, &srv);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);

        ps_constant.x = test->ps_constant;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &ps_constant, 0, 0);

        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

        draw_quad(&test_context);

        get_texture_readback(test_context.backbuffer, 0, &rb);
        for (x = 0; x < 4; ++x)
        {
            color = get_readback_color(&rb, 80 + x * 160, 0, 0);
            ok(compare_color(color, test->expected_colors[x], 1),
                    "Test %u: Got unexpected colour 0x%08lx at (%u).\n", i, color, x);
        }
        release_resource_readback(&rb);
    }
    ID3D11PixelShader_Release(ps);
    ID3D11Texture1D_Release(texture);
    ID3D11ShaderResourceView_Release(srv);
    ID3D11SamplerState_Release(sampler);

    ID3D11Buffer_Release(cb);
    release_test_context(&test_context);
}

static void test_texture(void)
{
    struct shader
    {
        const DWORD *code;
        size_t size;
    };
    struct texture
    {
        UINT width;
        UINT height;
        UINT miplevel_count;
        UINT array_size;
        DXGI_FORMAT format;
        D3D11_SUBRESOURCE_DATA data[3];
    };

    D3D11_SHADER_RESOURCE_VIEW_DESC srv_desc;
    struct d3d11_test_context test_context;
    const struct texture *current_texture;
    D3D11_TEXTURE2D_DESC texture_desc;
    D3D11_SAMPLER_DESC sampler_desc;
    const struct shader *current_ps;
    D3D_FEATURE_LEVEL feature_level;
    ID3D11ShaderResourceView *srv;
    ID3D11DeviceContext *context;
    ID3D11SamplerState *sampler;
    struct resource_readback rb;
    ID3D11Texture2D *texture;
    struct vec4 ps_constant;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    unsigned int i, x, y;
    ID3D11Buffer *cb;
    DWORD color;
    HRESULT hr;

    static const DWORD ps_ld_code[] =
    {
#if 0
        Texture2D t;

        float miplevel;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float3 p;
            t.GetDimensions(miplevel, p.x, p.y, p.z);
            p.z = miplevel;
            p *= float3(position.x / 640.0f, position.y / 480.0f, 1.0f);
            return t.Load(int3(p));
        }
#endif
        0x43425844, 0xbdda6bdf, 0xc6ffcdf1, 0xa58596b3, 0x822383f0, 0x00000001, 0x000001ac, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000110, 0x00000040,
        0x00000044, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x04001858, 0x00107000, 0x00000000,
        0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x0600001c, 0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000,
        0x0700003d, 0x001000f2, 0x00000000, 0x0010000a, 0x00000000, 0x00107e46, 0x00000000, 0x07000038,
        0x00100032, 0x00000000, 0x00100046, 0x00000000, 0x00101046, 0x00000000, 0x06000036, 0x001000c2,
        0x00000000, 0x00208006, 0x00000000, 0x00000000, 0x0a000038, 0x001000f2, 0x00000000, 0x00100e46,
        0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x3f800000, 0x3f800000, 0x0500001b, 0x001000f2,
        0x00000000, 0x00100e46, 0x00000000, 0x0700002d, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000,
        0x00107e46, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_ld = {ps_ld_code, sizeof(ps_ld_code)};
    static const DWORD ps_ld_sint8_code[] =
    {
#if 0
        Texture2D<int4> t;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float3 p, s;
            int4 c;

            p = float3(position.x / 640.0f, position.y / 480.0f, 0.0f);
            t.GetDimensions(0, s.x, s.y, s.z);
            p *= s;

            c = t.Load(int3(p));
            return (max(c / (float4)127, (float4)-1) + (float4)1) / 2.0f;
        }
#endif
        0x43425844, 0xb3d0b0fc, 0x0e486f4a, 0xf67eec12, 0xfb9dd52f, 0x00000001, 0x00000240, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x000001a4, 0x00000040,
        0x00000069, 0x04001858, 0x00107000, 0x00000000, 0x00003333, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000002, 0x0700003d, 0x001000f2,
        0x00000000, 0x00004001, 0x00000000, 0x00107e46, 0x00000000, 0x0a000038, 0x00100032, 0x00000001,
        0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000, 0x08000036,
        0x001000c2, 0x00000001, 0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x07000038,
        0x001000f2, 0x00000000, 0x00100f46, 0x00000000, 0x00100e46, 0x00000001, 0x0500001b, 0x001000f2,
        0x00000000, 0x00100e46, 0x00000000, 0x0700002d, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000,
        0x00107e46, 0x00000000, 0x0500002b, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x0a000038,
        0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0x3c010204, 0x3c010204, 0x3c010204,
        0x3c010204, 0x0a000034, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0xbf800000,
        0xbf800000, 0xbf800000, 0xbf800000, 0x0a000000, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000,
        0x00004002, 0x3f800000, 0x3f800000, 0x3f800000, 0x3f800000, 0x0a000038, 0x001020f2, 0x00000000,
        0x00100e46, 0x00000000, 0x00004002, 0x3f000000, 0x3f000000, 0x3f000000, 0x3f000000, 0x0100003e,
    };
    static const struct shader ps_ld_sint8 = {ps_ld_sint8_code, sizeof(ps_ld_sint8_code)};
    static const DWORD ps_ld_uint8_code[] =
    {
#if 0
        Texture2D<uint4> t;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float3 p, s;

            p = float3(position.x / 640.0f, position.y / 480.0f, 0.0f);
            t.GetDimensions(0, s.x, s.y, s.z);
            p *= s;

            return t.Load(int3(p)) / (float4)255;
        }
#endif
        0x43425844, 0xd09917eb, 0x4508a07e, 0xb0b7250a, 0x228c1f0e, 0x00000001, 0x000001c8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x0000012c, 0x00000040,
        0x0000004b, 0x04001858, 0x00107000, 0x00000000, 0x00004444, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000002, 0x0700003d, 0x001000f2,
        0x00000000, 0x00004001, 0x00000000, 0x00107e46, 0x00000000, 0x0a000038, 0x00100032, 0x00000001,
        0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000, 0x08000036,
        0x001000c2, 0x00000001, 0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x07000038,
        0x001000f2, 0x00000000, 0x00100f46, 0x00000000, 0x00100e46, 0x00000001, 0x0500001b, 0x001000f2,
        0x00000000, 0x00100e46, 0x00000000, 0x0700002d, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000,
        0x00107e46, 0x00000000, 0x05000056, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x0a000038,
        0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0x3b808081, 0x3b808081, 0x3b808081,
        0x3b808081, 0x0100003e,
    };
    static const struct shader ps_ld_uint8 = {ps_ld_uint8_code, sizeof(ps_ld_uint8_code)};
    static const DWORD ps_sample_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;

            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;
            return t.Sample(s, p);
        }
#endif
        0x43425844, 0x1ce9b612, 0xc8176faa, 0xd37844af, 0xdb515605, 0x00000001, 0x00000134, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000098, 0x00000040,
        0x00000026, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555,
        0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x09000045, 0x001020f2, 0x00000000, 0x00100046, 0x00000000,
        0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_sample = {ps_sample_code, sizeof(ps_sample_code)};
    static const DWORD ps_sample_b_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float bias;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;

            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;
            return t.SampleBias(s, p, bias);
        }
#endif
        0x43425844, 0xc39b0686, 0x8244a7fc, 0x14c0b97a, 0x2900b3b7, 0x00000001, 0x00000150, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x000000b4, 0x00000040,
        0x0000002d, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000, 0x00000000,
        0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0a000038, 0x00100032, 0x00000000,
        0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000, 0x0c00004a,
        0x001020f2, 0x00000000, 0x00100046, 0x00000000, 0x00107e46, 0x00000000, 0x00106000, 0x00000000,
        0x0020800a, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_sample_b = {ps_sample_b_code, sizeof(ps_sample_b_code)};
    static const DWORD ps_sample_l_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float level;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;

            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;
            return t.SampleLevel(s, p, level);
        }
#endif
        0x43425844, 0x61e05d85, 0x2a7300fb, 0x0a83706b, 0x889d1683, 0x00000001, 0x00000150, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x000000b4, 0x00000040,
        0x0000002d, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000, 0x00000000,
        0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0a000038, 0x00100032, 0x00000000,
        0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000, 0x0c000048,
        0x001020f2, 0x00000000, 0x00100046, 0x00000000, 0x00107e46, 0x00000000, 0x00106000, 0x00000000,
        0x0020800a, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_sample_l = {ps_sample_l_code, sizeof(ps_sample_l_code)};
    static const DWORD ps_sample_2d_array_code[] =
    {
#if 0
        Texture2DArray t;
        SamplerState s;

        float layer;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float3 d;
            float3 p = float3(position.x / 640.0f, position.y / 480.0f, 1.0f);
            t.GetDimensions(d.x, d.y, d.z);
            d.z = layer;
            return t.Sample(s, p * d);
        }
#endif
        0x43425844, 0xa9457e44, 0xc0b3ef8e, 0x3d751ae8, 0x23fa4807, 0x00000001, 0x00000194, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x000000f8, 0x00000040,
        0x0000003e, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000, 0x00000000,
        0x04004058, 0x00107000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0700003d, 0x001000f2, 0x00000000,
        0x00004001, 0x00000000, 0x00107e46, 0x00000000, 0x0a000038, 0x001000c2, 0x00000000, 0x00101406,
        0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x3acccccd, 0x3b088889, 0x07000038, 0x00100032,
        0x00000000, 0x00100046, 0x00000000, 0x00100ae6, 0x00000000, 0x06000036, 0x00100042, 0x00000000,
        0x0020800a, 0x00000000, 0x00000000, 0x09000045, 0x001020f2, 0x00000000, 0x00100246, 0x00000000,
        0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_sample_2d_array = {ps_sample_2d_array_code, sizeof(ps_sample_2d_array_code)};
    static const DWORD red_data[] =
    {
        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0x00000000, 0x00000000,
        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0x00000000, 0x00000000,
        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0x00000000, 0x00000000,
        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff, 0x00000000, 0x00000000,
    };
    static const DWORD green_data[] =
    {
        0xff00ff00, 0xff00ff00, 0xff00ff00, 0xff00ff00,
        0xff00ff00, 0xff00ff00, 0xff00ff00, 0xff00ff00,
        0xff00ff00, 0xff00ff00, 0xff00ff00, 0xff00ff00,
        0xff00ff00, 0xff00ff00, 0xff00ff00, 0xff00ff00,
    };
    static const DWORD blue_data[] =
    {
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000, 0x00000000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000, 0x00000000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000, 0x00000000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000, 0x00000000,
    };
    static const DWORD rgba_level_0[] =
    {
        0xff0000ff, 0xff00ffff, 0xff00ff00, 0xffffff00,
        0xffff0000, 0xffff00ff, 0xff000000, 0xff7f7f7f,
        0xffffffff, 0xffffffff, 0xffffffff, 0xff000000,
        0xffffffff, 0xff000000, 0xff000000, 0xff000000,
    };
    static const DWORD rgba_level_1[] =
    {
        0xffffffff, 0xff0000ff,
        0xff000000, 0xff00ff00,
    };
    static const DWORD rgba_level_2[] =
    {
        0xffff0000,
    };
    static const DWORD srgb_data[] =
    {
        0x00000000, 0xffffffff, 0xff000000, 0x7f7f7f7f,
        0xff010203, 0xff102030, 0xff0a0b0c, 0xff8090a0,
        0xffb1c4de, 0xfff0f1f2, 0xfffafdfe, 0xff5a560f,
        0xffd5ff00, 0xffc8f99f, 0xffaa00aa, 0xffdd55bb,
    };
    static const WORD r8g8_data[] =
    {
        0x0000, 0xffff, 0x0000, 0x7fff,
        0x0203, 0xff10, 0x0b0c, 0x8000,
        0xc4de, 0xfff0, 0xfdfe, 0x5a6f,
        0xff00, 0xffc8, 0x00aa, 0xdd5b,
    };
    static const BYTE a8_data[] =
    {
        0x00, 0x10, 0x20, 0x30,
        0x40, 0x50, 0x60, 0x70,
        0x80, 0x90, 0xa0, 0xb0,
        0xc0, 0xd0, 0xe0, 0xf0,
    };
    static const BYTE bc1_data[] =
    {
        0x00, 0xf8, 0x00, 0xf8, 0x00, 0x00, 0x00, 0x00,
        0xe0, 0x07, 0xe0, 0x07, 0x00, 0x00, 0x00, 0x00,
        0x1f, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE bc2_data[] =
    {
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0xf8, 0x00, 0xf8, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x07, 0xe0, 0x07, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE bc3_data[] =
    {
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0xf8, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x07, 0xe0, 0x07, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE bc4_data[] =
    {
        0x10, 0x7f, 0x77, 0x39, 0x05, 0x00, 0x00, 0x00,
        0x10, 0x7f, 0x49, 0x92, 0x24, 0x49, 0x92, 0x24,
        0x10, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x10, 0x7f, 0xb6, 0x6d, 0xdb, 0xb6, 0x6d, 0xdb,
    };
    static const BYTE bc5_data[] =
    {
        0x10, 0x7f, 0x77, 0x39, 0x05, 0x00, 0x00, 0x00, 0x10, 0x7f, 0x77, 0x39, 0x05, 0x00, 0x00, 0x00,
        0x10, 0x7f, 0x49, 0x92, 0x24, 0x49, 0x92, 0x24, 0x10, 0x7f, 0x49, 0x92, 0x24, 0x49, 0x92, 0x24,
        0x10, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x10, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x10, 0x7f, 0xb6, 0x6d, 0xdb, 0xb6, 0x6d, 0xdb, 0x10, 0x7f, 0xb6, 0x6d, 0xdb, 0xb6, 0x6d, 0xdb,
    };
    static const BYTE bc6h_u_data[] =
    {
        0xe3, 0x5e, 0x00, 0x00, 0x78, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x80, 0x7b, 0x01, 0x00, 0xe0, 0x3d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0xee, 0x05, 0x00, 0x80, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xe3, 0xde, 0x7b, 0xef, 0x7d, 0xef, 0xbd, 0xf7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE bc6h_s_data[] =
    {
        0x63, 0x2f, 0x00, 0x00, 0xb8, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x80, 0xbd, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0xf6, 0x02, 0x00, 0x80, 0x7b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x63, 0xaf, 0xbd, 0xf6, 0xba, 0xe7, 0x9e, 0x7b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE bc7_data[] =
    {
        0x02, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const float r32_float[] =
    {
        0.0f, 1.0f,  0.5f, 0.50f,
        1.0f, 0.0f,  0.0f, 0.75f,
        0.0f, 1.0f,  0.5f, 0.25f,
        1.0f, 0.0f,  0.0f, 0.75f,
    };
    static const DWORD r32_uint[] =
    {
          0,   1,   2,   3,
        100, 200, 255, 128,
         40,  30,  20,  10,
        250, 210, 155, 190,
    };
    static const DWORD r9g9b9e5_data[] =
    {
        0x80000100, 0x80020000, 0x84000000, 0x84000100,
        0x78000100, 0x78020000, 0x7c000000, 0x78020100,
        0x70000133, 0x70026600, 0x74cc0000, 0x74cc0133,
        0x6800019a, 0x68033400, 0x6e680000, 0x6e6b359a,
    };
    static const struct texture rgba_texture =
    {
        4, 4, 3, 1, DXGI_FORMAT_R8G8B8A8_UNORM,
        {
            {rgba_level_0, 4 * sizeof(*rgba_level_0), 0},
            {rgba_level_1, 2 * sizeof(*rgba_level_1), 0},
            {rgba_level_2,     sizeof(*rgba_level_2), 0},
        }
    };
    static const struct texture srgb_texture = {4, 4, 1, 1, DXGI_FORMAT_R8G8B8A8_UNORM_SRGB,
            {{srgb_data, 4 * sizeof(*srgb_data)}}};
    static const struct texture srgb_typeless = {4, 4, 1, 1, DXGI_FORMAT_R8G8B8A8_TYPELESS,
            {{srgb_data, 4 * sizeof(*srgb_data)}}};
    static const struct texture a8_texture = {4, 4, 1, 1, DXGI_FORMAT_A8_UNORM,
            {{a8_data, 4 * sizeof(*a8_data)}}};
    static const struct texture bc1_texture = {8, 8, 1, 1, DXGI_FORMAT_BC1_UNORM, {{bc1_data, 2 * 8}}};
    static const struct texture bc2_texture = {8, 8, 1, 1, DXGI_FORMAT_BC2_UNORM, {{bc2_data, 2 * 16}}};
    static const struct texture bc3_texture = {8, 8, 1, 1, DXGI_FORMAT_BC3_UNORM, {{bc3_data, 2 * 16}}};
    static const struct texture bc4_texture = {8, 8, 1, 1, DXGI_FORMAT_BC4_UNORM, {{bc4_data, 2 * 8}}};
    static const struct texture bc5_texture = {8, 8, 1, 1, DXGI_FORMAT_BC5_UNORM, {{bc5_data, 2 * 16}}};
    static const struct texture bc6h_u_texture = {8, 8, 1, 1, DXGI_FORMAT_BC6H_UF16, {{bc6h_u_data, 2 * 16}}};
    static const struct texture bc6h_s_texture = {8, 8, 1, 1, DXGI_FORMAT_BC6H_SF16, {{bc6h_s_data, 2 * 16}}};
    static const struct texture bc7_texture = {8, 8, 1, 1, DXGI_FORMAT_BC7_UNORM, {{bc7_data, 2 * 16}}};
    static const struct texture bc1_texture_srgb = {8, 8, 1, 1, DXGI_FORMAT_BC1_UNORM_SRGB, {{bc1_data, 2 * 8}}};
    static const struct texture bc2_texture_srgb = {8, 8, 1, 1, DXGI_FORMAT_BC2_UNORM_SRGB, {{bc2_data, 2 * 16}}};
    static const struct texture bc3_texture_srgb = {8, 8, 1, 1, DXGI_FORMAT_BC3_UNORM_SRGB, {{bc3_data, 2 * 16}}};
    static const struct texture bc7_texture_srgb = {8, 8, 1, 1, DXGI_FORMAT_BC7_UNORM_SRGB, {{bc7_data, 2 * 16}}};
    static const struct texture bc1_typeless = {8, 8, 1, 1, DXGI_FORMAT_BC1_TYPELESS, {{bc1_data, 2 * 8}}};
    static const struct texture bc2_typeless = {8, 8, 1, 1, DXGI_FORMAT_BC2_TYPELESS, {{bc2_data, 2 * 16}}};
    static const struct texture bc3_typeless = {8, 8, 1, 1, DXGI_FORMAT_BC3_TYPELESS, {{bc3_data, 2 * 16}}};
    static const struct texture sint8_texture = {4, 4, 1, 1, DXGI_FORMAT_R8G8B8A8_SINT,
            {{rgba_level_0, 4 * sizeof(*rgba_level_0)}}};
    static const struct texture uint8_texture = {4, 4, 1, 1, DXGI_FORMAT_R8G8B8A8_UINT,
            {{rgba_level_0, 4 * sizeof(*rgba_level_0)}}};
    static const struct texture array_2d_texture =
    {
        4, 4, 1, 3, DXGI_FORMAT_R8G8B8A8_UNORM,
        {
            {red_data,   6 * sizeof(*red_data)},
            {green_data, 4 * sizeof(*green_data)},
            {blue_data,  5 * sizeof(*blue_data)},
        }
    };
    static const struct texture r32f_float = {4, 4, 1, 1, DXGI_FORMAT_R32_FLOAT,
        {{r32_float, 4 * sizeof(*r32_float)}}};
    static const struct texture r32f_typeless = {4, 4, 1, 1, DXGI_FORMAT_R32_TYPELESS,
        {{r32_float, 4 * sizeof(*r32_float)}}};
    static const struct texture r32u_typeless = {4, 4, 1, 1, DXGI_FORMAT_R32_TYPELESS,
        {{r32_uint, 4 * sizeof(*r32_uint)}}};
    static const struct texture r8g8_snorm = {4, 4, 1, 1, DXGI_FORMAT_R8G8_SNORM,
        {{r8g8_data, 4 * sizeof(*r8g8_data)}}};
    static const struct texture r9g9b9e5_texture = {4, 4, 1, 1, DXGI_FORMAT_R9G9B9E5_SHAREDEXP,
        {{r9g9b9e5_data, 4 * sizeof(*r9g9b9e5_data)}}};
    static const DWORD red_colors[] =
    {
        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff,
        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff,
        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff,
        0xff0000ff, 0xff0000ff, 0xff0000ff, 0xff0000ff,
    };
    static const DWORD blue_colors[] =
    {
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
    };
    static const DWORD level_1_colors[] =
    {
        0xffffffff, 0xffffffff, 0xff0000ff, 0xff0000ff,
        0xffffffff, 0xffffffff, 0xff0000ff, 0xff0000ff,
        0xff000000, 0xff000000, 0xff00ff00, 0xff00ff00,
        0xff000000, 0xff000000, 0xff00ff00, 0xff00ff00,
    };
    static const DWORD lerp_1_2_colors[] =
    {
        0xffff7f7f, 0xffff7f7f, 0xff7f007f, 0xff7f007f,
        0xffff7f7f, 0xffff7f7f, 0xff7f007f, 0xff7f007f,
        0xff7f0000, 0xff7f0000, 0xff7f7f00, 0xff7f7f00,
        0xff7f0000, 0xff7f0000, 0xff7f7f00, 0xff7f7f00,
    };
    static const DWORD level_2_colors[] =
    {
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
        0xffff0000, 0xffff0000, 0xffff0000, 0xffff0000,
    };
    static const DWORD srgb_colors[] =
    {
        0x00000001, 0xffffffff, 0xff000000, 0x7f363636,
        0xff000000, 0xff010408, 0xff010101, 0xff37475a,
        0xff708cba, 0xffdee0e2, 0xfff3fbfd, 0xff1a1801,
        0xffa9ff00, 0xff93f159, 0xff670067, 0xffb8177f,
    };
    static const DWORD a8_colors[] =
    {
        0x00000000, 0x10000000, 0x20000000, 0x30000000,
        0x40000000, 0x50000000, 0x60000000, 0x70000000,
        0x80000000, 0x90000000, 0xa0000000, 0xb0000000,
        0xc0000000, 0xd0000000, 0xe0000000, 0xf0000000,
    };
    static const DWORD bc_colors[] =
    {
        0xff0000ff, 0xff0000ff, 0xff00ff00, 0xff00ff00,
        0xff0000ff, 0xff0000ff, 0xff00ff00, 0xff00ff00,
        0xffff0000, 0xffff0000, 0xffffffff, 0xffffffff,
        0xffff0000, 0xffff0000, 0xffffffff, 0xffffffff,
    };
    static const DWORD bc4_colors[] =
    {
        0xff000026, 0xff000010, 0xff00007f, 0xff00007f,
        0xff000010, 0xff000010, 0xff00007f, 0xff00007f,
        0xff0000ff, 0xff0000ff, 0xff000000, 0xff000000,
        0xff0000ff, 0xff0000ff, 0xff000000, 0xff000000,
    };
    static const DWORD bc5_colors[] =
    {
        0xff002626, 0xff001010, 0xff007f7f, 0xff007f7f,
        0xff001010, 0xff001010, 0xff007f7f, 0xff007f7f,
        0xff00ffff, 0xff00ffff, 0xff000000, 0xff000000,
        0xff00ffff, 0xff00ffff, 0xff000000, 0xff000000,
    };
    static const DWORD bc7_colors[] =
    {
        0xff0000fc, 0xff0000fc, 0xff00fc00, 0xff00fc00,
        0xff0000fc, 0xff0000fc, 0xff00fc00, 0xff00fc00,
        0xfffc0000, 0xfffc0000, 0xffffffff, 0xffffffff,
        0xfffc0000, 0xfffc0000, 0xffffffff, 0xffffffff,
    };
    static const DWORD sint8_colors[] =
    {
        0x7e80807e, 0x7e807e7e, 0x7e807e80, 0x7e7e7e80,
        0x7e7e8080, 0x7e7e7f7f, 0x7e808080, 0x7effffff,
        0x7e7e7e7e, 0x7e7e7e7e, 0x7e7e7e7e, 0x7e808080,
        0x7e7e7e7e, 0x7e7f7f7f, 0x7e7f7f7f, 0x7e7f7f7f,
    };
    static const DWORD snorm_colors[] =
    {
        0xff000000, 0xff000000, 0xff000000, 0xff00ff00,
        0xff000406, 0xff000020, 0xff001618, 0xff000000,
        0xff000000, 0xff000000, 0xff000000, 0xff00b5df,
        0xff000000, 0xff000000, 0xff000000, 0xff0000b7,
    };
    static const DWORD r32f_colors[] =
    {
        0xff000000, 0xff0000ff, 0xff00007f, 0xff00007f,
        0xff0000ff, 0xff000000, 0xff000000, 0xff0000bf,
        0xff000000, 0xff0000ff, 0xff00007f, 0xff000040,
        0xff0000ff, 0xff000000, 0xff000000, 0xff0000bf,
    };
    static const DWORD r32u_colors[16] =
    {
        0x01000000, 0x01000001, 0x01000002, 0x01000003,
        0x01000064, 0x010000c8, 0x010000ff, 0x01000080,
        0x01000028, 0x0100001e, 0x01000014, 0x0100000a,
        0x010000fa, 0x010000d2, 0x0100009b, 0x010000be,
    };
    static const DWORD r9g9b9e5_colors[16] =
    {
        0xff0000ff, 0xff00ff00, 0xffff0000, 0xffff00ff,
        0xff00007f, 0xff007f00, 0xff7f0000, 0xff007f7f,
        0xff00004c, 0xff004c00, 0xff4c0000, 0xff4c004c,
        0xff000033, 0xff003300, 0xff330000, 0xff333333,
    };
    static const DWORD zero_colors[4 * 4] = {0};
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};

    static const struct texture_test
    {
        const struct shader *ps;
        const struct texture *texture;
        D3D11_FILTER filter;
        float lod_bias;
        float min_lod;
        float max_lod;
        float ps_constant;
        const DWORD *expected_colors;
    }
    texture_tests[] =
    {
#define POINT        D3D11_FILTER_MIN_MAG_MIP_POINT
#define POINT_LINEAR D3D11_FILTER_MIN_MAG_POINT_MIP_LINEAR
#define MIP_MAX      D3D11_FLOAT32_MAX
        {&ps_ld,              &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  0.0f, rgba_level_0},
        {&ps_ld,              &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  1.0f, level_1_colors},
        {&ps_ld,              &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  2.0f, level_2_colors},
        {&ps_ld,              &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  3.0f, zero_colors},
        {&ps_ld,              &srgb_texture,     POINT,        0.0f, 0.0f,    0.0f,  0.0f, srgb_colors},
        {&ps_ld,              &bc1_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_ld,              &bc1_texture,      POINT,        0.0f, 0.0f,    0.0f,  1.0f, zero_colors},
        {&ps_ld,              &bc2_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_ld,              &bc2_texture,      POINT,        0.0f, 0.0f,    0.0f,  1.0f, zero_colors},
        {&ps_ld,              &bc3_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_ld,              &bc3_texture,      POINT,        0.0f, 0.0f,    0.0f,  1.0f, zero_colors},
        {&ps_ld,              &bc1_texture_srgb, POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_ld,              &bc2_texture_srgb, POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_ld,              &bc3_texture_srgb, POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_ld,              &bc4_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc4_colors},
        {&ps_ld,              &bc5_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc5_colors},
        {&ps_ld,              &bc6h_u_texture,   POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_ld,              &bc6h_s_texture,   POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_ld,              &bc7_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc7_colors},
        {&ps_ld,              &bc7_texture_srgb, POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc7_colors},
        {&ps_ld,              &r9g9b9e5_texture, POINT,        0.0f, 0.0f,    0.0f,  0.0f, r9g9b9e5_colors},
        {&ps_ld,              NULL,              POINT,        0.0f, 0.0f,    0.0f,  0.0f, zero_colors},
        {&ps_ld,              NULL,              POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, zero_colors},
        {&ps_ld_sint8,        &sint8_texture,    POINT,        0.0f, 0.0f,    0.0f,  0.0f, sint8_colors},
        {&ps_ld_uint8,        &uint8_texture,    POINT,        0.0f, 0.0f,    0.0f,  0.0f, rgba_level_0},
        {&ps_sample,          &bc1_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_sample,          &bc2_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_sample,          &bc3_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_sample,          &bc4_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc4_colors},
        {&ps_sample,          &bc5_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc5_colors},
        {&ps_sample,          &bc6h_u_texture,   POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_sample,          &bc6h_s_texture,   POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc_colors},
        {&ps_sample,          &bc7_texture,      POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc7_colors},
        {&ps_sample,          &bc7_texture_srgb, POINT,        0.0f, 0.0f,    0.0f,  0.0f, bc7_colors},
        {&ps_sample,          &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  0.0f, rgba_level_0},
        {&ps_sample,          &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, rgba_level_0},
        {&ps_sample,          &rgba_texture,     POINT,        2.0f, 0.0f, MIP_MAX,  0.0f, rgba_level_0},
        {&ps_sample,          &rgba_texture,     POINT,        8.0f, 0.0f, MIP_MAX,  0.0f, level_1_colors},
        {&ps_sample,          &srgb_texture,     POINT,        0.0f, 0.0f,    0.0f,  0.0f, srgb_colors},
        {&ps_sample,          &a8_texture,       POINT,        0.0f, 0.0f,    0.0f,  0.0f, a8_colors},
        {&ps_sample,          &r9g9b9e5_texture, POINT,        0.0f, 0.0f,    0.0f,  0.0f, r9g9b9e5_colors},
        {&ps_sample,          NULL,              POINT,        0.0f, 0.0f,    0.0f,  0.0f, zero_colors},
        {&ps_sample,          NULL,              POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, zero_colors},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, rgba_level_0},
        {&ps_sample_b,        &rgba_texture,     POINT,        8.0f, 0.0f, MIP_MAX,  0.0f, level_1_colors},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  8.0f, level_1_colors},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  8.4f, level_1_colors},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  8.5f, level_2_colors},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  9.0f, level_2_colors},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f,    2.0f,  1.0f, rgba_level_0},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f,    2.0f,  9.0f, level_2_colors},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f,    1.0f,  9.0f, level_1_colors},
        {&ps_sample_b,        &rgba_texture,     POINT,        0.0f, 0.0f,    0.0f,  9.0f, rgba_level_0},
        {&ps_sample_b,        NULL,              POINT,        0.0f, 0.0f,    0.0f,  0.0f, zero_colors},
        {&ps_sample_b,        NULL,              POINT,        0.0f, 0.0f,    0.0f,  1.0f, zero_colors},
        {&ps_sample_b,        NULL,              POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, zero_colors},
        {&ps_sample_b,        NULL,              POINT,        0.0f, 0.0f, MIP_MAX,  1.0f, zero_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX, -1.0f, rgba_level_0},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, rgba_level_0},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  0.4f, rgba_level_0},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  0.5f, level_1_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  1.0f, level_1_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  1.4f, level_1_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  1.5f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  2.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  3.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        0.0f, 0.0f, MIP_MAX,  4.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 0.0f, 0.0f, MIP_MAX,  1.5f, lerp_1_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 0.0f, MIP_MAX, -2.0f, rgba_level_0},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 0.0f, MIP_MAX, -1.0f, level_1_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 0.0f, MIP_MAX,  0.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 0.0f, MIP_MAX,  1.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 0.0f, MIP_MAX,  1.5f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 2.0f,    2.0f, -9.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 2.0f,    2.0f, -1.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 2.0f,    2.0f,  0.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 2.0f,    2.0f,  1.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT_LINEAR, 2.0f, 2.0f,    2.0f,  9.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        2.0f, 2.0f,    2.0f, -9.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        2.0f, 2.0f,    2.0f, -1.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        2.0f, 2.0f,    2.0f,  0.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        2.0f, 2.0f,    2.0f,  1.0f, level_2_colors},
        {&ps_sample_l,        &rgba_texture,     POINT,        2.0f, 2.0f,    2.0f,  9.0f, level_2_colors},
        {&ps_sample_l,        NULL,              POINT,        2.0f, 2.0f,    0.0f,  0.0f, zero_colors},
        {&ps_sample_l,        NULL,              POINT,        2.0f, 2.0f,    0.0f,  1.0f, zero_colors},
        {&ps_sample_l,        NULL,              POINT,        2.0f, 2.0f, MIP_MAX,  0.0f, zero_colors},
        {&ps_sample_l,        NULL,              POINT,        2.0f, 2.0f, MIP_MAX,  1.0f, zero_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX, -9.0f, red_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX, -1.0f, red_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, red_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  0.4f, red_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  0.5f, red_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  1.0f, green_data},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  1.4f, green_data},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  2.0f, blue_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  2.1f, blue_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  3.0f, blue_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  3.1f, blue_colors},
        {&ps_sample_2d_array, &array_2d_texture, POINT,        0.0f, 0.0f, MIP_MAX,  9.0f, blue_colors},
        {&ps_sample_2d_array, NULL,              POINT,        0.0f, 0.0f,    0.0f,  0.0f, zero_colors},
        {&ps_sample_2d_array, NULL,              POINT,        0.0f, 0.0f,    0.0f,  1.0f, zero_colors},
        {&ps_sample_2d_array, NULL,              POINT,        0.0f, 0.0f,    0.0f,  2.0f, zero_colors},
        {&ps_sample_2d_array, NULL,              POINT,        0.0f, 0.0f, MIP_MAX,  0.0f, zero_colors},
        {&ps_sample_2d_array, NULL,              POINT,        0.0f, 0.0f, MIP_MAX,  1.0f, zero_colors},
        {&ps_sample_2d_array, NULL,              POINT,        0.0f, 0.0f, MIP_MAX,  2.0f, zero_colors},
#undef POINT
#undef POINT_LINEAR
#undef MIP_MAX
    };
    static const struct srv_test
    {
        const struct shader *ps;
        const struct texture *texture;
        struct srv_desc srv_desc;
        float ps_constant;
        const DWORD *expected_colors;
    }
    srv_tests[] =
    {
#define TEX_2D              D3D11_SRV_DIMENSION_TEXTURE2D
#define TEX_2D_ARRAY        D3D11_SRV_DIMENSION_TEXTURE2DARRAY
#define BC1_UNORM           DXGI_FORMAT_BC1_UNORM
#define BC1_UNORM_SRGB      DXGI_FORMAT_BC1_UNORM_SRGB
#define BC2_UNORM           DXGI_FORMAT_BC2_UNORM
#define BC2_UNORM_SRGB      DXGI_FORMAT_BC2_UNORM_SRGB
#define BC3_UNORM           DXGI_FORMAT_BC3_UNORM
#define BC3_UNORM_SRGB      DXGI_FORMAT_BC3_UNORM_SRGB
#define R8G8B8A8_UNORM_SRGB DXGI_FORMAT_R8G8B8A8_UNORM_SRGB
#define R8G8B8A8_UNORM      DXGI_FORMAT_R8G8B8A8_UNORM
#define R8G8_SNORM          DXGI_FORMAT_R8G8_SNORM
#define R32_FLOAT           DXGI_FORMAT_R32_FLOAT
#define R32_UINT            DXGI_FORMAT_R32_UINT
#define FMT_UNKNOWN         DXGI_FORMAT_UNKNOWN
        {&ps_sample,          &bc1_typeless,     {BC1_UNORM,           TEX_2D,       0, 1},       0.0f, bc_colors},
        {&ps_sample,          &bc1_typeless,     {BC1_UNORM_SRGB,      TEX_2D,       0, 1},       0.0f, bc_colors},
        {&ps_sample,          &bc2_typeless,     {BC2_UNORM,           TEX_2D,       0, 1},       0.0f, bc_colors},
        {&ps_sample,          &bc2_typeless,     {BC2_UNORM_SRGB,      TEX_2D,       0, 1},       0.0f, bc_colors},
        {&ps_sample,          &bc3_typeless,     {BC3_UNORM,           TEX_2D,       0, 1},       0.0f, bc_colors},
        {&ps_sample,          &bc3_typeless,     {BC3_UNORM_SRGB,      TEX_2D,       0, 1},       0.0f, bc_colors},
        {&ps_sample,          &srgb_typeless,    {R8G8B8A8_UNORM_SRGB, TEX_2D,       0, 1},       0.0f, srgb_colors},
        {&ps_sample,          &srgb_typeless,    {R8G8B8A8_UNORM,      TEX_2D,       0, 1},       0.0f, srgb_data},
        {&ps_sample,          &r32f_typeless,    {R32_FLOAT,           TEX_2D,       0, 1},       0.0f, r32f_colors},
        {&ps_sample,          &r32f_float,       {R32_FLOAT,           TEX_2D,       0, 1},       0.0f, r32f_colors},
        {&ps_sample,          &r8g8_snorm,       {R8G8_SNORM,          TEX_2D,       0, 1},       0.0f, snorm_colors},
        {&ps_sample,          &array_2d_texture, {FMT_UNKNOWN,         TEX_2D,       0, 1},       0.0f, red_colors},
        {&ps_sample_2d_array, &array_2d_texture, {FMT_UNKNOWN,         TEX_2D_ARRAY, 0, 1, 0, 1}, 0.0f, red_colors},
        {&ps_sample_2d_array, &array_2d_texture, {FMT_UNKNOWN,         TEX_2D_ARRAY, 0, 1, 1, 1}, 0.0f, green_data},
        {&ps_sample_2d_array, &array_2d_texture, {FMT_UNKNOWN,         TEX_2D_ARRAY, 0, 1, 2, 1}, 0.0f, blue_colors},
        {&ps_ld_uint8,        &r32u_typeless,    {R32_UINT,            TEX_2D,       0, 1},       0.0f, r32u_colors},
#undef TEX_2D
#undef TEX_2D_ARRAY
#undef BC1_UNORM
#undef BC1_UNORM_SRGB
#undef BC2_UNORM
#undef BC2_UNORM_SRGB
#undef BC3_UNORM
#undef BC3_UNORM_SRGB
#undef R8G8B8A8_UNORM_SRGB
#undef R8G8B8A8_UNORM
#undef R8G8_SNORM
#undef R32_FLOAT
#undef R32_UINT
#undef FMT_UNKNOWN
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;
    feature_level = ID3D11Device_GetFeatureLevel(device);

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(ps_constant), NULL);

    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = D3D11_FLOAT32_MAX;

    ps = NULL;
    srv = NULL;
    sampler = NULL;
    texture = NULL;
    current_ps = NULL;
    current_texture = NULL;
    for (i = 0; i < ARRAY_SIZE(texture_tests); ++i)
    {
        const struct texture_test *test = &texture_tests[i];

        if (test->texture && (test->texture->format == DXGI_FORMAT_BC7_UNORM
                || test->texture->format == DXGI_FORMAT_BC7_UNORM_SRGB)
                && feature_level < D3D_FEATURE_LEVEL_11_0)
        {
            skip("Feature level >= 11.0 is required for BC7 tests.\n");
            continue;
        }

        if (test->texture && (test->texture->format == DXGI_FORMAT_BC6H_UF16
                || test->texture->format == DXGI_FORMAT_BC6H_SF16)
                && feature_level < D3D_FEATURE_LEVEL_11_0)
        {
            skip("Feature level >= 11.0 is required for BC6H tests.\n");
            continue;
        }

        if (current_ps != test->ps)
        {
            if (ps)
                ID3D11PixelShader_Release(ps);

            current_ps = test->ps;

            hr = ID3D11Device_CreatePixelShader(device, current_ps->code, current_ps->size, NULL, &ps);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

            ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
        }

        if (current_texture != test->texture)
        {
            if (texture)
                ID3D11Texture2D_Release(texture);
            if (srv)
                ID3D11ShaderResourceView_Release(srv);

            current_texture = test->texture;

            if (current_texture)
            {
                texture_desc.Width = current_texture->width;
                texture_desc.Height = current_texture->height;
                texture_desc.MipLevels = current_texture->miplevel_count;
                texture_desc.ArraySize = current_texture->array_size;
                texture_desc.Format = current_texture->format;

                hr = ID3D11Device_CreateTexture2D(device, &texture_desc, current_texture->data, &texture);
                ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

                hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, NULL, &srv);
                ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            }
            else
            {
                texture = NULL;
                srv = NULL;
            }

            ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);
        }

        if (!sampler || (sampler_desc.Filter != test->filter
                || sampler_desc.MipLODBias != test->lod_bias
                || sampler_desc.MinLOD != test->min_lod
                || sampler_desc.MaxLOD != test->max_lod))
        {
            if (sampler)
                ID3D11SamplerState_Release(sampler);

            sampler_desc.Filter = test->filter;
            sampler_desc.MipLODBias = test->lod_bias;
            sampler_desc.MinLOD = test->min_lod;
            sampler_desc.MaxLOD = test->max_lod;

            hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

            ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);
        }

        ps_constant.x = test->ps_constant;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &ps_constant, 0, 0);

        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

        draw_quad(&test_context);

        get_texture_readback(test_context.backbuffer, 0, &rb);
        for (y = 0; y < 4; ++y)
        {
            for (x = 0; x < 4; ++x)
            {
                color = get_readback_color(&rb, 80 + x * 160, 60 + y * 120, 0);
                ok(compare_color(color, test->expected_colors[y * 4 + x], 2),
                        "Test %u: Got unexpected colour 0x%08lx at (%u, %u).\n", i, color, x, y);
            }
        }
        release_resource_readback(&rb);
    }
    if (srv)
        ID3D11ShaderResourceView_Release(srv);
    ID3D11SamplerState_Release(sampler);
    if (texture)
        ID3D11Texture2D_Release(texture);
    ID3D11PixelShader_Release(ps);

    if (is_warp_device(device) && feature_level < D3D_FEATURE_LEVEL_11_0)
    {
        win_skip("SRV tests are broken on WARP.\n");
        ID3D11Buffer_Release(cb);
        release_test_context(&test_context);
        return;
    }

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = D3D11_FLOAT32_MAX;

    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);

    ps = NULL;
    srv = NULL;
    texture = NULL;
    current_ps = NULL;
    current_texture = NULL;
    for (i = 0; i < ARRAY_SIZE(srv_tests); ++i)
    {
        const struct srv_test *test = &srv_tests[i];

        if (current_ps != test->ps)
        {
            if (ps)
                ID3D11PixelShader_Release(ps);

            current_ps = test->ps;

            hr = ID3D11Device_CreatePixelShader(device, current_ps->code, current_ps->size, NULL, &ps);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

            ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
        }

        if (current_texture != test->texture)
        {
            if (texture)
                ID3D11Texture2D_Release(texture);

            current_texture = test->texture;

            texture_desc.Width = current_texture->width;
            texture_desc.Height = current_texture->height;
            texture_desc.MipLevels = current_texture->miplevel_count;
            texture_desc.ArraySize = current_texture->array_size;
            texture_desc.Format = current_texture->format;

            hr = ID3D11Device_CreateTexture2D(device, &texture_desc, current_texture->data, &texture);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        }

        if (srv)
            ID3D11ShaderResourceView_Release(srv);

        get_srv_desc(&srv_desc, &test->srv_desc);
        hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, &srv_desc, &srv);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);

        ps_constant.x = test->ps_constant;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &ps_constant, 0, 0);

        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

        draw_quad(&test_context);

        get_texture_readback(test_context.backbuffer, 0, &rb);
        for (y = 0; y < 4; ++y)
        {
            for (x = 0; x < 4; ++x)
            {
                color = get_readback_color(&rb, 80 + x * 160, 60 + y * 120, 0);
                ok(compare_color(color, test->expected_colors[y * 4 + x], 1),
                        "Test %u: Got unexpected colour 0x%08lx at (%u, %u).\n", i, color, x, y);
            }
        }
        release_resource_readback(&rb);
    }
    ID3D11PixelShader_Release(ps);
    ID3D11Texture2D_Release(texture);
    ID3D11ShaderResourceView_Release(srv);
    ID3D11SamplerState_Release(sampler);

    ID3D11Buffer_Release(cb);
    release_test_context(&test_context);
}

static void test_cube_maps(void)
{
    struct shader
    {
        const DWORD *code;
        size_t size;
    };

    unsigned int i, j, sub_resource_idx, sub_resource_count;
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    const struct shader *current_ps;
    D3D_FEATURE_LEVEL feature_level;
    ID3D11ShaderResourceView *srv;
    ID3D11DeviceContext *context;
    ID3D11Texture2D *rtv_texture;
    ID3D11RenderTargetView *rtv;
    struct vec4 expected_result;
    ID3D11Resource *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    float data[64 * 64];
    ID3D11Buffer *cb;
    HRESULT hr;
    RECT rect;
    struct
    {
        unsigned int face;
        unsigned int level;
        unsigned int cube;
        unsigned int padding;
    } constant;

    static const DWORD ps_cube_code[] =
    {
#if 0
        TextureCube t;
        SamplerState s;

        uint face;
        uint level;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;
            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;

            float3 coord;
            switch (face)
            {
                case 0:
                    coord = float3(1.0f, p.x, p.y);
                    break;
                case 1:
                    coord = float3(-1.0f, p.x, p.y);
                    break;
                case 2:
                    coord = float3(p.x, 1.0f, p.y);
                    break;
                case 3:
                    coord = float3(p.x, -1.0f, p.y);
                    break;
                case 4:
                    coord = float3(p.x, p.y, 1.0f);
                    break;
                case 5:
                default:
                    coord = float3(p.x, p.y, -1.0f);
                    break;
            }
            return t.SampleLevel(s, coord, level);
        }
#endif
        0x43425844, 0x039aee18, 0xfd630453, 0xb884cf0f, 0x10100744, 0x00000001, 0x00000310, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000274, 0x00000040,
        0x0000009d, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000, 0x00000000,
        0x04003058, 0x00107000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0400004c, 0x0020800a, 0x00000000,
        0x00000000, 0x03000006, 0x00004001, 0x00000000, 0x05000036, 0x00100012, 0x00000000, 0x00004001,
        0x3f800000, 0x0a000038, 0x00100062, 0x00000000, 0x00101106, 0x00000000, 0x00004002, 0x00000000,
        0x3acccccd, 0x3b088889, 0x00000000, 0x01000002, 0x03000006, 0x00004001, 0x00000001, 0x05000036,
        0x00100012, 0x00000000, 0x00004001, 0xbf800000, 0x0a000038, 0x00100062, 0x00000000, 0x00101106,
        0x00000000, 0x00004002, 0x00000000, 0x3acccccd, 0x3b088889, 0x00000000, 0x01000002, 0x03000006,
        0x00004001, 0x00000002, 0x0a000038, 0x00100052, 0x00000000, 0x00101106, 0x00000000, 0x00004002,
        0x3acccccd, 0x00000000, 0x3b088889, 0x00000000, 0x05000036, 0x00100022, 0x00000000, 0x00004001,
        0x3f800000, 0x01000002, 0x03000006, 0x00004001, 0x00000003, 0x0a000038, 0x00100052, 0x00000000,
        0x00101106, 0x00000000, 0x00004002, 0x3acccccd, 0x00000000, 0x3b088889, 0x00000000, 0x05000036,
        0x00100022, 0x00000000, 0x00004001, 0xbf800000, 0x01000002, 0x03000006, 0x00004001, 0x00000004,
        0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889,
        0x00000000, 0x00000000, 0x05000036, 0x00100042, 0x00000000, 0x00004001, 0x3f800000, 0x01000002,
        0x0100000a, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x05000036, 0x00100042, 0x00000000, 0x00004001, 0xbf800000,
        0x01000002, 0x01000017, 0x06000056, 0x00100082, 0x00000000, 0x0020801a, 0x00000000, 0x00000000,
        0x0b000048, 0x001020f2, 0x00000000, 0x00100246, 0x00000000, 0x00107e46, 0x00000000, 0x00106000,
        0x00000000, 0x0010003a, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_cube = {ps_cube_code, sizeof(ps_cube_code)};
    static const DWORD ps_cube_array_code[] =
    {
#if 0
        TextureCubeArray t;
        SamplerState s;

        uint face;
        uint level;
        uint cube;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;
            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;

            float3 coord;
            switch (face)
            {
                case 0:
                    coord = float3(1.0f, p.x, p.y);
                    break;
                case 1:
                    coord = float3(-1.0f, p.x, p.y);
                    break;
                case 2:
                    coord = float3(p.x, 1.0f, p.y);
                    break;
                case 3:
                    coord = float3(p.x, -1.0f, p.y);
                    break;
                case 4:
                    coord = float3(p.x, p.y, 1.0f);
                    break;
                case 5:
                default:
                    coord = float3(p.x, p.y, -1.0f);
                    break;
            }
            return t.SampleLevel(s, float4(coord, cube), level);
        }
#endif
        0x43425844, 0xb8d5b94a, 0xdb4be034, 0x183aed19, 0xad4af415, 0x00000001, 0x00000328, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x0000028c, 0x00000041,
        0x000000a3, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000,
        0x00000000, 0x04005058, 0x00107000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000002, 0x0400004c, 0x0020800a,
        0x00000000, 0x00000000, 0x03000006, 0x00004001, 0x00000000, 0x05000036, 0x00100012, 0x00000000,
        0x00004001, 0x3f800000, 0x0a000038, 0x00100062, 0x00000000, 0x00101106, 0x00000000, 0x00004002,
        0x00000000, 0x3acccccd, 0x3b088889, 0x00000000, 0x01000002, 0x03000006, 0x00004001, 0x00000001,
        0x05000036, 0x00100012, 0x00000000, 0x00004001, 0xbf800000, 0x0a000038, 0x00100062, 0x00000000,
        0x00101106, 0x00000000, 0x00004002, 0x00000000, 0x3acccccd, 0x3b088889, 0x00000000, 0x01000002,
        0x03000006, 0x00004001, 0x00000002, 0x0a000038, 0x00100052, 0x00000000, 0x00101106, 0x00000000,
        0x00004002, 0x3acccccd, 0x00000000, 0x3b088889, 0x00000000, 0x05000036, 0x00100022, 0x00000000,
        0x00004001, 0x3f800000, 0x01000002, 0x03000006, 0x00004001, 0x00000003, 0x0a000038, 0x00100052,
        0x00000000, 0x00101106, 0x00000000, 0x00004002, 0x3acccccd, 0x00000000, 0x3b088889, 0x00000000,
        0x05000036, 0x00100022, 0x00000000, 0x00004001, 0xbf800000, 0x01000002, 0x03000006, 0x00004001,
        0x00000004, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x05000036, 0x00100042, 0x00000000, 0x00004001, 0x3f800000,
        0x01000002, 0x0100000a, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002,
        0x3acccccd, 0x3b088889, 0x00000000, 0x00000000, 0x05000036, 0x00100042, 0x00000000, 0x00004001,
        0xbf800000, 0x01000002, 0x01000017, 0x06000056, 0x00100032, 0x00000001, 0x00208a66, 0x00000000,
        0x00000000, 0x05000036, 0x00100082, 0x00000000, 0x0010000a, 0x00000001, 0x0b000048, 0x001020f2,
        0x00000000, 0x00100e46, 0x00000000, 0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0010001a,
        0x00000001, 0x0100003e,
    };
    static const struct shader ps_cube_array = {ps_cube_array_code, sizeof(ps_cube_array_code)};
    static const struct ps_test
    {
        const struct shader *ps;
        unsigned int miplevel_count;
        unsigned int array_size;
    }
    ps_tests[] =
    {
        {&ps_cube, 1, 6},
        {&ps_cube, 2, 6},
        {&ps_cube, 3, 6},
        {&ps_cube, 0, 6},

        {&ps_cube_array, 1, 12},
        {&ps_cube_array, 2, 12},
        {&ps_cube_array, 3, 12},
        {&ps_cube_array, 0, 12},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;
    feature_level = ID3D11Device_GetFeatureLevel(device);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &rtv_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)rtv_texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    memset(&constant, 0, sizeof(constant));
    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(constant), &constant);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    ps = NULL;
    current_ps = NULL;
    for (i = 0; i < ARRAY_SIZE(ps_tests); ++i)
    {
        const struct ps_test *test = &ps_tests[i];

        if (test->array_size / 6 > 1 && feature_level < D3D_FEATURE_LEVEL_10_1)
        {
            skip("Test %u: Cube map array textures require feature level 10_1.\n", i);
            continue;
        }

        if (current_ps != test->ps)
        {
            if (ps)
                ID3D11PixelShader_Release(ps);

            current_ps = test->ps;

            hr = ID3D11Device_CreatePixelShader(device, current_ps->code, current_ps->size, NULL, &ps);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
        }

        if (!test->miplevel_count)
        {
            srv = NULL;
            ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);

            memset(&expected_result, 0, sizeof(expected_result));

            memset(&constant, 0, sizeof(constant));
            ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
            draw_quad(&test_context);
            check_texture_vec4(rtv_texture, &expected_result, 0);
            constant.level = 1;
            ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
            draw_quad(&test_context);
            check_texture_vec4(rtv_texture, &expected_result, 0);
            continue;
        }

        texture_desc.Width = 64;
        texture_desc.Height = 64;
        texture_desc.MipLevels = test->miplevel_count;
        texture_desc.ArraySize = test->array_size;
        texture_desc.Format = DXGI_FORMAT_R32_FLOAT;
        texture_desc.SampleDesc.Count = 1;
        texture_desc.SampleDesc.Quality = 0;
        texture_desc.Usage = D3D11_USAGE_DEFAULT;
        texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
        texture_desc.CPUAccessFlags = 0;
        texture_desc.MiscFlags = D3D11_RESOURCE_MISC_TEXTURECUBE;
        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, (ID3D11Texture2D **)&texture);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        hr = ID3D11Device_CreateShaderResourceView(device, texture, NULL, &srv);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);

        sub_resource_count = texture_desc.MipLevels * texture_desc.ArraySize;
        for (sub_resource_idx = 0; sub_resource_idx < sub_resource_count; ++sub_resource_idx)
        {
            for (j = 0; j < ARRAY_SIZE(data); ++j)
                data[j] = sub_resource_idx;
            ID3D11DeviceContext_UpdateSubresource(context, texture, sub_resource_idx, NULL, data,
                    texture_desc.Width * sizeof(*data), 0);
        }

        expected_result.y = expected_result.z = 0.0f;
        expected_result.w = 1.0f;
        for (sub_resource_idx = 0; sub_resource_idx < sub_resource_count; ++sub_resource_idx)
        {
            constant.face = (sub_resource_idx / texture_desc.MipLevels) % 6;
            constant.level = sub_resource_idx % texture_desc.MipLevels;
            constant.cube = (sub_resource_idx / texture_desc.MipLevels) / 6;
            ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);

            draw_quad(&test_context);
            expected_result.x = sub_resource_idx;
            /* Avoid testing values affected by seamless cube map filtering. */
            SetRect(&rect, 100, 100, 540, 380);
            check_texture_sub_resource_vec4(rtv_texture, 0, &rect, &expected_result, 0);
        }

        ID3D11Resource_Release(texture);
        ID3D11ShaderResourceView_Release(srv);
    }
    ID3D11PixelShader_Release(ps);

    ID3D11Buffer_Release(cb);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11Texture2D_Release(rtv_texture);
    release_test_context(&test_context);
}

static void test_depth_stencil_sampling(void)
{
    ID3D11PixelShader *ps_cmp, *ps_depth, *ps_stencil, *ps_depth_stencil;
    ID3D11ShaderResourceView *depth_srv, *stencil_srv;
    ID3D11SamplerState *cmp_sampler, *sampler;
    D3D11_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D11_DEPTH_STENCIL_VIEW_DESC dsv_desc;
    struct d3d11_test_context test_context;
    ID3D11Texture2D *texture, *rt_texture;
    D3D11_TEXTURE2D_DESC texture_desc;
    D3D11_SAMPLER_DESC sampler_desc;
    ID3D11DeviceContext *context;
    ID3D11DepthStencilView *dsv;
    ID3D11RenderTargetView *rtv;
    struct vec4 ps_constant;
    ID3D11Device *device;
    ID3D11Buffer *cb;
    unsigned int i;
    HRESULT hr;

    static const float black[] = {0.0f, 0.0f, 0.0f, 0.0f};
    static const DWORD ps_compare_code[] =
    {
#if 0
        Texture2D t;
        SamplerComparisonState s;

        float ref;

        float4 main(float4 position : SV_Position) : SV_Target
        {
            return t.SampleCmp(s, float2(position.x / 640.0f, position.y / 480.0f), ref);
        }
#endif
        0x43425844, 0xc2e0d84e, 0x0522c395, 0x9ff41580, 0xd3ca29cc, 0x00000001, 0x00000164, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x000000c8, 0x00000040,
        0x00000032, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300085a, 0x00106000, 0x00000000,
        0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0a000038, 0x00100032, 0x00000000,
        0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000, 0x0c000046,
        0x00100012, 0x00000000, 0x00100046, 0x00000000, 0x00107006, 0x00000000, 0x00106000, 0x00000000,
        0x0020800a, 0x00000000, 0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100006, 0x00000000,
        0x0100003e,
    };
    static const DWORD ps_sample_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float4 main(float4 position : SV_Position) : SV_Target
        {
            return t.Sample(s, float2(position.x / 640.0f, position.y / 480.0f));
        }
#endif
        0x43425844, 0x7472c092, 0x5548f00e, 0xf4e007f1, 0x5970429c, 0x00000001, 0x00000134, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000098, 0x00000040,
        0x00000026, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555,
        0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x09000045, 0x001020f2, 0x00000000, 0x00100046, 0x00000000,
        0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_stencil_code[] =
    {
#if 0
        Texture2D<uint4> t;

        float4 main(float4 position : SV_Position) : SV_Target
        {
            float2 s;
            t.GetDimensions(s.x, s.y);
            return t.Load(int3(float3(s.x * position.x / 640.0f, s.y * position.y / 480.0f, 0))).y;
        }
#endif
        0x43425844, 0x929fced8, 0x2cd93320, 0x0591ece3, 0xee50d04a, 0x00000001, 0x000001a0, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000104, 0x00000040,
        0x00000041, 0x04001858, 0x00107000, 0x00000000, 0x00004444, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0700003d, 0x001000f2,
        0x00000000, 0x00004001, 0x00000000, 0x00107e46, 0x00000000, 0x07000038, 0x00100032, 0x00000000,
        0x00100046, 0x00000000, 0x00101046, 0x00000000, 0x0a000038, 0x00100032, 0x00000000, 0x00100046,
        0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000, 0x0500001b, 0x00100032,
        0x00000000, 0x00100046, 0x00000000, 0x08000036, 0x001000c2, 0x00000000, 0x00004002, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x0700002d, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000,
        0x00107e46, 0x00000000, 0x05000056, 0x001020f2, 0x00000000, 0x00100556, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_depth_stencil_code[] =
    {
#if 0
        SamplerState samp;
        Texture2D depth_tex;
        Texture2D<uint4> stencil_tex;

        float main(float4 position: SV_Position) : SV_Target
        {
            float2 s, p;
            float depth, stencil;
            depth_tex.GetDimensions(s.x, s.y);
            p = float2(s.x * position.x / 640.0f, s.y * position.y / 480.0f);
            depth = depth_tex.Sample(samp, p).r;
            stencil = stencil_tex.Load(int3(float3(p.x, p.y, 0))).y;
            return depth + stencil;
        }
#endif
        0x43425844, 0x348f8377, 0x977d1ee0, 0x8cca4f35, 0xff5c5afc, 0x00000001, 0x000001fc, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x00000e01, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000160, 0x00000040,
        0x00000058, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555,
        0x04001858, 0x00107000, 0x00000001, 0x00004444, 0x04002064, 0x00101032, 0x00000000, 0x00000001,
        0x03000065, 0x00102012, 0x00000000, 0x02000068, 0x00000002, 0x0700003d, 0x001000f2, 0x00000000,
        0x00004001, 0x00000000, 0x00107e46, 0x00000000, 0x07000038, 0x00100032, 0x00000000, 0x00100046,
        0x00000000, 0x00101046, 0x00000000, 0x0a000038, 0x00100032, 0x00000000, 0x00100046, 0x00000000,
        0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000, 0x0500001b, 0x00100032, 0x00000001,
        0x00100046, 0x00000000, 0x09000045, 0x001000f2, 0x00000000, 0x00100046, 0x00000000, 0x00107e46,
        0x00000000, 0x00106000, 0x00000000, 0x08000036, 0x001000c2, 0x00000001, 0x00004002, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x0700002d, 0x001000f2, 0x00000001, 0x00100e46, 0x00000001,
        0x00107e46, 0x00000001, 0x05000056, 0x00100022, 0x00000000, 0x0010001a, 0x00000001, 0x07000000,
        0x00102012, 0x00000000, 0x0010001a, 0x00000000, 0x0010000a, 0x00000000, 0x0100003e,
    };
    static const struct test
    {
        DXGI_FORMAT typeless_format;
        DXGI_FORMAT dsv_format;
        DXGI_FORMAT depth_view_format;
        DXGI_FORMAT stencil_view_format;
    }
    tests[] =
    {
        {DXGI_FORMAT_R32G8X24_TYPELESS, DXGI_FORMAT_D32_FLOAT_S8X24_UINT,
                DXGI_FORMAT_R32_FLOAT_X8X24_TYPELESS, DXGI_FORMAT_X32_TYPELESS_G8X24_UINT},
        {DXGI_FORMAT_R32_TYPELESS, DXGI_FORMAT_D32_FLOAT,
                DXGI_FORMAT_R32_FLOAT},
        {DXGI_FORMAT_R24G8_TYPELESS, DXGI_FORMAT_D24_UNORM_S8_UINT,
                DXGI_FORMAT_R24_UNORM_X8_TYPELESS, DXGI_FORMAT_X24_TYPELESS_G8_UINT},
        {DXGI_FORMAT_R16_TYPELESS, DXGI_FORMAT_D16_UNORM,
                DXGI_FORMAT_R16_UNORM},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    if (is_amd_device(device))
    {
        /* Reads from depth/stencil shader resource views return stale values on some AMD drivers. */
        win_skip("Some AMD drivers have a bug affecting the test.\n");
        release_test_context(&test_context);
        return;
    }

    sampler_desc.Filter = D3D11_FILTER_COMPARISON_MIN_MAG_MIP_POINT;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_GREATER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = 0.0f;
    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &cmp_sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture_desc.Width = 640;
    texture_desc.Height = 480;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R32_FLOAT;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &rt_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)rt_texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);

    memset(&ps_constant, 0, sizeof(ps_constant));
    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(ps_constant), &ps_constant);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    hr = ID3D11Device_CreatePixelShader(device, ps_compare_code, sizeof(ps_compare_code), NULL, &ps_cmp);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_sample_code, sizeof(ps_sample_code), NULL, &ps_depth);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_stencil_code, sizeof(ps_stencil_code), NULL, &ps_stencil);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_depth_stencil_code, sizeof(ps_depth_stencil_code), NULL,
            &ps_depth_stencil);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        texture_desc.Format = tests[i].typeless_format;
        texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE | D3D11_BIND_DEPTH_STENCIL;
        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        dsv_desc.Format = tests[i].dsv_format;
        dsv_desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2D;
        dsv_desc.Flags = 0;
        U(dsv_desc).Texture2D.MipSlice = 0;
        hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, &dsv_desc, &dsv);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        srv_desc.Format = tests[i].depth_view_format;
        srv_desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
        U(srv_desc).Texture2D.MostDetailedMip = 0;
        U(srv_desc).Texture2D.MipLevels = 1;
        hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, &srv_desc, &depth_srv);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        ID3D11DeviceContext_PSSetShader(context, ps_cmp, NULL, 0);
        ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &depth_srv);
        ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &cmp_sampler);

        ps_constant.x = 0.5f;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0,
                NULL, &ps_constant, 0, 0);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 0.0f, 2);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 0.0f, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 1.0f, 2);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 0.5f, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 0.0f, 2);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 0.6f, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 0.0f, 2);

        ps_constant.x = 0.7f;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0,
                NULL, &ps_constant, 0, 0);

        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 1.0f, 2);

        ID3D11DeviceContext_PSSetShader(context, ps_depth, NULL, 0);
        ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 1.0f, 2);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 0.2f, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 0.2f, 2);

        if (!tests[i].stencil_view_format)
        {
            ID3D11DepthStencilView_Release(dsv);
            ID3D11ShaderResourceView_Release(depth_srv);
            ID3D11Texture2D_Release(texture);
            continue;
        }

        srv_desc.Format = tests[i].stencil_view_format;
        hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, &srv_desc, &stencil_srv);
        if (hr == E_OUTOFMEMORY)
        {
            skip("Could not create SRV for format %#x.\n", srv_desc.Format);
            ID3D11DepthStencilView_Release(dsv);
            ID3D11ShaderResourceView_Release(depth_srv);
            ID3D11Texture2D_Release(texture);
            continue;
        }
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        ID3D11DeviceContext_PSSetShader(context, ps_stencil, NULL, 0);
        ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &stencil_srv);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_STENCIL, 0.0f, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 0.0f, 0);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_STENCIL, 0.0f, 100);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 100.0f, 0);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_STENCIL, 0.0f, 255);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 255.0f, 0);

        ID3D11DeviceContext_PSSetShader(context, ps_depth_stencil, NULL, 0);
        ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &depth_srv);
        ID3D11DeviceContext_PSSetShaderResources(context, 1, 1, &stencil_srv);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 0.3f, 3);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 3.3f, 2);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 1.0f, 3);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 4.0f, 2);

        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 0.0f, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, black);
        draw_quad(&test_context);
        check_texture_float(rt_texture, 0.0f, 2);

        ID3D11DepthStencilView_Release(dsv);
        ID3D11ShaderResourceView_Release(depth_srv);
        ID3D11ShaderResourceView_Release(stencil_srv);
        ID3D11Texture2D_Release(texture);
    }

    ID3D11Buffer_Release(cb);
    ID3D11PixelShader_Release(ps_cmp);
    ID3D11PixelShader_Release(ps_depth);
    ID3D11PixelShader_Release(ps_depth_stencil);
    ID3D11PixelShader_Release(ps_stencil);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11SamplerState_Release(cmp_sampler);
    ID3D11SamplerState_Release(sampler);
    ID3D11Texture2D_Release(rt_texture);
    release_test_context(&test_context);
}

static void test_sample_c_lz(void)
{
    D3D11_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D11_DEPTH_STENCIL_VIEW_DESC dsv_desc;
    struct d3d11_test_context test_context;
    ID3D11Texture2D *texture, *rt_texture;
    D3D11_TEXTURE2D_DESC texture_desc;
    D3D11_SAMPLER_DESC sampler_desc;
    ID3D11ShaderResourceView *srv;
    ID3D11DeviceContext *context;
    ID3D11DepthStencilView *dsv;
    ID3D11RenderTargetView *rtv;
    ID3D11SamplerState *sampler;
    struct vec4 ps_constant;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *cb;
    unsigned int i;
    HRESULT hr;
    RECT rect;

    static const float clear_color[] = {0.5f, 0.5f, 0.5f, 0.5f};
    static const DWORD ps_array_code[] =
    {
#if 0
        Texture2DArray t;
        SamplerComparisonState s;

        float ref;
        float layer;

        float4 main(float4 position : SV_Position) : SV_Target
        {
            return t.SampleCmpLevelZero(s, float3(position.x / 640.0f, position.y / 480.0f, layer), ref);
        }
#endif
        0x43425844, 0xfe28b3c3, 0xdd7ef404, 0x8d5874a1, 0x984ff182, 0x00000001, 0x00000180, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x000000e4, 0x00000041,
        0x00000039, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300085a, 0x00106000,
        0x00000000, 0x04004058, 0x00107000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0a000038, 0x00100032,
        0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000,
        0x06000036, 0x00100042, 0x00000000, 0x0020801a, 0x00000000, 0x00000000, 0x0c000047, 0x00100012,
        0x00000000, 0x00100246, 0x00000000, 0x00107006, 0x00000000, 0x00106000, 0x00000000, 0x0020800a,
        0x00000000, 0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100006, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_cube_code[] =
    {
#if 0
        TextureCube t;
        SamplerComparisonState s;

        float ref;
        float face;

        float4 main(float4 position : SV_Position) : SV_Target
        {
            float2 p;
            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;

            float3 coord;
            switch ((uint)face)
            {
                case 0:
                    coord = float3(1.0f, p.x, p.y);
                    break;
                case 1:
                    coord = float3(-1.0f, p.x, p.y);
                    break;
                case 2:
                    coord = float3(p.x, 1.0f, p.y);
                    break;
                case 3:
                    coord = float3(p.x, -1.0f, p.y);
                    break;
                case 4:
                    coord = float3(p.x, p.y, 1.0f);
                    break;
                case 5:
                default:
                    coord = float3(p.x, p.y, -1.0f);
                    break;
            }

            return t.SampleCmpLevelZero(s, coord, ref);
        }
#endif
        0x43425844, 0xde5655e5, 0x1b116fa1, 0xfce9e757, 0x41c28aac, 0x00000001, 0x00000328, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x0000028c, 0x00000041,
        0x000000a3, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300085a, 0x00106000,
        0x00000000, 0x04003058, 0x00107000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0600001c, 0x00100012,
        0x00000000, 0x0020801a, 0x00000000, 0x00000000, 0x0300004c, 0x0010000a, 0x00000000, 0x03000006,
        0x00004001, 0x00000000, 0x05000036, 0x00100012, 0x00000000, 0x00004001, 0x3f800000, 0x0a000038,
        0x00100062, 0x00000000, 0x00101106, 0x00000000, 0x00004002, 0x00000000, 0x3acccccd, 0x3b088889,
        0x00000000, 0x01000002, 0x03000006, 0x00004001, 0x00000001, 0x05000036, 0x00100012, 0x00000000,
        0x00004001, 0xbf800000, 0x0a000038, 0x00100062, 0x00000000, 0x00101106, 0x00000000, 0x00004002,
        0x00000000, 0x3acccccd, 0x3b088889, 0x00000000, 0x01000002, 0x03000006, 0x00004001, 0x00000002,
        0x0a000038, 0x00100052, 0x00000000, 0x00101106, 0x00000000, 0x00004002, 0x3acccccd, 0x00000000,
        0x3b088889, 0x00000000, 0x05000036, 0x00100022, 0x00000000, 0x00004001, 0x3f800000, 0x01000002,
        0x03000006, 0x00004001, 0x00000003, 0x0a000038, 0x00100052, 0x00000000, 0x00101106, 0x00000000,
        0x00004002, 0x3acccccd, 0x00000000, 0x3b088889, 0x00000000, 0x05000036, 0x00100022, 0x00000000,
        0x00004001, 0xbf800000, 0x01000002, 0x03000006, 0x00004001, 0x00000004, 0x0a000038, 0x00100032,
        0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000, 0x00000000,
        0x05000036, 0x00100042, 0x00000000, 0x00004001, 0x3f800000, 0x01000002, 0x0100000a, 0x0a000038,
        0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd, 0x3b088889, 0x00000000,
        0x00000000, 0x05000036, 0x00100042, 0x00000000, 0x00004001, 0xbf800000, 0x01000002, 0x01000017,
        0x0c000047, 0x00100012, 0x00000000, 0x00100246, 0x00000000, 0x00107006, 0x00000000, 0x00106000,
        0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100006,
        0x00000000, 0x0100003e,
    };
    static const float depth_values[] = {1.0f, 0.0f, 0.5f, 0.6f, 0.4f, 0.1f};
    static const struct
    {
        unsigned int layer;
        float d_ref;
        float expected;
    }
    tests[] =
    {
        {0, 0.5f, 0.0f},
        {1, 0.5f, 1.0f},
        {2, 0.5f, 0.0f},
        {3, 0.5f, 0.0f},
        {4, 0.5f, 1.0f},
        {5, 0.5f, 1.0f},

        {0, 0.0f, 0.0f},
        {1, 0.0f, 0.0f},
        {2, 0.0f, 0.0f},
        {3, 0.0f, 0.0f},
        {4, 0.0f, 0.0f},
        {5, 0.0f, 0.0f},

        {0, 1.0f, 0.0f},
        {1, 1.0f, 1.0f},
        {2, 1.0f, 1.0f},
        {3, 1.0f, 1.0f},
        {4, 1.0f, 1.0f},
        {5, 1.0f, 1.0f},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    sampler_desc.Filter = D3D11_FILTER_COMPARISON_MIN_MAG_MIP_LINEAR;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_GREATER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = 10.0f;
    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_R32_FLOAT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &rt_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)rt_texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);

    memset(&ps_constant, 0, sizeof(ps_constant));
    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(ps_constant), &ps_constant);

    /* 2D array texture */
    texture_desc.Width = 32;
    texture_desc.Height = 32;
    texture_desc.MipLevels = 2;
    texture_desc.ArraySize = ARRAY_SIZE(depth_values);
    texture_desc.Format = DXGI_FORMAT_R32_TYPELESS;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE | D3D11_BIND_DEPTH_STENCIL;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(depth_values); ++i)
    {
        dsv_desc.Format = DXGI_FORMAT_D32_FLOAT;
        dsv_desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2DARRAY;
        dsv_desc.Flags = 0;
        U(dsv_desc).Texture2DArray.MipSlice = 0;
        U(dsv_desc).Texture2DArray.FirstArraySlice = i;
        U(dsv_desc).Texture2DArray.ArraySize = 1;

        hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, &dsv_desc, &dsv);
        ok(hr == S_OK, "Slice %u: Got unexpected hr %#lx.\n", i, hr);
        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, depth_values[i], 0);
        ID3D11DepthStencilView_Release(dsv);

        U(dsv_desc).Texture2DArray.MipSlice = 1;
        hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, &dsv_desc, &dsv);
        ok(hr == S_OK, "Slice %u: Got unexpected hr %#lx.\n", i, hr);
        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0);
        ID3D11DepthStencilView_Release(dsv);
    }

    srv_desc.Format = DXGI_FORMAT_R32_FLOAT;
    srv_desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2DARRAY;
    U(srv_desc).Texture2DArray.MostDetailedMip = 0;
    U(srv_desc).Texture2DArray.MipLevels = ~0u;
    U(srv_desc).Texture2DArray.FirstArraySlice = 0;
    U(srv_desc).Texture2DArray.ArraySize = ~0u;
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, &srv_desc, &srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_array_code, sizeof(ps_array_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        ps_constant.x = tests[i].d_ref;
        ps_constant.y = tests[i].layer;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0,
                NULL, &ps_constant, 0, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, clear_color);
        draw_quad(&test_context);
        check_texture_float(rt_texture, tests[i].expected, 2);
    }

    ID3D11Texture2D_Release(texture);
    ID3D11ShaderResourceView_Release(srv);
    ID3D11PixelShader_Release(ps);

    /* cube texture */
    texture_desc.MiscFlags = D3D11_RESOURCE_MISC_TEXTURECUBE;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(depth_values); ++i)
    {
        dsv_desc.Format = DXGI_FORMAT_D32_FLOAT;
        dsv_desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2DARRAY;
        dsv_desc.Flags = 0;
        U(dsv_desc).Texture2DArray.MipSlice = 0;
        U(dsv_desc).Texture2DArray.FirstArraySlice = i;
        U(dsv_desc).Texture2DArray.ArraySize = 1;

        hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, &dsv_desc, &dsv);
        ok(hr == S_OK, "Slice %u: Got unexpected hr %#lx.\n", i, hr);
        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, depth_values[i], 0);
        ID3D11DepthStencilView_Release(dsv);

        U(dsv_desc).Texture2DArray.MipSlice = 1;
        hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, &dsv_desc, &dsv);
        ok(hr == S_OK, "Slice %u: Got unexpected hr %#lx.\n", i, hr);
        ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0);
        ID3D11DepthStencilView_Release(dsv);
    }

    srv_desc.Format = DXGI_FORMAT_R32_FLOAT;
    srv_desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURECUBE;
    U(srv_desc).TextureCube.MostDetailedMip = 0;
    U(srv_desc).TextureCube.MipLevels = ~0u;
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, &srv_desc, &srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_cube_code, sizeof(ps_cube_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);
    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        ps_constant.x = tests[i].d_ref;
        ps_constant.y = tests[i].layer;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0,
                NULL, &ps_constant, 0, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, clear_color);
        draw_quad(&test_context);
        /* Avoid testing values affected by seamless cube map filtering. */
        SetRect(&rect, 100, 100, 540, 380);
        check_texture_sub_resource_float(rt_texture, 0, &rect, tests[i].expected, 2);
    }

    ID3D11Texture2D_Release(texture);
    ID3D11ShaderResourceView_Release(srv);

    ID3D11Buffer_Release(cb);
    ID3D11PixelShader_Release(ps);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11SamplerState_Release(sampler);
    ID3D11Texture2D_Release(rt_texture);
    release_test_context(&test_context);
}

static void test_multiple_render_targets(void)
{
    ID3D11RenderTargetView *rtv[4], *tmp_rtv[4];
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11InputLayout *input_layout;
    unsigned int stride, offset, i;
    ID3D11DeviceContext *context;
    ID3D11Texture2D *rt[4];
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *vb;
    ULONG refcount;
    HRESULT hr;

    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    static const DWORD vs_code[] =
    {
#if 0
        float4 main(float4 position : POSITION) : SV_POSITION
        {
            return position;
        }
#endif
        0x43425844, 0xa7a2f22d, 0x83ff2560, 0xe61638bd, 0x87e3ce90, 0x00000001, 0x000000d8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x0000003c, 0x00010040,
        0x0000000f, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        struct output
        {
            float4 t1 : SV_TARGET0;
            float4 t2 : SV_Target1;
            float4 t3 : SV_TARGET2;
            float4 t4 : SV_Target3;
        };

        output main(float4 position : SV_POSITION)
        {
            struct output o;
            o.t1 = (float4)1.0f;
            o.t2 = (float4)0.5f;
            o.t3 = (float4)0.2f;
            o.t4 = float4(0.0f, 0.2f, 0.5f, 1.0f);
            return o;
        }
#endif
        0x43425844, 0x8701ad18, 0xe3d5291d, 0x7b4288a6, 0x01917515, 0x00000001, 0x000001a8, 0x00000003,
        0x0000002c, 0x00000060, 0x000000e4, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000007c, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x00000072, 0x00000001, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x00000068, 0x00000002, 0x00000000, 0x00000003, 0x00000002, 0x0000000f, 0x00000072, 0x00000003,
        0x00000000, 0x00000003, 0x00000003, 0x0000000f, 0x545f5653, 0x45475241, 0x56530054, 0x7261545f,
        0x00746567, 0x52444853, 0x000000bc, 0x00000040, 0x0000002f, 0x03000065, 0x001020f2, 0x00000000,
        0x03000065, 0x001020f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000002, 0x03000065, 0x001020f2,
        0x00000003, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x3f800000, 0x3f800000, 0x3f800000,
        0x3f800000, 0x08000036, 0x001020f2, 0x00000001, 0x00004002, 0x3f000000, 0x3f000000, 0x3f000000,
        0x3f000000, 0x08000036, 0x001020f2, 0x00000002, 0x00004002, 0x3e4ccccd, 0x3e4ccccd, 0x3e4ccccd,
        0x3e4ccccd, 0x08000036, 0x001020f2, 0x00000003, 0x00004002, 0x00000000, 0x3e4ccccd, 0x3f000000,
        0x3f800000, 0x0100003e,
    };
    static const struct vec2 quad[] =
    {
        {-1.0f, -1.0f},
        {-1.0f,  1.0f},
        { 1.0f, -1.0f},
        { 1.0f,  1.0f},
    };
    static const float red[] = {1.0f, 0.0f, 0.0f, 1.0f};

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(quad), quad);

    texture_desc.Width = 640;
    texture_desc.Height = 480;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    for (i = 0; i < ARRAY_SIZE(rt); ++i)
    {
        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &rt[i]);
        ok(hr == S_OK, "RT %u: Got unexpected hr %#lx.\n", i, hr);

        hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)rt[i], NULL, &rtv[i]);
        ok(hr == S_OK, "RT %u: Got unexpected hr %#lx.\n", i, hr);
    }

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11Device_GetImmediateContext(device, &context);

    ID3D11DeviceContext_OMSetRenderTargets(context, 4, rtv, NULL);
    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    stride = sizeof(*quad);
    offset = 0;
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb, &stride, &offset);
    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    set_viewport(context, 0.0f, 0.0f, 640.0f, 480.0f, 0.0f, 1.0f);

    for (i = 0; i < ARRAY_SIZE(rtv); ++i)
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv[i], red);
    ID3D11DeviceContext_Draw(context, 4, 0);
    check_texture_color(rt[0], 0xffffffff, 2);
    check_texture_color(rt[1], 0x7f7f7f7f, 2);
    check_texture_color(rt[2], 0x33333333, 2);
    check_texture_color(rt[3], 0xff7f3300, 2);

    for (i = 0; i < ARRAY_SIZE(rtv); ++i)
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv[i], red);
    for (i = 0; i < ARRAY_SIZE(tmp_rtv); ++i)
    {
        memset(tmp_rtv, 0, sizeof(tmp_rtv));
        tmp_rtv[i] = rtv[i];
        ID3D11DeviceContext_OMSetRenderTargets(context, 4, tmp_rtv, NULL);
        ID3D11DeviceContext_Draw(context, 4, 0);
    }
    check_texture_color(rt[0], 0xffffffff, 2);
    check_texture_color(rt[1], 0x7f7f7f7f, 2);
    check_texture_color(rt[2], 0x33333333, 2);
    check_texture_color(rt[3], 0xff7f3300, 2);

    ID3D11Buffer_Release(vb);
    ID3D11PixelShader_Release(ps);
    ID3D11VertexShader_Release(vs);
    ID3D11InputLayout_Release(input_layout);
    for (i = 0; i < ARRAY_SIZE(rtv); ++i)
    {
        ID3D11RenderTargetView_Release(rtv[i]);
        ID3D11Texture2D_Release(rt[i]);
    }
    ID3D11DeviceContext_Release(context);
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_render_target_views(void)
{
    struct texture
    {
        UINT miplevel_count;
        UINT array_size;
    };

    static const struct vec4 red = {1.0f, 0.0f, 0.0f, 1.0f};
    static struct test
    {
        struct texture texture;
        struct rtv_desc rtv;
        DWORD expected_colors[4];
    }
    tests[] =
    {
        {{2, 1}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2D, 0},
                {0xff0000ff, 0x00000000}},
        {{2, 1}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2D, 1},
                {0x00000000, 0xff0000ff}},
        {{2, 1}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 0, 0, 1},
                {0xff0000ff, 0x00000000}},
        {{2, 1}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 1, 0, 1},
                {0x00000000, 0xff0000ff}},
        {{1, 4}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2D, 0},
                {0xff0000ff, 0x00000000, 0x00000000, 0x00000000}},
        {{1, 4}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 0, 0, 1},
                {0xff0000ff, 0x00000000, 0x00000000, 0x00000000}},
        {{1, 4}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 0, 1, 1},
                {0x00000000, 0xff0000ff, 0x00000000, 0x00000000}},
        {{1, 4}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 0, 2, 1},
                {0x00000000, 0x00000000, 0xff0000ff, 0x00000000}},
        {{1, 4}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 0, 3, 1},
                {0x00000000, 0x00000000, 0x00000000, 0xff0000ff}},
        {{1, 4}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 0, 0, 4},
                {0xff0000ff, 0x00000000, 0x00000000, 0x00000000}},
        {{2, 2}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2D, 0},
                {0xff0000ff, 0x00000000, 0x00000000, 0x00000000}},
        {{2, 2}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 0, 0, 1},
                {0xff0000ff, 0x00000000, 0x00000000, 0x00000000}},
        {{2, 2}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 0, 1, 1},
                {0x00000000, 0x00000000, 0xff0000ff, 0x00000000}},
        {{2, 2}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 1, 0, 1},
                {0x00000000, 0xff0000ff, 0x00000000, 0x00000000}},
        {{2, 2}, {DXGI_FORMAT_UNKNOWN, D3D11_RTV_DIMENSION_TEXTURE2DARRAY, 1, 1, 1},
                {0x00000000, 0x00000000, 0x00000000, 0xff0000ff}},
    };
#define FMT_UNKNOWN  DXGI_FORMAT_UNKNOWN
#define RGBA8_UNORM  DXGI_FORMAT_R8G8B8A8_UNORM
#define RGBA8_SRGB   DXGI_FORMAT_R8G8B8A8_UNORM_SRGB
#define RGBA8_UINT   DXGI_FORMAT_R8G8B8A8_UINT
#define RGBA8_TL     DXGI_FORMAT_R8G8B8A8_TYPELESS
#define DIM_UNKNOWN  D3D11_RTV_DIMENSION_UNKNOWN
#define TEX_1D       D3D11_RTV_DIMENSION_TEXTURE1D
#define TEX_1D_ARRAY D3D11_RTV_DIMENSION_TEXTURE1DARRAY
#define TEX_2D       D3D11_RTV_DIMENSION_TEXTURE2D
#define TEX_2D_ARRAY D3D11_RTV_DIMENSION_TEXTURE2DARRAY
#define TEX_3D       D3D11_RTV_DIMENSION_TEXTURE3D
    static const struct
    {
        struct
        {
            D3D11_RTV_DIMENSION dimension;
            unsigned int miplevel_count;
            unsigned int depth_or_array_size;
            DXGI_FORMAT format;
        } texture;
        struct rtv_desc rtv_desc;
    }
    invalid_desc_tests[] =
    {
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, DIM_UNKNOWN}},
        {{TEX_2D, 6, 4, RGBA8_UNORM}, {RGBA8_UNORM, DIM_UNKNOWN}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0, ~0u}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_2D,        0}},
        {{TEX_2D, 1, 1, RGBA8_TL},    {FMT_UNKNOWN, TEX_2D,        0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  1, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  2}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 1,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UINT,  TEX_2D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UINT},  {RGBA8_UNORM, TEX_2D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_SRGB,  TEX_2D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_SRGB},  {RGBA8_UNORM, TEX_2D,        0, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_3D,        0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_TL,    TEX_3D,        0, 0,  1}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  9}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        3, 0,  2}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        2, 0,  4}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 0,  8}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 8, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 4, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        2, 2, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        3, 1, ~0u}},
    };
#undef FMT_UNKNOWN
#undef RGBA8_UNORM
#undef RGBA8_SRGB
#undef RGBA8_UINT
#undef RGBA8_TL
#undef DIM_UNKNOWN
#undef TEX_1D
#undef TEX_1D_ARRAY
#undef TEX_2D
#undef TEX_2D_ARRAY
#undef TEX_3D

    struct d3d11_test_context test_context;
    D3D11_RENDER_TARGET_VIEW_DESC rtv_desc;
    D3D11_TEXTURE3D_DESC texture3d_desc;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11Texture3D *texture3d;
    ID3D11Texture2D *texture;
    ID3D11Resource *resource;
    ID3D11Device *device;
    unsigned int i, j, k;
    void *data;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 32;
    texture_desc.Height = 32;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    data = heap_alloc_zero(texture_desc.Width * texture_desc.Height * 4);
    ok(!!data, "Failed to allocate memory.\n");

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        const struct test *test = &tests[i];
        unsigned int sub_resource_count;

        texture_desc.MipLevels = test->texture.miplevel_count;
        texture_desc.ArraySize = test->texture.array_size;

        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        get_rtv_desc(&rtv_desc, &test->rtv);
        hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        for (j = 0; j < texture_desc.ArraySize; ++j)
        {
            for (k = 0; k < texture_desc.MipLevels; ++k)
            {
                unsigned int sub_resource_idx = j * texture_desc.MipLevels + k;
                ID3D11DeviceContext_UpdateSubresource(context,
                        (ID3D11Resource *)texture, sub_resource_idx, NULL, data, texture_desc.Width * 4, 0);
            }
        }
        check_texture_color(texture, 0, 0);

        ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
        draw_color_quad(&test_context, &red);

        sub_resource_count = texture_desc.MipLevels * texture_desc.ArraySize;
        assert(sub_resource_count <= ARRAY_SIZE(test->expected_colors));
        for (j = 0; j < sub_resource_count; ++j)
            check_texture_sub_resource_color(texture, j, NULL, test->expected_colors[j], 1);

        ID3D11RenderTargetView_Release(rtv);
        ID3D11Texture2D_Release(texture);
    }

    texture3d_desc.Width = 32;
    texture3d_desc.Height = 32;
    texture3d_desc.Depth = 32;
    texture3d_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture3d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture3d_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture3d_desc.CPUAccessFlags = 0;
    texture3d_desc.MiscFlags = 0;

    for (i = 0; i < ARRAY_SIZE(invalid_desc_tests); ++i)
    {
        assert(invalid_desc_tests[i].texture.dimension == D3D11_RTV_DIMENSION_TEXTURE2D
                || invalid_desc_tests[i].texture.dimension == D3D11_RTV_DIMENSION_TEXTURE3D);

        if (invalid_desc_tests[i].texture.dimension == D3D11_RTV_DIMENSION_TEXTURE2D)
        {
            texture_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
            texture_desc.ArraySize = invalid_desc_tests[i].texture.depth_or_array_size;
            texture_desc.Format = invalid_desc_tests[i].texture.format;
            texture_desc.MiscFlags = 0;

            hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            resource = (ID3D11Resource *)texture;
        }
        else
        {
            texture3d_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
            texture3d_desc.Depth = invalid_desc_tests[i].texture.depth_or_array_size;
            texture3d_desc.Format = invalid_desc_tests[i].texture.format;

            hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &texture3d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            resource = (ID3D11Resource *)texture3d;
        }

        get_rtv_desc(&rtv_desc, &invalid_desc_tests[i].rtv_desc);
        hr = ID3D11Device_CreateRenderTargetView(device, resource, &rtv_desc, &rtv);
        ok(hr == E_INVALIDARG, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        ID3D11Resource_Release(resource);
    }

    heap_free(data);
    release_test_context(&test_context);
}

static void test_layered_rendering(void)
{
    struct
    {
        unsigned int layer_offset;
        unsigned int draw_id;
        unsigned int padding[2];
    } constant;
    struct d3d11_test_context test_context;
    D3D11_RENDER_TARGET_VIEW_DESC rtv_desc;
    unsigned int i, sub_resource_count;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11Texture2D *texture;
    ID3D11GeometryShader *gs;
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *cb;
    HRESULT hr;
    BOOL warp;

    static const DWORD vs_code[] =
    {
#if 0
        uint layer_offset;

        void main(float4 position : POSITION,
                out float4 out_position : SV_POSITION,
                out uint layer : SV_RenderTargetArrayIndex)
        {
            out_position = position;
            layer = layer_offset;
        }
#endif
        0x43425844, 0x71f7b9cd, 0x2ab8c713, 0x53e77663, 0x54a9ba68, 0x00000001, 0x00000158, 0x00000004,
        0x00000030, 0x00000064, 0x000000cc, 0x00000148, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954,
        0xababab00, 0x4e47534f, 0x00000060, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000004, 0x00000001, 0x00000001,
        0x00000e01, 0x505f5653, 0x5449534f, 0x004e4f49, 0x525f5653, 0x65646e65, 0x72615472, 0x41746567,
        0x79617272, 0x65646e49, 0xabab0078, 0x52444853, 0x00000074, 0x00010040, 0x0000001d, 0x04000059,
        0x00208e46, 0x00000000, 0x00000001, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2,
        0x00000000, 0x00000001, 0x04000067, 0x00102012, 0x00000001, 0x00000004, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x06000036, 0x00102012, 0x00000001, 0x0020800a, 0x00000000,
        0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00002000, 0x00000000,
    };
    static const DWORD gs_5_code[] =
    {
#if 0
        uint layer_offset;

        struct gs_in
        {
            float4 pos : SV_Position;
        };

        struct gs_out
        {
            float4 pos : SV_Position;
            uint layer : SV_RenderTargetArrayIndex;
        };

        [instance(4)]
        [maxvertexcount(3)]
        void main(triangle gs_in vin[3], in uint instance_id : SV_GSInstanceID,
                inout TriangleStream<gs_out> vout)
        {
            gs_out o;
            o.layer = layer_offset + instance_id;
            for (uint i = 0; i < 3; ++i)
            {
                o.pos = vin[i].pos;
                vout.Append(o);
            }
        }
#endif
        0x43425844, 0xb52da162, 0x9a13d8ee, 0xf7c30b50, 0xe80bc2e7, 0x00000001, 0x00000218, 0x00000003,
        0x0000002c, 0x00000060, 0x000000d0, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x3547534f, 0x00000068, 0x00000002, 0x00000008, 0x00000000, 0x00000040, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x00000000, 0x0000004c, 0x00000000, 0x00000004, 0x00000001,
        0x00000001, 0x00000e01, 0x505f5653, 0x7469736f, 0x006e6f69, 0x525f5653, 0x65646e65, 0x72615472,
        0x41746567, 0x79617272, 0x65646e49, 0xabab0078, 0x58454853, 0x00000140, 0x00020050, 0x00000050,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x05000061, 0x002010f2, 0x00000003,
        0x00000000, 0x00000001, 0x0200005f, 0x00025000, 0x02000068, 0x00000001, 0x020000ce, 0x00000004,
        0x0100185d, 0x0300008f, 0x00110000, 0x00000000, 0x0100285c, 0x04000067, 0x001020f2, 0x00000000,
        0x00000001, 0x04000067, 0x00102012, 0x00000001, 0x00000004, 0x0200005e, 0x00000003, 0x0700001e,
        0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x0002500a, 0x05000036, 0x00100022,
        0x00000000, 0x00004001, 0x00000000, 0x01000030, 0x07000050, 0x00100042, 0x00000000, 0x0010001a,
        0x00000000, 0x00004001, 0x00000003, 0x03040003, 0x0010002a, 0x00000000, 0x07000036, 0x001020f2,
        0x00000000, 0x00a01e46, 0x0010001a, 0x00000000, 0x00000000, 0x05000036, 0x00102012, 0x00000001,
        0x0010000a, 0x00000000, 0x03000075, 0x00110000, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010001a, 0x00000000, 0x00004001, 0x00000001, 0x01000016, 0x0100003e,
    };
    static const DWORD gs_4_code[] =
    {
#if 0
        uint layer_offset;

        struct gs_in
        {
            float4 pos : SV_Position;
        };

        struct gs_out
        {
            float4 pos : SV_Position;
            uint layer : SV_RenderTargetArrayIndex;
        };

        [maxvertexcount(12)]
        void main(triangle gs_in vin[3], inout TriangleStream<gs_out> vout)
        {
            gs_out o;
            for (uint instance_id = 0; instance_id < 4; ++instance_id)
            {
                o.layer = layer_offset + instance_id;
                for (uint i = 0; i < 3; ++i)
                {
                    o.pos = vin[i].pos;
                    vout.Append(o);
                }
                vout.RestartStrip();
            }
        }
#endif
        0x43425844, 0x7eabd7c5, 0x8af1468e, 0xd585cade, 0xfe0d761d, 0x00000001, 0x00000250, 0x00000003,
        0x0000002c, 0x00000060, 0x000000c8, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x4e47534f, 0x00000060, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000004, 0x00000001, 0x00000001, 0x00000e01,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x525f5653, 0x65646e65, 0x72615472, 0x41746567, 0x79617272,
        0x65646e49, 0xabab0078, 0x52444853, 0x00000180, 0x00020040, 0x00000060, 0x04000059, 0x00208e46,
        0x00000000, 0x00000001, 0x05000061, 0x002010f2, 0x00000003, 0x00000000, 0x00000001, 0x02000068,
        0x00000001, 0x0100185d, 0x0100285c, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x04000067,
        0x00102012, 0x00000001, 0x00000004, 0x0200005e, 0x0000000c, 0x05000036, 0x00100012, 0x00000000,
        0x00004001, 0x00000000, 0x01000030, 0x07000050, 0x00100022, 0x00000000, 0x0010000a, 0x00000000,
        0x00004001, 0x00000004, 0x03040003, 0x0010001a, 0x00000000, 0x0800001e, 0x00100022, 0x00000000,
        0x0020800a, 0x00000000, 0x00000000, 0x0010000a, 0x00000000, 0x05000036, 0x00100042, 0x00000000,
        0x00004001, 0x00000000, 0x01000030, 0x07000050, 0x00100082, 0x00000000, 0x0010002a, 0x00000000,
        0x00004001, 0x00000003, 0x03040003, 0x0010003a, 0x00000000, 0x07000036, 0x001020f2, 0x00000000,
        0x00a01e46, 0x0010002a, 0x00000000, 0x00000000, 0x05000036, 0x00102012, 0x00000001, 0x0010001a,
        0x00000000, 0x01000013, 0x0700001e, 0x00100042, 0x00000000, 0x0010002a, 0x00000000, 0x00004001,
        0x00000001, 0x01000016, 0x01000009, 0x0700001e, 0x00100012, 0x00000000, 0x0010000a, 0x00000000,
        0x00004001, 0x00000001, 0x01000016, 0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        uint layer_offset;
        uint draw_id;

        float4 main(in float4 pos : SV_Position,
                in uint layer : SV_RenderTargetArrayIndex) : SV_Target
        {
            return float4(layer, draw_id, 0, 0);
        }
#endif
        0x43425844, 0x5fa6ae84, 0x3f893c81, 0xf15892d6, 0x142e2e6b, 0x00000001, 0x00000154, 0x00000003,
        0x0000002c, 0x00000094, 0x000000c8, 0x4e475349, 0x00000060, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000004,
        0x00000001, 0x00000001, 0x00000101, 0x505f5653, 0x7469736f, 0x006e6f69, 0x525f5653, 0x65646e65,
        0x72615472, 0x41746567, 0x79617272, 0x65646e49, 0xabab0078, 0x4e47534f, 0x0000002c, 0x00000001,
        0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x545f5653,
        0x65677261, 0xabab0074, 0x52444853, 0x00000084, 0x00000040, 0x00000021, 0x04000059, 0x00208e46,
        0x00000000, 0x00000001, 0x04000864, 0x00101012, 0x00000001, 0x00000004, 0x03000065, 0x001020f2,
        0x00000000, 0x05000056, 0x00102012, 0x00000000, 0x0010100a, 0x00000001, 0x06000056, 0x00102022,
        0x00000000, 0x0020801a, 0x00000000, 0x00000000, 0x08000036, 0x001020c2, 0x00000000, 0x00004002,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const struct vec4 expected_values[] =
    {
        {0.0f, 0.0f}, {0.0f, 3.0f}, {3.0f, 11.0f}, {1.0f, 0.0f}, {1.0f, 3.0f}, {3.0f, 10.0f},
        {2.0f, 0.0f}, {2.0f, 3.0f}, {3.0f,  9.0f}, {4.0f, 2.0f}, {3.0f, 3.0f}, {3.0f,  8.0f},
        {4.0f, 1.0f}, {4.0f, 3.0f}, {3.0f,  7.0f}, {5.0f, 1.0f}, {5.0f, 3.0f}, {3.0f,  6.0f},
        {6.0f, 1.0f}, {6.0f, 3.0f}, {3.0f,  5.0f}, {7.0f, 1.0f}, {7.0f, 3.0f}, {3.0f,  4.0f},
    };
    static const struct vec4 vs_expected_value = {1.0f, 42.0f};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    warp = is_warp_device(device);

    memset(&constant, 0, sizeof(constant));
    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(constant), &constant);
    ID3D11DeviceContext_VSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext_GSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    /* Geometry shader instancing seems broken on WARP. */
    if (ID3D11Device_GetFeatureLevel(device) < D3D_FEATURE_LEVEL_11_0 || warp)
    {
        hr = ID3D11Device_CreateGeometryShader(device, gs_4_code, sizeof(gs_4_code), NULL, &gs);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    }
    else
    {
        hr = ID3D11Device_CreateGeometryShader(device, gs_5_code, sizeof(gs_5_code), NULL, &gs);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    }
    ID3D11DeviceContext_GSSetShader(context, gs, NULL, 0);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    texture_desc.Width = 32;
    texture_desc.Height = 32;
    texture_desc.MipLevels = 3;
    texture_desc.ArraySize = 8;
    texture_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
    constant.layer_offset = 0;
    constant.draw_id = 0;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
    draw_quad(&test_context);
    constant.layer_offset = 4;
    constant.draw_id = 1;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
    draw_quad(&test_context);
    ID3D11RenderTargetView_Release(rtv);

    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DARRAY;
    rtv_desc.Format = texture_desc.Format;
    U(rtv_desc).Texture2DArray.MipSlice = 0;
    U(rtv_desc).Texture2DArray.FirstArraySlice = 3;
    U(rtv_desc).Texture2DArray.ArraySize = 1;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
    constant.layer_offset = 1;
    constant.draw_id = 2;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
    draw_quad(&test_context);
    ID3D11RenderTargetView_Release(rtv);

    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DARRAY;
    U(rtv_desc).Texture2DArray.MipSlice = 1;
    U(rtv_desc).Texture2DArray.FirstArraySlice = 0;
    U(rtv_desc).Texture2DArray.ArraySize = ~0u;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
    constant.layer_offset = 0;
    constant.draw_id = 3;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
    draw_quad(&test_context);
    constant.layer_offset = 4;
    constant.draw_id = 3;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
    draw_quad(&test_context);
    ID3D11RenderTargetView_Release(rtv);

    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DARRAY;
    U(rtv_desc).Texture2DArray.MipSlice = 2;
    U(rtv_desc).Texture2DArray.ArraySize = 1;
    for (i = 0; i < texture_desc.ArraySize; ++i)
    {
        U(rtv_desc).Texture2DArray.FirstArraySlice = texture_desc.ArraySize - 1 - i;
        hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv);
        ok(hr == S_OK, "Slice %u: Got unexpected hr %#lx.\n", i, hr);
        ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
        constant.layer_offset = 0;
        constant.draw_id = 4 + i;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
        draw_quad(&test_context);
        ID3D11RenderTargetView_Release(rtv);
    }

    sub_resource_count = texture_desc.MipLevels * texture_desc.ArraySize;
    assert(ARRAY_SIZE(expected_values) == sub_resource_count);
    for (i = 0; i < sub_resource_count; ++i)
    {
        if (warp && (i == 3 || i == 4)) /* Broken on WARP. */
            continue;
        check_texture_sub_resource_vec4(texture, i, NULL, &expected_values[i], 1);
    }

    /* layered rendering without GS */
    if (!check_viewport_array_index_from_any_shader_support(device))
    {
        hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
        todo_wine ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
        if (SUCCEEDED(hr))
            ID3D11VertexShader_Release(vs);
        skip("Viewport array index not supported in vertex shaders.\n");
        goto done;
    }

    ID3D11DeviceContext_GSSetShader(context, NULL, NULL, 0);

    constant.layer_offset = 1;
    constant.draw_id = 42;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &constant, 0, 0);
    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DARRAY;
    U(rtv_desc).Texture2DArray.MipSlice = 0;
    U(rtv_desc).Texture2DArray.FirstArraySlice = 0;
    U(rtv_desc).Texture2DArray.ArraySize = ~0u;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
    draw_quad_vs(&test_context, vs_code, sizeof(vs_code));
    check_texture_sub_resource_vec4(texture,
            constant.layer_offset * texture_desc.MipLevels, NULL, &vs_expected_value, 1);
    ID3D11RenderTargetView_Release(rtv);

done:
    ID3D11Texture2D_Release(texture);

    ID3D11Buffer_Release(cb);
    ID3D11GeometryShader_Release(gs);
    ID3D11PixelShader_Release(ps);
    release_test_context(&test_context);
}

static void test_scissor(void)
{
    struct d3d11_test_context test_context;
    ID3D11DeviceContext *immediate_context;
    D3D11_RASTERIZER_DESC rs_desc;
    ID3D11RasterizerState *rs;
    D3D11_RECT scissor_rect;
    ID3D11Device *device;
    DWORD color;
    HRESULT hr;

    static const float red[] = {1.0f, 0.0f, 0.0f, 1.0f};
    static const struct vec4 green = {0.0f, 1.0f, 0.0f, 1.0f};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    immediate_context = test_context.immediate_context;

    rs_desc.FillMode = D3D11_FILL_SOLID;
    rs_desc.CullMode = D3D11_CULL_BACK;
    rs_desc.FrontCounterClockwise = FALSE;
    rs_desc.DepthBias = 0;
    rs_desc.DepthBiasClamp = 0.0f;
    rs_desc.SlopeScaledDepthBias = 0.0f;
    rs_desc.DepthClipEnable = TRUE;
    rs_desc.ScissorEnable = TRUE;
    rs_desc.MultisampleEnable = FALSE;
    rs_desc.AntialiasedLineEnable = FALSE;
    hr = ID3D11Device_CreateRasterizerState(device, &rs_desc, &rs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    SetRect(&scissor_rect, 160, 120, 480, 360);
    ID3D11DeviceContext_RSSetScissorRects(immediate_context, 1, &scissor_rect);

    ID3D11DeviceContext_ClearRenderTargetView(immediate_context, test_context.backbuffer_rtv, red);
    check_texture_color(test_context.backbuffer, 0xff0000ff, 1);

    draw_color_quad(&test_context, &green);
    color = get_texture_color(test_context.backbuffer, 320, 60);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 80, 240);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 240);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 560, 240);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 420);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11DeviceContext_ClearRenderTargetView(immediate_context, test_context.backbuffer_rtv, red);
    ID3D11DeviceContext_RSSetState(immediate_context, rs);
    draw_color_quad(&test_context, &green);
    color = get_texture_color(test_context.backbuffer, 320, 60);
    ok(compare_color(color, 0xff0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 80, 240);
    ok(compare_color(color, 0xff0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 240);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 560, 240);
    ok(compare_color(color, 0xff0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 420);
    ok(compare_color(color, 0xff0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);

    set_viewport(immediate_context, -1.0f, 0.0f, 641, 480, 0.0f, 1.0f);
    SetRect(&scissor_rect, -1, 0, 640, 480);
    ID3D11DeviceContext_RSSetScissorRects(immediate_context, 1, &scissor_rect);
    ID3D11DeviceContext_ClearRenderTargetView(immediate_context, test_context.backbuffer_rtv, red);
    check_texture_color(test_context.backbuffer, 0xff0000ff, 1);
    draw_color_quad(&test_context, &green);
    color = get_texture_color(test_context.backbuffer, 320, 60);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 80, 240);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 240);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 560, 240);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 420);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11RasterizerState_Release(rs);
    release_test_context(&test_context);
}

static void test_clear_state(void)
{
    static const D3D_FEATURE_LEVEL feature_level = D3D_FEATURE_LEVEL_11_0;
    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
#if 0
    float4 main(float4 pos : POSITION) : POSITION
    {
        return pos;
    }
#endif
    static const DWORD simple_vs[] =
    {
        0x43425844, 0x66689e7c, 0x643f0971, 0xb7f67ff4, 0xabc48688, 0x00000001, 0x000000d4, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x49534f50, 0x4e4f4954, 0xababab00, 0x52444853, 0x00000038, 0x00010040,
        0x0000000e, 0x0300005f, 0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
#if 0
    struct data
    {
        float4 position : SV_Position;
    };

    struct patch_constant_data
    {
        float edges[3] : SV_TessFactor;
        float inside : SV_InsideTessFactor;
    };

    void patch_constant(InputPatch<data, 3> input, out patch_constant_data output)
    {
        output.edges[0] = output.edges[1] = output.edges[2] = 1.0f;
        output.inside = 1.0f;
    }

    [domain("tri")]
    [outputcontrolpoints(3)]
    [partitioning("integer")]
    [outputtopology("triangle_ccw")]
    [patchconstantfunc("patch_constant")]
    data hs_main(InputPatch<data, 3> input, uint i : SV_OutputControlPointID)
    {
        return input[i];
    }

    [domain("tri")]
    void ds_main(patch_constant_data input,
            float3 tess_coord : SV_DomainLocation,
            const OutputPatch<data, 3> patch,
            out data output)
    {
        output.position = tess_coord.x * patch[0].position
                + tess_coord.y * patch[1].position
                + tess_coord.z * patch[2].position;
    }
#endif
    static const DWORD simple_hs[] =
    {
        0x43425844, 0x42b5df25, 0xfd8aa2b1, 0xbe5490cb, 0xb595f8b1, 0x00000001, 0x0000020c, 0x00000004,
        0x00000030, 0x00000064, 0x00000098, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x47534350, 0x0000008c,
        0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d, 0x00000003, 0x00000000, 0x00000e01,
        0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001, 0x00000e01, 0x00000068, 0x00000002,
        0x0000000d, 0x00000003, 0x00000002, 0x00000e01, 0x00000076, 0x00000000, 0x0000000e, 0x00000003,
        0x00000003, 0x00000e01, 0x545f5653, 0x46737365, 0x6f746361, 0x56530072, 0x736e495f, 0x54656469,
        0x46737365, 0x6f746361, 0xabab0072, 0x58454853, 0x000000d8, 0x00030050, 0x00000036, 0x01000071,
        0x01001893, 0x01001894, 0x01001095, 0x01000896, 0x01002097, 0x0100086a, 0x01000073, 0x02000099,
        0x00000003, 0x0200005f, 0x00017000, 0x04000067, 0x00102012, 0x00000000, 0x00000011, 0x04000067,
        0x00102012, 0x00000001, 0x00000012, 0x04000067, 0x00102012, 0x00000002, 0x00000013, 0x02000068,
        0x00000001, 0x0400005b, 0x00102012, 0x00000000, 0x00000003, 0x04000036, 0x00100012, 0x00000000,
        0x0001700a, 0x06000036, 0x00902012, 0x0010000a, 0x00000000, 0x00004001, 0x3f800000, 0x0100003e,
        0x01000073, 0x04000067, 0x00102012, 0x00000003, 0x00000014, 0x05000036, 0x00102012, 0x00000003,
        0x00004001, 0x3f800000, 0x0100003e,
    };
    static const DWORD simple_ds[] =
    {
        0x43425844, 0xb7e35b82, 0x1b930ff2, 0x48d3a0f2, 0x375219ed, 0x00000001, 0x000001e0, 0x00000004,
        0x00000030, 0x00000064, 0x000000f8, 0x0000012c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x47534350, 0x0000008c, 0x00000004, 0x00000008, 0x00000068, 0x00000000, 0x0000000d,
        0x00000003, 0x00000000, 0x00000001, 0x00000068, 0x00000001, 0x0000000d, 0x00000003, 0x00000001,
        0x00000001, 0x00000068, 0x00000002, 0x0000000d, 0x00000003, 0x00000002, 0x00000001, 0x00000076,
        0x00000000, 0x0000000e, 0x00000003, 0x00000003, 0x00000001, 0x545f5653, 0x46737365, 0x6f746361,
        0x56530072, 0x736e495f, 0x54656469, 0x46737365, 0x6f746361, 0xabab0072, 0x4e47534f, 0x0000002c,
        0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x58454853, 0x000000ac, 0x00040050, 0x0000002b, 0x01001893,
        0x01001095, 0x0100086a, 0x0200005f, 0x0001c072, 0x0400005f, 0x002190f2, 0x00000003, 0x00000000,
        0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x02000068, 0x00000001, 0x07000038, 0x001000f2,
        0x00000000, 0x0001c556, 0x00219e46, 0x00000001, 0x00000000, 0x09000032, 0x001000f2, 0x00000000,
        0x0001c006, 0x00219e46, 0x00000000, 0x00000000, 0x00100e46, 0x00000000, 0x09000032, 0x001020f2,
        0x00000000, 0x0001caa6, 0x00219e46, 0x00000002, 0x00000000, 0x00100e46, 0x00000000, 0x0100003e,
    };
#if 0
    struct gs_out
    {
        float4 pos : SV_POSITION;
    };

    [maxvertexcount(4)]
    void main(point float4 vin[1] : POSITION, inout TriangleStream<gs_out> vout)
    {
        float offset = 0.1 * vin[0].w;
        gs_out v;

        v.pos = float4(vin[0].x - offset, vin[0].y - offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x - offset, vin[0].y + offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x + offset, vin[0].y - offset, vin[0].z, vin[0].w);
        vout.Append(v);
        v.pos = float4(vin[0].x + offset, vin[0].y + offset, vin[0].z, vin[0].w);
        vout.Append(v);
    }
#endif
    static const DWORD simple_gs[] =
    {
        0x43425844, 0x000ee786, 0xc624c269, 0x885a5cbe, 0x444b3b1f, 0x00000001, 0x0000023c, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x000001a0, 0x00020040,
        0x00000068, 0x0400005f, 0x002010f2, 0x00000001, 0x00000000, 0x02000068, 0x00000001, 0x0100085d,
        0x0100285c, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x0200005e, 0x00000004, 0x0f000032,
        0x00100032, 0x00000000, 0x80201ff6, 0x00000041, 0x00000000, 0x00000000, 0x00004002, 0x3dcccccd,
        0x3dcccccd, 0x00000000, 0x00000000, 0x00201046, 0x00000000, 0x00000000, 0x05000036, 0x00102032,
        0x00000000, 0x00100046, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000,
        0x00000000, 0x01000013, 0x05000036, 0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x0e000032,
        0x00100052, 0x00000000, 0x00201ff6, 0x00000000, 0x00000000, 0x00004002, 0x3dcccccd, 0x00000000,
        0x3dcccccd, 0x00000000, 0x00201106, 0x00000000, 0x00000000, 0x05000036, 0x00102022, 0x00000000,
        0x0010002a, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000, 0x00000000,
        0x01000013, 0x05000036, 0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x05000036, 0x00102022,
        0x00000000, 0x0010001a, 0x00000000, 0x06000036, 0x001020c2, 0x00000000, 0x00201ea6, 0x00000000,
        0x00000000, 0x01000013, 0x05000036, 0x00102032, 0x00000000, 0x00100086, 0x00000000, 0x06000036,
        0x001020c2, 0x00000000, 0x00201ea6, 0x00000000, 0x00000000, 0x01000013, 0x0100003e,
    };
#if 0
    float4 main(float4 color : COLOR) : SV_TARGET
    {
        return color;
    }
#endif
    static const DWORD simple_ps[] =
    {
        0x43425844, 0x08c2b568, 0x17d33120, 0xb7d82948, 0x13a570fb, 0x00000001, 0x000000d0, 0x00000003,
        0x0000002c, 0x0000005c, 0x00000090, 0x4e475349, 0x00000028, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x4f4c4f43, 0xabab0052, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000038, 0x00000040, 0x0000000e,
        0x03001062, 0x001010f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000000, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };
#if 0
    [numthreads(1, 1, 1)]
    void main() { }
#endif
    static const DWORD simple_cs[] =
    {
        0x43425844, 0x1acc3ad0, 0x71c7b057, 0xc72c4306, 0xf432cb57, 0x00000001, 0x00000074, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000020, 0x00050050, 0x00000008, 0x0100086a,
        0x0400009b, 0x00000001, 0x00000001, 0x00000001, 0x0100003e,
    };

    D3D11_VIEWPORT tmp_viewport[D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE];
    ID3D11ShaderResourceView *tmp_srv[D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT];
    ID3D11ShaderResourceView *srv[D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT];
    ID3D11RenderTargetView *tmp_rtv[D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT];
    RECT tmp_rect[D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE];
    ID3D11SamplerState *tmp_sampler[D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT];
    ID3D11RenderTargetView *rtv[D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT];
    ID3D11Texture2D *rt_texture[D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT];
    ID3D11Buffer *cb[D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT];
    ID3D11Buffer *tmp_buffer[D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT];
    ID3D11SamplerState *sampler[D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT];
    ID3D11UnorderedAccessView *tmp_uav[D3D11_PS_CS_UAV_REGISTER_COUNT];
    ID3D11UnorderedAccessView *cs_uav[D3D11_PS_CS_UAV_REGISTER_COUNT];
    ID3D11Buffer *buffer[D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT];
    ID3D11Buffer *cs_uav_buffer[D3D11_PS_CS_UAV_REGISTER_COUNT];
    UINT offset[D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT];
    UINT stride[D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT];
    ID3D11Buffer *so_buffer[D3D11_SO_BUFFER_SLOT_COUNT];
    ID3D11InputLayout *tmp_input_layout, *input_layout;
    ID3D11DepthStencilState *tmp_ds_state, *ds_state;
    ID3D11BlendState *tmp_blend_state, *blend_state;
    ID3D11RasterizerState *tmp_rs_state, *rs_state;
    ID3D11Predicate *tmp_predicate, *predicate;
    D3D11_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D11_SHADER_RESOURCE_VIEW_DESC srv_desc;
    ID3D11DepthStencilView *tmp_dsv, *dsv;
    ID3D11UnorderedAccessView *ps_uav;
    D3D11_PRIMITIVE_TOPOLOGY topology;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11GeometryShader *tmp_gs, *gs;
    ID3D11ComputeShader *tmp_cs, *cs;
    D3D11_DEPTH_STENCIL_DESC ds_desc;
    ID3D11VertexShader *tmp_vs, *vs;
    ID3D11DomainShader *tmp_ds, *ds;
    D3D11_SAMPLER_DESC sampler_desc;
    D3D11_QUERY_DESC predicate_desc;
    struct device_desc device_desc;
    ID3D11PixelShader *tmp_ps, *ps;
    ID3D11HullShader *tmp_hs, *hs;
    D3D11_RASTERIZER_DESC rs_desc;
    ID3D11DeviceContext *context;
    D3D11_BLEND_DESC blend_desc;
    ID3D11Texture2D *ds_texture;
    ID3D11Buffer *ps_uav_buffer;
    float blend_factor[4];
    ID3D11Device *device;
    BOOL predicate_value;
    UINT instance_count;
    DXGI_FORMAT format;
    UINT sample_mask;
    UINT stencil_ref;
    ULONG refcount;
    UINT count, i;
    HRESULT hr;

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }

    ID3D11Device_GetImmediateContext(device, &context);

    /* Verify the initial state after device creation. */

    ID3D11DeviceContext_VSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_VSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_VSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    instance_count = 100;
    ID3D11DeviceContext_VSGetShader(context, &tmp_vs, NULL, &instance_count);
    ok(!tmp_vs, "Got unexpected vertex shader %p.\n", tmp_vs);
    ok(!instance_count, "Got unexpected instance count %u.\n", instance_count);

    ID3D11DeviceContext_HSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_HSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_HSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    instance_count = 100;
    ID3D11DeviceContext_HSGetShader(context, &tmp_hs, NULL, &instance_count);
    ok(!tmp_hs, "Got unexpected hull shader %p.\n", tmp_hs);
    ok(!instance_count, "Got unexpected instance count %u.\n", instance_count);

    ID3D11DeviceContext_DSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_DSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_DSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    instance_count = 100;
    ID3D11DeviceContext_DSGetShader(context, &tmp_ds, NULL, &instance_count);
    ok(!tmp_ds, "Got unexpected domain shader %p.\n", tmp_ds);
    ok(!instance_count, "Got unexpected instance count %u.\n", instance_count);

    ID3D11DeviceContext_GSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_GSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_GSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    instance_count = 100;
    ID3D11DeviceContext_GSGetShader(context, &tmp_gs, NULL, &instance_count);
    ok(!tmp_gs, "Got unexpected geometry shader %p.\n", tmp_gs);
    ok(!instance_count, "Got unexpected instance count %u.\n", instance_count);

    ID3D11DeviceContext_PSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_PSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT,
            tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_PSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    instance_count = 100;
    ID3D11DeviceContext_PSGetShader(context, &tmp_ps, NULL, &instance_count);
    ok(!tmp_ps, "Got unexpected pixel shader %p.\n", tmp_ps);
    ok(!instance_count, "Got unexpected instance count %u.\n", instance_count);

    ID3D11DeviceContext_CSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_CSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT,
            tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_CSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    instance_count = 100;
    ID3D11DeviceContext_CSGetShader(context, &tmp_cs, NULL, &instance_count);
    ok(!tmp_cs, "Got unexpected compute shader %p.\n", tmp_cs);
    ok(!instance_count, "Got unexpected instance count %u.\n", instance_count);
    ID3D11DeviceContext_CSGetUnorderedAccessViews(context, 0, D3D11_PS_CS_UAV_REGISTER_COUNT, tmp_uav);
    for (i = 0; i < D3D11_PS_CS_UAV_REGISTER_COUNT; ++i)
    {
        ok(!tmp_uav[i], "Got unexpected unordered access view %p in slot %u.\n", tmp_uav[i], i);
    }

    ID3D11DeviceContext_IAGetVertexBuffers(context, 0, D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT,
            tmp_buffer, stride, offset);
    for (i = 0; i < D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected vertex buffer %p in slot %u.\n", tmp_buffer[i], i);
        ok(!stride[i], "Got unexpected stride %u in slot %u.\n", stride[i], i);
        ok(!offset[i], "Got unexpected offset %u in slot %u.\n", offset[i], i);
    }
    ID3D11DeviceContext_IAGetIndexBuffer(context, tmp_buffer, &format, offset);
    ok(!tmp_buffer[0], "Got unexpected index buffer %p.\n", tmp_buffer[0]);
    ok(format == DXGI_FORMAT_UNKNOWN, "Got unexpected index buffer format %#x.\n", format);
    ok(!offset[0], "Got unexpected index buffer offset %u.\n", offset[0]);
    ID3D11DeviceContext_IAGetInputLayout(context, &tmp_input_layout);
    ok(!tmp_input_layout, "Got unexpected input layout %p.\n", tmp_input_layout);
    ID3D11DeviceContext_IAGetPrimitiveTopology(context, &topology);
    ok(topology == D3D11_PRIMITIVE_TOPOLOGY_UNDEFINED, "Got unexpected primitive topology %#x.\n", topology);

    ID3D11DeviceContext_OMGetBlendState(context, &tmp_blend_state, blend_factor, &sample_mask);
    ok(!tmp_blend_state, "Got unexpected blend state %p.\n", tmp_blend_state);
    ok(blend_factor[0] == 1.0f && blend_factor[1] == 1.0f
            && blend_factor[2] == 1.0f && blend_factor[3] == 1.0f,
            "Got unexpected blend factor {%.8e, %.8e, %.8e, %.8e}.\n",
            blend_factor[0], blend_factor[1], blend_factor[2], blend_factor[3]);
    ok(sample_mask == D3D11_DEFAULT_SAMPLE_MASK, "Got unexpected sample mask %#x.\n", sample_mask);
    ID3D11DeviceContext_OMGetDepthStencilState(context, &tmp_ds_state, &stencil_ref);
    ok(!tmp_ds_state, "Got unexpected depth stencil state %p.\n", tmp_ds_state);
    ok(!stencil_ref, "Got unexpected stencil ref %u.\n", stencil_ref);
    ID3D11DeviceContext_OMGetRenderTargets(context, D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT, tmp_rtv, &tmp_dsv);
    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT; ++i)
    {
        ok(!tmp_rtv[i], "Got unexpected render target view %p in slot %u.\n", tmp_rtv[i], i);
    }
    ok(!tmp_dsv, "Got unexpected depth stencil view %p.\n", tmp_dsv);
    ID3D11DeviceContext_OMGetRenderTargetsAndUnorderedAccessViews(context,
            D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT, tmp_rtv, &tmp_dsv,
            0, D3D11_PS_CS_UAV_REGISTER_COUNT, tmp_uav);
    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT; ++i)
    {
        ok(!tmp_rtv[i], "Got unexpected render target view %p in slot %u.\n", tmp_rtv[i], i);
    }
    ok(!tmp_dsv, "Got unexpected depth stencil view %p.\n", tmp_dsv);
    for (i = 0; i < D3D11_PS_CS_UAV_REGISTER_COUNT; ++i)
    {
        ok(!tmp_uav[i], "Got unexpected unordered access view %p in slot %u.\n", tmp_uav[i], i);
    }

    if (!enable_debug_layer)
    {
        ID3D11DeviceContext_RSGetScissorRects(context, &count, NULL);
        ok(!count, "Got unexpected scissor rect count %u.\n", count);
    }
    memset(tmp_rect, 0x55, sizeof(tmp_rect));
    count = D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE;
    ID3D11DeviceContext_RSGetScissorRects(context, &count, tmp_rect);
    for (i = 0; i < D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE; ++i)
    {
        ok(!tmp_rect[i].left && !tmp_rect[i].top && !tmp_rect[i].right && !tmp_rect[i].bottom,
                "Got unexpected scissor rect %s in slot %u.\n", wine_dbgstr_rect(&tmp_rect[i]), i);
    }
    ID3D11DeviceContext_RSGetViewports(context, &count, NULL);
    ok(!count, "Got unexpected viewport count %u.\n", count);
    memset(tmp_viewport, 0x55, sizeof(tmp_viewport));
    count = D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE;
    ID3D11DeviceContext_RSGetViewports(context, &count, tmp_viewport);
    for (i = 0; i < D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE; ++i)
    {
        ok(!tmp_viewport[i].TopLeftX && !tmp_viewport[i].TopLeftY && !tmp_viewport[i].Width
                && !tmp_viewport[i].Height && !tmp_viewport[i].MinDepth && !tmp_viewport[i].MaxDepth,
                "Got unexpected viewport {%.8e, %.8e, %.8e, %.8e, %.8e, %.8e} in slot %u.\n",
                tmp_viewport[i].TopLeftX, tmp_viewport[i].TopLeftY, tmp_viewport[i].Width,
                tmp_viewport[i].Height, tmp_viewport[i].MinDepth, tmp_viewport[i].MaxDepth, i);
    }
    ID3D11DeviceContext_RSGetState(context, &tmp_rs_state);
    ok(!tmp_rs_state, "Got unexpected rasterizer state %p.\n", tmp_rs_state);

    ID3D11DeviceContext_SOGetTargets(context, D3D11_SO_BUFFER_SLOT_COUNT, tmp_buffer);
    for (i = 0; i < D3D11_SO_BUFFER_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected stream output %p in slot %u.\n", tmp_buffer[i], i);
    }

    ID3D11DeviceContext_GetPredication(context, &tmp_predicate, &predicate_value);
    ok(!tmp_predicate, "Got unexpected predicate %p.\n", tmp_predicate);
    ok(!predicate_value, "Got unexpected predicate value %#x.\n", predicate_value);

    /* Create resources. */

    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
        cb[i] = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, 1024, NULL);

    for (i = 0; i < D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        buffer[i] = create_buffer(device,
                D3D11_BIND_VERTEX_BUFFER | D3D11_BIND_INDEX_BUFFER | D3D11_BIND_SHADER_RESOURCE,
                1024, NULL);

        stride[i] = (i + 1) * 4;
        offset[i] = (i + 1) * 16;
    }

    for (i = 0; i < D3D11_SO_BUFFER_SLOT_COUNT; ++i)
        so_buffer[i] = create_buffer(device, D3D11_BIND_STREAM_OUTPUT, 1024, NULL);

    srv_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    srv_desc.ViewDimension = D3D11_SRV_DIMENSION_BUFFER;
    U(srv_desc).Buffer.ElementOffset = 0;
    U(srv_desc).Buffer.ElementWidth = 64;

    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        hr = ID3D11Device_CreateShaderResourceView(device,
                (ID3D11Resource *)buffer[i % D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT], &srv_desc, &srv[i]);
        ok(hr == S_OK, "View %u: Got unexpected hr %#lx.\n", i, hr);
    }

    uav_desc.Format = DXGI_FORMAT_R32_UINT;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 8;
    U(uav_desc).Buffer.Flags = 0;

    for (i = 0; i < D3D11_PS_CS_UAV_REGISTER_COUNT; ++i)
    {
        cs_uav_buffer[i] = create_buffer(device, D3D11_BIND_UNORDERED_ACCESS, 32, NULL);
        hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)cs_uav_buffer[i],
                &uav_desc, &cs_uav[i]);
        ok(hr == S_OK, "View %u: Got unexpected hr %#lx.\n", i, hr);
    }

    ps_uav_buffer = create_buffer(device, D3D11_BIND_UNORDERED_ACCESS, 32, NULL);
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)ps_uav_buffer,
            &uav_desc, &ps_uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 16;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = 16.0f;

    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        sampler_desc.MinLOD = (float)i;

        hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler[i]);
        ok(hr == S_OK, "Sampler %u: Got unexpected hr %#lx.\n", i, hr);
    }

    hr = ID3D11Device_CreateVertexShader(device, simple_vs, sizeof(simple_vs), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateHullShader(device, simple_hs, sizeof(simple_hs), NULL, &hs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateDomainShader(device, simple_ds, sizeof(simple_ds), NULL, &ds);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateGeometryShader(device, simple_gs, sizeof(simple_gs), NULL, &gs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, simple_ps, sizeof(simple_ps), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateComputeShader(device, simple_cs, sizeof(simple_cs), NULL, &cs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            simple_vs, sizeof(simple_vs), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    memset(&blend_desc, 0, sizeof(blend_desc));
    blend_desc.AlphaToCoverageEnable = FALSE;
    blend_desc.IndependentBlendEnable = FALSE;
    blend_desc.RenderTarget[0].BlendEnable = TRUE;
    blend_desc.RenderTarget[0].SrcBlend = D3D11_BLEND_ONE;
    blend_desc.RenderTarget[0].DestBlend = D3D11_BLEND_ZERO;
    blend_desc.RenderTarget[0].BlendOp = D3D11_BLEND_OP_ADD;
    blend_desc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_ONE;
    blend_desc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_ZERO;
    blend_desc.RenderTarget[0].BlendOpAlpha = D3D11_BLEND_OP_ADD;
    blend_desc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;

    hr = ID3D11Device_CreateBlendState(device, &blend_desc, &blend_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ds_desc.DepthEnable = TRUE;
    ds_desc.DepthWriteMask = D3D11_DEPTH_WRITE_MASK_ALL;
    ds_desc.DepthFunc = D3D11_COMPARISON_LESS;
    ds_desc.StencilEnable = FALSE;
    ds_desc.StencilReadMask = D3D11_DEFAULT_STENCIL_READ_MASK;
    ds_desc.StencilWriteMask = D3D11_DEFAULT_STENCIL_WRITE_MASK;
    ds_desc.FrontFace.StencilFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilDepthFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilPassOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.FrontFace.StencilFunc = D3D11_COMPARISON_ALWAYS;
    ds_desc.BackFace.StencilFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilDepthFailOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilPassOp = D3D11_STENCIL_OP_KEEP;
    ds_desc.BackFace.StencilFunc = D3D11_COMPARISON_ALWAYS;

    hr = ID3D11Device_CreateDepthStencilState(device, &ds_desc, &ds_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture_desc.Width = 512;
    texture_desc.Height = 512;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT; ++i)
    {
        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &rt_texture[i]);
        ok(hr == S_OK, "Texture %u: Got unexpected hr %#lx.\n", i, hr);
    }

    texture_desc.Format = DXGI_FORMAT_D24_UNORM_S8_UINT;
    texture_desc.BindFlags = D3D11_BIND_DEPTH_STENCIL;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &ds_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT; ++i)
    {
        hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)rt_texture[i], NULL, &rtv[i]);
        ok(hr == S_OK, "View %u: Got unexpected hr %#lx.\n", i, hr);
    }

    hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)ds_texture, NULL, &dsv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE; ++i)
    {
        SetRect(&tmp_rect[i], i, i * 2, i + 1, (i + 1) * 2);

        tmp_viewport[i].TopLeftX = i * 3;
        tmp_viewport[i].TopLeftY = i * 4;
        tmp_viewport[i].Width = 3;
        tmp_viewport[i].Height = 4;
        tmp_viewport[i].MinDepth = i * 0.01f;
        tmp_viewport[i].MaxDepth = (i + 1) * 0.01f;
    }

    rs_desc.FillMode = D3D11_FILL_SOLID;
    rs_desc.CullMode = D3D11_CULL_BACK;
    rs_desc.FrontCounterClockwise = FALSE;
    rs_desc.DepthBias = 0;
    rs_desc.DepthBiasClamp = 0.0f;
    rs_desc.SlopeScaledDepthBias = 0.0f;
    rs_desc.DepthClipEnable = TRUE;
    rs_desc.ScissorEnable = FALSE;
    rs_desc.MultisampleEnable = FALSE;
    rs_desc.AntialiasedLineEnable = FALSE;

    hr = ID3D11Device_CreateRasterizerState(device, &rs_desc, &rs_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    predicate_desc.Query = D3D11_QUERY_OCCLUSION_PREDICATE;
    predicate_desc.MiscFlags = 0;

    hr = ID3D11Device_CreatePredicate(device, &predicate_desc, &predicate);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    /* Setup state. */

    /* Some versions of Windows AMD drivers hang while the device is being
     * released, if the total number of used resource slots exceeds some limit.
     * Do not use all constant buffers slots in order to not trigger this
     * driver bug. */
    ID3D11DeviceContext_VSSetConstantBuffers(context, 0, 1, &cb[0]);
    ID3D11DeviceContext_VSSetConstantBuffers(context, 7, 1, &cb[7]);
    ID3D11DeviceContext_VSSetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, srv);
    ID3D11DeviceContext_VSSetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, sampler);
    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);

    ID3D11DeviceContext_HSSetConstantBuffers(context, 0, 1, &cb[0]);
    ID3D11DeviceContext_HSSetConstantBuffers(context, 7, 1, &cb[7]);
    ID3D11DeviceContext_HSSetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, srv);
    ID3D11DeviceContext_HSSetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, sampler);
    ID3D11DeviceContext_HSSetShader(context, hs, NULL, 0);

    ID3D11DeviceContext_DSSetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT, cb);
    ID3D11DeviceContext_DSSetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, srv);
    ID3D11DeviceContext_DSSetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, sampler);
    ID3D11DeviceContext_DSSetShader(context, ds, NULL, 0);

    ID3D11DeviceContext_GSSetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT, cb);
    ID3D11DeviceContext_GSSetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, srv);
    ID3D11DeviceContext_GSSetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, sampler);
    ID3D11DeviceContext_GSSetShader(context, gs, NULL, 0);

    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT, cb);
    ID3D11DeviceContext_PSSetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, srv);
    ID3D11DeviceContext_PSSetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, sampler);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11DeviceContext_CSSetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT, cb);
    ID3D11DeviceContext_CSSetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, srv);
    ID3D11DeviceContext_CSSetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, sampler);
    ID3D11DeviceContext_CSSetShader(context, cs, NULL, 0);
    ID3D11DeviceContext_CSSetUnorderedAccessViews(context, 0, D3D11_PS_CS_UAV_REGISTER_COUNT, cs_uav, NULL);

    ID3D11DeviceContext_IASetVertexBuffers(context, 0, D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT,
            buffer, stride, offset);
    ID3D11DeviceContext_IASetIndexBuffer(context, buffer[0], DXGI_FORMAT_R32_UINT, offset[0]);
    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);

    blend_factor[0] = 0.1f;
    blend_factor[1] = 0.2f;
    blend_factor[2] = 0.3f;
    blend_factor[3] = 0.4f;
    ID3D11DeviceContext_OMSetBlendState(context, blend_state, blend_factor, 0xff00ff00);
    ID3D11DeviceContext_OMSetDepthStencilState(context, ds_state, 3);
    ID3D11DeviceContext_OMSetRenderTargetsAndUnorderedAccessViews(context,
            D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT - 1, rtv, dsv,
            D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT - 1, 1, &ps_uav, NULL);

    ID3D11DeviceContext_RSSetScissorRects(context, D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE,
            tmp_rect);
    ID3D11DeviceContext_RSSetViewports(context, D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE,
            tmp_viewport);
    ID3D11DeviceContext_RSSetState(context, rs_state);

    ID3D11DeviceContext_SOSetTargets(context, D3D11_SO_BUFFER_SLOT_COUNT, so_buffer, offset);

    ID3D11DeviceContext_SetPredication(context, predicate, TRUE);

    /* Verify the set state. */

    ID3D11DeviceContext_VSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ID3D11Buffer *expected_cb = i % 7 ? NULL : cb[i];
        ok(tmp_buffer[i] == expected_cb, "Got unexpected constant buffer %p in slot %u, expected %p.\n",
                tmp_buffer[i], i, expected_cb);
        if (tmp_buffer[i])
            ID3D11Buffer_Release(tmp_buffer[i]);
    }
    ID3D11DeviceContext_VSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(tmp_srv[i] == srv[i], "Got unexpected shader resource view %p in slot %u, expected %p.\n",
                tmp_srv[i], i, srv[i]);
        ID3D11ShaderResourceView_Release(tmp_srv[i]);
    }
    ID3D11DeviceContext_VSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(tmp_sampler[i] == sampler[i], "Got unexpected sampler %p in slot %u, expected %p.\n",
                tmp_sampler[i], i, sampler[i]);
        ID3D11SamplerState_Release(tmp_sampler[i]);
    }
    ID3D11DeviceContext_VSGetShader(context, &tmp_vs, NULL, 0);
    ok(tmp_vs == vs, "Got unexpected vertex shader %p, expected %p.\n", tmp_vs, vs);
    ID3D11VertexShader_Release(tmp_vs);

    ID3D11DeviceContext_HSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ID3D11Buffer *expected_cb = i % 7 ? NULL : cb[i];
        ok(tmp_buffer[i] == expected_cb, "Got unexpected constant buffer %p in slot %u, expected %p.\n",
                tmp_buffer[i], i, expected_cb);
        if (tmp_buffer[i])
            ID3D11Buffer_Release(tmp_buffer[i]);
    }
    ID3D11DeviceContext_HSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(tmp_srv[i] == srv[i], "Got unexpected shader resource view %p in slot %u, expected %p.\n",
                tmp_srv[i], i, srv[i]);
        ID3D11ShaderResourceView_Release(tmp_srv[i]);
    }
    ID3D11DeviceContext_HSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(tmp_sampler[i] == sampler[i], "Got unexpected sampler %p in slot %u, expected %p.\n",
                tmp_sampler[i], i, sampler[i]);
        ID3D11SamplerState_Release(tmp_sampler[i]);
    }
    ID3D11DeviceContext_HSGetShader(context, &tmp_hs, NULL, 0);
    ok(tmp_hs == hs, "Got unexpected hull shader %p, expected %p.\n", tmp_hs, hs);
    ID3D11HullShader_Release(tmp_hs);

    ID3D11DeviceContext_DSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(tmp_buffer[i] == cb[i], "Got unexpected constant buffer %p in slot %u, expected %p.\n",
                tmp_buffer[i], i, cb[i]);
        ID3D11Buffer_Release(tmp_buffer[i]);
    }
    ID3D11DeviceContext_DSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(tmp_srv[i] == srv[i], "Got unexpected shader resource view %p in slot %u, expected %p.\n",
                tmp_srv[i], i, srv[i]);
        ID3D11ShaderResourceView_Release(tmp_srv[i]);
    }
    ID3D11DeviceContext_DSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(tmp_sampler[i] == sampler[i], "Got unexpected sampler %p in slot %u, expected %p.\n",
                tmp_sampler[i], i, sampler[i]);
        ID3D11SamplerState_Release(tmp_sampler[i]);
    }
    ID3D11DeviceContext_DSGetShader(context, &tmp_ds, NULL, 0);
    ok(tmp_ds == ds, "Got unexpected domain shader %p, expected %p.\n", tmp_ds, ds);
    ID3D11DomainShader_Release(tmp_ds);

    ID3D11DeviceContext_GSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(tmp_buffer[i] == cb[i], "Got unexpected constant buffer %p in slot %u, expected %p.\n",
                tmp_buffer[i], i, cb[i]);
        ID3D11Buffer_Release(tmp_buffer[i]);
    }
    ID3D11DeviceContext_GSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(tmp_srv[i] == srv[i], "Got unexpected shader resource view %p in slot %u, expected %p.\n",
                tmp_srv[i], i, srv[i]);
        ID3D11ShaderResourceView_Release(tmp_srv[i]);
    }
    ID3D11DeviceContext_GSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(tmp_sampler[i] == sampler[i], "Got unexpected sampler %p in slot %u, expected %p.\n",
                tmp_sampler[i], i, sampler[i]);
        ID3D11SamplerState_Release(tmp_sampler[i]);
    }
    ID3D11DeviceContext_GSGetShader(context, &tmp_gs, NULL, 0);
    ok(tmp_gs == gs, "Got unexpected geometry shader %p, expected %p.\n", tmp_gs, gs);
    ID3D11GeometryShader_Release(tmp_gs);

    ID3D11DeviceContext_PSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(tmp_buffer[i] == cb[i], "Got unexpected constant buffer %p in slot %u, expected %p.\n",
                tmp_buffer[i], i, cb[i]);
        ID3D11Buffer_Release(tmp_buffer[i]);
    }
    ID3D11DeviceContext_PSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(tmp_srv[i] == srv[i], "Got unexpected shader resource view %p in slot %u, expected %p.\n",
                tmp_srv[i], i, srv[i]);
        ID3D11ShaderResourceView_Release(tmp_srv[i]);
    }
    ID3D11DeviceContext_PSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(tmp_sampler[i] == sampler[i], "Got unexpected sampler %p in slot %u, expected %p.\n",
                tmp_sampler[i], i, sampler[i]);
        ID3D11SamplerState_Release(tmp_sampler[i]);
    }
    ID3D11DeviceContext_PSGetShader(context, &tmp_ps, NULL, 0);
    ok(tmp_ps == ps, "Got unexpected pixel shader %p, expected %p.\n", tmp_ps, ps);
    ID3D11PixelShader_Release(tmp_ps);

    ID3D11DeviceContext_CSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(tmp_buffer[i] == cb[i], "Got unexpected constant buffer %p in slot %u, expected %p.\n",
                tmp_buffer[i], i, cb[i]);
        ID3D11Buffer_Release(tmp_buffer[i]);
    }
    ID3D11DeviceContext_CSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(tmp_srv[i] == srv[i], "Got unexpected shader resource view %p in slot %u, expected %p.\n",
                tmp_srv[i], i, srv[i]);
        ID3D11ShaderResourceView_Release(tmp_srv[i]);
    }
    ID3D11DeviceContext_CSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(tmp_sampler[i] == sampler[i], "Got unexpected sampler %p in slot %u, expected %p.\n",
                tmp_sampler[i], i, sampler[i]);
        ID3D11SamplerState_Release(tmp_sampler[i]);
    }
    ID3D11DeviceContext_CSGetShader(context, &tmp_cs, NULL, 0);
    ok(tmp_cs == cs, "Got unexpected compute shader %p, expected %p.\n", tmp_cs, cs);
    ID3D11ComputeShader_Release(tmp_cs);
    ID3D11DeviceContext_CSGetUnorderedAccessViews(context, 0, D3D11_PS_CS_UAV_REGISTER_COUNT, tmp_uav);
    for (i = 0; i < D3D11_PS_CS_UAV_REGISTER_COUNT; ++i)
    {
        ok(tmp_uav[i] == cs_uav[i], "Got unexpected unordered access view %p in slot %u, expected %p.\n",
                tmp_uav[i], i, cs_uav[i]);
        ID3D11UnorderedAccessView_Release(tmp_uav[i]);
    }

    ID3D11DeviceContext_IAGetVertexBuffers(context, 0, D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT,
            tmp_buffer, stride, offset);
    for (i = 0; i < D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        todo_wine_if(i >= D3D10_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT)
        {
        ok(tmp_buffer[i] == buffer[i], "Got unexpected vertex buffer %p in slot %u, expected %p.\n",
                tmp_buffer[i], i, buffer[i]);
        ok(stride[i] == (i + 1) * 4, "Got unexpected stride %u in slot %u.\n", stride[i], i);
        ok(offset[i] == (i + 1) * 16, "Got unexpected offset %u in slot %u.\n", offset[i], i);
        }
        if (tmp_buffer[i])
            ID3D11Buffer_Release(tmp_buffer[i]);
    }
    ID3D11DeviceContext_IAGetIndexBuffer(context, tmp_buffer, &format, offset);
    ok(tmp_buffer[0] == buffer[0], "Got unexpected index buffer %p, expected %p.\n", tmp_buffer[0], buffer[0]);
    ID3D11Buffer_Release(tmp_buffer[0]);
    ok(format == DXGI_FORMAT_R32_UINT, "Got unexpected index buffer format %#x.\n", format);
    ok(offset[0] == 16, "Got unexpected index buffer offset %u.\n", offset[0]);
    ID3D11DeviceContext_IAGetInputLayout(context, &tmp_input_layout);
    ok(tmp_input_layout == input_layout, "Got unexpected input layout %p, expected %p.\n",
            tmp_input_layout, input_layout);
    ID3D11InputLayout_Release(tmp_input_layout);
    ID3D11DeviceContext_IAGetPrimitiveTopology(context, &topology);
    ok(topology == D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP, "Got unexpected primitive topology %#x.\n", topology);

    ID3D11DeviceContext_OMGetBlendState(context, &tmp_blend_state, blend_factor, &sample_mask);
    ok(tmp_blend_state == blend_state, "Got unexpected blend state %p, expected %p.\n", tmp_blend_state, blend_state);
    ID3D11BlendState_Release(tmp_blend_state);
    ok(blend_factor[0] == 0.1f && blend_factor[1] == 0.2f
            && blend_factor[2] == 0.3f && blend_factor[3] == 0.4f,
            "Got unexpected blend factor {%.8e, %.8e, %.8e, %.8e}.\n",
            blend_factor[0], blend_factor[1], blend_factor[2], blend_factor[3]);
    ok(sample_mask == 0xff00ff00, "Got unexpected sample mask %#x.\n", sample_mask);
    ID3D11DeviceContext_OMGetDepthStencilState(context, &tmp_ds_state, &stencil_ref);
    ok(tmp_ds_state == ds_state, "Got unexpected depth stencil state %p, expected %p.\n", tmp_ds_state, ds_state);
    ID3D11DepthStencilState_Release(tmp_ds_state);
    ok(stencil_ref == 3, "Got unexpected stencil ref %u.\n", stencil_ref);
    /* For OMGetDepthStencilState() both arguments are optional. */
    ID3D11DeviceContext_OMGetDepthStencilState(context, NULL, NULL);
    stencil_ref = 0;
    ID3D11DeviceContext_OMGetDepthStencilState(context, NULL, &stencil_ref);
    ok(stencil_ref == 3, "Got unexpected stencil ref %u.\n", stencil_ref);
    tmp_ds_state = NULL;
    ID3D11DeviceContext_OMGetDepthStencilState(context, &tmp_ds_state, NULL);
    ok(tmp_ds_state == ds_state, "Got unexpected depth stencil state %p, expected %p.\n", tmp_ds_state, ds_state);
    ID3D11DepthStencilState_Release(tmp_ds_state);
    /* OMGetBlendState() arguments are optional */
    ID3D11DeviceContext_OMGetBlendState(context, NULL, NULL, NULL);
    ID3D11DeviceContext_OMGetBlendState(context, &tmp_blend_state, NULL, NULL);
    ok(tmp_blend_state == blend_state, "Got unexpected blend state %p, expected %p.\n", tmp_blend_state, blend_state);
    ID3D11BlendState_Release(tmp_blend_state);
    sample_mask = 0;
    ID3D11DeviceContext_OMGetBlendState(context, NULL, NULL, &sample_mask);
    ok(sample_mask == 0xff00ff00, "Got unexpected sample mask %#x.\n", sample_mask);
    memset(blend_factor, 0, sizeof(blend_factor));
    ID3D11DeviceContext_OMGetBlendState(context, NULL, blend_factor, NULL);
    ok(blend_factor[0] == 0.1f && blend_factor[1] == 0.2f
            && blend_factor[2] == 0.3f && blend_factor[3] == 0.4f,
            "Got unexpected blend factor {%.8e, %.8e, %.8e, %.8e}.\n",
            blend_factor[0], blend_factor[1], blend_factor[2], blend_factor[3]);

    ID3D11DeviceContext_OMGetRenderTargets(context, D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT, tmp_rtv, &tmp_dsv);
    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT - 1; ++i)
    {
        ok(tmp_rtv[i] == rtv[i], "Got unexpected render target view %p in slot %u, expected %p.\n",
                tmp_rtv[i], i, rtv[i]);
        ID3D11RenderTargetView_Release(tmp_rtv[i]);
    }
    ok(!tmp_rtv[i], "Got unexpected render target view %p in slot %u.\n", tmp_rtv[i], i);
    ok(tmp_dsv == dsv, "Got unexpected depth stencil view %p, expected %p.\n", tmp_dsv, dsv);
    ID3D11DepthStencilView_Release(tmp_dsv);
    ID3D11DeviceContext_OMGetRenderTargetsAndUnorderedAccessViews(context,
            D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT, tmp_rtv, &tmp_dsv,
            0, D3D11_PS_CS_UAV_REGISTER_COUNT, tmp_uav);
    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT - 1; ++i)
    {
        ok(tmp_rtv[i] == rtv[i], "Got unexpected render target view %p in slot %u, expected %p.\n",
                tmp_rtv[i], i, rtv[i]);
        ID3D11RenderTargetView_Release(tmp_rtv[i]);
    }
    ok(!tmp_rtv[i], "Got unexpected render target view %p in slot %u.\n", tmp_rtv[i], i);
    ok(tmp_dsv == dsv, "Got unexpected depth stencil view %p, expected %p.\n", tmp_dsv, dsv);
    ID3D11DepthStencilView_Release(tmp_dsv);
    for (i = 0; i < D3D11_PS_CS_UAV_REGISTER_COUNT - 1; ++i)
    {
        ok(!tmp_uav[i], "Got unexpected unordered access view %p in slot %u.\n", tmp_uav[i], i);
    }
    ok(tmp_uav[i] == ps_uav, "Got unexpected unordered access view %p in slot %u, expected %p.\n",
            tmp_uav[i], i, ps_uav);
    ID3D11UnorderedAccessView_Release(tmp_uav[i]);

    ID3D11DeviceContext_RSGetScissorRects(context, &count, NULL);
    ok(count == D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE,
            "Got unexpected scissor rect count %u.\n", count);
    memset(tmp_rect, 0x55, sizeof(tmp_rect));
    ID3D11DeviceContext_RSGetScissorRects(context, &count, tmp_rect);
    for (i = 0; i < count; ++i)
    {
        ok(tmp_rect[i].left == i
                && tmp_rect[i].top == i * 2
                && tmp_rect[i].right == i + 1
                && tmp_rect[i].bottom == (i + 1) * 2,
                "Got unexpected scissor rect %s in slot %u.\n", wine_dbgstr_rect(&tmp_rect[i]), i);
    }
    ID3D11DeviceContext_RSGetViewports(context, &count, NULL);
    ok(count == D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE,
            "Got unexpected viewport count %u.\n", count);
    memset(tmp_viewport, 0x55, sizeof(tmp_viewport));
    ID3D11DeviceContext_RSGetViewports(context, &count, tmp_viewport);
    for (i = 0; i < count; ++i)
    {
        ok(tmp_viewport[i].TopLeftX == i * 3
                && tmp_viewport[i].TopLeftY == i * 4
                && tmp_viewport[i].Width == 3
                && tmp_viewport[i].Height == 4
                && compare_float(tmp_viewport[i].MinDepth, i * 0.01f, 16)
                && compare_float(tmp_viewport[i].MaxDepth, (i + 1) * 0.01f, 16),
                "Got unexpected viewport {%.8e, %.8e, %.8e, %.8e, %.8e, %.8e} in slot %u.\n",
                tmp_viewport[i].TopLeftX, tmp_viewport[i].TopLeftY, tmp_viewport[i].Width,
                tmp_viewport[i].Height, tmp_viewport[i].MinDepth, tmp_viewport[i].MaxDepth, i);
    }
    ID3D11DeviceContext_RSGetState(context, &tmp_rs_state);
    ok(tmp_rs_state == rs_state, "Got unexpected rasterizer state %p, expected %p.\n", tmp_rs_state, rs_state);
    ID3D11RasterizerState_Release(tmp_rs_state);

    ID3D11DeviceContext_SOGetTargets(context, D3D11_SO_BUFFER_SLOT_COUNT, tmp_buffer);
    for (i = 0; i < D3D11_SO_BUFFER_SLOT_COUNT; ++i)
    {
        ok(tmp_buffer[i] == so_buffer[i], "Got unexpected stream output %p in slot %u, expected %p.\n",
                tmp_buffer[i], i, so_buffer[i]);
        ID3D11Buffer_Release(tmp_buffer[i]);
    }

    ID3D11DeviceContext_GetPredication(context, &tmp_predicate, &predicate_value);
    ok(tmp_predicate == predicate, "Got unexpected predicate %p, expected %p.\n", tmp_predicate, predicate);
    ID3D11Predicate_Release(tmp_predicate);
    ok(predicate_value, "Got unexpected predicate value %#x.\n", predicate_value);

    /* Verify ClearState(). */

    ID3D11DeviceContext_ClearState(context);

    ID3D11DeviceContext_VSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_VSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_VSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    ID3D11DeviceContext_VSGetShader(context, &tmp_vs, NULL, 0);
    ok(!tmp_vs, "Got unexpected vertex shader %p.\n", tmp_vs);

    ID3D11DeviceContext_HSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_HSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_HSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    ID3D11DeviceContext_HSGetShader(context, &tmp_hs, NULL, 0);
    ok(!tmp_hs, "Got unexpected hull shader %p.\n", tmp_hs);

    ID3D11DeviceContext_DSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_DSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_DSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    ID3D11DeviceContext_DSGetShader(context, &tmp_ds, NULL, 0);
    ok(!tmp_ds, "Got unexpected domain shader %p.\n", tmp_ds);

    ID3D11DeviceContext_GSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_GSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_GSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    ID3D11DeviceContext_GSGetShader(context, &tmp_gs, NULL, 0);
    ok(!tmp_gs, "Got unexpected geometry shader %p.\n", tmp_gs);

    ID3D11DeviceContext_PSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_PSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT, tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_PSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    ID3D11DeviceContext_PSGetShader(context, &tmp_ps, NULL, 0);
    ok(!tmp_ps, "Got unexpected pixel shader %p.\n", tmp_ps);

    ID3D11DeviceContext_CSGetConstantBuffers(context, 0, D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT,
            tmp_buffer);
    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected constant buffer %p in slot %u.\n", tmp_buffer[i], i);
    }
    ID3D11DeviceContext_CSGetShaderResources(context, 0, D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT,
            tmp_srv);
    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_srv[i], "Got unexpected shader resource view %p in slot %u.\n", tmp_srv[i], i);
    }
    ID3D11DeviceContext_CSGetSamplers(context, 0, D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT, tmp_sampler);
    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ok(!tmp_sampler[i], "Got unexpected sampler %p in slot %u.\n", tmp_sampler[i], i);
    }
    ID3D11DeviceContext_CSGetShader(context, &tmp_cs, NULL, 0);
    ok(!tmp_cs, "Got unexpected compute shader %p.\n", tmp_cs);
    ID3D11DeviceContext_CSGetUnorderedAccessViews(context, 0, D3D11_PS_CS_UAV_REGISTER_COUNT, tmp_uav);
    for (i = 0; i < D3D11_PS_CS_UAV_REGISTER_COUNT; ++i)
    {
        ok(!tmp_uav[i], "Got unexpected unordered access view %p in slot %u.\n", tmp_uav[i], i);
    }

    ID3D11DeviceContext_IAGetVertexBuffers(context, 0, D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT,
            tmp_buffer, stride, offset);
    for (i = 0; i < D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected vertex buffer %p in slot %u.\n", tmp_buffer[i], i);
        ok(!stride[i], "Got unexpected stride %u in slot %u.\n", stride[i], i);
        ok(!offset[i], "Got unexpected offset %u in slot %u.\n", offset[i], i);
    }
    ID3D11DeviceContext_IAGetIndexBuffer(context, tmp_buffer, &format, offset);
    ok(!tmp_buffer[0], "Got unexpected index buffer %p.\n", tmp_buffer[0]);
    ok(format == DXGI_FORMAT_UNKNOWN, "Got unexpected index buffer format %#x.\n", format);
    ok(!offset[0], "Got unexpected index buffer offset %u.\n", offset[0]);
    ID3D11DeviceContext_IAGetInputLayout(context, &tmp_input_layout);
    ok(!tmp_input_layout, "Got unexpected input layout %p.\n", tmp_input_layout);
    ID3D11DeviceContext_IAGetPrimitiveTopology(context, &topology);
    ok(topology == D3D11_PRIMITIVE_TOPOLOGY_UNDEFINED, "Got unexpected primitive topology %#x.\n", topology);

    ID3D11DeviceContext_OMGetBlendState(context, &tmp_blend_state, blend_factor, &sample_mask);
    ok(!tmp_blend_state, "Got unexpected blend state %p.\n", tmp_blend_state);
    ok(blend_factor[0] == 1.0f && blend_factor[1] == 1.0f
            && blend_factor[2] == 1.0f && blend_factor[3] == 1.0f,
            "Got unexpected blend factor {%.8e, %.8e, %.8e, %.8e}.\n",
            blend_factor[0], blend_factor[1], blend_factor[2], blend_factor[3]);
    ok(sample_mask == D3D11_DEFAULT_SAMPLE_MASK, "Got unexpected sample mask %#x.\n", sample_mask);
    ID3D11DeviceContext_OMGetDepthStencilState(context, &tmp_ds_state, &stencil_ref);
    ok(!tmp_ds_state, "Got unexpected depth stencil state %p.\n", tmp_ds_state);
    ok(!stencil_ref, "Got unexpected stencil ref %u.\n", stencil_ref);
    ID3D11DeviceContext_OMGetRenderTargets(context, D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT, tmp_rtv, &tmp_dsv);
    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT; ++i)
    {
        ok(!tmp_rtv[i], "Got unexpected render target view %p in slot %u.\n", tmp_rtv[i], i);
    }
    ok(!tmp_dsv, "Got unexpected depth stencil view %p.\n", tmp_dsv);
    ID3D11DeviceContext_OMGetRenderTargetsAndUnorderedAccessViews(context,
            D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT, tmp_rtv, &tmp_dsv,
            0, D3D11_PS_CS_UAV_REGISTER_COUNT, tmp_uav);
    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT; ++i)
    {
        ok(!tmp_rtv[i], "Got unexpected render target view %p in slot %u.\n", tmp_rtv[i], i);
    }
    ok(!tmp_dsv, "Got unexpected depth stencil view %p.\n", tmp_dsv);
    for (i = 0; i < D3D11_PS_CS_UAV_REGISTER_COUNT; ++i)
    {
        ok(!tmp_uav[i], "Got unexpected unordered access view %p in slot %u.\n", tmp_uav[i], i);
    }

    ID3D11DeviceContext_RSGetScissorRects(context, &count, NULL);
    ok(!count, "Got unexpected scissor rect count %u.\n", count);
    memset(tmp_rect, 0x55, sizeof(tmp_rect));
    count = D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE;
    ID3D11DeviceContext_RSGetScissorRects(context, &count, tmp_rect);
    for (i = 0; i < D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE; ++i)
    {
        ok(!tmp_rect[i].left && !tmp_rect[i].top && !tmp_rect[i].right && !tmp_rect[i].bottom,
                "Got unexpected scissor rect %s in slot %u.\n",
                wine_dbgstr_rect(&tmp_rect[i]), i);
    }
    ID3D11DeviceContext_RSGetViewports(context, &count, NULL);
    ok(!count, "Got unexpected viewport count %u.\n", count);
    memset(tmp_viewport, 0x55, sizeof(tmp_viewport));
    count = D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE;
    ID3D11DeviceContext_RSGetViewports(context, &count, tmp_viewport);
    for (i = 0; i < D3D11_VIEWPORT_AND_SCISSORRECT_OBJECT_COUNT_PER_PIPELINE; ++i)
    {
        ok(!tmp_viewport[i].TopLeftX && !tmp_viewport[i].TopLeftY && !tmp_viewport[i].Width
                && !tmp_viewport[i].Height && !tmp_viewport[i].MinDepth && !tmp_viewport[i].MaxDepth,
                "Got unexpected viewport {%.8e, %.8e, %.8e, %.8e, %.8e, %.8e} in slot %u.\n",
                tmp_viewport[i].TopLeftX, tmp_viewport[i].TopLeftY, tmp_viewport[i].Width,
                tmp_viewport[i].Height, tmp_viewport[i].MinDepth, tmp_viewport[i].MaxDepth, i);
    }
    ID3D11DeviceContext_RSGetState(context, &tmp_rs_state);
    ok(!tmp_rs_state, "Got unexpected rasterizer state %p.\n", tmp_rs_state);

    ID3D11DeviceContext_SOGetTargets(context, D3D11_SO_BUFFER_SLOT_COUNT, tmp_buffer);
    for (i = 0; i < D3D11_SO_BUFFER_SLOT_COUNT; ++i)
    {
        ok(!tmp_buffer[i], "Got unexpected stream output %p in slot %u.\n", tmp_buffer[i], i);
    }

    ID3D11DeviceContext_GetPredication(context, &tmp_predicate, &predicate_value);
    ok(!tmp_predicate, "Got unexpected predicate %p.\n", tmp_predicate);
    ok(!predicate_value, "Got unexpected predicate value %#x.\n", predicate_value);

    /* Cleanup. */

    ID3D11Predicate_Release(predicate);
    ID3D11RasterizerState_Release(rs_state);
    ID3D11DepthStencilView_Release(dsv);
    ID3D11Texture2D_Release(ds_texture);

    for (i = 0; i < D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT; ++i)
    {
        ID3D11RenderTargetView_Release(rtv[i]);
        ID3D11Texture2D_Release(rt_texture[i]);
    }

    ID3D11DepthStencilState_Release(ds_state);
    ID3D11BlendState_Release(blend_state);
    ID3D11InputLayout_Release(input_layout);
    ID3D11VertexShader_Release(vs);
    ID3D11HullShader_Release(hs);
    ID3D11DomainShader_Release(ds);
    ID3D11GeometryShader_Release(gs);
    ID3D11PixelShader_Release(ps);
    ID3D11ComputeShader_Release(cs);

    for (i = 0; i < D3D11_COMMONSHADER_SAMPLER_SLOT_COUNT; ++i)
    {
        ID3D11SamplerState_Release(sampler[i]);
    }

    for (i = 0; i < D3D11_COMMONSHADER_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ID3D11ShaderResourceView_Release(srv[i]);
    }

    for (i = 0; i < D3D11_PS_CS_UAV_REGISTER_COUNT; ++i)
    {
        ID3D11UnorderedAccessView_Release(cs_uav[i]);
        ID3D11Buffer_Release(cs_uav_buffer[i]);
    }
    ID3D11UnorderedAccessView_Release(ps_uav);
    ID3D11Buffer_Release(ps_uav_buffer);

    for (i = 0; i < D3D11_SO_BUFFER_SLOT_COUNT; ++i)
    {
        ID3D11Buffer_Release(so_buffer[i]);
    }

    for (i = 0; i < D3D11_IA_VERTEX_INPUT_RESOURCE_SLOT_COUNT; ++i)
    {
        ID3D11Buffer_Release(buffer[i]);
    }

    for (i = 0; i < D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT; ++i)
    {
        ID3D11Buffer_Release(cb[i]);
    }

    ID3D11DeviceContext_Release(context);
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_il_append_aligned(void)
{
    struct d3d11_test_context test_context;
    ID3D11InputLayout *input_layout;
    ID3D11DeviceContext *context;
    unsigned int stride, offset;
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *vb[3];
    DWORD color;
    HRESULT hr;

    /* Semantic names are case-insensitive. */
    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"CoLoR",    2, DXGI_FORMAT_R32G32_FLOAT,       1, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D11_INPUT_PER_INSTANCE_DATA, 2},
        {"ColoR",    3, DXGI_FORMAT_R32G32_FLOAT,       2, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D11_INPUT_PER_INSTANCE_DATA, 1},
        {"POSITION", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"ColoR",    0, DXGI_FORMAT_R32G32_FLOAT,       2, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D11_INPUT_PER_INSTANCE_DATA, 1},
        {"cOLOr",    1, DXGI_FORMAT_R32G32_FLOAT,       1, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D11_INPUT_PER_INSTANCE_DATA, 2},
    };
    static const DWORD vs_code[] =
    {
#if 0
        struct vs_in
        {
            float4 position : POSITION;
            float2 color_xy : COLOR0;
            float2 color_zw : COLOR1;
            unsigned int instance_id : SV_INSTANCEID;
        };

        struct vs_out
        {
            float4 position : SV_POSITION;
            float2 color_xy : COLOR0;
            float2 color_zw : COLOR1;
        };

        struct vs_out main(struct vs_in i)
        {
            struct vs_out o;

            o.position = i.position;
            o.position.x += i.instance_id * 0.5;
            o.color_xy = i.color_xy;
            o.color_zw = i.color_zw;

            return o;
        }
#endif
        0x43425844, 0x52e3bf46, 0x6300403d, 0x624cffe4, 0xa4fc0013, 0x00000001, 0x00000214, 0x00000003,
        0x0000002c, 0x000000bc, 0x00000128, 0x4e475349, 0x00000088, 0x00000004, 0x00000008, 0x00000068,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000071, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000303, 0x00000071, 0x00000001, 0x00000000, 0x00000003, 0x00000002,
        0x00000303, 0x00000077, 0x00000000, 0x00000008, 0x00000001, 0x00000003, 0x00000101, 0x49534f50,
        0x4e4f4954, 0x4c4f4300, 0x5300524f, 0x4e495f56, 0x4e415453, 0x44494543, 0xababab00, 0x4e47534f,
        0x00000064, 0x00000003, 0x00000008, 0x00000050, 0x00000000, 0x00000001, 0x00000003, 0x00000000,
        0x0000000f, 0x0000005c, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x00000c03, 0x0000005c,
        0x00000001, 0x00000000, 0x00000003, 0x00000001, 0x0000030c, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4f4c4f43, 0xabab0052, 0x52444853, 0x000000e4, 0x00010040, 0x00000039, 0x0300005f, 0x001010f2,
        0x00000000, 0x0300005f, 0x00101032, 0x00000001, 0x0300005f, 0x00101032, 0x00000002, 0x04000060,
        0x00101012, 0x00000003, 0x00000008, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x03000065,
        0x00102032, 0x00000001, 0x03000065, 0x001020c2, 0x00000001, 0x02000068, 0x00000001, 0x05000056,
        0x00100012, 0x00000000, 0x0010100a, 0x00000003, 0x09000032, 0x00102012, 0x00000000, 0x0010000a,
        0x00000000, 0x00004001, 0x3f000000, 0x0010100a, 0x00000000, 0x05000036, 0x001020e2, 0x00000000,
        0x00101e56, 0x00000000, 0x05000036, 0x00102032, 0x00000001, 0x00101046, 0x00000001, 0x05000036,
        0x001020c2, 0x00000001, 0x00101406, 0x00000002, 0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        struct vs_out
        {
            float4 position : SV_POSITION;
            float2 color_xy : COLOR0;
            float2 color_zw : COLOR1;
        };

        float4 main(struct vs_out i) : SV_TARGET
        {
            return float4(i.color_xy.xy, i.color_zw.xy);
        }
#endif
        0x43425844, 0x64e48a09, 0xaa484d46, 0xe40a6e78, 0x9885edf3, 0x00000001, 0x00000118, 0x00000003,
        0x0000002c, 0x00000098, 0x000000cc, 0x4e475349, 0x00000064, 0x00000003, 0x00000008, 0x00000050,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x0000005c, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000303, 0x0000005c, 0x00000001, 0x00000000, 0x00000003, 0x00000001,
        0x00000c0c, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052, 0x4e47534f, 0x0000002c,
        0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f,
        0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000044, 0x00000040, 0x00000011, 0x03001062,
        0x00101032, 0x00000001, 0x03001062, 0x001010c2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000001, 0x0100003e,
    };
    static const struct
    {
        struct vec4 position;
    }
    stream0[] =
    {
        {{-1.0f, -1.0f, 0.0f, 1.0f}},
        {{-1.0f,  1.0f, 0.0f, 1.0f}},
        {{-0.5f, -1.0f, 0.0f, 1.0f}},
        {{-0.5f,  1.0f, 0.0f, 1.0f}},
    };
    static const struct
    {
        struct vec2 color2;
        struct vec2 color1;
    }
    stream1[] =
    {
        {{0.5f, 0.5f}, {0.0f, 1.0f}},
        {{0.5f, 0.5f}, {1.0f, 1.0f}},
    };
    static const struct
    {
        struct vec2 color3;
        struct vec2 color0;
    }
    stream2[] =
    {
        {{0.5f, 0.5f}, {1.0f, 0.0f}},
        {{0.5f, 0.5f}, {0.0f, 1.0f}},
        {{0.5f, 0.5f}, {0.0f, 0.0f}},
        {{0.5f, 0.5f}, {1.0f, 0.0f}},
    };
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    vb[0] = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(stream0), stream0);
    vb[1] = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(stream1), stream1);
    vb[2] = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(stream2), stream2);

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    offset = 0;
    stride = sizeof(*stream0);
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb[0], &stride, &offset);
    stride = sizeof(*stream1);
    ID3D11DeviceContext_IASetVertexBuffers(context, 1, 1, &vb[1], &stride, &offset);
    stride = sizeof(*stream2);
    ID3D11DeviceContext_IASetVertexBuffers(context, 2, 1, &vb[2], &stride, &offset);
    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

    ID3D11DeviceContext_DrawInstanced(context, 4, 4, 0, 0);

    color = get_texture_color(test_context.backbuffer,  80, 240);
    ok(compare_color(color, 0xff0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 240, 240);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 400, 240);
    ok(compare_color(color, 0xffff0000, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 560, 240);
    ok(compare_color(color, 0xffff00ff, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11PixelShader_Release(ps);
    ID3D11VertexShader_Release(vs);
    ID3D11Buffer_Release(vb[2]);
    ID3D11Buffer_Release(vb[1]);
    ID3D11Buffer_Release(vb[0]);
    ID3D11InputLayout_Release(input_layout);
    release_test_context(&test_context);
}

static void test_instanced_draw(void)
{
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11InputLayout *input_layout;
    ID3D11RenderTargetView *rtvs[2];
    ID3D11Texture2D *render_target;
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    unsigned int stride, offset;
    ID3D11Buffer *args_buffer;
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *vb[4];
    unsigned int i;
    HRESULT hr;

    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"position", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"color",    0, DXGI_FORMAT_R8_UNORM,           1, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D11_INPUT_PER_INSTANCE_DATA, 1},
        {"color",    1, DXGI_FORMAT_R8_UNORM,           2, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D10_INPUT_PER_INSTANCE_DATA, 0},
        {"color",    2, DXGI_FORMAT_R8_UNORM,           3, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D10_INPUT_PER_INSTANCE_DATA, 2},
        {"v_offset", 0, DXGI_FORMAT_R32_FLOAT,          1, D3D11_APPEND_ALIGNED_ELEMENT,
                D3D11_INPUT_PER_INSTANCE_DATA, 1},
    };
    static const DWORD vs_code[] =
    {
#if 0
        struct vs_in
        {
            float4 position : Position;
            float r : color0;
            float g : color1;
            float b : color2;
            float v_offset : V_Offset;
            uint instance_id : SV_InstanceId;
        };

        struct vs_out
        {
            float4 position : SV_Position;
            float r : color0;
            float g : color1;
            float b : color2;
            uint instance_id : InstanceId;
        };

        void main(vs_in i, out vs_out o)
        {
            o.position = i.position;
            o.position.x += i.v_offset;
            o.r = i.r;
            o.g = i.g;
            o.b = i.b;
            o.instance_id = i.instance_id;
        }
#endif
        0x43425844, 0x036df42e, 0xff0da346, 0x7b23a14a, 0xc26ec9be, 0x00000001, 0x000002bc, 0x00000003,
        0x0000002c, 0x000000f4, 0x0000019c, 0x4e475349, 0x000000c0, 0x00000006, 0x00000008, 0x00000098,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x000000a1, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000101, 0x000000a1, 0x00000001, 0x00000000, 0x00000003, 0x00000002,
        0x00000101, 0x000000a1, 0x00000002, 0x00000000, 0x00000003, 0x00000003, 0x00000101, 0x000000a7,
        0x00000000, 0x00000000, 0x00000003, 0x00000004, 0x00000101, 0x000000b0, 0x00000000, 0x00000008,
        0x00000001, 0x00000005, 0x00000101, 0x69736f50, 0x6e6f6974, 0x6c6f6300, 0x5600726f, 0x66664f5f,
        0x00746573, 0x495f5653, 0x6174736e, 0x4965636e, 0xabab0064, 0x4e47534f, 0x000000a0, 0x00000005,
        0x00000008, 0x00000080, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x0000008c,
        0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x00000e01, 0x0000008c, 0x00000001, 0x00000000,
        0x00000003, 0x00000001, 0x00000d02, 0x0000008c, 0x00000002, 0x00000000, 0x00000003, 0x00000001,
        0x00000b04, 0x00000092, 0x00000000, 0x00000000, 0x00000001, 0x00000002, 0x00000e01, 0x505f5653,
        0x7469736f, 0x006e6f69, 0x6f6c6f63, 0x6e490072, 0x6e617473, 0x64496563, 0xababab00, 0x52444853,
        0x00000118, 0x00010040, 0x00000046, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x00101012,
        0x00000001, 0x0300005f, 0x00101012, 0x00000002, 0x0300005f, 0x00101012, 0x00000003, 0x0300005f,
        0x00101012, 0x00000004, 0x04000060, 0x00101012, 0x00000005, 0x00000008, 0x04000067, 0x001020f2,
        0x00000000, 0x00000001, 0x03000065, 0x00102012, 0x00000001, 0x03000065, 0x00102022, 0x00000001,
        0x03000065, 0x00102042, 0x00000001, 0x03000065, 0x00102012, 0x00000002, 0x07000000, 0x00102012,
        0x00000000, 0x0010100a, 0x00000000, 0x0010100a, 0x00000004, 0x05000036, 0x001020e2, 0x00000000,
        0x00101e56, 0x00000000, 0x05000036, 0x00102012, 0x00000001, 0x0010100a, 0x00000001, 0x05000036,
        0x00102022, 0x00000001, 0x0010100a, 0x00000002, 0x05000036, 0x00102042, 0x00000001, 0x0010100a,
        0x00000003, 0x05000036, 0x00102012, 0x00000002, 0x0010100a, 0x00000005, 0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        struct vs_out
        {
            float4 position : SV_Position;
            float r : color0;
            float g : color1;
            float b : color2;
            uint instance_id : InstanceId;
        };

        void main(vs_out i, out float4 o0 : SV_Target0, out uint4 o1 : SV_Target1)
        {
            o0 = float4(i.r, i.g, i.b, 1.0f);
            o1 = i.instance_id;
        }
#endif
        0x43425844, 0xc9f9c86d, 0xa24d87aa, 0xff75d05b, 0xfbe0581a, 0x00000001, 0x000001b8, 0x00000003,
        0x0000002c, 0x000000d4, 0x00000120, 0x4e475349, 0x000000a0, 0x00000005, 0x00000008, 0x00000080,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x0000008c, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000101, 0x0000008c, 0x00000001, 0x00000000, 0x00000003, 0x00000001,
        0x00000202, 0x0000008c, 0x00000002, 0x00000000, 0x00000003, 0x00000001, 0x00000404, 0x00000092,
        0x00000000, 0x00000000, 0x00000001, 0x00000002, 0x00000101, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x6f6c6f63, 0x6e490072, 0x6e617473, 0x64496563, 0xababab00, 0x4e47534f, 0x00000044, 0x00000002,
        0x00000008, 0x00000038, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000038,
        0x00000001, 0x00000000, 0x00000001, 0x00000001, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074,
        0x52444853, 0x00000090, 0x00000040, 0x00000024, 0x03001062, 0x00101012, 0x00000001, 0x03001062,
        0x00101022, 0x00000001, 0x03001062, 0x00101042, 0x00000001, 0x03000862, 0x00101012, 0x00000002,
        0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x001020f2, 0x00000001, 0x05000036, 0x00102072,
        0x00000000, 0x00101246, 0x00000001, 0x05000036, 0x00102082, 0x00000000, 0x00004001, 0x3f800000,
        0x05000036, 0x001020f2, 0x00000001, 0x00101006, 0x00000002, 0x0100003e,
    };
    static const struct vec4 stream0[] =
    {
        {-1.00f, 0.0f, 0.0f, 1.0f},
        {-1.00f, 1.0f, 0.0f, 1.0f},
        {-0.75f, 0.0f, 0.0f, 1.0f},
        {-0.75f, 1.0f, 0.0f, 1.0f},
        /* indirect draws data */
        {-1.00f, -1.0f, 0.0f, 1.0f},
        {-1.00f,  0.0f, 0.0f, 1.0f},
        {-0.75f, -1.0f, 0.0f, 1.0f},
        {-0.75f,  0.0f, 0.0f, 1.0f},
    };
    static const struct
    {
        BYTE red;
        float v_offset;
    }
    stream1[] =
    {
        {0xf0, 0.00f},
        {0x80, 0.25f},
        {0x10, 0.50f},
        {0x40, 0.75f},

        {0xaa, 1.00f},
        {0xbb, 1.25f},
        {0xcc, 1.50f},
        {0x90, 1.75f},
    };
    static const BYTE stream2[] = {0xf0, 0x80, 0x10, 0x40, 0xaa, 0xbb, 0xcc, 0x90};
    static const BYTE stream3[] = {0xf0, 0x80, 0x10, 0x40, 0xaa, 0xbb, 0xcc, 0x90};
    static const D3D11_DRAW_INSTANCED_INDIRECT_ARGS argument_data[] =
    {
        {4, 4, 4, 0},
        {4, 4, 4, 4},
    };
    static const struct
    {
        RECT rect;
        unsigned int color;
        unsigned int instance_id;
    }
    expected_results[] =
    {
        {{  0, 0,  80, 240}, 0xfff0f0f0, 0},
        {{ 80, 0, 160, 240}, 0xfff0f080, 1},
        {{160, 0, 240, 240}, 0xff80f010, 2},
        {{240, 0, 320, 240}, 0xff80f040, 3},
        {{320, 0, 400, 240}, 0xffaaaaaa, 0},
        {{400, 0, 480, 240}, 0xffaaaabb, 1},
        {{480, 0, 560, 240}, 0xffbbaacc, 2},
        {{560, 0, 640, 240}, 0xffbbaa90, 3},
        /* indirect draws results */
        {{  0, 240,  80, 480}, 0xfff0f0f0, 0},
        {{ 80, 240, 160, 480}, 0xfff0f080, 1},
        {{160, 240, 240, 480}, 0xff80f010, 2},
        {{240, 240, 320, 480}, 0xff80f040, 3},
        {{320, 240, 400, 480}, 0xffaaaaaa, 0},
        {{400, 240, 480, 480}, 0xffaaaabb, 1},
        {{480, 240, 560, 480}, 0xffbbaacc, 2},
        {{560, 240, 640, 480}, 0xffbbaa90, 3},
    };
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    static const D3D_FEATURE_LEVEL feature_level = D3D_FEATURE_LEVEL_11_0;

    if (!init_test_context(&test_context, &feature_level))
        return;
    device = test_context.device;
    context = test_context.immediate_context;

    rtvs[0] = test_context.backbuffer_rtv;

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_R32_UINT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &render_target);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)render_target, NULL, &rtvs[1]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    vb[0] = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(stream0), stream0);
    vb[1] = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(stream1), stream1);
    vb[2] = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(stream2), stream2);
    vb[3] = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(stream3), stream3);

    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    offset = 0;
    stride = sizeof(*stream0);
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb[0], &stride, &offset);
    stride = sizeof(*stream1);
    ID3D11DeviceContext_IASetVertexBuffers(context, 1, 1, &vb[1], &stride, &offset);
    stride = sizeof(*stream2);
    ID3D11DeviceContext_IASetVertexBuffers(context, 2, 1, &vb[2], &stride, &offset);
    stride = sizeof(*stream3);
    ID3D11DeviceContext_IASetVertexBuffers(context, 3, 1, &vb[3], &stride, &offset);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtvs[0], white);
    ID3D11DeviceContext_ClearRenderTargetView(context, rtvs[1], white);

    ID3D11DeviceContext_OMSetRenderTargets(context, ARRAY_SIZE(rtvs), rtvs, NULL);
    ID3D11DeviceContext_DrawInstanced(context, 4, 4, 0, 0);
    ID3D11DeviceContext_DrawInstanced(context, 4, 4, 0, 4);

    args_buffer = create_buffer_misc(device, 0, D3D11_RESOURCE_MISC_DRAWINDIRECT_ARGS,
            sizeof(argument_data), argument_data);

    ID3D11DeviceContext_DrawInstancedIndirect(context, args_buffer, 0);
    ID3D11DeviceContext_DrawInstancedIndirect(context, args_buffer, sizeof(*argument_data));

    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < ARRAY_SIZE(expected_results); ++i)
        check_readback_data_color(&rb, &expected_results[i].rect, expected_results[i].color, 1);
    release_resource_readback(&rb);

    get_texture_readback(render_target, 0, &rb);
    for (i = 0; i < ARRAY_SIZE(expected_results); ++i)
        check_readback_data_color(&rb, &expected_results[i].rect, expected_results[i].instance_id, 0);
    release_resource_readback(&rb);

    ID3D11Buffer_Release(vb[0]);
    ID3D11Buffer_Release(vb[1]);
    ID3D11Buffer_Release(vb[2]);
    ID3D11Buffer_Release(vb[3]);
    ID3D11Buffer_Release(args_buffer);
    ID3D11RenderTargetView_Release(rtvs[1]);
    ID3D11Texture2D_Release(render_target);
    ID3D11VertexShader_Release(vs);
    ID3D11PixelShader_Release(ps);
    ID3D11InputLayout_Release(input_layout);
    release_test_context(&test_context);
}

static void test_vertex_id(void)
{
    static const DWORD vs_code[] =
    {
#if 0
        uint4 main(uint id : ID, uint instance_id : SV_InstanceID, uint vertex_id : SV_VertexID) : OUTPUT
        {
            return uint4(id, instance_id, vertex_id, 0);
        }
#endif
        0x43425844, 0x5625197b, 0x588ccf8f, 0x48694905, 0x961d19ca, 0x00000001, 0x00000170, 0x00000003,
        0x0000002c, 0x000000a4, 0x000000d4, 0x4e475349, 0x00000070, 0x00000003, 0x00000008, 0x00000050,
        0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x00000101, 0x00000053, 0x00000000, 0x00000008,
        0x00000001, 0x00000001, 0x00000101, 0x00000061, 0x00000000, 0x00000006, 0x00000001, 0x00000002,
        0x00000101, 0x53004449, 0x6e495f56, 0x6e617473, 0x44496563, 0x5f565300, 0x74726556, 0x44497865,
        0xababab00, 0x4e47534f, 0x00000028, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000001, 0x00000000, 0x0000000f, 0x5054554f, 0xab005455, 0x52444853, 0x00000094, 0x00010040,
        0x00000025, 0x0300005f, 0x00101012, 0x00000000, 0x04000060, 0x00101012, 0x00000001, 0x00000008,
        0x04000060, 0x00101012, 0x00000002, 0x00000006, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x00102012, 0x00000000, 0x0010100a, 0x00000000, 0x05000036, 0x00102022, 0x00000000, 0x0010100a,
        0x00000001, 0x05000036, 0x00102042, 0x00000000, 0x0010100a, 0x00000002, 0x05000036, 0x00102082,
        0x00000000, 0x00004001, 0x00000000, 0x0100003e,
    };
    D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"ID", 0, DXGI_FORMAT_R32_UINT, 0, D3D11_APPEND_ALIGNED_ELEMENT, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    static const D3D11_SO_DECLARATION_ENTRY so_declaration[] =
    {
        {0, "OUTPUT", 0, 0, 4, 0},
    };
    static const unsigned int vertices[] =
    {
        0,
        1,
        2,

        3,
        4,
        5,

        6,
        7,
        8,

        5,
        6,
        7,
    };
    static const unsigned int indices[] =
    {
        6, 7, 8,

        0, 1, 2,
    };
    struct uvec4 expected_values[] =
    {
        {0, 0, 0},
        {1, 0, 1},
        {2, 0, 2},
        {0, 1, 0},
        {1, 1, 1},
        {2, 1, 2},

        {3, 0, 0},
        {4, 0, 1},
        {5, 0, 2},

        {6, 0, 6},
        {7, 0, 7},
        {8, 0, 8},
        {6, 1, 6},
        {7, 1, 7},
        {8, 1, 8},

        {5, 0, 0},
        {6, 0, 1},
        {7, 0, 2},
    };

    BOOL found_values[ARRAY_SIZE(expected_values)] = {0};
    BOOL used_values[ARRAY_SIZE(expected_values)] = {0};
    struct d3d11_test_context test_context;
    D3D11_QUERY_DATA_SO_STATISTICS data;
    ID3D11Buffer *vb, *ib, *so_buffer;
    ID3D11InputLayout *input_layout;
    ID3D11DeviceContext *context;
    D3D11_QUERY_DESC query_desc;
    struct resource_readback rb;
    unsigned int stride, offset;
    ID3D11Asynchronous *query;
    ID3D11GeometryShader *gs;
    ID3D11VertexShader *vs;
    ID3D11Device *device;
    unsigned int count;
    unsigned int i, j;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;
    device = test_context.device;
    context = test_context.immediate_context;

    query_desc.Query = D3D11_QUERY_SO_STATISTICS;
    query_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateQuery(device, &query_desc, (ID3D11Query **)&query);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    stride = 16;
    hr = ID3D11Device_CreateGeometryShaderWithStreamOutput(device, vs_code, sizeof(vs_code),
            so_declaration, ARRAY_SIZE(so_declaration), &stride, 1, 0, NULL, &gs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(vertices), vertices);
    ib = create_buffer(device, D3D11_BIND_INDEX_BUFFER, sizeof(indices), indices);
    so_buffer = create_buffer(device, D3D11_BIND_STREAM_OUTPUT, 1024, NULL);

    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext_GSSetShader(context, gs, NULL, 0);
    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_POINTLIST);
    ID3D11DeviceContext_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, 0);
    offset = 0;
    stride = sizeof(*vertices);
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb, &stride, &offset);

    offset = 0;
    ID3D11DeviceContext_SOSetTargets(context, 1, &so_buffer, &offset);

    ID3D11DeviceContext_Begin(context, query);

    ID3D11DeviceContext_DrawInstanced(context, 3, 2, 0, 0);
    ID3D11DeviceContext_DrawInstanced(context, 3, 1, 3, 16);

    ID3D11DeviceContext_DrawIndexedInstanced(context, 3, 2, 0, 0, 0);
    ID3D11DeviceContext_DrawIndexedInstanced(context, 3, 1, 3, 9, 7);

    ID3D11DeviceContext_End(context, query);

    get_query_data(context, query, &data, sizeof(data));
    count = data.NumPrimitivesWritten;
    ok(count == ARRAY_SIZE(expected_values), "Got unexpected value %u.\n", count);

    count = min(count, ARRAY_SIZE(used_values));
    get_buffer_readback(so_buffer, &rb);
    for (i = 0; i < ARRAY_SIZE(expected_values); ++i)
    {
        for (j = 0; j < count; ++j)
        {
            if (!used_values[j] && compare_uvec4(get_readback_uvec4(&rb, j, 0), &expected_values[i]))
            {
                found_values[i] = TRUE;
                used_values[j] = TRUE;
                break;
            }
        }
    }

    for (i = 0; i < count; ++i)
    {
        const struct uvec4 *v = get_readback_uvec4(&rb, i, 0);
        ok(used_values[i], "Found unexpected value {0x%08x, 0x%08x, 0x%08x, 0x%08x}.\n", v->x, v->y, v->z, v->w);
    }
    release_resource_readback(&rb);

    for (i = 0; i < ARRAY_SIZE(expected_values); ++i)
    {
        ok(found_values[i], "Failed to find value {0x%08x, 0x%08x, 0x%08x, 0x%08x}.\n",
                expected_values[i].x, expected_values[i].y, expected_values[i].z, expected_values[i].w);
    }

    ID3D11Asynchronous_Release(query);
    ID3D11Buffer_Release(so_buffer);
    ID3D11Buffer_Release(vb);
    ID3D11Buffer_Release(ib);
    ID3D11GeometryShader_Release(gs);
    ID3D11VertexShader_Release(vs);
    ID3D11InputLayout_Release(input_layout);
    release_test_context(&test_context);
}

static void test_fragment_coords(void)
{
    struct d3d11_test_context test_context;
    ID3D11PixelShader *ps, *ps_frac;
    ID3D11DeviceContext *context;
    ID3D11Device *device;
    ID3D11Buffer *ps_cb;
    DWORD color;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        float2 cutoff;

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float4 ret = float4(0.0, 0.0, 0.0, 1.0);

            if (position.x > cutoff.x)
                ret.y = 1.0;
            if (position.y > cutoff.y)
                ret.z = 1.0;

            return ret;
        }
#endif
        0x43425844, 0x49fc9e51, 0x8068867d, 0xf20cfa39, 0xb8099e6b, 0x00000001, 0x00000144, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x000000a8, 0x00000040,
        0x0000002a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x08000031, 0x00100032,
        0x00000000, 0x00208046, 0x00000000, 0x00000000, 0x00101046, 0x00000000, 0x0a000001, 0x00102062,
        0x00000000, 0x00100106, 0x00000000, 0x00004002, 0x00000000, 0x3f800000, 0x3f800000, 0x00000000,
        0x08000036, 0x00102092, 0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x3f800000,
        0x0100003e,
    };
    static const DWORD ps_frac_code[] =
    {
#if 0
        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            return float4(frac(position.xy), 0.0, 1.0);
        }
#endif
        0x43425844, 0x86d9d78a, 0x190b72c2, 0x50841fd6, 0xdc24022e, 0x00000001, 0x000000f8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x0000005c, 0x00000040,
        0x00000017, 0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x0500001a, 0x00102032, 0x00000000, 0x00101046, 0x00000000, 0x08000036, 0x001020c2, 0x00000000,
        0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x3f800000, 0x0100003e,
    };
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};
    struct vec4 cutoff = {320.0f, 240.0f, 0.0f, 0.0f};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    ps_cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(cutoff), &cutoff);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_frac_code, sizeof(ps_frac_code), NULL, &ps_frac);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &ps_cb);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

    draw_quad(&test_context);

    color = get_texture_color(test_context.backbuffer, 319, 239);
    ok(compare_color(color, 0xff000000, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 239);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 319, 240);
    ok(compare_color(color, 0xffff0000, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 320, 240);
    ok(compare_color(color, 0xffffff00, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11Buffer_Release(ps_cb);
    cutoff.x = 16.0f;
    cutoff.y = 16.0f;
    ps_cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(cutoff), &cutoff);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &ps_cb);

    draw_quad(&test_context);

    color = get_texture_color(test_context.backbuffer, 14, 14);
    ok(compare_color(color, 0xff000000, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 18, 14);
    ok(compare_color(color, 0xff00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 14, 18);
    ok(compare_color(color, 0xffff0000, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(test_context.backbuffer, 18, 18);
    ok(compare_color(color, 0xffffff00, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11DeviceContext_PSSetShader(context, ps_frac, NULL, 0);
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

    ID3D11DeviceContext_Draw(context, 4, 0);

    color = get_texture_color(test_context.backbuffer, 14, 14);
    ok(compare_color(color, 0xff008080, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11Buffer_Release(ps_cb);
    ID3D11PixelShader_Release(ps_frac);
    ID3D11PixelShader_Release(ps);
    release_test_context(&test_context);
}

static void test_initial_texture_data(void)
{
    ID3D11Texture2D *texture, *staging_texture;
    struct d3d11_test_context test_context;
    D3D11_SUBRESOURCE_DATA resource_data;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11SamplerState *sampler_state;
    ID3D11ShaderResourceView *ps_srv;
    D3D11_SAMPLER_DESC sampler_desc;
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    unsigned int i, j;
    DWORD color;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;

            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;
            return t.Sample(s, p);
        }
#endif
        0x43425844, 0x1ce9b612, 0xc8176faa, 0xd37844af, 0xdb515605, 0x00000001, 0x00000134, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000098, 0x00000040,
        0x00000026, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555,
        0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x09000045, 0x001020f2, 0x00000000, 0x00100046, 0x00000000,
        0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0100003e,
    };
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};
    static const DWORD bitmap_data[] =
    {
        0xffffffff, 0xff000000, 0xffffffff, 0xff000000,
        0xff00ff00, 0xff0000ff, 0xff00ffff, 0x00000000,
        0xffffff00, 0xffff0000, 0xffff00ff, 0x00000000,
        0xff000000, 0xff7f7f7f, 0xffffffff, 0x00000000,
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 4;
    texture_desc.Height = 4;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_STAGING;
    texture_desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    texture_desc.BindFlags = 0;
    texture_desc.MiscFlags = 0;

    resource_data.pSysMem = bitmap_data;
    resource_data.SysMemPitch = texture_desc.Width * sizeof(*bitmap_data);
    resource_data.SysMemSlicePitch = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &resource_data, &staging_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_CopyResource(context, (ID3D11Resource *)texture, (ID3D11Resource *)staging_texture);

    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, NULL, &ps_srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = 0.0f;
    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &ps_srv);
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler_state);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);
    draw_quad(&test_context);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
            ok(compare_color(color, bitmap_data[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, bitmap_data[j + i * 4]);
        }
    }
    release_resource_readback(&rb);

    ID3D11PixelShader_Release(ps);
    ID3D11SamplerState_Release(sampler_state);
    ID3D11ShaderResourceView_Release(ps_srv);
    ID3D11Texture2D_Release(staging_texture);
    ID3D11Texture2D_Release(texture);
    release_test_context(&test_context);
}

static void test_update_subresource(void)
{
    struct d3d11_test_context test_context;
    D3D11_SUBRESOURCE_DATA resource_data;
    D3D11_TEXTURE3D_DESC texture_desc_3d;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11SamplerState *sampler_state;
    ID3D11ShaderResourceView *ps_srv;
    D3D11_SAMPLER_DESC sampler_desc;
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    ID3D11Texture3D *texture_3d;
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    unsigned int i, j;
    D3D11_BOX box;
    DWORD color;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;

            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;
            return t.Sample(s, p);
        }
#endif
        0x43425844, 0x1ce9b612, 0xc8176faa, 0xd37844af, 0xdb515605, 0x00000001, 0x00000134, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000098, 0x00000040,
        0x00000026, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555,
        0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x09000045, 0x001020f2, 0x00000000, 0x00100046, 0x00000000,
        0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_code_3d[] =
    {
#if 0
        Texture3D t;
        SamplerState s;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float3 p1, p2;
            p2.x = p1.x = position.x / 640.0f;
            p2.y = p1.y = position.y / 480.0f;
            p1.z = 0.25;
            p2.z = 0.75;
            return 0.5 * (t.Sample(s, p1) + t.Sample(s, p2));
        }
#endif
        0x43425844, 0x4d466d63, 0xa3d10db1, 0xd6534470, 0x16d738ef, 0x00000001, 0x000001ec, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000150, 0x00000040,
        0x00000054, 0x0300005a, 0x00106000, 0x00000000, 0x04002858, 0x00107000, 0x00000000, 0x00005555,
        0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000002, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x05000036, 0x00100042, 0x00000000, 0x00004001, 0x3e800000,
        0x09000045, 0x001000f2, 0x00000000, 0x00100246, 0x00000000, 0x00107e46, 0x00000000, 0x00106000,
        0x00000000, 0x0a000038, 0x00100032, 0x00000001, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x05000036, 0x00100042, 0x00000001, 0x00004001, 0x3f400000,
        0x09000045, 0x001000f2, 0x00000001, 0x00100246, 0x00000001, 0x00107e46, 0x00000000, 0x00106000,
        0x00000000, 0x07000000, 0x001000f2, 0x00000000, 0x00100e46, 0x00000000, 0x00100e46, 0x00000001,
        0x0a000038, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0x3f000000, 0x3f000000,
        0x3f000000, 0x3f000000, 0x0100003e,
    };
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};
    static const DWORD initial_data[32] = {0};
    static const DWORD bitmap_data[] =
    {
        0xff0000ff, 0xff00ffff, 0xff00ff00, 0xffffff00,
        0xffff0000, 0xffff00ff, 0xff000000, 0xff7f7f7f,
        0xffffffff, 0xffffffff, 0xffffffff, 0xff000000,
        0xffffffff, 0xff000000, 0xff000000, 0xff000000,
    };
    static const DWORD expected_colors[] =
    {
        0xffffffff, 0xff000000, 0xffffffff, 0xff000000,
        0xff00ff00, 0xff0000ff, 0xff00ffff, 0x00000000,
        0xffffff00, 0xffff0000, 0xffff00ff, 0x00000000,
        0xff000000, 0xff7f7f7f, 0xffffffff, 0x00000000,
    };
    static const DWORD bc7_data[] =
    {
        0x3a7b944b, 0x982a5800, 0x9cab4983, 0xc6a09579,
        0x5f7f2bfe, 0xa95d98f2, 0x3bfb4c03, 0x8be16a41,
        0x8362e6c0, 0x358ed7a2, 0xec3e130b, 0x86cebc86,
        0xf045be66, 0x7a16507f, 0xfe9ccc9f, 0x3f103e16,
        0x84d466c5, 0xfaf5cb5a, 0x9b9e1859, 0x384589b0,
        0x9268b4b8, 0x212b3643, 0x813f853a, 0x4a2bd7c2,
        0x1809f3e0, 0xf646d5ef, 0x40e80679, 0x05791fe5,
        0x6604e7e5, 0x5c28b55d, 0x1ef211f5, 0x632d47f6,
    };
    static const DWORD bc7_expected_colors[] =
    {
        0xc1752752, 0xc39859a9, 0xff79c08e, 0xff63bf6c,
        0xbf7d2756, 0xb89f3d40, 0xffda3a77, 0xffd08099,
        0x415f1f37, 0x43671d3f, 0xffc64758, 0xff57a194,
        0x405a2032, 0x39422619, 0xff749b76, 0xffabb879,
    };
    static const DWORD expected_colors_3d[] = { 0xffff8000, 0xffff8080, 0x80008000, 0xff8080ff };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 4;
    texture_desc.Height = 4;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    resource_data.pSysMem = initial_data;
    resource_data.SysMemPitch = texture_desc.Width * sizeof(*initial_data);
    resource_data.SysMemSlicePitch = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &resource_data, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, NULL, &ps_srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = 0.0f;

    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &ps_srv);
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler_state);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);
    check_texture_color(test_context.backbuffer, 0x7f0000ff, 1);

    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0x00000000, 0);

    set_box(&box, 1, 1, 0, 3, 3, 1);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture, 0, &box,
            bitmap_data, 4 * sizeof(*bitmap_data), 0);
    set_box(&box, 0, 3, 0, 3, 4, 1);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture, 0, &box,
            &bitmap_data[6], 4 * sizeof(*bitmap_data), 0);
    set_box(&box, 0, 0, 0, 4, 1, 1);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture, 0, &box,
            &bitmap_data[10], 4 * sizeof(*bitmap_data), 0);
    set_box(&box, 0, 1, 0, 1, 3, 1);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture, 0, &box,
            &bitmap_data[2], sizeof(*bitmap_data), 0);
    set_box(&box, 4, 4, 0, 3, 1, 1);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture, 0, &box,
            bitmap_data, sizeof(*bitmap_data), 0);
    set_box(&box, 0, 0, 0, 4, 4, 0);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture, 0, &box,
            bitmap_data, 4 * sizeof(*bitmap_data), 0);
    draw_quad(&test_context);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
            ok(compare_color(color, expected_colors[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, expected_colors[j + i * 4]);
        }
    }
    release_resource_readback(&rb);

    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture, 0, NULL,
            bitmap_data, 4 * sizeof(*bitmap_data), 0);
    draw_quad(&test_context);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
            ok(compare_color(color, bitmap_data[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, bitmap_data[j + i * 4]);
        }
    }
    release_resource_readback(&rb);

    ID3D11ShaderResourceView_Release(ps_srv);
    ID3D11Texture2D_Release(texture);
    ID3D11PixelShader_Release(ps);

    hr = ID3D11Device_CreatePixelShader(device, ps_code_3d, sizeof(ps_code_3d), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture_desc_3d.Width = 2;
    texture_desc_3d.Height = 2;
    texture_desc_3d.Depth = 2;
    texture_desc_3d.MipLevels = 1;
    texture_desc_3d.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc_3d.Usage = D3D11_USAGE_DEFAULT;
    texture_desc_3d.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture_desc_3d.CPUAccessFlags = 0;
    texture_desc_3d.MiscFlags = 0;

    resource_data.SysMemPitch = texture_desc_3d.Width * sizeof(*initial_data);
    resource_data.SysMemSlicePitch = texture_desc_3d.Width * texture_desc_3d.Height * sizeof(*initial_data);

    hr = ID3D11Device_CreateTexture3D(device, &texture_desc_3d, &resource_data, &texture_3d);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture_3d, NULL, &ps_srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &ps_srv);

    set_box(&box, 0, 0, 0, 1, 2, 1);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bitmap_data, 8, 16);
    set_box(&box, 0, 0, 0, 1, 1, 2);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bitmap_data + 4, 16, 32);
    set_box(&box, 1, 0, 0, 2, 1, 2);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bitmap_data + 8, 4, 0);
    set_box(&box, 0, 0, 1, 2, 1, 2);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bitmap_data + 2, 4, 5);
    set_box(&box, 0, 0, 1, 2, 1, 2);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bitmap_data + 3, 12, 0);
    set_box(&box, 1, 1, 0, 2, 2, 2);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bitmap_data, 0, 32);

    draw_quad(&test_context);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 2; ++i)
    {
        for (j = 0; j < 2; ++j)
        {
            color = get_readback_color(&rb, 160 + j * 320, 120 + i * 240, 0);
            ok(compare_color(color, expected_colors_3d[j + i * 2], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, expected_colors_3d[j + i * 2]);
        }
    }
    release_resource_readback(&rb);
    ID3D11ShaderResourceView_Release(ps_srv);
    ID3D11Texture3D_Release(texture_3d);

    texture_desc_3d.Width = 8;
    texture_desc_3d.Height = 8;
    texture_desc_3d.Depth = 2;
    texture_desc_3d.Format = DXGI_FORMAT_BC7_UNORM;

    resource_data.SysMemPitch = 32;
    resource_data.SysMemSlicePitch = 64;

    hr = ID3D11Device_CreateTexture3D(device, &texture_desc_3d, &resource_data, &texture_3d);
    if (FAILED(hr))
    {
        skip("Failed to create BC7 3d texture, hr %#lx.\n", hr);
    }
    else
    {
        hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture_3d, NULL, &ps_srv);
        ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

        ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &ps_srv);
        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

        set_box(&box, 0, 0, 0, 8, 8, 2);
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bc7_data, 32, 64);
        set_box(&box, 0, 0, 1, 8, 8, 2);
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bc7_data, 16, 0);
        set_box(&box, 0, 0, 0, 4, 4, 1);
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bc7_data + 8, 0, 0);
        set_box(&box, 4, 4, 0, 8, 8, 2);
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bc7_data + 16, 0, 16);
        set_box(&box, 0, 4, 1, 8, 8, 2);
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bc7_data + 1, 4, 32);
        set_box(&box, 4, 0, 0, 8, 4, 2);
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)texture_3d, 0, &box, bc7_data + 2, 0, 1);

        draw_quad(&test_context);
        get_texture_readback(test_context.backbuffer, 0, &rb);
        for (i = 0; i < 4; ++i)
        {
            for (j = 0; j < 4; ++j)
            {
                color = get_readback_color(&rb, 70 + j * 160, 50 + i * 120, 0);
                ok(compare_color(color, bc7_expected_colors[j + i * 4], 1),
                        "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                        color, j, i, bc7_expected_colors[j + i * 4]);
            }
        }
        release_resource_readback(&rb);
        ID3D11ShaderResourceView_Release(ps_srv);
        ID3D11Texture3D_Release(texture_3d);
    }

    ID3D11PixelShader_Release(ps);
    ID3D11SamplerState_Release(sampler_state);
    release_test_context(&test_context);
}

static void test_copy_subresource_region(void)
{
    ID3D11Texture2D *dst_texture, *src_texture;
    struct d3d11_test_context test_context;
    ID3D11Buffer *dst_buffer, *src_buffer;
    D3D11_SUBRESOURCE_DATA resource_data;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11SamplerState *sampler_state;
    ID3D11ShaderResourceView *ps_srv;
    D3D11_SAMPLER_DESC sampler_desc;
    ID3D11DeviceContext1 *context1;
    ID3D11DeviceContext *context;
    struct vec4 float_colors[16];
    struct resource_readback rb;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    unsigned int i, j;
    D3D11_BOX box;
    DWORD color;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;

            p.x = position.x / 640.0f;
            p.y = position.y / 480.0f;
            return t.Sample(s, p);
        }
#endif
        0x43425844, 0x1ce9b612, 0xc8176faa, 0xd37844af, 0xdb515605, 0x00000001, 0x00000134, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000098, 0x00000040,
        0x00000026, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555,
        0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x09000045, 0x001020f2, 0x00000000, 0x00100046, 0x00000000,
        0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_buffer_code[] =
    {
#if 0
        float4 buffer[16];

        float4 main(float4 position : SV_POSITION) : SV_TARGET
        {
            float2 p = (float2)4;
            p *= float2(position.x / 640.0f, position.y / 480.0f);
            return buffer[(int)p.y * 4 + (int)p.x];
        }
#endif
        0x43425844, 0x57e7139f, 0x4f0c9e52, 0x598b77e3, 0x5a239132, 0x00000001, 0x0000016c, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x000000d0, 0x00000040,
        0x00000034, 0x04000859, 0x00208e46, 0x00000000, 0x00000010, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0a000038, 0x00100032,
        0x00000000, 0x00101516, 0x00000000, 0x00004002, 0x3c088889, 0x3bcccccd, 0x00000000, 0x00000000,
        0x0500001b, 0x00100032, 0x00000000, 0x00100046, 0x00000000, 0x07000029, 0x00100012, 0x00000000,
        0x0010000a, 0x00000000, 0x00004001, 0x00000002, 0x0700001e, 0x00100012, 0x00000000, 0x0010000a,
        0x00000000, 0x0010001a, 0x00000000, 0x07000036, 0x001020f2, 0x00000000, 0x04208e46, 0x00000000,
        0x0010000a, 0x00000000, 0x0100003e,
    };
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};
    static const DWORD initial_data[16] = {0};
    static const DWORD bitmap_data[] =
    {
        0xff0000ff, 0xff00ffff, 0xff00ff00, 0xffffff00,
        0xffff0000, 0xffff00ff, 0xff000000, 0xff7f7f7f,
        0xffffffff, 0xffffffff, 0xffffffff, 0xff000000,
        0xffffffff, 0xff000000, 0xff000000, 0xff000000,
    };
    static const DWORD expected_colors[] =
    {
        0xffffffff, 0xff000000, 0xff000000, 0xff000000,
        0xffffff00, 0xff0000ff, 0xff00ffff, 0x00000000,
        0xff7f7f7f, 0xffff0000, 0xffff00ff, 0xff7f7f7f,
        0xffffffff, 0xffffffff, 0xff000000, 0x00000000,
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 4;
    texture_desc.Height = 4;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    resource_data.pSysMem = initial_data;
    resource_data.SysMemPitch = texture_desc.Width * sizeof(*initial_data);
    resource_data.SysMemSlicePitch = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &resource_data, &dst_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture_desc.Usage = D3D11_USAGE_IMMUTABLE;

    resource_data.pSysMem = bitmap_data;
    resource_data.SysMemPitch = texture_desc.Width * sizeof(*bitmap_data);
    resource_data.SysMemSlicePitch = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &resource_data, &src_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)dst_texture, NULL, &ps_srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = 0.0f;

    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &ps_srv);
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler_state);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);

    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            1, 1, 0, NULL, 0, &box);
    ID3D11DeviceContext_CopySubresourceRegion(context, NULL, 0,
            1, 1, 0, (ID3D11Resource *)src_texture, 0, &box);

    set_box(&box, 0, 0, 0, 2, 2, 1);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            1, 1, 0, (ID3D11Resource *)src_texture, 0, &box);
    set_box(&box, 1, 2, 0, 4, 3, 1);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            0, 3, 0, (ID3D11Resource *)src_texture, 0, &box);
    set_box(&box, 0, 3, 0, 4, 4, 1);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            0, 0, 0, (ID3D11Resource *)src_texture, 0, &box);
    set_box(&box, 3, 0, 0, 4, 2, 1);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            0, 1, 0, (ID3D11Resource *)src_texture, 0, &box);
    set_box(&box, 3, 1, 0, 4, 2, 1);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            3, 2, 0, (ID3D11Resource *)src_texture, 0, &box);
    set_box(&box, 0, 0, 0, 4, 4, 0);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            0, 0, 0, (ID3D11Resource *)src_texture, 0, &box);
    draw_quad(&test_context);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
            ok(compare_color(color, expected_colors[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, expected_colors[j + i * 4]);
        }
    }
    release_resource_readback(&rb);

    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            0, 0, 0, (ID3D11Resource *)src_texture, 0, NULL);
    draw_quad(&test_context);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
            ok(compare_color(color, bitmap_data[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, bitmap_data[j + i * 4]);
        }
    }
    release_resource_readback(&rb);

    hr = ID3D11DeviceContext_QueryInterface(context, &IID_ID3D11DeviceContext1, (void **)&context1);
    ok(hr == S_OK || broken(hr == E_NOINTERFACE) /* Not available on all Windows versions. */,
            "Got unexpected hr %#lx.\n", hr);

    if (SUCCEEDED(hr))
    {
        ID3D11DeviceContext1_ClearRenderTargetView(context1, test_context.backbuffer_rtv, red);
        check_texture_color(test_context.backbuffer, 0x800000ff, 2);

        memset(float_colors, 0, sizeof(float_colors));
        for (i = 0; i < texture_desc.Width; ++i)
            ((unsigned int *)float_colors)[i] = 0x45454545;

        ID3D11DeviceContext1_UpdateSubresource1(context1, (ID3D11Resource *)dst_texture, 0, NULL,
                float_colors, 0, 0, 0);
        draw_quad(&test_context);
        check_texture_color(test_context.backbuffer, 0x45454545, 1);

        ID3D11DeviceContext1_CopySubresourceRegion1(context1, (ID3D11Resource *)dst_texture, 0,
                0, 0, 0, (ID3D11Resource *)src_texture, 0, NULL, 0);
        draw_quad(&test_context);

        get_texture_readback(test_context.backbuffer, 0, &rb);
        for (i = 0; i < 4; ++i)
        {
            for (j = 0; j < 4; ++j)
            {
                color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
                ok(compare_color(color, bitmap_data[j + i * 4], 1),
                        "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                        color, j, i, bitmap_data[j + i * 4]);
            }
        }
        release_resource_readback(&rb);

        ID3D11DeviceContext1_Release(context1);
    }


    ID3D11PixelShader_Release(ps);
    hr = ID3D11Device_CreatePixelShader(device, ps_buffer_code, sizeof(ps_buffer_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11ShaderResourceView_Release(ps_srv);
    ps_srv = NULL;

    ID3D11SamplerState_Release(sampler_state);
    sampler_state = NULL;

    ID3D11Texture2D_Release(dst_texture);
    ID3D11Texture2D_Release(src_texture);

    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &ps_srv);
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler_state);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    memset(float_colors, 0, sizeof(float_colors));
    dst_buffer = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(float_colors), float_colors);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &dst_buffer);

    src_buffer = create_buffer(device, 0, 256 * sizeof(*float_colors), NULL);

    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            float_colors[j + i * 4].x = ((bitmap_data[j + i * 4] >>  0) & 0xff) / 255.0f;
            float_colors[j + i * 4].y = ((bitmap_data[j + i * 4] >>  8) & 0xff) / 255.0f;
            float_colors[j + i * 4].z = ((bitmap_data[j + i * 4] >> 16) & 0xff) / 255.0f;
            float_colors[j + i * 4].w = ((bitmap_data[j + i * 4] >> 24) & 0xff) / 255.0f;
        }
    }
    set_box(&box, 0, 0, 0, sizeof(float_colors), 1, 1);
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)src_buffer, 0, &box, float_colors, 0, 0);

    set_box(&box, 0, 0, 0, sizeof(float_colors), 0, 1);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_buffer, 0,
            0, 0, 0, (ID3D11Resource *)src_buffer, 0, &box);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0x00000000, 0);

    set_box(&box, 0, 0, 0, sizeof(float_colors), 1, 0);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_buffer, 0,
            0, 0, 0, (ID3D11Resource *)src_buffer, 0, &box);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0x00000000, 0);

    set_box(&box, 0, 0, 0, sizeof(float_colors), 0, 0);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_buffer, 0,
            0, 0, 0, (ID3D11Resource *)src_buffer, 0, &box);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0x00000000, 0);

    set_box(&box, 0, 0, 0, sizeof(float_colors), 1, 1);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_buffer, 0,
            0, 0, 0, (ID3D11Resource *)src_buffer, 0, &box);
    draw_quad(&test_context);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
            ok(compare_color(color, bitmap_data[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, bitmap_data[j + i * 4]);
        }
    }
    release_resource_readback(&rb);

    ID3D11Buffer_Release(dst_buffer);
    ID3D11Buffer_Release(src_buffer);
    ID3D11PixelShader_Release(ps);
    release_test_context(&test_context);
}

static void test_copy_subresource_region_1d(void)
{
    D3D11_SUBRESOURCE_DATA resource_data[4];
    struct d3d11_test_context test_context;
    D3D11_TEXTURE1D_DESC texture1d_desc;
    D3D11_TEXTURE2D_DESC texture2d_desc;
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    ID3D11Texture1D *texture1d;
    ID3D11Texture2D *texture2d;
    ID3D11Device *device;
    unsigned int i, j;
    D3D11_BOX box;
    DWORD color;
    HRESULT hr;

    static const DWORD bitmap_data[] =
    {
        0xff0000ff, 0xff00ffff, 0xff00ff00, 0xffffff00,
        0xffff0000, 0xffff00ff, 0xff000000, 0xff7f7f7f,
        0xffffffff, 0xffffffff, 0xffffffff, 0xff000000,
        0xffffffff, 0xff000000, 0xff000000, 0xff000000,
    };

    if (!init_test_context(&test_context, NULL))
        return;
    device = test_context.device;
    context = test_context.immediate_context;

    texture1d_desc.Width = 4;
    texture1d_desc.MipLevels = 1;
    texture1d_desc.ArraySize = 4;
    texture1d_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture1d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture1d_desc.BindFlags = 0;
    texture1d_desc.CPUAccessFlags = 0;
    texture1d_desc.MiscFlags = 0;

    for (i = 0; i < ARRAY_SIZE(resource_data); ++i)
    {
        resource_data[i].pSysMem = &bitmap_data[4 * i];
        resource_data[i].SysMemPitch = texture1d_desc.Width * sizeof(bitmap_data);
        resource_data[i].SysMemSlicePitch = 0;
    }

    hr = ID3D11Device_CreateTexture1D(device, &texture1d_desc, resource_data, &texture1d);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture2d_desc.Width = 4;
    texture2d_desc.Height = 4;
    texture2d_desc.MipLevels = 1;
    texture2d_desc.ArraySize = 1;
    texture2d_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture2d_desc.SampleDesc.Count = 1;
    texture2d_desc.SampleDesc.Quality = 0;
    texture2d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture2d_desc.BindFlags = 0;
    texture2d_desc.CPUAccessFlags = 0;
    texture2d_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, NULL, &texture2d);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    set_box(&box, 0, 0, 0, 4, 1, 1);
    for (i = 0; i < ARRAY_SIZE(resource_data); ++i)
    {
        ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)texture2d, 0,
                0, i, 0, (ID3D11Resource *)texture1d, i, &box);
    }

    get_texture_readback(texture2d, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, j, i, 0);
            ok(compare_color(color, bitmap_data[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, bitmap_data[j + i * 4]);
        }
    }
    release_resource_readback(&rb);

    get_texture1d_readback(texture1d, 0, &rb);
    for (i = 0; i < texture1d_desc.Width; ++i)
    {
        color = get_readback_color(&rb, i, 0, 0);
        ok(compare_color(color, bitmap_data[i], 1),
                "Got colour 0x%08lx at %u, expected 0x%08lx.\n",
                color, i, bitmap_data[i]);
    }
    release_resource_readback(&rb);

    ID3D11Texture1D_Release(texture1d);
    ID3D11Texture2D_Release(texture2d);
    release_test_context(&test_context);
}

static void test_copy_subresource_region_3d(void)
{
    ID3D11ShaderResourceView *dst_srv, *src_srv;
    ID3D11Texture3D *dst_texture, *src_texture;
    D3D11_SUBRESOURCE_DATA resource_data[4];
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture2d_desc;
    D3D11_TEXTURE3D_DESC texture3d_desc;
    ID3D11SamplerState *sampler_state;
    D3D11_SAMPLER_DESC sampler_desc;
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    ID3D11Texture2D *texture2d;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    unsigned int i, j;
    DWORD data[4][16];
    D3D11_BOX box;
    DWORD color;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        Texture3D t;
        SamplerState s;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            return t.Sample(s, position.xyz / float3(640, 480, 1));
        }
#endif
        0x43425844, 0x27b15ae8, 0xbebf46f7, 0x6cd88d8d, 0x5118de51, 0x00000001, 0x00000134, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000070f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000098, 0x00000040,
        0x00000026, 0x0300005a, 0x00106000, 0x00000000, 0x04002858, 0x00107000, 0x00000000, 0x00005555,
        0x04002064, 0x00101072, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0a000038, 0x00100072, 0x00000000, 0x00101246, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x3f800000, 0x00000000, 0x09000045, 0x001020f2, 0x00000000, 0x00100246, 0x00000000,
        0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0100003e,
    };
    static const DWORD bitmap_data[] =
    {
        0xff0000ff, 0xff00ffff, 0xff00ff00, 0xffffff00,
        0xffff0000, 0xffff00ff, 0xff000000, 0xff7f7f7f,
        0xffffffff, 0xffffffff, 0xffffffff, 0xff000000,
        0xffffffff, 0xff000000, 0xff000000, 0xff000000,
    };

    if (!init_test_context(&test_context, NULL))
        return;
    device = test_context.device;
    context = test_context.immediate_context;

    texture3d_desc.Width = 4;
    texture3d_desc.Height = 4;
    texture3d_desc.Depth = 4;
    texture3d_desc.MipLevels = 1;
    texture3d_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture3d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture3d_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture3d_desc.CPUAccessFlags = 0;
    texture3d_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &src_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &dst_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture2d_desc.Width = 4;
    texture2d_desc.Height = 4;
    texture2d_desc.MipLevels = 1;
    texture2d_desc.ArraySize = 4;
    texture2d_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture2d_desc.SampleDesc.Count = 1;
    texture2d_desc.SampleDesc.Quality = 0;
    texture2d_desc.Usage = D3D11_USAGE_IMMUTABLE;
    texture2d_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture2d_desc.CPUAccessFlags = 0;
    texture2d_desc.MiscFlags = 0;

    for (i = 0; i < ARRAY_SIZE(*data); ++i)
    {
        data[0][i] = 0xff0000ff;
        data[1][i] = bitmap_data[i];
        data[2][i] = 0xff00ff00;
        data[3][i] = 0xffff00ff;
    }

    for (i = 0; i < ARRAY_SIZE(resource_data); ++i)
    {
        resource_data[i].pSysMem = data[i];
        resource_data[i].SysMemPitch = texture2d_desc.Width * sizeof(data[0][0]);
        resource_data[i].SysMemSlicePitch = 0;
    }

    hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, resource_data, &texture2d);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)src_texture, NULL, &src_srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)dst_texture, NULL, &dst_srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    sampler_desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 0;
    sampler_desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    sampler_desc.BorderColor[0] = 0.0f;
    sampler_desc.BorderColor[1] = 0.0f;
    sampler_desc.BorderColor[2] = 0.0f;
    sampler_desc.BorderColor[3] = 0.0f;
    sampler_desc.MinLOD = 0.0f;
    sampler_desc.MaxLOD = 0.0f;

    hr = ID3D11Device_CreateSamplerState(device, &sampler_desc, &sampler_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &src_srv);
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler_state);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    set_box(&box, 0, 0, 0, 4, 4, 1);
    for (i = 0; i < ARRAY_SIZE(resource_data); ++i)
    {
        ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)src_texture, 0,
                0, 0, i, (ID3D11Resource *)texture2d, i, &box);
    }
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0xff0000ff, 1);
    draw_quad_z(&test_context, 0.25f);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
            ok(compare_color(color, bitmap_data[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, bitmap_data[j + i * 4]);
        }
    }
    release_resource_readback(&rb);
    draw_quad_z(&test_context, 0.5f);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 1);
    draw_quad_z(&test_context, 1.0f);
    check_texture_color(test_context.backbuffer, 0xffff00ff, 1);

    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &dst_srv);

    set_box(&box, 0, 0, 0, 4, 4, 2);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            0, 0, 2, (ID3D11Resource *)src_texture, 0, &box);
    set_box(&box, 0, 0, 2, 4, 4, 4);
    ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)dst_texture, 0,
            0, 0, 0, (ID3D11Resource *)src_texture, 0, &box);

    set_box(&box, 0, 0, 0, 4, 4, 1);
    for (i = 0; i < ARRAY_SIZE(resource_data); ++i)
    {
        ID3D11DeviceContext_CopySubresourceRegion(context, (ID3D11Resource *)src_texture, 0,
                0, 0, i, (ID3D11Resource *)texture2d, i, &box);
    }
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 1);
    draw_quad_z(&test_context, 0.25f);
    check_texture_color(test_context.backbuffer, 0xffff00ff, 1);
    draw_quad_z(&test_context, 0.5f);
    check_texture_color(test_context.backbuffer, 0xff0000ff, 1);
    draw_quad_z(&test_context, 1.0f);
    get_texture_readback(test_context.backbuffer, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            color = get_readback_color(&rb, 80 + j * 160, 60 + i * 120, 0);
            ok(compare_color(color, bitmap_data[j + i * 4], 1),
                    "Got colour 0x%08lx at (%u, %u), expected 0x%08lx.\n",
                    color, j, i, bitmap_data[j + i * 4]);
        }
    }
    release_resource_readback(&rb);

    ID3D11PixelShader_Release(ps);
    ID3D11SamplerState_Release(sampler_state);
    ID3D11ShaderResourceView_Release(dst_srv);
    ID3D11ShaderResourceView_Release(src_srv);
    ID3D11Texture2D_Release(texture2d);
    ID3D11Texture3D_Release(dst_texture);
    ID3D11Texture3D_Release(src_texture);
    release_test_context(&test_context);
}

static void test_resource_map(void)
{
    D3D11_MAPPED_SUBRESOURCE mapped_subresource;
    D3D11_TEXTURE3D_DESC texture3d_desc;
    D3D11_TEXTURE2D_DESC texture2d_desc;
    D3D11_BUFFER_DESC buffer_desc;
    ID3D11DeviceContext *context;
    ID3D11Texture3D *texture3d;
    ID3D11Texture2D *texture2d;
    ID3D11Buffer *buffer;
    ID3D11Device *device;
    ULONG refcount;
    HRESULT hr;
    DWORD data;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    ID3D11Device_GetImmediateContext(device, &context);

    buffer_desc.ByteWidth = 1024;
    buffer_desc.Usage = D3D11_USAGE_STAGING;
    buffer_desc.BindFlags = 0;
    buffer_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ | D3D11_CPU_ACCESS_WRITE;
    buffer_desc.MiscFlags = 0;
    buffer_desc.StructureByteStride = 0;

    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, &buffer);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    mapped_subresource.pData = (void *)0xdeadbeef;
    mapped_subresource.RowPitch = 0xabab;
    mapped_subresource.DepthPitch = 0xcdcd;
    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)buffer, 1, D3D11_MAP_READ, 0, &mapped_subresource);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!mapped_subresource.pData, "Unexpected pointer %p.\n", mapped_subresource.pData);
    ok(mapped_subresource.RowPitch == 0xabab, "Unexpected row pitch value %u.\n", mapped_subresource.RowPitch);
    ok(mapped_subresource.DepthPitch == 0xcdcd, "Unexpected depth pitch value %u.\n", mapped_subresource.DepthPitch);

    memset(&mapped_subresource, 0, sizeof(mapped_subresource));
    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)buffer, 0, D3D11_MAP_WRITE, 0, &mapped_subresource);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(mapped_subresource.RowPitch == 1024, "Got unexpected row pitch %u.\n", mapped_subresource.RowPitch);
    ok(mapped_subresource.DepthPitch == 1024, "Got unexpected depth pitch %u.\n", mapped_subresource.DepthPitch);
    *((DWORD *)mapped_subresource.pData) = 0xdeadbeef;
    ID3D11DeviceContext_Unmap(context, (ID3D11Resource *)buffer, 0);

    memset(&mapped_subresource, 0, sizeof(mapped_subresource));
    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)buffer, 0, D3D11_MAP_READ, 0, &mapped_subresource);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(mapped_subresource.RowPitch == 1024, "Got unexpected row pitch %u.\n", mapped_subresource.RowPitch);
    ok(mapped_subresource.DepthPitch == 1024, "Got unexpected depth pitch %u.\n", mapped_subresource.DepthPitch);
    data = *((DWORD *)mapped_subresource.pData);
    ok(data == 0xdeadbeef, "Got unexpected data %#lx.\n", data);
    ID3D11DeviceContext_Unmap(context, (ID3D11Resource *)buffer, 0);

    refcount = ID3D11Buffer_Release(buffer);
    ok(!refcount, "Buffer has %lu references left.\n", refcount);

    texture2d_desc.Width = 512;
    texture2d_desc.Height = 512;
    texture2d_desc.MipLevels = 1;
    texture2d_desc.ArraySize = 1;
    texture2d_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture2d_desc.SampleDesc.Count = 1;
    texture2d_desc.SampleDesc.Quality = 0;
    texture2d_desc.Usage = D3D11_USAGE_STAGING;
    texture2d_desc.BindFlags = 0;
    texture2d_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ | D3D11_CPU_ACCESS_WRITE;
    texture2d_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, NULL, &texture2d);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    mapped_subresource.pData = (void *)0xdeadbeef;
    mapped_subresource.RowPitch = 0xabab;
    mapped_subresource.DepthPitch = 0xcdcd;
    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)texture2d, 1, D3D11_MAP_READ, 0, &mapped_subresource);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!mapped_subresource.pData, "Unexpected pointer %p.\n", mapped_subresource.pData);
    ok(mapped_subresource.RowPitch == 0xabab, "Unexpected row pitch value %u.\n", mapped_subresource.RowPitch);
    ok(mapped_subresource.DepthPitch == 0xcdcd, "Unexpected depth pitch value %u.\n", mapped_subresource.DepthPitch);

    memset(&mapped_subresource, 0, sizeof(mapped_subresource));
    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)texture2d, 0, D3D11_MAP_WRITE, 0, &mapped_subresource);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(mapped_subresource.RowPitch == 4 * 512, "Got unexpected row pitch %u.\n", mapped_subresource.RowPitch);
    ok(mapped_subresource.DepthPitch == 4 * 512 * 512, "Got unexpected depth pitch %u.\n",
            mapped_subresource.DepthPitch);
    *((DWORD *)mapped_subresource.pData) = 0xdeadbeef;
    ID3D11DeviceContext_Unmap(context, (ID3D11Resource *)texture2d, 0);

    memset(&mapped_subresource, 0, sizeof(mapped_subresource));
    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)texture2d, 0, D3D11_MAP_READ, 0, &mapped_subresource);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(mapped_subresource.RowPitch == 4 * 512, "Got unexpected row pitch %u.\n", mapped_subresource.RowPitch);
    ok(mapped_subresource.DepthPitch == 4 * 512 * 512, "Got unexpected depth pitch %u.\n",
            mapped_subresource.DepthPitch);
    data = *((DWORD *)mapped_subresource.pData);
    ok(data == 0xdeadbeef, "Got unexpected data %#lx.\n", data);
    ID3D11DeviceContext_Unmap(context, (ID3D11Resource *)texture2d, 0);

    refcount = ID3D11Texture2D_Release(texture2d);
    ok(!refcount, "2D texture has %lu references left.\n", refcount);

    texture3d_desc.Width = 64;
    texture3d_desc.Height = 64;
    texture3d_desc.Depth = 64;
    texture3d_desc.MipLevels = 1;
    texture3d_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture3d_desc.Usage = D3D11_USAGE_STAGING;
    texture3d_desc.BindFlags = 0;
    texture3d_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ | D3D11_CPU_ACCESS_WRITE;
    texture3d_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &texture3d);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)texture3d, 1, D3D11_MAP_READ, 0, &mapped_subresource);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    memset(&mapped_subresource, 0, sizeof(mapped_subresource));
    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)texture3d, 0, D3D11_MAP_WRITE, 0, &mapped_subresource);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(mapped_subresource.RowPitch == 4 * 64, "Got unexpected row pitch %u.\n", mapped_subresource.RowPitch);
    ok(mapped_subresource.DepthPitch == 4 * 64 * 64, "Got unexpected depth pitch %u.\n",
            mapped_subresource.DepthPitch);
    *((DWORD *)mapped_subresource.pData) = 0xdeadbeef;
    ID3D11DeviceContext_Unmap(context, (ID3D11Resource *)texture3d, 0);

    memset(&mapped_subresource, 0, sizeof(mapped_subresource));
    hr = ID3D11DeviceContext_Map(context, (ID3D11Resource *)texture3d, 0, D3D11_MAP_READ, 0, &mapped_subresource);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(mapped_subresource.RowPitch == 4 * 64, "Got unexpected row pitch %u.\n", mapped_subresource.RowPitch);
    ok(mapped_subresource.DepthPitch == 4 * 64 * 64, "Got unexpected depth pitch %u.\n",
            mapped_subresource.DepthPitch);
    data = *((DWORD *)mapped_subresource.pData);
    ok(data == 0xdeadbeef, "Got unexpected data %#lx.\n", data);
    ID3D11DeviceContext_Unmap(context, (ID3D11Resource *)texture3d, 0);

    refcount = ID3D11Texture3D_Release(texture3d);
    ok(!refcount, "3D texture has %lu references left.\n", refcount);

    ID3D11DeviceContext_Release(context);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

#define check_resource_cpu_access(a, b, c, d, e) check_resource_cpu_access_(__LINE__, a, b, c, d, e)
static void check_resource_cpu_access_(unsigned int line, ID3D11DeviceContext *context,
        ID3D11Resource *resource, D3D11_USAGE usage, UINT bind_flags, UINT cpu_access)
{
    BOOL cpu_write = cpu_access & D3D11_CPU_ACCESS_WRITE;
    BOOL cpu_read = cpu_access & D3D11_CPU_ACCESS_READ;
    BOOL dynamic = usage == D3D11_USAGE_DYNAMIC;
    D3D11_MAPPED_SUBRESOURCE map_desc;
    HRESULT hr, expected_hr;
    ID3D11Device *device;

    expected_hr = cpu_read ? S_OK : E_INVALIDARG;
    hr = ID3D11DeviceContext_Map(context, resource, 0, D3D11_MAP_READ, 0, &map_desc);
    ok_(__FILE__, line)(hr == expected_hr, "Got hr %#lx for READ.\n", hr);
    if (SUCCEEDED(hr))
        ID3D11DeviceContext_Unmap(context, resource, 0);

    /* WRITE_DISCARD and WRITE_NO_OVERWRITE are the only allowed options for dynamic resources. */
    expected_hr = !dynamic && cpu_write ? S_OK : E_INVALIDARG;
    hr = ID3D11DeviceContext_Map(context, resource, 0, D3D11_MAP_WRITE, 0, &map_desc);
    todo_wine_if(dynamic && cpu_write)
    ok_(__FILE__, line)(hr == expected_hr, "Got hr %#lx for WRITE.\n", hr);
    if (SUCCEEDED(hr))
        ID3D11DeviceContext_Unmap(context, resource, 0);

    expected_hr = cpu_read && cpu_write ? S_OK : E_INVALIDARG;
    hr = ID3D11DeviceContext_Map(context, resource, 0, D3D11_MAP_READ_WRITE, 0, &map_desc);
    ok_(__FILE__, line)(hr == expected_hr, "Got hr %#lx for READ_WRITE.\n", hr);
    if (SUCCEEDED(hr))
        ID3D11DeviceContext_Unmap(context, resource, 0);

    expected_hr = dynamic ? S_OK : E_INVALIDARG;
    hr = ID3D11DeviceContext_Map(context, resource, 0, D3D11_MAP_WRITE_DISCARD, 0, &map_desc);
    todo_wine_if(!dynamic && cpu_write)
    ok_(__FILE__, line)(hr == expected_hr, "Got hr %#lx for WRITE_DISCARD.\n", hr);
    if (SUCCEEDED(hr))
        ID3D11DeviceContext_Unmap(context, resource, 0);

    if (!dynamic)
        return;

    ID3D11DeviceContext_GetDevice(context, &device);

    /* WRITE_NO_OVERWRITE is supported only for buffers. */
    expected_hr = is_buffer(resource) ? S_OK : E_INVALIDARG;
    hr = ID3D11DeviceContext_Map(context, resource, 0, D3D11_MAP_WRITE_NO_OVERWRITE, 0, &map_desc);
    /* D3D11.1 is required for constant and shader buffers. */
    todo_wine_if(expected_hr != S_OK)
    ok_(__FILE__, line)(hr == expected_hr
            || broken(bind_flags & (D3D11_BIND_CONSTANT_BUFFER | D3D11_BIND_SHADER_RESOURCE)),
            "Got hr %#lx for WRITE_NO_OVERWRITE.\n", hr);
    if (SUCCEEDED(hr))
        ID3D11DeviceContext_Unmap(context, resource, 0);

    ID3D11Device_Release(device);
}

static void test_resource_access(const D3D_FEATURE_LEVEL feature_level)
{
    D3D11_TEXTURE2D_DESC texture_desc;
    struct device_desc device_desc;
    D3D11_BUFFER_DESC buffer_desc;
    ID3D11DeviceContext *context;
    D3D11_SUBRESOURCE_DATA data;
    ID3D11Resource *resource;
    BOOL required_cpu_access;
    BOOL cpu_write, cpu_read;
    HRESULT hr, expected_hr;
    UINT allowed_cpu_access;
    BOOL broken_validation;
    ID3D11Device *device;
    unsigned int i;
    ULONG refcount;

    static const struct
    {
        D3D11_USAGE usage;
        UINT bind_flags;
        BOOL is_valid;
        UINT allowed_cpu_access;
    }
    tests[] =
    {
        /* Default resources cannot be written by CPU. */
        {D3D11_USAGE_DEFAULT, D3D11_BIND_VERTEX_BUFFER,    TRUE, 0},
        {D3D11_USAGE_DEFAULT, D3D11_BIND_INDEX_BUFFER,     TRUE, 0},
        {D3D11_USAGE_DEFAULT, D3D11_BIND_CONSTANT_BUFFER,  TRUE, 0},
        {D3D11_USAGE_DEFAULT, D3D11_BIND_SHADER_RESOURCE,  TRUE, 0},
        {D3D11_USAGE_DEFAULT, D3D11_BIND_STREAM_OUTPUT,    TRUE, 0},
        {D3D11_USAGE_DEFAULT, D3D11_BIND_RENDER_TARGET,    TRUE, 0},
        {D3D11_USAGE_DEFAULT, D3D11_BIND_DEPTH_STENCIL,    TRUE, 0},
        {D3D11_USAGE_DEFAULT, D3D11_BIND_UNORDERED_ACCESS, TRUE, 0},

        /* Immutable resources cannot be written by CPU and GPU. */
        {D3D11_USAGE_IMMUTABLE, 0,                           FALSE, 0},
        {D3D11_USAGE_IMMUTABLE, D3D11_BIND_VERTEX_BUFFER,    TRUE,  0},
        {D3D11_USAGE_IMMUTABLE, D3D11_BIND_INDEX_BUFFER,     TRUE,  0},
        {D3D11_USAGE_IMMUTABLE, D3D11_BIND_CONSTANT_BUFFER,  TRUE,  0},
        {D3D11_USAGE_IMMUTABLE, D3D11_BIND_SHADER_RESOURCE,  TRUE,  0},
        {D3D11_USAGE_IMMUTABLE, D3D11_BIND_STREAM_OUTPUT,    FALSE, 0},
        {D3D11_USAGE_IMMUTABLE, D3D11_BIND_RENDER_TARGET,    FALSE, 0},
        {D3D11_USAGE_IMMUTABLE, D3D11_BIND_DEPTH_STENCIL,    FALSE, 0},
        {D3D11_USAGE_IMMUTABLE, D3D11_BIND_UNORDERED_ACCESS, FALSE, 0},

        /* Dynamic resources cannot be written by GPU. */
        {D3D11_USAGE_DYNAMIC, 0,                           FALSE, D3D11_CPU_ACCESS_WRITE},
        {D3D11_USAGE_DYNAMIC, D3D11_BIND_VERTEX_BUFFER,    TRUE,  D3D11_CPU_ACCESS_WRITE},
        {D3D11_USAGE_DYNAMIC, D3D11_BIND_INDEX_BUFFER,     TRUE,  D3D11_CPU_ACCESS_WRITE},
        {D3D11_USAGE_DYNAMIC, D3D11_BIND_CONSTANT_BUFFER,  TRUE,  D3D11_CPU_ACCESS_WRITE},
        {D3D11_USAGE_DYNAMIC, D3D11_BIND_SHADER_RESOURCE,  TRUE,  D3D11_CPU_ACCESS_WRITE},
        {D3D11_USAGE_DYNAMIC, D3D11_BIND_STREAM_OUTPUT,    FALSE, D3D11_CPU_ACCESS_WRITE},
        {D3D11_USAGE_DYNAMIC, D3D11_BIND_RENDER_TARGET,    FALSE, D3D11_CPU_ACCESS_WRITE},
        {D3D11_USAGE_DYNAMIC, D3D11_BIND_DEPTH_STENCIL,    FALSE, D3D11_CPU_ACCESS_WRITE},
        {D3D11_USAGE_DYNAMIC, D3D11_BIND_UNORDERED_ACCESS, FALSE, D3D11_CPU_ACCESS_WRITE},

        /* Staging resources support only data transfer. */
        {D3D11_USAGE_STAGING, 0,                           TRUE,  D3D11_CPU_ACCESS_WRITE | D3D11_CPU_ACCESS_READ},
        {D3D11_USAGE_STAGING, D3D11_BIND_VERTEX_BUFFER,    FALSE, 0},
        {D3D11_USAGE_STAGING, D3D11_BIND_INDEX_BUFFER,     FALSE, 0},
        {D3D11_USAGE_STAGING, D3D11_BIND_CONSTANT_BUFFER,  FALSE, 0},
        {D3D11_USAGE_STAGING, D3D11_BIND_SHADER_RESOURCE,  FALSE, 0},
        {D3D11_USAGE_STAGING, D3D11_BIND_STREAM_OUTPUT,    FALSE, 0},
        {D3D11_USAGE_STAGING, D3D11_BIND_RENDER_TARGET,    FALSE, 0},
        {D3D11_USAGE_STAGING, D3D11_BIND_DEPTH_STENCIL,    FALSE, 0},
        {D3D11_USAGE_STAGING, D3D11_BIND_UNORDERED_ACCESS, FALSE, 0},
    };

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }
    ID3D11Device_GetImmediateContext(device, &context);

    data.SysMemPitch = 0;
    data.SysMemSlicePitch = 0;
    data.pSysMem = heap_alloc(10240);
    ok(!!data.pSysMem, "Failed to allocate memory.\n");

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        switch (tests[i].bind_flags)
        {
            case D3D11_BIND_DEPTH_STENCIL:
                continue;

            case D3D11_BIND_SHADER_RESOURCE:
            case D3D11_BIND_STREAM_OUTPUT:
            case D3D11_BIND_RENDER_TARGET:
                if (feature_level < D3D_FEATURE_LEVEL_10_0)
                    continue;
                break;

            case D3D11_BIND_UNORDERED_ACCESS:
                if (feature_level < D3D_FEATURE_LEVEL_11_0)
                    continue;
                break;

            default:
                break;
        }

        allowed_cpu_access = tests[i].allowed_cpu_access;
        if (feature_level >= D3D_FEATURE_LEVEL_11_0 && is_d3d11_2_runtime(device)
                && tests[i].usage == D3D11_USAGE_DEFAULT
                && (tests[i].bind_flags == D3D11_BIND_SHADER_RESOURCE
                || tests[i].bind_flags == D3D11_BIND_UNORDERED_ACCESS))
            allowed_cpu_access |= D3D11_CPU_ACCESS_WRITE | D3D11_CPU_ACCESS_READ;

        required_cpu_access = tests[i].usage == D3D11_USAGE_DYNAMIC || tests[i].usage == D3D11_USAGE_STAGING;
        cpu_write = allowed_cpu_access & D3D11_CPU_ACCESS_WRITE;
        cpu_read = allowed_cpu_access & D3D11_CPU_ACCESS_READ;

        buffer_desc.ByteWidth = 1024;
        buffer_desc.Usage = tests[i].usage;
        buffer_desc.BindFlags = tests[i].bind_flags;
        buffer_desc.MiscFlags = 0;
        buffer_desc.StructureByteStride = 0;

        buffer_desc.CPUAccessFlags = 0;
        expected_hr = tests[i].is_valid && !required_cpu_access ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreateBuffer(device, &buffer_desc, &data, (ID3D11Buffer **)&resource);
        ok(hr == expected_hr, "Test %u: Got hr %#lx, expected %#lx.\n", i, hr, expected_hr);
        if (SUCCEEDED(hr))
        {
            check_resource_cpu_access(context, resource,
                    buffer_desc.Usage, buffer_desc.BindFlags, buffer_desc.CPUAccessFlags);
            ID3D11Resource_Release(resource);
        }

        buffer_desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
        expected_hr = tests[i].is_valid && cpu_write ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreateBuffer(device, &buffer_desc, &data, (ID3D11Buffer **)&resource);
        ok(hr == expected_hr, "Test %u: Got hr %#lx, expected %#lx.\n", i, hr, expected_hr);
        if (SUCCEEDED(hr))
        {
            check_resource_cpu_access(context, resource,
                    buffer_desc.Usage, buffer_desc.BindFlags, buffer_desc.CPUAccessFlags);
            ID3D11Resource_Release(resource);
        }

        buffer_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
        expected_hr = tests[i].is_valid && cpu_read ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreateBuffer(device, &buffer_desc, &data, (ID3D11Buffer **)&resource);
        ok(hr == expected_hr, "Test %u: Got hr %#lx, expected %#lx.\n", i, hr, expected_hr);
        if (SUCCEEDED(hr))
        {
            check_resource_cpu_access(context, resource,
                    buffer_desc.Usage, buffer_desc.BindFlags, buffer_desc.CPUAccessFlags);
            ID3D11Resource_Release(resource);
        }

        buffer_desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE | D3D11_CPU_ACCESS_READ;
        expected_hr = tests[i].is_valid && cpu_write && cpu_read ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreateBuffer(device, &buffer_desc, &data, (ID3D11Buffer **)&resource);
        ok(hr == expected_hr, "Test %u: Got hr %#lx, expected %#lx.\n", i, hr, expected_hr);
        if (SUCCEEDED(hr))
        {
            check_resource_cpu_access(context, resource,
                    buffer_desc.Usage, buffer_desc.BindFlags, buffer_desc.CPUAccessFlags);
            ID3D11Resource_Release(resource);
        }
    }

    data.SysMemPitch = 16;

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        switch (tests[i].bind_flags)
        {
            case D3D11_BIND_VERTEX_BUFFER:
            case D3D11_BIND_INDEX_BUFFER:
            case D3D11_BIND_CONSTANT_BUFFER:
            case D3D11_BIND_STREAM_OUTPUT:
                continue;

            case D3D11_BIND_UNORDERED_ACCESS:
                if (feature_level < D3D_FEATURE_LEVEL_11_0)
                    continue;
                break;

            default:
                break;
        }

        broken_validation = tests[i].usage == D3D11_USAGE_DEFAULT
                && (tests[i].bind_flags == D3D11_BIND_SHADER_RESOURCE
                || tests[i].bind_flags == D3D11_BIND_RENDER_TARGET
                || tests[i].bind_flags == D3D11_BIND_UNORDERED_ACCESS);

        required_cpu_access = tests[i].usage == D3D11_USAGE_DYNAMIC || tests[i].usage == D3D11_USAGE_STAGING;
        cpu_write = tests[i].allowed_cpu_access & D3D11_CPU_ACCESS_WRITE;
        cpu_read = tests[i].allowed_cpu_access & D3D11_CPU_ACCESS_READ;

        texture_desc.Width = 4;
        texture_desc.Height = 4;
        texture_desc.MipLevels = 1;
        texture_desc.ArraySize = 1;
        texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
        texture_desc.SampleDesc.Count = 1;
        texture_desc.SampleDesc.Quality = 0;
        texture_desc.Usage = tests[i].usage;
        texture_desc.BindFlags = tests[i].bind_flags;
        texture_desc.MiscFlags = 0;
        if (tests[i].bind_flags == D3D11_BIND_DEPTH_STENCIL)
            texture_desc.Format = DXGI_FORMAT_D16_UNORM;

        texture_desc.CPUAccessFlags = 0;
        expected_hr = tests[i].is_valid && !required_cpu_access ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &data, (ID3D11Texture2D **)&resource);
        ok(hr == expected_hr, "Test %u: Got hr %#lx, expected %#lx.\n", i, hr, expected_hr);
        if (SUCCEEDED(hr))
        {
            check_resource_cpu_access(context, resource,
                    texture_desc.Usage, texture_desc.BindFlags, texture_desc.CPUAccessFlags);
            ID3D11Resource_Release(resource);
        }

        texture_desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
        expected_hr = tests[i].is_valid && cpu_write ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &data, (ID3D11Texture2D **)&resource);
        ok(hr == expected_hr || (hr == S_OK && broken_validation),
                "Test %u: Got hr %#lx, expected %#lx.\n", i, hr, expected_hr);
        if (SUCCEEDED(hr))
        {
            if (broken_validation)
                texture_desc.CPUAccessFlags = 0;
            check_resource_cpu_access(context, resource,
                    texture_desc.Usage, texture_desc.BindFlags, texture_desc.CPUAccessFlags);
            ID3D11Resource_Release(resource);
        }

        texture_desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
        expected_hr = tests[i].is_valid && cpu_read ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &data, (ID3D11Texture2D **)&resource);
        ok(hr == expected_hr || (hr == S_OK && broken_validation),
                "Test %u: Got hr %#lx, expected %#lx.\n", i, hr, expected_hr);
        if (SUCCEEDED(hr))
        {
            if (broken_validation)
                texture_desc.CPUAccessFlags = 0;
            check_resource_cpu_access(context, resource,
                    texture_desc.Usage, texture_desc.BindFlags, texture_desc.CPUAccessFlags);
            ID3D11Resource_Release(resource);
        }

        texture_desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE | D3D11_CPU_ACCESS_READ;
        expected_hr = tests[i].is_valid && cpu_write && cpu_read ? S_OK : E_INVALIDARG;
        hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &data, (ID3D11Texture2D **)&resource);
        ok(hr == expected_hr || (hr == S_OK && broken_validation),
                "Test %u: Got hr %#lx, expected %#lx.\n", i, hr, expected_hr);
        if (SUCCEEDED(hr))
        {
            if (broken_validation)
                texture_desc.CPUAccessFlags = 0;
            check_resource_cpu_access(context, resource,
                    texture_desc.Usage, texture_desc.BindFlags, texture_desc.CPUAccessFlags);
            ID3D11Resource_Release(resource);
        }
    }

    heap_free((void *)data.pSysMem);

    ID3D11DeviceContext_Release(context);
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_check_multisample_quality_levels(void)
{
    ID3D11Device *device;
    UINT quality_levels;
    ULONG refcount;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 2, &quality_levels);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    if (!quality_levels)
    {
        skip("Multisampling not supported for DXGI_FORMAT_R8G8B8A8_UNORM, skipping test.\n");
        goto done;
    }

    quality_levels = 0xdeadbeef;
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_UNKNOWN, 2, &quality_levels);
    todo_wine ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(!quality_levels, "Got unexpected quality_levels %u.\n", quality_levels);
    quality_levels = 0xdeadbeef;
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, 65536, 2, &quality_levels);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    todo_wine ok(quality_levels == 0xdeadbeef, "Got unexpected quality_levels %u.\n", quality_levels);

    if (!enable_debug_layer)
    {
        hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 0, NULL);
        ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
        hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 1, NULL);
        ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
        hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 2, NULL);
        ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    }

    quality_levels = 0xdeadbeef;
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 0, &quality_levels);
    ok(hr == E_FAIL, "Got unexpected hr %#lx.\n", hr);
    ok(!quality_levels, "Got unexpected quality_levels %u.\n", quality_levels);

    quality_levels = 0xdeadbeef;
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 1, &quality_levels);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(quality_levels == 1, "Got unexpected quality_levels %u.\n", quality_levels);

    quality_levels = 0xdeadbeef;
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 2, &quality_levels);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(quality_levels, "Got unexpected quality_levels %u.\n", quality_levels);

    /* We assume 15 samples multisampling is never supported in practice. */
    quality_levels = 0xdeadbeef;
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 15, &quality_levels);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(!quality_levels, "Got unexpected quality_levels %u.\n", quality_levels);
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 32, &quality_levels);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    quality_levels = 0xdeadbeef;
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 33, &quality_levels);
    ok(hr == E_FAIL, "Got unexpected hr %#lx.\n", hr);
    ok(!quality_levels, "Got unexpected quality_levels %u.\n", quality_levels);
    quality_levels = 0xdeadbeef;
    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_R8G8B8A8_UNORM, 64, &quality_levels);
    ok(hr == E_FAIL, "Got unexpected hr %#lx.\n", hr);
    ok(!quality_levels, "Got unexpected quality_levels %u.\n", quality_levels);

    hr = ID3D11Device_CheckMultisampleQualityLevels(device, DXGI_FORMAT_BC3_UNORM, 2, &quality_levels);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(!quality_levels, "Got unexpected quality_levels %u.\n", quality_levels);

done:
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_swapchain_formats(const D3D_FEATURE_LEVEL feature_level)
{
    DXGI_SWAP_CHAIN_DESC swapchain_desc;
    struct device_desc device_desc;
    IDXGISwapChain *swapchain;
    IDXGIDevice *dxgi_device;
    HRESULT hr, expected_hr;
    IDXGIAdapter *adapter;
    IDXGIFactory *factory;
    ID3D11Device *device;
    unsigned int i;
    ULONG refcount;

    swapchain_desc.BufferDesc.Width = 800;
    swapchain_desc.BufferDesc.Height = 600;
    swapchain_desc.BufferDesc.RefreshRate.Numerator = 60;
    swapchain_desc.BufferDesc.RefreshRate.Denominator = 60;
    swapchain_desc.BufferDesc.ScanlineOrdering = DXGI_MODE_SCANLINE_ORDER_UNSPECIFIED;
    swapchain_desc.BufferDesc.Scaling = DXGI_MODE_SCALING_UNSPECIFIED;
    swapchain_desc.SampleDesc.Count = 1;
    swapchain_desc.SampleDesc.Quality = 0;
    swapchain_desc.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
    swapchain_desc.BufferCount = 1;
    swapchain_desc.OutputWindow = CreateWindowA("static", "d3d11_test", 0, 0, 0, 0, 0, 0, 0, 0, 0);
    swapchain_desc.Windowed = TRUE;
    swapchain_desc.SwapEffect = DXGI_SWAP_EFFECT_DISCARD;
    swapchain_desc.Flags = 0;

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }

    hr = ID3D11Device_QueryInterface(device, &IID_IDXGIDevice, (void **)&dxgi_device);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = IDXGIDevice_GetAdapter(dxgi_device, &adapter);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    IDXGIDevice_Release(dxgi_device);
    hr = IDXGIAdapter_GetParent(adapter, &IID_IDXGIFactory, (void **)&factory);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    IDXGIAdapter_Release(adapter);

    swapchain_desc.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_TYPELESS;
    hr = IDXGIFactory_CreateSwapChain(factory, (IUnknown *)device, &swapchain_desc, &swapchain);
    todo_wine ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
    if (SUCCEEDED(hr))
        IDXGISwapChain_Release(swapchain);

    for (i = 0; i < ARRAY_SIZE(display_format_support); ++i)
    {
        DXGI_FORMAT format = display_format_support[i].format;
        BOOL todo = FALSE;

        if (display_format_support[i].fl_required <= feature_level)
        {
            expected_hr = S_OK;
            if (format == DXGI_FORMAT_R10G10B10_XR_BIAS_A2_UNORM)
                todo = TRUE;
        }
        else if (!display_format_support[i].fl_optional
                || display_format_support[i].fl_optional > feature_level)
        {
            expected_hr = E_INVALIDARG;
            if (format != DXGI_FORMAT_R10G10B10_XR_BIAS_A2_UNORM)
                todo = TRUE;
        }
        else
        {
            continue;
        }

        swapchain_desc.BufferDesc.Format = format;
        hr = IDXGIFactory_CreateSwapChain(factory, (IUnknown *)device, &swapchain_desc, &swapchain);
        todo_wine_if(todo)
            ok(hr == expected_hr || broken(hr == E_OUTOFMEMORY),
                    "Got hr %#lx, expected %#lx (format %#x).\n",
                    hr, expected_hr, format);
        if (FAILED(hr))
            continue;
        refcount = IDXGISwapChain_Release(swapchain);
        ok(!refcount, "Swapchain has %lu references left.\n", refcount);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
    refcount = IDXGIFactory_Release(factory);
    ok(!refcount, "Factory has %lu references left.\n", refcount);
    DestroyWindow(swapchain_desc.OutputWindow);
}

static void test_swapchain_views(void)
{
    D3D11_SHADER_RESOURCE_VIEW_DESC srv_desc;
    struct d3d11_test_context test_context;
    D3D11_RENDER_TARGET_VIEW_DESC rtv_desc;
    ID3D11ShaderResourceView *srv;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11Device *device;
    ULONG refcount;
    HRESULT hr;

    static const struct vec4 color = {0.2f, 0.3f, 0.5f, 1.0f};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    refcount = get_refcount(test_context.backbuffer);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    draw_color_quad(&test_context, &color);
    check_texture_color(test_context.backbuffer, 0xff7f4c33, 1);

    rtv_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM_SRGB;
    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2D;
    U(rtv_desc).Texture2D.MipSlice = 0;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)test_context.backbuffer, &rtv_desc, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);

    refcount = get_refcount(test_context.backbuffer);
    ok(refcount == 1, "Got unexpected refcount %lu.\n", refcount);

    draw_color_quad(&test_context, &color);
    check_texture_color(test_context.backbuffer, 0xffbc957c, 1);

    srv_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM_SRGB;
    srv_desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
    U(srv_desc).Texture2D.MostDetailedMip = 0;
    U(srv_desc).Texture2D.MipLevels = 1;
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)test_context.backbuffer, &srv_desc, &srv);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    if (SUCCEEDED(hr))
        ID3D11ShaderResourceView_Release(srv);

    ID3D11RenderTargetView_Release(rtv);
    release_test_context(&test_context);
}

static void test_swapchain_flip(void)
{
    ID3D11Texture2D *backbuffer_0, *backbuffer_1, *backbuffer_2, *offscreen;
    ID3D11ShaderResourceView *backbuffer_0_srv, *backbuffer_1_srv;
    ID3D11RenderTargetView *backbuffer_0_rtv, *offscreen_rtv;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11InputLayout *input_layout;
    ID3D11DeviceContext *context;
    unsigned int stride, offset;
    struct swapchain_desc desc;
    IDXGISwapChain *swapchain;
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *vb;
    ULONG refcount;
    DWORD color;
    HWND window;
    HRESULT hr;
    RECT rect;

    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32_FLOAT, 0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    static const DWORD vs_code[] =
    {
#if 0
        float4 main(float4 position : POSITION) : SV_POSITION
        {
            return position;
        }
#endif
        0x43425844, 0xa7a2f22d, 0x83ff2560, 0xe61638bd, 0x87e3ce90, 0x00000001, 0x000000d8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x0000003c, 0x00010040,
        0x0000000f, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };

    static const DWORD ps_code[] =
    {
#if 0
        Texture2D t0, t1;
        SamplerState s;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            float2 p;

            p.x = 0.5;
            p.y = 0.5;
            if (position.x < 320)
                return t0.Sample(s, p);
            return t1.Sample(s, p);
        }
#endif
        0x43425844, 0xc00961ea, 0x48558efd, 0x5eec7aed, 0xb597e6d1, 0x00000001, 0x00000188, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000010f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x000000ec, 0x00000040,
        0x0000003b, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555,
        0x04001858, 0x00107000, 0x00000001, 0x00005555, 0x04002064, 0x00101012, 0x00000000, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x07000031, 0x00100012, 0x00000000,
        0x0010100a, 0x00000000, 0x00004001, 0x43a00000, 0x0304001f, 0x0010000a, 0x00000000, 0x0c000045,
        0x001020f2, 0x00000000, 0x00004002, 0x3f000000, 0x3f000000, 0x00000000, 0x00000000, 0x00107e46,
        0x00000000, 0x00106000, 0x00000000, 0x0100003e, 0x01000015, 0x0c000045, 0x001020f2, 0x00000000,
        0x00004002, 0x3f000000, 0x3f000000, 0x00000000, 0x00000000, 0x00107e46, 0x00000001, 0x00106000,
        0x00000000, 0x0100003e,
    };
    static const struct vec2 quad[] =
    {
        {-1.0f, -1.0f},
        {-1.0f,  1.0f},
        { 1.0f, -1.0f},
        { 1.0f,  1.0f},
    };
    static const float red[] = {1.0f, 0.0f, 0.0f, 0.5f};
    static const float green[] = {0.0f, 1.0f, 0.0f, 0.5f};
    static const float blue[] = {0.0f, 0.0f, 1.0f, 0.5f};

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device, skipping tests.\n");
        return;
    }
    SetRect(&rect, 0, 0, 640, 480);
    AdjustWindowRect(&rect, WS_OVERLAPPEDWINDOW | WS_VISIBLE, FALSE);
    window = CreateWindowA("static", "d3d11_test", WS_OVERLAPPEDWINDOW | WS_VISIBLE,
            0, 0, rect.right - rect.left, rect.bottom - rect.top, NULL, NULL, NULL, NULL);
    desc.buffer_count = 3;
    desc.width = desc.height = 0;
    desc.swap_effect = DXGI_SWAP_EFFECT_SEQUENTIAL;
    desc.windowed = TRUE;
    desc.flags = SWAPCHAIN_FLAG_SHADER_INPUT;
    swapchain = create_swapchain(device, window, &desc);

    hr = IDXGISwapChain_GetBuffer(swapchain, 0, &IID_ID3D11Texture2D, (void **)&backbuffer_0);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = IDXGISwapChain_GetBuffer(swapchain, 1, &IID_ID3D11Texture2D, (void **)&backbuffer_1);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = IDXGISwapChain_GetBuffer(swapchain, 2, &IID_ID3D11Texture2D, (void **)&backbuffer_2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)backbuffer_0, NULL, &backbuffer_0_rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)backbuffer_0, NULL, &backbuffer_0_srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)backbuffer_1, NULL, &backbuffer_1_srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11Texture2D_GetDesc(backbuffer_0, &texture_desc);
    ok((texture_desc.BindFlags & (D3D11_BIND_RENDER_TARGET | D3D11_BIND_SHADER_RESOURCE))
            == (D3D11_BIND_RENDER_TARGET | D3D11_BIND_SHADER_RESOURCE),
            "Got unexpected bind flags %x.\n", texture_desc.BindFlags);
    ok(texture_desc.Usage == D3D11_USAGE_DEFAULT, "Got unexpected usage %u.\n", texture_desc.Usage);

    ID3D11Texture2D_GetDesc(backbuffer_1, &texture_desc);
    ok((texture_desc.BindFlags & (D3D11_BIND_RENDER_TARGET | D3D11_BIND_SHADER_RESOURCE))
            == (D3D11_BIND_RENDER_TARGET | D3D11_BIND_SHADER_RESOURCE),
            "Got unexpected bind flags %x.\n", texture_desc.BindFlags);
    ok(texture_desc.Usage == D3D11_USAGE_DEFAULT, "Got unexpected usage %u.\n", texture_desc.Usage);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)backbuffer_1, NULL, &offscreen_rtv);
    todo_wine ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    if (SUCCEEDED(hr))
        ID3D11RenderTargetView_Release(offscreen_rtv);

    ID3D11Device_GetImmediateContext(device, &context);

    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &backbuffer_0_srv);
    ID3D11DeviceContext_PSSetShaderResources(context, 1, 1, &backbuffer_1_srv);

    texture_desc.Width = 640;
    texture_desc.Height = 480;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &offscreen);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)offscreen, NULL, &offscreen_rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &offscreen_rtv, NULL);
    set_viewport(context, 0.0f, 0.0f, 640.0f, 480.0f, 0.0f, 1.0f);

    vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(quad), quad);

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    stride = sizeof(*quad);
    offset = 0;
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb, &stride, &offset);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, backbuffer_0_rtv, red);

    ID3D11DeviceContext_Draw(context, 4, 0);
    color = get_texture_color(offscreen, 120, 240);
    ok(compare_color(color, 0x7f0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);

    /* DXGI moves buffers in the same direction as earlier versions. Buffer 2
     * becomes buffer 1, buffer 1 becomes the new buffer 0, and buffer 0
     * becomes buffer n - 1. However, only buffer 0 can be rendered to.
     *
     * What is this good for? I don't know. Ad-hoc tests suggest that
     * Present() always waits for the next V-sync interval, even if there are
     * still untouched buffers. Buffer 0 is the buffer that is shown on the
     * screen, just like in <= d3d9. Present() also doesn't discard buffers if
     * rendering finishes before the V-sync interval is over. I haven't found
     * any productive use for more than one buffer. */
    IDXGISwapChain_Present(swapchain, 0, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, backbuffer_0_rtv, green);

    ID3D11DeviceContext_Draw(context, 4, 0);
    color = get_texture_color(offscreen, 120, 240); /* green, buf 0 */
    ok(compare_color(color, 0x7f00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    /* Buffer 1 is still untouched. */

    color = get_texture_color(backbuffer_0, 320, 240); /* green */
    ok(compare_color(color, 0x7f00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(backbuffer_2, 320, 240); /* red */
    ok(compare_color(color, 0x7f0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);

    IDXGISwapChain_Present(swapchain, 0, 0);

    ID3D11DeviceContext_ClearRenderTargetView(context, backbuffer_0_rtv, blue);

    ID3D11DeviceContext_Draw(context, 4, 0);
    color = get_texture_color(offscreen, 120, 240); /* blue, buf 0 */
    ok(compare_color(color, 0x7fff0000, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(offscreen, 360, 240); /* red, buf 1 */
    ok(compare_color(color, 0x7f0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);

    color = get_texture_color(backbuffer_0, 320, 240); /* blue */
    ok(compare_color(color, 0x7fff0000, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(backbuffer_1, 320, 240); /* red */
    ok(compare_color(color, 0x7f0000ff, 1), "Got unexpected colour 0x%08lx.\n", color);
    color = get_texture_color(backbuffer_2, 320, 240); /* green */
    ok(compare_color(color, 0x7f00ff00, 1), "Got unexpected colour 0x%08lx.\n", color);

    ID3D11VertexShader_Release(vs);
    ID3D11PixelShader_Release(ps);
    ID3D11Buffer_Release(vb);
    ID3D11InputLayout_Release(input_layout);
    ID3D11ShaderResourceView_Release(backbuffer_0_srv);
    ID3D11ShaderResourceView_Release(backbuffer_1_srv);
    ID3D11RenderTargetView_Release(backbuffer_0_rtv);
    ID3D11RenderTargetView_Release(offscreen_rtv);
    ID3D11Texture2D_Release(offscreen);
    ID3D11Texture2D_Release(backbuffer_0);
    ID3D11Texture2D_Release(backbuffer_1);
    ID3D11Texture2D_Release(backbuffer_2);
    IDXGISwapChain_Release(swapchain);

    ID3D11DeviceContext_Release(context);
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
    DestroyWindow(window);
}

static void test_clear_render_target_view_1d(void)
{
    static const float color[] = {0.1f, 0.5f, 0.3f, 0.75f};
    static const float green[] = {0.0f, 1.0f, 0.0f, 0.5f};

    struct d3d11_test_context test_context;
    D3D11_TEXTURE1D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11Texture1D *texture;
    ID3D11Device *device;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 64;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture1D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv, color);
    check_texture1d_color(texture, 0xbf4c7f19, 1);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv, green);
    check_texture1d_color(texture, 0x8000ff00, 1);

    ID3D11RenderTargetView_Release(rtv);
    ID3D11Texture1D_Release(texture);
    release_test_context(&test_context);
}

static void test_clear_render_target_view_2d(void)
{
    static const DWORD expected_color = 0xbf4c7f19, expected_srgb_color = 0xbf95bc59;
    static const float clear_colour[] = {0.1f, 0.5f, 0.3f, 0.75f};
    static const float green[] = {0.0f, 1.0f, 0.0f, 0.5f};
    static const float blue[] = {0.0f, 0.0f, 1.0f, 0.5f};

    ID3D11RenderTargetView *rtv[3], *srgb_rtv;
    ID3D11Texture2D *texture, *srgb_texture;
    struct d3d11_test_context test_context;
    D3D11_RENDER_TARGET_VIEW_DESC rtv_desc;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    ID3D11Device *device;
    unsigned int i, j;
    DWORD colour;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 640;
    texture_desc.Height = 480;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM_SRGB;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &srgb_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv[0]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)srgb_texture, NULL, &srgb_rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, clear_colour);
    check_texture_color(test_context.backbuffer, expected_color, 1);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv[0], clear_colour);
    check_texture_color(texture, expected_color, 1);

    if (!enable_debug_layer)
        ID3D11DeviceContext_ClearRenderTargetView(context, NULL, green);
    check_texture_color(texture, expected_color, 1);

    ID3D11DeviceContext_ClearRenderTargetView(context, srgb_rtv, clear_colour);
    check_texture_color(srgb_texture, expected_srgb_color, 1);

    ID3D11RenderTargetView_Release(srgb_rtv);
    ID3D11RenderTargetView_Release(rtv[0]);
    ID3D11Texture2D_Release(srgb_texture);
    ID3D11Texture2D_Release(texture);

    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_TYPELESS;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    rtv_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM_SRGB;
    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2D;
    U(rtv_desc).Texture2D.MipSlice = 0;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &srgb_rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    rtv_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2D;
    U(rtv_desc).Texture2D.MipSlice = 0;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv[0]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv[0], clear_colour);
    check_texture_color(texture, expected_color, 1);

    ID3D11DeviceContext_ClearRenderTargetView(context, srgb_rtv, clear_colour);
    get_texture_readback(texture, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            BOOL broken_device = is_warp_device(device) || is_nvidia_device(device);
            colour = get_readback_color(&rb, 80 + i * 160, 60 + j * 120, 0);
            ok(compare_color(colour, expected_srgb_color, 1)
                    || broken(compare_color(colour, expected_color, 1) && broken_device),
                    "Got unexpected colour 0x%08lx.\n", colour);
        }
    }
    release_resource_readback(&rb);

    ID3D11RenderTargetView_Release(srgb_rtv);
    ID3D11RenderTargetView_Release(rtv[0]);
    ID3D11Texture2D_Release(texture);

    texture_desc.Width = 16;
    texture_desc.Height = 16;
    texture_desc.ArraySize = 5;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DARRAY;
    U(rtv_desc).Texture2DArray.MipSlice = 0;
    U(rtv_desc).Texture2DArray.FirstArraySlice = 0;
    U(rtv_desc).Texture2DArray.ArraySize = 5;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv[0]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    U(rtv_desc).Texture2DArray.FirstArraySlice = 1;
    U(rtv_desc).Texture2DArray.ArraySize = 3;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv[1]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    U(rtv_desc).Texture2DArray.FirstArraySlice = 2;
    U(rtv_desc).Texture2DArray.ArraySize = 1;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv[2]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv[0], blue);
    ID3D11DeviceContext_ClearRenderTargetView(context, rtv[1], green);
    ID3D11DeviceContext_ClearRenderTargetView(context, rtv[2], clear_colour);

    get_texture_readback(texture, 0, &rb);
    colour = get_readback_color(&rb, 8, 8, 0);
    ok(compare_color(colour, 0x80ff0000, 1), "Got unexpected colour 0x%08lx.\n", colour);
    release_resource_readback(&rb);

    get_texture_readback(texture, 1, &rb);
    colour = get_readback_color(&rb, 8, 8, 0);
    ok(compare_color(colour, 0x8000ff00, 1), "Got unexpected colour 0x%08lx.\n", colour);
    release_resource_readback(&rb);

    get_texture_readback(texture, 2, &rb);
    colour = get_readback_color(&rb, 8, 8, 0);
    ok(compare_color(colour, 0xbf4c7f19, 1), "Got unexpected colour 0x%08lx.\n", colour);
    release_resource_readback(&rb);

    get_texture_readback(texture, 3, &rb);
    colour = get_readback_color(&rb, 8, 8, 0);
    ok(compare_color(colour, 0x8000ff00, 1), "Got unexpected colour 0x%08lx.\n", colour);
    release_resource_readback(&rb);

    get_texture_readback(texture, 4, &rb);
    colour = get_readback_color(&rb, 8, 8, 0);
    ok(compare_color(colour, 0x80ff0000, 1), "Got unexpected colour 0x%08lx.\n", colour);
    release_resource_readback(&rb);

    ID3D11RenderTargetView_Release(rtv[2]);
    ID3D11RenderTargetView_Release(rtv[1]);
    ID3D11RenderTargetView_Release(rtv[0]);
    ID3D11Texture2D_Release(texture);

    release_test_context(&test_context);
}

static void test_clear_render_target_view_3d(void)
{
    static const float color[] = {0.1f, 0.5f, 0.3f, 0.75f};
    static const float green[] = {0.0f, 1.0f, 0.0f, 0.5f};

    struct d3d11_test_context test_context;
    D3D11_RENDER_TARGET_VIEW_DESC rtv_desc;
    D3D11_TEXTURE3D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11Texture3D *texture;
    ID3D11Device *device;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;
    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 8;
    texture_desc.Height = 8;
    texture_desc.Depth = 4;
    texture_desc.MipLevels = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture3D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv, color);
    check_texture3d_color(texture, 0xbf4c7f19, 1);
    ID3D11DeviceContext_ClearRenderTargetView(context, rtv, green);
    check_texture3d_color(texture, 0x8000ff00, 1);

    ID3D11RenderTargetView_Release(rtv);
    ID3D11Texture3D_Release(texture);

    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_TYPELESS;
    hr = ID3D11Device_CreateTexture3D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    rtv_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM_SRGB;
    rtv_desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE3D;
    U(rtv_desc).Texture3D.MipSlice = 0;
    U(rtv_desc).Texture3D.FirstWSlice = 0;
    U(rtv_desc).Texture3D.WSize = ~0u;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, &rtv_desc, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv, color);
    check_texture3d_color(texture, 0xbf95bc59, 1);
    ID3D11DeviceContext_ClearRenderTargetView(context, rtv, green);
    check_texture3d_color(texture, 0x8000ff00, 1);

    ID3D11RenderTargetView_Release(rtv);
    ID3D11Texture3D_Release(texture);
    release_test_context(&test_context);
}

static void test_clear_depth_stencil_view(void)
{
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11Texture2D *depth_texture;
    ID3D11DeviceContext *context;
    ID3D11DepthStencilView *dsv;
    ID3D11Device *device;
    ULONG refcount;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    ID3D11Device_GetImmediateContext(device, &context);

    texture_desc.Width = 640;
    texture_desc.Height = 480;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_D32_FLOAT;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_DEPTH_STENCIL;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &depth_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)depth_texture, NULL, &dsv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0);
    check_texture_float(depth_texture, 1.0f, 0);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 0.25f, 0);
    check_texture_float(depth_texture, 0.25f, 0);

    if (!enable_debug_layer)
        ID3D11DeviceContext_ClearDepthStencilView(context, NULL, D3D11_CLEAR_DEPTH, 1.0f, 0);
    check_texture_float(depth_texture, 0.25f, 0);

    ID3D11Texture2D_Release(depth_texture);
    ID3D11DepthStencilView_Release(dsv);

    texture_desc.Format = DXGI_FORMAT_D24_UNORM_S8_UINT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &depth_texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)depth_texture, NULL, &dsv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 1.0f, 0);
    todo_wine check_texture_color(depth_texture, 0x00ffffff, 0);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 0.0f, 0xff);
    todo_wine check_texture_color(depth_texture, 0xff000000, 0);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 1.0f, 0xff);
    check_texture_color(depth_texture, 0xffffffff, 0);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 0.0f, 0);
    check_texture_color(depth_texture, 0x00000000, 0);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0xff);
    todo_wine check_texture_color(depth_texture, 0x00ffffff, 0);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_STENCIL, 0.0f, 0xff);
    check_texture_color(depth_texture, 0xffffffff, 0);

    ID3D11Texture2D_Release(depth_texture);
    ID3D11DepthStencilView_Release(dsv);

    ID3D11DeviceContext_Release(context);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static unsigned int to_sint8(unsigned int x)
{
    union
    {
        signed int s;
        unsigned int u;
    } bits;
    bits.u = x;
    return min(max(bits.s, -128), 127) & 0xff;
}

#define check_rgba_sint8(data, uvec) check_rgba_sint8_(__LINE__, data, uvec)
static void check_rgba_sint8_(unsigned int line, DWORD data, const struct uvec4 *v)
{
    unsigned int x = to_sint8(v->x);
    unsigned int y = to_sint8(v->y);
    unsigned int z = to_sint8(v->z);
    unsigned int w = to_sint8(v->w);
    DWORD expected[] =
    {
        /* Windows 7 - Nvidia, WARP */
        (v->x & 0xff) | (v->y & 0xff) << 8 | (v->z & 0xff) << 16 | (v->w & 0xff) << 24,
        /* Windows 10 - AMD */
        x | y << 8 | z << 16 | w << 24,
        /* Windows 10 - Intel */
        x | x << 8 | x << 16 | x << 24,
    };

    ok_(__FILE__, line)(data == expected[0] || data == expected[1] || broken(data == expected[2]),
            "Got %#lx, expected %#lx or %#lx at %u, uvec4 %#x, %#x, %#x, %#x.\n",
            data, expected[0], expected[1], x, v->x, v->y, v->z, v->w);
}

static void test_clear_buffer_unordered_access_view(void)
{
    D3D11_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    ID3D11UnorderedAccessView *uav, *uav2;
    struct device_desc device_desc;
    D3D11_BUFFER_DESC buffer_desc;
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    ID3D11Buffer *buffer;
    ID3D11Device *device;
    struct uvec4 uvec4;
    unsigned int i, x;
    ULONG refcount;
    HRESULT hr;
    RECT rect;

    static const D3D_FEATURE_LEVEL feature_level = D3D_FEATURE_LEVEL_11_0;
    static const struct uvec4 fe_uvec4 = {0xfefefefe, 0xfefefefe, 0xfefefefe, 0xfefefefe};
    static const struct uvec4 uvec4_data[] =
    {
        {0x00000000, 0x00000000, 0x00000000, 0x00000000},

        {0x00000000, 0xffffffff, 0xffffffff, 0xffffffff},
        {0xffffffff, 0x00000000, 0x00000000, 0x00000000},
        {0x00000000, 0xffffffff, 0x00000000, 0x00000000},
        {0x00000000, 0x00000000, 0xffffffff, 0x00000000},
        {0x00000000, 0x00000000, 0x00000000, 0xffffffff},

        {0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff},
        {0x80000000, 0x80000000, 0x80000000, 0x80000000},
        {0x000000ff, 0x00000080, 0x80000080, 0x00000080},
        {0x000000ff, 0x0000007f, 0x000000ef, 0x000000fe},
        {0x800000ff, 0x8000007f, 0x800000ef, 0x800000fe},
        {0xfefefefe, 0xf0f0f0f0, 0xefefefef, 0x0f0f0f0f},
        {0xaaaaaaaa, 0xdeadbeef, 0xdeadbabe, 0xdeadf00d},

        {0x00000001, 0x00000002, 0x00000003, 0x00000004},
        {0x000000ff, 0x000000fe, 0x000000fd, 0x000000fc},
        {0x000000f2, 0x000000f1, 0x000000f0, 0x000000ef},
        {0x0000000a, 0x0000000d, 0x0000000e, 0x0000000f},
        {0x0000001a, 0x0000002d, 0x0000003e, 0x0000004f},
        {0x00000050, 0x00000060, 0x00000070, 0x00000080},
        {0x00000090, 0x000000a0, 0x000000b0, 0x000000c0},
        {0x000000d0, 0x000000e0, 0x000000f0, 0x000000ff},
        {0x00000073, 0x00000077, 0x0000007a, 0x0000007b},
        {0x0000007c, 0x0000007d, 0x0000007e, 0x0000007f},

        {0x80000001, 0x80000002, 0x80000003, 0x80000004},
        {0x800000ff, 0x800000fe, 0x800000fd, 0x800000fc},
        {0x800000f2, 0x800000f1, 0x800000f0, 0x800000ef},
        {0x8000000a, 0x0000000d, 0x8000000e, 0x8000000f},
        {0x8000001a, 0x8000002d, 0x8000003e, 0x8000004f},
        {0x80000050, 0x80000060, 0x80000070, 0x00000080},
        {0x80000090, 0x800000a0, 0x800000b0, 0x800000c0},
        {0x800000d0, 0x800000e0, 0x800000f0, 0x800000ff},
        {0x80000073, 0x80000077, 0x8000007a, 0x8000007b},
        {0x8000007c, 0x8000007d, 0x8000007e, 0x8000007f},

        {0x7fffff01, 0x7fffff02, 0x7fffff03, 0x7fffff04},
        {0x7fffffff, 0x7ffffffe, 0x7ffffffd, 0x7ffffffc},
        {0x7ffffff2, 0x7ffffff1, 0x7ffffff0, 0x7fffffef},
        {0x7fffff0a, 0x7fffff0d, 0x7fffff0e, 0x7fffff0f},
        {0x7fffff1a, 0x7fffff2d, 0x7fffff3e, 0x7fffff4f},
        {0x7fffff50, 0x7fffff60, 0x7fffff70, 0x7fffff80},
        {0x8fffff90, 0x7fffffa0, 0x7fffffb0, 0x7fffffc0},
        {0x7fffffd0, 0x7fffffe0, 0x7ffffff0, 0x7fffffff},
        {0x7fffff73, 0x7fffff77, 0x7fffff7a, 0x7fffff7b},
        {0x7fffff7c, 0x7fffff7d, 0x7fffff7e, 0x7fffff7f},

        {0xffffff01, 0xffffff02, 0xffffff03, 0xffffff04},
        {0xffffffff, 0xfffffffe, 0xfffffffd, 0xfffffffc},
        {0xfffffff2, 0xfffffff1, 0xfffffff0, 0xffffffef},
        {0xffffff0a, 0xffffff0d, 0xffffff0e, 0xffffff0f},
        {0xffffff1a, 0xffffff2d, 0xffffff3e, 0xffffff4f},
        {0xffffff50, 0xffffff60, 0xffffff70, 0xffffff80},
        {0xffffff90, 0xffffffa0, 0xffffffb0, 0xffffffc0},
        {0xffffffd0, 0xffffffe0, 0xfffffff0, 0xffffffff},
        {0xffffff73, 0xffffff77, 0xffffff7a, 0xffffff7b},
        {0xffffff7c, 0xffffff7d, 0xffffff7e, 0xffffff7f},
    };

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device for feature level %#x.\n", feature_level);
        return;
    }

    ID3D11Device_GetImmediateContext(device, &context);

    /* Structured buffer views */
    buffer_desc.ByteWidth = 64;
    buffer_desc.Usage = D3D11_USAGE_DEFAULT;
    buffer_desc.BindFlags = D3D11_BIND_UNORDERED_ACCESS;
    buffer_desc.CPUAccessFlags = 0;
    buffer_desc.MiscFlags = D3D11_RESOURCE_MISC_BUFFER_STRUCTURED;
    buffer_desc.StructureByteStride = 8;
    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, &buffer);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, NULL, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    uav_desc.Format = DXGI_FORMAT_UNKNOWN;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 4;
    U(uav_desc).Buffer.Flags = 0;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(uvec4_data); ++i)
    {
        uvec4 = uvec4_data[i];
        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav, &uvec4.x);
        get_buffer_readback(buffer, &rb);
        SetRect(&rect, 0, 0, buffer_desc.ByteWidth / sizeof(uvec4.x), 1);
        check_readback_data_color(&rb, &rect, uvec4.x, 0);
        release_resource_readback(&rb);

        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav2, &fe_uvec4.x);
        get_buffer_readback(buffer, &rb);
        SetRect(&rect, 0, 0, U(uav_desc).Buffer.NumElements * buffer_desc.StructureByteStride / sizeof(uvec4.x), 1);
        check_readback_data_color(&rb, &rect, fe_uvec4.x, 0);
        rect.left = rect.right;
        rect.right = buffer_desc.ByteWidth / sizeof(uvec4.x);
        check_readback_data_color(&rb, &rect, uvec4.x, 0);
        release_resource_readback(&rb);
    }

    ID3D11Buffer_Release(buffer);
    ID3D11UnorderedAccessView_Release(uav);
    ID3D11UnorderedAccessView_Release(uav2);

    /* Raw buffer views */
    buffer_desc.MiscFlags = D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS;
    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, &buffer);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    uav_desc.Format = DXGI_FORMAT_R32_TYPELESS;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 16;
    U(uav_desc).Buffer.Flags = D3D11_BUFFER_UAV_FLAG_RAW;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    U(uav_desc).Buffer.FirstElement = 8;
    U(uav_desc).Buffer.NumElements = 8;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(uvec4_data); ++i)
    {
        uvec4 = uvec4_data[i];
        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav, &uvec4.x);
        get_buffer_readback(buffer, &rb);
        SetRect(&rect, 0, 0, buffer_desc.ByteWidth / sizeof(uvec4.x), 1);
        check_readback_data_color(&rb, &rect, uvec4.x, 0);
        release_resource_readback(&rb);

        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav2, &fe_uvec4.x);
        get_buffer_readback(buffer, &rb);
        SetRect(&rect, 0, 0, U(uav_desc).Buffer.FirstElement, 1);
        check_readback_data_color(&rb, &rect, uvec4.x, 0);
        SetRect(&rect, U(uav_desc).Buffer.FirstElement, 0, buffer_desc.ByteWidth / sizeof(uvec4.x), 1);
        check_readback_data_color(&rb, &rect, fe_uvec4.x, 0);
        release_resource_readback(&rb);
    }

    ID3D11Buffer_Release(buffer);
    ID3D11UnorderedAccessView_Release(uav);
    ID3D11UnorderedAccessView_Release(uav2);

    /* Typed buffer views */
    buffer_desc.MiscFlags = 0;
    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, &buffer);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    uav_desc.Format = DXGI_FORMAT_R32_SINT;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 16;
    U(uav_desc).Buffer.Flags = 0;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    U(uav_desc).Buffer.FirstElement = 9;
    U(uav_desc).Buffer.NumElements = 7;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(uvec4_data); ++i)
    {
        uvec4 = uvec4_data[i];
        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav, &uvec4.x);
        get_buffer_readback(buffer, &rb);
        SetRect(&rect, 0, 0, buffer_desc.ByteWidth / sizeof(uvec4.x), 1);
        check_readback_data_color(&rb, &rect, uvec4.x, 0);
        release_resource_readback(&rb);

        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav2, &fe_uvec4.x);
        get_buffer_readback(buffer, &rb);
        SetRect(&rect, 0, 0, U(uav_desc).Buffer.FirstElement, 1);
        check_readback_data_color(&rb, &rect, uvec4.x, 0);
        SetRect(&rect, U(uav_desc).Buffer.FirstElement, 0, buffer_desc.ByteWidth / sizeof(uvec4.x), 1);
        check_readback_data_color(&rb, &rect, fe_uvec4.x, 0);
        release_resource_readback(&rb);
    }

    ID3D11UnorderedAccessView_Release(uav);
    ID3D11UnorderedAccessView_Release(uav2);

    uav_desc.Format = DXGI_FORMAT_R32G32B32A32_SINT;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 4;
    U(uav_desc).Buffer.Flags = 0;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    U(uav_desc).Buffer.FirstElement = 2;
    U(uav_desc).Buffer.NumElements = 2;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(uvec4_data); ++i)
    {
        const struct uvec4 *data = NULL;
        BOOL all_match;

        uvec4 = uvec4_data[i];
        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav, &uvec4.x);
        get_buffer_readback(buffer, &rb);
        for (x = 0, all_match = TRUE; x < buffer_desc.ByteWidth / sizeof(uvec4); ++x)
        {
            const struct uvec4 broken_result = {uvec4.x, uvec4.x, uvec4.x, uvec4.x}; /* Intel */
            data = get_readback_uvec4(&rb, x, 0);
            if (!(compare_uvec4(data, &uvec4) || broken(compare_uvec4(data, &broken_result))))
            {
                all_match = FALSE;
                break;
            }
        }
        ok(all_match, "Got {%#x, %#x, %#x, %#x}, expected {%#x, %#x, %#x, %#x} at %u.\n",
                data->x, data->y, data->z, data->w, uvec4.x, uvec4.y, uvec4.z, uvec4.w, x);
        release_resource_readback(&rb);

        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav2, &fe_uvec4.x);
        get_buffer_readback(buffer, &rb);
        for (x = 0, all_match = TRUE; x < buffer_desc.ByteWidth / sizeof(uvec4); ++x)
        {
            struct uvec4 broken_result;
            data = get_readback_uvec4(&rb, x, 0);
            uvec4 = U(uav_desc).Buffer.FirstElement <= x ? fe_uvec4 : uvec4_data[i];
            broken_result.x = broken_result.y = broken_result.z = broken_result.w = uvec4.x;
            if (!(compare_uvec4(data, &uvec4) || broken(compare_uvec4(data, &broken_result))))
            {
                all_match = FALSE;
                break;
            }
        }
        ok(all_match, "Got {%#x, %#x, %#x, %#x}, expected {%#x, %#x, %#x, %#x} at %u.\n",
                data->x, data->y, data->z, data->w, uvec4.x, uvec4.y, uvec4.z, uvec4.w, x);
        release_resource_readback(&rb);
    }

    uvec4.x = uvec4.y = uvec4.z = uvec4.w = 0xdeadbeef;
    ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav, &uvec4.x);
    ID3D11UnorderedAccessView_Release(uav);
    ID3D11UnorderedAccessView_Release(uav2);

    uav_desc.Format = DXGI_FORMAT_R8G8B8A8_SINT;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 16;
    U(uav_desc).Buffer.Flags = 0;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    U(uav_desc).Buffer.FirstElement = 8;
    U(uav_desc).Buffer.NumElements = 8;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav2);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(uvec4_data); ++i)
    {
        uvec4 = uvec4_data[i];
        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav, &uvec4.x);
        get_buffer_readback(buffer, &rb);
        todo_wine check_rgba_sint8(get_readback_color(&rb, 0, 0, 0), &uvec4);
        todo_wine check_rgba_sint8(get_readback_color(&rb, 7, 0, 0), &uvec4);
        todo_wine check_rgba_sint8(get_readback_color(&rb, 15, 0, 0), &uvec4);
        release_resource_readback(&rb);

        ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav2, &fe_uvec4.x);
        get_buffer_readback(buffer, &rb);
        todo_wine check_rgba_sint8(get_readback_color(&rb, 0, 0, 0), &uvec4);
        todo_wine check_rgba_sint8(get_readback_color(&rb, U(uav_desc).Buffer.FirstElement - 1, 0, 0), &uvec4);
        todo_wine check_rgba_sint8(get_readback_color(&rb, U(uav_desc).Buffer.FirstElement, 0, 0), &fe_uvec4);
        release_resource_readback(&rb);
    }

    ID3D11UnorderedAccessView_Release(uav);
    ID3D11UnorderedAccessView_Release(uav2);

    ID3D11Buffer_Release(buffer);

    ID3D11DeviceContext_Release(context);
    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_clear_image_unordered_access_view(void)
{
    unsigned int expected_colour, actual_colour;
    D3D11_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    unsigned int i, j, d, p, x, y, z, layer;
    struct d3d11_test_context test_context;
    unsigned int image_size, image_depth;
    struct resource_desc resource_desc;
    ID3D11UnorderedAccessView *uav[2];
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    BOOL is_small_float_format;
    ID3D11Resource *resource;
    BOOL is_inside, success;
    ID3D11Device *device;
    UINT clear_value[4];
    HRESULT hr;

#define IMAGE_SIZE 16
    static const struct
    {
        DXGI_FORMAT format;
        unsigned int image_mips;
        unsigned int image_layers;
        unsigned int mip_level;
        unsigned int first_layer;
        unsigned int layer_count;
        unsigned int values[4];
        unsigned int expected;
        BOOL is_float;
        unsigned int clamped;
    }
    tests[] =
    {
        /* Test clearing a specific mip level. */
        {DXGI_FORMAT_R32_FLOAT,       2, 1, 0, 0, 1, {1,          0, 0, 0}, 0x00000001},
        {DXGI_FORMAT_R32_FLOAT,       2, 1, 1, 0, 1, {1,          0, 0, 0}, 0x00000001},
        {DXGI_FORMAT_R32_FLOAT,       2, 1, 0, 0, 1, {0x3f000000, 0, 0, 0}, 0x3f000000, TRUE},
        {DXGI_FORMAT_R32_FLOAT,       2, 1, 1, 0, 1, {0x3f000000, 0, 0, 0}, 0x3f000000, TRUE},
        /* Test clearing specific array layers. */
        {DXGI_FORMAT_R32_FLOAT,       1, IMAGE_SIZE, 0, 0, IMAGE_SIZE, {1, 0, 0, 0}, 0x00000001},
        {DXGI_FORMAT_R32_FLOAT,       1, IMAGE_SIZE, 0, 3, 2,          {1, 0, 0, 0}, 0x00000001},
        {DXGI_FORMAT_R32_FLOAT,       1, IMAGE_SIZE, 0, 0, IMAGE_SIZE, {0x3f000000, 0, 0, 0}, 0x3f000000, TRUE},
        {DXGI_FORMAT_R32_FLOAT,       1, IMAGE_SIZE, 0, 3, 2,          {0x3f000000, 0, 0, 0}, 0x3f000000, TRUE},
        /* Test uint clears with formats. */
        {DXGI_FORMAT_R16G16_UINT,     1, 1, 0, 0, 1, {1,       2, 3, 4}, 0x00020001},
        {DXGI_FORMAT_R16G16_UINT,     1, 1, 0, 0, 1, {0x12345, 0, 0, 0}, 0x00002345, FALSE, 0x0000ffff},
        {DXGI_FORMAT_R16G16_UNORM,    1, 1, 0, 0, 1, {1,       2, 3, 4}, 0x00020001},
        {DXGI_FORMAT_R16G16_FLOAT,    1, 1, 0, 0, 1, {1,       2, 3, 4}, 0x00020001},
        {DXGI_FORMAT_R8G8B8A8_UINT,   1, 1, 0, 0, 1, {1,       2, 3, 4}, 0x04030201},
        {DXGI_FORMAT_R8G8B8A8_UINT,   1, 1, 0, 0, 1, {0x123,   0, 0, 0}, 0x00000023, FALSE, 0x000000ff},
        {DXGI_FORMAT_R8G8B8A8_UNORM,  1, 1, 0, 0, 1, {1,       2, 3, 4}, 0x04030201},
        {DXGI_FORMAT_R11G11B10_FLOAT, 1, 1, 0, 0, 1, {1,       2, 3, 4}, 0x00c01001},
        /* Test float clears with formats. */
        {DXGI_FORMAT_R16G16_UNORM,    1, 1, 0, 0, 1,
                {0x3f000000 /* 0.5f */, 0x3f800000 /* 1.0f */, 0, 0}, 0xffff8000, TRUE},
        {DXGI_FORMAT_R16G16_FLOAT,    1, 1, 0, 0, 1,
                {0x3f000000 /* 0.5f */, 0x3f800000 /* 1.0f */, 0, 0}, 0x3c003800, TRUE},
        {DXGI_FORMAT_R8G8B8A8_UNORM,  1, 1, 0, 0, 1,
                {0x3f000000 /* 0.5f */, 0x3f800000 /* 1.0f */, 0, 0}, 0x0000ff80, TRUE},
        {DXGI_FORMAT_R8G8B8A8_UNORM,  1, 1, 0, 0, 1,
                {0, 0, 0x3f000000 /* 0.5f */, 0x3f800000 /* 1.0f */}, 0xff800000, TRUE},
        {DXGI_FORMAT_R11G11B10_FLOAT, 1, 1, 0, 0, 1,
                {0x3f000000 /* 1.0f */, 0 /* 0.0f */, 0xbf800000 /* -1.0f */, 0x3f000000 /* 1.0f */},
                0x00000380, TRUE},
    };

    static const struct
    {
        D3D11_RESOURCE_DIMENSION resource_dim;
        D3D11_UAV_DIMENSION view_dim;
        BOOL is_layered;
    }
    uav_dimensions[] =
    {
        {D3D11_RESOURCE_DIMENSION_TEXTURE2D, D3D11_UAV_DIMENSION_TEXTURE2D,      FALSE},
        {D3D11_RESOURCE_DIMENSION_TEXTURE2D, D3D11_UAV_DIMENSION_TEXTURE2DARRAY, TRUE },
        {D3D11_RESOURCE_DIMENSION_TEXTURE3D, D3D11_UAV_DIMENSION_TEXTURE3D,      TRUE },
    };

    if (!init_test_context(&test_context, NULL))
        return;
    device = test_context.device;
    context = test_context.immediate_context;

    memset(&resource_desc, 0, sizeof(resource_desc));
    resource_desc.width = IMAGE_SIZE;
    resource_desc.height = IMAGE_SIZE;
    resource_desc.sample_desc.Count = 1;
    resource_desc.usage = D3D11_USAGE_DEFAULT;
    resource_desc.bind_flags = D3D11_BIND_UNORDERED_ACCESS;

    for (d = 0; d < ARRAY_SIZE(uav_dimensions); ++d)
    {
        for (i = 0; i < ARRAY_SIZE(tests); ++i)
        {
            if (tests[i].image_layers > 1 && !uav_dimensions[d].is_layered)
                continue;

            /* Expected behaviour with partial layer coverage is unclear. */
            if (uav_dimensions[d].view_dim == D3D11_UAV_DIMENSION_TEXTURE3D
                    && tests[i].image_layers != tests[i].layer_count)
                continue;

            winetest_push_context("Dim %u, Test %u", d, i);

            resource_desc.dimension = uav_dimensions[d].resource_dim;
            resource_desc.depth_or_array_size = tests[i].image_layers;
            resource_desc.level_count = tests[i].image_mips;
            resource_desc.format = tests[i].format;
            if (FAILED(hr = create_resource(device, &resource_desc, NULL, &resource)))
            {
                skip("Failed to create resource, hr %#lx.\n", hr);
                winetest_pop_context();
                continue;
            }

            uav_desc.Format = tests[i].format;
            uav_desc.ViewDimension = uav_dimensions[d].view_dim;

            for (j = 0; j < 2; ++j)
            {
                unsigned int first_layer = j ? 0 : tests[i].first_layer;
                unsigned int layer_count = j ? tests[i].image_layers : tests[i].layer_count;

                switch (uav_desc.ViewDimension)
                {
                    case D3D11_UAV_DIMENSION_TEXTURE2D:
                        uav_desc.Texture2D.MipSlice = tests[i].mip_level;
                        break;

                    case D3D11_UAV_DIMENSION_TEXTURE2DARRAY:
                        uav_desc.Texture2DArray.MipSlice = tests[i].mip_level;
                        uav_desc.Texture2DArray.FirstArraySlice = first_layer;
                        uav_desc.Texture2DArray.ArraySize = layer_count;
                        break;

                    case D3D11_UAV_DIMENSION_TEXTURE3D:
                        uav_desc.Texture3D.MipSlice = tests[i].mip_level;
                        uav_desc.Texture3D.FirstWSlice = first_layer;
                        uav_desc.Texture3D.WSize = layer_count;
                        break;

                    default:
                        ok(0, "Unhandled uav dimension %#x.\n", uav_dimensions[d].view_dim);
                        break;
                }

                hr = ID3D11Device_CreateUnorderedAccessView(device, resource, &uav_desc, &uav[j]);
                ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
            }

            for (j = 0; j < 4; ++j)
            {
                clear_value[j] = tests[i].expected ? 0u : ~0u;
            }

            ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav[1], clear_value);
            if (tests[i].is_float)
                ID3D11DeviceContext_ClearUnorderedAccessViewFloat(context, uav[0], (const float *)tests[i].values);
            else
                ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav[0], tests[i].values);

            image_depth = uav_dimensions[d].resource_dim == D3D11_RESOURCE_DIMENSION_TEXTURE3D
                    ? max(tests[i].image_layers >> tests[i].mip_level, 1u) : 1;
            image_size = max(IMAGE_SIZE >> tests[i].mip_level, 1u);

            is_small_float_format = tests[i].format == DXGI_FORMAT_R16G16_UNORM
                    || tests[i].format == DXGI_FORMAT_R16G16_FLOAT
                    || tests[i].format == DXGI_FORMAT_R11G11B10_FLOAT
                    || tests[i].format == DXGI_FORMAT_R8G8B8A8_UNORM;
            for (layer = 0; layer < tests[i].image_layers / image_depth; ++layer)
            {
                get_resource_readback(resource, tests[i].mip_level + (layer * tests[i].image_mips), &rb);

                success = TRUE;
                expected_colour = actual_colour = x = y = z = 0;
                for (p = 0; p < image_depth * image_size * image_size; ++p)
                {
                    x = p % image_size;
                    y = (p / image_size) % image_size;
                    z = p / (image_size * image_size);

                    if (uav_dimensions[d].resource_dim == D3D11_RESOURCE_DIMENSION_TEXTURE3D)
                        is_inside = z >= tests[i].first_layer
                                && z < tests[i].first_layer + tests[i].layer_count;
                    else
                        is_inside = layer >= tests[i].first_layer
                                && layer < tests[i].first_layer + tests[i].layer_count;

                    expected_colour = is_inside ? tests[i].expected : clear_value[0];
                    actual_colour = get_readback_u32(&rb, x, y, z);
                    if (!(success = compare_color(actual_colour, expected_colour, tests[i].is_float ? 1 : 0)
                            /* Some drivers/GPUs clamp clear values that can't
                             * be represented by the format. (Windows 7
                             * testbot, AMD PALM) */
                            || broken(is_inside && tests[i].clamped && actual_colour == tests[i].clamped)
                            /* Some drivers/GPUs mishandle integer clears of
                             * small float/normalised formats. (AMD PALM) */
                            || broken(is_inside && !tests[i].is_float && is_small_float_format && !actual_colour)))
                        break;
                }
                ok(success, "At layer %u, (%u,%u,%u), expected 0x%08x, got 0x%08x.\n",
                        layer, x, y, z, expected_colour, actual_colour);

                release_resource_readback(&rb);
            }

            ID3D11UnorderedAccessView_Release(uav[1]);
            ID3D11UnorderedAccessView_Release(uav[0]);
            ID3D11Resource_Release(resource);
            winetest_pop_context();
        }
    }

    release_test_context(&test_context);
#undef IMAGE_SIZE
}

static void test_initial_depth_stencil_state(void)
{
    static const struct vec4 green = {0.0f, 1.0f, 0.0f, 1.0f};
    static const struct vec4 red = {1.0f, 0.0f, 0.0f, 1.0f};
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11DepthStencilView *dsv;
    ID3D11Texture2D *texture;
    ID3D11Device *device;
    unsigned int count;
    D3D11_VIEWPORT vp;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_D32_FLOAT;
    texture_desc.BindFlags = D3D11_BIND_DEPTH_STENCIL;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, NULL, &dsv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &test_context.backbuffer_rtv, dsv);

    count = 1;
    ID3D11DeviceContext_RSGetViewports(context, &count, &vp);

    /* check if depth function is D3D11_COMPARISON_LESS */
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);
    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 0.5f, 0);
    set_viewport(context, vp.TopLeftX, vp.TopLeftY, vp.Width, vp.Height, 0.4f, 0.4f);
    draw_color_quad(&test_context, &green);
    draw_color_quad(&test_context, &red);
    set_viewport(context, vp.TopLeftX, vp.TopLeftY, vp.Width, vp.Height, 0.6f, 0.6f);
    draw_color_quad(&test_context, &red);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 1);
    check_texture_float(texture, 0.4f, 1);

    ID3D11DepthStencilView_Release(dsv);
    ID3D11Texture2D_Release(texture);
    release_test_context(&test_context);
}

static void test_draw_depth_only(void)
{
    struct d3d11_test_context test_context;
    ID3D11PixelShader *ps_color, *ps_depth;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11DepthStencilView *dsv;
    struct resource_readback rb;
    ID3D11Texture2D *texture;
    ID3D11Device *device;
    unsigned int i, j;
    struct vec4 depth;
    ID3D11Buffer *cb;
    HRESULT hr;

    static const DWORD ps_color_code[] =
    {
#if 0
        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            return float4(0.0, 1.0, 0.0, 1.0);
        }
#endif
        0x43425844, 0x30240e72, 0x012f250c, 0x8673c6ea, 0x392e4cec, 0x00000001, 0x000000d4, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000038, 0x00000040,
        0x0000000e, 0x03000065, 0x001020f2, 0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002,
        0x00000000, 0x3f800000, 0x00000000, 0x3f800000, 0x0100003e,
    };
    static const DWORD ps_depth_code[] =
    {
#if 0
        float depth;

        float main() : SV_Depth
        {
            return depth;
        }
#endif
        0x43425844, 0x91af6cd0, 0x7e884502, 0xcede4f54, 0x6f2c9326, 0x00000001, 0x000000b0, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0xffffffff,
        0x00000e01, 0x445f5653, 0x68747065, 0xababab00, 0x52444853, 0x00000038, 0x00000040, 0x0000000e,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x02000065, 0x0000c001, 0x05000036, 0x0000c001,
        0x0020800a, 0x00000000, 0x00000000, 0x0100003e,
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(depth), NULL);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_D32_FLOAT;
    texture_desc.BindFlags = D3D11_BIND_DEPTH_STENCIL;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateDepthStencilView(device, (ID3D11Resource *)texture, NULL, &dsv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_color_code, sizeof(ps_color_code), NULL, &ps_color);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_depth_code, sizeof(ps_depth_code), NULL, &ps_depth);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);
    ID3D11DeviceContext_PSSetShader(context, ps_color, NULL, 0);
    ID3D11DeviceContext_OMSetRenderTargets(context, 0, NULL, dsv);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0);
    check_texture_float(texture, 1.0f, 1);
    draw_quad(&test_context);
    check_texture_float(texture, 0.0f, 1);

    ID3D11DeviceContext_PSSetShader(context, ps_depth, NULL, 0);

    depth.x = 0.7f;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &depth, 0, 0);
    draw_quad(&test_context);
    check_texture_float(texture, 0.0f, 1);
    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0);
    check_texture_float(texture, 1.0f, 1);
    draw_quad(&test_context);
    check_texture_float(texture, 0.7f, 1);
    depth.x = 0.8f;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &depth, 0, 0);
    draw_quad(&test_context);
    check_texture_float(texture, 0.7f, 1);
    depth.x = 0.5f;
    ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &depth, 0, 0);
    draw_quad(&test_context);
    check_texture_float(texture, 0.5f, 1);

    ID3D11DeviceContext_ClearDepthStencilView(context, dsv, D3D11_CLEAR_DEPTH, 1.0f, 0);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            depth.x = 1.0f / 16.0f * (j + 4 * i);
            ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &depth, 0, 0);

            set_viewport(context, 160.0f * j, 120.0f * i, 160.0f, 120.0f, 0.0f, 1.0f);

            draw_quad(&test_context);
        }
    }
    get_texture_readback(texture, 0, &rb);
    for (i = 0; i < 4; ++i)
    {
        for (j = 0; j < 4; ++j)
        {
            float obtained_depth, expected_depth;

            obtained_depth = get_readback_float(&rb, 80 + j * 160, 60 + i * 120);
            expected_depth = 1.0f / 16.0f * (j + 4 * i);
            ok(compare_float(obtained_depth, expected_depth, 1),
                    "Got unexpected depth %.8e at (%u, %u), expected %.8e.\n",
                    obtained_depth, j, i, expected_depth);
        }
    }
    release_resource_readback(&rb);

    ID3D11Buffer_Release(cb);
    ID3D11PixelShader_Release(ps_color);
    ID3D11PixelShader_Release(ps_depth);
    ID3D11DepthStencilView_Release(dsv);
    ID3D11Texture2D_Release(texture);
    release_test_context(&test_context);
}

static void test_draw_uav_only(void)
{
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11UnorderedAccessView *uav;
    ID3D11DeviceContext *context;
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        RWTexture2D<int> u;

        void main()
        {
            InterlockedAdd(u[uint2(0, 0)], 1);
        }
#endif
        0x43425844, 0x237a8398, 0xe7b34c17, 0xa28c91a4, 0xb3614d73, 0x00000001, 0x0000009c, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000048, 0x00000050, 0x00000012, 0x0100086a,
        0x0400189c, 0x0011e000, 0x00000000, 0x00003333, 0x0a0000ad, 0x0011e000, 0x00000000, 0x00004002,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00004001, 0x00000001, 0x0100003e,
    };
    static const D3D_FEATURE_LEVEL feature_level = D3D_FEATURE_LEVEL_11_0;
    static const UINT values[4] = {0};

    if (!init_test_context(&test_context, &feature_level))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 1;
    texture_desc.Height = 1;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R32_SINT;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_UNORDERED_ACCESS;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)texture, NULL, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_OMSetRenderTargetsAndUnorderedAccessViews(context, 0, NULL, NULL,
            0, 1, &uav, NULL);

    ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav, values);
    set_viewport(context, 0.0f, 0.0f, 10.0f, 10.0f, 0.0f, 0.0f);
    ID3D11DeviceContext_ClearUnorderedAccessViewUint(context, uav, values);
    draw_quad(&test_context);
    check_texture_color(texture, 100, 1);

    draw_quad(&test_context);
    draw_quad(&test_context);
    draw_quad(&test_context);
    draw_quad(&test_context);
    check_texture_color(texture, 500, 1);

    ID3D11PixelShader_Release(ps);
    ID3D11Texture2D_Release(texture);
    ID3D11UnorderedAccessView_Release(uav);
    release_test_context(&test_context);
}

static void test_cb_relative_addressing(void)
{
    struct d3d11_test_context test_context;
    ID3D11Buffer *colors_cb, *index_cb;
    unsigned int i, index[4] = {0};
    ID3D11DeviceContext *context;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
int color_index;

cbuffer colors
{
    float4 colors[8];
};

struct vs_in
{
    float4 position : POSITION;
};

struct vs_out
{
    float4 position : SV_POSITION;
    float4 color : COLOR;
};

vs_out main(const vs_in v)
{
    vs_out o;

    o.position = v.position;
    o.color = colors[color_index];

    return o;
}
#endif
        0x43425844, 0xc2eb30bf, 0x2868c855, 0xaa34b609, 0x1f4957d4, 0x00000001, 0x00000164, 0x00000003,
        0x0000002c, 0x00000060, 0x000000b4, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052, 0x58454853, 0x000000a8, 0x00010050,
        0x0000002a, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x04000859, 0x00208e46,
        0x00000001, 0x00000008, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x02000068, 0x00000001, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x06000036, 0x00100012, 0x00000000, 0x0020800a, 0x00000000,
        0x00000000, 0x07000036, 0x001020f2, 0x00000001, 0x04208e46, 0x00000001, 0x0010000a, 0x00000000,
        0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
struct ps_in
{
    float4 position : SV_POSITION;
    float4 color : COLOR;
};

float4 main(const ps_in v) : SV_TARGET
{
    return v.color;
}
#endif
        0x43425844, 0x1a6def50, 0x9c069300, 0x7cce68f0, 0x621239b9, 0x00000001, 0x000000f8, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x0000003c, 0x00000050,
        0x0000000f, 0x0100086a, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000001, 0x0100003e,
    };
    static const struct
    {
        float color[4];
    }
    colors[10] =
    {
        {{0.0f, 0.0f, 0.0f, 1.0f}},
        {{0.0f, 0.0f, 1.0f, 0.0f}},
        {{0.0f, 0.0f, 1.0f, 1.0f}},
        {{0.0f, 1.0f, 0.0f, 0.0f}},
        {{0.0f, 1.0f, 0.0f, 1.0f}},
        {{0.0f, 1.0f, 1.0f, 0.0f}},
        {{0.0f, 1.0f, 1.0f, 1.0f}},
        {{1.0f, 0.0f, 0.0f, 0.0f}},
        {{1.0f, 0.0f, 0.0f, 1.0f}},
        {{1.0f, 0.0f, 1.0f, 0.0f}},
    };
    static const struct
    {
        unsigned int index;
        DWORD expected;
    }
    test_data[] =
    {
        {0, 0xff000000},
        {1, 0x00ff0000},
        {2, 0xffff0000},
        {3, 0x0000ff00},
        {4, 0xff00ff00},
        {5, 0x00ffff00},
        {6, 0xffffff00},
        {7, 0x000000ff},

        {8, 0xff0000ff},
        {9, 0x00ff00ff},
    };
    static const float white_color[] = {1.0f, 1.0f, 1.0f, 1.0f};
    static const D3D_FEATURE_LEVEL feature_level = D3D_FEATURE_LEVEL_11_0;

    if (!init_test_context(&test_context, &feature_level))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    colors_cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(colors), &colors);
    index_cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(index), NULL);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_VSSetConstantBuffers(context, 0, 1, &index_cb);
    ID3D11DeviceContext_VSSetConstantBuffers(context, 1, 1, &colors_cb);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    for (i = 0; i < ARRAY_SIZE(test_data); ++i)
    {
        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white_color);

        index[0] = test_data[i].index;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)index_cb, 0, NULL, &index, 0, 0);

        draw_quad_vs(&test_context, vs_code, sizeof(vs_code));
        check_texture_color(test_context.backbuffer, test_data[i].expected, 1);
    }

    ID3D11Buffer_Release(index_cb);
    ID3D11Buffer_Release(colors_cb);
    ID3D11PixelShader_Release(ps);

    release_test_context(&test_context);
}

static void test_vs_input_relative_addressing(void)
{
    struct d3d11_test_context test_context;
    ID3D11DeviceContext *context;
    unsigned int offset, stride;
    unsigned int index[4] = {0};
    ID3D11PixelShader *ps;
    ID3D11Buffer *vb, *cb;
    ID3D11Device *device;
    unsigned int i;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        struct vertex
        {
            float4 position : POSITION;
            float4 colors[4] : COLOR;
        };

        uint index;

        void main(vertex vin, out float4 position : SV_Position,
                out float4 color : COLOR)
        {
            position = vin.position;
            color = vin.colors[index];
        }
#endif
        0x43425844, 0x8623dd89, 0xe37fecf5, 0xea3fdfe1, 0xdf36e4e4, 0x00000001, 0x000001f4, 0x00000003,
        0x0000002c, 0x000000c4, 0x00000118, 0x4e475349, 0x00000090, 0x00000005, 0x00000008, 0x00000080,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000089, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x00000089, 0x00000001, 0x00000000, 0x00000003, 0x00000002,
        0x00000f0f, 0x00000089, 0x00000002, 0x00000000, 0x00000003, 0x00000003, 0x00000f0f, 0x00000089,
        0x00000003, 0x00000000, 0x00000003, 0x00000004, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0x4c4f4300,
        0xab00524f, 0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001,
        0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0xabab0052, 0x52444853, 0x000000d4,
        0x00010040, 0x00000035, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005f, 0x001010f2,
        0x00000000, 0x0300005f, 0x001010f2, 0x00000001, 0x0300005f, 0x001010f2, 0x00000002, 0x0300005f,
        0x001010f2, 0x00000003, 0x0300005f, 0x001010f2, 0x00000004, 0x04000067, 0x001020f2, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x02000068, 0x00000001, 0x0400005b, 0x001010f2,
        0x00000001, 0x00000004, 0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x06000036,
        0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x07000036, 0x001020f2, 0x00000001,
        0x00d01e46, 0x00000001, 0x0010000a, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        struct vs_out
        {
            float4 position : SV_POSITION;
            float4 color : COLOR;
        };

        float4 main(struct vs_out i) : SV_TARGET
        {
            return i.color;
        }
#endif
        0x43425844, 0xe2087fa6, 0xa35fbd95, 0x8e585b3f, 0x67890f54, 0x00000001, 0x000000f4, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000038, 0x00000040,
        0x0000000e, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000001, 0x0100003e,
    };
    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32_FLOAT,   0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"COLOR",    0, DXGI_FORMAT_R8G8B8A8_UNORM, 1,  0, D3D11_INPUT_PER_INSTANCE_DATA, 1},
        {"COLOR",    1, DXGI_FORMAT_R8G8B8A8_UNORM, 1,  4, D3D11_INPUT_PER_INSTANCE_DATA, 1},
        {"COLOR",    2, DXGI_FORMAT_R8G8B8A8_UNORM, 1,  8, D3D11_INPUT_PER_INSTANCE_DATA, 1},
        {"COLOR",    3, DXGI_FORMAT_R8G8B8A8_UNORM, 1, 12, D3D11_INPUT_PER_INSTANCE_DATA, 1},
    };
    static const unsigned int colors[] = {0xff0000ff, 0xff00ff00, 0xffff0000, 0xff0f0f0f};
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};

    if (!init_test_context(&test_context, NULL))
        return;
    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &test_context.input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(index), NULL);
    ID3D11DeviceContext_VSSetConstantBuffers(context, 0, 1, &cb);

    vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(colors), colors);
    stride = sizeof(colors);
    offset = 0;
    ID3D11DeviceContext_IASetVertexBuffers(context, 1, 1, &vb, &stride, &offset);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    for (i = 0; i < ARRAY_SIZE(colors); ++i)
    {
        *index = i;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, index, 0, 0);
        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);
        draw_quad_vs(&test_context, vs_code, sizeof(vs_code));
        check_texture_color(test_context.backbuffer, colors[i], 1);
    }

    ID3D11Buffer_Release(cb);
    ID3D11Buffer_Release(vb);
    ID3D11PixelShader_Release(ps);
    release_test_context(&test_context);
}

static void test_getdc(void)
{
    static const struct
    {
        const char *name;
        DXGI_FORMAT format;
        BOOL getdc_supported;
    }
    testdata[] =
    {
        {"B8G8R8A8_UNORM",      DXGI_FORMAT_B8G8R8A8_UNORM,      TRUE },
        {"B8G8R8A8_TYPELESS",   DXGI_FORMAT_B8G8R8A8_TYPELESS,   TRUE },
        {"B8G8R8A8_UNORM_SRGB", DXGI_FORMAT_B8G8R8A8_UNORM_SRGB, TRUE },
        {"B8G8R8X8_UNORM",      DXGI_FORMAT_B8G8R8X8_UNORM,      FALSE },
        {"B8G8R8X8_TYPELESS",   DXGI_FORMAT_B8G8R8X8_TYPELESS,   FALSE },
        {"B8G8R8X8_UNORM_SRGB", DXGI_FORMAT_B8G8R8X8_UNORM_SRGB, FALSE },
    };
    struct device_desc device_desc;
    D3D11_TEXTURE2D_DESC desc;
    ID3D11Texture2D *texture;
    IDXGISurface1 *surface;
    ID3D11Device *device;
    unsigned int i;
    ULONG refcount;
    HRESULT hr;
    HDC dc;

    device_desc.feature_level = NULL;
    device_desc.flags = D3D11_CREATE_DEVICE_BGRA_SUPPORT;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }

    /* Without D3D11_RESOURCE_MISC_GDI_COMPATIBLE. */
    desc.Width = 512;
    desc.Height = 512;
    desc.MipLevels = 1;
    desc.ArraySize = 1;
    desc.Format = DXGI_FORMAT_B8G8R8A8_UNORM;
    desc.SampleDesc.Count = 1;
    desc.SampleDesc.Quality = 0;
    desc.Usage = D3D11_USAGE_DEFAULT;
    desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    desc.CPUAccessFlags = 0;
    desc.MiscFlags = 0;
    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Texture2D_QueryInterface(texture, &IID_IDXGISurface1, (void**)&surface);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = IDXGISurface1_GetDC(surface, FALSE, &dc);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);

    IDXGISurface1_Release(surface);
    ID3D11Texture2D_Release(texture);

    desc.MiscFlags = D3D11_RESOURCE_MISC_GDI_COMPATIBLE;
    hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Texture2D_QueryInterface(texture, &IID_IDXGISurface1, (void**)&surface);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = IDXGISurface1_ReleaseDC(surface, NULL);
    todo_wine ok(hr == DXGI_ERROR_INVALID_CALL, "Got unexpected hr %#lx.\n", hr);

    hr = IDXGISurface1_GetDC(surface, FALSE, &dc);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = IDXGISurface1_ReleaseDC(surface, NULL);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    IDXGISurface1_Release(surface);
    ID3D11Texture2D_Release(texture);

    for (i = 0; i < ARRAY_SIZE(testdata); ++i)
    {
        static const unsigned int bit_count = 32;
        unsigned int width_bytes;
        DIBSECTION dib;
        HBITMAP bitmap;
        DWORD type;
        int size;

        desc.Width = 64;
        desc.Height = 64;
        desc.MipLevels = 1;
        desc.ArraySize = 1;
        desc.Format = testdata[i].format;
        desc.SampleDesc.Count = 1;
        desc.SampleDesc.Quality = 0;
        desc.Usage = D3D11_USAGE_STAGING;
        desc.BindFlags = 0;
        desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
        desc.MiscFlags = 0;

        hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
        ok(hr == S_OK, "Format %s: Got unexpected hr %#lx.\n", testdata[i].name, hr);
        ID3D11Texture2D_Release(texture);

        /* STAGING usage, requesting GDI compatibility mode. */
        desc.MiscFlags = D3D11_RESOURCE_MISC_GDI_COMPATIBLE;
        hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
        ok(FAILED(hr), "Format %s: Got unexpected hr %#lx.\n", testdata[i].name, hr);

        desc.Usage = D3D11_USAGE_DEFAULT;
        desc.BindFlags = D3D11_BIND_RENDER_TARGET;
        desc.CPUAccessFlags = 0;
        hr = ID3D11Device_CreateTexture2D(device, &desc, NULL, &texture);
        if (testdata[i].getdc_supported)
            ok(hr == S_OK, "Format %s: Got unexpected hr %#lx.\n", testdata[i].name, hr);
        else
            ok(FAILED(hr), "Format %s: Got unexpected hr %#lx.\n", testdata[i].name, hr);

        if (FAILED(hr))
            continue;

        hr = ID3D11Texture2D_QueryInterface(texture, &IID_IDXGISurface1, (void**)&surface);
        ok(hr == S_OK, "Format %s: Got unexpected hr %#lx.\n", testdata[i].name, hr);

        dc = (void *)0x1234;
        hr = IDXGISurface1_GetDC(surface, FALSE, &dc);
        ok(hr == S_OK, "Format %s: Got unexpected hr %#lx.\n", testdata[i].name, hr);

        if (FAILED(hr))
        {
            IDXGISurface1_Release(surface);
            ID3D11Texture2D_Release(texture);
            continue;
        }

        type = GetObjectType(dc);
        ok(type == OBJ_MEMDC, "Format %s: Got unexpected object type %#lx.\n", testdata[i].name, type);
        bitmap = GetCurrentObject(dc, OBJ_BITMAP);
        type = GetObjectType(bitmap);
        ok(type == OBJ_BITMAP, "Format %s: Got unexpected object type %#lx.\n", testdata[i].name, type);

        size = GetObjectA(bitmap, sizeof(dib), &dib);
        ok(size == sizeof(dib) || broken(size == sizeof(dib.dsBm)),
                "Got unexpected size %d for format %s.\n", size, testdata[i].name);

        ok(!dib.dsBm.bmType, "Got unexpected type %#x for format %s.\n",
                dib.dsBm.bmType, testdata[i].name);
        ok(dib.dsBm.bmWidth == 64, "Got unexpected width %d for format %s.\n",
                dib.dsBm.bmWidth, testdata[i].name);
        ok(dib.dsBm.bmHeight == 64, "Got unexpected height %d for format %s.\n",
                dib.dsBm.bmHeight, testdata[i].name);
        width_bytes = ((dib.dsBm.bmWidth * bit_count + 31) >> 3) & ~3;
        ok(dib.dsBm.bmWidthBytes == width_bytes, "Got unexpected width bytes %d for format %s.\n",
                dib.dsBm.bmWidthBytes, testdata[i].name);
        ok(dib.dsBm.bmPlanes == 1, "Got unexpected plane count %d for format %s.\n",
                dib.dsBm.bmPlanes, testdata[i].name);
        ok(dib.dsBm.bmBitsPixel == bit_count, "Got unexpected bit count %d for format %s.\n",
                dib.dsBm.bmBitsPixel, testdata[i].name);

        if (size == sizeof(dib))
            ok(!!dib.dsBm.bmBits, "Got unexpected bits %p for format %s.\n",
                    dib.dsBm.bmBits, testdata[i].name);
        else
            ok(!dib.dsBm.bmBits, "Got unexpected bits %p for format %s.\n",
                    dib.dsBm.bmBits, testdata[i].name);

        if (size == sizeof(dib))
        {
            ok(dib.dsBmih.biSize == sizeof(dib.dsBmih), "Got unexpected size %lu for format %s.\n",
                    dib.dsBmih.biSize, testdata[i].name);
            ok(dib.dsBmih.biWidth == 64, "Got unexpected width %ld for format %s.\n",
                    dib.dsBmih.biHeight, testdata[i].name);
            ok(dib.dsBmih.biHeight == 64, "Got unexpected height %ld for format %s.\n",
                    dib.dsBmih.biHeight, testdata[i].name);
            ok(dib.dsBmih.biPlanes == 1, "Got unexpected plane count %u for format %s.\n",
                    dib.dsBmih.biPlanes, testdata[i].name);
            ok(dib.dsBmih.biBitCount == bit_count, "Got unexpected bit count %u for format %s.\n",
                    dib.dsBmih.biBitCount, testdata[i].name);
            ok(dib.dsBmih.biCompression == BI_RGB, "Got unexpected compression %#lx for format %s.\n",
                    dib.dsBmih.biCompression, testdata[i].name);
            ok(!dib.dsBmih.biSizeImage, "Got unexpected image size %lu for format %s.\n",
                    dib.dsBmih.biSizeImage, testdata[i].name);
            ok(!dib.dsBmih.biXPelsPerMeter, "Got unexpected horizontal resolution %ld for format %s.\n",
                    dib.dsBmih.biXPelsPerMeter, testdata[i].name);
            ok(!dib.dsBmih.biYPelsPerMeter, "Got unexpected vertical resolution %ld for format %s.\n",
                    dib.dsBmih.biYPelsPerMeter, testdata[i].name);
            ok(!dib.dsBmih.biClrUsed, "Got unexpected used colour count %lu for format %s.\n",
                    dib.dsBmih.biClrUsed, testdata[i].name);
            ok(!dib.dsBmih.biClrImportant, "Got unexpected important colour count %lu for format %s.\n",
                    dib.dsBmih.biClrImportant, testdata[i].name);
            ok(!dib.dsBitfields[0] && !dib.dsBitfields[1] && !dib.dsBitfields[2],
                    "Got unexpected colour masks 0x%08lx 0x%08lx 0x%08lx for format %s.\n",
                    dib.dsBitfields[0], dib.dsBitfields[1], dib.dsBitfields[2], testdata[i].name);
            ok(!dib.dshSection, "Got unexpected section %p for format %s.\n", dib.dshSection, testdata[i].name);
            ok(!dib.dsOffset, "Got unexpected offset %lu for format %s.\n", dib.dsOffset, testdata[i].name);
        }

        hr = IDXGISurface1_ReleaseDC(surface, NULL);
        ok(hr == S_OK, "Format %s: Got unexpected hr %#lx.\n", testdata[i].name, hr);

        IDXGISurface1_Release(surface);
        ID3D11Texture2D_Release(texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_shader_stage_input_output_matching(void)
{
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11Texture2D *render_target;
    ID3D11RenderTargetView *rtv[2];
    ID3D11DeviceContext *context;
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        struct output
        {
            float4 position : SV_PoSiTion;
            float4 color0 : COLOR0;
            float4 color1 : COLOR1;
        };

        void main(uint id : SV_VertexID, out output o)
        {
            float2 coords = float2((id << 1) & 2, id & 2);
            o.position = float4(coords * float2(2, -2) + float2(-1, 1), 0, 1);
            o.color0 = float4(1.0f, 0.0f, 0.0f, 1.0f);
            o.color1 = float4(0.0f, 1.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x93c216a1, 0xbaa7e8d4, 0xd5368c6a, 0x4e889e07, 0x00000001, 0x00000224, 0x00000003,
        0x0000002c, 0x00000060, 0x000000cc, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000006, 0x00000001, 0x00000000, 0x00000101, 0x565f5653, 0x65747265, 0x00444978,
        0x4e47534f, 0x00000064, 0x00000003, 0x00000008, 0x00000050, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x0000005c, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x0000005c, 0x00000001, 0x00000000, 0x00000003, 0x00000002, 0x0000000f, 0x505f5653, 0x5469536f,
        0x006e6f69, 0x4f4c4f43, 0xabab0052, 0x52444853, 0x00000150, 0x00010040, 0x00000054, 0x04000060,
        0x00101012, 0x00000000, 0x00000006, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x03000065,
        0x001020f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000002, 0x02000068, 0x00000001, 0x07000029,
        0x00100012, 0x00000000, 0x0010100a, 0x00000000, 0x00004001, 0x00000001, 0x07000001, 0x00100012,
        0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0x00000002, 0x07000001, 0x00100042, 0x00000000,
        0x0010100a, 0x00000000, 0x00004001, 0x00000002, 0x05000056, 0x00100032, 0x00000000, 0x00100086,
        0x00000000, 0x0f000032, 0x00102032, 0x00000000, 0x00100046, 0x00000000, 0x00004002, 0x40000000,
        0xc0000000, 0x00000000, 0x00000000, 0x00004002, 0xbf800000, 0x3f800000, 0x00000000, 0x00000000,
        0x08000036, 0x001020c2, 0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x3f800000,
        0x08000036, 0x001020f2, 0x00000001, 0x00004002, 0x3f800000, 0x00000000, 0x00000000, 0x3f800000,
        0x08000036, 0x001020f2, 0x00000002, 0x00004002, 0x00000000, 0x3f800000, 0x00000000, 0x3f800000,
        0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        struct input
        {
            float4 position : SV_PoSiTiOn;
            float4 color1 : COLOR1;
            float4 color0 : COLOR0;
        };

        struct output
        {
            float4 target0 : SV_Target0;
            float4 target1 : SV_Target1;
        };

        void main(const in input i, out output o)
        {
            o.target0 = i.color0;
            o.target1 = i.color1;
        }
#endif
        0x43425844, 0x620ef963, 0xed8f19fe, 0x7b3a0a53, 0x126ce021, 0x00000001, 0x00000150, 0x00000003,
        0x0000002c, 0x00000098, 0x000000e4, 0x4e475349, 0x00000064, 0x00000003, 0x00000008, 0x00000050,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x0000005c, 0x00000001, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x0000005c, 0x00000000, 0x00000000, 0x00000003, 0x00000002,
        0x00000f0f, 0x505f5653, 0x5469536f, 0x006e4f69, 0x4f4c4f43, 0xabab0052, 0x4e47534f, 0x00000044,
        0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f,
        0x00000038, 0x00000001, 0x00000000, 0x00000003, 0x00000001, 0x0000000f, 0x545f5653, 0x65677261,
        0xabab0074, 0x52444853, 0x00000064, 0x00000040, 0x00000019, 0x03001062, 0x001010f2, 0x00000001,
        0x03001062, 0x001010f2, 0x00000002, 0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x001020f2,
        0x00000001, 0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000002, 0x05000036, 0x001020f2,
        0x00000001, 0x00101e46, 0x00000001, 0x0100003e,
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &render_target);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    rtv[0] = test_context.backbuffer_rtv;
    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)render_target, NULL, &rtv[1]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST);
    ID3D11DeviceContext_OMSetRenderTargets(context, 2, rtv, NULL);
    ID3D11DeviceContext_Draw(context, 3, 0);

    check_texture_color(test_context.backbuffer, 0xff00ff00, 0);
    check_texture_color(render_target, 0xff0000ff, 0);

    ID3D11RenderTargetView_Release(rtv[1]);
    ID3D11Texture2D_Release(render_target);
    ID3D11PixelShader_Release(ps);
    ID3D11VertexShader_Release(vs);
    release_test_context(&test_context);
}

static void test_unbound_streams(void)
{
    struct d3d11_test_context test_context;
    ID3D11DeviceContext *context;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        struct vs_ps
        {
            float4 position : SV_POSITION;
            float4 color    : COLOR0;
        };

        vs_ps vs_main(float4 position : POSITION, float4 color : COLOR0)
        {
            vs_ps result;
            result.position = position;
            result.color = color;
            result.color.w = 1.0;
            return result;
        }
#endif
        0x43425844, 0x4a9efaec, 0xe2c6cdf5, 0x15dd28a7, 0xae68e320, 0x00000001, 0x00000154, 0x00000003,
        0x0000002c, 0x0000007c, 0x000000d0, 0x4e475349, 0x00000048, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000041, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x0000070f, 0x49534f50, 0x4e4f4954, 0x4c4f4300, 0xab00524f, 0x4e47534f,
        0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000,
        0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f, 0x505f5653,
        0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052, 0x52444853, 0x0000007c, 0x00010040, 0x0000001f,
        0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x00101072, 0x00000001, 0x04000067, 0x001020f2,
        0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x05000036, 0x001020f2, 0x00000000,
        0x00101e46, 0x00000000, 0x05000036, 0x00102072, 0x00000001, 0x00101246, 0x00000001, 0x05000036,
        0x00102082, 0x00000001, 0x00004001, 0x3f800000, 0x0100003e,
    };

    static const DWORD ps_code[] =
    {
#if 0
        float4 ps_main(vs_ps input) : SV_TARGET
        {
            return input.color;
        }
#endif
        0x43425844, 0xe2087fa6, 0xa35fbd95, 0x8e585b3f, 0x67890f54, 0x00000001, 0x000000f4, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000038, 0x00000040,
        0x0000000e, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x05000036,
        0x001020f2, 0x00000000, 0x00101e46, 0x00000001, 0x0100003e,
    };

    static const float white[4] = {1.0f, 1.0f, 1.0f, 1.0f};

    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT,    0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"COLOR",    0, DXGI_FORMAT_R32G32B32A32_FLOAT, 1, 16, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &test_context.input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);
    draw_quad_vs(&test_context, vs_code, sizeof(vs_code));
    check_texture_color(test_context.backbuffer, 0xff000000, 1);

    ID3D11PixelShader_Release(ps);
    release_test_context(&test_context);
}

static void test_shader_interstage_interface(void)
{
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11InputLayout *input_layout;
    ID3D11Texture2D *render_target;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    UINT stride, offset;
    ID3D11Buffer *vb;
    unsigned int i;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        struct vertex
        {
            float4 position : SV_Position;
            float2 t0 : TEXCOORD0;
            nointerpolation float t1 : TEXCOORD1;
            uint t2 : TEXCOORD2;
            uint t3 : TEXCOORD3;
            float t4 : TEXCOORD4;
        };

        void main(in vertex vin, out vertex vout)
        {
            vout = vin;
        }
#endif
        0x43425844, 0xd55780bf, 0x76866b06, 0x45d697a2, 0xafac2ecd, 0x00000001, 0x000002bc, 0x00000003,
        0x0000002c, 0x000000e4, 0x0000019c, 0x4e475349, 0x000000b0, 0x00000006, 0x00000008, 0x00000098,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x000000a4, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000303, 0x000000a4, 0x00000001, 0x00000000, 0x00000003, 0x00000002,
        0x00000101, 0x000000a4, 0x00000002, 0x00000000, 0x00000001, 0x00000003, 0x00000101, 0x000000a4,
        0x00000003, 0x00000000, 0x00000001, 0x00000004, 0x00000101, 0x000000a4, 0x00000004, 0x00000000,
        0x00000003, 0x00000005, 0x00000101, 0x505f5653, 0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f,
        0xababab00, 0x4e47534f, 0x000000b0, 0x00000006, 0x00000008, 0x00000098, 0x00000000, 0x00000001,
        0x00000003, 0x00000000, 0x0000000f, 0x000000a4, 0x00000000, 0x00000000, 0x00000003, 0x00000001,
        0x00000c03, 0x000000a4, 0x00000004, 0x00000000, 0x00000003, 0x00000001, 0x00000b04, 0x000000a4,
        0x00000001, 0x00000000, 0x00000003, 0x00000002, 0x00000e01, 0x000000a4, 0x00000002, 0x00000000,
        0x00000001, 0x00000002, 0x00000d02, 0x000000a4, 0x00000003, 0x00000000, 0x00000001, 0x00000002,
        0x00000b04, 0x505f5653, 0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f, 0xababab00, 0x52444853,
        0x00000118, 0x00010040, 0x00000046, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x00101032,
        0x00000001, 0x0300005f, 0x00101012, 0x00000002, 0x0300005f, 0x00101012, 0x00000003, 0x0300005f,
        0x00101012, 0x00000004, 0x0300005f, 0x00101012, 0x00000005, 0x04000067, 0x001020f2, 0x00000000,
        0x00000001, 0x03000065, 0x00102032, 0x00000001, 0x03000065, 0x00102042, 0x00000001, 0x03000065,
        0x00102012, 0x00000002, 0x03000065, 0x00102022, 0x00000002, 0x03000065, 0x00102042, 0x00000002,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x05000036, 0x00102032, 0x00000001,
        0x00101046, 0x00000001, 0x05000036, 0x00102042, 0x00000001, 0x0010100a, 0x00000005, 0x05000036,
        0x00102012, 0x00000002, 0x0010100a, 0x00000002, 0x05000036, 0x00102022, 0x00000002, 0x0010100a,
        0x00000003, 0x05000036, 0x00102042, 0x00000002, 0x0010100a, 0x00000004, 0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        void main(float4 position : SV_Position, float2 t0 : TEXCOORD0,
                nointerpolation float t1 : TEXCOORD1, uint t2 : TEXCOORD2,
                uint t3 : TEXCOORD3, float t4 : TEXCOORD4, out float4 o : SV_Target)
        {
            o.x = t0.y + t1;
            o.y = t2 + t3;
            o.z = t4;
            o.w = t0.x;
        }
#endif
        0x43425844, 0x8a7ef706, 0xc8f2cbf1, 0x83a05df1, 0xfab8e613, 0x00000001, 0x000001dc, 0x00000003,
        0x0000002c, 0x000000e4, 0x00000118, 0x4e475349, 0x000000b0, 0x00000006, 0x00000008, 0x00000098,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x000000a4, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000303, 0x000000a4, 0x00000004, 0x00000000, 0x00000003, 0x00000001,
        0x00000404, 0x000000a4, 0x00000001, 0x00000000, 0x00000003, 0x00000002, 0x00000101, 0x000000a4,
        0x00000002, 0x00000000, 0x00000001, 0x00000002, 0x00000202, 0x000000a4, 0x00000003, 0x00000000,
        0x00000001, 0x00000002, 0x00000404, 0x505f5653, 0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f,
        0xababab00, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x000000bc,
        0x00000040, 0x0000002f, 0x03001062, 0x00101032, 0x00000001, 0x03001062, 0x00101042, 0x00000001,
        0x03000862, 0x00101012, 0x00000002, 0x03000862, 0x00101022, 0x00000002, 0x03000862, 0x00101042,
        0x00000002, 0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0700001e, 0x00100012,
        0x00000000, 0x0010101a, 0x00000002, 0x0010102a, 0x00000002, 0x05000056, 0x00102022, 0x00000000,
        0x0010000a, 0x00000000, 0x07000000, 0x00102012, 0x00000000, 0x0010101a, 0x00000001, 0x0010100a,
        0x00000002, 0x05000036, 0x001020c2, 0x00000000, 0x001012a6, 0x00000001, 0x0100003e,
    };
    static const DWORD ps_partial_input_code[] =
    {
#if 0
        void main(float4 position : SV_Position, float2 t0 : TEXCOORD0,
                nointerpolation float t1 : TEXCOORD1, uint t2 : TEXCOORD2,
                uint t3 : TEXCOORD3, out float4 o : SV_Target)
        {
            o.x = t0.y + t1;
            o.y = t2 + t3;
            o.z = 0.0f;
            o.w = t0.x;
        }
#endif
        0x43425844, 0x5b1db356, 0xaa5a5e9d, 0xb916a081, 0x61e6dcb1, 0x00000001, 0x000001cc, 0x00000003,
        0x0000002c, 0x000000cc, 0x00000100, 0x4e475349, 0x00000098, 0x00000005, 0x00000008, 0x00000080,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x0000008c, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000303, 0x0000008c, 0x00000001, 0x00000000, 0x00000003, 0x00000002,
        0x00000101, 0x0000008c, 0x00000002, 0x00000000, 0x00000001, 0x00000002, 0x00000202, 0x0000008c,
        0x00000003, 0x00000000, 0x00000001, 0x00000002, 0x00000404, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x43584554, 0x44524f4f, 0xababab00, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074,
        0x52444853, 0x000000c4, 0x00000040, 0x00000031, 0x03001062, 0x00101032, 0x00000001, 0x03000862,
        0x00101012, 0x00000002, 0x03000862, 0x00101022, 0x00000002, 0x03000862, 0x00101042, 0x00000002,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x0700001e, 0x00100012, 0x00000000,
        0x0010102a, 0x00000002, 0x0010101a, 0x00000002, 0x05000056, 0x00102022, 0x00000000, 0x0010000a,
        0x00000000, 0x07000000, 0x00102012, 0x00000000, 0x0010101a, 0x00000001, 0x0010100a, 0x00000002,
        0x05000036, 0x00102042, 0x00000000, 0x00004001, 0x00000000, 0x05000036, 0x00102082, 0x00000000,
        0x0010100a, 0x00000001, 0x0100003e,
    };
    static const DWORD ps_single_input_code[] =
    {
#if 0
        void main(float4 position : SV_Position, float2 t0 : TEXCOORD0, out float4 o : SV_Target)
        {
            o.x = t0.x;
            o.y = t0.y;
            o.z = 1.0f;
            o.w = 2.0f;
        }
#endif
        0x43425844, 0x7cc601b6, 0xc65b8bdb, 0x54d0f606, 0x9cc74d3d, 0x00000001, 0x00000118, 0x00000003,
        0x0000002c, 0x00000084, 0x000000b8, 0x4e475349, 0x00000050, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000303, 0x505f5653, 0x7469736f, 0x006e6f69, 0x43584554, 0x44524f4f,
        0xababab00, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000058,
        0x00000040, 0x00000016, 0x03001062, 0x00101032, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x05000036, 0x00102032, 0x00000000, 0x00101046, 0x00000001, 0x08000036, 0x001020c2, 0x00000000,
        0x00004002, 0x00000000, 0x00000000, 0x3f800000, 0x40000000, 0x0100003e,
    };
    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"SV_POSITION", 0, DXGI_FORMAT_R32G32_FLOAT, 0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"TEXCOORD",    0, DXGI_FORMAT_R32G32_FLOAT, 0,  8, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"TEXCOORD",    1, DXGI_FORMAT_R32_FLOAT,    0, 16, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"TEXCOORD",    2, DXGI_FORMAT_R32_UINT,     0, 20, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"TEXCOORD",    3, DXGI_FORMAT_R32_UINT,     0, 24, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"TEXCOORD",    4, DXGI_FORMAT_R32_FLOAT,    0, 28, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    static const struct
    {
        struct vec2 position;
        struct vec2 t0;
        float t1;
        unsigned int t2;
        unsigned int t3;
        float t4;
    }
    quad[] =
    {
        {{-1.0f, -1.0f}, {3.0f, 5.0f}, 5.0f, 2, 6, 7.0f},
        {{-1.0f,  1.0f}, {3.0f, 5.0f}, 5.0f, 2, 6, 7.0f},
        {{ 1.0f, -1.0f}, {3.0f, 5.0f}, 5.0f, 2, 6, 7.0f},
        {{ 1.0f,  1.0f}, {3.0f, 5.0f}, 5.0f, 2, 6, 7.0f},
    };
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    static const struct
    {
        const DWORD *ps_code;
        size_t ps_size;
        struct vec4 expected_result;
    }
    tests[] =
    {
        {ps_code, sizeof(ps_code), {10.0f, 8.0f, 7.0f, 3.0f}},
        {ps_partial_input_code, sizeof(ps_partial_input_code), {10.0f, 8.0f, 0.0f, 3.0f}},
        {ps_single_input_code, sizeof(ps_single_input_code), {3.0f, 5.0f, 1.0f, 2.0f}},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &render_target);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)render_target, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(quad), quad);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv, white);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    offset = 0;
    stride = sizeof(*quad);
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb, &stride, &offset);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        hr = ID3D11Device_CreatePixelShader(device, tests[i].ps_code, tests[i].ps_size, NULL, &ps);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
        ID3D11DeviceContext_Draw(context, 4, 0);
        check_texture_vec4(render_target, &tests[i].expected_result, 0);
        ID3D11PixelShader_Release(ps);
    }

    ID3D11InputLayout_Release(input_layout);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11Texture2D_Release(render_target);
    ID3D11VertexShader_Release(vs);
    ID3D11Buffer_Release(vb);
    release_test_context(&test_context);
}

static void test_sm4_if_instruction(void)
{
    struct d3d11_test_context test_context;
    ID3D11PixelShader *ps_if_nz, *ps_if_z;
    ID3D11DeviceContext *context;
    ID3D11Device *device;
    unsigned int bits[4];
    DWORD expected_color;
    ID3D11Buffer *cb;
    unsigned int i;
    HRESULT hr;

    static const DWORD ps_if_nz_code[] =
    {
#if 0
        uint bits;

        float4 main() : SV_TARGET
        {
            if (bits)
                return float4(0.0f, 1.0f, 0.0f, 1.0f);
            else
                return float4(1.0f, 0.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x2a94f6f1, 0xdbe88943, 0x3426a708, 0x09cec990, 0x00000001, 0x00000100, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000088, 0x00000040, 0x00000022,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x0404001f,
        0x0020800a, 0x00000000, 0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x00000000,
        0x3f800000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000012, 0x08000036, 0x001020f2, 0x00000000,
        0x00004002, 0x3f800000, 0x00000000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000015, 0x0100003e,
    };
    static const DWORD ps_if_z_code[] =
    {
#if 0
        uint bits;

        float4 main() : SV_TARGET
        {
            if (!bits)
                return float4(0.0f, 1.0f, 0.0f, 1.0f);
            else
                return float4(1.0f, 0.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x2e3030ca, 0x94c8610c, 0xdf0c1b1f, 0x80f2ca2c, 0x00000001, 0x00000100, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000088, 0x00000040, 0x00000022,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x0400001f,
        0x0020800a, 0x00000000, 0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x00000000,
        0x3f800000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000012, 0x08000036, 0x001020f2, 0x00000000,
        0x00004002, 0x3f800000, 0x00000000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000015, 0x0100003e,
    };
    static unsigned int bit_patterns[] =
    {
        0x00000000, 0x00000001, 0x10010001, 0x10000000, 0x80000000, 0xffff0000, 0x0000ffff, 0xffffffff,
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreatePixelShader(device, ps_if_nz_code, sizeof(ps_if_nz_code), NULL, &ps_if_nz);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_if_z_code, sizeof(ps_if_z_code), NULL, &ps_if_z);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(bits), NULL);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    for (i = 0; i < ARRAY_SIZE(bit_patterns); ++i)
    {
        *bits = bit_patterns[i];
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, bits, 0, 0);

        ID3D11DeviceContext_PSSetShader(context, ps_if_nz, NULL, 0);
        expected_color = *bits ? 0xff00ff00 : 0xff0000ff;
        draw_quad(&test_context);
        check_texture_color(test_context.backbuffer, expected_color, 0);

        ID3D11DeviceContext_PSSetShader(context, ps_if_z, NULL, 0);
        expected_color = *bits ? 0xff0000ff : 0xff00ff00;
        draw_quad(&test_context);
        check_texture_color(test_context.backbuffer, expected_color, 0);
    }

    ID3D11Buffer_Release(cb);
    ID3D11PixelShader_Release(ps_if_z);
    ID3D11PixelShader_Release(ps_if_nz);
    release_test_context(&test_context);
}

static void test_sm4_breakc_instruction(void)
{
    struct d3d11_test_context test_context;
    ID3D11DeviceContext *context;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DWORD ps_breakc_nz_code[] =
    {
#if 0
        float4 main() : SV_TARGET
        {
            uint counter = 0;

            for (uint i = 0; i < 255; ++i)
                ++counter;

            if (counter == 255)
                return float4(0.0f, 1.0f, 0.0f, 1.0f);
            else
                return float4(1.0f, 0.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x065ac80a, 0x24369e7e, 0x218d5dc1, 0x3532868c, 0x00000001, 0x00000188, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000110, 0x00000040, 0x00000044,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000001, 0x08000036, 0x00100032, 0x00000000,
        0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000030, 0x07000050, 0x00100042,
        0x00000000, 0x0010001a, 0x00000000, 0x00004001, 0x000000ff, 0x03040003, 0x0010002a, 0x00000000,
        0x0a00001e, 0x00100032, 0x00000000, 0x00100046, 0x00000000, 0x00004002, 0x00000001, 0x00000001,
        0x00000000, 0x00000000, 0x01000016, 0x07000020, 0x00100012, 0x00000000, 0x0010000a, 0x00000000,
        0x00004001, 0x000000ff, 0x0304001f, 0x0010000a, 0x00000000, 0x08000036, 0x001020f2, 0x00000000,
        0x00004002, 0x00000000, 0x3f800000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000012, 0x08000036,
        0x001020f2, 0x00000000, 0x00004002, 0x3f800000, 0x00000000, 0x00000000, 0x3f800000, 0x0100003e,
        0x01000015, 0x0100003e,
    };
    static const DWORD ps_breakc_z_code[] =
    {
#if 0
        float4 main() : SV_TARGET
        {
            uint counter = 0;

            for (int i = 0, j = 254; i < 255 && j >= 0; ++i, --j)
                ++counter;

            if (counter == 255)
                return float4(0.0f, 1.0f, 0.0f, 1.0f);
            else
                return float4(1.0f, 0.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x687406ef, 0x7bdeb7d1, 0xb3282292, 0x934a9101, 0x00000001, 0x000001c0, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000148, 0x00000040, 0x00000052,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000002, 0x08000036, 0x00100072, 0x00000000,
        0x00004002, 0x00000000, 0x00000000, 0x000000fe, 0x00000000, 0x01000030, 0x07000022, 0x00100082,
        0x00000000, 0x0010001a, 0x00000000, 0x00004001, 0x000000ff, 0x07000021, 0x00100012, 0x00000001,
        0x0010002a, 0x00000000, 0x00004001, 0x00000000, 0x07000001, 0x00100082, 0x00000000, 0x0010003a,
        0x00000000, 0x0010000a, 0x00000001, 0x03000003, 0x0010003a, 0x00000000, 0x0a00001e, 0x00100072,
        0x00000000, 0x00100246, 0x00000000, 0x00004002, 0x00000001, 0x00000001, 0xffffffff, 0x00000000,
        0x01000016, 0x07000020, 0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0x000000ff,
        0x0304001f, 0x0010000a, 0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x00000000,
        0x3f800000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000012, 0x08000036, 0x001020f2, 0x00000000,
        0x00004002, 0x3f800000, 0x00000000, 0x00000000, 0x3f800000, 0x0100003e, 0x01000015, 0x0100003e,
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreatePixelShader(device, ps_breakc_nz_code, sizeof(ps_breakc_nz_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 0);
    ID3D11PixelShader_Release(ps);

    hr = ID3D11Device_CreatePixelShader(device, ps_breakc_z_code, sizeof(ps_breakc_z_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 0);
    ID3D11PixelShader_Release(ps);

    release_test_context(&test_context);
}

static void test_sm4_continuec_instruction(void)
{
    struct d3d11_test_context test_context;
    ID3D11DeviceContext *context;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    /* To get fxc to output continuec_z/continuec_nz instead of an if-block
     * with a normal continue inside, the shaders have been compiled with
     * the /Gfa flag. */
    static const DWORD ps_continuec_nz_code[] =
    {
#if 0
        float4 main() : SV_TARGET
        {
            uint counter = 0;
            int i = -1;

            while (i < 255) {
                ++i;

                if (i != 0)
                    continue;

                ++counter;
            }

            if (counter == 1)
                return float4(0.0f, 1.0f, 0.0f, 1.0f);
            else
                return float4(1.0f, 0.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0xaadaac96, 0xbe00fdfb, 0x29356be0, 0x47e79bd6, 0x00000001, 0x00000208, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000190, 0x00000040, 0x00000064,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000003, 0x08000036, 0x00100032, 0x00000000,
        0x00004002, 0x00000000, 0xffffffff, 0x00000000, 0x00000000, 0x01000030, 0x07000021, 0x00100042,
        0x00000000, 0x0010001a, 0x00000000, 0x00004001, 0x000000ff, 0x03040003, 0x0010002a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000001, 0x0010001a, 0x00000000, 0x00004001, 0x00000001, 0x09000037,
        0x00100022, 0x00000002, 0x0010001a, 0x00000001, 0x0010001a, 0x00000001, 0x00004001, 0x00000000,
        0x05000036, 0x00100012, 0x00000002, 0x0010000a, 0x00000000, 0x05000036, 0x00100032, 0x00000000,
        0x00100046, 0x00000002, 0x05000036, 0x00100042, 0x00000000, 0x0010001a, 0x00000001, 0x03040008,
        0x0010002a, 0x00000000, 0x0700001e, 0x00100012, 0x00000001, 0x0010000a, 0x00000000, 0x00004001,
        0x00000001, 0x05000036, 0x00100032, 0x00000000, 0x00100046, 0x00000001, 0x01000016, 0x07000020,
        0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0x00000001, 0x08000036, 0x001020f2,
        0x00000000, 0x00004002, 0x00000000, 0x3f800000, 0x00000000, 0x3f800000, 0x0304003f, 0x0010000a,
        0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x3f800000, 0x00000000, 0x00000000,
        0x3f800000, 0x0100003e,

    };
    static const DWORD ps_continuec_z_code[] =
    {
#if 0
        float4 main() : SV_TARGET
        {
            uint counter = 0;
            int i = -1;

            while (i < 255) {
                ++i;

                if (i == 0)
                    continue;

                ++counter;
            }

            if (counter == 255)
                return float4(0.0f, 1.0f, 0.0f, 1.0f);
            else
                return float4(1.0f, 0.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x0322b23d, 0x52b25dc8, 0xa625f5f1, 0x271e3f46, 0x00000001, 0x000001d0, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x45475241, 0xabab0054, 0x52444853, 0x00000158, 0x00000040, 0x00000056,
        0x03000065, 0x001020f2, 0x00000000, 0x02000068, 0x00000002, 0x08000036, 0x00100032, 0x00000000,
        0x00004002, 0x00000000, 0xffffffff, 0x00000000, 0x00000000, 0x01000030, 0x07000021, 0x00100042,
        0x00000000, 0x0010001a, 0x00000000, 0x00004001, 0x000000ff, 0x03040003, 0x0010002a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000001, 0x0010001a, 0x00000000, 0x00004001, 0x00000001, 0x05000036,
        0x00100042, 0x00000001, 0x0010000a, 0x00000000, 0x05000036, 0x00100072, 0x00000000, 0x00100966,
        0x00000001, 0x03000008, 0x0010002a, 0x00000000, 0x0700001e, 0x00100012, 0x00000001, 0x0010000a,
        0x00000000, 0x00004001, 0x00000001, 0x05000036, 0x00100032, 0x00000000, 0x00100046, 0x00000001,
        0x01000016, 0x07000020, 0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0x000000ff,
        0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x00000000, 0x3f800000, 0x00000000, 0x3f800000,
        0x0304003f, 0x0010000a, 0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x3f800000,
        0x00000000, 0x00000000, 0x3f800000, 0x0100003e,
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreatePixelShader(device, ps_continuec_nz_code, sizeof(ps_continuec_nz_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 0);
    ID3D11PixelShader_Release(ps);

    hr = ID3D11Device_CreatePixelShader(device, ps_continuec_z_code, sizeof(ps_continuec_z_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 0);
    ID3D11PixelShader_Release(ps);

    release_test_context(&test_context);
}

static void test_sm4_discard_instruction(void)
{
    ID3D11PixelShader *ps_discard_nz, *ps_discard_z;
    struct d3d11_test_context test_context;
    ID3D11DeviceContext *context;
    ID3D11Device *device;
    ID3D11Buffer *cb;
    unsigned int i;
    HRESULT hr;

    static const DWORD ps_discard_nz_code[] =
    {
#if 0
        uint data;

        float4 main() : SV_Target
        {
            if (data)
                discard;
            return float4(0.0f, 0.5f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0xfa7e5758, 0xd8716ffc, 0x5ad6a940, 0x2b99bba2, 0x00000001, 0x000000d0, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000058, 0x00000040, 0x00000016,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x0404000d,
        0x0020800a, 0x00000000, 0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x00000000,
        0x3f000000, 0x00000000, 0x3f800000, 0x0100003e,
    };
    static const DWORD ps_discard_z_code[] =
    {
#if 0
        uint data;

        float4 main() : SV_Target
        {
            if (!data)
                discard;
            return float4(0.0f, 1.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x5c4dd108, 0x1eb43558, 0x7c02c98c, 0xd81eb34c, 0x00000001, 0x000000d0, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000058, 0x00000040, 0x00000016,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x0400000d,
        0x0020800a, 0x00000000, 0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x00000000,
        0x3f800000, 0x00000000, 0x3f800000, 0x0100003e,
    };
    static const float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
    static const struct uvec4 values[] =
    {
        {0x0000000},
        {0x0000001},
        {0x8000000},
        {0xfffffff},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(*values), NULL);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    hr = ID3D11Device_CreatePixelShader(device, ps_discard_nz_code, sizeof(ps_discard_nz_code),
            NULL, &ps_discard_nz);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_discard_z_code, sizeof(ps_discard_z_code),
            NULL, &ps_discard_z);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(values); ++i)
    {
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &values[i], 0, 0);

        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);
        ID3D11DeviceContext_PSSetShader(context, ps_discard_nz, NULL, 0);
        draw_quad(&test_context);
        check_texture_color(test_context.backbuffer, values[i].x ? 0xffffffff : 0xff007f00, 1);

        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, white);
        ID3D11DeviceContext_PSSetShader(context, ps_discard_z, NULL, 0);
        draw_quad(&test_context);
        check_texture_color(test_context.backbuffer, values[i].x ? 0xff00ff00 : 0xffffffff, 1);
    }

    ID3D11Buffer_Release(cb);
    ID3D11PixelShader_Release(ps_discard_nz);
    ID3D11PixelShader_Release(ps_discard_z);
    release_test_context(&test_context);
}

static void test_sm5_swapc_instruction(void)
{
    struct input
    {
        struct uvec4 src0;
        struct uvec4 src1;
        struct uvec4 src2;
    };

    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps[6];
    ID3D11Device *device;
    ID3D11Buffer *cb;
    unsigned int i;
    HRESULT hr;

    static const D3D_FEATURE_LEVEL feature_level = D3D_FEATURE_LEVEL_11_0;
    static const DWORD ps_swapc0_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[3], immediateIndexed
        dcl_output o0.xyzw
        dcl_temps 2
        swapc r0.xyzw, r1.xyzw, cb0[0].xyzw, cb0[1].xyzw, cb0[2].xyzw
        mov o0.xyzw, r0.xyzw
        ret
#endif
        0x43425844, 0x9e089246, 0x9f8b5cbe, 0xbac66faf, 0xaef23488, 0x00000001, 0x000000f8, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000080, 0x00000050, 0x00000020,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000003, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000002, 0x0e00008e, 0x001000f2, 0x00000000, 0x001000f2, 0x00000001, 0x00208e46,
        0x00000000, 0x00000000, 0x00208e46, 0x00000000, 0x00000001, 0x00208e46, 0x00000000, 0x00000002,
        0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_swapc1_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[3], immediateIndexed
        dcl_output o0.xyzw
        dcl_temps 2
        swapc r0.xyzw, r1.xyzw, cb0[0].xyzw, cb0[1].xyzw, cb0[2].xyzw
        mov o0.xyzw, r1.xyzw
        ret
#endif
        0x43425844, 0xf2daed61, 0xede211f7, 0x7300dbea, 0x573ed49f, 0x00000001, 0x000000f8, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000080, 0x00000050, 0x00000020,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000003, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000002, 0x0e00008e, 0x001000f2, 0x00000000, 0x001000f2, 0x00000001, 0x00208e46,
        0x00000000, 0x00000000, 0x00208e46, 0x00000000, 0x00000001, 0x00208e46, 0x00000000, 0x00000002,
        0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000001, 0x0100003e,
    };
    static const DWORD ps_swapc2_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[3], immediateIndexed
        dcl_output o0.xyzw
        dcl_temps 2
        mov r0.xyzw, cb0[1].xyzw
        mov r1.xyzw, cb0[2].xyzw
        swapc r0.xyzw, r1.xyzw, cb0[0].xyzw, r0.xyzw, r1.xyzw
        mov o0.xyzw, r0.xyzw
        ret
#endif
        0x43425844, 0x230fcb22, 0x70d99148, 0x65814d89, 0x97473498, 0x00000001, 0x00000120, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x000000a8, 0x00000050, 0x0000002a,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000003, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000002, 0x06000036, 0x001000f2, 0x00000000, 0x00208e46, 0x00000000, 0x00000001,
        0x06000036, 0x001000f2, 0x00000001, 0x00208e46, 0x00000000, 0x00000002, 0x0c00008e, 0x001000f2,
        0x00000000, 0x001000f2, 0x00000001, 0x00208e46, 0x00000000, 0x00000000, 0x00100e46, 0x00000000,
        0x00100e46, 0x00000001, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_swapc3_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[3], immediateIndexed
        dcl_output o0.xyzw
        dcl_temps 2
        mov r0.xyzw, cb0[1].xyzw
        mov r1.xyzw, cb0[2].xyzw
        swapc r0.xyzw, r1.xyzw, cb0[0].xyzw, r0.xyzw, r1.xyzw
        mov o0.xyzw, r1.xyzw
        ret
#endif
        0x43425844, 0xce595d62, 0x98305541, 0xb04e74c8, 0xfc010f3a, 0x00000001, 0x00000120, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x000000a8, 0x00000050, 0x0000002a,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000003, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000002, 0x06000036, 0x001000f2, 0x00000000, 0x00208e46, 0x00000000, 0x00000001,
        0x06000036, 0x001000f2, 0x00000001, 0x00208e46, 0x00000000, 0x00000002, 0x0c00008e, 0x001000f2,
        0x00000000, 0x001000f2, 0x00000001, 0x00208e46, 0x00000000, 0x00000000, 0x00100e46, 0x00000000,
        0x00100e46, 0x00000001, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000001, 0x0100003e,
    };
    static const DWORD ps_swapc4_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[3], immediateIndexed
        dcl_output o0.xyzw
        dcl_temps 2
        mov r0.xyzw, cb0[0].xyzw
        swapc r0.xyzw, r1.xyzw, r0.xyzw, cb0[1].xyzw, cb0[2].xyzw
        mov o0.xyzw, r0.xyzw
        ret
#endif
        0x43425844, 0x72067c48, 0xb53572a0, 0x9dd9e0fd, 0x903e37e3, 0x00000001, 0x0000010c, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000094, 0x00000050, 0x00000025,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000003, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000002, 0x06000036, 0x001000f2, 0x00000000, 0x00208e46, 0x00000000, 0x00000000,
        0x0d00008e, 0x001000f2, 0x00000000, 0x001000f2, 0x00000001, 0x00100e46, 0x00000000, 0x00208e46,
        0x00000000, 0x00000001, 0x00208e46, 0x00000000, 0x00000002, 0x05000036, 0x001020f2, 0x00000000,
        0x00100e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_swapc5_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[3], immediateIndexed
        dcl_output o0.xyzw
        dcl_temps 2
        mov r1.xyzw, cb0[0].xyzw
        swapc r0.xyzw, r1.xyzw, r1.xyzw, cb0[1].xyzw, cb0[2].xyzw
        mov o0.xyzw, r1.xyzw
        ret
#endif
        0x43425844, 0x7078fb08, 0xdd24cd44, 0x469d3258, 0x9e33a0bc, 0x00000001, 0x0000010c, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000094, 0x00000050, 0x00000025,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000003, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000002, 0x06000036, 0x001000f2, 0x00000001, 0x00208e46, 0x00000000, 0x00000000,
        0x0d00008e, 0x001000f2, 0x00000000, 0x001000f2, 0x00000001, 0x00100e46, 0x00000001, 0x00208e46,
        0x00000000, 0x00000001, 0x00208e46, 0x00000000, 0x00000002, 0x05000036, 0x001020f2, 0x00000000,
        0x00100e46, 0x00000001, 0x0100003e,
    };
    static const struct
    {
        struct input input;
        struct uvec4 dst0;
        struct uvec4 dst1;
    }
    tests[] =
    {
        {
            {{0, 0, 0, 0}, {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}
        },
        {
            {{1, 1, 1, 1}, {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}, {0xdead, 0xc0de, 0xffff, 0xeeee},
        },
        {
            {{0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff},
                    {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}, {0xdead, 0xc0de, 0xffff, 0xeeee},
        },
        {
            {{1, 0, 1, 0}, {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xaaaa, 0xc0de, 0xcccc, 0xeeee}, {0xdead, 0xbbbb, 0xffff, 0xdddd},
        },
        {
            {{1, 0, 0, 1}, {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xaaaa, 0xc0de, 0xffff, 0xdddd}, {0xdead, 0xbbbb, 0xcccc, 0xeeee},
        },
        {
            {{1, 0, 0, 0}, {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xaaaa, 0xc0de, 0xffff, 0xeeee}, {0xdead, 0xbbbb, 0xcccc, 0xdddd}
        },
        {
            {{0, 1, 0, 0}, {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xdead, 0xbbbb, 0xffff, 0xeeee}, {0xaaaa, 0xc0de, 0xcccc, 0xdddd}
        },
        {
            {{0, 0, 1, 0}, {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xdead, 0xc0de, 0xcccc, 0xeeee}, {0xaaaa, 0xbbbb, 0xffff, 0xdddd}
        },
        {
            {{0, 0, 0, 1}, {0xdead, 0xc0de, 0xffff, 0xeeee}, {0xaaaa, 0xbbbb, 0xcccc, 0xdddd}},
            {0xdead, 0xc0de, 0xffff, 0xdddd}, {0xaaaa, 0xbbbb, 0xcccc, 0xeeee},
        },
    };

    if (!init_test_context(&test_context, &feature_level))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_R32G32B32A32_UINT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(struct input), NULL);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    hr = ID3D11Device_CreatePixelShader(device, ps_swapc0_code, sizeof(ps_swapc0_code), NULL, &ps[0]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_swapc1_code, sizeof(ps_swapc1_code), NULL, &ps[1]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_swapc2_code, sizeof(ps_swapc2_code), NULL, &ps[2]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_swapc3_code, sizeof(ps_swapc3_code), NULL, &ps[3]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_swapc4_code, sizeof(ps_swapc4_code), NULL, &ps[4]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreatePixelShader(device, ps_swapc5_code, sizeof(ps_swapc5_code), NULL, &ps[5]);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, &tests[i].input, 0, 0);

        ID3D11DeviceContext_PSSetShader(context, ps[0], NULL, 0);
        draw_quad(&test_context);
        check_texture_uvec4(texture, &tests[i].dst0);

        ID3D11DeviceContext_PSSetShader(context, ps[1], NULL, 0);
        draw_quad(&test_context);
        check_texture_uvec4(texture, &tests[i].dst1);

        ID3D11DeviceContext_PSSetShader(context, ps[2], NULL, 0);
        draw_quad(&test_context);
        check_texture_uvec4(texture, &tests[i].dst0);

        ID3D11DeviceContext_PSSetShader(context, ps[3], NULL, 0);
        draw_quad(&test_context);
        check_texture_uvec4(texture, &tests[i].dst1);

        ID3D11DeviceContext_PSSetShader(context, ps[4], NULL, 0);
        draw_quad(&test_context);
        check_texture_uvec4(texture, &tests[i].dst0);

        ID3D11DeviceContext_PSSetShader(context, ps[5], NULL, 0);
        draw_quad(&test_context);
        check_texture_uvec4(texture, &tests[i].dst1);
    }

    for (i = 0; i < ARRAY_SIZE(ps); ++i)
        ID3D11PixelShader_Release(ps[i]);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11Texture2D_Release(texture);
    ID3D11Buffer_Release(cb);
    release_test_context(&test_context);
}

static BOOL is_vs_sysval_semantic(const char *name)
{
    return !strcmp(name, "SV_InstanceID") || !strcmp(name, "SV_VertexID");
}

static void check_layout_element_exclusion(ID3D11Device *device, const D3D11_INPUT_ELEMENT_DESC *elements,
        unsigned int element_count, const DWORD *shader_code, unsigned int shader_code_size, BOOL is_cs,
        const char *test_context)
{
    D3D11_INPUT_ELEMENT_DESC elements_temp[32];
    ID3D11InputLayout *layout;
    unsigned int i;
    HRESULT hr;

    winetest_push_context(test_context);

    hr = ID3D11Device_CreateInputLayout(device, elements, element_count, shader_code, shader_code_size, &layout);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);
    ID3D11InputLayout_Release(layout);

    for (i = 0; i < element_count; ++i)
    {
        winetest_push_context("%s excluded", elements[i].SemanticName);
        memcpy(elements_temp, elements, sizeof(*elements_temp) * i);
        memcpy(elements_temp + i, elements + i + 1, sizeof(*elements_temp) * (element_count - i - 1));
        hr = ID3D11Device_CreateInputLayout(device, elements_temp, element_count - 1,
                shader_code, shader_code_size, &layout);
        if (is_cs || is_vs_sysval_semantic(elements[i].SemanticName))
        {
            /* sysval semantic. */
            ok(hr == S_OK, "Got hr %#lx.\n", hr);
            ID3D11InputLayout_Release(layout);
        }
        else
        {
            ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
        }
        winetest_pop_context();
    }

    winetest_pop_context();
}

static void test_create_input_layout(void)
{
    D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"POSITION", 0, DXGI_FORMAT_UNKNOWN, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"COLOR", 0, DXGI_FORMAT_B8G8R8A8_UNORM, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };

    ULONG refcount, expected_refcount;
    ID3D11InputLayout *input_layout;
    ID3D11Device *device;
    unsigned int i;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        float4 main(float4 position : POSITION) : SV_POSITION
        {
            return position;
        }
#endif
        0x43425844, 0xa7a2f22d, 0x83ff2560, 0xe61638bd, 0x87e3ce90, 0x00000001, 0x000000d8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x0000003c, 0x00010040,
        0x0000000f, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };

    static D3D11_INPUT_ELEMENT_DESC vs2_layout_desc[] =
    {
        {"CUSTOMDATA", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"POSITION", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"SV_Position", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"SV_ClipDistance", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"COLOR", 0, DXGI_FORMAT_B8G8R8A8_UNORM, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"SV_InstanceID", 0, DXGI_FORMAT_R32_UINT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"SV_VertexID", 0, DXGI_FORMAT_R32_UINT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
    };
#if 0
    float4 main(float4 d : CUSTOMDATA, float4 p1 : POSITION, float4 p2 : SV_Position, float4 c : COLOR,
                float cd : SV_ClipDistance, uint iid : SV_InstanceID, uint vid : SV_VertexID) : SV_POSITION
    {
        return p1;
    }
#endif
    static const DWORD vs2_code[] =
    {
        0x43425844, 0x3e9b7242, 0x048c7a6e, 0x31fd9fc7, 0x1105dc32, 0x00000001, 0x000001ac, 0x00000003,
        0x0000002c, 0x00000134, 0x00000168, 0x4e475349, 0x00000100, 0x00000007, 0x00000008, 0x000000b0,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x000000bb, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x000000c4, 0x00000000, 0x00000000, 0x00000003, 0x00000002,
        0x0000000f, 0x000000d0, 0x00000000, 0x00000000, 0x00000003, 0x00000003, 0x0000000f, 0x000000d6,
        0x00000000, 0x00000000, 0x00000003, 0x00000004, 0x00000001, 0x000000e6, 0x00000000, 0x00000008,
        0x00000001, 0x00000005, 0x00000001, 0x000000f4, 0x00000000, 0x00000006, 0x00000001, 0x00000006,
        0x00000001, 0x54535543, 0x41444d4f, 0x50004154, 0x5449534f, 0x004e4f49, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4f4c4f43, 0x56530052, 0x696c435f, 0x73694470, 0x636e6174, 0x56530065, 0x736e495f,
        0x636e6174, 0x00444965, 0x565f5653, 0x65747265, 0x00444978, 0x4e47534f, 0x0000002c, 0x00000001,
        0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x505f5653,
        0x5449534f, 0x004e4f49, 0x52444853, 0x0000003c, 0x00010040, 0x0000000f, 0x0300005f, 0x001010f2,
        0x00000001, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x05000036, 0x001020f2, 0x00000000,
        0x00101e46, 0x00000001, 0x0100003e
    };

#if 0
float4 main() : SV_POSITION
{
    return float4(0, 0, 0, 1);
}
#endif
    static const DWORD vs3_code[] =
    {
        0x43425844, 0xcf477c49, 0x343fc59b, 0x52dff555, 0x73ebaa82, 0x00000001, 0x000000b4, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000,
        0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x0000003c, 0x00010040, 0x0000000f,
        0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x08000036, 0x001020f2, 0x00000000, 0x00004002,
        0x00000000, 0x00000000, 0x00000000, 0x3f800000, 0x0100003e,
    };

    static D3D11_INPUT_ELEMENT_DESC cs_layout_desc[] =
    {
        {"SV_GroupID", 0, DXGI_FORMAT_R32_UINT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"SV_GroupThreadID", 0, DXGI_FORMAT_R32_UINT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"SV_DispatchThreadID", 0, DXGI_FORMAT_R32_UINT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"SV_GroupIndex", 0, DXGI_FORMAT_R32_UINT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
#if 0
    [numthreads(256, 1, 1)]
    void main(uint p1 : SV_GroupID, uint p2 : SV_GroupThreadID, uint p3 : SV_DispatchThreadID, uint p4 : SV_GroupIndex)
    {
    }
#endif
    static const DWORD cs_code[] =
    {
        0x43425844, 0x4ea5ca3f, 0xc6e47bfe, 0x5409d22b, 0xff4f75b7, 0x00000001, 0x00000074, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000020, 0x00050040, 0x00000008, 0x0100086a,
        0x0400009b, 0x00000100, 0x00000001, 0x00000001, 0x0100003e,
    };

    static D3D11_INPUT_ELEMENT_DESC ps_layout_desc[] =
    {
        {"CUSTOMDATA", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"POSITION", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"COLOR", 0, DXGI_FORMAT_B8G8R8A8_UNORM, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"SV_Position", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
        {"SV_InstanceID", 0, DXGI_FORMAT_R32_UINT, 0, 0, D3D11_INPUT_PER_INSTANCE_DATA, 0},
    };
    static const DWORD ps_code[] =
    {
#if 0
    float4 main(float4 d : CUSTOMDATA, float4 p1 : POSITION, float4 p2 : SV_Position, float4 c : COLOR,
                uint iid : SV_InstanceID) : SV_Target
    {
        return c;
    }
#endif
        0x43425844, 0x0138f525, 0x4a391fec, 0xaf43a6b1, 0xab08068f, 0x00000001, 0x0000015c, 0x00000003,
        0x0000002c, 0x000000e8, 0x0000011c, 0x4e475349, 0x000000b4, 0x00000005, 0x00000008, 0x00000080,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x0000008b, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x0000000f, 0x00000094, 0x00000000, 0x00000001, 0x00000003, 0x00000002,
        0x0000000f, 0x000000a0, 0x00000000, 0x00000000, 0x00000003, 0x00000003, 0x00000f0f, 0x000000a6,
        0x00000000, 0x00000000, 0x00000001, 0x00000004, 0x00000001, 0x54535543, 0x41444d4f, 0x50004154,
        0x5449534f, 0x004e4f49, 0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0x56530052, 0x736e495f,
        0x636e6174, 0x00444965, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853,
        0x00000038, 0x00000040, 0x0000000e, 0x03001062, 0x001010f2, 0x00000003, 0x03000065, 0x001020f2,
        0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000003, 0x0100003e,
    };

    static const DXGI_FORMAT vertex_formats[] =
    {
        DXGI_FORMAT_R32G32_FLOAT,
        DXGI_FORMAT_R32G32_UINT,
        DXGI_FORMAT_R32G32_SINT,
        DXGI_FORMAT_R16G16_FLOAT,
        DXGI_FORMAT_R16G16_UINT,
        DXGI_FORMAT_R16G16_SINT,
        DXGI_FORMAT_R11G11B10_FLOAT,
        DXGI_FORMAT_R32_FLOAT,
        DXGI_FORMAT_R32_UINT,
        DXGI_FORMAT_R32_SINT,
        DXGI_FORMAT_R16_FLOAT,
        DXGI_FORMAT_R16_UINT,
        DXGI_FORMAT_R16_SINT,
        DXGI_FORMAT_R8G8_UNORM,
        DXGI_FORMAT_R8_UINT,
        DXGI_FORMAT_R8_SINT,
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    for (i = 0; i < ARRAY_SIZE(vertex_formats); ++i)
    {
        expected_refcount = get_refcount(device) + 1;
        layout_desc->Format = vertex_formats[i];
        hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
                vs_code, sizeof(vs_code), &input_layout);
        ok(hr == S_OK, "Format %#x: Got unexpected hr %#lx.\n", vertex_formats[i], hr);
        refcount = get_refcount(device);
        ok(refcount == expected_refcount, "Format %#x: Got refcount %lu, expected %lu.\n",
                vertex_formats[i], refcount, expected_refcount);
        ID3D11InputLayout_Release(input_layout);
    }

    layout_desc[0].Format = vertex_formats[0];

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, 0, vs_code, sizeof(vs_code), &input_layout);
    ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, 0, vs3_code, sizeof(vs3_code), &input_layout);
    ID3D11InputLayout_Release(input_layout);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);

    hr = ID3D11Device_CreateInputLayout(device, NULL, 0, vs3_code, sizeof(vs3_code), &input_layout);
    ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);

    layout_desc->SemanticIndex = 1;
    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
    layout_desc->SemanticIndex = 0;

    layout_desc->InputSlotClass = D3D11_INPUT_PER_INSTANCE_DATA;
    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    todo_wine ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);
    if (SUCCEEDED(hr))
        ID3D11InputLayout_Release(input_layout);
    hr = ID3D11Device_CreateInputLayout(device, layout_desc, 1, vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);
    ID3D11InputLayout_Release(input_layout);
    layout_desc->InputSlotClass = D3D11_INPUT_PER_VERTEX_DATA;

    layout_desc->SemanticName = "SV_POSITION";
    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);

    layout_desc->SemanticName = "POSITION2";
    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == E_INVALIDARG, "Got hr %#lx.\n", hr);

    check_layout_element_exclusion(device, vs2_layout_desc, ARRAY_SIZE(vs2_layout_desc), vs2_code,
            sizeof(vs2_code), FALSE, "vs2");
    check_layout_element_exclusion(device, cs_layout_desc, ARRAY_SIZE(cs_layout_desc), cs_code,
            sizeof(cs_code), TRUE, "cs");
    check_layout_element_exclusion(device, ps_layout_desc, ARRAY_SIZE(ps_layout_desc), ps_code,
            sizeof(ps_code), FALSE, "ps");

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_input_layout_alignment(void)
{
    ID3D11InputLayout *layout;
    ID3D11Device *device;
    unsigned int i;
    ULONG refcount;
    HRESULT hr;

    static const DWORD vs_code[] =
    {
#if 0
        float4 main(float4 position : POSITION) : SV_POSITION
        {
            return position;
        }
#endif
        0x43425844, 0xa7a2f22d, 0x83ff2560, 0xe61638bd, 0x87e3ce90, 0x00000001, 0x000000d8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0xababab00,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52444853, 0x0000003c, 0x00010040,
        0x0000000f, 0x0300005f, 0x001010f2, 0x00000000, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x05000036, 0x001020f2, 0x00000000, 0x00101e46, 0x00000000, 0x0100003e,
    };

    static const struct
    {
        D3D11_INPUT_ELEMENT_DESC elements[2];
        HRESULT hr;
    }
    test_data[] =
    {
        {{
            {"POSITION", 0, DXGI_FORMAT_R8_UINT,            0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R16_UINT,           0,  2, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, S_OK},
        {{
            {"POSITION", 0, DXGI_FORMAT_R8_UINT,            0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R16_UINT,           0,  1, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, E_INVALIDARG},
        {{
            {"POSITION", 0, DXGI_FORMAT_R8_UINT,            0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R8_UINT,            0,  1, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, S_OK},
        {{
            {"POSITION", 0, DXGI_FORMAT_R16_UINT,           0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R32_UINT,           0,  2, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, E_INVALIDARG},
        {{
            {"POSITION", 0, DXGI_FORMAT_R16_UINT,           0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,  2, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, E_INVALIDARG},
        {{
            {"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT,    0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 16, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, S_OK},
        {{
            {"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT,    0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 17, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, E_INVALIDARG},
        {{
            {"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT,    0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 18, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, E_INVALIDARG},
        {{
            {"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT,    0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 19, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, E_INVALIDARG},
        {{
            {"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT,    0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
            {"COLOR",    0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 20, D3D11_INPUT_PER_VERTEX_DATA, 0},
        }, S_OK},
    };

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    for (i = 0; i < ARRAY_SIZE(test_data); ++i)
    {
        hr = ID3D11Device_CreateInputLayout(device, test_data[i].elements, 2, vs_code, sizeof(vs_code), &layout);
        ok(hr == test_data[i].hr, "Test %u: Got unexpected hr %#lx, expected %#lx.\n", i, hr, test_data[i].hr);
        if (SUCCEEDED(hr))
            ID3D11InputLayout_Release(layout);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_input_assembler(void)
{
    enum layout_id
    {
        LAYOUT_FLOAT32,
        LAYOUT_UINT16,
        LAYOUT_SINT16,
        LAYOUT_UNORM16,
        LAYOUT_SNORM16,
        LAYOUT_UINT8,
        LAYOUT_SINT8,
        LAYOUT_UNORM8,
        LAYOUT_SNORM8,
        LAYOUT_UNORM10_2,
        LAYOUT_UINT10_2,

        LAYOUT_COUNT,
    };

    D3D11_INPUT_ELEMENT_DESC input_layout_desc[] =
    {
        {"POSITION",  0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"ATTRIBUTE", 0, DXGI_FORMAT_UNKNOWN,      1, 0, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    ID3D11VertexShader *vs_float, *vs_uint, *vs_sint;
    ID3D11InputLayout *input_layout[LAYOUT_COUNT];
    ID3D11Buffer *vb_position, *vb_attribute;
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    unsigned int i, j, stride, offset;
    ID3D11Texture2D *render_target;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DXGI_FORMAT layout_formats[LAYOUT_COUNT] =
    {
        DXGI_FORMAT_R32G32B32A32_FLOAT,
        DXGI_FORMAT_R16G16B16A16_UINT,
        DXGI_FORMAT_R16G16B16A16_SINT,
        DXGI_FORMAT_R16G16B16A16_UNORM,
        DXGI_FORMAT_R16G16B16A16_SNORM,
        DXGI_FORMAT_R8G8B8A8_UINT,
        DXGI_FORMAT_R8G8B8A8_SINT,
        DXGI_FORMAT_R8G8B8A8_UNORM,
        DXGI_FORMAT_R8G8B8A8_SNORM,
        DXGI_FORMAT_R10G10B10A2_UNORM,
        DXGI_FORMAT_R10G10B10A2_UINT,
    };
    static const struct vec2 quad[] =
    {
        {-1.0f, -1.0f},
        {-1.0f,  1.0f},
        { 1.0f, -1.0f},
        { 1.0f,  1.0f},
    };
    static const DWORD ps_code[] =
    {
#if 0
        float4 main(float4 position : POSITION, float4 color: COLOR) : SV_Target
        {
            return color;
        }
#endif
        0x43425844, 0xa9150342, 0x70e18d2e, 0xf7769835, 0x4c3a7f02, 0x00000001, 0x000000f0, 0x00000003,
        0x0000002c, 0x0000007c, 0x000000b0, 0x4e475349, 0x00000048, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000041, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0x4c4f4300, 0xab00524f, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000038, 0x00000040, 0x0000000e,
        0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000001, 0x0100003e,
    };
    static const DWORD vs_float_code[] =
    {
#if 0
        struct output
        {
            float4 position : SV_Position;
            float4 color : COLOR;
        };

        void main(float4 position : POSITION, float4 color : ATTRIBUTE, out output o)
        {
            o.position = position;
            o.color = color;
        }
#endif
        0x43425844, 0xf6051ffd, 0xd9e49503, 0x171ad197, 0x3764fe47, 0x00000001, 0x00000144, 0x00000003,
        0x0000002c, 0x00000080, 0x000000d4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000041, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0x54544100, 0x55424952, 0xab004554,
        0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0xabab0052, 0x52444853, 0x00000068, 0x00010040,
        0x0000001a, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x001010f2, 0x00000001, 0x04000067,
        0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x05000036, 0x001020f2, 0x00000001, 0x00101e46, 0x00000001,
        0x0100003e,
    };
    static const DWORD vs_uint_code[] =
    {
#if 0
        struct output
        {
            float4 position : SV_Position;
            float4 color : COLOR;
        };

        void main(float4 position : POSITION, uint4 color : ATTRIBUTE, out output o)
        {
            o.position = position;
            o.color = color;
        }
#endif
        0x43425844, 0x0bae0bc0, 0xf6473aa5, 0x4ecf4a25, 0x414fac23, 0x00000001, 0x00000144, 0x00000003,
        0x0000002c, 0x00000080, 0x000000d4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000041, 0x00000000, 0x00000000,
        0x00000001, 0x00000001, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0x54544100, 0x55424952, 0xab004554,
        0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0xabab0052, 0x52444853, 0x00000068, 0x00010040,
        0x0000001a, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x001010f2, 0x00000001, 0x04000067,
        0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x05000056, 0x001020f2, 0x00000001, 0x00101e46, 0x00000001,
        0x0100003e,
    };
    static const DWORD vs_sint_code[] =
    {
#if 0
        struct output
        {
            float4 position : SV_Position;
            float4 color : COLOR;
        };

        void main(float4 position : POSITION, int4 color : ATTRIBUTE, out output o)
        {
            o.position = position;
            o.color = color;
        }
#endif
        0x43425844, 0xaf60aad9, 0xba91f3a4, 0x2015d384, 0xf746fdf5, 0x00000001, 0x00000144, 0x00000003,
        0x0000002c, 0x00000080, 0x000000d4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000041, 0x00000000, 0x00000000,
        0x00000002, 0x00000001, 0x00000f0f, 0x49534f50, 0x4e4f4954, 0x54544100, 0x55424952, 0xab004554,
        0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x4f4c4f43, 0xabab0052, 0x52444853, 0x00000068, 0x00010040,
        0x0000001a, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x001010f2, 0x00000001, 0x04000067,
        0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x0500002b, 0x001020f2, 0x00000001, 0x00101e46, 0x00000001,
        0x0100003e,
    };
    static const float float32_data[] = {1.0f, 2.0f, 3.0f, 4.0f};
    static const unsigned short uint16_data[] = {6, 8, 55, 777};
    static const short sint16_data[] = {-1, 33, 8, -77};
    static const unsigned short unorm16_data[] = {0, 16383, 32767, 65535};
    static const short snorm16_data[] = {-32768, 0, 32767, 0};
    static const unsigned char uint8_data[] = {0, 64, 128, 255};
    static const signed char sint8_data[] = {-128, 0, 127, 64};
    static const unsigned int uint32_zero = 0;
    static const unsigned int uint32_max = 0xffffffff;
    static const unsigned int unorm10_2_data= 0xa00003ff;
    static const unsigned int g10_data = 0x000ffc00;
    static const unsigned int a2_data = 0xc0000000;
    static const struct
    {
        enum layout_id layout_id;
        unsigned int stride;
        const void *data;
        struct vec4 expected_color;
        BOOL todo;
    }
    tests[] =
    {
        {LAYOUT_FLOAT32,   sizeof(float32_data),   float32_data,
                {1.0f, 2.0f, 3.0f, 4.0f}},
        {LAYOUT_UINT16,    sizeof(uint16_data),    uint16_data,
                {6.0f, 8.0f, 55.0f, 777.0f}, TRUE},
        {LAYOUT_SINT16,    sizeof(sint16_data),    sint16_data,
                {-1.0f, 33.0f, 8.0f, -77.0f}, TRUE},
        {LAYOUT_UNORM16,   sizeof(unorm16_data),   unorm16_data,
                {0.0f, 16383.0f / 65535.0f, 32767.0f / 65535.0f, 1.0f}},
        {LAYOUT_SNORM16,   sizeof(snorm16_data),   snorm16_data,
                {-1.0f, 0.0f, 1.0f, 0.0f}},
        {LAYOUT_UINT8,     sizeof(uint32_zero),    &uint32_zero,
                {0.0f, 0.0f, 0.0f, 0.0f}},
        {LAYOUT_UINT8,     sizeof(uint32_max),     &uint32_max,
                {255.0f, 255.0f, 255.0f, 255.0f}},
        {LAYOUT_UINT8,     sizeof(uint8_data),     uint8_data,
                {0.0f, 64.0f, 128.0f, 255.0f}},
        {LAYOUT_SINT8,     sizeof(uint32_zero),    &uint32_zero,
                {0.0f, 0.0f, 0.0f, 0.0f}},
        {LAYOUT_SINT8,     sizeof(uint32_max),     &uint32_max,
                {-1.0f, -1.0f, -1.0f, -1.0f}},
        {LAYOUT_SINT8,     sizeof(sint8_data),     sint8_data,
                {-128.0f, 0.0f, 127.0f, 64.0f}},
        {LAYOUT_UNORM8,    sizeof(uint32_zero),    &uint32_zero,
                {0.0f, 0.0f, 0.0f, 0.0f}},
        {LAYOUT_UNORM8,    sizeof(uint32_max),     &uint32_max,
                {1.0f, 1.0f, 1.0f, 1.0f}},
        {LAYOUT_UNORM8,    sizeof(uint8_data),     uint8_data,
                {0.0f, 64.0f / 255.0f, 128.0f / 255.0f, 1.0f}},
        {LAYOUT_SNORM8,    sizeof(uint32_zero),    &uint32_zero,
                {0.0f, 0.0f, 0.0f, 0.0f}},
        {LAYOUT_SNORM8,    sizeof(sint8_data),     sint8_data,
                {-1.0f, 0.0f, 1.0f, 64.0f / 127.0f}},
        {LAYOUT_UNORM10_2, sizeof(uint32_zero),    &uint32_zero,
                {0.0f, 0.0f, 0.0f, 0.0f}},
        {LAYOUT_UNORM10_2, sizeof(uint32_max),     &uint32_max,
                {1.0f, 1.0f, 1.0f, 1.0f}},
        {LAYOUT_UNORM10_2, sizeof(g10_data),       &g10_data,
                {0.0f, 1.0f, 0.0f, 0.0f}},
        {LAYOUT_UNORM10_2, sizeof(a2_data),        &a2_data,
                {0.0f, 0.0f, 0.0f, 1.0f}},
        {LAYOUT_UNORM10_2, sizeof(unorm10_2_data), &unorm10_2_data,
                {1.0f, 0.0f, 512.0f / 1023.0f, 2.0f / 3.0f}},
        {LAYOUT_UINT10_2,  sizeof(uint32_zero),    &uint32_zero,
                {0.0f, 0.0f, 0.0f, 0.0f}},
        {LAYOUT_UINT10_2,  sizeof(uint32_max),     &uint32_max,
                {1023.0f, 1023.0f, 1023.0f, 3.0f}},
        {LAYOUT_UINT10_2,  sizeof(g10_data),       &g10_data,
                {0.0f, 1023.0f, 0.0f, 0.0f}},
        {LAYOUT_UINT10_2,  sizeof(a2_data),        &a2_data,
                {0.0f, 0.0f, 0.0f, 3.0f}},
        {LAYOUT_UINT10_2,  sizeof(unorm10_2_data), &unorm10_2_data,
                {1023.0f, 0.0f, 512.0f, 2.0f}},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateVertexShader(device, vs_float_code, sizeof(vs_float_code), NULL, &vs_float);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateVertexShader(device, vs_uint_code, sizeof(vs_uint_code), NULL, &vs_uint);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CreateVertexShader(device, vs_sint_code, sizeof(vs_sint_code), NULL, &vs_sint);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    for (i = 0; i < LAYOUT_COUNT; ++i)
    {
        input_layout_desc[1].Format = layout_formats[i];
        input_layout[i] = NULL;
        hr = ID3D11Device_CreateInputLayout(device, input_layout_desc, ARRAY_SIZE(input_layout_desc),
                vs_float_code, sizeof(vs_float_code), &input_layout[i]);
        todo_wine_if(input_layout_desc[1].Format == DXGI_FORMAT_R10G10B10A2_UINT)
        ok(hr == S_OK, "Format %#x: Got unexpected hr %#lx.\n", layout_formats[i], hr);
    }

    vb_position = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(quad), quad);
    vb_attribute = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, 1024, NULL);

    texture_desc.Width = 640;
    texture_desc.Height = 480;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &render_target);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)render_target, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    offset = 0;
    stride = sizeof(*quad);
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb_position, &stride, &offset);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        D3D11_BOX box = {0, 0, 0, 1, 1, 1};

        if (tests[i].layout_id == LAYOUT_UINT10_2)
            continue;

        assert(tests[i].layout_id < LAYOUT_COUNT);
        ID3D11DeviceContext_IASetInputLayout(context, input_layout[tests[i].layout_id]);

        assert(4 * tests[i].stride <= 1024);
        box.right = tests[i].stride;
        for (j = 0; j < 4; ++j)
        {
            ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)vb_attribute, 0,
                    &box, tests[i].data, 0, 0);
            box.left += tests[i].stride;
            box.right += tests[i].stride;
        }

        stride = tests[i].stride;
        ID3D11DeviceContext_IASetVertexBuffers(context, 1, 1, &vb_attribute, &stride, &offset);

        switch (layout_formats[tests[i].layout_id])
        {
            case DXGI_FORMAT_R16G16B16A16_UINT:
            case DXGI_FORMAT_R10G10B10A2_UINT:
            case DXGI_FORMAT_R8G8B8A8_UINT:
                ID3D11DeviceContext_VSSetShader(context, vs_uint, NULL, 0);
                break;
            case DXGI_FORMAT_R16G16B16A16_SINT:
            case DXGI_FORMAT_R8G8B8A8_SINT:
                ID3D11DeviceContext_VSSetShader(context, vs_sint, NULL, 0);
                break;

            default:
                trace("Unhandled format %#x.\n", layout_formats[tests[i].layout_id]);
                /* Fall through. */
            case DXGI_FORMAT_R32G32B32A32_FLOAT:
            case DXGI_FORMAT_R16G16B16A16_UNORM:
            case DXGI_FORMAT_R16G16B16A16_SNORM:
            case DXGI_FORMAT_R10G10B10A2_UNORM:
            case DXGI_FORMAT_R8G8B8A8_UNORM:
            case DXGI_FORMAT_R8G8B8A8_SNORM:
                ID3D11DeviceContext_VSSetShader(context, vs_float, NULL, 0);
                break;
        }

        ID3D11DeviceContext_Draw(context, 4, 0);
        check_texture_vec4(render_target, &tests[i].expected_color, 2);
    }

    ID3D11Texture2D_Release(render_target);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11Buffer_Release(vb_attribute);
    ID3D11Buffer_Release(vb_position);
    for (i = 0; i < LAYOUT_COUNT; ++i)
    {
        if (input_layout[i])
            ID3D11InputLayout_Release(input_layout[i]);
    }
    ID3D11PixelShader_Release(ps);
    ID3D11VertexShader_Release(vs_float);
    ID3D11VertexShader_Release(vs_uint);
    ID3D11VertexShader_Release(vs_sint);
    release_test_context(&test_context);
}

static void test_null_sampler(void)
{
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11ShaderResourceView *srv;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11SamplerState *sampler;
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float4 main(float4 position : SV_POSITION) : SV_Target
        {
            return t.Sample(s, float2(position.x / 640.0f, position.y / 480.0f));
        }
#endif
        0x43425844, 0x1ce9b612, 0xc8176faa, 0xd37844af, 0xdb515605, 0x00000001, 0x00000134, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f, 0x004e4f49,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000098, 0x00000040,
        0x00000026, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555,
        0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0a000038, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x00004002, 0x3acccccd,
        0x3b088889, 0x00000000, 0x00000000, 0x09000045, 0x001020f2, 0x00000000, 0x00100046, 0x00000000,
        0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0100003e,
    };
    static const float blue[] = {0.0f, 0.0f, 1.0f, 1.0f};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    texture_desc.Width = 64;
    texture_desc.Height = 64;
    texture_desc.MipLevels = 1;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_RENDER_TARGET | D3D11_BIND_SHADER_RESOURCE;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;

    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, NULL, &srv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_ClearRenderTargetView(context, rtv, blue);

    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);
    sampler = NULL;
    ID3D11DeviceContext_PSSetSamplers(context, 0, 1, &sampler);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0xffff0000, 0);

    ID3D11ShaderResourceView_Release(srv);
    ID3D11RenderTargetView_Release(rtv);
    ID3D11Texture2D_Release(texture);
    ID3D11PixelShader_Release(ps);
    release_test_context(&test_context);
}

static void test_check_feature_support(void)
{
    D3D11_FEATURE_DATA_THREADING threading[2];
    D3D11_FEATURE_DATA_D3D10_X_HARDWARE_OPTIONS hwopts;
    D3D11_FEATURE_DATA_ARCHITECTURE_INFO archinfo;
    ID3D11Device *device;
    ULONG refcount;
    HRESULT hr;

    if (!(device = create_device(NULL)))
    {
        skip("Failed to create device.\n");
        return;
    }

    memset(threading, 0xef, sizeof(threading));

    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_THREADING, NULL, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_THREADING, threading, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_THREADING, threading, sizeof(*threading) - 1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_THREADING, threading, sizeof(*threading) / 2);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_THREADING, threading, sizeof(*threading) + 1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_THREADING, threading, sizeof(*threading) * 2);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    ok(threading[0].DriverConcurrentCreates == 0xefefefef,
            "Got unexpected concurrent creates %#x.\n", threading[0].DriverConcurrentCreates);
    ok(threading[0].DriverCommandLists == 0xefefefef,
            "Got unexpected command lists %#x.\n", threading[0].DriverCommandLists);
    ok(threading[1].DriverConcurrentCreates == 0xefefefef,
            "Got unexpected concurrent creates %#x.\n", threading[1].DriverConcurrentCreates);
    ok(threading[1].DriverCommandLists == 0xefefefef,
            "Got unexpected command lists %#x.\n", threading[1].DriverCommandLists);

    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_THREADING, threading, sizeof(*threading));
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ok(threading->DriverConcurrentCreates == TRUE || threading->DriverConcurrentCreates == FALSE,
            "Got unexpected concurrent creates %#x.\n", threading->DriverConcurrentCreates);
    ok(threading->DriverCommandLists == TRUE || threading->DriverCommandLists == FALSE,
            "Got unexpected command lists %#x.\n", threading->DriverCommandLists);

    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_D3D10_X_HARDWARE_OPTIONS, NULL, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_D3D10_X_HARDWARE_OPTIONS, &hwopts, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_D3D10_X_HARDWARE_OPTIONS, &hwopts, sizeof(hwopts) - 1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_D3D10_X_HARDWARE_OPTIONS, &hwopts, sizeof(hwopts) / 2);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_D3D10_X_HARDWARE_OPTIONS, &hwopts, sizeof(hwopts) + 1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_D3D10_X_HARDWARE_OPTIONS, &hwopts, sizeof(hwopts) * 2);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_D3D10_X_HARDWARE_OPTIONS, &hwopts, sizeof(hwopts));
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    trace("Compute shader support via SM4 %#x.\n", hwopts.ComputeShaders_Plus_RawAndStructuredBuffers_Via_Shader_4_x);

    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_ARCHITECTURE_INFO, &archinfo, sizeof(archinfo));
    ok(hr == S_OK || broken(hr == E_INVALIDARG) /* Not available on all Windows versions. */,
            "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_ARCHITECTURE_INFO, &archinfo, sizeof(archinfo)*2);
    ok(hr == E_INVALIDARG /* Not available on all Windows versions but they will return E_INVALIDARG anyways. */,
            "Got unexpected hr %#lx.\n", hr);

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_create_unordered_access_view(void)
{
    static const D3D_FEATURE_LEVEL feature_level = D3D_FEATURE_LEVEL_11_0;
    D3D11_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D11_TEXTURE3D_DESC texture3d_desc;
    D3D11_TEXTURE2D_DESC texture2d_desc;
    ULONG refcount, expected_refcount;
    D3D11_SUBRESOURCE_DATA data = {0};
    ID3D11UnorderedAccessView *uav;
    struct device_desc device_desc;
    D3D11_BUFFER_DESC buffer_desc;
    ID3D11Device *device, *tmp;
    ID3D11Texture3D *texture3d;
    ID3D11Texture2D *texture2d;
    ID3D11Resource *texture;
    ID3D11Buffer *buffer;
    unsigned int i;
    HRESULT hr;

#define FMT_UNKNOWN  DXGI_FORMAT_UNKNOWN
#define RGBA8_UNORM  DXGI_FORMAT_R8G8B8A8_UNORM
#define RGBA8_SRGB   DXGI_FORMAT_R8G8B8A8_UNORM_SRGB
#define RGBA8_UINT   DXGI_FORMAT_R8G8B8A8_UINT
#define RGBA8_TL     DXGI_FORMAT_R8G8B8A8_TYPELESS
#define DIM_UNKNOWN  D3D11_UAV_DIMENSION_UNKNOWN
#define TEX_1D       D3D11_UAV_DIMENSION_TEXTURE1D
#define TEX_1D_ARRAY D3D11_UAV_DIMENSION_TEXTURE1DARRAY
#define TEX_2D       D3D11_UAV_DIMENSION_TEXTURE2D
#define TEX_2D_ARRAY D3D11_UAV_DIMENSION_TEXTURE2DARRAY
#define TEX_3D       D3D11_UAV_DIMENSION_TEXTURE3D
    static const struct
    {
        struct
        {
            unsigned int miplevel_count;
            unsigned int depth_or_array_size;
            DXGI_FORMAT format;
        } texture;
        struct uav_desc uav_desc;
        struct uav_desc expected_uav_desc;
    }
    tests[] =
    {
        {{ 1, 1, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_2D,       0}},
        {{10, 1, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_2D,       0}},
        {{10, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D,       0},         {RGBA8_UNORM, TEX_2D,       0}},
        {{10, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D,       1},         {RGBA8_UNORM, TEX_2D,       1}},
        {{10, 1, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D,       9},         {RGBA8_UNORM, TEX_2D,       9}},
        {{ 1, 1, RGBA8_TL},    {RGBA8_UNORM, TEX_2D,       0},         {RGBA8_UNORM, TEX_2D,       0}},
        {{10, 1, RGBA8_TL},    {RGBA8_UNORM, TEX_2D,       0},         {RGBA8_UNORM, TEX_2D,       0}},
        {{ 1, 1, RGBA8_UINT},  {0},                                    {RGBA8_UINT,  TEX_2D,       0, 0, 1}},
        {{ 1, 1, RGBA8_TL},    {RGBA8_UINT,  TEX_2D,       0, 0, ~0u}, {RGBA8_UINT,  TEX_2D,       0, 0, 1}},
        {{ 1, 4, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 1, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 1, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 3, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 3, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 5, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 5, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 9, 0, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 9, 0, 4}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 1, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 1, 3}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 2, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 2, 2}},
        {{10, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_2D_ARRAY, 0, 3, ~0u}, {RGBA8_UNORM, TEX_2D_ARRAY, 0, 3, 1}},
        {{ 1, 6, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_3D,       0, 0, 6}},
        {{ 2, 6, RGBA8_UNORM}, {0},                                    {RGBA8_UNORM, TEX_3D,       0, 0, 6}},
        {{ 2, 6, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 0, 6}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       1, 0, 2}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       1, 0, 2}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 1, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 1, 3}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 2, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 2, 2}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 3, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 3, 1}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 1,  1},  {RGBA8_UNORM, TEX_3D,       0, 1, 1}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 1,  1},  {RGBA8_UNORM, TEX_3D,       1, 1, 1}},
        {{ 2, 4, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 1, ~0u}, {RGBA8_UNORM, TEX_3D,       1, 1, 1}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       0, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       0, 0, 8}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       1, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       1, 0, 4}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       2, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       2, 0, 2}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       3, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       3, 0, 1}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       4, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       4, 0, 1}},
        {{ 6, 8, RGBA8_UNORM}, {FMT_UNKNOWN, TEX_3D,       5, 0, ~0u}, {RGBA8_UNORM, TEX_3D,       5, 0, 1}},
    };
    static const struct
    {
        struct
        {
            D3D11_UAV_DIMENSION dimension;
            unsigned int miplevel_count;
            unsigned int depth_or_array_size;
            DXGI_FORMAT format;
        } texture;
        struct uav_desc uav_desc;
    }
    invalid_desc_tests[] =
    {
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, DIM_UNKNOWN}},
        {{TEX_2D, 6, 4, RGBA8_UNORM}, {RGBA8_UNORM, DIM_UNKNOWN}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0, ~0u}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_2D,        0}},
        {{TEX_2D, 1, 1, RGBA8_TL},    {FMT_UNKNOWN, TEX_2D,        0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  0}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  1, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  2}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 1,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_UINT,  TEX_2D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UINT},  {RGBA8_UNORM, TEX_2D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_UNORM}, {RGBA8_SRGB,  TEX_2D,        0, 0,  1}},
        {{TEX_2D, 1, 1, RGBA8_SRGB},  {RGBA8_UNORM, TEX_2D,        0, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D,        0}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_1D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D,        0}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_UNORM, TEX_2D_ARRAY,  0, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  0}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 0,  1}},
        {{TEX_3D, 1, 1, RGBA8_UNORM}, {RGBA8_TL,    TEX_3D,        0, 0,  1}},
        {{TEX_3D, 1, 9, RGBA8_UNORM}, {RGBA8_TL,    TEX_3D,        0, 0,  1}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 0,  9}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        3, 0,  2}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        2, 0,  4}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 0,  8}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        0, 8, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        1, 4, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        2, 2, ~0u}},
        {{TEX_3D, 4, 8, RGBA8_UNORM}, {RGBA8_UNORM, TEX_3D,        3, 1, ~0u}},
    };
#undef FMT_UNKNOWN
#undef RGBA8_UNORM
#undef RGBA8_SRGB
#undef RGBA8_UINT
#undef RGBA8_TL
#undef DIM_UNKNOWN
#undef TEX_1D
#undef TEX_1D_ARRAY
#undef TEX_2D
#undef TEX_2D_ARRAY
#undef TEX_3D

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }

    buffer_desc.ByteWidth = 1024;
    buffer_desc.Usage = D3D11_USAGE_DEFAULT;
    buffer_desc.BindFlags = D3D11_BIND_UNORDERED_ACCESS;
    buffer_desc.CPUAccessFlags = 0;
    buffer_desc.MiscFlags = 0;
    buffer_desc.StructureByteStride = 0;

    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, &data, &buffer);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, &buffer);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11Buffer_GetDevice(buffer, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    uav_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 64;
    U(uav_desc).Buffer.Flags = 0;

    uav = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, NULL, &uav);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!uav, "Unexpected pointer %p.\n", uav);

    expected_refcount = get_refcount(device) + 1;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    refcount = get_refcount(device);
    ok(refcount >= expected_refcount, "Got unexpected refcount %lu, expected >= %lu.\n", refcount, expected_refcount);
    tmp = NULL;
    expected_refcount = refcount + 1;
    ID3D11UnorderedAccessView_GetDevice(uav, &tmp);
    ok(tmp == device, "Got unexpected device %p, expected %p.\n", tmp, device);
    refcount = get_refcount(device);
    ok(refcount == expected_refcount, "Got unexpected refcount %lu, expected %lu.\n", refcount, expected_refcount);
    ID3D11Device_Release(tmp);

    ID3D11UnorderedAccessView_Release(uav);
    ID3D11Buffer_Release(buffer);

    buffer_desc.MiscFlags = D3D11_RESOURCE_MISC_BUFFER_STRUCTURED;
    buffer_desc.StructureByteStride = 4;

    hr = ID3D11Device_CreateBuffer(device, &buffer_desc, NULL, &buffer);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, NULL, &uav);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    memset(&uav_desc, 0, sizeof(uav_desc));
    ID3D11UnorderedAccessView_GetDesc(uav, &uav_desc);

    ok(uav_desc.Format == DXGI_FORMAT_UNKNOWN, "Got unexpected format %#x.\n", uav_desc.Format);
    ok(uav_desc.ViewDimension == D3D11_UAV_DIMENSION_BUFFER, "Got unexpected view dimension %#x.\n",
            uav_desc.ViewDimension);
    ok(!U(uav_desc).Buffer.FirstElement, "Got unexpected first element %u.\n", U(uav_desc).Buffer.FirstElement);
    ok(U(uav_desc).Buffer.NumElements == 256, "Got unexpected num elements %u.\n", U(uav_desc).Buffer.NumElements);
    ok(!U(uav_desc).Buffer.Flags, "Got unexpected flags %u.\n", U(uav_desc).Buffer.Flags);

    ID3D11UnorderedAccessView_Release(uav);
    ID3D11Buffer_Release(buffer);

    /* Without D3D11_BIND_UNORDERED_ACCESS. */
    buffer = create_buffer(device, 0, 1024, NULL);

    uav_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    uav_desc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
    U(uav_desc).Buffer.FirstElement = 0;
    U(uav_desc).Buffer.NumElements = 64;
    U(uav_desc).Buffer.Flags = 0;

    uav = (void *)0xdeadbeef;
    hr = ID3D11Device_CreateUnorderedAccessView(device, (ID3D11Resource *)buffer, &uav_desc, &uav);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    ok(!uav, "Unexpected pointer %p.\n", uav);

    ID3D11Buffer_Release(buffer);

    texture2d_desc.Width = 512;
    texture2d_desc.Height = 512;
    texture2d_desc.SampleDesc.Count = 1;
    texture2d_desc.SampleDesc.Quality = 0;
    texture2d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture2d_desc.BindFlags = D3D11_BIND_UNORDERED_ACCESS;
    texture2d_desc.CPUAccessFlags = 0;
    texture2d_desc.MiscFlags = 0;

    texture3d_desc.Width = 64;
    texture3d_desc.Height = 64;
    texture3d_desc.Usage = D3D11_USAGE_DEFAULT;
    texture3d_desc.BindFlags = D3D11_BIND_UNORDERED_ACCESS;
    texture3d_desc.CPUAccessFlags = 0;
    texture3d_desc.MiscFlags = 0;

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        D3D11_UNORDERED_ACCESS_VIEW_DESC *current_desc;

        if (tests[i].expected_uav_desc.dimension != D3D11_UAV_DIMENSION_TEXTURE3D)
        {
            texture2d_desc.MipLevels = tests[i].texture.miplevel_count;
            texture2d_desc.ArraySize = tests[i].texture.depth_or_array_size;
            texture2d_desc.Format = tests[i].texture.format;

            hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, NULL, &texture2d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture2d;
        }
        else
        {
            texture3d_desc.MipLevels = tests[i].texture.miplevel_count;
            texture3d_desc.Depth = tests[i].texture.depth_or_array_size;
            texture3d_desc.Format = tests[i].texture.format;

            hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &texture3d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture3d;
        }

        if (tests[i].uav_desc.dimension == D3D11_UAV_DIMENSION_UNKNOWN)
        {
            current_desc = NULL;
        }
        else
        {
            current_desc = &uav_desc;
            get_uav_desc(current_desc, &tests[i].uav_desc);
        }

        expected_refcount = get_refcount(texture);
        hr = ID3D11Device_CreateUnorderedAccessView(device, texture, current_desc, &uav);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        refcount = get_refcount(texture);
        ok(refcount == expected_refcount, "Test %u: Got refcount %lu, expected %lu.\n",
                i, refcount, expected_refcount);

        memset(&uav_desc, 0, sizeof(uav_desc));
        ID3D11UnorderedAccessView_GetDesc(uav, &uav_desc);
        check_uav_desc(&uav_desc, &tests[i].expected_uav_desc);

        ID3D11UnorderedAccessView_Release(uav);
        ID3D11Resource_Release(texture);
    }

    for (i = 0; i < ARRAY_SIZE(invalid_desc_tests); ++i)
    {
        assert(invalid_desc_tests[i].texture.dimension == D3D11_UAV_DIMENSION_TEXTURE2D
                || invalid_desc_tests[i].texture.dimension == D3D11_UAV_DIMENSION_TEXTURE3D);

        if (invalid_desc_tests[i].texture.dimension != D3D11_UAV_DIMENSION_TEXTURE3D)
        {
            texture2d_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
            texture2d_desc.ArraySize = invalid_desc_tests[i].texture.depth_or_array_size;
            texture2d_desc.Format = invalid_desc_tests[i].texture.format;

            hr = ID3D11Device_CreateTexture2D(device, &texture2d_desc, NULL, &texture2d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture2d;
        }
        else
        {
            texture3d_desc.MipLevels = invalid_desc_tests[i].texture.miplevel_count;
            texture3d_desc.Depth = invalid_desc_tests[i].texture.depth_or_array_size;
            texture3d_desc.Format = invalid_desc_tests[i].texture.format;

            hr = ID3D11Device_CreateTexture3D(device, &texture3d_desc, NULL, &texture3d);
            ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
            texture = (ID3D11Resource *)texture3d;
        }

        uav = (void *)0xdeadbeef;
        get_uav_desc(&uav_desc, &invalid_desc_tests[i].uav_desc);
        hr = ID3D11Device_CreateUnorderedAccessView(device, texture, &uav_desc, &uav);
        ok(hr == E_INVALIDARG, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        ok(!uav, "Unexpected pointer %p.\n", uav);

        ID3D11Resource_Release(texture);
    }

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_immediate_constant_buffer(void)
{
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    unsigned int index[4] = {0};
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *cb;
    unsigned int i;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        uint index;

        static const int int_array[6] =
        {
            310, 111, 212, -513, -318, 0,
        };

        static const uint uint_array[6] =
        {
            2, 7, 0x7f800000, 0xff800000, 0x7fc00000, 0
        };

        static const float float_array[6] =
        {
            76, 83.5f, 0.5f, 0.75f, -0.5f, 0.0f,
        };

        float4 main() : SV_Target
        {
            return float4(int_array[index], uint_array[index], float_array[index], 1.0f);
        }
#endif
        0x43425844, 0xbad068da, 0xd631ea3c, 0x41648374, 0x3ccd0120, 0x00000001, 0x00000184, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x0000010c, 0x00000040, 0x00000043,
        0x00001835, 0x0000001a, 0x00000136, 0x00000002, 0x42980000, 0x00000000, 0x0000006f, 0x00000007,
        0x42a70000, 0x00000000, 0x000000d4, 0x7f800000, 0x3f000000, 0x00000000, 0xfffffdff, 0xff800000,
        0x3f400000, 0x00000000, 0xfffffec2, 0x7fc00000, 0xbf000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2,
        0x00000000, 0x02000068, 0x00000001, 0x05000036, 0x00102082, 0x00000000, 0x00004001, 0x3f800000,
        0x06000036, 0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x06000056, 0x00102022,
        0x00000000, 0x0090901a, 0x0010000a, 0x00000000, 0x0600002b, 0x00102012, 0x00000000, 0x0090900a,
        0x0010000a, 0x00000000, 0x06000036, 0x00102042, 0x00000000, 0x0090902a, 0x0010000a, 0x00000000,
        0x0100003e,
    };
    static struct vec4 expected_result[] =
    {
        { 310.0f,          2.0f, 76.00f, 1.0f},
        { 111.0f,          7.0f, 83.50f, 1.0f},
        { 212.0f, 2139095040.0f,  0.50f, 1.0f},
        {-513.0f, 4286578688.0f,  0.75f, 1.0f},
        {-318.0f, 2143289344.0f, -0.50f, 1.0f},
        {   0.0f,          0.0f,  0.0f,  1.0f},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(index), NULL);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);

    for (i = 0; i < ARRAY_SIZE(expected_result); ++i)
    {
        *index = i;
        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, index, 0, 0);

        draw_quad(&test_context);
        check_texture_vec4(texture, &expected_result[i], 0);
    }

    ID3D11Buffer_Release(cb);
    ID3D11PixelShader_Release(ps);
    ID3D11Texture2D_Release(texture);
    ID3D11RenderTargetView_Release(rtv);
    release_test_context(&test_context);
}

static void test_fp_specials(void)
{
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DWORD ps_code[] =
    {
#if 0
        float4 main() : SV_Target
        {
            return float4(0.0f / 0.0f, 1.0f / 0.0f, -1.0f / 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x86d7f319, 0x14cde598, 0xe7ce83a8, 0x0e06f3f0, 0x00000001, 0x000000b0, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000038, 0x00000040, 0x0000000e,
        0x03000065, 0x001020f2, 0x00000000, 0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0xffc00000,
        0x7f800000, 0xff800000, 0x3f800000, 0x0100003e,
    };
    static const struct uvec4 expected_result = {BITS_NNAN, BITS_INF, BITS_NINF, BITS_1_0};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);

    draw_quad(&test_context);
    check_texture_uvec4(texture, &expected_result);

    ID3D11PixelShader_Release(ps);
    ID3D11Texture2D_Release(texture);
    ID3D11RenderTargetView_Release(rtv);
    release_test_context(&test_context);
}

static void test_uint_shader_instructions(void)
{
    struct shader
    {
        const DWORD *code;
        size_t size;
        D3D_FEATURE_LEVEL required_feature_level;
    };

    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    D3D_FEATURE_LEVEL feature_level;
    ID3D11DeviceContext *context;
    ID3D11RenderTargetView *rtv;
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *cb;
    unsigned int i;
    HRESULT hr;

    static const DWORD ps_bfi_code[] =
    {
#if 0
        uint bits, offset, insert, base;

        uint4 main() : SV_Target
        {
            uint mask = ((1 << bits) - 1) << offset;
            return ((insert << offset) & mask) | (base & ~mask);
        }
#endif
        0x43425844, 0xbe9af688, 0xf5caec6f, 0x63ed2522, 0x5f91f209, 0x00000001, 0x000000e0, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000068, 0x00000050, 0x0000001a,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x0f00008c, 0x001020f2, 0x00000000, 0x00208006, 0x00000000, 0x00000000, 0x00208556, 0x00000000,
        0x00000000, 0x00208aa6, 0x00000000, 0x00000000, 0x00208ff6, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_bfi2_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[1], immediateIndexed
        dcl_output o0.xyzw
        dcl_temps 1
        mov r0.xyzw, cb0[0].xyzw
        bfi r0.xyzw, r0.xxxx, r0.yyyy, r0.zzzz, r0.wwww
        mov o0.xyzw, r0.xyzw
        ret
#endif
        0x43425844, 0x900f86b6, 0x73f4dabf, 0xea1b1106, 0x591ac790, 0x00000001, 0x00000104, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x0000008c, 0x00000050, 0x00000023,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x06000036, 0x001000f2, 0x00000000, 0x00208e46, 0x00000000, 0x00000000,
        0x0b00008c, 0x001000f2, 0x00000000, 0x00100006, 0x00000000, 0x00100556, 0x00000000, 0x00100aa6,
        0x00000000, 0x00100ff6, 0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000,
        0x0100003e,
    };
    static const DWORD ps_ibfe_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[1], immediateIndexed
        dcl_output o0.xyzw
        ibfe o0.xyzw, cb0[0].xxxx, cb0[0].yyyy, cb0[0].zzzz
        ret
#endif
        0x43425844, 0x4b2225f7, 0xd0860f66, 0xe38775bb, 0x6d23d1d2, 0x00000001, 0x000000d4, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x0000005c, 0x00000050, 0x00000017,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x0c00008b, 0x001020f2, 0x00000000, 0x00208006, 0x00000000, 0x00000000, 0x00208556, 0x00000000,
        0x00000000, 0x00208aa6, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_ibfe2_code[] =
    {
#if 0
        ps_5_0
        dcl_globalFlags refactoringAllowed
        dcl_constantbuffer cb0[1], immediateIndexed
        dcl_output o0.xyzw
        dcl_temps 1
        mov r0.xyzw, cb0[0].xyzw
        ibfe r0.xyzw, r0.xxxx, r0.yyyy, r0.zzzz
        mov o0.xyzw, r0.xyzw
        ret
#endif
        0x43425844, 0x347a9c0e, 0x3eff39a4, 0x3dd41cc5, 0xff87ec8d, 0x00000001, 0x000000fc, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000084, 0x00000050, 0x00000021,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x06000036, 0x001000f2, 0x00000000, 0x00208e46, 0x00000000, 0x00000000,
        0x0900008b, 0x001000f2, 0x00000000, 0x00100006, 0x00000000, 0x00100556, 0x00000000, 0x00100aa6,
        0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_ubfe_code[] =
    {
#if 0
        uint u;

        uint4 main() : SV_Target
        {
            return uint4((u & 0xf0) >> 4, (u & 0x7fffff00) >> 8, (u & 0xfe) >> 1, (u & 0x7fffffff) >> 1);
        }
#endif
        0x43425844, 0xc4ac0509, 0xaea83154, 0xf1fb3b80, 0x4c22e3cc, 0x00000001, 0x000000e4, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x0000006c, 0x00000050, 0x0000001b,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x1000008a, 0x001020f2, 0x00000000, 0x00004002, 0x00000004, 0x00000017, 0x00000007, 0x0000001e,
        0x00004002, 0x00000004, 0x00000008, 0x00000001, 0x00000001, 0x00208006, 0x00000000, 0x00000000,
        0x0100003e,
    };
    static const DWORD ps_ubfe2_code[] =
    {
#if 0
       ps_5_0
       dcl_globalFlags refactoringAllowed
       dcl_constantbuffer cb0[1], immediateIndexed
       dcl_output o0.xyzw
       dcl_temps 1
       mov r0.xyzw, cb0[0].xyzw
       ubfe r0.xyzw, r0.xxxx, r0.yyyy, r0.zzzz
       mov o0.xyzw, r0.xyzw
       ret
#endif
        0x43425844, 0xf377b7fc, 0x1e4cb9e7, 0xb9b1020d, 0xde484388, 0x00000001, 0x000000fc, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000084, 0x00000050, 0x00000021,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x06000036, 0x001000f2, 0x00000000, 0x00208e46, 0x00000000, 0x00000000,
        0x0900008a, 0x001000f2, 0x00000000, 0x00100006, 0x00000000, 0x00100556, 0x00000000, 0x00100aa6,
        0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_bfrev_code[] =
    {
#if 0
        uint bits;

        uint4 main() : SV_Target
        {
            return uint4(reversebits(bits), reversebits(reversebits(bits)),
                    reversebits(bits & 0xFFFF), reversebits(bits >> 16));
        }
#endif
        0x43425844, 0x73daef82, 0xe52befa3, 0x8504d5f0, 0xebdb321d, 0x00000001, 0x00000154, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x000000dc, 0x00000050, 0x00000037,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x08000001, 0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000,
        0x00004001, 0x0000ffff, 0x0500008d, 0x00102042, 0x00000000, 0x0010000a, 0x00000000, 0x08000055,
        0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x00004001, 0x00000010, 0x0500008d,
        0x00102082, 0x00000000, 0x0010000a, 0x00000000, 0x0600008d, 0x00100012, 0x00000000, 0x0020800a,
        0x00000000, 0x00000000, 0x0500008d, 0x00102022, 0x00000000, 0x0010000a, 0x00000000, 0x05000036,
        0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_bits_code[] =
    {
#if 0
        uint u;
        int i;

        uint4 main() : SV_Target
        {
            return uint4(countbits(u), firstbitlow(u), firstbithigh(u), firstbithigh(i));
        }
#endif
        0x43425844, 0x23fee911, 0x145287d1, 0xea904419, 0x8aa59a6a, 0x00000001, 0x000001b4, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x0000013c, 0x00000050, 0x0000004f,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x06000089, 0x00100012, 0x00000000, 0x0020801a, 0x00000000, 0x00000000,
        0x07000020, 0x00100022, 0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0xffffffff, 0x0800001e,
        0x00100012, 0x00000000, 0x00004001, 0x0000001f, 0x8010000a, 0x00000041, 0x00000000, 0x09000037,
        0x00102082, 0x00000000, 0x0010001a, 0x00000000, 0x00004001, 0xffffffff, 0x0010000a, 0x00000000,
        0x06000087, 0x00100012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x0800001e, 0x00100012,
        0x00000000, 0x00004001, 0x0000001f, 0x8010000a, 0x00000041, 0x00000000, 0x0a000037, 0x00102042,
        0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0xffffffff,
        0x06000086, 0x00102012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x06000088, 0x00102022,
        0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_ftou_code[] =
    {
#if 0
        float f;

        uint4 main() : SV_Target
        {
            return uint4(f, -f, 0, 0);
        }
#endif
        0x43425844, 0xfde0ee2d, 0x812b339a, 0xb9fc36d2, 0x5820bec6, 0x00000001, 0x000000f4, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x0000007c, 0x00000040, 0x0000001f,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x0600001c,
        0x00102012, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x0700001c, 0x00102022, 0x00000000,
        0x8020800a, 0x00000041, 0x00000000, 0x00000000, 0x08000036, 0x001020c2, 0x00000000, 0x00004002,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_f16tof32_code[] =
    {
#if 0
        uint4 hf;

        uint4 main() : SV_Target
        {
            return f16tof32(hf);
        }
#endif
        0x43425844, 0xc1816e6e, 0x27562d96, 0x56980fa2, 0x421e6640, 0x00000001, 0x000000d8, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000060, 0x00000050, 0x00000018,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x06000083, 0x001000f2, 0x00000000, 0x00208e46, 0x00000000, 0x00000000,
        0x0500001c, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_f32tof16_code[] =
    {
#if 0
        float4 f;

        uint4 main() : SV_Target
        {
            return f32tof16(f);
        }
#endif
        0x43425844, 0x523a765c, 0x1a5be3a9, 0xaed69c80, 0xd26fe296, 0x00000001, 0x000000bc, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000044, 0x00000050, 0x00000011,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x06000082, 0x001020f2, 0x00000000, 0x00208e46, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const DWORD ps_not_code[] =
    {
#if 0
        uint2 bits;

        uint4 main() : SV_Target
        {
            return uint4(~bits.x, ~(bits.x ^ ~0u), ~bits.y, ~(bits.y ^ ~0u));
        }
#endif
        0x43425844, 0xaed0fd26, 0xf719a878, 0xc832efd6, 0xba03c264, 0x00000001, 0x00000100, 0x00000003,
        0x0000002c, 0x0000003c, 0x00000070, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000088, 0x00000040, 0x00000022,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0b000057, 0x00100032, 0x00000000, 0x00208046, 0x00000000, 0x00000000, 0x00004002,
        0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x0500003b, 0x001020a2, 0x00000000, 0x00100406,
        0x00000000, 0x0600003b, 0x00102052, 0x00000000, 0x00208106, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const struct shader ps_bfi = {ps_bfi_code, sizeof(ps_bfi_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_bfi2 = {ps_bfi2_code, sizeof(ps_bfi2_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_ibfe = {ps_ibfe_code, sizeof(ps_ibfe_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_ibfe2 = {ps_ibfe2_code, sizeof(ps_ibfe2_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_ubfe = {ps_ubfe_code, sizeof(ps_ubfe_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_ubfe2 = {ps_ubfe2_code, sizeof(ps_ubfe2_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_bfrev = {ps_bfrev_code, sizeof(ps_bfrev_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_bits = {ps_bits_code, sizeof(ps_bits_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_ftou = {ps_ftou_code, sizeof(ps_ftou_code), D3D_FEATURE_LEVEL_10_0};
    static const struct shader ps_f16tof32 = {ps_f16tof32_code, sizeof(ps_f16tof32_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_f32tof16 = {ps_f32tof16_code, sizeof(ps_f32tof16_code), D3D_FEATURE_LEVEL_11_0};
    static const struct shader ps_not = {ps_not_code, sizeof(ps_not_code), D3D_FEATURE_LEVEL_10_0};
    static const struct
    {
        const struct shader *ps;
        unsigned int bits[4];
        struct uvec4 expected_result;
    }
    tests[] =
    {
        {&ps_bfi, {     0,      0,    0,    0}, {         0,          0,          0,          0}},
        {&ps_bfi, {     0,      0,    0,    1}, {         1,          1,          1,          1}},
        {&ps_bfi, {   ~0u,      0,  ~0u,    0}, {0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff}},
        {&ps_bfi, {   ~0u,    ~0u,  ~0u,    0}, {0x80000000, 0x80000000, 0x80000000, 0x80000000}},
        {&ps_bfi, {   ~0u,  0x1fu,  ~0u,    0}, {0x80000000, 0x80000000, 0x80000000, 0x80000000}},
        {&ps_bfi, {   ~0u, ~0x1fu,  ~0u,    0}, {0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff}},
        {&ps_bfi, {     0,      0, 0xff,    1}, {         1,          1,          1,          1}},
        {&ps_bfi, {     0,      0, 0xff,    2}, {         2,          2,          2,          2}},
        {&ps_bfi, {    16,     16, 0xff, 0xff}, {0x00ff00ff, 0x00ff00ff, 0x00ff00ff, 0x00ff00ff}},
        {&ps_bfi, {     0,      0,  ~0u,  ~0u}, {       ~0u,        ~0u,        ~0u,        ~0u}},
        {&ps_bfi, {~0x1fu,      0,  ~0u,    0}, {         0,          0,          0,          0}},
        {&ps_bfi, {~0x1fu,      0,  ~0u,    1}, {         1,          1,          1,          1}},
        {&ps_bfi, {~0x1fu,      0,  ~0u,    2}, {         2,          2,          2,          2}},
        {&ps_bfi, {     0, ~0x1fu,  ~0u,    0}, {         0,          0,          0,          0}},
        {&ps_bfi, {     0, ~0x1fu,  ~0u,    1}, {         1,          1,          1,          1}},
        {&ps_bfi, {     0, ~0x1fu,  ~0u,    2}, {         2,          2,          2,          2}},
        {&ps_bfi, {~0x1fu, ~0x1fu,  ~0u,    0}, {         0,          0,          0,          0}},
        {&ps_bfi, {~0x1fu, ~0x1fu,  ~0u,    1}, {         1,          1,          1,          1}},
        {&ps_bfi, {~0x1fu, ~0x1fu,  ~0u,    2}, {         2,          2,          2,          2}},

        {&ps_bfi2, {   ~0u,      0,  ~0u,    0}, {0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff}},
        {&ps_bfi2, {   ~0u,    ~0u,  ~0u,    0}, {0x80000000, 0x80000000, 0x80000000, 0x80000000}},
        {&ps_bfi2, {   ~0u,  0x1fu,  ~0u,    0}, {0x80000000, 0x80000000, 0x80000000, 0x80000000}},
        {&ps_bfi2, {   ~0u, ~0x1fu,  ~0u,    0}, {0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff}},

        {&ps_ibfe, { 0,  4, 0x00000000}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ibfe, { 0,  4, 0xffffffff}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ibfe, { 0,  4, 0x7fffffff}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ibfe, { 4,  0, 0x00000000}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ibfe, { 4,  0, 0xfffffffa}, {0xfffffffa, 0xfffffffa, 0xfffffffa, 0xfffffffa}},
        {&ps_ibfe, { 4,  0, 0x7ffffffc}, {0xfffffffc, 0xfffffffc, 0xfffffffc, 0xfffffffc}},
        {&ps_ibfe, { 4,  4, 0x00000000}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ibfe, { 4,  4, 0xffffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, { 4,  4, 0xffffff1f}, {0x00000001, 0x00000001, 0x00000001, 0x00000001}},
        {&ps_ibfe, { 4,  4, 0x7fffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {23,  8, 0x00000000}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ibfe, {23,  8, 0xffffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {23,  8, 0x7fffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {30,  1, 0x00000000}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ibfe, {30,  1, 0xffffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {30,  1, 0x7fffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {15, 15, 0x7fffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {15, 15, 0x3fffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {15, 15, 0x1fffffff}, {0x00003fff, 0x00003fff, 0x00003fff, 0x00003fff}},
        {&ps_ibfe, {15, 15, 0xffff00ff}, {0xfffffffe, 0xfffffffe, 0xfffffffe, 0xfffffffe}},
        {&ps_ibfe, {16, 15, 0xffffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {16, 15, 0x3fffffff}, {0x00007fff, 0x00007fff, 0x00007fff, 0x00007fff}},
        {&ps_ibfe, {20, 15, 0xffffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {31, 31, 0xffffffff}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {31, 31, 0x80000000}, {0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff}},
        {&ps_ibfe, {31, 31, 0x7fffffff}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},

        {&ps_ibfe2, {16, 15, 0x3fffffff}, {0x00007fff, 0x00007fff, 0x00007fff, 0x00007fff}},

        {&ps_ubfe, {0x00000000}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ubfe, {0xffffffff}, {0x0000000f, 0x007fffff, 0x0000007f, 0x3fffffff}},
        {&ps_ubfe, {0xff000000}, {0x00000000, 0x007f0000, 0x00000000, 0x3f800000}},
        {&ps_ubfe, {0x00ff0000}, {0x00000000, 0x0000ff00, 0x00000000, 0x007f8000}},
        {&ps_ubfe, {0x000000ff}, {0x0000000f, 0x00000000, 0x0000007f, 0x0000007f}},
        {&ps_ubfe, {0x80000001}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},
        {&ps_ubfe, {0xc0000003}, {0x00000000, 0x00400000, 0x00000001, 0x20000001}},

        {&ps_ubfe2, { 4,  4, 0x7fffffff}, {0x0000000f, 0x0000000f, 0x0000000f, 0x0000000f}},
        {&ps_ubfe2, {23,  8, 0xffffffff}, {0x007fffff, 0x007fffff, 0x007fffff, 0x007fffff}},
        {&ps_ubfe2, {30,  1, 0xffffffff}, {0x3fffffff, 0x3fffffff, 0x3fffffff, 0x3fffffff}},
        {&ps_ubfe2, {15, 15, 0x7fffffff}, {0x00007fff, 0x00007fff, 0x00007fff, 0x00007fff}},
        {&ps_ubfe2, {15, 15, 0xffff00ff}, {0x00007ffe, 0x00007ffe, 0x00007ffe, 0x00007ffe}},
        {&ps_ubfe2, {16, 15, 0xffffffff}, {0x0000ffff, 0x0000ffff, 0x0000ffff, 0x0000ffff}},
        {&ps_ubfe2, {16, 15, 0x3fffffff}, {0x00007fff, 0x00007fff, 0x00007fff, 0x00007fff}},
        {&ps_ubfe2, {20, 15, 0xffffffff}, {0x0001ffff, 0x0001ffff, 0x0001ffff, 0x0001ffff}},
        {&ps_ubfe2, {31, 31, 0xffffffff}, {0x00000001, 0x00000001, 0x00000001, 0x00000001}},
        {&ps_ubfe2, {31, 31, 0x80000000}, {0x00000001, 0x00000001, 0x00000001, 0x00000001}},
        {&ps_ubfe2, {31, 31, 0x7fffffff}, {0x00000000, 0x00000000, 0x00000000, 0x00000000}},

        {&ps_bfrev, {0x12345678}, {0x1e6a2c48, 0x12345678, 0x1e6a0000, 0x2c480000}},
        {&ps_bfrev, {0xffff0000}, {0x0000ffff, 0xffff0000, 0x00000000, 0xffff0000}},
        {&ps_bfrev, {0xffffffff}, {0xffffffff, 0xffffffff, 0xffff0000, 0xffff0000}},

        {&ps_bits, {         0,          0}, { 0, ~0u, ~0u, ~0u}},
        {&ps_bits, {       ~0u,        ~0u}, {32,   0,  31, ~0u}},
        {&ps_bits, {0x7fffffff, 0x7fffffff}, {31,   0,  30,  30}},
        {&ps_bits, {0x80000000, 0x80000000}, { 1,  31,  31,  30}},
        {&ps_bits, {0x00000001, 0x00000001}, { 1,   0,   0,   0}},
        {&ps_bits, {0x80000001, 0x80000001}, { 2,   0,  31,  30}},
        {&ps_bits, {0x88888888, 0x88888888}, { 8,   3,  31,  30}},
        {&ps_bits, {0xcccccccc, 0xcccccccc}, {16,   2,  31,  29}},
        {&ps_bits, {0x11111111, 0x11111c11}, { 8,   0,  28,  28}},
        {&ps_bits, {0x0000000f, 0x0000000f}, { 4,   0,   3,   3}},
        {&ps_bits, {0x8000000f, 0x8000000f}, { 5,   0,  31,  30}},
        {&ps_bits, {0x00080000, 0x00080000}, { 1,  19,  19,  19}},

        {&ps_ftou, {BITS_NNAN}, { 0,  0}},
        {&ps_ftou, {BITS_NAN},  { 0,  0}},
        {&ps_ftou, {BITS_NINF}, { 0, ~0u}},
        {&ps_ftou, {BITS_INF},  {~0u, 0}},
        {&ps_ftou, {BITS_N1_0}, { 0,  1}},
        {&ps_ftou, {BITS_1_0},  { 1,  0}},

        {&ps_f16tof32, {0x00000000, 0x00003c00, 0x00005640, 0x00005bd0}, {0, 1, 100, 250}},
        {&ps_f16tof32, {0x00010000, 0x00013c00, 0x00015640, 0x00015bd0}, {0, 1, 100, 250}},
        {&ps_f16tof32, {0x000f0000, 0x000f3c00, 0x000f5640, 0x000f5bd0}, {0, 1, 100, 250}},
        {&ps_f16tof32, {0xffff0000, 0xffff3c00, 0xffff5640, 0xffff5bd0}, {0, 1, 100, 250}},

        {&ps_f32tof16, {0, BITS_1_0, BITS_N1_0, 0x44268000}, {0, 0x3c00, 0xbc00, 0x6134}},

        {&ps_not, {0x00000000, 0xffffffff}, {0xffffffff, 0x00000000, 0x00000000, 0xffffffff}},
        {&ps_not, {0xf0f0f0f0, 0x0f0f0f0f}, {0x0f0f0f0f, 0xf0f0f0f0, 0xf0f0f0f0, 0x0f0f0f0f}},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;
    feature_level = ID3D11Device_GetFeatureLevel(device);

    cb = create_buffer(device, D3D11_BIND_CONSTANT_BUFFER, sizeof(tests[0].bits), NULL);
    ID3D11DeviceContext_PSSetConstantBuffers(context, 0, 1, &cb);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    texture_desc.Format = DXGI_FORMAT_R32G32B32A32_UINT;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        if (feature_level < tests[i].ps->required_feature_level)
            continue;

        hr = ID3D11Device_CreatePixelShader(device, tests[i].ps->code, tests[i].ps->size, NULL, &ps);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);
        ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

        ID3D11DeviceContext_UpdateSubresource(context, (ID3D11Resource *)cb, 0, NULL, tests[i].bits, 0, 0);

        draw_quad(&test_context);
        check_texture_uvec4(texture, &tests[i].expected_result);

        ID3D11PixelShader_Release(ps);
    }

    ID3D11Buffer_Release(cb);
    ID3D11Texture2D_Release(texture);
    ID3D11RenderTargetView_Release(rtv);
    release_test_context(&test_context);
}

static void test_index_buffer_offset(void)
{
    ID3D11Buffer *vb, *ib, *so_buffer, *args_buffer;
    struct d3d11_test_context test_context;
    ID3D11InputLayout *input_layout;
    ID3D11DeviceContext *context;
    struct resource_readback rb;
    ID3D11GeometryShader *gs;
    const struct vec4 *data;
    ID3D11VertexShader *vs;
    ID3D11Device *device;
    UINT stride, offset;
    unsigned int i;
    HRESULT hr;

    static const D3D_FEATURE_LEVEL feature_level = D3D_FEATURE_LEVEL_11_0;
    static const DWORD vs_code[] =
    {
#if 0
        void main(float4 position : SV_POSITION, float4 attrib : ATTRIB,
                out float4 out_position : SV_Position, out float4 out_attrib : ATTRIB)
        {
            out_position = position;
            out_attrib = attrib;
        }
#endif
        0x43425844, 0xd7716716, 0xe23207f3, 0xc8af57c0, 0x585e2919, 0x00000001, 0x00000144, 0x00000003,
        0x0000002c, 0x00000080, 0x000000d4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52545441, 0xab004249,
        0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x505f5653, 0x7469736f, 0x006e6f69, 0x52545441, 0xab004249, 0x52444853, 0x00000068, 0x00010040,
        0x0000001a, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x001010f2, 0x00000001, 0x04000067,
        0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x05000036, 0x001020f2, 0x00000001, 0x00101e46, 0x00000001,
        0x0100003e,
    };
    static const DWORD gs_code[] =
    {
#if 0
        struct vertex
        {
            float4 position : SV_POSITION;
            float4 attrib : ATTRIB;
        };

        [maxvertexcount(1)]
        void main(point vertex input[1], inout PointStream<vertex> output)
        {
            output.Append(input[0]);
            output.RestartStrip();
        }
#endif
        0x43425844, 0x3d1dc497, 0xdf450406, 0x284ab03b, 0xa4ec0fd6, 0x00000001, 0x00000170, 0x00000003,
        0x0000002c, 0x00000080, 0x000000d4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x00000f0f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x52545441, 0xab004249,
        0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x505f5653, 0x5449534f, 0x004e4f49, 0x52545441, 0xab004249, 0x52444853, 0x00000094, 0x00020040,
        0x00000025, 0x05000061, 0x002010f2, 0x00000001, 0x00000000, 0x00000001, 0x0400005f, 0x002010f2,
        0x00000001, 0x00000001, 0x0100085d, 0x0100085c, 0x04000067, 0x001020f2, 0x00000000, 0x00000001,
        0x03000065, 0x001020f2, 0x00000001, 0x0200005e, 0x00000001, 0x06000036, 0x001020f2, 0x00000000,
        0x00201e46, 0x00000000, 0x00000000, 0x06000036, 0x001020f2, 0x00000001, 0x00201e46, 0x00000000,
        0x00000001, 0x01000013, 0x01000009, 0x0100003e,
    };
    static const D3D11_INPUT_ELEMENT_DESC input_desc[] =
    {
        {"SV_POSITION", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"ATTRIB",      0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 16, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
    static const D3D11_SO_DECLARATION_ENTRY so_declaration[] =
    {
        {0, "SV_Position", 0, 0, 4, 0},
        {0, "ATTRIB",      0, 0, 4, 0},
    };
    static const struct
    {
        struct vec4 position;
        struct vec4 attrib;
    }
    vertices[] =
    {
        {{-1.0f, -1.0f, 0.0f, 1.0f}, {1.0f}},
        {{-1.0f,  1.0f, 0.0f, 1.0f}, {2.0f}},
        {{ 1.0f, -1.0f, 0.0f, 1.0f}, {3.0f}},
        {{ 1.0f,  1.0f, 0.0f, 1.0f}, {4.0f}},
    };
    static const unsigned int indices[] =
    {
        0, 1, 2, 3,
        3, 2, 1, 0,
        1, 3, 2, 0,
    };
    static const struct vec4 expected_data[] =
    {
        {-1.0f, -1.0f, 0.0f, 1.0f}, {1.0f},
        {-1.0f,  1.0f, 0.0f, 1.0f}, {2.0f},
        { 1.0f, -1.0f, 0.0f, 1.0f}, {3.0f},
        { 1.0f,  1.0f, 0.0f, 1.0f}, {4.0f},

        { 1.0f,  1.0f, 0.0f, 1.0f}, {4.0f},
        { 1.0f, -1.0f, 0.0f, 1.0f}, {3.0f},
        {-1.0f,  1.0f, 0.0f, 1.0f}, {2.0f},
        {-1.0f, -1.0f, 0.0f, 1.0f}, {1.0f},

        {-1.0f,  1.0f, 0.0f, 1.0f}, {2.0f},
        { 1.0f,  1.0f, 0.0f, 1.0f}, {4.0f},
        { 1.0f, -1.0f, 0.0f, 1.0f}, {3.0f},
        {-1.0f, -1.0f, 0.0f, 1.0f}, {1.0f},
    };
    static const struct vec4 broken_result = {0.0f, 0.0f, 0.0f, 1.0f};
    static const D3D11_DRAW_INDEXED_INSTANCED_INDIRECT_ARGS argument_data[] =
    {
        {4, 1, 0, 0, 0},
    };

    if (!init_test_context(&test_context, &feature_level))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    hr = ID3D11Device_CreateInputLayout(device, input_desc, ARRAY_SIZE(input_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    stride = 32;
    hr = ID3D11Device_CreateGeometryShaderWithStreamOutput(device, gs_code, sizeof(gs_code),
            so_declaration, ARRAY_SIZE(so_declaration),
            &stride, 1, D3D11_SO_NO_RASTERIZED_STREAM, NULL, &gs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(vertices), vertices);
    ib = create_buffer(device, D3D11_BIND_INDEX_BUFFER, sizeof(indices), indices);
    so_buffer = create_buffer(device, D3D11_BIND_STREAM_OUTPUT, 1024, NULL);

    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);
    ID3D11DeviceContext_GSSetShader(context, gs, NULL, 0);

    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_POINTLIST);
    stride = sizeof(*vertices);
    offset = 0;
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb, &stride, &offset);

    offset = 0;
    ID3D11DeviceContext_SOSetTargets(context, 1, &so_buffer, &offset);

    ID3D11DeviceContext_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, 0);
    ID3D11DeviceContext_DrawIndexed(context, 4, 0, 0);

    ID3D11DeviceContext_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, 4 * sizeof(*indices));
    ID3D11DeviceContext_DrawIndexed(context, 4, 0, 0);

    ID3D11DeviceContext_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, 8 * sizeof(*indices));
    ID3D11DeviceContext_DrawIndexed(context, 4, 0, 0);

    get_buffer_readback(so_buffer, &rb);
    for (i = 0; i < ARRAY_SIZE(expected_data); ++i)
    {
        data = get_readback_vec4(&rb, i, 0);
        ok(compare_vec4(data, &expected_data[i], 0)
                || broken(is_nvidia_device(device) && !(i % 2) && compare_vec4(data, &broken_result, 0)),
                "Got unexpected result {%.8e, %.8e, %.8e, %.8e} at %u.\n",
                data->x, data->y, data->z, data->w, i);
    }
    release_resource_readback(&rb);

    /* indirect draws */
    args_buffer = create_buffer_misc(device, 0, D3D11_RESOURCE_MISC_DRAWINDIRECT_ARGS,
            sizeof(argument_data), argument_data);

    offset = 0;
    ID3D11DeviceContext_SOSetTargets(context, 1, &so_buffer, &offset);

    ID3D11DeviceContext_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, 0);
    ID3D11DeviceContext_DrawIndexedInstancedIndirect(context, args_buffer, 0);

    ID3D11DeviceContext_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, 4 * sizeof(*indices));
    ID3D11DeviceContext_DrawIndexedInstancedIndirect(context, args_buffer, 0);

    ID3D11DeviceContext_IASetIndexBuffer(context, ib, DXGI_FORMAT_R32_UINT, 8 * sizeof(*indices));
    ID3D11DeviceContext_DrawIndexedInstancedIndirect(context, args_buffer, 0);

    get_buffer_readback(so_buffer, &rb);
    for (i = 0; i < ARRAY_SIZE(expected_data); ++i)
    {
        data = get_readback_vec4(&rb, i, 0);
        todo_wine_if(i >= 8 && i != 20 && i != 21)
        ok(compare_vec4(data, &expected_data[i], 0)
                || broken(is_nvidia_device(device) && !(i % 2) && compare_vec4(data, &broken_result, 0)),
                "Got unexpected result {%.8e, %.8e, %.8e, %.8e} at %u.\n",
                data->x, data->y, data->z, data->w, i);
    }
    release_resource_readback(&rb);

    ID3D11Buffer_Release(so_buffer);
    ID3D11Buffer_Release(args_buffer);
    ID3D11Buffer_Release(ib);
    ID3D11Buffer_Release(vb);
    ID3D11VertexShader_Release(vs);
    ID3D11GeometryShader_Release(gs);
    ID3D11InputLayout_Release(input_layout);
    release_test_context(&test_context);
}

static void test_face_culling(void)
{
    struct d3d11_test_context test_context;
    D3D11_RASTERIZER_DESC rasterizer_desc;
    ID3D11RasterizerState *state;
    ID3D11DeviceContext *context;
    ID3D11Buffer *cw_vb, *ccw_vb;
    ID3D11Device *device;
    BOOL broken_warp;
    unsigned int i;
    HRESULT hr;

    static const struct vec4 red = {1.0f, 0.0f, 0.0f, 1.0f};
    static const struct vec4 green = {0.0f, 1.0f, 0.0f, 1.0f};
    static const DWORD ps_code[] =
    {
#if 0
        float4 main(uint front : SV_IsFrontFace) : SV_Target
        {
            return (front == ~0u) ? float4(0.0f, 1.0f, 0.0f, 1.0f) : float4(0.0f, 0.0f, 1.0f, 1.0f);
        }
#endif
        0x43425844, 0x92002fad, 0xc5c620b9, 0xe7a154fb, 0x78b54e63, 0x00000001, 0x00000128, 0x00000003,
        0x0000002c, 0x00000064, 0x00000098, 0x4e475349, 0x00000030, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000009, 0x00000001, 0x00000000, 0x00000101, 0x495f5653, 0x6f724673, 0x6146746e,
        0xab006563, 0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x00000088,
        0x00000040, 0x00000022, 0x04000863, 0x00101012, 0x00000000, 0x00000009, 0x03000065, 0x001020f2,
        0x00000000, 0x02000068, 0x00000001, 0x07000020, 0x00100012, 0x00000000, 0x0010100a, 0x00000000,
        0x00004001, 0xffffffff, 0x0f000037, 0x001020f2, 0x00000000, 0x00100006, 0x00000000, 0x00004002,
        0x00000000, 0x3f800000, 0x00000000, 0x3f800000, 0x00004002, 0x00000000, 0x00000000, 0x3f800000,
        0x3f800000, 0x0100003e,
    };
    static const struct vec3 ccw_quad[] =
    {
        {-1.0f,  1.0f, 0.0f},
        {-1.0f, -1.0f, 0.0f},
        { 1.0f,  1.0f, 0.0f},
        { 1.0f, -1.0f, 0.0f},
    };
    static const struct
    {
        D3D11_CULL_MODE cull_mode;
        BOOL front_ccw;
        BOOL expected_cw;
        BOOL expected_ccw;
    }
    tests[] =
    {
        {D3D11_CULL_NONE,  FALSE, TRUE,  TRUE},
        {D3D11_CULL_NONE,  TRUE,  TRUE,  TRUE},
        {D3D11_CULL_FRONT, FALSE, FALSE, TRUE},
        {D3D11_CULL_FRONT, TRUE,  TRUE,  FALSE},
        {D3D11_CULL_BACK,  FALSE, TRUE,  FALSE},
        {D3D11_CULL_BACK,  TRUE,  FALSE, TRUE},
    };

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 0);

    cw_vb = test_context.vb;
    ccw_vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(ccw_quad), ccw_quad);

    test_context.vb = ccw_vb;
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    check_texture_color(test_context.backbuffer, 0xff0000ff, 0);

    rasterizer_desc.FillMode = D3D11_FILL_SOLID;
    rasterizer_desc.CullMode = D3D11_CULL_BACK;
    rasterizer_desc.FrontCounterClockwise = FALSE;
    rasterizer_desc.DepthBias = 0;
    rasterizer_desc.DepthBiasClamp = 0.0f;
    rasterizer_desc.SlopeScaledDepthBias = 0.0f;
    rasterizer_desc.DepthClipEnable = TRUE;
    rasterizer_desc.ScissorEnable = FALSE;
    rasterizer_desc.MultisampleEnable = FALSE;
    rasterizer_desc.AntialiasedLineEnable = FALSE;

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        rasterizer_desc.CullMode = tests[i].cull_mode;
        rasterizer_desc.FrontCounterClockwise = tests[i].front_ccw;
        hr = ID3D11Device_CreateRasterizerState(device, &rasterizer_desc, &state);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        ID3D11DeviceContext_RSSetState(context, state);

        test_context.vb = cw_vb;
        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
        draw_color_quad(&test_context, &green);
        check_texture_color(test_context.backbuffer, tests[i].expected_cw ? 0xff00ff00 : 0xff0000ff, 0);

        test_context.vb = ccw_vb;
        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
        draw_color_quad(&test_context, &green);
        check_texture_color(test_context.backbuffer, tests[i].expected_ccw ? 0xff00ff00 : 0xff0000ff, 0);

        ID3D11RasterizerState_Release(state);
    }

    broken_warp = is_warp_device(device) && ID3D11Device_GetFeatureLevel(device) < D3D_FEATURE_LEVEL_11_0;

    /* Test SV_IsFrontFace. */
    ID3D11PixelShader_Release(test_context.ps);
    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &test_context.ps);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    rasterizer_desc.CullMode = D3D11_CULL_NONE;
    rasterizer_desc.FrontCounterClockwise = FALSE;
    hr = ID3D11Device_CreateRasterizerState(device, &rasterizer_desc, &state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_RSSetState(context, state);

    test_context.vb = cw_vb;
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 0);
    test_context.vb = ccw_vb;
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    if (!broken_warp)
        check_texture_color(test_context.backbuffer, 0xffff0000, 0);
    else
        win_skip("Broken WARP.\n");

    ID3D11RasterizerState_Release(state);

    rasterizer_desc.CullMode = D3D11_CULL_NONE;
    rasterizer_desc.FrontCounterClockwise = TRUE;
    hr = ID3D11Device_CreateRasterizerState(device, &rasterizer_desc, &state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_RSSetState(context, state);

    test_context.vb = cw_vb;
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    if (!broken_warp)
        check_texture_color(test_context.backbuffer, 0xffff0000 , 0);
    else
        win_skip("Broken WARP.\n");
    test_context.vb = ccw_vb;
    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    check_texture_color(test_context.backbuffer, 0xff00ff00, 0);

    ID3D11RasterizerState_Release(state);

    test_context.vb = cw_vb;
    ID3D11Buffer_Release(ccw_vb);
    release_test_context(&test_context);
}

static void test_line_antialiasing_blending(void)
{
    ID3D11RasterizerState *rasterizer_state;
    struct d3d11_test_context test_context;
    D3D11_RASTERIZER_DESC rasterizer_desc;
    ID3D11BlendState *blend_state;
    ID3D11DeviceContext *context;
    D3D11_BLEND_DESC blend_desc;
    ID3D11Device *device;
    HRESULT hr;

    static const struct vec4 red = {1.0f, 0.0f, 0.0f, 0.8f};
    static const struct vec4 green = {0.0f, 1.0f, 0.0f, 0.5f};

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    memset(&blend_desc, 0, sizeof(blend_desc));
    blend_desc.AlphaToCoverageEnable = FALSE;
    blend_desc.IndependentBlendEnable = FALSE;
    blend_desc.RenderTarget[0].BlendEnable = TRUE;
    blend_desc.RenderTarget[0].SrcBlend = D3D11_BLEND_SRC_ALPHA;
    blend_desc.RenderTarget[0].DestBlend = D3D11_BLEND_DEST_ALPHA;
    blend_desc.RenderTarget[0].BlendOp = D3D11_BLEND_OP_ADD;
    blend_desc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_SRC_ALPHA;
    blend_desc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_DEST_ALPHA;
    blend_desc.RenderTarget[0].BlendOpAlpha = D3D11_BLEND_OP_ADD;
    blend_desc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;

    hr = ID3D11Device_CreateBlendState(device, &blend_desc, &blend_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_OMSetBlendState(context, blend_state, NULL, D3D11_DEFAULT_SAMPLE_MASK);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    check_texture_color(test_context.backbuffer, 0xe2007fcc, 1);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &green.x);
    draw_color_quad(&test_context, &red);
    check_texture_color(test_context.backbuffer, 0xe2007fcc, 1);

    ID3D11DeviceContext_OMSetBlendState(context, NULL, NULL, D3D11_DEFAULT_SAMPLE_MASK);
    ID3D11BlendState_Release(blend_state);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    check_texture_color(test_context.backbuffer, 0x7f00ff00, 1);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &green.x);
    draw_color_quad(&test_context, &red);
    check_texture_color(test_context.backbuffer, 0xcc0000ff, 1);

    rasterizer_desc.FillMode = D3D11_FILL_SOLID;
    rasterizer_desc.CullMode = D3D11_CULL_BACK;
    rasterizer_desc.FrontCounterClockwise = FALSE;
    rasterizer_desc.DepthBias = 0;
    rasterizer_desc.DepthBiasClamp = 0.0f;
    rasterizer_desc.SlopeScaledDepthBias = 0.0f;
    rasterizer_desc.DepthClipEnable = TRUE;
    rasterizer_desc.ScissorEnable = FALSE;
    rasterizer_desc.MultisampleEnable = FALSE;
    rasterizer_desc.AntialiasedLineEnable = TRUE;

    hr = ID3D11Device_CreateRasterizerState(device, &rasterizer_desc, &rasterizer_state);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);
    ID3D11DeviceContext_RSSetState(context, rasterizer_state);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &red.x);
    draw_color_quad(&test_context, &green);
    check_texture_color(test_context.backbuffer, 0x7f00ff00, 1);

    ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, &green.x);
    draw_color_quad(&test_context, &red);
    check_texture_color(test_context.backbuffer, 0xcc0000ff, 1);

    ID3D11RasterizerState_Release(rasterizer_state);
    release_test_context(&test_context);
}

static void check_format_support(ID3D11Device *device, const unsigned int *format_support,
        D3D_FEATURE_LEVEL feature_level, const struct format_support *formats,
        unsigned int format_count, unsigned int feature_flag, const char *feature_name)
{
    unsigned int i;

    for (i = 0; i < format_count; ++i)
    {
        DXGI_FORMAT format = formats[i].format;
        unsigned int supported = format_support[format] & feature_flag;

        if (formats[i].fl_required <= feature_level)
        {
            todo_wine_if (feature_flag == D3D11_FORMAT_SUPPORT_DISPLAY)
                ok(supported || broken(is_warp_device(device)),
                        "Format %#x - %s not supported, format support %#x.\n",
                        format, feature_name, format_support[format]);
            continue;
        }

        if (formats[i].fl_optional && formats[i].fl_optional <= feature_level)
        {
            if (supported)
                trace("Optional format %#x - %s supported.\n", format, feature_name);
            continue;
        }

        todo_wine_if (feature_flag != D3D11_FORMAT_SUPPORT_DISPLAY)
            ok(!supported, "Format %#x - %s supported, format support %#x.\n",
                    format, feature_name, format_support[format]);
    }
}

static void test_format_support(const D3D_FEATURE_LEVEL feature_level)
{
    unsigned int format_support[DXGI_FORMAT_B4G4R4A4_UNORM + 1];
    D3D11_FEATURE_DATA_FORMAT_SUPPORT feature_data;
    struct device_desc device_desc;
    ID3D11Device *device;
    DXGI_FORMAT format;
    ULONG refcount;
    UINT support;
    HRESULT hr;

    static const struct format_support index_buffers[] =
    {
        {DXGI_FORMAT_R32_UINT, D3D_FEATURE_LEVEL_9_2},
        {DXGI_FORMAT_R16_UINT, D3D_FEATURE_LEVEL_9_1},
    };

    static const struct format_support vertex_buffers[] =
    {
        {DXGI_FORMAT_R8G8_UINT, D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R11G11B10_FLOAT, D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R16_FLOAT, D3D_FEATURE_LEVEL_10_0},
    };

    static const struct format_support blend[] =
    {
        {DXGI_FORMAT_R16G16B16A16_UNORM,    D3D_FEATURE_LEVEL_10_1, D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R16G16_UNORM,          D3D_FEATURE_LEVEL_10_1, D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R16_UNORM,             D3D_FEATURE_LEVEL_10_1, D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R16G16B16A16_SNORM,    D3D_FEATURE_LEVEL_10_1},
        {DXGI_FORMAT_R16G16_SNORM,          D3D_FEATURE_LEVEL_10_1},
        {DXGI_FORMAT_R16_SNORM,             D3D_FEATURE_LEVEL_10_1},
        {DXGI_FORMAT_R8G8B8A8_SNORM,        D3D_FEATURE_LEVEL_10_1},
        {DXGI_FORMAT_R8G8_SNORM,            D3D_FEATURE_LEVEL_10_1},
        {DXGI_FORMAT_R8_SNORM,              D3D_FEATURE_LEVEL_10_1},

        {DXGI_FORMAT_R32G32B32A32_FLOAT,    D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R32G32_FLOAT,          D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R32_FLOAT,             D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R10G10B10A2_UNORM,     D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R11G11B10_FLOAT,       D3D_FEATURE_LEVEL_10_0},
        {DXGI_FORMAT_R16_FLOAT,             D3D_FEATURE_LEVEL_10_0},

        {DXGI_FORMAT_R16G16B16A16_FLOAT,    D3D_FEATURE_LEVEL_9_3},

        {DXGI_FORMAT_R8G8B8A8_UNORM,        D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_R8G8B8A8_UNORM_SRGB,   D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_R8G8_UNORM,            D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_R8_UNORM,              D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_A8_UNORM,              D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_B5G6R5_UNORM,          D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_B8G8R8A8_UNORM,        D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_B8G8R8X8_UNORM,        D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_B8G8R8A8_UNORM_SRGB,   D3D_FEATURE_LEVEL_9_1},
        {DXGI_FORMAT_B8G8R8X8_UNORM_SRGB,   D3D_FEATURE_LEVEL_9_1},
    };

    device_desc.feature_level = &feature_level;
    device_desc.flags = 0;
    if (!(device = create_device(&device_desc)))
    {
        skip("Failed to create device.\n");
        return;
    }

    feature_data.InFormat = DXGI_FORMAT_R8G8B8A8_UNORM;

    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_FORMAT_SUPPORT, NULL, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_FORMAT_SUPPORT, &feature_data, 0);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_FORMAT_SUPPORT, &feature_data, sizeof(feature_data) - 1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_FORMAT_SUPPORT, &feature_data, sizeof(feature_data) / 2);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_FORMAT_SUPPORT, &feature_data, sizeof(feature_data) + 1);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_FORMAT_SUPPORT, &feature_data, sizeof(feature_data) * 2);
    ok(hr == E_INVALIDARG, "Got unexpected hr %#lx.\n", hr);

    support = 0xdeadbeef;
    hr = ID3D11Device_CheckFormatSupport(device, ~0u, &support);
    ok(hr == E_FAIL, "Got unexpected hr %#lx.\n", hr);
    ok(!support, "Got unexpected format support %#x.\n", support);

    feature_data.InFormat = ~0u;
    feature_data.OutFormatSupport = 0xdeadbeef;
    hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_FORMAT_SUPPORT, &feature_data, sizeof(feature_data));
    ok(hr == E_FAIL, "Got unexpected hr %#lx.\n", hr);
    ok(!feature_data.OutFormatSupport, "Got unexpected format support %#x.\n", feature_data.OutFormatSupport);

    memset(format_support, 0, sizeof(format_support));
    for (format = DXGI_FORMAT_UNKNOWN; format <= DXGI_FORMAT_B4G4R4A4_UNORM; ++format)
    {
        winetest_push_context("format %#x", format);

        hr = ID3D11Device_CheckFormatSupport(device, format, &format_support[format]);
        ok((hr == S_OK && format_support[format]) || (hr == E_FAIL && !format_support[format]),
                "Got unexpected hr %#lx, format_support %#x.\n", hr, format_support[format]);
        if (format_support[format] & D3D11_FORMAT_SUPPORT_MIP_AUTOGEN)
        {
            ok(format_support[format] & D3D11_FORMAT_SUPPORT_TEXTURE2D,
                    "Got unexpected format support %#x", format);
        }

        feature_data.InFormat = format;
        hr = ID3D11Device_CheckFeatureSupport(device, D3D11_FEATURE_FORMAT_SUPPORT,
                &feature_data, sizeof(feature_data));
        ok((hr == S_OK && feature_data.OutFormatSupport) || (hr == E_FAIL && !feature_data.OutFormatSupport),
                "Got unexpected hr %#lx, format_support %#x.\n", hr, feature_data.OutFormatSupport);
        ok(feature_data.OutFormatSupport == format_support[format], "Expected format support %#x, got %#x.\n",
                format_support[format], feature_data.OutFormatSupport);

        winetest_pop_context();
    }

    for (format = DXGI_FORMAT_UNKNOWN; format <= DXGI_FORMAT_B4G4R4A4_UNORM; ++format)
    {
        if (feature_level < D3D_FEATURE_LEVEL_10_0)
        {
            /* SHADER_SAMPLE_COMPARISON is never advertised as supported on feature level 9. */
            ok(!(format_support[format] & D3D11_FORMAT_SUPPORT_SHADER_SAMPLE_COMPARISON),
                    "Unexpected SHADER_SAMPLE_COMPARISON for format %#x.\n", format);
            ok(!(format_support[format] & D3D11_FORMAT_SUPPORT_BUFFER),
                    "Unexpected BUFFER for format %#x.\n", format);
        }
        if (feature_level < D3D_FEATURE_LEVEL_10_1)
        {
            ok(!(format_support[format] & D3D11_FORMAT_SUPPORT_SHADER_GATHER),
                    "Unexpected SHADER_GATHER for format %#x.\n", format);
            ok(!(format_support[format] & D3D11_FORMAT_SUPPORT_SHADER_GATHER_COMPARISON),
                    "Unexpected SHADER_GATHER_COMPARISON for format %#x.\n", format);
        }
    }

    ok(format_support[DXGI_FORMAT_R8G8B8A8_UNORM] & D3D11_FORMAT_SUPPORT_SHADER_SAMPLE,
            "SHADER_SAMPLE is not supported for R8G8B8A8_UNORM.\n");
    todo_wine
    ok(!(format_support[DXGI_FORMAT_R32G32B32A32_UINT] & D3D11_FORMAT_SUPPORT_SHADER_SAMPLE),
            "SHADER_SAMPLE is supported for R32G32B32A32_UINT.\n");
    if (feature_level >= D3D_FEATURE_LEVEL_10_0)
    {
        ok(format_support[DXGI_FORMAT_R32G32B32A32_UINT] & D3D11_FORMAT_SUPPORT_SHADER_LOAD,
                "SHADER_LOAD is not supported for R32G32B32A32_UINT.\n");
    }

    check_format_support(device, format_support, feature_level,
            index_buffers, ARRAY_SIZE(index_buffers),
            D3D11_FORMAT_SUPPORT_IA_INDEX_BUFFER, "index buffer");

    check_format_support(device, format_support, feature_level,
            vertex_buffers, ARRAY_SIZE(vertex_buffers),
            D3D11_FORMAT_SUPPORT_IA_VERTEX_BUFFER, "vertex buffer");

    check_format_support(device, format_support, feature_level,
            display_format_support, ARRAY_SIZE(display_format_support),
            D3D11_FORMAT_SUPPORT_DISPLAY, "display");

    check_format_support(device, format_support, feature_level, blend, ARRAY_SIZE(blend),
            D3D11_FORMAT_SUPPORT_BLENDABLE, "blendable");

    refcount = ID3D11Device_Release(device);
    ok(!refcount, "Device has %lu references left.\n", refcount);
}

static void test_fl9_draw(const D3D_FEATURE_LEVEL feature_level)
{
    struct d3d11_test_context test_context;
    D3D11_SUBRESOURCE_DATA resource_data;
    D3D11_TEXTURE2D_DESC texture_desc;
    ID3D11ShaderResourceView *srv;
    ID3D11DeviceContext *context;
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const struct vec4 color = {0.2f, 0.3f, 0.0f, 1.0f};
    static const DWORD ps_code[] =
    {
#if 0
        float4 main() : SV_TARGET
        {
            return float4(1.0f, 0.0f, 0.0f, 0.5f);
        }
#endif
        0x43425844, 0xb70eda74, 0xc9a7f982, 0xebc31bbf, 0x952a1360, 0x00000001, 0x00000168, 0x00000005,
        0x00000034, 0x0000008c, 0x000000e4, 0x00000124, 0x00000134, 0x53414e58, 0x00000050, 0x00000050,
        0xffff0200, 0x0000002c, 0x00000024, 0x00240000, 0x00240000, 0x00240000, 0x00240000, 0x00240000,
        0xffff0200, 0x05000051, 0xa00f0000, 0x3f800000, 0x00000000, 0x00000000, 0x3f000000, 0x02000001,
        0x800f0800, 0xa0e40000, 0x0000ffff, 0x396e6f41, 0x00000050, 0x00000050, 0xffff0200, 0x0000002c,
        0x00000024, 0x00240000, 0x00240000, 0x00240000, 0x00240000, 0x00240000, 0xffff0200, 0x05000051,
        0xa00f0000, 0x3f800000, 0x00000000, 0x00000000, 0x3f000000, 0x02000001, 0x800f0800, 0xa0e40000,
        0x0000ffff, 0x52444853, 0x00000038, 0x00000040, 0x0000000e, 0x03000065, 0x001020f2, 0x00000000,
        0x08000036, 0x001020f2, 0x00000000, 0x00004002, 0x3f800000, 0x00000000, 0x00000000, 0x3f000000,
        0x0100003e, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f, 0x0000002c, 0x00000001,
        0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x545f5653,
        0x45475241, 0xabab0054,
    };
    static const DWORD ps_texture_code[] =
    {
#if 0
        Texture2D t;
        SamplerState s;

        float4 main() : SV_TARGET
        {
            return t.Sample(s, (float2)0);
        }
#endif
        0x43425844, 0xf876c2db, 0x13725f1f, 0xcb6d3d65, 0x9994473f, 0x00000001, 0x000001d4, 0x00000005,
        0x00000034, 0x000000a0, 0x00000124, 0x00000190, 0x000001a0, 0x53414e58, 0x00000064, 0x00000064,
        0xffff0200, 0x0000003c, 0x00000028, 0x00280000, 0x00280000, 0x00280000, 0x00240001, 0x00280000,
        0x00000000, 0xffff0200, 0x05000051, 0xa00f0000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x0200001f, 0x90000000, 0xa00f0800, 0x03000042, 0x800f0800, 0xa0000000, 0xa0e40800, 0x0000ffff,
        0x396e6f41, 0x0000007c, 0x0000007c, 0xffff0200, 0x00000054, 0x00000028, 0x00280000, 0x00280000,
        0x00280000, 0x00240001, 0x00280000, 0x00000000, 0xffff0200, 0x05000051, 0xa00f0000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x0200001f, 0x90000000, 0xa00f0800, 0x02000001, 0x80030000,
        0xa0000000, 0x03000042, 0x800f0000, 0x80e40000, 0xa0e40800, 0x02000001, 0x800f0800, 0x80e40000,
        0x0000ffff, 0x52444853, 0x00000064, 0x00000040, 0x00000019, 0x0300005a, 0x00106000, 0x00000000,
        0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x03000065, 0x001020f2, 0x00000000, 0x0c000045,
        0x001020f2, 0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00107e46,
        0x00000000, 0x00106000, 0x00000000, 0x0100003e, 0x4e475349, 0x00000008, 0x00000000, 0x00000008,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x45475241, 0xabab0054,
    };
    static const DWORD texture_data[] = {0xffffff00};

    if (!init_test_context(&test_context, &feature_level))
        return;

    device = test_context.device;
    context = test_context.immediate_context;

    texture_desc.Width = 1;
    texture_desc.Height = 1;
    texture_desc.MipLevels = 0;
    texture_desc.ArraySize = 1;
    texture_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    texture_desc.SampleDesc.Count = 1;
    texture_desc.SampleDesc.Quality = 0;
    texture_desc.Usage = D3D11_USAGE_DEFAULT;
    texture_desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    texture_desc.CPUAccessFlags = 0;
    texture_desc.MiscFlags = 0;
    resource_data.pSysMem = texture_data;
    resource_data.SysMemPitch = sizeof(texture_data);
    resource_data.SysMemSlicePitch = 0;
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, &resource_data, &texture);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);
    hr = ID3D11Device_CreateShaderResourceView(device, (ID3D11Resource *)texture, NULL, &srv);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);

    hr = ID3D11Device_CreatePixelShader(device, ps_code, sizeof(ps_code), NULL, &ps);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0x7f0000ff, 1);
    ID3D11PixelShader_Release(ps);

    draw_color_quad(&test_context, &color);
    todo_wine check_texture_color(test_context.backbuffer, 0xff004c33, 1);

    hr = ID3D11Device_CreatePixelShader(device, ps_texture_code, sizeof(ps_texture_code), NULL, &ps);
    ok(hr == S_OK, "Got hr %#lx.\n", hr);
    ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);
    ID3D11DeviceContext_PSSetShaderResources(context, 0, 1, &srv);
    draw_quad(&test_context);
    check_texture_color(test_context.backbuffer, 0xffffff00, 1);
    ID3D11PixelShader_Release(ps);

    ID3D11ShaderResourceView_Release(srv);
    ID3D11Texture2D_Release(texture);
    release_test_context(&test_context);
}

static void queue_for_each_feature_level_in_range(D3D_FEATURE_LEVEL begin,
        D3D_FEATURE_LEVEL end, void (*test_func)(const D3D_FEATURE_LEVEL fl))
{
    static const D3D_FEATURE_LEVEL feature_levels[] =
    {
        D3D_FEATURE_LEVEL_11_1,
        D3D_FEATURE_LEVEL_11_0,
        D3D_FEATURE_LEVEL_10_1,
        D3D_FEATURE_LEVEL_10_0,
        D3D_FEATURE_LEVEL_9_3,
        D3D_FEATURE_LEVEL_9_2,
        D3D_FEATURE_LEVEL_9_1
    };
    unsigned int i;

    assert(begin <= end);
    for (i = 0; i < ARRAY_SIZE(feature_levels); ++i)
    {
        if (begin <= feature_levels[i] && feature_levels[i] <= end)
            queue_test_fl(test_func, feature_levels[i]);
    }
}

static void queue_for_each_feature_level(void (*test_func)(const D3D_FEATURE_LEVEL fl))
{
    queue_for_each_feature_level_in_range(D3D_FEATURE_LEVEL_9_1,
            D3D_FEATURE_LEVEL_11_1, test_func);
}

static void queue_for_each_9_x_feature_level(void (*test_func)(const D3D_FEATURE_LEVEL fl))
{
    queue_for_each_feature_level_in_range(D3D_FEATURE_LEVEL_9_1,
            D3D_FEATURE_LEVEL_9_3, test_func);
}

static void test_ddy(void)
{
    static const struct
    {
        struct vec4 position;
        unsigned int color;
    }
    quad[] =
    {
        {{-1.0f, -1.0f, 0.0f, 1.0f}, 0x00ff0000},
        {{-1.0f,  1.0f, 0.0f, 1.0f}, 0x0000ff00},
        {{ 1.0f, -1.0f, 0.0f, 1.0f}, 0x00ff0000},
        {{ 1.0f,  1.0f, 0.0f, 1.0f}, 0x0000ff00},
    };
    static const D3D11_INPUT_ELEMENT_DESC layout_desc[] =
    {
        {"SV_POSITION", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,  0, D3D11_INPUT_PER_VERTEX_DATA, 0},
        {"COLOR",       0, DXGI_FORMAT_R8G8B8A8_UNORM,     0, 16, D3D11_INPUT_PER_VERTEX_DATA, 0},
    };
#if 0
struct vs_data
{
    float4 pos : SV_POSITION;
    float4 color : COLOR;
};

void main(in struct vs_data vs_input, out struct vs_data vs_output)
{
    vs_output.pos = vs_input.pos;
    vs_output.color = vs_input.color;
}
#endif
    static const DWORD vs_code[] =
    {
        0x43425844, 0xd5b32785, 0x35332906, 0x4d05e031, 0xf66a58af, 0x00000001, 0x00000144, 0x00000003,
        0x0000002c, 0x00000080, 0x000000d4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000000, 0x00000003, 0x00000000, 0x00000f0f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000004c, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x0000000f,
        0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052, 0x52444853, 0x00000068, 0x00010040,
        0x0000001a, 0x0300005f, 0x001010f2, 0x00000000, 0x0300005f, 0x001010f2, 0x00000001, 0x04000067,
        0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000001, 0x05000036, 0x001020f2,
        0x00000000, 0x00101e46, 0x00000000, 0x05000036, 0x001020f2, 0x00000001, 0x00101e46, 0x00000001,
        0x0100003e,
    };
#if 0
struct ps_data
{
    float4 pos : SV_POSITION;
    float4 color : COLOR;
};

float4 main(struct ps_data ps_input) : SV_Target
{
    return ddy(ps_input.color) * 240.0 + 0.5;
}
#endif
    static const DWORD ps_code_ddy[] =
    {
        0x43425844, 0x423712f6, 0x786c59c2, 0xa6023c60, 0xb79faad2, 0x00000001, 0x00000138, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x52444853, 0x0000007c, 0x00000040,
        0x0000001f, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x02000068,
        0x00000001, 0x0500000c, 0x001000f2, 0x00000000, 0x00101e46, 0x00000001, 0x0f000032, 0x001020f2,
        0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0x43700000, 0x43700000, 0x43700000, 0x43700000,
        0x00004002, 0x3f000000, 0x3f000000, 0x3f000000, 0x3f000000, 0x0100003e,
    };
#if 0
struct ps_data
{
    float4 pos : SV_POSITION;
    float4 color : COLOR;
};

float4 main(struct ps_data ps_input) : SV_Target
{
    return ddy_coarse(ps_input.color) * 240.0 + 0.5;
}
#endif
    static const DWORD ps_code_ddy_coarse[] =
    {
        0x43425844, 0xbf9a31cb, 0xb42695b6, 0x629119b8, 0x6962d5dd, 0x00000001, 0x0000013c, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000080, 0x00000050,
        0x00000020, 0x0100086a, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x0500007c, 0x001000f2, 0x00000000, 0x00101e46, 0x00000001, 0x0f000032,
        0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0x43700000, 0x43700000, 0x43700000,
        0x43700000, 0x00004002, 0x3f000000, 0x3f000000, 0x3f000000, 0x3f000000, 0x0100003e,
    };
#if 0
struct ps_data
{
    float4 pos : SV_POSITION;
    float4 color : COLOR;
};

float4 main(struct ps_data ps_input) : SV_Target
{
    return ddy_fine(ps_input.color) * 240.0 + 0.5;
}
#endif
    static const DWORD ps_code_ddy_fine[] =
    {
        0x43425844, 0xea6563ae, 0x3ee0da50, 0x4c2b3ef3, 0xa69a4077, 0x00000001, 0x0000013c, 0x00000003,
        0x0000002c, 0x00000080, 0x000000b4, 0x4e475349, 0x0000004c, 0x00000002, 0x00000008, 0x00000038,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000,
        0x00000003, 0x00000001, 0x00000f0f, 0x505f5653, 0x5449534f, 0x004e4f49, 0x4f4c4f43, 0xabab0052,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x00000080, 0x00000050,
        0x00000020, 0x0100086a, 0x03001062, 0x001010f2, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x02000068, 0x00000001, 0x0500007d, 0x001000f2, 0x00000000, 0x00101e46, 0x00000001, 0x0f000032,
        0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x00004002, 0x43700000, 0x43700000, 0x43700000,
        0x43700000, 0x00004002, 0x3f000000, 0x3f000000, 0x3f000000, 0x3f000000, 0x0100003e,
    };
    static const struct
    {
        D3D_FEATURE_LEVEL min_feature_level;
        const DWORD *ps_code;
        unsigned int ps_code_size;
    }
    tests[] =
    {
        {D3D_FEATURE_LEVEL_10_0, ps_code_ddy, sizeof(ps_code_ddy)},
        {D3D_FEATURE_LEVEL_11_0, ps_code_ddy_coarse, sizeof(ps_code_ddy_coarse)},
        {D3D_FEATURE_LEVEL_11_0, ps_code_ddy_fine, sizeof(ps_code_ddy_fine)},
    };
    static const float red[] = {1.0f, 0.0f, 0.0f, 1.0f};
    struct d3d11_test_context test_context;
    D3D11_TEXTURE2D_DESC texture_desc;
    D3D_FEATURE_LEVEL feature_level;
    ID3D11InputLayout *input_layout;
    ID3D11DeviceContext *context;
    unsigned int stride, offset;
    struct resource_readback rb;
    ID3D11RenderTargetView *rtv;
    ID3D11Texture2D *texture;
    ID3D11VertexShader *vs;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    ID3D11Buffer *vb;
    unsigned int i;
    DWORD color;
    HRESULT hr;

    if (!init_test_context(&test_context, NULL))
        return;

    device = test_context.device;
    context = test_context.immediate_context;
    feature_level = ID3D11Device_GetFeatureLevel(device);

    ID3D11Texture2D_GetDesc(test_context.backbuffer, &texture_desc);
    hr = ID3D11Device_CreateTexture2D(device, &texture_desc, NULL, &texture);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateRenderTargetView(device, (ID3D11Resource *)texture, NULL, &rtv);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    hr = ID3D11Device_CreateInputLayout(device, layout_desc, ARRAY_SIZE(layout_desc),
            vs_code, sizeof(vs_code), &input_layout);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    vb = create_buffer(device, D3D11_BIND_VERTEX_BUFFER, sizeof(quad), quad);

    hr = ID3D11Device_CreateVertexShader(device, vs_code, sizeof(vs_code), NULL, &vs);
    ok(hr == S_OK, "Got unexpected hr %#lx.\n", hr);

    ID3D11DeviceContext_IASetInputLayout(context, input_layout);
    ID3D11DeviceContext_IASetPrimitiveTopology(context, D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    stride = sizeof(*quad);
    offset = 0;
    ID3D11DeviceContext_IASetVertexBuffers(context, 0, 1, &vb, &stride, &offset);
    ID3D11DeviceContext_VSSetShader(context, vs, NULL, 0);

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        if (feature_level < tests[i].min_feature_level)
        {
            skip("Skipping test %u, feature_level %#x lower than minimum required %#x.\n", i,
                    feature_level, tests[i].min_feature_level);
            continue;
        }

        hr = ID3D11Device_CreatePixelShader(device, tests[i].ps_code, tests[i].ps_code_size, NULL, &ps);
        ok(hr == S_OK, "Test %u: Got unexpected hr %#lx.\n", i, hr);

        ID3D11DeviceContext_PSSetShader(context, ps, NULL, 0);

        ID3D11DeviceContext_OMSetRenderTargets(context, 1, &rtv, NULL);
        ID3D11DeviceContext_ClearRenderTargetView(context, rtv, red);
        ID3D11DeviceContext_Draw(context, 4, 0);

        get_texture_readback(texture, 0, &rb);
        color = get_readback_color(&rb, 320, 190, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        color = get_readback_color(&rb, 255, 240, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        color = get_readback_color(&rb, 320, 240, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        color = get_readback_color(&rb, 385, 240, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        color = get_readback_color(&rb, 320, 290, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        release_resource_readback(&rb);

        ID3D11DeviceContext_OMSetRenderTargets(context, 1, &test_context.backbuffer_rtv, NULL);
        ID3D11DeviceContext_ClearRenderTargetView(context, test_context.backbuffer_rtv, red);
        ID3D11DeviceContext_Draw(context, 4, 0);

        get_texture_readback(test_context.backbuffer, 0, &rb);
        color = get_readback_color(&rb, 320, 190, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        color = get_readback_color(&rb, 255, 240, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        color = get_readback_color(&rb, 320, 240, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        color = get_readback_color(&rb, 385, 240, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        color = get_readback_color(&rb, 320, 290, 0);
        ok(compare_color(color, 0x7fff007f, 1), "Got unexpected colour 0x%08lx.\n", color);
        release_resource_readback(&rb);

        ID3D11PixelShader_Release(ps);
    }

    ID3D11VertexShader_Release(vs);
    ID3D11Buffer_Release(vb);
    ID3D11InputLayout_Release(input_layout);
    ID3D11Texture2D_Release(texture);
    ID3D11RenderTargetView_Release(rtv);
    release_test_context(&test_context);
}

static void test_shader_input_registers_limits(void)
{
    struct d3d11_test_context test_context;
    D3D11_SUBRESOURCE_DATA resource_data;
    D3D11_TEXTURE2D_DESC texture_desc;
    D3D11_SAMPLER_DESC sampler_desc;
    ID3D11ShaderResourceView *srv;
    ID3D11DeviceContext *context;
    ID3D11SamplerState *sampler;
    ID3D11Texture2D *texture;
    ID3D11PixelShader *ps;
    ID3D11Device *device;
    HRESULT hr;

    static const DWORD ps_last_register_code[] =
    {
#if 0
        Texture2D t : register(t127);
        SamplerState s : register(s15);

        void main(out float4 target : SV_Target)
        {
            target = t.Sample(s, float2(0, 0));
        }
#endif
        0x43425844, 0xd81ff2f8, 0x8c704b9c, 0x8c6f4857, 0xd02949ac, 0x0000000