/*
 * Unit tests for RtlNtStatusToDosError function
 *
 * Copyright (c) 2002 Andriy Palamarchuk
 * Copyright (c) 2010 André Hentschel
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <stdarg.h>
#include <stdio.h>

#include "ntstatus.h"
#define WIN32_NO_STATUS

#include "wine/test.h"

#include "windef.h"
#include "winbase.h"
#include "rpcnterr.h"
#include "winreg.h"
#include "winternl.h"

/* FIXME!!! this test checks only mappings, defined by MSDN
 * It is necessary to add other mappings and to test them
 * up to the latest Windows platform.
 *
 * Some Windows platforms don't know about all the mappings, and in such
 * cases they return somewhat strange results (Win98) or a generic error
 * like ERROR_MR_MID_NOT_FOUND (NT4). Our tests have to know about these to
 * not fail, but we would very much prefer Wine not to return such garbage.
 * So you can pass the 'strict' option to this test to force it to only check
 * results against the first listed value. This test should pass in strict
 * mode on the latest Windows platform and in Wine.
 * (of course older Windows platforms will fail to pass the strict mode)
 */

static ULONG (WINAPI *pRtlNtStatusToDosError)(NTSTATUS Status);
static int strict;

static BOOL prepare_test(void)
{
    HMODULE ntdll;
    int argc;
    char** argv;

    ntdll = LoadLibraryA("ntdll.dll");
    pRtlNtStatusToDosError = (void*)GetProcAddress(ntdll, "RtlNtStatusToDosError");
    if (!pRtlNtStatusToDosError)
    {
        win_skip("RtlNtStatusToDosError is not available\n");
        return FALSE;
    }

    argc = winetest_get_mainargs(&argv);
    strict=(argc >= 3 && strcmp(argv[2],"strict")==0);
    return TRUE;
}

static void cmp_call(NTSTATUS win_nt, ULONG win32, const char* message)
{
    ULONG err;

    err = pRtlNtStatusToDosError(win_nt);
    ok(err == win32,
       "%s (%x): got %u, expected %u\n",
            message, win_nt, err, win32);
}

static void cmp_call2(NTSTATUS win_nt, ULONG win32, const char* message)
{
    ULONG err;

    err = pRtlNtStatusToDosError(win_nt);
    ok(err == win32 ||
       (!strict && err == ERROR_MR_MID_NOT_FOUND),
       "%s (%x): got %u, expected %u (or MID_NOT_FOUND)\n",
       message, win_nt, err, win32);
}

static void cmp_call3(NTSTATUS win_nt, ULONG win32_1, ULONG win32_2, const char* message)
{
    ULONG err;

    err = pRtlNtStatusToDosError(win_nt);
    ok(err == win32_1 || (!strict && err == win32_2),
       "%s (%x): got %u, expected %u or %u\n",
            message, win_nt, err, win32_1, win32_2);
}

static void cmp_call4(NTSTATUS win_nt, ULONG win32_1, ULONG win32_2, const char* message)
{
    ULONG err;

    err = pRtlNtStatusToDosError(win_nt);
    ok(err == win32_1 ||
       (!strict && (err == win32_2 || err == ERROR_MR_MID_NOT_FOUND)),
       "%s (%x): got %u, expected %u or %u\n",
            message, win_nt, err, win32_1, win32_2);
}

#define cmp(status, error) \
        cmp_call(status, error, #status)
#define cmp2(status, error) \
        cmp_call2(status, error, #status)
#define cmp3(status, error1, error2) \
        cmp_call3(status, error1, error2, #status)
#define cmp4(status, error1, error2) \
        cmp_call4(status, error1, error2, #status)

static void run_error_tests(void)
{
    cmp(STATUS_DATATYPE_MISALIGNMENT,            ERROR_NOACCESS);
    cmp(STATUS_ACCESS_VIOLATION,                 ERROR_NOACCESS);
    cmp2(STATUS_DATATYPE_MISALIGNMENT_ERROR,     ERROR_NOACCESS);
    cmp2(STATUS_MULTIPLE_FAULT_VIOLATION,        ERROR_MULTIPLE_FAULT_VIOLATION);
    cmp(STATUS_CTL_FILE_NOT_SUPPORTED,           ERROR_NOT_SUPPORTED);
    cmp(STATUS_PORT_ALREADY_SET,                 ERROR_INVALID_PARAMETER);
    cmp2(STATUS_PORT_NOT_SET,                    ERROR_PORT_NOT_SET);
    cmp(STATUS_SECTION_NOT_IMAGE,                ERROR_INVALID_PARAMETER);
    cmp(STATUS_BAD_WORKING_SET_LIMIT,            ERROR_INVALID_PARAMETER);
    cmp(STATUS_WORKING_SET_LIMIT_RANGE,          ERROR_INVALID_PARAMETER);
    cmp(STATUS_INCOMPATIBLE_FILE_MAP,            ERROR_INVALID_PARAMETER);
    cmp(STATUS_PORT_DISCONNECTED,                ERROR_INVALID_HANDLE);
    cmp(STATUS_NOT_LOCKED,                       ERROR_NOT_LOCKED);
    cmp(STATUS_NOT_MAPPED_VIEW,                  ERROR_INVALID_ADDRESS);
    cmp(STATUS_UNABLE_TO_FREE_VM,                ERROR_INVALID_PARAMETER);
    cmp(STATUS_UNABLE_TO_DELETE_SECTION,         ERROR_INVALID_PARAMETER);
    cmp(STATUS_MORE_PROCESSING_REQUIRED,         ERROR_MORE_DATA);
    cmp(STATUS_INVALID_CID,                      ERROR_INVALID_PARAMETER);
    cmp(STATUS_STACK_OVERFLOW,                   ERROR_STACK_OVERFLOW);
    cmp(STATUS_BAD_INITIAL_STACK,                ERROR_STACK_OVERFLOW);
    cmp4(STATUS_STACK_OVERFLOW_READ,             ERROR_STACK_OVERFLOW_READ, ERROR_STACK_OVERFLOW);
    cmp(STATUS_INVALID_VOLUME_LABEL,             ERROR_LABEL_TOO_LONG);
    cmp(STATUS_SECTION_NOT_EXTENDED,             ERROR_OUTOFMEMORY);
    cmp(STATUS_NOT_MAPPED_DATA,                  ERROR_INVALID_ADDRESS);
    cmp2(STATUS_NO_LDT,                          ERROR_INVALID_THREAD_ID);
    cmp(STATUS_INFO_LENGTH_MISMATCH,             ERROR_BAD_LENGTH);
    cmp(STATUS_INVALID_INFO_CLASS,               ERROR_INVALID_PARAMETER);
    cmp(STATUS_SUSPEND_COUNT_EXCEEDED,           ERROR_SIGNAL_REFUSED);
    cmp(STATUS_NOTIFY_ENUM_DIR,                  ERROR_NOTIFY_ENUM_DIR);
    cmp2(STATUS_BAD_CURRENT_DIRECTORY,           ERROR_BAD_CURRENT_DIRECTORY);
    cmp(STATUS_REGISTRY_RECOVERED,               ERROR_REGISTRY_RECOVERED);
    cmp2(STATUS_REGISTRY_HIVE_RECOVERED,         ERROR_REGISTRY_HIVE_RECOVERED);
    cmp(STATUS_REGISTRY_IO_FAILED,               ERROR_REGISTRY_IO_FAILED);
    cmp(STATUS_NOT_REGISTRY_FILE,                ERROR_NOT_REGISTRY_FILE);
    cmp2(STATUS_CANNOT_LOAD_REGISTRY_FILE,       ERROR_CANNOT_LOAD_REGISTRY_FILE);
    cmp2(STATUS_SYSTEM_HIVE_TOO_LARGE,           ERROR_SYSTEM_HIVE_TOO_LARGE);
    cmp(STATUS_KEY_DELETED,                      ERROR_KEY_DELETED);
    cmp(STATUS_NO_LOG_SPACE,                     ERROR_NO_LOG_SPACE);
    cmp(STATUS_KEY_HAS_CHILDREN,                 ERROR_KEY_HAS_CHILDREN);
    cmp(STATUS_CHILD_MUST_BE_VOLATILE,           ERROR_CHILD_MUST_BE_VOLATILE);
    cmp(STATUS_REGISTRY_CORRUPT,                 ERROR_BADDB);
    cmp2(STATUS_REGISTRY_QUOTA_LIMIT,            ERROR_REGISTRY_QUOTA_LIMIT);
    cmp(STATUS_DLL_NOT_FOUND,                    ERROR_MOD_NOT_FOUND);
    cmp2(STATUS_OPEN_FAILED,                     ERROR_NET_OPEN_FAILED);
    cmp2(STATUS_IO_PRIVILEGE_FAILED,             ERROR_IO_PRIVILEGE_FAILED);
    cmp(STATUS_DLL_INIT_FAILED,                  ERROR_DLL_INIT_FAILED);
    cmp2(STATUS_DLL_INIT_FAILED_LOGOFF,          ERROR_DLL_INIT_FAILED_LOGOFF);
    cmp2(STATUS_INVALID_IMPORT_OF_NON_DLL,       ERROR_INVALID_IMPORT_OF_NON_DLL);
    cmp(STATUS_ORDINAL_NOT_FOUND,                ERROR_INVALID_ORDINAL);
    cmp(STATUS_DRIVER_ORDINAL_NOT_FOUND,         ERROR_INVALID_ORDINAL);
    cmp2(STATUS_DRIVER_UNABLE_TO_LOAD,           ERROR_BAD_DRIVER);
    cmp(STATUS_ENTRYPOINT_NOT_FOUND,             ERROR_PROC_NOT_FOUND);
    cmp(STATUS_DRIVER_ENTRYPOINT_NOT_FOUND,      ERROR_PROC_NOT_FOUND);
    cmp2(STATUS_BAD_DLL_ENTRYPOINT,              ERROR_BAD_DLL_ENTRYPOINT);
    cmp2(STATUS_BAD_SERVICE_ENTRYPOINT,          ERROR_BAD_SERVICE_ENTRYPOINT);
    cmp2(STATUS_FAILED_DRIVER_ENTRY,             ERROR_FAILED_DRIVER_ENTRY);
    cmp2(STATUS_DLL_MIGHT_BE_INSECURE,           ERROR_DLL_MIGHT_BE_INSECURE);
    cmp2(STATUS_DLL_MIGHT_BE_INCOMPATIBLE,       ERROR_DLL_MIGHT_BE_INCOMPATIBLE);
    cmp2(STATUS_ILLEGAL_DLL_RELOCATION,          ERROR_ILLEGAL_DLL_RELOCATION);
    cmp2(STATUS_NOT_SAFE_MODE_DRIVER,            ERROR_NOT_SAFE_MODE_DRIVER);
    cmp2(STATUS_DRIVER_DATABASE_ERROR,           ERROR_DRIVER_DATABASE_ERROR);
    cmp(STATUS_PENDING,                          ERROR_IO_PENDING);
    cmp(STATUS_MORE_ENTRIES,                     ERROR_MORE_DATA);
    cmp(STATUS_INTEGER_OVERFLOW,                 ERROR_ARITHMETIC_OVERFLOW);
    cmp(STATUS_BUFFER_OVERFLOW,                  ERROR_MORE_DATA);
    cmp2(STATUS_MARSHALL_OVERFLOW,               ERROR_MARSHALL_OVERFLOW);
    cmp(STATUS_NO_MORE_FILES,                    ERROR_NO_MORE_FILES);
    cmp2(STATUS_REDIRECTOR_HAS_OPEN_HANDLES,     ERROR_REDIRECTOR_HAS_OPEN_HANDLES);
    cmp2(STATUS_SERVER_HAS_OPEN_HANDLES,         ERROR_SERVER_HAS_OPEN_HANDLES);
    cmp2(STATUS_HANDLES_CLOSED,                  ERROR_HANDLES_CLOSED);
    cmp(STATUS_NO_INHERITANCE,                   ERROR_NO_INHERITANCE);
    cmp(STATUS_NO_MORE_EAS,                      ERROR_NO_MORE_ITEMS);
    cmp(STATUS_NO_MORE_ENTRIES,                  ERROR_NO_MORE_ITEMS);
    cmp(STATUS_GUIDS_EXHAUSTED,                  ERROR_NO_MORE_ITEMS);
    cmp(STATUS_AGENTS_EXHAUSTED,                 ERROR_NO_MORE_ITEMS);
    cmp(STATUS_UNSUCCESSFUL,                     ERROR_GEN_FAILURE);
    cmp(STATUS_TOO_MANY_LINKS,                   ERROR_TOO_MANY_LINKS);
    cmp(STATUS_NOT_IMPLEMENTED,                  ERROR_INVALID_FUNCTION);
    cmp(STATUS_ILLEGAL_FUNCTION,                 ERROR_INVALID_FUNCTION);
    cmp(STATUS_IN_PAGE_ERROR,                    ERROR_SWAPERROR);
    cmp(STATUS_PAGEFILE_QUOTA,                   ERROR_PAGEFILE_QUOTA);
    cmp2(STATUS_PAGEFILE_QUOTA_EXCEEDED,         ERROR_PAGEFILE_QUOTA_EXCEEDED);
    cmp2(STATUS_PAGE_FAULT_TRANSITION,           ERROR_PAGE_FAULT_TRANSITION);
    cmp2(STATUS_PAGE_FAULT_DEMAND_ZERO,          ERROR_PAGE_FAULT_DEMAND_ZERO);
    cmp2(STATUS_PAGE_FAULT_COPY_ON_WRITE,        ERROR_PAGE_FAULT_COPY_ON_WRITE);
    cmp2(STATUS_PAGE_FAULT_GUARD_PAGE,           ERROR_PAGE_FAULT_GUARD_PAGE);
    cmp2(STATUS_PAGE_FAULT_PAGING_FILE,          ERROR_PAGE_FAULT_PAGING_FILE);
    cmp2(STATUS_CACHE_PAGE_LOCKED,               ERROR_CACHE_PAGE_LOCKED);
    cmp2(STATUS_PAGEFILE_CREATE_FAILED,          ERROR_PAGEFILE_CREATE_FAILED);
    cmp2(STATUS_NO_PAGEFILE,                     ERROR_NO_PAGEFILE);
    cmp(STATUS_COMMITMENT_LIMIT,                 ERROR_COMMITMENT_LIMIT);
    cmp2(STATUS_COMMITMENT_MINIMUM,              ERROR_COMMITMENT_MINIMUM);
    cmp(STATUS_SECTION_TOO_BIG,                  ERROR_NOT_ENOUGH_MEMORY);
    cmp2(STATUS_SEGMENT_NOTIFICATION,            ERROR_SEGMENT_NOTIFICATION);
    cmp(RPC_NT_SS_IN_NULL_CONTEXT,               ERROR_INVALID_HANDLE);
    cmp(RPC_NT_INVALID_BINDING,                  ERROR_INVALID_HANDLE);
    cmp(STATUS_INVALID_HANDLE,                   ERROR_INVALID_HANDLE);
    cmp(STATUS_OBJECT_TYPE_MISMATCH,             ERROR_INVALID_HANDLE);
    cmp(STATUS_FILE_CLOSED,                      ERROR_INVALID_HANDLE);
    cmp(STATUS_INVALID_PORT_HANDLE,              ERROR_INVALID_HANDLE);
    cmp(STATUS_HANDLE_NOT_CLOSABLE,              ERROR_INVALID_HANDLE);
    cmp(STATUS_NOT_COMMITTED,                    ERROR_INVALID_ADDRESS);
    cmp(STATUS_PARTIAL_COPY,                     ERROR_PARTIAL_COPY);
    cmp3(STATUS_LPC_REPLY_LOST,                  ERROR_CONNECTION_ABORTED, ERROR_INTERNAL_ERROR);
    cmp2(STATUS_INVALID_VARIANT,                 ERROR_INVALID_VARIANT);
    cmp(STATUS_INVALID_PARAMETER,                ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_1,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_2,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_3,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_4,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_5,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_6,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_7,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_8,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_9,              ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_10,             ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_11,             ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_12,             ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PARAMETER_MIX,            ERROR_INVALID_PARAMETER);
    cmp(STATUS_INVALID_PAGE_PROTECTION,          ERROR_INVALID_PARAMETER);
    cmp(STATUS_SECTION_PROTECTION,               ERROR_INVALID_PARAMETER);
    cmp(STATUS_RESOURCE_DATA_NOT_FOUND,          ERROR_RESOURCE_DATA_NOT_FOUND);
    cmp(STATUS_RESOURCE_TYPE_NOT_FOUND,          ERROR_RESOURCE_TYPE_NOT_FOUND);
    cmp(STATUS_RESOURCE_NAME_NOT_FOUND,          ERROR_RESOURCE_NAME_NOT_FOUND);
    cmp(STATUS_RESOURCE_LANG_NOT_FOUND,          ERROR_RESOURCE_LANG_NOT_FOUND);
    cmp(STATUS_NO_SUCH_DEVICE,                   ERROR_FILE_NOT_FOUND);
    cmp(STATUS_NO_SUCH_FILE,                     ERROR_FILE_NOT_FOUND);
    cmp(STATUS_INVALID_DEVICE_REQUEST,           ERROR_INVALID_FUNCTION);
    cmp2(STATUS_VOLUME_NOT_UPGRADED,             ERROR_INVALID_FUNCTION);
    cmp2(STATUS_CONVERT_TO_LARGE,                ERROR_CONVERT_TO_LARGE);
    cmp(STATUS_END_OF_FILE,                      ERROR_HANDLE_EOF);
    cmp(STATUS_FILE_FORCED_CLOSED,               ERROR_HANDLE_EOF);
    cmp(STATUS_WRONG_VOLUME,                     ERROR_WRONG_DISK);
    cmp(STATUS_NO_MEDIA,                         ERROR_NO_MEDIA_IN_DRIVE);
    cmp(STATUS_NO_MEDIA_IN_DEVICE,               ERROR_NOT_READY);
    cmp2(STATUS_VOLUME_DISMOUNTED,               ERROR_NOT_READY);
    cmp2(STATUS_VOLUME_MOUNTED,                  ERROR_VOLUME_MOUNTED);
    cmp2(STATUS_MOUNT_POINT_NOT_RESOLVED,        ERROR_MOUNT_POINT_NOT_RESOLVED);
    cmp(STATUS_NONEXISTENT_SECTOR,               ERROR_SECTOR_NOT_FOUND);
    cmp(STATUS_WORKING_SET_QUOTA,                ERROR_WORKING_SET_QUOTA);
    cmp(STATUS_NO_MEMORY,                        ERROR_NOT_ENOUGH_MEMORY);
    cmp(STATUS_CONFLICTING_ADDRESSES,            ERROR_INVALID_ADDRESS);
    cmp(STATUS_INVALID_SYSTEM_SERVICE,           ERROR_INVALID_FUNCTION);
    cmp2(STATUS_THREAD_WAS_SUSPENDED,            ERROR_THREAD_WAS_SUSPENDED);
    cmp2(STATUS_TOO_MANY_THREADS,                ERROR_TOO_MANY_THREADS);
    cmp2(STATUS_THREAD_NOT_IN_PROCESS,           ERROR_THREAD_NOT_IN_PROCESS);
    cmp(STATUS_THREAD_IS_TERMINATING,            ERROR_ACCESS_DENIED);
    cmp(STATUS_PROCESS_IS_TERMINATING,           ERROR_ACCESS_DENIED);
    cmp2(STATUS_SYSTEM_PROCESS_TERMINATED,       ERROR_SYSTEM_PROCESS_TERMINATED);
    cmp(STATUS_INVALID_LOCK_SEQUENCE,            ERROR_ACCESS_DENIED);
    cmp(STATUS_INVALID_VIEW_SIZE,                ERROR_ACCESS_DENIED);
    cmp(STATUS_ALREADY_COMMITTED,                ERROR_ACCESS_DENIED);
    cmp(STATUS_ACCESS_DENIED,                    ERROR_ACCESS_DENIED);
    cmp(STATUS_FILE_IS_A_DIRECTORY,              ERROR_ACCESS_DENIED);
    cmp(STATUS_CANNOT_DELETE,                    ERROR_ACCESS_DENIED);
    cmp(STATUS_INVALID_COMPUTER_NAME,            ERROR_INVALID_COMPUTERNAME);
    cmp(STATUS_FILE_DELETED,                     ERROR_ACCESS_DENIED);
    cmp2(STATUS_FILE_RENAMED,                    ERROR_ACCESS_DENIED);
    cmp(STATUS_DELETE_PENDING,                   ERROR_ACCESS_DENIED);
    cmp(STATUS_PORT_CONNECTION_REFUSED,          ERROR_ACCESS_DENIED);
    cmp(STATUS_NO_SUCH_PRIVILEGE,                ERROR_NO_SUCH_PRIVILEGE);
    cmp(STATUS_PRIVILEGE_NOT_HELD,               ERROR_PRIVILEGE_NOT_HELD);
    cmp(STATUS_CANNOT_IMPERSONATE,               ERROR_CANNOT_IMPERSONATE);
    cmp(STATUS_LOGON_FAILURE,                    ERROR_LOGON_FAILURE);
    cmp2(STATUS_MUTUAL_AUTHENTICATION_FAILED,    ERROR_MUTUAL_AUTH_FAILED);
    cmp2(STATUS_TIME_DIFFERENCE_AT_DC,           ERROR_TIME_SKEW);
    cmp2(STATUS_PKINIT_FAILURE,                  ERROR_PKINIT_FAILURE);
    cmp2(STATUS_SMARTCARD_SUBSYSTEM_FAILURE,     ERROR_SMARTCARD_SUBSYSTEM_FAILURE);
    cmp2(STATUS_DOWNGRADE_DETECTED,              ERROR_DOWNGRADE_DETECTED);
    cmp4(STATUS_SMARTCARD_CERT_REVOKED,          SEC_E_SMARTCARD_CERT_REVOKED, 1266); /* FIXME: real name? */
    cmp4(STATUS_ISSUING_CA_UNTRUSTED,            SEC_E_ISSUING_CA_UNTRUSTED, 1267); /* FIXME: real name? */
    cmp4(STATUS_REVOCATION_OFFLINE_C,            SEC_E_REVOCATION_OFFLINE_C, 1268); /* FIXME: real name? */
    cmp4(STATUS_PKINIT_CLIENT_FAILURE,           SEC_E_PKINIT_CLIENT_FAILURE, 1269); /* FIXME: real name? */
    cmp4(STATUS_SMARTCARD_CERT_EXPIRED,          SEC_E_SMARTCARD_CERT_EXPIRED, 1270); /* FIXME: real name? */
    cmp2(STATUS_NO_KERB_KEY,                     SEC_E_NO_KERB_KEY);
    cmp2(STATUS_CURRENT_DOMAIN_NOT_ALLOWED,      ERROR_CURRENT_DOMAIN_NOT_ALLOWED);
    cmp2(STATUS_SMARTCARD_WRONG_PIN,             SCARD_W_WRONG_CHV);
    cmp2(STATUS_SMARTCARD_CARD_BLOCKED,          SCARD_W_CHV_BLOCKED);
    cmp2(STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED,SCARD_W_CARD_NOT_AUTHENTICATED);
    cmp2(STATUS_SMARTCARD_NO_CARD,               SCARD_E_NO_SMARTCARD);
    cmp2(STATUS_SMARTCARD_NO_KEY_CONTAINER,      NTE_NO_KEY);
    cmp2(STATUS_SMARTCARD_NO_CERTIFICATE,        SCARD_E_NO_SUCH_CERTIFICATE);
    cmp2(STATUS_SMARTCARD_NO_KEYSET,             NTE_BAD_KEYSET);
    cmp2(STATUS_SMARTCARD_IO_ERROR,              SCARD_E_COMM_DATA_LOST);
    cmp(STATUS_ACCOUNT_RESTRICTION,              ERROR_ACCOUNT_RESTRICTION);
    cmp(STATUS_INVALID_LOGON_HOURS,              ERROR_INVALID_LOGON_HOURS);
    cmp(STATUS_INVALID_WORKSTATION,              ERROR_INVALID_WORKSTATION);
    cmp(STATUS_BUFFER_TOO_SMALL,                 ERROR_INSUFFICIENT_BUFFER);
    cmp(STATUS_UNABLE_TO_DECOMMIT_VM,            ERROR_INVALID_ADDRESS);
    cmp(STATUS_DISK_CORRUPT_ERROR,               ERROR_DISK_CORRUPT);
    cmp2(STATUS_FT_READ_RECOVERY_FROM_BACKUP,    ERROR_FT_READ_RECOVERY_FROM_BACKUP);
    cmp2(STATUS_FT_WRITE_RECOVERY,               ERROR_FT_WRITE_RECOVERY);
    cmp(STATUS_FT_MISSING_MEMBER,                ERROR_IO_DEVICE);
    cmp(STATUS_FT_ORPHANING,                     ERROR_IO_DEVICE);
    cmp2(STATUS_BACKUP_CONTROLLER,               ERROR_BACKUP_CONTROLLER);
    cmp(STATUS_VARIABLE_NOT_FOUND,               ERROR_ENVVAR_NOT_FOUND);
    cmp(STATUS_OBJECT_NAME_INVALID,              ERROR_INVALID_NAME);
    cmp(STATUS_OBJECT_NAME_NOT_FOUND,            ERROR_FILE_NOT_FOUND);
    cmp(STATUS_OBJECT_NAME_COLLISION,            ERROR_ALREADY_EXISTS);
    cmp2(STATUS_OBJECT_NAME_EXISTS,              ERROR_OBJECT_NAME_EXISTS);
    cmp(STATUS_OBJECT_PATH_INVALID,              ERROR_BAD_PATHNAME);
    cmp(STATUS_OBJECT_PATH_NOT_FOUND,            ERROR_PATH_NOT_FOUND);
    cmp(STATUS_DFS_EXIT_PATH_FOUND,              ERROR_PATH_NOT_FOUND);
    cmp2(STATUS_DFS_UNAVAILABLE,                 ERROR_CONNECTION_UNAVAIL);
    cmp(STATUS_OBJECT_PATH_SYNTAX_BAD,           ERROR_BAD_PATHNAME);
    cmp(STATUS_NAME_TOO_LONG,                    ERROR_FILENAME_EXCED_RANGE);
    cmp(STATUS_DATA_OVERRUN,                     ERROR_IO_DEVICE);
    cmp(STATUS_DATA_LATE_ERROR,                  ERROR_IO_DEVICE);
    cmp(STATUS_DATA_ERROR,                       ERROR_CRC);
    cmp2(STATUS_DATA_NOT_ACCEPTED,               ERROR_DATA_NOT_ACCEPTED);
    cmp2(STATUS_LOST_WRITEBEHIND_DATA,           ERROR_LOST_WRITEBEHIND_DATA);
    cmp(STATUS_CRC_ERROR,                        ERROR_CRC);
    cmp(STATUS_SHARING_VIOLATION,                ERROR_SHARING_VIOLATION);
    cmp(STATUS_QUOTA_EXCEEDED,                   ERROR_NOT_ENOUGH_QUOTA);
    cmp2(STATUS_FS_DRIVER_REQUIRED,              ERROR_FS_DRIVER_REQUIRED);
    cmp(STATUS_MUTANT_NOT_OWNED,                 ERROR_NOT_OWNER);
    cmp2(STATUS_MUTANT_LIMIT_EXCEEDED,           ERROR_MUTANT_LIMIT_EXCEEDED);
    cmp(STATUS_SEMAPHORE_LIMIT_EXCEEDED,         ERROR_TOO_MANY_POSTS);
    cmp(STATUS_DISK_FULL,                        ERROR_DISK_FULL);
    cmp(STATUS_LOCK_NOT_GRANTED,                 ERROR_LOCK_VIOLATION);
    cmp(STATUS_FILE_LOCK_CONFLICT,               ERROR_LOCK_VIOLATION);
    cmp2(STATUS_NOT_TINY_STREAM,                 ERROR_NOT_TINY_STREAM);
    cmp(STATUS_NOT_A_DIRECTORY,                  ERROR_DIRECTORY);
    cmp2(STATUS_CANNOT_MAKE,                     ERROR_CANNOT_MAKE);
    cmp(STATUS_UNKNOWN_REVISION,                 ERROR_UNKNOWN_REVISION);
    cmp(STATUS_REVISION_MISMATCH,                ERROR_REVISION_MISMATCH);
    cmp(STATUS_INVALID_OWNER,                    ERROR_INVALID_OWNER);
    cmp(STATUS_INVALID_PRIMARY_GROUP,            ERROR_INVALID_PRIMARY_GROUP);
    cmp(STATUS_NO_IMPERSONATION_TOKEN,           ERROR_NO_IMPERSONATION_TOKEN);
    cmp(STATUS_CANT_DISABLE_MANDATORY,           ERROR_CANT_DISABLE_MANDATORY);
    cmp(STATUS_NO_LOGON_SERVERS,                 ERROR_NO_LOGON_SERVERS);
    cmp(STATUS_DOMAIN_CONTROLLER_NOT_FOUND,      ERROR_DOMAIN_CONTROLLER_NOT_FOUND);
    cmp(STATUS_NO_SUCH_LOGON_SESSION,            ERROR_NO_SUCH_LOGON_SESSION);
    cmp(STATUS_INVALID_ACCOUNT_NAME,             ERROR_INVALID_ACCOUNT_NAME);
    cmp(STATUS_USER_EXISTS,                      ERROR_USER_EXISTS);
    cmp(STATUS_NO_SUCH_USER,                     ERROR_NO_SUCH_USER);
    cmp(STATUS_GROUP_EXISTS,                     ERROR_GROUP_EXISTS);
    cmp(STATUS_NO_SUCH_GROUP,                    ERROR_NO_SUCH_GROUP);
    cmp(STATUS_SPECIAL_GROUP,                    ERROR_SPECIAL_GROUP);
    cmp(STATUS_MEMBER_IN_GROUP,                  ERROR_MEMBER_IN_GROUP);
    cmp(STATUS_MEMBER_NOT_IN_GROUP,              ERROR_MEMBER_NOT_IN_GROUP);
    cmp(STATUS_LAST_ADMIN,                       ERROR_LAST_ADMIN);
    cmp(STATUS_WRONG_PASSWORD,                   ERROR_INVALID_PASSWORD);
    cmp(STATUS_WRONG_PASSWORD_CORE,              ERROR_INVALID_PASSWORD);
    cmp(STATUS_ILL_FORMED_PASSWORD,              ERROR_ILL_FORMED_PASSWORD);
    cmp(STATUS_PASSWORD_RESTRICTION,             ERROR_PASSWORD_RESTRICTION);
    cmp(STATUS_PASSWORD_EXPIRED,                 ERROR_PASSWORD_EXPIRED);
    cmp(STATUS_PASSWORD_MUST_CHANGE,             ERROR_PASSWORD_MUST_CHANGE);
    cmp2(STATUS_PWD_TOO_SHORT,                   ERROR_PWD_TOO_SHORT);
    cmp2(STATUS_PWD_TOO_RECENT,                  ERROR_PWD_TOO_RECENT);
    cmp2(STATUS_PWD_HISTORY_CONFLICT,            ERROR_PWD_HISTORY_CONFLICT);
    cmp(STATUS_ACCOUNT_DISABLED,                 ERROR_ACCOUNT_DISABLED);
    cmp(STATUS_ACCOUNT_LOCKED_OUT,               ERROR_ACCOUNT_LOCKED_OUT);
    cmp(STATUS_NONE_MAPPED,                      ERROR_NONE_MAPPED);
    cmp(STATUS_TOO_MANY_LUIDS_REQUESTED,         ERROR_TOO_MANY_LUIDS_REQUESTED);
    cmp(STATUS_LUIDS_EXHAUSTED,                  ERROR_LUIDS_EXHAUSTED);
    cmp(STATUS_INVALID_SUB_AUTHORITY,            ERROR_INVALID_SUB_AUTHORITY);
    cmp(STATUS_INVALID_ACL,                      ERROR_INVALID_ACL);
    cmp(STATUS_INVALID_SID,                      ERROR_INVALID_SID);
    cmp(STATUS_INVALID_SECURITY_DESCR,           ERROR_INVALID_SECURITY_DESCR);
    cmp(STATUS_PROCEDURE_NOT_FOUND,              ERROR_PROC_NOT_FOUND);
    cmp2(STATUS_VDM_HARD_ERROR,                  ERROR_VDM_HARD_ERROR);
    cmp2(STATUS_INVALID_LDT_SIZE,                ERROR_INVALID_LDT_SIZE);
    cmp2(STATUS_INVALID_LDT_OFFSET,              ERROR_INVALID_LDT_OFFSET);
    cmp2(STATUS_INVALID_LDT_DESCRIPTOR,          ERROR_INVALID_LDT_DESCRIPTOR);
    cmp(STATUS_BAD_INITIAL_PC,                   ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_INVALID_FILE_FOR_SECTION,         ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_INVALID_IMAGE_FORMAT,             ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_IMAGE_MP_UP_MISMATCH,             ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_INVALID_IMAGE_NOT_MZ,             ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_IMAGE_CHECKSUM_MISMATCH,          ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_INVALID_IMAGE_PROTECT,            ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_INVALID_IMAGE_LE_FORMAT,          ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_INVALID_IMAGE_NE_FORMAT,          ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_INVALID_IMAGE_WIN_16,             ERROR_BAD_EXE_FORMAT);
    cmp2(STATUS_INVALID_IMAGE_WIN_32,            ERROR_BAD_EXE_FORMAT);
    cmp2(STATUS_INVALID_IMAGE_WIN_64,            ERROR_BAD_EXE_FORMAT);
    cmp(STATUS_NO_TOKEN,                         ERROR_NO_TOKEN);
    cmp(STATUS_RANGE_NOT_LOCKED,                 ERROR_NOT_LOCKED);
    cmp2(STATUS_RANGE_LIST_CONFLICT,             ERROR_RANGE_LIST_CONFLICT);
    cmp2(STATUS_RANGE_NOT_FOUND,                 ERROR_RANGE_NOT_FOUND);
    cmp(STATUS_SERVER_DISABLED,                  ERROR_SERVER_DISABLED);
    cmp(STATUS_SERVER_NOT_DISABLED,              ERROR_SERVER_NOT_DISABLED);
    cmp(STATUS_INVALID_ID_AUTHORITY,             ERROR_INVALID_ID_AUTHORITY);
    cmp(STATUS_ALLOTTED_SPACE_EXCEEDED,          ERROR_ALLOTTED_SPACE_EXCEEDED);
    cmp(STATUS_TOO_MANY_PAGING_FILES,            ERROR_NOT_ENOUGH_MEMORY);
    cmp2(STATUS_RESOURCE_REQUIREMENTS_CHANGED,   ERROR_RESOURCE_REQUIREMENTS_CHANGED);
    cmp(STATUS_INSUFFICIENT_RESOURCES,           ERROR_NO_SYSTEM_RESOURCES);
    cmp(STATUS_INSUFF_SERVER_RESOURCES,          ERROR_NOT_ENOUGH_SERVER_MEMORY);
    cmp(STATUS_FILE_INVALID,                     ERROR_FILE_INVALID);
    cmp(STATUS_MAPPED_FILE_SIZE_ZERO,            ERROR_FILE_INVALID);
    cmp(STATUS_DEVICE_PAPER_EMPTY,               ERROR_OUT_OF_PAPER);
    cmp(STATUS_DEVICE_POWERED_OFF,               ERROR_NOT_READY);
    cmp(STATUS_DEVICE_OFF_LINE,                  ERROR_NOT_READY);
    cmp(STATUS_DEVICE_DATA_ERROR,                ERROR_CRC);
    cmp(STATUS_DEVICE_NOT_READY,                 ERROR_NOT_READY);
    cmp3(STATUS_DEVICE_NOT_CONNECTED,            ERROR_DEVICE_NOT_CONNECTED, ERROR_NOT_READY);
    cmp(STATUS_DEVICE_POWER_FAILURE,             ERROR_NOT_READY);
    cmp2(STATUS_INSUFFICIENT_POWER,              ERROR_INSUFFICIENT_POWER);
    cmp2(STATUS_DRIVER_FAILED_SLEEP,             ERROR_DRIVER_FAILED_SLEEP);
    cmp2(STATUS_NOT_FOUND,                       ERROR_NOT_FOUND);
    cmp2(STATUS_NO_MATCH,                        ERROR_NO_MATCH);
    cmp2(STATUS_NO_MORE_MATCHES,                 ERROR_NO_MORE_MATCHES);
    cmp2(STATUS_PROPSET_NOT_FOUND,               ERROR_SET_NOT_FOUND);
    cmp2(STATUS_FOUND_OUT_OF_SCOPE,              ERROR_FOUND_OUT_OF_SCOPE);
    cmp(STATUS_DEVICE_BUSY,                      ERROR_BUSY);
    cmp(STATUS_FREE_VM_NOT_AT_BASE,              ERROR_INVALID_ADDRESS);
    cmp(STATUS_MEMORY_NOT_ALLOCATED,             ERROR_INVALID_ADDRESS);
    cmp2(STATUS_ALLOCATE_BUCKET,                 ERROR_ALLOCATE_BUCKET);
    cmp(STATUS_NOT_SAME_DEVICE,                  ERROR_NOT_SAME_DEVICE);
    cmp(STATUS_NOT_SUPPORTED,                    ERROR_NOT_SUPPORTED);
    cmp(STATUS_REMOTE_NOT_LISTENING,             ERROR_REM_NOT_LIST);
    cmp(STATUS_DUPLICATE_NAME,                   ERROR_DUP_NAME);
    cmp(STATUS_BAD_NETWORK_PATH,                 ERROR_BAD_NETPATH);
    cmp(STATUS_NETWORK_BUSY,                     ERROR_NETWORK_BUSY);
    cmp2(STATUS_ONLY_IF_CONNECTED,               ERROR_ONLY_IF_CONNECTED);
    cmp(STATUS_DEVICE_DOES_NOT_EXIST,            ERROR_DEV_NOT_EXIST);
    cmp(STATUS_TOO_MANY_COMMANDS,                ERROR_TOO_MANY_CMDS);
    cmp(STATUS_ADAPTER_HARDWARE_ERROR,           ERROR_ADAP_HDW_ERR);
    cmp2(STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT,ERROR_BIOS_FAILED_TO_CONNECT_INTERRUPT);
    cmp(STATUS_REDIRECTOR_NOT_STARTED,           ERROR_PATH_NOT_FOUND);
    cmp(STATUS_INVALID_EA_NAME,                  ERROR_INVALID_EA_NAME);
    cmp(STATUS_EA_LIST_INCONSISTENT,             ERROR_EA_LIST_INCONSISTENT);
    cmp(STATUS_EA_TOO_LARGE,                     ERROR_EA_LIST_INCONSISTENT);
    cmp(STATUS_INVALID_EA_FLAG,                  ERROR_EA_LIST_INCONSISTENT);
    cmp2(STATUS_EAS_NOT_SUPPORTED,               ERROR_EAS_NOT_SUPPORTED);
    cmp(STATUS_FILE_CORRUPT_ERROR,               ERROR_FILE_CORRUPT);
    cmp(STATUS_EA_CORRUPT_ERROR,                 ERROR_FILE_CORRUPT);
    cmp(STATUS_NONEXISTENT_EA_ENTRY,             ERROR_FILE_CORRUPT);
    cmp(STATUS_NO_EAS_ON_FILE,                   ERROR_FILE_CORRUPT);
    cmp2(STATUS_REPARSE,                         ERROR_REPARSE);
    cmp2(STATUS_REPARSE_OBJECT,                  ERROR_REPARSE_OBJECT);
    cmp2(STATUS_NOT_A_REPARSE_POINT,             ERROR_NOT_A_REPARSE_POINT);
    cmp4(STATUS_IO_REPARSE_TAG_INVALID,          ERROR_REPARSE_TAG_INVALID, ERROR_INVALID_PARAMETER);
    cmp4(STATUS_IO_REPARSE_TAG_MISMATCH,         ERROR_REPARSE_TAG_MISMATCH, ERROR_INVALID_PARAMETER);
    cmp2(STATUS_IO_REPARSE_TAG_NOT_HANDLED,      ERROR_CANT_ACCESS_FILE);
    cmp2(STATUS_REPARSE_POINT_NOT_RESOLVED,      ERROR_CANT_RESOLVE_FILENAME);
    cmp2(STATUS_DIRECTORY_IS_A_REPARSE_POINT,    ERROR_BAD_PATHNAME);
    cmp2(STATUS_REPARSE_ATTRIBUTE_CONFLICT,      ERROR_REPARSE_ATTRIBUTE_CONFLICT);
    cmp4(STATUS_IO_REPARSE_DATA_INVALID,         ERROR_INVALID_REPARSE_DATA, ERROR_INVALID_PARAMETER);
    cmp2(STATUS_FILE_IS_OFFLINE,                 ERROR_FILE_OFFLINE);
    cmp2(STATUS_REMOTE_STORAGE_NOT_ACTIVE,       ERROR_REMOTE_STORAGE_NOT_ACTIVE);
    cmp2(STATUS_REMOTE_STORAGE_MEDIA_ERROR,      ERROR_REMOTE_STORAGE_MEDIA_ERROR);
    cmp2(STATUS_NO_TRACKING_SERVICE,             ERROR_NO_TRACKING_SERVICE);
    cmp2(STATUS_JOURNAL_DELETE_IN_PROGRESS,      ERROR_JOURNAL_DELETE_IN_PROGRESS);
    cmp2(STATUS_JOURNAL_NOT_ACTIVE,              ERROR_JOURNAL_NOT_ACTIVE);
    cmp2(STATUS_JOURNAL_ENTRY_DELETED,           ERROR_JOURNAL_ENTRY_DELETED);
    cmp2(STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED,ERROR_PRIMARY_TRANSPORT_CONNECT_FAILED);
    cmp(STATUS_INVALID_NETWORK_RESPONSE,         ERROR_BAD_NET_RESP);
    cmp(STATUS_USER_SESSION_DELETED,             ERROR_UNEXP_NET_ERR);
    cmp(STATUS_UNEXPECTED_NETWORK_ERROR,         ERROR_UNEXP_NET_ERR);
    cmp(STATUS_BAD_REMOTE_ADAPTER,               ERROR_BAD_REM_ADAP);
    cmp(STATUS_PRINT_QUEUE_FULL,                 ERROR_PRINTQ_FULL);
    cmp(STATUS_NO_SPOOL_SPACE,                   ERROR_NO_SPOOL_SPACE);
    cmp(STATUS_PRINT_CANCELLED,                  ERROR_PRINT_CANCELLED);
    cmp(STATUS_NETWORK_NAME_DELETED,             ERROR_NETNAME_DELETED);
    cmp(STATUS_NETWORK_ACCESS_DENIED,            ERROR_NETWORK_ACCESS_DENIED);
    cmp(STATUS_BAD_DEVICE_TYPE,                  ERROR_BAD_DEV_TYPE);
    cmp(STATUS_BAD_NETWORK_NAME,                 ERROR_BAD_NET_NAME);
    cmp(STATUS_TOO_MANY_NAMES,                   ERROR_TOO_MANY_NAMES);
    cmp(STATUS_TOO_MANY_GUIDS_REQUESTED,         ERROR_TOO_MANY_NAMES);
    cmp(STATUS_TOO_MANY_ADDRESSES,               ERROR_TOO_MANY_NAMES);
    cmp(STATUS_TOO_MANY_NODES,                   ERROR_TOO_MANY_NAMES);
    cmp(STATUS_TOO_MANY_SESSIONS,                ERROR_TOO_MANY_SESS);
    cmp(STATUS_SHARING_PAUSED,                   ERROR_SHARING_PAUSED);
    cmp2(STATUS_CANT_ENABLE_DENY_ONLY,           ERROR_CANT_ENABLE_DENY_ONLY);
    cmp(STATUS_REQUEST_NOT_ACCEPTED,             ERROR_REQ_NOT_ACCEP);
    cmp(STATUS_REDIRECTOR_PAUSED,                ERROR_REDIR_PAUSED);
    cmp(STATUS_NET_WRITE_FAULT,                  ERROR_NET_WRITE_FAULT);
    cmp(STATUS_VIRTUAL_CIRCUIT_CLOSED,           ERROR_VC_DISCONNECTED);
    cmp(STATUS_INVALID_PIPE_STATE,               ERROR_BAD_PIPE);
    cmp(STATUS_INVALID_READ_MODE,                ERROR_BAD_PIPE);
    cmp(STATUS_PIPE_CLOSING,                     ERROR_NO_DATA);
    cmp(STATUS_PIPE_EMPTY,                       ERROR_NO_DATA);
    cmp(STATUS_PIPE_CONNECTED,                   ERROR_PIPE_CONNECTED);
    cmp(STATUS_PIPE_DISCONNECTED,                ERROR_PIPE_NOT_CONNECTED);
    cmp(STATUS_PIPE_LISTENING,                   ERROR_PIPE_LISTENING);
    cmp(STATUS_PIPE_NOT_AVAILABLE,               ERROR_PIPE_BUSY);
    cmp(STATUS_INSTANCE_NOT_AVAILABLE,           ERROR_PIPE_BUSY);
    cmp(STATUS_PIPE_BUSY,                        ERROR_PIPE_BUSY);
    cmp(STATUS_PIPE_BROKEN,                      ERROR_BROKEN_PIPE);
    cmp(STATUS_DIRECTORY_NOT_EMPTY,              ERROR_DIR_NOT_EMPTY);
    cmp(STATUS_TOO_MANY_OPENED_FILES,            ERROR_TOO_MANY_OPEN_FILES);
    cmp(STATUS_IO_TIMEOUT,                       ERROR_SEM_TIMEOUT);
    cmp(STATUS_CANCELLED,                        ERROR_OPERATION_ABORTED);
    cmp(STATUS_UNRECOGNIZED_MEDIA,               ERROR_UNRECOGNIZED_MEDIA);
    cmp(STATUS_INVALID_LEVEL,                    ERROR_INVALID_LEVEL);
    cmp(STATUS_UNRECOGNIZED_VOLUME,              ERROR_UNRECOGNIZED_VOLUME);
    cmp(STATUS_MEDIA_WRITE_PROTECTED,            ERROR_WRITE_PROTECT);
    cmp(STATUS_TOO_LATE,                         ERROR_WRITE_PROTECT);
    cmp(STATUS_SUCCESS,                          NO_ERROR);
    cmp(STATUS_FULLSCREEN_MODE,                  ERROR_FULLSCREEN_MODE);
    cmp(STATUS_END_OF_MEDIA,                     ERROR_END_OF_MEDIA);
    cmp(STATUS_EOM_OVERFLOW,                     ERROR_EOM_OVERFLOW);
    cmp(STATUS_BEGINNING_OF_MEDIA,               ERROR_BEGINNING_OF_MEDIA);
    cmp2(STATUS_MEDIA_CHECK,                     ERROR_MEDIA_CHECK);
    cmp(STATUS_MEDIA_CHANGED,                    ERROR_MEDIA_CHANGED);
    cmp(STATUS_BUS_RESET,                        ERROR_BUS_RESET);
    cmp(STATUS_FILEMARK_DETECTED,                ERROR_FILEMARK_DETECTED);
    cmp(STATUS_SETMARK_DETECTED,                 ERROR_SETMARK_DETECTED);
    cmp(STATUS_NO_DATA_DETECTED,                 ERROR_NO_DATA_DETECTED);
    cmp(STATUS_PARTITION_FAILURE,                ERROR_PARTITION_FAILURE);
    cmp(STATUS_INVALID_BLOCK_LENGTH,             ERROR_INVALID_BLOCK_LENGTH);
    cmp(STATUS_DEVICE_NOT_PARTITIONED,           ERROR_DEVICE_NOT_PARTITIONED);
    cmp(STATUS_UNABLE_TO_LOCK_MEDIA,             ERROR_UNABLE_TO_LOCK_MEDIA);
    cmp(STATUS_UNABLE_TO_UNLOAD_MEDIA,           ERROR_UNABLE_TO_UNLOAD_MEDIA);
    cmp2(STATUS_ILLEGAL_CHARACTER,               ERROR_ILLEGAL_CHARACTER);
    cmp(STATUS_UNMAPPABLE_CHARACTER,             ERROR_NO_UNICODE_TRANSLATION);
    cmp2(STATUS_UNDEFINED_CHARACTER,             ERROR_UNDEFINED_CHARACTER);
    cmp2(STATUS_TRANSLATION_COMPLETE,            ERROR_TRANSLATION_COMPLETE);
    cmp(STATUS_NOT_ALL_ASSIGNED,                 ERROR_NOT_ALL_ASSIGNED);
    cmp(STATUS_SOME_NOT_MAPPED,                  ERROR_SOME_NOT_MAPPED);
    cmp(STATUS_NO_QUOTAS_FOR_ACCOUNT,            ERROR_NO_QUOTAS_FOR_ACCOUNT);
    cmp2(STATUS_QUOTA_LIST_INCONSISTENT,         ERROR_QUOTA_LIST_INCONSISTENT);
    cmp(STATUS_LOCAL_USER_SESSION_KEY,           ERROR_LOCAL_USER_SESSION_KEY);
    cmp(STATUS_NULL_LM_PASSWORD,                 ERROR_NULL_LM_PASSWORD);
    cmp2(STATUS_SYNCHRONIZATION_REQUIRED,        ERROR_SYNCHRONIZATION_REQUIRED);
    cmp2(STATUS_WAKE_SYSTEM_DEBUGGER,            ERROR_WAKE_SYSTEM_DEBUGGER);
    cmp2(DBG_EXCEPTION_HANDLED,                  ERROR_DBG_EXCEPTION_HANDLED);
    cmp2(DBG_CONTINUE,                           ERROR_DBG_CONTINUE);
    cmp2(DBG_REPLY_LATER,                        ERROR_DBG_REPLY_LATER);
    cmp2(DBG_UNABLE_TO_PROVIDE_HANDLE,           ERROR_DBG_UNABLE_TO_PROVIDE_HANDLE);
    cmp2(DBG_TERMINATE_THREAD,                   ERROR_DBG_TERMINATE_THREAD);
    cmp2(DBG_TERMINATE_PROCESS,                  ERROR_DBG_TERMINATE_PROCESS);
    cmp2(DBG_CONTROL_C,                          ERROR_DBG_CONTROL_C);
    cmp2(DBG_PRINTEXCEPTION_C,                   ERROR_DBG_PRINTEXCEPTION_C);
    cmp2(DBG_RIPEXCEPTION,                       ERROR_DBG_RIPEXCEPTION);
    cmp2(DBG_CONTROL_BREAK,                      ERROR_DBG_CONTROL_BREAK);
    cmp2(DBG_COMMAND_EXCEPTION,                  ERROR_DBG_COMMAND_EXCEPTION);
    cmp2(DBG_PRINTEXCEPTION_WIDE_C,              ERROR_DBG_PRINTEXCEPTION_C);
    cmp2(DBG_EXCEPTION_NOT_HANDLED,              ERROR_DBG_EXCEPTION_NOT_HANDLED);
    cmp2(STATUS_DEBUG_ATTACH_FAILED,             ERROR_DEBUG_ATTACH_FAILED);
    cmp2(STATUS_VALIDATE_CONTINUE,               ERROR_VALIDATE_CONTINUE);
    cmp2(STATUS_EXTRANEOUS_INFORMATION,          ERROR_EXTRANEOUS_INFORMATION);
    cmp2(STATUS_GUID_SUBSTITUTION_MADE,          ERROR_GUID_SUBSTITUTION_MADE);
    cmp2(STATUS_IMAGE_MACHINE_TYPE_MISMATCH,     ERROR_IMAGE_MACHINE_TYPE_MISMATCH);
    cmp2(STATUS_RECEIVE_PARTIAL,                 ERROR_RECEIVE_PARTIAL);
    cmp2(STATUS_RECEIVE_EXPEDITED,               ERROR_RECEIVE_EXPEDITED);
    cmp2(STATUS_RECEIVE_PARTIAL_EXPEDITED,       ERROR_RECEIVE_PARTIAL_EXPEDITED);
    cmp2(STATUS_EVENT_DONE,                      ERROR_EVENT_DONE);
    cmp2(STATUS_EVENT_PENDING,                   ERROR_EVENT_PENDING);
    cmp2(STATUS_CHECKING_FILE_SYSTEM,            ERROR_CHECKING_FILE_SYSTEM);
    cmp2(STATUS_FATAL_APP_EXIT,                  ERROR_FATAL_APP_EXIT);
    cmp2(STATUS_PREDEFINED_HANDLE,               ERROR_PREDEFINED_HANDLE);
    cmp2(STATUS_WAS_UNLOCKED,                    ERROR_WAS_UNLOCKED);
    cmp2(STATUS_SERVICE_NOTIFICATION,            ERROR_SERVICE_NOTIFICATION);
    cmp2(STATUS_WAS_LOCKED,                      ERROR_WAS_LOCKED);
    cmp2(STATUS_LOG_HARD_ERROR,                  ERROR_LOG_HARD_ERROR);
    cmp2(STATUS_ALREADY_WIN32,                   ERROR_ALREADY_WIN32);
    cmp2(STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE, ERROR_IMAGE_MACHINE_TYPE_MISMATCH_EXE);
    cmp2(STATUS_NO_YIELD_PERFORMED,              ERROR_NO_YIELD_PERFORMED);
    cmp2(STATUS_TIMER_RESUME_IGNORED,            ERROR_TIMER_RESUME_IGNORED);
    cmp2(STATUS_TIMER_RESOLUTION_NOT_SET,        ERROR_TIMER_RESOLUTION_NOT_SET);
    cmp2(STATUS_ARBITRATION_UNHANDLED,           ERROR_ARBITRATION_UNHANDLED);
    cmp2(STATUS_CARDBUS_NOT_SUPPORTED,           ERROR_CARDBUS_NOT_SUPPORTED);
    cmp2(STATUS_MP_PROCESSOR_MISMATCH,           ERROR_MP_PROCESSOR_MISMATCH);
    cmp2(STATUS_HIBERNATED,                      ERROR_HIBERNATED);
    cmp2(STATUS_RESUME_HIBERNATION,              ERROR_RESUME_HIBERNATION);
    cmp2(STATUS_FIRMWARE_UPDATED,                ERROR_FIRMWARE_UPDATED);
    cmp2(STATUS_WAKE_SYSTEM,                     ERROR_WAKE_SYSTEM);
    cmp2(STATUS_WAIT_1,                          ERROR_WAIT_1);
    cmp2(STATUS_WAIT_2,                          ERROR_WAIT_2);
    cmp2(STATUS_WAIT_3,                          ERROR_WAIT_3);
    cmp2(STATUS_WAIT_63,                         ERROR_WAIT_63);
    cmp2(STATUS_ABANDONED_WAIT_0,                ERROR_WAIT_NO_CHILDREN);
    cmp2(STATUS_ABANDONED_WAIT_63,               ERROR_ABANDONED_WAIT_63);
    cmp2(STATUS_USER_APC,                        ERROR_USER_APC);
    cmp2(STATUS_KERNEL_APC,                      ERROR_KERNEL_APC);
    cmp2(STATUS_ALERTED,                         ERROR_ALERTED);
    cmp2(STATUS_NOTIFY_CLEANUP,                  ERROR_NOTIFY_CLEANUP);
    cmp2(STATUS_NO_CALLBACK_ACTIVE,              ERROR_NO_CALLBACK_ACTIVE);
    cmp4(STATUS_NOINTERFACE,                     ERROR_NOINTERFACE, E_NOINTERFACE);
    cmp(STATUS_BAD_INHERITANCE_ACL,              ERROR_BAD_INHERITANCE_ACL);
    cmp(STATUS_INVALID_GROUP_ATTRIBUTES,         ERROR_INVALID_GROUP_ATTRIBUTES);
    cmp(STATUS_BAD_IMPERSONATION_LEVEL,          ERROR_BAD_IMPERSONATION_LEVEL);
    cmp2(STATUS_ELEVATION_REQUIRED,              ERROR_ELEVATION_REQUIRED);
    cmp(STATUS_CANT_OPEN_ANONYMOUS,              ERROR_CANT_OPEN_ANONYMOUS);
    cmp(STATUS_BAD_VALIDATION_CLASS,             ERROR_BAD_VALIDATION_CLASS);
    cmp(STATUS_BAD_TOKEN_TYPE,                   ERROR_BAD_TOKEN_TYPE);
    cmp2(STATUS_BAD_MASTER_BOOT_RECORD,          ERROR_INVALID_PARAMETER);
    cmp(STATUS_NO_SECURITY_ON_OBJECT,            ERROR_NO_SECURITY_ON_OBJECT);
    cmp(STATUS_CANT_ACCESS_DOMAIN_INFO,          ERROR_CANT_ACCESS_DOMAIN_INFO);
    cmp(STATUS_INVALID_SERVER_STATE,             ERROR_INVALID_SERVER_STATE);
    cmp(STATUS_INVALID_DOMAIN_STATE,             ERROR_INVALID_DOMAIN_STATE);
    cmp(STATUS_INVALID_DOMAIN_ROLE,              ERROR_INVALID_DOMAIN_ROLE);
    cmp(STATUS_NO_SUCH_DOMAIN,                   ERROR_NO_SUCH_DOMAIN);
    cmp(STATUS_DOMAIN_EXISTS,                    ERROR_DOMAIN_EXISTS);
    cmp(STATUS_DOMAIN_LIMIT_EXCEEDED,            ERROR_DOMAIN_LIMIT_EXCEEDED);
    cmp2(STATUS_OPLOCK_NOT_GRANTED,              ERROR_OPLOCK_NOT_GRANTED);
    cmp2(STATUS_INVALID_OPLOCK_PROTOCOL,         ERROR_INVALID_OPLOCK_PROTOCOL);
    cmp2(STATUS_OPLOCK_BREAK_IN_PROGRESS,        ERROR_OPLOCK_BREAK_IN_PROGRESS);
    cmp(STATUS_INTERNAL_DB_CORRUPTION,           ERROR_INTERNAL_DB_CORRUPTION);
    cmp(STATUS_INTERNAL_ERROR,                   ERROR_INTERNAL_ERROR);
    cmp(STATUS_GENERIC_NOT_MAPPED,               ERROR_GENERIC_NOT_MAPPED);
    cmp(STATUS_BAD_DESCRIPTOR_FORMAT,            ERROR_BAD_DESCRIPTOR_FORMAT);
    cmp(STATUS_NOT_LOGON_PROCESS,                ERROR_NOT_LOGON_PROCESS);
    cmp(STATUS_LOGON_SESSION_EXISTS,             ERROR_LOGON_SESSION_EXISTS);
    cmp(STATUS_NO_SUCH_PACKAGE,                  ERROR_NO_SUCH_PACKAGE);
    cmp(STATUS_BAD_LOGON_SESSION_STATE,          ERROR_BAD_LOGON_SESSION_STATE);
    cmp(STATUS_LOGON_SESSION_COLLISION,          ERROR_LOGON_SESSION_COLLISION);
    cmp(STATUS_INVALID_LOGON_TYPE,               ERROR_INVALID_LOGON_TYPE);
    cmp2(STATUS_LOGON_SERVER_CONFLICT,           ERROR_LOGON_SERVER_CONFLICT);
    cmp2(STATUS_SERVER_SID_MISMATCH,             ERROR_SERVER_SID_MISMATCH);
    cmp2(STATUS_RXACT_STATE_CREATED,             ERROR_RXACT_STATE_CREATED);
    cmp(STATUS_RXACT_INVALID_STATE,              ERROR_RXACT_INVALID_STATE);
    cmp2(STATUS_RXACT_COMMITTED,                 ERROR_RXACT_COMMITTED);
    cmp2(STATUS_RXACT_COMMIT_NECESSARY,          ERROR_RXACT_COMMIT_NECESSARY);
    cmp(STATUS_RXACT_COMMIT_FAILURE,             ERROR_RXACT_COMMIT_FAILURE);
    cmp(STATUS_SPECIAL_ACCOUNT,                  ERROR_SPECIAL_ACCOUNT);
    cmp(STATUS_SPECIAL_USER,                     ERROR_SPECIAL_USER);
    cmp(STATUS_MEMBERS_PRIMARY_GROUP,            ERROR_MEMBERS_PRIMARY_GROUP);
    cmp(STATUS_TOKEN_ALREADY_IN_USE,             ERROR_TOKEN_ALREADY_IN_USE);
    cmp(STATUS_NO_SUCH_ALIAS,                    ERROR_NO_SUCH_ALIAS);
    cmp(STATUS_MEMBER_NOT_IN_ALIAS,              ERROR_MEMBER_NOT_IN_ALIAS);
    cmp(STATUS_MEMBER_IN_ALIAS,                  ERROR_MEMBER_IN_ALIAS);
    cmp(STATUS_ALIAS_EXISTS,                     ERROR_ALIAS_EXISTS);
    cmp(STATUS_LOGON_NOT_GRANTED,                ERROR_LOGON_NOT_GRANTED);
    cmp(STATUS_TOO_MANY_SECRETS,                 ERROR_TOO_MANY_SECRETS);
    cmp(STATUS_SECRET_TOO_LONG,                  ERROR_SECRET_TOO_LONG);
    cmp(STATUS_INTERNAL_DB_ERROR,                ERROR_INTERNAL_DB_ERROR);
    cmp(STATUS_TOO_MANY_CONTEXT_IDS,             ERROR_TOO_MANY_CONTEXT_IDS);
    cmp(STATUS_LOGON_TYPE_NOT_GRANTED,           ERROR_LOGON_TYPE_NOT_GRANTED);
    cmp(STATUS_NT_CROSS_ENCRYPTION_REQUIRED,     ERROR_NT_CROSS_ENCRYPTION_REQUIRED);
    cmp(STATUS_NO_SUCH_MEMBER,                   ERROR_NO_SUCH_MEMBER);
    cmp(STATUS_INVALID_MEMBER,                   ERROR_INVALID_MEMBER);
    cmp(STATUS_TOO_MANY_SIDS,                    ERROR_TOO_MANY_SIDS);
    cmp2(STATUS_ALL_SIDS_FILTERED,               ERROR_ALL_SIDS_FILTERED);
    cmp2(STATUS_INSUFFICIENT_LOGON_INFO,         ERROR_INSUFFICIENT_LOGON_INFO);
    cmp(STATUS_LM_CROSS_ENCRYPTION_REQUIRED,     ERROR_LM_CROSS_ENCRYPTION_REQUIRED);
    cmp(STATUS_MESSAGE_NOT_FOUND,                ERROR_MR_MID_NOT_FOUND);
    cmp2(STATUS_CONTROL_C_EXIT,                  ERROR_CONTROL_C_EXIT);
    cmp(STATUS_LOCAL_DISCONNECT,                 ERROR_NETNAME_DELETED);
    cmp(STATUS_REMOTE_DISCONNECT,                ERROR_NETNAME_DELETED);
    cmp(STATUS_REMOTE_RESOURCES,                 ERROR_REM_NOT_LIST);
    cmp(STATUS_LINK_FAILED,                      ERROR_UNEXP_NET_ERR);
    cmp(STATUS_LINK_TIMEOUT,                     ERROR_UNEXP_NET_ERR);
    cmp(STATUS_INVALID_CONNECTION,               ERROR_UNEXP_NET_ERR);
    cmp(STATUS_INVALID_ADDRESS,                  ERROR_UNEXP_NET_ERR);
    cmp2(STATUS_IP_ADDRESS_CONFLICT1,            ERROR_IP_ADDRESS_CONFLICT1);
    cmp2(STATUS_IP_ADDRESS_CONFLICT2,            ERROR_IP_ADDRESS_CONFLICT2);
    cmp2(STATUS_MISSING_SYSTEMFILE,              ERROR_MISSING_SYSTEMFILE);
    cmp2(STATUS_CORRUPT_SYSTEM_FILE,             ERROR_CORRUPT_SYSTEM_FILE);
    cmp2(STATUS_SYSTEM_IMAGE_BAD_SIGNATURE,      ERROR_SYSTEM_IMAGE_BAD_SIGNATURE);
    cmp2(STATUS_CRASH_DUMP,                      ERROR_CRASH_DUMP);
    cmp2(STATUS_LONGJUMP,                        ERROR_LONGJUMP);
    cmp2(STATUS_UNWIND_CONSOLIDATE,              ERROR_UNWIND_CONSOLIDATE);
    cmp2(STATUS_UNHANDLED_EXCEPTION,             ERROR_UNHANDLED_EXCEPTION);
    cmp2(STATUS_MCA_OCCURED,                     ERROR_MCA_OCCURED);
    cmp2(STATUS_APP_INIT_FAILURE,                ERROR_APP_INIT_FAILURE);
    cmp2(STATUS_ILLEGAL_FLOAT_CONTEXT,           ERROR_ILLEGAL_FLOAT_CONTEXT);
    cmp2(STATUS_FLOAT_MULTIPLE_FAULTS,           ERROR_FLOAT_MULTIPLE_FAULTS);
    cmp2(STATUS_FLOAT_MULTIPLE_TRAPS,            ERROR_FLOAT_MULTIPLE_TRAPS);
    cmp2(STATUS_NO_EVENT_PAIR,                   ERROR_NO_EVENT_PAIR);
    cmp2(STATUS_DOMAIN_CTRLR_CONFIG_ERROR,       ERROR_DOMAIN_CTRLR_CONFIG_ERROR);
    cmp(STATUS_IO_DEVICE_ERROR,                  ERROR_IO_DEVICE);
    cmp(STATUS_DEVICE_PROTOCOL_ERROR,            ERROR_IO_DEVICE);
    cmp(STATUS_DRIVER_INTERNAL_ERROR,            ERROR_IO_DEVICE);
    cmp(STATUS_INVALID_DEVICE_STATE,             ERROR_BAD_COMMAND);
    cmp(STATUS_DEVICE_CONFIGURATION_ERROR,       ERROR_INVALID_PARAMETER);
    cmp2(STATUS_DEVICE_ENUMERATION_ERROR,        ERROR_DEVICE_ENUMERATION_ERROR);
    cmp2(STATUS_INVALID_DEVICE_OBJECT_PARAMETER, ERROR_INVALID_DEVICE_OBJECT_PARAMETER);
    cmp2(STATUS_SOURCE_ELEMENT_EMPTY,            ERROR_SOURCE_ELEMENT_EMPTY);
    cmp2(STATUS_DESTINATION_ELEMENT_FULL,        ERROR_DESTINATION_ELEMENT_FULL);
    cmp2(STATUS_ILLEGAL_ELEMENT_ADDRESS,         ERROR_ILLEGAL_ELEMENT_ADDRESS);
    cmp2(STATUS_MAGAZINE_NOT_PRESENT,            ERROR_MAGAZINE_NOT_PRESENT);
    cmp2(STATUS_REINITIALIZATION_NEEDED,         ERROR_DEVICE_REINITIALIZATION_NEEDED);
    cmp2(STATUS_DEVICE_REQUIRES_CLEANING,        ERROR_DEVICE_REQUIRES_CLEANING);
    cmp2(STATUS_DEVICE_DOOR_OPEN,                ERROR_DEVICE_DOOR_OPEN);
    cmp2(STATUS_TRANSPORT_FULL,                  ERROR_TRANSPORT_FULL);
    cmp2(STATUS_CLEANER_CARTRIDGE_INSTALLED,     ERROR_CLEANER_CARTRIDGE_INSTALLED);
    cmp2(STATUS_REG_NAT_CONSUMPTION,             ERROR_REG_NAT_CONSUMPTION);
    cmp4(STATUS_ENCRYPTION_FAILED,               ERROR_ACCESS_DENIED, ERROR_ENCRYPTION_FAILED);
    cmp4(STATUS_DECRYPTION_FAILED,               ERROR_ACCESS_DENIED, ERROR_DECRYPTION_FAILED);
    cmp4(STATUS_NO_RECOVERY_POLICY,              ERROR_ACCESS_DENIED, ERROR_NO_RECOVERY_POLICY);
    cmp4(STATUS_NO_EFS,                          ERROR_ACCESS_DENIED, ERROR_NO_EFS);
    cmp4(STATUS_WRONG_EFS,                       ERROR_ACCESS_DENIED, ERROR_WRONG_EFS);
    cmp4(STATUS_NO_USER_KEYS,                    ERROR_ACCESS_DENIED, ERROR_NO_USER_KEYS);
    cmp2(STATUS_FILE_NOT_ENCRYPTED,              ERROR_FILE_NOT_ENCRYPTED);
    cmp2(STATUS_NOT_EXPORT_FORMAT,               ERROR_NOT_EXPORT_FORMAT);
    cmp2(STATUS_FILE_ENCRYPTED,                  ERROR_FILE_ENCRYPTED);
    cmp2(STATUS_EFS_ALG_BLOB_TOO_BIG,            ERROR_EFS_ALG_BLOB_TOO_BIG);
    cmp2(STATUS_BUFFER_ALL_ZEROS,                ERROR_BUFFER_ALL_ZEROS);
    cmp(STATUS_INVALID_USER_BUFFER,              ERROR_INVALID_USER_BUFFER);
    cmp2(STATUS_BAD_COMPRESSION_BUFFER,          ERROR_BAD_COMPRESSION_BUFFER);
    cmp2(STATUS_UNSUPPORTED_COMPRESSION,         ERROR_UNSUPPORTED_COMPRESSION);
    cmp(STATUS_SERIAL_NO_DEVICE_INITED,          ERROR_SERIAL_NO_DEVICE);
    cmp(STATUS_SHARED_IRQ_BUSY,                  ERROR_IRQ_BUSY);
    cmp(STATUS_SERIAL_MORE_WRITES,               ERROR_MORE_WRITES);
    cmp(STATUS_SERIAL_COUNTER_TIMEOUT,           ERROR_COUNTER_TIMEOUT);
    cmp(STATUS_FLOPPY_ID_MARK_NOT_FOUND,         ERROR_FLOPPY_ID_MARK_NOT_FOUND);
    cmp(STATUS_FLOPPY_WRONG_CYLINDER,            ERROR_FLOPPY_WRONG_CYLINDER);
    cmp(STATUS_FLOPPY_UNKNOWN_ERROR,             ERROR_FLOPPY_UNKNOWN_ERROR);
    cmp(STATUS_FLOPPY_BAD_REGISTERS,             ERROR_FLOPPY_BAD_REGISTERS);
    cmp2(STATUS_FLOPPY_VOLUME,                   ERROR_FLOPPY_VOLUME);
    cmp(STATUS_DISK_RECALIBRATE_FAILED,          ERROR_DISK_RECALIBRATE_FAILED);
    cmp(STATUS_DISK_OPERATION_FAILED,            ERROR_DISK_OPERATION_FAILED);
    cmp(STATUS_DISK_RESET_FAILED,                ERROR_DISK_RESET_FAILED);
    cmp(STATUS_EVENTLOG_FILE_CORRUPT,            ERROR_EVENTLOG_FILE_CORRUPT);
    cmp(STATUS_EVENTLOG_CANT_START,              ERROR_EVENTLOG_CANT_START);
    cmp(STATUS_NETLOGON_NOT_STARTED,             ERROR_NETLOGON_NOT_STARTED);
    cmp(STATUS_ACCOUNT_EXPIRED,                  ERROR_ACCOUNT_EXPIRED);
    cmp(STATUS_NETWORK_CREDENTIAL_CONFLICT,      ERROR_SESSION_CREDENTIAL_CONFLICT);
    cmp(STATUS_REMOTE_SESSION_LIMIT,             ERROR_REMOTE_SESSION_LIMIT_EXCEEDED);
    cmp(STATUS_INVALID_BUFFER_SIZE,              ERROR_INVALID_USER_BUFFER);
    cmp(STATUS_INVALID_ADDRESS_COMPONENT,        ERROR_INVALID_NETNAME);
    cmp(STATUS_INVALID_ADDRESS_WILDCARD,         ERROR_INVALID_NETNAME);
    cmp(STATUS_ADDRESS_ALREADY_EXISTS,           ERROR_DUP_NAME);
    cmp(STATUS_ADDRESS_CLOSED,                   ERROR_NETNAME_DELETED);
    cmp(STATUS_CONNECTION_DISCONNECTED,          ERROR_NETNAME_DELETED);
    cmp(STATUS_CONNECTION_RESET,                 ERROR_NETNAME_DELETED);
    cmp(STATUS_TRANSACTION_ABORTED,              ERROR_UNEXP_NET_ERR);
    cmp(STATUS_TRANSACTION_TIMED_OUT,            ERROR_UNEXP_NET_ERR);
    cmp(STATUS_TRANSACTION_NO_RELEASE,           ERROR_UNEXP_NET_ERR);
    cmp(STATUS_TRANSACTION_NO_MATCH,             ERROR_UNEXP_NET_ERR);
    cmp(STATUS_TRANSACTION_RESPONDED,            ERROR_UNEXP_NET_ERR);
    cmp(STATUS_TRANSACTION_INVALID_ID,           ERROR_UNEXP_NET_ERR);
    cmp(STATUS_TRANSACTION_INVALID_TYPE,         ERROR_UNEXP_NET_ERR);
    cmp(STATUS_NOT_SERVER_SESSION,               ERROR_NOT_SUPPORTED);
    cmp(STATUS_NOT_CLIENT_SESSION,               ERROR_NOT_SUPPORTED);
    cmp2(STATUS_CLIENT_SERVER_PARAMETERS_INVALID,ERROR_CLIENT_SERVER_PARAMETERS_INVALID);
    cmp(STATUS_USER_MAPPED_FILE,                 ERROR_USER_MAPPED_FILE);
    cmp2(STATUS_INVALID_HW_PROFILE,              ERROR_INVALID_HW_PROFILE);
    cmp(STATUS_PLUGPLAY_NO_DEVICE,               ERROR_SERVICE_DISABLED);
    cmp2(STATUS_PLUGPLAY_QUERY_VETOED,           ERROR_PLUGPLAY_QUERY_VETOED);
    cmp2(STATUS_INVALID_PLUGPLAY_DEVICE_PATH,    ERROR_INVALID_PLUGPLAY_DEVICE_PATH);
    cmp2(STATUS_PNP_RESTART_ENUMERATION,         ERROR_PNP_RESTART_ENUMERATION);
    cmp2(STATUS_PNP_REBOOT_REQUIRED,             ERROR_PNP_REBOOT_REQUIRED);
    cmp2(STATUS_WMI_GUID_NOT_FOUND,              ERROR_WMI_GUID_NOT_FOUND);
    cmp2(STATUS_WMI_INSTANCE_NOT_FOUND,          ERROR_WMI_INSTANCE_NOT_FOUND);
    cmp2(STATUS_WMI_ITEMID_NOT_FOUND,            ERROR_WMI_ITEMID_NOT_FOUND);
    cmp2(STATUS_WMI_TRY_AGAIN,                   ERROR_WMI_TRY_AGAIN);
    cmp2(STATUS_WMI_READ_ONLY,                   ERROR_WMI_READ_ONLY);
    cmp2(STATUS_WMI_SET_FAILURE,                 ERROR_WMI_SET_FAILURE);
    cmp2(STATUS_WMI_NOT_SUPPORTED,               ERROR_NOT_SUPPORTED);
    cmp2(STATUS_WMI_GUID_DISCONNECTED,           ERROR_WMI_GUID_DISCONNECTED);
    cmp2(STATUS_WMI_ALREADY_DISABLED,            ERROR_WMI_ALREADY_DISABLED);
    cmp2(STATUS_WMI_ALREADY_ENABLED,             ERROR_WMI_ALREADY_ENABLED);
    cmp2(STATUS_COPY_PROTECTION_FAILURE,         STG_E_STATUS_COPY_PROTECTION_FAILURE);
    cmp2(STATUS_CSS_AUTHENTICATION_FAILURE,      STG_E_CSS_AUTHENTICATION_FAILURE);
    cmp2(STATUS_CSS_KEY_NOT_PRESENT,             STG_E_CSS_KEY_NOT_PRESENT);
    cmp2(STATUS_CSS_KEY_NOT_ESTABLISHED,         STG_E_CSS_KEY_NOT_ESTABLISHED);
    cmp2(STATUS_CSS_SCRAMBLED_SECTOR,            STG_E_CSS_SCRAMBLED_SECTOR);
    cmp2(STATUS_CSS_REGION_MISMATCH,             STG_E_CSS_REGION_MISMATCH);
    cmp2(STATUS_CSS_RESETS_EXHAUSTED,            STG_E_RESETS_EXHAUSTED);
    cmp(RPC_NT_SERVER_UNAVAILABLE,               RPC_S_SERVER_UNAVAILABLE);
    cmp(RPC_NT_INVALID_STRING_BINDING,           RPC_S_INVALID_STRING_BINDING);
    cmp(RPC_NT_WRONG_KIND_OF_BINDING,            RPC_S_WRONG_KIND_OF_BINDING);
    cmp(RPC_NT_PROTSEQ_NOT_SUPPORTED,            RPC_S_PROTSEQ_NOT_SUPPORTED);
    cmp(RPC_NT_INVALID_RPC_PROTSEQ,              RPC_S_INVALID_RPC_PROTSEQ);
    cmp(RPC_NT_INVALID_STRING_UUID,              RPC_S_INVALID_STRING_UUID);
    cmp(RPC_NT_INVALID_ENDPOINT_FORMAT,          RPC_S_INVALID_ENDPOINT_FORMAT);
    cmp(RPC_NT_INVALID_NET_ADDR,                 RPC_S_INVALID_NET_ADDR);
    cmp(RPC_NT_NO_ENDPOINT_FOUND,                RPC_S_NO_ENDPOINT_FOUND);
    cmp(RPC_NT_INVALID_TIMEOUT,                  RPC_S_INVALID_TIMEOUT);
    cmp(RPC_NT_OBJECT_NOT_FOUND,                 RPC_S_OBJECT_NOT_FOUND);
    cmp(RPC_NT_ALREADY_REGISTERED,               RPC_S_ALREADY_REGISTERED);
    cmp(RPC_NT_TYPE_ALREADY_REGISTERED,          RPC_S_TYPE_ALREADY_REGISTERED);
    cmp(RPC_NT_ALREADY_LISTENING,                RPC_S_ALREADY_LISTENING);
    cmp(RPC_NT_NO_PROTSEQS_REGISTERED,           RPC_S_NO_PROTSEQS_REGISTERED);
    cmp(RPC_NT_NOT_LISTENING,                    RPC_S_NOT_LISTENING);
    cmp(RPC_NT_UNKNOWN_MGR_TYPE,                 RPC_S_UNKNOWN_MGR_TYPE);
    cmp(RPC_NT_UNKNOWN_IF,                       RPC_S_UNKNOWN_IF);
    cmp(RPC_NT_NO_BINDINGS,                      RPC_S_NO_BINDINGS);
    cmp(RPC_NT_NO_MORE_BINDINGS,                 RPC_S_NO_MORE_BINDINGS);
    cmp(RPC_NT_NO_PROTSEQS,                      RPC_S_NO_PROTSEQS);
    cmp(RPC_NT_CANT_CREATE_ENDPOINT,             RPC_S_CANT_CREATE_ENDPOINT);
    cmp(RPC_NT_OUT_OF_RESOURCES,                 RPC_S_OUT_OF_RESOURCES);
    cmp(RPC_NT_SERVER_TOO_BUSY,                  RPC_S_SERVER_TOO_BUSY);
    cmp(RPC_NT_INVALID_NETWORK_OPTIONS,          RPC_S_INVALID_NETWORK_OPTIONS);
    cmp(RPC_NT_NO_CALL_ACTIVE,                   RPC_S_NO_CALL_ACTIVE);
    cmp(RPC_NT_CALL_FAILED,                      RPC_S_CALL_FAILED);
    cmp(RPC_NT_CALL_FAILED_DNE,                  RPC_S_CALL_FAILED_DNE);
    cmp(RPC_NT_PROTOCOL_ERROR,                   RPC_S_PROTOCOL_ERROR);
    cmp(RPC_NT_UNSUPPORTED_TRANS_SYN,            RPC_S_UNSUPPORTED_TRANS_SYN);
    cmp(RPC_NT_UNSUPPORTED_TYPE,                 RPC_S_UNSUPPORTED_TYPE);
    cmp(RPC_NT_INVALID_TAG,                      RPC_S_INVALID_TAG);
    cmp(RPC_NT_INVALID_BOUND,                    RPC_S_INVALID_BOUND);
    cmp(RPC_NT_NO_ENTRY_NAME,                    RPC_S_NO_ENTRY_NAME);
    cmp(RPC_NT_INVALID_NAME_SYNTAX,              RPC_S_INVALID_NAME_SYNTAX);
    cmp(RPC_NT_UNSUPPORTED_NAME_SYNTAX,          RPC_S_UNSUPPORTED_NAME_SYNTAX);
    cmp(RPC_NT_UUID_NO_ADDRESS,                  RPC_S_UUID_NO_ADDRESS);
    cmp(RPC_NT_DUPLICATE_ENDPOINT,               RPC_S_DUPLICATE_ENDPOINT);
    cmp(RPC_NT_UNKNOWN_AUTHN_TYPE,               RPC_S_UNKNOWN_AUTHN_TYPE);
    cmp(RPC_NT_MAX_CALLS_TOO_SMALL,              RPC_S_MAX_CALLS_TOO_SMALL);
    cmp(RPC_NT_STRING_TOO_LONG,                  RPC_S_STRING_TOO_LONG);
    cmp(RPC_NT_PROTSEQ_NOT_FOUND,                RPC_S_PROTSEQ_NOT_FOUND);
    cmp(RPC_NT_PROCNUM_OUT_OF_RANGE,             RPC_S_PROCNUM_OUT_OF_RANGE);
    cmp(RPC_NT_BINDING_HAS_NO_AUTH,              RPC_S_BINDING_HAS_NO_AUTH);
    cmp(RPC_NT_UNKNOWN_AUTHN_SERVICE,            RPC_S_UNKNOWN_AUTHN_SERVICE);
    cmp(RPC_NT_UNKNOWN_AUTHN_LEVEL,              RPC_S_UNKNOWN_AUTHN_LEVEL);
    cmp(RPC_NT_INVALID_AUTH_IDENTITY,            RPC_S_INVALID_AUTH_IDENTITY);
    cmp(RPC_NT_UNKNOWN_AUTHZ_SERVICE,            RPC_S_UNKNOWN_AUTHZ_SERVICE);
    cmp(EPT_NT_INVALID_ENTRY,                    EPT_S_INVALID_ENTRY);
    cmp(EPT_NT_CANT_PERFORM_OP,                  EPT_S_CANT_PERFORM_OP);
    cmp(EPT_NT_NOT_REGISTERED,                   EPT_S_NOT_REGISTERED);
    cmp(RPC_NT_NOTHING_TO_EXPORT,                RPC_S_NOTHING_TO_EXPORT);
    cmp(RPC_NT_INCOMPLETE_NAME,                  RPC_S_INCOMPLETE_NAME);
    cmp(RPC_NT_INVALID_VERS_OPTION,              RPC_S_INVALID_VERS_OPTION);
    cmp(RPC_NT_NO_MORE_MEMBERS,                  RPC_S_NO_MORE_MEMBERS);
    cmp(RPC_NT_NOT_ALL_OBJS_UNEXPORTED,          RPC_S_NOT_ALL_OBJS_UNEXPORTED);
    cmp(RPC_NT_INTERFACE_NOT_FOUND,              RPC_S_INTERFACE_NOT_FOUND);
    cmp(RPC_NT_ENTRY_ALREADY_EXISTS,             RPC_S_ENTRY_ALREADY_EXISTS);
    cmp(RPC_NT_ENTRY_NOT_FOUND,                  RPC_S_ENTRY_NOT_FOUND);
    cmp(RPC_NT_NAME_SERVICE_UNAVAILABLE,         RPC_S_NAME_SERVICE_UNAVAILABLE);
    cmp(RPC_NT_INVALID_NAF_ID,                   RPC_S_INVALID_NAF_ID);
    cmp(RPC_NT_CANNOT_SUPPORT,                   RPC_S_CANNOT_SUPPORT);
    cmp(RPC_NT_NO_CONTEXT_AVAILABLE,             RPC_S_NO_CONTEXT_AVAILABLE);
    cmp(RPC_NT_INTERNAL_ERROR,                   RPC_S_INTERNAL_ERROR);
    cmp(RPC_NT_ZERO_DIVIDE,                      RPC_S_ZERO_DIVIDE);
    cmp(RPC_NT_ADDRESS_ERROR,                    RPC_S_ADDRESS_ERROR);
    cmp(RPC_NT_FP_DIV_ZERO,                      RPC_S_FP_DIV_ZERO);
    cmp(RPC_NT_FP_UNDERFLOW,                     RPC_S_FP_UNDERFLOW);
    cmp(RPC_NT_FP_OVERFLOW,                      RPC_S_FP_OVERFLOW);
    cmp(RPC_NT_NO_MORE_ENTRIES,                  RPC_X_NO_MORE_ENTRIES);
    cmp(RPC_NT_SS_CHAR_TRANS_OPEN_FAIL,          RPC_X_SS_CHAR_TRANS_OPEN_FAIL);
    cmp(RPC_NT_SS_CHAR_TRANS_SHORT_FILE,         RPC_X_SS_CHAR_TRANS_SHORT_FILE);
    cmp(RPC_NT_SS_CONTEXT_MISMATCH,              ERROR_INVALID_HANDLE);
    cmp(RPC_NT_SS_CONTEXT_DAMAGED,               RPC_X_SS_CONTEXT_DAMAGED);
    cmp(RPC_NT_SS_HANDLES_MISMATCH,              RPC_X_SS_HANDLES_MISMATCH);
    cmp(RPC_NT_SS_CANNOT_GET_CALL_HANDLE,        RPC_X_SS_CANNOT_GET_CALL_HANDLE);
    cmp(RPC_NT_NULL_REF_POINTER,                 RPC_X_NULL_REF_POINTER);
    cmp(RPC_NT_ENUM_VALUE_OUT_OF_RANGE,          RPC_X_ENUM_VALUE_OUT_OF_RANGE);
    cmp(RPC_NT_BYTE_COUNT_TOO_SMALL,             RPC_X_BYTE_COUNT_TOO_SMALL);
    cmp(RPC_NT_BAD_STUB_DATA,                    RPC_X_BAD_STUB_DATA);
    cmp(RPC_NT_INVALID_OBJECT,                   RPC_S_INVALID_OBJECT);
    cmp(STATUS_NO_TRUST_LSA_SECRET,              ERROR_NO_TRUST_LSA_SECRET);
    cmp(STATUS_NO_TRUST_SAM_ACCOUNT,             ERROR_NO_TRUST_SAM_ACCOUNT);
    cmp(STATUS_TRUSTED_DOMAIN_FAILURE,           ERROR_TRUSTED_DOMAIN_FAILURE);
    cmp(STATUS_TRUSTED_RELATIONSHIP_FAILURE,     ERROR_TRUSTED_RELATIONSHIP_FAILURE);
    cmp(STATUS_TRUST_FAILURE,                    ERROR_TRUST_FAILURE);
    cmp(RPC_NT_CALL_IN_PROGRESS,                 RPC_S_CALL_IN_PROGRESS);
    cmp(STATUS_LOG_FILE_FULL,                    ERROR_LOG_FILE_FULL);
    cmp(STATUS_EVENTLOG_FILE_CHANGED,            ERROR_EVENTLOG_FILE_CHANGED);
    cmp(STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT, ERROR_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT);
    cmp(STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT, ERROR_NOLOGON_WORKSTATION_TRUST_ACCOUNT);
    cmp(STATUS_NOLOGON_SERVER_TRUST_ACCOUNT,     ERROR_NOLOGON_SERVER_TRUST_ACCOUNT);
    cmp(STATUS_DOMAIN_TRUST_INCONSISTENT,        ERROR_DOMAIN_TRUST_INCONSISTENT);
    cmp(STATUS_NO_USER_SESSION_KEY,              ERROR_NO_USER_SESSION_KEY);
    cmp(STATUS_POSSIBLE_DEADLOCK,                ERROR_POSSIBLE_DEADLOCK);
    cmp(STATUS_IMAGE_ALREADY_LOADED,             ERROR_SERVICE_ALREADY_RUNNING);
    cmp(RPC_NT_GROUP_MEMBER_NOT_FOUND,           RPC_S_GROUP_MEMBER_NOT_FOUND);
    cmp(RPC_NT_NO_INTERFACES,                    RPC_S_NO_INTERFACES);
    cmp(RPC_NT_CALL_CANCELLED,                   RPC_S_CALL_CANCELLED);
    cmp(RPC_NT_BINDING_INCOMPLETE,               RPC_S_BINDING_INCOMPLETE);
    cmp(RPC_NT_COMM_FAILURE,                     RPC_S_COMM_FAILURE);
    cmp(RPC_NT_UNSUPPORTED_AUTHN_LEVEL,          RPC_S_UNSUPPORTED_AUTHN_LEVEL);
    cmp(RPC_NT_NO_PRINC_NAME,                    RPC_S_NO_PRINC_NAME);
    cmp(RPC_NT_NOT_RPC_ERROR,                    RPC_S_NOT_RPC_ERROR);
    cmp(RPC_NT_UUID_LOCAL_ONLY,                  RPC_S_UUID_LOCAL_ONLY);
    cmp(RPC_NT_SEC_PKG_ERROR,                    RPC_S_SEC_PKG_ERROR);
    cmp(RPC_NT_NOT_CANCELLED,                    RPC_S_NOT_CANCELLED);
    cmp(RPC_NT_INVALID_ES_ACTION,                RPC_X_INVALID_ES_ACTION);
    cmp(RPC_NT_WRONG_ES_VERSION,                 RPC_X_WRONG_ES_VERSION);
    cmp(RPC_NT_WRONG_STUB_VERSION,               RPC_X_WRONG_STUB_VERSION);
    cmp(RPC_NT_INVALID_PIPE_OBJECT,              RPC_X_INVALID_PIPE_OBJECT);
    cmp(RPC_NT_INVALID_PIPE_OPERATION,           RPC_X_INVALID_PIPE_OPERATION);
    cmp(RPC_NT_WRONG_PIPE_VERSION,               RPC_X_WRONG_PIPE_VERSION);
    cmp(EPT_NT_CANT_CREATE,                      EPT_S_CANT_CREATE);
    cmp(RPC_NT_SEND_INCOMPLETE,                  RPC_S_SEND_INCOMPLETE);
    cmp2(RPC_NT_INVALID_ASYNC_HANDLE,            RPC_S_INVALID_ASYNC_HANDLE);
    cmp2(RPC_NT_INVALID_ASYNC_CALL,              RPC_S_INVALID_ASYNC_CALL);
    cmp2(RPC_NT_PIPE_CLOSED,                     RPC_X_PIPE_CLOSED);
    cmp2(RPC_NT_PIPE_EMPTY,                      RPC_X_PIPE_EMPTY);
    cmp2(RPC_NT_PIPE_DISCIPLINE_ERROR,           RPC_X_PIPE_DISCIPLINE_ERROR);
    cmp(STATUS_NO_BROWSER_SERVERS_FOUND,         ERROR_NO_BROWSER_SERVERS_FOUND);
    cmp(STATUS_MAPPED_ALIGNMENT,                 ERROR_MAPPED_ALIGNMENT);
    cmp(STATUS_CONNECTION_IN_USE,                ERROR_DEVICE_IN_USE);
    cmp(STATUS_VERIFY_REQUIRED,                  ERROR_MEDIA_CHANGED);
    cmp(STATUS_ALREADY_DISCONNECTED,             ERROR_ACTIVE_CONNECTIONS);
    cmp(STATUS_CONNECTION_REFUSED,               ERROR_CONNECTION_REFUSED);
    cmp(STATUS_GRACEFUL_DISCONNECT,              ERROR_GRACEFUL_DISCONNECT);
    cmp(STATUS_ADDRESS_ALREADY_ASSOCIATED,       ERROR_ADDRESS_ALREADY_ASSOCIATED);
    cmp(STATUS_ADDRESS_NOT_ASSOCIATED,           ERROR_ADDRESS_NOT_ASSOCIATED);
    cmp(STATUS_CONNECTION_INVALID,               ERROR_CONNECTION_INVALID);
    cmp(STATUS_CONNECTION_ACTIVE,                ERROR_CONNECTION_ACTIVE);
    cmp(STATUS_NETWORK_UNREACHABLE,              ERROR_NETWORK_UNREACHABLE);
    cmp(STATUS_HOST_UNREACHABLE,                 ERROR_HOST_UNREACHABLE);
    cmp2(STATUS_HOST_DOWN,                       ERROR_HOST_DOWN);
    cmp(STATUS_PROTOCOL_UNREACHABLE,             ERROR_PROTOCOL_UNREACHABLE);
    cmp(STATUS_PORT_UNREACHABLE,                 ERROR_PORT_UNREACHABLE);
    cmp(STATUS_REQUEST_ABORTED,                  ERROR_REQUEST_ABORTED);
    cmp(STATUS_CONNECTION_ABORTED,               ERROR_CONNECTION_ABORTED);
    cmp(STATUS_CONNECTION_COUNT_LIMIT,           ERROR_CONNECTION_COUNT_LIMIT);
    cmp(STATUS_PATH_NOT_COVERED,                 ERROR_HOST_UNREACHABLE);
    cmp(STATUS_LOGIN_TIME_RESTRICTION,           ERROR_LOGIN_TIME_RESTRICTION);
    cmp(STATUS_LOGIN_WKSTA_RESTRICTION,          ERROR_LOGIN_WKSTA_RESTRICTION);
    cmp(STATUS_LICENSE_QUOTA_EXCEEDED,           ERROR_LICENSE_QUOTA_EXCEEDED);
    cmp(STATUS_RESOURCE_NOT_OWNED,               ERROR_NOT_OWNER);
    cmp3(STATUS_DUPLICATE_OBJECTID,              ERROR_OBJECT_ALREADY_EXISTS, STATUS_DUPLICATE_OBJECTID);
    cmp3(STATUS_OBJECTID_EXISTS,                 ERROR_OBJECT_ALREADY_EXISTS, STATUS_OBJECTID_EXISTS);
    cmp2(STATUS_OBJECTID_NOT_FOUND,              ERROR_FILE_NOT_FOUND);
    cmp2(STATUS_MFT_TOO_FRAGMENTED,              ERROR_DISK_TOO_FRAGMENTED);
    cmp(SEC_E_INSUFFICIENT_MEMORY,               ERROR_NO_SYSTEM_RESOURCES);
    cmp(SEC_E_INVALID_HANDLE,                    ERROR_INVALID_HANDLE);
    cmp(SEC_E_UNSUPPORTED_FUNCTION,              ERROR_INVALID_FUNCTION);
    cmp(SEC_E_TARGET_UNKNOWN,                    ERROR_BAD_NETPATH);
    cmp(SEC_E_INTERNAL_ERROR,                    ERROR_INTERNAL_ERROR);
    cmp(SEC_E_SECPKG_NOT_FOUND,                  ERROR_NO_SUCH_PACKAGE);
    cmp(SEC_E_NOT_OWNER,                         ERROR_NOT_OWNER);
    cmp(SEC_E_CANNOT_INSTALL,                    ERROR_NO_SUCH_PACKAGE);
    cmp(SEC_E_INVALID_TOKEN,                     ERROR_INVALID_PARAMETER);
    cmp(SEC_E_CANNOT_PACK,                       ERROR_INVALID_PARAMETER);
    cmp(SEC_E_QOP_NOT_SUPPORTED,                 ERROR_NOT_SUPPORTED);
    cmp(SEC_E_NO_IMPERSONATION,                  ERROR_CANNOT_IMPERSONATE);
    cmp2(SEC_E_MULTIPLE_ACCOUNTS,                ERROR_CANNOT_IMPERSONATE);
    cmp(SEC_E_LOGON_DENIED,                      ERROR_LOGON_FAILURE);
    cmp(SEC_E_UNKNOWN_CREDENTIALS,               ERROR_INVALID_PARAMETER);
    cmp2(SEC_E_INCOMPLETE_CREDENTIALS,           ERROR_INVALID_PARAMETER);
    cmp(SEC_E_NO_CREDENTIALS,                    ERROR_NO_SUCH_LOGON_SESSION);
    cmp(SEC_E_MESSAGE_ALTERED,                   ERROR_ACCESS_DENIED);
    cmp(SEC_E_OUT_OF_SEQUENCE,                   ERROR_ACCESS_DENIED);
    cmp(SEC_E_NO_AUTHENTICATING_AUTHORITY,       ERROR_NO_LOGON_SERVERS);
    cmp(SEC_E_BAD_PKGID,                         ERROR_NO_SUCH_PACKAGE);
    cmp4(SEC_E_WRONG_PRINCIPAL,                  ERROR_WRONG_TARGET_NAME, 1462);
    cmp2(SEC_E_INCOMPLETE_MESSAGE,               ERROR_INVALID_USER_BUFFER);
    cmp2(SEC_E_BUFFER_TOO_SMALL,                 ERROR_INSUFFICIENT_BUFFER);
    cmp2(SEC_E_UNTRUSTED_ROOT,                   ERROR_TRUST_FAILURE);
    cmp2(SEC_E_ILLEGAL_MESSAGE,                  ERROR_INVALID_PARAMETER);
    cmp2(SEC_E_CERT_UNKNOWN,                     ERROR_INVALID_PARAMETER);
    cmp2(SEC_E_CERT_EXPIRED,                     ERROR_PASSWORD_EXPIRED);
    cmp2(SEC_E_ENCRYPT_FAILURE,                  ERROR_ENCRYPTION_FAILED);
    cmp2(SEC_E_DECRYPT_FAILURE,                  ERROR_DECRYPTION_FAILED);
    cmp2(SEC_E_ALGORITHM_MISMATCH,               ERROR_INVALID_FUNCTION);
    cmp2(SEC_E_CONTEXT_EXPIRED,                  ERROR_CONTEXT_EXPIRED);
    cmp2(STATUS_BAD_BINDINGS,                    SEC_E_BAD_BINDINGS);
    cmp2(TRUST_E_CERT_SIGNATURE,                 ERROR_MUTUAL_AUTH_FAILED);
    cmp2(CRYPT_E_REVOKED,                        ERROR_MUTUAL_AUTH_FAILED);
    cmp2(CRYPT_E_NO_REVOCATION_CHECK,            ERROR_MUTUAL_AUTH_FAILED);
    cmp2(CRYPT_E_REVOCATION_OFFLINE,             ERROR_MUTUAL_AUTH_FAILED);
    cmp2(STATUS_SHUTDOWN_IN_PROGRESS,            ERROR_SHUTDOWN_IN_PROGRESS);
    cmp2(STATUS_SERVER_SHUTDOWN_IN_PROGRESS,     ERROR_SERVER_SHUTDOWN_IN_PROGRESS);
    cmp2(STATUS_SYSTEM_SHUTDOWN,                 ERROR_SYSTEM_SHUTDOWN);
    cmp4(STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY, ERROR_DS_MEMBERSHIP_EVALUATED_LOCALLY, 1922);
    cmp4(STATUS_DS_NO_ATTRIBUTE_OR_VALUE,        ERROR_DS_NO_ATTRIBUTE_OR_VALUE, 1923);
    cmp4(STATUS_DS_INVALID_ATTRIBUTE_SYNTAX,     ERROR_DS_INVALID_ATTRIBUTE_SYNTAX, 1924);
    cmp4(STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED,     ERROR_DS_ATTRIBUTE_TYPE_UNDEFINED, 1925);
    cmp4(STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS,    ERROR_DS_ATTRIBUTE_OR_VALUE_EXISTS, 1926);
    cmp4(STATUS_DS_BUSY,                         ERROR_DS_BUSY, 1927);
    cmp4(STATUS_DS_UNAVAILABLE,                  ERROR_DS_UNAVAILABLE, 1928);
    cmp4(STATUS_DS_NO_RIDS_ALLOCATED,            ERROR_DS_NO_RIDS_ALLOCATED, 1929);
    cmp4(STATUS_DS_NO_MORE_RIDS,                 ERROR_DS_NO_MORE_RIDS, 1930);
    cmp4(STATUS_DS_INCORRECT_ROLE_OWNER,         ERROR_DS_INCORRECT_ROLE_OWNER, 1931);
    cmp4(STATUS_DS_RIDMGR_INIT_ERROR,            ERROR_DS_RIDMGR_INIT_ERROR, 1932);
    cmp4(STATUS_DS_OBJ_CLASS_VIOLATION,          ERROR_DS_OBJ_CLASS_VIOLATION, 1933);
    cmp4(STATUS_DS_CANT_ON_NON_LEAF,             ERROR_DS_CANT_ON_NON_LEAF, 1934);
    cmp4(STATUS_DS_CANT_ON_RDN,                  ERROR_DS_CANT_ON_RDN, 1935);
    cmp4(STATUS_DS_CROSS_DOM_MOVE_FAILED,        ERROR_DS_CROSS_DOM_MOVE_ERROR, 1937);
    cmp4(STATUS_DS_GC_NOT_AVAILABLE,             ERROR_DS_GC_NOT_AVAILABLE, 1938);
    cmp2(STATUS_DS_CANT_MOD_OBJ_CLASS,           ERROR_DS_CANT_MOD_OBJ_CLASS);
    cmp2(STATUS_DS_ADMIN_LIMIT_EXCEEDED,         ERROR_DS_ADMIN_LIMIT_EXCEEDED);
    cmp2(STATUS_DIRECTORY_SERVICE_REQUIRED,      ERROR_DS_DS_REQUIRED);
    cmp2(STATUS_DS_SAM_INIT_FAILURE,             ERROR_DS_SAM_INIT_FAILURE);
    cmp2(STATUS_DS_CANT_START,                   ERROR_DS_CANT_START);
    cmp2(STATUS_DS_INIT_FAILURE,                 ERROR_DS_INIT_FAILURE);
    cmp2(STATUS_SAM_INIT_FAILURE,                ERROR_SAM_INIT_FAILURE);
    cmp2(STATUS_DS_SENSITIVE_GROUP_VIOLATION,    ERROR_DS_SENSITIVE_GROUP_VIOLATION);
    cmp2(STATUS_DS_CANT_MOD_PRIMARYGROUPID,      ERROR_DS_CANT_MOD_PRIMARYGROUPID);
    cmp2(STATUS_DS_INVALID_GROUP_TYPE,           ERROR_DS_INVALID_GROUP_TYPE);
    cmp2(STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN,           ERROR_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN);
    cmp2(STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN,            ERROR_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN);
    cmp2(STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER,                ERROR_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER);
    cmp2(STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER,            ERROR_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER);
    cmp2(STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER,             ERROR_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER);
    cmp2(STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER,          ERROR_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER);
    cmp2(STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER,     ERROR_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER);
    cmp2(STATUS_DS_HAVE_PRIMARY_MEMBERS,         ERROR_DS_HAVE_PRIMARY_MEMBERS);
    cmp2(STATUS_DS_GC_REQUIRED,                  ERROR_DS_GC_REQUIRED);
    cmp2(STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY,   ERROR_DS_LOCAL_MEMBER_OF_LOCAL_ONLY);
    cmp2(STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS,   ERROR_DS_NO_FPO_IN_UNIVERSAL_GROUPS);
    cmp2(STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED,ERROR_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED);
    cmp2(STATUS_SAM_NEED_BOOTKEY_PASSWORD,       ERROR_DS_SAM_NEED_BOOTKEY_PASSWORD);
    cmp2(STATUS_SAM_NEED_BOOTKEY_FLOPPY,         ERROR_DS_SAM_NEED_BOOTKEY_FLOPPY);
    cmp2(STATUS_DS_INIT_FAILURE_CONSOLE,         ERROR_DS_INIT_FAILURE_CONSOLE);
    cmp2(STATUS_DS_SAM_INIT_FAILURE_CONSOLE,     ERROR_DS_SAM_INIT_FAILURE_CONSOLE);
    cmp2(STATUS_DS_VERSION_CHECK_FAILURE,        ERROR_DS_VERSION_CHECK_FAILURE);
    cmp2(STATUS_UNFINISHED_CONTEXT_DELETED,      SEC_E_UNFINISHED_CONTEXT_DELETED);
    cmp2(STATUS_NO_TGT_REPLY,                    SEC_E_NO_TGT_REPLY);
    cmp2(STATUS_NO_IP_ADDRESSES,                 SEC_E_NO_IP_ADDRESSES);
    cmp2(STATUS_WRONG_CREDENTIAL_HANDLE,         SEC_E_WRONG_CREDENTIAL_HANDLE);
    cmp2(STATUS_CRYPTO_SYSTEM_INVALID,           SEC_E_CRYPTO_SYSTEM_INVALID);
    cmp2(STATUS_MAX_REFERRALS_EXCEEDED,          SEC_E_MAX_REFERRALS_EXCEEDED);
    cmp2(STATUS_MUST_BE_KDC,                     SEC_E_MUST_BE_KDC);
    cmp2(STATUS_STRONG_CRYPTO_NOT_SUPPORTED,     SEC_E_STRONG_CRYPTO_NOT_SUPPORTED);
    cmp2(STATUS_TOO_MANY_PRINCIPALS,             SEC_E_TOO_MANY_PRINCIPALS);
    cmp2(STATUS_NO_PA_DATA,                      SEC_E_NO_PA_DATA);
    cmp2(STATUS_PKINIT_NAME_MISMATCH,            SEC_E_PKINIT_NAME_MISMATCH);
    cmp2(STATUS_SMARTCARD_LOGON_REQUIRED,        SEC_E_SMARTCARD_LOGON_REQUIRED);
    cmp2(STATUS_KDC_INVALID_REQUEST,             SEC_E_KDC_INVALID_REQUEST);
    cmp2(STATUS_KDC_UNABLE_TO_REFER,             SEC_E_KDC_UNABLE_TO_REFER);
    cmp2(STATUS_KDC_UNKNOWN_ETYPE,               SEC_E_KDC_UNKNOWN_ETYPE);
    cmp2(STATUS_UNSUPPORTED_PREAUTH,             SEC_E_UNSUPPORTED_PREAUTH);
    cmp4(STATUS_SHARED_POLICY,                   ERROR_SHARED_POLICY, 1939);
    cmp4(STATUS_POLICY_OBJECT_NOT_FOUND,         ERROR_POLICY_OBJECT_NOT_FOUND, 1940);
    cmp4(STATUS_POLICY_ONLY_IN_DS,               ERROR_POLICY_ONLY_IN_DS, 1941);
    cmp4(STATUS_DEVICE_REMOVED,                  ERROR_DEVICE_REMOVED, 617);
    cmp2(STATUS_RETRY,                           ERROR_RETRY);
    cmp2(STATUS_NOT_SUPPORTED_ON_SBS,            ERROR_NOT_SUPPORTED_ON_SBS);
    cmp2(STATUS_DRIVER_BLOCKED_CRITICAL,         ERROR_DRIVER_BLOCKED);
    cmp2(STATUS_DRIVER_BLOCKED,                  ERROR_DRIVER_BLOCKED);
    cmp2(STATUS_DRIVER_CANCEL_TIMEOUT,           ERROR_DRIVER_CANCEL_TIMEOUT);
    cmp2(STATUS_PRENT4_MACHINE_ACCOUNT,          ERROR_DS_MACHINE_ACCOUNT_CREATED_PRENT4);
    cmp2(STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER,ERROR_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER);
    cmp2(STATUS_DS_SHUTTING_DOWN,                ERROR_DS_SHUTTING_DOWN);
    cmp2(STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT,       ERROR_ACCESS_DISABLED_BY_POLICY);
    cmp2(STATUS_ACCESS_DISABLED_BY_POLICY_PATH,          ERROR_ACCESS_DISABLED_BY_POLICY);
    cmp2(STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER,     ERROR_ACCESS_DISABLED_BY_POLICY);
    cmp2(STATUS_ACCESS_DISABLED_BY_POLICY_OTHER, ERROR_ACCESS_DISABLED_BY_POLICY);
    cmp2(STATUS_FAIL_CHECK,                      ERROR_INVALID_PARAMETER);
    cmp2(STATUS_CTX_CLOSE_PENDING,               ERROR_CTX_CLOSE_PENDING);
    cmp2(STATUS_CTX_NO_OUTBUF,                   ERROR_CTX_NO_OUTBUF);
    cmp2(STATUS_CTX_MODEM_INF_NOT_FOUND,         ERROR_CTX_MODEM_INF_NOT_FOUND);
    cmp2(STATUS_CTX_INVALID_MODEMNAME,           ERROR_CTX_INVALID_MODEMNAME);
    cmp2(STATUS_CTX_RESPONSE_ERROR,              ERROR_CTX_MODEM_RESPONSE_ERROR);
    cmp2(STATUS_CTX_MODEM_RESPONSE_TIMEOUT,      ERROR_CTX_MODEM_RESPONSE_TIMEOUT);
    cmp2(STATUS_CTX_MODEM_RESPONSE_NO_CARRIER,   ERROR_CTX_MODEM_RESPONSE_NO_CARRIER);
    cmp2(STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE,  ERROR_CTX_MODEM_RESPONSE_NO_DIALTONE);
    cmp2(STATUS_CTX_MODEM_RESPONSE_BUSY,         ERROR_CTX_MODEM_RESPONSE_BUSY);
    cmp2(STATUS_CTX_MODEM_RESPONSE_VOICE,        ERROR_CTX_MODEM_RESPONSE_VOICE);
    cmp2(STATUS_CTX_TD_ERROR,                    ERROR_CTX_TD_ERROR);
    cmp2(STATUS_CTX_WINSTATION_NAME_INVALID,     ERROR_CTX_WINSTATION_NAME_INVALID);
    cmp2(STATUS_CTX_WINSTATION_NOT_FOUND,        ERROR_CTX_WINSTATION_NOT_FOUND);
    cmp2(STATUS_CTX_WINSTATION_NAME_COLLISION,   ERROR_CTX_WINSTATION_ALREADY_EXISTS);
    cmp2(STATUS_CTX_WINSTATION_BUSY,             ERROR_CTX_WINSTATION_BUSY);
    cmp2(STATUS_CTX_GRAPHICS_INVALID,            ERROR_CTX_GRAPHICS_INVALID);
    cmp2(STATUS_CTX_BAD_VIDEO_MODE,              ERROR_CTX_BAD_VIDEO_MODE);
    cmp2(STATUS_CTX_NOT_CONSOLE,                 ERROR_CTX_NOT_CONSOLE);
    cmp2(STATUS_CTX_CLIENT_QUERY_TIMEOUT,        ERROR_CTX_CLIENT_QUERY_TIMEOUT);
    cmp2(STATUS_CTX_CONSOLE_DISCONNECT,          ERROR_CTX_CONSOLE_DISCONNECT);
    cmp2(STATUS_CTX_CONSOLE_CONNECT,             ERROR_CTX_CONSOLE_CONNECT);
    cmp2(STATUS_CTX_SHADOW_DENIED,               ERROR_CTX_SHADOW_DENIED);
    cmp2(STATUS_CTX_SHADOW_INVALID,              ERROR_CTX_SHADOW_INVALID);
    cmp2(STATUS_CTX_SHADOW_DISABLED,             ERROR_CTX_SHADOW_DISABLED);
    cmp2(STATUS_CTX_WINSTATION_ACCESS_DENIED,    ERROR_CTX_WINSTATION_ACCESS_DENIED);
    cmp2(STATUS_CTX_INVALID_PD,                  ERROR_CTX_INVALID_PD);
    cmp2(STATUS_CTX_PD_NOT_FOUND,                ERROR_CTX_PD_NOT_FOUND);
    cmp2(STATUS_CTX_INVALID_WD,                  ERROR_CTX_INVALID_WD);
    cmp2(STATUS_CTX_WD_NOT_FOUND,                ERROR_CTX_WD_NOT_FOUND);
    cmp2(STATUS_CTX_CLIENT_LICENSE_IN_USE,       ERROR_CTX_CLIENT_LICENSE_IN_USE);
    cmp2(STATUS_CTX_CLIENT_LICENSE_NOT_SET,      ERROR_CTX_CLIENT_LICENSE_NOT_SET);
    cmp2(STATUS_CTX_LICENSE_NOT_AVAILABLE,       ERROR_CTX_LICENSE_NOT_AVAILABLE);
    cmp2(STATUS_CTX_LICENSE_CLIENT_INVALID,      ERROR_CTX_LICENSE_CLIENT_INVALID);
    cmp2(STATUS_CTX_LICENSE_EXPIRED,             ERROR_CTX_LICENSE_EXPIRED);
    cmp2(STATUS_CTX_SHADOW_ENDED_BY_MODE_CHANGE, ERROR_CTX_SHADOW_ENDED_BY_MODE_CHANGE);
    cmp2(STATUS_CTX_SHADOW_NOT_RUNNING,          ERROR_CTX_SHADOW_NOT_RUNNING);
    cmp2(STATUS_LICENSE_VIOLATION,               ERROR_CTX_LICENSE_NOT_AVAILABLE);
    cmp2(STATUS_EVALUATION_EXPIRATION,           ERROR_EVALUATION_EXPIRATION);
    cmp2(STATUS_NETWORK_SESSION_EXPIRED,         ERROR_NO_USER_SESSION_KEY);
    cmp2(STATUS_FILES_OPEN,                      ERROR_OPEN_FILES);
    cmp2(STATUS_SXS_SECTION_NOT_FOUND,           ERROR_SXS_SECTION_NOT_FOUND);
    cmp2(STATUS_SXS_CANT_GEN_ACTCTX,             ERROR_SXS_CANT_GEN_ACTCTX);
    cmp2(STATUS_SXS_INVALID_ACTCTXDATA_FORMAT,   ERROR_SXS_INVALID_ACTCTXDATA_FORMAT);
    cmp2(STATUS_SXS_ASSEMBLY_NOT_FOUND,          ERROR_SXS_ASSEMBLY_NOT_FOUND);
    cmp2(STATUS_SXS_MANIFEST_FORMAT_ERROR,       ERROR_SXS_MANIFEST_FORMAT_ERROR);
    cmp2(STATUS_SXS_MANIFEST_PARSE_ERROR,        ERROR_SXS_MANIFEST_PARSE_ERROR);
    cmp2(STATUS_SXS_ACTIVATION_CONTEXT_DISABLED, ERROR_SXS_ACTIVATION_CONTEXT_DISABLED);
    cmp2(STATUS_SXS_KEY_NOT_FOUND,               ERROR_SXS_KEY_NOT_FOUND);
    cmp2(STATUS_SXS_VERSION_CONFLICT,            ERROR_SXS_VERSION_CONFLICT);
    cmp2(STATUS_SXS_WRONG_SECTION_TYPE,          ERROR_SXS_WRONG_SECTION_TYPE);
    cmp2(STATUS_SXS_THREAD_QUERIES_DISABLED,     ERROR_SXS_THREAD_QUERIES_DISABLED);
    cmp2(STATUS_SXS_ASSEMBLY_MISSING,            ERROR_SXS_ASSEMBLY_MISSING);
    cmp2(STATUS_SXS_PROCESS_DEFAULT_ALREADY_SET, ERROR_SXS_PROCESS_DEFAULT_ALREADY_SET);
    cmp2(STATUS_SXS_EARLY_DEACTIVATION,          ERROR_SXS_EARLY_DEACTIVATION);
    cmp2(STATUS_SXS_INVALID_DEACTIVATION,        ERROR_SXS_INVALID_DEACTIVATION);
    cmp2(STATUS_SXS_MULTIPLE_DEACTIVATION,       ERROR_SXS_MULTIPLE_DEACTIVATION);
    cmp2(STATUS_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY,        ERROR_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY);
    cmp2(STATUS_SXS_PROCESS_TERMINATION_REQUESTED,                  ERROR_SXS_PROCESS_TERMINATION_REQUESTED);
    cmp2(STATUS_SXS_CORRUPT_ACTIVATION_STACK,    ERROR_SXS_CORRUPT_ACTIVATION_STACK);
    cmp2(STATUS_SXS_CORRUPTION,                  ERROR_SXS_CORRUPTION);
    cmp2(STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE,               ERROR_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE);
    cmp2(STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME,                ERROR_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME);
    cmp2(STATUS_SXS_IDENTITY_DUPLICATE_ATTRIBUTE,                   ERROR_SXS_IDENTITY_DUPLICATE_ATTRIBUTE);
    cmp2(STATUS_SXS_IDENTITY_PARSE_ERROR,        ERROR_SXS_IDENTITY_PARSE_ERROR);
    cmp2(STATUS_SXS_COMPONENT_STORE_CORRUPT,     ERROR_SXS_COMPONENT_STORE_CORRUPT);
    cmp2(STATUS_SXS_FILE_HASH_MISMATCH,          ERROR_SXS_FILE_HASH_MISMATCH);
    cmp2(STATUS_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT,  ERROR_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT);
    cmp2(STATUS_SXS_IDENTITIES_DIFFERENT,        ERROR_SXS_IDENTITIES_DIFFERENT);
    cmp2(STATUS_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT,                   ERROR_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT);
    cmp2(STATUS_SXS_FILE_NOT_PART_OF_ASSEMBLY,   ERROR_SXS_FILE_NOT_PART_OF_ASSEMBLY);
    cmp2(STATUS_ADVANCED_INSTALLER_FAILED,       ERROR_ADVANCED_INSTALLER_FAILED);
    cmp2(STATUS_XML_ENCODING_MISMATCH,           ERROR_XML_ENCODING_MISMATCH);
    cmp2(STATUS_SXS_MANIFEST_TOO_BIG,            ERROR_SXS_MANIFEST_TOO_BIG);
    cmp2(STATUS_SXS_SETTING_NOT_REGISTERED,      ERROR_SXS_SETTING_NOT_REGISTERED);
    cmp2(STATUS_SXS_TRANSACTION_CLOSURE_INCOMPLETE,                 ERROR_SXS_TRANSACTION_CLOSURE_INCOMPLETE);
    cmp2(STATUS_SMI_PRIMITIVE_INSTALLER_FAILED,  ERROR_SMI_PRIMITIVE_INSTALLER_FAILED);
    cmp2(STATUS_GENERIC_COMMAND_FAILED,          ERROR_GENERIC_COMMAND_FAILED);
    cmp2(STATUS_SXS_FILE_HASH_MISSING,           ERROR_SXS_FILE_HASH_MISSING);
    cmp2(STATUS_REDIRECTOR_STARTED,              ERROR_SERVICE_ALREADY_RUNNING);
    cmp2(STATUS_AUDITING_DISABLED,               ERROR_AUDITING_DISABLED);
    cmp2(STATUS_AUDIT_FAILED,                    ERROR_AUDIT_FAILED);
    cmp2(STATUS_CLUSTER_NODE_ALREADY_UP,         ERROR_CLUSTER_NODE_ALREADY_UP);
    cmp2(STATUS_CLUSTER_NODE_ALREADY_DOWN,       ERROR_CLUSTER_NODE_ALREADY_DOWN);
    cmp2(STATUS_CLUSTER_NETWORK_ALREADY_ONLINE,  ERROR_CLUSTER_NETWORK_ALREADY_ONLINE);
    cmp2(STATUS_CLUSTER_NETWORK_ALREADY_OFFLINE, ERROR_CLUSTER_NETWORK_ALREADY_OFFLINE);
    cmp2(STATUS_CLUSTER_NODE_ALREADY_MEMBER,     ERROR_CLUSTER_NODE_ALREADY_MEMBER);
    cmp2(STATUS_CLUSTER_INVALID_NODE,            ERROR_CLUSTER_INVALID_NODE);
    cmp2(STATUS_CLUSTER_NODE_EXISTS,             ERROR_CLUSTER_NODE_EXISTS);
    cmp2(STATUS_CLUSTER_JOIN_IN_PROGRESS,        ERROR_CLUSTER_JOIN_IN_PROGRESS);
    cmp2(STATUS_CLUSTER_NODE_NOT_FOUND,          ERROR_CLUSTER_NODE_NOT_FOUND);
    cmp2(STATUS_CLUSTER_LOCAL_NODE_NOT_FOUND,    ERROR_CLUSTER_LOCAL_NODE_NOT_FOUND);
    cmp2(STATUS_CLUSTER_NETWORK_EXISTS,          ERROR_CLUSTER_NETWORK_EXISTS);
    cmp2(STATUS_CLUSTER_NETWORK_NOT_FOUND,       ERROR_CLUSTER_NETWORK_NOT_FOUND);
    cmp2(STATUS_CLUSTER_NETINTERFACE_EXISTS,     ERROR_CLUSTER_NETINTERFACE_EXISTS);
    cmp2(STATUS_CLUSTER_NETINTERFACE_NOT_FOUND,  ERROR_CLUSTER_NETINTERFACE_NOT_FOUND);
    cmp2(STATUS_CLUSTER_INVALID_REQUEST,         ERROR_CLUSTER_INVALID_REQUEST);
    cmp2(STATUS_CLUSTER_INVALID_NETWORK_PROVIDER,ERROR_CLUSTER_INVALID_NETWORK_PROVIDER);
    cmp2(STATUS_CLUSTER_NODE_DOWN,               ERROR_CLUSTER_NODE_DOWN);
    cmp2(STATUS_CLUSTER_NODE_UNREACHABLE,        ERROR_CLUSTER_NODE_UNREACHABLE);
    cmp2(STATUS_CLUSTER_NODE_NOT_MEMBER,         ERROR_CLUSTER_NODE_NOT_MEMBER);
    cmp2(STATUS_CLUSTER_JOIN_NOT_IN_PROGRESS,    ERROR_CLUSTER_JOIN_NOT_IN_PROGRESS);
    cmp2(STATUS_CLUSTER_INVALID_NETWORK,         ERROR_CLUSTER_INVALID_NETWORK);
    cmp2(STATUS_CLUSTER_NODE_UP,                 ERROR_CLUSTER_NODE_UP);
    cmp2(STATUS_CLUSTER_NODE_PAUSED,             ERROR_CLUSTER_NODE_PAUSED);
    cmp2(STATUS_CLUSTER_NODE_NOT_PAUSED,         ERROR_CLUSTER_NODE_NOT_PAUSED);
    cmp2(STATUS_CLUSTER_NO_SECURITY_CONTEXT,     ERROR_CLUSTER_NO_SECURITY_CONTEXT);
    cmp2(STATUS_CLUSTER_NETWORK_NOT_INTERNAL,    ERROR_CLUSTER_NETWORK_NOT_INTERNAL);
    cmp2(STATUS_NOTHING_TO_TERMINATE,            ERROR_NOTHING_TO_TERMINATE);
    cmp2(STATUS_PROCESS_NOT_IN_JOB,              ERROR_PROCESS_NOT_IN_JOB);
    cmp2(STATUS_PROCESS_IN_JOB,                  ERROR_PROCESS_IN_JOB);
    cmp2(STATUS_VOLSNAP_HIBERNATE_READY,         ERROR_VOLSNAP_HIBERNATE_READY);
    cmp2(STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY, ERROR_FSFILTER_OP_COMPLETED_SUCCESSFULLY);
    cmp2(STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED, ERROR_INTERRUPT_VECTOR_ALREADY_CONNECTED);
    cmp2(STATUS_INTERRUPT_STILL_CONNECTED,       ERROR_INTERRUPT_STILL_CONNECTED);
    cmp2(STATUS_WX86_UNSIMULATE,                 ERROR_WX86_WARNING);
    cmp2(STATUS_WX86_CONTINUE,                   ERROR_WX86_WARNING);
    cmp2(STATUS_WX86_SINGLE_STEP,                ERROR_WX86_WARNING);
    cmp2(STATUS_WX86_BREAKPOINT,                 ERROR_WX86_WARNING);
    cmp2(STATUS_WX86_EXCEPTION_CONTINUE,         ERROR_WX86_WARNING);
    cmp2(STATUS_WX86_EXCEPTION_LASTCHANCE,       ERROR_WX86_WARNING);
    cmp2(STATUS_WX86_EXCEPTION_CHAIN,            ERROR_WX86_WARNING);
    cmp2(STATUS_WX86_CREATEWX86TIB,              ERROR_WX86_WARNING);
    cmp2(STATUS_DRIVERS_LEAKING_LOCKED_PAGES,    ERROR_DRIVERS_LEAKING_LOCKED_PAGES);
    cmp2(STATUS_MESSAGE_RETRIEVED,               ERROR_INTERNAL_ERROR);
    cmp2(STATUS_SYSTEM_POWERSTATE_TRANSITION,    ERROR_SYSTEM_POWERSTATE_TRANSITION);
    cmp2(STATUS_ALPC_CHECK_COMPLETION_LIST,      ERROR_INTERNAL_ERROR);
    cmp2(STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION, ERROR_SYSTEM_POWERSTATE_COMPLEX_TRANSITION);
    cmp2(STATUS_ACCESS_AUDIT_BY_POLICY,          ERROR_ACCESS_AUDIT_BY_POLICY);
    cmp2(STATUS_ABANDON_HIBERFILE,               ERROR_ABANDON_HIBERFILE);
    cmp2(STATUS_BIZRULES_NOT_ENABLED,            ERROR_BIZRULES_NOT_ENABLED);
    cmp2(STATUS_IMAGE_AT_DIFFERENT_BASE,         ERROR_IMAGE_AT_DIFFERENT_BASE);
    cmp2(STATUS_STOPPED_ON_SYMLINK,              ERROR_STOPPED_ON_SYMLINK);
    cmp2(STATUS_CANNOT_GRANT_REQUESTED_OPLOCK,   ERROR_CANNOT_GRANT_REQUESTED_OPLOCK);
    cmp2(STATUS_NO_ACE_CONDITION,                ERROR_NO_ACE_CONDITION);
    cmp2(STATUS_DEVICE_SUPPORT_IN_PROGRESS,      ERROR_DEVICE_SUPPORT_IN_PROGRESS);
    cmp2(STATUS_DEVICE_POWER_CYCLE_REQUIRED,     STG_S_POWER_CYCLE_REQUIRED);
    cmp2(STATUS_NO_WORK_DONE,                    ERROR_NO_WORK_DONE);
    cmp2(SEC_E_UNFINISHED_CONTEXT_DELETED,       ERROR_CONTEXT_EXPIRED);
    cmp2(SEC_E_SHUTDOWN_IN_PROGRESS,             ERROR_SHUTDOWN_IN_PROGRESS);
    cmp2(SEC_E_CERT_WRONG_USAGE,                 ERROR_BAD_TOKEN_TYPE);
    cmp2(SEC_E_INVALID_PARAMETER,                ERROR_INVALID_PARAMETER);
    cmp2(SEC_E_DELEGATION_POLICY,                ERROR_ACCESS_DENIED);
    cmp2(SEC_E_POLICY_NLTM_ONLY,                 ERROR_ACCESS_DENIED);
    cmp2(SEC_E_MUTUAL_AUTH_FAILED,               ERROR_MUTUAL_AUTH_FAILED);
    cmp2(SEC_E_APPLICATION_PROTOCOL_MISMATCH,    ERROR_NOT_SUPPORTED);
    cmp2(STATUS_TIMER_NOT_CANCELED,              ERROR_TIMER_NOT_CANCELED);
    cmp2(STATUS_ILLEGAL_INSTRUCTION,             STATUS_ILLEGAL_INSTRUCTION);
    cmp2(STATUS_UNWIND,                          ERROR_UNWIND);
    cmp2(STATUS_BAD_STACK,                       ERROR_BAD_STACK);
    cmp2(STATUS_INVALID_UNWIND_TARGET,           ERROR_INVALID_UNWIND_TARGET);
    cmp2(STATUS_INVALID_PORT_ATTRIBUTES,         ERROR_INVALID_PORT_ATTRIBUTES);
    cmp2(STATUS_PORT_MESSAGE_TOO_LONG,           ERROR_PORT_MESSAGE_TOO_LONG);
    cmp2(STATUS_INVALID_QUOTA_LOWER,             ERROR_INVALID_QUOTA_LOWER);
    cmp2(STATUS_PORT_DO_NOT_DISTURB,             RPC_S_DO_NOT_DISTURB);
    cmp2(STATUS_DEVICE_ALREADY_ATTACHED,         ERROR_DEVICE_ALREADY_ATTACHED);
    cmp2(STATUS_INSTRUCTION_MISALIGNMENT,        ERROR_INSTRUCTION_MISALIGNMENT);
    cmp2(STATUS_PROFILING_NOT_STARTED,           ERROR_PROFILING_NOT_STARTED);
    cmp2(STATUS_PROFILING_NOT_STOPPED,           ERROR_PROFILING_NOT_STOPPED);
    cmp2(STATUS_COULD_NOT_INTERPRET,             ERROR_COULD_NOT_INTERPRET);
    cmp2(STATUS_PROFILING_AT_LIMIT,              ERROR_PROFILING_AT_LIMIT);
    cmp2(STATUS_CANT_WAIT,                       ERROR_CANT_WAIT);
    cmp2(STATUS_CANT_TERMINATE_SELF,             ERROR_CANT_TERMINATE_SELF);
    cmp2(STATUS_UNEXPECTED_IO_ERROR,             ERROR_IO_DEVICE);
    cmp2(STATUS_UNEXPECTED_MM_CREATE_ERR,        ERROR_UNEXPECTED_MM_CREATE_ERR);
    cmp2(STATUS_UNEXPECTED_MM_MAP_ERROR,         ERROR_UNEXPECTED_MM_MAP_ERROR);
    cmp2(STATUS_UNEXPECTED_MM_EXTEND_ERR,        ERROR_UNEXPECTED_MM_EXTEND_ERR);
    cmp2(STATUS_BAD_FUNCTION_TABLE,              ERROR_BAD_FUNCTION_TABLE);
    cmp2(STATUS_NO_GUID_TRANSLATION,             ERROR_NO_GUID_TRANSLATION);
    cmp2(STATUS_ABIOS_NOT_PRESENT,               ERROR_ABIOS_ERROR);
    cmp2(STATUS_ABIOS_LID_NOT_EXIST,             ERROR_ABIOS_ERROR);
    cmp2(STATUS_ABIOS_LID_ALREADY_OWNED,         ERROR_ABIOS_ERROR);
    cmp2(STATUS_ABIOS_NOT_LID_OWNER,             ERROR_ABIOS_ERROR);
    cmp2(STATUS_ABIOS_INVALID_COMMAND,           ERROR_ABIOS_ERROR);
    cmp2(STATUS_ABIOS_INVALID_LID,               ERROR_ABIOS_ERROR);
    cmp2(STATUS_ABIOS_SELECTOR_NOT_AVAILABLE,    ERROR_ABIOS_ERROR);
    cmp2(STATUS_ABIOS_INVALID_SELECTOR,          ERROR_ABIOS_ERROR);
    cmp2(STATUS_ILL_FORMED_SERVICE_ENTRY,        ERROR_SERVICE_NOT_FOUND);
    cmp2(STATUS_IMAGE_ALREADY_LOADED_AS_DLL,     ERROR_SERVICE_ALREADY_RUNNING);
    cmp2(STATUS_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING, ERROR_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING);
    cmp2(STATUS_SHORT_NAMES_NOT_ENABLED_ON_VOLUME, ERROR_SHORT_NAMES_NOT_ENABLED_ON_VOLUME);
    cmp2(STATUS_SECURITY_STREAM_IS_INCONSISTENT, ERROR_SECURITY_STREAM_IS_INCONSISTENT);
    cmp2(STATUS_INVALID_LOCK_RANGE,              ERROR_INVALID_LOCK_RANGE);
    cmp2(STATUS_INVALID_ACE_CONDITION,           ERROR_INVALID_ACE_CONDITION);
    cmp2(STATUS_IMAGE_SUBSYSTEM_NOT_PRESENT,     ERROR_IMAGE_SUBSYSTEM_NOT_PRESENT);
    cmp2(STATUS_NOTIFICATION_GUID_ALREADY_DEFINED, ERROR_NOTIFICATION_GUID_ALREADY_DEFINED);
    cmp2(STATUS_INVALID_EXCEPTION_HANDLER,       ERROR_INVALID_EXCEPTION_HANDLER);
    cmp2(STATUS_DUPLICATE_PRIVILEGES,            ERROR_DUPLICATE_PRIVILEGES);
    cmp2(STATUS_NOT_ALLOWED_ON_SYSTEM_FILE,      ERROR_NOT_ALLOWED_ON_SYSTEM_FILE);
    cmp2(STATUS_REPAIR_NEEDED,                   ERROR_DATA_LOST_REPAIR);
    cmp2(STATUS_QUOTA_NOT_ENABLED,               ERROR_NOT_SUPPORTED);
    cmp2(STATUS_NO_APPLICATION_PACKAGE,          APPMODEL_ERROR_NO_PACKAGE);
    cmp2(STATUS_FILE_METADATA_OPTIMIZATION_IN_PROGRESS, ERROR_FILE_METADATA_OPTIMIZATION_IN_PROGRESS);
    cmp2(STATUS_NOT_SAME_OBJECT,                 ERROR_NOT_SAME_OBJECT);
    cmp2(STATUS_FATAL_MEMORY_EXHAUSTION,         ERROR_NOT_ENOUGH_MEMORY);
    cmp2(STATUS_ERROR_PROCESS_NOT_IN_JOB,        ERROR_PROCESS_NOT_IN_JOB);
    cmp2(STATUS_CPU_SET_INVALID,                 ERROR_CPU_SET_INVALID);
    cmp2(STATUS_NETWORK_OPEN_RESTRICTION,        ERROR_NETWORK_ACCESS_DENIED);
    cmp2(STATUS_RECOVERY_FAILURE,                ERROR_RECOVERY_FAILURE);
    cmp2(STATUS_WX86_INTERNAL_ERROR,             ERROR_WX86_ERROR);
    cmp2(STATUS_WX86_FLOAT_STACK_CHECK,          ERROR_WX86_ERROR);
    cmp2(STATUS_PWD_TOO_LONG,                    ERROR_PWD_TOO_LONG);
    cmp2(STATUS_STOWED_EXCEPTION,                ERROR_INTERNAL_ERROR);
    cmp2(STATUS_CONTEXT_STOWED_EXCEPTION,        ERROR_INTERNAL_ERROR);
    cmp2(STATUS_DS_RIDMGR_DISABLED,              ERROR_DS_RIDMGR_DISABLED);
    cmp2(STATUS_POWER_STATE_INVALID,             ERROR_NOT_READY);
    cmp2(STATUS_PASSWORD_CHANGE_REQUIRED,        ERROR_PASSWORD_CHANGE_REQUIRED);
    cmp2(STATUS_LOST_MODE_LOGON_RESTRICTION,     ERROR_LOST_MODE_LOGON_RESTRICTION);
    cmp2(STATUS_DEBUGGER_INACTIVE,               ERROR_DEBUGGER_INACTIVE);
    cmp2(STATUS_APPHELP_BLOCK,                   ERROR_APPHELP_BLOCK);
    cmp2(STATUS_NO_SECRETS,                      ERROR_NO_SECRETS);
    cmp2(STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY, ERROR_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY);
    cmp2(STATUS_FAILED_STACK_SWITCH,             ERROR_NOT_ENOUGH_MEMORY);
    cmp2(STATUS_HEAP_CORRUPTION,                 ERROR_INTERNAL_ERROR);
    cmp2(STATUS_DRIVER_FAILED_PRIOR_UNLOAD,      ERROR_DRIVER_FAILED_PRIOR_UNLOAD);
    cmp2(STATUS_SMARTCARD_SILENT_CONTEXT,        NTE_SILENT_CONTEXT);
    cmp2(STATUS_PER_USER_TRUST_QUOTA_EXCEEDED,   ERROR_PER_USER_TRUST_QUOTA_EXCEEDED);
    cmp2(STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED,   ERROR_ALL_USER_TRUST_QUOTA_EXCEEDED);
    cmp2(STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED, ERROR_USER_DELETE_TRUST_QUOTA_EXCEEDED);
    cmp2(STATUS_DS_NAME_NOT_UNIQUE,              ERROR_DS_NAME_NOT_UNIQUE);
    cmp2(STATUS_DS_DUPLICATE_ID_FOUND,           ERROR_DS_DUPLICATE_ID_FOUND);
    cmp2(STATUS_DS_GROUP_CONVERSION_ERROR,       ERROR_DS_GROUP_CONVERSION_ERROR);
    cmp2(STATUS_VOLSNAP_PREPARE_HIBERNATE,       ERROR_VOLSNAP_PREPARE_HIBERNATE);
    cmp2(STATUS_USER2USER_REQUIRED,              ERROR_LOGON_FAILURE);
    cmp2(STATUS_STACK_BUFFER_OVERRUN,            ERROR_STACK_BUFFER_OVERRUN);
    cmp2(STATUS_NO_S4U_PROT_SUPPORT,             SEC_E_NO_S4U_PROT_SUPPORT);
    cmp2(STATUS_CROSSREALM_DELEGATION_FAILURE,   SEC_E_CROSSREALM_DELEGATION_FAILURE);
    cmp2(STATUS_REVOCATION_OFFLINE_KDC,          SEC_E_REVOCATION_OFFLINE_KDC);
    cmp2(STATUS_ISSUING_CA_UNTRUSTED_KDC,        SEC_E_ISSUING_CA_UNTRUSTED_KDC);
    cmp2(STATUS_KDC_CERT_EXPIRED,                SEC_E_KDC_CERT_EXPIRED);
    cmp2(STATUS_KDC_CERT_REVOKED,                SEC_E_KDC_CERT_REVOKED);
    cmp2(STATUS_PARAMETER_QUOTA_EXCEEDED,        ERROR_PARAMETER_QUOTA_EXCEEDED);
    cmp2(STATUS_HIBERNATION_FAILURE,             ERROR_HIBERNATION_FAILURE);
    cmp2(STATUS_DELAY_LOAD_FAILED,               ERROR_DELAY_LOAD_FAILED);
    cmp2(STATUS_AUTHENTICATION_FIREWALL_FAILED,  ERROR_AUTHENTICATION_FIREWALL_FAILED);
    cmp2(STATUS_VDM_DISALLOWED,                  ERROR_VDM_DISALLOWED);
    cmp2(STATUS_HUNG_DISPLAY_DRIVER_THREAD,      ERROR_HUNG_DISPLAY_DRIVER_THREAD);
    cmp2(STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE, ERROR_NOT_ENOUGH_MEMORY);
    cmp2(STATUS_INVALID_CRUNTIME_PARAMETER,      ERROR_INVALID_CRUNTIME_PARAMETER);
    cmp2(STATUS_NTLM_BLOCKED,                    ERROR_NTLM_BLOCKED);
    cmp2(STATUS_DS_SRC_SID_EXISTS_IN_FOREST,     ERROR_DS_SRC_SID_EXISTS_IN_FOREST);
    cmp2(STATUS_DS_DOMAIN_NAME_EXISTS_IN_FOREST, ERROR_DS_DOMAIN_NAME_EXISTS_IN_FOREST);
    cmp2(STATUS_DS_FLAT_NAME_EXISTS_IN_FOREST,   ERROR_DS_FLAT_NAME_EXISTS_IN_FOREST);
    cmp2(STATUS_INVALID_USER_PRINCIPAL_NAME,     ERROR_INVALID_USER_PRINCIPAL_NAME);
    cmp2(STATUS_FATAL_USER_CALLBACK_EXCEPTION,   ERROR_FATAL_APP_EXIT);
    cmp2(STATUS_ASSERTION_FAILURE,               ERROR_ASSERTION_FAILURE);
    cmp2(STATUS_VERIFIER_STOP,                   ERROR_VERIFIER_STOP);
    cmp2(STATUS_CALLBACK_POP_STACK,              ERROR_CALLBACK_POP_STACK);
    cmp2(STATUS_INCOMPATIBLE_DRIVER_BLOCKED,     ERROR_DRIVER_BLOCKED);
    cmp2(STATUS_HIVE_UNLOADED,                   ERROR_KEY_DELETED);
    cmp2(STATUS_COMPRESSION_DISABLED,            ERROR_COMPRESSION_DISABLED);
    cmp2(STATUS_FILE_SYSTEM_LIMITATION,          ERROR_FILE_SYSTEM_LIMITATION);
    cmp2(STATUS_INVALID_IMAGE_HASH,              ERROR_INVALID_IMAGE_HASH);
    cmp2(STATUS_NOT_CAPABLE,                     ERROR_NOT_CAPABLE);
    cmp2(STATUS_REQUEST_OUT_OF_SEQUENCE,         ERROR_REQUEST_OUT_OF_SEQUENCE);
    cmp2(STATUS_IMPLEMENTATION_LIMIT,            ERROR_IMPLEMENTATION_LIMIT);
    cmp2(STATUS_NO_SECURITY_CONTEXT,             SEC_E_NO_CONTEXT);
    cmp2(STATUS_PKU2U_CERT_FAILURE,              SEC_E_PKU2U_CERT_FAILURE);
    cmp2(STATUS_BEYOND_VDL,                      ERROR_BEYOND_VDL);
    cmp2(STATUS_ENCOUNTERED_WRITE_IN_PROGRESS,   ERROR_BUSY);
    cmp2(STATUS_PTE_CHANGED,                     ERROR_BUSY);
    cmp2(STATUS_PURGE_FAILED,                    ERROR_USER_MAPPED_FILE);
    cmp2(STATUS_CRED_REQUIRES_CONFIRMATION,      ERROR_CRED_REQUIRES_CONFIRMATION);
    cmp2(STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE, ERROR_CS_ENCRYPTION_INVALID_SERVER_RESPONSE);
    cmp2(STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER, ERROR_CS_ENCRYPTION_UNSUPPORTED_SERVER);
    cmp2(STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE, ERROR_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE);
    cmp2(STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE, ERROR_CS_ENCRYPTION_NEW_ENCRYPTED_FILE);
    cmp2(STATUS_CS_ENCRYPTION_FILE_NOT_CSE,      ERROR_CS_ENCRYPTION_FILE_NOT_CSE);
    cmp2(STATUS_INVALID_LABEL,                   ERROR_INVALID_LABEL);
    cmp2(STATUS_DRIVER_PROCESS_TERMINATED,       ERROR_DRIVER_PROCESS_TERMINATED);
    cmp2(STATUS_AMBIGUOUS_SYSTEM_DEVICE,         ERROR_AMBIGUOUS_SYSTEM_DEVICE);
    cmp2(STATUS_SYSTEM_DEVICE_NOT_FOUND,         ERROR_SYSTEM_DEVICE_NOT_FOUND);
    cmp2(STATUS_RESTART_BOOT_APPLICATION,        ERROR_RESTART_APPLICATION);
    cmp2(STATUS_INSUFFICIENT_NVRAM_RESOURCES,    ERROR_NO_NVRAM_RESOURCES);
    cmp2(STATUS_INVALID_SESSION,                 ERROR_INVALID_HANDLE);
    cmp2(STATUS_THREAD_ALREADY_IN_SESSION,       ERROR_INVALID_PARAMETER);
    cmp2(STATUS_THREAD_NOT_IN_SESSION,           ERROR_INVALID_PARAMETER);
    cmp2(STATUS_INVALID_WEIGHT,                  ERROR_INVALID_PARAMETER);
    cmp2(STATUS_REQUEST_PAUSED,                  ERROR_REQUEST_PAUSED);
    cmp2(STATUS_NO_RANGES_PROCESSED,             ERROR_NO_RANGES_PROCESSED);
    cmp2(STATUS_DISK_RESOURCES_EXHAUSTED,        ERROR_DISK_RESOURCES_EXHAUSTED);
    cmp2(STATUS_NEEDS_REMEDIATION,               ERROR_NEEDS_REMEDIATION);
    cmp2(STATUS_DEVICE_FEATURE_NOT_SUPPORTED,    ERROR_DEVICE_FEATURE_NOT_SUPPORTED);
    cmp2(STATUS_DEVICE_UNREACHABLE,              ERROR_DEVICE_UNREACHABLE);
    cmp2(STATUS_INVALID_TOKEN,                   ERROR_INVALID_TOKEN);
    cmp2(STATUS_SERVER_UNAVAILABLE,              ERROR_NETNAME_DELETED);
    cmp2(STATUS_FILE_NOT_AVAILABLE,              ERROR_SHARING_VIOLATION);
    cmp2(STATUS_DEVICE_INSUFFICIENT_RESOURCES,   ERROR_DEVICE_NO_RESOURCES);
    cmp2(STATUS_PACKAGE_UPDATING,                ERROR_PACKAGE_UPDATING);
    cmp2(STATUS_NOT_READ_FROM_COPY,              ERROR_NOT_READ_FROM_COPY);
    cmp2(STATUS_FT_WRITE_FAILURE,                ERROR_FT_WRITE_FAILURE);
    cmp2(STATUS_FT_DI_SCAN_REQUIRED,             ERROR_FT_DI_SCAN_REQUIRED);
    cmp2(STATUS_OBJECT_NOT_EXTERNALLY_BACKED,    ERROR_OBJECT_NOT_EXTERNALLY_BACKED);
    cmp2(STATUS_EXTERNAL_BACKING_PROVIDER_UNKNOWN, ERROR_EXTERNAL_BACKING_PROVIDER_UNKNOWN);
    cmp2(STATUS_COMPRESSION_NOT_BENEFICIAL,      ERROR_COMPRESSION_NOT_BENEFICIAL);
    cmp2(STATUS_DATA_CHECKSUM_ERROR,             ERROR_DATA_CHECKSUM_ERROR);
    cmp2(STATUS_INTERMIXED_KERNEL_EA_OPERATION,  ERROR_INTERMIXED_KERNEL_EA_OPERATION);
    cmp2(STATUS_TRIM_READ_ZERO_NOT_SUPPORTED,    ERROR_FILE_LEVEL_TRIM_NOT_SUPPORTED);
    cmp2(STATUS_TOO_MANY_SEGMENT_DESCRIPTORS,    ERROR_TOO_MANY_DESCRIPTORS);
    cmp2(STATUS_INVALID_OFFSET_ALIGNMENT,        ERROR_OFFSET_ALIGNMENT_VIOLATION);
    cmp2(STATUS_INVALID_FIELD_IN_PARAMETER_LIST, ERROR_INVALID_FIELD_IN_PARAMETER_LIST);
    cmp2(STATUS_OPERATION_IN_PROGRESS,           ERROR_OPERATION_IN_PROGRESS);
    cmp2(STATUS_INVALID_INITIATOR_TARGET_PATH,   ERROR_BAD_DEVICE_PATH);
    cmp2(STATUS_SCRUB_DATA_DISABLED,             ERROR_SCRUB_DATA_DISABLED);
    cmp2(STATUS_NOT_REDUNDANT_STORAGE,           ERROR_NOT_REDUNDANT_STORAGE);
    cmp2(STATUS_RESIDENT_FILE_NOT_SUPPORTED,     ERROR_RESIDENT_FILE_NOT_SUPPORTED);
    cmp2(STATUS_COMPRESSED_FILE_NOT_SUPPORTED,   ERROR_COMPRESSED_FILE_NOT_SUPPORTED);
    cmp2(STATUS_DIRECTORY_NOT_SUPPORTED,         ERROR_DIRECTORY_NOT_SUPPORTED);
    cmp2(STATUS_IO_OPERATION_TIMEOUT,            ERROR_TIMEOUT);
    cmp2(STATUS_SYSTEM_NEEDS_REMEDIATION,        ERROR_SYSTEM_NEEDS_REMEDIATION);
    cmp2(STATUS_APPX_INTEGRITY_FAILURE_CLR_NGEN, ERROR_APPX_INTEGRITY_FAILURE_CLR_NGEN);
    cmp2(STATUS_SHARE_UNAVAILABLE,               ERROR_NETNAME_DELETED);
    cmp2(STATUS_APISET_NOT_HOSTED,               ERROR_MOD_NOT_FOUND);
    cmp2(STATUS_APISET_NOT_PRESENT,              ERROR_MOD_NOT_FOUND);
    cmp2(STATUS_DEVICE_HARDWARE_ERROR,           ERROR_DEVICE_HARDWARE_ERROR);
    cmp2(STATUS_FIRMWARE_SLOT_INVALID,           STG_E_FIRMWARE_SLOT_INVALID);
    cmp2(STATUS_FIRMWARE_IMAGE_INVALID,          STG_E_FIRMWARE_IMAGE_INVALID);
    cmp2(STATUS_STORAGE_TOPOLOGY_ID_MISMATCH,    ERROR_STORAGE_TOPOLOGY_ID_MISMATCH);
    cmp2(STATUS_WIM_NOT_BOOTABLE,                ERROR_GEN_FAILURE);
    cmp2(STATUS_BLOCKED_BY_PARENTAL_CONTROLS,    ERROR_BLOCKED_BY_PARENTAL_CONTROLS);
    cmp2(STATUS_NEEDS_REGISTRATION,              ERROR_NEEDS_REGISTRATION);
    cmp2(STATUS_QUOTA_ACTIVITY,                  ERROR_QUOTA_ACTIVITY);
    cmp2(STATUS_CALLBACK_INVOKE_INLINE,          ERROR_CALLBACK_INVOKE_INLINE);
    cmp2(STATUS_BLOCK_TOO_MANY_REFERENCES,       ERROR_BLOCK_TOO_MANY_REFERENCES);
    cmp2(STATUS_MARKED_TO_DISALLOW_WRITES,       ERROR_MARKED_TO_DISALLOW_WRITES);
    cmp2(STATUS_NETWORK_ACCESS_DENIED_EDP,       ERROR_NETWORK_ACCESS_DENIED_EDP);
    cmp2(STATUS_ENCLAVE_FAILURE,                 ERROR_ENCLAVE_FAILURE);
    cmp2(STATUS_PNP_NO_COMPAT_DRIVERS,           ERROR_NO_MATCH);
    cmp2(STATUS_PNP_DRIVER_PACKAGE_NOT_FOUND,    ERROR_FILE_NOT_FOUND);
    cmp2(STATUS_PNP_DRIVER_CONFIGURATION_NOT_FOUND, ERROR_NOT_FOUND);
    cmp2(STATUS_PNP_DRIVER_CONFIGURATION_INCOMPLETE, ERROR_SET_NOT_FOUND);
    cmp2(STATUS_PNP_FUNCTION_DRIVER_REQUIRED,    ERROR_NOT_CAPABLE);
    cmp2(STATUS_PNP_DEVICE_CONFIGURATION_PENDING, ERROR_NOT_READY);
    cmp2(STATUS_DEVICE_HINT_NAME_BUFFER_TOO_SMALL, ERROR_DEVICE_HINT_NAME_BUFFER_TOO_SMALL);
    cmp2(STATUS_PACKAGE_NOT_AVAILABLE,           APPMODEL_ERROR_PACKAGE_NOT_AVAILABLE);
    cmp2(STATUS_DEVICE_IN_MAINTENANCE,           ERROR_DEVICE_IN_MAINTENANCE);
    cmp2(STATUS_NOT_SUPPORTED_ON_DAX,            ERROR_NOT_SUPPORTED_ON_DAX);
    cmp2(STATUS_FREE_SPACE_TOO_FRAGMENTED,       ERROR_DISK_TOO_FRAGMENTED);
    cmp2(STATUS_DAX_MAPPING_EXISTS,              ERROR_DAX_MAPPING_EXISTS);
    cmp2(STATUS_CHILD_PROCESS_BLOCKED,           ERROR_CHILD_PROCESS_BLOCKED);
    cmp2(STATUS_STORAGE_LOST_DATA_PERSISTENCE,   ERROR_STORAGE_LOST_DATA_PERSISTENCE);
    cmp2(STATUS_VRF_CFG_ENABLED,                 ERROR_VRF_CFG_ENABLED);
    cmp2(STATUS_PARTITION_TERMINATING,           ERROR_PARTITION_TERMINATING);
    cmp2(STATUS_EXTERNAL_SYSKEY_NOT_SUPPORTED,   ERROR_EXTERNAL_SYSKEY_NOT_SUPPORTED);
    cmp2(STATUS_FAIL_FAST_EXCEPTION,             ERROR_FAIL_FAST_EXCEPTION);
    cmp2(STATUS_IMAGE_CERT_REVOKED,              CERT_E_REVOKED);
    cmp2(STATUS_DYNAMIC_CODE_BLOCKED,            ERROR_DYNAMIC_CODE_BLOCKED);
    cmp2(STATUS_IMAGE_CERT_EXPIRED,              CERT_E_EXPIRED);
    cmp2(STATUS_STRICT_CFG_VIOLATION,            ERROR_STRICT_CFG_VIOLATION);
    cmp2(STATUS_SET_CONTEXT_DENIED,              ERROR_SET_CONTEXT_DENIED);
    cmp2(STATUS_CROSS_PARTITION_VIOLATION,       ERROR_CROSS_PARTITION_VIOLATION);
    cmp2(STATUS_PORT_CLOSED,                     ERROR_INTERNAL_ERROR);
    cmp2(STATUS_MESSAGE_LOST,                    ERROR_INTERNAL_ERROR);
    cmp2(STATUS_INVALID_MESSAGE,                 ERROR_INVALID_PARAMETER);
    cmp2(STATUS_REQUEST_CANCELED,                ERROR_INTERNAL_ERROR);
    cmp2(STATUS_RECURSIVE_DISPATCH,              ERROR_NOT_SUPPORTED);
    cmp2(STATUS_LPC_RECEIVE_BUFFER_EXPECTED,     ERROR_INVALID_PARAMETER);
    cmp2(STATUS_LPC_INVALID_CONNECTION_USAGE,    ERROR_INVALID_PARAMETER);
    cmp2(STATUS_LPC_REQUESTS_NOT_ALLOWED,        ERROR_NOT_SUPPORTED);
    cmp2(STATUS_RESOURCE_IN_USE,                 ERROR_INTERNAL_ERROR);
    cmp2(STATUS_HARDWARE_MEMORY_ERROR,           ERROR_MEMORY_HARDWARE);
    cmp2(STATUS_THREADPOOL_HANDLE_EXCEPTION,     ERROR_INVALID_HANDLE);
    cmp2(STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED, ERROR_INVALID_HANDLE);
    cmp2(STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED, ERROR_INVALID_HANDLE);
    cmp2(STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED, ERROR_INVALID_HANDLE);
    cmp2(STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED, ERROR_INVALID_HANDLE);
    cmp2(STATUS_THREADPOOL_RELEASED_DURING_OPERATION, ERROR_INVALID_HANDLE);
    cmp2(STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING, ERROR_INVALID_FUNCTION);
    cmp2(STATUS_APC_RETURNED_WHILE_IMPERSONATING, ERROR_INVALID_FUNCTION);
    cmp2(STATUS_PROCESS_IS_PROTECTED,            ERROR_PROCESS_IS_PROTECTED);
    cmp2(STATUS_MCA_EXCEPTION,                   ERROR_MCA_EXCEPTION);
    cmp2(STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE,  ERROR_LOGON_FAILURE);
    cmp2(STATUS_SYMLINK_CLASS_DISABLED,          ERROR_SYMLINK_CLASS_DISABLED);
    cmp2(STATUS_INVALID_IDN_NORMALIZATION,       ERROR_INVALID_NAME);
    cmp2(STATUS_NO_UNICODE_TRANSLATION,          ERROR_NO_UNICODE_TRANSLATION);
    cmp2(STATUS_ALREADY_REGISTERED,              ERROR_INTERNAL_ERROR);
    cmp2(STATUS_CONTEXT_MISMATCH,                ERROR_INTERNAL_ERROR);
    cmp2(STATUS_PORT_ALREADY_HAS_COMPLETION_LIST, ERROR_INTERNAL_ERROR);
    cmp2(STATUS_CALLBACK_RETURNED_THREAD_PRIORITY, ERROR_INVALID_FUNCTION);
    cmp2(STATUS_INVALID_THREAD,                  ERROR_INVALID_PARAMETER);
    cmp2(STATUS_CALLBACK_RETURNED_TRANSACTION,   ERROR_INVALID_FUNCTION);
    cmp2(STATUS_CALLBACK_RETURNED_LDR_LOCK,      ERROR_INVALID_FUNCTION);
    cmp2(STATUS_CALLBACK_RETURNED_LANG,          ERROR_INVALID_FUNCTION);
    cmp2(STATUS_CALLBACK_RETURNED_PRI_BACK,      ERROR_INVALID_FUNCTION);
    cmp2(STATUS_CALLBACK_RETURNED_THREAD_AFFINITY, ERROR_INVALID_FUNCTION);
    cmp2(STATUS_LPC_HANDLE_COUNT_EXCEEDED,       RPC_S_SYSTEM_HANDLE_COUNT_EXCEEDED);
    cmp2(STATUS_EXECUTABLE_MEMORY_WRITE,         ERROR_GEN_FAILURE);
    cmp2(STATUS_KERNEL_EXECUTABLE_MEMORY_WRITE,  ERROR_GEN_FAILURE);
    cmp2(STATUS_ATTACHED_EXECUTABLE_MEMORY_WRITE, ERROR_GEN_FAILURE);
    cmp2(STATUS_TRIGGERED_EXECUTABLE_MEMORY_WRITE, ERROR_GEN_FAILURE);
    cmp2(STATUS_DISK_REPAIR_DISABLED,            ERROR_DISK_REPAIR_DISABLED);
    cmp2(STATUS_DS_DOMAIN_RENAME_IN_PROGRESS,    ERROR_DS_DOMAIN_RENAME_IN_PROGRESS);
    cmp2(STATUS_DISK_QUOTA_EXCEEDED,             ERROR_DISK_QUOTA_EXCEEDED);
    cmp2(STATUS_CONTENT_BLOCKED,                 ERROR_CONTENT_BLOCKED);
    cmp2(STATUS_BAD_CLUSTERS,                    ERROR_BAD_CLUSTERS);
    cmp2(STATUS_VOLUME_DIRTY,                    ERROR_VOLUME_DIRTY);
    cmp2(STATUS_DISK_REPAIR_UNSUCCESSFUL,        ERROR_DISK_REPAIR_UNSUCCESSFUL);
    cmp2(STATUS_CORRUPT_LOG_OVERFULL,            ERROR_CORRUPT_LOG_OVERFULL);
    cmp2(STATUS_CORRUPT_LOG_CORRUPTED,           ERROR_CORRUPT_LOG_CORRUPTED);
    cmp2(STATUS_CORRUPT_LOG_UNAVAILABLE,         ERROR_CORRUPT_LOG_UNAVAILABLE);
    cmp2(STATUS_CORRUPT_LOG_DELETED_FULL,        ERROR_CORRUPT_LOG_DELETED_FULL);
    cmp2(STATUS_CORRUPT_LOG_CLEARED,             ERROR_CORRUPT_LOG_CLEARED);
    cmp2(STATUS_ORPHAN_NAME_EXHAUSTED,           ERROR_ORPHAN_NAME_EXHAUSTED);
    cmp2(STATUS_PROACTIVE_SCAN_IN_PROGRESS,      ERROR_RETRY);
    cmp2(STATUS_ENCRYPTED_IO_NOT_POSSIBLE,       ERROR_ENCRYPTED_IO_NOT_POSSIBLE);
    cmp2(STATUS_CORRUPT_LOG_UPLEVEL_RECORDS,     ERROR_INTERNAL_ERROR);
    cmp2(STATUS_FILE_CHECKED_OUT,                ERROR_FILE_CHECKED_OUT);
    cmp2(STATUS_CHECKOUT_REQUIRED,               ERROR_CHECKOUT_REQUIRED);
    cmp2(STATUS_BAD_FILE_TYPE,                   ERROR_BAD_FILE_TYPE);
    cmp2(STATUS_FILE_TOO_LARGE,                  ERROR_FILE_TOO_LARGE);
    cmp2(STATUS_FORMS_AUTH_REQUIRED,             ERROR_FORMS_AUTH_REQUIRED);
    cmp2(STATUS_VIRUS_INFECTED,                  ERROR_VIRUS_INFECTED);
    cmp2(STATUS_VIRUS_DELETED,                   ERROR_VIRUS_DELETED);
    cmp2(STATUS_BAD_MCFG_TABLE,                  ERROR_BAD_MCFG_TABLE);
    cmp2(STATUS_CANNOT_BREAK_OPLOCK,             ERROR_CANNOT_BREAK_OPLOCK);
    cmp2(STATUS_BAD_KEY,                         NTE_BAD_KEY);
    cmp2(STATUS_BAD_DATA,                        NTE_BAD_DATA);
    cmp2(STATUS_NO_KEY,                          NTE_NO_KEY);
    cmp2(STATUS_FILE_HANDLE_REVOKED,             ERROR_FILE_HANDLE_REVOKED);
    cmp2(RPC_NT_PROXY_ACCESS_DENIED,             RPC_S_PROXY_ACCESS_DENIED);
    cmp2(RPC_NT_COOKIE_AUTH_FAILED,              RPC_S_COOKIE_AUTH_FAILED);
    cmp2(STATUS_RDP_PROTOCOL_ERROR,              ERROR_RDP_PROTOCOL_ERROR);
    cmp2(STATUS_CTX_LOGON_DISABLED,              ERROR_CTX_LOGON_DISABLED);
    cmp2(STATUS_CTX_SECURITY_LAYER_ERROR,        ERROR_CTX_SECURITY_LAYER_ERROR);
    cmp2(STATUS_TS_INCOMPATIBLE_SESSIONS,        ERROR_TS_INCOMPATIBLE_SESSIONS);
    cmp2(STATUS_TS_VIDEO_SUBSYSTEM_ERROR,        ERROR_TS_VIDEO_SUBSYSTEM_ERROR);
    cmp2(STATUS_MUI_FILE_NOT_FOUND,              ERROR_MUI_FILE_NOT_FOUND);
    cmp2(STATUS_MUI_INVALID_FILE,                ERROR_MUI_INVALID_FILE);
    cmp2(STATUS_MUI_INVALID_RC_CONFIG,           ERROR_MUI_INVALID_RC_CONFIG);
    cmp2(STATUS_MUI_INVALID_LOCALE_NAME,         ERROR_MUI_INVALID_LOCALE_NAME);
    cmp2(STATUS_MUI_INVALID_ULTIMATEFALLBACK_NAME, ERROR_MUI_INVALID_ULTIMATEFALLBACK_NAME);
    cmp2(STATUS_MUI_FILE_NOT_LOADED,             ERROR_MUI_FILE_NOT_LOADED);
    cmp2(STATUS_RESOURCE_ENUM_USER_STOP,         ERROR_RESOURCE_ENUM_USER_STOP);
    cmp2(STATUS_CLUSTER_NO_NET_ADAPTERS,         ERROR_CLUSTER_NO_NET_ADAPTERS);
    cmp2(STATUS_CLUSTER_POISONED,                ERROR_CLUSTER_POISONED);
    cmp2(STATUS_CLUSTER_NON_CSV_PATH,            ERROR_NON_CSV_PATH);
    cmp2(STATUS_CLUSTER_CSV_VOLUME_NOT_LOCAL,    ERROR_CSV_VOLUME_NOT_LOCAL);
    cmp2(STATUS_CLUSTER_CSV_READ_OPLOCK_BREAK_IN_PROGRESS, ERROR_RETRY);
    cmp2(STATUS_CLUSTER_CSV_AUTO_PAUSE_ERROR,    ERROR_RETRY);
    cmp2(STATUS_CLUSTER_CSV_REDIRECTED,          ERROR_NOT_SUPPORTED);
    cmp2(STATUS_CLUSTER_CSV_NOT_REDIRECTED,      ERROR_NOT_SUPPORTED);
    cmp2(STATUS_CLUSTER_CSV_VOLUME_DRAINING,     ERROR_NOT_SUPPORTED);
    cmp2(STATUS_CLUSTER_CSV_SNAPSHOT_CREATION_IN_PROGRESS, ERROR_RETRY);
    cmp2(STATUS_CLUSTER_CSV_VOLUME_DRAINING_SUCCEEDED_DOWNLEVEL, ERROR_NOT_SUPPORTED);
    cmp2(STATUS_CLUSTER_CSV_NO_SNAPSHOTS,        ERROR_RETRY);
    cmp2(STATUS_CSV_IO_PAUSE_TIMEOUT,            ERROR_CLUSTER_CSV_IO_PAUSE_TIMEOUT);
    cmp2(STATUS_CLUSTER_CSV_INVALID_HANDLE,      ERROR_CLUSTER_CSV_INVALID_HANDLE);
    cmp2(STATUS_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR, ERROR_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR);
    cmp2(STATUS_CLUSTER_CAM_TICKET_REPLAY_DETECTED, ERROR_ACCESS_DENIED);
    cmp2(STATUS_VOLMGR_DATABASE_FULL,            ERROR_VOLMGR_DATABASE_FULL);
    cmp2(STATUS_VOLMGR_DISK_CONFIGURATION_CORRUPTED, ERROR_VOLMGR_DISK_CONFIGURATION_CORRUPTED);
    cmp2(STATUS_VOLMGR_DISK_CONFIGURATION_NOT_IN_SYNC, ERROR_VOLMGR_DISK_CONFIGURATION_NOT_IN_SYNC);
    cmp2(STATUS_VOLMGR_PACK_CONFIG_UPDATE_FAILED, ERROR_VOLMGR_PACK_CONFIG_UPDATE_FAILED);
    cmp2(STATUS_VOLMGR_DISK_CONTAINS_NON_SIMPLE_VOLUME, ERROR_VOLMGR_DISK_CONTAINS_NON_SIMPLE_VOLUME);
    cmp2(STATUS_VOLMGR_DISK_DUPLICATE,           ERROR_VOLMGR_DISK_DUPLICATE);
    cmp2(STATUS_VOLMGR_DISK_DYNAMIC,             ERROR_VOLMGR_DISK_DYNAMIC);
    cmp2(STATUS_VOLMGR_DISK_ID_INVALID,          ERROR_VOLMGR_DISK_ID_INVALID);
    cmp2(STATUS_VOLMGR_DISK_INVALID,             ERROR_VOLMGR_DISK_INVALID);
    cmp2(STATUS_VOLMGR_DISK_LAST_VOTER,          ERROR_VOLMGR_DISK_LAST_VOTER);
    cmp2(STATUS_VOLMGR_DISK_LAYOUT_INVALID,      ERROR_VOLMGR_DISK_LAYOUT_INVALID);
    cmp2(STATUS_VOLMGR_DISK_LAYOUT_NON_BASIC_BETWEEN_BASIC_PARTITIONS, ERROR_VOLMGR_DISK_LAYOUT_NON_BASIC_BETWEEN_BASIC_PARTITIONS);
    cmp2(STATUS_VOLMGR_DISK_LAYOUT_NOT_CYLINDER_ALIGNED, ERROR_VOLMGR_DISK_LAYOUT_NOT_CYLINDER_ALIGNED);
    cmp2(STATUS_VOLMGR_DISK_LAYOUT_PARTITIONS_TOO_SMALL, ERROR_VOLMGR_DISK_LAYOUT_PARTITIONS_TOO_SMALL);
    cmp2(STATUS_VOLMGR_DISK_LAYOUT_PRIMARY_BETWEEN_LOGICAL_PARTITIONS, ERROR_VOLMGR_DISK_LAYOUT_PRIMARY_BETWEEN_LOGICAL_PARTITIONS);
    cmp2(STATUS_VOLMGR_DISK_LAYOUT_TOO_MANY_PARTITIONS, ERROR_VOLMGR_DISK_LAYOUT_TOO_MANY_PARTITIONS);
    cmp2(STATUS_VOLMGR_DISK_MISSING,             ERROR_VOLMGR_DISK_MISSING);
    cmp2(STATUS_VOLMGR_DISK_NOT_EMPTY,           ERROR_VOLMGR_DISK_NOT_EMPTY);
    cmp2(STATUS_VOLMGR_DISK_NOT_ENOUGH_SPACE,    ERROR_VOLMGR_DISK_NOT_ENOUGH_SPACE);
    cmp2(STATUS_VOLMGR_DISK_REVECTORING_FAILED,  ERROR_VOLMGR_DISK_REVECTORING_FAILED);
    cmp2(STATUS_VOLMGR_DISK_SECTOR_SIZE_INVALID, ERROR_VOLMGR_DISK_SECTOR_SIZE_INVALID);
    cmp2(STATUS_VOLMGR_DISK_SET_NOT_CONTAINED,   ERROR_VOLMGR_DISK_SET_NOT_CONTAINED);
    cmp2(STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_MEMBERS, ERROR_VOLMGR_DISK_USED_BY_MULTIPLE_MEMBERS);
    cmp2(STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_PLEXES, ERROR_VOLMGR_DISK_USED_BY_MULTIPLE_PLEXES);
    cmp2(STATUS_VOLMGR_DYNAMIC_DISK_NOT_SUPPORTED, ERROR_VOLMGR_DYNAMIC_DISK_NOT_SUPPORTED);
    cmp2(STATUS_VOLMGR_EXTENT_ALREADY_USED,      ERROR_VOLMGR_EXTENT_ALREADY_USED);
    cmp2(STATUS_VOLMGR_EXTENT_NOT_CONTIGUOUS,    ERROR_VOLMGR_EXTENT_NOT_CONTIGUOUS);
    cmp2(STATUS_VOLMGR_EXTENT_NOT_IN_PUBLIC_REGION, ERROR_VOLMGR_EXTENT_NOT_IN_PUBLIC_REGION);
    cmp2(STATUS_VOLMGR_EXTENT_NOT_SECTOR_ALIGNED, ERROR_VOLMGR_EXTENT_NOT_SECTOR_ALIGNED);
    cmp2(STATUS_VOLMGR_EXTENT_OVERLAPS_EBR_PARTITION, ERROR_VOLMGR_EXTENT_OVERLAPS_EBR_PARTITION);
    cmp2(STATUS_VOLMGR_EXTENT_VOLUME_LENGTHS_DO_NOT_MATCH, ERROR_VOLMGR_EXTENT_VOLUME_LENGTHS_DO_NOT_MATCH);
    cmp2(STATUS_VOLMGR_FAULT_TOLERANT_NOT_SUPPORTED, ERROR_VOLMGR_FAULT_TOLERANT_NOT_SUPPORTED);
    cmp2(STATUS_VOLMGR_INTERLEAVE_LENGTH_INVALID, ERROR_VOLMGR_INTERLEAVE_LENGTH_INVALID);
    cmp2(STATUS_VOLMGR_MAXIMUM_REGISTERED_USERS, ERROR_VOLMGR_MAXIMUM_REGISTERED_USERS);
    cmp2(STATUS_VOLMGR_MEMBER_IN_SYNC,           ERROR_VOLMGR_MEMBER_IN_SYNC);
    cmp2(STATUS_VOLMGR_MEMBER_INDEX_DUPLICATE,   ERROR_VOLMGR_MEMBER_INDEX_DUPLICATE);
    cmp2(STATUS_VOLMGR_MEMBER_INDEX_INVALID,     ERROR_VOLMGR_MEMBER_INDEX_INVALID);
    cmp2(STATUS_VOLMGR_MEMBER_MISSING,           ERROR_VOLMGR_MEMBER_MISSING);
    cmp2(STATUS_VOLMGR_MEMBER_NOT_DETACHED,      ERROR_VOLMGR_MEMBER_NOT_DETACHED);
    cmp2(STATUS_VOLMGR_MEMBER_REGENERATING,      ERROR_VOLMGR_MEMBER_REGENERATING);
    cmp2(STATUS_VOLMGR_ALL_DISKS_FAILED,         ERROR_VOLMGR_ALL_DISKS_FAILED);
    cmp2(STATUS_VOLMGR_NO_REGISTERED_USERS,      ERROR_VOLMGR_NO_REGISTERED_USERS);
    cmp2(STATUS_VOLMGR_NO_SUCH_USER,             ERROR_VOLMGR_NO_SUCH_USER);
    cmp2(STATUS_VOLMGR_NOTIFICATION_RESET,       ERROR_VOLMGR_NOTIFICATION_RESET);
    cmp2(STATUS_VOLMGR_NUMBER_OF_MEMBERS_INVALID,ERROR_VOLMGR_NUMBER_OF_MEMBERS_INVALID);
    cmp2(STATUS_VOLMGR_NUMBER_OF_PLEXES_INVALID, ERROR_VOLMGR_NUMBER_OF_PLEXES_INVALID);
    cmp2(STATUS_VOLMGR_PACK_DUPLICATE,           ERROR_VOLMGR_PACK_DUPLICATE);
    cmp2(STATUS_VOLMGR_PACK_ID_INVALID,          ERROR_VOLMGR_PACK_ID_INVALID);
    cmp2(STATUS_VOLMGR_PACK_INVALID,             ERROR_VOLMGR_PACK_INVALID);
    cmp2(STATUS_VOLMGR_PACK_NAME_INVALID,        ERROR_VOLMGR_PACK_NAME_INVALID);
    cmp2(STATUS_VOLMGR_PACK_OFFLINE,             ERROR_VOLMGR_PACK_OFFLINE);
    cmp2(STATUS_VOLMGR_PACK_HAS_QUORUM,          ERROR_VOLMGR_PACK_HAS_QUORUM);
    cmp2(STATUS_VOLMGR_PACK_WITHOUT_QUORUM,      ERROR_VOLMGR_PACK_WITHOUT_QUORUM);
    cmp2(STATUS_VOLMGR_PARTITION_STYLE_INVALID,  ERROR_VOLMGR_PARTITION_STYLE_INVALID);
    cmp2(STATUS_VOLMGR_PARTITION_UPDATE_FAILED,  ERROR_VOLMGR_PARTITION_UPDATE_FAILED);
    cmp2(STATUS_VOLMGR_PLEX_IN_SYNC,             ERROR_VOLMGR_PLEX_IN_SYNC);
    cmp2(STATUS_VOLMGR_PLEX_INDEX_DUPLICATE,     ERROR_VOLMGR_PLEX_INDEX_DUPLICATE);
    cmp2(STATUS_VOLMGR_PLEX_INDEX_INVALID,       ERROR_VOLMGR_PLEX_INDEX_INVALID);
    cmp2(STATUS_VOLMGR_PLEX_LAST_ACTIVE,         ERROR_VOLMGR_PLEX_LAST_ACTIVE);
    cmp2(STATUS_VOLMGR_PLEX_MISSING,             ERROR_VOLMGR_PLEX_MISSING);
    cmp2(STATUS_VOLMGR_PLEX_REGENERATING,        ERROR_VOLMGR_PLEX_REGENERATING);
    cmp2(STATUS_VOLMGR_PLEX_TYPE_INVALID,        ERROR_VOLMGR_PLEX_TYPE_INVALID);
    cmp2(STATUS_VOLMGR_PLEX_NOT_RAID5,           ERROR_VOLMGR_PLEX_NOT_RAID5);
    cmp2(STATUS_VOLMGR_PLEX_NOT_SIMPLE,          ERROR_VOLMGR_PLEX_NOT_SIMPLE);
    cmp2(STATUS_VOLMGR_STRUCTURE_SIZE_INVALID,   ERROR_VOLMGR_STRUCTURE_SIZE_INVALID);
    cmp2(STATUS_VOLMGR_TOO_MANY_NOTIFICATION_REQUESTS, ERROR_VOLMGR_TOO_MANY_NOTIFICATION_REQUESTS);
    cmp2(STATUS_VOLMGR_TRANSACTION_IN_PROGRESS,  ERROR_VOLMGR_TRANSACTION_IN_PROGRESS);
    cmp2(STATUS_VOLMGR_UNEXPECTED_DISK_LAYOUT_CHANGE, ERROR_VOLMGR_UNEXPECTED_DISK_LAYOUT_CHANGE);
    cmp2(STATUS_VOLMGR_VOLUME_CONTAINS_MISSING_DISK, ERROR_VOLMGR_VOLUME_CONTAINS_MISSING_DISK);
    cmp2(STATUS_VOLMGR_VOLUME_ID_INVALID,        ERROR_VOLMGR_VOLUME_ID_INVALID);
    cmp2(STATUS_VOLMGR_VOLUME_LENGTH_INVALID,    ERROR_VOLMGR_VOLUME_LENGTH_INVALID);
    cmp2(STATUS_VOLMGR_VOLUME_LENGTH_NOT_SECTOR_SIZE_MULTIPLE, ERROR_VOLMGR_VOLUME_LENGTH_NOT_SECTOR_SIZE_MULTIPLE);
    cmp2(STATUS_VOLMGR_VOLUME_NOT_MIRRORED,      ERROR_VOLMGR_VOLUME_NOT_MIRRORED);
    cmp2(STATUS_VOLMGR_VOLUME_NOT_RETAINED,      ERROR_VOLMGR_VOLUME_NOT_RETAINED);
    cmp2(STATUS_VOLMGR_VOLUME_OFFLINE,           ERROR_VOLMGR_VOLUME_OFFLINE);
    cmp2(STATUS_VOLMGR_VOLUME_RETAINED,          ERROR_VOLMGR_VOLUME_RETAINED);
    cmp2(STATUS_VOLMGR_NUMBER_OF_EXTENTS_INVALID, ERROR_VOLMGR_NUMBER_OF_EXTENTS_INVALID);
    cmp2(STATUS_VOLMGR_DIFFERENT_SECTOR_SIZE,    ERROR_VOLMGR_DIFFERENT_SECTOR_SIZE);
    cmp2(STATUS_VOLMGR_BAD_BOOT_DISK,            ERROR_VOLMGR_BAD_BOOT_DISK);
    cmp2(STATUS_VOLMGR_PACK_CONFIG_OFFLINE,      ERROR_VOLMGR_PACK_CONFIG_OFFLINE);
    cmp2(STATUS_VOLMGR_PACK_CONFIG_ONLINE,       ERROR_VOLMGR_PACK_CONFIG_ONLINE);
    cmp2(STATUS_VOLMGR_NOT_PRIMARY_PACK,         ERROR_VOLMGR_NOT_PRIMARY_PACK);
    cmp2(STATUS_VOLMGR_PACK_LOG_UPDATE_FAILED,   ERROR_VOLMGR_PACK_LOG_UPDATE_FAILED);
    cmp2(STATUS_VOLMGR_NUMBER_OF_DISKS_IN_PLEX_INVALID, ERROR_VOLMGR_NUMBER_OF_DISKS_IN_PLEX_INVALID);
    cmp2(STATUS_VOLMGR_NUMBER_OF_DISKS_IN_MEMBER_INVALID, ERROR_VOLMGR_NUMBER_OF_DISKS_IN_MEMBER_INVALID);
    cmp2(STATUS_VOLMGR_VOLUME_MIRRORED,          ERROR_VOLMGR_VOLUME_MIRRORED);
    cmp2(STATUS_VOLMGR_PLEX_NOT_SIMPLE_SPANNED,  ERROR_VOLMGR_PLEX_NOT_SIMPLE_SPANNED);
    cmp2(STATUS_VOLMGR_NO_VALID_LOG_COPIES,      ERROR_VOLMGR_NO_VALID_LOG_COPIES);
    cmp2(STATUS_VOLMGR_PRIMARY_PACK_PRESENT,     ERROR_VOLMGR_PRIMARY_PACK_PRESENT);
    cmp2(STATUS_VOLMGR_NUMBER_OF_DISKS_INVALID,  ERROR_VOLMGR_NUMBER_OF_DISKS_INVALID);
    cmp2(STATUS_VOLMGR_MIRROR_NOT_SUPPORTED,     ERROR_VOLMGR_MIRROR_NOT_SUPPORTED);
    cmp2(STATUS_VOLMGR_RAID5_NOT_SUPPORTED,      ERROR_VOLMGR_RAID5_NOT_SUPPORTED);
}

START_TEST(error)
{
    if (prepare_test())
        run_error_tests();
}
