# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2012 Canonical
# Author: Thomi Richards
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""Functional tests for the window-mocker application."""

import os.path
import subprocess
import tempfile
from testtools import TestCase
from testtools.matchers import Equals


def get_app_path():
    """Return the full path to the test application.

    This function prefers the local copy if these tests are being run from
    source.

    """
    source_path = os.path.abspath(
        os.path.join(
            os.path.dirname(__file__),
            '..',
            'bin',
            'window-mocker'))
    if os.path.exists(source_path):
        return source_path

    return subprocess.check_output(['which', 'window-mocker'])


def run_app_with_args(*args):
    os.putenv('PYTHONPATH',
        os.path.abspath(
            os.path.join(
                os.path.dirname(__file__),
                '..')
            )
        )
    application = [get_app_path()]
    application.extend(args)
    process = subprocess.Popen(
            application,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            shell=False
            )

    stdout, stderr = process.communicate()
    retcode = process.poll()

    return (retcode, stdout, stderr)


class FunctionalTests(TestCase):

    """Functional tests for the window-mocker application."""

    def run_with_spec_content(self, content_string):
        temp_filename = tempfile.mktemp()
        open(temp_filename, 'w').write(content_string)
        self.addCleanup(os.remove, temp_filename)
        return run_app_with_args(temp_filename)

    def test_errors_with_missing_file(self):
        retcode, stdout, stderr = run_app_with_args("bad_file_name")
        self.assertThat(stdout, Equals(''))
        self.assertThat(stderr, Equals('Error: Unable to open file: No such file or directory\n'))
        self.assertThat(retcode, Equals(1))

    def test_errors_with_inaccessible_file(self):
        open('cannot_read','w')
        os.chmod('cannot_read', 0)
        self.addCleanup(os.remove, 'cannot_read')

        retcode, stdout, stderr = run_app_with_args("cannot_read")
        self.assertThat(stdout, Equals(''))
        self.assertThat(stderr, Equals('Error: Unable to open file: Permission denied\n'))
        self.assertThat(retcode, Equals(1))

    def test_errors_on_empty_file(self):
        retcode, stdout, stderr = self.run_with_spec_content("")

        self.assertThat(stdout, Equals(""))
        self.assertThat(retcode, Equals(3))
        self.assertThat(stderr, Equals("Error: Unable to process request: No JSON object could be decoded\n"))

