/* whoopsie
 * 
 * Copyright © 2011 Canonical Ltd.
 * Author: Evan Dandrea <evan.dandrea@canonical.com>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _XOPEN_SOURCE
#define _GNU_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gio/gio.h>
#include <assert.h>
#include <curl/curl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/file.h>
#include <errno.h>
#include <signal.h>
#include <gcrypt.h>
#include <pwd.h>
#include <grp.h>
#include <sys/capability.h>
#include <sys/prctl.h>
#include <sys/mount.h>

#include "bson/bson.h"

/* Test cases:
 * - Hundreds of crash reports; do not balloon memory massively.
 * - Tests of various busted reports. ":\n", for example.
 * - No Internet, but lots of reports. Do not balloon memory massively.
 */

/* If true, we have an active Internet connection. True by default in case we
 * can't bring up GNetworkMonitor */
static gboolean online_state = TRUE;

/* TODO This queue could get quite large if a lot of crashes occur when there
 * isn't an Internet connection. Should we just iterate /var/crash/ *.upload
 * every so often? Or should we clear the queue when we don't have an Internet
 * connection, and rebuild it when we do?
 */
static GQueue* report_queue = NULL;
static GMainLoop* loop = NULL;

/* The URL of the crash database. */
static char* crash_db_url = NULL;

/*  The system UUID, taken from the DMI tables and SHA-512 hashed */
static char sha512_system_uuid[129] = {0};

/* The URL for sending the initial crash report */
static char* crash_db_submit_url = NULL;

/* Username we will run under */
static const char *username = "whoopsie";

/* Options */

static gboolean foreground = FALSE;

static GOptionEntry option_entries[] = {
    { "foreground", 'f', 0, G_OPTION_ARG_NONE, &foreground,
      "Run in the foreground", NULL },
    { NULL }
};

static void
parse_arguments (int* argc, char** argv[])
{
    GError* err = NULL;
    GOptionContext* context;
    context = g_option_context_new ("Reporting client");
    g_option_context_add_main_entries (context, option_entries, NULL);
    g_option_context_parse (context, argc, argv, &err);
    g_option_context_free (context);
    if (err)
        goto error;
    return;

    error:
        g_print ("whoopsie: %s\n", err->message);
        g_error_free (err);
        exit (EXIT_FAILURE);
}

char*
upload_to_crash_file (const char* upload_file)
{
    /* TODO: Why are we adding crash files instead of upload files to the list?
     */
    char* crash_file = NULL;
    const char* p = NULL;
    int len, i = 0;

    /* Find the file extension. */
    p = upload_file;
    while (*p != '\0') {
        p++; i++;
        if (*p == '.')
            len = i;
    }
    crash_file = malloc (len + 7); /* .crash */
    memcpy (crash_file, upload_file, len);
    strcpy (crash_file + len, ".crash");
    return crash_file;
}

void
append_key_value (gpointer key, gpointer value, gpointer bson_string)
{
    /* Takes a key and its value from a GHashTable and adds it to a BSON string
     * as key and its string value. */

    /* We don't send the core dump in the first upload, as the server might not
     * need it */
    if (!strcmp ("CoreDump", key))
        return;
    bson_append_string ((bson*)bson_string, (char*)key, (char*)value);
}

size_t
server_response (char* ptr, size_t size, size_t nmemb, void** user_data)
{
    char* buffer = NULL;

    buffer = malloc (size * nmemb + 1);
    memcpy (buffer, ptr, size * nmemb);
    buffer[size * nmemb] = '\0';
    *user_data = buffer;
    return size * nmemb;
}

void
split_string (char* head, char** tail)
{
    *tail = strchr (head, ' ');
    if (*tail) {
        **tail = '\0';
        (*tail)++;
    }
}

void
bsonify (GHashTable* report, bson* b, const char** bson_message,
         int* bson_message_len)
{
    assert (report != NULL);

    bson_init (b);
    g_hash_table_foreach (report, append_key_value, b);
    bson_finish (b);

    printf ("Report:\n");
    bson_print (b);
    *bson_message = bson_data (b);
    *bson_message_len = bson_size (b);
}

gboolean
upload_report (const char* message_data, int message_len, char** response_data)
{
    CURL* curl = NULL;
    CURLcode result_code = 0;
    long response_code = 0;
    struct curl_slist* list = NULL;

    if ((curl = curl_easy_init ()) == NULL) {
        printf ("Couldn't init curl.\n");
        return FALSE;
    }
    curl_easy_setopt (curl, CURLOPT_POST, 1);
    curl_easy_setopt (curl, CURLOPT_NOPROGRESS, 1);
    list = curl_slist_append (list, "Content-Type: application/octet-stream");
    curl_easy_setopt (curl, CURLOPT_URL, crash_db_submit_url);
    curl_easy_setopt (curl, CURLOPT_HTTPHEADER, list);
    curl_easy_setopt (curl, CURLOPT_POSTFIELDSIZE, message_len);
    curl_easy_setopt (curl, CURLOPT_POSTFIELDS, (void*)message_data);
    curl_easy_setopt (curl, CURLOPT_WRITEFUNCTION, server_response);
    curl_easy_setopt (curl, CURLOPT_WRITEDATA, response_data);
    curl_easy_setopt (curl, CURLOPT_VERBOSE, NULL);

    result_code = curl_easy_perform (curl);
    curl_slist_free_all(list);
    curl_easy_getinfo (curl, CURLINFO_RESPONSE_CODE, &response_code);

    printf ("Sent; server replied with: %s\n",
        curl_easy_strerror (result_code));
    printf ("Response code: %ld\n", response_code);
    curl_easy_cleanup (curl);

    return result_code == CURLE_OK && response_code == 200;
}

void
destroy_key_and_value (gpointer key, gpointer value, gpointer user_data)
{
    if (key)
        g_free (key);
    if (value)
        g_free (value);
}

GHashTable*
parse_report (const char* report_path)
{
    /* We'll eventually modify the contents of the report, rather than sending
     * it as-is, to make it more amenable to what the server has to stick in
     * the database, and thus creating less work server-side.
     */

    GMappedFile* fp = NULL;
    GHashTable* hash_table = NULL;
    gchar* contents = NULL;
    gsize file_len = 0;
    /* Our position in the file. */
    size_t p = 0;
    /* The end or length of the token. */
    size_t token_p = 0;
    char* key = NULL;
    char* value = NULL;
    size_t value_p = 0;
    GError* err = NULL;

    /* TODO handle the file being modified underneath us. */
    fp = g_mapped_file_new (report_path, FALSE, &err);
    if (err) {
        g_warning ("Unable to map report: %s\n", err->message);
        g_error_free (err);
        return NULL;
    }
        
    contents = g_mapped_file_get_contents (fp);
    file_len = g_mapped_file_get_length (fp);
    hash_table = g_hash_table_new (g_str_hash, g_str_equal);

    while (p < file_len) {
        assert (p == 0 || contents[p - 1] == '\n');
        if (contents[p] == ' ') {
            /* Skip the space. */
            p++;
            token_p = p;
            while (token_p < file_len && contents[token_p] != '\n') token_p++;
            token_p = token_p - p;
            /* Space for the leading newline too. */
            value = realloc (value, 1 + value_p + token_p + 1);
            value[value_p] = '\n';
            /* Again, skip past the leading newline. */
            value_p += 1;
            memcpy (value + value_p, contents + p, token_p);
            /* We don't extend past the null terminator, because we want to
             * overwrite it on the next pass. */
            value_p += token_p;
            value[value_p] = '\0';
            p += token_p + 1;
            g_hash_table_insert (hash_table, key, value);
        } else {
            /* Reset the value pointer. */
            value_p = 0;
            /* Key. */
            token_p = p;
            while (token_p < file_len) {
                if (contents[token_p] != ':' && contents[token_p] != '\n')
                    token_p++;
                else
                    break;
            }
            token_p = token_p - p;
            /* When we load the whole file into memory, we wont need to malloc
             * anymore, as we can add null-terminators in the existing string
             * buffer. */
            key = malloc (token_p + 1);
            memcpy (key, contents + p, token_p);
            key[token_p] = '\0';
            p += token_p + 1;

            token_p = p;
            while (token_p < file_len && contents[token_p] != '\n') token_p++;
            /* Skip any leading spaces. */
            while (p < file_len && contents[p] == ' ') p++;
            token_p = token_p - p;
            if (token_p == 0) {
                /* Empty value. The key likely has a child. */
                value = NULL;
            } else {
                /* Value. */
                value = malloc (token_p + 1);
                memcpy (value, contents + p, token_p);
                value[token_p] = '\0';
            }
            p += token_p + 1;
            g_hash_table_insert (hash_table, key, value);
        }
    }
    g_mapped_file_unref (fp);
    return hash_table;
}

gboolean
upload_core (const char* uuid, const char* core_data) {
    CURL* curl = NULL;
    CURLcode result_code = 0;
    char* response_data = NULL;
    long response_code = 0;
    struct curl_slist* list = NULL;
    char* crash_db_core_url = NULL;

    asprintf (&crash_db_core_url, "%s/%s/submit-core/%s",
        crash_db_url, uuid, sha512_system_uuid);

    /* TODO use CURLOPT_READFUNCTION to transparently compress data with
     * Snappy. */
    if ((curl = curl_easy_init ()) == NULL) {
        printf ("Couldn't init curl.\n");
        return FALSE;
    }
    curl_easy_setopt (curl, CURLOPT_POST, 1);
    curl_easy_setopt (curl, CURLOPT_NOPROGRESS, 1);
    list = curl_slist_append (list, "Content-Type: application/octet-stream");
    curl_easy_setopt (curl, CURLOPT_URL, crash_db_core_url);
    curl_easy_setopt (curl, CURLOPT_HTTPHEADER, list);
    curl_easy_setopt (curl, CURLOPT_POSTFIELDS, (void*)core_data);
    curl_easy_setopt (curl, CURLOPT_WRITEFUNCTION, server_response);
    curl_easy_setopt (curl, CURLOPT_WRITEDATA, &response_data);
    curl_easy_setopt (curl, CURLOPT_VERBOSE, NULL);

    result_code = curl_easy_perform (curl);
    curl_slist_free_all(list);
    free (response_data);

    curl_easy_getinfo (curl, CURLINFO_RESPONSE_CODE, &response_code);

    printf ("Sent; server replied with: %s\n",
        curl_easy_strerror (result_code));
    printf ("Response code: %ld\n", response_code);
    curl_easy_cleanup (curl);
    free (crash_db_core_url);

    return result_code == CURLE_OK && response_code == 200;
}

gboolean
parse_and_upload_report (const char* crash_file)
{
    GHashTable* report = NULL;
    gboolean success = FALSE;
    int message_len = 0;
    const char* message_data = NULL;
    char* response_data = NULL;
    char* command = NULL;
    char* core = NULL;
    bson b[1];

    report = parse_report (crash_file);
    if (!report) {
        g_warning ("Unable to parse report: %s\n", crash_file);
        return FALSE;
    }

    bsonify (report, b, &message_data, &message_len);
    success = upload_report (message_data, message_len, &response_data);
    bson_destroy (b);

    /* Command could be CORE, which requests the core dump, BUG ######, if in a
     * development release, which points to the bug report, or UPDATE, if this
     * is fixed in an update. */
    if (success && response_data) {
        split_string (response_data, &command);
        if (command) {
            if (strcmp (command, "CORE") == 0) {
                core = g_hash_table_lookup (report, "CoreDump");
                if (core) {
                    if (!upload_core (response_data, core)) {
                        /* TODO handle retrying? */
                        printf ("Upload of the core dump failed.\n");
                    }
                } else {
                    printf ("Asked for a core dump that we don't have.\n");
                }
            } else {
                printf ("Got command: %s\n", command);
            }
        }
    }

    free (response_data);
    g_hash_table_foreach (report, destroy_key_and_value, NULL);
    g_hash_table_destroy (report);

    return success;
}

void
create_file (const char* upload)
{
    char* upload_file = g_strdup (upload);
    char* crash_file = upload_to_crash_file (upload_file);

    if (g_file_test (crash_file, G_FILE_TEST_EXISTS)) {
        g_message ("%s exists", crash_file);
        if (online_state && parse_and_upload_report (crash_file)) {
            if (g_unlink (upload_file))
                g_warning ("Unable to remove: %s", upload_file);
            free (crash_file);
        } else {
            g_warning ("Adding to queue: %s", upload_file);
            g_queue_push_head (report_queue, (gpointer)upload_file);
        }
    } else {
        /* Already cleaned up? Nothing more we can do. */
        if (g_unlink (upload_file))
            g_warning ("Unable to remove: %s", upload_file);
        free (crash_file);
    }
}

gboolean
remove_from_report_queue (const char* upload_file) {
    GList* queued_crash = NULL;
    queued_crash = g_queue_find_custom (report_queue, upload_file,
                                        (GCompareFunc)strcmp);
    if (queued_crash) {
        gpointer data = queued_crash->data;
        if (!g_queue_remove (report_queue, queued_crash->data))
            g_warning ("Unable to remove from queue: %s", upload_file);
        g_free (data);
        g_list_free (queued_crash);
        return TRUE;
    } else {
        return FALSE;
    }
}

void
delete_file (const char* upload_file)
{
    g_message ("deleted: %s", upload_file);
    if (remove_from_report_queue (upload_file))
        g_warning ("%s deleted before being processed.", upload_file);
}

void
changed_event (GFileMonitor* monitor, GFile *file, GFile *other_file,
               GFileMonitorEvent event_type, gpointer user_data)
{
    char* path = NULL;
    const char *ext = NULL;

    path = g_file_get_path (file);
    /* Find the file extension. */
    ext = strrchr(path, '.');
    if (ext == NULL || strcmp(++ext, "upload") != 0)
        return;
    if (event_type == G_FILE_MONITOR_EVENT_CREATED)
        create_file(path);
    else if (event_type == G_FILE_MONITOR_EVENT_DELETED)
        delete_file(path);

    g_free(path);
}

gboolean
process_queue (void) {
    g_warning ("Processing queue.");
    GList* list = NULL;
    char *upload_file, *crash_file = NULL;
    list = report_queue->head;
    while (list) {
        GList* next = list->next;
        upload_file = list->data;
        if (g_file_test (upload_file, G_FILE_TEST_EXISTS)) {
            crash_file = upload_to_crash_file (upload_file);
            if (online_state && parse_and_upload_report (crash_file)) {
                if (g_unlink (upload_file))
                    g_warning ("Unable to remove: %s", upload_file);
                remove_from_report_queue (upload_file);
            }
            free (crash_file);
        } else {
            remove_from_report_queue (upload_file);
        }
        list = next;
    }
    return TRUE;
}

void
process_existing_files (void)
{
    GDir* dir = NULL;
    const gchar *file, *ext = NULL;
    gchar* upload_file = NULL;
    char* crash_file = NULL;

    dir = g_dir_open ("/var/crash", 0, NULL);
    while ((file = g_dir_read_name (dir)) != NULL) {
        upload_file = g_build_filename ("/var/crash", file, NULL);
        ext = strrchr(upload_file, '.');
        if (ext && strcmp(++ext, "upload") == 0) {
            crash_file = upload_to_crash_file (upload_file);
            if (online_state && parse_and_upload_report (crash_file)) {
                if (g_unlink (upload_file))
                    g_warning ("Unable to remove: %s", upload_file);
                free (crash_file);
                free (upload_file);
            } else {
                g_queue_push_head (report_queue, upload_file);
            }
        } else {
            free (upload_file);
        }
    }
    g_dir_close (dir);
}

void daemonize (void)
{
    pid_t pid, sid;

    pid = fork();
    if (pid < 0)
        exit (EXIT_FAILURE);
    if (pid > 0)
        exit (EXIT_SUCCESS);
    umask (0);
    sid = setsid ();
    if (sid < 0)
        exit (EXIT_FAILURE);

    if ((chdir ("/")) < 0)
        exit (EXIT_FAILURE);

    close (STDIN_FILENO);
    close (STDOUT_FILENO);
    close (STDERR_FILENO);
}

void
exit_if_already_running (void)
{
    int lock_fd = 0;
    int rc = 0;
    lock_fd = open ("/tmp/.whoopsie-lock", O_CREAT | O_RDWR, 0600);
    rc = flock (lock_fd, LOCK_EX | LOCK_NB);
    if (rc) {
        if (EWOULDBLOCK == errno) {
            g_warning ("Another instance is already running.");
            exit (1);
        } else {
            g_warning ("Could not create lock file: %s", strerror (errno));
        }
    }
}

static void
handle_signals (int signo)
{
    g_unlink ("/tmp/.whoopsie-lock");
    if (loop)
        g_main_loop_quit (loop);
    else
        exit (0);
}

static void
setup_signals (void)
{
    struct sigaction action;
    sigset_t mask;

    sigemptyset (&mask);
    action.sa_handler = handle_signals;
    action.sa_mask = mask;
    action.sa_flags = 0;
    sigaction (SIGTERM, &action, NULL);
    sigaction (SIGINT, &action, NULL);
}

void
hex_to_char (char* buf, const unsigned char *str, int len)
{
    char* p = NULL;
    int i = 0;

    p = buf;
    for (i = 0; i < len; i++) {
        snprintf(p, 3, "%02x", str[i]);
        p += 2;
    }
    buf[2*len] = 0;
}

void
get_system_uuid (char* res)
{
    int fp;
    char system_uuid[37] = {0};
    int md_len;
    unsigned char* id;
    gcry_md_hd_t sha512;

    fp = open ("/sys/class/dmi/id/product_uuid", O_RDONLY);
    if (read (fp, system_uuid, 36) == 36) {
        system_uuid[36] = '\0';
        close (fp);
    } else {
        close (fp);
        return;
    }

    gcry_control (GCRYCTL_DISABLE_SECMEM, 0);
    gcry_control (GCRYCTL_INITIALIZATION_FINISHED, 0);
    md_len = gcry_md_get_algo_dlen(GCRY_MD_SHA512);
    gcry_md_open (&sha512, GCRY_MD_SHA512, 0);
    gcry_md_write (sha512, system_uuid, 36);
    gcry_md_final (sha512);
    id = gcry_md_read (sha512, GCRY_MD_SHA512);
    hex_to_char (res, id, md_len);
    gcry_md_close (sha512);
}

char*
get_crash_db_url (void)
{
    /* TODO validate URL */
    char* url = NULL;
    url = getenv ("CRASH_DB_URL");
    if (url)
        return g_strdup (url);
    else
        return NULL;
}

static void
create_namespace (void)
{
    /* We're going to override +t globally, so let's play it safe and restrict
     * ourselves to only being able to write in /var/crash. */

    mkdir ("/var/tmp/whoopsie", 0755);

    /* Set up a private mount namespace. */
    if (unshare (CLONE_NEWNS) == -1)
        g_error ("CLONE_NEWNS failed.");

    if (mount ("/", "/var/tmp/whoopsie", NULL, MS_BIND | MS_REC | MS_RDONLY, NULL))
        g_error ("Could not bind mount /.");

    if (mount ("/var/crash", "/var/tmp/whoopsie/var/crash", NULL, MS_BIND, NULL))
        g_error ("Could not rw mount /var/crash.");

    if (chroot ("/var/tmp/whoopsie"))
        g_error ("Could not chroot.");

    if (chdir ("/"))
        g_error ("Could not chdir to /.");

    /* We don't need to worry about unmounting the above bind mounts, as once
     * we leave the namespace, they will be released:
     * http://lxr.linux.no/linux+v3.2.1/fs/namespace.c#L2736 */
}

static void
drop_privileges (void)
{
    struct passwd *pw = NULL;
    cap_t cap;
    /* Specify that we want to ignore the directory sticky bit */
    cap_value_t cap_list[] = {CAP_FOWNER};

    if (!CAP_IS_SUPPORTED (CAP_SETFCAP))
        g_error ("SETFCAP is not supported.");


    /* Ensure that we don't lose the capabilities when we drop privileges */
    if (prctl (PR_SET_KEEPCAPS, 1) < 0)
        g_error ("prctl failed.");

    if (!(pw = getpwnam (username)))
        g_error ("Failed to find user: %s", username);

    /* Drop privileges */
    if (setgroups (1, &pw->pw_gid) < 0 ||
        setresgid (pw->pw_gid, pw->pw_gid, pw->pw_gid) < 0 ||
        setresuid (pw->pw_uid, pw->pw_uid, pw->pw_uid) < 0) {
        g_error ("Failed to become user: %s", username);
    }

    setenv ("USER", username, 1);
    setenv ("USERNAME", username, 1);
    
    /* Now drop all capabilities but CAP_SETFCAP and CAP_FOWNER */
    cap = cap_init ();
    if (cap == NULL)
        g_error ("cap_get_proc failed.");
    if (cap_set_flag (cap, CAP_EFFECTIVE, 1, cap_list, CAP_SET) == -1)
        g_error ("cap_set_flag CAP_EFFECTIVE failed.");
    if (cap_set_flag (cap, CAP_PERMITTED, 1, cap_list, CAP_SET) == -1)
        g_error ("cap_set_flag CAP_PERMITTED failed.");
    if (cap_set_proc (cap) == -1)
        g_error ("cap_set_proc failed.");
    cap_free (cap);

    cap_clear (cap);
    cap = cap_get_proc ();
    g_warning ("capabilities: %s\n", cap_to_text(cap, NULL));
    cap_free (cap);
}

void
network_changed (GNetworkMonitor *nm, gboolean available, gpointer addr)
{
    if (!online_state && available &&
        g_network_monitor_can_reach (nm, addr, NULL, NULL)) {
            online_state = TRUE;
            process_queue ();
    } else
        online_state = FALSE;
}

void
setup_network_changed (void)
{
    GNetworkMonitor* nm = NULL;
    GSocketConnectable *addr = NULL;
    addr = g_network_address_parse_uri (crash_db_url, 80, NULL);

    nm = g_network_monitor_get_default ();
    if (!nm)
        return;

    if (!g_network_monitor_get_network_available (nm) ||
        !g_network_monitor_can_reach (nm, addr, NULL, NULL))
            online_state = FALSE;

    g_signal_connect (nm, "network-changed", G_CALLBACK (network_changed), addr);
}

#ifndef TEST
int
main (int argc, char** argv)
{
    GFile* path = NULL;
    GFileMonitor* monitor = NULL;
    GError* err = NULL;
    gulong handler = 0;
    char* system_uuid = NULL;

    setup_signals ();
    parse_arguments (&argc, &argv);

    if ((crash_db_url = get_crash_db_url ()) == NULL) {
        g_warning ("Could not get crash database location.");
        return 1;
    }
    get_system_uuid (sha512_system_uuid);
    if (*sha512_system_uuid != '\0') {
        asprintf (&crash_db_submit_url, "%s/%s",
            crash_db_url, sha512_system_uuid);
    } else {
        crash_db_submit_url = crash_db_url;
    }
    free (system_uuid);

    create_namespace ();
    drop_privileges ();
    exit_if_already_running ();

    if (!foreground)
        daemonize ();

    g_type_init ();

    /* TODO use curl_share for DNS caching. */
    if (curl_global_init (CURL_GLOBAL_NOTHING)) {
        g_warning ("Unable to initialize curl.\n");
        return 1;
    }
    report_queue = g_queue_new ();
    mkdir ("/var/crash", 0755);
    path = g_file_new_for_path ("/var/crash");
    monitor = g_file_monitor_directory (path, G_FILE_MONITOR_NONE, NULL, &err);
    g_object_unref (path);
    if (err) {
        g_warning ("Unable to monitor /var/crash: %s\n", err->message);
        g_error_free (err);
    } else {
        handler = g_signal_connect (monitor, "changed",
                                    G_CALLBACK (changed_event), NULL);
    }
    g_timeout_add_seconds (60 * 5, (GSourceFunc) process_queue, NULL);

    setup_network_changed ();
    process_existing_files ();
    loop = g_main_loop_new (NULL, FALSE);
    g_main_loop_run (loop);

    curl_global_cleanup ();
    g_signal_handler_disconnect (monitor, handler);
    g_object_unref (monitor);
    g_queue_foreach (report_queue, (GFunc)g_free, NULL);
    g_queue_free (report_queue);
    g_free (crash_db_url);
    g_free (crash_db_submit_url);
	return 0;
}
#endif
