/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "wfmetaport.h"
#include "wfindent.h"
#include "list1.h"
#include "defs.h"


bool
wf_metaport::add(const wf_metaport& metaport) {
  list<wf_metaport_elem>::const_iterator first = metaport.elems.begin(),
    last = metaport.elems.end();
  for (; first != last; ++first) {
    if (add(*first) == false)
      return false;
  }
  return true;
}

bool
wf_metaport::add(const wf_metaport_elem& elem) {
  if (elem.gettype() == WF_METAPORT_NONE)
    return false;

  elems.push_back(elem);
  return true;
}

void
wf_metaport::del(const wf_metaport_elem& elem) {
  elems.remove(elem);
}

void
wf_metaport::clear() {
  name = "";
  comment = "";
  elems.clear();
}

void
wf_metaport::substitute(const string& substname, const wf_metaport& metaport) {
  if (name == substname) {
    *this = metaport;
    name = "";
    /* This is a hack: we use this to make everyone believe it is not
       a reference, and thus output substituted value instead of name. */
  }
  else {
    list<wf_metaport_elem>::iterator first = elems.begin(), last = elems.end();
    for (; first != last; ++first) {
      if (first->gettype() == WF_METAPORT_METAPORT)
	first->get_metaport()->substitute(substname, metaport);
    }
  }
}

bool
wf_metaport::isreference() const {
  return (name.empty() == false && elems.empty());
}

unsigned int
wf_metaport::count_ports() const {
  unsigned int count = 0;
  
  list<wf_metaport_elem>::const_iterator first = elems.begin(),
    last = elems.end();
  for (; first != last; ++first) {
    if (first->gettype() == WF_METAPORT_METAPORT)
      count += first->get_metaport()->count_ports();
    else
      count++;
  }
  return count;
}

ostream&
wf_metaport::print_value(ostream& os) const {
  list1_print(elems, 0, os);
  return os;
}

ostream&
wf_metaport::print(ostream& os) const {
  if (name.empty())
    list1_print(elems, 0, os);
  else
    os << '$' << name;
  return os;
}

ostream&
wf_metaport::output_xml(ostream& os, unsigned int indent_level) const {
  os << wf_indent(indent_level++) << "<metaport>\n";
  if (name.empty() == false)
    os << wf_indent(indent_level) << "<name>" << name << "</name>\n";

  list<wf_metaport_elem>::const_iterator first = elems.begin(),
    last = elems.end();
  for (; first != last; ++first)
    first->output_xml(os, indent_level);

  if (comment.empty() == false)
    os << wf_indent(indent_level) << "<comment>" << comment << "</comment>\n";
  return os << wf_indent(--indent_level) << "</metaport>" << endl;
}

ostream&
wf_metaport::debugprint(ostream& os) const {
  os << _("Metaport:") << endl;

  os << _("name:\t\t") << name << endl;

  os << _("elems:\t\t");
  /* replace 0 (always_parenthesis) by config variable RV@@4 */
  list1_print(elems, 0, os);
  os << endl;

  os << _("comment:\t") << comment << endl;

  return os;
}


ostream&
operator<<(ostream& os, const wf_metaport& metaport) {
  return metaport.print(os);
}
