/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "wfipaddr_range.h"
#include "wfindent.h"
#include "defs.h"


wf_ipaddr_range::wf_ipaddr_range(const wf_ipaddr& min_addr,
				 const wf_ipaddr& max_addr) {
  set(min_addr, max_addr);
}

wf_ipaddr_range::wf_ipaddr_range(const string& min_str,
				 const string& max_str) {
  set(min_str, max_str);
}


bool
wf_ipaddr_range::set(const wf_ipaddr& min_addr, const wf_ipaddr& max_addr) {
  /* No need to check each address: we let the know they are ok. */
  /* Now we are tolerant, so we ensure that they are in the right order. */
  if (min_addr < max_addr) {
    _min_addr = min_addr;
    _max_addr = max_addr;
  }
  else { /* min was not the min ;-/ */
    _min_addr = max_addr;
    _max_addr = min_addr;
  }
  return true;
}

bool
wf_ipaddr_range::set(const string& min_str, const string& max_str) {
  wf_ipaddr min_addr, max_addr;

  /* First we check the addresses in order to guaranty atomicity. */
  if (min_addr.set(min_str) == false || max_addr.set(max_str) == false)
    return false;
  
  return set(min_addr, max_addr);
}

bool
wf_ipaddr_range::isdefined() const {
  return _min_addr.isdefined() && _max_addr.isdefined();
}

string
wf_ipaddr_range::tostr() const {
  return _min_addr.tostr() + '-' + _max_addr.tostr();
}

bool
wf_ipaddr_range::belong(const wf_ipaddr& addr) const {
  return (addr.get() >= _min_addr.get() &&
	  addr.get() <= _max_addr.get());
}

ostream&
wf_ipaddr_range::print(ostream& os) const {
  return os << tostr();
}

ostream&
wf_ipaddr_range::output_xml(ostream& os, unsigned int indent_level) const {
  return os << wf_indent(indent_level) << "<ipaddr_range>\n" <<
    wf_indent(indent_level + 1) << "<min>" << _min_addr.tostr() <<
    "</min>\n" <<
    wf_indent(indent_level + 1) << "<max>" << _max_addr.tostr() <<
    "</max>\n" <<
    wf_indent(indent_level) << "</ipaddr_range>" << endl;
}

ostream&
wf_ipaddr_range::debugprint(ostream& os) const {
  return os << _("IP address range:\t") << *this << endl;
}

ostream&
operator<<(ostream& os, const wf_ipaddr_range& ipaddr_range) {
  return ipaddr_range.print(os);
}


bool
wf_ipaddr_range::check(const string& min_addr, const string& max_addr) {
  return (wf_ipaddr::check(min_addr) == true &&
	  wf_ipaddr::check(max_addr) == true);
}
