/* 
 * Copyright © 2012 Intel Corporation
 * 
 * Permission to use, copy, modify, distribute, and sell this
 * software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notice appear in
 * all copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of
 * the copyright holders not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 * 
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

#ifndef TEXT_SERVER_PROTOCOL_H
#define TEXT_SERVER_PROTOCOL_H

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stddef.h>
#include "wayland-util.h"

struct wl_client;
struct wl_resource;

struct text_model;
struct text_model_factory;

extern const struct wl_interface text_model_interface;
extern const struct wl_interface text_model_factory_interface;

/**
 * text_model - text model
 * @set_surrounding_text: sets the surrounding text
 * @activate: request activation
 * @deactivate: request deactivation
 * @reset: reset
 * @set_micro_focus: (none)
 * @set_preedit: (none)
 * @set_content_type: (none)
 *
 * A model for text input. Adds support for text input and input methods
 * to applications. A text_model object is created from a
 * text_model_factory and corresponds typically to a text entry in an
 * application. Requests are used to activate/deactivate the model and set
 * information like surrounding and selected text or the content type. The
 * information about entered text is sent to the model via the pre-edit and
 * commit events. Using this interface removes the need for applications to
 * directly process hardware key events and compose text out of them.
 */
struct text_model_interface {
	/**
	 * set_surrounding_text - sets the surrounding text
	 * @text: (none)
	 * @cursor: (none)
	 * @anchor: (none)
	 *
	 * Sets the plain surrounding text around the input position.
	 * Cursor is the byte index within the surrounding text. Anchor is
	 * the byte index of the selection anchor within the surrounding
	 * text. If there is no selected text anchor is the same as cursor.
	 */
	void (*set_surrounding_text)(struct wl_client *client,
				     struct wl_resource *resource,
				     const char *text,
				     uint32_t cursor,
				     uint32_t anchor);
	/**
	 * activate - request activation
	 * @seat: (none)
	 * @surface: (none)
	 *
	 * Requests the model to be activated (typically when the text
	 * entry gets focus). The seat argument is a wl_seat which
	 * maintains the focus for this activation. The surface argument is
	 * a wl_surface assigned to the model and tracked for focus lost.
	 * The activated event is emitted on successful activation.
	 */
	void (*activate)(struct wl_client *client,
			 struct wl_resource *resource,
			 struct wl_resource *seat,
			 struct wl_resource *surface);
	/**
	 * deactivate - request deactivation
	 * @seat: (none)
	 *
	 * Requests the model to be deactivated (typically when the text
	 * entry lost focus). The seat argument is a wl_seat which was used
	 * for activation.
	 */
	void (*deactivate)(struct wl_client *client,
			   struct wl_resource *resource,
			   struct wl_resource *seat);
	/**
	 * reset - reset
	 *
	 * Should be called by an editor widget when the input state
	 * should be reseted, for example after the text was changed
	 * outside of the normal input method flow.
	 */
	void (*reset)(struct wl_client *client,
		      struct wl_resource *resource);
	/**
	 * set_micro_focus - (none)
	 * @x: (none)
	 * @y: (none)
	 * @width: (none)
	 * @height: (none)
	 */
	void (*set_micro_focus)(struct wl_client *client,
				struct wl_resource *resource,
				int32_t x,
				int32_t y,
				int32_t width,
				int32_t height);
	/**
	 * set_preedit - (none)
	 */
	void (*set_preedit)(struct wl_client *client,
			    struct wl_resource *resource);
	/**
	 * set_content_type - (none)
	 */
	void (*set_content_type)(struct wl_client *client,
				 struct wl_resource *resource);
};

#define TEXT_MODEL_COMMIT_STRING	0
#define TEXT_MODEL_PREEDIT_STRING	1
#define TEXT_MODEL_DELETE_SURROUNDING_TEXT	2
#define TEXT_MODEL_PREEDIT_STYLING	3
#define TEXT_MODEL_KEY	4
#define TEXT_MODEL_SELECTION_REPLACEMENT	5
#define TEXT_MODEL_DIRECTION	6
#define TEXT_MODEL_LOCALE	7
#define TEXT_MODEL_ENTER	8
#define TEXT_MODEL_LEAVE	9

static inline void
text_model_send_commit_string(struct wl_resource *resource_, const char *text, uint32_t index)
{
	wl_resource_post_event(resource_, TEXT_MODEL_COMMIT_STRING, text, index);
}

static inline void
text_model_send_preedit_string(struct wl_resource *resource_, const char *text, uint32_t index)
{
	wl_resource_post_event(resource_, TEXT_MODEL_PREEDIT_STRING, text, index);
}

static inline void
text_model_send_delete_surrounding_text(struct wl_resource *resource_, int32_t index, uint32_t length)
{
	wl_resource_post_event(resource_, TEXT_MODEL_DELETE_SURROUNDING_TEXT, index, length);
}

static inline void
text_model_send_preedit_styling(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, TEXT_MODEL_PREEDIT_STYLING);
}

static inline void
text_model_send_key(struct wl_resource *resource_, uint32_t key, uint32_t state)
{
	wl_resource_post_event(resource_, TEXT_MODEL_KEY, key, state);
}

static inline void
text_model_send_selection_replacement(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, TEXT_MODEL_SELECTION_REPLACEMENT);
}

static inline void
text_model_send_direction(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, TEXT_MODEL_DIRECTION);
}

static inline void
text_model_send_locale(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, TEXT_MODEL_LOCALE);
}

static inline void
text_model_send_enter(struct wl_resource *resource_, struct wl_resource *surface)
{
	wl_resource_post_event(resource_, TEXT_MODEL_ENTER, surface);
}

static inline void
text_model_send_leave(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, TEXT_MODEL_LEAVE);
}

/**
 * text_model_factory - text model factory
 * @create_text_model: create text model
 *
 * A factory for text models. This object is a singleton global.
 */
struct text_model_factory_interface {
	/**
	 * create_text_model - create text model
	 * @id: (none)
	 *
	 * Creates a new text model object.
	 */
	void (*create_text_model)(struct wl_client *client,
				  struct wl_resource *resource,
				  uint32_t id);
};

#ifdef  __cplusplus
}
#endif

#endif
