/* 
 * Copyright © 2012 Intel Corporation
 * 
 * Permission to use, copy, modify, distribute, and sell this
 * software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notice appear in
 * all copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of
 * the copyright holders not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 * 
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

#ifndef INPUT_METHOD_SERVER_PROTOCOL_H
#define INPUT_METHOD_SERVER_PROTOCOL_H

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stddef.h>
#include "wayland-util.h"

struct wl_client;
struct wl_resource;

struct input_method_context;
struct input_method;

extern const struct wl_interface input_method_context_interface;
extern const struct wl_interface input_method_interface;

/**
 * input_method_context - input method context
 * @destroy: (none)
 * @commit_string: commit string
 * @preedit_string: pre-edit string
 * @delete_surrounding_text: (none)
 * @key: (none)
 *
 * Corresponds to a text model on input method side. An input method
 * context is created on text mode activation on the input method side. It
 * allows to receive information about the text model from the application
 * via events. Input method contexts do not keep state after deactivation
 * and should be destroyed after deactivation is handled.
 */
struct input_method_context_interface {
	/**
	 * destroy - (none)
	 */
	void (*destroy)(struct wl_client *client,
			struct wl_resource *resource);
	/**
	 * commit_string - commit string
	 * @text: (none)
	 * @index: (none)
	 *
	 * Send the commit string text to the applications text model.
	 */
	void (*commit_string)(struct wl_client *client,
			      struct wl_resource *resource,
			      const char *text,
			      uint32_t index);
	/**
	 * preedit_string - pre-edit string
	 * @text: (none)
	 * @index: (none)
	 *
	 * Send the pre-edit string text to the applications text model.
	 */
	void (*preedit_string)(struct wl_client *client,
			       struct wl_resource *resource,
			       const char *text,
			       uint32_t index);
	/**
	 * delete_surrounding_text - (none)
	 * @index: (none)
	 * @length: (none)
	 */
	void (*delete_surrounding_text)(struct wl_client *client,
					struct wl_resource *resource,
					int32_t index,
					uint32_t length);
	/**
	 * key - (none)
	 * @key: (none)
	 * @state: (none)
	 */
	void (*key)(struct wl_client *client,
		    struct wl_resource *resource,
		    uint32_t key,
		    uint32_t state);
};

#define INPUT_METHOD_CONTEXT_SURROUNDING_TEXT	0
#define INPUT_METHOD_CONTEXT_RESET	1

static inline void
input_method_context_send_surrounding_text(struct wl_resource *resource_, const char *text, uint32_t cursor, uint32_t anchor)
{
	wl_resource_post_event(resource_, INPUT_METHOD_CONTEXT_SURROUNDING_TEXT, text, cursor, anchor);
}

static inline void
input_method_context_send_reset(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, INPUT_METHOD_CONTEXT_RESET);
}

#define INPUT_METHOD_ACTIVATE	0
#define INPUT_METHOD_DEACTIVATE	1

static inline void
input_method_send_activate(struct wl_resource *resource_, struct wl_resource *id)
{
	wl_resource_post_event(resource_, INPUT_METHOD_ACTIVATE, id);
}

static inline void
input_method_send_deactivate(struct wl_resource *resource_, struct wl_resource *context)
{
	wl_resource_post_event(resource_, INPUT_METHOD_DEACTIVATE, context);
}

#ifdef  __cplusplus
}
#endif

#endif
