/* 
 * Copyright © 2012 Intel Corporation
 * 
 * Permission to use, copy, modify, distribute, and sell this
 * software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notice appear in
 * all copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of
 * the copyright holders not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 * 
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

#ifndef TEXT_CLIENT_PROTOCOL_H
#define TEXT_CLIENT_PROTOCOL_H

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stddef.h>
#include "wayland-client.h"

struct wl_client;
struct wl_resource;

struct text_model;
struct text_model_factory;

extern const struct wl_interface text_model_interface;
extern const struct wl_interface text_model_factory_interface;

/**
 * text_model - text model
 * @commit_string: commit
 * @preedit_string: pre-edit
 * @delete_surrounding_text: delete surrounding text
 * @preedit_styling: (none)
 * @key: key
 * @selection_replacement: (none)
 * @direction: (none)
 * @locale: (none)
 * @enter: enter event
 * @leave: leave event
 *
 * A model for text input. Adds support for text input and input methods
 * to applications. A text_model object is created from a
 * text_model_factory and corresponds typically to a text entry in an
 * application. Requests are used to activate/deactivate the model and set
 * information like surrounding and selected text or the content type. The
 * information about entered text is sent to the model via the pre-edit and
 * commit events. Using this interface removes the need for applications to
 * directly process hardware key events and compose text out of them.
 */
struct text_model_listener {
	/**
	 * commit_string - commit
	 * @text: (none)
	 * @index: (none)
	 *
	 * Notify when text should be inserted into the editor widget.
	 * The text to commit could be either just a single character after
	 * a key press or the result of some composing (pre-edit). It also
	 * sets the new cursor position (as byte index) relative to the
	 * inserted text.
	 */
	void (*commit_string)(void *data,
			      struct text_model *text_model,
			      const char *text,
			      uint32_t index);
	/**
	 * preedit_string - pre-edit
	 * @text: (none)
	 * @index: (none)
	 *
	 * Notify when a new composing text (pre-edit) should be set
	 * around the current cursor position. Any previously set composing
	 * text should be removed. It also sets the cursor position (as
	 * byte index) relative to the start of the composing text.
	 */
	void (*preedit_string)(void *data,
			       struct text_model *text_model,
			       const char *text,
			       uint32_t index);
	/**
	 * delete_surrounding_text - delete surrounding text
	 * @index: (none)
	 * @length: (none)
	 *
	 * Notify when the text around the current cursor position should
	 * be deleted. Index is relative to the current cursor (as byte
	 * index). Length is the length of deleted text (as bytes).
	 */
	void (*delete_surrounding_text)(void *data,
					struct text_model *text_model,
					int32_t index,
					uint32_t length);
	/**
	 * preedit_styling - (none)
	 */
	void (*preedit_styling)(void *data,
				struct text_model *text_model);
	/**
	 * key - key
	 * @key: (none)
	 * @state: (none)
	 *
	 * Notify when a key event was sent. Key events should not be
	 * used for normal text input operations, which should be done with
	 * commit_string, delete_surrounfing_text, etc. The key event
	 * follows the wl_keyboard key event convention. Key is a XKB
	 * keycode, state a wl_keyboard key_state.
	 */
	void (*key)(void *data,
		    struct text_model *text_model,
		    uint32_t key,
		    uint32_t state);
	/**
	 * selection_replacement - (none)
	 */
	void (*selection_replacement)(void *data,
				      struct text_model *text_model);
	/**
	 * direction - (none)
	 */
	void (*direction)(void *data,
			  struct text_model *text_model);
	/**
	 * locale - (none)
	 */
	void (*locale)(void *data,
		       struct text_model *text_model);
	/**
	 * enter - enter event
	 * @surface: (none)
	 *
	 * Notify the model when it is activated. Typically in response
	 * to an activate request.
	 */
	void (*enter)(void *data,
		      struct text_model *text_model,
		      struct wl_surface *surface);
	/**
	 * leave - leave event
	 *
	 * Notify the model when it is deactivated. Either in response to
	 * a deactivate request or when the assigned surface lost focus or
	 * was destroyed.
	 */
	void (*leave)(void *data,
		      struct text_model *text_model);
};

static inline int
text_model_add_listener(struct text_model *text_model,
			const struct text_model_listener *listener, void *data)
{
	return wl_proxy_add_listener((struct wl_proxy *) text_model,
				     (void (**)(void)) listener, data);
}

#define TEXT_MODEL_SET_SURROUNDING_TEXT	0
#define TEXT_MODEL_ACTIVATE	1
#define TEXT_MODEL_DEACTIVATE	2
#define TEXT_MODEL_RESET	3
#define TEXT_MODEL_SET_MICRO_FOCUS	4
#define TEXT_MODEL_SET_PREEDIT	5
#define TEXT_MODEL_SET_CONTENT_TYPE	6

static inline void
text_model_set_user_data(struct text_model *text_model, void *user_data)
{
	wl_proxy_set_user_data((struct wl_proxy *) text_model, user_data);
}

static inline void *
text_model_get_user_data(struct text_model *text_model)
{
	return wl_proxy_get_user_data((struct wl_proxy *) text_model);
}

static inline void
text_model_destroy(struct text_model *text_model)
{
	wl_proxy_destroy((struct wl_proxy *) text_model);
}

static inline void
text_model_set_surrounding_text(struct text_model *text_model, const char *text, uint32_t cursor, uint32_t anchor)
{
	wl_proxy_marshal((struct wl_proxy *) text_model,
			 TEXT_MODEL_SET_SURROUNDING_TEXT, text, cursor, anchor);
}

static inline void
text_model_activate(struct text_model *text_model, struct wl_seat *seat, struct wl_surface *surface)
{
	wl_proxy_marshal((struct wl_proxy *) text_model,
			 TEXT_MODEL_ACTIVATE, seat, surface);
}

static inline void
text_model_deactivate(struct text_model *text_model, struct wl_seat *seat)
{
	wl_proxy_marshal((struct wl_proxy *) text_model,
			 TEXT_MODEL_DEACTIVATE, seat);
}

static inline void
text_model_reset(struct text_model *text_model)
{
	wl_proxy_marshal((struct wl_proxy *) text_model,
			 TEXT_MODEL_RESET);
}

static inline void
text_model_set_micro_focus(struct text_model *text_model, int32_t x, int32_t y, int32_t width, int32_t height)
{
	wl_proxy_marshal((struct wl_proxy *) text_model,
			 TEXT_MODEL_SET_MICRO_FOCUS, x, y, width, height);
}

static inline void
text_model_set_preedit(struct text_model *text_model)
{
	wl_proxy_marshal((struct wl_proxy *) text_model,
			 TEXT_MODEL_SET_PREEDIT);
}

static inline void
text_model_set_content_type(struct text_model *text_model)
{
	wl_proxy_marshal((struct wl_proxy *) text_model,
			 TEXT_MODEL_SET_CONTENT_TYPE);
}

#define TEXT_MODEL_FACTORY_CREATE_TEXT_MODEL	0

static inline void
text_model_factory_set_user_data(struct text_model_factory *text_model_factory, void *user_data)
{
	wl_proxy_set_user_data((struct wl_proxy *) text_model_factory, user_data);
}

static inline void *
text_model_factory_get_user_data(struct text_model_factory *text_model_factory)
{
	return wl_proxy_get_user_data((struct wl_proxy *) text_model_factory);
}

static inline void
text_model_factory_destroy(struct text_model_factory *text_model_factory)
{
	wl_proxy_destroy((struct wl_proxy *) text_model_factory);
}

static inline struct text_model *
text_model_factory_create_text_model(struct text_model_factory *text_model_factory)
{
	struct wl_proxy *id;

	id = wl_proxy_create((struct wl_proxy *) text_model_factory,
			     &text_model_interface);
	if (!id)
		return NULL;

	wl_proxy_marshal((struct wl_proxy *) text_model_factory,
			 TEXT_MODEL_FACTORY_CREATE_TEXT_MODEL, id);

	return (struct text_model *) id;
}

#ifdef  __cplusplus
}
#endif

#endif
