/* 
 * Copyright © 2012 Intel Corporation
 * 
 * Permission to use, copy, modify, distribute, and sell this
 * software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notice appear in
 * all copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of
 * the copyright holders not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 * 
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

#ifndef INPUT_METHOD_CLIENT_PROTOCOL_H
#define INPUT_METHOD_CLIENT_PROTOCOL_H

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stddef.h>
#include "wayland-client.h"

struct wl_client;
struct wl_resource;

struct input_method_context;
struct input_method;

extern const struct wl_interface input_method_context_interface;
extern const struct wl_interface input_method_interface;

/**
 * input_method_context - input method context
 * @surrounding_text: surrounding text event
 * @reset: (none)
 *
 * Corresponds to a text model on input method side. An input method
 * context is created on text mode activation on the input method side. It
 * allows to receive information about the text model from the application
 * via events. Input method contexts do not keep state after deactivation
 * and should be destroyed after deactivation is handled.
 */
struct input_method_context_listener {
	/**
	 * surrounding_text - surrounding text event
	 * @text: (none)
	 * @cursor: (none)
	 * @anchor: (none)
	 *
	 * The plain surrounding text around the input position. Cursor
	 * is the position within the surrounding text. Anchor is the
	 * position of the selection anchor within the surrounding text. If
	 * there is no selected text anchor is the same as cursor.
	 */
	void (*surrounding_text)(void *data,
				 struct input_method_context *input_method_context,
				 const char *text,
				 uint32_t cursor,
				 uint32_t anchor);
	/**
	 * reset - (none)
	 */
	void (*reset)(void *data,
		      struct input_method_context *input_method_context);
};

static inline int
input_method_context_add_listener(struct input_method_context *input_method_context,
				  const struct input_method_context_listener *listener, void *data)
{
	return wl_proxy_add_listener((struct wl_proxy *) input_method_context,
				     (void (**)(void)) listener, data);
}

#define INPUT_METHOD_CONTEXT_DESTROY	0
#define INPUT_METHOD_CONTEXT_COMMIT_STRING	1
#define INPUT_METHOD_CONTEXT_PREEDIT_STRING	2
#define INPUT_METHOD_CONTEXT_DELETE_SURROUNDING_TEXT	3
#define INPUT_METHOD_CONTEXT_KEY	4

static inline void
input_method_context_set_user_data(struct input_method_context *input_method_context, void *user_data)
{
	wl_proxy_set_user_data((struct wl_proxy *) input_method_context, user_data);
}

static inline void *
input_method_context_get_user_data(struct input_method_context *input_method_context)
{
	return wl_proxy_get_user_data((struct wl_proxy *) input_method_context);
}

static inline void
input_method_context_destroy(struct input_method_context *input_method_context)
{
	wl_proxy_marshal((struct wl_proxy *) input_method_context,
			 INPUT_METHOD_CONTEXT_DESTROY);

	wl_proxy_destroy((struct wl_proxy *) input_method_context);
}

static inline void
input_method_context_commit_string(struct input_method_context *input_method_context, const char *text, uint32_t index)
{
	wl_proxy_marshal((struct wl_proxy *) input_method_context,
			 INPUT_METHOD_CONTEXT_COMMIT_STRING, text, index);
}

static inline void
input_method_context_preedit_string(struct input_method_context *input_method_context, const char *text, uint32_t index)
{
	wl_proxy_marshal((struct wl_proxy *) input_method_context,
			 INPUT_METHOD_CONTEXT_PREEDIT_STRING, text, index);
}

static inline void
input_method_context_delete_surrounding_text(struct input_method_context *input_method_context, int32_t index, uint32_t length)
{
	wl_proxy_marshal((struct wl_proxy *) input_method_context,
			 INPUT_METHOD_CONTEXT_DELETE_SURROUNDING_TEXT, index, length);
}

static inline void
input_method_context_key(struct input_method_context *input_method_context, uint32_t key, uint32_t state)
{
	wl_proxy_marshal((struct wl_proxy *) input_method_context,
			 INPUT_METHOD_CONTEXT_KEY, key, state);
}

/**
 * input_method - input method
 * @activate: activate event
 * @deactivate: activate event
 *
 * An input method object is responsible to compose text in response to
 * input from hardware or virtual keyboards. There is one input method
 * object per seat. On activate there is a new input method context object
 * created which allows the input method to communicate with the text
 * model.
 */
struct input_method_listener {
	/**
	 * activate - activate event
	 * @id: (none)
	 *
	 * A text model was activated. Creates an input method context
	 * object which allows communication with the text model.
	 */
	void (*activate)(void *data,
			 struct input_method *input_method,
			 struct input_method_context *id);
	/**
	 * deactivate - activate event
	 * @context: (none)
	 *
	 * The text model corresponding to the context argument was
	 * deactivated. The input method context should be destroyed after
	 * deactivation is handled.
	 */
	void (*deactivate)(void *data,
			   struct input_method *input_method,
			   struct input_method_context *context);
};

static inline int
input_method_add_listener(struct input_method *input_method,
			  const struct input_method_listener *listener, void *data)
{
	return wl_proxy_add_listener((struct wl_proxy *) input_method,
				     (void (**)(void)) listener, data);
}

static inline void
input_method_set_user_data(struct input_method *input_method, void *user_data)
{
	wl_proxy_set_user_data((struct wl_proxy *) input_method, user_data);
}

static inline void *
input_method_get_user_data(struct input_method *input_method)
{
	return wl_proxy_get_user_data((struct wl_proxy *) input_method);
}

static inline void
input_method_destroy(struct input_method *input_method)
{
	wl_proxy_destroy((struct wl_proxy *) input_method);
}

#ifdef  __cplusplus
}
#endif

#endif
