
#include "webauthp.h"

#include <stdio.h>
#include <krb5.h>
#include <com_err.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

typedef struct {
    krb5_context ctx;
    krb5_ccache cc;
    krb5_principal princ;
    krb5_error_code code;
    int keep_cache;
} WEBAUTH_KRB5_CTXTP;

/*#define WA_CRED_DEBUG 1 */

/* these names are kept to a minimum since encoded creds end up
   in cookies,etc */

#define CR_ADDRTYPE "A%d"
#define CR_ADDRCONT "a%d"
#define CR_CLIENT "c"
#define CR_AUTHDATATYPE "D%d"
#define CR_AUTHDATACONT "d%d"
#define CR_TICKETFLAGS "f"
#define CR_ISSKEY "i"
#define CR_SERVER "s"
#define CR_KEYBLOCK_CONTENTS "k"
#define CR_KEYBLOCK_ENCTYPE "K"
#define CR_NUMADDRS "na"
#define CR_NUMAUTHDATA "nd"
#define CR_TICKET "t"
#define CR_TICKET2 "t2"
#define CR_AUTHTIME  "ta"
#define CR_STARTTIME  "ts"
#define CR_ENDTIME  "te"
#define CR_RENEWTILL "tr"

/*
 * take a single krb5 cred an externalize it to a buffer
 */

static int
cred_to_attr_encoding(WEBAUTH_KRB5_CTXTP *c, 
                      krb5_creds *creds, 
                      unsigned char **output, 
                      int *length,
                      time_t *expiration)
{
    WEBAUTH_ATTR_LIST *list;
    int s, length_max;

    assert(c != NULL);
    assert(creds != NULL);
    assert(output != NULL);
    assert(length != NULL);
    assert(expiration != NULL);

    list = webauth_attr_list_new(128);

    /* clent principal */
    if (creds->client) {
        char *princ;
        c->code = krb5_unparse_name(c->ctx, creds->client, &princ);
        if (c->code != 0) {
            s = WA_ERR_KRB5;
            goto cleanup;
        }
        
        s = webauth_attr_list_add_str(list, CR_CLIENT, princ, 0, 
                                      WA_F_COPY_VALUE);
        free(princ);
        if (s != WA_ERR_NONE) 
            goto cleanup;
    }

    /* server principal */
    if (creds->server) {
        char *princ;
        c->code = krb5_unparse_name(c->ctx, creds->server, &princ);
        if (c->code != 0) {
            s = WA_ERR_KRB5;
            goto cleanup;
        }
        s = webauth_attr_list_add_str(list, CR_SERVER, princ, 0, 
                                      WA_F_COPY_VALUE);
        free(princ);
        if (s != WA_ERR_NONE) 
            goto cleanup;
    }
    /* keyblock */
    s = webauth_attr_list_add_int32(list, CR_KEYBLOCK_ENCTYPE, 
                                    creds->keyblock.enctype, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    s = webauth_attr_list_add(list, CR_KEYBLOCK_CONTENTS,
                              creds->keyblock.contents,
                              creds->keyblock.length, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    /* times */
    s = webauth_attr_list_add_int32(list, 
                                    CR_AUTHTIME, 
                                    creds->times.authtime, WA_F_NONE);
    if (s == WA_ERR_NONE) 
        s = webauth_attr_list_add_int32(list, CR_STARTTIME, 
                                        creds->times.starttime, WA_F_NONE);
    if ( s== WA_ERR_NONE) 
        s = webauth_attr_list_add_int32(list, CR_ENDTIME,
                                        creds->times.endtime, WA_F_NONE);
    if (s == WA_ERR_NONE) 
        s = webauth_attr_list_add_int32(list, CR_RENEWTILL,
                                        creds->times.renew_till, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    *expiration = creds->times.endtime;

    /* is_skey */
    s = webauth_attr_list_add_int32(list, 
                                    CR_ISSKEY, creds->is_skey, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    /* ticket_flags */
    s = webauth_attr_list_add_int32(list, 
                                    CR_TICKETFLAGS, 
                                    creds->ticket_flags, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    /* addresses */
    /* FIXME: We might never want to send these? */
    if (creds->addresses) {
        int num = 0, i;
        char name[32];
        krb5_address **temp = creds->addresses;
        while (*temp++)
            num++;
        s = webauth_attr_list_add_int32(list, CR_NUMADDRS, num, WA_F_NONE);
        if (s != WA_ERR_NONE)
            goto cleanup;
        for (i=0; i < num; i++) {
            sprintf(name, CR_ADDRTYPE, i);
            s = webauth_attr_list_add_int32(list, name,
                                            creds->addresses[i]->addrtype,
                                            WA_F_COPY_NAME);
            if (s != WA_ERR_NONE)
                goto cleanup;
            sprintf(name, CR_ADDRCONT, i);
            s = webauth_attr_list_add(list, name,
                                      creds->addresses[i]->contents,
                                      creds->addresses[i]->length,
                                      WA_F_COPY_NAME);
            if (s != WA_ERR_NONE)
                goto cleanup;
        }
        
    }

    /* ticket */
    if (creds->ticket.length) {
        s = webauth_attr_list_add(list, CR_TICKET,
                                  creds->ticket.data, 
                                  creds->ticket.length, WA_F_NONE);
        if (s != WA_ERR_NONE)
            goto cleanup;
    }

    /* second_ticket */
    if (creds->second_ticket.length) {
        s = webauth_attr_list_add(list, CR_TICKET2,
                                  creds->second_ticket.data,
                                  creds->second_ticket.length, WA_F_NONE);
    }

    if (s != WA_ERR_NONE)
        goto cleanup;

    /* authdata */
    if (creds->authdata) {
        int num = 0, i;
        char name[32];
        krb5_authdata **temp = creds->authdata;
        while (*temp++)
            num++;
        s = webauth_attr_list_add_int32(list, CR_NUMAUTHDATA, num, WA_F_NONE);
        if (s != WA_ERR_NONE)
            goto cleanup;
        for (i=0; i < num; i++) {
            sprintf(name, CR_AUTHDATATYPE, i);
            s = webauth_attr_list_add_int32(list, name,
                                            creds->authdata[i]->ad_type,
                                            WA_F_COPY_NAME);
            if (s != WA_ERR_NONE)
                goto cleanup;
            sprintf(name, CR_AUTHDATACONT, i);
            s = webauth_attr_list_add(list, name,
                                      creds->authdata[i]->contents,
                                      creds->authdata[i]->length,
                                      WA_F_COPY_NAME);
            if (s != WA_ERR_NONE)
                goto cleanup;
        }
    }


    length_max =  webauth_attrs_encoded_length(list);

    *output = malloc(length_max);
    
    if (*output == NULL) {
        s = WA_ERR_NO_MEM;
        goto cleanup;
    }

    s = webauth_attrs_encode(list, *output, length, length_max);
    if (s != WA_ERR_NONE) {
        free (*output);
        *output = NULL;
    }
 cleanup:
    webauth_attr_list_free(list);
    return s;
}


/*
 * take an externalized cred and turn it back into a cred
 */

static int
cred_from_attr_encoding(WEBAUTH_KRB5_CTXTP *c, 
                        unsigned char *input,
                        int input_length,
                        krb5_creds *creds)

{
    WEBAUTH_ATTR_LIST *list;
    int s, f;
    unsigned char *buff;

    assert(c != NULL);
    assert(creds != NULL);
    assert(input != NULL);

    memset(creds, 0, sizeof(krb5_creds));

    list = NULL;
    buff = malloc(input_length);

    if (buff == NULL)
        return WA_ERR_NO_MEM;

    memcpy(buff, input, input_length);

    s = webauth_attrs_decode(buff, input_length, &list);

    if (s != WA_ERR_NONE) 
        goto cleanup;

    /* clent principal */
    webauth_attr_list_find(list, CR_CLIENT, &f);
    if (f != -1) {
        c->code = krb5_parse_name(c->ctx,
                                  (char *)list->attrs[f].value,
                                  &creds->client);
        if (c->code != 0) {
            s = WA_ERR_KRB5;
            goto cleanup;
        }
    }

    /* server principal */
    webauth_attr_list_find(list, CR_SERVER, &f);
    if (f != -1) {
        c->code = krb5_parse_name(c->ctx,
                                  (char *)list->attrs[f].value,
                                  &creds->server);
        if (c->code != 0) {
            s = WA_ERR_KRB5;
            goto cleanup;
        }
    }

    /* keyblock */
    creds->keyblock.magic = KV5M_KEYBLOCK;
    s = webauth_attr_list_get_int32(list, CR_KEYBLOCK_ENCTYPE, 
                                    &creds->keyblock.enctype, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    s = webauth_attr_list_get(list, CR_KEYBLOCK_CONTENTS,
                              (void**)&creds->keyblock.contents,
                              &creds->keyblock.length, WA_F_COPY_VALUE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    /* times */
    s = webauth_attr_list_get_int32(list, CR_AUTHTIME, 
                                    &creds->times.authtime, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    s = webauth_attr_list_get_int32(list, CR_STARTTIME,
                                    &creds->times.starttime, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    s = webauth_attr_list_get_int32(list, CR_ENDTIME, 
                                    &creds->times.endtime, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    s = webauth_attr_list_get_int32(list, CR_RENEWTILL,
                                    &creds->times.renew_till, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;
    /* is_skey */
    s = webauth_attr_list_get_int32(list, CR_ISSKEY,
                                    (int32_t *) &creds->is_skey, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    /* ticket_flags */
    s = webauth_attr_list_get_int32(list, CR_TICKETFLAGS, 
                                    &creds->ticket_flags, WA_F_NONE);
    if (s != WA_ERR_NONE)
        goto cleanup;

    /* addresses */
    /* FIXME: We might never want to add these? */
    /* they might not even exist if we got forwardable/proxiable tickets */
    webauth_attr_list_find(list, CR_NUMADDRS, &f);
    if (f != -1) {
        int num = 0, i;
        char name[32];
        
        s = webauth_attr_list_get_int32(list, CR_NUMADDRS, &num, WA_F_NONE);
        if (s != WA_ERR_NONE)
            goto cleanup;

        /* don't forget to add 1 to num for the null address at the
           end of the list */
        creds->addresses = 
            (krb5_address **)calloc(num+1, sizeof(krb5_address *));
        if (creds->addresses == NULL) {
            s = WA_ERR_NO_MEM;
            goto cleanup;
        }

        for (i=0; i < num; i++) {
            creds->addresses[i] = 
                (krb5_address *) malloc(sizeof(krb5_address));
            if (creds->addresses[i] == NULL) {
                goto cleanup;
            }

            creds->addresses[i]->magic = KV5M_ADDRESS;
            sprintf(name, CR_ADDRTYPE, i);
            s = webauth_attr_list_get_int32(list, name,
                                            &creds->addresses[i]->addrtype,
                                            WA_F_NONE);
            if (s != WA_ERR_NONE)
                goto cleanup;
            sprintf(name, CR_ADDRCONT, i);

            s = webauth_attr_list_get(list, name,
                                      (void**)&creds->addresses[i]->contents,
                                      &creds->addresses[i]->length, 
                                      WA_F_COPY_VALUE);
            if (s != WA_ERR_NONE)
                goto cleanup;
        }
    }

    /* ticket */
    webauth_attr_list_find(list, CR_TICKET, &f);
    if (f != -1) {
        creds->ticket.magic = KV5M_DATA;

        s = webauth_attr_list_get(list, CR_TICKET,
                                  (void**)&creds->ticket.data,
                                  &creds->ticket.length, WA_F_COPY_VALUE);
        if (s != WA_ERR_NONE)
            goto cleanup;
    }

    /* second_ticket */
    webauth_attr_list_find(list, CR_TICKET2, &f);
    if (f != -1) {
        creds->ticket.magic = KV5M_DATA;

        s = webauth_attr_list_get(list, CR_TICKET2,
                                  (void**)&creds->second_ticket.data,
                                  &creds->second_ticket.length, 
                                  WA_F_COPY_VALUE);
        if (s != WA_ERR_NONE)
            goto cleanup;
    }

    /* authdata */
    webauth_attr_list_find(list, CR_NUMAUTHDATA, &f);
    if (f != -1) {
        int num = 0, i;
        char name[32];
        
        s = webauth_attr_list_get_int32(list, CR_NUMAUTHDATA, &num, WA_F_NONE);
        if (s != WA_ERR_NONE)
            goto cleanup;

        /* don't forget to add 1 to num for the null address at the
           end of the list */
        creds->authdata =
            (krb5_authdata **)calloc(num+1, sizeof(krb5_authdata *));
        if (creds->authdata == NULL) {
            s = WA_ERR_NO_MEM;
            goto cleanup;
        }

        for (i=0; i < num; i++) {
            creds->authdata[i] = 
                (krb5_authdata *) malloc(sizeof(krb5_authdata));
            if (creds->authdata[i] == NULL) {
                goto cleanup;
            }

            creds->authdata[i]->magic = KV5M_AUTHDATA;
            sprintf(name, CR_AUTHDATATYPE, i);
            s = webauth_attr_list_get_int32(list, name,
                                            &creds->authdata[i]->ad_type,
                                            WA_F_NONE);
            if (s != WA_ERR_NONE)
                goto cleanup;
            sprintf(name, CR_AUTHDATACONT, i);

            s = webauth_attr_list_get(list, name,
                                      (void**)&creds->authdata[i]->contents,
                                      &creds->authdata[i]->length, 
                                      WA_F_COPY_VALUE);
            if (s != WA_ERR_NONE)
                goto cleanup;
        }
    }

    s = WA_ERR_NONE;

 cleanup:
    if (buff != NULL) 
        free(buff);

    if (list != NULL) 
        webauth_attr_list_free(list);

    if  (s != WA_ERR_NONE)
        krb5_free_cred_contents(c->ctx, creds);

    return s;
}

/* 
 * open up a keytab and return a krb5_principal to use with that
 * keytab. If in _principal is NULL, returned out_principal is first
 * principal found in keytab.
 */
static int
open_keytab(WEBAUTH_KRB5_CTXTP *c,
            const char *keytab_path,
            const char *in_principal,
            krb5_principal *out_principal,
            krb5_keytab *id_out)
{
    krb5_keytab id;
    krb5_kt_cursor cursor;
    krb5_keytab_entry entry;
    krb5_error_code tcode;

    assert(c != NULL);
    assert(keytab_path != NULL);
    assert(out_principal != NULL);
    assert(id_out != NULL);

    c->code = krb5_kt_resolve(c->ctx, keytab_path, &id);
    if (c->code != 0)
        return WA_ERR_KRB5;


    if (in_principal != NULL) {
        c->code = krb5_parse_name(c->ctx, (char *)in_principal, 
                                  out_principal);
    } else {
        c->code = krb5_kt_start_seq_get(c->ctx, id, &cursor);
        if (c->code != 0) {
            /* FIXME: when logging is in, log if error if tcode != 0*/
            tcode = krb5_kt_close(c->ctx, id);
            return WA_ERR_KRB5;
        }

        c->code = krb5_kt_next_entry(c->ctx, id, &entry, &cursor);
        if (c->code == 0) {
            c->code = krb5_copy_principal(c->ctx, entry.principal, 
                                          out_principal);
            /* use tcode fromt this point so we don't lose value of c->code */
            /* FIXME: when logging is in, log if error if tcode != 0 */
#ifdef HAVE_KRB5_FREE_KEYTAB_ENTRY_CONTENTS
            tcode = krb5_free_keytab_entry_contents(c->ctx, &entry);
#else
            tcode = krb5_kt_free_entry(c->ctx, &entry);
#endif 
        }
        /* FIXME: when logging is in, log if error if tcode != 0 */
        tcode = krb5_kt_end_seq_get(c->ctx, id, &cursor);
    }

    if (c->code == 0) {
        *id_out = id;
        return WA_ERR_NONE;
    } else {
        *id_out = NULL;
        tcode = krb5_kt_close(c->ctx, id);
        return WA_ERR_KRB5;
    }
}


/* like krb5_mk_req, but takes a principal instead of a service/host */
static krb5_error_code
mk_req_with_principal(krb5_context context, 
                      krb5_auth_context *auth_context, 
                      krb5_flags ap_req_options, 
                      krb5_principal server, 
                      krb5_data *in_data,
                      krb5_ccache ccache,
                      krb5_data *outbuf)
{
    krb5_error_code 	  retval;
    krb5_creds 		* credsp;
    krb5_creds 		  creds;

    /* obtain ticket & session key */
    memset((char *)&creds, 0, sizeof(creds));
    if ((retval = krb5_copy_principal(context, server, &creds.server)))
        return retval;

    if ((retval = krb5_cc_get_principal(context, ccache, &creds.client)))
	goto cleanup_creds;

    if ((retval = krb5_get_credentials(context, 0,
				       ccache, &creds, &credsp)))
	goto cleanup_creds;

    retval = krb5_mk_req_extended(context, auth_context, ap_req_options, 
				  in_data, credsp, outbuf);

    krb5_free_creds(context, credsp);

 cleanup_creds:
    krb5_free_cred_contents(context, &creds);
    return retval;
}

static int
verify_tgt(WEBAUTH_KRB5_CTXTP *c, const char *keytab_path,
           const char *server_principal,
           char **server_principal_out)
{
    krb5_principal server;
    krb5_keytab keytab;
    krb5_auth_context auth;
    krb5_data outbuf;
    int s;

    assert(c != NULL);
    assert(keytab_path != NULL);
    assert(server_principal_out != NULL);

    *server_principal_out = NULL;

    s = open_keytab(c, keytab_path, server_principal, &server, &keytab);
    if (s != WA_ERR_NONE)
        return s;

    auth = NULL;
    c->code = mk_req_with_principal(c->ctx, &auth, 0, server, 
                                    NULL, c->cc, &outbuf);
    if (c->code != 0) {
        krb5_free_principal(c->ctx, server);
        return WA_ERR_KRB5;
    }

    if (auth != NULL) {
        krb5_auth_con_free(c->ctx, auth);
    }

    auth = NULL;
    c->code = krb5_rd_req(c->ctx, &auth, &outbuf, server, keytab, NULL, NULL);
    if (auth != NULL) {
        krb5_auth_con_free(c->ctx, auth);
    }
                          
    krb5_free_data_contents(c->ctx, &outbuf);
    krb5_kt_close(c->ctx, keytab);

    if (c->code == 0) {
        c->code = krb5_unparse_name(c->ctx, server, server_principal_out);
    }

    krb5_free_principal(c->ctx, server);

    return (c->code == 0) ? WA_ERR_NONE : WA_ERR_KRB5;
}

int
webauth_krb5_new(WEBAUTH_KRB5_CTXT **ctxt)
{
    WEBAUTH_KRB5_CTXTP *c;
    assert(ctxt);

    *ctxt = NULL;

    c = malloc(sizeof(WEBAUTH_KRB5_CTXTP));
    if (c == NULL)
        return WA_ERR_NO_MEM;

    c->cc = NULL;
    c->princ = NULL;
    c->keep_cache = 0;
    c->ctx = NULL;
    c->code = krb5_init_context(&c->ctx);
    *ctxt = (WEBAUTH_KRB5_CTXT*)c;
    return (c->code == 0) ? WA_ERR_NONE : WA_ERR_KRB5;
}

int
webauth_krb5_error_code(WEBAUTH_KRB5_CTXT *context)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    assert(c);
    return c->code;
}

const char *
webauth_krb5_error_message(WEBAUTH_KRB5_CTXT *context)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    assert(c);
    if (c->code == 0) {
        return "success";
    } else {
        return error_message(c->code);
    }
}


int
webauth_krb5_init_via_password(WEBAUTH_KRB5_CTXT *context,
                               const char *username,
                               const char *password,
                               const char *keytab,
                               const char *server_principal,
                               const char *cache_name,
                               char **server_principal_out)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    char ccname[128];
    char *tpassword;
    krb5_creds creds;
    krb5_get_init_creds_opt opts;

    assert(c != NULL);
    assert(username != NULL);
    assert(password != NULL);
    assert(keytab != NULL);
    assert(server_principal_out != NULL);

    c->code = krb5_parse_name(c->ctx, username, &c->princ);

    if (c->code != 0) {
        return WA_ERR_KRB5;
    }

    if (cache_name == NULL) {
        sprintf(ccname, "MEMORY:%p", c);
        cache_name = ccname;
    }

    c->code = krb5_cc_resolve(c->ctx, cache_name, &c->cc);
    if (c->code != 0) 
        return WA_ERR_KRB5;

    c->code = krb5_cc_initialize(c->ctx, c->cc, c->princ);
    if (c->code != 0) 
        return WA_ERR_KRB5;

    krb5_get_init_creds_opt_init(&opts);
    krb5_get_init_creds_opt_set_forwardable(&opts, 1);
    /* FIXME: we'll need to pull some options from config
       once config is in */
    /*krb5_get_init_creds_opt_set_tkt_life(&opts, KRB5_DEFAULT_LIFE);*/

    tpassword = strdup(password);
    if (tpassword == NULL) {
        return WA_ERR_NO_MEM;
    }

    c->code = krb5_get_init_creds_password(c->ctx,
                                           &creds,
                                           c->princ,
                                           (char*)tpassword,
                                           NULL, /* prompter */
                                           NULL, /* data */
                                           0, /* start_time */
                                           NULL, /* in_tkt_service */
                                           &opts);

    memset(tpassword, 0, strlen(tpassword));
    free(tpassword);

    if (c->code != 0) {
        /*printf("code = %d (%s)\n", c->code, error_message(c->code));*/
        switch (c->code) {
            case KRB5KRB_AP_ERR_BAD_INTEGRITY:
            case KRB5KDC_ERR_PREAUTH_FAILED:
            case KRB5KDC_ERR_C_PRINCIPAL_UNKNOWN:
                return WA_ERR_LOGIN_FAILED;
            default:
                /* FIXME: log once logging is in */
                return WA_ERR_KRB5;
        }

    }

    /* add the creds to the cache */
    c->code = krb5_cc_store_cred(c->ctx, c->cc, &creds);
    krb5_free_cred_contents(c->ctx, &creds);
    if (c->code != 0) {
        return WA_ERR_KRB5;
    } else {
        /* lets see if the credentials are valid */
        return verify_tgt(c, keytab, server_principal, server_principal_out);
    }
}

int
webauth_krb5_init_via_cache(WEBAUTH_KRB5_CTXT *context,
                            const char *cache_name)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;

    assert(c != NULL);

    if (cache_name != NULL) {
        c->code = krb5_cc_resolve(c->ctx, cache_name, &c->cc);
    } else {
        c->code = krb5_cc_default(c->ctx, &c->cc);
    }

    if (c->code != 0) 
        return WA_ERR_KRB5;

    c->code = krb5_cc_get_principal(c->ctx, c->cc, &c->princ);

    return (c->code == 0) ? WA_ERR_NONE : WA_ERR_KRB5;
}

int
webauth_krb5_keep_cred_cache(WEBAUTH_KRB5_CTXT *context) 
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    assert(c != NULL);
    c->keep_cache = 1;
    return WA_ERR_NONE;
}

int
webauth_krb5_free(WEBAUTH_KRB5_CTXT *context)
{    
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    assert(c != NULL);

    if (c->cc) {
        if (c->keep_cache) {
            krb5_cc_close(c->ctx, c->cc);
        } else {
            krb5_cc_destroy(c->ctx, c->cc);
        }
    }
    if (c->princ) {
        krb5_free_principal(c->ctx, c->princ);
    }
    if (c->ctx != NULL) {
        krb5_free_context(c->ctx);
    }
    free(context);
    return WA_ERR_NONE;
}

int
webauth_krb5_mk_req(WEBAUTH_KRB5_CTXT *context,
                    const char *server_principal,
                    unsigned char **output,
                    int *length)
{
    return webauth_krb5_mk_req_with_data(context, 
                                         server_principal, output, length,
                                         NULL, 0, NULL, NULL);
#if 0
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_auth_context auth;
    krb5_data outbuf;
    krb5_principal princ;
    int s;

    assert(c != NULL);
    assert(server_principal != NULL);
    assert(output != NULL);
    assert(length != NULL);

    c->code = krb5_parse_name(c->ctx, server_principal, &princ);
    if (c->code != 0)
        return WA_ERR_KRB5;

    auth = NULL;
    c->code = mk_req_with_principal(c->ctx, &auth, 0, princ,
                                    NULL, c->cc, &outbuf);
    krb5_free_principal(c->ctx, princ);

    if (c->code != 0)
        return WA_ERR_KRB5;

    if (auth != NULL)
        krb5_auth_con_free(c->ctx, auth);

    *output = malloc(outbuf.length);
    if (*output == NULL) {
        s = WA_ERR_NO_MEM;
    } else {
        *length = outbuf.length;
        memcpy(*output, outbuf.data, outbuf.length);
        s = WA_ERR_NONE;
    }
    krb5_free_data_contents(c->ctx, &outbuf);
    return s;
#endif 
}


int
webauth_krb5_mk_req_with_data(WEBAUTH_KRB5_CTXT *context,
                              const char *server_principal,
                              unsigned char **output,
                              int *length,
                              unsigned char *in_data,
                              int in_length,
                              unsigned char **out_data,
                              int *out_length)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_auth_context auth;
    krb5_data outbuf;
    krb5_principal princ;
    int s;

    assert(c != NULL);
    assert(server_principal != NULL);
    assert(output != NULL);
    assert(length != NULL);

    memset(&outbuf, 0, sizeof(krb5_data));
    *output = NULL;
    if (out_data)
        *output = NULL;

    c->code = krb5_parse_name(c->ctx, server_principal, &princ);
    if (c->code != 0)
        return WA_ERR_KRB5;

    auth = NULL;
    c->code = mk_req_with_principal(c->ctx, &auth, 0, princ,
                                    NULL, c->cc, &outbuf);
    krb5_free_principal(c->ctx, princ);

    if (c->code != 0)
        return WA_ERR_KRB5;

    *output = malloc(outbuf.length);
    if (*output == NULL) {
        s = WA_ERR_NO_MEM;
        krb5_free_data_contents(c->ctx, &outbuf);
        goto cleanup;
    } else {
        *length = outbuf.length;
        memcpy(*output, outbuf.data, outbuf.length);
        s = WA_ERR_NONE;
        krb5_free_data_contents(c->ctx, &outbuf);
    }

    if (in_data != NULL && out_data != NULL) {
        krb5_data indata, outdata;
        /*krb5_address **laddrs;*/
        krb5_address laddr;
        char lh[4] = {127, 0, 0, 1};

        laddr.magic = KV5M_ADDRESS;
        laddr.addrtype = ADDRTYPE_INET;
        laddr.length = 4;
        laddr.contents = (void*)&lh;

        indata.data = (char *) in_data;
        indata.length = in_length;

        krb5_auth_con_setflags(c->ctx, auth, 0);
        /*krb5_os_localaddr(c->ctx, &laddrs);*/
        /*krb5_auth_con_setaddrs(c->ctx, auth, laddrs[0], NULL);*/
        /*krb5_free_addresses(c->ctx, laddrs);*/
        krb5_auth_con_setaddrs(c->ctx, auth, &laddr, NULL);

        c->code = krb5_mk_priv(c->ctx, auth, &indata, &outdata, NULL);
        if (c->code == 0) {
            s = WA_ERR_NONE;

            *out_data = malloc(outdata.length);
            if (*out_data == NULL) {
                s = WA_ERR_NO_MEM;
            } else {
                *out_length = outdata.length;
                memcpy(*out_data, outdata.data, outdata.length);
                s = WA_ERR_NONE;
            }
            krb5_free_data_contents(c->ctx, &outdata);
        } else {
            s = WA_ERR_KRB5;
        }
    }

 cleanup:

    if (s != WA_ERR_NONE) {
        if (*output != NULL)
            free(*output);
    }
        
    if (auth != NULL)
        krb5_auth_con_free(c->ctx, auth);

    return s;
}

int
webauth_krb5_rd_req(WEBAUTH_KRB5_CTXT *context,
                    const unsigned char *req,
                    int length,
                    const char *keytab_path,
                    const char *server_principal,
                    char **client_principal,
                    int local)
{
    return webauth_krb5_rd_req_with_data(context, req, length, 
                                         keytab_path,
                                         server_principal, 
                                         NULL,
                                         client_principal, local,
                                         NULL, 0, NULL, NULL);
#if 0
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_principal server;
    krb5_keytab keytab;
    krb5_auth_context auth;
    krb5_data buf;
    int s;

    assert(c != NULL);
    assert(keytab_path != NULL);
    assert(req != NULL);
    assert(client_principal);

    s = open_keytab(c, keytab_path, server_principal, &server, &keytab);
    if (s != WA_ERR_NONE)
        return s;

    auth = NULL;

    buf.data = (char*) req;
    buf.length = length;
    c->code = krb5_rd_req(c->ctx, &auth, &buf, server, keytab, NULL, NULL);
    if (c->code == 0) {
        if (auth != NULL) {
            krb5_authenticator *ka;
            c->code = krb5_auth_con_getauthenticator(c->ctx, auth, &ka);
            if (c->code == 0) {
                int local_ok = 0;

                if (local) {
                    krb5_error_code tcode;
                    char lname[256];

                    tcode = krb5_aname_to_localname(c->ctx, ka->client,
                                                    sizeof(lname)-1, lname);
                    if (tcode == 0) {
                        *client_principal = malloc(strlen(lname)+1);
                        strcpy(*client_principal, lname);
                        local_ok = 1;
                    } 
                }

                if (!local_ok)
                    c->code = krb5_unparse_name(c->ctx, ka->client, 
                                                client_principal);

                krb5_free_authenticator(c->ctx, ka);

            } else {
                *client_principal = NULL;
            }
            krb5_auth_con_free(c->ctx, auth);
        }
    }

    krb5_kt_close(c->ctx, keytab);
    krb5_free_principal(c->ctx, server);

    return (c->code == 0) ? WA_ERR_NONE : WA_ERR_KRB5;
#endif
}


int
webauth_krb5_rd_req_with_data(WEBAUTH_KRB5_CTXT *context,
                              const unsigned char *req,
                              int length,
                              const char *keytab_path,
                              const char *server_principal,
                              char **out_server_principal,
                              char **client_principal,
                              int local,
                              unsigned char *in_data,
                              int in_length,
                              unsigned char **out_data,
                              int *out_length)

{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_principal server;
    krb5_keytab keytab;
    krb5_auth_context auth;
    krb5_data buf;
    int s;

    assert(c != NULL);
    assert(keytab_path != NULL);
    assert(req != NULL);
    assert(client_principal);

    s = open_keytab(c, keytab_path, server_principal, &server, &keytab);
    if (s != WA_ERR_NONE)
        return s;

    auth = NULL;

    if (out_server_principal)
        *out_server_principal = NULL;

    buf.data = (char*) req;
    buf.length = length;
    c->code = krb5_rd_req(c->ctx, &auth, &buf, server, keytab, NULL, NULL);
    if (c->code == 0) {
        if (out_server_principal)
            krb5_unparse_name(c->ctx, server, out_server_principal);
        if (auth != NULL) {
            krb5_authenticator *ka;
            c->code = krb5_auth_con_getauthenticator(c->ctx, auth, &ka);
            if (c->code == 0) {
                int local_ok = 0;

                if (local) {
                    krb5_error_code tcode;
                    char lname[256];

                    tcode = krb5_aname_to_localname(c->ctx, ka->client,
                                                    sizeof(lname)-1, lname);
                    if (tcode == 0) {
                        *client_principal = malloc(strlen(lname)+1);
                        strcpy(*client_principal, lname);
                        local_ok = 1;
                    } 
                }

                if (!local_ok)
                    c->code = krb5_unparse_name(c->ctx, ka->client, 
                                                client_principal);

                if (in_data != NULL && out_data != NULL) {
                    krb5_data inbuf, outbuf;
                    krb5_address raddr;
                    char rh[4] = {127, 0, 0, 1};

                    raddr.magic = KV5M_ADDRESS;
                    raddr.addrtype = ADDRTYPE_INET;
                    raddr.length = 4;
                    raddr.contents = (void*)&rh;

                    inbuf.data = (char *) in_data;
                    inbuf.length = in_length;
                    krb5_auth_con_setflags(c->ctx, auth, 0);
                    krb5_auth_con_setaddrs(c->ctx, auth, NULL, &raddr);
                    c->code = krb5_rd_priv(c->ctx, auth, 
                                           &inbuf, &outbuf, NULL);
                    if (c->code == 0) {
                        *out_data = malloc(outbuf.length);
                        if (*out_data == NULL) {
                            s = WA_ERR_NO_MEM;
                        } else {
                            s = WA_ERR_NONE;
                            *out_length = outbuf.length;
                            memcpy(*out_data, outbuf.data, outbuf.length);
                        }
                        krb5_free_data_contents(c->ctx, &outbuf);
                    }

                }
                krb5_free_authenticator(c->ctx, ka);

            } else {
                *client_principal = NULL;
            }
            krb5_auth_con_free(c->ctx, auth);
        }
    }

    krb5_kt_close(c->ctx, keytab);
    krb5_free_principal(c->ctx, server);

    if (s == WA_ERR_NONE && c->code != 0)
        s = WA_ERR_KRB5;

    if (s != WA_ERR_NONE) {
        if (out_server_principal && *out_server_principal != NULL)
            free(*out_server_principal);
    }

    return s;
}

int
webauth_krb5_init_via_keytab(WEBAUTH_KRB5_CTXT *context, 
                             const char *keytab_path,
                             const char *server_principal,
                             const char *cache_name)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    char ccname[128];
    krb5_creds creds;
    krb5_get_init_creds_opt opts;
    krb5_keytab keytab;
    krb5_error_code tcode;
    int s;

    assert(c != NULL);
    assert(keytab_path != NULL);

    if (c->princ != NULL)
        krb5_free_principal(c->ctx, c->princ);

    s = open_keytab(c, keytab_path, server_principal, &c->princ, &keytab);
    if (s != WA_ERR_NONE) 
        return WA_ERR_KRB5;

    if (cache_name == NULL) {
        sprintf(ccname, "MEMORY:%p", c);
        cache_name = ccname;
    }

    c->code = krb5_cc_resolve(c->ctx, cache_name, &c->cc);
    if (c->code != 0) {
        tcode = krb5_kt_close(c->ctx, keytab);
        return WA_ERR_KRB5;
    }

    c->code = krb5_cc_initialize(c->ctx, c->cc, c->princ);
    if (c->code != 0) {
        tcode = krb5_kt_close(c->ctx, keytab);
        return WA_ERR_KRB5;
    }

    krb5_get_init_creds_opt_init(&opts);

    c->code = krb5_get_init_creds_keytab(c->ctx,
                                         &creds,
                                         c->princ,
                                         keytab,
                                         0, /* start_time */
                                         NULL, /* in_tkt_service */
                                         &opts);

    /* FIXME: when logging is in, log if error if tcode != 0*/
    tcode = krb5_kt_close(c->ctx, keytab);


    if (c->code != 0) {
        /*printf("code = %d\n", c->code);*/
        switch (c->code) {
            case KRB5KRB_AP_ERR_BAD_INTEGRITY:
            case KRB5KDC_ERR_PREAUTH_FAILED:
            case KRB5KDC_ERR_C_PRINCIPAL_UNKNOWN:
                return WA_ERR_LOGIN_FAILED;
            default:
                /* FIXME: log once logging is in */
                return WA_ERR_KRB5;
        }

    }

    /* add the creds to the cache */
    c->code = krb5_cc_store_cred(c->ctx, c->cc, &creds);
    krb5_free_cred_contents(c->ctx, &creds);
    if (c->code != 0) {
        return WA_ERR_KRB5;
    } else {
        return WA_ERR_NONE;
    }
}

int
webauth_krb5_init_via_cred(WEBAUTH_KRB5_CTXT *context,
                           unsigned char *cred,
                           int cred_len,
                           const char *cache_name)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_creds creds;
    char ccname[128];
    int s;

    assert(c != NULL);
    assert(cred != NULL);

    s = cred_from_attr_encoding(c, cred, cred_len, &creds);

    if (s!= WA_ERR_NONE) 
        return s;

    if (cache_name == NULL) {
        sprintf(ccname, "MEMORY:%p", c);
        cache_name = ccname;
    }
    c->code = krb5_cc_resolve(c->ctx, cache_name, &c->cc);

    if (c->code != 0) {
        return WA_ERR_KRB5;
    }

    c->code = krb5_copy_principal(c->ctx, creds.client, &c->princ);
    if (c->code != 0) {
        return WA_ERR_KRB5;
    }

    c->code = krb5_cc_initialize(c->ctx, c->cc, c->princ);

    if (c->code != 0) {
        return WA_ERR_KRB5;
    }

    /* add the creds to the cache */
    c->code = krb5_cc_store_cred(c->ctx, c->cc, &creds);
    krb5_free_cred_contents(c->ctx, &creds);

    if (c->code != 0) {
        return WA_ERR_KRB5;
    } else {
        return WA_ERR_NONE;
    }
}

int
webauth_krb5_export_tgt(WEBAUTH_KRB5_CTXT *context,
                        unsigned char **tgt,
                        int *tgt_len,
                        time_t *expiration)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_principal tgtprinc, client;
    krb5_data *client_realm;
    krb5_creds creds, tgtq;
    int s;

    assert(c != NULL);
    assert(tgt != NULL);
    assert(tgt_len != NULL);
    assert(expiration != NULL);

    /* first we need to find tgt in cache */
    c->code = krb5_cc_get_principal(c->ctx, c->cc, &client);
    if (c->code != 0) {
        return WA_ERR_KRB5;
    }

    client_realm = krb5_princ_realm(c->ctx, client);
    c->code = krb5_build_principal_ext(c->ctx,
                                       &tgtprinc,
                                       client_realm->length,
                                       client_realm->data,
                                       KRB5_TGS_NAME_SIZE,
                                       KRB5_TGS_NAME,
                                       client_realm->length,
                                       client_realm->data,
                                       0);

    if (c->code != 0) {
        krb5_free_principal(c->ctx, client);        
        return WA_ERR_KRB5;
    }

    memset(&tgtq, 0, sizeof(tgtq));
    memset(&creds, 0, sizeof(creds));

    tgtq.server = tgtprinc;
    tgtq.client = client;

    c->code = krb5_cc_retrieve_cred(c->ctx, 
                                    c->cc,
                                    KRB5_TC_MATCH_SRV_NAMEONLY,
                                    &tgtq,
                                    &creds);

    if (c->code == 0) {
        s = cred_to_attr_encoding(c, &creds, tgt, tgt_len, expiration);
        krb5_free_cred_contents(c->ctx, &creds);
    } else {
        s = WA_ERR_KRB5;
    }

    krb5_free_principal(c->ctx, client);
    krb5_free_principal(c->ctx, tgtprinc);

    return s;
}

int
webauth_krb5_import_cred(WEBAUTH_KRB5_CTXT *context,
                         unsigned char *cred,
                         int cred_len)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_creds creds;
    int s;

    assert(c != NULL);
    assert(cred != NULL);

    s = cred_from_attr_encoding(c, cred, cred_len, &creds);
    if (s!= WA_ERR_NONE) 
        return s;

    /* add the creds to the cache */
    c->code = krb5_cc_store_cred(c->ctx, c->cc, &creds);
    krb5_free_cred_contents(c->ctx, &creds);
    if (c->code != 0) {
        return WA_ERR_KRB5;
    } else {
        return WA_ERR_NONE;
    }
}

int
webauth_krb5_export_ticket(WEBAUTH_KRB5_CTXT *context,
                           char *server_principal,
                           unsigned char **ticket,
                           int *ticket_len,
                           time_t *expiration)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_creds *credsp, creds;
    int s;

    s = WA_ERR_KRB5;
    memset((char *)&creds, 0, sizeof(creds));

    c->code = krb5_parse_name(c->ctx, server_principal, &creds.server);
    if (c->code != 0)
        goto cleanup_creds;

    c->code = krb5_cc_get_principal(c->ctx, c->cc, &creds.client);
    if (c->code != 0)
	goto cleanup_creds;

    c->code = krb5_get_credentials(c->ctx, 0, c->cc, &creds, &credsp);
    if (c->code != 0)
	goto cleanup_creds;

    s = cred_to_attr_encoding(c, credsp, ticket, ticket_len, expiration);
    krb5_free_creds(c->ctx, credsp);

 cleanup_creds:
    krb5_free_cred_contents(c->ctx, &creds);
    return s;
}

int
webauth_krb5_service_principal(WEBAUTH_KRB5_CTXT *context,
                               const char *service,
                               const char *hostname,
                               char **server_principal)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;
    krb5_principal princ;

    c->code = krb5_sname_to_principal(c->ctx,
                                      hostname,
                                      service,
                                      KRB5_NT_SRV_HST,
                                      &princ);
    if (c->code != 0)
        return WA_ERR_KRB5;

    c->code = krb5_unparse_name(c->ctx, princ, server_principal);
    krb5_free_principal(c->ctx, princ);

    return c->code == 0 ? WA_ERR_NONE : WA_ERR_KRB5;
}

int
webauth_krb5_get_principal(WEBAUTH_KRB5_CTXT *context,
                               char **principal, int local)
{
    WEBAUTH_KRB5_CTXTP *c = (WEBAUTH_KRB5_CTXTP*)context;

    if (c->princ == NULL)
        return WA_ERR_INVALID_CONTEXT;

    if (local) {
        krb5_error_code tcode;
        char lname[256];

        tcode = krb5_aname_to_localname(c->ctx, c->princ,
                                        sizeof(lname)-1,
                                        lname);
        if (tcode == 0) {
            *principal = malloc(strlen(lname)+1);
            strcpy(*principal, lname);
            return WA_ERR_NONE;
        } 
    }

    /* fall through to fully-qualified on krb5_aname_to_localname errors */
    c->code = krb5_unparse_name(c->ctx, c->princ, principal);
    return c->code == 0 ? WA_ERR_NONE : WA_ERR_KRB5;
}
