/*
 * This file is part of the webaccounts-browser-plugin.
 * Copyright (C) Canonical Ltd. 2012
 *
 * Author: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include "fake-capture-service.h"
#include "login-handler.h"

#include <glib-object.h>
#include <gtest/gtest.h>
#include <libaccounts-glib/ag-account.h>
#include <libaccounts-glib/ag-manager.h>

TEST (npapi, wrong_domain) {
    FakeCaptureService service;
    static const gchar json_string[] = "{"
        "  'login': 'my user name', "
        "  'domain': 'invalid.com' "
        "}";

    fake_capture_service_init (&service);

    /* This should fail because there isn't a provider handling the domain
     * "invalid.com" */
    webaccounts_login_handler_set_json (json_string);
    EXPECT_FALSE (service.called);

    fake_capture_service_unset (&service);
}

TEST (npapi, existing_account) {
    AgManager *manager;
    AgAccount *account;
    FakeCaptureService service;

    static const gchar json_string[] = "{"
        "  'login': 'user@test.com', "
        "  'domain': 'example.com' "
        "}";

    /* first, create an account for user@test.com */
    manager = ag_manager_new ();
    ASSERT_TRUE (manager != NULL);

    account = ag_manager_create_account (manager, "MyProvider");
    ASSERT_TRUE (account != NULL);
    ag_account_set_display_name (account, "user@test.com");
    ag_account_store_blocking (account, NULL);

    g_object_unref (account);
    g_object_unref (manager);

    fake_capture_service_init (&service);

    /* This should fail because an account for that username already exists */
    webaccounts_login_handler_set_json (json_string);
    EXPECT_FALSE (service.called);

    fake_capture_service_unset (&service);
}

TEST (npapi, username) {
    FakeCaptureService service;
    static const gchar json_string[] = "{"
        "  'login': 'my user name', "
        "  'domain': 'example.com' "
        "}";

    fake_capture_service_init (&service);

    webaccounts_login_handler_set_json (json_string);
    EXPECT_TRUE (service.called);
    EXPECT_STREQ ("my user name", service.username);
    EXPECT_STREQ ("MyProvider", service.provider_name);

    fake_capture_service_unset (&service);
}

TEST (npapi, blacklist) {
    FakeCaptureService service;
    static const gchar json_string[] = "{"
        "  'login': 'my user name', "
        "  'domain': 'example.com' "
        "}";

    fake_capture_service_init (&service);

    webaccounts_login_handler_set_json (json_string);
    EXPECT_FALSE (service.called);

    fake_capture_service_unset (&service);
}

TEST (npapi, cookies) {
    FakeCaptureService service;
    gboolean ok;
    gchar *cookie;
    static const gchar json_string[] = "{"
        "  'login': 'Happy User', "
        "  'domain': 'example.com', "
        "  'cookies': { "
        "    'example.com': 'a list of cookies', "
        "    'test.com': 'more cookies' "
        "  } "
        "}";

    fake_capture_service_init (&service);

    webaccounts_login_handler_set_json (json_string);
    EXPECT_TRUE (service.called);
    EXPECT_STREQ ("Happy User", service.username);
    EXPECT_STREQ ("MyProvider", service.provider_name);
    ASSERT_TRUE (service.cookies != NULL);

    /* check the cookie contents */
    ok = g_variant_lookup (service.cookies, "example.com", "s", &cookie);
    EXPECT_TRUE (ok);
    EXPECT_STREQ ("a list of cookies", cookie);
    g_free (cookie);

    ok = g_variant_lookup (service.cookies, "test.com", "s", &cookie);
    EXPECT_TRUE (ok);
    EXPECT_STREQ ("more cookies", cookie);
    g_free (cookie);

    fake_capture_service_unset (&service);
}

int main (int argc, char **argv)
{
    ::testing::InitGoogleTest (&argc, argv);

    g_type_init ();

    return RUN_ALL_TESTS ();
}
