# -*- coding: utf-8 -*-
"""
web2ldap plugin classes for schema elements defined in RFC2307

$Id: nis.py,v 1.20 2012/05/28 12:29:02 michael Exp $
"""

import re,pyweblib.forms

from w2lapp.schema.syntaxes import SelectList,IA5String,IntegerRange,IPServicePortNumber, \
                         DaysSinceEpoch,DNSDomain,DynamicValueSelectList,\
                         syntax_registry

import ldaputil.schema

class RFC2307BootParameter(IA5String):
  oid = '1.3.6.1.1.1.0.1'
  desc = 'RFC2307 Boot Parameter'
  reObj=None # just a stub, should be made stricter


class GidNumber(DynamicValueSelectList,IntegerRange):
  oid = 'GidNumber-oid'
  desc = 'RFC2307: An integer uniquely identifying a group in an administrative domain'
  minValue = 0
  maxValue = 4294967295L
  ldap_url = 'ldap:///_?gidNumber,cn?sub?(objectClass=posixGroup)'

  def _validate(self,attrValue):
    return IntegerRange._validate(self,attrValue)

  def _get_ocs(self):
    return [ oc.lower() for oc in self._entry.get('objectClass',[]) ]

  def _doSearch(self,initial_attr_value_dict=None):
    # Enable empty value in any case
    self.attr_value_dict = initial_attr_value_dict or {}
    ocs = self._get_ocs()
    if 'posixaccount' in ocs or 'shadowaccount' in ocs:
      DynamicValueSelectList._doSearch(self,initial_attr_value_dict)

  def displayValue(self,valueindex=0,commandbutton=0):
    # Possibly display a link
    ocs = self._get_ocs()
    if 'posixaccount' in ocs or 'shadowaccount' in ocs:
      return DynamicValueSelectList.displayValue(self,valueindex,commandbutton=0)
    else:
      r = [IntegerRange.displayValue(self,valueindex,commandbutton=0)]
      if not commandbutton:
        return r[0]
      if 'posixgroup' in ocs:
        title = u'Search group members'
        filterstr_template = '(&(|(objectClass=posixAccount)(objectClass=shadowAccount))(gidNumber=%s))'
      else:
        title = None
        filterstr_template = None
      if title and filterstr_template:
        r.append(self._form.applAnchor(
            'searchform','&raquo;',self._sid,
            [
              ('dn',self._dn),
              ('filterstr', filterstr_template % (
                  self._form.utf2display(self._ls.uc_decode(self.attrValue)[0])
                )
              ),
              ('searchform_mode','exp'),
            ],
            title=title,
        ))
      return ' '.join(r)

  def formField(self):
    ocs = self._get_ocs()
    if 'posixaccount' in ocs or 'shadowaccount' in ocs:
      return DynamicValueSelectList.formField(self)
    else:
      return IntegerRange.formField(self)

syntax_registry.registerAttrType(
  GidNumber.oid,[
    '1.3.6.1.1.1.1.1', # gidNumber
  ]
)


class MemberUID(IA5String):
  oid = 'MemberUID-oid'
  desc = 'RFC2307 numerical UID of group member(s)'

  def displayValue(self,valueindex=0,commandbutton=0):
    r = [IA5String.displayValue(self,valueindex,commandbutton=0)]
    if commandbutton:
      r.append(self._form.applAnchor(
          'searchform','&raquo;',self._sid,
          [
            ('dn',self._dn),
            ('filterstr','(&(objectClass=posixAccount)(uid=%s))' % (
                self._form.utf2display(self._ls.uc_decode(self.attrValue)[0])
              )
            ),
            ('searchform_mode','exp'),
          ],
          title=u'Search for user entry',
      ))
    return ' '.join(r)

syntax_registry.registerAttrType(
  MemberUID.oid,[
    '1.3.6.1.1.1.1.12', # memberUID
  ]
)


class RFC2307NISNetgroupTriple(IA5String):
  oid = '1.3.6.1.1.1.0.0'
  desc = 'RFC2307 NIS Netgroup Triple'
  reObj=re.compile('^\([a-z0-9.-]*,[a-z0-9.-]*,[a-z0-9.-]*\)$')


class UidNumber(IntegerRange):
  oid = 'UidNumber-oid'
  desc = 'Numerical user ID for Posix systems'
  minValue = 0
  maxValue = 4294967295L

syntax_registry.registerAttrType(
  UidNumber.oid,[
    '1.3.6.1.1.1.1.0', # uidNumber
  ]
)


class Shell(SelectList):
  oid = 'Shell-oid'
  desc = 'Shell for user of Posix systems'

  attr_value_dict = {
    u'/bin/bash':u'Bourne-Again SHell /bin/bash',
    u'/bin/csh':u'/bin/csh',
    u'/bin/tcsh':u'/bin/tcsh',
    u'/bin/ksh':u'Korn shell /bin/ksh',
    u'/bin/passwd':u'Password change /bin/passwd',
    u'/bin/true':u'/bin/true',
    u'/bin/false':u'/bin/false',
    u'/bin/zsh':u'Zsh /bin/zsh',
    u'/usr/bin/bash':u'Bourne-Again SHell /usr/bin/bash',
    u'/usr/bin/csh':u'/usr/bin/csh',
    u'/usr/bin/tcsh':u'/usr/bin/csh',
    u'/usr/bin/ksh':u'Korn shell /usr/bin/ksh',
    u'/usr/bin/zsh':u'Zsh /usr/bin/zsh',
  }

syntax_registry.registerAttrType(
  Shell.oid,[
    '1.3.6.1.1.1.1.4', # loginShell
  ]
)


syntax_registry.registerAttrType(
  DNSDomain.oid,[
    '1.3.6.1.1.1.1.30', # nisDomain
  ]
)


syntax_registry.registerAttrType(
  DaysSinceEpoch.oid,[
    '1.3.6.1.1.1.1.10', # shadowExpire
    '1.3.6.1.1.1.1.5', # shadowLastChange
  ]
)


syntax_registry.registerAttrType(
  IPServicePortNumber.oid,[
    '1.3.6.1.1.1.1.15', # ipServicePort
  ]
)


# Register all syntax classes in this module
for name in dir():
  syntax_registry.registerSyntaxClass(eval(name))



