'''
sitemapReader.py

Copyright 2006 Andres Riancho

This file is part of w3af, w3af.sourceforge.net .

w3af is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation version 2 of the License.

w3af is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with w3af; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

'''

import core.controllers.outputManager as om

# options
from core.data.options.option import option
from core.data.options.optionList import optionList

from core.controllers.basePlugin.baseDiscoveryPlugin import baseDiscoveryPlugin
import core.data.kb.knowledgeBase as kb
import core.data.parsers.urlParser as urlParser
from core.controllers.w3afException import w3afException, w3afRunOnce


class sitemapReader(baseDiscoveryPlugin):
    '''
    Analyze the sitemap.xml file and find new URLs
    
    @author: Andres Riancho ( andres.riancho@gmail.com )
    '''

    def __init__(self):
        baseDiscoveryPlugin.__init__(self)
        
        # Internal variables
        self._exec = True

    def discover(self, fuzzableRequest ):
        '''
        Get the sitemap.xml file and parse it.
        
        @parameter fuzzableRequest: A fuzzableRequest instance that contains (among other things) the URL to test.
        '''
        if not self._exec:
            # This will remove the plugin from the discovery plugins to be runned.
            raise w3afRunOnce()
        else:
            # Only run once
            self._exec = False
            self._fuzzableRequests = []
            is_404 = kb.kb.getData( 'error404page', '404' )
            
            base_url = urlParser.baseUrl( fuzzableRequest.getURL() )
            sitemap_url = urlParser.urlJoin(  base_url , 'sitemap.xml' )
            response = self._urlOpener.GET( sitemap_url, useCache=True )
            
            # Remember that httpResponse objects have a faster "__in__" than
            # the one in strings; so string in response.getBody() is slower than
            # string in response
            if '</urlset>' in response and not is_404( response ):
                om.out.debug('Analyzing sitemap.xml file.')
                
                self._fuzzableRequests.extend( self._createFuzzableRequests( response ) )
                
                import xml.dom.minidom
                om.out.debug('Parsing xml file with xml.dom.minidom.')
                try:
                    dom = xml.dom.minidom.parseString( response.getBody() )
                except:
                    raise w3afException('Error while parsing sitemap.xml')
                urlList = dom.getElementsByTagName("loc")
                for url in urlList:
                    url = url.childNodes[0].data 
                    om.out.information( 'sitemapReader found a new URL: ' + url )
                    response = self._urlOpener.GET( url, useCache=True )
                    self._fuzzableRequests.extend( self._createFuzzableRequests( response ) )

        return self._fuzzableRequests
        
    def getOptions( self ):
        '''
        @return: A list of option objects for this plugin.
        '''    
        ol = optionList()
        return ol
        
    def setOptions( self, OptionList ):
        '''
        This method sets all the options that are configured using the user interface 
        generated by the framework using the result of getOptions().
        
        @parameter OptionList: A dictionary with the options for the plugin.
        @return: No value is returned.
        ''' 
        pass

    def getPluginDeps( self ):
        '''
        @return: A list with the names of the plugins that should be runned before the
        current one.
        '''
        return []
    
    def getLongDesc( self ):
        '''
        @return: A DETAILED description of the plugin functions and features.
        '''
        return '''
        This plugin searches for the sitemap.xml file, and parses it.
        
        The sitemap.xml file is used by the site administrator to give the Google crawler more
        information about the site. By parsing this file, the plugin finds new URLs and other
        usefull information.
        '''
