'''
domain_dot.py

Copyright 2006 Andres Riancho

This file is part of w3af, w3af.sourceforge.net .

w3af is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation version 2 of the License.

w3af is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with w3af; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

'''

import core.controllers.outputManager as om

# options
from core.data.options.option import option
from core.data.options.optionList import optionList

from core.controllers.basePlugin.baseDiscoveryPlugin import baseDiscoveryPlugin

import core.data.kb.knowledgeBase as kb
import core.data.kb.info as info
import core.data.constants.severity as severity

import core.data.parsers.urlParser as urlParser
from core.controllers.w3afException import w3afException
from core.controllers.misc.levenshtein import relative_distance


class domain_dot(baseDiscoveryPlugin):
    '''
    Send a specially crafted request with a dot after the domain (http://host.tld./) and analyze response.
    
    @author: Andres Riancho ( andres.riancho@gmail.com )
    '''

    def __init__(self):
        baseDiscoveryPlugin.__init__(self)
        
        # Internal variables
        self._already_tested = []

    def discover(self, fuzzableRequest ):
        '''
        Sends the special request.
        
        @parameter fuzzableRequest: A fuzzableRequest instance that contains
                                                    (among other things) the URL to test.
        '''
        domain = urlParser.getDomain(fuzzableRequest.getURL())
        extension = urlParser.getDomain(fuzzableRequest.getURL())
        
        if (domain, extension) not in self._already_tested:
            
            # Do it only one time
            self._already_tested.append( (domain, extension) )
            
            # Generate the new URL
            domain += '.'
            path = urlParser.getPath( fuzzableRequest.getURL() )
            protocol = urlParser.getProtocol( fuzzableRequest.getURL() )
            new_URL = protocol + '://' + domain + path
            try:
                # GET the original response
                original_response = self._urlOpener.GET( fuzzableRequest.getURL(), useCache=False )
                # GET the response with the modified domain (with the trailing dot)
                response = self._urlOpener.GET( new_URL, useCache=False )
            except KeyboardInterrupt,e:
                raise e
            except w3afException,w3:
                om.out.error( str(w3) )
            else:
                self._analyze_response( original_response, response )

    def _analyze_response(self, original_response, response):
        '''
        @parameter response: The httpResponse object that holds the ORIGINAL response.
        @parameter response: The httpResponse object that holds the content of the response to analyze.
        '''
        if relative_distance( original_response.getBody(), response.getBody() ) < 0.7:
            i = info.info( response )
            i.setId( [original_response.id, response.id] )
            i.setName( 'Responses differ' )
            msg = '[Manual verification required] The response body for a request with a trailing'
            msg += ' dot in the domain, and the response body without a trailing dot in the domain'
            msg += ' differ. This could indicate a misconfiguration in the virtual host settings.'
            msg += ' In some cases, this misconfiguration permits the attacker to read the source'
            msg += ' code of the web application.'
            i.setDesc( msg )
            
            om.out.information( i.getDesc() )
            
            kb.kb.append( self, 'domain_dot', i )
                
    def getOptions( self ):
        '''
        @return: A list of option objects for this plugin.
        '''    
        ol = optionList()
        return ol

    def setOptions( self, OptionList ):
        '''
        This method sets all the options that are configured using the user interface 
        generated by the framework using the result of getOptions().
        
        @parameter OptionList: A dictionary with the options for the plugin.
        @return: No value is returned.
        ''' 
        pass

    def getPluginDeps( self ):
        '''
        @return: A list with the names of the plugins that should be runned before the
        current one.
        '''
        return []
        
    def getLongDesc( self ):
        '''
        @return: A DETAILED description of the plugin functions and features.
        '''
        return '''
        This plugin finds misconfigurations in the virtual host settings by sending a specially crafted
        request with a trailing dot in the domain name. For example, if the input for this plugin is
        http://host.tld/ , the plugin will perform a request to http://host.tld./ .
        
        In some misconfigurations, the attacker is able to read the web application source code by
        requesting any of the files in the "dotted" domain like this:
            - http://host.tld/login.php
        '''
