# ###################################################
# Copyright (C) 2012 The Unknown Horizons Team
# team@unknown-horizons.org
# This file is part of Unknown Horizons.
#
# Unknown Horizons is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
# ###################################################

import re

import horizons.globals
from horizons.constants import GROUND
from horizons.ext.dummy import Dummy
from horizons.gui.keylisteners import IngameKeyListener
from horizons.gui.tabs import TabWidget
from horizons.gui.tabs.tabinterface import TabInterface
from horizons.gui.util import LazyWidgetsDict
from horizons.gui.widgets.imagebutton import OkButton, CancelButton
from horizons.gui.widgets.minimap import Minimap
from horizons.util.living import LivingObject, livingProperty
from horizons.util.loaders.tilesetloader import TileSetLoader
from horizons.util.python.callback import Callback


class IngameGui(LivingObject):
	minimap = livingProperty()
	keylistener = livingProperty()

	def __init__(self, session, main_gui):
		self.session = session
		self.main_gui = main_gui
		self.keylistener = IngameKeyListener(self.session)

		# Mocks needed to act like the real IngameGui
		self.message_widget = Dummy
		self.show_menu = Dummy
		self.hide_menu = Dummy

		self.widgets = LazyWidgetsDict({})
		minimap = self.widgets['minimap']
		minimap.position_technique = "right+0:top+0"

		icon = minimap.findChild(name="minimap")
		self.minimap = Minimap(icon,
		                       targetrenderer=horizons.globals.fife.targetrenderer,
		                       imagemanager=horizons.globals.fife.imagemanager,
		                       session=self.session,
		                       view=self.session.view)

		minimap.mapEvents({
			'zoomIn': self.session.view.zoom_in,
			'zoomOut': self.session.view.zoom_out,
			'rotateRight': Callback.ChainedCallbacks(self.session.view.rotate_right, self.minimap.rotate_right),
			'rotateLeft': Callback.ChainedCallbacks(self.session.view.rotate_left, self.minimap.rotate_left),
			'gameMenuButton' : self.main_gui.toggle_pause,
		})

		minimap.show()

		# Hide unnecessary buttons in hud
		for widget in ("build", "speedUp", "speedDown", "destroy_tool", "diplomacyButton", "logbook"):
			self.widgets['minimap'].findChild(name=widget).hide()

		self.save_map_dialog = SaveMapDialog(self.main_gui, self.session, self.widgets['save_map'])

	def end(self):
		self.widgets['minimap'].mapEvents({
			'zoomIn': None,
			'zoomOut': None,
			'rotateRight': None,
			'rotateLeft': None,
			'gameMenuButton': None
		})
		self.minimap = None
		self.keylistener = None
		super(IngameGui, self).end()

	def load(self, savegame):
		self.minimap.draw()

	def setup(self):
		"""Called after the world editor was initialized."""
		self._settings_tab = TabWidget(self, tabs=[SettingsTab(self.session.world_editor, self.session)])
		self._settings_tab.show()

	def minimap_to_front(self):
		"""Make sure the full right top gui is visible and not covered by some dialog"""
		self.widgets['minimap'].hide()
		self.widgets['minimap'].show()

	def show_save_map_dialog(self):
		"""Shows a dialog where the user can set the name of the saved map."""
		self.save_map_dialog.show()

	def on_escape(self):
		pass

	def on_key_press(self, action, evt):
		pass


class SettingsTab(TabInterface):
	widget = 'editor_settings.xml'

	def __init__(self, world_editor, session):
		super(SettingsTab, self).__init__(widget=self.widget)

		self._world_editor = world_editor

		# Brush size
		for i in range(1, 4):
			b = self.widget.findChild(name='size_%d' % i)
			b.capture(Callback(self._change_brush_size, i))

		# Activate radio button for default brush size
		self._change_brush_size(self._world_editor.brush_size)

		# Tile selection
		for tile_type in ('default_land', 'sand', 'shallow_water', 'water'):
			image = self.widget.findChild(name=tile_type)
			tile = getattr(GROUND, tile_type.upper())
			image.up_image = self._get_tile_image(tile)
			image.size = image.min_size = image.max_size = (64, 32)
			image.capture(Callback(session.set_cursor, 'tile_layer', tile))

	def _get_tile_image(self, tile):
		# TODO TileLayingTool does almost the same thing, perhaps put this in a better place
		tile_sets = TileSetLoader.get_sets()

		ground_id, action_id, rotation = tile
		set_id = horizons.globals.db.get_random_tile_set(ground_id)
		return tile_sets[set_id][action_id][rotation].keys()[0]

	def _change_brush_size(self, size):
		"""Change the brush size and update the gui."""
		images = {
		  'box_highlighted': 'content/gui/icons/ship/smallbutton_a.png',
		  'box': 'content/gui/icons/ship/smallbutton.png',
		}

		b = self.widget.findChild(name='size_%d' % self._world_editor.brush_size)
		b.up_image = images['box']

		self._world_editor.brush_size = size
		b = self.widget.findChild(name='size_%d' % self._world_editor.brush_size)
		b.up_image = images['box_highlighted']


class SaveMapDialog(object):
	"""Shows a dialog where the user can set the name of the saved map."""

	def __init__(self, main_gui, session, widget):
		self._main_gui = main_gui
		self._session = session
		self._widget = widget

		events = {
			OkButton.DEFAULT_NAME: self._do_save,
			CancelButton.DEFAULT_NAME: self.hide
		}
		self._widget.mapEvents(events)

	def show(self):
		self._main_gui.on_escape = self.hide

		name = self._widget.findChild(name='map_name')
		name.text = u''
		name.capture(self._do_save)

		self._widget.show()
		name.requestFocus()

	def hide(self):
		self._main_gui.on_escape = self._main_gui.toggle_pause
		self._widget.hide()

	def _do_save(self):
		name = self._widget.collectData('map_name')
		if re.match('^[a-zA-Z0-9_-]+$', name):
			self._session.save_map(name)
			self.hide()
		else:
			#xgettext:python-format
			message = _('Valid map names are in the following form: {expression}').format(expression='[a-zA-Z0-9_-]+')
			#xgettext:python-format
			advice = _('Try a name that only contains letters and numbers.')
			self._main_gui.show_error_popup(_('Error'), message, advice)
